// SPDX-License-Identifier: BSD-3-Clause
/*
 * Clock drivers for Qualcomm SDM845
 *
 * (C) Copyright 2017 Jorge Ramirez Ortiz <jorge.ramirez-ortiz@linaro.org>
 * (C) Copyright 2021 Dzmitry Sankouski <dsankouski@gmail.com>
 *
 * Based on Little Kernel driver, simplified
 */

#include <clk-uclass.h>
#include <dm.h>
#include <linux/delay.h>
#include <errno.h>
#include <asm/io.h>
#include <linux/bitops.h>
#include <dt-bindings/clock/qcom,gcc-sdm845.h>

#include "clock-qcom.h"

#define SE9_UART_APPS_CMD_RCGR	0x18148

#define USB30_PRIM_MASTER_CLK_CMD_RCGR 0xf018
#define USB30_PRIM_MOCK_UTMI_CLK_CMD_RCGR 0xf030
#define USB3_PRIM_PHY_AUX_CMD_RCGR 0xf05c
#define SDCC1_APPS_CLK_CMD_RCGR 0x26028
#define SDCC2_APPS_CLK_CMD_RCGR 0x1400c

static const struct freq_tbl ftbl_gcc_qupv3_wrap0_s0_clk_src[] = {
	F(7372800, CFG_CLK_SRC_GPLL0_EVEN, 1, 384, 15625),
	F(14745600, CFG_CLK_SRC_GPLL0_EVEN, 1, 768, 15625),
	F(19200000, CFG_CLK_SRC_CXO, 1, 0, 0),
	F(29491200, CFG_CLK_SRC_GPLL0_EVEN, 1, 1536, 15625),
	F(32000000, CFG_CLK_SRC_GPLL0_EVEN, 1, 8, 75),
	F(48000000, CFG_CLK_SRC_GPLL0_EVEN, 1, 4, 25),
	F(64000000, CFG_CLK_SRC_GPLL0_EVEN, 1, 16, 75),
	F(80000000, CFG_CLK_SRC_GPLL0_EVEN, 1, 4, 15),
	F(96000000, CFG_CLK_SRC_GPLL0_EVEN, 1, 8, 25),
	F(100000000, CFG_CLK_SRC_GPLL0_EVEN, 3, 0, 0),
	F(102400000, CFG_CLK_SRC_GPLL0_EVEN, 1, 128, 375),
	F(112000000, CFG_CLK_SRC_GPLL0_EVEN, 1, 28, 75),
	F(117964800, CFG_CLK_SRC_GPLL0_EVEN, 1, 6144, 15625),
	F(120000000, CFG_CLK_SRC_GPLL0_EVEN, 2.5, 0, 0),
	F(128000000, CFG_CLK_SRC_GPLL0, 1, 16, 75),
	{ }
};

static const struct freq_tbl ftbl_gcc_sdcc1_apps_clk_src[] = {
	F(144000, CFG_CLK_SRC_CXO, 16, 3, 25),
	F(400000, CFG_CLK_SRC_CXO, 12, 1, 4),
	F(20000000, CFG_CLK_SRC_GPLL0_EVEN, 5, 1, 3),
	F(25000000, CFG_CLK_SRC_GPLL0_EVEN, 6, 1, 2),
	F(50000000, CFG_CLK_SRC_GPLL0_EVEN, 6, 0, 0),
	F(100000000, CFG_CLK_SRC_GPLL0, 6, 0, 0),
	F(192000000, CFG_CLK_SRC_GPLL6, 2, 0, 0),
	F(384000000, CFG_CLK_SRC_GPLL6, 1, 0, 0),
	{ }
};

static const struct freq_tbl ftbl_gcc_sdcc2_apps_clk_src[] = {
	F(400000, CFG_CLK_SRC_CXO, 12, 1, 4),
	F(9600000, CFG_CLK_SRC_CXO, 2, 0, 0),
	F(19200000, CFG_CLK_SRC_CXO, 1, 0, 0),
	F(25000000, CFG_CLK_SRC_GPLL0_EVEN, 12, 0, 0),
	F(50000000, CFG_CLK_SRC_GPLL0_EVEN, 6, 0, 0),
	F(100000000, CFG_CLK_SRC_GPLL0, 6, 0, 0),
	F(201500000, CFG_CLK_SRC_GPLL4, 4, 0, 0),
	{ }
};

static ulong sdm670_clk_set_rate(struct clk *clk, ulong rate)
{
	struct msm_clk_priv *priv = dev_get_priv(clk->dev);
	const struct freq_tbl *freq;

	switch (clk->id) {
	case GCC_SDCC1_APPS_CLK:
		freq = qcom_find_freq(ftbl_gcc_sdcc1_apps_clk_src, rate);
		clk_rcg_set_rate_mnd(priv->base, SDCC1_APPS_CLK_CMD_RCGR,
				     freq->pre_div, freq->m, freq->n, freq->src, 8);
		return freq->freq;
	default:
		return 0;
	}
}

static ulong sdm845_clk_set_rate(struct clk *clk, ulong rate)
{
	struct msm_clk_priv *priv = dev_get_priv(clk->dev);
	const struct freq_tbl *freq;

	switch (clk->id) {
	case GCC_QUPV3_WRAP1_S1_CLK: /* UART9 */
		freq = qcom_find_freq(ftbl_gcc_qupv3_wrap0_s0_clk_src, rate);
		clk_rcg_set_rate_mnd(priv->base, SE9_UART_APPS_CMD_RCGR,
				     freq->pre_div, freq->m, freq->n, freq->src, 16);
		return freq->freq;
	case GCC_SDCC2_APPS_CLK:
		freq = qcom_find_freq(ftbl_gcc_sdcc2_apps_clk_src, rate);
		clk_rcg_set_rate_mnd(priv->base, SDCC2_APPS_CLK_CMD_RCGR,
				     freq->pre_div, freq->m, freq->n, freq->src, 8);
		return freq->freq;
	default:
		return 0;
	}
}

static const struct gate_clk sdm670_clks[] = {
	GATE_CLK(GCC_AGGRE_USB3_PRIM_AXI_CLK,		0x8201c, 0x00000001),
	GATE_CLK(GCC_CFG_NOC_USB3_PRIM_AXI_CLK,		0x0502c, 0x00000001),
	GATE_CLK(GCC_QUPV3_WRAP0_S0_CLK,		0x5200c, 0x00000400),
	GATE_CLK(GCC_QUPV3_WRAP0_S1_CLK,		0x5200c, 0x00000800),
	GATE_CLK(GCC_QUPV3_WRAP0_S2_CLK,		0x5200c, 0x00001000),
	GATE_CLK(GCC_QUPV3_WRAP0_S3_CLK,		0x5200c, 0x00002000),
	GATE_CLK(GCC_QUPV3_WRAP0_S4_CLK,		0x5200c, 0x00004000),
	GATE_CLK(GCC_QUPV3_WRAP0_S5_CLK,		0x5200c, 0x00008000),
	GATE_CLK(GCC_QUPV3_WRAP0_S6_CLK,		0x5200c, 0x00010000),
	GATE_CLK(GCC_QUPV3_WRAP0_S7_CLK,		0x5200c, 0x00020000),
	GATE_CLK(GCC_QUPV3_WRAP1_S0_CLK,		0x5200c, 0x00400000),
	GATE_CLK(GCC_QUPV3_WRAP1_S1_CLK,		0x5200c, 0x00800000),
	GATE_CLK(GCC_QUPV3_WRAP1_S3_CLK,		0x5200c, 0x02000000),
	GATE_CLK(GCC_QUPV3_WRAP1_S4_CLK,		0x5200c, 0x04000000),
	GATE_CLK(GCC_QUPV3_WRAP1_S5_CLK,		0x5200c, 0x08000000),
	GATE_CLK(GCC_QUPV3_WRAP1_S6_CLK,		0x5200c, 0x10000000),
	GATE_CLK(GCC_QUPV3_WRAP1_S7_CLK,		0x5200c, 0x20000000),
	GATE_CLK(GCC_QUPV3_WRAP_0_M_AHB_CLK,		0x5200c, 0x00000040),
	GATE_CLK(GCC_QUPV3_WRAP_0_S_AHB_CLK,		0x5200c, 0x00000080),
	GATE_CLK(GCC_QUPV3_WRAP_1_M_AHB_CLK,		0x5200c, 0x00100000),
	GATE_CLK(GCC_QUPV3_WRAP_1_S_AHB_CLK,		0x5200c, 0x00200000),
	GATE_CLK(GCC_SDCC1_AHB_CLK,			0x26008, 0x00000001),
	GATE_CLK(GCC_SDCC1_APPS_CLK,			0x26004, 0x00000001),
	GATE_CLK(GCC_SDCC1_ICE_CORE_CLK,		0x2600c, 0x00000001),
	GATE_CLK(GCC_SDCC2_AHB_CLK,			0x14008, 0x00000001),
	GATE_CLK(GCC_SDCC2_APPS_CLK,			0x14004, 0x00000001),
	GATE_CLK(GCC_SDCC4_AHB_CLK,			0x16008, 0x00000001),
	GATE_CLK(GCC_SDCC4_APPS_CLK,			0x16004, 0x00000001),
	GATE_CLK(GCC_UFS_MEM_CLKREF_CLK,		0x8c000, 0x00000001),
	GATE_CLK(GCC_AGGRE_UFS_PHY_AXI_CLK,		0x82024, 0x00000001),
	GATE_CLK(GCC_UFS_PHY_AHB_CLK,			0x77010, 0x00000001),
	GATE_CLK(GCC_UFS_PHY_AXI_CLK,			0x7700c, 0x00000001),
	GATE_CLK(GCC_UFS_PHY_ICE_CORE_CLK,		0x77058, 0x00000001),
	GATE_CLK(GCC_UFS_PHY_PHY_AUX_CLK,		0x7708c, 0x00000001),
	GATE_CLK(GCC_UFS_PHY_RX_SYMBOL_0_CLK,		0x77018, 0x00000001),
	GATE_CLK(GCC_UFS_PHY_TX_SYMBOL_0_CLK,		0x77014, 0x00000001),
	GATE_CLK(GCC_UFS_PHY_UNIPRO_CORE_CLK,		0x77054, 0x00000001),
	GATE_CLK(GCC_USB30_PRIM_MASTER_CLK,		0x0f00c, 0x00000001),
	GATE_CLK(GCC_USB30_PRIM_MOCK_UTMI_CLK,		0x0f014, 0x00000001),
	GATE_CLK(GCC_USB30_PRIM_SLEEP_CLK,		0x0f010, 0x00000001),
	GATE_CLK(GCC_USB3_PRIM_CLKREF_CLK,		0x8c008, 0x00000001),
	GATE_CLK(GCC_USB3_PRIM_PHY_AUX_CLK,		0x0f04c, 0x00000001),
	GATE_CLK(GCC_USB3_PRIM_PHY_COM_AUX_CLK,		0x0f050, 0x00000001),
	GATE_CLK(GCC_USB3_PRIM_PHY_PIPE_CLK,		0x0f054, 0x00000001),
	GATE_CLK(GCC_USB_PHY_CFG_AHB2PHY_CLK,		0x6a004, 0x00000001),
};

static const struct gate_clk sdm845_clks[] = {
	GATE_CLK(GCC_AGGRE_USB3_PRIM_AXI_CLK,		0x8201c, 0x00000001),
	GATE_CLK(GCC_AGGRE_USB3_SEC_AXI_CLK,		0x82020, 0x00000001),
	GATE_CLK(GCC_CFG_NOC_USB3_PRIM_AXI_CLK,		0x0502c, 0x00000001),
	GATE_CLK(GCC_CFG_NOC_USB3_SEC_AXI_CLK,		0x05030, 0x00000001),
	GATE_CLK(GCC_QUPV3_WRAP0_S0_CLK,		0x5200c, 0x00000400),
	GATE_CLK(GCC_QUPV3_WRAP0_S1_CLK,		0x5200c, 0x00000800),
	GATE_CLK(GCC_QUPV3_WRAP0_S2_CLK,		0x5200c, 0x00001000),
	GATE_CLK(GCC_QUPV3_WRAP0_S3_CLK,		0x5200c, 0x00002000),
	GATE_CLK(GCC_QUPV3_WRAP0_S4_CLK,		0x5200c, 0x00004000),
	GATE_CLK(GCC_QUPV3_WRAP0_S5_CLK,		0x5200c, 0x00008000),
	GATE_CLK(GCC_QUPV3_WRAP0_S6_CLK,		0x5200c, 0x00010000),
	GATE_CLK(GCC_QUPV3_WRAP0_S7_CLK,		0x5200c, 0x00020000),
	GATE_CLK(GCC_QUPV3_WRAP1_S0_CLK,		0x5200c, 0x00400000),
	GATE_CLK(GCC_QUPV3_WRAP1_S1_CLK,		0x5200c, 0x00800000),
	GATE_CLK(GCC_QUPV3_WRAP1_S3_CLK,		0x5200c, 0x02000000),
	GATE_CLK(GCC_QUPV3_WRAP1_S4_CLK,		0x5200c, 0x04000000),
	GATE_CLK(GCC_QUPV3_WRAP1_S5_CLK,		0x5200c, 0x08000000),
	GATE_CLK(GCC_QUPV3_WRAP1_S6_CLK,		0x5200c, 0x10000000),
	GATE_CLK(GCC_QUPV3_WRAP1_S7_CLK,		0x5200c, 0x20000000),
	GATE_CLK(GCC_QUPV3_WRAP_0_M_AHB_CLK,		0x5200c, 0x00000040),
	GATE_CLK(GCC_QUPV3_WRAP_0_S_AHB_CLK,		0x5200c, 0x00000080),
	GATE_CLK(GCC_QUPV3_WRAP_1_M_AHB_CLK,		0x5200c, 0x00100000),
	GATE_CLK(GCC_QUPV3_WRAP_1_S_AHB_CLK,		0x5200c, 0x00200000),
	GATE_CLK(GCC_SDCC2_AHB_CLK,			0x14008, 0x00000001),
	GATE_CLK(GCC_SDCC2_APPS_CLK,			0x14004, 0x00000001),
	GATE_CLK(GCC_SDCC4_AHB_CLK,			0x16008, 0x00000001),
	GATE_CLK(GCC_SDCC4_APPS_CLK,			0x16004, 0x00000001),
	GATE_CLK(GCC_UFS_CARD_AHB_CLK,			0x75010, 0x00000001),
	GATE_CLK(GCC_UFS_CARD_AXI_CLK,			0x7500c, 0x00000001),
	GATE_CLK(GCC_UFS_CARD_CLKREF_CLK,		0x8c004, 0x00000001),
	GATE_CLK(GCC_UFS_CARD_ICE_CORE_CLK,		0x75058, 0x00000001),
	GATE_CLK(GCC_UFS_CARD_PHY_AUX_CLK,		0x7508c, 0x00000001),
	GATE_CLK(GCC_UFS_CARD_RX_SYMBOL_0_CLK,		0x75018, 0x00000001),
	GATE_CLK(GCC_UFS_CARD_RX_SYMBOL_1_CLK,		0x750a8, 0x00000001),
	GATE_CLK(GCC_UFS_CARD_TX_SYMBOL_0_CLK,		0x75014, 0x00000001),
	GATE_CLK(GCC_UFS_CARD_UNIPRO_CORE_CLK,		0x75054, 0x00000001),
	GATE_CLK(GCC_UFS_MEM_CLKREF_CLK,		0x8c000, 0x00000001),
	GATE_CLK(GCC_AGGRE_UFS_PHY_AXI_CLK,		0x82024, 0x00000001),
	GATE_CLK(GCC_UFS_PHY_AHB_CLK,			0x77010, 0x00000001),
	GATE_CLK(GCC_UFS_PHY_AXI_CLK,			0x7700c, 0x00000001),
	GATE_CLK(GCC_UFS_PHY_ICE_CORE_CLK,		0x77058, 0x00000001),
	GATE_CLK(GCC_UFS_PHY_PHY_AUX_CLK,		0x7708c, 0x00000001),
	GATE_CLK(GCC_UFS_PHY_RX_SYMBOL_0_CLK,		0x77018, 0x00000001),
	GATE_CLK(GCC_UFS_PHY_RX_SYMBOL_1_CLK,		0x770a8, 0x00000001),
	GATE_CLK(GCC_UFS_PHY_TX_SYMBOL_0_CLK,		0x77014, 0x00000001),
	GATE_CLK(GCC_UFS_PHY_UNIPRO_CORE_CLK,		0x77054, 0x00000001),
	GATE_CLK(GCC_USB30_PRIM_MASTER_CLK,		0x0f00c, 0x00000001),
	GATE_CLK(GCC_USB30_PRIM_MOCK_UTMI_CLK,		0x0f014, 0x00000001),
	GATE_CLK(GCC_USB30_PRIM_SLEEP_CLK,		0x0f010, 0x00000001),
	GATE_CLK(GCC_USB30_SEC_MASTER_CLK,		0x1000c, 0x00000001),
	GATE_CLK(GCC_USB30_SEC_MOCK_UTMI_CLK,		0x10014, 0x00000001),
	GATE_CLK(GCC_USB30_SEC_SLEEP_CLK,		0x10010, 0x00000001),
	GATE_CLK(GCC_USB3_PRIM_CLKREF_CLK,		0x8c008, 0x00000001),
	GATE_CLK(GCC_USB3_PRIM_PHY_AUX_CLK,		0x0f04c, 0x00000001),
	GATE_CLK(GCC_USB3_PRIM_PHY_COM_AUX_CLK,		0x0f050, 0x00000001),
	GATE_CLK(GCC_USB3_PRIM_PHY_PIPE_CLK,		0x0f054, 0x00000001),
	GATE_CLK(GCC_USB3_SEC_CLKREF_CLK,		0x8c028, 0x00000001),
	GATE_CLK(GCC_USB3_SEC_PHY_AUX_CLK,		0x1004c, 0x00000001),
	GATE_CLK(GCC_USB3_SEC_PHY_PIPE_CLK,		0x10054, 0x00000001),
	GATE_CLK(GCC_USB3_SEC_PHY_COM_AUX_CLK,		0x10050, 0x00000001),
	GATE_CLK(GCC_USB_PHY_CFG_AHB2PHY_CLK,		0x6a004, 0x00000001),
};

static int sdm670_clk_enable(struct clk *clk)
{
	struct msm_clk_priv *priv = dev_get_priv(clk->dev);

	debug("%s: clk %s\n", __func__, sdm670_clks[clk->id].name);

	switch (clk->id) {
	case GCC_USB30_PRIM_MASTER_CLK:
		qcom_gate_clk_en(priv, GCC_USB_PHY_CFG_AHB2PHY_CLK);
		/* These numbers are just pulled from the frequency tables in the Linux driver */
		clk_rcg_set_rate_mnd(priv->base, USB30_PRIM_MASTER_CLK_CMD_RCGR,
				     (4.5 * 2) - 1, 0, 0, 1 << 8, 8);
		clk_rcg_set_rate_mnd(priv->base, USB30_PRIM_MOCK_UTMI_CLK_CMD_RCGR,
				     1, 0, 0, 0, 8);
		clk_rcg_set_rate_mnd(priv->base, USB3_PRIM_PHY_AUX_CMD_RCGR,
				     1, 0, 0, 0, 8);
		break;
	}

	return qcom_gate_clk_en(priv, clk->id);
}

static int sdm845_clk_enable(struct clk *clk)
{
	struct msm_clk_priv *priv = dev_get_priv(clk->dev);

	debug("%s: clk %s\n", __func__, sdm845_clks[clk->id].name);

	switch (clk->id) {
	case GCC_USB30_PRIM_MASTER_CLK:
		qcom_gate_clk_en(priv, GCC_USB_PHY_CFG_AHB2PHY_CLK);
		/* These numbers are just pulled from the frequency tables in the Linux driver */
		clk_rcg_set_rate_mnd(priv->base, USB30_PRIM_MASTER_CLK_CMD_RCGR,
				     (4.5 * 2) - 1, 0, 0, 1 << 8, 8);
		clk_rcg_set_rate_mnd(priv->base, USB30_PRIM_MOCK_UTMI_CLK_CMD_RCGR,
				     1, 0, 0, 0, 8);
		clk_rcg_set_rate_mnd(priv->base, USB3_PRIM_PHY_AUX_CMD_RCGR,
				     1, 0, 0, 0, 8);
		break;
	case GCC_USB30_SEC_MASTER_CLK:
		qcom_gate_clk_en(priv, GCC_USB3_SEC_PHY_AUX_CLK);

		qcom_gate_clk_en(priv, GCC_USB3_SEC_CLKREF_CLK);
		qcom_gate_clk_en(priv, GCC_USB3_SEC_PHY_COM_AUX_CLK);
		break;
	}

	return qcom_gate_clk_en(priv, clk->id);
}

static const struct qcom_reset_map sdm845_gcc_resets[] = {
	[GCC_QUPV3_WRAPPER_0_BCR] = { 0x17000 },
	[GCC_QUPV3_WRAPPER_1_BCR] = { 0x18000 },
	[GCC_QUSB2PHY_PRIM_BCR] = { 0x12000 },
	[GCC_QUSB2PHY_SEC_BCR] = { 0x12004 },
	[GCC_SDCC2_BCR] = { 0x14000 },
	[GCC_SDCC4_BCR] = { 0x16000 },
	[GCC_UFS_CARD_BCR] = { 0x75000 },
	[GCC_UFS_PHY_BCR] = { 0x77000 },
	[GCC_USB30_PRIM_BCR] = { 0xf000 },
	[GCC_USB30_SEC_BCR] = { 0x10000 },
	[GCC_USB3_PHY_PRIM_BCR] = { 0x50000 },
	[GCC_USB3PHY_PHY_PRIM_BCR] = { 0x50004 },
	[GCC_USB3_DP_PHY_PRIM_BCR] = { 0x50008 },
	[GCC_USB3_PHY_SEC_BCR] = { 0x5000c },
	[GCC_USB3PHY_PHY_SEC_BCR] = { 0x50010 },
	[GCC_USB3_DP_PHY_SEC_BCR] = { 0x50014 },
	[GCC_USB_PHY_CFG_AHB2PHY_BCR] = { 0x6a000 },
};

static const struct qcom_power_map sdm670_gdscs[] = {
	[UFS_PHY_GDSC] = { 0x77004 },
	[USB30_PRIM_GDSC] = { 0xf004 },
	[HLOS1_VOTE_AGGRE_NOC_MMU_AUDIO_TBU_GDSC] = { 0x7d030 },
	[HLOS1_VOTE_AGGRE_NOC_MMU_TBU1_GDSC] = { 0x7d034 },
	[HLOS1_VOTE_AGGRE_NOC_MMU_TBU2_GDSC] = { 0x7d038 },
	[HLOS1_VOTE_MMNOC_MMU_TBU_HF0_GDSC] = { 0x7d040 },
	[HLOS1_VOTE_MMNOC_MMU_TBU_HF1_GDSC] = { 0x7d048 },
	[HLOS1_VOTE_MMNOC_MMU_TBU_SF_GDSC] = { 0x7d044 },
};

static const struct qcom_power_map sdm845_gdscs[] = {
	[PCIE_0_GDSC] = { 0x6b004 },
	[PCIE_1_GDSC] = { 0x8d004 },
	[UFS_CARD_GDSC] = { 0x75004 },
	[UFS_PHY_GDSC] = { 0x77004 },
	[USB30_PRIM_GDSC] = { 0xf004 },
	[USB30_SEC_GDSC] = { 0x10004 },
	[HLOS1_VOTE_AGGRE_NOC_MMU_AUDIO_TBU_GDSC] = { 0x7d030 },
	[HLOS1_VOTE_AGGRE_NOC_MMU_PCIE_TBU_GDSC] = { 0x7d03c },
	[HLOS1_VOTE_AGGRE_NOC_MMU_TBU1_GDSC] = { 0x7d034 },
	[HLOS1_VOTE_AGGRE_NOC_MMU_TBU2_GDSC] = { 0x7d038 },
	[HLOS1_VOTE_MMNOC_MMU_TBU_HF0_GDSC] = { 0x7d040 },
	[HLOS1_VOTE_MMNOC_MMU_TBU_HF1_GDSC] = { 0x7d048 },
	[HLOS1_VOTE_MMNOC_MMU_TBU_SF_GDSC] = { 0x7d044 },
};

static const phys_addr_t sdm845_gpll_addrs[] = {
	0x00100000, // GCC_GPLL0_MODE
	0x00101000, // GCC_GPLL1_MODE
	0x00102000, // GCC_GPLL2_MODE
	0x00103000, // GCC_GPLL3_MODE
	0x00176000, // GCC_GPLL4_MODE
	0x00174000, // GCC_GPLL5_MODE
	0x00113000, // GCC_GPLL6_MODE
};

static const phys_addr_t sdm845_rcg_addrs[] = {
	0x0010f018, // GCC_USB30_PRIM_MASTER
	0x0010f030, // GCC_USB30_PRIM_MOCK_UTMI
	0x0010f05c, // GCC_USB3_PRIM_PHY_AUX
	0x00110018, // GCC_USB30_SEC_MASTER
	0x00110030, // GCC_USB30_SEC_MOCK_UTMI
	0x0011005c, // GCC_USB3_SEC_PHY_AUX
	0x0011400c, // GCC_SDCC2_APPS
	0x0011600c, // GCC_SDCC4_APPS
	0x00117018, // GCC_QUPV3_WRAP0_CORE_2X
	0x00117034, // GCC_QUPV3_WRAP0_S0
	0x00117164, // GCC_QUPV3_WRAP0_S1
	0x00117294, // GCC_QUPV3_WRAP0_S2
	0x001173c4, // GCC_QUPV3_WRAP0_S3
	0x001174f4, // GCC_QUPV3_WRAP0_S4
	0x00117624, // GCC_QUPV3_WRAP0_S5
	0x00117754, // GCC_QUPV3_WRAP0_S6
	0x00117884, // GCC_QUPV3_WRAP0_S7
	0x00118018, // GCC_QUPV3_WRAP1_S0
	0x00118148, // GCC_QUPV3_WRAP1_S1
	0x00118278, // GCC_QUPV3_WRAP1_S2
	0x001183a8, // GCC_QUPV3_WRAP1_S3
	0x001184d8, // GCC_QUPV3_WRAP1_S4
	0x00118608, // GCC_QUPV3_WRAP1_S5
	0x00118738, // GCC_QUPV3_WRAP1_S6
	0x00118868, // GCC_QUPV3_WRAP1_S7
	0x0016b028, // GCC_PCIE_0_AUX
	0x0018d028, // GCC_PCIE_1_AUX
	0x0016f014, // GCC_PCIE_PHY_REFGEN
	0x0017501c, // GCC_UFS_CARD_AXI
	0x0017505c, // GCC_UFS_CARD_ICE_CORE
	0x00175074, // GCC_UFS_CARD_UNIPRO_CORE
	0x00175090, // GCC_UFS_CARD_PHY_AUX
	0x0017701c, // GCC_UFS_PHY_AXI
	0x0017705c, // GCC_UFS_PHY_ICE_CORE
	0x00177074, // GCC_UFS_PHY_UNIPRO_CORE
	0x00177090, // GCC_UFS_PHY_PHY_AUX
};

static const char *const sdm845_rcg_names[] = {
	"GCC_USB30_PRIM_MASTER",
	"GCC_USB30_PRIM_MOCK_UTMI",
	"GCC_USB3_PRIM_PHY_AUX",
	"GCC_USB30_SEC_MASTER",
	"GCC_USB30_SEC_MOCK_UTMI",
	"GCC_USB3_SEC_PHY_AUX",
	"GCC_SDCC2_APPS",
	"GCC_SDCC4_APPS",
	"GCC_QUPV3_WRAP0_CORE_2X",
	"GCC_QUPV3_WRAP0_S0",
	"GCC_QUPV3_WRAP0_S1",
	"GCC_QUPV3_WRAP0_S2",
	"GCC_QUPV3_WRAP0_S3",
	"GCC_QUPV3_WRAP0_S4",
	"GCC_QUPV3_WRAP0_S5",
	"GCC_QUPV3_WRAP0_S6",
	"GCC_QUPV3_WRAP0_S7",
	"GCC_QUPV3_WRAP1_S0",
	"GCC_QUPV3_WRAP1_S1",
	"GCC_QUPV3_WRAP1_S2",
	"GCC_QUPV3_WRAP1_S3",
	"GCC_QUPV3_WRAP1_S4",
	"GCC_QUPV3_WRAP1_S5",
	"GCC_QUPV3_WRAP1_S6",
	"GCC_QUPV3_WRAP1_S7",
	"GCC_PCIE_0_AUX",
	"GCC_PCIE_1_AUX",
	"GCC_PCIE_PHY_REFGEN",
	"GCC_UFS_CARD_AXI",
	"GCC_UFS_CARD_ICE_CORE",
	"GCC_UFS_CARD_UNIPRO_CORE",
	"GCC_UFS_CARD_PHY_AUX",
	"GCC_UFS_PHY_AXI",
	"GCC_UFS_PHY_ICE_CORE",
	"GCC_UFS_PHY_UNIPRO_CORE",
	"GCC_UFS_PHY_PHY_AUX",
};

static struct msm_clk_data sdm670_clk_data = {
	/* Snapdragon 670 can function without its own exclusive resets. */
	.resets = sdm845_gcc_resets,
	.num_resets = ARRAY_SIZE(sdm845_gcc_resets),
	.clks = sdm670_clks,
	.num_clks = ARRAY_SIZE(sdm670_clks),
	.power_domains = sdm670_gdscs,
	.num_power_domains = ARRAY_SIZE(sdm670_gdscs),

	.enable = sdm670_clk_enable,
	.set_rate = sdm670_clk_set_rate,
};

static struct msm_clk_data sdm845_clk_data = {
	.resets = sdm845_gcc_resets,
	.num_resets = ARRAY_SIZE(sdm845_gcc_resets),
	.clks = sdm845_clks,
	.num_clks = ARRAY_SIZE(sdm845_clks),
	.power_domains = sdm845_gdscs,
	.num_power_domains = ARRAY_SIZE(sdm845_gdscs),

	.enable = sdm845_clk_enable,
	.set_rate = sdm845_clk_set_rate,
	.dbg_pll_addrs = sdm845_gpll_addrs,
	.num_plls = ARRAY_SIZE(sdm845_gpll_addrs),
	.dbg_rcg_addrs = sdm845_rcg_addrs,
	.num_rcgs = ARRAY_SIZE(sdm845_rcg_addrs),
	.dbg_rcg_names = sdm845_rcg_names,
};

static const struct udevice_id gcc_sdm845_of_match[] = {
	{
		.compatible = "qcom,gcc-sdm670",
		.data = (ulong)&sdm670_clk_data,
	},
	{
		.compatible = "qcom,gcc-sdm845",
		.data = (ulong)&sdm845_clk_data,
	},
	{ }
};

U_BOOT_DRIVER(gcc_sdm845) = {
	.name		= "gcc_sdm845",
	.id		= UCLASS_NOP,
	.of_match	= gcc_sdm845_of_match,
	.bind		= qcom_cc_bind,
	.flags		= DM_FLAG_PRE_RELOC | DM_FLAG_DEFAULT_PD_CTRL_OFF,
};
