#!/bin/bash
set -euo pipefail

# this script has been adapted from:
# https://devopscube.com/create-self-signed-certificates-openssl/

if [ "$#" -ne 1 ]
then
  echo "Error: No domain name argument provided"
  echo "Usage: Provide a domain name as an argument"
  exit 1
fi

DOMAIN=$1

# Create root CA & Private key

openssl req -x509 \
            -sha256 -days 356 \
            -nodes \
            -newkey rsa:2048 \
            -subj "/CN=${DOMAIN}/C=CZ/L=Prague" \
            -keyout rootCA.key -out rootCA.crt

# Generate Private key

openssl genrsa -out "${DOMAIN}.key" 2048

# Create csf conf
cat > csr.conf <<EOF
[ req ]
default_bits = 2048
prompt = no
default_md = sha256
req_extensions = req_ext
distinguished_name = dn

[ dn ]
C = CZ
L = Prague
O = Me
OU = Myself
CN = ${DOMAIN}

[ req_ext ]
subjectAltName = @alt_names

[ alt_names ]
DNS.1 = ${DOMAIN}
DNS.2 = www.${DOMAIN}
IP.1 = 127.0.0.1
IP.2 = 127.0.0.2

EOF

# create CSR request using private key
openssl req -new -key "${DOMAIN}.key" -out "${DOMAIN}.csr" -config csr.conf

# Create a external config file for the certificate
cat > cert.conf <<EOF

authorityKeyIdentifier=keyid,issuer
basicConstraints=CA:FALSE
keyUsage = digitalSignature, nonRepudiation, keyEncipherment, dataEncipherment
subjectAltName = @alt_names

[alt_names]
DNS.1 = ${DOMAIN}

EOF

# Create SSl with self signed CA
openssl x509 -req \
    -in "${DOMAIN}.csr" \
    -CA rootCA.crt -CAkey rootCA.key \
    -CAcreateserial -out "${DOMAIN}.crt" \
    -days 365 \
    -sha256 -extfile cert.conf
