"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PanelsRouter = PanelsRouter;
var _configSchema = require("@osd/config-schema");
var _custom_panel_adaptor = require("../../adaptors/custom_panels/custom_panel_adaptor");
var _custom_panels = require("../../../common/constants/custom_panels");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function PanelsRouter(router) {
  const customPanelBackend = new _custom_panel_adaptor.CustomPanelsAdaptor();
  // Fetch all the custom panels available
  router.get({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/panels`,
    validate: {}
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const panelsList = await customPanelBackend.viewPanelList(opensearchNotebooksClient);
      return response.ok({
        body: {
          panels: panelsList
        }
      });
    } catch (error) {
      console.error('Issue in fetching panel list:', error);
      return response.custom({
        statusCode: error.statusCode || 400,
        body: error.message
      });
    }
  });

  // Fetch the required panel by id
  router.get({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/panels/{panelId}`,
    validate: {
      params: _configSchema.schema.object({
        panelId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const panelObject = await customPanelBackend.getPanel(opensearchNotebooksClient, request.params.panelId);
      return response.ok({
        body: panelObject
      });
    } catch (error) {
      console.error('Issue in fetching panel:', error);
      return response.custom({
        statusCode: error.statusCode || 400,
        body: error.message
      });
    }
  });

  // Create a new panel
  router.post({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/panels`,
    validate: {
      body: _configSchema.schema.object({
        panelName: _configSchema.schema.string(),
        applicationId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const newPanelId = await customPanelBackend.createNewPanel(opensearchNotebooksClient, request.body.panelName, request.body.applicationId || '');
      return response.ok({
        body: {
          message: 'Panel Created',
          newPanelId
        }
      });
    } catch (error) {
      console.error('Issue in creating new panel', error);
      return response.custom({
        statusCode: error.statusCode || 400,
        body: error.message
      });
    }
  });

  // update an existing panel
  router.post({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/panels/update`,
    validate: {
      body: _configSchema.schema.object({
        panelId: _configSchema.schema.string(),
        panel: _configSchema.schema.any()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const _responseBody = await customPanelBackend.updatePanel(opensearchNotebooksClient, request.body.panelId, request.body.panel);
      return response.ok({
        body: {
          message: 'Panel Updated'
        }
      });
    } catch (error) {
      console.error('Issue in updating panel', error);
      return response.custom({
        statusCode: error.statusCode || 400,
        body: error.message
      });
    }
  });

  // rename an existing panel
  router.post({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/panels/rename`,
    validate: {
      body: _configSchema.schema.object({
        panelId: _configSchema.schema.string(),
        panelName: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const _responseBody = await customPanelBackend.renamePanel(opensearchNotebooksClient, request.body.panelId, request.body.panelName);
      return response.ok({
        body: {
          message: 'Panel Renamed'
        }
      });
    } catch (error) {
      console.error('Issue in renaming panel', error);
      return response.custom({
        statusCode: error.statusCode || 400,
        body: error.message
      });
    }
  });

  // clones an existing panel
  // returns new panel Id
  router.post({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/panels/clone`,
    validate: {
      body: _configSchema.schema.object({
        panelId: _configSchema.schema.string(),
        panelName: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const cloneResponse = await customPanelBackend.clonePanel(opensearchNotebooksClient, request.body.panelId, request.body.panelName);
      return response.ok({
        body: {
          message: 'Panel Cloned',
          clonePanelId: cloneResponse.clonePanelId,
          dateCreated: cloneResponse.dateCreated,
          dateModified: cloneResponse.dateModified
        }
      });
    } catch (error) {
      console.error('Issue in cloning panel', error);
      return response.custom({
        statusCode: error.statusCode || 400,
        body: error.message
      });
    }
  });

  // delete an existing panel
  router.delete({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/panels/{panelId}`,
    validate: {
      params: _configSchema.schema.object({
        panelId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const _deleteResponse = await customPanelBackend.deletePanel(opensearchNotebooksClient, request.params.panelId);
      return response.ok({
        body: {
          message: 'Panel Deleted'
        }
      });
    } catch (error) {
      console.error('Issue in deleting panel', error);
      return response.custom({
        statusCode: error.statusCode || 400,
        body: error.message
      });
    }
  });

  // delete an existing panel(s)
  router.delete({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/panelList/{panelIdList}`,
    validate: {
      params: _configSchema.schema.object({
        panelIdList: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const _deleteResponse = await customPanelBackend.deletePanelList(opensearchNotebooksClient, request.params.panelIdList);
      return response.ok({
        body: {
          message: 'Panel Deleted'
        }
      });
    } catch (error) {
      console.error('Issue in deleting panel', error);
      return response.custom({
        statusCode: error.statusCode || 400,
        body: error.message
      });
    }
  });

  // replaces the ppl query filter in panel
  router.post({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/panels/filter`,
    validate: {
      body: _configSchema.schema.object({
        panelId: _configSchema.schema.string(),
        query: _configSchema.schema.string(),
        language: _configSchema.schema.string(),
        to: _configSchema.schema.string(),
        from: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const _panelFilterResponse = await customPanelBackend.addPanelFilter(opensearchNotebooksClient, request.body.panelId, request.body.query, request.body.language, request.body.to, request.body.from);
      return response.ok({
        body: {
          message: 'Panel PPL Filter Changed'
        }
      });
    } catch (error) {
      console.error('Issue in adding query filter', error);
      return response.custom({
        statusCode: error.statusCode || 400,
        body: error.message
      });
    }
  });

  // Add Sample Panels
  router.post({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/panels/addSamplePanels`,
    validate: {
      body: _configSchema.schema.object({
        savedVisualizationIds: _configSchema.schema.arrayOf(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const panelsData = await customPanelBackend.addSamplePanels(opensearchNotebooksClient, request.body.savedVisualizationIds);
      return response.ok({
        body: {
          demoPanelsData: panelsData
        }
      });
    } catch (error) {
      console.error('Issue in fetching panel list:', error);
      return response.custom({
        statusCode: error.statusCode || 400,
        body: error.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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