﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/workdocs/WorkDocsRequest.h>
#include <aws/workdocs/WorkDocs_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace WorkDocs {
namespace Model {

/**
 */
class DescribeResourcePermissionsRequest : public WorkDocsRequest {
 public:
  AWS_WORKDOCS_API DescribeResourcePermissionsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeResourcePermissions"; }

  AWS_WORKDOCS_API Aws::String SerializePayload() const override;

  AWS_WORKDOCS_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  AWS_WORKDOCS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>Amazon WorkDocs authentication token. Not required when using Amazon Web
   * Services administrator credentials to access the API.</p>
   */
  inline const Aws::String& GetAuthenticationToken() const { return m_authenticationToken; }
  inline bool AuthenticationTokenHasBeenSet() const { return m_authenticationTokenHasBeenSet; }
  template <typename AuthenticationTokenT = Aws::String>
  void SetAuthenticationToken(AuthenticationTokenT&& value) {
    m_authenticationTokenHasBeenSet = true;
    m_authenticationToken = std::forward<AuthenticationTokenT>(value);
  }
  template <typename AuthenticationTokenT = Aws::String>
  DescribeResourcePermissionsRequest& WithAuthenticationToken(AuthenticationTokenT&& value) {
    SetAuthenticationToken(std::forward<AuthenticationTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the resource.</p>
   */
  inline const Aws::String& GetResourceId() const { return m_resourceId; }
  inline bool ResourceIdHasBeenSet() const { return m_resourceIdHasBeenSet; }
  template <typename ResourceIdT = Aws::String>
  void SetResourceId(ResourceIdT&& value) {
    m_resourceIdHasBeenSet = true;
    m_resourceId = std::forward<ResourceIdT>(value);
  }
  template <typename ResourceIdT = Aws::String>
  DescribeResourcePermissionsRequest& WithResourceId(ResourceIdT&& value) {
    SetResourceId(std::forward<ResourceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the principal to filter permissions by.</p>
   */
  inline const Aws::String& GetPrincipalId() const { return m_principalId; }
  inline bool PrincipalIdHasBeenSet() const { return m_principalIdHasBeenSet; }
  template <typename PrincipalIdT = Aws::String>
  void SetPrincipalId(PrincipalIdT&& value) {
    m_principalIdHasBeenSet = true;
    m_principalId = std::forward<PrincipalIdT>(value);
  }
  template <typename PrincipalIdT = Aws::String>
  DescribeResourcePermissionsRequest& WithPrincipalId(PrincipalIdT&& value) {
    SetPrincipalId(std::forward<PrincipalIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of items to return with this call.</p>
   */
  inline int GetLimit() const { return m_limit; }
  inline bool LimitHasBeenSet() const { return m_limitHasBeenSet; }
  inline void SetLimit(int value) {
    m_limitHasBeenSet = true;
    m_limit = value;
  }
  inline DescribeResourcePermissionsRequest& WithLimit(int value) {
    SetLimit(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The marker for the next set of results. (You received this marker from a
   * previous call)</p>
   */
  inline const Aws::String& GetMarker() const { return m_marker; }
  inline bool MarkerHasBeenSet() const { return m_markerHasBeenSet; }
  template <typename MarkerT = Aws::String>
  void SetMarker(MarkerT&& value) {
    m_markerHasBeenSet = true;
    m_marker = std::forward<MarkerT>(value);
  }
  template <typename MarkerT = Aws::String>
  DescribeResourcePermissionsRequest& WithMarker(MarkerT&& value) {
    SetMarker(std::forward<MarkerT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_authenticationToken;

  Aws::String m_resourceId;

  Aws::String m_principalId;

  int m_limit{0};

  Aws::String m_marker;
  bool m_authenticationTokenHasBeenSet = false;
  bool m_resourceIdHasBeenSet = false;
  bool m_principalIdHasBeenSet = false;
  bool m_limitHasBeenSet = false;
  bool m_markerHasBeenSet = false;
};

}  // namespace Model
}  // namespace WorkDocs
}  // namespace Aws
