const char swim_lua[] =
"local ffi = require('ffi')\n"
"local uuid = require('uuid')\n"
"local buffer = require('buffer')\n"
"local msgpack = require('msgpack')\n"
"local crypto = require('crypto')\n"
"local fiber = require('fiber')\n"
"local internal = require('swim.lib')\n"
"local schedule_task = fiber._internal.schedule_task\n"
"local cord_ibuf_take = buffer.internal.cord_ibuf_take\n"
"local cord_ibuf_put = buffer.internal.cord_ibuf_put\n"
"\n"
"ffi.cdef[[\n"
"    struct swim;\n"
"    struct tt_uuid;\n"
"    struct swim_iterator;\n"
"    struct swim_member;\n"
"\n"
"    enum swim_gc_mode {\n"
"        SWIM_GC_DEFAULT = -1,\n"
"        SWIM_GC_OFF = 0,\n"
"        SWIM_GC_ON = 1,\n"
"    };\n"
"\n"
"    enum swim_member_status {\n"
"        MEMBER_ALIVE = 0,\n"
"        MEMBER_SUSPECTED,\n"
"        MEMBER_DEAD,\n"
"        MEMBER_LEFT,\n"
"    };\n"
"\n"
"    enum swim_ev_mask {\n"
"        SWIM_EV_NEW             = 0b00000001,\n"
"        SWIM_EV_NEW_STATUS      = 0b00000010,\n"
"        SWIM_EV_NEW_URI         = 0b00000100,\n"
"        SWIM_EV_NEW_GENERATION  = 0b00001000,\n"
"        SWIM_EV_NEW_VERSION     = 0b00010000,\n"
"        SWIM_EV_NEW_INCARNATION = 0b00011000,\n"
"        SWIM_EV_NEW_PAYLOAD     = 0b00100000,\n"
"        SWIM_EV_UPDATE          = 0b00111110,\n"
"        SWIM_EV_DROP            = 0b01000000,\n"
"    };\n"
"\n"
"    struct swim_incarnation {\n"
"        uint64_t generation;\n"
"        uint64_t version;\n"
"    };\n"
"\n"
"    bool\n"
"    swim_is_configured(const struct swim *swim);\n"
"\n"
"    int\n"
"    swim_cfg(struct swim *swim, const char *uri, double heartbeat_rate,\n"
"             double ack_timeout, enum swim_gc_mode gc_mode,\n"
"             const struct tt_uuid *uuid);\n"
"\n"
"    int\n"
"    swim_set_payload(struct swim *swim, const char *payload, int payload_size);\n"
"\n"
"    int\n"
"    swim_set_codec(struct swim *swim, enum crypto_algo algo,\n"
"                   enum crypto_mode mode, const char *key, int key_size);\n"
"\n"
"    int\n"
"    swim_add_member(struct swim *swim, const char *uri,\n"
"                    const struct tt_uuid *uuid);\n"
"\n"
"    int\n"
"    swim_remove_member(struct swim *swim, const struct tt_uuid *uuid);\n"
"\n"
"    int\n"
"    swim_probe_member(struct swim *swim, const char *uri);\n"
"\n"
"    int\n"
"    swim_broadcast(struct swim *swim, int port);\n"
"\n"
"    int\n"
"    swim_size(const struct swim *swim);\n"
"\n"
"    struct swim_member *\n"
"    swim_self(struct swim *swim);\n"
"\n"
"    struct swim_member *\n"
"    swim_member_by_uuid(struct swim *swim, const struct tt_uuid *uuid);\n"
"\n"
"    enum swim_member_status\n"
"    swim_member_status(const struct swim_member *member);\n"
"\n"
"    struct swim_iterator *\n"
"    swim_iterator_open(struct swim *swim);\n"
"\n"
"    struct swim_member *\n"
"    swim_iterator_next(struct swim_iterator *iterator);\n"
"\n"
"    void\n"
"    swim_iterator_close(struct swim_iterator *iterator);\n"
"\n"
"    const char *\n"
"    swim_member_uri(const struct swim_member *member);\n"
"\n"
"    const struct tt_uuid *\n"
"    swim_member_uuid(const struct swim_member *member);\n"
"\n"
"    struct swim_incarnation\n"
"    swim_member_incarnation(const struct swim_member *member);\n"
"\n"
"    const char *\n"
"    swim_member_payload(const struct swim_member *member, int *size);\n"
"\n"
"    void\n"
"    swim_member_ref(struct swim_member *member);\n"
"\n"
"    void\n"
"    swim_member_unref(struct swim_member *member);\n"
"\n"
"    bool\n"
"    swim_member_is_dropped(const struct swim_member *member);\n"
"\n"
"    bool\n"
"    swim_member_is_payload_up_to_date(const struct swim_member *member);\n"
"]]\n"
"\n"
"-- Shortcut to avoid unnecessary lookups in 'ffi' table.\n"
"local capi = ffi.C\n"
"\n"
"local swim_t = ffi.typeof('struct swim *')\n"
"local swim_member_t = ffi.typeof('struct swim_member *')\n"
"\n"
"local swim_member_status_strs = {\n"
"    [capi.MEMBER_ALIVE] = 'alive',\n"
"    [capi.MEMBER_SUSPECTED] = 'suspected',\n"
"    [capi.MEMBER_DEAD] = 'dead',\n"
"    [capi.MEMBER_LEFT] = 'left'\n"
"}\n"
"\n"
"local swim_incarnation_mt = {\n"
"    __eq = function(l, r)\n"
"        return l.version == r.version and l.generation == r.generation\n"
"    end,\n"
"    __lt = function(l, r)\n"
"        return l.generation < r.generation or\n"
"               l.generation == r.generation and l.version < r.version\n"
"    end,\n"
"    __le = function(l, r)\n"
"        return l.generation < r.generation or\n"
"               l.generation == r.generation and l.version <= r.version\n"
"    end,\n"
"    __tostring = function(i)\n"
"        return string.format('cdata {generation = %s, version = %s}',\n"
"                             i.generation, i.version)\n"
"    end,\n"
"}\n"
"ffi.metatype(ffi.typeof('struct swim_incarnation'), swim_incarnation_mt)\n"
"\n"
"--\n"
"-- Check if @a value is something that can be passed as a\n"
"-- URI parameter. Note, it does not validate URI, because it is\n"
"-- done in C module. Here only dynamic typing errors are checked.\n"
"-- Throws on invalid type.\n"
"--\n"
"-- @param value Value to check and probably convert.\n"
"-- @param func_name Caller function name to include into an error\n"
"--        message.\n"
"-- @return String that can be passed as a URI parameter.\n"
"--\n"
"local function swim_check_uri(value, func_name)\n"
"    if value == nil then\n"
"        return nil\n"
"    end\n"
"    if type(value) == 'string' then\n"
"        return value\n"
"    end\n"
"    if type(value) == 'number' then\n"
"        return tostring(value)\n"
"    end\n"
"    return error(func_name..': expected string URI or port number')\n"
"end\n"
"\n"
"--\n"
"-- Check if @a value is a number, that can be passed as a timeout.\n"
"-- Throws on invalid type.\n"
"--\n"
"-- @param value Value to check.\n"
"-- @param func_name Caller function name to include into an error\n"
"--        message.\n"
"-- @param param_name Timeout parameter name to include into an\n"
"--        error message. Examples: 'heartbeat_rate',\n"
"--        'ack_timeout'.\n"
"-- @return Timeout value. Can be negative - SWIM treats negative\n"
"--         value as an instruction to keep an old timeout.\n"
"--\n"
"local function swim_check_timeout(value, func_name, param_name)\n"
"    if value == nil then\n"
"        return -1\n"
"    end\n"
"    if type(value) ~= 'number' then\n"
"        return error(func_name..': expected number '..param_name)\n"
"    end\n"
"    return value\n"
"end\n"
"\n"
"--\n"
"-- Check if @a value is a valid garbage collection mode.\n"
"-- Throws on invalid type and unknown mode.\n"
"--\n"
"-- @param value Value to check and convert.\n"
"-- @param func_name Caller function name to include into an error\n"
"--        message.\n"
"-- @return GC mode cdata enum value.\n"
"--\n"
"local function swim_check_gc_mode(value, func_name)\n"
"    if value == nil then\n"
"        return capi.SWIM_GC_DEFAULT\n"
"    end\n"
"    if value == 'on' then\n"
"        return capi.SWIM_GC_ON\n"
"    elseif value == 'off' then\n"
"        return capi.SWIM_GC_OFF\n"
"    else\n"
"        return error(func_name..': unknown gc_mode')\n"
"    end\n"
"end\n"
"\n"
"--\n"
"-- Check if @a value is a valid UUID. Throws on invalid type and\n"
"-- UUID.\n"
"--\n"
"-- @param value Value to check.\n"
"-- @param func_name Caller function name to include into an error\n"
"--        message.\n"
"-- @return Struct UUID cdata.\n"
"--\n"
"local function swim_check_uuid(value, func_name)\n"
"    if value == nil then\n"
"        return nil\n"
"    end\n"
"    if type(value) ~= 'string' then\n"
"        if ffi.istype('struct tt_uuid', value) then\n"
"            return value\n"
"        end\n"
"        return error(func_name..': expected string UUID or struct tt_uuid')\n"
"    end\n"
"    value = uuid.fromstr(value)\n"
"    if not value then\n"
"        return error(func_name..': invalid UUID')\n"
"    end\n"
"    return value\n"
"end\n"
"\n"
"--\n"
"-- Check if @a value is something that can be passed as const\n"
"-- char *, as binary data. It should be either string, or cdata.\n"
"-- @param value Value to check if can be converted to\n"
"--        const char *.\n"
"-- @param size Size of @a value in bytes. In case of cdata the\n"
"--        argument is mandatory. In case of string @a size is\n"
"--        optional and can be <= @a value length.\n"
"-- @param func_name Caller function name to include into an error\n"
"--        message.\n"
"-- @param param_name Parameter name to include into an error\n"
"--        message. Examples: 'payload', 'key'.\n"
"-- @return Value compatible with const char *, and size in bytes.\n"
"--\n"
"local function swim_check_const_char(value, size, func_name, param_name)\n"
"    if size ~= nil and type(size) ~= 'number' then\n"
"        return error(func_name..': expected number '..param_name..' size')\n"
"    end\n"
"    if type(value) == 'cdata' then\n"
"        if not size then\n"
"            return error(func_name..': size is mandatory for cdata '..\n"
"                         param_name)\n"
"        end\n"
"        value = ffi.cast('const char *', value)\n"
"    elseif type(value) == 'string' then\n"
"        if not size then\n"
"            size = value:len()\n"
"        elseif size > value:len() then\n"
"            return error(func_name..': explicit '..param_name..\n"
"                         ' size > string length')\n"
"        end\n"
"    elseif value == nil then\n"
"        if size then\n"
"            return error(func_name..': size can not be set without '..\n"
"                         param_name)\n"
"        end\n"
"        size = 0\n"
"    else\n"
"        return error(func_name..': '..param_name..' should be either string '..\n"
"                     'or cdata')\n"
"    end\n"
"    return value, size\n"
"end\n"
"\n"
"--\n"
"-- Check if @a s is a SWIM instance. It should be a table with\n"
"-- cdata struct swim in 'ptr' attribute. Throws on invalid type.\n"
"--\n"
"-- @param value Value to check.\n"
"-- @param func_name Caller function name to include into an error\n"
"--        message.\n"
"-- @return Pointer to struct swim.\n"
"--\n"
"local function swim_check_instance(s, func_name)\n"
"    if type(s) == 'table' then\n"
"        local ptr = s.ptr\n"
"        if ffi.istype(swim_t, ptr) then\n"
"            return ptr\n"
"        end\n"
"    end\n"
"    return error(func_name..': first argument is not a SWIM instance')\n"
"end\n"
"\n"
"--\n"
"-- The same for SWIM member.\n"
"--\n"
"local function swim_check_member(m, func_name)\n"
"    if type(m) == 'table' then\n"
"        local ptr = m.ptr\n"
"        if ffi.istype(swim_member_t, ptr) then\n"
"            return ptr\n"
"        end\n"
"    end\n"
"    return error(func_name..': first argument is not a SWIM member')\n"
"end\n"
"\n"
"local function swim_check_event(event, func_name)\n"
"    if type(event) == 'table' then\n"
"        local value = event[1]\n"
"        if type(value) == 'number' then\n"
"            return value\n"
"        end\n"
"    end\n"
"    return error(func_name..': first argument is not a SWIM event')\n"
"end\n"
"\n"
"--\n"
"-- Member methods. Most of them are one-liners, not much to\n"
"-- comment.\n"
"--\n"
"\n"
"local function swim_member_status(m)\n"
"    local ptr = swim_check_member(m, 'member:status()')\n"
"    return swim_member_status_strs[tonumber(capi.swim_member_status(ptr))]\n"
"end\n"
"\n"
"local function swim_member_uri(m)\n"
"    local ptr = swim_check_member(m, 'member:uri()')\n"
"    return ffi.string(capi.swim_member_uri(ptr))\n"
"end\n"
"\n"
"local function swim_member_incarnation(m)\n"
"    local ptr = swim_check_member(m, 'member:incarnation()')\n"
"    return capi.swim_member_incarnation(ptr)\n"
"end\n"
"\n"
"local function swim_member_is_dropped(m)\n"
"    local ptr = swim_check_member(m, 'member:is_dropped()')\n"
"    return capi.swim_member_is_dropped(ptr)\n"
"end\n"
"\n"
"local function swim_member_payload_raw(ptr)\n"
"    local int = ffi.new('int[1]')\n"
"    local cdata = capi.swim_member_payload(ptr, int)\n"
"    return cdata, int[0]\n"
"end\n"
"\n"
"--\n"
"-- Payload can be bigger than KB, and probably it is undesirable\n"
"-- to copy it into a Lua string or decode MessagePack into a\n"
"-- Lua object. This method is the cheapest way of taking payload.\n"
"--\n"
"local function swim_member_payload_cdata(m)\n"
"    local ptr = swim_check_member(m, 'member:payload_cdata()')\n"
"    return swim_member_payload_raw(ptr)\n"
"end\n"
"\n"
"--\n"
"-- Cdata requires to keep explicit size, besides not all user\n"
"-- methods can be able to work with cdata. This is why it may be\n"
"-- needed to take payload as a string - text or binary.\n"
"--\n"
"local function swim_member_payload_str(m)\n"
"    local ptr = swim_check_member(m, 'member:payload_str()')\n"
"    local _, size = swim_member_payload_raw(ptr)\n"
"    if size > 0 then\n"
"        return ffi.string(swim_member_payload_raw(ptr))\n"
"    end\n"
"    return nil\n"
"end\n"
"\n"
"--\n"
"-- Since this is a Lua module, a user is likely to use Lua objects\n"
"-- as a payload - tables, numbers, string etc. And it is natural\n"
"-- to expect that member:payload() should return the same object\n"
"-- which was passed into swim:set_payload() on another instance.\n"
"-- This member method tries to interpret payload as MessagePack,\n"
"-- and if fails, returns the payload as a string.\n"
"--\n"
"-- This function caches its result. It means, that only first call\n"
"-- actually decodes cdata payload. All the next calls return\n"
"-- pointer to the same result, until payload is changed with a new\n"
"-- incarnation.\n"
"--\n"
"local function swim_member_payload(m)\n"
"    local ptr = swim_check_member(m, 'member:payload()')\n"
"    -- Two keys are needed. Incarnation is not enough, because a\n"
"    -- new incarnation can be disseminated earlier than a new\n"
"    -- payload. For example, via ACK messages.\n"
"    local key1 = capi.swim_member_incarnation(ptr)\n"
"    local key2 = capi.swim_member_is_payload_up_to_date(ptr)\n"
"    if m.p_key1 and key1 == m.p_key1 and key2 == m.p_key2 then\n"
"        return m.p\n"
"    end\n"
"    local cdata, size = swim_member_payload_raw(ptr)\n"
"    local ok, result\n"
"    if size == 0 then\n"
"        result = nil\n"
"    else\n"
"        ok, result = pcall(msgpack.decode, cdata, size)\n"
"        if not ok then\n"
"            result = ffi.string(cdata, size)\n"
"        end\n"
"    end\n"
"    -- Member bans new indexes. Only rawset() can be used.\n"
"    rawset(m, 'p', result)\n"
"    rawset(m, 'p_key1', key1)\n"
"    rawset(m, 'p_key2', key2)\n"
"    return result\n"
"end\n"
"\n"
"--\n"
"-- Cdata UUID. It is ok to return cdata, because struct tt_uuid\n"
"-- type has strong support by 'uuid' Lua module with nice\n"
"-- metatable, serialization, string conversions etc.\n"
"--\n"
"local function swim_member_uuid(m)\n"
"    local ptr = swim_check_member(m, 'member:uuid()')\n"
"    local u = m.u\n"
"    if not u then\n"
"        ptr = capi.swim_member_uuid(ptr)\n"
"        u = ffi.new('struct tt_uuid')\n"
"        ffi.copy(u, ptr, ffi.sizeof('struct tt_uuid'))\n"
"        rawset(m, 'u', u)\n"
"    end\n"
"    return u\n"
"end\n"
"\n"
"local function swim_member_serialize(m)\n"
"    local ptr = swim_check_member(m, 'member:__serialize()')\n"
"    local _, size = swim_member_payload_raw(ptr)\n"
"    return {\n"
"        status = swim_member_status(m),\n"
"        uuid = swim_member_uuid(m),\n"
"        uri = swim_member_uri(m),\n"
"        incarnation = swim_member_incarnation(m),\n"
"        -- There are many ways to interpret a payload, and it is\n"
"        -- not a job of a serialization method. Only binary size\n"
"        -- here is returned to allow a user to detect, whether a\n"
"        -- payload exists.\n"
"        payload_size = size,\n"
"    }\n"
"end\n"
"\n"
"local swim_member_mt = {\n"
"    __index = {\n"
"        status = swim_member_status,\n"
"        uuid = swim_member_uuid,\n"
"        uri = swim_member_uri,\n"
"        incarnation = swim_member_incarnation,\n"
"        payload_cdata = swim_member_payload_cdata,\n"
"        payload_str = swim_member_payload_str,\n"
"        payload = swim_member_payload,\n"
"        is_dropped = swim_member_is_dropped,\n"
"    },\n"
"    __serialize = swim_member_serialize,\n"
"    __newindex = function(self)\n"
"        return error('swim_member is a read-only object')\n"
"    end\n"
"}\n"
"\n"
"--\n"
"-- Wrap a SWIM member into a table with proper metamethods. The\n"
"-- table-wrapper stores not only a pointer, but also cached\n"
"-- decoded payload.\n"
"--\n"
"local function swim_wrap_member(s, ptr)\n"
"    -- Lua tables can't normally work with cdata keys. Even when\n"
"    -- cdata is a simple number, table can't do search by it.\n"
"    local key = tonumber(ffi.cast('unsigned long', ptr))\n"
"    local cache = s.cache_table\n"
"    local wrapped = cache[key]\n"
"    if wrapped == nil then\n"
"        capi.swim_member_ref(ptr)\n"
"        ffi.gc(ptr, capi.swim_member_unref)\n"
"        wrapped = setmetatable({ptr = ptr}, swim_member_mt)\n"
"        cache[key] = wrapped\n"
"    end\n"
"    return wrapped\n"
"end\n"
"\n"
"--\n"
"-- When a SWIM instance is deleted or has quited, it can't be used\n"
"-- anymore. This function replaces all methods of a deleted\n"
"-- instance to throw an error on a usage attempt.\n"
"--\n"
"local function swim_error_deleted()\n"
"    return error('the swim instance is deleted')\n"
"end\n"
"-- This is done without 'if' in the original methods, but rather\n"
"-- via metatable replacement after deletion has happened.\n"
"local swim_mt_deleted = {\n"
"    __index = swim_error_deleted\n"
"}\n"
"\n"
"--\n"
"-- Delete a SWIM instance immediately, do not notify cluster\n"
"-- members about that.\n"
"--\n"
"local function swim_delete(s)\n"
"    local ptr = swim_check_instance(s, 'swim:delete')\n"
"    internal.swim_delete(ffi.gc(ptr, nil))\n"
"    s.ptr = nil\n"
"    setmetatable(s, swim_mt_deleted)\n"
"end\n"
"\n"
"--\n"
"-- Quit from a cluster gracefully, notify other members. The SWIM\n"
"-- instance is considered deleted immediately after this function\n"
"-- returned, and can't be used anymore.\n"
"--\n"
"local function swim_quit(s)\n"
"    local ptr = swim_check_instance(s, 'swim:quit')\n"
"    internal.swim_quit(ffi.gc(ptr, nil))\n"
"    s.ptr = nil\n"
"    setmetatable(s, swim_mt_deleted)\n"
"end\n"
"\n"
"--\n"
"-- Size of the local member table.\n"
"--\n"
"local function swim_size(s)\n"
"    return capi.swim_size(swim_check_instance(s, 'swim:size'))\n"
"end\n"
"\n"
"--\n"
"-- Check if a SWIM instance is configured already. Not configured\n"
"-- instance does not provide any methods.\n"
"--\n"
"local function swim_is_configured(s)\n"
"    return capi.swim_is_configured(swim_check_instance(s, 'swim:is_configured'))\n"
"end\n"
"\n"
"--\n"
"-- Configuration options are printed when a SWIM instance is\n"
"-- serialized, for example, in a console.\n"
"--\n"
"local function swim_serialize(s)\n"
"    swim_check_instance(s, 'swim:__serialize()')\n"
"    return s.cfg.index\n"
"end\n"
"\n"
"--\n"
"-- Ping a member probably located at @a uri.\n"
"--\n"
"local function swim_probe_member(s, uri)\n"
"    local func_name = 'swim:probe_member'\n"
"    local ptr = swim_check_instance(s, func_name)\n"
"    uri = swim_check_uri(uri, func_name)\n"
"    if capi.swim_probe_member(ptr, uri) ~= 0 then\n"
"        return nil, box.error.last()\n"
"    end\n"
"    return true\n"
"end\n"
"\n"
"--\n"
"-- Add a new member to the member table explicitly.\n"
"--\n"
"local function swim_add_member(s, cfg)\n"
"    local func_name = 'swim:add_member'\n"
"    local ptr = swim_check_instance(s, func_name)\n"
"    if type(cfg) ~= 'table' then\n"
"        return error(func_name..': expected table member definition')\n"
"    end\n"
"    local uri = swim_check_uri(cfg.uri, func_name)\n"
"    local uuid = swim_check_uuid(cfg.uuid, func_name)\n"
"    if capi.swim_add_member(ptr, uri, uuid) ~= 0 then\n"
"        return nil, box.error.last()\n"
"    end\n"
"    return true\n"
"end\n"
"\n"
"--\n"
"-- Remove a member by @a uuid immediately from the local member\n"
"-- table.\n"
"--\n"
"local function swim_remove_member(s, uuid)\n"
"    local func_name = 'swim:remove_member'\n"
"    local ptr = swim_check_instance(s, func_name)\n"
"    uuid = swim_check_uuid(uuid, func_name)\n"
"    if capi.swim_remove_member(ptr, uuid) ~= 0 then\n"
"        return nil, box.error.last()\n"
"    end\n"
"    return true\n"
"end\n"
"\n"
"--\n"
"-- Broadcast a ping message on all interfaces with @a port\n"
"-- destination. Port can be omitted, then currently bound is used.\n"
"--\n"
"local function swim_broadcast(s, port)\n"
"    local func_name = 'swim:broadcast'\n"
"    local ptr = swim_check_instance(s, func_name)\n"
"    if port == nil then\n"
"        port = -1\n"
"    else\n"
"        if type(port) == 'string' then\n"
"            port = tonumber(port)\n"
"        end\n"
"        if type(port) ~= 'number' then\n"
"            return error(func_name..': expected number port')\n"
"        end\n"
"    end\n"
"    if capi.swim_broadcast(ptr, port) ~= 0 then\n"
"        return nil, box.error.last()\n"
"    end\n"
"    return true\n"
"end\n"
"\n"
"--\n"
"-- Shortcut to get the self member in O(1) not making a lookup\n"
"-- into the member table.\n"
"--\n"
"local function swim_self(s)\n"
"    local ptr = swim_check_instance(s, 'swim:self')\n"
"    return swim_wrap_member(s, capi.swim_self(ptr))\n"
"end\n"
"\n"
"--\n"
"-- Find a member by UUID in the local member table.\n"
"--\n"
"local function swim_member_by_uuid(s, uuid)\n"
"    local func_name = 'swim:member_by_uuid'\n"
"    local ptr = swim_check_instance(s, func_name)\n"
"    uuid = swim_check_uuid(uuid, func_name)\n"
"    local m = capi.swim_member_by_uuid(ptr, uuid)\n"
"    if m == nil then\n"
"        return nil\n"
"    end\n"
"    return swim_wrap_member(s, m)\n"
"end\n"
"\n"
"--\n"
"-- Set raw payload without any preprocessing nor encoding. It can\n"
"-- be anything, not necessary MessagePack.\n"
"--\n"
"local function swim_set_payload_raw(s, payload, payload_size)\n"
"    local func_name = 'swim:set_payload_raw'\n"
"    local ptr = swim_check_instance(s, func_name)\n"
"    payload, payload_size =\n"
"        swim_check_const_char(payload, payload_size, func_name, 'payload')\n"
"    if capi.swim_set_payload(ptr, payload, payload_size) ~= 0 then\n"
"        return nil, box.error.last()\n"
"    end\n"
"    return true\n"
"end\n"
"\n"
"--\n"
"-- Set Lua object as a payload. It is encoded into MessagePack.\n"
"--\n"
"local function swim_set_payload(s, payload)\n"
"    local func_name = 'swim:set_payload'\n"
"    local ptr = swim_check_instance(s, func_name)\n"
"    local rc\n"
"    if payload == nil then\n"
"        rc = capi.swim_set_payload(ptr, nil, 0)\n"
"    else\n"
"        local buf = cord_ibuf_take()\n"
"        local payload_size = msgpack.encode(payload, buf)\n"
"        payload = buf.rpos\n"
"        rc = capi.swim_set_payload(ptr, payload, payload_size)\n"
"        cord_ibuf_put(buf)\n"
"    end\n"
"    if rc ~= 0 then\n"
"        return nil, box.error.last()\n"
"    end\n"
"    return true\n"
"end\n"
"\n"
"--\n"
"-- Set encryption algorithm, mode, and a private key.\n"
"--\n"
"local function swim_set_codec(s, cfg)\n"
"    local func_name = 'swim:set_codec'\n"
"    local ptr = swim_check_instance(s, func_name)\n"
"    if type(cfg) ~= 'table' then\n"
"        error(func_name..': expected table codec configuration')\n"
"    end\n"
"    local algo = crypto.cipher_algo[cfg.algo]\n"
"    if algo == nil then\n"
"        error(func_name..': unknown crypto algorithm')\n"
"    end\n"
"    local mode = cfg.mode\n"
"    if mode == nil then\n"
"        mode = crypto.cipher_mode.cbc\n"
"    else\n"
"        mode = crypto.cipher_mode[mode]\n"
"        if mode == nil then\n"
"            error(func_name..': unknown crypto algorithm mode')\n"
"        end\n"
"    end\n"
"    local key, key_size =\n"
"        swim_check_const_char(cfg.key, cfg.key_size, func_name, 'key')\n"
"    if capi.swim_set_codec(ptr, algo, mode, key, key_size) ~= 0 then\n"
"        return nil, box.error.last()\n"
"    end\n"
"    return true\n"
"end\n"
"\n"
"--\n"
"-- Lua pairs() or similar function should return 3 values:\n"
"-- iterator function, iterator object, a key before first. This is\n"
"-- iterator function. On each iteration it returns UUID as a key,\n"
"-- member object as a value.\n"
"--\n"
"local function swim_pairs_next(ctx)\n"
"    local s = ctx.swim\n"
"    if s.ptr == nil then\n"
"        return swim_error_deleted()\n"
"    end\n"
"    local iterator = ctx.iterator\n"
"    local m = capi.swim_iterator_next(iterator)\n"
"    if m ~= nil then\n"
"        m = swim_wrap_member(s, m)\n"
"        return m:uuid(), m\n"
"    end\n"
"    capi.swim_iterator_close(ffi.gc(iterator, nil))\n"
"    ctx.iterator = nil\n"
"end\n"
"\n"
"--\n"
"-- Pairs() to use in 'for' cycles.\n"
"--\n"
"local function swim_pairs(s)\n"
"    local ptr = swim_check_instance(s, 'swim:pairs')\n"
"    local iterator = capi.swim_iterator_open(ptr)\n"
"    if iterator == nil then\n"
"        return nil, box.error.last()\n"
"    end\n"
"    ffi.gc(iterator, capi.swim_iterator_close)\n"
"    return swim_pairs_next, {swim = s, iterator = iterator}, nil\n"
"end\n"
"\n"
"local swim_member_event_index = {\n"
"    is_new = function(self)\n"
"        local value = swim_check_event(self, 'event:is_new()')\n"
"        return bit.band(value, capi.SWIM_EV_NEW) ~= 0\n"
"    end,\n"
"    is_drop = function(self)\n"
"        local value = swim_check_event(self, 'event:is_drop()')\n"
"        return bit.band(value, capi.SWIM_EV_DROP) ~= 0\n"
"    end,\n"
"    is_update = function(self)\n"
"        local value = swim_check_event(self, 'event:is_update()')\n"
"        return bit.band(value, capi.SWIM_EV_UPDATE) ~= 0\n"
"    end,\n"
"    is_new_status = function(self)\n"
"        local value = swim_check_event(self, 'event:is_new_status()')\n"
"        return bit.band(value, capi.SWIM_EV_NEW_STATUS) ~= 0\n"
"    end,\n"
"    is_new_uri = function(self)\n"
"        local value = swim_check_event(self, 'event:is_new_uri()')\n"
"        return bit.band(value, capi.SWIM_EV_NEW_URI) ~= 0\n"
"    end,\n"
"    is_new_incarnation = function(self)\n"
"        local value = swim_check_event(self, 'event:is_new_incarnation()')\n"
"        return bit.band(value, capi.SWIM_EV_NEW_INCARNATION) ~= 0\n"
"    end,\n"
"    is_new_generation = function(self)\n"
"        local value = swim_check_event(self, 'event:is_new_generation()')\n"
"        return bit.band(value, capi.SWIM_EV_NEW_GENERATION) ~= 0\n"
"    end,\n"
"    is_new_version = function(self)\n"
"        local value = swim_check_event(self, 'event:is_new_version()')\n"
"        return bit.band(value, capi.SWIM_EV_NEW_VERSION) ~= 0\n"
"    end,\n"
"    is_new_payload = function(self)\n"
"        local value = swim_check_event(self, 'event:is_new_payload()')\n"
"        return bit.band(value, capi.SWIM_EV_NEW_PAYLOAD) ~= 0\n"
"    end,\n"
"}\n"
"\n"
"local swim_member_event_mt = {\n"
"    __index = swim_member_event_index,\n"
"    __serialize = function(self)\n"
"        local res = {}\n"
"        for k, v in pairs(swim_member_event_index) do\n"
"            v = v(self)\n"
"            if v then\n"
"                res[k] = v\n"
"            end\n"
"        end\n"
"        return res\n"
"    end,\n"
"}\n"
"\n"
"--\n"
"-- Create a closure function for preprocessing raw SWIM member\n"
"-- event trigger parameters.\n"
"-- @param s SWIM instance.\n"
"-- @param callback User functions to call.\n"
"-- @param ctx An optional parameter for @a callback passed as is.\n"
"-- @return A function to set as a trigger.\n"
"--\n"
"local function swim_on_member_event_new(s, callback, ctx)\n"
"    -- Do not keep a hard reference to a SWIM instance. Otherwise\n"
"    -- it is a cyclic reference, and both the instance and the\n"
"    -- trigger will never be GC-ed.\n"
"    s = setmetatable({s}, {__mode = 'v'})\n"
"    return function(member_ptr, event_mask)\n"
"        local s = s[1]\n"
"        if s then\n"
"            local m = swim_wrap_member(s, member_ptr)\n"
"            local event = setmetatable({event_mask}, swim_member_event_mt)\n"
"            return callback(m, event, ctx)\n"
"        end\n"
"    end\n"
"end\n"
"\n"
"--\n"
"-- Add or/and delete a trigger on member event. Possible usages:\n"
"--\n"
"-- * on_member_event(new[, ctx]) - add a new trigger. It should\n"
"--   accept 3 arguments: an updated member, an events object, an\n"
"--   optional @a ctx parameter passed as is.\n"
"--\n"
"-- * on_member_event(new, old[, ctx]) - add a new trigger @a new\n"
"--   if not nil, in place of @a old trigger.\n"
"--\n"
"-- * on_member_event() - get a list of triggers.\n"
"--\n"
"local function swim_on_member_event(s, new, old, ctx)\n"
"    local ptr = swim_check_instance(s, 'swim:on_member_event')\n"
"    if type(old) ~= 'function' then\n"
"        ctx = old\n"
"        old = nil\n"
"    end\n"
"    if new ~= nil then\n"
"        new = swim_on_member_event_new(s, new, ctx)\n"
"    end\n"
"    return internal.swim_on_member_event(ptr, new, old)\n"
"end\n"
"\n"
"--\n"
"-- Normal metatable of a configured SWIM instance.\n"
"--\n"
"local swim_mt = {\n"
"    __index = {\n"
"        delete = swim_delete,\n"
"        quit = swim_quit,\n"
"        size = swim_size,\n"
"        is_configured = swim_is_configured,\n"
"        probe_member = swim_probe_member,\n"
"        add_member = swim_add_member,\n"
"        remove_member = swim_remove_member,\n"
"        broadcast = swim_broadcast,\n"
"        self = swim_self,\n"
"        member_by_uuid = swim_member_by_uuid,\n"
"        set_payload_raw = swim_set_payload_raw,\n"
"        set_payload = swim_set_payload,\n"
"        set_codec = swim_set_codec,\n"
"        pairs = swim_pairs,\n"
"        on_member_event = swim_on_member_event,\n"
"    },\n"
"    __serialize = swim_serialize\n"
"}\n"
"\n"
"local swim_cfg_options = {\n"
"    uri = true, heartbeat_rate = true, ack_timeout = true,\n"
"    gc_mode = true, uuid = true\n"
"}\n"
"\n"
"--\n"
"-- SWIM 'cfg' attribute is not a trivial table nor function. It is\n"
"-- a callable table. It allows to strongly restrict use cases of\n"
"-- swim.cfg down to 2 applications:\n"
"--\n"
"-- - Cfg-table. 'swim.cfg' is a read-only table of configured\n"
"--   parameters. A user can't write 'swim.cfg.<key> = value' - an\n"
"--   error is thrown. But it can be indexed like\n"
"--   'opt = swim.cfg.<key>'. Configuration is cached in Lua, so\n"
"--   the latter example won't even call SWIM C API functions.\n"
"--\n"
"-- - Cfg-function. 'swim:cfg(<new config>)' reconfigures a SWIM\n"
"--   instance and returns normal values: nil+error or true. The\n"
"--   new configuration is cached for further indexing.\n"
"--\n"
"-- All the other combinations are banned. The function below\n"
"-- implements configuration call.\n"
"--\n"
"local function swim_cfg_call(c, s, cfg)\n"
"    local func_name = 'swim:cfg'\n"
"    local ptr = swim_check_instance(s, func_name)\n"
"    if type(cfg) ~= 'table' then\n"
"        return error(func_name..': expected table configuration')\n"
"    end\n"
"    for k in pairs(cfg) do\n"
"        if not swim_cfg_options[k] then\n"
"            return error(func_name..': unknown option '..k)\n"
"        end\n"
"    end\n"
"    local uri = swim_check_uri(cfg.uri, func_name)\n"
"    local heartbeat_rate = swim_check_timeout(cfg.heartbeat_rate,\n"
"                                              func_name, 'heartbeat_rate')\n"
"    local ack_timeout = swim_check_timeout(cfg.ack_timeout, func_name,\n"
"                                           'ack_timeout');\n"
"    local gc_mode = swim_check_gc_mode(cfg.gc_mode, func_name)\n"
"    local uuid = swim_check_uuid(cfg.uuid, func_name)\n"
"    if capi.swim_cfg(ptr, uri, heartbeat_rate, ack_timeout,\n"
"                     gc_mode, uuid) ~= 0 then\n"
"        return nil, box.error.last()\n"
"    end\n"
"    local index = c.index\n"
"    for k, v in pairs(cfg) do\n"
"        index[k] = v\n"
"    end\n"
"    return true\n"
"end\n"
"\n"
"local swim_cfg_mt = {\n"
"    __call = swim_cfg_call,\n"
"    __index = function(c, k)\n"
"        return c.index[k]\n"
"    end,\n"
"    __serialize = function(c)\n"
"        return c.index\n"
"    end,\n"
"    __newindex = function()\n"
"        return error('please, use swim:cfg{key = value} instead of '..\n"
"                     'swim.cfg.key = value')\n"
"    end\n"
"}\n"
"\n"
"--\n"
"-- First 'swim:cfg()' call is different from others. On success it\n"
"-- replaces swim metatable with a full one, where all the variety\n"
"-- of methods is available.\n"
"--\n"
"local function swim_cfg_first_call(c, s, cfg)\n"
"    local ok, err = swim_cfg_call(c, s, cfg)\n"
"    if not ok then\n"
"        return ok, err\n"
"    end\n"
"    -- Update 'cfg' metatable as well to never call this\n"
"    -- function again and use ordinary swim_cfg_call() directly.\n"
"    setmetatable(c, swim_cfg_mt)\n"
"    setmetatable(s, swim_mt)\n"
"    return ok\n"
"end\n"
"\n"
"local swim_not_configured_mt = {\n"
"    __index = {\n"
"        delete = swim_delete,\n"
"        is_configured = swim_is_configured,\n"
"        set_codec = swim_set_codec,\n"
"        on_member_event = swim_on_member_event,\n"
"    },\n"
"    __serialize = swim_serialize\n"
"}\n"
"\n"
"local swim_cfg_not_configured_mt = table.deepcopy(swim_cfg_mt)\n"
"swim_cfg_not_configured_mt.__call = swim_cfg_first_call\n"
"\n"
"-- Member cache stores week references so as to do not care about\n"
"-- removed members erasure - GC drops them automatically.\n"
"local cache_table_mt = { __mode = 'v' }\n"
"\n"
"--\n"
"-- SWIM garbage collection function. It can't delete the SWIM\n"
"-- instance immediately, because it is invoked by Lua GC. Firstly,\n"
"-- it is not safe to yield in FFI - Jit can't survive a yield.\n"
"-- Secondly, it is not safe to yield in any GC function, because\n"
"-- it stops garbage collection. Instead, here GC is delayed, works\n"
"-- at the end of the event loop, and deletes the instance\n"
"-- asynchronously.\n"
"--\n"
"local function swim_gc(ptr)\n"
"    schedule_task(internal.swim_delete, ptr)\n"
"end\n"
"\n"
"--\n"
"-- Create a new SWIM instance, and configure if @a cfg is\n"
"-- provided.\n"
"--\n"
"local function swim_new(cfg)\n"
"    local generation\n"
"    if cfg and type(cfg) == 'table' and cfg.generation then\n"
"        generation = cfg.generation\n"
"        if type(generation) ~= 'number' or generation < 0 or\n"
"           math.floor(generation) ~= generation then\n"
"            return error('swim.new: generation should be non-negative integer')\n"
"        end\n"
"        cfg = table.copy(cfg)\n"
"        -- Nullify in order to do not raise errors in the\n"
"        -- following swim:cfg() about unknown parameters.\n"
"        cfg.generation = nil\n"
"        -- When only 'generation' is specified, empty config will\n"
"        -- raise an error at swim:cfg(). It should not.\n"
"        if next(cfg) == nil then\n"
"            cfg = nil\n"
"        end\n"
"    else\n"
"        generation = fiber.time64()\n"
"    end\n"
"    local ptr = internal.swim_new(generation)\n"
"    if ptr == nil then\n"
"        return nil, box.error.last()\n"
"    end\n"
"    ffi.gc(ptr, swim_gc)\n"
"    local s = setmetatable({\n"
"        ptr = ptr,\n"
"        cfg = setmetatable({index = {}}, swim_cfg_not_configured_mt),\n"
"        cache_table = setmetatable({}, cache_table_mt)\n"
"    }, swim_not_configured_mt)\n"
"    if cfg then\n"
"        local ok, err = s:cfg(cfg)\n"
"        if not ok then\n"
"            s:delete()\n"
"            return ok, err\n"
"        end\n"
"    end\n"
"    return s\n"
"end\n"
"\n"
"return {\n"
"    new = swim_new,\n"
"}\n"
""
;
