// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CONTENT_SERVICES_AUCTION_WORKLET_TRUSTED_SIGNALS_H_
#define CONTENT_SERVICES_AUCTION_WORKLET_TRUSTED_SIGNALS_H_

#include <map>
#include <memory>
#include <optional>
#include <set>
#include <string>
#include <vector>

#include "base/containers/flat_map.h"
#include "base/functional/callback.h"
#include "base/memory/ref_counted.h"
#include "base/memory/scoped_refptr.h"
#include "base/task/sequenced_task_runner.h"
#include "base/time/time.h"
#include "content/common/content_export.h"
#include "content/services/auction_worklet/auction_v8_helper.h"
#include "content/services/auction_worklet/public/mojom/auction_network_events_handler.mojom.h"
#include "content/services/auction_worklet/public/mojom/seller_worklet.mojom-forward.h"
#include "net/http/http_response_headers.h"
#include "services/network/public/mojom/url_loader_factory.mojom-forward.h"
#include "third_party/blink/public/common/interest_group/ad_display_size.h"
#include "url/gurl.h"
#include "url/origin.h"
#include "v8/include/v8-forward.h"

namespace auction_worklet {

class AuctionDownloader;

// Represents the trusted bidding/scoring signals that are part of the FLEDGE
// bidding system (https://github.com/WICG/turtledove/blob/main/FLEDGE.md).
// Fetches and parses the hosted JSON data files needed by worklets. There are
// separate methods for fetching bidding and scoring signals. A single
// TrustedSignals object can only be used to fetch bidding signals or scoring
// signals, even if a single URL is used for both types of signals.
class CONTENT_EXPORT TrustedSignals {
 public:
  // Contains the values returned by the server.
  //
  // This can be created and destroyed on any thread, but GetSignals() can only
  // be used on the V8 thread.
  class CONTENT_EXPORT Result : public base::RefCountedThreadSafe<Result> {
   public:
    using PriorityVector = base::flat_map<std::string, double>;

    struct CONTENT_EXPORT PerGroupData {
      PerGroupData(std::optional<PriorityVector> priority_vector,
                   std::optional<base::TimeDelta> update_if_older_than);
      ~PerGroupData();

      PerGroupData(PerGroupData&&);
      PerGroupData& operator=(PerGroupData&&);

      std::optional<PriorityVector> priority_vector;
      std::optional<base::TimeDelta> update_if_older_than;
    };

    using PerInterestGroupDataMap = std::map<std::string, PerGroupData>;

    // Constructor for bidding signals.
    Result(PerInterestGroupDataMap per_interest_group_data,
           std::map<std::string, AuctionV8Helper::SerializedValue> bidder_data,
           std::optional<uint32_t> data_version);

    // Constructor for scoring signals.
    Result(
        std::map<std::string, AuctionV8Helper::SerializedValue> render_url_data,
        std::map<std::string, AuctionV8Helper::SerializedValue>
            ad_component_data,
        std::optional<uint32_t> data_version);

    explicit Result(const Result&) = delete;
    Result& operator=(const Result&) = delete;

    // Retrieves the `perInterestGroupData` associated with the passed in
    // interest group. `this` must have been generated by fetching bidding
    // signals. Unlike other methods, this does not call into v8, so may be
    // called on any thread.
    //
    // Returns nullptr if there's no matching `perInterestGroupData` in the
    // source dict. The `priority_vector` flat_map will be empty if it has no
    // valid entries, and the `update_if_older_than` will be std::nullopt if
    // `updateIfOlderThanMs` was not specified in the response JSON.
    const TrustedSignals::Result::PerGroupData* GetPerGroupData(
        const std::string& interest_group_name) const;

    // Retrieves the trusted bidding signals associated with the passed in keys,
    // in the format expected by a worklet's generateBid() method. `this` must
    // have been generated by fetching bidding signals. `v8_helper`'s Isolate
    // must be active (in particular, this must be on the v8 thread), and
    // `context` must be the active context. `bidding_signals_keys` must be
    // subsets of the keys provided when creating the TrustedSignals object.
    // Always returns a non-empty value.
    v8::Local<v8::Object> GetBiddingSignals(
        AuctionV8Helper* v8_helper,
        v8::Local<v8::Context> context,
        const std::vector<std::string>& bidding_signals_keys) const;

    // Retrieves the trusted scoring signals associated with the passed in urls,
    // in the format expected by a worklet's scoreAd() method. `this` must have
    // been generated by fetching scoring signals. `v8_helper`'s Isolate must be
    // active (in particular, this must be on the v8 thread), and `context` must
    // be the active context. `render_url` and `ad_component_render_urls` must
    // be subsets of the corresponding sets of GURLs provided when creating the
    // TrustedSignals object. Always returns a non-empty value.
    v8::Local<v8::Object> GetScoringSignals(
        AuctionV8Helper* v8_helper,
        v8::Local<v8::Context> context,
        const GURL& render_url,
        const std::vector<mojom::CreativeInfoWithoutOwnerPtr>& ad_components)
        const;

    std::optional<uint32_t> GetDataVersion() const { return data_version_; }

    // If `signals` is not undefined or not null, returns an object containing
    // `signals` as a field under the key `source_origin`.
    // Otherwise, returns JS null.
    static v8::Local<v8::Value> WrapCrossOriginSignals(
        AuctionV8Helper* v8_helper,
        v8::Local<v8::Context> context,
        const url::Origin& source_origin,
        v8::Local<v8::Value> signals);

   private:
    friend class base::RefCountedThreadSafe<Result>;

    ~Result();

    // Per-interest-group data returned by the trusted bidding server.
    const std::optional<PerInterestGroupDataMap> per_interest_group_data_;
    // Map of keys to the associated data for trusted bidding signals.
    const std::optional<std::map<std::string, AuctionV8Helper::SerializedValue>>
        bidder_data_;

    // Map of keys to the associated data for trusted scoring signals.
    const std::optional<std::map<std::string, AuctionV8Helper::SerializedValue>>
        render_url_data_;
    const std::optional<std::map<std::string, AuctionV8Helper::SerializedValue>>
        ad_component_data_;

    // Data version associated with the trusted signals.
    const std::optional<uint32_t> data_version_;
  };

  // Info about a creative, either ad or component ad, that's sent to trusted
  // scoring signals server, corresponding to one chosen by a generateBid()
  // invocation. `buyer_and_seller_reporting_id` is only applicable, and only
  // sent, for ads - not for ad components - as ad components may not provide a
  // value for `buyer_and_seller_reporting_id` or any other reporting IDs.
  //
  // If operating with `send_creative_scanning_metadata` true, the same URL may
  // need to be repeated, in cases like it occurring in multiple interest groups
  // with the same ad creative but different scanning metadata.
  //
  // When `send_creative_scanning_metadata` is false, all fields other than
  // `ad_descriptor`'s `url` must be kept empty to avoid needlessly duplicating
  // URLs.
  struct CONTENT_EXPORT CreativeInfo {
    CreativeInfo();
    CreativeInfo(blink::AdDescriptor ad_descriptor,
                 std::string creative_scanning_metadata,
                 std::optional<url::Origin> interest_group_owner,
                 std::string buyer_and_seller_reporting_id);
    CreativeInfo(bool send_creative_scanning_metadata,
                 const mojom::CreativeInfoWithoutOwner& mojo_creative_info,
                 const url::Origin& in_interest_group_owner,
                 const std::optional<std::string>&
                     browser_signal_buyer_and_seller_reporting_id);
    ~CreativeInfo();

    CreativeInfo(CreativeInfo&&);
    CreativeInfo(const CreativeInfo&);
    CreativeInfo& operator=(CreativeInfo&&);
    CreativeInfo& operator=(const CreativeInfo&);

    bool operator<(const CreativeInfo& other) const;

    // The ad and size selected by generateBid().
    blink::AdDescriptor ad_descriptor;

    // From `InterestGroup::Ad::creative_scanning_metadata`, with nullopt
    // converted to empty string.
    std::string creative_scanning_metadata;

    // From `InterestGroup::owner`.
    std::optional<url::Origin> interest_group_owner;

    // From `InterestGroup::Ad::buyer_and_seller_reporting_id`, with nullopt
    // converted to empty string.
    std::string buyer_and_seller_reporting_id;
  };

  using LoadSignalsCallback =
      base::OnceCallback<void(scoped_refptr<Result> result,
                              std::optional<std::string> error_msg)>;

  explicit TrustedSignals(const TrustedSignals&) = delete;
  TrustedSignals& operator=(const TrustedSignals&) = delete;
  ~TrustedSignals();

  static GURL BuildTrustedBiddingSignalsURL(
      const std::string& hostname,
      const GURL& trusted_bidding_signals_url,
      const std::set<std::string>& interest_group_names,
      const std::set<std::string>& bidding_signals_keys,
      std::optional<uint16_t> experiment_group_id,
      const std::string& trusted_bidding_signals_slot_size_param);

  // `ads` and `component_ads` are set<CreativeInfo> rather than
  // map<GURL, something> because the same URL can have different creative
  // scanning metadata in different IGs, or different size in difference
  // occurrences as a component ad, etc.
  static GURL BuildTrustedScoringSignalsURL(
      bool send_creative_scanning_metadata,
      const std::string& hostname,
      const GURL& trusted_scoring_signals_url,
      const std::set<CreativeInfo>& ads,
      const std::set<CreativeInfo>& component_ads,
      std::optional<uint16_t> experiment_group_id);

  // Constructs a TrustedSignals for fetching bidding signals, and starts
  // the fetch. `trusted_bidding_signals_url` must be the base URL (no query
  // params added).  Callback will be invoked asynchronously once the data
  // has been fetched or an error has occurred. De-duplicates keys when
  // assembling the full URL for the fetch. Fails if the URL already has a
  // query param (or has a location or embedded credentials) or if the
  // response is not JSON. If some or all of the render URLs are missing,
  // still succeeds, and GetSignals() will populate them with nulls.
  //
  // If non-empty, "&`trusted_bidding_signals_slot_size_param`" is appended
  // to the end of the query string. It's expected to already be escaped if
  // necessary.
  //
  // There are no lifetime constraints of `url_loader_factory`.
  static std::unique_ptr<TrustedSignals> LoadBiddingSignals(
      network::mojom::URLLoaderFactory* url_loader_factory,
      mojo::PendingRemote<auction_worklet::mojom::AuctionNetworkEventsHandler>
          auction_network_events_handler,
      std::set<std::string> interest_group_names,
      std::set<std::string> bidding_signals_keys,
      const std::string& hostname,
      const GURL& trusted_bidding_signals_url,
      std::optional<uint16_t> experiment_group_id,
      const std::string& trusted_bidding_signals_slot_size_param,
      scoped_refptr<AuctionV8Helper> v8_helper,
      LoadSignalsCallback load_signals_callback);

  // Just like LoadBiddingSignals() above, but for fetching seller signals.
  static std::unique_ptr<TrustedSignals> LoadScoringSignals(
      network::mojom::URLLoaderFactory* url_loader_factory,
      mojo::PendingRemote<auction_worklet::mojom::AuctionNetworkEventsHandler>
          auction_network_events_handler,
      std::set<CreativeInfo> ads,
      std::set<CreativeInfo> ad_components,
      const std::string& hostname,
      const GURL& trusted_scoring_signals_url,
      std::optional<uint16_t> experiment_group_id,
      bool send_creative_scanning_metadata,
      scoped_refptr<AuctionV8Helper> v8_helper,
      LoadSignalsCallback load_signals_callback);

  // Attempts to parse the `priorityVector` value in
  // `v8_per_interest_group_data`,
  // expecting it to be a string-to-number mapping. Returns the parsed mapping,
  // or nullopt upon failure to find or parse the field. Any case where
  // `priorityVector` exists and is an object is considered a success, even if
  // it's empty, or some/all keys in it are mapped to things other than numbers.
  // Must be called on `v8_helper`'s sequence.
  static std::optional<TrustedSignals::Result::PriorityVector>
  ParsePriorityVector(AuctionV8Helper* v8_helper,
                      v8::Local<v8::Object> v8_per_interest_group_data);

  // Attempts to parse the `updateIfOlderThanMs` value in
  // `v8_per_interest_group_data`, expecting it to be a double duration in
  // milliseconds. Returns the time delta, or nullopt upon failure to find or
  // parse the value. Must be called on `v8_helper`'s sequence.
  static std::optional<base::TimeDelta> ParseUpdateIfOlderThan(
      AuctionV8Helper* v8_helper,
      v8::Local<v8::Object> v8_per_interest_group_data);

 private:
  TrustedSignals(
      std::optional<std::set<std::string>> interest_group_names,
      std::optional<std::set<std::string>> bidding_signals_keys,
      std::optional<std::set<CreativeInfo>> ads,
      std::optional<std::set<CreativeInfo>> ad_components,
      const GURL& trusted_signals_url,
      mojo::PendingRemote<auction_worklet::mojom::AuctionNetworkEventsHandler>
          auction_network_events_handler,
      scoped_refptr<AuctionV8Helper> v8_helper,
      LoadSignalsCallback load_signals_callback);

  // Starts downloading `url`, which should be the bidding or scoring signals
  // URL with the query parameter correctly set.
  void StartDownload(network::mojom::URLLoaderFactory* url_loader_factory,
                     const GURL& full_signals_url);

  void OnDownloadComplete(std::unique_ptr<std::string> body,
                          scoped_refptr<net::HttpResponseHeaders> headers,
                          std::optional<std::string> error_msg);

  // Parses the response body on the V8 thread, and extracts values associated
  // with the requested keys.
  static void HandleDownloadResultOnV8Thread(
      scoped_refptr<AuctionV8Helper> v8_helper,
      const GURL& signals_url,
      std::optional<std::set<std::string>> interest_group_names,
      std::optional<std::set<std::string>> bidding_signals_keys,
      std::optional<std::set<CreativeInfo>> ads,
      std::optional<std::set<CreativeInfo>> ad_components,
      std::unique_ptr<std::string> body,
      scoped_refptr<net::HttpResponseHeaders> headers,
      std::optional<std::string> error_msg,
      scoped_refptr<base::SequencedTaskRunner> user_thread_task_runner,
      base::WeakPtr<TrustedSignals> weak_instance,
      base::TimeDelta download_time);

  // Called from V8 thread.
  static void PostCallbackToUserThread(
      scoped_refptr<base::SequencedTaskRunner> user_thread_task_runner,
      base::WeakPtr<TrustedSignals> weak_instance,
      scoped_refptr<Result> result,
      std::optional<std::string> error_msg);

  // Called on user thread.
  void DeliverCallbackOnUserThread(scoped_refptr<Result> result,
                                   std::optional<std::string> error_msg);

  // Keys being fetched. For bidding signals, only `bidding_signals_keys_` and
  // `interest_group_names_` are non-null. For scoring signals, only
  // `ads_` and `ad_components_` are non-null. These are
  // cleared and ownership is passed to the V8 thread once the download
  // completes, as they're no longer on the main thread after that point.
  std::optional<std::set<std::string>> interest_group_names_;
  std::optional<std::set<std::string>> bidding_signals_keys_;
  std::optional<std::set<CreativeInfo>> ads_;
  std::optional<std::set<CreativeInfo>> ad_components_;

  const GURL trusted_signals_url_;  // original, for error messages.
  const scoped_refptr<AuctionV8Helper> v8_helper_;

  LoadSignalsCallback load_signals_callback_;
  std::unique_ptr<AuctionDownloader> auction_downloader_;
  // Used only for metrics; time when download started.
  base::TimeTicks download_start_time_;

  mojo::PendingRemote<auction_worklet::mojom::AuctionNetworkEventsHandler>
      auction_network_events_handler_;

  base::WeakPtrFactory<TrustedSignals> weak_ptr_factory{this};
};

}  // namespace auction_worklet

#endif  // CONTENT_SERVICES_AUCTION_WORKLET_TRUSTED_SIGNALS_H_
