// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.attn_fwd.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_attn_fwd__A4__F406

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"PERSISTENT_TYPE": 2, "GRID_CU_MULTIP": 2, "BLOCK_M": 16, "BLOCK_N": 16, "PRE_LOAD_V": false})xyzw",
  R"xyzw({"PERSISTENT_TYPE": 2, "GRID_CU_MULTIP": 2, "BLOCK_M": 16, "BLOCK_N": 16, "PRE_LOAD_V": false})xyzw",
  R"xyzw({"PERSISTENT_TYPE": 2, "GRID_CU_MULTIP": 2, "BLOCK_M": 16, "BLOCK_N": 16, "PRE_LOAD_V": false})xyzw",
  R"xyzw({"PERSISTENT_TYPE": 2, "GRID_CU_MULTIP": 2, "BLOCK_M": 16, "BLOCK_N": 16, "PRE_LOAD_V": false})xyzw",
  R"xyzw({"PERSISTENT_TYPE": 2, "GRID_CU_MULTIP": 2, "BLOCK_M": 16, "BLOCK_N": 16, "PRE_LOAD_V": true})xyzw",
  R"xyzw({"PERSISTENT_TYPE": 2, "GRID_CU_MULTIP": 2, "BLOCK_M": 16, "BLOCK_N": 16, "PRE_LOAD_V": true})xyzw",
  R"xyzw({"PERSISTENT_TYPE": 2, "GRID_CU_MULTIP": 2, "BLOCK_M": 16, "BLOCK_N": 16, "PRE_LOAD_V": true})xyzw",
  R"xyzw({"PERSISTENT_TYPE": 2, "GRID_CU_MULTIP": 2, "BLOCK_M": 16, "BLOCK_N": 16, "PRE_LOAD_V": true})xyzw",
  R"xyzw({"PERSISTENT_TYPE": 2, "GRID_CU_MULTIP": 2, "BLOCK_M": 32, "BLOCK_N": 32, "PRE_LOAD_V": true})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
    int8_t  PERSISTENT_TYPE;
    int8_t  GRID_CU_MULTIP;
    bool    PRE_LOAD_V;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16, .PERSISTENT_TYPE = 2, .GRID_CU_MULTIP = 2, .PRE_LOAD_V = false },
    { .BLOCK_M = 16, .BLOCK_N = 16, .PERSISTENT_TYPE = 2, .GRID_CU_MULTIP = 2, .PRE_LOAD_V = false },
    { .BLOCK_M = 16, .BLOCK_N = 16, .PERSISTENT_TYPE = 2, .GRID_CU_MULTIP = 2, .PRE_LOAD_V = false },
    { .BLOCK_M = 16, .BLOCK_N = 16, .PERSISTENT_TYPE = 2, .GRID_CU_MULTIP = 2, .PRE_LOAD_V = false },
    { .BLOCK_M = 16, .BLOCK_N = 16, .PERSISTENT_TYPE = 2, .GRID_CU_MULTIP = 2, .PRE_LOAD_V = true },
    { .BLOCK_M = 16, .BLOCK_N = 16, .PERSISTENT_TYPE = 2, .GRID_CU_MULTIP = 2, .PRE_LOAD_V = true },
    { .BLOCK_M = 16, .BLOCK_N = 16, .PERSISTENT_TYPE = 2, .GRID_CU_MULTIP = 2, .PRE_LOAD_V = true },
    { .BLOCK_M = 16, .BLOCK_N = 16, .PERSISTENT_TYPE = 2, .GRID_CU_MULTIP = 2, .PRE_LOAD_V = true },
    { .BLOCK_M = 32, .BLOCK_N = 32, .PERSISTENT_TYPE = 2, .GRID_CU_MULTIP = 2, .PRE_LOAD_V = true }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr std::string_view PACKAGE_PATH { R"xyzw(amd-gfx120x/flash/attn_fwd/FONLY__＊fp32@16_32_F_T_3_0___gfx120x)xyzw" };
constexpr std::string_view FUNC_NAME { R"xyzw(＊fp32@16_32_F_T_3_0)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx1200)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0x03577c69u, 0x721ce5ceu, 566, 48 }, // 03577c69721ce5ce = b2sum -l 64 <<< amd-gfx120x/flash/attn_fwd/FONLY__＊fp32@16_32_F_T_3_0___gfx120x__P__2_2_16_16_False__CO__wave1_warp2_stg1--Arch_gfx1200
    { 0x5f883b58u, 0x2253566du, 566, 65 }, // 5f883b582253566d = b2sum -l 64 <<< amd-gfx120x/flash/attn_fwd/FONLY__＊fp32@16_32_F_T_3_0___gfx120x__P__2_2_16_16_False__CO__wave2_warp2_stg1--Arch_gfx1200
    { 0x6d993850u, 0x80f5272cu, 566, 82 }, // 6d99385080f5272c = b2sum -l 64 <<< amd-gfx120x/flash/attn_fwd/FONLY__＊fp32@16_32_F_T_3_0___gfx120x__P__2_2_16_16_False__CO__wave3_warp2_stg1--Arch_gfx1200
    { 0xfc920304u, 0xe4baea89u, 566, 16 }, // fc920304e4baea89 = b2sum -l 64 <<< amd-gfx120x/flash/attn_fwd/FONLY__＊fp32@16_32_F_T_3_0___gfx120x__P__2_2_16_16_False__CO__wave4_warp2_stg1--Arch_gfx1200
    { 0x5b723dc5u, 0x3421d3deu, 520, 48 }, // 5b723dc53421d3de = b2sum -l 64 <<< amd-gfx120x/flash/attn_fwd/FONLY__＊fp32@16_32_F_T_3_0___gfx120x__P__2_2_16_16_True__CO__wave1_warp2_stg1--Arch_gfx1200
    { 0xeab0eadeu, 0x1ad8f31eu, 520, 65 }, // eab0eade1ad8f31e = b2sum -l 64 <<< amd-gfx120x/flash/attn_fwd/FONLY__＊fp32@16_32_F_T_3_0___gfx120x__P__2_2_16_16_True__CO__wave2_warp2_stg1--Arch_gfx1200
    { 0x61a6c739u, 0xc20b8f71u, 520, 82 }, // 61a6c739c20b8f71 = b2sum -l 64 <<< amd-gfx120x/flash/attn_fwd/FONLY__＊fp32@16_32_F_T_3_0___gfx120x__P__2_2_16_16_True__CO__wave3_warp2_stg1--Arch_gfx1200
    { 0x6d14ecbeu, 0x7d0658a4u, 520, 16 }, // 6d14ecbe7d0658a4 = b2sum -l 64 <<< amd-gfx120x/flash/attn_fwd/FONLY__＊fp32@16_32_F_T_3_0___gfx120x__P__2_2_16_16_True__CO__wave4_warp2_stg1--Arch_gfx1200
    { 0x75023d02u, 0xaf31c630u, 535, 195 }, // 75023d02af31c630 = b2sum -l 64 <<< amd-gfx120x/flash/attn_fwd/FONLY__＊fp32@16_32_F_T_3_0___gfx120x__P__2_2_32_32_True__CO__wave4_warp4_stg1--Arch_gfx1200
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::attn_fwd_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               attn_fwd_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx1200_mod0
{{7,0,1,3,1,3,0,1,2,2},
 {4,4,4,7,4,4,4,7,7,4},
 {4,7,7,5,7,4,7,4,7,4},
 {6,5,8,8,8,8,8,8,8,8},
 {4,8,8,8,8,8,8,8,8,8},
 {5,8,8,8,8,8,8,8,8,6},
 {6,8,8,8,8,8,8,8,8,6},
 {4,8,8,8,8,8,8,8,8,8},
 {0,8,8,8,8,8,8,8,8,8},
 {3,8,8,8,8,8,8,8,8,8}}
// End of GPU gfx1200_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::AttnFwdContext;

void CURRENT_ENTRY_PUBLIC(AttnFwdContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_attn_fwd__A4__F406 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 5;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.PERSISTENT_TYPE = perf.PERSISTENT_TYPE;
    context.GRID_CU_MULTIP = perf.GRID_CU_MULTIP;
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
    context.PRE_LOAD_V = perf.PRE_LOAD_V;
        return ;
    }
#endif
    auto kernel_index = attn_fwd__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 5;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.PERSISTENT_TYPE = perf.PERSISTENT_TYPE;
    context.GRID_CU_MULTIP = perf.GRID_CU_MULTIP;
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
    context.PRE_LOAD_V = perf.PRE_LOAD_V;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*fp32:16"
// B = 0
// A = 0
// Sm_scale = "fp32"
// L = "*fp32:16"
// Q_descale = 0
// stride_bz = 0
// stride_az = 0
// Num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// BLOCK_DMODEL = 32
// Head_dim = 32
// PADDED_HEAD = False
// ENABLE_DROPOUT = True
// dropout_p = "fp32"
// philox_seed_ptr = "*u64"
// philox_offset1 = "*u64"
// philox_offset2 = "u64"
// RETURN_ENCODED_SOFTMAX = False
// encoded_softmax = 0
// CAUSAL_TYPE = 3
// Window_left = "i32"
// BIAS_TYPE = 0
// USE_ALIBI = False
// INT8 = False
// persistent_atomic_counter = "*i32"
// Num_CU = "i32"

// vim: set fileencoding=utf-8

