// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.attn_fwd.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_attn_fwd__A2__F321

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"PERSISTENT_TYPE": 0, "GRID_CU_MULTIP": 2, "BLOCK_M": 16, "BLOCK_N": 16, "PRE_LOAD_V": false})xyzw",
  R"xyzw({"PERSISTENT_TYPE": 0, "GRID_CU_MULTIP": 2, "BLOCK_M": 16, "BLOCK_N": 16, "PRE_LOAD_V": false})xyzw",
  R"xyzw({"PERSISTENT_TYPE": 0, "GRID_CU_MULTIP": 2, "BLOCK_M": 16, "BLOCK_N": 16, "PRE_LOAD_V": true})xyzw",
  R"xyzw({"PERSISTENT_TYPE": 0, "GRID_CU_MULTIP": 2, "BLOCK_M": 32, "BLOCK_N": 16, "PRE_LOAD_V": false})xyzw",
  R"xyzw({"PERSISTENT_TYPE": 0, "GRID_CU_MULTIP": 2, "BLOCK_M": 32, "BLOCK_N": 16, "PRE_LOAD_V": false})xyzw",
  R"xyzw({"PERSISTENT_TYPE": 0, "GRID_CU_MULTIP": 2, "BLOCK_M": 32, "BLOCK_N": 16, "PRE_LOAD_V": true})xyzw",
  R"xyzw({"PERSISTENT_TYPE": 0, "GRID_CU_MULTIP": 2, "BLOCK_M": 32, "BLOCK_N": 32, "PRE_LOAD_V": false})xyzw",
  R"xyzw({"PERSISTENT_TYPE": 0, "GRID_CU_MULTIP": 2, "BLOCK_M": 32, "BLOCK_N": 32, "PRE_LOAD_V": false})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
    int8_t  PERSISTENT_TYPE;
    int8_t  GRID_CU_MULTIP;
    bool    PRE_LOAD_V;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16, .PERSISTENT_TYPE = 0, .GRID_CU_MULTIP = 2, .PRE_LOAD_V = false },
    { .BLOCK_M = 16, .BLOCK_N = 16, .PERSISTENT_TYPE = 0, .GRID_CU_MULTIP = 2, .PRE_LOAD_V = false },
    { .BLOCK_M = 16, .BLOCK_N = 16, .PERSISTENT_TYPE = 0, .GRID_CU_MULTIP = 2, .PRE_LOAD_V = true },
    { .BLOCK_M = 32, .BLOCK_N = 16, .PERSISTENT_TYPE = 0, .GRID_CU_MULTIP = 2, .PRE_LOAD_V = false },
    { .BLOCK_M = 32, .BLOCK_N = 16, .PERSISTENT_TYPE = 0, .GRID_CU_MULTIP = 2, .PRE_LOAD_V = false },
    { .BLOCK_M = 32, .BLOCK_N = 16, .PERSISTENT_TYPE = 0, .GRID_CU_MULTIP = 2, .PRE_LOAD_V = true },
    { .BLOCK_M = 32, .BLOCK_N = 32, .PERSISTENT_TYPE = 0, .GRID_CU_MULTIP = 2, .PRE_LOAD_V = false },
    { .BLOCK_M = 32, .BLOCK_N = 32, .PERSISTENT_TYPE = 0, .GRID_CU_MULTIP = 2, .PRE_LOAD_V = false }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr std::string_view PACKAGE_PATH { R"xyzw(amd-gfx11xx/flash/attn_fwd/FONLY__＊bf16@16_192_F_F_0_1___gfx11xx)xyzw" };
constexpr std::string_view FUNC_NAME { R"xyzw(＊bf16@16_192_F_F_0_1)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx1150)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0x47540e06u, 0xecdea007u, 458, 65 }, // 47540e06ecdea007 = b2sum -l 64 <<< amd-gfx11xx/flash/attn_fwd/FONLY__＊bf16@16_192_F_F_0_1___gfx11xx__P__0_2_16_16_False__CO__wave2_warp2_stg1--Arch_gfx1150
    { 0x4af96d56u, 0xfbbef187u, 458, 82 }, // 4af96d56fbbef187 = b2sum -l 64 <<< amd-gfx11xx/flash/attn_fwd/FONLY__＊bf16@16_192_F_F_0_1___gfx11xx__P__0_2_16_16_False__CO__wave3_warp2_stg1--Arch_gfx1150
    { 0xf5ba1c3cu, 0xe0537b2au, 474, 16 }, // f5ba1c3ce0537b2a = b2sum -l 64 <<< amd-gfx11xx/flash/attn_fwd/FONLY__＊bf16@16_192_F_F_0_1___gfx11xx__P__0_2_16_16_True__CO__wave4_warp2_stg1--Arch_gfx1150
    { 0xdf925497u, 0x5107bb7du, 0, 48 }, // df9254975107bb7d = b2sum -l 64 <<< amd-gfx11xx/flash/attn_fwd/FONLY__＊bf16@16_192_F_F_0_1___gfx11xx__P__0_2_32_16_False__CO__wave1_warp2_stg1--Arch_gfx1150
    { 0xee1dbec9u, 0xaadb9319u, 0, 195 }, // ee1dbec9aadb9319 = b2sum -l 64 <<< amd-gfx11xx/flash/attn_fwd/FONLY__＊bf16@16_192_F_F_0_1___gfx11xx__P__0_2_32_16_False__CO__wave4_warp4_stg1--Arch_gfx1150
    { 0x4584993fu, 0xc90be804u, 33, 48 }, // 4584993fc90be804 = b2sum -l 64 <<< amd-gfx11xx/flash/attn_fwd/FONLY__＊bf16@16_192_F_F_0_1___gfx11xx__P__0_2_32_16_True__CO__wave1_warp2_stg1--Arch_gfx1150
    { 0x8e686adau, 0x0df0d9feu, 489, 263 }, // 8e686ada0df0d9fe = b2sum -l 64 <<< amd-gfx11xx/flash/attn_fwd/FONLY__＊bf16@16_192_F_F_0_1___gfx11xx__P__0_2_32_32_False__CO__wave1_warp4_stg1--Arch_gfx1150
    { 0xd71b9c55u, 0x8a3c5c99u, 489, 178 }, // d71b9c558a3c5c99 = b2sum -l 64 <<< amd-gfx11xx/flash/attn_fwd/FONLY__＊bf16@16_192_F_F_0_1___gfx11xx__P__0_2_32_32_False__CO__wave2_warp4_stg1--Arch_gfx1150
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::attn_fwd_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               attn_fwd_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx1150_mod0
{{4,6,6,6,6,0,0,0,0,0},
 {6,4,4,4,4,0,0,0,0,0},
 {7,4,4,3,5,5,5,3,3,3},
 {7,4,4,4,5,3,3,5,5,2},
 {7,4,4,3,3,3,3,3,0,1},
 {7,4,4,3,3,3,3,0,0,0},
 {7,4,4,3,3,5,0,0,0,0},
 {7,4,4,3,3,3,3,3,5,3},
 {7,4,4,3,5,3,3,3,3,5},
 {7,4,4,3,3,3,5,5,5,3}}
// End of GPU gfx1150_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::AttnFwdContext;

void CURRENT_ENTRY_PUBLIC(AttnFwdContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_attn_fwd__A2__F321 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 1;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.PERSISTENT_TYPE = perf.PERSISTENT_TYPE;
    context.GRID_CU_MULTIP = perf.GRID_CU_MULTIP;
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
    context.PRE_LOAD_V = perf.PRE_LOAD_V;
        return ;
    }
#endif
    auto kernel_index = attn_fwd__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 1;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.PERSISTENT_TYPE = perf.PERSISTENT_TYPE;
    context.GRID_CU_MULTIP = perf.GRID_CU_MULTIP;
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
    context.PRE_LOAD_V = perf.PRE_LOAD_V;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*bf16:16"
// B = "*bf16:16"
// A = 0
// Sm_scale = "fp32"
// L = "*fp32:16"
// Q_descale = 0
// stride_az = 0
// Num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// BLOCK_DMODEL = 192
// Head_dim = 192
// PADDED_HEAD = False
// ENABLE_DROPOUT = False
// dropout_p = 0
// philox_seed_ptr = 0
// philox_offset1 = 0
// philox_offset2 = 0
// RETURN_ENCODED_SOFTMAX = False
// encoded_softmax = 0
// CAUSAL_TYPE = 0
// Window_left = 0
// BIAS_TYPE = 1
// USE_ALIBI = False
// INT8 = False
// persistent_atomic_counter = 0
// Num_CU = "i32"

// vim: set fileencoding=utf-8

