{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- A @PangoTabArray@ contains an array of tab stops.
-- 
-- @PangoTabArray@ can be used to set tab stops in a @PangoLayout@.
-- Each tab stop has an alignment, a position, and optionally
-- a character to use as decimal point.

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Pango.Structs.TabArray
    ( 

-- * Exported types
    TabArray(..)                            ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [copy]("GI.Pango.Structs.TabArray#g:method:copy"), [free]("GI.Pango.Structs.TabArray#g:method:free"), [resize]("GI.Pango.Structs.TabArray#g:method:resize"), [sort]("GI.Pango.Structs.TabArray#g:method:sort"), [toString]("GI.Pango.Structs.TabArray#g:method:toString").
-- 
-- ==== Getters
-- [getDecimalPoint]("GI.Pango.Structs.TabArray#g:method:getDecimalPoint"), [getPositionsInPixels]("GI.Pango.Structs.TabArray#g:method:getPositionsInPixels"), [getSize]("GI.Pango.Structs.TabArray#g:method:getSize"), [getTab]("GI.Pango.Structs.TabArray#g:method:getTab").
-- 
-- ==== Setters
-- [setDecimalPoint]("GI.Pango.Structs.TabArray#g:method:setDecimalPoint"), [setPositionsInPixels]("GI.Pango.Structs.TabArray#g:method:setPositionsInPixels"), [setTab]("GI.Pango.Structs.TabArray#g:method:setTab").

#if defined(ENABLE_OVERLOADING)
    ResolveTabArrayMethod                   ,
#endif

-- ** copy #method:copy#

#if defined(ENABLE_OVERLOADING)
    TabArrayCopyMethodInfo                  ,
#endif
    tabArrayCopy                            ,


-- ** free #method:free#

#if defined(ENABLE_OVERLOADING)
    TabArrayFreeMethodInfo                  ,
#endif
    tabArrayFree                            ,


-- ** fromString #method:fromString#

    tabArrayFromString                      ,


-- ** getDecimalPoint #method:getDecimalPoint#

#if defined(ENABLE_OVERLOADING)
    TabArrayGetDecimalPointMethodInfo       ,
#endif
    tabArrayGetDecimalPoint                 ,


-- ** getPositionsInPixels #method:getPositionsInPixels#

#if defined(ENABLE_OVERLOADING)
    TabArrayGetPositionsInPixelsMethodInfo  ,
#endif
    tabArrayGetPositionsInPixels            ,


-- ** getSize #method:getSize#

#if defined(ENABLE_OVERLOADING)
    TabArrayGetSizeMethodInfo               ,
#endif
    tabArrayGetSize                         ,


-- ** getTab #method:getTab#

#if defined(ENABLE_OVERLOADING)
    TabArrayGetTabMethodInfo                ,
#endif
    tabArrayGetTab                          ,


-- ** new #method:new#

    tabArrayNew                             ,


-- ** resize #method:resize#

#if defined(ENABLE_OVERLOADING)
    TabArrayResizeMethodInfo                ,
#endif
    tabArrayResize                          ,


-- ** setDecimalPoint #method:setDecimalPoint#

#if defined(ENABLE_OVERLOADING)
    TabArraySetDecimalPointMethodInfo       ,
#endif
    tabArraySetDecimalPoint                 ,


-- ** setPositionsInPixels #method:setPositionsInPixels#

#if defined(ENABLE_OVERLOADING)
    TabArraySetPositionsInPixelsMethodInfo  ,
#endif
    tabArraySetPositionsInPixels            ,


-- ** setTab #method:setTab#

#if defined(ENABLE_OVERLOADING)
    TabArraySetTabMethodInfo                ,
#endif
    tabArraySetTab                          ,


-- ** sort #method:sort#

#if defined(ENABLE_OVERLOADING)
    TabArraySortMethodInfo                  ,
#endif
    tabArraySort                            ,


-- ** toString #method:toString#

#if defined(ENABLE_OVERLOADING)
    TabArrayToStringMethodInfo              ,
#endif
    tabArrayToString                        ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import {-# SOURCE #-} qualified GI.Pango.Enums as Pango.Enums

#else
import {-# SOURCE #-} qualified GI.Pango.Enums as Pango.Enums

#endif

-- | Memory-managed wrapper type.
newtype TabArray = TabArray (SP.ManagedPtr TabArray)
    deriving (Eq)

instance SP.ManagedPtrNewtype TabArray where
    toManagedPtr (TabArray p) = p

foreign import ccall "pango_tab_array_get_type" c_pango_tab_array_get_type :: 
    IO GType

type instance O.ParentTypes TabArray = '[]
instance O.HasParentTypes TabArray

instance B.Types.TypedObject TabArray where
    glibType = c_pango_tab_array_get_type

instance B.Types.GBoxed TabArray

-- | Convert t'TabArray' to and from 'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe TabArray) where
    gvalueGType_ = c_pango_tab_array_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_boxed gv (FP.nullPtr :: FP.Ptr TabArray)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_boxed gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_boxed gv :: IO (Ptr TabArray)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newBoxed TabArray ptr
        else return P.Nothing
        
    


#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList TabArray
type instance O.AttributeList TabArray = TabArrayAttributeList
type TabArrayAttributeList = ('[ ] :: [(Symbol, DK.Type)])
#endif

-- method TabArray::new
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "initial_size"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "Initial number of tab stops to allocate, can be 0"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "positions_in_pixels"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "whether positions are in pixel units"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "TabArray" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_tab_array_new" pango_tab_array_new :: 
    Int32 ->                                -- initial_size : TBasicType TInt
    CInt ->                                 -- positions_in_pixels : TBasicType TBoolean
    IO (Ptr TabArray)

-- | Creates an array of /@initialSize@/ tab stops.
-- 
-- Tab stops are specified in pixel units if /@positionsInPixels@/ is 'P.True',
-- otherwise in Pango units. All stops are initially at position 0.
tabArrayNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Int32
    -- ^ /@initialSize@/: Initial number of tab stops to allocate, can be 0
    -> Bool
    -- ^ /@positionsInPixels@/: whether positions are in pixel units
    -> m TabArray
    -- ^ __Returns:__ the newly allocated @PangoTabArray@, which should
    --   be freed with 'GI.Pango.Structs.TabArray.tabArrayFree'.
tabArrayNew initialSize positionsInPixels = liftIO $ do
    let positionsInPixels' = (P.fromIntegral . P.fromEnum) positionsInPixels
    result <- pango_tab_array_new initialSize positionsInPixels'
    checkUnexpectedReturnNULL "tabArrayNew" result
    result' <- (wrapBoxed TabArray) result
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method TabArray::copy
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "src"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "TabArray" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "`PangoTabArray` to copy"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "TabArray" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_tab_array_copy" pango_tab_array_copy :: 
    Ptr TabArray ->                         -- src : TInterface (Name {namespace = "Pango", name = "TabArray"})
    IO (Ptr TabArray)

-- | Copies a @PangoTabArray@.
tabArrayCopy ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    TabArray
    -- ^ /@src@/: @PangoTabArray@ to copy
    -> m TabArray
    -- ^ __Returns:__ the newly allocated @PangoTabArray@, which should
    --   be freed with 'GI.Pango.Structs.TabArray.tabArrayFree'.
tabArrayCopy src = liftIO $ do
    src' <- unsafeManagedPtrGetPtr src
    result <- pango_tab_array_copy src'
    checkUnexpectedReturnNULL "tabArrayCopy" result
    result' <- (wrapBoxed TabArray) result
    touchManagedPtr src
    return result'

#if defined(ENABLE_OVERLOADING)
data TabArrayCopyMethodInfo
instance (signature ~ (m TabArray), MonadIO m) => O.OverloadedMethod TabArrayCopyMethodInfo TabArray signature where
    overloadedMethod = tabArrayCopy

instance O.OverloadedMethodInfo TabArrayCopyMethodInfo TabArray where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Pango.Structs.TabArray.tabArrayCopy",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-pango-1.0.30/docs/GI-Pango-Structs-TabArray.html#v:tabArrayCopy"
        })


#endif

-- method TabArray::free
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "tab_array"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "TabArray" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `PangoTabArray`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "pango_tab_array_free" pango_tab_array_free :: 
    Ptr TabArray ->                         -- tab_array : TInterface (Name {namespace = "Pango", name = "TabArray"})
    IO ()

-- | Frees a tab array and associated resources.
tabArrayFree ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    TabArray
    -- ^ /@tabArray@/: a @PangoTabArray@
    -> m ()
tabArrayFree tabArray = liftIO $ do
    tabArray' <- unsafeManagedPtrGetPtr tabArray
    pango_tab_array_free tabArray'
    touchManagedPtr tabArray
    return ()

#if defined(ENABLE_OVERLOADING)
data TabArrayFreeMethodInfo
instance (signature ~ (m ()), MonadIO m) => O.OverloadedMethod TabArrayFreeMethodInfo TabArray signature where
    overloadedMethod = tabArrayFree

instance O.OverloadedMethodInfo TabArrayFreeMethodInfo TabArray where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Pango.Structs.TabArray.tabArrayFree",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-pango-1.0.30/docs/GI-Pango-Structs-TabArray.html#v:tabArrayFree"
        })


#endif

-- method TabArray::get_decimal_point
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "tab_array"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "TabArray" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `PangoTabArray`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "tab_index"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the index of a tab stop"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUniChar)
-- throws : False
-- Skip return : False

foreign import ccall "pango_tab_array_get_decimal_point" pango_tab_array_get_decimal_point :: 
    Ptr TabArray ->                         -- tab_array : TInterface (Name {namespace = "Pango", name = "TabArray"})
    Int32 ->                                -- tab_index : TBasicType TInt
    IO CInt

-- | Gets the Unicode character to use as decimal point.
-- 
-- This is only relevant for tabs with 'GI.Pango.Enums.TabAlignDecimal' alignment,
-- which align content at the first occurrence of the decimal point
-- character.
-- 
-- The default value of 0 means that Pango will use the
-- decimal point according to the current locale.
-- 
-- /Since: 1.50/
tabArrayGetDecimalPoint ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    TabArray
    -- ^ /@tabArray@/: a @PangoTabArray@
    -> Int32
    -- ^ /@tabIndex@/: the index of a tab stop
    -> m Char
tabArrayGetDecimalPoint tabArray tabIndex = liftIO $ do
    tabArray' <- unsafeManagedPtrGetPtr tabArray
    result <- pango_tab_array_get_decimal_point tabArray' tabIndex
    let result' = (chr . fromIntegral) result
    touchManagedPtr tabArray
    return result'

#if defined(ENABLE_OVERLOADING)
data TabArrayGetDecimalPointMethodInfo
instance (signature ~ (Int32 -> m Char), MonadIO m) => O.OverloadedMethod TabArrayGetDecimalPointMethodInfo TabArray signature where
    overloadedMethod = tabArrayGetDecimalPoint

instance O.OverloadedMethodInfo TabArrayGetDecimalPointMethodInfo TabArray where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Pango.Structs.TabArray.tabArrayGetDecimalPoint",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-pango-1.0.30/docs/GI-Pango-Structs-TabArray.html#v:tabArrayGetDecimalPoint"
        })


#endif

-- method TabArray::get_positions_in_pixels
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "tab_array"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "TabArray" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `PangoTabArray`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "pango_tab_array_get_positions_in_pixels" pango_tab_array_get_positions_in_pixels :: 
    Ptr TabArray ->                         -- tab_array : TInterface (Name {namespace = "Pango", name = "TabArray"})
    IO CInt

-- | Returns 'P.True' if the tab positions are in pixels,
-- 'P.False' if they are in Pango units.
tabArrayGetPositionsInPixels ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    TabArray
    -- ^ /@tabArray@/: a @PangoTabArray@
    -> m Bool
    -- ^ __Returns:__ whether positions are in pixels.
tabArrayGetPositionsInPixels tabArray = liftIO $ do
    tabArray' <- unsafeManagedPtrGetPtr tabArray
    result <- pango_tab_array_get_positions_in_pixels tabArray'
    let result' = (/= 0) result
    touchManagedPtr tabArray
    return result'

#if defined(ENABLE_OVERLOADING)
data TabArrayGetPositionsInPixelsMethodInfo
instance (signature ~ (m Bool), MonadIO m) => O.OverloadedMethod TabArrayGetPositionsInPixelsMethodInfo TabArray signature where
    overloadedMethod = tabArrayGetPositionsInPixels

instance O.OverloadedMethodInfo TabArrayGetPositionsInPixelsMethodInfo TabArray where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Pango.Structs.TabArray.tabArrayGetPositionsInPixels",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-pango-1.0.30/docs/GI-Pango-Structs-TabArray.html#v:tabArrayGetPositionsInPixels"
        })


#endif

-- method TabArray::get_size
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "tab_array"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "TabArray" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `PangoTabArray`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "pango_tab_array_get_size" pango_tab_array_get_size :: 
    Ptr TabArray ->                         -- tab_array : TInterface (Name {namespace = "Pango", name = "TabArray"})
    IO Int32

-- | Gets the number of tab stops in /@tabArray@/.
tabArrayGetSize ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    TabArray
    -- ^ /@tabArray@/: a @PangoTabArray@
    -> m Int32
    -- ^ __Returns:__ the number of tab stops in the array.
tabArrayGetSize tabArray = liftIO $ do
    tabArray' <- unsafeManagedPtrGetPtr tabArray
    result <- pango_tab_array_get_size tabArray'
    touchManagedPtr tabArray
    return result

#if defined(ENABLE_OVERLOADING)
data TabArrayGetSizeMethodInfo
instance (signature ~ (m Int32), MonadIO m) => O.OverloadedMethod TabArrayGetSizeMethodInfo TabArray signature where
    overloadedMethod = tabArrayGetSize

instance O.OverloadedMethodInfo TabArrayGetSizeMethodInfo TabArray where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Pango.Structs.TabArray.tabArrayGetSize",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-pango-1.0.30/docs/GI-Pango-Structs-TabArray.html#v:tabArrayGetSize"
        })


#endif

-- method TabArray::get_tab
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "tab_array"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "TabArray" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `PangoTabArray`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "tab_index"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "tab stop index" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "alignment"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "TabAlign" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "location to store alignment"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "location"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "location to store tab position"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "pango_tab_array_get_tab" pango_tab_array_get_tab :: 
    Ptr TabArray ->                         -- tab_array : TInterface (Name {namespace = "Pango", name = "TabArray"})
    Int32 ->                                -- tab_index : TBasicType TInt
    Ptr CUInt ->                            -- alignment : TInterface (Name {namespace = "Pango", name = "TabAlign"})
    Ptr Int32 ->                            -- location : TBasicType TInt
    IO ()

-- | Gets the alignment and position of a tab stop.
tabArrayGetTab ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    TabArray
    -- ^ /@tabArray@/: a @PangoTabArray@
    -> Int32
    -- ^ /@tabIndex@/: tab stop index
    -> m ((Pango.Enums.TabAlign, Int32))
tabArrayGetTab tabArray tabIndex = liftIO $ do
    tabArray' <- unsafeManagedPtrGetPtr tabArray
    alignment <- allocMem :: IO (Ptr CUInt)
    location <- allocMem :: IO (Ptr Int32)
    pango_tab_array_get_tab tabArray' tabIndex alignment location
    alignment' <- peek alignment
    let alignment'' = (toEnum . fromIntegral) alignment'
    location' <- peek location
    touchManagedPtr tabArray
    freeMem alignment
    freeMem location
    return (alignment'', location')

#if defined(ENABLE_OVERLOADING)
data TabArrayGetTabMethodInfo
instance (signature ~ (Int32 -> m ((Pango.Enums.TabAlign, Int32))), MonadIO m) => O.OverloadedMethod TabArrayGetTabMethodInfo TabArray signature where
    overloadedMethod = tabArrayGetTab

instance O.OverloadedMethodInfo TabArrayGetTabMethodInfo TabArray where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Pango.Structs.TabArray.tabArrayGetTab",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-pango-1.0.30/docs/GI-Pango-Structs-TabArray.html#v:tabArrayGetTab"
        })


#endif

-- XXX Could not generate method TabArray::get_tabs
{-  Bad introspection data: `TCArray False (-1) (-1) (TBasicType TInt)' is an array type, but contains no length information,
    so it cannot be unpacked.
-}
-- method TabArray::resize
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "tab_array"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "TabArray" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `PangoTabArray`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "new_size"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "new size of the array"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "pango_tab_array_resize" pango_tab_array_resize :: 
    Ptr TabArray ->                         -- tab_array : TInterface (Name {namespace = "Pango", name = "TabArray"})
    Int32 ->                                -- new_size : TBasicType TInt
    IO ()

-- | Resizes a tab array.
-- 
-- You must subsequently initialize any tabs
-- that were added as a result of growing the array.
tabArrayResize ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    TabArray
    -- ^ /@tabArray@/: a @PangoTabArray@
    -> Int32
    -- ^ /@newSize@/: new size of the array
    -> m ()
tabArrayResize tabArray newSize = liftIO $ do
    tabArray' <- unsafeManagedPtrGetPtr tabArray
    pango_tab_array_resize tabArray' newSize
    touchManagedPtr tabArray
    return ()

#if defined(ENABLE_OVERLOADING)
data TabArrayResizeMethodInfo
instance (signature ~ (Int32 -> m ()), MonadIO m) => O.OverloadedMethod TabArrayResizeMethodInfo TabArray signature where
    overloadedMethod = tabArrayResize

instance O.OverloadedMethodInfo TabArrayResizeMethodInfo TabArray where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Pango.Structs.TabArray.tabArrayResize",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-pango-1.0.30/docs/GI-Pango-Structs-TabArray.html#v:tabArrayResize"
        })


#endif

-- method TabArray::set_decimal_point
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "tab_array"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "TabArray" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `PangoTabArray`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "tab_index"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the index of a tab stop"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "decimal_point"
--           , argType = TBasicType TUniChar
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the decimal point to use"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "pango_tab_array_set_decimal_point" pango_tab_array_set_decimal_point :: 
    Ptr TabArray ->                         -- tab_array : TInterface (Name {namespace = "Pango", name = "TabArray"})
    Int32 ->                                -- tab_index : TBasicType TInt
    CInt ->                                 -- decimal_point : TBasicType TUniChar
    IO ()

-- | Sets the Unicode character to use as decimal point.
-- 
-- This is only relevant for tabs with 'GI.Pango.Enums.TabAlignDecimal' alignment,
-- which align content at the first occurrence of the decimal point
-- character.
-- 
-- By default, Pango uses the decimal point according
-- to the current locale.
-- 
-- /Since: 1.50/
tabArraySetDecimalPoint ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    TabArray
    -- ^ /@tabArray@/: a @PangoTabArray@
    -> Int32
    -- ^ /@tabIndex@/: the index of a tab stop
    -> Char
    -- ^ /@decimalPoint@/: the decimal point to use
    -> m ()
tabArraySetDecimalPoint tabArray tabIndex decimalPoint = liftIO $ do
    tabArray' <- unsafeManagedPtrGetPtr tabArray
    let decimalPoint' = (P.fromIntegral . SP.ord) decimalPoint
    pango_tab_array_set_decimal_point tabArray' tabIndex decimalPoint'
    touchManagedPtr tabArray
    return ()

#if defined(ENABLE_OVERLOADING)
data TabArraySetDecimalPointMethodInfo
instance (signature ~ (Int32 -> Char -> m ()), MonadIO m) => O.OverloadedMethod TabArraySetDecimalPointMethodInfo TabArray signature where
    overloadedMethod = tabArraySetDecimalPoint

instance O.OverloadedMethodInfo TabArraySetDecimalPointMethodInfo TabArray where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Pango.Structs.TabArray.tabArraySetDecimalPoint",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-pango-1.0.30/docs/GI-Pango-Structs-TabArray.html#v:tabArraySetDecimalPoint"
        })


#endif

-- method TabArray::set_positions_in_pixels
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "tab_array"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "TabArray" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `PangoTabArray`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "positions_in_pixels"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "whether positions are in pixels"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "pango_tab_array_set_positions_in_pixels" pango_tab_array_set_positions_in_pixels :: 
    Ptr TabArray ->                         -- tab_array : TInterface (Name {namespace = "Pango", name = "TabArray"})
    CInt ->                                 -- positions_in_pixels : TBasicType TBoolean
    IO ()

-- | Sets whether positions in this array are specified in
-- pixels.
-- 
-- /Since: 1.50/
tabArraySetPositionsInPixels ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    TabArray
    -- ^ /@tabArray@/: a @PangoTabArray@
    -> Bool
    -- ^ /@positionsInPixels@/: whether positions are in pixels
    -> m ()
tabArraySetPositionsInPixels tabArray positionsInPixels = liftIO $ do
    tabArray' <- unsafeManagedPtrGetPtr tabArray
    let positionsInPixels' = (P.fromIntegral . P.fromEnum) positionsInPixels
    pango_tab_array_set_positions_in_pixels tabArray' positionsInPixels'
    touchManagedPtr tabArray
    return ()

#if defined(ENABLE_OVERLOADING)
data TabArraySetPositionsInPixelsMethodInfo
instance (signature ~ (Bool -> m ()), MonadIO m) => O.OverloadedMethod TabArraySetPositionsInPixelsMethodInfo TabArray signature where
    overloadedMethod = tabArraySetPositionsInPixels

instance O.OverloadedMethodInfo TabArraySetPositionsInPixelsMethodInfo TabArray where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Pango.Structs.TabArray.tabArraySetPositionsInPixels",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-pango-1.0.30/docs/GI-Pango-Structs-TabArray.html#v:tabArraySetPositionsInPixels"
        })


#endif

-- method TabArray::set_tab
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "tab_array"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "TabArray" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `PangoTabArray`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "tab_index"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the index of a tab stop"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "alignment"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "TabAlign" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "tab alignment" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "location"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "tab location in Pango units"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "pango_tab_array_set_tab" pango_tab_array_set_tab :: 
    Ptr TabArray ->                         -- tab_array : TInterface (Name {namespace = "Pango", name = "TabArray"})
    Int32 ->                                -- tab_index : TBasicType TInt
    CUInt ->                                -- alignment : TInterface (Name {namespace = "Pango", name = "TabAlign"})
    Int32 ->                                -- location : TBasicType TInt
    IO ()

-- | Sets the alignment and location of a tab stop.
tabArraySetTab ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    TabArray
    -- ^ /@tabArray@/: a @PangoTabArray@
    -> Int32
    -- ^ /@tabIndex@/: the index of a tab stop
    -> Pango.Enums.TabAlign
    -- ^ /@alignment@/: tab alignment
    -> Int32
    -- ^ /@location@/: tab location in Pango units
    -> m ()
tabArraySetTab tabArray tabIndex alignment location = liftIO $ do
    tabArray' <- unsafeManagedPtrGetPtr tabArray
    let alignment' = (fromIntegral . fromEnum) alignment
    pango_tab_array_set_tab tabArray' tabIndex alignment' location
    touchManagedPtr tabArray
    return ()

#if defined(ENABLE_OVERLOADING)
data TabArraySetTabMethodInfo
instance (signature ~ (Int32 -> Pango.Enums.TabAlign -> Int32 -> m ()), MonadIO m) => O.OverloadedMethod TabArraySetTabMethodInfo TabArray signature where
    overloadedMethod = tabArraySetTab

instance O.OverloadedMethodInfo TabArraySetTabMethodInfo TabArray where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Pango.Structs.TabArray.tabArraySetTab",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-pango-1.0.30/docs/GI-Pango-Structs-TabArray.html#v:tabArraySetTab"
        })


#endif

-- method TabArray::sort
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "tab_array"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "TabArray" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `PangoTabArray`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "pango_tab_array_sort" pango_tab_array_sort :: 
    Ptr TabArray ->                         -- tab_array : TInterface (Name {namespace = "Pango", name = "TabArray"})
    IO ()

-- | Utility function to ensure that the tab stops are in increasing order.
-- 
-- /Since: 1.50/
tabArraySort ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    TabArray
    -- ^ /@tabArray@/: a @PangoTabArray@
    -> m ()
tabArraySort tabArray = liftIO $ do
    tabArray' <- unsafeManagedPtrGetPtr tabArray
    pango_tab_array_sort tabArray'
    touchManagedPtr tabArray
    return ()

#if defined(ENABLE_OVERLOADING)
data TabArraySortMethodInfo
instance (signature ~ (m ()), MonadIO m) => O.OverloadedMethod TabArraySortMethodInfo TabArray signature where
    overloadedMethod = tabArraySort

instance O.OverloadedMethodInfo TabArraySortMethodInfo TabArray where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Pango.Structs.TabArray.tabArraySort",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-pango-1.0.30/docs/GI-Pango-Structs-TabArray.html#v:tabArraySort"
        })


#endif

-- method TabArray::to_string
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "tab_array"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "TabArray" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `PangoTabArray`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "pango_tab_array_to_string" pango_tab_array_to_string :: 
    Ptr TabArray ->                         -- tab_array : TInterface (Name {namespace = "Pango", name = "TabArray"})
    IO CString

-- | Serializes a @PangoTabArray@ to a string.
-- 
-- In the resulting string, serialized tabs are separated by newlines or commas.
-- 
-- Individual tabs are serialized to a string of the form
-- 
--     [ALIGNMENT:]POSITION[:DECIMAL_POINT]
-- 
-- Where ALIGNMENT is one of _left_, _right_, _center_ or _decimal_, and
-- POSITION is the position of the tab, optionally followed by the unit _px_.
-- If ALIGNMENT is omitted, it defaults to _left_. If ALIGNMENT is _decimal_,
-- the DECIMAL_POINT character may be specified as a Unicode codepoint.
-- 
-- Note that all tabs in the array must use the same unit.
-- 
-- A typical example:
-- 
--     100px 200px center:300px right:400px
-- 
-- /Since: 1.50/
tabArrayToString ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    TabArray
    -- ^ /@tabArray@/: a @PangoTabArray@
    -> m T.Text
    -- ^ __Returns:__ a newly allocated string
tabArrayToString tabArray = liftIO $ do
    tabArray' <- unsafeManagedPtrGetPtr tabArray
    result <- pango_tab_array_to_string tabArray'
    checkUnexpectedReturnNULL "tabArrayToString" result
    result' <- cstringToText result
    freeMem result
    touchManagedPtr tabArray
    return result'

#if defined(ENABLE_OVERLOADING)
data TabArrayToStringMethodInfo
instance (signature ~ (m T.Text), MonadIO m) => O.OverloadedMethod TabArrayToStringMethodInfo TabArray signature where
    overloadedMethod = tabArrayToString

instance O.OverloadedMethodInfo TabArrayToStringMethodInfo TabArray where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Pango.Structs.TabArray.tabArrayToString",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-pango-1.0.30/docs/GI-Pango-Structs-TabArray.html#v:tabArrayToString"
        })


#endif

-- method TabArray::from_string
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "text"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a string" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "TabArray" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_tab_array_from_string" pango_tab_array_from_string :: 
    CString ->                              -- text : TBasicType TUTF8
    IO (Ptr TabArray)

-- | Deserializes a @PangoTabArray@ from a string.
-- 
-- This is the counterpart to 'GI.Pango.Structs.TabArray.tabArrayToString'.
-- See that functions for details about the format.
-- 
-- /Since: 1.50/
tabArrayFromString ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@text@/: a string
    -> m (Maybe TabArray)
    -- ^ __Returns:__ a new @PangoTabArray@
tabArrayFromString text = liftIO $ do
    text' <- textToCString text
    result <- pango_tab_array_from_string text'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed TabArray) result'
        return result''
    freeMem text'
    return maybeResult

#if defined(ENABLE_OVERLOADING)
#endif

#if defined(ENABLE_OVERLOADING)
type family ResolveTabArrayMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveTabArrayMethod "copy" o = TabArrayCopyMethodInfo
    ResolveTabArrayMethod "free" o = TabArrayFreeMethodInfo
    ResolveTabArrayMethod "resize" o = TabArrayResizeMethodInfo
    ResolveTabArrayMethod "sort" o = TabArraySortMethodInfo
    ResolveTabArrayMethod "toString" o = TabArrayToStringMethodInfo
    ResolveTabArrayMethod "getDecimalPoint" o = TabArrayGetDecimalPointMethodInfo
    ResolveTabArrayMethod "getPositionsInPixels" o = TabArrayGetPositionsInPixelsMethodInfo
    ResolveTabArrayMethod "getSize" o = TabArrayGetSizeMethodInfo
    ResolveTabArrayMethod "getTab" o = TabArrayGetTabMethodInfo
    ResolveTabArrayMethod "setDecimalPoint" o = TabArraySetDecimalPointMethodInfo
    ResolveTabArrayMethod "setPositionsInPixels" o = TabArraySetPositionsInPixelsMethodInfo
    ResolveTabArrayMethod "setTab" o = TabArraySetTabMethodInfo
    ResolveTabArrayMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveTabArrayMethod t TabArray, O.OverloadedMethod info TabArray p) => OL.IsLabel t (TabArray -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveTabArrayMethod t TabArray, O.OverloadedMethod info TabArray p, R.HasField t TabArray p) => R.HasField t TabArray p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveTabArrayMethod t TabArray, O.OverloadedMethodInfo info TabArray) => OL.IsLabel t (O.MethodProxy info TabArray) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif


