{-# LANGUAGE ImplicitParams, RankNTypes, TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- An interface for content that can be painted.
-- 
-- The content of a @GdkPaintable@ can be painted anywhere at any size
-- without requiring any sort of layout. The interface is inspired by
-- similar concepts elsewhere, such as
-- <https://developer.gnome.org/clutter/stable/ClutterContent.html ClutterContent>,
-- <https://www.w3.org/TR/css-images-4/#paint-source HTML/CSS Paint Sources>,
-- or <https://www.w3.org/TR/SVG2/pservers.html SVG Paint Servers>.
-- 
-- A @GdkPaintable@ can be snapshot at any time and size using
-- 'GI.Gdk.Interfaces.Paintable.paintableSnapshot'. How the paintable interprets that size and
-- if it scales or centers itself into the given rectangle is implementation
-- defined, though if you are implementing a @GdkPaintable@ and don\'t know what
-- to do, it is suggested that you scale your paintable ignoring any potential
-- aspect ratio.
-- 
-- The contents that a @GdkPaintable@ produces may depend on the t'GI.Gdk.Objects.Snapshot.Snapshot'
-- passed to it. For example, paintables may decide to use more detailed images
-- on higher resolution screens or when OpenGL is available. A @GdkPaintable@
-- will however always produce the same output for the same snapshot.
-- 
-- A @GdkPaintable@ may change its contents, meaning that it will now produce
-- a different output with the same snapshot. Once that happens, it will call
-- 'GI.Gdk.Interfaces.Paintable.paintableInvalidateContents' which will emit the
-- [Paintable::invalidateContents]("GI.Gdk.Interfaces.Paintable#g:signal:invalidateContents") signal. If a paintable is known
-- to never change its contents, it will set the 'GI.Gdk.Flags.PaintableFlagsContents'
-- flag. If a consumer cannot deal with changing contents, it may call
-- 'GI.Gdk.Interfaces.Paintable.paintableGetCurrentImage' which will return a static
-- paintable and use that.
-- 
-- A paintable can report an intrinsic (or preferred) size or aspect ratio it
-- wishes to be rendered at, though it doesn\'t have to. Consumers of the interface
-- can use this information to layout thepaintable appropriately. Just like the
-- contents, the size of a paintable can change. A paintable will indicate this
-- by calling 'GI.Gdk.Interfaces.Paintable.paintableInvalidateSize' which will emit the
-- [Paintable::invalidateSize]("GI.Gdk.Interfaces.Paintable#g:signal:invalidateSize") signal. And just like for contents,
-- if a paintable is known to never change its size, it will set the
-- 'GI.Gdk.Flags.PaintableFlagsSize' flag.
-- 
-- Besides API for applications, there are some functions that are only
-- useful for implementing subclasses and should not be used by applications:
-- 'GI.Gdk.Interfaces.Paintable.paintableInvalidateContents',
-- 'GI.Gdk.Interfaces.Paintable.paintableInvalidateSize',
-- 'GI.Gdk.Interfaces.Paintable.paintableNewEmpty'.

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gdk.Interfaces.Paintable
    ( 

-- * Exported types
    Paintable(..)                           ,
    IsPaintable                             ,
    toPaintable                             ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [computeConcreteSize]("GI.Gdk.Interfaces.Paintable#g:method:computeConcreteSize"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [invalidateContents]("GI.Gdk.Interfaces.Paintable#g:method:invalidateContents"), [invalidateSize]("GI.Gdk.Interfaces.Paintable#g:method:invalidateSize"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [snapshot]("GI.Gdk.Interfaces.Paintable#g:method:snapshot"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getCurrentImage]("GI.Gdk.Interfaces.Paintable#g:method:getCurrentImage"), [getData]("GI.GObject.Objects.Object#g:method:getData"), [getFlags]("GI.Gdk.Interfaces.Paintable#g:method:getFlags"), [getIntrinsicAspectRatio]("GI.Gdk.Interfaces.Paintable#g:method:getIntrinsicAspectRatio"), [getIntrinsicHeight]("GI.Gdk.Interfaces.Paintable#g:method:getIntrinsicHeight"), [getIntrinsicWidth]("GI.Gdk.Interfaces.Paintable#g:method:getIntrinsicWidth"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata").
-- 
-- ==== Setters
-- [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty").

#if defined(ENABLE_OVERLOADING)
    ResolvePaintableMethod                  ,
#endif

-- ** computeConcreteSize #method:computeConcreteSize#

#if defined(ENABLE_OVERLOADING)
    PaintableComputeConcreteSizeMethodInfo  ,
#endif
    paintableComputeConcreteSize            ,


-- ** getCurrentImage #method:getCurrentImage#

#if defined(ENABLE_OVERLOADING)
    PaintableGetCurrentImageMethodInfo      ,
#endif
    paintableGetCurrentImage                ,


-- ** getFlags #method:getFlags#

#if defined(ENABLE_OVERLOADING)
    PaintableGetFlagsMethodInfo             ,
#endif
    paintableGetFlags                       ,


-- ** getIntrinsicAspectRatio #method:getIntrinsicAspectRatio#

#if defined(ENABLE_OVERLOADING)
    PaintableGetIntrinsicAspectRatioMethodInfo,
#endif
    paintableGetIntrinsicAspectRatio        ,


-- ** getIntrinsicHeight #method:getIntrinsicHeight#

#if defined(ENABLE_OVERLOADING)
    PaintableGetIntrinsicHeightMethodInfo   ,
#endif
    paintableGetIntrinsicHeight             ,


-- ** getIntrinsicWidth #method:getIntrinsicWidth#

#if defined(ENABLE_OVERLOADING)
    PaintableGetIntrinsicWidthMethodInfo    ,
#endif
    paintableGetIntrinsicWidth              ,


-- ** invalidateContents #method:invalidateContents#

#if defined(ENABLE_OVERLOADING)
    PaintableInvalidateContentsMethodInfo   ,
#endif
    paintableInvalidateContents             ,


-- ** invalidateSize #method:invalidateSize#

#if defined(ENABLE_OVERLOADING)
    PaintableInvalidateSizeMethodInfo       ,
#endif
    paintableInvalidateSize                 ,


-- ** newEmpty #method:newEmpty#

    paintableNewEmpty                       ,


-- ** snapshot #method:snapshot#

#if defined(ENABLE_OVERLOADING)
    PaintableSnapshotMethodInfo             ,
#endif
    paintableSnapshot                       ,




 -- * Signals


-- ** invalidateContents #signal:invalidateContents#

    PaintableInvalidateContentsCallback     ,
#if defined(ENABLE_OVERLOADING)
    PaintableInvalidateContentsSignalInfo   ,
#endif
    afterPaintableInvalidateContents        ,
    onPaintableInvalidateContents           ,


-- ** invalidateSize #signal:invalidateSize#

    PaintableInvalidateSizeCallback         ,
#if defined(ENABLE_OVERLOADING)
    PaintableInvalidateSizeSignalInfo       ,
#endif
    afterPaintableInvalidateSize            ,
    onPaintableInvalidateSize               ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.GObject.Objects.Object as GObject.Object
import {-# SOURCE #-} qualified GI.Gdk.Flags as Gdk.Flags
import {-# SOURCE #-} qualified GI.Gdk.Objects.Snapshot as Gdk.Snapshot

#else
import qualified GI.GObject.Objects.Object as GObject.Object
import {-# SOURCE #-} qualified GI.Gdk.Flags as Gdk.Flags
import {-# SOURCE #-} qualified GI.Gdk.Objects.Snapshot as Gdk.Snapshot

#endif

-- interface Paintable 
-- | Memory-managed wrapper type.
newtype Paintable = Paintable (SP.ManagedPtr Paintable)
    deriving (Eq)

instance SP.ManagedPtrNewtype Paintable where
    toManagedPtr (Paintable p) = p

foreign import ccall "gdk_paintable_get_type"
    c_gdk_paintable_get_type :: IO B.Types.GType

instance B.Types.TypedObject Paintable where
    glibType = c_gdk_paintable_get_type

instance B.Types.GObject Paintable

-- | Type class for types which can be safely cast to t'Paintable', for instance with `toPaintable`.
class (SP.GObject o, O.IsDescendantOf Paintable o) => IsPaintable o
instance (SP.GObject o, O.IsDescendantOf Paintable o) => IsPaintable o

instance O.HasParentTypes Paintable
type instance O.ParentTypes Paintable = '[GObject.Object.Object]

-- | Cast to t'Paintable', for types for which this is known to be safe. For general casts, use 'Data.GI.Base.ManagedPtr.castTo'.
toPaintable :: (MIO.MonadIO m, IsPaintable o) => o -> m Paintable
toPaintable = MIO.liftIO . B.ManagedPtr.unsafeCastTo Paintable

-- | Convert t'Paintable' to and from t'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe Paintable) where
    gvalueGType_ = c_gdk_paintable_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr Paintable)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr Paintable)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject Paintable ptr
        else return P.Nothing
        
    

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList Paintable
type instance O.AttributeList Paintable = PaintableAttributeList
type PaintableAttributeList = ('[ ] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
#endif

#if defined(ENABLE_OVERLOADING)
type family ResolvePaintableMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolvePaintableMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolvePaintableMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolvePaintableMethod "computeConcreteSize" o = PaintableComputeConcreteSizeMethodInfo
    ResolvePaintableMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolvePaintableMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolvePaintableMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolvePaintableMethod "invalidateContents" o = PaintableInvalidateContentsMethodInfo
    ResolvePaintableMethod "invalidateSize" o = PaintableInvalidateSizeMethodInfo
    ResolvePaintableMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolvePaintableMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolvePaintableMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolvePaintableMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolvePaintableMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolvePaintableMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolvePaintableMethod "snapshot" o = PaintableSnapshotMethodInfo
    ResolvePaintableMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolvePaintableMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolvePaintableMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolvePaintableMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolvePaintableMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolvePaintableMethod "getCurrentImage" o = PaintableGetCurrentImageMethodInfo
    ResolvePaintableMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolvePaintableMethod "getFlags" o = PaintableGetFlagsMethodInfo
    ResolvePaintableMethod "getIntrinsicAspectRatio" o = PaintableGetIntrinsicAspectRatioMethodInfo
    ResolvePaintableMethod "getIntrinsicHeight" o = PaintableGetIntrinsicHeightMethodInfo
    ResolvePaintableMethod "getIntrinsicWidth" o = PaintableGetIntrinsicWidthMethodInfo
    ResolvePaintableMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolvePaintableMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolvePaintableMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolvePaintableMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolvePaintableMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolvePaintableMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolvePaintableMethod t Paintable, O.OverloadedMethod info Paintable p) => OL.IsLabel t (Paintable -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolvePaintableMethod t Paintable, O.OverloadedMethod info Paintable p, R.HasField t Paintable p) => R.HasField t Paintable p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolvePaintableMethod t Paintable, O.OverloadedMethodInfo info Paintable) => OL.IsLabel t (O.MethodProxy info Paintable) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

-- method Paintable::compute_concrete_size
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "paintable"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "Paintable" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GdkPaintable`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "specified_width"
--           , argType = TBasicType TDouble
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the width @paintable could be drawn into or\n  0.0 if unknown"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "specified_height"
--           , argType = TBasicType TDouble
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the height @paintable could be drawn into or\n  0.0 if unknown"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "default_width"
--           , argType = TBasicType TDouble
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the width @paintable would be drawn into if\n  no other constraints were given"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "default_height"
--           , argType = TBasicType TDouble
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the height @paintable would be drawn into if\n  no other constraints were given"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "concrete_width"
--           , argType = TBasicType TDouble
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "will be set to the concrete width computed"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "concrete_height"
--           , argType = TBasicType TDouble
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "will be set to the concrete height computed"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gdk_paintable_compute_concrete_size" gdk_paintable_compute_concrete_size :: 
    Ptr Paintable ->                        -- paintable : TInterface (Name {namespace = "Gdk", name = "Paintable"})
    CDouble ->                              -- specified_width : TBasicType TDouble
    CDouble ->                              -- specified_height : TBasicType TDouble
    CDouble ->                              -- default_width : TBasicType TDouble
    CDouble ->                              -- default_height : TBasicType TDouble
    Ptr CDouble ->                          -- concrete_width : TBasicType TDouble
    Ptr CDouble ->                          -- concrete_height : TBasicType TDouble
    IO ()

-- | Compute a concrete size for the @GdkPaintable@.
-- 
-- Applies the sizing algorithm outlined in the
-- <https://drafts.csswg.org/css-images-3/#default-sizing CSS Image spec>
-- to the given /@paintable@/. See that link for more details.
-- 
-- It is not necessary to call this function when both /@specifiedWidth@/
-- and /@specifiedHeight@/ are known, but it is useful to call this
-- function in GtkWidget:measure implementations to compute the
-- other dimension when only one dimension is given.
paintableComputeConcreteSize ::
    (B.CallStack.HasCallStack, MonadIO m, IsPaintable a) =>
    a
    -- ^ /@paintable@/: a @GdkPaintable@
    -> Double
    -- ^ /@specifiedWidth@/: the width /@paintable@/ could be drawn into or
    --   0.0 if unknown
    -> Double
    -- ^ /@specifiedHeight@/: the height /@paintable@/ could be drawn into or
    --   0.0 if unknown
    -> Double
    -- ^ /@defaultWidth@/: the width /@paintable@/ would be drawn into if
    --   no other constraints were given
    -> Double
    -- ^ /@defaultHeight@/: the height /@paintable@/ would be drawn into if
    --   no other constraints were given
    -> m ((Double, Double))
paintableComputeConcreteSize paintable specifiedWidth specifiedHeight defaultWidth defaultHeight = liftIO $ do
    paintable' <- unsafeManagedPtrCastPtr paintable
    let specifiedWidth' = realToFrac specifiedWidth
    let specifiedHeight' = realToFrac specifiedHeight
    let defaultWidth' = realToFrac defaultWidth
    let defaultHeight' = realToFrac defaultHeight
    concreteWidth <- allocMem :: IO (Ptr CDouble)
    concreteHeight <- allocMem :: IO (Ptr CDouble)
    gdk_paintable_compute_concrete_size paintable' specifiedWidth' specifiedHeight' defaultWidth' defaultHeight' concreteWidth concreteHeight
    concreteWidth' <- peek concreteWidth
    let concreteWidth'' = realToFrac concreteWidth'
    concreteHeight' <- peek concreteHeight
    let concreteHeight'' = realToFrac concreteHeight'
    touchManagedPtr paintable
    freeMem concreteWidth
    freeMem concreteHeight
    return (concreteWidth'', concreteHeight'')

#if defined(ENABLE_OVERLOADING)
data PaintableComputeConcreteSizeMethodInfo
instance (signature ~ (Double -> Double -> Double -> Double -> m ((Double, Double))), MonadIO m, IsPaintable a) => O.OverloadedMethod PaintableComputeConcreteSizeMethodInfo a signature where
    overloadedMethod = paintableComputeConcreteSize

instance O.OverloadedMethodInfo PaintableComputeConcreteSizeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gdk.Interfaces.Paintable.paintableComputeConcreteSize",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gdk-4.0.9/docs/GI-Gdk-Interfaces-Paintable.html#v:paintableComputeConcreteSize"
        })


#endif

-- method Paintable::get_current_image
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "paintable"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "Paintable" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GdkPaintable`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gdk" , name = "Paintable" })
-- throws : False
-- Skip return : False

foreign import ccall "gdk_paintable_get_current_image" gdk_paintable_get_current_image :: 
    Ptr Paintable ->                        -- paintable : TInterface (Name {namespace = "Gdk", name = "Paintable"})
    IO (Ptr Paintable)

-- | Gets an immutable paintable for the current contents displayed by /@paintable@/.
-- 
-- This is useful when you want to retain the current state of an animation,
-- for example to take a screenshot of a running animation.
-- 
-- If the /@paintable@/ is already immutable, it will return itself.
paintableGetCurrentImage ::
    (B.CallStack.HasCallStack, MonadIO m, IsPaintable a) =>
    a
    -- ^ /@paintable@/: a @GdkPaintable@
    -> m Paintable
    -- ^ __Returns:__ An immutable paintable for the current
    --   contents of /@paintable@/
paintableGetCurrentImage paintable = liftIO $ do
    paintable' <- unsafeManagedPtrCastPtr paintable
    result <- gdk_paintable_get_current_image paintable'
    checkUnexpectedReturnNULL "paintableGetCurrentImage" result
    result' <- (wrapObject Paintable) result
    touchManagedPtr paintable
    return result'

#if defined(ENABLE_OVERLOADING)
data PaintableGetCurrentImageMethodInfo
instance (signature ~ (m Paintable), MonadIO m, IsPaintable a) => O.OverloadedMethod PaintableGetCurrentImageMethodInfo a signature where
    overloadedMethod = paintableGetCurrentImage

instance O.OverloadedMethodInfo PaintableGetCurrentImageMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gdk.Interfaces.Paintable.paintableGetCurrentImage",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gdk-4.0.9/docs/GI-Gdk-Interfaces-Paintable.html#v:paintableGetCurrentImage"
        })


#endif

-- method Paintable::get_flags
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "paintable"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "Paintable" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GdkPaintable`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gdk" , name = "PaintableFlags" })
-- throws : False
-- Skip return : False

foreign import ccall "gdk_paintable_get_flags" gdk_paintable_get_flags :: 
    Ptr Paintable ->                        -- paintable : TInterface (Name {namespace = "Gdk", name = "Paintable"})
    IO CUInt

-- | Get flags for the paintable.
-- 
-- This is oftentimes useful for optimizations.
-- 
-- See [flags/@gdk@/.PaintableFlags] for the flags and what they mean.
paintableGetFlags ::
    (B.CallStack.HasCallStack, MonadIO m, IsPaintable a) =>
    a
    -- ^ /@paintable@/: a @GdkPaintable@
    -> m [Gdk.Flags.PaintableFlags]
    -- ^ __Returns:__ The @GdkPaintableFlags@ for this paintable
paintableGetFlags paintable = liftIO $ do
    paintable' <- unsafeManagedPtrCastPtr paintable
    result <- gdk_paintable_get_flags paintable'
    let result' = wordToGFlags result
    touchManagedPtr paintable
    return result'

#if defined(ENABLE_OVERLOADING)
data PaintableGetFlagsMethodInfo
instance (signature ~ (m [Gdk.Flags.PaintableFlags]), MonadIO m, IsPaintable a) => O.OverloadedMethod PaintableGetFlagsMethodInfo a signature where
    overloadedMethod = paintableGetFlags

instance O.OverloadedMethodInfo PaintableGetFlagsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gdk.Interfaces.Paintable.paintableGetFlags",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gdk-4.0.9/docs/GI-Gdk-Interfaces-Paintable.html#v:paintableGetFlags"
        })


#endif

-- method Paintable::get_intrinsic_aspect_ratio
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "paintable"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "Paintable" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GdkPaintable`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TDouble)
-- throws : False
-- Skip return : False

foreign import ccall "gdk_paintable_get_intrinsic_aspect_ratio" gdk_paintable_get_intrinsic_aspect_ratio :: 
    Ptr Paintable ->                        -- paintable : TInterface (Name {namespace = "Gdk", name = "Paintable"})
    IO CDouble

-- | Gets the preferred aspect ratio the /@paintable@/ would like to be displayed at.
-- 
-- The aspect ratio is the width divided by the height, so a value of 0.5
-- means that the /@paintable@/ prefers to be displayed twice as high as it
-- is wide. Consumers of this interface can use this to preserve aspect
-- ratio when displaying the paintable.
-- 
-- This is a purely informational value and does not in any way limit the
-- values that may be passed to 'GI.Gdk.Interfaces.Paintable.paintableSnapshot'.
-- 
-- Usually when a /@paintable@/ returns nonzero values from
-- 'GI.Gdk.Interfaces.Paintable.paintableGetIntrinsicWidth' and
-- 'GI.Gdk.Interfaces.Paintable.paintableGetIntrinsicHeight' the aspect ratio
-- should conform to those values, though that is not required.
-- 
-- If the /@paintable@/ does not have a preferred aspect ratio,
-- it returns 0. Negative values are never returned.
paintableGetIntrinsicAspectRatio ::
    (B.CallStack.HasCallStack, MonadIO m, IsPaintable a) =>
    a
    -- ^ /@paintable@/: a @GdkPaintable@
    -> m Double
    -- ^ __Returns:__ the intrinsic aspect ratio of /@paintable@/ or 0 if none.
paintableGetIntrinsicAspectRatio paintable = liftIO $ do
    paintable' <- unsafeManagedPtrCastPtr paintable
    result <- gdk_paintable_get_intrinsic_aspect_ratio paintable'
    let result' = realToFrac result
    touchManagedPtr paintable
    return result'

#if defined(ENABLE_OVERLOADING)
data PaintableGetIntrinsicAspectRatioMethodInfo
instance (signature ~ (m Double), MonadIO m, IsPaintable a) => O.OverloadedMethod PaintableGetIntrinsicAspectRatioMethodInfo a signature where
    overloadedMethod = paintableGetIntrinsicAspectRatio

instance O.OverloadedMethodInfo PaintableGetIntrinsicAspectRatioMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gdk.Interfaces.Paintable.paintableGetIntrinsicAspectRatio",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gdk-4.0.9/docs/GI-Gdk-Interfaces-Paintable.html#v:paintableGetIntrinsicAspectRatio"
        })


#endif

-- method Paintable::get_intrinsic_height
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "paintable"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "Paintable" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GdkPaintable`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "gdk_paintable_get_intrinsic_height" gdk_paintable_get_intrinsic_height :: 
    Ptr Paintable ->                        -- paintable : TInterface (Name {namespace = "Gdk", name = "Paintable"})
    IO Int32

-- | Gets the preferred height the /@paintable@/ would like to be displayed at.
-- 
-- Consumers of this interface can use this to reserve enough space to draw
-- the paintable.
-- 
-- This is a purely informational value and does not in any way limit the
-- values that may be passed to 'GI.Gdk.Interfaces.Paintable.paintableSnapshot'.
-- 
-- If the /@paintable@/ does not have a preferred height, it returns 0.
-- Negative values are never returned.
paintableGetIntrinsicHeight ::
    (B.CallStack.HasCallStack, MonadIO m, IsPaintable a) =>
    a
    -- ^ /@paintable@/: a @GdkPaintable@
    -> m Int32
    -- ^ __Returns:__ the intrinsic height of /@paintable@/ or 0 if none.
paintableGetIntrinsicHeight paintable = liftIO $ do
    paintable' <- unsafeManagedPtrCastPtr paintable
    result <- gdk_paintable_get_intrinsic_height paintable'
    touchManagedPtr paintable
    return result

#if defined(ENABLE_OVERLOADING)
data PaintableGetIntrinsicHeightMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsPaintable a) => O.OverloadedMethod PaintableGetIntrinsicHeightMethodInfo a signature where
    overloadedMethod = paintableGetIntrinsicHeight

instance O.OverloadedMethodInfo PaintableGetIntrinsicHeightMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gdk.Interfaces.Paintable.paintableGetIntrinsicHeight",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gdk-4.0.9/docs/GI-Gdk-Interfaces-Paintable.html#v:paintableGetIntrinsicHeight"
        })


#endif

-- method Paintable::get_intrinsic_width
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "paintable"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "Paintable" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GdkPaintable`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "gdk_paintable_get_intrinsic_width" gdk_paintable_get_intrinsic_width :: 
    Ptr Paintable ->                        -- paintable : TInterface (Name {namespace = "Gdk", name = "Paintable"})
    IO Int32

-- | Gets the preferred width the /@paintable@/ would like to be displayed at.
-- 
-- Consumers of this interface can use this to reserve enough space to draw
-- the paintable.
-- 
-- This is a purely informational value and does not in any way limit the
-- values that may be passed to 'GI.Gdk.Interfaces.Paintable.paintableSnapshot'.
-- 
-- If the /@paintable@/ does not have a preferred width, it returns 0.
-- Negative values are never returned.
paintableGetIntrinsicWidth ::
    (B.CallStack.HasCallStack, MonadIO m, IsPaintable a) =>
    a
    -- ^ /@paintable@/: a @GdkPaintable@
    -> m Int32
    -- ^ __Returns:__ the intrinsic width of /@paintable@/ or 0 if none.
paintableGetIntrinsicWidth paintable = liftIO $ do
    paintable' <- unsafeManagedPtrCastPtr paintable
    result <- gdk_paintable_get_intrinsic_width paintable'
    touchManagedPtr paintable
    return result

#if defined(ENABLE_OVERLOADING)
data PaintableGetIntrinsicWidthMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsPaintable a) => O.OverloadedMethod PaintableGetIntrinsicWidthMethodInfo a signature where
    overloadedMethod = paintableGetIntrinsicWidth

instance O.OverloadedMethodInfo PaintableGetIntrinsicWidthMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gdk.Interfaces.Paintable.paintableGetIntrinsicWidth",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gdk-4.0.9/docs/GI-Gdk-Interfaces-Paintable.html#v:paintableGetIntrinsicWidth"
        })


#endif

-- method Paintable::invalidate_contents
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "paintable"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "Paintable" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GdkPaintable`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gdk_paintable_invalidate_contents" gdk_paintable_invalidate_contents :: 
    Ptr Paintable ->                        -- paintable : TInterface (Name {namespace = "Gdk", name = "Paintable"})
    IO ()

-- | Called by implementations of @GdkPaintable@ to invalidate their contents.
-- 
-- Unless the contents are invalidated, implementations must guarantee that
-- multiple calls of 'GI.Gdk.Interfaces.Paintable.paintableSnapshot' produce the same output.
-- 
-- This function will emit the [Paintable::invalidateContents]("GI.Gdk.Interfaces.Paintable#g:signal:invalidateContents")
-- signal.
-- 
-- If a /@paintable@/ reports the 'GI.Gdk.Flags.PaintableFlagsContents' flag,
-- it must not call this function.
paintableInvalidateContents ::
    (B.CallStack.HasCallStack, MonadIO m, IsPaintable a) =>
    a
    -- ^ /@paintable@/: a @GdkPaintable@
    -> m ()
paintableInvalidateContents paintable = liftIO $ do
    paintable' <- unsafeManagedPtrCastPtr paintable
    gdk_paintable_invalidate_contents paintable'
    touchManagedPtr paintable
    return ()

#if defined(ENABLE_OVERLOADING)
data PaintableInvalidateContentsMethodInfo
instance (signature ~ (m ()), MonadIO m, IsPaintable a) => O.OverloadedMethod PaintableInvalidateContentsMethodInfo a signature where
    overloadedMethod = paintableInvalidateContents

instance O.OverloadedMethodInfo PaintableInvalidateContentsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gdk.Interfaces.Paintable.paintableInvalidateContents",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gdk-4.0.9/docs/GI-Gdk-Interfaces-Paintable.html#v:paintableInvalidateContents"
        })


#endif

-- method Paintable::invalidate_size
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "paintable"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "Paintable" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GdkPaintable`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gdk_paintable_invalidate_size" gdk_paintable_invalidate_size :: 
    Ptr Paintable ->                        -- paintable : TInterface (Name {namespace = "Gdk", name = "Paintable"})
    IO ()

-- | Called by implementations of @GdkPaintable@ to invalidate their size.
-- 
-- As long as the size is not invalidated, /@paintable@/ must return the same
-- values for its intrinsic width, height and aspect ratio.
-- 
-- This function will emit the [Paintable::invalidateSize]("GI.Gdk.Interfaces.Paintable#g:signal:invalidateSize")
-- signal.
-- 
-- If a /@paintable@/ reports the 'GI.Gdk.Flags.PaintableFlagsSize' flag,
-- it must not call this function.
paintableInvalidateSize ::
    (B.CallStack.HasCallStack, MonadIO m, IsPaintable a) =>
    a
    -- ^ /@paintable@/: a @GdkPaintable@
    -> m ()
paintableInvalidateSize paintable = liftIO $ do
    paintable' <- unsafeManagedPtrCastPtr paintable
    gdk_paintable_invalidate_size paintable'
    touchManagedPtr paintable
    return ()

#if defined(ENABLE_OVERLOADING)
data PaintableInvalidateSizeMethodInfo
instance (signature ~ (m ()), MonadIO m, IsPaintable a) => O.OverloadedMethod PaintableInvalidateSizeMethodInfo a signature where
    overloadedMethod = paintableInvalidateSize

instance O.OverloadedMethodInfo PaintableInvalidateSizeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gdk.Interfaces.Paintable.paintableInvalidateSize",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gdk-4.0.9/docs/GI-Gdk-Interfaces-Paintable.html#v:paintableInvalidateSize"
        })


#endif

-- method Paintable::snapshot
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "paintable"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "Paintable" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GdkPaintable`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "snapshot"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "Snapshot" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GdkSnapshot` to snapshot to"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "width"
--           , argType = TBasicType TDouble
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "width to snapshot in"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "height"
--           , argType = TBasicType TDouble
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "height to snapshot in"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gdk_paintable_snapshot" gdk_paintable_snapshot :: 
    Ptr Paintable ->                        -- paintable : TInterface (Name {namespace = "Gdk", name = "Paintable"})
    Ptr Gdk.Snapshot.Snapshot ->            -- snapshot : TInterface (Name {namespace = "Gdk", name = "Snapshot"})
    CDouble ->                              -- width : TBasicType TDouble
    CDouble ->                              -- height : TBasicType TDouble
    IO ()

-- | Snapshots the given paintable with the given /@width@/ and /@height@/.
-- 
-- The paintable is drawn at the current (0,0) offset of the /@snapshot@/.
-- If /@width@/ and /@height@/ are not larger than zero, this function will
-- do nothing.
paintableSnapshot ::
    (B.CallStack.HasCallStack, MonadIO m, IsPaintable a, Gdk.Snapshot.IsSnapshot b) =>
    a
    -- ^ /@paintable@/: a @GdkPaintable@
    -> b
    -- ^ /@snapshot@/: a @GdkSnapshot@ to snapshot to
    -> Double
    -- ^ /@width@/: width to snapshot in
    -> Double
    -- ^ /@height@/: height to snapshot in
    -> m ()
paintableSnapshot paintable snapshot width height = liftIO $ do
    paintable' <- unsafeManagedPtrCastPtr paintable
    snapshot' <- unsafeManagedPtrCastPtr snapshot
    let width' = realToFrac width
    let height' = realToFrac height
    gdk_paintable_snapshot paintable' snapshot' width' height'
    touchManagedPtr paintable
    touchManagedPtr snapshot
    return ()

#if defined(ENABLE_OVERLOADING)
data PaintableSnapshotMethodInfo
instance (signature ~ (b -> Double -> Double -> m ()), MonadIO m, IsPaintable a, Gdk.Snapshot.IsSnapshot b) => O.OverloadedMethod PaintableSnapshotMethodInfo a signature where
    overloadedMethod = paintableSnapshot

instance O.OverloadedMethodInfo PaintableSnapshotMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gdk.Interfaces.Paintable.paintableSnapshot",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gdk-4.0.9/docs/GI-Gdk-Interfaces-Paintable.html#v:paintableSnapshot"
        })


#endif

-- method Paintable::new_empty
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "intrinsic_width"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "The intrinsic width to report. Can be 0 for no width."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "intrinsic_height"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "The intrinsic height to report. Can be 0 for no height."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gdk" , name = "Paintable" })
-- throws : False
-- Skip return : False

foreign import ccall "gdk_paintable_new_empty" gdk_paintable_new_empty :: 
    Int32 ->                                -- intrinsic_width : TBasicType TInt
    Int32 ->                                -- intrinsic_height : TBasicType TInt
    IO (Ptr Paintable)

-- | Returns a paintable that has the given intrinsic size and draws nothing.
-- 
-- This is often useful for implementing the
-- t'GI.Gdk.Interfaces.Paintable.Paintable'.@/get_current_image/@() virtual function
-- when the paintable is in an incomplete state (like a
-- <http://developer.gnome.org/gdk/stable/../gtk4/class.MediaStream.html GtkMediaStream> before receiving
-- the first frame).
paintableNewEmpty ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Int32
    -- ^ /@intrinsicWidth@/: The intrinsic width to report. Can be 0 for no width.
    -> Int32
    -- ^ /@intrinsicHeight@/: The intrinsic height to report. Can be 0 for no height.
    -> m Paintable
    -- ^ __Returns:__ a @GdkPaintable@
paintableNewEmpty intrinsicWidth intrinsicHeight = liftIO $ do
    result <- gdk_paintable_new_empty intrinsicWidth intrinsicHeight
    checkUnexpectedReturnNULL "paintableNewEmpty" result
    result' <- (wrapObject Paintable) result
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- signal Paintable::invalidate-contents
-- | Emitted when the contents of the /@paintable@/ change.
-- 
-- Examples for such an event would be videos changing to the next frame or
-- the icon theme for an icon changing.
type PaintableInvalidateContentsCallback =
    IO ()

type C_PaintableInvalidateContentsCallback =
    Ptr Paintable ->                        -- object
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_PaintableInvalidateContentsCallback`.
foreign import ccall "wrapper"
    mk_PaintableInvalidateContentsCallback :: C_PaintableInvalidateContentsCallback -> IO (FunPtr C_PaintableInvalidateContentsCallback)

wrap_PaintableInvalidateContentsCallback :: 
    GObject a => (a -> PaintableInvalidateContentsCallback) ->
    C_PaintableInvalidateContentsCallback
wrap_PaintableInvalidateContentsCallback gi'cb gi'selfPtr _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self) 


-- | Connect a signal handler for the [invalidateContents](#signal:invalidateContents) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' paintable #invalidateContents callback
-- @
-- 
-- 
onPaintableInvalidateContents :: (IsPaintable a, MonadIO m) => a -> ((?self :: a) => PaintableInvalidateContentsCallback) -> m SignalHandlerId
onPaintableInvalidateContents obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_PaintableInvalidateContentsCallback wrapped
    wrapped'' <- mk_PaintableInvalidateContentsCallback wrapped'
    connectSignalFunPtr obj "invalidate-contents" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [invalidateContents](#signal:invalidateContents) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' paintable #invalidateContents callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterPaintableInvalidateContents :: (IsPaintable a, MonadIO m) => a -> ((?self :: a) => PaintableInvalidateContentsCallback) -> m SignalHandlerId
afterPaintableInvalidateContents obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_PaintableInvalidateContentsCallback wrapped
    wrapped'' <- mk_PaintableInvalidateContentsCallback wrapped'
    connectSignalFunPtr obj "invalidate-contents" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data PaintableInvalidateContentsSignalInfo
instance SignalInfo PaintableInvalidateContentsSignalInfo where
    type HaskellCallbackType PaintableInvalidateContentsSignalInfo = PaintableInvalidateContentsCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_PaintableInvalidateContentsCallback cb
        cb'' <- mk_PaintableInvalidateContentsCallback cb'
        connectSignalFunPtr obj "invalidate-contents" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gdk.Interfaces.Paintable::invalidate-contents"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gdk-4.0.9/docs/GI-Gdk-Interfaces-Paintable.html#g:signal:invalidateContents"})

#endif

-- signal Paintable::invalidate-size
-- | Emitted when the intrinsic size of the /@paintable@/ changes.
-- 
-- This means the values reported by at least one of
-- 'GI.Gdk.Interfaces.Paintable.paintableGetIntrinsicWidth',
-- 'GI.Gdk.Interfaces.Paintable.paintableGetIntrinsicHeight' or
-- 'GI.Gdk.Interfaces.Paintable.paintableGetIntrinsicAspectRatio'
-- has changed.
-- 
-- Examples for such an event would be a paintable displaying
-- the contents of a toplevel surface being resized.
type PaintableInvalidateSizeCallback =
    IO ()

type C_PaintableInvalidateSizeCallback =
    Ptr Paintable ->                        -- object
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_PaintableInvalidateSizeCallback`.
foreign import ccall "wrapper"
    mk_PaintableInvalidateSizeCallback :: C_PaintableInvalidateSizeCallback -> IO (FunPtr C_PaintableInvalidateSizeCallback)

wrap_PaintableInvalidateSizeCallback :: 
    GObject a => (a -> PaintableInvalidateSizeCallback) ->
    C_PaintableInvalidateSizeCallback
wrap_PaintableInvalidateSizeCallback gi'cb gi'selfPtr _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self) 


-- | Connect a signal handler for the [invalidateSize](#signal:invalidateSize) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' paintable #invalidateSize callback
-- @
-- 
-- 
onPaintableInvalidateSize :: (IsPaintable a, MonadIO m) => a -> ((?self :: a) => PaintableInvalidateSizeCallback) -> m SignalHandlerId
onPaintableInvalidateSize obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_PaintableInvalidateSizeCallback wrapped
    wrapped'' <- mk_PaintableInvalidateSizeCallback wrapped'
    connectSignalFunPtr obj "invalidate-size" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [invalidateSize](#signal:invalidateSize) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' paintable #invalidateSize callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterPaintableInvalidateSize :: (IsPaintable a, MonadIO m) => a -> ((?self :: a) => PaintableInvalidateSizeCallback) -> m SignalHandlerId
afterPaintableInvalidateSize obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_PaintableInvalidateSizeCallback wrapped
    wrapped'' <- mk_PaintableInvalidateSizeCallback wrapped'
    connectSignalFunPtr obj "invalidate-size" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data PaintableInvalidateSizeSignalInfo
instance SignalInfo PaintableInvalidateSizeSignalInfo where
    type HaskellCallbackType PaintableInvalidateSizeSignalInfo = PaintableInvalidateSizeCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_PaintableInvalidateSizeCallback cb
        cb'' <- mk_PaintableInvalidateSizeCallback cb'
        connectSignalFunPtr obj "invalidate-size" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gdk.Interfaces.Paintable::invalidate-size"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gdk-4.0.9/docs/GI-Gdk-Interfaces-Paintable.html#g:signal:invalidateSize"})

#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList Paintable = PaintableSignalList
type PaintableSignalList = ('[ '("invalidateContents", PaintableInvalidateContentsSignalInfo), '("invalidateSize", PaintableInvalidateSizeSignalInfo), '("notify", GObject.Object.ObjectNotifySignalInfo)] :: [(Symbol, DK.Type)])

#endif


