/*
 * nUFRaw - Unidentified Flying Raw converter for digital camera images
 *
 * nufraw_ui.c - GUI for controlling all the image manipulations
 * Copyright 2016 by Matteo Lucarelli
 *
 * based on ufraw 0.23 Copyright 2004-2015 by Udi Fuchs
 * based on the GIMP plug-in by Pawel T. Jochym jochym at ifj edu pl,
 * based on the GIMP plug-in by Dave Coffin http://www.cybercom.net/~dcoffin/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include "nufraw.h"
#include "uf_gtk.h"
#include "nufraw_ui.h"
#include "curveeditor_widget.h"
#include <gdk/gdkkeysyms.h>
#include <glib/gi18n.h>
#include <gtkimageview/gtkimagescrollwin.h>
#include <gtkimageview/gtkimageview.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <errno.h>

#ifdef _OPENMP
#include <omp.h>
#define uf_omp_get_thread_num() omp_get_thread_num()
#define uf_omp_get_max_threads() omp_get_max_threads()
#else
#define uf_omp_get_thread_num() 0
#define uf_omp_get_max_threads() 1
#endif

#ifdef _WIN32
// GDK threads are not supported on the Windows platform
#define gdk_threads_add_timeout g_timeout_add
#define gdk_threads_add_idle_full g_idle_add_full
#endif

// Response can be any unique positive integer
#define NUFRAW_RESPONSE_DELETE  1
#define NUFRAW_NO_RESPONSE      2

// from nufraw_ui_saver.c
extern long nufraw_save_dialog(nufraw_data *uf, ui_conf *uiconf, void *widget);
extern long nufraw_save_now(nufraw_data *uf, void *widget);
extern long nufraw_send_to_gimp(nufraw_data *uf, char* remoteGimpCommand);
extern int nufraw_confirm_dialog(const char *message, const char* title, void *widget);

// from nufraw_ui_lensfun.c
extern void lens_fill_interface(ui_data *data, GtkTable *table);

// from nufraw_ui_chooser.c
extern void nufraw_hidden_chooser_toggle(GtkToggleButton *button, GtkFileChooser *filechooser);
extern GtkFileChooser *nufraw_raw_chooser(conf_data *conf, ui_conf* uiconf, const char *defPath,
                                         const gchar *label, GtkWindow *toplevel,
                                         const gchar *cancel, gboolean multiple, gboolean idonly);

// from nufraw_ui_options.c
extern void show_options_dialog();

// from nufraw_ui_histogram.c
extern gboolean render_raw_histogram(gpointer unused);
extern gboolean render_live_histogram(gpointer unused);
extern void rawhistogram_fill_interface(GtkTable* table);
extern void livehistogram_fill_interface(GtkTable* table);
extern void histogram_expander(GtkWidget *expander, void *unused);
extern void panel_size_allocate(GtkWidget *panel, GtkAllocation *allocation, gpointer unused);
extern void collect_raw_histogram_data();
extern void valuetable_fill_interface(GtkTable *table);

static const char *grayscaleModeNames[5] = {
	N_("None"),
	N_("Lightness"),
	N_("Luminance"),
	N_("Value"),
	N_("Channel Mixer")
};

struct spot {
	int StartY;
	int EndY;
	int StartX;
	int EndX;
	int Size;
};

// the drag mode cursor is managed by gtk_image_viewer
static const GdkCursorType Cursors[cursor_num] = {
	GDK_HAND1,               // color picker mode
	GDK_LEFT_PTR,            // crop mode outside
	GDK_LEFT_SIDE,           // crop left side
	GDK_RIGHT_SIDE,          // crop right side
	GDK_TOP_SIDE,            // crop top side
	GDK_BOTTOM_SIDE,         // crop bottom side
	GDK_TOP_LEFT_CORNER,     // crop top left corner
	GDK_TOP_RIGHT_CORNER,    // crop top right corner
	GDK_BOTTOM_LEFT_CORNER,  // crop bottom left corner
	GDK_BOTTOM_RIGHT_CORNER, // crop bottom right corner
	GDK_CROSS                // crop move
};

// Global ui data (defined in nufraw_ui.h)
ui_data gUiData;

// configurations
conf_data *gConf=NULL;    // adjustments
ui_conf *gUiConf=NULL;    // gui

gboolean gImageHasModifications=FALSE;
gboolean gFirstRender=TRUE;

// utilities ///////////////////////////////////////////////////////////////////

void get_expander_state(GtkWidget *widget, void *index)
{
	int *i=(int*)index;
	if (*i<UF_MAX_EXPANDERS) gUiConf->expander_open[*i]=gtk_expander_get_expanded(GTK_EXPANDER(widget));
	(*i)++;
}

// set/unser window as modal
void nufraw_focus(void *window, gboolean focus)
{
	if (focus) {
		GtkWindow *parentWindow = (void *)nufraw_message(NUFRAW_SET_PARENT, (char *)window);
		g_object_set_data(G_OBJECT(window), "WindowParent", parentWindow);
		if (parentWindow != NULL) gtk_window_set_accept_focus(GTK_WINDOW(parentWindow), FALSE);
	} else {
		GtkWindow *parentWindow = g_object_get_data(G_OBJECT(window), "WindowParent");
		nufraw_message(NUFRAW_SET_PARENT, (char *)parentWindow);
		if (parentWindow != NULL) gtk_window_set_accept_focus(GTK_WINDOW(parentWindow), TRUE);
	}
}

void nufraw_messenger(char *message,  void *parentWindow)
{
	GtkDialog *dialog;

	if (parentWindow == NULL) {
		nufraw_batch_messenger(message);
	} else {
		dialog = GTK_DIALOG(gtk_message_dialog_new(GTK_WINDOW(parentWindow),
		                    GTK_DIALOG_DESTROY_WITH_PARENT,
		                    GTK_MESSAGE_WARNING, GTK_BUTTONS_OK, "%s", message));
		gtk_window_set_title(GTK_WINDOW(dialog), _("nUFRaw Message"));
		gtk_dialog_run(dialog);
		gtk_widget_destroy(GTK_WIDGET(dialog));
	}
}

static void list_store_add(GtkListStore *store, char *name, char *var)
{
	if (var != NULL && strlen(var) > 0) {
		GtkTreeIter iter;
		gtk_list_store_append(store, &iter);
		gtk_list_store_set(store, &iter, 0, name, 1, var, -1);
	}
}

// Return numeric zone representation from 0-1 luminance value.
// Unlike Adams, we use arabic for now.
// v0.18=z5 :  0stop
// v/2=z-1  : -1stop
// v*2=z+1  : +1stop
static double value2zone(double v)
{
	double zone=0;
	if (v>0) zone = log(v/0.18) / log(2.0) + 5.0;
	return zone;
}

gboolean has_camera_curve()
{
     return (gConf->BaseCurve[camera_curve].m_numAnchors>0);
}

colorLabels *color_labels_new(GtkTable *table, int x, int y, char *label, int format, gboolean forspot)
{
	colorLabels *l;
	GtkWidget *lbl;
	int c;
	int labelscount = (forspot ? 5 : 3); // 0-2 for RGB, 3 for value, 4 for zone
	int i=0;

	l = g_new(colorLabels, 1);
	l->format = format;
	l->forspot = forspot;

	if (label != NULL) {
		lbl = gtk_label_new(label);
		gtk_misc_set_alignment(GTK_MISC(lbl), 1, 0.5);
		gtk_table_attach(table, lbl, x, x + 1, y, y + 1, GTK_EXPAND | GTK_FILL, 0, 0, 0);
		i++;
	}

	for (c = 0; c < labelscount; c++, i++) {
		l->labels[c] = GTK_LABEL(gtk_label_new(NULL));
		gtk_misc_set_alignment(GTK_MISC(l->labels[c]),0.5,0.5);
		gtk_table_attach(table,GTK_WIDGET(l->labels[c]) , x + i, x + i + 1, y, y + 1, GTK_FILL|GTK_EXPAND,0, 0, 0);
	}
	return l;
}

void color_labels_set(colorLabels *l, double data[])
{
	int c;
	char buf[UF_MAX_NAME];char buf2[UF_MAX_NAME];
	const char lbl[]=UI_SPOT_LABELS;

	for (c = 0; c < 3; c++) {

		switch (l->format) {

			case pixel_format:
				g_snprintf(buf, UF_MAX_NAME, "%.0f", data[c]);
				break;

			case percent_format:
				if (data[c] < 10.0) g_snprintf(buf, UF_MAX_NAME, "%.2f%%", data[c]);
				else if (data[c] < 100.0) g_snprintf(buf, UF_MAX_NAME, "%.1f%%", data[c]);
				else g_snprintf(buf, UF_MAX_NAME, "%.0f%%", data[c]);
				break;
		}

		if (l->forspot){
			g_snprintf(buf2, UF_MAX_NAME, " %c:%s",lbl[c],buf);
			gtk_label_set(l->labels[c], buf2);
		}else{
			gtk_label_set(l->labels[c], buf);
		}
	}
	// If value/zone not requested, just return now
	if (!l->forspot) return;

	// Value
	c = 3;
	g_snprintf(buf, UF_MAX_NAME, " %c:%.3f", lbl[3], data[3]);
	gtk_label_set(l->labels[c], buf);

	// Zone
	c = 4;
	g_snprintf(buf, UF_MAX_NAME, " %c:%.1f", lbl[4], data[4]);
	gtk_label_set(l->labels[c], buf);
}

// transfrom spot from display to width/height image
static void calculate_spot(struct spot *spot, int width, int height)
{
	int spotHeight = abs(gUiData.SpotY1 - gUiData.SpotY2) * height / gUiData.UF->rotatedHeight + 1;
	spot->StartY = MIN(gUiData.SpotY1, gUiData.SpotY2) * height / gUiData.UF->rotatedHeight;
	if (spotHeight + spot->StartY > height) spot->StartY = height - spotHeight;
	spot->EndY = spot->StartY + spotHeight;

	int spotWidth = abs(gUiData.SpotX1 - gUiData.SpotX2) * width / gUiData.UF->rotatedWidth + 1;
	spot->StartX = MIN(gUiData.SpotX1, gUiData.SpotX2) * width / gUiData.UF->rotatedWidth;
	if (spotWidth + spot->StartX > width) spot->StartX = width - spotWidth;
	spot->EndX = spot->StartX + spotWidth;

	spot->Size = spotWidth * spotHeight;
}

// if zoom <=100 return shrink value else return 1
static int zoom_to_shrink(int zoom)
{
	if (zoom<100) return (int)(100.0/zoom+0.5);
	return 1;
}

// update size value (0 if shrink is set)
static void update_size()
{
	if (gConf->shrink == 1) {
		int cropSize = MAX(gConf->CropY2 - gConf->CropY1, gConf->CropX2 - gConf->CropX1);
		gConf->size = MIN(gUiConf->Zoom, 100.0) / 100.0 * cropSize;
	} else {
		gConf->size = 0;
	}
}

gboolean ui_is_rendering()
{
	return (gUiData.RenderSubArea >= 0);
}

// passes > window makes no sense. Either the number of passes is
// ridiculously large or the same effect can be achieved by replacing
// decay by pow(decay, original_passes).
static void despeckle_apply_constraints(GtkAdjustment **adjp, int ch)
{
	conf_data *c = gConf;
	double value = gtk_adjustment_get_value(adjp[ch]);

	gUiData.FreezeDialog++;

	if (adjp == gUiData.DespeckleWindowAdj && value < c->despecklePasses[ch] && value) {
		c->despecklePasses[ch] = value;
		gtk_adjustment_set_value(gUiData.DespecklePassesAdj[ch], value);
	}
	if (adjp == gUiData.DespecklePassesAdj && value > c->despeckleWindow[ch] && c->despeckleWindow[ch]) {
		c->despeckleWindow[ch] = value;
		gtk_adjustment_set_value(gUiData.DespeckleWindowAdj[ch], value);
	}
	gUiData.FreezeDialog--;
}

static void lch_to_color(float lch[3], GdkColor *color)
{
	gint64 rgb[3];
	uf_cielch_to_rgb(lch, rgb);
	color->red = pow((double)MIN(rgb[0], 0xFFFF) / 0xFFFF, 0.45) * 0xFFFF;
	color->green = pow((double)MIN(rgb[1], 0xFFFF) / 0xFFFF, 0.45) * 0xFFFF;
	color->blue = pow((double)MIN(rgb[2], 0xFFFF) / 0xFFFF, 0.45) * 0xFFFF;
}

static void widget_set_hue(GtkWidget *widget, double hue)
{
	// FIXME: sometimes colors are strange

	float lch[3];
	GdkColor hueColor;
	lch[1] = 181.019336; // max_colorfulness = sqrt(128*128+128*128)
	lch[2] = hue * M_PI / 180;

	lch[0] = 0.75 * 100.0; // max_luminance = 100
	lch_to_color(lch, &hueColor);
	gtk_widget_modify_bg(widget, GTK_STATE_NORMAL, &hueColor);

	lch[0] = 1.0 * 100.0; // max_luminance = 100
	lch_to_color(lch, &hueColor);
	gtk_widget_modify_bg(widget, GTK_STATE_PRELIGHT, &hueColor);

	lch[0] = 0.5 * 100.0; // max_luminance = 100
	lch_to_color(lch, &hueColor);
	gtk_widget_modify_bg(widget, GTK_STATE_ACTIVE, &hueColor);
}

// curve management ////////////////////////////////////////////////////////////

static void load_curve(GtkWidget *widget, long curveType)
{
	GtkFileChooser *fileChooser;
	GtkFileFilter *filter;
	GSList *list, *saveList;
	CurveData c;
	char *cp;

	if (gUiData.FreezeDialog) return;

	if ((curveType == base_curve && gConf->BaseCurveCount >= UF_MAX_CURVES) ||
	    (curveType == luminosity_curve && gConf->curveCount >= UF_MAX_CURVES)) {

		// TODO disable load button instead
		nufraw_message(NUFRAW_ERROR, _("No more room for new curves."));
		return;
	}

	fileChooser = GTK_FILE_CHOOSER(gtk_file_chooser_dialog_new(_("Load curve"),
	                                                           GTK_WINDOW(gtk_widget_get_toplevel(widget)),
	                                                           GTK_FILE_CHOOSER_ACTION_OPEN,
	                                                           GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
	                                                           GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT, NULL));

	nufraw_focus(fileChooser, TRUE);

	gtk_file_chooser_set_select_multiple(fileChooser, TRUE);
	gtk_file_chooser_set_show_hidden(fileChooser, FALSE);

	GtkWidget *button = gtk_check_button_new_with_label(_("Show hidden files"));
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), FALSE);
	g_signal_connect(G_OBJECT(button), "toggled",
	G_CALLBACK(nufraw_hidden_chooser_toggle), fileChooser);
	gtk_file_chooser_set_extra_widget(fileChooser, button);

	filter = GTK_FILE_FILTER(gtk_file_filter_new());
	gtk_file_filter_set_name(filter, _("All curve formats"));
	gtk_file_filter_add_pattern(filter, "*.ntc");
	gtk_file_filter_add_pattern(filter, "*.NTC");
	gtk_file_filter_add_pattern(filter, "*.ncv");
	gtk_file_filter_add_pattern(filter, "*.NCV");
	gtk_file_filter_add_pattern(filter, "*.curve");
	gtk_file_filter_add_pattern(filter, "*.CURVE");
	gtk_file_chooser_add_filter(fileChooser, filter);

	filter = GTK_FILE_FILTER(gtk_file_filter_new());
	gtk_file_filter_set_name(filter, _("nUFRaw curve format"));
	gtk_file_filter_add_pattern(filter, "*.curve");
	gtk_file_filter_add_pattern(filter, "*.CURVE");
	gtk_file_chooser_add_filter(fileChooser, filter);

	filter = GTK_FILE_FILTER(gtk_file_filter_new());
	gtk_file_filter_set_name(filter, _("Nikon curve format"));
	gtk_file_filter_add_pattern(filter, "*.ntc");
	gtk_file_filter_add_pattern(filter, "*.NTC");
	gtk_file_filter_add_pattern(filter, "*.ncv");
	gtk_file_filter_add_pattern(filter, "*.NCV");
	gtk_file_chooser_add_filter(fileChooser, filter);

	filter = GTK_FILE_FILTER(gtk_file_filter_new());
	gtk_file_filter_set_name(filter, _("All files"));
	gtk_file_filter_add_pattern(filter, "*");
	gtk_file_chooser_add_filter(fileChooser, filter);

	if (strlen(gUiConf->curvePath) > 0) gtk_file_chooser_set_current_folder(fileChooser, gUiConf->curvePath);

	if (gtk_dialog_run(GTK_DIALOG(fileChooser)) == GTK_RESPONSE_ACCEPT) {

		for (list = saveList = gtk_file_chooser_get_filenames(fileChooser); list != NULL; list = g_slist_next(list)) {

			c = conf_default.curve[0];
			if (curve_load(&c, list->data) != NUFRAW_SUCCESS) continue;
			if (curveType == base_curve) {

				if (gConf->BaseCurveCount >= UF_MAX_CURVES) break;

				gtk_combo_box_append_text(gUiData.BaseCurveCombo, c.name);
				gConf->BaseCurve[gConf->BaseCurveCount] = c;
				gConf->BaseCurveIndex = gConf->BaseCurveCount;
				gConf->BaseCurveCount++;

				if (has_camera_curve())  gtk_combo_box_set_active(gUiData.BaseCurveCombo, gConf->BaseCurveIndex);
				else gtk_combo_box_set_active(gUiData.BaseCurveCombo, gConf->BaseCurveIndex - 2);
			} else {

				// curveType==luminosity_curve
				if (gConf->curveCount >= UF_MAX_CURVES) break;

				gtk_combo_box_append_text(gUiData.CurveCombo, c.name);
				gConf->curve[gConf->curveCount] = c;
				gConf->curveIndex = gConf->curveCount;
				gConf->curveCount++;

				gtk_combo_box_set_active(gUiData.CurveCombo, gConf->curveIndex);
			}

			cp = g_path_get_dirname(list->data);
			g_strlcpy(gUiConf->curvePath, cp, UF_MAX_PATH);

			g_free(cp);
			g_free(list->data);
		}

		// TODO disable load button instead
		if (list != NULL) nufraw_message(NUFRAW_ERROR, _("No more room for new curves."));
		g_slist_free(saveList);
	}
	nufraw_focus(fileChooser, FALSE);
	gtk_widget_destroy(GTK_WIDGET(fileChooser));
}

static void save_curve(GtkWidget *widget, long curveType)
{
	GtkFileChooser *fileChooser;
	GtkFileFilter *filter;
	char defFilename[UF_MAX_NAME];
	char *filename, *cp;

	if (gUiData.FreezeDialog) return;

	fileChooser = GTK_FILE_CHOOSER(gtk_file_chooser_dialog_new(_("Save curve"),
	                                                           GTK_WINDOW(gtk_widget_get_toplevel(widget)),
	                                                           GTK_FILE_CHOOSER_ACTION_SAVE,
	                                                           GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
	                                                           GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT, NULL));

	nufraw_focus(fileChooser, TRUE);

	gtk_file_chooser_set_show_hidden(fileChooser, FALSE);
	GtkWidget *button = gtk_check_button_new_with_label(_("Show hidden files"));
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), FALSE);
	g_signal_connect(G_OBJECT(button), "toggled",
	G_CALLBACK(nufraw_hidden_chooser_toggle), fileChooser);
	gtk_file_chooser_set_extra_widget(fileChooser, button);

	filter = GTK_FILE_FILTER(gtk_file_filter_new());
	gtk_file_filter_set_name(filter, _("All curve formats"));
	gtk_file_filter_add_pattern(filter, "*.ntc");
	gtk_file_filter_add_pattern(filter, "*.NTC");
	gtk_file_filter_add_pattern(filter, "*.ncv");
	gtk_file_filter_add_pattern(filter, "*.NCV");
	gtk_file_filter_add_pattern(filter, "*.curve");
	gtk_file_filter_add_pattern(filter, "*.CURVE");
	gtk_file_chooser_add_filter(fileChooser, filter);

	filter = GTK_FILE_FILTER(gtk_file_filter_new());
	gtk_file_filter_set_name(filter, _("nUFRaw curve format"));
	gtk_file_filter_add_pattern(filter, "*.curve");
	gtk_file_filter_add_pattern(filter, "*.CURVE");
	gtk_file_chooser_add_filter(fileChooser, filter);

	filter = GTK_FILE_FILTER(gtk_file_filter_new());
	gtk_file_filter_set_name(filter, _("Nikon curve format"));
	gtk_file_filter_add_pattern(filter, "*.ntc");
	gtk_file_filter_add_pattern(filter, "*.NTC");
	gtk_file_filter_add_pattern(filter, "*.ncv");
	gtk_file_filter_add_pattern(filter, "*.NCV");
	gtk_file_chooser_add_filter(fileChooser, filter);

	filter = GTK_FILE_FILTER(gtk_file_filter_new());
	gtk_file_filter_set_name(filter, _("All files"));
	gtk_file_filter_add_pattern(filter, "*");
	gtk_file_chooser_add_filter(fileChooser, filter);

	if (strlen(gUiConf->curvePath) > 0) gtk_file_chooser_set_current_folder(fileChooser, gUiConf->curvePath);

	if (curveType == base_curve) g_snprintf(defFilename, UF_MAX_NAME, "%s.curve", gConf->BaseCurve[gConf->BaseCurveIndex].name);
	else	g_snprintf(defFilename, UF_MAX_NAME, "%s.curve", gConf->curve[gConf->curveIndex].name);

	gtk_file_chooser_set_current_name(fileChooser, defFilename);

	if (gtk_dialog_run(GTK_DIALOG(fileChooser)) == GTK_RESPONSE_ACCEPT) {
		filename = gtk_file_chooser_get_filename(fileChooser);

		if (curveType == base_curve) curve_save(&gConf->BaseCurve[gConf->BaseCurveIndex], filename);
		else curve_save(&gConf->curve[gConf->curveIndex], filename);

		cp = g_path_get_dirname(filename);
		g_strlcpy(gUiConf->curvePath, cp, UF_MAX_PATH);

		g_free(cp);
		g_free(filename);
	}
	nufraw_focus(fileChooser, FALSE);
	gtk_widget_destroy(GTK_WIDGET(fileChooser));
}

// profile /////////////////////////////////////////////////////////////////////

void load_profile(GtkWidget *widget, long type)
{
	GtkFileChooser *fileChooser;
	GSList *list, *saveList;
	GtkFileFilter *filter;
	char *filename, *cp;
	profile_data p;

	if (gUiData.FreezeDialog) return;
	if (gConf->profileCount[type] == UF_MAX_PROFILES) {
		// TODO: disable load button instead
		nufraw_message(NUFRAW_ERROR, _("No more room for new profiles."));
		return;
	}

	fileChooser = GTK_FILE_CHOOSER(gtk_file_chooser_dialog_new(_("Load color profile"),
	                                                           GTK_WINDOW(gtk_widget_get_toplevel(widget)),
	                                                           GTK_FILE_CHOOSER_ACTION_OPEN,
	                                                           GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
	                                                           GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT, NULL));

	nufraw_focus(fileChooser, TRUE);

	gtk_file_chooser_set_select_multiple(fileChooser, TRUE);
	gtk_file_chooser_set_show_hidden(fileChooser, FALSE);
	GtkWidget *button = gtk_check_button_new_with_label(_("Show hidden files"));
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), FALSE);
	g_signal_connect(G_OBJECT(button), "toggled",
	G_CALLBACK(nufraw_hidden_chooser_toggle), fileChooser);
	gtk_file_chooser_set_extra_widget(fileChooser, button);

	if (strlen(gUiConf->profilePath) > 0) gtk_file_chooser_set_current_folder(fileChooser, gUiConf->profilePath);

	filter = GTK_FILE_FILTER(gtk_file_filter_new());
	gtk_file_filter_set_name(filter, _("Color Profiles"));
	gtk_file_filter_add_pattern(filter, "*.icm");
	gtk_file_filter_add_pattern(filter, "*.ICM");
	gtk_file_filter_add_pattern(filter, "*.icc");
	gtk_file_filter_add_pattern(filter, "*.ICC");
	gtk_file_chooser_add_filter(fileChooser, filter);

	filter = GTK_FILE_FILTER(gtk_file_filter_new());
	gtk_file_filter_set_name(filter, _("All files"));
	gtk_file_filter_add_pattern(filter, "*");
	gtk_file_chooser_add_filter(fileChooser, filter);

	if (gtk_dialog_run(GTK_DIALOG(fileChooser)) == GTK_RESPONSE_ACCEPT) {

		for (list = saveList = gtk_file_chooser_get_filenames(fileChooser);
		     list != NULL && gConf->profileCount[type] < UF_MAX_PROFILES;
		     list = g_slist_next(list)) {

			filename = list->data;
			p = conf_default.profile[type][conf_default.profileCount[type]];
			g_strlcpy(p.file, filename, UF_MAX_PATH);

			// Make sure we update product name
			gUiData.UF->developer->updateTransform = TRUE;
			developer_profile(gUiData.UF->developer, type, &p);

			if (gUiData.UF->developer->profile[type] == NULL) {
				g_free(list->data);
				continue;
			}

			char *base = g_path_get_basename(filename);
			char *name = uf_file_set_type(base, "");
			char *utf8 = g_filename_display_name(name);
			g_strlcpy(p.name, utf8, UF_MAX_NAME);
			g_free(utf8);
			g_free(name);
			g_free(base);

			// Set some defaults to the profile's curve
			p.gamma = profile_default_gamma(&p);
			gConf->profile[type][gConf->profileCount[type]++] = p;
			gtk_combo_box_append_text(gUiData.ProfileCombo[type], p.name);
			gConf->profileIndex[type] = gConf->profileCount[type] - 1;

			cp = g_path_get_dirname(list->data);
			g_strlcpy(gUiConf->profilePath, cp, UF_MAX_PATH);

			g_free(cp);
			g_free(list->data);
		}
		gtk_combo_box_set_active(gUiData.ProfileCombo[type],
		gConf->profileIndex[type]);

		if (list != NULL) nufraw_message(NUFRAW_ERROR, _("No more room for new profiles."));
		g_slist_free(saveList);
	}
	nufraw_focus(fileChooser, FALSE);
	gtk_widget_destroy(GTK_WIDGET(fileChooser));
}

// render //////////////////////////////////////////////////////////////////////

// write display_phase image in gdk_pixbuf widget
// Modify the preview image to mark crop, spot and specal render
// Note that all coordinate intervals are semi-inclusive, e.g.
// X1 <= pixels < X2 and Y1 <= pixels < Y2
// This approach makes computing width/height just a matter of
// substracting X1 from X2 or Y1 from Y2.
static void preview_draw_area(int x, int y, int width, int height)
{
	int pixbufHeight = gdk_pixbuf_get_height(gUiData.PreviewPixbuf);

	if (y < 0 || y >= pixbufHeight) g_error("preview_draw_area(): y:%d out of range 0 <= y < %d", y, pixbufHeight);
	if (y + height > pixbufHeight) g_error("preview_draw_area(): y+height:%d out of range y+height <= %d", y + height, pixbufHeight);
	if (height == 0) return; // Nothing to do

	int pixbufWidth = gdk_pixbuf_get_width(gUiData.PreviewPixbuf);

	if (x < 0 || x >= pixbufWidth) g_error("preview_draw_area(): x:%d out of range 0 <= x < %d", x, pixbufWidth);
	if (x + width > pixbufWidth) g_error("preview_draw_area(): x+width:%d out of range x+width <= %d", x + width, pixbufWidth);
	if (width == 0) return; // Nothing to do

	gboolean blinkOver = gUiConf->showOverExp &&
	                     (!gUiConf->blinkOverUnder || (gUiData.OverUnderTicker & 3) == 1);
	gboolean blinkUnder = gUiConf->showUnderExp &&
	                      (!gUiConf->blinkOverUnder || (gUiData.OverUnderTicker & 3) == 3);

	UFRectangle Crop;
	nufraw_get_scaled_crop(gUiData.UF, &Crop);
	int CropX2 = Crop.x + Crop.width;
	int CropY2 = Crop.y + Crop.height;

	int drawLines = gUiConf->gridCount + 1;

	// Scale spot image coordinates to pixbuf coordinates
	float scale_x = ((float)pixbufWidth) / gUiData.UF->rotatedWidth;
	float scale_y = ((float)pixbufHeight) / gUiData.UF->rotatedHeight;
	int SpotY1 = floor(MIN(gUiData.SpotY1, gUiData.SpotY2) * scale_y);
	int SpotY2 =  ceil(MAX(gUiData.SpotY1, gUiData.SpotY2) * scale_y);
	int SpotX1 = floor(MIN(gUiData.SpotX1, gUiData.SpotX2) * scale_x);
	int SpotX2 =  ceil(MAX(gUiData.SpotX1, gUiData.SpotX2) * scale_x);

	int rowstride = gdk_pixbuf_get_rowstride(gUiData.PreviewPixbuf);
	guint8 *pixies = gdk_pixbuf_get_pixels(gUiData.PreviewPixbuf) + x * 3;

	// This is bad. `img' should have been a parameter because we
	// cannot request an up to date buffer but it must be up to
	// date to some extend. In theory we could get the wrong buffer
	nufraw_image_data *displayImage = nufraw_get_image(gUiData.UF, nufraw_display_phase, FALSE);
	guint8 *displayPixies = displayImage->buffer + x * displayImage->depth;

	nufraw_image_data *workingImage = nufraw_get_image(gUiData.UF, nufraw_develop_phase, FALSE);
	guint8 *workingPixies = workingImage->buffer + x * workingImage->depth;

	int xx, yy, c;
	for (yy = y; yy < y + height; yy++) {

		guint8 *p8 = pixies + yy * rowstride;
		memcpy(p8, displayPixies + yy * displayImage->rowstride, width * displayImage->depth);

		if (gUiData.ChannelSelect >= 0) {
			// show grayscale for selected channel
			guint8 *p = p8;
			for (xx = 0; xx < width; xx++, p += 3) {
				guint8 px = p[gUiData.ChannelSelect];
				p[0] = p[1] = p[2] = px;
			}
		}

		guint8 *p8working = workingPixies + yy * workingImage->rowstride;
		for (xx = x; xx < x + width; xx++, p8 += 3, p8working += workingImage->depth) {

			// draw spot dashed line
			if (gUiData.SpotDraw && gUiConf->cursor_mode==1 &&
			    (((yy == SpotY1 - 1 || yy == SpotY2) && xx >= SpotX1 - 1 && xx <= SpotX2)||
			     ((xx == SpotX1 - 1 || xx == SpotX2) && yy >= SpotY1 - 1 && yy <= SpotY2))) {
				if (((xx + yy) & 7) >= 4) p8[0] = p8[1] = p8[2] = 0;
				else p8[0] = p8[1] = p8[2] = 255;
				continue;
			}

			// Draw white frame around crop area
			if (((yy == Crop.y - 1 || yy == CropY2) && xx >= Crop.x - 1 && xx <= CropX2) ||
			    ((xx == Crop.x - 1 || xx == CropX2) && yy >= Crop.y - 1 && yy <= CropY2)) {
				p8[0] = p8[1] = p8[2] = 255;
				continue;
			}

			// Shade the cropped out area
			else if (yy < Crop.y || yy >= CropY2 || xx < Crop.x || xx >= CropX2) {
				for (c = 0; c < 3; c++) p8[c] = p8[c] / 4;
				continue;
			}
			if (gUiData.RenderMode == render_default) {

				// Shade out the grid lines
				if (gUiConf->gridCount && yy > Crop.y + 1 && yy < CropY2 - 2 && xx > Crop.x + 1 && xx < CropX2 - 2) {

					int dx = (xx - Crop.x) * drawLines % Crop.width / drawLines;
					int dy = (yy - Crop.y) * drawLines % Crop.height / drawLines;
					if (dx == 0 || dy == 0) {
						p8[0] /= 2;
						p8[1] /= 2;
						p8[2] /= 2;
					} else if (dx == 1 || dy == 1) {
						p8[0] = 255 - (255 - p8[0]) / 2;
						p8[1] = 255 - (255 - p8[1]) / 2;
						p8[2] = 255 - (255 - p8[2]) / 2;
					}
				}

				// Blink the overexposed/underexposed spots
				if (blinkOver && (p8working[0] == 255 || p8working[1] == 255 || p8working[2] == 255)){
					p8[0] = p8[1] = p8[2] = 0;
				}else if (blinkUnder && (p8working[0] == 0 || p8working[1] == 0 || p8working[2] == 0)){
					p8[0] = p8[1] = p8[2] = 255;
				}

			} else if (gUiData.RenderMode == render_overexposed) {

				for (c = 0; c < 3; c++) if (p8working[c] != 255) p8[c] = 0;

			} else if (gUiData.RenderMode == render_underexposed) {

				for (c = 0; c < 3; c++) if (p8working[c] != 0) p8[c] = 255;

			} else if (gUiData.RenderMode == render_zone) {

				// very rough conversion to zones as integer values
				double yValue = 0.5;
				extern const double xyz_rgb[3][3];
				for (c = 0; c < 3; c++) yValue += xyz_rgb[1][c] * p8[c];
				yValue /= 0xFF;
				for (c = 0; c < 3; c++){
					p8[c] = (guint8)(25.5 * (int)(yValue*10+0.5));
				}
			}
		}
	}

#ifdef _OPENMP
    #pragma omp critical
	{
#endif
		// Redraw the changed areas
		GtkWidget *widget = gUiData.PreviewWidget;
		GtkImageView *view = GTK_IMAGE_VIEW(widget);
		GdkRectangle rect;
		rect.x = x;
		rect.y = y;
		rect.width = width;
		rect.height = height;
		gtk_image_view_damage_pixels(view, &rect);
#ifdef _OPENMP
	}
#endif
}

static void render_status_text()
{
    if (gUiData.ProgressBar == NULL) return;

    char progressText[UF_MAX_NAME];

    g_snprintf(progressText, UF_MAX_NAME, _("size %dx%d, scale 1/%d"), gUiData.UF->rotatedWidth,
                                                                       gUiData.UF->rotatedHeight,
                                                                       gConf->shrink);
    gtk_progress_bar_set_text(gUiData.ProgressBar, progressText);
    gtk_progress_bar_set_fraction(gUiData.ProgressBar, 0);
}

// the progress bar callback
static void preview_progress(int what, int ticks)
{
    static int last_what, todo, done;
    gboolean update = FALSE;

#ifdef _OPENMP
    #pragma omp master
#endif
    update = TRUE;
#ifdef _OPENMP
    #pragma omp critical(preview_progress)
    {
#endif
        if (ticks < 0) {
            todo = -ticks;
            done = 0;
            last_what = what;
        } else {
            if (last_what == what) done += ticks;
            else update = FALSE;
        }
#ifdef _OPENMP
    }
#endif
    // wrong thread for GTK or "what" mismatch
    if (!update) return;

    // avoid progress bar rendering hog
    if (g_timer_elapsed(gUiData.ProgressTimer, NULL) < 0.07 && ticks >= 0) return;

    g_timer_start(gUiData.ProgressTimer);

    gboolean events = TRUE;
    double start = 0.0, stop = 1.0, fraction = 0.0;
    char *text = NULL;
    switch (what) {
        case PROGRESS_WAVELET_DENOISE:
            text = _("Wavelet denoising");
            break;
        case PROGRESS_DESPECKLE:
            text = _("Despeckling");
            break;
        case PROGRESS_INTERPOLATE:
            text = _("Interpolating");
            break;
        case PROGRESS_RENDER:
            text = _("Rendering");
            stop = 0.0;
            events = FALSE;		// not needed in render_preview_image()
            break;
        case PROGRESS_LOAD:
            text = _("Loading preview");
            break;
        case PROGRESS_SAVE:
            text = _("Saving image");
            break;
    }
    if (ticks < 0 && text) gtk_progress_bar_set_text(gUiData.ProgressBar, text);
    if (!events) return;
    fraction = todo ? start + (stop - start) * done / todo : 0;
    if (fraction > stop) fraction = stop;
    gtk_progress_bar_set_fraction(gUiData.ProgressBar, fraction);

    while (gtk_events_pending()) gtk_main_iteration();
}

static void preview_progress_enable()
{
    gUiData.ProgressTimer = g_timer_new();
    nufraw_progress_cb = preview_progress;
}

static void preview_progress_disable()
{
    nufraw_progress_cb = NULL;
    if (gUiData.ProgressTimer != NULL) g_timer_destroy(gUiData.ProgressTimer);
    render_status_text();
}

static int choose_subarea(int *chosen)
{
    int subarea = -1;
    int max_area = -1;

    // First of all, find the maximally visible yet unrendered subarea.
    // Refreshing visible subareas in the first place improves visual
    // feedback and overall user experience.
    nufraw_image_data *img = nufraw_get_image(gUiData.UF, nufraw_display_phase, FALSE);

    GdkRectangle viewport;
    gtk_image_view_get_viewport(GTK_IMAGE_VIEW(gUiData.PreviewWidget), &viewport);

    int i;
    for (i = 0; i < 32; i++) {

        // Skip valid subareas
        if (img->valid & (1 << i)) continue;

        // Skip areas chosen by other threads
        if (*chosen & (1 << i)) continue;

        UFRectangle rec = nufraw_image_get_subarea_rectangle(img, i);

        gboolean noclip = TRUE;
        if (rec.x < viewport.x) {
            rec.width -= (viewport.x - rec.x);
            rec.x = viewport.x;
            noclip = FALSE;
        }
        if (rec.x + rec.width > viewport.x + viewport.width) {
            rec.width = viewport.x + viewport.width - rec.x;
            noclip = FALSE;
        }
        if (rec.y < viewport.y) {
            rec.height -= (viewport.y - rec.y);
            rec.y = viewport.y;
            noclip = FALSE;
        }
        if (rec.y + rec.height > viewport.y + viewport.height) {
            rec.height = viewport.y + viewport.height - rec.y;
            noclip = FALSE;
        }

        // Compute the visible area of the subarea
        int area = (rec.width > 0 && rec.height > 0) ? rec.width * rec.height : 0;
        if (area > max_area) {
            max_area = area;
            subarea = i;
            // If this area is fully visible, stop searching
            if (noclip) break;
        }
    }
    if (subarea >= 0) *chosen |= 1 << subarea;
    return subarea;
}

// show and hide underexposed and overexposed highlights
static gboolean switch_highlights(gpointer unused)
{
    (void)unused;

    // Redraw the highlights only in the default rendering mode.
    if (gUiData.RenderMode != render_default || gUiData.FreezeDialog) return TRUE;

    if (!ui_is_rendering()) {

        // Set the area to redraw based on the crop rectangle and view port.
        UFRectangle Crop;
        nufraw_get_scaled_crop(gUiData.UF, &Crop);
        GdkRectangle viewRect;
        gtk_image_view_get_viewport(GTK_IMAGE_VIEW(gUiData.PreviewWidget),
                                    &viewRect);
        gdouble zoom = gtk_image_view_get_zoom(GTK_IMAGE_VIEW(gUiData.PreviewWidget));

        int x1 = MAX(Crop.x, viewRect.x / zoom);
        int width = MIN(Crop.width, viewRect.width);
        int y1 = MAX(Crop.y, viewRect.y / zoom);
        int height = MIN(Crop.height, viewRect.height);

        gUiData.OverUnderTicker++;
        preview_draw_area(x1, y1, width, height);
    }

    // If no blinking is needed, disable this timeout function.
    if (!gUiConf->blinkOverUnder || (!gUiConf->showOverExp && !gUiConf->showUnderExp)) {
        gUiData.BlinkTimer = 0;
        return FALSE;
    }
    return TRUE;
}

void start_blink()
{
	if (gUiConf->blinkOverUnder && (gUiConf->showOverExp || gUiConf->showUnderExp)) {
		if (!gUiData.BlinkTimer) {
			gUiData.BlinkTimer = gdk_threads_add_timeout(UI_BLINK_MS, switch_highlights, NULL);
		}
	}
}

static void stop_blink()
{
	if (gUiData.BlinkTimer) {
		gUiData.OverUnderTicker = 0;
		switch_highlights(NULL);
		g_source_remove(gUiData.BlinkTimer);
		gUiData.BlinkTimer = 0;
	}
}

static gboolean preview_draw_crop(gpointer unused)
{
	(void)unused;

	UFRectangle Crop;
	nufraw_get_scaled_crop(gUiData.UF, &Crop);
	preview_draw_area(Crop.x, Crop.y, Crop.width, Crop.height);
	return FALSE;
}
static void render_special_mode(GtkWidget *widget, long mode)
{
	(void)widget;
	if (gUiData.FreezeDialog) return;

	gUiData.RenderMode = mode;
	preview_draw_crop(NULL);
}

static void render_init()
{
	// Check if we need a new pixbuf
	int width = gdk_pixbuf_get_width(gUiData.PreviewPixbuf);
	int height = gdk_pixbuf_get_height(gUiData.PreviewPixbuf);
	nufraw_image_data *image = nufraw_get_image(gUiData.UF, nufraw_display_phase, FALSE);
	if (width != image->width || height != image->height) {

		// Calculate current viewport center
		GdkRectangle vp;
		gtk_image_view_get_viewport(GTK_IMAGE_VIEW(gUiData.PreviewWidget), &vp);
		double xc = (vp.x + vp.width / 2.0) / width * image->width;
		double yc = (vp.y + vp.height / 2.0) / height * image->height;

		gUiData.PreviewPixbuf = gdk_pixbuf_new(GDK_COLORSPACE_RGB, FALSE, 8, image->width, image->height);

		// Clear the pixbuffer to avoid displaying garbage
		gdk_pixbuf_fill(gUiData.PreviewPixbuf, 0);
		gtk_image_view_set_pixbuf(GTK_IMAGE_VIEW(gUiData.PreviewWidget), gUiData.PreviewPixbuf, FALSE);
		g_object_unref(gUiData.PreviewPixbuf);

		// restore the viewport center
		gtk_image_view_set_offset(GTK_IMAGE_VIEW(gUiData.PreviewWidget), xc - vp.width / 2, yc - vp.height / 2, FALSE);
	}
}

// draw the spot rectangle over the preview
static void draw_spot(gboolean draw)
{
	if (gUiData.SpotX1 < 0) return;
	int width = gdk_pixbuf_get_width(gUiData.PreviewPixbuf);
	int height = gdk_pixbuf_get_height(gUiData.PreviewPixbuf);
	gUiData.SpotDraw = draw;

	// Scale spot image coordinates to pixbuf coordinates
	int SpotY1 = MAX(MIN(gUiData.SpotY1, gUiData.SpotY2) * height / gUiData.UF->rotatedHeight - 1, 0);
	int SpotY2 = MIN(MAX(gUiData.SpotY1, gUiData.SpotY2) * height / gUiData.UF->rotatedHeight + 1, height - 1);
	int SpotX1 = MAX(MIN(gUiData.SpotX1, gUiData.SpotX2) * width / gUiData.UF->rotatedWidth - 1, 0);
	int SpotX2 = MIN(MAX(gUiData.SpotX1, gUiData.SpotX2) * width / gUiData.UF->rotatedWidth + 1, width - 1);

	preview_draw_area(SpotX1, SpotY1, SpotX2 - SpotX1 + 1, 1);
	preview_draw_area(SpotX1, SpotY2, SpotX2 - SpotX1 + 1, 1);
	preview_draw_area(SpotX1, SpotY1, 1, SpotY2 - SpotY1 + 1);
	preview_draw_area(SpotX2, SpotY1, 1, SpotY2 - SpotY1 + 1);
}

// compute the spot values and update spot labels
static gboolean render_spot(gpointer unused)
{
	(void)unused;
	if (gUiData.FreezeDialog) return FALSE;

	if (gUiData.SpotX1 < 0) return FALSE;
	if (gUiData.SpotX1 >= gUiData.UF->rotatedWidth ||
	    gUiData.SpotY1 >= gUiData.UF->rotatedHeight) return FALSE;

	nufraw_image_data *de_img = nufraw_get_image(gUiData.UF, nufraw_develop_phase, TRUE);
	int width = de_img->width;
	int height = de_img->height;
	int outDepth = de_img->depth;
	void *outBuffer = de_img->buffer;

	// We assume that transform_phase and develop_phase buffer sizes are the same.
	nufraw_image_data *tr_img = nufraw_get_image(gUiData.UF, nufraw_transform_phase, TRUE);
	int rawDepth = tr_img->depth;
	void *rawBuffer = tr_img->buffer;

	struct spot spot;
	calculate_spot(&spot, width, height);

	guint64 rawSum[4], outSum[3];
	guint16 *rawPixie;
	guint8 *outPixie;

	int c, y, x;
	for (c = 0; c < 3; c++) rawSum[c] = outSum[c] = 0;

	for (y = spot.StartY; y < spot.EndY; y++) {

		rawPixie = rawBuffer + (y * width + spot.StartX) * rawDepth;
		outPixie = outBuffer + (y * width + spot.StartX) * outDepth;

		for (x = spot.StartX; x < spot.EndX; x++) {

			rawPixie += rawDepth/2;
			outPixie += outDepth;

			for (c = 0; c < gUiData.UF->colors; c++) rawSum[c] += rawPixie[c];
			for (c = 0; c < 3; c++) outSum[c] += outPixie[c];
		}
	}
	double rgb[5];
	for (c = 0; c < 3; c++) rgb[c] = outSum[c] / spot.Size;

	// Convert RGB pixel value to 0-1 space, intending to represent,
	// absent contrast manipulation and color issues, luminance relative
	// to an 18% grey card at 0.18.
	// The RGB color space is approximately linearized sRGB as it is not
	// affected from the ICC profile.
	guint16 rawChannels[4];
	for (c = 0; c < gUiData.UF->colors; c++) rawChannels[c] = rawSum[c] / spot.Size;

	guint16 linearChannels[3];
	develop_linear(rawChannels, linearChannels, gUiData.UF->developer);

	double yValue = 0.5;
	extern const double xyz_rgb[3][3];
	for (c = 0; c < 3; c++) yValue += xyz_rgb[1][c] * linearChannels[c];
	yValue /= 0xFFFF;
	if (gUiData.UF->developer->clipHighlights == film_highlights)
		yValue *= (double)gUiData.UF->developer->exposure / 0x10000;
	rgb[3] = yValue;
	rgb[4] = value2zone(yValue);

	// set labels and color in spot table
	color_labels_set(gUiData.SpotLabels, rgb);
	uf_label_set_background(gUiData.SpotPatch,rgb[0], rgb[1], rgb[2], 255);

	draw_spot(TRUE);
	return FALSE;
}

// render_preview_image() is called after all non-tiled phases are rendered.
// OpenMP notes:
// Unfortunately nufraw_convert_image_area() still has some OpenMP awareness
// which is related to OpenMP here. That should not be necessary.
static gboolean render_preview_image(gpointer unused)
{
	if (gUiData.FreezeDialog) return FALSE;

	(void)unused;
	gboolean again = FALSE;
	int chosen = 0;

	int subarea[uf_omp_get_max_threads()];
	int i;

	for (i = 0; i < uf_omp_get_max_threads(); i++) subarea[i] = -1;

#ifdef _OPENMP
	#pragma omp parallel shared(chosen,gUiData) reduction(||:again)
	{
		#pragma omp critical
#endif
		subarea[uf_omp_get_thread_num()] = choose_subarea(&chosen);

		if (subarea[uf_omp_get_thread_num()] < 0) {
			gUiData.RenderSubArea = -1;
		} else {
			nufraw_convert_image_area(gUiData.UF, subarea[uf_omp_get_thread_num()], nufraw_phases_num - 1);
			again = TRUE;
		}

#ifdef _OPENMP
	}
#endif
	nufraw_image_data *img = nufraw_get_image(gUiData.UF, nufraw_display_phase, FALSE);
	for (i = 0; i < uf_omp_get_max_threads(); i++) {
		if (subarea[i] >= 0) {
			UFRectangle area = nufraw_image_get_subarea_rectangle(img, subarea[i]);
			preview_draw_area(area.x, area.y, area.width, area.height);
			nufraw_progress_set(PROGRESS_RENDER, 1);
		}
	}
	if (!again) {
		preview_progress_disable();
		gdk_threads_add_idle_full(G_PRIORITY_DEFAULT_IDLE, (GSourceFunc)(render_raw_histogram), &gUiData, NULL);
		gdk_threads_add_idle_full(G_PRIORITY_DEFAULT_IDLE, (GSourceFunc)(render_live_histogram), &gUiData, NULL);
		gdk_threads_add_idle_full(G_PRIORITY_DEFAULT_IDLE, (GSourceFunc)(render_spot), &gUiData, NULL);
	}
	return again;
}

// refresh/update/set //////////////////////////////////////////////////////////

static void refresh_aspect()
{
	size_t i;
	for (i = 0; i < sizeof(predef_aspects) / sizeof(predef_aspects[0]); i++){

		// if current aspect ratio is in range with a predefined one
		if (gConf->aspectRatio >= predef_aspects[i].val * (1-UI_ASPECT_TOL) &&
		    gConf->aspectRatio <= predef_aspects[i].val * (1+UI_ASPECT_TOL)) {

			// set aspect ratio as string
			gUiData.FreezeDialog++;
			gtk_entry_set_text(gUiData.AspectEntry, predef_aspects[i].text);
			gUiData.FreezeDialog--;
			return;
		}
	}

	// set aspect ratio as float
	char *text = g_strdup_printf("%.4g", gConf->aspectRatio);
	gUiData.FreezeDialog++;
	gtk_entry_set_text(gUiData.AspectEntry, text);
	gUiData.FreezeDialog--;
	g_free(text);
}

static void update_shrink_ranges()
{
	if (gUiData.FreezeDialog) return;
	gUiData.FreezeDialog++;

	int croppedWidth = gConf->CropX2 - gConf->CropX1;
	int croppedHeight = gConf->CropY2 - gConf->CropY1;
	if (gUiData.shrink != 0 && fabs(gUiData.shrink - floor(gUiData.shrink + 0.0005)) < 0.0005) {
		gUiData.shrink = floor(gUiData.shrink + 0.0005);
		gUiData.height = croppedHeight / gUiData.shrink;
		gUiData.width = croppedWidth / gUiData.shrink;
	} else {
		int size = floor(MAX(gUiData.height, gUiData.width) + 0.5);
		if (size == 0 || (croppedHeight == 0 && croppedWidth == 0)){
			gUiData.height = gUiData.width = gUiData.shrink = 0;
		}else if (croppedHeight > croppedWidth) {
			gUiData.height = size;
			gUiData.width = size * croppedWidth / croppedHeight;
			gUiData.shrink = (double)croppedHeight / size;
		} else {
			gUiData.width = size;
			gUiData.height = size * croppedHeight / croppedWidth;
			gUiData.shrink = (double)croppedWidth / size;
		}
	}

	gtk_adjustment_set_upper(gUiData.HeightAdjustment,croppedHeight);
	gtk_adjustment_set_upper(gUiData.WidthAdjustment,croppedWidth);
	gtk_adjustment_set_value(gUiData.ShrinkAdjustment, gUiData.shrink);
	gtk_adjustment_set_value(gUiData.HeightAdjustment, gUiData.height);
	gtk_adjustment_set_value(gUiData.WidthAdjustment, gUiData.width);

	gUiData.FreezeDialog--;
}

static void update_crop_ranges(gboolean render)
{
	if (gUiData.FreezeDialog) return;

	// Avoid recursive handling of the same event
	gUiData.FreezeDialog++;

	gtk_adjustment_set_upper(gUiData.CropX1Adjustment, gConf->CropX2 - 1);
	gtk_adjustment_set_upper(gUiData.CropY1Adjustment, gConf->CropY2 - 1);
	gtk_adjustment_set_lower(gUiData.CropX2Adjustment,gConf->CropX1 + 1);
	gtk_adjustment_set_upper(gUiData.CropX2Adjustment,gUiData.UF->rotatedWidth);
	gtk_adjustment_set_lower(gUiData.CropY2Adjustment, gConf->CropY1 + 1);
	gtk_adjustment_set_upper(gUiData.CropY2Adjustment,gUiData.UF->rotatedHeight);

	gtk_adjustment_set_value(gUiData.CropX1Adjustment, gConf->CropX1);
	gtk_adjustment_set_value(gUiData.CropY1Adjustment, gConf->CropY1);
	gtk_adjustment_set_value(gUiData.CropX2Adjustment, gConf->CropX2);
	gtk_adjustment_set_value(gUiData.CropY2Adjustment, gConf->CropY2);

	gUiData.FreezeDialog--;

	UFRectangle Crop;
	nufraw_get_scaled_crop(gUiData.UF, &Crop);
	int CropX2 = Crop.x + Crop.width;
	int CropY2 = Crop.y + Crop.height;

	int x1[4], x2[4], y1[4], y2[4], i = 0;

	if (Crop.x != gUiData.DrawnCropX1) {
		x1[i] = MIN(Crop.x, gUiData.DrawnCropX1);
		x2[i] = MAX(Crop.x, gUiData.DrawnCropX1);
		y1[i] = MIN(Crop.y, gUiData.DrawnCropY1);
		y2[i] = MAX(CropY2, gUiData.DrawnCropY2);
		gUiData.DrawnCropX1 = Crop.x;
		i++;
	}

	if (CropX2 != gUiData.DrawnCropX2) {
		x1[i] = MIN(CropX2, gUiData.DrawnCropX2);
		x2[i] = MAX(CropX2, gUiData.DrawnCropX2);
		y1[i] = MIN(Crop.y, gUiData.DrawnCropY1);
		y2[i] = MAX(CropY2, gUiData.DrawnCropY2);
		gUiData.DrawnCropX2 = CropX2;
		i++;
	}

	if (Crop.y != gUiData.DrawnCropY1) {
		y1[i] = MIN(Crop.y, gUiData.DrawnCropY1);
		y2[i] = MAX(Crop.y, gUiData.DrawnCropY1);
		x1[i] = MIN(Crop.x, gUiData.DrawnCropX1);
		x2[i] = MAX(CropX2, gUiData.DrawnCropX2);
		gUiData.DrawnCropY1 = Crop.y;
		i++;
	}

	if (CropY2 != gUiData.DrawnCropY2) {
		y1[i] = MIN(CropY2, gUiData.DrawnCropY2);
		y2[i] = MAX(CropY2, gUiData.DrawnCropY2);
		x1[i] = MIN(Crop.x, gUiData.DrawnCropX1);
		x2[i] = MAX(CropX2, gUiData.DrawnCropX2);
		gUiData.DrawnCropY2 = CropY2;
		i++;
	}
	update_shrink_ranges();
	if (!render) return;

	// It is better not to draw lines than to draw partial lines:
	int saveDrawLines = gUiConf->gridCount;
	gUiConf->gridCount = 0;
	int pixbufHeight = gdk_pixbuf_get_height(gUiData.PreviewPixbuf);
	int pixbufWidth = gdk_pixbuf_get_width(gUiData.PreviewPixbuf);
	for (i--; i >= 0; i--) {
		x1[i] = MAX(x1[i] - 1, 0);
		x2[i] = MIN(x2[i] + 1, pixbufWidth);
		y1[i] = MAX(y1[i] - 1, 0);
		y2[i] = MIN(y2[i] + 1, pixbufHeight);
		preview_draw_area(x1[i], y1[i], x2[i] - x1[i], y2[i] - y1[i]);
	}
	gUiConf->gridCount = saveDrawLines;

	// Draw lines when idle if necessary
	if (gUiConf->gridCount > 0 && gUiData.BlinkTimer == 0) {
		//if (gUiData.DrawCropID != 0) g_source_remove(gUiData.DrawCropID);
		gUiData.DrawCropID = gdk_threads_add_idle_full(G_PRIORITY_DEFAULT_IDLE + 30,(GSourceFunc)(preview_draw_crop), &gUiData, NULL);
	}
	if (!ui_is_rendering()){
		gdk_threads_add_idle_full(G_PRIORITY_DEFAULT_IDLE,(GSourceFunc)(render_live_histogram), &gUiData, NULL);
	}
}

static void check_crop_aspect()
{
	if (gConf->CropX2 < gConf->CropX1) gConf->CropX2 = gConf->CropX1;
	if (gConf->CropY2 < gConf->CropY1) gConf->CropY2 = gConf->CropY1;
	if (gConf->CropX1 < 0) gConf->CropX1 = 0;
	if (gConf->CropX2 > gUiData.UF->rotatedWidth) gConf->CropX2 = gUiData.UF->rotatedWidth;
	if (gConf->CropY1 < 0) gConf->CropY1 = 0;
	if (gConf->CropY2 > gUiData.UF->rotatedHeight) gConf->CropY2 = gUiData.UF->rotatedHeight;
}

static void fix_crop_aspect(CursorType cursor, gboolean render)
{
	double aspect;

	check_crop_aspect();

	if (!gUiConf->lockAspect) {

		update_crop_ranges(render);
		int dy = gConf->CropY2 - gConf->CropY1;
		gConf->aspectRatio = dy ? ((gConf->CropX2 - gConf->CropX1) / (float)dy) : 1.0;
		refresh_aspect();
		return;
	}
	if (gConf->aspectRatio == 0) aspect = ((double)gUiData.UF->rotatedWidth) / gUiData.UF->rotatedHeight;
	else aspect = gConf->aspectRatio;

	switch (cursor) {

		case left_cursor:
		case right_cursor: {

				// Try to expand the rectangle evenly vertically, if space permits
				double cy = (gConf->CropY1 + gConf->CropY2) / 2.0;
				double dy = (gConf->CropX2 - gConf->CropX1) * 0.5 / aspect;
				int fix_dx = 0;
				if (dy > cy) dy = cy, fix_dx++;
				if (cy + dy > gUiData.UF->rotatedHeight) dy = gUiData.UF->rotatedHeight - cy, fix_dx++;
				if (fix_dx) {
					double dx = floor(dy * 2.0 * aspect + 0.5);
					if (cursor == left_cursor) gConf->CropX1 = gConf->CropX2 - dx;
					else gConf->CropX2 = gConf->CropX1 + dx;
				}
				gConf->CropY1 = floor(cy - dy + 0.5);
				gConf->CropY2 = floor(cy + dy + 0.5);
			}
			break;

		case top_cursor:
		case bottom_cursor: {

				// Try to expand the rectangle evenly horizontally, if space permits
				double cx = (gConf->CropX1 + gConf->CropX2) / 2.0;
				double dx = (gConf->CropY2 - gConf->CropY1) * 0.5 * aspect;
				int fix_dy = 0;
				if (dx > cx) dx = cx, fix_dy++;
				if (cx + dx > gUiData.UF->rotatedWidth) dx = gUiData.UF->rotatedWidth - cx, fix_dy++;
				if (fix_dy) {
					double dy = floor(dx * 2.0 / aspect + 0.5);
					if (cursor == top_cursor) gConf->CropY1 = gConf->CropY2 - dy;
					else gConf->CropY2 = gConf->CropY1 + dy;
				}
				gConf->CropX1 = floor(cx - dx + 0.5);
				gConf->CropX2 = floor(cx + dx + 0.5);
			}
			break;

		case top_left_cursor:
		case top_right_cursor:
		case bottom_left_cursor:
		case bottom_right_cursor: {
				// Adjust rectangle width/height according to aspect ratio
				// trying to preserve the area of the crop rectangle.
				// See the comment in set_new_aspect()
				double dy, dx = sqrt(aspect * (gConf->CropX2 - gConf->CropX1) * (gConf->CropY2 - gConf->CropY1));
				int i;
				for (i = 0; i < 20; i++) {
					if (cursor == top_left_cursor || cursor == bottom_left_cursor) {
						if (gConf->CropX2 < dx)
						dx = gConf->CropX2;
						gConf->CropX1 = gConf->CropX2 - dx;
					} else {
						if (gConf->CropX1 + dx > gUiData.UF->rotatedWidth)
						dx = gUiData.UF->rotatedWidth - gConf->CropX1;
						gConf->CropX2 = gConf->CropX1 + dx;
					}

					dy = dx / aspect;

					if (cursor == top_left_cursor || cursor == top_right_cursor) {
						if (gConf->CropY2 < dy) {
							dx = gConf->CropY2 * aspect;
							continue;
						}
						gConf->CropY1 = gConf->CropY2 - dy;
					} else {
						if (gConf->CropY1 + dy > gUiData.UF->rotatedHeight) {
							dx = (gUiData.UF->rotatedHeight - gConf->CropY1) * aspect;
							continue;
						}
						gConf->CropY2 = gConf->CropY1 + dy;
					}
					break;
				}
				if (i == 10)
				    g_warning("fix_crop_aspect(): loop not converging. "
						  "aspect:%f dx:%f dy:%f X1:%d X2:%d Y1:%d Y2:%d\n",
						  aspect, dx, dy, gConf->CropX1, gConf->CropX2,
						  gConf->CropY1, gConf->CropY2);
			}
			break;

		case move_cursor:
			break;

		default:
			g_warning("fix_crop_aspect(): unknown cursor %d", cursor);
			return;
	}
	update_crop_ranges(render);
}

// Crop and spot area cannot be rotated but we make sure their coordinates
// are valid. Try to preserve them over a rotate forth and back and try to
// preserve their geometry.
void resize_canvas()
{
	int d;

	if (gConf->autoCrop == enabled_state) gConf->autoCrop = apply_state;
	gConf->fullCrop = gConf->CropX1 == 0 && gConf->CropX2 == gUiData.UF->rotatedWidth &&
	                  gConf->CropY1 == 0 && gConf->CropY2 == gUiData.UF->rotatedHeight;

	nufraw_get_image_dimensions(gUiData.UF);

	if (gConf->fullCrop) {

		if (gUiConf->lockAspect) {

			double newAspect = (double)gUiData.UF->rotatedWidth / gUiData.UF->rotatedHeight;

			// Allow 1/aspect switch even if aspect is locked.
			if (fabs(gConf->aspectRatio - 1 / newAspect) < 0.0001) {
			    gConf->CropX2 = gUiData.UF->rotatedWidth;
			    gConf->CropY2 = gUiData.UF->rotatedHeight;
			    gConf->aspectRatio = newAspect;
			    refresh_aspect();
			}

		} else { // Keep full crop

			gConf->CropX2 = gUiData.UF->rotatedWidth;
			gConf->CropY2 = gUiData.UF->rotatedHeight;
		}
	}

	d = MIN(gConf->CropX1, gConf->CropX2 - gUiData.UF->rotatedWidth);
	if (d > 0) {
		gConf->CropX1 -= d;
		gConf->CropX2 -= d;
	}

	d = MIN(gConf->CropY1, gConf->CropY2 - gUiData.UF->rotatedHeight);
	if (d > 0) {
		gConf->CropY1 -= d;
		gConf->CropY2 -= d;
	}

	d = MIN(gUiData.SpotX1, gUiData.SpotX2 - gUiData.UF->rotatedWidth);
	if (d > 0) {
		gUiData.SpotX1 -= d;
		gUiData.SpotX2 -= d;
	}

	d = MIN(gUiData.SpotY1, gUiData.SpotY2 - gUiData.UF->rotatedHeight);
	if (d > 0) {
		gUiData.SpotY1 -= d;
		gUiData.SpotY2 -= d;
	}

	if (gUiData.SpotX2 > gUiData.UF->rotatedWidth || gUiData.SpotY2 > gUiData.UF->rotatedHeight) {
		gUiData.SpotDraw = FALSE;
		gUiData.SpotX1 = -1;
		gUiData.SpotX2 = -1;
		gUiData.SpotY1 = -1;
		gUiData.SpotY2 = -1;
	}
	if (gConf->CropX2 > gUiData.UF->rotatedWidth) fix_crop_aspect(top_right_cursor, FALSE);
	else if (gConf->CropY2 > gUiData.UF->rotatedHeight)  fix_crop_aspect(bottom_left_cursor, FALSE);
	else update_crop_ranges(FALSE);
}

static gboolean render_preview_now(gpointer unused)
{
	(void)unused;
	if (gUiData.FreezeDialog) return FALSE;

	while (g_idle_remove_by_data(&gUiData));

	gUiData.RenderSubArea = 0;

	if (gUiData.UF->Images[nufraw_transform_phase].invalidate_event) resize_canvas();

	if (gConf->autoExposure == apply_state) {

		nufraw_invalidate_layer(gUiData.UF, nufraw_develop_phase);
		nufraw_auto_expose(gUiData.UF);
		gdouble lower, upper;
		g_object_get(gUiData.ExposureAdjustment, "lower", &lower, "upper", &upper, NULL);

		// Clip the exposure to prevent a "changed" signal
		if (gConf->exposure > upper) gConf->exposure = upper;
		if (gConf->exposure < lower) gConf->exposure = lower;
		gtk_adjustment_set_value(gUiData.ExposureAdjustment, gConf->exposure);
		gtk_widget_set_sensitive(gUiData.ResetExposureButton,fabs(conf_default.exposure - gConf->exposure) > 0.001);
		if (gConf->autoBlack == enabled_state) gConf->autoBlack = apply_state;
	}

	if (gConf->autoBlack == apply_state) {

		nufraw_invalidate_layer(gUiData.UF, nufraw_develop_phase);
		nufraw_auto_black(gUiData.UF);
		curveeditor_widget_set_curve(gUiData.CurveWidget, &gConf->curve[gConf->curveIndex]);
		gtk_widget_set_sensitive(gUiData.ResetBlackButton,
		                         gConf->curve[gConf->curveIndex].m_anchors[0].x != 0.0);
	}

	if (gConf->profileIndex[display_profile] == 0) {
		uf_get_display_profile(gUiData.PreviewWidget, &gUiData.UF->displayProfile, &gUiData.UF->displayProfileSize);
	}

	if (gConf->autoCrop == apply_state) {

		nufraw_invalidate_layer(gUiData.UF, nufraw_transform_phase);
		nufraw_get_image_dimensions(gUiData.UF);

		gConf->CropX1 = (gUiData.UF->rotatedWidth - gUiData.UF->autoCropWidth) / 2;
		gConf->CropX2 = gConf->CropX1 + gUiData.UF->autoCropWidth;
		gConf->CropY1 = (gUiData.UF->rotatedHeight - gUiData.UF->autoCropHeight) / 2;
		gConf->CropY2 = gConf->CropY1 + gUiData.UF->autoCropHeight;
		update_crop_ranges(FALSE);
		gConf->autoCrop = enabled_state;
	}

	// We need to freeze the dialog in case an error message pops up in lcms error handler.
	gUiData.FreezeDialog = TRUE;
	if (nufraw_developer_prepare(gUiData.UF, display_developer)!=NUFRAW_SUCCESS){

		// reset curve
		gConf->curve[gConf->curveIndex].m_numAnchors = 2;
		gConf->curve[gConf->curveIndex].m_anchors[0].x = 0;
		gConf->curve[gConf->curveIndex].m_anchors[0].y = 0;
		gConf->curve[gConf->curveIndex].m_anchors[1].x = 1.0;
		gConf->curve[gConf->curveIndex].m_anchors[1].y = 1.0;
		curveeditor_widget_set_curve(gUiData.CurveWidget, &gConf->curve[gConf->curveIndex]);
	}
	gUiData.FreezeDialog = FALSE;
	render_init();

	// This will trigger the untiled phases if necessary. The progress bar
	// updates require gtk_main_iteration() calls which can only be
	// done when there are no pending idle tasks which could recurse
	// into nufraw_convert_image_area().
	preview_progress_enable();
	gUiData.FreezeDialog = TRUE;
	nufraw_convert_image_area(gUiData.UF, 0, nufraw_first_phase);
	gUiData.FreezeDialog = FALSE;

	preview_progress(PROGRESS_RENDER, -32);
	// Since we are already inside an idle callback, we should not use gdk_threads_add_idle_full().
	g_idle_add_full(G_PRIORITY_DEFAULT_IDLE, (GSourceFunc)(render_preview_image), &gUiData, NULL);

	if (!gFirstRender) gImageHasModifications=TRUE;
	gFirstRender=FALSE;
	return FALSE;
}

void render_preview()
{
	if (gFirstRender) return;
    while (g_idle_remove_by_data(&gUiData));
    gdk_threads_add_idle_full(G_PRIORITY_DEFAULT_IDLE,(GSourceFunc)(render_preview_now), &gUiData, NULL);
}

static void despeckle_update_sensitive()
{
    conf_data *c = gConf;
    gboolean b;
    int i;

    b = FALSE;
    for (i = 0; i < gUiData.UF->colors; ++i) {
        b |= fabs(c->despeckleWindow[i] - conf_default.despeckleWindow[i]) > 0.1;
        b |= fabs(c->despeckleDecay[i] - conf_default.despeckleDecay[i]) > 0.001;
        b |= fabs(c->despecklePasses[i] - conf_default.despecklePasses[i]) > 0.1;
    }
    gtk_widget_set_sensitive(gUiData.ResetDespeckleButton, b);
    b = FALSE;
    for (i = 1; i < gUiData.UF->colors; ++i) {
        b |= c->despeckleWindow[0] != c->despeckleWindow[i];
        b |= c->despeckleDecay[0] != c->despeckleDecay[i];
        b |= c->despecklePasses[0] != c->despecklePasses[i];
    }
    gtk_widget_set_sensitive(GTK_WIDGET(gUiData.DespeckleLockChannelsButton), !b);
}

static void set_darkframe_label()
{
	if (gConf->darkframeFile[0] != '\0') {
		char *basename = g_path_get_basename(gConf->darkframeFile);
		gtk_label_set_text(GTK_LABEL(gUiData.DarkFrameLabel), basename);
		g_free(basename);
	} else {
		// No darkframe file
		gtk_label_set_text(GTK_LABEL(gUiData.DarkFrameLabel), _("None"));
	}
}

static void restore_details_button_set()
{
	const char *state;
	switch (gConf->restoreDetails) {

		case clip_details:
			gtk_button_set_image(GTK_BUTTON(gUiData.DetailButton), gtk_image_new_from_stock(GTK_STOCK_CUT, UI_BT_SIZE));
			state = _("clip");
			break;

		case restore_lch_details:
			gtk_button_set_image(GTK_BUTTON(gUiData.DetailButton), gtk_image_new_from_stock("restore-highlights-lch", UI_BT_SIZE));
			state = _("restore in LCH space for soft details");
			break;

		case restore_hsv_details:
			gtk_button_set_image(GTK_BUTTON(gUiData.DetailButton), gtk_image_new_from_stock("restore-highlights-hsv", UI_BT_SIZE));
			state = _("restore in HSV space for sharp details");
			break;

		default:
			state = "Error";
			break;
	}
	char *text = g_strdup_printf(_("Restore details for negative EV\nCurrent state: %s"), state);
	gtk_widget_set_tooltip_text(gUiData.DetailButton, text);
	g_free(text);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gUiData.DetailButton), FALSE);
}

static void clip_highlights_button_set()
{
	const char *state;
	switch (gConf->clipHighlights) {

		case digital_highlights:
			gtk_button_set_image(GTK_BUTTON(gUiData.HighlightButton), gtk_image_new_from_stock("clip-highlights-digital", UI_BT_SIZE));
			state = _("digital linear");
			break;

		case film_highlights:
			gtk_button_set_image(GTK_BUTTON(gUiData.HighlightButton), gtk_image_new_from_stock("clip-highlights-film", UI_BT_SIZE));
			state = _("soft film like");
			break;

		default:
			state = "Error";
			break;
	}
	char *text = g_strdup_printf(_("Clip highlights for positive EV\n" "Current state: %s"), state);
	gtk_widget_set_tooltip_text(GTK_WIDGET(gUiData.HighlightButton), text);
	g_free(text);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gUiData.HighlightButton), FALSE);
}

// update the UI entries that could have changed automatically
static void update_scales()
{
	if (gUiData.FreezeDialog) return;
	gUiData.FreezeDialog = TRUE;

	int i;
	double max;

	// interpolation and smoothing
	uf_combo_box_set_data(gUiData.InterpolationCombo, &gConf->interpolation);
	gtk_toggle_button_set_active(gUiData.SmoothingButton, gConf->smoothing);

	// ICC profile
	uf_combo_box_set_data(gUiData.ProfileCombo[in_profile], &gConf->profileIndex[in_profile]);

	// Gamma and linearity
	gtk_adjustment_set_value(gUiData.GammaAdjustment, gConf->profile[0][gConf->profileIndex[0]].gamma);
	gtk_widget_set_sensitive(gUiData.ResetGammaButton,
	                         fabs(profile_default_gamma(&gConf->profile[0][gConf->profileIndex[0]])
	                         - gConf->profile[0][gConf->profileIndex[0]].gamma) > 0.001);
	gtk_adjustment_set_value(gUiData.LinearAdjustment, gConf->profile[0][gConf->profileIndex[0]].linear);
	gtk_widget_set_sensitive(gUiData.ResetLinearButton,
	                        fabs(profile_default_linear(&gConf->profile[0][gConf->profileIndex[0]])
	                        - gConf->profile[0][gConf->profileIndex[0]].linear) > 0.001);
	// base curve
	if (gConf->BaseCurveIndex > camera_curve && !has_camera_curve()){
		gtk_combo_box_set_active(gUiData.BaseCurveCombo, gConf->BaseCurveIndex - 2);
	}else{
		gtk_combo_box_set_active(gUiData.BaseCurveCombo, gConf->BaseCurveIndex);
	}
	gtk_combo_box_set_active(gUiData.CurveCombo, gConf->curveIndex);
	gtk_widget_set_sensitive(gUiData.ResetBaseCurveButton,
	                         gConf->BaseCurve[gConf->BaseCurveIndex].m_numAnchors > 2 ||
	                         gConf->BaseCurve[gConf->BaseCurveIndex].m_anchors[0].x != 0.0 ||
	                         gConf->BaseCurve[gConf->BaseCurveIndex].m_anchors[0].y != 0.0 ||
	                         gConf->BaseCurve[gConf->BaseCurveIndex].m_anchors[1].x != 1.0 ||
	                         gConf->BaseCurve[gConf->BaseCurveIndex].m_anchors[1].y != 1.0);

	// exposure
	gtk_adjustment_set_value(gUiData.ExposureAdjustment, gConf->exposure);
	gtk_widget_set_sensitive(gUiData.ResetExposureButton, fabs(conf_default.exposure - gConf->exposure) > 0.001);
	restore_details_button_set();
	clip_highlights_button_set();

	// Grayscale
	for (i = 0; i < grayscale_invalid; i++) {
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gUiData.GrayscaleButtons[i]), gConf->grayscaleMode == i);
	}
	gtk_widget_set_sensitive(GTK_WIDGET(gUiData.GrayscaleMixerTable), gConf->grayscaleMode == grayscale_mixer);
	for (i = 0; i < 3; ++i) gtk_adjustment_set_value(gUiData.GrayscaleMixers[i], gConf->grayscaleMixer[i]);
	for (max = 1, i = 0; i < 3; ++i) max = MAX(max, gConf->grayscaleMixer[i]);
	uf_label_set_background(gUiData.GrayscaleMixerColor,
	                        gConf->grayscaleMixer[0],
	                        gConf->grayscaleMixer[1],
	                        gConf->grayscaleMixer[2],
	                        max);	gtk_widget_set_sensitive(gUiData.ResetGrayscaleChannelMixerButton,
		                 (gConf->grayscaleMixer[0] != conf_default.grayscaleMixer[0])
		                 || (gConf->grayscaleMixer[1] != conf_default.grayscaleMixer[1])
		                 || (gConf->grayscaleMixer[2] != conf_default.grayscaleMixer[2]));


	// Luminosity and saturation
#ifdef NUFRAW_CONTRAST
	gtk_adjustment_set_value(gUiData.ContrastAdjustment, gConf->contrast);
	gtk_widget_set_sensitive(gUiData.ResetContrastButton, fabs(conf_default.contrast - gConf->contrast) > 0.001);
#endif
	gtk_adjustment_set_value(gUiData.SaturationAdjustment, gConf->saturation);
	gtk_widget_set_sensitive(gUiData.ResetSaturationButton, fabs(conf_default.saturation - gConf->saturation) > 0.001);

	curveeditor_widget_set_curve(gUiData.CurveWidget, &gConf->curve[gConf->curveIndex]);
	gtk_widget_set_sensitive(gUiData.ResetCurveButton,
	                         gConf->curve[gConf->curveIndex].m_numAnchors > 2 ||
	                         gConf->curve[gConf->curveIndex].m_anchors[1].x != 1.0 ||
	                         gConf->curve[gConf->curveIndex].m_anchors[1].y != 1.0 ||
	                         gConf->curve[gConf->curveIndex].m_anchors[0].y != 0.0);

	// black point adjustment
	gtk_adjustment_set_value(gUiData.BlackpointAdjustment,gConf->curve[gConf->curveIndex].m_anchors[0].x);
	gtk_widget_set_sensitive(gUiData.ResetBlackButton,
	                         gConf->curve[gConf->curveIndex].m_anchors[0].x != 0.0);

	// lightness adjustment
	for (i = 0; i < UF_MAX_LIGHTNESS_ADJUSTMENTS; i++){
		if (i < gConf->lightnessAdjustmentCount) gtk_widget_show(GTK_WIDGET(gUiData.LightnessAdjustmentTable[i]));
		else gtk_widget_hide(GTK_WIDGET(gUiData.LightnessAdjustmentTable[i]));
	}
	for (i = 0; i < gConf->lightnessAdjustmentCount; ++i) {
		gtk_adjustment_set_value(gUiData.LightnessAdjustment[i], gConf->lightnessAdjustment[i].adjustment);
		widget_set_hue(gUiData.LightnessHueSelectButton[i], gConf->lightnessAdjustment[i].hue);
		gtk_widget_set_sensitive(gUiData.ResetLightnessAdjustmentButton[i], fabs(gConf->lightnessAdjustment[i].adjustment - 1.0) >= 0.01);
	}
	gtk_widget_set_sensitive(gUiData.LightnessHueSelectNewButton, gConf->lightnessAdjustmentCount < UF_MAX_LIGHTNESS_ADJUSTMENTS);

	// crop and rotate
	gtk_widget_set_sensitive(gUiData.ResetCropButton,
	                         gConf->CropX1 != 0 ||
	                         gConf->CropY1 != 0 ||
	                         gConf->CropX2 != gUiData.UF->rotatedWidth ||
	                         gConf->CropY2 != gUiData.UF->rotatedHeight ||
	                         gConf->aspectRatio > ((float)gUiData.UF->rotatedWidth)/gUiData.UF->rotatedHeight * (1+UI_ASPECT_TOL) ||
	                         gConf->aspectRatio < ((float)gUiData.UF->rotatedWidth)/gUiData.UF->rotatedHeight * (1-UI_ASPECT_TOL) );

	// denoising
	gtk_adjustment_set_value(gUiData.ThresholdAdjustment, gConf->wdThreshold);
	gtk_widget_set_sensitive(gUiData.ResetThresholdButton, fabs(conf_default.wdThreshold - gConf->wdThreshold) > 1);
	gtk_adjustment_set_value(gUiData.HotpixelAdjustment, gConf->hotpixel);
	gtk_widget_set_sensitive(gUiData.ResetHotpixelButton, fabs(conf_default.hotpixel - gConf->hotpixel) > 0);
	set_darkframe_label();
	gtk_widget_set_sensitive(gUiData.ResetDarkframe, gConf->darkframeFile[0] != '\0');

	// despeckling
	for (i = 0; i < gUiData.UF->colors; ++i) {
		gtk_adjustment_set_value(gUiData.DespeckleWindowAdj[i], gConf->despeckleWindow[i]);
		gtk_adjustment_set_value(gUiData.DespeckleDecayAdj[i], gConf->despeckleDecay[i]);
		gtk_adjustment_set_value(gUiData.DespecklePassesAdj[i], gConf->despecklePasses[i]);
	}
	despeckle_update_sensitive();

	// output
	// TODO: gtk_adjustment_set_value(gUiData.ShrinkAdjustment,gConf->shrink);
	gtk_widget_set_sensitive(gUiData.ResetShrink, gtk_adjustment_get_value(gUiData.ShrinkAdjustment) > 1.000);

	uf_combo_box_set_data(gUiData.ProfileCombo[out_profile], &gConf->profileIndex[out_profile]);
	uf_combo_box_set_data(GTK_COMBO_BOX(gUiData.OutputIntentCombo),(int *)&gConf->intent[out_profile]);
	uf_combo_box_set_data(GTK_COMBO_BOX(gUiData.BitDepthCombo),&gConf->profile[out_profile][gConf->profileIndex[out_profile]].BitDepth);

	gUiData.FreezeDialog = FALSE;
	update_shrink_ranges();
}

// callbacks ///////////////////////////////////////////////////////////////////

static void paned_resized(GtkPaned *paned)
{
	gUiData.LeftSpace=LIM(gtk_paned_get_position(paned)-40,UI_HIS_CURVE_MINWIDTH,UI_HIS_CURVE_MAXWIDTH);

	gtk_widget_set_size_request(gUiData.RawHisto, -1, -1);
	gdk_threads_add_idle_full(G_PRIORITY_DEFAULT_IDLE,(GSourceFunc)(render_raw_histogram),&gUiData, NULL);

	gtk_widget_set_size_request(gUiData.LiveHisto, -1, -1);
	gdk_threads_add_idle_full(G_PRIORITY_DEFAULT_IDLE,(GSourceFunc)(render_live_histogram),&gUiData, NULL);

	gtk_widget_set_size_request(gUiData.BaseCurveWidget, -1, -1);
	curveeditor_widget_set_width(gUiData.LeftSpace,gUiData.BaseCurveWidget);

	gtk_widget_set_size_request(gUiData.CurveWidget, -1, -1);
	curveeditor_widget_set_width(gUiData.LeftSpace,gUiData.CurveWidget);
}

static void auto_button_toggle(GtkToggleButton *button, gboolean *valuep)
{
    if (gtk_toggle_button_get_active(button)) {
        // the button is inactive most of the time, clicking on it activates it, but
        //  we immediately deactivate it here so this function is called again recursively
        // via callback from gtk_toggle_button_set_active
        *valuep = !*valuep;
        gtk_toggle_button_set_active(button, FALSE);
    }

    // if this function is called directly, the condition above is false and we only update the button image
    if (*valuep) gtk_button_set_image(GTK_BUTTON(button), gtk_image_new_from_stock("object-automatic", UI_BT_SIZE));
    else gtk_button_set_image(GTK_BUTTON(button), gtk_image_new_from_stock("object-manual", UI_BT_SIZE));
}

static void curve_update(GtkWidget *widget, long curveType)
{
	(void)widget;
	if (curveType == base_curve) {
		gConf->BaseCurveIndex = manual_curve;
		gConf->BaseCurve[gConf->BaseCurveIndex] = *curveeditor_widget_get_curve(gUiData.BaseCurveWidget);
		if (gConf->autoExposure == enabled_state) gConf->autoExposure = apply_state;
		if (gConf->autoBlack == enabled_state) gConf->autoBlack = apply_state;
	} else {
		gConf->curveIndex = manual_curve;
		gConf->curve[gConf->curveIndex] = *curveeditor_widget_get_curve(gUiData.CurveWidget);
		gConf->autoBlack = FALSE;
		auto_button_toggle(gUiData.AutoBlackButton, &gConf->autoBlack);
	}
	nufraw_invalidate_layer(gUiData.UF, nufraw_develop_phase);
	update_scales();
	render_preview();
}

static void spot_wb_event(GtkWidget *widget, gpointer unused)
{
	(void)unused;
	(void)widget;

	struct spot spot;
	int width, height, x, y, c;
	guint64 rgb[4];

	if (gUiData.FreezeDialog) return;
	if (gUiData.SpotX1 <= 0) return;
	width = gdk_pixbuf_get_width(gUiData.PreviewPixbuf);
	height = gdk_pixbuf_get_height(gUiData.PreviewPixbuf);

	// Scale image coordinates to pixbuf coordinates
	calculate_spot(&spot, width, height);
	nufraw_image_data *image = nufraw_get_image(gUiData.UF, nufraw_transform_phase, TRUE);

	for (c = 0; c < 4; c++) rgb[c] = 0;

	for (y = spot.StartY; y < spot.EndY; y++){

		for (x = spot.StartX; x < spot.EndX; x++) {
			guint16 *pixie = (guint16*)(image->buffer + (y * image->width + x) * image->depth);
			for (c = 0; c < gUiData.UF->colors; c++) rgb[c] += pixie[c];
		}
	}

	for (c = 0; c < 4; c++) rgb[c] = MAX(rgb[c], 1);

	double chanMulArray[4];
	for (c = 0; c < gUiData.UF->colors; c++) chanMulArray[c] = (double)spot.Size * gUiData.UF->rgbMax / rgb[c];

	if (gUiData.UF->colors < 4) chanMulArray[3] = chanMulArray[1];
	UFObject *chanMul = ufgroup_element(gConf->ufobject, ufChannelMultipliers);
	ufnumber_array_set(chanMul, chanMulArray);
}

static void calculate_hue(int i)
{
	nufraw_image_data *img = nufraw_get_image(gUiData.UF, nufraw_transform_phase, FALSE);
	int width = img->width;
	int height = img->height;
	int rawDepth = img->depth;
	void *rawBuffer = img->buffer;

	struct spot spot;
	// TODO: explain and cleanup if necessary
	calculate_spot(&spot, width, height);

	guint64 rawSum[4];
	int c, y, x;
	for (c = 0; c < 3; c++) rawSum[c] = 0;

	for (y = spot.StartY; y < spot.EndY; y++) {
		guint16 *rawPixie = rawBuffer + (y * width + spot.StartX) * rawDepth;
		for (x = spot.StartX; x < spot.EndX; x++, rawPixie += rawDepth) {
			for (c = 0; c < gUiData.UF->colors; c++) rawSum[c] += rawPixie[c];
		}
	}
	guint16 rawChannels[4];
	for (c = 0; c < gUiData.UF->colors; c++) rawChannels[c] = rawSum[c] / spot.Size;

	float lch[3];
	uf_raw_to_cielch(gUiData.UF->developer, rawChannels, lch);
	gConf->lightnessAdjustment[i].hue = lch[2];

	double hue = lch[2];
	double sum = 0;
	for (y = spot.StartY; y < spot.EndY; y++) {
		guint16 *rawPixie = rawBuffer + (y * width + spot.StartX) * rawDepth;
		for (x = spot.StartX; x < spot.EndX; x++, rawPixie += rawDepth) {
			uf_raw_to_cielch(gUiData.UF->developer, rawPixie, lch);
			double diff = fabs(hue - lch[2]);
			if (diff > 180.0) diff = 360.0 - diff;
			sum += diff * diff;
		}
	}
	double stddev = sqrt(sum / spot.Size);

	// TODO: hue width adjuster
	gConf->lightnessAdjustment[i].hueWidth = MIN(stddev * 2, 60);
}

static void select_hue_event(GtkWidget *widget, gpointer index)
{
	(void)widget;
	if (gUiData.FreezeDialog) return;
	if (gUiData.SpotX1 == -1) return;

	uf_long i = (uf_long)index;

	if (i < 0) {
		if (gConf->lightnessAdjustmentCount >= UF_MAX_LIGHTNESS_ADJUSTMENTS)  return;
		i = gConf->lightnessAdjustmentCount++;
	}

	calculate_hue(i);
	widget_set_hue(gUiData.LightnessHueSelectButton[i], gConf->lightnessAdjustment[i].hue);
	gtk_widget_show_all(GTK_WIDGET(gUiData.LightnessAdjustmentTable[i]));
	update_scales();
}

static void remove_hue_event(GtkWidget *widget, gpointer index)
{
	(void)widget;
	uf_long i = (uf_long)index;

	for (; i < gConf->lightnessAdjustmentCount - 1; i++) {
		gConf->lightnessAdjustment[i] = gConf->lightnessAdjustment[i + 1];
		widget_set_hue(gUiData.LightnessHueSelectButton[i], gConf->lightnessAdjustment[i].hue);
	}

	gConf->lightnessAdjustment[i] = conf_default.lightnessAdjustment[i];
	gtk_widget_hide(GTK_WIDGET(gUiData.LightnessAdjustmentTable[i]));
	gConf->lightnessAdjustmentCount--;

	nufraw_invalidate_layer(gUiData.UF, nufraw_develop_phase);
	update_scales();
	render_preview();
}

// rescale coordinates from pixview to image reference
// regardless of zoom and pixview scroll
static void event_coordinate_rescale(gdouble *x, gdouble *y)
{
	int width = gdk_pixbuf_get_width(gUiData.PreviewPixbuf);
	int height = gdk_pixbuf_get_height(gUiData.PreviewPixbuf);

	// Find location of event in the view
	GdkRectangle viewRect;
	gtk_image_view_get_viewport(GTK_IMAGE_VIEW(gUiData.PreviewWidget),&viewRect);
	int viewWidth = gUiData.PreviewWidget->allocation.width;
	int viewHeight = gUiData.PreviewWidget->allocation.height;

	if (viewWidth < width) *x += viewRect.x;
	else  *x -= (viewWidth - width) / 2;

	if (viewHeight < height) *y += viewRect.y;
	else *y -= (viewHeight - height) / 2;

	if (*x < 0) *x = 0;
	if (*x > width) *x = width;
	if (*y < 0) *y = 0;
	if (*y > height) *y = height;

	// Scale pixbuf coordinates to image coordinates
	// gtk_image_view_get_zoom acts only when zoom>100%
	double zoom = gtk_image_view_get_zoom(GTK_IMAGE_VIEW(gUiData.PreviewWidget));
	*x = *x * gUiData.UF->rotatedWidth / width / zoom;
	*y = *y * gUiData.UF->rotatedHeight / height / zoom;
}

static gboolean preview_button_press_event(GtkWidget *widget, GdkEventButton *event, gpointer unused)
{
	(void)unused;
	(void)widget;

	if (gUiData.FreezeDialog) return FALSE;
	if (event->button != 1) return FALSE;

	//if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(gUiData.CursorButton[0]))){
	if (gUiConf->cursor_mode==0){

		// leave event management to GtkImageView
		return FALSE;

	//}else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(gUiData.CursorButton[1]))){
	}else if (gUiConf->cursor_mode==1){

		event_coordinate_rescale(&event->x, &event->y);
		draw_spot(FALSE);
		gUiData.SpotX1 = gUiData.SpotX2 = event->x;
		gUiData.SpotY1 = gUiData.SpotY2 = event->y;
		if (!ui_is_rendering()){
			gdk_threads_add_idle_full(G_PRIORITY_DEFAULT_IDLE, (GSourceFunc)(render_spot), &gUiData, NULL);
		}
	}
	gUiData.PreviewButtonPressed = TRUE;
	return TRUE;
}

static gboolean preview_button_release_event(GtkWidget *widget, GdkEventButton *event, gpointer unused)
{
	(void)unused;
	(void)widget;

	if (event->button != 1) return FALSE;
	gUiData.PreviewButtonPressed = FALSE;
	return TRUE;
}

static gboolean crop_motion_notify(GdkEventMotion *event)
{
	event_coordinate_rescale(&event->x, &event->y);
	int x = event->x;
	int y = event->y;
	int pixbufHeight = gdk_pixbuf_get_height(gUiData.PreviewPixbuf);
	int pixbufWidth = gdk_pixbuf_get_width(gUiData.PreviewPixbuf);
	int sideSizeX = MIN(16 * gUiData.UF->rotatedWidth / pixbufWidth,(gConf->CropX2 - gConf->CropX1) / 3);
	int sideSizeY = MIN(16 * gUiData.UF->rotatedHeight / pixbufHeight,(gConf->CropY2 - gConf->CropY1) / 3);

	if ((event->state & GDK_BUTTON1_MASK) == 0) {

		// While mouse button is not clicked we set the cursor type
		// according to mouse pointer location.
		const CursorType tr_cursor[16] = {
			crop_cursor, crop_cursor, crop_cursor, crop_cursor,
			crop_cursor, top_left_cursor, left_cursor, bottom_left_cursor,
			crop_cursor, top_cursor, move_cursor, bottom_cursor,
			crop_cursor, top_right_cursor, right_cursor, bottom_right_cursor
		};

		int sel_cursor = 0;

		if (y >= gConf->CropY1 - 1) {
		if (y < gConf->CropY1 + sideSizeY) sel_cursor |= 1;
		else if (y <= gConf->CropY2 - sideSizeY) sel_cursor |= 2;
		else if (y <= gConf->CropY2) sel_cursor |= 3;
		}
		if (x >= gConf->CropX1 - 1) {
		if (x < gConf->CropX1 + sideSizeX) sel_cursor |= 4;
		else if (x <= gConf->CropX2 - sideSizeX) sel_cursor |= 8;
		else if (x <= gConf->CropX2) sel_cursor |= 12;
		}
		gUiData.CropMotionType = tr_cursor[sel_cursor];

		gdk_window_set_cursor(gUiData.PreviewEventBox->window,gUiData.Cursor[gUiData.CropMotionType]);

	} else {

		// While mouse button is clicked we change crop according to cursor
		// type and mouse pointer location.
		if (gUiData.CropMotionType == top_cursor || gUiData.CropMotionType == top_left_cursor || gUiData.CropMotionType == top_right_cursor)
		if (y < gConf->CropY2) gConf->CropY1 = y;

		if (gUiData.CropMotionType == bottom_cursor || gUiData.CropMotionType == bottom_left_cursor || gUiData.CropMotionType == bottom_right_cursor)
		if (y > gConf->CropY1) gConf->CropY2 = y;

		if (gUiData.CropMotionType == left_cursor || gUiData.CropMotionType == top_left_cursor || gUiData.CropMotionType == bottom_left_cursor)
		if (x < gConf->CropX2) gConf->CropX1 = x;

		if (gUiData.CropMotionType == right_cursor || gUiData.CropMotionType == top_right_cursor || gUiData.CropMotionType == bottom_right_cursor)
		if (x > gConf->CropX1) gConf->CropX2 = x;

		if (gUiData.CropMotionType == move_cursor) {

			int d = x - gUiData.OldMouseX;
			if (gConf->CropX1 + d < 0)  d = -gConf->CropX1;
			if (gConf->CropX2 + d >= gUiData.UF->rotatedWidth) d = gUiData.UF->rotatedWidth - gConf->CropX2;
			gConf->CropX1 += d;
			gConf->CropX2 += d;

			d = y - gUiData.OldMouseY;
			if (gConf->CropY1 + d < 0)
			    d = -gConf->CropY1;
			if (gConf->CropY2 + d >= gUiData.UF->rotatedHeight)
			    d = gUiData.UF->rotatedHeight - gConf->CropY2;
			gConf->CropY1 += d;
			gConf->CropY2 += d;
		}

		if (gUiData.CropMotionType != crop_cursor) {

			fix_crop_aspect(gUiData.CropMotionType, TRUE);
			gConf->autoCrop = disabled_state;
			auto_button_toggle(gUiData.AutoCropButton, &gConf->autoCrop);
		}
		update_scales();
	}

	gUiData.OldMouseX = x;
	gUiData.OldMouseY = y;
	return TRUE;
}

static gboolean preview_motion_notify_event(GtkWidget *event_box, GdkEventMotion *event, gpointer unused)
{
	(void)unused;
	if (!gtk_event_box_get_above_child(GTK_EVENT_BOX(event_box))) return FALSE;

	//if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(gUiData.CursorButton[2]))){
	if (gUiConf->cursor_mode==2){
		return crop_motion_notify(event);
	}

	if ((event->state & GDK_BUTTON1_MASK) == 0) return FALSE;
	if (!gUiData.PreviewButtonPressed) return FALSE;

	draw_spot(FALSE);
	event_coordinate_rescale(&event->x, &event->y);
	gUiData.SpotX2 = event->x;
	gUiData.SpotY2 = event->y;

	if (!ui_is_rendering()){
		gdk_threads_add_idle_full(G_PRIORITY_DEFAULT_IDLE,  (GSourceFunc)(render_spot), &gUiData, NULL);
	}
	return TRUE;
}

static gboolean(*gtk_image_view_scroll_event)(GtkWidget *widget,GdkEventScroll *event);
static gboolean preview_scroll_event(GtkWidget *widget, GdkEventScroll *event)
{
	// GtkImageView only knows how to handle scroll up or down
	// We also disable Ctrl+scroll which activates the zoom
	if ((event->direction == GDK_SCROLL_UP || event->direction == GDK_SCROLL_DOWN) && event->state & GDK_CONTROL_MASK)
		(*gtk_image_view_scroll_event)(widget, event);

	return TRUE;
}

static void flip_image(GtkWidget *widget, int flip)
{
	(void)widget;
	int oldOrientation = gConf->orientation;
	double oldAngle = gConf->rotationAngle;
	nufraw_flip_orientation(gUiData.UF, flip);

	gUiData.FreezeDialog++;
	nufraw_unnormalize_rotation(gUiData.UF);
	gtk_adjustment_set_value(gUiData.RotationAdjustment, gConf->rotationAngle);
	gUiData.UnnormalizedOrientation = gConf->orientation;
	nufraw_normalize_rotation(gUiData.UF);
	gUiData.FreezeDialog--;

	gConf->orientation = oldOrientation;
	gConf->rotationAngle = oldAngle;
	gtk_adjustment_value_changed(gUiData.RotationAdjustment);
}

// Modify current crop area so that it fits current aspect ratio
static void set_new_aspect()
{
	float cx, cy, dx, dy;

	// Crop area center never changes
	cx = (gConf->CropX1 + gConf->CropX2) / 2.0;
	cy = (gConf->CropY1 + gConf->CropY2) / 2.0;

	///Adjust the current crop area width/height taking into account
	// the new aspect ratio. The rule is to keep one of the dimensions
	// and modify other dimension in such a way that the area of the
	// new crop area will be maximal.
	dx = gConf->CropX2 - cx;
	dy = gConf->CropY2 - cy;
	if (dx / dy > gConf->aspectRatio) dy = dx / gConf->aspectRatio;
	else dx = dy * gConf->aspectRatio;

	if (dx > cx) {
		dx = cx;
		dy = dx / gConf->aspectRatio;
	}

	if (cx + dx > gUiData.UF->rotatedWidth) {
		dx = gUiData.UF->rotatedWidth - cx;
		dy = dx / gConf->aspectRatio;
	}

	if (dy > cy) {
		dy = cy;
		dx = dy * gConf->aspectRatio;
	}

	if (cy + dy > gUiData.UF->rotatedHeight) {
		dy = gUiData.UF->rotatedHeight - cy;
		dx = dy * gConf->aspectRatio;
	}

	gConf->CropX1 = floor(cx - dx + 0.5);
	gConf->CropX2 = floor(cx + dx + 0.5);
	gConf->CropY1 = floor(cy - dy + 0.5);
	gConf->CropY2 = floor(cy + dy + 0.5);

	update_crop_ranges(TRUE);
}

static void lock_aspect(GtkToggleButton *button, gboolean *valuep)
{
	if (gtk_toggle_button_get_active(button)) {
		*valuep = !*valuep;
		gtk_toggle_button_set_active(button, FALSE);
	}

	if (*valuep) {
		gtk_button_set_image(GTK_BUTTON(button), gtk_image_new_from_stock("object-lock", UI_BT_SIZE));
		gtk_widget_set_tooltip_text(GTK_WIDGET(button), _("Aspect ratio locked, click to unlock"));
	} else {
		gtk_button_set_image(GTK_BUTTON(button), gtk_image_new_from_stock("object-unlock", UI_BT_SIZE));
		gtk_widget_set_tooltip_text(GTK_WIDGET(button), _("Aspect ratio unlocked, click to lock"));
	}
}

static void aspect_modify(GtkWidget *widget)
{
	(void)widget;
	if (gUiData.FreezeDialog) return;

	const gchar *text = gtk_entry_get_text(gUiData.AspectEntry);

	if (text) {
		float aspect = 0.0, aspect_div, aspect_quot;
		if (sscanf(text, "%f : %f", &aspect_div, &aspect_quot) == 2) {
			if (aspect_quot) aspect = aspect_div / aspect_quot;
		} else {
			sscanf(text, "%g", &aspect);
		}
		if (aspect >= 0.1 && aspect <= 10.0) gConf->aspectRatio = aspect;
	}
	set_new_aspect();
	gUiConf->lockAspect = TRUE;
	lock_aspect(gUiData.LockAspectButton, &gUiConf->lockAspect);

	if (gConf->autoCrop == enabled_state) {
		gConf->autoCrop = apply_state;
		update_scales();
	}
}

// callbacks ///////////////////////////////////////////////////////////////////

static void cursor_button_cb(GtkWidget *button, gpointer unused)
{
	(void)unused;
	if (!gUiData.PreviewEventBox) return;

	int i;
	for (i = 0; i < 3; ++i){
		if (button == gUiData.CursorButton[i] && gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button))){
			gUiConf->cursor_mode = i;
		}
	}

	if (gUiConf->cursor_mode==0){  // DRAG

		draw_spot(FALSE);

		// leave event to GtkImageView
		gtk_event_box_set_above_child(GTK_EVENT_BOX(gUiData.PreviewEventBox), FALSE);
		gUiConf->cursor_mode=0;

	}else if (gUiConf->cursor_mode==1){  // SPOT

		draw_spot(TRUE);

		// set cursor mode spot
		gtk_event_box_set_above_child(GTK_EVENT_BOX(gUiData.PreviewEventBox), TRUE);
		gdk_window_set_cursor(gUiData.PreviewEventBox->window, gUiData.Cursor[spot_cursor]);
		gUiConf->cursor_mode=1;


	}else if (gUiConf->cursor_mode==2){  // CROP

		draw_spot(FALSE);

		// set cursor mode crop
		gtk_event_box_set_above_child(GTK_EVENT_BOX(gUiData.PreviewEventBox), TRUE);
		gdk_window_set_cursor(gUiData.PreviewEventBox->window, gUiData.Cursor[crop_cursor]);
		gUiConf->cursor_mode=2;
	}
}

static void crop_reset(GtkWidget *widget, gpointer unused)
{
	(void)unused;

	gConf->CropX1 = 0;
	gConf->CropY1 = 0;
	gConf->CropX2 = gUiData.UF->rotatedWidth;
	gConf->CropY2 = gUiData.UF->rotatedHeight;
	gConf->aspectRatio = ((float)gUiData.UF->rotatedWidth)/gUiData.UF->rotatedHeight;

	refresh_aspect();
	update_crop_ranges(widget!=NULL); // do not render if called by ResetAllButton
	gConf->fullCrop = enabled_state;
	gConf->autoCrop = disabled_state;
	auto_button_toggle(gUiData.AutoCropButton, &gConf->autoCrop);

	update_scales();
}

static void zoom_update(GtkWidget *widget, gpointer unused)
{
	(void)unused;
	(void)widget;
	if (gUiData.FreezeDialog) return;

	double oldZoom = gUiConf->Zoom;
	gUiConf->Zoom = gtk_adjustment_get_value(gUiData.ZoomAdjustment);
	if (gUiConf->Zoom == oldZoom) return;

	// Set shrink/size values for preview rendering
	gConf->shrink = zoom_to_shrink(gUiConf->Zoom);
	update_size();
	render_status_text();
	gtk_image_view_set_zoom(GTK_IMAGE_VIEW(gUiData.PreviewWidget), MAX(gUiConf->Zoom, 100.0) / 100.0);

	if (gConf->shrink != zoom_to_shrink(oldZoom)) {
		nufraw_invalidate_layer(gUiData.UF, nufraw_first_phase);
		gFirstRender=TRUE; // to avoid set gImageHasModifications
		render_preview_now(NULL);
	}
}

static void zoom_in_event(GtkWidget *widget, gpointer unused)
{
	(void)unused;
	(void)widget;
	if (gUiData.FreezeDialog) return;

	double zoom = gUiConf->Zoom;

	if (zoom >= 100.0) zoom = (floor((zoom + 0.5) / 100.0) + 1) * 100.0;
	else zoom = MIN(MAX(floor(100.0 / (zoom_to_shrink(zoom)-1)), zoom + 1), 100.0);

	gtk_adjustment_set_value(gUiData.ZoomAdjustment, zoom);
}

static void zoom_out_event(GtkWidget *widget, gpointer unused)
{
	(void)unused;
	(void)widget;
	if (gUiData.FreezeDialog) return;

	double zoom = gUiConf->Zoom;

	if (zoom > 100.0) zoom = (ceil((zoom - 0.5) / 100.0) - 1) * 100.0;
	else zoom = floor(100.0 / (zoom_to_shrink(zoom)+1));

	gtk_adjustment_set_value(gUiData.ZoomAdjustment, zoom);
}

static void zoom_fit_event(GtkWidget *widget, gpointer unused)
{
	(void)unused;
	(void)widget;
	if (gUiData.FreezeDialog) return;

	int previewWidth = gUiData.PreviewScroll->allocation.width;
	int previewHeight = gUiData.PreviewScroll->allocation.height;

	// zoom to cropped area
	double wScale = (double)(gConf->CropX2 - gConf->CropX1) / previewWidth;
	double hScale = (double)(gConf->CropY2 - gConf->CropY1) / previewHeight;

	double Scale = ceil(MAX(wScale, hScale));
	gtk_adjustment_set_value(gUiData.ZoomAdjustment, 100.0/Scale );

	// scroll to cropped area
	gFirstRender=TRUE; // to avoid set gImageHasModifications
	render_preview_now(NULL);
	double sx,sy;
	sx=(gConf->CropX1+gConf->CropX2)/(2*Scale)-(previewWidth/2);
	sy=(gConf->CropY1+gConf->CropY2)/(2*Scale)-(previewHeight/2);

	sx=LIM(sx,0,gUiData.UF->rotatedWidth/Scale-previewWidth);
	sy=LIM(sy,0,gUiData.UF->rotatedHeight/Scale-previewHeight);

	gtk_adjustment_set_value(GTK_IMAGE_VIEW(gUiData.PreviewWidget)->hadj, sx);
	gtk_adjustment_set_value(GTK_IMAGE_VIEW(gUiData.PreviewWidget)->vadj, sy);
}

static void zoom_100_event(GtkWidget *widget, gpointer unused)
{
	if (gUiData.FreezeDialog) return;

	(void)unused;
	(void)widget;

	gtk_adjustment_set_value(gUiData.ZoomAdjustment, 100.0);
}

static int aspect_activate(GtkWidget *widget, gpointer unused)
{
	(void)unused;
	aspect_modify(widget);
	return FALSE;
}

static void aspect_changed(GtkWidget *widget, gpointer unused)
{
	(void)unused;
	if (gtk_combo_box_get_active((GtkComboBox *)widget) >= 0) aspect_modify(widget);
}

static void grayscale_update(GtkWidget *button, gpointer unused)
{
	(void)unused;
	int i;

	for (i = 0; i < grayscale_invalid; ++i){
		if (button == gUiData.GrayscaleButtons[i] && gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button))){
			gConf->grayscaleMode = i;
			nufraw_invalidate_layer(gUiData.UF, nufraw_develop_phase);
			update_scales();
			render_preview();
		}
	}
}

static void adjustment_reset_rotation(GtkWidget *widget, gpointer unused)
{
	(void)unused;
	(void)widget;

	int oldOrientation = gConf->orientation;
	double oldAngle = gConf->rotationAngle;
	gConf->orientation = gConf->CameraOrientation;
	gConf->rotationAngle = 0;

	gUiData.FreezeDialog++;
	nufraw_unnormalize_rotation(gUiData.UF);
	gtk_adjustment_set_value(gUiData.RotationAdjustment, gConf->rotationAngle);
	gUiData.UnnormalizedOrientation = gConf->orientation;
	nufraw_normalize_rotation(gUiData.UF);
	gUiData.FreezeDialog--;

	gConf->orientation = oldOrientation;
	gConf->rotationAngle = oldAngle;
	gtk_adjustment_value_changed(gUiData.RotationAdjustment);
}

static void set_darkframe()
{
	set_darkframe_label();
	nufraw_invalidate_darkframe_layer(gUiData.UF);
	render_preview();
}

static void load_darkframe(GtkWidget *widget, void *unused)
{
	(void)unused;

	GtkFileChooser *fileChooser;
	char *basedir;

	if (gUiData.FreezeDialog) return;
	basedir = g_path_get_dirname(gConf->darkframeFile);
	fileChooser = nufraw_raw_chooser(gConf, gUiConf, basedir, _("Load dark frame"),
	                                 GTK_WINDOW(gtk_widget_get_toplevel(widget)),
	                                 GTK_STOCK_CANCEL, FALSE, FALSE);
	free(basedir);

	if (gtk_dialog_run(GTK_DIALOG(fileChooser)) == GTK_RESPONSE_ACCEPT) {

		char *filename = gtk_file_chooser_get_filename(fileChooser);
		g_strlcpy(gConf->darkframeFile, filename, UF_MAX_PATH);
		g_free(filename);
		nufraw_load_darkframe(gUiData.UF);
		set_darkframe();
	}
	nufraw_focus(fileChooser, FALSE);
	gtk_widget_destroy(GTK_WIDGET(fileChooser));
}

static void reset_darkframe(GtkWidget *widget, void *unused)
{
	(void)unused;
	(void)widget;

	if (gUiData.FreezeDialog) return;
	if (gConf->darkframe == NULL) return;

	nufraw_close_darkframe(gConf);
	set_darkframe();
}

static void toggle_button_cb(GtkToggleButton *button, gboolean *valuep)
{
	if (valuep == &gConf->restoreDetails) {

		// Our untoggling of the button creates a redundant event.
		if (gtk_toggle_button_get_active(button)) {

			// Scroll through the settings.
			gConf->restoreDetails = (gConf->restoreDetails + 1) % restore_types;
			restore_details_button_set();
			nufraw_invalidate_layer(gUiData.UF, nufraw_develop_phase);
			update_scales();
			render_preview();
		}
	} else if (valuep == &gConf->clipHighlights) {

		// Our untoggling of the button creates a redundant event
		if (gtk_toggle_button_get_active(button)) {

			// Scroll through the settings
			gConf->clipHighlights = (gConf->clipHighlights + 1) % highlights_types;
			clip_highlights_button_set();
			nufraw_invalidate_layer(gUiData.UF, nufraw_develop_phase);
			update_scales();
			render_preview();
		}

	} else if (valuep == (void*)gUiData.ChannelSelectButton) {

		if (gUiData.ChannelSelect >= -1) {

			int i, b = 0;
			while (gUiData.ChannelSelectButton[b] != button) ++b;

			if (gtk_toggle_button_get_active(button)) {

				// ignore generated events, for render_preview()
				gUiData.ChannelSelect = -2;
				for (i = 0; i < gUiData.UF->colors; ++i)
				if (i != b)
				gtk_toggle_button_set_active(
				gUiData.ChannelSelectButton[i], FALSE);
				gUiData.ChannelSelect = b;

			} else {
				gUiData.ChannelSelect = -1;
			}
			nufraw_invalidate_layer(gUiData.UF, nufraw_develop_phase);
			render_preview();
		}

	} else {

		*valuep = gtk_toggle_button_get_active(button);

		if (valuep == &gUiConf->showOverExp || valuep == &gUiConf->showUnderExp) {

			start_blink();
			switch_highlights(NULL);

		} else if (valuep == &gConf->smoothing) {

			nufraw_invalidate_smoothing_layer(gUiData.UF);
			render_preview();

		} else if (valuep == &gUiData.UF->mark_hotpixels) {

			if (gUiData.UF->hotpixels) {
				nufraw_invalidate_hotpixel_layer(gUiData.UF);
				render_preview();
			}
		}
	}
}

static void grayscale_filter_clicked(GtkWidget *unused, gpointer index)
{
	(void)unused;
	long i=(long)index;

	gConf->grayscaleMixer[0] = grayscale_filters[i][0];
	gConf->grayscaleMixer[1] = grayscale_filters[i][1];
	gConf->grayscaleMixer[2] = grayscale_filters[i][2];

	nufraw_invalidate_layer(gUiData.UF, nufraw_develop_phase);
	update_scales();
	render_preview();
}

static void reset_button_cb(GtkWidget *button, gpointer unused)
{
	(void)unused;
	int i;

	// INPUT /////////////////////////////////////////////////////////////////

	if (button==gUiData.ResetAllButton) {

		// interpolation and smoothing
		gConf->interpolation=conf_default.interpolation;
		gConf->smoothing=conf_default.smoothing;

		// input profile
		gConf->profileIndex[in_profile]=conf_default.profileIndex[in_profile];
	}

	if ((button == gUiData.ResetGammaButton)||(button==gUiData.ResetAllButton)) {
		gConf->profile[0][gConf->profileIndex[0]].gamma = profile_default_gamma(&gConf->profile[0][gConf->profileIndex[0]]);
	}

	if ((button == gUiData.ResetLinearButton)||(button==gUiData.ResetAllButton)) {
		gConf->profile[0][gConf->profileIndex[0]].linear = profile_default_linear(&gConf->profile[0][gConf->profileIndex[0]]);
	}

	if ((button == gUiData.ResetBaseCurveButton)||(button==gUiData.ResetAllButton)) {

		if (gConf->BaseCurveIndex == manual_curve) {
			gConf->BaseCurve[gConf->BaseCurveIndex].m_numAnchors = 2;
			gConf->BaseCurve[gConf->BaseCurveIndex].m_anchors[0].x = 0.0;
			gConf->BaseCurve[gConf->BaseCurveIndex].m_anchors[0].y = 0.0;
			gConf->BaseCurve[gConf->BaseCurveIndex].m_anchors[1].x = 1.0;
			gConf->BaseCurve[gConf->BaseCurveIndex].m_anchors[1].y = 1.0;
		} else {
			gConf->BaseCurveIndex = linear_curve;
		}
		curveeditor_widget_set_curve(gUiData.BaseCurveWidget, &gConf->BaseCurve[gConf->BaseCurveIndex]);
	}

	// EXPOSURE //////////////////////////////////////////////////////////////

	if ((button == gUiData.ResetExposureButton)||(button==gUiData.ResetAllButton)) {
		gConf->exposure = conf_default.exposure;
		gConf->autoExposure = FALSE;
		auto_button_toggle(gUiData.AutoExposureButton, &gConf->autoExposure);
		gConf->restoreDetails = conf_default.restoreDetails;
		gConf->clipHighlights = conf_default.clipHighlights;
	}

	// GRAYSCALE /////////////////////////////////////////////////////////////

	if ((button == gUiData.ResetGrayscaleChannelMixerButton)||(button==gUiData.ResetAllButton)) {
		gConf->grayscaleMixer[0] = conf_default.grayscaleMixer[0];
		gConf->grayscaleMixer[1] = conf_default.grayscaleMixer[1];
		gConf->grayscaleMixer[2] = conf_default.grayscaleMixer[2];
	}

	// LUMINOSITY AND SATURATION /////////////////////////////////////////////

#ifdef NUFRAW_CONTRAST
	if ((button == gUiData.ResetContrastButton)||(button==gUiData.ResetAllButton)) {
		gConf->contrast = conf_default.contrast;
	}

#endif

	if ((button == gUiData.ResetSaturationButton)||(button==gUiData.ResetAllButton)) {
		gConf->saturation = conf_default.saturation;
	}

	if ((button == gUiData.ResetBlackButton)||(button==gUiData.ResetAllButton)) {
		CurveDataSetPoint(&gConf->curve[gConf->curveIndex],0,conf_default.black,gConf->curve[gConf->curveIndex].m_anchors[0].y );
		gConf->autoBlack = FALSE;
		auto_button_toggle(gUiData.AutoBlackButton, &gConf->autoBlack);
		gConf->black = conf_default.black;
	}

	if (button == gUiData.AutoCurveButton){
		gConf->curveIndex = manual_curve;
		nufraw_auto_curve(gUiData.UF);
		gConf->autoBlack = enabled_state;
		auto_button_toggle(gUiData.AutoBlackButton, &gConf->autoBlack);
	}

	if ((button == gUiData.ResetCurveButton)||(button==gUiData.ResetAllButton)) {
		if (gConf->curveIndex == manual_curve) {
			gConf->curve[gConf->curveIndex].m_numAnchors = 2;
			gConf->curve[gConf->curveIndex].m_anchors[0].y = 0;
			gConf->curve[gConf->curveIndex].m_anchors[1].x = 1.0;
			gConf->curve[gConf->curveIndex].m_anchors[1].y = 1.0;
		} else {
			gConf->curveIndex = linear_curve;
		}
	}

	// LIGTHNESS ADJUSTMENTS /////////////////////////////////////////////////

	for (i = 0; i < UF_MAX_LIGHTNESS_ADJUSTMENTS; ++i) {
		if (button == gUiData.ResetLightnessAdjustmentButton[i]) {
			gConf->lightnessAdjustment[i].adjustment = 1.0;
		}
	}

	// DENOISING /////////////////////////////////////////////////////////////

	if ((button == gUiData.ResetThresholdButton)||(button==gUiData.ResetAllButton)) {
		gConf->wdThreshold = conf_default.wdThreshold;
		nufraw_invalidate_denoise_layer(gUiData.UF);
	}
	if ((button == gUiData.ResetHotpixelButton)||(button==gUiData.ResetAllButton)) {
		gConf->hotpixel = conf_default.hotpixel;
		nufraw_invalidate_hotpixel_layer(gUiData.UF);
	}

	// DESPECKILNG ///////////////////////////////////////////////////////////

	if ((button == gUiData.ResetDespeckleButton)||(button==gUiData.ResetAllButton)) {
		memcpy(gConf->despeckleWindow, conf_default.despeckleWindow, sizeof(gConf->despeckleWindow));
		memcpy(gConf->despeckleDecay, conf_default.despeckleDecay,  sizeof(gConf->despeckleDecay));
		memcpy(gConf->despecklePasses, conf_default.despecklePasses, sizeof(gConf->despecklePasses));
		nufraw_invalidate_despeckle_layer(gUiData.UF);
	}

	// OUTPUT ////////////////////////////////////////////////////////////////

	if ((button==gUiData.ResetShrink)||(button==gUiData.ResetAllButton)) {
		gtk_adjustment_set_value(gUiData.ShrinkAdjustment,conf_default.shrink);
	}

	// output profile, intent, bitDepth
	if (button==gUiData.ResetAllButton) {
		gConf->profileIndex[out_profile]=conf_default.profileIndex[out_profile];
		gConf->intent[out_profile]=conf_default.intent[out_profile];
		gConf->profile[out_profile][gConf->profileIndex[out_profile]].BitDepth=conf_default.profile[out_profile][gConf->profileIndex[out_profile]].BitDepth;
	}

	if (button==gUiData.ResetAllButton){

		while (gConf->lightnessAdjustmentCount>0)
			remove_hue_event(gUiData.LightnessHueRemoveButton[gConf->lightnessAdjustmentCount-1]
				    ,(gpointer)((long)gConf->lightnessAdjustmentCount-1));

		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gUiData.GrayscaleButtons[0]),TRUE);
		grayscale_update(gUiData.GrayscaleButtons[0],NULL);

		adjustment_reset_rotation(gUiData.ResetRotationAdjustment, NULL);
		ufobject_reset_clicked(gUiData.ResetWbButton);

#ifdef HAVE_LENSFUN
		ufobject_reset_clicked(gUiData.ResetLens);
#endif

		reset_darkframe(gUiData.ResetDarkframe, NULL);

		if (gUiData.ChannelSelect>=0){
			gtk_toggle_button_set_active(gUiData.ChannelSelectButton[gUiData.ChannelSelect], FALSE);
			gUiData.ChannelSelect = -1;
		}
		crop_reset(NULL, NULL);
	}

	if (gConf->autoExposure == enabled_state) gConf->autoExposure = apply_state;
	if (gConf->autoBlack == enabled_state) gConf->autoBlack = apply_state;
	nufraw_invalidate_layer(gUiData.UF, nufraw_develop_phase);
	update_scales();
	render_preview();
}

static void auto_button_toggled(GtkToggleButton *button, gboolean *valuep)
{
	auto_button_toggle(button, valuep);

	if (*valuep == enabled_state) {
		*valuep = apply_state;
		render_preview();
	}
}

static void adjustment_update_int(GtkAdjustment *adj, int *valuep)
{
	int value = (int)floor(gtk_adjustment_get_value(adj) + 0.5);
	if (value == *valuep) return;
	*valuep = value;

	if (gUiData.FreezeDialog) return;

	if (valuep == &gUiConf->gridCount) {
		//if (gUiData.DrawCropID != 0) g_source_remove(gUiData.DrawCropID);
		gUiData.DrawCropID = gdk_threads_add_idle_full(G_PRIORITY_DEFAULT_IDLE + 30,(GSourceFunc)(preview_draw_crop), &gUiData, NULL);
	}
}

static gboolean despeckle_adjustment_update(double *p)
{
	conf_data *c = gConf;
	int i, j, match;
	GtkAdjustment **adjp = NULL;

	match = 0;
	for (i = 0; i < gUiData.UF->colors; ++i) {

		if (p == &c->despeckleWindow[i]) {
			adjp = gUiData.DespeckleWindowAdj;
			match = c->despecklePasses[i] ? 1 : -1;
			break;
		}
		if (p == &c->despeckleDecay[i]) {
			adjp = gUiData.DespeckleDecayAdj;
			match = c->despeckleWindow[i] && c->despecklePasses[i] ? 1 : -1;
			break;
		}
		if (p == &c->despecklePasses[i]) {
			adjp = gUiData.DespecklePassesAdj;
			match = c->despeckleWindow[i] ? 1 : -1;
			break;
		}
	}
	if (match > 0) despeckle_apply_constraints(adjp, i);
	if (match) {
		if (gtk_toggle_button_get_active(gUiData.DespeckleLockChannelsButton)) {
			gUiData.FreezeDialog++;
			for (j = 0; j < gUiData.UF->colors; ++j) {
				p[j - i] = *p;
				gtk_adjustment_set_value(adjp[j], *p);
				despeckle_apply_constraints(adjp, j);
			}
			gUiData.FreezeDialog--;
		}
		despeckle_update_sensitive();
	}
	if (match > 0) {
		nufraw_invalidate_despeckle_layer(gUiData.UF);
		render_preview();
	}
	return match ? TRUE : FALSE;
}

static void adjustment_update(GtkAdjustment *adj, double *valuep)
{
	if (adj==gUiData.BlackpointAdjustment){

		if (gConf->autoBlack==apply_state){

			// force manual curve
			gConf->curveIndex = manual_curve;
			// disable autoblack
			gConf->autoBlack = FALSE;
			auto_button_toggle(gUiData.AutoBlackButton, &gConf->autoBlack);
		}

		// do not use valuep because curveIndex can change
		valuep = &gConf->curve[gConf->curveIndex].m_anchors[0].x;
	}

	if (valuep == &gConf->profile[0][0].gamma) valuep = (void *)&gConf->profile[0][gConf->profileIndex[0]].gamma;
	if (valuep == &gConf->profile[0][0].linear) valuep = (void *)&gConf->profile[0][gConf->profileIndex[0]].linear;

	if ((int *)valuep == &gConf->CropX1 || (int *)valuep == &gConf->CropX2 ||
	    (int *)valuep == &gConf->CropY1 || (int *)valuep == &gConf->CropY2) {

		// values set by update_crop_ranges are ok and do not have to be fixed
		if (!gUiData.FreezeDialog) {

			*((int *)valuep) = (int)gtk_adjustment_get_value(adj);
			CursorType cursor = left_cursor;
			if ((int *)valuep == &gConf->CropX1) cursor = left_cursor;
			if ((int *)valuep == &gConf->CropX2) cursor = right_cursor;
			if ((int *)valuep == &gConf->CropY1) cursor = top_cursor;
			if ((int *)valuep == &gConf->CropY2) cursor = bottom_cursor;
			fix_crop_aspect(cursor, TRUE);
			gConf->fullCrop = disabled_state;
			gConf->autoCrop = disabled_state;
			auto_button_toggle(gUiData.AutoCropButton, &gConf->autoCrop);
		}
		update_scales();
		return;
	}

	// Do nothing if value didn't really change
	uf_long accuracy = (uf_long)g_object_get_data(G_OBJECT(adj), "Adjustment-Accuracy");
	float change = fabs(*valuep - gtk_adjustment_get_value(adj));
	float min_change = pow(10, -accuracy) / 2;

	if (change < min_change) return;
	else *valuep = gtk_adjustment_get_value(adj);

	if (valuep == &gConf->exposure) {

		gConf->autoExposure = FALSE;
		auto_button_toggle(gUiData.AutoExposureButton, &gConf->autoExposure);
		if (gConf->autoBlack == enabled_state) gConf->autoBlack = apply_state;

	} else if (valuep == &gConf->wdThreshold) {
		nufraw_invalidate_denoise_layer(gUiData.UF);
	} else if (valuep == &gConf->hotpixel) {
		nufraw_invalidate_hotpixel_layer(gUiData.UF);
	} else if (despeckle_adjustment_update(valuep)) {
		return;
	} else {
		if (gConf->autoExposure == enabled_state) gConf->autoExposure = apply_state;
		if (gConf->autoBlack == enabled_state) gConf->autoBlack = apply_state;
	}

	int croppedWidth = gConf->CropX2 - gConf->CropX1;
	int croppedHeight = gConf->CropY2 - gConf->CropY1;

	if ((valuep == &gUiData.shrink)&&(gUiData.shrink!=0)) {
		gUiData.height = croppedHeight / gUiData.shrink;
		gUiData.width = croppedWidth / gUiData.shrink;
		update_scales();
		return;
	}

	if (valuep == &gUiData.height) {
		if (croppedHeight!=0) gUiData.width = gUiData.height * croppedWidth / croppedHeight;
		if (gUiData.height!=0) gUiData.shrink = (double)croppedHeight / gUiData.height;
		update_scales();
		return;
	}

	if (valuep == &gUiData.width) {
		if (croppedWidth!=0) gUiData.height = gUiData.width * croppedHeight / croppedWidth;
		if (gUiData.width!=0) gUiData.shrink = (double)croppedWidth / gUiData.width;
		update_scales();
		return;
	}

	nufraw_invalidate_layer(gUiData.UF, nufraw_develop_phase);
	update_scales();
	render_preview();
}

static void adjustment_update_rotation(GtkAdjustment *adj, gpointer unused)
{
	(void)adj;
	(void)unused;

	// Normalize the "unnormalized" value displayed to the user to
	// -180 < a <= 180, though we later normalize to an orientation
	// and flip plus 0 <= a < 90 rotation for processing.
	if (gUiData.FreezeDialog) return;

	int oldFlip = gConf->orientation;
	nufraw_unnormalize_rotation(gUiData.UF);
	gConf->rotationAngle = gtk_adjustment_get_value(gUiData.RotationAdjustment);
	gConf->orientation = gUiData.UnnormalizedOrientation;
	nufraw_normalize_rotation(gUiData.UF);
	int newFlip = gConf->orientation;
	int flip;
	for (flip = 0; flip < 8; flip++) {
		gConf->orientation = oldFlip;
		nufraw_flip_orientation(gUiData.UF, flip);
		if (gConf->orientation == newFlip) break;
	}
	gConf->orientation = oldFlip;
	nufraw_flip_image(gUiData.UF, flip);
	gtk_widget_set_sensitive(gUiData.ResetRotationAdjustment, gConf->rotationAngle != 0 || gConf->orientation != gConf->CameraOrientation);

	nufraw_invalidate_layer(gUiData.UF, nufraw_transform_phase);
	render_preview();
}

static void nufraw_image_changed(UFObject *obj, UFEventType type)
{
	(void)obj;
	if (type != uf_value_changed) return;
	render_preview();
}

static void combo_update(GtkWidget *combo, gint *valuep)
{
	if (gUiData.FreezeDialog) return;

	*valuep = gtk_combo_box_get_active(GTK_COMBO_BOX(combo));

	if (valuep == &gConf->BaseCurveIndex) {

		if (!has_camera_curve() && gConf->BaseCurveIndex > camera_curve - 2) gConf->BaseCurveIndex += 2;
		curveeditor_widget_set_curve(gUiData.BaseCurveWidget, &gConf->BaseCurve[gConf->BaseCurveIndex]);

	} else if (valuep == &gConf->curveIndex) {

		curveeditor_widget_set_curve(gUiData.CurveWidget, &gConf->curve[gConf->curveIndex]);
	}

	if (gConf->autoExposure == enabled_state) gConf->autoExposure = apply_state;
	if (gConf->autoBlack == enabled_state) gConf->autoBlack = apply_state;
	nufraw_invalidate_layer(gUiData.UF, nufraw_develop_phase);
	update_scales();
	render_preview();
}

void combo_update_simple(GtkWidget *combo, nUFRawPhase phase)
{
	(void)combo;
	if (gUiData.FreezeDialog) return;

	if (gConf->autoExposure == enabled_state) gConf->autoExposure = apply_state;
	if (gConf->autoBlack == enabled_state) gConf->autoBlack = apply_state;

	nufraw_invalidate_layer(gUiData.UF, phase);
	update_scales();
	render_preview();
}

static void despeckling_expander(GtkWidget *expander, gpointer index)
{
    long i=(long)index;

    if (!gtk_expander_get_expanded(GTK_EXPANDER(expander))) {
        if (gUiData.ChannelSelect >= 0)
            gtk_toggle_button_set_active(gUiData.ChannelSelectButton[gUiData.ChannelSelect], FALSE);
    }
    if (i>=UF_MAX_EXPANDERS) return;
    gUiConf->expander_open[i]=gtk_expander_get_expanded(GTK_EXPANDER(expander));
}

static void window_map_event(GtkWidget *widget, GdkEvent *event, gpointer unused)
{
	(void)event;
	(void)unused;
	(void)widget;

	start_blink();
}

static void window_unmap_event(GtkWidget *widget, GdkEvent *event, gpointer unused)
{
	(void)event;
	(void)unused;
	(void)widget;

	stop_blink();
}

static gboolean window_delete_event(GtkWidget *widget, GdkEvent *event, gpointer unused)
{
	(void)unused;
	(void)event;

	if (gUiData.FreezeDialog) return TRUE;

	if (!gImageHasModifications){
			g_object_set_data(G_OBJECT(widget), "WindowResponse", (gpointer)GTK_RESPONSE_CANCEL);
			gtk_main_quit();
	}else{
		char message[UF_MAX_PATH];
		snprintf(message, UF_MAX_PATH, "%s?", _("Close window"));
		if (nufraw_confirm_dialog(message, _("Close window"), widget)==GTK_RESPONSE_YES){
			g_object_set_data(G_OBJECT(widget), "WindowResponse", (gpointer)GTK_RESPONSE_CANCEL);
			gtk_main_quit();
		}
	}

	return TRUE;
}

// save user default settings
void user_store_cb(GtkWidget *widget, void* unused)
{
 	if (gUiData.FreezeDialog) return;

 	(void)widget;
	(void)unused;

	char message[UF_MAX_PATH];
	snprintf(message, UF_MAX_PATH, "%s?", _("Store user default"));
	if (nufraw_confirm_dialog(message, _("Store user default"), widget)!=GTK_RESPONSE_YES) return;

	int shrink = gConf->shrink;
	gConf->shrink=gUiData.shrink;
	conf_save(gConf,NULL,NULL);
	gConf->shrink=shrink;
}

void reset(const char *IDFilename){

	conf_data rc;
	conf_load(&rc, IDFilename);

	// TODO: same code in nufraw_main.c : put in a function
	if (gUiData.plugin == nufraw_standalone){
		// Half interpolation is an option only for the GIMP plug-in.
		// For the stand-alone tool it is disabled
		if (rc.interpolation == half_interpolation)  rc.interpolation = ahd_interpolation;
	}

	g_strlcpy(rc.outputPath, "", UF_MAX_PATH);
	g_strlcpy(rc.inputFilename, "", UF_MAX_PATH);
	g_strlcpy(rc.outputFilename, "", UF_MAX_PATH);

	conf_copy_image(gConf, &rc);

 	// some conf_copy_save
	gConf->type = rc.type;
	gConf->compression = rc.compression;
	gConf->createID =rc.createID;
	gConf->embedExif = rc.embedExif;
	if (rc.shrink!=0) gUiData.shrink = rc.shrink;
	gConf->overwrite = rc.overwrite;
	gConf->progressiveJPEG = rc.progressiveJPEG;
	gConf->losslessCompress = rc.losslessCompress;

	// some conf_copy_transform
	gConf->CropX1 = rc.CropX1;
	gConf->CropY1 = rc.CropY1;
	gConf->CropX2 = rc.CropX2;
	gConf->CropY2 = rc.CropY2;
	gConf->aspectRatio = rc.aspectRatio;
	gConf->rotationAngle = rc.rotationAngle;
	gConf->orientation = rc.orientation;

	nufraw_invalidate_layer(gUiData.UF, nufraw_first_phase);
	update_scales();
	render_preview();
}

// load user default settings
void user_reset_cb(GtkWidget *widget, void* unused)
{
 	if (gUiData.FreezeDialog) return;

 	(void)widget;
	(void)unused;

	char message[UF_MAX_PATH];
	snprintf(message, UF_MAX_PATH, "%s?", _("Reset to user default"));
	if (nufraw_confirm_dialog(message, _("Reset to user default"), widget)!=GTK_RESPONSE_YES) return;

	reset(g_build_filename(uf_get_home_dir(), UF_RC_FILE, NULL));
}

// load id settings
void id_reset_cb(GtkWidget *widget, void* unused)
{
 	if (gUiData.FreezeDialog) return;

	(void)unused;

	GtkFileChooser *fileChooser;
	fileChooser = nufraw_raw_chooser(gConf, gUiConf, NULL, _("Reset to ID file"),
	                                 GTK_WINDOW(gtk_widget_get_toplevel(widget)),
	                                 GTK_STOCK_CANCEL, FALSE, TRUE);

	if (gtk_dialog_run(GTK_DIALOG(fileChooser)) == GTK_RESPONSE_ACCEPT) {
		reset(gtk_file_chooser_get_filename(fileChooser));
	}
	nufraw_focus(fileChooser, FALSE);
	gtk_widget_destroy(GTK_WIDGET(fileChooser));
}

// load program default settings
void program_reset_cb(GtkWidget *widget, void* unused)
{
 	if (gUiData.FreezeDialog) return;

 	(void)widget;
	(void)unused;

	char message[UF_MAX_PATH];
	snprintf(message, UF_MAX_PATH, "%s?", _("Reset to program default"));
	if (nufraw_confirm_dialog(message, _("Reset to program default"), widget)!=GTK_RESPONSE_YES) return;

	reset_button_cb(gUiData.ResetAllButton,NULL);
}

void options_button_cb(GtkWidget *widget, void* unused)
{
	(void)widget;
	(void)unused;

	if (gUiData.FreezeDialog == TRUE) return;
	show_options_dialog();
}

// control buttons are the ones that can end session (except cancel)
static void control_button_cb(GtkWidget *widget, long type)
{
	if (gUiData.FreezeDialog == TRUE) return;
	gUiData.FreezeDialog = TRUE;

	GtkWindow *window = GTK_WINDOW(gtk_widget_get_toplevel(widget));
	uf_long response = NUFRAW_NO_RESPONSE;
	int status = NUFRAW_SUCCESS;

	// Switch from preview shrink/size to output shrink/size
	int shrinkSave = gConf->shrink;
	int sizeSave = gConf->size;
	if (fabs(gUiData.shrink - floor(gUiData.shrink + 0.0005)) < 0.0005) {
		gConf->shrink = floor(gUiData.shrink + 0.0005);
		gConf->size = 0;
	} else {
		gConf->shrink = 1;
		gConf->size = floor(MAX(gUiData.height, gUiData.width) + 0.5);
	}
	switch (type) {

		case save_button:

			// save for standalone mode
			if (nufraw_save_dialog(gUiData.UF,gUiConf,widget)==GTK_RESPONSE_ACCEPT){

				preview_progress_enable();
				nufraw_save_now(gUiData.UF, widget);
				preview_progress_disable();

				nufraw_invalidate_layer(gUiData.UF, nufraw_raw_phase);

				render_preview_now(NULL);
				gImageHasModifications=FALSE;

				// TODO: close on save could be an option
				// response = GTK_RESPONSE_OK;
			}

			break;

		case gimp_button:

			// send to GIMP for standalone mode
			if (nufraw_send_to_gimp(gUiData.UF,gUiConf->remoteGimpCommand) == NUFRAW_SUCCESS){

				// TODO: close on send to GIMP cuold be an option
				// response = GTK_RESPONSE_OK;
			}
			break;

		case ok_button:

			// send to gimp for plugin mode
			preview_progress_enable();
			status = (*gUiData.SaveFunc)(gUiData.UF, widget);
			preview_progress_disable();
			if (status == NUFRAW_SUCCESS)
			response = GTK_RESPONSE_OK;
			break;

		default:
			break;
	}
	if (response != NUFRAW_NO_RESPONSE) {
		// Finish this session
		g_object_set_data(G_OBJECT(window), "WindowResponse",(gpointer)response);
		gtk_main_quit();
	}

	// Restore setting
	gConf->shrink = shrinkSave;
	gConf->size = sizeSave;
	gUiData.FreezeDialog = FALSE;

	// cases that set error status require redrawing of the preview image
	if (status != NUFRAW_SUCCESS) {
		nufraw_invalidate_layer(gUiData.UF, nufraw_raw_phase);
		render_preview();
	}
}

// main dialog /////////////////////////////////////////////////////////////////

static void input_fill_interface(GtkTable *table)
{
	GtkWidget *button;
	GtkBox *box;
	int i;
	GtkWidget *label;

	// Interpolation
	gUiData.InterpolationCombo = GTK_COMBO_BOX(uf_combo_box_new_text());
	if (gUiData.UF->HaveFilters) {
		if (gUiData.UF->IsXTrans) {
			uf_combo_box_append_text(gUiData.InterpolationCombo, _("X-Trans interpolation"), (void*)xtrans_interpolation);
		} else if (gUiData.UF->colors == 4) {
			uf_combo_box_append_text(gUiData.InterpolationCombo, _("VNG four color interpolation"), (void*)four_color_interpolation);
		} else {
			uf_combo_box_append_text(gUiData.InterpolationCombo, _("AHD interpolation"), (void*)ahd_interpolation);
			uf_combo_box_append_text(gUiData.InterpolationCombo, _("VNG interpolation"), (void*)vng_interpolation);
			uf_combo_box_append_text(gUiData.InterpolationCombo, _("VNG four color interpolation"), (void*)four_color_interpolation);
			uf_combo_box_append_text(gUiData.InterpolationCombo, _("PPG interpolation"), (void*)ppg_interpolation);
		}
		uf_combo_box_append_text(gUiData.InterpolationCombo, _("Bilinear interpolation"), (void*)bilinear_interpolation);
#ifdef ENABLE_INTERP_NONE

#endif
		uf_combo_box_set_data(gUiData.InterpolationCombo, &gConf->interpolation);
		g_signal_connect_after(G_OBJECT(gUiData.InterpolationCombo), "changed", G_CALLBACK(combo_update_simple), (gpointer)nufraw_first_phase);
	} else {
		gtk_combo_box_append_text(gUiData.InterpolationCombo, _("No color filter array"));
		gtk_combo_box_set_active(gUiData.InterpolationCombo, 0);
		gtk_widget_set_sensitive(GTK_WIDGET(gUiData.InterpolationCombo), FALSE);
	}
	gtk_table_attach(table, GTK_WIDGET(gUiData.InterpolationCombo), 0, 3, 0, 1, GTK_EXPAND | GTK_FILL, 0, 0, 0);

	// Color smoothing button
	gUiData.SmoothingButton = GTK_TOGGLE_BUTTON(gtk_toggle_button_new());
	gtk_container_add(GTK_CONTAINER(gUiData.SmoothingButton), gtk_image_new_from_stock(GTK_STOCK_CLEAR, UI_BT_SIZE));
	gtk_widget_set_tooltip_text(GTK_WIDGET(gUiData.SmoothingButton), _("Apply color smoothing"));
	gtk_toggle_button_set_active(gUiData.SmoothingButton, gConf->smoothing);
	g_signal_connect(G_OBJECT(gUiData.SmoothingButton), "toggled", G_CALLBACK(toggle_button_cb), &gConf->smoothing);
	if (!gUiData.UF->HaveFilters) gtk_widget_set_sensitive(GTK_WIDGET(gUiData.SmoothingButton), FALSE);
	gtk_table_attach(table, GTK_WIDGET(gUiData.SmoothingButton), 3, 4, 0, 1, 0, 0, 0, 0);

	// camera profile
	label = gtk_label_new(_("ICC profile"));
	gtk_table_attach(table, GTK_WIDGET(label), 0, 1, 1, 2, 0, 0, 0, 0);
	gUiData.ProfileCombo[in_profile] = GTK_COMBO_BOX(uf_combo_box_new_text());
	for (i = 0; i < gConf->profileCount[in_profile]; i++){
		if (i < conf_default.profileCount[in_profile]) {
			gtk_combo_box_append_text(gUiData.ProfileCombo[in_profile],_(gConf->profile[in_profile][i].name));
		} else {
			gtk_combo_box_append_text(gUiData.ProfileCombo[in_profile], gConf->profile[in_profile][i].name);
		}
	}
	uf_combo_box_set_data(gUiData.ProfileCombo[in_profile], &gConf->profileIndex[in_profile]);
	g_signal_connect_after(G_OBJECT(gUiData.ProfileCombo[in_profile]), "changed", G_CALLBACK(combo_update_simple), (gpointer)nufraw_develop_phase);
	gtk_table_attach(table, GTK_WIDGET(gUiData.ProfileCombo[in_profile]),  1, 3, 1, 2, GTK_EXPAND | GTK_FILL, 0, 0, 0);
	button = uf_stock_icon_button(GTK_STOCK_OPEN, NULL, G_CALLBACK(load_profile), (gpointer)in_profile);
	gtk_table_attach(table, button, 3, 4, 1, 2, 0, 0, 0, 0);

	// gamma and linearity
	gUiData.GammaAdjustment = uf_adjustment_scale(table, 0, 2, _("Gamma"),
		                                        gConf->profile[0][gConf->profileIndex[0]].gamma,
		                                        &gConf->profile[0][0].gamma, 0.1, 1.0, 0.01, 0.05, 2, FALSE,
		                                        _("Gamma correction for the input profile"),
		                                        G_CALLBACK(adjustment_update),
		                                        &gUiData.ResetGammaButton, _("Reset gamma to default"),
		                                        G_CALLBACK(reset_button_cb));

	gUiData.LinearAdjustment = uf_adjustment_scale(table, 0, 3, _("Linearity"),
		                                         gConf->profile[0][gConf->profileIndex[0]].linear,
		                                         &gConf->profile[0][0].linear, 0.0, 1.0, 0.01, 0.05, 3, FALSE,
		                                         _("Linear part of the gamma correction"),
		                                         G_CALLBACK(adjustment_update),
		                                         &gUiData.ResetLinearButton, _("Reset linearity to default"),
		                                         G_CALLBACK(reset_button_cb));

	box = GTK_BOX(gtk_hbox_new(FALSE, 0));
	gtk_table_attach(table, GTK_WIDGET(box), 0, 4, 4, 5, GTK_EXPAND | GTK_FILL, 0, 0, 0);

	// base curve
	gUiData.BaseCurveCombo = GTK_COMBO_BOX(uf_combo_box_new_text());
	// Fill in the curve names, skipping custom and camera curves if there is
	// no cameraCurve. This will make some mess later with the counting
	for (i = 0; i < gConf->BaseCurveCount; i++) {
		if ((i == custom_curve || i == camera_curve) && !has_camera_curve()) continue;
		if (i <= camera_curve)  gtk_combo_box_append_text(gUiData.BaseCurveCombo, _(gConf->BaseCurve[i].name));
		else  gtk_combo_box_append_text(gUiData.BaseCurveCombo, gConf->BaseCurve[i].name);
	}

	// This is part of the mess with the combo_box counting
	if (gConf->BaseCurveIndex > camera_curve && !has_camera_curve()){
		gtk_combo_box_set_active(gUiData.BaseCurveCombo, gConf->BaseCurveIndex - 2);
	}else{
		gtk_combo_box_set_active(gUiData.BaseCurveCombo, gConf->BaseCurveIndex);
	}
	g_signal_connect(G_OBJECT(gUiData.BaseCurveCombo), "changed",  G_CALLBACK(combo_update), &gConf->BaseCurveIndex);
	gtk_box_pack_start(box, GTK_WIDGET(gUiData.BaseCurveCombo), TRUE, TRUE, 0);
	button = uf_stock_icon_button(GTK_STOCK_OPEN, _("Load base curve"), G_CALLBACK(load_curve), (gpointer)base_curve);
	gtk_box_pack_start(box, button, FALSE, FALSE, 0);
	button = uf_stock_icon_button(GTK_STOCK_SAVE_AS, _("Save base curve"), G_CALLBACK(save_curve), (gpointer)base_curve);
	gtk_box_pack_start(box, button, FALSE, FALSE, 0);

	gUiData.ResetBaseCurveButton = uf_reset_button(_("Reset base curve to default"), G_CALLBACK(reset_button_cb), NULL);
	gtk_box_pack_start(box, gUiData.ResetBaseCurveButton, FALSE, FALSE, 0);

	box = GTK_BOX(gtk_hbox_new(FALSE, 0));
	gtk_table_attach(table, GTK_WIDGET(box), 0, 9, 5, 6, GTK_EXPAND | GTK_FILL, 0, 0, 0);

	gUiData.BaseCurveWidget = curveeditor_widget_new(UI_CURVE_HEIGHT, gUiData.LeftSpace, curve_update, (gpointer)base_curve);
	curveeditor_widget_set_curve(gUiData.BaseCurveWidget, &gConf->BaseCurve[gConf->BaseCurveIndex]);
	gtk_box_pack_start(box,GTK_WIDGET(gUiData.BaseCurveWidget), TRUE, TRUE, 0);
}

static void exposure_fill_interface(GtkTable *table)
{
	gUiData.ExposureAdjustment = uf_adjustment_scale(table, 0, 0, NULL,
		                   gConf->exposure, &gConf->exposure,
		                   -UI_EXP_RANGE, UI_EXP_RANGE, 0.01, 1.0 / 6, 2, FALSE,
		                   _("Exposure compensation in EV"),
		                   G_CALLBACK(adjustment_update), NULL, NULL, NULL);

	gUiData.DetailButton = gtk_toggle_button_new();
	gtk_table_attach(table, gUiData.DetailButton, 7, 8, 0, 1, 0, 0, 0, 0);
	restore_details_button_set();
	g_signal_connect(G_OBJECT(gUiData.DetailButton), "toggled", G_CALLBACK(toggle_button_cb), &gConf->restoreDetails);

	gUiData.HighlightButton = gtk_toggle_button_new();
	gtk_table_attach(table, gUiData.HighlightButton, 8, 9, 0, 1, 0, 0, 0, 0);
	clip_highlights_button_set();
	g_signal_connect(G_OBJECT(gUiData.HighlightButton), "toggled", G_CALLBACK(toggle_button_cb), &gConf->clipHighlights);

	gUiData.AutoExposureButton = GTK_TOGGLE_BUTTON(gtk_toggle_button_new());
	gtk_table_attach(table, GTK_WIDGET(gUiData.AutoExposureButton), 9, 10, 0, 1, 0, 0, 0, 0);
	gtk_widget_set_tooltip_text(GTK_WIDGET(gUiData.AutoExposureButton), _("Auto adjust exposure"));
	auto_button_toggle(gUiData.AutoExposureButton, &gConf->autoExposure);
	g_signal_connect(G_OBJECT(gUiData.AutoExposureButton), "toggled", G_CALLBACK(auto_button_toggled), &gConf->autoExposure);

	gUiData.ResetExposureButton = uf_reset_button(_("Reset exposure to default"), G_CALLBACK(reset_button_cb), NULL);
	gtk_table_attach(table, gUiData.ResetExposureButton, 10, 11, 0, 1, 0, 0, 0, 0);
}

static void whitebalance_fill_interface(GtkTable *table)
{
	int i;
	GtkWidget *event_box;
	GtkWidget *button;
	GtkWidget *label;
	GtkComboBox *combo;
	nufraw_data *uf = gUiData.UF;
	UFObject *image = gConf->ufobject;
	int c=0;

	UFObject *wb = ufgroup_element(image, ufWB);

	if (!ufgroup_has(wb, uf_camera_wb)) {

		event_box = gtk_event_box_new();
		label = gtk_image_new_from_stock(GTK_STOCK_DIALOG_WARNING,UI_BT_SIZE);
		gtk_container_add(GTK_CONTAINER(event_box), label);
		gtk_table_attach(table, GTK_WIDGET(event_box), 0, 1, 0, 1, 0, 0, 0, 0);
		gtk_widget_set_tooltip_text(event_box, _("Cannot use camera white balance."));
		c++;

	} else if (!uf->wb_presets_make_model_match) {

		event_box = gtk_event_box_new();
		label = gtk_image_new_from_stock(GTK_STOCK_DIALOG_WARNING,UI_BT_SIZE);
		gtk_container_add(GTK_CONTAINER(event_box), label);
		gtk_table_attach(table, GTK_WIDGET(event_box), 0, 1, 0, 1, 0, 0, 0, 0);
		gtk_widget_set_tooltip_text(event_box, _("There are no white balance presets for your camera model.\n"
		                                         "Check nUFRaw's webpage for information on how to get your\n"
		                                         "camera supported."));
		c++;
	}

	combo = GTK_COMBO_BOX(ufarray_combo_box_new(wb));
	gtk_table_attach(table, GTK_WIDGET(combo), c, 5, 0, 1,GTK_EXPAND | GTK_FILL, 0, 0, 0);
	gtk_widget_set_tooltip_text(GTK_WIDGET(combo), _("White Balance"));

	// TODO: hide if camera has no fine tunings
	button = ufnumber_spin_button_new(ufgroup_element(image, ufWBFineTuning));
	gtk_table_attach(table, GTK_WIDGET(button), 5, 6, 0, 1, GTK_EXPAND | GTK_FILL, 0, 0, 0);

	// Spot WB button:
	button = uf_stock_icon_button(GTK_STOCK_COLOR_PICKER,
	                              _("Select a spot on the preview image to apply spot white balance"),
	                              G_CALLBACK(spot_wb_event), NULL);
	gtk_table_attach(table, GTK_WIDGET(button), 6, 7, 0, 1, 0, 0, 0, 0);

	// WB reset button
	gUiData.ResetWbButton = ufobject_reset_button_new(_("Reset white balance to initial value"));
	ufobject_reset_button_add(gUiData.ResetWbButton, ufgroup_element(image, ufWB));
	ufobject_reset_button_add(gUiData.ResetWbButton, ufgroup_element(image, ufWBFineTuning));
	gtk_table_attach(table, GTK_WIDGET(gUiData.ResetWbButton), 7, 8, 0, 1, 0, 0, 0, 0);

	ufobject_set_changed_event_handle(image, nufraw_image_changed);

	// wb sliders
	ufnumber_adjustment_scale(ufgroup_element(image, ufTemperature), table,0, 1, _("Temperature"), _("White balance color temperature (K)"));
	ufnumber_adjustment_scale(ufgroup_element(image, ufGreen), table, 0, 2, _("Green"), _("Green component"));

	// channle multipliers
	label = gtk_label_new(_("Chan. multipliers"));
	gtk_misc_set_alignment(GTK_MISC(label),1,0.5);
	gtk_table_attach(table, label, 4, 5, 3, 4, GTK_SHRINK|GTK_FILL, 0, 0, 0);
	for (i = 0; i < gUiData.UF->colors; i++) {
		button = ufnumber_array_spin_button_new(ufgroup_element(image, ufChannelMultipliers), i);
		gtk_table_attach(table, button, 5, 7, 3+i, 4+i, GTK_EXPAND | GTK_FILL, 0, 0, 0);
	}
	ufobject_reset_button_add(gUiData.ResetWbButton,ufgroup_element(image, ufChannelMultipliers));
}

static void grayscale_fill_interface(GtkTable *basetable)
{
	GtkTable *table;
	GtkWidget *label;
	GtkWidget *button;
	long i;

	// mode buttons
	table = GTK_TABLE(gtk_table_new(2, grayscale_invalid-1, TRUE));
	gtk_table_attach(basetable, GTK_WIDGET(table), 0, 1, 0, 1, GTK_EXPAND|GTK_FILL, 0, 0, 0);

	for (i = 0; i < grayscale_invalid; i++) {
		gUiData.GrayscaleButtons[i] = gtk_radio_button_new_with_label_from_widget(
			                        (i > 0) ? GTK_RADIO_BUTTON(gUiData.GrayscaleButtons[0]) : NULL,
			                        _(grayscaleModeNames[i]));
		gtk_toggle_button_set_mode(GTK_TOGGLE_BUTTON(gUiData.GrayscaleButtons[i]), FALSE);
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gUiData.GrayscaleButtons[i]), gConf->grayscaleMode == i);

		if (i==grayscale_mixer){
			// channel mixer stay on a line by itself
			gtk_table_attach(table, gUiData.GrayscaleButtons[i], 0, grayscale_mixer, 1, 2, GTK_EXPAND | GTK_FILL, 0, 0, 0);
		}else{
			gtk_table_attach(table, gUiData.GrayscaleButtons[i], i, i+1, 0, 1, GTK_EXPAND | GTK_FILL, 0, 0, 0);
		}
		g_signal_connect(G_OBJECT(gUiData.GrayscaleButtons[i]), "toggled", G_CALLBACK(grayscale_update), NULL);
	}

	// grayscale mixer
	gUiData.GrayscaleMixerTable = GTK_TABLE(gtk_table_new(4, 8, FALSE));
	gtk_table_attach(basetable, GTK_WIDGET(gUiData.GrayscaleMixerTable), 0, 1, 1, 2, GTK_EXPAND|GTK_FILL, 0, 0, 0);
	for (i = 0; i < 3; i++) {
		gUiData.GrayscaleMixers[i] = uf_adjustment_scale(gUiData.GrayscaleMixerTable, 0, i,
		                                                 i == 0 ? "@channel-red" : i == 1 ? "@channel-green" : "@channel-blue",
		                                                 gConf->grayscaleMixer[i], &gConf->grayscaleMixer[i],
		                                                 -2.0, 2.0, .01, 0.10, 2, FALSE, NULL,
		                                                 G_CALLBACK(adjustment_update),NULL, NULL, NULL);
	}

	gUiData.ResetGrayscaleChannelMixerButton = uf_reset_button(_("Reset channel mixer"), G_CALLBACK(reset_button_cb), NULL);
	gtk_table_attach(gUiData.GrayscaleMixerTable, gUiData.ResetGrayscaleChannelMixerButton, 7, 8, 0, 1, GTK_SHRINK | GTK_FILL, 0, 0, 0);

	gUiData.GrayscaleMixerColor = GTK_LABEL(gtk_label_new(NULL));
	gtk_table_attach(gUiData.GrayscaleMixerTable,GTK_WIDGET(gUiData.GrayscaleMixerColor), 7, 8, 1, 3, GTK_SHRINK | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

	// filters buttons
	table = GTK_TABLE(gtk_table_new(1, sizeof(grayscale_filters)/sizeof(grayscale_filters[0]), TRUE));
	gtk_table_attach(gUiData.GrayscaleMixerTable,GTK_WIDGET(table),0, 8, 3, 4, GTK_EXPAND | GTK_FILL, 0, 0, 0);

	for (i=0; i<(long)(sizeof(grayscale_filters)/sizeof(grayscale_filters[0])); i++){
		button = gtk_button_new();
		label = gtk_label_new("");
		gtk_container_add(GTK_CONTAINER(button),label);
		uf_label_set_background(GTK_LABEL(label),grayscale_filters[i][0],grayscale_filters[i][1],grayscale_filters[i][2],1);
		gtk_table_attach(table,button, i, i+1, 0, 1, GTK_EXPAND | GTK_FILL, 0, 0, 0);
		g_signal_connect(G_OBJECT(button), "clicked", G_CALLBACK(grayscale_filter_clicked), (gpointer)i);
	}
}

static void corrections_fill_interface(GtkTable *table)
{
	GtkWidget *button;
	int i;
	GtkTable *subTable;

#ifdef NUFRAW_CONTRAST
	gUiData.ContrastAdjustment = uf_adjustment_scale(table, 0, 0, _("Contrast"),
		                                           gConf->contrast, &gConf->contrast,
		                                           0, 8.0, 0.01, 0.1, 2, FALSE,
		                                           _("Global contrast adjustment"),
		                                           G_CALLBACK(adjustment_update),
		                                           &gUiData.ResetContrastButton,
		                                           _("Reset global contrast to default"),
		                                           G_CALLBACK(reset_button_cb));
#endif
	gUiData.SaturationAdjustment = uf_adjustment_scale(table, 0, 1, _("Saturation"),
		                                             gConf->saturation, &gConf->saturation,
		                                             0.0, 8.0, 0.01, 0.1, 2, FALSE,
		                                             _("Saturation"), G_CALLBACK(adjustment_update),
		                                             &gUiData.ResetSaturationButton,
		                                             _("Reset saturation to default"),
		                                             G_CALLBACK(reset_button_cb));

	gUiData.BlackpointAdjustment = uf_adjustment_scale(table, 0, 2, _("Black point"),
		                                             gConf->curve[gConf->curveIndex].m_anchors[0].x,
		                                             &gConf->curve[gConf->curveIndex].m_anchors[0].x,
		                                             0.0, 1.0, 0.001, 0.1, 3, FALSE,
		                                             _("Black point"), G_CALLBACK(adjustment_update),
		                                             &gUiData.ResetBlackButton,
		                                             _("Reset black-point to default"),
		                                             G_CALLBACK(reset_button_cb));

	gUiData.AutoBlackButton = GTK_TOGGLE_BUTTON(gtk_toggle_button_new());
	gtk_table_attach(table, GTK_WIDGET(gUiData.AutoBlackButton), 8, 9, 2, 3, GTK_SHRINK, 0, 0, 0);
	gtk_widget_set_tooltip_text(GTK_WIDGET(gUiData.AutoBlackButton), _("Auto adjust black-point"));
	auto_button_toggle(gUiData.AutoBlackButton, &gConf->autoBlack);
	g_signal_connect(G_OBJECT(gUiData.AutoBlackButton), "toggled", G_CALLBACK(auto_button_toggled), &gConf->autoBlack);

	// the curve combo
	subTable = GTK_TABLE(gtk_table_new(5, 2, FALSE));
	gtk_table_attach(table, GTK_WIDGET(subTable), 0, 9, 3, 4, GTK_EXPAND|GTK_FILL, 0, 0, 0);

	gUiData.CurveCombo = GTK_COMBO_BOX(uf_combo_box_new_text());
	for (i = 0; i < gConf->curveCount; i++){
		if (i <= linear_curve) gtk_combo_box_append_text(gUiData.CurveCombo, _(gConf->curve[i].name));
		else gtk_combo_box_append_text(gUiData.CurveCombo, gConf->curve[i].name);
	}
	gtk_combo_box_set_active(gUiData.CurveCombo, gConf->curveIndex);
	g_signal_connect(G_OBJECT(gUiData.CurveCombo), "changed", G_CALLBACK(combo_update), &gConf->curveIndex);
	gtk_table_attach(subTable, GTK_WIDGET(gUiData.CurveCombo), 0, 1, 0, 1, GTK_EXPAND|GTK_FILL, 0, 0, 0);

	button = uf_stock_icon_button(GTK_STOCK_OPEN, _("Load curve"),G_CALLBACK(load_curve), (gpointer)luminosity_curve);
	gtk_table_attach(subTable, GTK_WIDGET(button), 1, 2, 0, 1, 0, 0, 0, 0);

	button = uf_stock_icon_button(GTK_STOCK_SAVE_AS, _("Save curve"), G_CALLBACK(save_curve), (gpointer)luminosity_curve);
	gtk_table_attach(subTable, GTK_WIDGET(button), 2, 3, 0, 1, 0, 0, 0, 0);

	gUiData.AutoCurveButton = uf_stock_icon_button("object-manual", _("Auto adjust curve\n(Flatten histogram)"),G_CALLBACK(reset_button_cb), NULL);
	gtk_table_attach(subTable, GTK_WIDGET(gUiData.AutoCurveButton), 3, 4, 0, 1, 0, 0, 0, 0);

	gUiData.ResetCurveButton = uf_reset_button( _("Reset curve to default"), G_CALLBACK(reset_button_cb), NULL);
	gtk_table_attach(subTable, GTK_WIDGET(gUiData.ResetCurveButton), 4, 5, 0, 1, 0, 0, 0, 0);

	gUiData.CurveWidget = curveeditor_widget_new(UI_CURVE_HEIGHT, gUiData.LeftSpace, curve_update, (gpointer)luminosity_curve);
	curveeditor_widget_set_curve(gUiData.CurveWidget, &gConf->curve[gConf->curveIndex]);
	gtk_table_attach(table, GTK_WIDGET(gUiData.CurveWidget), 0, 9, 4, 5, GTK_EXPAND|GTK_FILL, 0, 0, 0);
}

static void lightness_fill_interface(GtkTable *table)
{
	uf_long i;

	// add button
	gUiData.LightnessHueSelectNewButton = uf_stock_icon_button( GTK_STOCK_COLOR_PICKER,
	                                                           _("Select a spot on the preview image to choose hue"),
	                                                           G_CALLBACK(select_hue_event), (gpointer) -1);
	gtk_table_attach(table, gUiData.LightnessHueSelectNewButton, 0, 1, 0, 1, GTK_EXPAND | GTK_FILL, 0, 0, 0);

	for (i = 0; i < UF_MAX_LIGHTNESS_ADJUSTMENTS; ++i) {

		GtkTable *subTable = GTK_TABLE(gtk_table_new(10, 1, FALSE));
		gtk_table_attach(table, GTK_WIDGET(subTable), 0, 1, i+1, i+2, GTK_EXPAND | GTK_FILL, 0, 0, 0);

		gUiData.LightnessAdjustmentTable[i] = subTable;
		gUiData.LightnessAdjustment[i] = uf_adjustment_scale(subTable, 0, 0, NULL,
		                                                     gConf->lightnessAdjustment[i].adjustment,
		                                                     &gConf->lightnessAdjustment[i].adjustment,
		                                                     0, 2, 0.01, 0.10, 2, FALSE, NULL,
		                                                     G_CALLBACK(adjustment_update),
		                                                     &gUiData.ResetLightnessAdjustmentButton[i],
		                                                     _("Reset adjustment"), G_CALLBACK(reset_button_cb));

		gUiData.LightnessHueSelectButton[i] = uf_stock_icon_button(GTK_STOCK_COLOR_PICKER,
				                        _("Select a spot on the preview image to choose hue"),
				                        G_CALLBACK(select_hue_event), (gpointer)i);

		widget_set_hue(gUiData.LightnessHueSelectButton[i], gConf->lightnessAdjustment[i].hue);
		gtk_table_attach(subTable, gUiData.LightnessHueSelectButton[i], 8, 9, 0, 1, 0, 0, 0, 0);

		gUiData.LightnessHueRemoveButton[i] = uf_stock_icon_button(GTK_STOCK_CLOSE,
		                                                           _("Remove adjustment"),
		                                                           G_CALLBACK(remove_hue_event), (gpointer)i);
		gtk_table_attach(subTable, gUiData.LightnessHueRemoveButton[i], 9, 10, 0, 1, 0, 0, 0, 0);
	}
}

static void transformations_fill_interface(GtkTable *table)
{
	GtkTable *subTable;
	GtkWidget *button;
	GtkWidget *entry;
	GtkWidget *label;

	// Aspect ratio controls
	label = gtk_label_new(_("Aspect ratio"));
	gtk_table_attach(table, label, 0, 1, 0, 1, 0, 0, 0, 0);

	subTable = GTK_TABLE(gtk_table_new(5, 1, FALSE));
	gtk_table_attach(table, GTK_WIDGET(subTable), 1, 9, 0, 1, GTK_EXPAND|GTK_FILL, 0, 0, 0);

	entry = gtk_combo_box_entry_new_text();
	gUiData.AspectEntry = GTK_ENTRY(gtk_bin_get_child(GTK_BIN(entry)));
	gtk_entry_set_width_chars(gUiData.AspectEntry, 6);
	gtk_entry_set_alignment(gUiData.AspectEntry, 0.5);
	gtk_table_attach(subTable, GTK_WIDGET(entry), 1, 2, 0, 1, GTK_EXPAND | GTK_FILL, 0, 0, 0);
	gtk_widget_set_tooltip_text(GTK_WIDGET(gUiData.AspectEntry),
		                      _("Crop area aspect ratio.\n"
		                        "Can be entered in decimal notation (1.273)\n"
		                        "or as a ratio of two numbers (14:11)"));

	size_t s;
	for (s = 0; s < sizeof(predef_aspects) / sizeof(predef_aspects[0]); s++) {
		gtk_combo_box_append_text(GTK_COMBO_BOX(entry), predef_aspects[s].text);
	}

	g_signal_connect(G_OBJECT(entry), "changed", G_CALLBACK(aspect_changed), NULL);
	g_signal_connect(G_OBJECT(gUiData.AspectEntry), "focus-out-event", G_CALLBACK(aspect_activate), NULL);
	g_signal_connect(G_OBJECT(gUiData.AspectEntry), "activate", G_CALLBACK(aspect_activate), NULL);

	// lock aspect button
	gUiData.LockAspectButton = GTK_TOGGLE_BUTTON(gtk_toggle_button_new());
	gtk_table_attach(subTable, GTK_WIDGET(gUiData.LockAspectButton), 2, 3, 0, 1, 0, 0, 0, 0);
	g_signal_connect(G_OBJECT(gUiData.LockAspectButton), "clicked",G_CALLBACK(lock_aspect), &gUiConf->lockAspect);
	lock_aspect(gUiData.LockAspectButton, &gUiConf->lockAspect);

	// autocrop button
	gUiData.AutoCropButton = GTK_TOGGLE_BUTTON(gtk_toggle_button_new());
	gtk_table_attach(subTable, GTK_WIDGET(gUiData.AutoCropButton), 3, 4, 0, 1, 0, 0, 0, 0);
	gtk_widget_set_tooltip_text(GTK_WIDGET(gUiData.AutoCropButton), _("Auto fit crop area"));
	auto_button_toggle(gUiData.AutoCropButton, &gConf->autoCrop);
	g_signal_connect(G_OBJECT(gUiData.AutoCropButton), "toggled", G_CALLBACK(auto_button_toggled), &gConf->autoCrop);

	// Crop reset button
	gUiData.ResetCropButton = uf_reset_button( _("Reset the crop area"), G_CALLBACK(crop_reset), NULL);
	gtk_table_attach(subTable, gUiData.ResetCropButton, 4, 5, 0, 1, 0, 0, 0, 0);

	// TODO gtk_widget_set_sensitive(gUiData.ResetCropButton

	// Get initial aspect ratio
	if (gConf->aspectRatio != 0.0){
		set_new_aspect();
	}else{
		int dy = gConf->CropY2 - gConf->CropY1;
		gConf->aspectRatio = dy ? ((gConf->CropX2 - gConf->CropX1) / (float)dy) : 1.0;
	}
	refresh_aspect();

	// Crop controls
	gUiData.CropY1Adjustment = uf_adjustment_scale(table, 0, 1, _("Top"),
		                                         gConf->CropY1, &gConf->CropY1,
		                                         0, gUiData.UF->rotatedHeight, 1, 10, 0, FALSE,
		                                         "",G_CALLBACK(adjustment_update),
		                                         NULL, "", NULL);

	gUiData.CropY2Adjustment = uf_adjustment_scale(table, 0, 2, _("Bottom"),
		                                         gConf->CropY2, &gConf->CropY2,
		                                         0, gUiData.UF->rotatedHeight, 1, 10, 0, FALSE,
		                                         "",G_CALLBACK(adjustment_update),
		                                         NULL, "", NULL);


	gUiData.CropX1Adjustment = uf_adjustment_scale(table, 0, 3, _("Left"),
		                                         gConf->CropX1, &gConf->CropX1,
		                                         0, gUiData.UF->rotatedWidth, 1, 10, 0, FALSE,
		                                         "",G_CALLBACK(adjustment_update),
		                                         NULL, "", NULL);

	gUiData.CropX2Adjustment = uf_adjustment_scale(table, 0, 4, _("Right"),
		                                         gConf->CropX2, &gConf->CropX2,
		                                         0, gUiData.UF->rotatedWidth, 1, 10, 0, FALSE,
		                                         "",G_CALLBACK(adjustment_update),
		                                         NULL, "", NULL);

	// Rotation controls
	nufraw_unnormalize_rotation(gUiData.UF);
	gUiData.RotationAdjustment = uf_adjustment_scale(table, 0, 5, _("Rotation"),
		                                           gConf->rotationAngle, NULL,
		                                           -180, 180, 0.1, 1, 2, TRUE, _("Rotation angle"),
		                                           G_CALLBACK(adjustment_update_rotation),
		                                           &gUiData.ResetRotationAdjustment, _("Reset rotation angle"),
		                                           G_CALLBACK(adjustment_reset_rotation));

	gUiData.UnnormalizedOrientation = gConf->orientation;
	nufraw_normalize_rotation(gUiData.UF);
	gtk_widget_set_sensitive(gUiData.ResetRotationAdjustment, gConf->rotationAngle != 0 || gConf->orientation != gConf->CameraOrientation);

	// Orientation controls
	label = gtk_label_new(_("Orientation"));
	gtk_table_attach(table, label, 0, 1, 6, 7, 0, 0, 0, 0);

	subTable = GTK_TABLE(gtk_table_new(5, 1, FALSE));
	gtk_table_attach(table, GTK_WIDGET(subTable), 1, 3, 6, 7, GTK_EXPAND|GTK_FILL, 0, 0, 0);

	button = uf_stock_image_button("object-rotate-right", UI_BT_SIZE, NULL,G_CALLBACK(flip_image), (gpointer)6);
	gtk_table_attach(subTable, button, 0, 1, 0, 1, GTK_EXPAND | GTK_FILL, 0, 0, 0);

	button = uf_stock_image_button("object-rotate-left",UI_BT_SIZE, NULL,G_CALLBACK(flip_image), (gpointer)5);
	gtk_table_attach(subTable, button, 1, 2, 0, 1, GTK_EXPAND | GTK_FILL, 0, 0, 0);

	button = uf_stock_image_button("object-flip-horizontal",UI_BT_SIZE, NULL,G_CALLBACK(flip_image), (gpointer)1);
	gtk_table_attach(subTable, button, 2, 3, 0, 1, GTK_EXPAND | GTK_FILL, 0, 0, 0);

	button = uf_stock_image_button("object-flip-vertical",UI_BT_SIZE, NULL,G_CALLBACK(flip_image), (gpointer)2);
	gtk_table_attach(subTable, button, 3, 4, 0, 1, GTK_EXPAND | GTK_FILL, 0, 0, 0);
}

static void denoise_fill_interface(GtkTable *table)
{
	GtkWidget *button;
	GtkWidget *label;
	GtkTable *subTable;

	// Denoising
	// TODO update on release
	gUiData.ThresholdAdjustment = uf_adjustment_scale(table, 0, 0, _("Denoising"),
		                                            gConf->wdThreshold, &gConf->wdThreshold,
		                                            0.0, 1000.0, 10, 50, 0, FALSE,
		                                            _("Threshold for wavelet denoising"),
		                                            G_CALLBACK(adjustment_update),
		                                            &gUiData.ResetThresholdButton,
		                                            _("Reset denoise threshold to default"),
		                                            G_CALLBACK(reset_button_cb));

	// TODO: color denoise: convert YcbCr and wavelet denoise on Cb and Cr

	// hot pixels
	label = gtk_label_new(_("Hot pixels"));
	gtk_misc_set_alignment(GTK_MISC(label), 1, 0.5);
	gtk_table_attach(table, GTK_WIDGET(label), 0, 1, 1, 2, 0, 0, 0, 0);

	subTable = GTK_TABLE(gtk_table_new(4, 1, FALSE));
	gtk_table_attach(table, GTK_WIDGET(subTable), 1, 4, 1, 2, GTK_EXPAND | GTK_FILL, 0, 0, 0);

	button = gtk_check_button_new();
	gtk_container_add(GTK_CONTAINER(button),gtk_image_new_from_stock(GTK_STOCK_APPLY, UI_BT_SIZE));
	gtk_toggle_button_set_mode(GTK_TOGGLE_BUTTON(button), FALSE);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button),gUiData.UF->mark_hotpixels);
	g_signal_connect(G_OBJECT(button), "toggled",G_CALLBACK(toggle_button_cb), &gUiData.UF->mark_hotpixels);
	gtk_table_attach(subTable, GTK_WIDGET(button), 0, 1, 0, 1, 0, 0, 0, 0);

	gUiData.HotpixelCount = GTK_LABEL(gtk_label_new(NULL));
	gtk_table_attach(subTable, GTK_WIDGET(gUiData.HotpixelCount), 1, 2, 0, 1, 0, 0, 0, 0);
	gtk_label_set_width_chars(gUiData.HotpixelCount,4);

	// TODO update on release
	gUiData.HotpixelAdjustment = uf_adjustment_scale(subTable, 2, 0, NULL,
		                                           gConf->hotpixel, &gConf->hotpixel,
		                                           0.0, 10, 0.01, 0.1, 3, FALSE,
		                                           _("Hot pixel sensitivity"),
		                                           G_CALLBACK(adjustment_update),
		                                           &gUiData.ResetHotpixelButton,
		                                           _("Reset hot pixel sensitivity"),
		                                           G_CALLBACK(reset_button_cb));

	// darkframe
	label = gtk_label_new(_("Dark frame"));
	gtk_misc_set_alignment(GTK_MISC(label), 1, 0.5);
	gtk_table_attach(table, GTK_WIDGET(label), 0, 1, 2, 3, 0, 0, 0, 0);

	subTable = GTK_TABLE(gtk_table_new(3, 1, FALSE));
	gtk_table_attach(table, GTK_WIDGET(subTable), 1, 4, 2, 3, GTK_EXPAND | GTK_FILL, 0, 0, 0);

	label = gtk_label_new("");
	gtk_misc_set_alignment(GTK_MISC(label), 0.5, 0.5);
	gtk_table_attach(subTable, GTK_WIDGET(label), 0, 1, 0, 1, GTK_EXPAND | GTK_FILL, 0, 0, 0);
	gUiData.DarkFrameLabel = GTK_LABEL(label);
	set_darkframe_label();

	button = uf_stock_icon_button(GTK_STOCK_OPEN, _("Load dark frame"), G_CALLBACK(load_darkframe), NULL);
	gtk_table_attach(subTable, GTK_WIDGET(button), 1, 2, 0, 1, 0, 0, 0, 0);

	gUiData.ResetDarkframe = uf_reset_button(_("Reset dark frame"), G_CALLBACK(reset_darkframe), NULL);
	gtk_table_attach(subTable, GTK_WIDGET(gUiData.ResetDarkframe), 2, 3, 0, 1, 0, 0, 0, 0);
}

static void despeckling_fill_interface(GtkTable *table)
{
	GtkWidget *button;
	GtkWidget *label;
	GtkWidget *icon;
	GtkBox *box;
	int i;

	int right = 6 + gUiData.UF->colors; // Right column location
	icon = gtk_image_new_from_stock(GTK_STOCK_INFO, UI_BT_SIZE);
	gtk_table_attach(table, icon, right, right + 1, 0, 1, 0, 0, 0, 0);
	gtk_widget_set_tooltip_text(icon, _(
		                        "Despeckling is mainly useful when combining a high ISO number "
		                        "with a high channel multiplier: when one channel has a very bad "
		                        "signal to noise ratio. Try setting window size, color decay and "
		                        "number of passes to 50,0,5 for that channel. When a channel "
		                        "contains only noise then try 1,0.6,1.\n"
		                        "Despeckling is off when window size or passes equals zero. When "
		                        "on then window size cannot be smaller than the number of passes."));

	// buttons on the right
	box = GTK_BOX(gtk_vbox_new(FALSE, 0));
	button = gtk_toggle_button_new();
	gtk_container_add(GTK_CONTAINER(button), gtk_image_new_from_stock("object-lock", UI_BT_SIZE));
	gUiData.DespeckleLockChannelsButton = GTK_TOGGLE_BUTTON(button);
	gtk_box_pack_start(box, button, FALSE, FALSE, 0);
	gtk_table_attach(table, GTK_WIDGET(box), right, right + 1, 1, 4, GTK_FILL, 0, 0, 0);
	gtk_widget_set_tooltip_text(button, _("Update channel parameters together"));
	gUiData.ResetDespeckleButton = uf_reset_button(_("Reset despeckle parameters"), G_CALLBACK(reset_button_cb), NULL);
	gtk_box_pack_start(box, gUiData.ResetDespeckleButton, FALSE, FALSE, 0);

	// channel to view
	label = gtk_label_new(_("View channel"));
	gtk_misc_set_alignment(GTK_MISC(label), 1, 1);
	gtk_table_attach(table, label, 0, 6, 0, 1, 0, 0, 0, 0);
	for (i = 0; i < gUiData.UF->colors; ++i) {
		button = gtk_toggle_button_new();
		gtk_container_add(GTK_CONTAINER(button), gtk_image_new_from_stock(
				      i == 0 ? gUiData.UF->colors == 1 ? "channel-grey" : "channel-red" :
				      i == 1 || i == 3 ? "channel-green" : "channel-blue",
				      UI_BT_SIZE));
		gUiData.ChannelSelectButton[i] = GTK_TOGGLE_BUTTON(button);
		g_signal_connect(G_OBJECT(button), "toggled", G_CALLBACK(toggle_button_cb), gUiData.ChannelSelectButton);
		gtk_table_attach(table, button, 6 + i, 6 + i + 1, 0, 1, GTK_FILL | GTK_EXPAND, 0, 0, 0);
	}
	gUiData.ChannelSelect = -1;

	// Parameters
	label = gtk_label_new(_("Window size"));
	gtk_misc_set_alignment(GTK_MISC(label), 1, 1);
	gtk_table_attach(table, label, 0, 6, 1, 2, 0, 0, 0, 0);
	for (i = 0; i < gUiData.UF->colors; ++i) {
		gUiData.DespeckleWindowAdj[i] = GTK_ADJUSTMENT(  gtk_adjustment_new(gConf->despeckleWindow[i],  0.0, 999.0, 1.0, 1.0, 0));
		g_object_set_data(G_OBJECT(gUiData.DespeckleWindowAdj[i]), "Adjustment-Accuracy", (gpointer)0);
		button = gtk_spin_button_new(gUiData.DespeckleWindowAdj[i], 1.0, 0);
		g_object_set_data(G_OBJECT(gUiData.DespeckleWindowAdj[i]), "Parent-Widget", button);
		g_signal_connect(G_OBJECT(gUiData.DespeckleWindowAdj[i]), "value-changed", G_CALLBACK(adjustment_update), &gConf->despeckleWindow[i]);
		gtk_table_attach(table, button, 6 + i, 6 + i + 1, 1, 2, GTK_FILL | GTK_EXPAND, 0, 0, 0);
	}

	label = gtk_label_new(_("Color decay"));
	gtk_misc_set_alignment(GTK_MISC(label), 1, 1);
	gtk_table_attach(table, label, 0, 6, 2, 3, 0, 0, 0, 0);
	for (i = 0; i < gUiData.UF->colors; ++i) {
		gUiData.DespeckleDecayAdj[i] = GTK_ADJUSTMENT(gtk_adjustment_new(gConf->despeckleDecay[i], 0.0, 1.0, 0.1, 0.1, 0));
		g_object_set_data(G_OBJECT(gUiData.DespeckleDecayAdj[i]), "Adjustment-Accuracy", (gpointer)2);
		button = gtk_spin_button_new(gUiData.DespeckleDecayAdj[i], 1.0, 2);
		g_object_set_data(G_OBJECT(gUiData.DespeckleDecayAdj[i]), "Parent-Widget", button);
		g_signal_connect(G_OBJECT(gUiData.DespeckleDecayAdj[i]), "value-changed", G_CALLBACK(adjustment_update), &gConf->despeckleDecay[i]);
		gtk_table_attach(table, button, 6 + i, 6 + i + 1, 2, 3, GTK_FILL | GTK_EXPAND, 0, 0, 0);
	}

	label = gtk_label_new(_("Passes"));
	gtk_misc_set_alignment(GTK_MISC(label), 1, 1);
	gtk_table_attach(table, label, 0, 6, 3, 4, 0, 0, 0, 0);
	for (i = 0; i < gUiData.UF->colors; ++i) {
		gUiData.DespecklePassesAdj[i] = GTK_ADJUSTMENT(gtk_adjustment_new(gConf->despecklePasses[i],0.0, 99.0, 1.0, 1.0, 0));
		g_object_set_data(G_OBJECT(gUiData.DespecklePassesAdj[i]),"Adjustment-Accuracy", (gpointer)0);
		button = gtk_spin_button_new(gUiData.DespecklePassesAdj[i], 1.0, 0);
		g_object_set_data(G_OBJECT(gUiData.DespecklePassesAdj[i]),"Parent-Widget", button);
		g_signal_connect(G_OBJECT(gUiData.DespecklePassesAdj[i]), "value-changed",G_CALLBACK(adjustment_update), &gConf->despecklePasses[i]);
		gtk_table_attach(table, button, 6 + i, 6 + i + 1, 3, 4, GTK_FILL | GTK_EXPAND, 0, 0, 0);
	}
}

static void output_fill_interface(GtkTable *table)
{
	GtkWidget  *label, *button;
	uf_long i;

	// shrink
	gUiData.shrink = gConf->shrink;
	if (gConf->size == 0) {
		gUiData.width = 0;
		gUiData.height = 0;
	} else {
		gUiData.shrink = 0;
		if (gUiData.UF->rotatedWidth > gUiData.UF->rotatedHeight) {
			gUiData.width = gConf->size;
			gUiData.height = gConf->size * gUiData.UF->rotatedHeight / gUiData.UF->rotatedWidth;
		} else {
			gUiData.width = gConf->size * gUiData.UF->rotatedWidth / gUiData.UF->rotatedHeight;
			gUiData.height = gConf->size;
		}
	}

	gUiData.ShrinkAdjustment = uf_adjustment_scale(
		                  table, 0, 1, _("Shrink factor"),
		                  gUiData.shrink, &gUiData.shrink,
		                  1, 10, 0.001, 0.1, 3, FALSE,
		                  "", G_CALLBACK(adjustment_update),
		                  &gUiData.ResetShrink, "",G_CALLBACK(reset_button_cb));

	gUiData.WidthAdjustment = uf_adjustment_scale(
		                  table, 0, 2, _("Width"),
		                  0, &gUiData.width,
		                  0, 99999, 10, 100, 0, FALSE,
		                  "", G_CALLBACK(adjustment_update),
		                  NULL, "", NULL);

	gUiData.HeightAdjustment = uf_adjustment_scale(
		                  table, 0, 3, _("Height"),
		                  0, &gUiData.height,
		                  0, 99999, 10, 100, 0, FALSE,
		                  "", G_CALLBACK(adjustment_update),
		                  NULL, "", NULL);

	// output profile
	label = gtk_label_new(_("ICC profile"));
	gtk_table_attach(table, label, 0, 1, 4, 5, GTK_SHRINK|GTK_FILL, 0, 0, 0);
	gtk_misc_set_alignment(GTK_MISC(label), 1.0 ,0.5);
	gUiData.ProfileCombo[out_profile] = GTK_COMBO_BOX(uf_combo_box_new_text());
	for (i = 0; i < gConf->profileCount[out_profile]; i++){
		if (i < conf_default.profileCount[out_profile]) {
			gtk_combo_box_append_text(gUiData.ProfileCombo[out_profile],_(gConf->profile[out_profile][i].name));
		} else {
			gtk_combo_box_append_text(gUiData.ProfileCombo[out_profile], gConf->profile[out_profile][i].name);
		}
	}
	uf_combo_box_set_data(gUiData.ProfileCombo[out_profile], &gConf->profileIndex[out_profile]);
	gtk_table_attach(table, GTK_WIDGET(gUiData.ProfileCombo[out_profile]),  1, 3, 4, 5, GTK_FILL, GTK_FILL, 0, 0);
	button = uf_stock_icon_button(GTK_STOCK_OPEN, NULL, G_CALLBACK(load_profile), (gpointer)out_profile);
	gtk_table_attach(table, button, 3, 4, 4, 5, GTK_SHRINK, GTK_FILL, 0, 0);

	// output intent
	label = gtk_label_new(_("Intent"));
	gtk_table_attach(table, label, 0, 1, 5, 6, 0, 0, 0, 0);
	gtk_misc_set_alignment(GTK_MISC(label), 1.0 ,0.5);
	gUiData.OutputIntentCombo = GTK_COMBO_BOX(uf_combo_box_new_text());
	gtk_combo_box_append_text(gUiData.OutputIntentCombo, _("Perceptual"));
	gtk_combo_box_append_text(gUiData.OutputIntentCombo, _("Relative colorimetric"));
	gtk_combo_box_append_text(gUiData.OutputIntentCombo, _("Saturation"));
	gtk_combo_box_append_text(gUiData.OutputIntentCombo, _("Absolute colorimetric"));
	uf_combo_box_set_data(GTK_COMBO_BOX(gUiData.OutputIntentCombo),(int *)&gConf->intent[out_profile]);
	gtk_table_attach(table, GTK_WIDGET(gUiData.OutputIntentCombo), 1, 3, 5, 6, GTK_EXPAND|GTK_FILL, 0, 0, 0);

	// output depth
	label = gtk_label_new(_("Bit depth"));
	gtk_table_attach(table, label, 0, 1, 6, 7, 0, 0, 0, 0);
	gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);

	gUiData.BitDepthCombo = GTK_COMBO_BOX(uf_combo_box_new_text());
	uf_combo_box_append_text(gUiData.BitDepthCombo, "8", (void*)8);
#if !HAVE_GIMP_2_9
	if (gUiData.plugin != nufraw_gimp_plugin)
#endif
		uf_combo_box_append_text(gUiData.BitDepthCombo, "16", (void*)16);

	uf_combo_box_set_data(GTK_COMBO_BOX(gUiData.BitDepthCombo),&gConf->profile[out_profile][gConf->profileIndex[out_profile]].BitDepth);
	gtk_table_attach(table, GTK_WIDGET(gUiData.BitDepthCombo), 1, 3, 6, 7, GTK_EXPAND|GTK_FILL, 0, 0, 0);
}

static void exif_fill_interface(GtkTable *table)
{
	GtkWidget *label;

	if (gUiData.UF->inputExifBuf == NULL) {
		label = gtk_label_new(NULL);
		char *text = g_strdup_printf("<span foreground='red'>%s</span>",
				             _("Warning: EXIF data will not be sent to output"));
		gtk_label_set_markup(GTK_LABEL(label), text);
		g_free(text);
		gtk_table_attach(table, GTK_WIDGET(label), 0, 1, 0, 1, 0, 0, 0, 0);
	}

	label = gtk_label_new(NULL);
	gchar *message = g_strdup_printf(_("EXIF data read by %s"),gConf->exifSource);
	gtk_label_set_markup(GTK_LABEL(label), message);
	gtk_table_attach(table, GTK_WIDGET(label), 0, 1, 1, 2, 0, 0, 0, 0);
	g_free(message);

	GtkListStore *store = gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_STRING);
	GtkWidget *treeview = gtk_tree_view_new_with_model(GTK_TREE_MODEL(store));
	gtk_tree_view_set_search_column(GTK_TREE_VIEW(treeview), 0);
	gtk_tree_view_set_rules_hint(GTK_TREE_VIEW(treeview), TRUE);
	g_object_unref(store);

	gtk_table_attach(table, GTK_WIDGET(treeview), 0, 1, 2, 3, GTK_EXPAND|GTK_FILL, GTK_EXPAND|GTK_FILL, 0, 0);

	GtkTreeViewColumn *column = gtk_tree_view_column_new_with_attributes(
		                        _("Tag"), gtk_cell_renderer_text_new(), "text", 0, NULL);
	gtk_tree_view_column_set_sort_column_id(column, 0);
	gtk_tree_view_column_set_resizable(column,TRUE);
	gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), column);

	column = gtk_tree_view_column_new_with_attributes(
		     _("Value"), gtk_cell_renderer_text_new(), "text", 1, NULL);
	gtk_tree_view_column_set_sort_column_id(column, 1);
	gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), column);

	// Fill table with EXIF tags
	list_store_add(store, _("Camera maker"), gConf->make);
	list_store_add(store, _("Camera model"), gConf->model);
	list_store_add(store, _("Timestamp"), gConf->timestampText);
	list_store_add(store, _("Shutter time"), gConf->shutterText);
	list_store_add(store, _("Aperture"), gConf->apertureText);
	list_store_add(store, _("ISO speed"), gConf->isoText);
	list_store_add(store, _("Focal length"), gConf->focalLenText);
	list_store_add(store, _("35mm focal length"), gConf->focalLen35Text);
	list_store_add(store, _("Lens"), gConf->lensText);
	list_store_add(store, _("Flash"), gConf->flashText);
	list_store_add(store, _("White balance"), gConf->whiteBalanceText);
}

int nufraw_ui(nufraw_data *uf, ui_conf *uiconf, int plugin, long(*save_func)())
{
	GtkWidget *previewWindow, *leftBoxUp, *leftBoxDown,  *rightBox;
	GtkPaned *hPaned;
	GtkPaned *vPaned;
	GtkScrolledWindow *leftScrollUp;
	GtkScrolledWindow *leftScrollDown;

	GtkTable *table;
	GtkBox *box,*hbox;
	GtkWidget *button;
	int i;
	uf_long status;

	// Fill the whole structure with zeros, to avoid surprises
	memset(&gUiData, 0, sizeof(gUiData));

	gUiConf=uiconf;
	gUiData.UF = uf;
	gUiData.SaveFunc = save_func;
	gUiData.plugin=plugin;
	gUiData.LeftSpace=LIM(gUiConf->left_size-40,UI_HIS_CURVE_MINWIDTH,UI_HIS_CURVE_MAXWIDTH);
	gConf=gUiData.UF->conf;

	// init defaults
	gUiData.SpotX1 = -1;
	gUiData.SpotX2 = -1;
	gUiData.SpotY1 = -1;
	gUiData.SpotY2 = -1;
	gUiData.SpotDraw = FALSE;
	gUiData.FreezeDialog = TRUE;
	gUiData.DrawnCropX1 = 0;
	gUiData.DrawnCropX2 = 99999;
	gUiData.DrawnCropY1 = 0;
	gUiData.DrawnCropY2 = 99999;

	gUiData.BlinkTimer = 0;
	gUiData.DrawCropID = 0;

	// MAIN WINDOW ///////////////////////////////////////////////////////////

	previewWindow = gtk_window_new(GTK_WINDOW_TOPLEVEL);

	gtk_window_set_default_size(GTK_WINDOW(previewWindow), gUiConf->win_width, gUiConf->win_height);

	char *utf8_filename = g_filename_display_name(uf->filename);
	char *previewHeader = g_strdup_printf(_("%s - nUFRaw"), utf8_filename);
	gtk_window_set_title(GTK_WINDOW(previewWindow), previewHeader);
	g_free(previewHeader);
	g_free(utf8_filename);

	nufraw_icons_init();
	gtk_window_set_icon_name(GTK_WINDOW(previewWindow), "nufraw");
	g_signal_connect(G_OBJECT(previewWindow), "delete-event",G_CALLBACK(window_delete_event), NULL);
	g_signal_connect(G_OBJECT(previewWindow), "map-event",G_CALLBACK(window_map_event), NULL);
	g_signal_connect(G_OBJECT(previewWindow), "unmap-event",G_CALLBACK(window_unmap_event), NULL);
	nufraw_focus(previewWindow, TRUE);

	hPaned = GTK_PANED(gtk_hpaned_new());
	gtk_paned_set_position(hPaned,gUiConf->left_size);
	g_signal_connect(G_OBJECT(hPaned), "notify::position", G_CALLBACK(paned_resized), GTK_PANED(hPaned));
	gtk_container_add(GTK_CONTAINER(previewWindow), GTK_WIDGET(hPaned));

	// LEFT SIDE ///////////////////////////////////////////////////////////////

	vPaned = GTK_PANED(gtk_vpaned_new());
	gtk_paned_add1(GTK_PANED(hPaned),GTK_WIDGET(vPaned));
	gtk_paned_set_position(vPaned,gUiConf->up_size);

	leftScrollUp = GTK_SCROLLED_WINDOW(gtk_scrolled_window_new(NULL,NULL));
	leftBoxUp = gtk_vbox_new(FALSE, 0);
	gtk_scrolled_window_set_policy (leftScrollUp,GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_add_with_viewport(leftScrollUp, GTK_WIDGET(leftBoxUp));
	gtk_paned_add1(GTK_PANED(vPaned),GTK_WIDGET(leftScrollUp));

	leftScrollDown = GTK_SCROLLED_WINDOW(gtk_scrolled_window_new(NULL,NULL));
	leftBoxDown = gtk_vbox_new(FALSE, 0);
	gtk_scrolled_window_set_policy (leftScrollDown,GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_add_with_viewport(leftScrollDown, GTK_WIDGET(leftBoxDown));
	gtk_paned_add2(GTK_PANED(vPaned),GTK_WIDGET(leftScrollDown));

	g_signal_connect(G_OBJECT(leftBoxDown), "size-allocate",G_CALLBACK(panel_size_allocate), NULL);

	// Expanders
	int index=0;

	table = uf_expander_with_frame(leftBoxUp, _("Input"), gUiConf->expander_open[index++], NULL, FALSE);
	input_fill_interface(table);

	table = uf_expander_with_frame(leftBoxUp, _("Exposure"), gUiConf->expander_open[index++], NULL, FALSE);
	exposure_fill_interface(table);

	table = uf_expander_with_frame(leftBoxUp, _("White balance"), gUiConf->expander_open[index++], NULL, FALSE);
	whitebalance_fill_interface(table);

	table = uf_expander_with_frame(leftBoxUp, _("Grayscale conversion"), gUiConf->expander_open[index++], NULL, FALSE);
	grayscale_fill_interface(table);

	table = uf_expander_with_frame(leftBoxUp, _("Luminosity and saturation"), gUiConf->expander_open[index++], NULL, FALSE);
	corrections_fill_interface(table);

	table = uf_expander_with_frame(leftBoxUp, _("Lightness adjustments"), gUiConf->expander_open[index++], NULL, FALSE);
	lightness_fill_interface(table);

	table = uf_expander_with_frame(leftBoxUp, _("Crop and rotate"), gUiConf->expander_open[index++], NULL, FALSE);
	transformations_fill_interface(table);

#ifdef HAVE_LENSFUN
	table = uf_expander_with_frame(leftBoxUp, _("Lens correction"), gUiConf->expander_open[index++], NULL, FALSE);
	lens_fill_interface(&gUiData, table);
#endif /* HAVE_LENSFUN */

	table = uf_expander_with_frame(leftBoxUp, _("Denoising"), gUiConf->expander_open[index++], NULL, FALSE);
	denoise_fill_interface(table);

	table = uf_expander_with_frame(leftBoxUp, _("Despeckling"), gUiConf->expander_open[index++], G_CALLBACK(despeckling_expander), FALSE);
	despeckling_fill_interface(table);

	table = uf_expander_with_frame(leftBoxUp, _("Output"), gUiConf->expander_open[index++], NULL, FALSE);
	output_fill_interface(table);

	table = uf_expander_with_frame(leftBoxDown, _("Raw histogram and conversion curves"), gUiConf->expander_open[index++], G_CALLBACK(histogram_expander), gUiConf->expander_open[18]);
	rawhistogram_fill_interface(table);

	table = uf_expander_with_frame(leftBoxDown, _("Live histogram"), gUiConf->expander_open[index++], G_CALLBACK(histogram_expander), gUiConf->expander_open[19]);
	livehistogram_fill_interface(table);

	table = uf_expander_with_frame(leftBoxDown, _("Live values"), gUiConf->expander_open[index++], NULL, FALSE);
	valuetable_fill_interface(table);

	table = uf_expander_with_frame(leftBoxDown, _("Exif data"),gUiConf->expander_open[index++], NULL, FALSE);
	exif_fill_interface(table);

	// RIGHT SIDE ////////////////////////////////////////////////////////////

	rightBox = gtk_vbox_new(FALSE, 0);
	gtk_paned_add2(GTK_PANED(hPaned),rightBox);

	// top buttons
	hbox = GTK_BOX(gtk_hbox_new(FALSE, 0));
	gtk_box_pack_start(GTK_BOX(rightBox), GTK_WIDGET(hbox), FALSE, FALSE, 6);

	// Zoom bar
	table = GTK_TABLE(gtk_table_new(1,11, FALSE));
	gtk_box_pack_start(GTK_BOX(hbox), GTK_WIDGET(table), TRUE, TRUE, 0);

	// Zoom adjust
	gUiData.ZoomAdjustment = uf_adjustment_spin(table, 0, 0, _("Zoom"),
	                                             gUiConf->Zoom, &gUiConf->Zoom,
	                                             UI_MIN_ZOOM, UI_MAX_ZOOM,
	                                             1, 1, 0, TRUE, _("Zoom percentage"),
	                                             G_CALLBACK(zoom_update),
	                                             NULL, NULL, NULL);

	// Zoom out button
	button = uf_stock_icon_button(GTK_STOCK_ZOOM_OUT, NULL, G_CALLBACK(zoom_out_event), NULL);
	gtk_table_attach(table, button, 2, 3, 0, 1, 0, 0, 0, 0);

	// Zoom in button
	button = uf_stock_icon_button(GTK_STOCK_ZOOM_IN, NULL, G_CALLBACK(zoom_in_event), NULL);
	gtk_table_attach(table, button, 3, 4, 0, 1, 0, 0, 0, 0);

	// Zoom fit button
	button = uf_stock_icon_button(GTK_STOCK_ZOOM_FIT, NULL, G_CALLBACK(zoom_fit_event), NULL);
	gtk_table_attach(table, button, 4, 5, 0, 1, 0, 0, 0, 0);

	// Zoom max button
	button = uf_stock_icon_button(GTK_STOCK_ZOOM_100, NULL, G_CALLBACK(zoom_100_event), NULL);
	gtk_table_attach(table, button, 5, 6, 0, 1, 0, 0, 0, 0);

	// grid adjustment
	box = GTK_BOX(gtk_hbox_new(TRUE, 0));
	gtk_box_pack_start(GTK_BOX(hbox), GTK_WIDGET(box), TRUE, TRUE, 0);

	table = GTK_TABLE(gtk_table_new(1,2, FALSE));
	gtk_box_pack_start(GTK_BOX(box), GTK_WIDGET(table), TRUE, TRUE, 0);
	uf_adjustment_scale(GTK_TABLE(table), 0, 0, _("Grid"),
		              gUiConf->gridCount, &gUiConf->gridCount,
		              0, 20, 1, 1, 0, FALSE,
		              _("Number of grid lines to overlay in the crop area"),
		              G_CALLBACK(adjustment_update_int), NULL, NULL, NULL);


	// special render buttons
	box = GTK_BOX(gtk_hbox_new(FALSE, 0));
	gtk_box_pack_start(GTK_BOX(hbox), GTK_WIDGET(box), FALSE, FALSE, 6);

	button = gtk_button_new();
	gtk_container_add(GTK_CONTAINER(button),gtk_image_new_from_stock("object-grayscale", UI_BT_SIZE));
	gtk_widget_set_tooltip_text(button, _("Show luminance zones"));
	gtk_box_pack_start(GTK_BOX(box), GTK_WIDGET(button), FALSE, FALSE, 0);
	g_signal_connect(G_OBJECT(button), "pressed", G_CALLBACK(render_special_mode), (void*)render_zone);
	g_signal_connect(G_OBJECT(button), "released", G_CALLBACK(render_special_mode), (void*)render_default);

	button = gtk_button_new();
	gtk_container_add(GTK_CONTAINER(button),gtk_image_new_from_stock("renderover", UI_BT_SIZE));
	gtk_widget_set_tooltip_text(button, _("Show overexposure"));
	gtk_box_pack_start(GTK_BOX(box), GTK_WIDGET(button), FALSE, FALSE, 0);
	g_signal_connect(G_OBJECT(button), "pressed", G_CALLBACK(render_special_mode), (void*)render_overexposed);
	g_signal_connect(G_OBJECT(button), "released", G_CALLBACK(render_special_mode), (void*)render_default);

	button = gtk_button_new();
	gtk_container_add(GTK_CONTAINER(button),gtk_image_new_from_stock("renderunder", UI_BT_SIZE));
	gtk_widget_set_tooltip_text(button, _("Show underexposure"));
	gtk_box_pack_start(GTK_BOX(box), GTK_WIDGET(button), FALSE, FALSE, 0);
	g_signal_connect(G_OBJECT(button), "pressed", G_CALLBACK(render_special_mode), (void *)render_underexposed);
	g_signal_connect(G_OBJECT(button), "released", G_CALLBACK(render_special_mode), (void *)render_default);

	// blink buttons
	box = GTK_BOX(gtk_hbox_new(FALSE, 0));
	gtk_box_pack_start(GTK_BOX(hbox), GTK_WIDGET(box), FALSE, FALSE, 6);

	button=gtk_check_button_new();
	gtk_container_add(GTK_CONTAINER(button),gtk_image_new_from_stock("warnover", UI_BT_SIZE));
	gtk_toggle_button_set_mode(GTK_TOGGLE_BUTTON(button), FALSE);
	gtk_widget_set_tooltip_text(button, _("Overexpose warning"));
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), gUiConf->showOverExp);
	g_signal_connect(G_OBJECT(button), "toggled",G_CALLBACK(toggle_button_cb), (gboolean *)(&gUiConf->showOverExp));
	gtk_box_pack_start(GTK_BOX(box), GTK_WIDGET(button), FALSE, FALSE, 0);

	button=gtk_check_button_new();
	gtk_toggle_button_set_mode(GTK_TOGGLE_BUTTON(button), FALSE);
	gtk_container_add(GTK_CONTAINER(button),gtk_image_new_from_stock("warnunder", UI_BT_SIZE));
	gtk_widget_set_tooltip_text(button, _("Underexpose warning"));
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), gUiConf->showUnderExp);
	g_signal_connect(G_OBJECT(button), "toggled",G_CALLBACK(toggle_button_cb), (gboolean *)(&gUiConf->showUnderExp));
	gtk_box_pack_start(GTK_BOX(box), GTK_WIDGET(button), FALSE, FALSE, 0);

	// curson buttons
	box = GTK_BOX(gtk_hbox_new(FALSE, 0));
	gtk_box_pack_start(GTK_BOX(hbox), GTK_WIDGET(box), FALSE, FALSE, 6);

	gUiData.CursorButton[0]=gtk_radio_button_new_from_widget(NULL);
	gtk_toggle_button_set_mode(GTK_TOGGLE_BUTTON(gUiData.CursorButton[0]), FALSE);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gUiData.CursorButton[0]), gUiConf->cursor_mode==0);
	gtk_container_add(GTK_CONTAINER(gUiData.CursorButton[0]), gtk_image_new_from_stock(GTK_STOCK_INDEX,UI_BT_SIZE));
	gtk_box_pack_start(GTK_BOX(box), GTK_WIDGET(gUiData.CursorButton[0]), FALSE, FALSE, 0);
	g_signal_connect(G_OBJECT(gUiData.CursorButton[0]), "toggled",G_CALLBACK(cursor_button_cb), NULL);
	gtk_widget_set_tooltip_text(gUiData.CursorButton[0],_("Drag cursor"));

	gUiData.CursorButton[1]=gtk_radio_button_new_from_widget(GTK_RADIO_BUTTON(gUiData.CursorButton[0]));
	gtk_toggle_button_set_mode(GTK_TOGGLE_BUTTON(gUiData.CursorButton[1]), FALSE);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gUiData.CursorButton[1]), gUiConf->cursor_mode==1);
	gtk_container_add(GTK_CONTAINER(gUiData.CursorButton[1]), gtk_image_new_from_stock(GTK_STOCK_COLOR_PICKER,UI_BT_SIZE));
	gtk_box_pack_start(GTK_BOX(box), GTK_WIDGET(gUiData.CursorButton[1]), FALSE, FALSE, 0);
	g_signal_connect(G_OBJECT(gUiData.CursorButton[1]), "toggled",G_CALLBACK(cursor_button_cb), NULL);
	gtk_widget_set_tooltip_text(gUiData.CursorButton[1],_("Color picker cursor"));

	gUiData.CursorButton[2]=gtk_radio_button_new_from_widget(GTK_RADIO_BUTTON(gUiData.CursorButton[1]));
	gtk_toggle_button_set_mode(GTK_TOGGLE_BUTTON(gUiData.CursorButton[2]), FALSE);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gUiData.CursorButton[2]), gUiConf->cursor_mode==2);
	gtk_container_add(GTK_CONTAINER(gUiData.CursorButton[2]), gtk_image_new_from_stock(GTK_STOCK_CUT, UI_IC_SIZE));
	gtk_box_pack_start(GTK_BOX(box), GTK_WIDGET(gUiData.CursorButton[2]), FALSE, FALSE, 0);
	g_signal_connect(G_OBJECT(gUiData.CursorButton[2]), "toggled",G_CALLBACK(cursor_button_cb), NULL);
	gtk_widget_set_tooltip_text(gUiData.CursorButton[2],_("Cut cursor"));

	// the preview
	gUiData.PreviewEventBox = gtk_event_box_new();
	gUiData.PreviewWidget = gtk_image_view_new();
	gtk_image_view_set_interpolation(GTK_IMAGE_VIEW(gUiData.PreviewWidget), GDK_INTERP_NEAREST);
	gtk_image_view_set_zoom(GTK_IMAGE_VIEW(gUiData.PreviewWidget), 1.0);

	gUiData.PreviewScroll = gtk_image_scroll_win_new(GTK_IMAGE_VIEW(gUiData.PreviewWidget));
	GtkWidget *container = gtk_widget_get_ancestor(gUiData.PreviewWidget, GTK_TYPE_TABLE);
	g_object_ref(G_OBJECT(gUiData.PreviewWidget));
	gtk_container_remove(GTK_CONTAINER(container), gUiData.PreviewWidget);
	gtk_container_add(GTK_CONTAINER(gUiData.PreviewEventBox), gUiData.PreviewWidget);
	g_object_unref(G_OBJECT(gUiData.PreviewWidget));
	gtk_table_attach(GTK_TABLE(container), gUiData.PreviewEventBox, 0, 1, 0, 1, GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
	gtk_box_pack_start(GTK_BOX(rightBox), gUiData.PreviewScroll, TRUE, TRUE, 0);

	gUiData.PreviewButtonPressed = FALSE;
	g_signal_connect(G_OBJECT(gUiData.PreviewEventBox), "button-press-event",G_CALLBACK(preview_button_press_event), NULL);
	g_signal_connect(G_OBJECT(gUiData.PreviewEventBox), "button-release-event",G_CALLBACK(preview_button_release_event), NULL);
	g_signal_connect(G_OBJECT(gUiData.PreviewEventBox), "motion-notify-event", G_CALLBACK(preview_motion_notify_event), NULL);
	gtk_widget_add_events(gUiData.PreviewEventBox, GDK_POINTER_MOTION_MASK);

	// Hide zoom key bindings from GtkImageView
	GtkImageViewClass *klass = GTK_IMAGE_VIEW_GET_CLASS(GTK_IMAGE_VIEW(gUiData.PreviewWidget));
	GtkBindingSet *binding_set = gtk_binding_set_by_class(klass);
	gtk_binding_entry_remove(binding_set, GDK_1, 0);
	gtk_binding_entry_remove(binding_set, GDK_2, 0);
	gtk_binding_entry_remove(binding_set, GDK_3, 0);
	gtk_binding_entry_remove(binding_set, GDK_plus, 0);
	gtk_binding_entry_remove(binding_set, GDK_equal, 0);
	gtk_binding_entry_remove(binding_set, GDK_KP_Add, 0);
	gtk_binding_entry_remove(binding_set, GDK_minus, 0);
	gtk_binding_entry_remove(binding_set, GDK_KP_Subtract, 0);
	gtk_binding_entry_remove(binding_set, GDK_x, 0);

	// GtkImageView should only get the scroll up/down events
	GtkWidgetClass *widget_class = (GtkWidgetClass *)klass;
	gtk_image_view_scroll_event = widget_class->scroll_event;
	widget_class->scroll_event = preview_scroll_event;

	gUiData.ProgressBar = GTK_PROGRESS_BAR(gtk_progress_bar_new());
	gtk_box_pack_start(GTK_BOX(rightBox), GTK_WIDGET(gUiData.ProgressBar), FALSE, FALSE, 0);

	// bottom buttons
	hbox = GTK_BOX(gtk_hbox_new(FALSE, 0));
	gtk_box_pack_start(GTK_BOX(rightBox), GTK_WIDGET(hbox), FALSE, FALSE, 6);

	// Global reset
	gUiData.ResetAllButton = uf_stock_icon_button(GTK_STOCK_REFRESH,_("Reset to program default"), G_CALLBACK(program_reset_cb), NULL);
	gtk_box_pack_start(hbox, gUiData.ResetAllButton, FALSE, FALSE, 0);
	button = uf_stock_icon_button(GTK_STOCK_SAVE,_("Store user default") , G_CALLBACK(user_store_cb), NULL);
	gtk_box_pack_start(hbox, button, FALSE, FALSE, 0);
	button = uf_stock_icon_button(GTK_STOCK_REVERT_TO_SAVED,_("Reset to user default") , G_CALLBACK(user_reset_cb), NULL);
	gtk_box_pack_start(hbox, button, FALSE, FALSE, 0);
	button = uf_stock_icon_button(GTK_STOCK_OPEN,_("Reset to ID file") , G_CALLBACK(id_reset_cb), NULL);
	gtk_box_pack_start(hbox, button, FALSE, FALSE, 0);

	// Options button
	button = uf_stock_icon_button(GTK_STOCK_PREFERENCES, _("Options"), G_CALLBACK(options_button_cb), NULL);
	gtk_box_pack_start(hbox, button, FALSE, FALSE, 0);

	// spot table
	box = GTK_BOX(gtk_hbox_new(FALSE, 0)); // spacer box
	gtk_box_pack_start(GTK_BOX(hbox), GTK_WIDGET(box), TRUE, TRUE, 0);
	box = GTK_BOX(gtk_hbox_new(FALSE, 0));
	gtk_box_pack_start(GTK_BOX(hbox), GTK_WIDGET(box), FALSE, FALSE, 0);
	gtk_container_add(GTK_CONTAINER(box), gtk_image_new_from_stock(GTK_STOCK_COLOR_PICKER,UI_BT_SIZE));
	table = GTK_TABLE(uf_table_with_frame(GTK_WIDGET(box)));
	gUiData.SpotPatch = GTK_LABEL(gtk_label_new(NULL));
	gtk_table_attach(table, GTK_WIDGET(gUiData.SpotPatch), 0, 1, 1, 2, GTK_FILL|GTK_EXPAND, 0, 0, 0);
	gUiData.SpotLabels = color_labels_new(table, 1, 1, NULL, pixel_format, TRUE);
	double rgb[5]={0,0,0,0,0};
	color_labels_set(gUiData.SpotLabels, rgb);
	uf_label_set_background(gUiData.SpotPatch,rgb[0], rgb[1], rgb[2], 255);
	box = GTK_BOX(gtk_hbox_new(FALSE, 0)); // spacer box
	gtk_box_pack_start(GTK_BOX(hbox), GTK_WIDGET(box), TRUE, TRUE, 0);

	// buttons on the right
	box = GTK_BOX(gtk_hbox_new(FALSE, 0));
	gtk_box_pack_start(GTK_BOX(hbox), GTK_WIDGET(box), FALSE, FALSE, 6);

	if (gUiData.plugin == nufraw_standalone) {

		// Send to Gimp button
		button = uf_stock_image_button("gimp", UI_BT_SIZE, _("Send to Gimp"),G_CALLBACK(control_button_cb), (gpointer)gimp_button);
		gtk_box_pack_start(box, button, FALSE, FALSE, 0);

		// Save
		gUiData.SaveButton = uf_stock_icon_button(GTK_STOCK_SAVE_AS, _("Save"),G_CALLBACK(control_button_cb), (gpointer)save_button);
		gtk_box_pack_start(box, gUiData.SaveButton, FALSE, FALSE, 0);

	}else{
		// OK button for the plug-in
		gUiData.SaveButton = uf_stock_icon_button(GTK_STOCK_OK, _("Save"),G_CALLBACK(control_button_cb), (gpointer)ok_button);
		gtk_box_pack_start(box, gUiData.SaveButton, FALSE, FALSE, 0);
	}

	// Cancel button
	button = uf_stock_icon_button(GTK_STOCK_CLOSE, _("Close"),G_CALLBACK(window_delete_event), NULL);
	gtk_box_pack_start(box, button, FALSE, FALSE, 0);

	gtk_widget_show_all(previewWindow);

	for (i = gConf->lightnessAdjustmentCount; i < UF_MAX_LIGHTNESS_ADJUSTMENTS; i++)
		gtk_widget_hide(GTK_WIDGET(gUiData.LightnessAdjustmentTable[i]));

	// Allocate the preview pixbuf - size don't matter because will be resized
	gUiData.PreviewPixbuf = gdk_pixbuf_new(GDK_COLORSPACE_RGB, FALSE, 8, 10, 10);
	gdk_pixbuf_fill(gUiData.PreviewPixbuf, 0);
	gtk_image_view_set_pixbuf(GTK_IMAGE_VIEW(gUiData.PreviewWidget), gUiData.PreviewPixbuf, FALSE);
	g_object_unref(gUiData.PreviewPixbuf);

	// allocate cursors
	for (i = 0; i < cursor_num; i++) gUiData.Cursor[i] = gdk_cursor_new(Cursors[i]);
	cursor_button_cb(NULL, gUiData.CursorButton[gUiConf->cursor_mode]);

	preview_progress_enable();
	nufraw_load_raw(uf);
	preview_progress_disable();

	// prevents division by zero, Should only happen if nufraw_load_raw() failed:
	if (gUiData.UF->rgbMax == 0) gUiData.UF->rgbMax = 0xffff;

	// After window size was set, the user may want to re-size it.
	// This function is called after the progress-bar text was set,
	// to make sure that there are no scroll-bars on the initial preview.
	gtk_widget_set_size_request(gUiData.PreviewScroll, -1, -1);

	// Set shrink/size values for preview rendering
	gConf->shrink = zoom_to_shrink(gUiConf->Zoom);
	update_size();

	nufraw_invalidate_layer(gUiData.UF, nufraw_first_phase);

	// Save initial WB data for the sake of "Reset WB"
	// TODO: reset to camera WB if any
	UFObject *Image = gConf->ufobject;
	ufobject_set_default(ufgroup_element(Image, ufChannelMultipliers));
	ufobject_set_default(ufgroup_element(Image, ufWB));
	ufobject_set_default(ufgroup_element(Image, ufWBFineTuning));

	// Update the curve editor in case nufraw_convert_image() modified it.
	curveeditor_widget_set_curve(gUiData.CurveWidget, &gConf->curve[gConf->curveIndex]);

	gUiData.RenderSubArea = -1;
	gUiData.FreezeDialog = FALSE;
	gUiData.RenderMode = render_default;
	gUiData.OverUnderTicker = 0;

	// start the conversion and enqueue rendering functions
	update_scales();
	render_preview_now(NULL);
	update_crop_ranges(FALSE);
	collect_raw_histogram_data();

	// GTK MAIN //////////////////////////////////////////////////////////////
	gImageHasModifications=FALSE;
	gtk_main();

	// get exit status
	status = (uf_long)g_object_get_data(G_OBJECT(previewWindow),"WindowResponse");

	// get expander state
	int expIndex=0;
	gtk_container_foreach(GTK_CONTAINER(leftBoxUp),(GtkCallback)(get_expander_state), &expIndex);
	gtk_container_foreach(GTK_CONTAINER(leftBoxDown),(GtkCallback)(get_expander_state), &expIndex);

	// get ui_conf values
	gtk_window_get_size(GTK_WINDOW(previewWindow),&gUiConf->win_width,&gUiConf->win_height);
	gUiConf->left_size=gtk_paned_get_position(hPaned);
	gUiConf->up_size=gtk_paned_get_position(vPaned);

	nufraw_focus(previewWindow, FALSE);
	gtk_widget_destroy(previewWindow);
	for (i = 0; i < cursor_num; i++) gdk_cursor_unref(gUiData.Cursor[i]);

	// Make sure that there are no preview idle task remaining
	while (g_idle_remove_by_data(&gUiData));

	if (gUiData.BlinkTimer) {
		g_source_remove(gUiData.BlinkTimer);
		gUiData.BlinkTimer = 0;
	}

	//if (gUiData.DrawCropID != 0) g_source_remove(gUiData.DrawCropID);

	if (gConf->darkframe!=NULL) nufraw_close_darkframe(gConf);
	nufraw_close(gUiData.UF);

	g_free(gUiData.SpotLabels);
	g_free(gUiData.DevLabels);
	g_free(gUiData.OverLabels);
	g_free(gUiData.UnderLabels);

	if (status != GTK_RESPONSE_OK) return NUFRAW_CANCEL;
	return NUFRAW_SUCCESS;
}
