// Copyright 2019 Drone.IO Inc. All rights reserved.
// Use of this source code is governed by the Drone Non-Commercial License
// that can be found in the LICENSE file.

// +build !oss

package crons

import (
	"encoding/json"
	"net/http"

	"github.com/drone/drone/core"
	"github.com/drone/drone/handler/api/render"

	"github.com/go-chi/chi"
)

type cronUpdate struct {
	Branch   *string `json:"branch"`
	Target   *string `json:"target"`
	Disabled *bool   `json:"disabled"`
}

// HandleUpdate returns an http.HandlerFunc that processes http
// requests to enable or disable a cron job.
func HandleUpdate(
	repos core.RepositoryStore,
	crons core.CronStore,
) http.HandlerFunc {
	return func(w http.ResponseWriter, r *http.Request) {
		var (
			namespace = chi.URLParam(r, "owner")
			name      = chi.URLParam(r, "name")
			cron      = chi.URLParam(r, "cron")
		)
		repo, err := repos.FindName(r.Context(), namespace, name)
		if err != nil {
			render.NotFound(w, err)
			return
		}
		cronjob, err := crons.FindName(r.Context(), repo.ID, cron)
		if err != nil {
			render.NotFound(w, err)
			return
		}

		in := new(cronUpdate)
		json.NewDecoder(r.Body).Decode(in)
		if in.Branch != nil {
			cronjob.Branch = *in.Branch
		}
		if in.Target != nil {
			cronjob.Target = *in.Target
		}
		if in.Disabled != nil {
			cronjob.Disabled = *in.Disabled
		}

		err = crons.Update(r.Context(), cronjob)
		if err != nil {
			render.InternalError(w, err)
			return
		}
		render.JSON(w, cronjob, 200)
	}
}
