﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/clouddirectory/CloudDirectoryRequest.h>
#include <aws/clouddirectory/CloudDirectory_EXPORTS.h>
#include <aws/clouddirectory/model/ObjectReference.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace CloudDirectory {
namespace Model {

/**
 */
class DetachObjectRequest : public CloudDirectoryRequest {
 public:
  AWS_CLOUDDIRECTORY_API DetachObjectRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DetachObject"; }

  AWS_CLOUDDIRECTORY_API Aws::String SerializePayload() const override;

  AWS_CLOUDDIRECTORY_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) that is associated with the <a>Directory</a>
   * where objects reside. For more information, see <a>arns</a>.</p>
   */
  inline const Aws::String& GetDirectoryArn() const { return m_directoryArn; }
  inline bool DirectoryArnHasBeenSet() const { return m_directoryArnHasBeenSet; }
  template <typename DirectoryArnT = Aws::String>
  void SetDirectoryArn(DirectoryArnT&& value) {
    m_directoryArnHasBeenSet = true;
    m_directoryArn = std::forward<DirectoryArnT>(value);
  }
  template <typename DirectoryArnT = Aws::String>
  DetachObjectRequest& WithDirectoryArn(DirectoryArnT&& value) {
    SetDirectoryArn(std::forward<DirectoryArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The parent reference from which the object with the specified link name is
   * detached.</p>
   */
  inline const ObjectReference& GetParentReference() const { return m_parentReference; }
  inline bool ParentReferenceHasBeenSet() const { return m_parentReferenceHasBeenSet; }
  template <typename ParentReferenceT = ObjectReference>
  void SetParentReference(ParentReferenceT&& value) {
    m_parentReferenceHasBeenSet = true;
    m_parentReference = std::forward<ParentReferenceT>(value);
  }
  template <typename ParentReferenceT = ObjectReference>
  DetachObjectRequest& WithParentReference(ParentReferenceT&& value) {
    SetParentReference(std::forward<ParentReferenceT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The link name associated with the object that needs to be detached.</p>
   */
  inline const Aws::String& GetLinkName() const { return m_linkName; }
  inline bool LinkNameHasBeenSet() const { return m_linkNameHasBeenSet; }
  template <typename LinkNameT = Aws::String>
  void SetLinkName(LinkNameT&& value) {
    m_linkNameHasBeenSet = true;
    m_linkName = std::forward<LinkNameT>(value);
  }
  template <typename LinkNameT = Aws::String>
  DetachObjectRequest& WithLinkName(LinkNameT&& value) {
    SetLinkName(std::forward<LinkNameT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_directoryArn;

  ObjectReference m_parentReference;

  Aws::String m_linkName;
  bool m_directoryArnHasBeenSet = false;
  bool m_parentReferenceHasBeenSet = false;
  bool m_linkNameHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudDirectory
}  // namespace Aws
