﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/acm-pca/ACMPCARequest.h>
#include <aws/acm-pca/ACMPCA_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace ACMPCA {
namespace Model {

/**
 */
class ListTagsRequest : public ACMPCARequest {
 public:
  AWS_ACMPCA_API ListTagsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListTags"; }

  AWS_ACMPCA_API Aws::String SerializePayload() const override;

  AWS_ACMPCA_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>Use this parameter when paginating results to specify the maximum number of
   * items to return in the response. If additional items exist beyond the number you
   * specify, the <b>NextToken</b> element is sent in the response. Use this
   * <b>NextToken</b> value in a subsequent request to retrieve additional items.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListTagsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Use this parameter when paginating results in a subsequent request after you
   * receive a response with truncated results. Set it to the value of
   * <b>NextToken</b> from the response you just received.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListTagsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) that was returned when you called the <a
   * href="https://docs.aws.amazon.com/privateca/latest/APIReference/API_CreateCertificateAuthority.html">CreateCertificateAuthority</a>
   * action. This must be of the form: </p> <p>
   * <code>arn:aws:acm-pca:<i>region</i>:<i>account</i>:certificate-authority/<i>12345678-1234-1234-1234-123456789012</i>
   * </code> </p>
   */
  inline const Aws::String& GetCertificateAuthorityArn() const { return m_certificateAuthorityArn; }
  inline bool CertificateAuthorityArnHasBeenSet() const { return m_certificateAuthorityArnHasBeenSet; }
  template <typename CertificateAuthorityArnT = Aws::String>
  void SetCertificateAuthorityArn(CertificateAuthorityArnT&& value) {
    m_certificateAuthorityArnHasBeenSet = true;
    m_certificateAuthorityArn = std::forward<CertificateAuthorityArnT>(value);
  }
  template <typename CertificateAuthorityArnT = Aws::String>
  ListTagsRequest& WithCertificateAuthorityArn(CertificateAuthorityArnT&& value) {
    SetCertificateAuthorityArn(std::forward<CertificateAuthorityArnT>(value));
    return *this;
  }
  ///@}
 private:
  int m_maxResults{0};

  Aws::String m_nextToken;

  Aws::String m_certificateAuthorityArn;
  bool m_maxResultsHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
  bool m_certificateAuthorityArnHasBeenSet = false;
};

}  // namespace Model
}  // namespace ACMPCA
}  // namespace Aws
