﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/events/CloudWatchEventsRequest.h>
#include <aws/events/CloudWatchEvents_EXPORTS.h>

#include <utility>

namespace Aws {
namespace CloudWatchEvents {
namespace Model {

/**
 */
class EnableRuleRequest : public CloudWatchEventsRequest {
 public:
  AWS_CLOUDWATCHEVENTS_API EnableRuleRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "EnableRule"; }

  AWS_CLOUDWATCHEVENTS_API Aws::String SerializePayload() const override;

  AWS_CLOUDWATCHEVENTS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the rule.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  EnableRuleRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name or ARN of the event bus associated with the rule. If you omit this,
   * the default event bus is used.</p>
   */
  inline const Aws::String& GetEventBusName() const { return m_eventBusName; }
  inline bool EventBusNameHasBeenSet() const { return m_eventBusNameHasBeenSet; }
  template <typename EventBusNameT = Aws::String>
  void SetEventBusName(EventBusNameT&& value) {
    m_eventBusNameHasBeenSet = true;
    m_eventBusName = std::forward<EventBusNameT>(value);
  }
  template <typename EventBusNameT = Aws::String>
  EnableRuleRequest& WithEventBusName(EventBusNameT&& value) {
    SetEventBusName(std::forward<EventBusNameT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;

  Aws::String m_eventBusName;
  bool m_nameHasBeenSet = false;
  bool m_eventBusNameHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudWatchEvents
}  // namespace Aws
