//
// Syd: rock-solid application kernel
// src/hook.rs: Secure computing hooks
//
// Copyright (c) 2023, 2024, 2025 Ali Polatel <alip@chesswob.org>
// Based in part upon greenhook which is under public domain.
// MDWE code is based in part upon systemd which is LGPL-2.1-or-later.
// Personality code is based on pacwrap which is GPL-3.0-only.
//
// SPDX-License-Identifier: GPL-3.0

use std::{
    borrow::Cow,
    cmp::Ordering,
    env,
    ffi::{OsStr, OsString},
    fs::{create_dir_all, File, OpenOptions},
    hash::Hash,
    io::{self, IoSlice, IoSliceMut, Read, Seek, SeekFrom, Write},
    ops::Deref,
    os::{
        fd::{AsFd, AsRawFd, OwnedFd, RawFd},
        unix::{ffi::OsStrExt, fs::OpenOptionsExt},
    },
    str::FromStr,
    sync::{atomic::AtomicBool, Arc, Condvar, LazyLock, Mutex, RwLock},
};

use bitflags::bitflags;
use btoi::btoi;
use data_encoding::HEXLOWER;
use libseccomp::{
    scmp_cmp, ScmpAction, ScmpArch, ScmpArgCompare, ScmpCompareOp, ScmpFilterContext,
    ScmpNotifResp, ScmpNotifRespFlags, ScmpSyscall, ScmpVersion,
};
use libseccomp_sys::__NR_SCMP_ERROR;
use memchr::{
    arch::all::{is_equal, is_suffix},
    memmem,
};
use nix::{
    errno::Errno,
    fcntl::{openat2, OFlag, ResolveFlag},
    libc::pid_t,
    mount::{mount, umount2, MntFlags, MsFlags},
    sched::{unshare, CloneFlags},
    sys::{
        prctl::{
            get_pdeathsig, set_child_subreaper, set_dumpable, set_no_new_privs, set_pdeathsig,
        },
        ptrace,
        resource::{getrlimit, setrlimit, Resource},
        signal::{kill, killpg, SaFlags, Signal},
        socket::UnixAddr,
        stat::{mkdirat, mknodat, umask, Mode, SFlag},
        time::TimeSpec,
        uio::{process_vm_readv, process_vm_writev, RemoteIoVec},
        wait::{Id, WaitPidFlag},
    },
    unistd::{
        chdir, close, fchdir, getpgid, getpgrp, mkdtemp, pipe2, pivot_root, read, symlinkat, Gid,
        Pid, Uid,
    },
    NixPath,
};
use serde::{ser::SerializeMap, Serialize};

use crate::{
    caps,
    compat::{fstatfs64, fstatx, waitid, TimeSpec32, TimeSpec64, WaitStatus, STATX_INO},
    config::*,
    confine::{
        confine_mdwe, confine_scmp_write, extend_ioctl, is_coredump, nsflag_name, scmp_add_mknod,
        scmp_add_setid_rules, scmp_arch, scmp_arch_bits, scmp_arch_raw, seccomp_add_architectures,
        seccomp_native_has_socketcall, ExportMode, ScmpNotifReq, SydArch, Sydcall, EIDRM, IPC_ARCH,
        NAMESPACE_FLAGS, NAMESPACE_FLAGS_ALL, NAMESPACE_NAMES, SCMP_ARCH,
    },
    cookie::{
        OPENAT2_COOKIE_ARG4, OPENAT2_COOKIE_ARG5, PROCMAP_QUERY_COOKIE_ARG3,
        PROCMAP_QUERY_COOKIE_ARG4, PROCMAP_QUERY_COOKIE_ARG5,
    },
    debug,
    elf::{ElfError, ElfFileType, ElfType, ExecutableFile, LinkingType},
    err::{err2no, SydJoinHandle, SydResult},
    error,
    fs::{
        closeexcept, fd_status_flags, file_type, pidfd_getfd, pidfd_open, pidfd_send_signal,
        process_mrelease, readlinkat, retry_on_eintr, safe_canonicalize, safe_open, safe_open_how,
        safe_open_msym, seccomp_export_pfc, seccomp_notify_addfd, seccomp_notify_id_valid,
        seccomp_notify_set_flags, to_valid_fd, CanonicalPath, FileType, FsFlags, AT_BADFD,
        AT_EXECVE_CHECK, PIDFD_THREAD, SECCOMP_USER_NOTIF_FD_SYNC_WAKE_UP,
    },
    hash::{key_ring_validate, SydHashMap, SydHashSet},
    info,
    landlock::{CompatLevel, RulesetStatus},
    log::{log_init_main, log_set_panic_hook},
    log_enabled,
    path::{dotdot_with_nul, XPath, XPathBuf, PATH_MAX, PATH_MIN},
    pool,
    proc::{
        proc_auxv, proc_executables, proc_fs_nr_open, proc_maps, proc_pid_max, proc_rand_fd,
        proc_set_at_secure, proc_stack_pointer, proc_status, proc_task_limit, proc_task_nr_syd,
        proc_task_nr_sys, proc_unix_get_inodes, PROCMAP_QUERY,
    },
    ptrace::{
        ptrace_get_syscall_info, ptrace_set_return, ptrace_skip_syscall, ptrace_syscall_info,
    },
    rwrite, rwriteln,
    sandbox::{
        Action, BindMount, Capability, Flags, IntegrityError, LockState, NetlinkFamily, Sandbox,
        SandboxGuard,
    },
    sealbox::SealableSydHashMap,
    set_cpu_priority_idle, set_io_priority_idle,
    spec::{
        speculation_get, speculation_set, SpeculationControlStatus, SpeculationFeature,
        SpeculationStatus, PR_SPEC_FORCE_DISABLE,
    },
    syslog::LogLevel,
    warn,
    workers::{aes::AesLock, ipc::IpcWorker, UnixMap, WorkerCache},
};

const NONE: Option<&XPathBuf> = None::<&XPathBuf>;

const FD_MAX: u64 = i32::MAX as u64;

// x32 compatibility
// See https://sourceware.org/bugzilla/show_bug.cgi?id=16437
#[cfg(all(target_arch = "x86_64", target_pointer_width = "32"))]
#[expect(non_camel_case_types)]
type timespec_tv_nsec_t = i64;
#[cfg(not(all(target_arch = "x86_64", target_pointer_width = "32")))]
#[expect(non_camel_case_types)]
type timespec_tv_nsec_t = libc::c_long;

/*
 * Macros
 */
bitflags! {
    /// Flags for `SysArg`.
    #[derive(Clone, Copy, Debug, Default, Eq, PartialEq)]
    pub(crate) struct SysFlags: u8 {
        /// Whether if it's ok for the path to be empty.
        const EMPTY_PATH = 1 << 0;
    }
}

impl Serialize for SysFlags {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: serde::Serializer,
    {
        let mut flags: Vec<&str> = vec![];

        if self.is_empty() {
            return serializer.collect_seq(flags);
        }

        if self.contains(Self::EMPTY_PATH) {
            flags.push("empty-path");
        }

        flags.sort();
        serializer.collect_seq(flags)
    }
}

/// `SysArg` represents a system call path argument,
/// coupled with a directory file descriptor as necessary.
#[derive(Copy, Clone, Debug, Default)]
pub(crate) struct SysArg {
    /// DirFd index in syscall args, if applicable.
    pub(crate) dirfd: Option<usize>,
    /// Path index in syscall args, if applicable.
    pub(crate) path: Option<usize>,
    /// Options for the system call.
    pub(crate) flags: SysFlags,
    /// Options for path canonicalization.
    pub(crate) fsflags: FsFlags,
    /// Whether dot as final component must return the given `Errno`.
    pub(crate) dotlast: Option<Errno>,
}

impl Serialize for SysArg {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: serde::Serializer,
    {
        let mut map = serializer.serialize_map(Some(5))?;
        map.serialize_entry("dirfd", &self.dirfd)?;
        map.serialize_entry("path", &self.path)?;
        map.serialize_entry("flags", &self.flags)?;
        map.serialize_entry("fsflags", &self.fsflags)?;
        map.serialize_entry("dotlast", &self.dotlast.map(|e| e as i32))?;
        map.end()
    }
}

impl SysArg {
    pub(crate) fn open(flags: OFlag, atfunc: bool, rflags: ResolveFlag) -> Self {
        let (dirfd, path) = if atfunc {
            (Some(0), Some(1))
        } else {
            (None, Some(0))
        };

        // SAFETY:
        // We do not resolve symbolic links if O_CREAT|O_EXCL is
        // specified to support creating files through dangling symbolic
        // links, see the creat_thru_dangling test for more information.
        // We also set MISS_LAST in this case so we get to assert EEXIST.
        let is_create = flags.contains(OFlag::O_CREAT);
        let is_exclusive_create = is_create && flags.contains(OFlag::O_EXCL);

        let mut fsflags = FsFlags::empty();
        if is_exclusive_create {
            fsflags.insert(FsFlags::MISS_LAST);
        } else if !is_create {
            fsflags.insert(FsFlags::MUST_PATH);
        };

        if flags.contains(OFlag::O_NOFOLLOW) || is_exclusive_create {
            fsflags |= FsFlags::NO_FOLLOW_LAST;
        }

        if rflags.contains(ResolveFlag::RESOLVE_BENEATH) {
            fsflags |= FsFlags::RESOLVE_BENEATH;
        }

        if rflags.contains(ResolveFlag::RESOLVE_NO_SYMLINKS) {
            fsflags |= FsFlags::NO_RESOLVE_PATH;
        }

        if rflags.contains(ResolveFlag::RESOLVE_NO_MAGICLINKS) {
            fsflags |= FsFlags::NO_RESOLVE_PROC;
        }

        if rflags.contains(ResolveFlag::RESOLVE_NO_XDEV) {
            fsflags |= FsFlags::NO_RESOLVE_XDEV;
        }

        Self {
            dirfd,
            path,
            fsflags,
            ..Default::default()
        }
    }
}

// Represents path arguments (max=2).
pub(crate) type PathArg<'a> = Option<CanonicalPath<'a>>;

#[derive(Debug)]
pub(crate) struct PathArgs<'a>(pub(crate) PathArg<'a>, pub(crate) PathArg<'a>);

/// `UNotifyEventRequest` is the type of parameter that user's function
/// would get.
pub(crate) struct UNotifyEventRequest {
    pub(crate) scmpreq: ScmpNotifReq,
    pub(crate) syscall: Sydcall,
    notify_fd: RawFd,
    pub(crate) cache: Arc<WorkerCache<'static>>,
    sandbox: Arc<RwLock<Sandbox>>,
    pub(crate) crypt_map: Option<AesLock>,
    unix_map: UnixMap,
}

impl Serialize for UNotifyEventRequest {
    #[expect(clippy::cognitive_complexity)]
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: serde::Serializer,
    {
        let mut map = serializer.serialize_map(Some(8))?;

        map.serialize_entry("pid", &self.scmpreq.pid)?;
        map.serialize_entry("sys", &self.syscall)?;
        map.serialize_entry("args", &self.scmpreq.data.args)?;
        map.serialize_entry("arch", &SydArch(self.scmpreq.data.arch))?;

        #[expect(clippy::cast_possible_wrap)]
        let pid = Pid::from_raw(self.scmpreq.pid as libc::pid_t);

        if let Ok(status) = proc_status(pid) {
            map.serialize_entry("cmd", &status.command)?;
            map.serialize_entry("tgid", &status.pid)?;
            map.serialize_entry("sig_caught", &status.sig_caught)?;
            map.serialize_entry("sig_blocked", &status.sig_blocked)?;
            map.serialize_entry("sig_ignored", &status.sig_ignored)?;
            map.serialize_entry("sig_pending_thread", &status.sig_pending_thread)?;
            map.serialize_entry("sig_pending_process", &status.sig_pending_process)?;
            map.serialize_entry("umask", &status.umask)?;
        }

        #[expect(clippy::unnecessary_cast)]
        if let Ok(auxv) = proc_auxv(pid) {
            // Note: libc::AT_* constant are u32 on 32-bit...

            // Base and entry addresses
            if let Some(val) = auxv.get(&(libc::AT_BASE as u64)) {
                map.serialize_entry("at_base", val)?;
            }
            if let Some(val) = auxv.get(&(libc::AT_ENTRY as u64)) {
                map.serialize_entry("at_entry", val)?;
            }

            // Program headers
            if let Some(val) = auxv.get(&(libc::AT_PHDR as u64)) {
                map.serialize_entry("at_phdr", val)?;
            }
            if let Some(val) = auxv.get(&(libc::AT_PHENT as u64)) {
                map.serialize_entry("at_phent", val)?;
            }
            if let Some(val) = auxv.get(&(libc::AT_PHNUM as u64)) {
                map.serialize_entry("at_phnum", val)?;
            }

            // Read AT_RANDOM bytes which is 16 bytes of
            // random data placed by the kernel at the
            // specified address.
            if let Some(addr) = auxv.get(&(libc::AT_RANDOM as u64)) {
                let mut at_random = [0u8; 16];
                if *addr >= *MMAP_MIN_ADDR && self.read_mem(&mut at_random, *addr).is_ok() {
                    map.serialize_entry("at_random", &HEXLOWER.encode(&at_random))?;
                }
            }

            // AT_SECURE: we set this ourselves
            // unless trace/allow_unsafe_libc:1 is passed at startup,
            // however when we set it, the value will still incorrectly
            // show as false because this file is not updated after
            // process startup.
            if let Some(val) = auxv.get(&(libc::AT_SECURE as u64)) {
                let sandbox = self.sandbox.read().unwrap_or_else(|err| err.into_inner());
                let seclibc = !sandbox.flags.allow_unsafe_libc();
                drop(sandbox); // release the read-lock.

                if seclibc {
                    map.serialize_entry("at_secure", &true)?;
                } else {
                    map.serialize_entry("at_secure", &(*val != 0))?;
                }
            }
        }

        let sp = proc_stack_pointer(pid)
            .map(|sp| (sp & !0xF).saturating_sub(16))
            .ok();
        map.serialize_entry("sp", &sp)?;

        let ip = self.scmpreq.data.instr_pointer;
        map.serialize_entry("ip", &ip)?;

        let mut ip_mem = [0u8; 64];
        let mut sp_mem = [0u8; 64];
        let mut ip_read = false;
        let mut sp_read = false;

        if ip >= *MMAP_MIN_ADDR && self.read_mem(&mut ip_mem, ip).is_ok() {
            ip_read = true;
        }

        if let Some(sp) = sp {
            if sp >= *MMAP_MIN_ADDR && self.read_mem(&mut sp_mem, sp).is_ok() {
                sp_read = true;
            }
        }

        map.serialize_entry(
            "sp_mem",
            &if sp_read {
                Some(HEXLOWER.encode(&sp_mem))
            } else {
                None
            },
        )?;

        map.serialize_entry(
            "ip_mem",
            &if ip_read {
                Some(HEXLOWER.encode(&ip_mem))
            } else {
                None
            },
        )?;

        map.serialize_entry("maps", &proc_maps(pid).ok())?;

        map.end()
    }
}

impl UNotifyEventRequest {
    pub(crate) fn new(
        scmpreq: ScmpNotifReq,
        syscall: Sydcall,
        notify_fd: RawFd,
        cache: Arc<WorkerCache<'static>>,
        sandbox: Arc<RwLock<Sandbox>>,
        crypt_map: Option<AesLock>,
        unix_map: UnixMap,
    ) -> Self {
        UNotifyEventRequest {
            scmpreq,
            syscall,
            notify_fd,
            cache,
            sandbox,
            crypt_map,
            unix_map,
        }
    }

    /// Get a read lock to the sandbox.
    pub(crate) fn get_sandbox(&self) -> SandboxGuard<'_> {
        // Note, if another user of this mutex panicked while holding
        // the mutex, then this call will return an error once the mutex
        // is acquired. We ignore this case here and fall through
        // because Syd emulator threads are free to panic independent of
        // each other.
        SandboxGuard::Read(self.sandbox.read().unwrap_or_else(|err| err.into_inner()))
    }

    /// Get a write lock to the sandbox.
    pub(crate) fn get_mut_sandbox(&self) -> SandboxGuard<'_> {
        // Note, if another user of this mutex panicked while holding
        // the mutex, then this call will return an error once the mutex
        // is acquired. We ignore this case here and fall through
        // because Syd emulator threads are free to panic independent of
        // each other.
        SandboxGuard::Write(self.sandbox.write().unwrap_or_else(|err| err.into_inner()))
    }

    /// Add a bind address to the BindMap. This has been split from the sandbox policy
    /// as of version 3.33.1 because it has no bearing on access rights and is provided
    /// for convenience with getsockname(2).
    pub(crate) fn add_unix<Fd: AsFd>(
        &self,
        fd: Fd,
        pid: Pid,
        path: Option<&XPath>,
    ) -> Result<(), Errno> {
        // Convert path to unix address.
        let addr = if let Some(path) = path {
            Some(UnixAddr::new(path)?)
        } else {
            None
        };

        // Get socket inode.
        let inode = fstatx(fd, STATX_INO).map(|statx| statx.stx_ino)?;

        // Record unix address.
        let mut unix_map = self.unix_map.write().unwrap_or_else(|err| err.into_inner());
        unix_map.insert(inode, (pid, addr));
        let unix_len = unix_map.len();

        // SAFETY: Do _not_ hold a write lock during /proc read.
        drop(unix_map);

        // Cleanup unix map from unused inodes as necessary.
        if unix_len > 128 {
            let inodes = proc_unix_get_inodes()?;
            let mut unix_map = self.unix_map.write().unwrap_or_else(|err| err.into_inner());
            unix_map.retain(|inode, _| inodes.contains(inode));
        }

        Ok(())
    }

    pub(crate) fn get_unix(&self, inode: u64) -> Option<(Pid, Option<UnixAddr>)> {
        self.unix_map
            .read()
            .unwrap_or_else(|err| err.into_inner())
            .get(&inode)
            .copied()
    }

    /// Read the sa_flags member of `struct sigaction` from the given address.
    pub(crate) fn read_sa_flags(&self, addr: u64) -> Result<SaFlags, Errno> {
        let req = self.scmpreq;

        // Determine the target word size. (4 for 32-bit, 8 for 64-bit).
        let is32 = scmp_arch_bits(req.data.arch) == 32;
        let word_size = if is32 { 4usize } else { 8usize };

        // Offset of sa_flags within struct sigaction.
        let offset = word_size as u64; // 4 on 32-bit, 8 on 64-bit.

        // Compute absolute read address, checking for overflow.
        let read_addr = addr.checked_add(offset).ok_or(Errno::EFAULT)?;

        // Initialize vector on stack.
        //
        // Buffer up to 8 bytes; will only use first `word_size` bytes.
        let mut buf = [0u8; 8];

        // Read from process memory.
        //
        // Loop until we've read `word_size` bytes,
        // or encounter EOF (zero-read).
        let process = RemoteProcess::new(self.scmpreq.pid());
        let mut nread = 0;
        while nread < word_size {
            // Adjust current slice.
            //
            // Compute absolute read address plus the offset, checking for overflow.
            let slice = &mut buf[nread..word_size];
            let read_addr = read_addr.checked_add(nread as u64).ok_or(Errno::EFAULT)?;

            // Read remote memory.
            //
            // SAFETY: The request is going to be validated.
            let n = unsafe { process.read_mem(slice, read_addr) }?;

            // SAFETY: Assume error on zero-read.
            if n == 0 {
                return Err(Errno::EFAULT);
            }

            // Compute next offset, check for overflow.
            nread = nread.checked_add(n).ok_or(Errno::EFAULT)?;
        }

        // SAFETY: Check request validity after memory read.
        if !self.is_valid() {
            return Err(Errno::ESRCH);
        }

        // Interpret raw bytes in native endianness.
        #[expect(clippy::cast_possible_truncation)]
        #[expect(clippy::cast_possible_wrap)]
        #[expect(clippy::disallowed_methods)]
        let raw = if word_size == 8 {
            u64::from_ne_bytes(buf) as libc::c_int
        } else {
            // SAFETY: `word_size` must always be 4 here.
            u32::from_ne_bytes(buf[..4].try_into().unwrap()) as libc::c_int
        };

        Ok(SaFlags::from_bits_truncate(raw))
    }

    /// Read the `libc::open_how` struct from process memory
    /// at the given address and size.
    pub(crate) fn remote_ohow(&self, addr: u64, size: u64) -> Result<libc::open_how, Errno> {
        const OPEN_HOW_SIZE: usize = std::mem::size_of::<libc::open_how>();
        const OPEN_HOW_ALIGN: usize = std::mem::align_of::<libc::open_how>();

        // SAFETY: Validate size argument.
        let size = usize::try_from(size).or(Err(Errno::EINVAL))?;
        match size.cmp(&OPEN_HOW_SIZE) {
            Ordering::Equal => {}
            Ordering::Less => return Err(Errno::EINVAL),
            Ordering::Greater if size % OPEN_HOW_ALIGN == 0 => return Err(Errno::E2BIG),
            Ordering::Greater => return Err(Errno::EFAULT),
        };

        // SAFETY: Validate address argument.
        if addr < *MMAP_MIN_ADDR {
            return Err(Errno::EFAULT);
        }

        let mut buf = [0u8; OPEN_HOW_SIZE];
        self.read_mem(&mut buf, addr)?;

        // SAFETY: The following unsafe block assumes that:
        // 1. The memory layout of open_how in our Rust environment
        //    matches that of the target process.
        // 2. The request.process.read_mem call has populated buf with valid data
        //    of the appropriate size (ensured by the size check above).
        // 3. The buffer is appropriately aligned for reading an
        //    open_how struct. If the remote process's representation of
        //    open_how was correctly aligned, our local buffer should be
        //    too, since it's an array on the stack.
        Ok(unsafe { std::ptr::read_unaligned(buf.as_ptr() as *const _) })
    }

    /// Read the `libc::utimbuf` struct from process memory at the given address.
    /// Convert it to a `libc::timespec[2]` for easy interoperability.
    pub(crate) fn remote_utimbuf(&self, addr: u64) -> Result<(TimeSpec, TimeSpec), Errno> {
        if addr == 0 {
            // utimbuf pointer is NULL: Set to current time.
            return Ok((TimeSpec::UTIME_NOW, TimeSpec::UTIME_NOW));
        } else if addr < *MMAP_MIN_ADDR {
            // utimbuf pointer is invalid: return EFAULT.
            return Err(Errno::EFAULT);
        }

        let mut buf = [0u8; std::mem::size_of::<libc::utimbuf>()];
        self.read_mem(&mut buf, addr)?;

        // SAFETY: The following unsafe block assumes that:
        // 1. The memory layout of utimbuf in our Rust environment
        //    matches that of the target process.
        // 2. The request.process.read_mem call has populated buf with valid data
        //    of the appropriate size (ensured by the size check above).
        // 3. The buffer is appropriately aligned for reading a utimbuf
        //    struct. If the remote process's representation of utimbuf
        //    was correctly aligned, our local buffer should be too,
        //    since it's an array on the stack.
        let utimbuf: libc::utimbuf = unsafe { std::ptr::read_unaligned(buf.as_ptr() as *const _) };

        Ok((
            TimeSpec::new(utimbuf.actime, 0),
            TimeSpec::new(utimbuf.modtime, 0),
        ))
    }

    /// Read the `libc::timeval[2]` struct from process memory at the given address.
    /// Convert it to a `libc::timespec[2]` for easy interoperability.
    pub(crate) fn remote_timeval(&self, addr: u64) -> Result<(TimeSpec, TimeSpec), Errno> {
        if addr == 0 {
            // timeval pointer is NULL: Set to current time.
            return Ok((TimeSpec::UTIME_NOW, TimeSpec::UTIME_NOW));
        } else if addr < *MMAP_MIN_ADDR {
            // timeval pointer is invalid: return EFAULT.
            return Err(Errno::EFAULT);
        }

        let mut buf = [0u8; std::mem::size_of::<libc::timeval>() * 2];
        self.read_mem(&mut buf, addr)?;

        // SAFETY: The following unsafe block assumes that:
        // 1. The memory layout of timeval in our Rust environment
        //    matches that of the target process.
        // 2. The request.process.read_mem call has populated buf with valid data
        //    of the appropriate size (ensured by the size check above).
        // 3. The buffer is appropriately aligned for reading a timeval
        //    struct. If the remote process's representation of timeval
        //    was correctly aligned, our local buffer should be too,
        //    since it's an array on the stack.
        #[expect(clippy::cast_ptr_alignment)]
        let timevals = unsafe {
            // Create a raw pointer to the buffer.
            let ptr = buf.as_ptr() as *const libc::timeval;

            // Read the timeval values from the buffer.
            [
                std::ptr::read_unaligned(ptr),
                std::ptr::read_unaligned(ptr.add(1)),
            ]
        };

        Ok((
            TimeSpec::new(
                timevals[0].tv_sec,
                (timevals[0].tv_usec as timespec_tv_nsec_t).saturating_mul(1_000), /* ms->ns */
            ),
            TimeSpec::new(
                timevals[1].tv_sec,
                (timevals[1].tv_usec as timespec_tv_nsec_t).saturating_mul(1_000), /* ms->ns */
            ),
        ))
    }

    /// Read the `TimeSpec32[2]` struct from process memory at the given address.
    pub(crate) fn remote_timespec32(&self, addr: u64) -> Result<(TimeSpec, TimeSpec), Errno> {
        if addr == 0 {
            // timespec pointer is NULL: Set to current time.
            return Ok((TimeSpec::UTIME_NOW, TimeSpec::UTIME_NOW));
        } else if addr < *MMAP_MIN_ADDR {
            // timespec pointer is invalid: return EFAULT.
            return Err(Errno::EFAULT);
        }

        let mut buf = [0u8; std::mem::size_of::<TimeSpec32>() * 2];
        self.read_mem(&mut buf, addr)?;

        // SAFETY: The following unsafe block assumes that:
        // 1. The memory layout of timespec in our Rust environment
        //    matches that of the target process.
        // 2. The request.process.read_mem call has populated buf with valid data
        //    of the appropriate size (ensured by the size check above).
        // 3. The buffer is appropriately aligned for reading a timespec
        //    struct. If the remote process's representation of timespec
        //    was correctly aligned, our local buffer should be too,
        //    since it's an array on the stack.
        #[expect(clippy::cast_ptr_alignment)]
        let timespecs = unsafe {
            // Create a raw pointer to the buffer.
            let ptr = buf.as_ptr() as *const TimeSpec32;

            // Read the timespec values from the buffer.
            [
                std::ptr::read_unaligned(ptr),
                std::ptr::read_unaligned(ptr.add(1)),
            ]
        };

        Ok((
            TimeSpec::new(timespecs[0].tv_sec.into(), timespecs[0].tv_nsec.into()),
            TimeSpec::new(timespecs[1].tv_sec.into(), timespecs[1].tv_nsec.into()),
        ))
    }

    /// Read the `TimeSpec64[2]` struct from process memory at the given address.
    // `as _` casts are used to write portable code for x32 and i386.
    #[expect(clippy::as_underscore)]
    pub(crate) fn remote_timespec64(&self, addr: u64) -> Result<(TimeSpec, TimeSpec), Errno> {
        if addr == 0 {
            // timespec pointer is NULL: Set to current time.
            return Ok((TimeSpec::UTIME_NOW, TimeSpec::UTIME_NOW));
        } else if addr < *MMAP_MIN_ADDR {
            // timespec pointer is invalid: return EFAULT.
            return Err(Errno::EFAULT);
        }

        let mut buf = [0u8; std::mem::size_of::<TimeSpec64>() * 2];
        self.read_mem(&mut buf, addr)?;

        // SAFETY: The following unsafe block assumes that:
        // 1. The memory layout of timespec in our Rust environment
        //    matches that of the target process.
        // 2. The request.process.read_mem call has populated buf with valid data
        //    of the appropriate size (ensured by the size check above).
        // 3. The buffer is appropriately aligned for reading a timespec
        //    struct. If the remote process's representation of timespec
        //    was correctly aligned, our local buffer should be too,
        //    since it's an array on the stack.
        #[expect(clippy::cast_ptr_alignment)]
        let timespecs = unsafe {
            // Create a raw pointer to the buffer.
            let ptr = buf.as_ptr() as *const TimeSpec64;

            // Read the timespec values from the buffer.
            [
                std::ptr::read_unaligned(ptr),
                std::ptr::read_unaligned(ptr.add(1)),
            ]
        };

        Ok((
            TimeSpec::new(timespecs[0].tv_sec as _, timespecs[0].tv_nsec as _),
            TimeSpec::new(timespecs[1].tv_sec as _, timespecs[1].tv_nsec as _),
        ))
    }

    /// Read path from the given system call argument with the given request.
    /// Check for magic prefix is magic is true.
    pub(crate) fn read_path<'b>(
        &self,
        sandbox: &SandboxGuard,
        arg: SysArg,
        magic: bool,
    ) -> Result<(CanonicalPath<'b>, bool), Errno> {
        let process = RemoteProcess::new(self.scmpreq.pid());

        // SAFETY: The request is validated by read_path.
        let (path, magic, doterr, empty_path) =
            process.read_path(sandbox, self.scmpreq.data.args, arg, magic, Some(self))?;

        // Determine FD-only system calls.
        // We return EACCES rather than ENOENT for these.
        let is_fd = empty_path && arg.flags.contains(SysFlags::EMPTY_PATH);

        // (a) Delayed dotlast Errno::ENOENT handler, see above for the rationale.
        // (b) SAFETY: the Missing check is skipped by fs::canonicalize on purpose,
        // so that EEXIST return value cannot be abused to locate hidden paths.
        if !doterr {
            Ok((path, magic))
        } else if path
            .typ
            .as_ref()
            .map(|typ| !typ.is_symlink())
            .unwrap_or(false)
        {
            // Path exists and is not a symbolic link.
            // Return EACCES if this is FD-only call.
            // Return ENOENT if either one of path or parent is hidden.
            // Return EEXIST if not.
            if is_fd {
                Err(Errno::EACCES)
            } else if sandbox.is_hidden(path.abs()) || sandbox.is_hidden(path.abs().parent()) {
                Err(Errno::ENOENT)
            } else {
                Err(Errno::EEXIST)
            }
        } else if is_fd {
            Err(Errno::EACCES)
        } else {
            Err(Errno::ENOENT)
        }
    }

    /// Read data from remote process's memory with `process_vm_readv()`.
    pub(crate) fn read_mem(
        &self,
        local_buffer: &mut [u8],
        remote_addr: u64,
    ) -> Result<usize, Errno> {
        let process = RemoteProcess::new(self.scmpreq.pid());

        // SAFETY: The request is validated.
        match unsafe { process.read_mem(local_buffer, remote_addr) } {
            Ok(n) => {
                if self.is_valid() {
                    Ok(n)
                } else {
                    Err(Errno::ESRCH)
                }
            }
            Err(errno) => Err(errno),
        }
    }

    /// Write data to remote process's memory with `process_vm_writev()`.
    #[inline(always)]
    pub(crate) fn write_mem(&self, local_buffer: &[u8], remote_addr: u64) -> Result<usize, Errno> {
        let process = RemoteProcess::new(self.scmpreq.pid());

        // SAFETY: The request is validated.
        match unsafe { process.write_mem(local_buffer, remote_addr) } {
            Ok(n) => {
                if self.is_valid() {
                    Ok(n)
                } else {
                    Err(Errno::ESRCH)
                }
            }
            Err(errno) => Err(errno),
        }
    }

    /// Get file descriptor from remote process with pidfd_getfd(2).
    ///
    /// This function requires Linux 5.6+.
    pub(crate) fn get_fd(&self, remote_fd: RawFd) -> Result<OwnedFd, Errno> {
        // SAFETY: Check if the RawFd is valid.
        if remote_fd < 0 {
            return Err(Errno::EBADF);
        }

        // Open a PidFd or use an already opened one.
        let pid_fd = self.pidfd_open()?;

        // Transfer fd using pidfd_getfd(2)
        pidfd_getfd(pid_fd, remote_fd)
    }

    /// Send a signal to the PIDFd of the process.
    pub(crate) fn pidfd_kill(&self, sig: i32) -> Result<(), Errno> {
        // Open a PidFd by validating it.
        let pid_fd = self.pidfd_open()?;
        pidfd_send_signal(&pid_fd, sig)?;

        // SAFETY: Release memory immediately using process_mrelease(2) if we
        // have sent a SIGKILL to the sandbox process. Above all, this is useful
        // for memory sandboxing.
        if sig == libc::SIGKILL {
            let _ = process_mrelease(&pid_fd);
        }

        Ok(())
    }

    /// Open a PidFd and validate it against the request.
    pub(crate) fn pidfd_open(&self) -> Result<OwnedFd, Errno> {
        // Open the PIDFd.
        let pid_fd = pidfd_open(self.scmpreq.pid(), PIDFD_THREAD)?;

        // SAFETY:
        // 1. Validate the PIDFd by validating the request ID if submitted.
        // 2. EAGAIN|EINTR is handled.
        // 3. ENOENT means child died mid-way.
        if seccomp_notify_id_valid(self.notify_fd, self.scmpreq.id).is_err() {
            return Err(Errno::ESRCH);
        }

        Ok(pid_fd)
    }

    /// Send the request pid a signal based on the given action.
    ///
    /// Non-signaling actions default to SIGKILL.
    pub(crate) fn kill(&self, action: Action) -> Result<(), Errno> {
        self.pidfd_kill(
            action
                .signal()
                .map(|sig| sig as libc::c_int)
                .unwrap_or(libc::SIGKILL),
        )
    }

    /// Let the kernel continue the syscall.
    ///
    /// # Safety
    /// CAUTION! This method is unsafe because it may suffer TOCTOU attack.
    /// Please read `seccomp_unotify(2)` "NOTES/Design goals; use of `SECCOMP_USER_NOTIF_FLAG_CONTINUE`"
    /// before using this method.
    pub(crate) unsafe fn continue_syscall(&self) -> ScmpNotifResp {
        ScmpNotifResp::new(self.scmpreq.id, 0, 0, ScmpNotifRespFlags::CONTINUE.bits())
    }

    /// Returns error to supervised process.
    pub(crate) fn fail_syscall(&self, err: Errno) -> ScmpNotifResp {
        assert!(err != Errno::UnknownErrno);
        #[expect(clippy::arithmetic_side_effects)]
        ScmpNotifResp::new(self.scmpreq.id, 0, -(err as i32), 0)
    }

    /// Returns value to supervised process.
    pub(crate) fn return_syscall(&self, val: i64) -> ScmpNotifResp {
        ScmpNotifResp::new(self.scmpreq.id, val, 0, 0)
    }

    /// Check if this event is still valid.
    /// In some cases this is necessary, please check `seccomp_unotify(2)` for more information.
    #[inline(always)]
    pub(crate) fn is_valid(&self) -> bool {
        // EAGAIN|EINTR is handled.
        // ENOENT means child died mid-way.
        seccomp_notify_id_valid(self.notify_fd, self.scmpreq.id).is_ok()
    }

    /// Add a file descriptor to the supervised process,
    /// and reply to the seccomp request at the same time.
    /// This could help avoid TOCTOU attack in some cases.
    pub(crate) fn send_fd<Fd: AsFd>(
        &self,
        src_fd: Fd,
        close_on_exec: bool,
        randomize_fds: bool,
    ) -> Result<ScmpNotifResp, Errno> {
        #[expect(clippy::cast_possible_truncation)]
        let (newfd, flags) = if randomize_fds {
            (
                proc_rand_fd(self.scmpreq.pid())?,
                (libc::SECCOMP_ADDFD_FLAG_SEND as u32 | libc::SECCOMP_ADDFD_FLAG_SETFD as u32),
            )
        } else {
            (0, libc::SECCOMP_ADDFD_FLAG_SEND as u32)
        };

        let newfd_flags = if close_on_exec {
            libc::O_CLOEXEC as u32
        } else {
            0
        };

        #[expect(clippy::cast_sign_loss)]
        let addfd: libc::seccomp_notif_addfd = libc::seccomp_notif_addfd {
            id: self.scmpreq.id,
            srcfd: src_fd.as_fd().as_raw_fd() as u32,
            newfd: newfd as u32,
            flags,
            newfd_flags,
        };

        // EAGAIN|EINTR is retried.
        // Other errors are fatal,
        // including ENOENT which means child died mid-way.
        seccomp_notify_addfd(self.notify_fd, std::ptr::addr_of!(addfd))?;

        // We do not need to send a response,
        // send a dummy response to the caller
        // can skip it gracefully.
        Ok(ScmpNotifResp::new(0, 0, EIDRM, 0))
    }
}

/// By using `RemoteProcess`, you can get information about the
/// supervised process.
#[derive(Clone, Debug)]
pub struct RemoteProcess {
    /// The process ID.
    pub pid: Pid,
}

impl PartialEq for RemoteProcess {
    fn eq(&self, other: &Self) -> bool {
        self.pid == other.pid
    }
}

impl Eq for RemoteProcess {}

impl Ord for RemoteProcess {
    fn cmp(&self, other: &Self) -> Ordering {
        self.pid.cmp(&other.pid)
    }
}

impl PartialOrd for RemoteProcess {
    fn partial_cmp(&self, other: &Self) -> Option<Ordering> {
        Some(self.cmp(other))
    }
}

impl RemoteProcess {
    /// Create a new `RemoteProcess` for the given TID.
    pub(crate) fn new(pid: Pid) -> Self {
        Self { pid }
    }

    /// Read path from the given system call argument with the given request.
    /// Check for magic prefix is magic is true.
    ///
    /// If `request` is `Some()` request is validated after
    /// actions that require validation such as proc reads
    /// and fd transfers. Otherwise, the caller must validate
    /// to verify the path read from sandbox process memory
    /// is what's expected.
    #[expect(clippy::cognitive_complexity)]
    #[expect(clippy::type_complexity)]
    pub(crate) fn read_path<'b>(
        &self,
        sandbox: &SandboxGuard,
        args: [u64; 6],
        arg: SysArg,
        mut magic: bool,
        request: Option<&UNotifyEventRequest>,
    ) -> Result<(CanonicalPath<'b>, bool, bool, bool), Errno> {
        let orig = match arg.path {
            Some(idx) => {
                // SAFETY: Check pointer against mmap_min_addr.
                if args[idx] < *MMAP_MIN_ADDR {
                    return Err(Errno::EFAULT);
                }

                Some(self.remote_path(args[idx], request)?)
            }
            None => None,
        };
        let mut doterr = false;

        // magic is both an in and out variable.
        // in=t: check for magic path.
        // out=t: path is magic path.
        let check_magic = magic;
        magic = false;

        let mut empty_path = false;
        let canonical_path = if let Some(path) = orig {
            empty_path = path.is_empty();
            if empty_path && !arg.flags.contains(SysFlags::EMPTY_PATH) {
                return Err(Errno::ENOENT);
            }

            if let Some(errno) = arg.dotlast {
                if path.ends_with_dot() {
                    if errno == Errno::ENOENT {
                        // This will be handled later, as we may
                        // need to return EEXIST instead of ENOENT
                        // if the path exists.
                        doterr = true;
                    } else {
                        return Err(errno);
                    }
                }
            }

            if check_magic && path.is_magic() {
                magic = true;
                CanonicalPath::new_magic(path)
            } else if empty_path || path.is_dot() {
                let dirfd = if let Some(idx) = arg.dirfd {
                    // Validate FD argument.
                    //
                    // Note about EMPTY_PATH:
                    // 1. execveat(fd, "", NULL, NULL, AT_EMPTY_PATH)
                    // 2. openat(fd, "", O_TMPFILE|O_RDWR, 0)
                    // In the first case AT_FDCWD is invalid,
                    // but in the second case AT_FDCWD is valid.
                    to_valid_fd(args[idx])?
                } else {
                    libc::AT_FDCWD
                };
                let is_dot = !empty_path;

                // SAFETY: The ends_with_dot check above
                // ensures we return ENOTDIR when e.g. path is
                // a dot and the file descriptor argument is a
                // regular file. This happens because in this
                // case, joining the directory with an empty
                // path on the next branch essentially adds a
                // trailing slash to the path, making the
                // system call emulator fail with ENOTDIR if
                // the argument is not a directory. This way,
                // we avoid stat'ing the path here to
                // determine whether it's a directory or not.
                if let Some(request) = request {
                    if dirfd == libc::AT_FDCWD {
                        let path = CanonicalPath::new_fd(libc::AT_FDCWD.into(), self.pid)?;

                        // Validate request after procfs(5) read.
                        if !request.is_valid() {
                            return Err(Errno::ESRCH);
                        }

                        path
                    } else {
                        // SAFETY: Get the file descriptor before access check
                        // as it may change after which is a TOCTOU vector.
                        let fd = request.get_fd(dirfd)?;

                        let path = CanonicalPath::new_fd(fd.into(), self.pid)?;

                        if is_dot && path.typ != Some(FileType::Dir) {
                            // FD-only call, no need to delay ENOTDIR.
                            return Err(Errno::ENOTDIR);
                        }

                        path
                    }
                } else if dirfd == libc::AT_FDCWD {
                    CanonicalPath::new_fd(libc::AT_FDCWD.into(), self.pid)?
                } else {
                    // SAFETY: Get the file descriptor before access check
                    // as it may change after which is a TOCTOU vector.
                    let pid_fd = pidfd_open(self.pid, PIDFD_THREAD)?;
                    let fd = pidfd_getfd(pid_fd, dirfd)?;

                    let path = CanonicalPath::new_fd(fd.into(), self.pid)?;

                    if is_dot && path.typ != Some(FileType::Dir) {
                        // FD-only call, no need to delay ENOTDIR.
                        return Err(Errno::ENOTDIR);
                    }

                    path
                }
            } else {
                let fd = if let Some(idx) = arg.dirfd {
                    // Using a bad directory is okay for absolute paths.
                    if path.is_absolute() {
                        None
                    } else {
                        Some(to_valid_fd(args[idx])?)
                    }
                } else {
                    None
                };

                let path =
                    safe_canonicalize(self.pid, fd, &path, arg.fsflags, Some(sandbox.deref()))?;

                // Validate request after canonicalize which may
                // have involved a procfs(5) read.
                if request.map(|req| !req.is_valid()).unwrap_or(false) {
                    return Err(Errno::ESRCH);
                }

                path
            }
        } else {
            // SAFETY: SysArg.path is None asserting dirfd is Some.
            #[expect(clippy::disallowed_methods)]
            let idx = arg.dirfd.unwrap();

            // Validate file descriptor.
            //
            // AT_FDCWD is an invalid file descriptor with NULL path.
            let remote_fd = RawFd::try_from(args[idx]).or(Err(Errno::EBADF))?;
            if remote_fd < 0 {
                // Negative file descriptors are invalid with NULL path.
                return Err(Errno::EBADF);
            }

            if let Some(request) = request {
                // SAFETY: Get the file descriptor before access check
                // as it may change after which is a TOCTOU vector.
                let fd = request.get_fd(remote_fd)?;

                // Validate WANT_READ against O_PATH.
                if arg.fsflags.want_read() && fd_status_flags(&fd)?.contains(OFlag::O_PATH) {
                    return Err(Errno::EBADF);
                }

                CanonicalPath::new_fd(fd.into(), self.pid)?
            } else {
                // SAFETY: Get the file descriptor before access check
                // as it may change after which is a TOCTOU vector.
                let pid_fd = pidfd_open(self.pid, PIDFD_THREAD)?;
                let fd = pidfd_getfd(pid_fd, remote_fd)?;

                // Validate WANT_READ against O_PATH.
                if arg.fsflags.want_read() && fd_status_flags(&fd)?.contains(OFlag::O_PATH) {
                    return Err(Errno::EBADF);
                }

                CanonicalPath::new_fd(fd.into(), self.pid)?
            }
        };

        if !magic && arg.path.is_some() {
            // SAFETY: Deny access to critical and/or suspicious paths.
            canonical_path.abs().check(
                self.pid,
                canonical_path.typ.as_ref(),
                None,
                !sandbox.flags.allow_unsafe_filename(),
                !sandbox.flags.allow_unsafe_mkbdev(),
            )?;
        }

        Ok((canonical_path, magic, doterr, empty_path))
    }

    /// Read data from remote process's memory with `process_vm_readv()`.
    ///
    /// # Safety
    ///
    /// This function is unsafe because the request is not validated.
    pub(crate) unsafe fn read_mem(
        &self,
        local_buffer: &mut [u8],
        remote_addr: u64,
    ) -> Result<usize, Errno> {
        static FORCE_PROC: LazyLock<bool> =
            LazyLock::new(|| std::env::var_os(ENV_NO_CROSS_MEMORY_ATTACH).is_some());
        // SAFETY: Falling back to proc_pid_mem(5) is insecure,
        // and is no longer the default as of 3.32.6.
        static PROC_MEM_F: LazyLock<bool> =
            LazyLock::new(|| std::env::var_os(ENV_PROC_PID_MEM_FALLBACK).is_some());
        if *FORCE_PROC || (*PROC_MEM_F && !*HAVE_CROSS_MEMORY_ATTACH) {
            return self.read_mem_proc(local_buffer, remote_addr);
        }

        // SAFETY: Check pointer against mmap_min_addr.
        if remote_addr < *MMAP_MIN_ADDR {
            return Err(Errno::EFAULT);
        }

        let len = local_buffer.len();
        process_vm_readv(
            self.pid,
            &mut [IoSliceMut::new(local_buffer)],
            &[RemoteIoVec {
                len,
                base: usize::try_from(remote_addr).or(Err(Errno::EFAULT))?,
            }],
        )
    }

    /// Fallback method to read data from `/proc/$pid/mem` when `process_vm_readv()` is unavailable.
    ///
    /// # Safety
    ///
    /// This function is unsafe because the request is not validated.
    pub(crate) unsafe fn read_mem_proc(
        &self,
        local_buffer: &mut [u8],
        remote_addr: u64,
    ) -> Result<usize, Errno> {
        // SAFETY: Check pointer against mmap_min_addr.
        if remote_addr < *MMAP_MIN_ADDR {
            return Err(Errno::EFAULT);
        }

        let mut path = XPathBuf::from_pid(self.pid);
        path.push(b"mem");

        let mut file = safe_open_msym(PROC_FILE(), &path, OFlag::O_RDONLY, ResolveFlag::empty())
            .map(File::from)
            .or(Err(Errno::EACCES))?;
        file.seek(SeekFrom::Start(remote_addr))
            .or(Err(Errno::EACCES))?;

        let mut nread = 0;
        #[expect(clippy::arithmetic_side_effects)]
        while nread < local_buffer.len() {
            match file.read(&mut local_buffer[nread..]) {
                Ok(0) => return Err(Errno::EACCES),
                Ok(n) => nread += n,
                Err(ref e) if e.kind() == io::ErrorKind::Interrupted => {}
                Err(_) => return Err(Errno::EACCES),
            }
        }

        Ok(nread)
    }

    /// Write data to remote process's memory with `process_vm_writev()`.
    ///
    /// # Safety
    ///
    /// This function is unsafe because the request is not validated.
    pub(crate) unsafe fn write_mem(
        &self,
        local_buffer: &[u8],
        remote_addr: u64,
    ) -> Result<usize, Errno> {
        static FORCE_PROC: LazyLock<bool> =
            LazyLock::new(|| std::env::var_os(ENV_NO_CROSS_MEMORY_ATTACH).is_some());
        // SAFETY: Falling back to proc_pid_mem(5) is insecure,
        // and is no longer the default as of 3.32.6.
        static PROC_MEM_F: LazyLock<bool> =
            LazyLock::new(|| std::env::var_os(ENV_PROC_PID_MEM_FALLBACK).is_some());
        if *FORCE_PROC || (*PROC_MEM_F && !*HAVE_CROSS_MEMORY_ATTACH) {
            return self.write_mem_proc(local_buffer, remote_addr);
        }

        // SAFETY: Check pointer against mmap_min_addr.
        if remote_addr < *MMAP_MIN_ADDR {
            return Err(Errno::EFAULT);
        }

        let len = local_buffer.len();
        process_vm_writev(
            self.pid,
            &[IoSlice::new(local_buffer)],
            &[RemoteIoVec {
                len,
                base: usize::try_from(remote_addr).or(Err(Errno::EFAULT))?,
            }],
        )
    }

    /// Fallback method to write data to `/proc/$pid/mem` when `process_vm_writev()` is unavailable.
    ///
    /// # Safety
    ///
    /// This function is unsafe because the request is not validated.
    pub(crate) unsafe fn write_mem_proc(
        &self,
        local_buffer: &[u8],
        remote_addr: u64,
    ) -> Result<usize, Errno> {
        // SAFETY: Check pointer against mmap_min_addr.
        if remote_addr < *MMAP_MIN_ADDR {
            return Err(Errno::EFAULT);
        }

        let mut path = XPathBuf::from_pid(self.pid);
        path.push(b"mem");

        let mut file = safe_open_msym(PROC_FILE(), &path, OFlag::O_WRONLY, ResolveFlag::empty())
            .map(File::from)
            .or(Err(Errno::EACCES))?;
        file.seek(SeekFrom::Start(remote_addr))
            .or(Err(Errno::EACCES))?;

        let mut nwritten = 0;
        #[expect(clippy::arithmetic_side_effects)]
        while nwritten < local_buffer.len() {
            match file.write(&local_buffer[nwritten..]) {
                Ok(0) => return Err(Errno::EACCES),
                Ok(n) => nwritten += n,
                Err(ref e) if e.kind() == io::ErrorKind::Interrupted => {}
                Err(_) => return Err(Errno::EACCES),
            }
        }

        Ok(nwritten)
    }

    /// Read the path from memory of the process with the given `Pid` with the given address.
    ///
    /// If `request` is `Some()` request is validated after
    /// actions that require validation such as proc reads
    /// and fd transfers. Otherwise, the caller must validate
    /// to verify the path read from sandbox process memory
    /// is what's expected.
    pub(crate) fn remote_path(
        &self,
        addr: u64,
        request: Option<&UNotifyEventRequest>,
    ) -> Result<XPathBuf, Errno> {
        // Initialize path on the heap,
        // bail out if memory allocation fails.
        let mut buf = Vec::new();
        buf.try_reserve(PATH_MIN).or(Err(Errno::ENOMEM))?;

        // Read from process memory.
        // We read PATH_MIN bytes at a time,
        // because most paths are short.
        let mut off = 0;
        #[expect(clippy::arithmetic_side_effects)]
        while off < PATH_MAX {
            // Ensure we have enough space for the next read.
            let len = PATH_MIN.min(PATH_MAX - off);

            if buf.len() < off + len {
                // Extend the buffer to the required size,
                // bail out if memory allocation fails.
                buf.try_reserve(len).or(Err(Errno::ENOMEM))?;
                buf.resize(off + len, 0);
            }

            let ptr = &mut buf[off..off + len];

            // Read remote memory.
            // SAFETY:
            // 1. Assume error on zero-read.
            // 2. Validate the request after memory read.
            let len = unsafe { self.read_mem(ptr, addr + off as u64) }?;
            if len == 0 {
                return Err(Errno::EFAULT);
            } else if request.map(|req| !req.is_valid()).unwrap_or(false) {
                return Err(Errno::ESRCH);
            }

            // Check for NUL-byte.
            if let Some(nul) = memchr::memchr(0, &ptr[..len]) {
                // Adjust to actual size up to NUL-byte.
                off += nul;
                buf.truncate(off);
                buf.shrink_to_fit();
                return Ok(buf.into());
            }

            off += len;
        }

        Err(Errno::ENAMETOOLONG)
    }
}

pub(crate) type Handler = Arc<Box<dyn Fn(UNotifyEventRequest) -> ScmpNotifResp + Send + Sync>>;
pub(crate) type HandlerMap = SealableSydHashMap<Sydcall, Handler>;
type AllowSet = SydHashSet<ScmpSyscall>;

/// Supervisor of a Syd sandbox.
pub struct Supervisor {
    handlers: Arc<HandlerMap>,
    sysallow: AllowSet,
    sandbox: Arc<RwLock<Sandbox>>,
    crypt_map: Option<AesLock>,
    pty_child: Option<OwnedFd>,
    ipc_worker: Option<IpcWorker>,
}

impl Supervisor {
    /// Create a new `Supervisor` object. You can specify the number of threads in the thread pool.
    /// This function will also check your kernel version and show warning or return error if necessary.
    pub(crate) fn new(
        sandbox: Sandbox,
        pty_child: Option<OwnedFd>,
        mut ipc_worker: Option<IpcWorker>,
    ) -> SydResult<Self> {
        let mut handlers = SealableSydHashMap::default();
        let mut sysallow = SydHashSet::default();
        Self::init(&sandbox, &mut handlers, &mut sysallow)?;

        // SAFETY: Seal handlers map to protect function pointers.
        handlers = handlers.seal(Some(c"syd"))?;

        let crypt_map = if sandbox.enabled(Capability::CAP_CRYPT) {
            Some(Arc::new((
                Mutex::new(SydHashMap::default()),
                Condvar::new(),
            )))
        } else {
            None
        };

        let sandbox = Arc::new(RwLock::new(sandbox));
        if let Some(ref mut ipc_worker) = ipc_worker {
            ipc_worker.set_sandbox(Arc::clone(&sandbox));
        }

        let supervisor = Supervisor {
            pty_child,
            ipc_worker,
            sandbox,
            sysallow,
            handlers: Arc::new(handlers),
            crypt_map: crypt_map.as_ref().map(Arc::clone),
        };

        Ok(supervisor)
    }

    /// Initializes the supervisor by adding the system call handlers.
    #[expect(clippy::cognitive_complexity)]
    fn init(
        sandbox: &Sandbox,
        handlers: &mut HandlerMap,
        sysallow: &mut AllowSet,
    ) -> SydResult<()> {
        // For performance reasons, we apply Memory sandboxing
        // at startup only. Disabling it after startup is still
        // possible though when the respective syscalls will not
        // do any memory related checks.
        let has_mem = sandbox.enabled(Capability::CAP_MEM);
        let has_safe_setuid = sandbox.has_uid_transit();
        let has_safe_setgid = sandbox.has_gid_transit();
        let restrict_chroot = !sandbox.flags.allow_unsafe_chroot();
        let restrict_memfd = !sandbox.flags.allow_unsafe_memfd();
        let restrict_mqueue = !sandbox.flags.allow_unsafe_mqueue();
        let restrict_ptrace = !sandbox.flags.allow_unsafe_ptrace();
        let restrict_shm = !sandbox.flags.allow_unsafe_shm();
        let restrict_spec_exec = !sandbox.flags.allow_unsafe_spec_exec();
        let restrict_sysinfo = !sandbox.flags.allow_unsafe_sysinfo();
        let skip_append = sandbox.skip_append();
        let flags = *sandbox.flags;

        let mut allow_calls = Vec::new();

        // PR_SET_NAME logging.
        Self::insert_handler(handlers, "prctl", crate::kernel::prctl::sys_prctl);

        // F_SETFL O_APPEND unset prevention for appendonly files.
        //
        // Note, pwritev2(2) flag RWF_NOAPPEND is denied as part
        // of a separate filter because of its arch-specific nature.
        if !skip_append {
            Self::insert_handler(handlers, "fcntl", crate::kernel::fcntl::sys_fcntl);
            Self::insert_handler(handlers, "fcntl64", crate::kernel::fcntl::sys_fcntl);
        } else {
            allow_calls.extend(["fcntl", "fcntl64", "pwritev2"]);
        }

        if !restrict_ptrace {
            // Exec sandboxing, only used with trace/allow_unsafe_ptrace:1
            //
            // Because with seccomp there's no TOCTTOU-free way to
            // implement these system calls.
            //
            // See: https://bugzilla.kernel.org/show_bug.cgi?id=218501
            Self::insert_handler(handlers, "execve", crate::kernel::exec::sys_execve);
            Self::insert_handler(handlers, "execveat", crate::kernel::exec::sys_execveat);
        }

        // SA_RESTART tracking for syscall interruption.
        Self::insert_handler(
            handlers,
            "sigaction",
            crate::kernel::sigaction::sys_sigaction,
        );
        Self::insert_handler(
            handlers,
            "rt_sigaction",
            crate::kernel::sigaction::sys_sigaction,
        );

        // Ioctl sandboxing
        Self::insert_handler(handlers, "ioctl", crate::kernel::ioctl::sys_ioctl);

        if has_mem {
            // Memory sandboxing
            Self::insert_handler(handlers, "brk", crate::kernel::mem::sys_brk);
            Self::insert_handler(handlers, "mremap", crate::kernel::mem::sys_mremap);
        } else {
            allow_calls.extend(["brk", "mremap"]);
        }

        // mmap{,2} are checked for Exec too!
        // We handle them specially in setup_seccomp,
        // as we only want to hook into PROT_EXEC and !MAP_ANONYMOUS.
        Self::insert_handler(handlers, "mmap", crate::kernel::mem::sys_mmap);
        Self::insert_handler(handlers, "mmap2", crate::kernel::mem::sys_mmap2);

        // SafeSetID
        // SAFETY: We do not support diverging FsID from Effective ID.
        // SAFETY: We do not support setgroups (due to pointer deref -> TOCTOU vector)
        // The parent seccomp filter stops setfs*id and setgroups.
        // Parent filter also stops {U,G}ID to privileged user/groups.
        if has_safe_setuid {
            Self::insert_handler(handlers, "setuid", crate::kernel::setid::sys_setuid);
            Self::insert_handler(handlers, "setuid32", crate::kernel::setid::sys_setuid);
            Self::insert_handler(handlers, "setreuid", crate::kernel::setid::sys_setreuid);
            Self::insert_handler(handlers, "setreuid32", crate::kernel::setid::sys_setreuid);
            Self::insert_handler(handlers, "setresuid", crate::kernel::setid::sys_setresuid);
            Self::insert_handler(handlers, "setresuid32", crate::kernel::setid::sys_setresuid);
        } else {
            allow_calls.extend([
                "setuid",
                "setuid32",
                "setreuid",
                "setreuid32",
                "setresuid",
                "setresuid32",
            ]);
        }
        if has_safe_setgid {
            Self::insert_handler(handlers, "setgid", crate::kernel::setid::sys_setgid);
            Self::insert_handler(handlers, "setgid32", crate::kernel::setid::sys_setgid);
            Self::insert_handler(handlers, "setregid", crate::kernel::setid::sys_setregid);
            Self::insert_handler(handlers, "setregid32", crate::kernel::setid::sys_setregid);
            Self::insert_handler(handlers, "setresgid", crate::kernel::setid::sys_setresgid);
            Self::insert_handler(handlers, "setresgid32", crate::kernel::setid::sys_setresgid);
        } else {
            allow_calls.extend([
                "setgid",
                "setgid32",
                "setregid",
                "setregid32",
                "setresgid",
                "setresgid32",
            ]);
        }

        // SAFETY: sysinfo() is a vector of information leak as it
        // provides identical information with the files /proc/meminfo
        // and /proc/loadavg.
        // Since 3.32.4, this can be relaxed with trace/allow_unsafe_sysinfo:1.
        if restrict_sysinfo {
            Self::insert_handler(handlers, "sysinfo", crate::kernel::sysinfo::sys_sysinfo);
        }

        // SAFETY: syslog(2) provides the syslog interface in case
        // the sandbox process has access to the sandbox lock.
        #[cfg(feature = "log")]
        Self::insert_handler(handlers, "syslog", crate::kernel::syslog::sys_syslog);

        // Sanitize uname(2) to protect against information leaks.
        // This is consistent with masking /proc/version.
        Self::insert_handler(handlers, "uname", crate::kernel::uname::sys_uname);

        // signal protection
        Self::insert_handler(handlers, "kill", crate::kernel::signal::sys_kill);
        Self::insert_handler(handlers, "tkill", crate::kernel::signal::sys_tkill);
        Self::insert_handler(handlers, "tgkill", crate::kernel::signal::sys_tgkill);
        Self::insert_handler(
            handlers,
            "rt_sigqueueinfo",
            crate::kernel::signal::sys_rt_sigqueueinfo,
        );
        Self::insert_handler(
            handlers,
            "rt_tgsigqueueinfo",
            crate::kernel::signal::sys_rt_tgsigqueueinfo,
        );
        Self::insert_handler(
            handlers,
            "pidfd_open",
            crate::kernel::signal::sys_pidfd_open,
        );

        // network sandboxing
        Self::insert_handler(handlers, "socketcall", crate::kernel::net::sys_socketcall);
        Self::insert_handler(handlers, "socket", crate::kernel::net::sys_socket);
        Self::insert_handler(handlers, "bind", crate::kernel::net::sys_bind);
        Self::insert_handler(handlers, "connect", crate::kernel::net::sys_connect);
        Self::insert_handler(handlers, "sendto", crate::kernel::net::sys_sendto);
        Self::insert_handler(handlers, "sendmsg", crate::kernel::net::sys_sendmsg);
        Self::insert_handler(handlers, "sendmmsg", crate::kernel::net::sys_sendmmsg);
        Self::insert_handler(handlers, "accept", crate::kernel::net::sys_accept);
        Self::insert_handler(handlers, "accept4", crate::kernel::net::sys_accept4);
        Self::insert_handler(handlers, "getsockname", crate::kernel::net::sys_getsockname);
        Self::insert_handler(handlers, "getsockopt", crate::kernel::net::sys_getsockopt);

        // chroot sandboxing
        if restrict_chroot {
            Self::insert_handler(handlers, "chroot", crate::kernel::chroot::sys_chroot);
        } // else trace/allow_unsafe_chroot:1

        // stat sandboxing
        if !restrict_ptrace {
            Self::insert_handler(handlers, "chdir", crate::kernel::chdir::sys_chdir);
        }
        Self::insert_handler(handlers, "fchdir", crate::kernel::chdir::sys_fchdir);
        Self::insert_handler(
            handlers,
            "getdents64",
            crate::kernel::getdents::sys_getdents64,
        );
        Self::insert_handler(handlers, "stat", crate::kernel::stat::sys_stat);
        Self::insert_handler(handlers, "stat64", crate::kernel::stat::sys_stat64);
        Self::insert_handler(handlers, "statx", crate::kernel::stat::sys_statx);
        Self::insert_handler(handlers, "lstat", crate::kernel::stat::sys_lstat);
        Self::insert_handler(handlers, "lstat64", crate::kernel::stat::sys_lstat64);
        Self::insert_handler(handlers, "fstatat64", crate::kernel::stat::sys_newfstatat);
        Self::insert_handler(handlers, "newfstatat", crate::kernel::stat::sys_newfstatat);
        Self::insert_handler(handlers, "fstat", crate::kernel::stat::sys_fstat);
        Self::insert_handler(handlers, "fstat64", crate::kernel::stat::sys_fstat64);
        Self::insert_handler(handlers, "statfs", crate::kernel::statfs::sys_statfs);
        Self::insert_handler(handlers, "statfs64", crate::kernel::statfs::sys_statfs64);
        Self::insert_handler(handlers, "fstatfs", crate::kernel::statfs::sys_fstatfs);
        Self::insert_handler(handlers, "fstatfs64", crate::kernel::statfs::sys_fstatfs64);
        Self::insert_handler(handlers, "getxattr", crate::kernel::xattr::sys_getxattr);
        Self::insert_handler(handlers, "fgetxattr", crate::kernel::xattr::sys_fgetxattr);
        Self::insert_handler(handlers, "lgetxattr", crate::kernel::xattr::sys_lgetxattr);
        Self::insert_handler(handlers, "getxattrat", crate::kernel::xattr::sys_getxattrat);
        Self::insert_handler(handlers, "listxattr", crate::kernel::xattr::sys_listxattr);
        Self::insert_handler(handlers, "flistxattr", crate::kernel::xattr::sys_flistxattr);
        Self::insert_handler(handlers, "llistxattr", crate::kernel::xattr::sys_llistxattr);
        Self::insert_handler(
            handlers,
            "listxattrat",
            crate::kernel::xattr::sys_listxattrat,
        );
        Self::insert_handler(
            handlers,
            "fanotify_mark",
            crate::kernel::fanotify::sys_fanotify_mark,
        );
        Self::insert_handler(
            handlers,
            "inotify_add_watch",
            crate::kernel::inotify::sys_inotify_add_watch,
        );

        // read/write sandboxing
        Self::insert_handler(handlers, "access", crate::kernel::access::sys_access);
        Self::insert_handler(handlers, "faccessat", crate::kernel::access::sys_faccessat);
        Self::insert_handler(
            handlers,
            "faccessat2",
            crate::kernel::access::sys_faccessat2,
        );
        Self::insert_handler(handlers, "chmod", crate::kernel::chmod::sys_chmod);
        Self::insert_handler(handlers, "fchmod", crate::kernel::chmod::sys_fchmod);
        Self::insert_handler(handlers, "fchmodat", crate::kernel::chmod::sys_fchmodat);
        Self::insert_handler(handlers, "fchmodat2", crate::kernel::chmod::sys_fchmodat2);
        Self::insert_handler(handlers, "chown", crate::kernel::chown::sys_chown);
        Self::insert_handler(handlers, "chown32", crate::kernel::chown::sys_chown);
        Self::insert_handler(handlers, "fchown", crate::kernel::chown::sys_fchown);
        Self::insert_handler(handlers, "fchown32", crate::kernel::chown::sys_fchown);
        Self::insert_handler(handlers, "lchown", crate::kernel::chown::sys_lchown);
        Self::insert_handler(handlers, "lchown32", crate::kernel::chown::sys_lchown);
        Self::insert_handler(handlers, "fchownat", crate::kernel::chown::sys_fchownat);
        Self::insert_handler(handlers, "link", crate::kernel::link::sys_link);
        Self::insert_handler(handlers, "linkat", crate::kernel::link::sys_linkat);
        Self::insert_handler(handlers, "symlink", crate::kernel::symlink::sys_symlink);
        Self::insert_handler(handlers, "symlinkat", crate::kernel::symlink::sys_symlinkat);
        Self::insert_handler(handlers, "unlink", crate::kernel::unlink::sys_unlink);
        Self::insert_handler(handlers, "unlinkat", crate::kernel::unlink::sys_unlinkat);
        Self::insert_handler(handlers, "rmdir", crate::kernel::unlink::sys_rmdir);
        Self::insert_handler(handlers, "mkdir", crate::kernel::mkdir::sys_mkdir);
        Self::insert_handler(handlers, "mkdirat", crate::kernel::mkdir::sys_mkdirat);
        Self::insert_handler(handlers, "mknod", crate::kernel::mknod::sys_mknod);
        Self::insert_handler(handlers, "mknodat", crate::kernel::mknod::sys_mknodat);
        Self::insert_handler(handlers, "creat", crate::kernel::open::sys_creat);
        Self::insert_handler(handlers, "open", crate::kernel::open::sys_open);
        Self::insert_handler(handlers, "openat", crate::kernel::open::sys_openat);
        Self::insert_handler(handlers, "openat2", crate::kernel::open::sys_openat2);
        Self::insert_handler(handlers, "rename", crate::kernel::rename::sys_rename);
        Self::insert_handler(handlers, "renameat", crate::kernel::rename::sys_renameat);
        Self::insert_handler(handlers, "renameat2", crate::kernel::rename::sys_renameat2);
        Self::insert_handler(handlers, "utime", crate::kernel::utime::sys_utime);
        Self::insert_handler(handlers, "utimes", crate::kernel::utime::sys_utimes);
        Self::insert_handler(handlers, "futimesat", crate::kernel::utime::sys_futimesat);
        Self::insert_handler(handlers, "utimensat", crate::kernel::utime::sys_utimensat);
        Self::insert_handler(
            handlers,
            "utimensat_time64",
            crate::kernel::utime::sys_utimensat64,
        );
        Self::insert_handler(handlers, "truncate", crate::kernel::truncate::sys_truncate);
        Self::insert_handler(
            handlers,
            "truncate64",
            crate::kernel::truncate::sys_truncate64,
        );
        Self::insert_handler(
            handlers,
            "ftruncate",
            crate::kernel::truncate::sys_ftruncate,
        );
        Self::insert_handler(
            handlers,
            "ftruncate64",
            crate::kernel::truncate::sys_ftruncate64,
        );
        Self::insert_handler(
            handlers,
            "fallocate",
            crate::kernel::truncate::sys_fallocate,
        );
        Self::insert_handler(handlers, "setxattr", crate::kernel::xattr::sys_setxattr);
        Self::insert_handler(handlers, "fsetxattr", crate::kernel::xattr::sys_fsetxattr);
        Self::insert_handler(handlers, "lsetxattr", crate::kernel::xattr::sys_lsetxattr);
        Self::insert_handler(handlers, "setxattrat", crate::kernel::xattr::sys_setxattrat);
        Self::insert_handler(
            handlers,
            "removexattr",
            crate::kernel::xattr::sys_removexattr,
        );
        Self::insert_handler(
            handlers,
            "removexattrat",
            crate::kernel::xattr::sys_removexattrat,
        );
        Self::insert_handler(
            handlers,
            "fremovexattr",
            crate::kernel::xattr::sys_fremovexattr,
        );
        Self::insert_handler(
            handlers,
            "lremovexattr",
            crate::kernel::xattr::sys_lremovexattr,
        );

        // memfds have mode 777 by default,
        // so we check it for all of Read, Write and Exec sandboxing.
        Self::insert_handler(
            handlers,
            "memfd_create",
            crate::kernel::memfd::sys_memfd_create,
        );
        if !restrict_memfd {
            allow_calls.push("memfd_secret");
        }

        // Prevent unsafe shared memory permissions.
        // We need NOTIFY hook due to ipc(2) multiplexer.
        if restrict_shm {
            Self::insert_handler(handlers, "ipc", crate::kernel::shm::sys_ipc);
            Self::insert_handler(handlers, "shmat", crate::kernel::shm::sys_shmat);
            Self::insert_handler(handlers, "msgctl", crate::kernel::shm::sys_msgctl);
            Self::insert_handler(handlers, "semctl", crate::kernel::shm::sys_semctl);
            Self::insert_handler(handlers, "shmctl", crate::kernel::shm::sys_shmctl);
            Self::insert_handler(handlers, "msgget", crate::kernel::shm::sys_msgget);
            Self::insert_handler(handlers, "semget", crate::kernel::shm::sys_semget);
            Self::insert_handler(handlers, "shmget", crate::kernel::shm::sys_shmget);
        } else {
            allow_calls.extend([
                "ipc", "shmat", "msgctl", "semctl", "shmctl", "msgget", "semget", "shmget",
            ]);
        }

        // For consistency with SHM calls,
        // we also handle mq_open in a NOTIFY hook.
        if restrict_mqueue {
            Self::insert_handler(handlers, "mq_open", crate::kernel::shm::sys_mq_open);
        } else {
            allow_calls.push("mq_open");
        }

        // Allowlist safe system calls.
        for sysname in SAFE_SYSCALLS
            .iter()
            .chain(&allow_calls)
            .chain(FUTEX_SYSCALLS)
            .chain(VDSO_SYSCALLS)
        // SAFETY: get id syscalls are handled by `root/fake' as necessary.
        // .chain(GET_ID_SYSCALLS)
        {
            match ScmpSyscall::from_name(sysname) {
                Ok(syscall) => Self::allow_syscall(sysallow, syscall),
                Err(_) => {
                    info!("ctx": "confine", "op": "allow_syscall",
                        "msg": format!("invalid or unsupported syscall {sysname}"));
                }
            }
        }

        // SAFETY: Set the no-new-privileges attribute.
        set_no_new_privs()?;
        info!("ctx": "set_no_new_privileges", "msg": "no-new-privileges attribute set");

        // SAFETY: Set Memory-Deny-Write-Execute protection mask.
        // REFUSE_EXEC_GAIN is available since Linux-6.3.
        // NO_INHERIT is available since Linux-6.6.
        // For older systems we also apply a mmap filter in load_seccomp_parent.
        let restrict_memory = !flags.allow_unsafe_memory();
        let mdwe_no_inherit = if restrict_memory {
            // Apply restriction globally.
            false
        } else {
            // Apply restrictions only for syd process.
            true
        };

        #[cfg(not(any(
            target_arch = "mips",
            target_arch = "mips32r6",
            target_arch = "mips64",
            target_arch = "mips64r6"
        )))]
        // Set memory-deny-write-execute attribute.
        // Note, mips requires executable stack so we skip on this arch.
        if let Err(errno) = confine_mdwe(mdwe_no_inherit) {
            // EINVAL: MDWE is not supported (Required Linux 6.3 or newer)
            // EPERM: Permission denied (MDWE already enabled?)
            info!("ctx": "set_memory_deny_write_execute",
                "msg": format!("memory-deny-write-execute error: {errno}"));
        } else {
            info!("ctx": "set_memory_deny_write_execute",
                "msg": format!("memory-deny-write-execute set with inherit:{restrict_memory}"));
        }

        if restrict_spec_exec {
            // SAFETY: Set mitigations for speculation misfeatures.
            let nstatus = SpeculationStatus::from_raw(PR_SPEC_FORCE_DISABLE);
            for spec_feat in [
                SpeculationFeature::StoreBypass,
                SpeculationFeature::IndirectBranch,
                SpeculationFeature::L1DFlush,
            ] {
                match speculation_get(spec_feat) {
                    Ok(cstatus) if cstatus.status.can_prctl_set() => {
                        // Fall-through and attempt to set.
                    }
                    Ok(cstatus) => {
                        // prctl cannot set, nothing left to do.
                        info!("ctx": "set_speculative_execution_mitigation",
                            "msg": cstatus.to_string(),
                            "feature": spec_feat.to_string(),
                            "status": cstatus.status.0);
                        continue;
                    }
                    Err(errno) => {
                        info!("ctx": "set_speculative_execution_mitigations",
                            "msg": format!("speculation-get error: {errno}"),
                            "feature": spec_feat.to_string());
                        // Fall-through and attempt to set.
                    }
                }

                match speculation_set(spec_feat, nstatus) {
                    Ok(_) => {
                        let cstatus = SpeculationControlStatus {
                            status: nstatus,
                            feature: spec_feat,
                        };
                        info!("ctx": "set_speculative_execution_mitigation",
                            "msg": cstatus.to_string(),
                            "feature": spec_feat.to_string(),
                            "status": cstatus.status.0);
                    }
                    Err(errno) => {
                        info!("ctx": "set_speculative_execution_mitigations",
                            "msg": format!("speculation-set error: {errno}"),
                            "feature": spec_feat.to_string(),
                            "status": nstatus.0);
                    }
                }
            }
        }

        let unsafe_caps = flags.allow_unsafe_caps();
        let unsafe_ptrace = flags.allow_unsafe_ptrace();
        if !unsafe_caps {
            // Get current effective caps.
            let capset = caps::read(None, caps::CapSet::Effective).unwrap_or_default();

            // Build allow mask based on sandbox flags.
            //
            // SAFETY: CAP_SYS_PTRACE is special because
            // Syd process needs this capability for
            // pidfd_getfd(2), that's why we drop this
            // capability late in the child before exec,
            // see below.
            let mut allow = caps::Capabilities::CAP_SYS_PTRACE;
            if flags.allow_safe_setuid() {
                allow |= caps::Capabilities::CAP_SETUID;
            }
            if flags.allow_safe_setgid() {
                allow |= caps::Capabilities::CAP_SETGID;
            }
            if flags.allow_unsafe_bind() {
                allow |= caps::Capabilities::CAP_NET_BIND_SERVICE;
            }
            if flags.allow_unsafe_chown() {
                allow |= caps::Capabilities::CAP_CHOWN;
            }
            if flags.allow_unsafe_mkbdev() || flags.allow_unsafe_mkcdev() {
                allow |= caps::Capabilities::CAP_MKNOD;
            }
            if flags.allow_unsafe_socket() {
                allow |= caps::Capabilities::CAP_NET_RAW;
            }
            if flags.allow_unsafe_syslog() {
                allow |= caps::Capabilities::CAP_SYSLOG;
            }
            if flags.allow_unsafe_time() {
                allow |= caps::Capabilities::CAP_SYS_TIME;
            }

            // Calculate capabilities to keep.
            //
            // SAFETY: Drop CAP_SYS_PTRACE from Inheritable capabilities,
            // unless trace/allow_unsafe_ptrace:1 is set to allow ptrace(2).
            let capeff = capset & allow;
            let mut capinh = capeff;
            if !unsafe_ptrace {
                capinh.remove(caps::Capabilities::CAP_SYS_PTRACE);
            }

            // SAFETY: Drop CAP_CHOWN from Inheritable capabilities,
            // even if trace/allow_unsafe_chmod is set.
            // This ensures only syd-emulator threads can ever run
            // the privileged chmod(2) calls after access check.
            capinh.remove(caps::Capabilities::CAP_CHOWN);

            // SAFETY: Drop CAP_MKNOD from Inheritable capabilities,
            // even if trace/allow_unsafe_mk{b,c}dev is set.
            // This ensures only syd-emulator threads can ever run
            // these privileged mknod(2) calls after access check.
            capinh.remove(caps::Capabilities::CAP_MKNOD);

            // SAFETY: Drop CAP_NET_BIND_SERVICE from Inheritable capabilities,
            // even if trace/allow_unsafe_bind is set.
            // This ensures only syd-emulator threads can ever run
            // the privileged bind(2) calls after access check.
            capinh.remove(caps::Capabilities::CAP_NET_BIND_SERVICE);

            // SAFETY: Drop CAP_NET_RAW from Inheritable capabilities,
            // even if trace/allow_unsafe_socket is set.
            // This ensures only syd-emulator threads can ever run
            // the privileged socket(2) calls after access check.
            capinh.remove(caps::Capabilities::CAP_NET_RAW);

            // SAFETY:
            // 1. Set effective capabilities.
            // 2. Set permitted capabilities to ensure we cannot gain caps back.
            // 3. Set inheritable capabilities to ensure sandbox process inherits.
            //
            // Set at once using set_all to reduce syscalls.
            caps::set_all(None, capeff, capeff, capinh)?;

            if log_enabled!(LogLevel::Info) {
                let caps_rem = (capset & !capeff)
                    .iter()
                    .filter_map(|f| caps::Capability::try_from(f).ok())
                    .map(|c| c.to_string())
                    .collect::<Vec<_>>();
                let caps_set = capeff
                    .iter()
                    .filter_map(|f| caps::Capability::try_from(f).ok())
                    .map(|c| c.to_string())
                    .collect::<Vec<_>>();
                if caps_rem.is_empty() && caps_set.is_empty() {
                    info!("ctx": "restrict_linux_capabilities",
                        "msg": "no Linux capabilities to restrict");
                } else {
                    info!("ctx": "restrict_linux_capabilities",
                        "msg": "Linux capabilities restricted",
                        "caps_set": caps_set, "caps_rem": caps_rem);
                }
            }
        }

        // Register as a process subreaper if we're not already pid1.
        // This is important because otherwise processes will be
        // reparented to the actual pid1, after which we can no longer
        // access their /proc/pid/mem without ptrace rights.
        let pid = Pid::this().as_raw();
        if pid != 1 {
            set_child_subreaper(true)?;
            info!("ctx": "set_child_subreaper",
                "msg": "child-subreaper attribute set",
                "sub": pid);
        }

        // Apply seccomp hardening for the Syd process itself.
        // This also inherits to the child process, and
        // unshare, mount etc. restrictions happen here.
        Self::load_seccomp_parent(sandbox)?;

        Ok(())
    }

    /// Insert this system call to the list of allowed system calls.
    /// No filtering is done one these system calls and they're allowed at the kernel level.
    fn allow_syscall(sysallow: &mut AllowSet, syscall: ScmpSyscall) {
        sysallow.insert(syscall);
    }

    /// Insert a system call handler.
    #[expect(clippy::cognitive_complexity)]
    #[expect(clippy::disallowed_methods)]
    fn insert_handler(
        handlers: &mut HandlerMap,
        syscall_name: &'static str,
        handler: impl Fn(UNotifyEventRequest) -> ScmpNotifResp + Clone + Send + Sync + 'static,
    ) {
        for arch in SCMP_ARCH {
            if let Ok(sys) = ScmpSyscall::from_name_by_arch(syscall_name, *arch) {
                #[expect(clippy::disallowed_methods)]
                handlers
                    .insert(
                        Sydcall(sys, scmp_arch_raw(*arch)),
                        Arc::new(Box::new(handler.clone())),
                    )
                    .unwrap();
            } else {
                info!("ctx": "confine", "op": "hook_syscall",
                    "msg": format!("invalid or unsupported syscall {syscall_name}"));
            }

            // Support the new non-multiplexed ipc syscalls.
            if IPC_ARCH.contains(arch) {
                let sys_ipc = match syscall_name {
                    "shmat" => Some(397),
                    "msgctl" => Some(402),
                    "semctl" => Some(394),
                    "shmctl" => Some(396),
                    "msgget" => Some(399),
                    "semget" => Some(393),
                    "shmget" => Some(395),
                    _ => None,
                };

                if let Some(sys) = sys_ipc {
                    #[expect(clippy::disallowed_methods)]
                    handlers
                        .insert(
                            Sydcall(ScmpSyscall::from(sys), scmp_arch_raw(*arch)),
                            Arc::new(Box::new(handler.clone())),
                        )
                        .unwrap();
                    continue;
                }
            }

            // Support the new non-multiplexed network syscalls on MIPS, PPC, S390 & X86.
            let sys = match *arch {
                ScmpArch::M68k => match syscall_name {
                    "socket" => 356,
                    "bind" => 358,
                    // no accept on m68k.
                    "accept4" => 361,
                    "connect" => 359,
                    "getsockname" => 364,
                    "getsockopt" => 362,
                    "sendto" => 366,
                    "sendmsg" => 367,
                    "sendmmsg" => 372,
                    _ => continue,
                },
                ScmpArch::Mips | ScmpArch::Mipsel => match syscall_name {
                    "socket" => 183,
                    "bind" => 169,
                    "accept" => 168,
                    "accept4" => 334,
                    "connect" => 170,
                    "getsockname" => 172,
                    "getsockopt" => 173,
                    "sendto" => 180,
                    "sendmsg" => 179,
                    "sendmmsg" => 343,
                    _ => continue,
                },
                ScmpArch::Ppc | ScmpArch::Ppc64 | ScmpArch::Ppc64Le => match syscall_name {
                    "socket" => 326,
                    "bind" => 327,
                    "accept" => 330,
                    "accept4" => 344,
                    "connect" => 328,
                    "getsockname" => 331,
                    "getsockopt" => 340,
                    "sendto" => 335,
                    "sendmsg" => 341,
                    "sendmmsg" => 349,
                    _ => continue,
                },
                ScmpArch::S390X | ScmpArch::S390 => match syscall_name {
                    "socket" => 359,
                    "bind" => 361,
                    // no accept on s390x.
                    "accept4" => 364,
                    "connect" => 362,
                    "getsockname" => 367,
                    "getsockopt" => 365,
                    "sendto" => 369,
                    "sendmsg" => 370,
                    "sendmmsg" => 358,
                    _ => continue,
                },
                ScmpArch::X86 => match syscall_name {
                    "socket" => 359,
                    "bind" => 361,
                    // no accept on x86.
                    "accept4" => 364,
                    "connect" => 362,
                    "getsockname" => 367,
                    "getsockopt" => 365,
                    "sendto" => 369,
                    "sendmsg" => 370,
                    "sendmmsg" => 345,
                    _ => continue,
                },
                _ => continue,
            };

            handlers
                .insert(
                    Sydcall(ScmpSyscall::from(sys), scmp_arch_raw(*arch)),
                    Arc::new(Box::new(handler.clone())),
                )
                .unwrap();

            #[expect(clippy::arithmetic_side_effects)]
            if matches!(*arch, ScmpArch::Mips | ScmpArch::Mipsel) {
                // This is a libseccomp oddity,
                // it could be a bug in the syscall multiplexer.
                // TODO: Investigate and submit a bug report.
                handlers
                    .insert(
                        Sydcall(ScmpSyscall::from(sys + 4000), scmp_arch_raw(*arch)),
                        Arc::new(Box::new(handler.clone())),
                    )
                    .unwrap();
            }
        }
    }

    /// Run a command with seccomp filter.
    /// This method will fork a child process, do some preparations and run the command in it.
    #[expect(clippy::cognitive_complexity)]
    #[expect(clippy::type_complexity)]
    fn spawn(
        mut self,
        mut command: crate::unshare::Command,
    ) -> SydResult<(
        Arc<WorkerCache<'static>>,
        Arc<RwLock<Sandbox>>,
        Option<AesLock>,
        SydJoinHandle<()>,
        Arc<AtomicBool>,
    )> {
        let mut sandbox = self.sandbox.write().unwrap_or_else(|err| err.into_inner());

        // SAFETY: Ensure Crypt sandboxing keys are wiped from memory
        // before sandbox process start to ensure there's no race with
        // the sandbox process.
        sandbox.set_crypt()?;

        if sandbox.lock.is_none() {
            // SAFETY: Set the sandbox lock if the state is unspecified.
            // This is safer than the previous default LockState::Exec.
            // We set this right before exec to ensure the initial configuration
            // passes through (ie config file and CLI options), however
            // we still do it pre-exec to ensure there's no race with
            // the sandbox process.
            //
            // !sandbox.is_running -> lock returns no errors.
            #[expect(clippy::disallowed_methods)]
            sandbox.lock(LockState::Set).expect("lock sandbox");
        }

        // SAFETY: Seal critical sandbox memory regions as read-only.
        // Tear down the sandbox if sealing is not possible.
        if sandbox.locked() {
            #[expect(clippy::disallowed_methods)]
            sandbox.seal().expect("seal sandbox");
        }

        let ssb = sandbox.flags.allow_unsafe_spec_exec();
        let ioctl_denylist = sandbox.get_ioctl_deny();
        drop(sandbox); // release the write-lock.

        // Set command PTY as necessary.
        command.pty(self.pty_child.as_ref().map(|fd| fd.as_raw_fd()));

        // Set seccomp filter to be applied.
        let seccomp_filter = self.setup_seccomp(ssb)?;
        command.seccomp_filter(seccomp_filter);

        // Set ioctl denylist to be applied.
        command.ioctl_denylist(Some(ioctl_denylist));

        // Spawn child under sandbox.
        //
        // Ready, set, go!
        let child = command.spawn()?;
        let pid = child.id();
        let seccomp_fd = child.seccomp_fd;

        // Having passed the PTY child FD to the sandbox process,
        // it is now safe to close our instance of it.
        if let Some(fd) = self.pty_child.take() {
            let _ = close(fd);
        }

        // Attempt to set file-max to hard limit overriding the soft limit.
        // Since this is just an attempt for convenience, we log errors with info.
        // We do this late to access the static PROC_FILE() and to ensure the
        // sandbox process does _not_ inherit the file limits.
        let nr_open = proc_fs_nr_open().unwrap_or(0x100000);
        match getrlimit(Resource::RLIMIT_NOFILE)? {
            (soft_limit, hard_limit) if soft_limit < hard_limit => {
                // Careful on 32-bit, setrlimit expects an u32 not an u64!
                #[expect(clippy::useless_conversion)]
                let hard_limit = hard_limit.min(nr_open.try_into().unwrap_or(0x100000));
                match setrlimit(Resource::RLIMIT_NOFILE, hard_limit, hard_limit) {
                    Ok(_) => {
                        info!("ctx": "run", "op": "set_rlimit_nofile",
                            "msg": format!("file-max limit increased from {soft_limit} to {hard_limit}"));
                    }
                    Err(errno) => {
                        info!("ctx": "run", "op": "set_rlimit_nofile",
                            "msg": format!("setrlimit error: {errno}"),
                            "err": errno as i32);
                    }
                }
            }
            (_, hard_limit) => {
                info!("ctx": "run", "op": "set_rlimit_nofile",
                    "msg": format!("file-max limit is already set to hard limit {hard_limit}"));
            }
        };

        #[expect(clippy::cast_possible_wrap)]
        let pid = Pid::from_raw(pid as i32);
        let mut sandbox = self.sandbox.write().unwrap_or_else(|err| err.into_inner());
        sandbox.set_child(pid, child.pid_fd);
        let locked = sandbox.locked();
        let restrict_dumps = !sandbox.flags.allow_unsafe_dumpable();
        let sync_scmp = sandbox.flags.sync_scmp();
        let flags = *sandbox.flags;
        drop(sandbox);

        // Set synchronous mode if requested and supported,
        // so each syscall handler thread wakes up
        // on the same CPU as the respective sandbox process.
        if sync_scmp {
            match seccomp_notify_set_flags(seccomp_fd, SECCOMP_USER_NOTIF_FD_SYNC_WAKE_UP) {
                Ok(_) => {
                    info!("ctx": "set_seccomp_synchronous_mode",
                        "msg": "set seccomp synchronous mode",
                        "fd": seccomp_fd);
                }
                Err(Errno::ENOSYS) => {
                    info!("ctx": "set_seccomp_synchronous_mode",
                        "msg": "seccomp synchronous mode isn't supported on this system",
                        "fd": seccomp_fd);
                }
                Err(errno) => {
                    error!("ctx": "set_seccomp_synchronous_mode",
                        "msg": format!("set seccomp synchronous mode error: {errno}"),
                        "fd": seccomp_fd, "err": errno as i32);
                    // Continue, as this mode is not significant to our use.
                }
            };
        }

        // SAFETY: Set Syd process dumpable attribute to false,
        // unless trace/allow_unsafe_dumpable:1 was passed at startup.
        // We do this after spawning the child but before spawning the
        // system call handler threads to ensure the sandbox process
        // does not inherit the attribute but cannot attach to Syd.
        if restrict_dumps {
            set_dumpable(false)?;
        }

        // SAFETY: At this point Syd has successfully forked a new
        // process to execute the sandbox process. As such Syd no longer
        // needs the execve, and execveat system calls. Let's
        // disable these critical system calls here to ensure a
        // compromised Syd process cannot abuse them.
        // EXCEPTION: Sandbox is not locked and we need exec for cmd/exec.
        let mut ctx = ScmpFilterContext::new(ScmpAction::Allow)?;
        // Enforce the NO_NEW_PRIVS functionality before
        // loading the seccomp filter into the kernel.
        ctx.set_ctl_nnp(true)?;
        // Disable Speculative Store Bypass mitigations
        // with trace/allow_unsafe_spec_exec:1
        ctx.set_ctl_ssb(ssb)?;
        // Synchronize filter to all threads.
        ctx.set_ctl_tsync(true)?;
        // We kill for bad system call and bad arch.
        ctx.set_act_badarch(ScmpAction::KillProcess)?;
        // Use a binary tree sorted by syscall number if possible.
        let _ = ctx.set_ctl_optimize(2);
        // SAFETY: Do NOT add supported architectures to the filter.
        // This ensures Syd can never run a non-native system call,
        // which we do not need at all.
        // seccomp_add_architectures(&mut ctx).map_err(|e| err2no(&e))?;

        // SAFETY: Mitigate ret2mprotect for a compromised Syd process.
        // Be swift and kill process as this attempt is most certainly
        // malicious and the kill action cannot be misused to DOS the
        // Syd process.
        // Note, mips requires executable stack so we skip on this arch.
        #[cfg(not(any(
            target_arch = "mips",
            target_arch = "mips32r6",
            target_arch = "mips64",
            target_arch = "mips64r6"
        )))]
        {
            const X: u64 = libc::PROT_EXEC as u64;
            for sysname in ["mprotect", "pkey_mprotect"] {
                #[expect(clippy::disallowed_methods)]
                let syscall = ScmpSyscall::from_name(sysname).unwrap();
                ctx.add_rule_conditional(
                    ScmpAction::KillProcess,
                    syscall,
                    &[scmp_cmp!($arg2 & X == X)],
                )?;
            }
        }

        // SAFETY:
        // Mitigate the following for a compromised Syd process:
        // 1. cachestat, mincore: Page-cache attacks
        // 2. msgsnd: Kernel heap spraying attacks
        // 3. sethostname, setdomainname: Change UTS host/domain name.
        // Be swift and kill the process as
        // this attempt it most certainly malicious and the kill action
        // cannot be misused to DOS the Syd process.
        for sysname in PAGE_CACHE_SYSCALLS
            .iter()
            .chain(UTS_SYSCALLS)
            .chain(&["msgsnd"])
        {
            match ScmpSyscall::from_name(sysname) {
                Ok(syscall) => {
                    ctx.add_rule(ScmpAction::KillProcess, syscall)?;
                }
                Err(_) => {
                    info!("ctx": "confine", "op": "kill_syscall",
                        "msg": format!("invalid or unsupported syscall {sysname}"));
                }
            }
        }

        if locked {
            // SAFETY:
            // 1. Deny exec(3) calls if locked.
            // 2. Allow execveat(2) with AT_EXECVE_CHECK for
            //    Linux>=6.14.
            // 3. Be swift and kill process as this attempt is most
            //    certainly malicious and the kill action cannot be misused
            //    to DOS the Syd process.
            #[expect(clippy::disallowed_methods)]
            let syscall = ScmpSyscall::from_name("execve").unwrap();
            ctx.add_rule(ScmpAction::KillProcess, syscall)?;

            #[expect(clippy::disallowed_methods)]
            let syscall = ScmpSyscall::from_name("execveat").unwrap();
            if *HAVE_AT_EXECVE_CHECK {
                #[expect(clippy::cast_sign_loss)]
                let atcheck = AT_EXECVE_CHECK.bits() as u64;
                ctx.add_rule_conditional(
                    ScmpAction::KillProcess,
                    syscall,
                    &[scmp_cmp!($arg4 & atcheck == 0)],
                )?;
            } else {
                ctx.add_rule(ScmpAction::KillProcess, syscall)?;
            }
        }

        // SAFETY: After this point we no longer need the PTRACE_SEIZE
        // operation. Since this is a powerful operation, we apply a
        // quick seccomp filter to disable it from ever happening again.
        // This way a compromised Syd process cannot attach to any other
        // process in the system.
        // SAFETY: PTRACE_ATTACH is most certainly malicious,
        // add to kill set.
        // SAFETY: We add these ptrace rules without checking the state
        // of allow_unsafe_ptrace, because if allow_unsafe_ptrace is off we
        // have already performed the initial PTRACE_SEIZE, and if
        // allow_unsafe_ptrace is on we never need PTRACE_SEIZE to begin
        // with.
        #[expect(clippy::disallowed_methods)]
        let syscall = ScmpSyscall::from_name("ptrace").unwrap();
        #[expect(clippy::cast_lossless)]
        for op in [libc::PTRACE_ATTACH, libc::PTRACE_SEIZE] {
            ctx.add_rule_conditional(
                ScmpAction::KillProcess,
                syscall,
                &[scmp_cmp!($arg0 == op as u64)],
            )?;
        }

        // Export seccomp rules if requested.
        let export = match ExportMode::from_env() {
            Some(ExportMode::BerkeleyPacketFilter) => {
                #[expect(clippy::disallowed_methods)]
                let file = OpenOptions::new()
                    .write(true)
                    .create_new(true)
                    .mode(0o400)
                    .open("syd_process.bpf")?;
                ctx.export_bpf(file)?;
                true
            }
            Some(ExportMode::PseudoFiltercode) => {
                // Lock stdout to prevent concurrent access.
                let mut stdout = std::io::stdout().lock();

                rwriteln!(stdout, "# Syd process rules with seccomp_fd {seccomp_fd}")?;
                rwrite!(stdout, "{}", seccomp_export_pfc(&ctx)?)?;
                true
            }
            _ => false,
        };

        // Load the seccomp filter unless:
        // a. We're running in debug mode with SYD_SKIP_SCMP=1.
        // b. We're exporting seccomp filters
        if !export && env::var_os(ENV_SKIP_SCMP).is_none() {
            ctx.load()?;
        }

        self.supervise(seccomp_fd, flags)
    }

    // Set up seccomp for the sandbox process.
    #[expect(clippy::cognitive_complexity)]
    fn setup_seccomp(&self, ssb: bool) -> SydResult<ScmpFilterContext> {
        let mut ctx = ScmpFilterContext::new(ScmpAction::Errno(libc::ENOSYS))?;
        // Enforce the NO_NEW_PRIVS functionality before
        // loading the seccomp filter into the kernel.
        ctx.set_ctl_nnp(true)?;
        // Disable Speculative Store Bypass mitigations
        // with trace/allow_unsafe_spec_exec:1
        ctx.set_ctl_ssb(ssb)?;
        // Synchronize filter to all threads.
        ctx.set_ctl_tsync(true)?;
        // Request wait killable semantics.
        #[cfg(libseccomp_v2_6)]
        ctx.set_ctl_waitkill(true)?;
        // We deny with ENOSYS for bad/unsupported system call,
        // and kill process for bad arch.
        ctx.set_act_badarch(ScmpAction::KillProcess)?;
        // Use a binary tree sorted by syscall number if possible.
        let _ = ctx.set_ctl_optimize(2);
        // We don't want ECANCELED, we want actual errnos.
        let _ = ctx.set_api_sysrawrc(true);

        seccomp_add_architectures(&mut ctx)?;

        // Acquire the read lock to sandbox configuration.
        let sandbox = self.sandbox.read().unwrap_or_else(|err| err.into_inner());
        // Note: if lock is None, it'll be set to Some(LockState::Set),
        // when the sandbox child starts executing.
        let is_lock = matches!(sandbox.lock, None | Some(LockState::Set));
        let safe_syslog = sandbox.flags.allow_safe_syslog();
        let deny_tsc = sandbox.flags.deny_tsc();
        let fake_root = sandbox.flags.fake_root();
        let has_mem = sandbox.enabled(Capability::CAP_MEM);
        let restrict_cbpf = !sandbox.flags.allow_unsafe_cbpf();
        let restrict_ebpf = !sandbox.flags.allow_unsafe_ebpf();
        let restrict_chroot = !sandbox.flags.allow_unsafe_chroot();
        let restrict_pivot_root = !sandbox.flags.allow_unsafe_pivot_root();
        let restrict_cpu = !sandbox.flags.allow_unsafe_cpu();
        let restrict_keyring = !sandbox.flags.allow_unsafe_keyring();
        let restrict_iouring = !sandbox.flags.allow_unsafe_iouring();
        let restrict_mount = !sandbox.flags.allow_unsafe_unshare_mount();
        let restrict_uts = !sandbox.flags.allow_unsafe_unshare_uts();
        let restrict_deprecated = !sandbox.flags.allow_unsafe_deprecated();
        let restrict_mbind = !sandbox.flags.allow_unsafe_mbind();
        let restrict_msgsnd = !sandbox.flags.allow_unsafe_msgsnd();
        let restrict_nice = !sandbox.flags.allow_unsafe_nice();
        let restrict_page_cache = !sandbox.flags.allow_unsafe_page_cache();
        let restrict_perf = !sandbox.flags.allow_unsafe_perf();
        let restrict_pkey = !sandbox.flags.allow_unsafe_pkey();
        let restrict_personality = !sandbox.flags.allow_unsafe_personality();
        let restrict_prctl = !sandbox.flags.allow_unsafe_prctl();
        let restrict_spec_exec = !sandbox.flags.allow_unsafe_spec_exec();
        let restrict_sysinfo = !sandbox.flags.allow_unsafe_sysinfo();
        let restrict_pipe = !sandbox.flags.allow_unsafe_pipe();
        let restrict_prlimit = !sandbox.flags.allow_unsafe_prlimit();
        let restrict_ptrace = !sandbox.flags.allow_unsafe_ptrace();
        let restrict_sigreturn = !sandbox.flags.allow_unsafe_sigreturn();
        let restrict_rseq = !sandbox.flags.allow_unsafe_rseq();
        let restrict_sync = !sandbox.flags.allow_unsafe_sync();
        let restrict_time = !sandbox.flags.allow_unsafe_time();
        let skip_append = sandbox.skip_append();
        drop(sandbox); // release the read lock.

        // Fakeroot
        let id_action = if fake_root {
            ScmpAction::Errno(0)
        } else {
            ScmpAction::Allow
        };
        for sysname in GET_ID_SYSCALLS {
            if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
                ctx.add_rule(id_action, syscall)?;
            } else {
                info!("ctx": "confine", "op": "filter_syscall",
                    "msg": format!("invalid or unsupported syscall {sysname}"));
            }
        }

        // Add notify rules for system calls with handlers.
        // Collect names into a HashSet to ensure uniqueness across architectures.
        let mut sydset: SydHashSet<String> = SydHashSet::default();
        for (&sydcall, _) in &*self.handlers {
            if i32::from(sydcall.0) == __NR_SCMP_ERROR {
                // Unsupported system call for the given architecture,
                // move on.
                continue;
            }

            // ? -> unsupported, see Sydcall::Display.
            // We handle chroot specially below.
            // We handle mmap{,2}, and sendto specially below.
            // We handle kill calls specially below.
            // We handle prctl specially where we only hook PR_SET_NAME.
            // We handle fcntl{,64} specially where we only hook F_SETFL with O_APPEND unset.
            // We handle syslog(2) calls specially below.
            // We allow/hook sysinfo(2) based on trace/allow_unsafe_sysinfo:1 since 3.32.4
            const SYS_EXCEPT: &[&str] = &[
                "?",
                "chroot",
                "fcntl",
                "fcntl64",
                "kill",
                "mmap",
                "mmap2",
                "prctl",
                "rt_sigaction",
                "rt_sigqueueinfo",
                "rt_tgsigqueueinfo",
                "sigaction",
                "sysinfo",
                "syslog",
                "tgkill",
                "tkill",
            ];
            let name = sydcall.to_string();
            if SYS_EXCEPT.binary_search(&name.as_str()).is_ok() {
                continue;
            }

            let syscall = if sydset.insert(name.clone()) {
                if let Ok(syscall) = ScmpSyscall::from_name(&name) {
                    syscall
                } else {
                    info!("ctx": "confine", "op": "hook_box_syscall",
                        "msg": format!("invalid or unsupported syscall {name}"));
                    continue;
                }
            } else {
                continue;
            };

            ctx.add_rule(ScmpAction::Notify, syscall)?;
        }

        // Add allow rules for system calls in the default allow list.
        let syscall_allow: Vec<_> = self.sysallow.iter().copied().collect();
        for syscall in &syscall_allow {
            ctx.add_rule(ScmpAction::Allow, *syscall)?;
        }

        // Skip hooking into kill syscalls which are called
        // with the dummy signal 0. This is used to determine
        // the existence of processes and is considered safe use.
        for sysname in ["kill", "rt_sigqueueinfo", "tkill"] {
            let syscall = ScmpSyscall::from_name(sysname)?;
            ctx.add_rule_conditional(ScmpAction::Allow, syscall, &[scmp_cmp!($arg1 == 0)])?;
            ctx.add_rule_conditional(ScmpAction::Notify, syscall, &[scmp_cmp!($arg1 != 0)])?;
        }
        for sysname in ["tgkill", "rt_tgsigqueueinfo"] {
            let syscall = ScmpSyscall::from_name(sysname)?;
            ctx.add_rule_conditional(ScmpAction::Allow, syscall, &[scmp_cmp!($arg2 == 0)])?;
            ctx.add_rule_conditional(ScmpAction::Notify, syscall, &[scmp_cmp!($arg2 != 0)])?;
        }

        // Hook {rt_}sigaction(2) for SA_RESTART tracking.
        // Skip hooking into sigaction calls where the new action is NULL.
        for sysname in ["sigaction", "rt_sigaction"] {
            let syscall = ScmpSyscall::from_name(sysname)?;
            ctx.add_rule_conditional(ScmpAction::Allow, syscall, &[scmp_cmp!($arg1 == 0)])?;
            ctx.add_rule_conditional(ScmpAction::Notify, syscall, &[scmp_cmp!($arg1 != 0)])?;
        }

        // Since 3.32.4, we skip hooking into sysinfo(2) syscalls
        // if trace/allow_unsafe_sysinfo:1 is given.
        let sysname = "sysinfo";
        if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
            if restrict_sysinfo {
                ctx.add_rule(ScmpAction::Notify, syscall)?;
            } else {
                ctx.add_rule(ScmpAction::Allow, syscall)?;
            }
        }

        // Skip hooking into syslog(2) syscalls if the log feature
        // is off or if the sandbox has already been locked.
        let sysname = "syslog";
        #[expect(clippy::disallowed_methods)]
        let syscall = ScmpSyscall::from_name(sysname).unwrap();
        if !cfg!(feature = "log") || (!safe_syslog && is_lock) {
            ctx.add_rule(ScmpAction::Errno(libc::EPERM), syscall)?;
        } else {
            ctx.add_rule(ScmpAction::Notify, syscall)?;
        }

        // mmap{,2} hooks vary based on options:
        // 0. If Memory sandboxing is on, hook without flag check.
        // 1. Hook into PROT_EXEC|!MAP_ANONYMOUS for exec check.
        // 2. Hook into MAP_SHARED for append-only check.
        let syscalls = ["mmap", "mmap2"];
        if has_mem {
            for sysname in syscalls {
                #[expect(clippy::disallowed_methods)]
                let syscall = ScmpSyscall::from_name(sysname).unwrap();
                ctx.add_rule(ScmpAction::Notify, syscall)?;
            }
        } else {
            const PROT_EXEC: u64 = libc::PROT_EXEC as u64;
            const MAP_ANONYMOUS: u64 = libc::MAP_ANONYMOUS as u64;
            const MAP_SHARED: u64 = libc::MAP_SHARED as u64;
            for sysname in syscalls {
                #[expect(clippy::disallowed_methods)]
                let syscall = ScmpSyscall::from_name(sysname).unwrap();

                // Hook into fd-based mappings with PROT_EXEC
                // but without MAP_ANONYMOUS.
                ctx.add_rule_conditional(
                    ScmpAction::Notify,
                    syscall,
                    &[
                        scmp_cmp!($arg2 & PROT_EXEC == PROT_EXEC),
                        scmp_cmp!($arg3 & MAP_ANONYMOUS == 0),
                        scmp_cmp!($arg4 <= FD_MAX),
                    ],
                )?;

                if !skip_append {
                    // Hook into fd-based mappings with MAP_SHARED.
                    ctx.add_rule_conditional(
                        ScmpAction::Notify,
                        syscall,
                        &[
                            scmp_cmp!($arg3 & MAP_SHARED == MAP_SHARED),
                            scmp_cmp!($arg4 <= FD_MAX),
                        ],
                    )?;
                }

                // Allow negations.
                ctx.add_rule_conditional(
                    ScmpAction::Allow,
                    syscall,
                    &[scmp_cmp!($arg2 & PROT_EXEC == 0)],
                )?;
                ctx.add_rule_conditional(
                    ScmpAction::Allow,
                    syscall,
                    &[scmp_cmp!($arg3 & MAP_ANONYMOUS == MAP_ANONYMOUS)],
                )?;
                ctx.add_rule_conditional(ScmpAction::Allow, syscall, &[scmp_cmp!($arg4 > FD_MAX)])?;
            }
        }

        // Hook chdir(2) via ptrace(2).
        #[expect(clippy::disallowed_methods)]
        let sys_chdir = ScmpSyscall::from_name("chdir").unwrap();
        if restrict_ptrace {
            ctx.add_rule(ScmpAction::Trace(PTRACE_DATA_CHDIR), sys_chdir)?;
        } else {
            // Hook into chdir with seccomp.
            // This was already done in init,
            // so we don't have to repeat here.
        }

        // Hook execve(2) and execveat(2) via ptrace(2).
        if restrict_ptrace {
            #[expect(clippy::disallowed_methods)]
            let sys_execve = ScmpSyscall::from_name("execve").unwrap();
            ctx.add_rule(ScmpAction::Trace(PTRACE_DATA_EXECVE), sys_execve)?;

            #[expect(clippy::disallowed_methods)]
            let sys_execveat = ScmpSyscall::from_name("execveat").unwrap();
            ctx.add_rule(ScmpAction::Trace(PTRACE_DATA_EXECVEAT), sys_execveat)?;
        }

        // Hook {rt_}sigreturn(2) via ptrace(2).
        #[expect(clippy::disallowed_methods)]
        let sys_sigreturn = ScmpSyscall::from_name("sigreturn").unwrap();
        #[expect(clippy::disallowed_methods)]
        let sys_rt_sigreturn = ScmpSyscall::from_name("rt_sigreturn").unwrap();
        if restrict_ptrace && restrict_sigreturn {
            ctx.add_rule(ScmpAction::Trace(PTRACE_DATA_SIGRETURN), sys_sigreturn)?;
            ctx.add_rule(
                ScmpAction::Trace(PTRACE_DATA_RT_SIGRETURN),
                sys_rt_sigreturn,
            )?;
        } else {
            // TODO: Research if something similar is doable with seccomp only.
            ctx.add_rule(ScmpAction::Allow, sys_sigreturn)?;
            ctx.add_rule(ScmpAction::Allow, sys_rt_sigreturn)?;
        }

        // Restriction 0: Handle no-op syscalls:
        // 1. Turn chroot(2) into no-op if trace/allow_unsafe_chroot:1.
        // 2. Turn pivot_root(2) into no-op if trace/allow_unsafe_pivot_root:1.
        let mut noop_syscalls = Vec::with_capacity(2);
        if restrict_chroot {
            let sysname = "chroot";
            if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
                ctx.add_rule(ScmpAction::Notify, syscall)?;
            } else {
                info!("ctx": "confine", "op": "notify_box_syscall",
                    "msg": format!("invalid or unsupported syscall {sysname}"));
            }
        } else {
            noop_syscalls.push("chroot");
        }
        if restrict_pivot_root {
            let sysname = "pivot_root";
            if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
                ctx.add_rule(ScmpAction::Errno(libc::EPERM), syscall)?;
            } else {
                info!("ctx": "confine", "op": "deny_box_syscall",
                    "msg": format!("invalid or unsupported syscall {sysname}"));
            }
        } else {
            noop_syscalls.push("pivot_root");
        }
        for sysname in noop_syscalls {
            if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
                ctx.add_rule(ScmpAction::Errno(0), syscall)?;
            } else {
                info!("ctx": "confine", "op": "noop_box_syscall",
                    "msg": format!("invalid or unsupported syscall {sysname}"));
            }
        }

        // Restriction 1: Deny unsafe set-id system calls.
        // Deny with Errno=0 -> Turn the system calls into no-op.
        // This is for compatibility, e.g. postgres invokes
        // setgroups before setuid and aborts on failure.
        for sysname in UNSAFE_ID_SYSCALLS {
            if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
                ctx.add_rule(ScmpAction::Errno(0), syscall)?;
            } else {
                info!("ctx": "confine", "op": "noop_box_syscall",
                    "msg": format!("invalid or unsupported syscall {sysname}"));
            }
        }

        // Restriction 2: Allowlist known-safe prctls.
        // Things like PR_SET_MM, PR_SET_PTRACER, and PR_SET_SPECULATION_CTRL are left out.
        // PR_SET_TSC is out if deny-tsc is set and allow_unsafe_prctl is unset.
        #[expect(clippy::disallowed_methods)]
        let sys_prctl = ScmpSyscall::from_name("prctl").unwrap();
        if restrict_prctl {
            for (name, opt) in ALLOWLIST_PRCTL {
                if deny_tsc && is_equal(name.as_bytes(), b"PR_SET_TSC") {
                    continue;
                }

                let act = if is_equal(name.as_bytes(), b"PR_SET_NAME") {
                    // SAFETY: Warn on PR_SET_NAME calls.
                    ScmpAction::Notify
                } else if restrict_cbpf && is_suffix(name.as_bytes(), b"_SECCOMP") {
                    // SAFETY:
                    // Deny all seccomp(2) operations with EINVAL,
                    // _unless_ trace/allow_unsafe_cbpf:1 is passed at startup.
                    //
                    // Note, allowing strict mode here is going to make no difference,
                    // as the kernel will return `EINVAL` anyway because a secure
                    // computing mode is already set by Syd and strict mode differs
                    // from the current mode (filter).
                    ScmpAction::Errno(libc::EINVAL)
                } else if restrict_perf && memmem::find(name.as_bytes(), b"_PERF_").is_some() {
                    // SAFETY:
                    // Deny perf prctl(2)s with EINVAL,
                    // _unless_ trace/allow_unsafe_perf:1 is passed at startup.
                    ScmpAction::Errno(libc::EINVAL)
                } else if restrict_spec_exec && is_suffix(name.as_bytes(), b"_SPECULATION_CTRL") {
                    // SAFETY:
                    // Deny speculation prctl(2)s with EINVAL,
                    // _unless_ trace/allow_unsafe_spec_exec:1 is passed at startup.
                    ScmpAction::Errno(libc::EINVAL)
                } else {
                    ScmpAction::Allow
                };

                let cmp = ScmpArgCompare::new(0, ScmpCompareOp::Equal, *opt);
                ctx.add_rule_conditional(act, sys_prctl, &[cmp])?;
            }
        } else {
            ctx.add_rule(ScmpAction::Allow, sys_prctl)?;
        }

        // Restriction 3: Disallow seccomp(2) operations with EINVAL
        // _unless_ trace/allow_unsafe_cbpf:1 is passed at startup.
        //
        // Note, allowing strict mode here is going to make no difference,
        // as the kernel will return `EINVAL` anyway because a secure
        // computing mode is already set by Syd and strict mode differs
        // from the current mode (filter).
        #[expect(clippy::disallowed_methods)]
        let sys_seccomp = ScmpSyscall::from_name("seccomp").unwrap();
        if restrict_cbpf {
            // 1. Allow SECCOMP_GET_ACTION_AVAIL & SECCOMP_GET_NOTIF_SIZES.
            // 2. Deny SECCOMP_SET_MODE_STRICT & SECCOMP_SET_MODE_FILTER with EINVAL.
            // 3. Deny all future seccomp(2) operations.
            for op in [
                libc::SECCOMP_GET_ACTION_AVAIL,
                libc::SECCOMP_GET_NOTIF_SIZES,
            ] {
                ctx.add_rule_conditional(
                    ScmpAction::Allow,
                    sys_seccomp,
                    &[scmp_cmp!($arg0 == u64::from(op))],
                )?;
            }

            for op in [libc::SECCOMP_SET_MODE_STRICT, libc::SECCOMP_SET_MODE_FILTER] {
                ctx.add_rule_conditional(
                    ScmpAction::Errno(libc::EINVAL),
                    sys_seccomp,
                    &[scmp_cmp!($arg0 == u64::from(op))],
                )?;
            }

            // Make the filter future-proof.
            const SECCOMP_OPERATION_MAX: u64 = libc::SECCOMP_GET_NOTIF_SIZES as u64;
            ctx.add_rule_conditional(
                ScmpAction::Errno(libc::EINVAL),
                sys_seccomp,
                &[scmp_cmp!($arg0 > SECCOMP_OPERATION_MAX)],
            )?;
        } else {
            ctx.add_rule(ScmpAction::Allow, sys_seccomp)?;
        }

        // Restriction 4: Disallow eBPF programs unless trace/allow_unsafe_ebpf:1
        if !restrict_ebpf {
            for sysname in EBPF_SYSCALLS {
                if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
                    if *sysname == "bpf" {
                        // SAFETY: Even with trace/allow_unsafe_ebpf:1,
                        // do not allow the BPF commands
                        // BPF_MAP_CREATE and BPF_PROG_LOAD
                        // which are privileged and require CAP_BPF.
                        // See:
                        // https://git.kernel.org/pub/scm/linux/kernel/git/torvalds/linux.git/commit/?id=c8644cd0efe719608ddcb341bcf087d4bc0bf6b8
                        const BPF_MAP_CREATE: u8 = 0;
                        const BPF_PROG_LOAD: u8 = 5;
                        const MAX_BPF_CMD: u8 = 38; /* As of Linux>=6.15 */
                        for bpf_cmd in 0..MAX_BPF_CMD {
                            let action = if matches!(bpf_cmd, BPF_MAP_CREATE | BPF_PROG_LOAD) {
                                ScmpAction::Errno(libc::EPERM)
                            } else {
                                ScmpAction::Allow
                            };
                            ctx.add_rule_conditional(
                                action,
                                syscall,
                                &[scmp_cmp!($arg0 == bpf_cmd.into())],
                            )?;
                        }
                        // Ensure future compatibility.
                        ctx.add_rule_conditional(
                            ScmpAction::Errno(libc::EPERM),
                            syscall,
                            &[scmp_cmp!($arg0 >= MAX_BPF_CMD.into())],
                        )?;
                    } else {
                        ctx.add_rule(ScmpAction::Allow, syscall)?;
                    }
                } else {
                    info!("ctx": "confine", "op": "allow_box_syscall",
                        "msg": format!("invalid or unsupported syscall {sysname}"));
                }
            }
        }

        // Restriction 5:
        //
        // Disallow unsetting O_APPEND for append-only files.
        //
        // Note, pwritev2(2) flag RWF_NOAPPEND is denied as part
        // of a separate filter because of its arch-specific nature.
        if !skip_append {
            const F_SETFL: u64 = libc::F_SETFL as u64;
            const O_APPEND: u64 = libc::O_APPEND as u64;
            for sysname in ["fcntl", "fcntl64"] {
                if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
                    ctx.add_rule_conditional(
                        ScmpAction::Allow,
                        syscall,
                        &[scmp_cmp!($arg1 != F_SETFL)],
                    )?;
                    ctx.add_rule_conditional(
                        ScmpAction::Allow,
                        syscall,
                        &[
                            scmp_cmp!($arg1 == F_SETFL),
                            scmp_cmp!($arg2 & O_APPEND == O_APPEND),
                        ],
                    )?;
                    ctx.add_rule_conditional(
                        ScmpAction::Notify,
                        syscall,
                        &[
                            scmp_cmp!($arg1 == F_SETFL),
                            scmp_cmp!($arg2 & O_APPEND == 0),
                        ],
                    )?;
                } else {
                    info!("ctx": "confine", "op": "allow_box_syscall",
                        "msg": format!("invalid or unsupported syscall {sysname}"));
                }
            }
        } else {
            // Allow fcntl and fcntl64 system calls.
            // This was already done in init,
            // so we don't have to repeat here.
        }

        // Restriction 6: Disallow prlimit from setting resources.
        #[expect(clippy::disallowed_methods)]
        let sys_prlimit = ScmpSyscall::from_name("prlimit64").unwrap();
        if restrict_prlimit {
            // prlimit(pid_t pid, int resource,
            //         const struct rlimit *_Nullable new_limit,
            //         struct rlimit *_Nullable old_limit);
            // SAFETY: new_limit==NULL is safe.
            ctx.add_rule_conditional(ScmpAction::Allow, sys_prlimit, &[scmp_cmp!($arg2 == 0)])?;
        } else {
            #[expect(clippy::disallowed_methods)]
            let sys_setrlimit = ScmpSyscall::from_name("setrlimit").unwrap();
            ctx.add_rule(ScmpAction::Allow, sys_prlimit)?;
            ctx.add_rule(ScmpAction::Allow, sys_setrlimit)?;
        }

        // Restriction 7: Disallow CPU emulation functionality.
        if !restrict_cpu {
            for sysname in CPU_SYSCALLS {
                if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
                    ctx.add_rule(ScmpAction::Allow, syscall)?;
                } else {
                    info!("ctx": "confine", "op": "allow_box_syscall",
                        "msg": format!("invalid or unsupported syscall {sysname}"));
                }
            }
        }

        // Restriction 8: Disallow Kernel keyring access.
        if !restrict_keyring {
            for sysname in KEYRING_SYSCALLS {
                if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
                    ctx.add_rule(ScmpAction::Allow, syscall)?;
                } else {
                    info!("ctx": "confine", "op": "allow_box_syscall",
                        "msg": format!("invalid or unsupported syscall {sysname}"));
                }
            }
        }

        // Restriction 9: Disallow adjusting system time.
        if !restrict_time {
            for sysname in TIME_SYSCALLS {
                if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
                    ctx.add_rule(ScmpAction::Allow, syscall)?;
                } else {
                    info!("ctx": "confine", "op": "allow_box_syscall",
                        "msg": format!("invalid or unsupported syscall {sysname}"));
                }
            }
        }

        // Restriction 10: Disallow io_uring interface.
        if !restrict_iouring {
            for sysname in IOURING_SYSCALLS {
                if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
                    ctx.add_rule(ScmpAction::Allow, syscall)?;
                } else {
                    info!("ctx": "confine", "op": "allow_box_syscall",
                        "msg": format!("invalid or unsupported syscall {sysname}"));
                }
            }
        }

        // Restriction 11: Disallow page cache system calls.
        if !restrict_page_cache {
            for sysname in PAGE_CACHE_SYSCALLS {
                if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
                    ctx.add_rule(ScmpAction::Allow, syscall)?;
                } else {
                    info!("ctx": "confine", "op": "allow_box_syscall",
                        "msg": format!("invalid or unsupported syscall {sysname}"));
                }
            }
        }

        // Restriction 12: Disallow deprecated system calls.
        if !restrict_deprecated {
            for sysname in DEPRECATED_SYSCALLS {
                if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
                    ctx.add_rule(ScmpAction::Allow, syscall)?;
                } else {
                    info!("ctx": "confine", "op": "allow_box_syscall",
                        "msg": format!("invalid or unsupported syscall {sysname}"));
                }
            }
        }

        // Restriction 13: Disallow mbind system call.
        if !restrict_mbind {
            if let Ok(syscall) = ScmpSyscall::from_name("mbind") {
                ctx.add_rule(ScmpAction::Allow, syscall)?;
            } else {
                info!("ctx": "confine", "op": "allow_box_syscall",
                    "msg": "invalid or unsupported syscall mbind");
            }
        }

        // Restriction 14: Disallow msgsnd system call.
        if !restrict_msgsnd {
            if let Ok(syscall) = ScmpSyscall::from_name("msgsnd") {
                ctx.add_rule(ScmpAction::Allow, syscall)?;
            } else {
                info!("ctx": "confine", "op": "allow_box_syscall",
                    "msg": "invalid or unsupported syscall msgsnd");
            }
        }

        // Restriction 15: Disallow sync(2) and syncfs(2) system calls.
        // Use trace/allow_unsafe_sync:1 to relax the restriction.
        let action = if restrict_sync {
            ScmpAction::Errno(0)
        } else {
            ScmpAction::Allow
        };
        for sysname in SYNC_SYSCALLS {
            if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
                ctx.add_rule(action, syscall)?;
            } else {
                info!("ctx": "confine", "op": "noop_box_syscall",
                    "msg": format!("invalid or unsupported syscall {sysname}"));
            }
        }

        // Restriction 16: Provide stealth for PTRACE_TRACEME operation.
        // This ptrace operation is the single one that is allowed in
        // the tracee and therefore is quite often used to detect the
        // existence of a ptracer. Here we provide a best-effort
        // mitigation against this and turn PTRACE_TRACEME into a no-op
        // that always succeeds. This way a naive approach is going to
        // fail to detect a ptracer.
        // As of version 3.19.0, we turn all ptrace operations into
        // no-ops so as to provide a best-effort mitigation against
        // using requests such as PTRACE_ATTACH or PTRACE_SEIZE to
        // detect a ptracer.
        // As of version 3.25.2, we log ptrace(2) calls in case we're
        // allowing them to help with malware analysis.
        let action = if restrict_ptrace {
            ScmpAction::Errno(0)
        } else {
            ScmpAction::Allow
        };
        for sysname in PTRACE_SYSCALLS {
            if !is_equal(sysname.as_bytes(), b"ptrace") && action != ScmpAction::Allow {
                // Return ENOSYS for all ptrace system calls but ptrace(2).
                // No need to add a rule, default action will do.
                continue;
            }
            if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
                ctx.add_rule(action, syscall)?;
            } else {
                info!("ctx": "confine", "op": "noop_box_syscall",
                    "msg": format!("invalid or unsupported syscall {sysname}"));
            }
        }

        // Restriction 17: Disallow perf.
        if !restrict_perf {
            for sysname in PERF_SYSCALLS {
                if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
                    ctx.add_rule(ScmpAction::Allow, syscall)?;
                } else {
                    info!("ctx": "confine", "op": "allow_box_syscall",
                        "msg": format!("invalid or unsupported syscall {sysname}"));
                }
            }
        }

        // Restriction 18: Disallow memory protection keys.
        if !restrict_pkey {
            for sysname in PKEY_SYSCALLS {
                if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
                    ctx.add_rule(ScmpAction::Allow, syscall)?;
                } else {
                    info!("ctx": "confine", "op": "allow_box_syscall",
                        "msg": format!("invalid or unsupported syscall {sysname}"));
                }
            }
        }

        // Restriction 19: Disallow mount family.
        if !restrict_mount {
            for sysname in MOUNT_SYSCALLS {
                if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
                    ctx.add_rule(ScmpAction::Allow, syscall)?;
                } else {
                    info!("ctx": "confine", "op": "allow_box_syscall",
                        "msg": format!("invalid or unsupported syscall {sysname}"));
                }
            }
        }

        // Restriction 20: Disallow UTS family.
        if !restrict_uts {
            for sysname in UTS_SYSCALLS {
                if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
                    ctx.add_rule(ScmpAction::Allow, syscall)?;
                } else {
                    info!("ctx": "confine", "op": "allow_box_syscall",
                        "msg": format!("invalid or unsupported syscall {sysname}"));
                }
            }
        }

        // Restriction 21: Disallow nice.
        if !restrict_nice {
            for sysname in NICE_SYSCALLS {
                if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
                    ctx.add_rule(ScmpAction::Allow, syscall)?;
                } else {
                    info!("ctx": "confine", "op": "allow_box_syscall",
                        "msg": format!("invalid or unsupported syscall {sysname}"));
                }
            }
        }

        // Restriction 22: Disallow rseq.
        if !restrict_rseq {
            let sysname = "rseq";
            if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
                ctx.add_rule(ScmpAction::Allow, syscall)?;
            } else {
                info!("ctx": "confine", "op": "allow_box_syscall",
                    "msg": format!("invalid or unsupported syscall {sysname}"));
            }
        }

        // Restriction 23: Disallow unsafe personality(2) personas.
        let sysname = "personality";
        if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
            if restrict_personality {
                #[expect(clippy::cast_sign_loss)]
                for persona in UNSAFE_PERSONAS {
                    let persona = persona.bits() as u64;
                    ctx.add_rule_conditional(
                        ScmpAction::KillProcess,
                        syscall,
                        &[scmp_cmp!($arg0 == persona)],
                    )?;
                }
                for &(_, persona) in SAFE_PERSONAS {
                    ctx.add_rule_conditional(
                        ScmpAction::Allow,
                        syscall,
                        &[scmp_cmp!($arg0 == persona)],
                    )?;
                }
            } else {
                ctx.add_rule(ScmpAction::Allow, syscall)?;
            }
        } else {
            info!("ctx": "confine", "op": "allow_box_syscall",
                "msg": format!("invalid or unsupported syscall {sysname}"));
        }

        // Restriction 24: Disallow O_NOTIFICATION_PIPE for pipe2(2).
        let sysname = "pipe2";
        #[expect(clippy::cast_sign_loss)]
        if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
            if restrict_pipe {
                // O_NOTIFICATION_PIPE is equivalent to O_EXCL,
                // see: linux/watch_queue.h
                const O_NOTIFICATION_PIPE: u64 = OFlag::O_EXCL.bits() as u64;

                ctx.add_rule_conditional(
                    ScmpAction::Allow,
                    syscall,
                    &[scmp_cmp!($arg1 & O_NOTIFICATION_PIPE == 0)],
                )?;

                // pipe2(2) returns ENOPKG when CONFIG_WATCH_QUEUE
                // is not enabled in the host Linux kernel.
                ctx.add_rule_conditional(
                    ScmpAction::Errno(Errno::ENOPKG as i32),
                    syscall,
                    &[scmp_cmp!($arg1 & O_NOTIFICATION_PIPE == O_NOTIFICATION_PIPE)],
                )?;
            } else {
                ctx.add_rule(ScmpAction::Allow, syscall)?;
            }
        } else {
            info!("ctx": "confine", "op": "allow_box_syscall",
                "msg": format!("invalid or unsupported syscall {sysname}"));
        }

        // Export seccomp rules if requested.
        if let Some(mode) = ExportMode::from_env() {
            self.seccomp_export(&ctx, mode)?;
        }

        // Precompute seccomp rules which ensures:
        // 1. We fail early before spawning sandbox process on errors.
        // 2. We reduce number of memory {de,}allocations that are
        //    going to happen in the sandbox process after loading
        //    the seccomp filter but before passing the notification
        //    file descriptor back to Syd. This issue can become apparent
        //    when memory sandboxing is enabled whereby a memory
        //    {de,}allocation request can deadlock this process.
        // Note, we precompute after exporting the seccomp filter to
        // ease debugging in case of potential errors during
        // precomputation.
        #[cfg(libseccomp_v2_6)]
        ctx.precompute()?;

        Ok(ctx)
    }

    /// Set up seccomp for the Syd process which will be inherited by
    /// the child. this is important to restrict potential attack space
    /// in case Syd process is compromised somehow.
    #[expect(clippy::cognitive_complexity)]
    fn load_seccomp_parent(sandbox: &Sandbox) -> SydResult<()> {
        let mut ctx = ScmpFilterContext::new(ScmpAction::Allow)?;
        // Enforce the NO_NEW_PRIVS functionality before
        // loading the seccomp filter into the kernel.
        ctx.set_ctl_nnp(true)?;
        // Disable Speculative Store Bypass mitigations
        // with trace/allow_unsafe_spec_exec:1
        let ssb = sandbox.flags.allow_unsafe_spec_exec();
        ctx.set_ctl_ssb(ssb)?;
        // Synchronize filter to all threads.
        ctx.set_ctl_tsync(true)?;
        // We deny with ENOSYS for bad/unsupported system call, and kill process for bad arch.
        ctx.set_act_badarch(ScmpAction::KillProcess)?;
        // Use a binary tree sorted by syscall number if possible.
        let _ = ctx.set_ctl_optimize(2);
        // We don't want ECANCELED, we want actual errnos.
        let _ = ctx.set_api_sysrawrc(true);

        // Add supported architectures.
        seccomp_add_architectures(&mut ctx)?;

        // Determine restrictions based on sandbox flags.
        let restrict_exec = !sandbox.flags.allow_unsafe_exec();
        let restrict_kcapi = !sandbox.flags.allow_safe_kcapi();
        let restrict_memory = !sandbox.flags.allow_unsafe_memory();
        let restrict_socket = !sandbox.flags.allow_unsupp_socket();
        let restrict_mkbdev = !sandbox.flags.allow_unsafe_mkbdev();
        let restrict_mkcdev = !sandbox.flags.allow_unsafe_mkcdev();

        // Restriction 1:
        // (a) Prevent execve where arg0==NULL||arg1==NULL||arg2==NULL
        // (b) Prevent execveat where arg1==NULL||arg2==NULL||arg3==NULL
        // On Linux, argv and envp can be specified as NULL. In
        // both cases, this has the same effect as specifying the
        // argument as a pointer to a list containing a single null
        // pointer. Do not take advantage of this nonstandard and
        // nonportable misfeature! On many other UNIX systems,
        // specifying argv as NULL will result in an error (EFAULT).
        // Some other UNIX systems treat the envp==NULL case the same as
        // Linux.
        // SAFETY: We kill the process rather than deny with EFAULT
        // because this call is most certainly malicious and this gives
        // the system administrator a notification via dmesg(1) about
        // the potentially malicious activity.
        //
        // This mitigation can be disabled with trace/allow_unsafe_exec:1.
        if restrict_exec {
            #[expect(clippy::disallowed_methods)]
            let sys_execve = ScmpSyscall::from_name("execve").unwrap();
            #[expect(clippy::disallowed_methods)]
            let sys_execveat = ScmpSyscall::from_name("execveat").unwrap();
            ctx.add_rule_conditional(
                ScmpAction::KillProcess,
                sys_execve,
                &[scmp_cmp!($arg0 == 0)],
            )?;
            ctx.add_rule_conditional(
                ScmpAction::KillProcess,
                sys_execve,
                &[scmp_cmp!($arg1 == 0)],
            )?;
            ctx.add_rule_conditional(
                ScmpAction::KillProcess,
                sys_execve,
                &[scmp_cmp!($arg2 == 0)],
            )?;
            ctx.add_rule_conditional(
                ScmpAction::KillProcess,
                sys_execveat,
                &[scmp_cmp!($arg1 == 0)],
            )?;
            ctx.add_rule_conditional(
                ScmpAction::KillProcess,
                sys_execveat,
                &[scmp_cmp!($arg2 == 0)],
            )?;
            ctx.add_rule_conditional(
                ScmpAction::KillProcess,
                sys_execveat,
                &[scmp_cmp!($arg3 == 0)],
            )?;
        }

        // Restriction 2: Prevent mmap(addr<${mmap_min_addr}, MAP_FIXED).
        // Arguably this does not give us much however ensuring mmap_min_addr
        // is constant after the start of the sandbox with zero-cost can't be bad.
        // In addition we kill the process directly rather than denying the call
        // like mmap_min_addr does, thereby giving the system administrator higher
        // chance to notice potentially malicious activity.
        if restrict_memory {
            const MAP_FIXED: u64 = libc::MAP_FIXED as u64;
            const MAP_FIXED_NOREPLACE: u64 = libc::MAP_FIXED_NOREPLACE as u64;
            for sysname in ["mmap", "mmap2"] {
                #[expect(clippy::disallowed_methods)]
                let syscall = ScmpSyscall::from_name(sysname).unwrap();
                ctx.add_rule_conditional(
                    ScmpAction::KillProcess,
                    syscall,
                    &[
                        scmp_cmp!($arg0 < *MMAP_MIN_ADDR),
                        scmp_cmp!($arg3 & MAP_FIXED == MAP_FIXED),
                    ],
                )?;
                ctx.add_rule_conditional(
                    ScmpAction::KillProcess,
                    syscall,
                    &[
                        scmp_cmp!($arg0 < *MMAP_MIN_ADDR),
                        scmp_cmp!($arg3 & MAP_FIXED_NOREPLACE == MAP_FIXED_NOREPLACE),
                    ],
                )?;
            }
        }

        // Restriction 3: Prohibit attempts to create memory mappings
        // that are writable and executable at the same time, or to
        // change existing memory mappings to become executable, or
        // mapping shared memory segments as executable.
        // Note, mips requires executable stack so we skip on this arch.
        #[cfg(not(any(
            target_arch = "mips",
            target_arch = "mips32r6",
            target_arch = "mips64",
            target_arch = "mips64r6"
        )))]
        if restrict_memory {
            const W: u64 = libc::PROT_WRITE as u64;
            const X: u64 = libc::PROT_EXEC as u64;
            const WX: u64 = W | X;
            const MAP_A: u64 = libc::MAP_ANONYMOUS as u64;
            const MAP_S: u64 = libc::MAP_SHARED as u64;
            for sysname in ["mmap", "mmap2"] {
                // Prevent writable and executable memory.
                #[expect(clippy::disallowed_methods)]
                let syscall = ScmpSyscall::from_name(sysname).unwrap();
                ctx.add_rule_conditional(
                    ScmpAction::KillProcess,
                    syscall,
                    &[scmp_cmp!($arg2 & WX == WX)],
                )?;

                // Prevent executable anonymous memory.
                ctx.add_rule_conditional(
                    ScmpAction::KillProcess,
                    syscall,
                    &[scmp_cmp!($arg2 & X == X), scmp_cmp!($arg3 & MAP_A == MAP_A)],
                )?;

                // Prevent executable shared memory.
                ctx.add_rule_conditional(
                    ScmpAction::KillProcess,
                    syscall,
                    &[scmp_cmp!($arg2 & X == X), scmp_cmp!($arg3 & MAP_S == MAP_S)],
                )?;
            }

            for sysname in ["mprotect", "pkey_mprotect"] {
                #[expect(clippy::disallowed_methods)]
                let syscall = ScmpSyscall::from_name(sysname).unwrap();
                ctx.add_rule_conditional(
                    ScmpAction::KillProcess,
                    syscall,
                    &[scmp_cmp!($arg2 & X == X)],
                )?;
            }
        }

        // Restriction 4: Deny creation of block and character devices.
        // Terminate the process on match because legit use cases inside
        // the sandbox are rare. For those cases we provide the options
        // trace/allow_unsafe_mkbdev:1 and trace/allow_unsafe_mkcdev:1.
        if restrict_mkbdev {
            scmp_add_mknod(&mut ctx, ScmpAction::KillProcess, FileType::Blk)?;
        }
        if restrict_mkcdev {
            scmp_add_mknod(&mut ctx, ScmpAction::KillProcess, FileType::Chr)?;
        }

        // Restriction 5: Deny unsafe set-id system calls.
        // Deny with Errno=0 -> Turn the system calls into no-op.
        // This is for compatibility, e.g. postgres invokes
        // setgroups before setuid and aborts on failure.
        for sysname in UNSAFE_ID_SYSCALLS {
            if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
                ctx.add_rule(ScmpAction::Errno(0), syscall)?;
            } else {
                info!("ctx": "confine", "op": "noop_syscall",
                    "msg": format!("invalid or unsupported syscall {sysname}"));
            }
        }

        // Restriction 6: Deny transition to privileged {U,G}IDs.
        // Step 1: arg0 for UIDs.
        for sysname in &[
            "setuid",
            "setuid32",
            "setreuid",
            "setreuid32",
            "setresuid",
            "setresuid32",
        ] {
            if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
                ctx.add_rule_conditional(
                    ScmpAction::Errno(libc::EACCES),
                    syscall,
                    &[scmp_cmp!($arg0 <= u64::from(UID_MIN.as_raw()))],
                )?;
            } else {
                info!("ctx": "confine", "op": "deny_syscall",
                    "msg": format!("invalid or unsupported syscall {sysname}"));
            }
        }

        // Step 2: arg0 for GIDs.
        for sysname in &[
            "setgid",
            "setgid32",
            "setregid",
            "setregid32",
            "setresgid",
            "setresgid32",
        ] {
            if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
                ctx.add_rule_conditional(
                    ScmpAction::Errno(libc::EACCES),
                    syscall,
                    &[scmp_cmp!($arg0 <= u64::from(GID_MIN.as_raw()))],
                )?;
            } else {
                info!("ctx": "confine", "op": "deny_syscall",
                    "msg": format!("invalid or unsupported syscall {sysname}"));
            }
        }

        // Step 3: arg1 for UIDs.
        for sysname in &["setreuid", "setreuid32", "setresuid", "setresuid32"] {
            if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
                ctx.add_rule_conditional(
                    ScmpAction::Errno(libc::EACCES),
                    syscall,
                    &[scmp_cmp!($arg1 <= u64::from(UID_MIN.as_raw()))],
                )?;
            } else {
                info!("ctx": "confine", "op": "deny_syscall",
                    "msg": format!("invalid or unsupported syscall {sysname}"));
            }
        }

        // Step 4: arg1 for GIDs.
        for sysname in &["setregid", "setregid32", "setresgid", "setresgid32"] {
            if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
                ctx.add_rule_conditional(
                    ScmpAction::Errno(libc::EACCES),
                    syscall,
                    &[scmp_cmp!($arg1 <= u64::from(GID_MIN.as_raw()))],
                )?;
            } else {
                info!("ctx": "confine", "op": "deny_syscall",
                    "msg": format!("invalid or unsupported syscall {sysname}"));
            }
        }

        // Step 5: arg2 for UIDS.
        for sysname in &["setresuid", "setresuid32"] {
            if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
                ctx.add_rule_conditional(
                    ScmpAction::Errno(libc::EACCES),
                    syscall,
                    &[scmp_cmp!($arg2 <= u64::from(UID_MIN.as_raw()))],
                )?;
            } else {
                info!("ctx": "confine", "op": "deny_syscall",
                    "msg": format!("invalid or unsupported syscall {sysname}"));
            }
        }

        // Step 6: arg2 for GIDs.
        for sysname in &["setresgid", "setresgid32"] {
            if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
                ctx.add_rule_conditional(
                    ScmpAction::Errno(libc::EACCES),
                    syscall,
                    &[scmp_cmp!($arg2 <= u64::from(GID_MIN.as_raw()))],
                )?;
            } else {
                info!("ctx": "confine", "op": "deny_syscall",
                    "msg": format!("invalid or unsupported syscall {sysname}"));
            }
        }

        // Restriction 7: Restrict sub namespace creation.
        let deny_namespaces = sandbox.flags.denied_namespaces();
        let (namespace_act, namespace_acl) = if deny_namespaces == 0 {
            (Action::Allow, None)
        } else {
            let sysname_ns = "setns";
            #[expect(clippy::disallowed_methods)]
            let syscall_ns = ScmpSyscall::from_name(sysname_ns).unwrap();
            let deny_all = deny_namespaces & NAMESPACE_FLAGS_ALL == NAMESPACE_FLAGS_ALL;
            let namespace_act = if deny_all {
                // If every single kind of namespace shall be
                // prohibited, then let's block the whole setns()
                // syscall altogether.
                ctx.add_rule(ScmpAction::Errno(libc::EACCES), syscall_ns)?;
                Action::Deny
            } else {
                // Otherwise, block only the invocations with the
                // appropriate flags in the loop below, but also the
                // special invocation with a zero flags argument, right
                // here.
                ctx.add_rule_conditional(
                    ScmpAction::Errno(libc::EACCES),
                    syscall_ns,
                    &[scmp_cmp!($arg1 == 0)],
                )?;
                Action::Filter
            };

            let sysname_cl = "clone";
            let sysname_un = "unshare";
            #[expect(clippy::disallowed_methods)]
            let syscall_cl = ScmpSyscall::from_name(sysname_cl).unwrap();
            #[expect(clippy::disallowed_methods)]
            let syscall_un = ScmpSyscall::from_name(sysname_un).unwrap();
            let mut ns_deny = vec![];
            let mut ns_allow = vec![];
            for flag in NAMESPACE_FLAGS {
                if deny_namespaces & flag == 0 {
                    ns_allow.push(nsflag_name(*flag));
                    continue;
                } else {
                    ns_deny.push(nsflag_name(*flag));
                }

                #[expect(clippy::cast_sign_loss)]
                let flag = *flag as u64;
                ctx.add_rule_conditional(
                    ScmpAction::Errno(libc::EACCES),
                    syscall_un,
                    &[scmp_cmp!($arg0 & flag == flag)],
                )?;

                // On s390/s390x the first two parameters to clone are switched.
                if !cfg!(target_arch = "s390x") {
                    ctx.add_rule_conditional(
                        ScmpAction::Errno(libc::EACCES),
                        syscall_cl,
                        &[scmp_cmp!($arg0 & flag == flag)],
                    )?;
                } else {
                    ctx.add_rule_conditional(
                        ScmpAction::Errno(libc::EACCES),
                        syscall_cl,
                        &[scmp_cmp!($arg1 & flag == flag)],
                    )?;
                }

                if !deny_all {
                    ctx.add_rule_conditional(
                        ScmpAction::Errno(libc::EACCES),
                        syscall_ns,
                        &[scmp_cmp!($arg1 & flag == flag)],
                    )?;
                }
            }

            if namespace_act == Action::Deny {
                (namespace_act, None)
            } else {
                (namespace_act, Some((ns_deny, ns_allow)))
            }
        };

        match (namespace_act, namespace_acl) {
            (Action::Allow, _) => info!(
                "ctx": "restrict_namespaces",
                "msg": "namespace creation allowed",
                "ns_allow": NAMESPACE_NAMES),
            (Action::Deny, _) => info!(
                "ctx": "restrict_namespaces",
                "msg": "namespace creation denied",
                "ns_deny": NAMESPACE_NAMES),
            (_, Some((acl_deny, acl_allow))) => info!(
                "ctx": "restrict_namespaces",
                "msg": "namespace creation filtered",
                "ns_deny": acl_deny,
                "ns_allow": acl_allow),
            _ => unreachable!(),
        };

        // Export seccomp rules if requested.
        let export = match ExportMode::from_env() {
            Some(ExportMode::BerkeleyPacketFilter) => {
                #[expect(clippy::disallowed_methods)]
                let file = OpenOptions::new()
                    .write(true)
                    .create_new(true)
                    .mode(0o400)
                    .open("syd_parent.bpf")?;
                ctx.export_bpf(file)?;
                true
            }
            Some(ExportMode::PseudoFiltercode) => {
                // Lock stdout to prevent concurrent access.
                let mut stdout = std::io::stdout().lock();

                rwriteln!(stdout, "# Syd parent rules")?;
                rwrite!(stdout, "{}", seccomp_export_pfc(&ctx)?)?;
                true
            }
            _ => false,
        };

        // Load the seccomp filter unless:
        // a. We're running in debug mode with SYD_SKIP_SCMP=1.
        // b. We're exporting seccomp filters
        if !export && env::var_os(ENV_SKIP_SCMP).is_none() {
            ctx.load()?;
        }

        // Restriction 8: Restrict socket domains based on flags.
        //
        // SAFETY: socket may not exist on every architecture.
        // On some arches such as x86 there's the socketcall
        // system call which involves a pointer indirection
        // for the domain argument therefore on these arches
        // we rely on our socketcall seccomp-notify hook to
        // achieve the same effect.
        if !restrict_socket && !restrict_kcapi {
            return Ok(()); // No need for an additional socket filter.
        } else if seccomp_native_has_socketcall() {
            // Export seccomp rules if requested.
            if matches!(ExportMode::from_env(), Some(ExportMode::PseudoFiltercode)) {
                // Lock stdout to prevent concurrent access.
                let mut stdout = std::io::stdout().lock();

                rwriteln!(stdout, "# Syd socket rules")?;
                rwriteln!(stdout, "# Not applicable on this architecture!")?;
            }
        } else {
            let mut ctx = ScmpFilterContext::new(ScmpAction::Allow)?;
            // Enforce the NO_NEW_PRIVS functionality before
            // loading the seccomp filter into the kernel.
            ctx.set_ctl_nnp(true)?;
            // Disable Speculative Store Bypass mitigations
            // with trace/allow_unsafe_spec_exec:1
            ctx.set_ctl_ssb(ssb)?;
            // Synchronize filter to all threads.
            ctx.set_ctl_tsync(true)?;
            // SAFETY: We do allow bad architecture, see the comment above.
            ctx.set_act_badarch(ScmpAction::Allow)?;
            // Use a binary tree sorted by syscall number if possible.
            let _ = ctx.set_ctl_optimize(2);
            // SAFETY: Do not add supported architectures, this filter
            // is for the native architecture only.
            // seccomp_add_architectures(&mut ctx)?;
            // We don't want ECANCELED, we want actual errnos.
            let _ = ctx.set_api_sysrawrc(true);

            #[expect(clippy::disallowed_methods)]
            let syscall = ScmpSyscall::from_name("socket").unwrap();
            if restrict_socket {
                // TODO: libc:: should define this!
                const AF_MAX: libc::c_int = 45;
                // Only allow AF_{UNIX,INET,INET6,NETLINK} by default
                let mut allow_domains: SydHashSet<libc::c_int> = SydHashSet::from_iter([
                    libc::AF_UNIX,
                    libc::AF_INET,
                    libc::AF_INET6,
                    libc::AF_NETLINK,
                ]);
                if !restrict_kcapi {
                    // Allow KCAPI as well.
                    allow_domains.insert(libc::AF_ALG);
                }

                for domain in 0..AF_MAX {
                    if allow_domains.contains(&domain) {
                        continue;
                    }
                    #[expect(clippy::cast_sign_loss)]
                    ctx.add_rule_conditional(
                        ScmpAction::Errno(libc::EAFNOSUPPORT),
                        syscall,
                        &[scmp_cmp!($arg0 == domain as u64)],
                    )?;
                }

                // SAFETY: Guard against new AF_* that may be added in the future.
                ctx.add_rule_conditional(
                    ScmpAction::Errno(libc::EINVAL),
                    syscall,
                    &[scmp_cmp!($arg0 >= AF_MAX as u64)],
                )?;

                // SAFETY: Restrict AF_NETLINK families.
                //
                // Include NETLINK_SOCK_DIAG into the set by default,
                // which is required by syd::fs::peer_inode at getsockopt(2)
                // boundary.
                let mut allow_netlink_families = sandbox.netlink_families;
                allow_netlink_families.insert(NetlinkFamily::NETLINK_SOCK_DIAG);
                if allow_netlink_families.is_empty() {
                    // No netlink families were allowed, deny all of AF_NETLINK.
                    // See comment above on the usage of _exact.
                    ctx.add_rule_conditional(
                        ScmpAction::Errno(libc::EAFNOSUPPORT),
                        syscall,
                        &[scmp_cmp!($arg0 == libc::AF_NETLINK as u64)],
                    )?;
                } else {
                    let allow_netlink_families = allow_netlink_families.to_vec();
                    let netlink_family_max = NetlinkFamily::max();
                    for netlink_family in 0..netlink_family_max {
                        if allow_netlink_families.contains(&netlink_family) {
                            continue;
                        }
                        // See comment above on the usage of _exact.
                        #[expect(clippy::cast_sign_loss)]
                        ctx.add_rule_conditional(
                            ScmpAction::Errno(libc::EAFNOSUPPORT),
                            syscall,
                            &[
                                scmp_cmp!($arg0 == libc::AF_NETLINK as u64),
                                scmp_cmp!($arg2 == netlink_family as u64),
                            ],
                        )?;
                    }
                    // SAFETY: Guard against new netlink families that may be added in the future.
                    #[expect(clippy::cast_sign_loss)]
                    ctx.add_rule_conditional(
                        ScmpAction::Errno(libc::EINVAL),
                        syscall,
                        &[
                            scmp_cmp!($arg0 == libc::AF_NETLINK as u64),
                            scmp_cmp!($arg2 > netlink_family_max as u64),
                        ],
                    )?;
                }
            } else if restrict_kcapi {
                ctx.add_rule_conditional(
                    ScmpAction::Errno(libc::EAFNOSUPPORT),
                    syscall,
                    &[scmp_cmp!($arg0 == libc::AF_ALG as u64)],
                )?;
            }

            // Export seccomp rules if requested.
            let export = match ExportMode::from_env() {
                Some(ExportMode::BerkeleyPacketFilter) => {
                    #[expect(clippy::disallowed_methods)]
                    let file = OpenOptions::new()
                        .write(true)
                        .create_new(true)
                        .mode(0o400)
                        .open("syd_socket.bpf")?;
                    ctx.export_bpf(file)?;
                    true
                }
                Some(ExportMode::PseudoFiltercode) => {
                    // Lock stdout to prevent concurrent access.
                    let mut stdout = std::io::stdout().lock();

                    rwriteln!(stdout, "# Syd socket rules")?;
                    rwrite!(stdout, "{}", seccomp_export_pfc(&ctx)?)?;
                    true
                }
                _ => false,
            };

            // Load the seccomp filter unless:
            // a. We're running in debug mode with SYD_SKIP_SCMP=1.
            // b. We're exporting seccomp filters
            if !export && env::var_os(ENV_SKIP_SCMP).is_none() {
                ctx.load()?;
            }
        }

        Ok(())
    }

    /// Export a seccomp context as bpf or pfc.
    fn seccomp_export(&self, ctx: &ScmpFilterContext, mode: ExportMode) -> SydResult<()> {
        match mode {
            ExportMode::BerkeleyPacketFilter => {
                #[expect(clippy::disallowed_methods)]
                let file = OpenOptions::new()
                    .write(true)
                    .create_new(true)
                    .mode(0o400)
                    .open("syd_child.bpf")?;
                Ok(ctx.export_bpf(file)?)
            }
            ExportMode::PseudoFiltercode => {
                // Lock stdout to prevent concurrent access.
                let mut stdout = std::io::stdout().lock();

                let libver = ScmpVersion::current()?;
                rwriteln!(
                    stdout,
                    "# Syd v{} seccomp rules generated by libseccomp v{}.{}.{}",
                    env!("CARGO_PKG_VERSION"),
                    libver.major,
                    libver.minor,
                    libver.micro
                )?;
                rwriteln!(
                    stdout,
                    "# API Version: {API_MAJOR_VERSION}.{API_MINOR_VERSION}"
                )?;

                #[expect(clippy::disallowed_methods)]
                let mut syscall_allow: Vec<_> = self
                    .sysallow
                    .iter()
                    .copied()
                    .map(|sys| sys.get_name().unwrap())
                    .collect();
                let syscall_notif: SydHashSet<String> = self.handlers
                    .into_iter()
                    .map(|(key, _)| key.0.to_string()) // Extract the name from keys
                    .collect(); // Collect names into a HashSet to ensure uniqueness
                let mut syscall_notif: Vec<String> = syscall_notif.into_iter().collect();
                syscall_allow.sort();
                syscall_notif.sort();

                rwriteln!(
                    stdout,
                    "# System calls with Action=ALLOW: {}",
                    syscall_allow.len()
                )?;
                rwriteln!(
                    stdout,
                    "# System calls with Action=NOTIF: {}",
                    syscall_notif.len()
                )?;

                let uidcall = GET_ID_SYSCALLS.to_vec().join(", ");
                let sandbox = self.sandbox.read().unwrap_or_else(|err| err.into_inner());
                rwriteln!(
                    stdout,
                    "# Fake Root: {} ( {uidcall} )",
                    if sandbox.flags.fake_root() {
                        "yes"
                    } else {
                        "no"
                    }
                )?;
                rwriteln!(
                    stdout,
                    "{}",
                    sandbox
                        .to_string()
                        .lines()
                        .map(|line| format!("# {line}"))
                        .collect::<Vec<_>>()
                        .join("\n")
                )?;
                drop(sandbox);

                rwriteln!(stdout, "# Action=NOTIF: {}", syscall_notif.len())?;
                for name in &syscall_notif {
                    rwriteln!(stdout, "#    - {name}")?;
                }
                rwriteln!(stdout, "# Action=ALLOW: {}", syscall_allow.len())?;
                for name in &syscall_allow {
                    rwriteln!(stdout, "#    - {name}")?;
                }

                // Finally, export PFC.
                rwrite!(stdout, "{}", seccomp_export_pfc(ctx)?)?;

                Ok(())
            }
        }
    }

    /// Logic for the supervise child thread.
    #[expect(clippy::type_complexity)]
    fn supervise(
        mut self,
        seccomp_fd: RawFd,
        flags: Flags,
    ) -> SydResult<(
        Arc<WorkerCache<'static>>,
        Arc<RwLock<Sandbox>>,
        Option<AesLock>,
        SydJoinHandle<()>,
        Arc<AtomicBool>,
    )> {
        // Spawn the IPC thread.
        // We have already setup the socket pre-startup.
        if let Some(ipc_worker) = self.ipc_worker.take() {
            // Unwrap is fine because IPC setup is already done.
            #[expect(clippy::disallowed_methods)]
            let ipc_poll = ipc_worker
                .epoll
                .as_ref()
                .map(|fd| fd.0.as_raw_fd())
                .unwrap();
            #[expect(clippy::disallowed_methods)]
            let ipc_sock = ipc_worker.sock.as_ref().map(|fd| fd.as_raw_fd()).unwrap();

            // Set up a notification pipe and wait for
            // the IPC worker to start and unshare CLONE_F{ILE,}S.
            let (pipe_rd, pipe_wr) = pipe2(OFlag::O_CLOEXEC)?;

            // Spawn the IPC worker.
            ipc_worker.try_spawn((pipe_rd.as_raw_fd(), pipe_wr.as_raw_fd()))?;

            // Wait for startup notification.
            let mut buf = [0u8; 1];
            match retry_on_eintr(|| read(&pipe_rd, &mut buf[..]))? {
                0 => {
                    // IPC thread died before unshare.
                    // This should ideally never happen.
                    return Err(Errno::EIO.into());
                }
                1 if buf[0] == 42 => {
                    // IPC thread unshared successfully.
                    // We can go ahead and close our copies now.
                }
                _ => unreachable!("BUG: The meaning of life is not {:#x}!", buf[0]),
            }

            let _ = close(ipc_poll);
            let _ = close(ipc_sock);
        }

        // Set (process-wide) umask to 0.
        let _ = umask(Mode::empty());

        // Set main thread ID to decide in panic hook.
        log_init_main()?;

        // Set panic hook that plays well with the log-fd.
        log_set_panic_hook();

        // Spawn the syscall handler pool.
        let should_exit = Arc::new(AtomicBool::new(false));
        let syshandler_pool = pool::ThreadPool::new(
            seccomp_fd,
            flags,
            *EMU_POOL_SIZE,
            EMU_KEEP_ALIVE,
            Arc::clone(&self.sandbox),
            Arc::clone(&self.handlers),
            Arc::clone(&should_exit),
            self.crypt_map.as_ref().map(Arc::clone),
        );

        // Clone the WorkerCache instance to pass to the main thread.
        let cache = Arc::clone(&syshandler_pool.cache);

        // Boot the thread pool!
        let monitor_handle = syshandler_pool.boot()?;

        // We return a clone of the cache and the sandbox to the caller.
        // exec-TOCTOU-mitigator uses this instance in the wait loop.
        Ok((
            cache,
            Arc::clone(&self.sandbox),
            self.crypt_map.as_ref().map(Arc::clone),
            monitor_handle,
            should_exit,
        ))
    }

    /// Wait for the child process to exit.
    /// It returns the exit code of the process.
    #[expect(clippy::cognitive_complexity)]
    fn wait(
        cache: Arc<WorkerCache>,
        sandbox: Arc<RwLock<Sandbox>>,
        crypt_map: Option<AesLock>,
        monitor_handle: SydJoinHandle<()>,
        should_exit: Arc<AtomicBool>,
    ) -> SydResult<u8> {
        let my_sandbox = SandboxGuard::Read(sandbox.read().unwrap_or_else(|err| err.into_inner()));
        let child = my_sandbox.get_child_pid();
        let wait_all = my_sandbox.flags.exit_wait_all();
        let safe_setuid = my_sandbox.flags.allow_safe_setuid();
        let safe_setgid = my_sandbox.flags.allow_safe_setgid();
        let safe_setid = safe_setuid || safe_setgid;
        let ssb = my_sandbox.flags.allow_unsafe_spec_exec();
        let restrict_sys = !my_sandbox.flags.allow_unsafe_nocookie();
        let transit_uids = my_sandbox.transit_uids.clone();
        let transit_gids = my_sandbox.transit_gids.clone();
        drop(my_sandbox); // release the read lock.

        // SAFETY: Confine the main thread.
        // Honour dry-run when exporting.
        let dry_run = if env::var_os(ENV_SKIP_SCMP).is_some() || ExportMode::from_env().is_some() {
            error!("ctx": "confine", "op": "confine_main_thread",
                "msg": "main thread is running unconfined in debug mode");
            true
        } else {
            false
        };

        let mut ctx = ScmpFilterContext::new(ScmpAction::KillProcess)?;

        // Enforce the NO_NEW_PRIVS functionality before
        // loading the seccomp filter into the kernel.
        ctx.set_ctl_nnp(true)?;

        // Disable Speculative Store Bypass mitigations
        // with trace/allow_unsafe_spec_exec:1
        ctx.set_ctl_ssb(ssb)?;

        // DO NOT synchronize filter to all threads.
        // Thread pool confines itself as necessary.
        ctx.set_ctl_tsync(false)?;

        // We kill for bad system call and bad arch.
        ctx.set_act_badarch(ScmpAction::KillProcess)?;

        // Use a binary tree sorted by syscall number if possible.
        let _ = ctx.set_ctl_optimize(2);

        // SAFETY: Do NOT add supported architectures to the filter.
        // This ensures Syd can never run a non-native system call,
        // which we do not need at all.
        // seccomp_add_architectures(&mut ctx)?;

        // Deny open and {l,}stat with ENOSYS rather than KillProcess.
        // We need this because std::thread::spawn has unwanted
        // side-effects such as opening /sys/devices/system/cpu/online
        // on some architectures.
        //
        // Note, we avoid this when profiling is enabled,
        // as gperf requires it to write profiling data.
        for sysname in ["open", "stat", "lstat"] {
            match ScmpSyscall::from_name(sysname) {
                Ok(syscall) => {
                    let action = if !cfg!(feature = "prof") {
                        ScmpAction::Errno(Errno::ENOSYS as i32)
                    } else {
                        ScmpAction::Allow
                    };
                    ctx.add_rule(action, syscall)?;
                }
                Err(_) => {
                    info!("ctx": "confine", "op": "allow_main_syscall",
                        "msg": format!("invalid or unsupported syscall {sysname}"));
                }
            }
        }

        // Allow writes to the log-fd and proc_pid_mem(5) as necessary.
        confine_scmp_write(&mut ctx, None, true)?;

        // Allow safe fcntl(2) utility calls.
        for sysname in ["fcntl", "fcntl64"] {
            let syscall = match ScmpSyscall::from_name(sysname) {
                Ok(syscall) => syscall,
                Err(_) => {
                    info!("ctx": "confine", "op": "allow_main_syscall",
                        "msg": format!("invalid or unsupported syscall {sysname}"));
                    continue;
                }
            };

            for op in MAIN_FCNTL_OPS {
                ctx.add_rule_conditional(ScmpAction::Allow, syscall, &[scmp_cmp!($arg1 == *op)])?;
            }
        }

        // Allow safe prctl(2) operations.
        let sysname = "prctl";
        if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
            for (_, op) in MAIN_PRCTL_OPS {
                ctx.add_rule_conditional(ScmpAction::Allow, syscall, &[scmp_cmp!($arg0 == *op)])?;
            }
        } else {
            info!("ctx": "confine", "op": "allow_main_syscall",
                "msg": format!("invalid or unsupported syscall {sysname}"));
        }

        // Deny installing new signal handlers for {rt_,}sigaction(2).
        for sysname in ["sigaction", "rt_sigaction"] {
            let syscall = match ScmpSyscall::from_name(sysname) {
                Ok(syscall) => syscall,
                Err(_) => {
                    info!("ctx": "confine", "op": "allow_main_syscall",
                        "msg": format!("invalid or unsupported syscall {sysname}"));
                    continue;
                }
            };

            // Installs a signal handler if first argument is non-NULL.
            // We deny this case, but allow returning the current handler.
            ctx.add_rule_conditional(ScmpAction::Allow, syscall, &[scmp_cmp!($arg1 == 0)])?;
        }

        // Allow the ioctl request PAGEMAP_QUERY to lookup _proc_pid_maps_(5) efficiently.
        // This request is new in Linux-6.11.
        if *HAVE_PROCMAP_QUERY {
            let sysname = "ioctl";
            #[expect(clippy::unnecessary_cast)]
            let ioctl_request = PROCMAP_QUERY as u64;
            #[expect(clippy::useless_conversion)]
            match ScmpSyscall::from_name(sysname) {
                Ok(syscall) => {
                    let mut rules = Vec::with_capacity(if restrict_sys { 4 } else { 1 });
                    if restrict_sys {
                        rules.extend(&[
                            scmp_cmp!($arg3 == (*PROCMAP_QUERY_COOKIE_ARG3).into()),
                            scmp_cmp!($arg4 == (*PROCMAP_QUERY_COOKIE_ARG4).into()),
                            scmp_cmp!($arg5 == (*PROCMAP_QUERY_COOKIE_ARG5).into()),
                        ]);
                    }

                    rules.push(scmp_cmp!($arg1 == ioctl_request));
                    ctx.add_rule_conditional(ScmpAction::Allow, syscall, &rules)?;

                    if let Some(ioctl_request) = extend_ioctl(ioctl_request) {
                        rules.pop();
                        rules.push(scmp_cmp!($arg1 == ioctl_request));
                        ctx.add_rule_conditional(ScmpAction::Allow, syscall, &rules)?;
                    }
                }
                Err(_) => {
                    info!("ctx": "confine", "op": "allow_main_syscall",
                        "msg": format!("invalid or unsupported syscall {sysname}"));
                }
            }
        }

        // Allow safe system calls.
        //
        // Note, `PROF_SYSCALLS` is empty in case `prof` feature is disabled.
        for sysname in MAIN_SYSCALLS
            .iter()
            .chain(PROF_SYSCALLS)
            .chain(VDSO_SYSCALLS)
        {
            if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
                ctx.add_rule(ScmpAction::Allow, syscall)?;
            } else {
                info!("ctx": "confine", "op": "allow_main_syscall",
                    "msg": format!("invalid or unsupported syscall {sysname}"));
            }
        }

        let sysname = "openat2";
        if restrict_sys {
            // openat2(2) may be used only with syscall argument cookies.
            // We also prevent AT_FDCWD usage as fd argument.
            #[expect(clippy::useless_conversion)]
            match ScmpSyscall::from_name(sysname) {
                Ok(syscall) => {
                    ctx.add_rule_conditional(
                        ScmpAction::Allow,
                        syscall,
                        &[
                            scmp_cmp!($arg0 <= RawFd::MAX as u64),
                            scmp_cmp!($arg4 == (*OPENAT2_COOKIE_ARG4).into()),
                            scmp_cmp!($arg5 == (*OPENAT2_COOKIE_ARG5).into()),
                        ],
                    )?;
                }
                Err(_) => {
                    info!("ctx": "confine", "op": "allow_main_syscall",
                        "msg": format!("invalid or unsupported syscall {sysname}"));
                }
            }
        } else {
            // trace_allow_unsafe_nocookie:1
            match ScmpSyscall::from_name(sysname) {
                Ok(syscall) => {
                    ctx.add_rule(ScmpAction::Allow, syscall)?;
                }
                Err(_) => {
                    info!("ctx": "confine", "op": "allow_main_syscall",
                        "msg": format!("invalid or unsupported syscall {sysname}"));
                }
            }
        }

        // openat(2) may be used to open the parent directory only by getdir_long()
        // The rest of the attempts are denied with ENOSYS for compat.
        //
        // Note we allow openat when profiling is enabled,
        // as gperf requires it to write profiling data.
        let sysname = "openat";
        match ScmpSyscall::from_name(sysname) {
            Ok(syscall) => {
                if !cfg!(feature = "prof") {
                    let dotdot = dotdot_with_nul();
                    let oflags = (libc::O_RDONLY
                        | libc::O_CLOEXEC
                        | libc::O_DIRECTORY
                        | libc::O_LARGEFILE
                        | libc::O_NOCTTY
                        | libc::O_NOFOLLOW) as u64;
                    ctx.add_rule_conditional(
                        ScmpAction::Allow,
                        syscall,
                        &[
                            scmp_cmp!($arg0 <= RawFd::MAX as u64),
                            scmp_cmp!($arg1 == dotdot),
                            scmp_cmp!($arg2 & oflags == oflags),
                        ],
                    )?;
                    ctx.add_rule_conditional(
                        ScmpAction::Errno(Errno::ENOSYS as i32),
                        syscall,
                        &[scmp_cmp!($arg0 > RawFd::MAX as u64)],
                    )?;
                    ctx.add_rule_conditional(
                        ScmpAction::Errno(Errno::ENOSYS as i32),
                        syscall,
                        &[scmp_cmp!($arg1 != dotdot)],
                    )?;
                } else {
                    // Profiling is enabled, allow openat(2).
                    ctx.add_rule(ScmpAction::Allow, syscall)?;
                }
            }
            Err(_) => {
                info!("ctx": "confine", "op": "allow_main_syscall",
                    "msg": format!("invalid or unsupported syscall {sysname}"));
            }
        }

        // Allow futex system calls.
        for sysname in FUTEX_SYSCALLS {
            if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
                ctx.add_rule(ScmpAction::Allow, syscall)?;
            } else {
                info!("ctx": "confine", "op": "allow_main_syscall",
                    "msg": format!("invalid or unsupported syscall {sysname}"));
            }
        }

        // Allow getid system calls.
        for sysname in GET_ID_SYSCALLS {
            if let Ok(syscall) = ScmpSyscall::from_name(sysname) {
                ctx.add_rule(ScmpAction::Allow, syscall)?;
            } else {
                info!("ctx": "confine", "op": "allow_main_syscall",
                    "msg": format!("invalid or unsupported syscall {sysname}"));
            }
        }

        // Allow execveat(2) with AT_EXECVE_CHECK for Linux>=6.14.
        if *HAVE_AT_EXECVE_CHECK {
            let sysname = "execveat";
            #[expect(clippy::cast_sign_loss)]
            let atcheck = AT_EXECVE_CHECK.bits() as u64;
            match ScmpSyscall::from_name(sysname) {
                Ok(syscall) => {
                    ctx.add_rule_conditional(
                        ScmpAction::Allow,
                        syscall,
                        &[scmp_cmp!($arg4 & atcheck == atcheck)],
                    )?;
                }
                Err(_) => {
                    info!("ctx": "confine", "op": "allow_syscall",
                        "msg": format!("invalid or unsupported syscall {sysname}"));
                }
            }
        }

        // Allow UID/GID changing system calls as necessary.
        if safe_setid {
            scmp_add_setid_rules(
                "main",
                &mut ctx,
                safe_setuid,
                safe_setgid,
                &transit_uids,
                &transit_gids,
            )?;
        }

        // Export seccomp rules if requested.
        match ExportMode::from_env() {
            Some(ExportMode::BerkeleyPacketFilter) => {
                #[expect(clippy::disallowed_methods)]
                let file = OpenOptions::new()
                    .write(true)
                    .create_new(true)
                    .mode(0o400)
                    .open("syd_main.bpf")?;
                ctx.export_bpf(file)?;
            }
            Some(ExportMode::PseudoFiltercode) => {
                // Lock stdout to prevent concurrent access.
                let mut stdout = std::io::stdout().lock();

                rwriteln!(stdout, "# Syd waiter rules")?;
                rwrite!(stdout, "{}", seccomp_export_pfc(&ctx)?)?;
            }
            _ => {}
        }

        // Unshare CLONE_FS|CLONE_FILES for isolation.
        unshare(CloneFlags::CLONE_FS | CloneFlags::CLONE_FILES)?;

        // SAFETY: The main (ptrace) worker needs to inherit:
        // 1. Static file descriptors.
        // 2. Log file descriptor.
        // We have to sort the set as the FDs are randomized.
        #[expect(clippy::cast_sign_loss)]
        let mut set = vec![
            ROOT_FD() as libc::c_uint,
            PROC_FD() as libc::c_uint,
            NULL_FD() as libc::c_uint,
            crate::log::LOG_FD.load(std::sync::atomic::Ordering::Relaxed) as libc::c_uint,
        ];
        set.sort_unstable();
        closeexcept(&set)?;
        drop(set);

        // All set, load the filter!
        if !dry_run {
            ctx.load()?;

            info!("ctx": "confine", "op": "confine_main_thread",
                "msg": format!("main thread confined with{} SROP mitigation",
                    if safe_setid { "out" } else { "" }));
        }
        drop(ctx);

        let mut exit_code: i32 = 127;
        'waitloop: loop {
            match waitid(Id::All, WaitPidFlag::WEXITED | WaitPidFlag::__WNOTHREAD) {
                Ok(WaitStatus::Exited(pid, code)) => {
                    cache.del_pid(pid);
                    if pid == child {
                        exit_code = code;
                        if !wait_all {
                            break;
                        }
                    }
                }
                Ok(WaitStatus::Signaled(pid, signal, _core)) => {
                    cache.del_pid(pid);
                    if pid == child {
                        exit_code = 128_i32.saturating_add(signal);
                        if !wait_all {
                            break;
                        }
                    }
                }
                Ok(WaitStatus::PtraceEvent(
                    pid,
                    libc::SIGSTOP | libc::SIGTSTP | libc::SIGTTIN | libc::SIGTTOU,
                    libc::PTRACE_EVENT_STOP,
                )) => {
                    // SAFETY: nix does not have a wrapper for PTRACE_LISTEN,s
                    // so we fallback to libc here.
                    let _ = unsafe { libc::ptrace(libc::PTRACE_LISTEN, pid.as_raw(), 0, 0) };
                }
                Ok(WaitStatus::PtraceEvent(
                    pid,
                    _, // Can this ever be !SIGTRAP?
                    libc::PTRACE_EVENT_STOP,
                )) => {
                    // ptrace-stop, do not forward the signal.
                    let _ = ptrace::cont(pid, None);
                }
                Ok(WaitStatus::PtraceEvent(pid, sig, 0)) => {
                    // Pid received genuine signal:
                    // 1. Check if this signal has a handler.
                    // 2. If (1) is yes, increase signal count for SROP mitigation.
                    //
                    // SAFETY: Check for signal status in /proc/pid/status.
                    let status = match proc_status(pid) {
                        Ok(status) => status,
                        Err(Errno::ESRCH) => {
                            // SAFETY: Process already dead, continue.
                            continue;
                        }
                        Err(errno) => {
                            // SAFETY: Failed to get process status, terminate the process.
                            error!("ctx": "handle_signal", "op": "read_status", "err": errno as i32,
                                "msg": format!("failed to read /proc/{}/status: {errno}", pid.as_raw()),
                                "tip": "check with SYD_LOG=debug and/or submit a bug report");
                            let _ = kill(pid, Some(Signal::SIGKILL));
                            continue;
                        }
                    };

                    if status.sig_caught.contains(sig) {
                        // SAFETY:
                        // 1. Increase expected sigreturn(2) count, now that
                        //    we're forwarding a signal to the sandbox
                        //    process.
                        // 2. Signal handlers are per-process not per-thread!
                        cache.inc_sig_handle(Pid::from_raw(status.pid));
                    }

                    // SAFETY: nix Signal type does not include realtime signals,
                    // so we fallback to libc here.
                    let _ = unsafe { libc::ptrace(libc::PTRACE_CONT, pid.as_raw(), 0, sig) };
                }
                Ok(WaitStatus::PtraceEvent(pid, libc::SIGTRAP, libc::PTRACE_EVENT_SECCOMP)) => {
                    // This is ptrace syscall entry stop.
                    //
                    // SAFETY: Verify with PTRACE_GET_SYSCALL_INFO.
                    let info = match ptrace_get_syscall_info(pid) {
                        Ok(info) if info.seccomp().is_none() => {
                            // unreachable!("BUG: Invalid syscall info returned by Linux: {info:?}")
                            // trinity manages to reliably trigger this with op=0.
                            // SAFETY: Failed to get syscall info but
                            // process is still alive. Terminate!
                            let _ = kill(pid, Some(Signal::SIGKILL));
                            continue;
                        }
                        Ok(info) => info,
                        Err(Errno::ESRCH) => continue,
                        Err(_) => {
                            // SAFETY: Failed to get syscall info but
                            // process is still alive. Terminate!
                            let _ = kill(pid, Some(Signal::SIGKILL));
                            continue;
                        }
                    };

                    // Handle system call entry.
                    let result = handle_ptrace_sysenter(pid, info, &cache, &sandbox);

                    // Stop at syscall exit or continue as necessary.
                    //
                    // SAFETY: continue here is unsafe and we only use
                    // it when skipping the system call.
                    match result {
                        Ok(_) => {
                            let _ = ptrace::syscall(pid, None);
                        }
                        Err(Errno::ECANCELED) => {
                            let _ = ptrace::cont(pid, None);
                        }
                        Err(Errno::ESRCH) => {}
                        Err(_) => {
                            // SAFETY: Unexpected error at sysenter,
                            // terminate the process.
                            let _ = kill(pid, Some(Signal::SIGKILL));
                        }
                    };
                }
                Ok(WaitStatus::PtraceSyscall(pid)) => {
                    // This is ptrace syscall exit stop.
                    //
                    // SAFETY: Verify with PTRACE_GET_SYSCALL_INFO.
                    let info = match ptrace_get_syscall_info(pid) {
                        Ok(info) if info.exit().is_none() => {
                            //unreachable!("BUG: Invalid syscall info returned by Linux: {info:?}")
                            // trinity manages to reliably trigger this with op=0.
                            // SAFETY: Failed to get syscall info but
                            // process is still alive. Terminate!
                            let _ = kill(pid, Some(Signal::SIGKILL));
                            continue;
                        }
                        Ok(info) => info,
                        Err(Errno::ESRCH) => continue,
                        Err(_) => {
                            // SAFETY: Failed to get syscall info but
                            // process is still alive. Terminate!
                            let _ = kill(pid, Some(Signal::SIGKILL));
                            continue;
                        }
                    };

                    // Handle system call exit and continue if Ok.
                    match handle_ptrace_sysexit(pid, info, &cache) {
                        Ok(_) => {
                            let _ = ptrace::cont(pid, None);
                        }
                        Err(Errno::ESRCH) => {}
                        Err(_) => {
                            // SAFETY: Unexpected error during sysexit,
                            // terminate the process.
                            let _ = kill(pid, Some(Signal::SIGKILL));
                        }
                    }
                }
                Ok(WaitStatus::PtraceEvent(
                    pid,
                    libc::SIGTRAP,
                    libc::PTRACE_EVENT_CLONE | libc::PTRACE_EVENT_FORK | libc::PTRACE_EVENT_VFORK,
                )) => {
                    // Read-lock the sandbox.
                    let sandbox =
                        SandboxGuard::Read(sandbox.read().unwrap_or_else(|err| err.into_inner()));

                    // Check for PID sandboxing.
                    //
                    // sandbox/pid may be used dynamically after startup.
                    if !sandbox.enabled(Capability::CAP_PID) {
                        // PID sandboxing disabled,
                        // nothing else to do.
                        let _ = ptrace::cont(pid, None);
                        continue;
                    }

                    let pid_max = if sandbox.pid_max > 0 {
                        sandbox.pid_max
                    } else {
                        // pid/max:0 disables PID sandboxing.
                        // pid/max may be used dynamically after startup.
                        let _ = ptrace::cont(pid, None);
                        continue;
                    };
                    let pid_act = sandbox.default_action(Capability::CAP_PID);
                    drop(sandbox); // release the read lock.

                    // Check for PID limit.
                    let errno = match proc_task_limit(pid, pid_max) {
                        Ok(false) => {
                            // Limit not exceeded, continue process.
                            let _ = ptrace::cont(pid, None);
                            continue;
                        }
                        Ok(true) => None,                 // Limit exceeded.
                        Err(errno) => Some(errno as i32), // Error during limit check.
                    };

                    // Report error as necessary.
                    let pgid = getpgid(Some(pid)).map(|p| p.as_raw()).unwrap_or(0);
                    let syd_pgid = getpgrp().as_raw();
                    let kill_gid = pgid != 0 && pgid != syd_pgid;
                    if pid_act != Action::Filter {
                        let cnt_sys = proc_task_nr_sys().unwrap_or(0);
                        let cnt_syd = proc_task_nr_syd().unwrap_or(0);
                        let syd_pid = Pid::this().as_raw();
                        #[expect(clippy::cast_possible_truncation)]
                        let cpid = ptrace::getevent(pid)
                            .map(|p| Pid::from_raw(p as pid_t))
                            .unwrap_or(pid);
                        match pid_act {
                            // Allow|Deny|Filter|Panic cannot happen.
                            action if action.is_signaling() => {
                                // SAFETY: is_signaling() ensures signal() returns Some.
                                #[expect(clippy::disallowed_methods)]
                                let kill_sig = action.signal().unwrap();
                                let kill_it = if kill_gid {
                                    format!("kill process group {pgid} with {kill_sig}")
                                } else {
                                    format!("kill process {pid} with {kill_sig}")
                                };
                                error!("ctx": "limit_pid",
                                    "msg": format!("process limit {pid_max} reached, {kill_it}"),
                                    "err": errno.unwrap_or(0), "tip": "increase `pid/max'",
                                    "pid_max": pid_max, "sig": kill_sig as libc::c_int,
                                    "sys_tasks": cnt_sys,  "syd_tasks": cnt_syd,
                                    "pid": cpid.as_raw(), "ppid": pid.as_raw(), "pgid": pgid,
                                    "syd_pid": syd_pid, "syd_pgid": syd_pgid);
                            }
                            Action::Warn => {
                                warn!("ctx": "pid_limit",
                                    "msg": format!("process limit {pid_max} reached with pid {pid}"),
                                    "err": errno.unwrap_or(0), "tip": "increase `pid/max'",
                                    "sys_tasks": cnt_sys,  "syd_tasks": cnt_syd,
                                    "pid": cpid.as_raw(), "ppid": pid.as_raw(), "pgid": pgid,
                                    "syd_pid": syd_pid, "syd_pgid": syd_pgid);
                            }
                            Action::Exit => {
                                let act = pid_act.to_string().to_ascii_lowercase();
                                error!("ctx": "limit_pid",
                                    "msg": format!("process limit {pid_max} reached with pid {cpid}, {act}ing!"),
                                    "err": errno.unwrap_or(0), "tip": "increase `pid/max'",
                                    "sys_tasks": cnt_sys,  "syd_tasks": cnt_syd,
                                    "pid": cpid.as_raw(), "ppid": pid.as_raw(), "pgid": pgid,
                                    "syd_pid": syd_pid, "syd_pgid": syd_pgid);
                            }
                            _ => unreachable!(),
                        };
                    }

                    let kill_sig = match pid_act {
                        // Allow|Deny|Panic cannot happen.
                        action if action.is_signaling() => action.signal(),
                        Action::Filter => Some(Signal::SIGKILL),
                        Action::Warn => None,
                        Action::Exit => std::process::exit(errno.unwrap_or(libc::EACCES)),
                        _ => unreachable!(),
                    };

                    // SAFETY: Send signal to the process group,
                    // unless process shares their process group
                    // with the current process.
                    if let Some(kill_sig) = kill_sig {
                        if kill_gid {
                            let _ = killpg(Pid::from_raw(pgid), Some(kill_sig));
                        } else {
                            let _ = kill(pid, Some(kill_sig));
                        }
                    } else {
                        let _ = ptrace::cont(pid, None);
                    }
                }
                Ok(WaitStatus::PtraceEvent(pid, libc::SIGTRAP, libc::PTRACE_EVENT_EXEC)) => {
                    // This is ptrace syscall exec stop.
                    //
                    // An important caveat is the TGID may have switched.

                    // Retrieve the exec record from the cache.
                    let (exe, file, ip, sp, args, ip_mem, sp_mem, memmap) = if let Some((
                        _,
                        result,
                    )) =
                        cache.get_exec(pid)
                    {
                        (
                            result.exe,
                            result.file,
                            result.ip,
                            result.sp,
                            result.args,
                            result.ip_mem,
                            result.sp_mem,
                            result.memmap,
                        )
                    } else {
                        // Note the pid may have been switched to the thread group ID,
                        // so we need to call getevent to get the actual thread ID.
                        #[expect(clippy::cast_possible_truncation)]
                        match ptrace::getevent(pid).map(|tid| Pid::from_raw(tid as i32)) {
                            Ok(tid) if pid != tid => {
                                if let Some((_, result)) = cache.get_exec(tid) {
                                    (
                                        result.exe,
                                        result.file,
                                        result.ip,
                                        result.sp,
                                        result.args,
                                        result.ip_mem,
                                        result.sp_mem,
                                        result.memmap,
                                    )
                                } else {
                                    // SAFETY: Exec sandboxing is/was disabled.
                                    let _ = ptrace::cont(pid, None);

                                    continue;
                                }
                            }
                            Ok(_) => {
                                // SAFETY: Exec sandboxing is/was disabled.
                                let _ = ptrace::cont(pid, None);

                                continue;
                            }
                            Err(Errno::ESRCH) => continue,
                            Err(errno) => {
                                error!("ctx": "exec", "op": "getevent",
                                        "msg": format!("failed to get ptrace event message: {errno}"),
                                        "err": errno as i32,
                                        "tip": "check with SYD_LOG=debug and/or submit a bug report");
                                let _ = kill(pid, Some(Signal::SIGKILL));
                                continue;
                            }
                        }
                    };

                    // Read executable file information.
                    let (exe_inode, exe_devid_maj, exe_devid_min) = match fstatx(&file, STATX_INO) {
                        Ok(stx) => (stx.stx_ino, stx.stx_dev_major, stx.stx_dev_minor),
                        Err(errno) => {
                            // This should never happen in an ideal world,
                            // let's handle it as gracefully as we can...
                            error!("ctx": "exec", "op": "read_stat",
                                "msg": format!("failed to read exec file stats: {errno}"),
                                "err": errno as i32,
                                "tip": "check with SYD_LOG=debug and/or submit a bug report");
                            let _ = kill(pid, Some(Signal::SIGKILL));
                            continue;
                        }
                    };
                    let mut exe = exe;

                    // Read executable paths.
                    // This includes the executable, and the loader if
                    // executable is dynamically linked.
                    let bins = match proc_executables(pid) {
                        Ok(bins) => bins,
                        Err(errno) => {
                            // This should never happen in an ideal world,
                            // let's handle it as gracefully as we can...
                            error!("ctx": "exec", "op": "read_maps",
                                "msg": format!("failed to read /proc/{}/maps: {errno}", pid.as_raw()),
                                "err": errno as i32,
                                "tip": "check with SYD_LOG=debug and/or submit a bug report");
                            let _ = kill(pid, Some(Signal::SIGKILL));
                            continue;
                        }
                    };
                    let path = &bins[0].0; // Path to the executable.
                    let mut deny_action: Option<Action> = None;

                    // Determine open flags.
                    let flags = if exe == ExecutableFile::Script {
                        // We will read from the file and parse ELF.
                        OFlag::O_RDONLY | OFlag::O_NOFOLLOW | OFlag::O_NOCTTY
                    } else {
                        // ELF parsing was done at syscall entry, verify paths.
                        OFlag::O_PATH | OFlag::O_NOFOLLOW
                    };

                    // Open paths and verify the open FDs
                    // match the device ID and inode information.
                    // The FDs will be used for two things:
                    // 1. Parsing ELF to determine bitness, PIE etc.
                    // 2. Checksumming binary for Force sandboxing.
                    let mut file = Some(file);
                    let mut files = Vec::with_capacity(2);
                    for (path, inode, devid_maj, devid_min) in &bins {
                        let mut is_exe;
                        #[expect(clippy::cast_sign_loss)]
                        let result = if file.is_some() {
                            is_exe = *inode == exe_inode
                                && *devid_maj as u32 == exe_devid_maj
                                && *devid_min as u32 == exe_devid_min;
                            if is_exe && file.is_some() {
                                #[expect(clippy::disallowed_methods)]
                                Ok(file.take().unwrap().into())
                            } else {
                                is_exe = false;
                                safe_open(AT_BADFD, path, flags, ResolveFlag::empty())
                            }
                        } else {
                            is_exe = false;
                            safe_open(AT_BADFD, path, flags, ResolveFlag::empty())
                        };

                        match result {
                            Ok(fd) if is_exe => {
                                // Executable file checked out!
                                files.push(File::from(fd));
                            }
                            Ok(fd) => {
                                // WORKAROUND: Check if the FS reports sane device ids.
                                // Check the comment on has_sane_device_id() function
                                // for more information.
                                // Assume true on errors for safety.
                                let dev_check = match retry_on_eintr(|| fstatfs64(&fd)) {
                                    Ok(statfs) => !statfs.has_broken_device_ids(),
                                    Err(Errno::ENOSYS) => {
                                        // Filesystem type does not support this call.
                                        // Assume true for safety.
                                        true
                                    }
                                    Err(errno) => {
                                        error!("ctx": "open_elf",
                                            "msg": format!("statfs error: {errno}"),
                                            "err": errno as i32,
                                            "pid": pid.as_raw(), "path": path);
                                        let _ = kill(pid, Some(Signal::SIGKILL));
                                        continue 'waitloop;
                                    }
                                };
                                let statx = match fstatx(&fd, STATX_INO) {
                                    Ok(stat) => stat,
                                    Err(errno) => {
                                        error!("ctx": "open_elf",
                                            "msg": format!("statx error: {errno}"),
                                            "err": errno as i32,
                                            "pid": pid.as_raw(), "path": path);
                                        let _ = kill(pid, Some(Signal::SIGKILL));
                                        continue 'waitloop;
                                    }
                                };
                                // SAFETY: Verify we opened the same file!
                                #[expect(clippy::cast_sign_loss)]
                                let devid_maj = *devid_maj as libc::c_uint;
                                #[expect(clippy::cast_sign_loss)]
                                let devid_min = *devid_min as libc::c_uint;
                                if *inode != statx.stx_ino
                                    || (dev_check
                                        && (devid_maj != statx.stx_dev_major
                                            || devid_min != statx.stx_dev_minor))
                                {
                                    let error = format!(
                                        "metadata mismatch: {}:{}={} is not {}:{}={}",
                                        statx.stx_dev_major,
                                        statx.stx_dev_minor,
                                        statx.stx_ino,
                                        devid_maj,
                                        devid_min,
                                        inode
                                    );
                                    error!("ctx": "open_elf",
                                        "msg": error,
                                        "pid": pid.as_raw(),"path": path);
                                    let _ = kill(pid, Some(Signal::SIGKILL));
                                    continue 'waitloop;
                                }
                                files.push(File::from(fd));
                            }
                            Err(errno) => {
                                error!("ctx": "open_elf",
                                    "msg": format!("open error: {errno}"),
                                    "err": errno as i32,
                                    "pid": pid.as_raw(), "path": path);
                                let _ = kill(pid, Some(Signal::SIGKILL));
                                continue 'waitloop;
                            }
                        }
                    }
                    drop(file);

                    // Parse ELF file to figure out type,
                    // if the original file we've checked
                    // was a script.
                    let mut my_sandbox =
                        SandboxGuard::Read(sandbox.read().unwrap_or_else(|err| err.into_inner()));
                    if exe == ExecutableFile::Script {
                        // Check SegvGuard.
                        if let Some(action) = my_sandbox.check_segvguard(path) {
                            if action != Action::Filter {
                                error!("ctx": "segvguard",
                                    "msg": format!("Max crashes {} exceeded, kill process {}",
                                        my_sandbox.segvguard_maxcrashes,
                                        pid.as_raw()),
                                    "tip": "increase `segvguard/maxcrashes'",
                                    "pid": pid.as_raw(), "path": path);
                            }
                            if action == Action::Exit {
                                std::process::exit(libc::EACCES);
                            } else if action.is_signaling() {
                                deny_action = Some(action);
                            } else if action.is_denying() {
                                deny_action = Some(Action::Kill);
                            }
                        }

                        // Check for Exec sandboxing.
                        if deny_action.is_none() && my_sandbox.enabled(Capability::CAP_EXEC) {
                            for (path, _, _, _) in &bins {
                                let path = &path;
                                let (mut action, filter) =
                                    my_sandbox.check_path(Capability::CAP_EXEC, path);
                                if action == Action::Deny {
                                    // ptrace-event-exec stop:
                                    // promote deny action to kill.
                                    action = Action::Kill;
                                }
                                if !filter {
                                    warn!("ctx": "access", "cap": Capability::CAP_EXEC, "act": action,
                                        "pid": pid.as_raw(), "sys": "exec", "path": path,
                                        "tip": format!("configure `allow/exec+{path}'"));
                                }
                                match action {
                                    Action::Allow | Action::Warn => {}
                                    Action::Stop => {
                                        deny_action = Some(Action::Stop);
                                        break;
                                    }
                                    Action::Abort => {
                                        deny_action = Some(Action::Abort);
                                        break;
                                    }
                                    Action::Exit => std::process::exit(libc::EACCES),
                                    _ => {
                                        // Deny|Filter|Kill
                                        deny_action = Some(Action::Kill);
                                        break;
                                    }
                                }
                            }
                        }

                        // Check for Trusted Path Execution (TPE).
                        if deny_action.is_none() && my_sandbox.enabled(Capability::CAP_TPE) {
                            for (idx, (path, _, _, _)) in bins.iter().enumerate() {
                                let file = &files[idx];
                                let path = &path;
                                let (action, msg) = my_sandbox.check_tpe(file, path);
                                if !matches!(action, Action::Allow | Action::Filter) {
                                    let msg = msg.as_deref().unwrap_or("?");
                                    error!("ctx": "trusted_path_execution", "err": libc::EACCES,
                                        "pid": pid.as_raw(), "sys": "exec", "path": path, "act": action,
                                        "msg": format!("exec from untrusted path blocked: {msg}"),
                                        "tip": "move the binary to a safe location or use `sandbox/tpe:off'");
                                }
                                match action {
                                    Action::Allow | Action::Warn => {}
                                    Action::Stop => deny_action = Some(Action::Stop),
                                    Action::Abort => deny_action = Some(Action::Abort),
                                    Action::Exit => std::process::exit(libc::EACCES),
                                    _ => {
                                        // Deny|Filter|Kill
                                        deny_action = Some(Action::Kill);
                                    }
                                }
                            }
                        }

                        // Parse ELF as necessary for restrictions.
                        let restrict_32 = my_sandbox.flags.deny_elf32();
                        let restrict_dyn = my_sandbox.flags.deny_elf_dynamic();
                        let restrict_sta = my_sandbox.flags.deny_elf_static();
                        let restrict_ldd = !my_sandbox.flags.allow_unsafe_ldso();
                        let restrict_pie = !my_sandbox.flags.allow_unsafe_nopie();
                        let restrict_xs = !my_sandbox.flags.allow_unsafe_stack();

                        let check_linking = restrict_ldd
                            || restrict_dyn
                            || restrict_sta
                            || restrict_pie
                            || restrict_xs;

                        // Drop sandbox lock before blocking operation.
                        drop(my_sandbox);

                        // Ensure the file offset is maintained,
                        // as the file might be sharing the OFD
                        // with the sandbox process.
                        let mut file = &files[0];
                        let offset = match file.stream_position().map_err(|err| err2no(&err)) {
                            Ok(offset) => offset,
                            Err(errno) => {
                                // This should never happen in an ideal world,
                                // let's handle it as gracefully as we can...
                                error!("ctx": "exec", "op": "read_offset",
                                    "msg": format!("failed to read exec file offset: {errno}"),
                                    "err": errno as i32,
                                    "tip": "check with SYD_LOG=debug and/or submit a bug report");
                                let _ = kill(pid, Some(Signal::SIGKILL));
                                continue;
                            }
                        };

                        let result = (|| -> Result<ExecutableFile, ElfError> {
                            // Parse ELF and reset the file offset.
                            if offset != 0 {
                                file.rewind().map_err(ElfError::IoError)?;
                            }
                            let result = ExecutableFile::parse(file, check_linking);
                            file.seek(SeekFrom::Start(offset))
                                .map_err(ElfError::IoError)?;
                            result
                        })();

                        // Re-acquire the read-lock.
                        my_sandbox = SandboxGuard::Read(
                            sandbox.read().unwrap_or_else(|err| err.into_inner()),
                        );

                        match result {
                            // Update ELF information.
                            Ok(exe_bin) => exe = exe_bin,
                            Err(ElfError::IoError(err)) => {
                                deny_action = Some(Action::Kill);
                                if !my_sandbox.filter_path(Capability::CAP_EXEC, path) {
                                    error!("ctx": "parse_elf",
                                        "msg": format!("io error: {}", err2no(&err)),
                                        "err": err2no(&err) as i32,
                                        "pid": pid.as_raw(), "path": path);
                                }
                            }
                            Err(ElfError::BadMagic) => {
                                deny_action = Some(Action::Kill);
                                if !my_sandbox.filter_path(Capability::CAP_EXEC, path) {
                                    error!("ctx": "parse_elf",
                                        "msg": format!("BUG: not an ELF"),
                                        "pid": pid.as_raw(), "path": path);
                                }
                            }
                            Err(ElfError::Malformed) => {
                                deny_action = Some(Action::Kill);
                                if !my_sandbox.filter_path(Capability::CAP_EXEC, path) {
                                    error!("ctx": "parse_elf",
                                        "msg": format!("BUG: malformed ELF"),
                                        "pid": pid.as_raw(), "path": path);
                                }
                            }
                        };

                        if restrict_ldd
                            && !matches!(
                                exe,
                                ExecutableFile::Elf {
                                    file_type: ElfFileType::Executable,
                                    ..
                                }
                            )
                        {
                            deny_action = Some(Action::Kill);
                            if !my_sandbox.filter_path(Capability::CAP_EXEC, path) {
                                error!("ctx": "check_elf",
                                    "msg": "ld.so(8) exec-indirection prevented",
                                    "pid": pid.as_raw(), "path": path,
                                    "tip": "configure `trace/allow_unsafe_ldso:1'",
                                    "exe": format!("{exe}"));
                            }
                        }

                        if deny_action.is_none()
                            && restrict_pie
                            && matches!(exe, ExecutableFile::Elf { pie: false, .. })
                        {
                            deny_action = Some(Action::Kill);
                            if !my_sandbox.filter_path(Capability::CAP_EXEC, path) {
                                error!("ctx": "check_elf",
                                    "msg": "ELF is not a Position Independent Executable (PIE)",
                                    "pid": pid.as_raw(), "path": path,
                                    "tip": "configure `trace/allow_unsafe_nopie:1'",
                                    "exe": format!("{exe}"));
                            }
                        }

                        if deny_action.is_none()
                            && restrict_xs
                            && matches!(exe, ExecutableFile::Elf { xs: true, .. })
                        {
                            deny_action = Some(Action::Kill);
                            if !my_sandbox.filter_path(Capability::CAP_EXEC, path) {
                                error!("ctx": "check_elf",
                                    "msg": "ELF has Executable Stack (PT_GNU_STACK)",
                                    "pid": pid.as_raw(), "path": path,
                                    "tip": "configure `trace/allow_unsafe_stack:1'",
                                    "exe": format!("{exe}"));
                            }
                        }

                        if deny_action.is_none()
                            && restrict_32
                            && matches!(
                                exe,
                                ExecutableFile::Elf {
                                    elf_type: ElfType::Elf32,
                                    ..
                                }
                            )
                        {
                            deny_action = Some(Action::Kill);
                            if !my_sandbox.filter_path(Capability::CAP_EXEC, path) {
                                error!("ctx": "check_elf",
                                    "msg": "32-bit execution prevented",
                                    "pid": pid.as_raw(), "path": path,
                                    "tip": "configure `trace/deny_elf32:0'",
                                    "exe": format!("{exe}"));
                            }
                        }

                        if deny_action.is_none()
                            && restrict_dyn
                            && matches!(
                                exe,
                                ExecutableFile::Elf {
                                    linking_type: Some(LinkingType::Dynamic),
                                    ..
                                }
                            )
                        {
                            deny_action = Some(Action::Kill);
                            if !my_sandbox.filter_path(Capability::CAP_EXEC, path) {
                                error!("ctx": "check_elf",
                                    "msg": "dynamic-link execution prevented",
                                    "pid": pid.as_raw(), "path": path,
                                    "tip": "configure `trace/deny_elf_dynamic:0'",
                                    "exe": format!("{exe}"));
                            }
                        }

                        if deny_action.is_none()
                            && restrict_sta
                            && matches!(
                                exe,
                                ExecutableFile::Elf {
                                    linking_type: Some(LinkingType::Static),
                                    ..
                                }
                            )
                        {
                            deny_action = Some(Action::Kill);
                            if !my_sandbox.filter_path(Capability::CAP_EXEC, path) {
                                error!("ctx": "check_elf",
                                    "msg": "static-link execution prevented",
                                    "pid": pid.as_raw(), "path": path,
                                    "tip": "configure `trace/deny_elf_static:0'",
                                    "exe": format!("{exe}"));
                            }
                        }

                        // Check for Force sandboxing.
                        if deny_action.is_none() && my_sandbox.enabled(Capability::CAP_FORCE) {
                            for (idx, (path, _, _, _)) in bins.iter().enumerate() {
                                let file = &mut files[idx];
                                let result = (|mut file: &mut File, idx, offset| -> Result<Action, IntegrityError> {
                                    // Calculate checksum and reset file offset as necessary.
                                    if idx == 0 {
                                        if offset != 0 {
                                            file.rewind()
                                                .map_err(IntegrityError::from)?;
                                        }
                                        let result = my_sandbox.check_force2(path, &mut file);
                                        file.seek(SeekFrom::Start(offset))
                                            .map_err(IntegrityError::from)?;
                                        result
                                    } else {
                                        my_sandbox.check_force2(path, &mut file)
                                    }
                                })(file, idx, offset);
                                match result {
                                    Ok(Action::Allow) => {}
                                    Ok(Action::Warn) => {
                                        warn!("ctx": "verify_elf", "act": Action::Warn,
                                            "pid": pid.as_raw(), "path": path,
                                            "tip": format!("configure `force+{path}:<checksum>'"));
                                    }
                                    Ok(Action::Stop) => {
                                        deny_action = Some(Action::Stop);
                                        warn!("ctx": "verify_elf", "act": Action::Stop,
                                            "pid": pid.as_raw(), "path": path,
                                            "tip": format!("configure `force+{path}:<checksum>'"));
                                    }
                                    Ok(Action::Abort) => {
                                        deny_action = Some(Action::Abort);
                                        warn!("ctx": "verify_elf", "act": Action::Abort,
                                            "pid": pid.as_raw(), "path": path,
                                            "tip": format!("configure `force+{path}:<checksum>'"));
                                    }
                                    Ok(Action::Exit) => {
                                        error!("ctx": "verify_elf", "act": Action::Exit,
                                            "pid": pid.as_raw(), "path": path,
                                            "tip": format!("configure `force+{path}:<checksum>'"));
                                        std::process::exit(libc::EACCES);
                                    }
                                    Ok(mut action) => {
                                        // Deny|Filter|Kill
                                        deny_action = Some(Action::Kill);
                                        if action == Action::Deny {
                                            // ptrace-event-exec stop:
                                            // promote deny action to kill.
                                            action = Action::Kill;
                                        }
                                        if action != Action::Filter {
                                            warn!("ctx": "verify_elf", "act": action,
                                                "pid": pid.as_raw(), "path": path,
                                                "tip": format!("configure `force+{path}:<checksum>'"));
                                        }
                                    }
                                    Err(IntegrityError::Sys(errno)) => {
                                        deny_action = Some(Action::Kill);
                                        error!("ctx": "verify_elf",
                                            "msg": format!("system error during ELF checksum calculation: {errno}"),
                                            "err": errno as i32,
                                            "pid": pid.as_raw(), "path": path,
                                            "tip": format!("configure `force+{path}:<checksum>'"));
                                    }
                                    Err(IntegrityError::Hash {
                                        mut action,
                                        expected,
                                        found,
                                    }) => {
                                        if action == Action::Deny {
                                            // ptrace-event-exec stop:
                                            // promote deny action to kill.
                                            action = Action::Kill;
                                        }
                                        if !matches!(action, Action::Allow | Action::Filter) {
                                            error!("ctx": "verify_elf", "act": action,
                                                "msg": format!("ELF checksum mismatch: {found} is not {expected}"),
                                                "pid": pid.as_raw(), "path": path,
                                                "tip": format!("configure `force+{path}:<checksum>'"));
                                        }
                                        match action {
                                            Action::Allow | Action::Warn => {}
                                            Action::Stop => deny_action = Some(Action::Stop),
                                            Action::Abort => deny_action = Some(Action::Abort),
                                            Action::Exit => std::process::exit(libc::EACCES),
                                            _ =>
                                            /*Deny|Filter|Kill*/
                                            {
                                                deny_action = Some(Action::Kill)
                                            }
                                        };
                                    }
                                }
                            }
                        }
                    }

                    if deny_action.is_none() && !my_sandbox.flags.allow_unsafe_libc() {
                        let elf_type = match exe {
                            ExecutableFile::Elf { elf_type, .. } => elf_type,
                            _ => unreachable!(), // Script is not possible here.
                        };

                        // SAFETY:
                        // 1. Sets AT_SECURE.
                        // 2. Verifies AT_{E,}{U,G}ID matches Syd's own.
                        match proc_set_at_secure(pid, elf_type) {
                            Ok(_) | Err(Errno::ESRCH) => {}
                            Err(errno) => {
                                deny_action = Some(Action::Kill);
                                if !my_sandbox.filter_path(Capability::CAP_EXEC, path) {
                                    error!("ctx": "secure_exec",
                                        "msg": format!("error setting AT_SECURE: {errno}"),
                                        "err": errno as i32,
                                        "tip": "configure `trace/allow_unsafe_libc:1'",
                                        "pid": pid.as_raw(), "path": path);
                                }
                            }
                        }
                    }

                    // Release the read lock.
                    drop(my_sandbox);

                    if let Some(action) = deny_action {
                        let _ = kill(
                            pid,
                            Some(
                                Signal::try_from(
                                    action
                                        .signal()
                                        .map(|sig| sig as i32)
                                        .unwrap_or(libc::SIGKILL),
                                )
                                .unwrap_or(Signal::SIGKILL),
                            ),
                        );
                    } else {
                        let _ = ptrace::cont(pid, None);

                        let ip_mem = ip_mem.map(|ip_mem| HEXLOWER.encode(&ip_mem));
                        let sp_mem = sp_mem.map(|sp_mem| HEXLOWER.encode(&sp_mem));

                        debug!("ctx": "exec", "op": "verify_exec",
                            "msg": format!("execution of `{path}' of type {exe} approved"),
                            "pid": pid.as_raw(),
                            "path": &path,
                            "exe": &exe.to_string(),
                            "args": args,
                            "ip": ip,
                            "sp": sp,
                            "ip_mem": ip_mem,
                            "sp_mem": sp_mem,
                            "memmap": memmap);
                    }
                }
                Ok(WaitStatus::PtraceEvent(pid, libc::SIGTRAP, libc::PTRACE_EVENT_EXIT)) => {
                    // We stopped before return from exit(2).
                    // Apply SegvGuard.
                    let mut my_sandbox =
                        SandboxGuard::Read(sandbox.read().unwrap_or_else(|err| err.into_inner()));
                    let has_segvguard = !my_sandbox.get_segvguard_expiry().is_zero();
                    drop(my_sandbox);

                    // Setting expiry timeout to 0 disables SegvGuard.
                    if has_segvguard {
                        // Step 1:
                        // (a) Check if process produced a core dump.
                        // (b) Check if process received a signal with default action Core.
                        let sig = match ptrace::getevent(pid) {
                            Ok(status) => {
                                #[expect(clippy::cast_possible_truncation)]
                                match WaitStatus::from_raw(pid, status as i32) {
                                    WaitStatus::Signaled(_, sig, true) => Some(sig),
                                    WaitStatus::Signaled(_, sig, _) if is_coredump(sig) => {
                                        Some(sig)
                                    }
                                    _ => None, // Process did not produce a core dump, move on.
                                }
                            }
                            Err(_) => None, // Process dead? move on.
                        };

                        // Step 2: Record the crash as necessary.
                        if let Some(sig) = sig {
                            // Child received a signal that produces a
                            // coredump and SegvGuard is enabled.
                            // Add the exec path to the segvguard expiry
                            // map.
                            let mut exe = XPathBuf::from_pid(pid);
                            exe.push(b"exe");

                            let path = match readlinkat(PROC_FILE(), &exe) {
                                Ok(path) => path,
                                Err(_) => continue,
                            };

                            // Upgrade the sandbox lock to writable.
                            my_sandbox = SandboxGuard::Write(
                                sandbox.write().unwrap_or_else(|err| err.into_inner()),
                            );

                            // Record the crashing program.
                            let (was_suspended, is_suspended, num_crashes) =
                                my_sandbox.add_segvguard_crash(&path);

                            drop(my_sandbox); // release the write-lock.

                            // Convert sig to Signal for pretty printing.
                            // Note, `Signal` does not support realtime signals,
                            // therefore we log the original raw signal number
                            // as well.
                            let signal = Signal::try_from(sig).unwrap_or(Signal::SIGKILL);
                            let crashes = if num_crashes > 1 { "crashes" } else { "crash" };
                            if is_suspended {
                                error!("ctx": "segvguard",
                                    "msg": format!("suspending after {signal} due to {num_crashes} {crashes}"),
                                    "tip": "increase `segvguard/maxcrashes'",
                                    "pid": pid.as_raw(), "path": path, "sig": sig);
                            } else {
                                info!("ctx": "segvguard",
                                    "msg": format!("{num_crashes} {crashes} recorded after {signal}{}",
                                        if was_suspended { " (suspended)" } else { "" }),
                                    "pid": pid.as_raw(), "path": path, "sig": sig);
                            }
                        }
                    }

                    // Step 3: Remove PID from cache.
                    cache.del_pid(pid);

                    // Step 4: Continue the process so it exits cleanly.
                    let _ = ptrace::cont(pid, None);
                }
                Ok(status) => panic!("Unhandled wait event: {status:?}"),
                Err(Errno::EINTR | Errno::EAGAIN) => {}
                Err(Errno::ECHILD) => break,
                Err(errno) => return Err(errno.into()),
            }
        }

        // Set should_exit to true and notify the syd_aes thread.
        should_exit.store(true, std::sync::atomic::Ordering::Relaxed);
        if let Some(ref crypt_map) = crypt_map {
            let (aes_map, cvar) = &**crypt_map;
            let _aes_map = aes_map.lock().unwrap_or_else(|e| e.into_inner());
            cvar.notify_one();
        } // Lock is released here.

        // Wait for the syd-mon thread, which in turn waits for syd-aes threads,
        // for a clean, graceful, group-exit.
        monitor_handle.join().or(Err(Errno::EAGAIN))??;

        // SAFETY: Since we do not run epoll on the main thread anymore,
        // seccomp_notify_receive may block forever on seccomp fd even
        // when all processes have exited... Hence we do not join the
        // pools here and this is safe since we can be at this point
        // under two conditions:
        // (a): wait_all == false && exec child has exited.
        // (b): wait_all == true && we received ECHILD on wait().
        // Under both cases it is safe to tear down the sandbox as we
        // ensure we do not interrupt any syscall processing.
        //if wait_all {
        // let _ = thread_handle.join().map_err(|_| Errno::EPIPE)?;
        // pool_handle.join();
        //}

        let exit_code = u8::try_from(exit_code).unwrap_or(127);
        info!("ctx": "wait", "op": "exit",
            "msg": format!("return code {exit_code}, sandboxing ended!"),
            "code": exit_code);

        Ok(exit_code)
    }

    /// Run the supervisor, main entry point.
    #[expect(clippy::cognitive_complexity)]
    pub fn run(
        mut sandbox: Sandbox,
        pty_child: Option<OwnedFd>,
        argv0: &OsStr,
        argv: Vec<OsString>,
        envp: Option<&SydHashSet<OsString>>,
        arg0: Option<OsString>,
    ) -> SydResult<u8> {
        let (major, minor) = *KERNEL_VERSION;
        if major < 5 {
            error!("ctx": "run", "op": "check_kernel_version",
                "msg": "Your kernel version is too old.");
            return Err(Errno::ENOSYS.into());
        } else if major == 5 && minor < 5 {
            error!("ctx": "run", "op": "check_kernel_version",
                "msg": "Your kernel version is too old: Does not support SECCOMP_USER_NOTIF_FLAG_CONTINUE, ...");
            return Err(Errno::ENOSYS.into());
        } else if major == 5 && minor < 6 {
            error!("ctx": "run", "op": "check_kernel_version",
                "msg": "Your kernel version is too old: Does not support pidfd_getfd(2) and SECCOMP_IOCTL_NOTIF_ADDFD.");
            return Err(Errno::ENOSYS.into());
        } else if major == 5 && minor < 9 {
            error!("ctx": "run", "op": "check_kernel_version",
                "msg": "Your kernel version is too old: Does not support SECCOMP_IOCTL_NOTIF_ADDFD.");
            return Err(Errno::ENOSYS.into());
        } else if major == 5 && minor < 19 {
            error!("ctx": "run", "op": "check_kernel_version",
                "msg": "Your kernel version is too old: Does not support SECCOMP_FILTER_FLAG_WAIT_KILLABLE_RECV.");
            return Err(Errno::ENOSYS.into());
        }
        // Crypt sandboxing requires Linux>=6.2.
        if sandbox.enabled(Capability::CAP_CRYPT) && (major < 6 || (major == 6 && minor < 2)) {
            error!("ctx": "run", "op": "check_kernel_version",
                "msg": "Your kernel version is too old: Does not support ALG_SET_KEY_BY_KEY_SERIAL.",
                "tip": "turn Crypt sandboxing off with `sandbox/crypt:off'");
            return Err(Errno::ENOSYS.into());
        }
        info!("ctx": "run", "op": "check_kernel_version",
            "msg": "kernel version is compatible",
            "major": major, "minor": minor,
            "pidfd_thread": *HAVE_PIDFD_THREAD,
            "seccomp_sync": *HAVE_SECCOMP_USER_NOTIF_FD_SYNC_WAKE_UP,
            "mountid_uniq": *HAVE_STATX_MNT_ID_UNIQUE);

        // Restore parent-death signal as necessary.
        let pds_old = get_pdeathsig().unwrap_or(None);
        let pds_new = if let Some(pds_new) = env::var_os(ENV_PDS) {
            let pds = if let Ok(pds_new) = btoi::<i32>(pds_new.as_bytes()) {
                Signal::try_from(pds_new).ok()
            } else if let Ok(pds_new) = std::str::from_utf8(pds_new.as_bytes()) {
                Signal::from_str(pds_new).ok()
            } else {
                None
            };

            if pds != pds_old {
                Some(pds)
            } else {
                None
            }
        } else {
            None
        };

        let pds_old = pds_old.map(|sig| sig as i32).unwrap_or(0);
        if let Some(pds_new) = pds_new {
            set_pdeathsig(pds_new)?;

            let pds_new = pds_new.map(|sig| sig as i32).unwrap_or(0);
            info!("ctx": "run", "op": "set_parent_death_signal",
                "msg": format!("set parent-death signal from {pds_old} to {pds_new}"),
                "sig": pds_new,
                "old": pds_old);
        } else {
            info!("ctx": "run", "op": "get_parent_death_signal",
                "msg": format!("parent-death signal is {pds_old}"),
                "sig": pds_old);
        }

        // SAFETY:
        // 1. Set trace/allow_safe_set{u,g}id to true,
        //    if at least one {U,G}ID transit was defined.
        // 2. Return ENODATA if trace/allow_safe_set{u,g}id
        //    was set to true but no {U,G}ID transit was defined.
        // 3. Return ENODATA if trace/allow_safe_set{u,g}id
        //    was set to true but no {U,G}ID transit was defined
        //    for the current user/group.
        if !sandbox.transit_uids.is_empty() {
            sandbox.flags.insert(Flags::FL_ALLOW_SAFE_SETUID);
        } else if sandbox.flags.allow_safe_setuid() {
            return Err(Errno::ENODATA.into());
        }
        if !sandbox.transit_gids.is_empty() {
            sandbox.flags.insert(Flags::FL_ALLOW_SAFE_SETGID);
        } else if sandbox.flags.allow_safe_setgid() {
            return Err(Errno::ENODATA.into());
        }
        let uid = Uid::current();
        if sandbox.flags.allow_safe_setuid()
            && !sandbox.transit_uids.iter().any(|(src, _)| *src == uid)
        {
            return Err(Errno::ENODATA.into());
        }
        let gid = Gid::current();
        if sandbox.flags.allow_safe_setgid()
            && !sandbox.transit_gids.iter().any(|(src, _)| *src == gid)
        {
            return Err(Errno::ENODATA.into());
        }

        // SAFETY: Set crypt sandboxing on if:
        // a. Encryption key was supplied.
        // b. Crypt sandboxing list was populated.
        //
        // This avoids the surprising case where a user
        // supplies a key/acl but forgets to turn Crypt
        // sandboxing on in which case the files marked
        // for encryption would silently be written
        // unencrypted.
        if sandbox.must_crypt() {
            sandbox.state.insert(Capability::CAP_CRYPT);
        }

        // If Crypt Sandboxing is on.
        // 1. Validate session keyring is attached to the user keyring.
        // 2. Set allow_safe_kcapi and exit_wait_all flags.
        if sandbox.enabled(Capability::CAP_CRYPT) {
            if let Err(errno) = key_ring_validate() {
                error!("ctx": "run", "op": "check_keyrings", "err": errno as i32,
                    "msg": format!("session keyring isn't attached to the user keyring: {errno}"),
                    "tip": "reconnect or start a fresh login session");
                return Err(errno.into());
            }
            sandbox.flags.insert(Flags::FL_ALLOW_SAFE_KCAPI);
            sandbox.flags.insert(Flags::FL_EXIT_WAIT_ALL);
        }

        // Check MMAP_MIN_ADDR.
        // We must initialize this LazyLock before sandbox is locked.
        let mmap_min_addr = *MMAP_MIN_ADDR;
        info!("ctx": "run", "op": "check_mmap_min_addr", "limit": mmap_min_addr);

        // Set process and i/o prorities.
        // See the "Process Priority and Resource Management" section of the syd(7) manual page.
        let restrict_nice = !sandbox.flags.allow_unsafe_nice();
        if restrict_nice {
            // SAFETY: Step 1: Set thread priority to a low value.
            match Errno::result(unsafe { libc::setpriority(libc::PRIO_PROCESS, 0, 20) }) {
                Ok(_) => {
                    info!("ctx": "run", "op": "set_program_scheduling_priority",
                        "msg": "set program scheduling priority to 20",
                        "val": 20);
                }
                Err(errno @ Errno::ENOSYS) => {
                    info!("ctx": "run", "op": "set_program_scheduling_priority",
                        "msg": format!("setpriority error: {errno}"),
                        "err": errno as i32);
                }
                Err(errno) => {
                    error!("ctx": "run", "op": "set_program_scheduling_priority",
                        "msg": format!("setpriority error: {errno}"),
                        "err": errno as i32);
                }
            }

            // SAFETY: Step 2: Set CPU scheduling priority to idle.
            match set_cpu_priority_idle() {
                Ok(_) => {
                    info!("ctx": "run", "op": "set_cpu_scheduling_priority",
                        "msg": "set CPU scheduling priority to idle",
                        "val": "idle");
                }
                Err(errno @ Errno::ENOSYS) => {
                    info!("ctx": "run", "op": "set_cpu_scheduling_priority",
                        "msg": format!("sched_setscheduler error: {errno}"),
                        "err": errno as i32);
                }
                Err(errno) => {
                    error!("ctx": "run", "op": "set_cpu_scheduling_priority",
                        "msg": format!("sched_setscheduler error: {errno}"),
                        "err": errno as i32);
                }
            }

            // SAFETY: Step 3: Set I/O priority to idle.
            match set_io_priority_idle() {
                Ok(_) => {
                    info!("ctx": "run", "op": "set_io_scheduling_priority",
                        "msg": "set i/o scheduling priority to idle",
                        "val": "idle");
                }
                Err(errno @ Errno::ENOSYS) => {
                    info!("ctx": "run", "op": "set_io_scheduling_priority",
                        "msg": format!("ioprio_set error: {errno}"),
                        "err": errno as i32);
                }
                Err(errno) => {
                    error!("ctx": "run", "op": "set_io_scheduling_priority",
                        "msg": format!("ioprio_set error: {errno}"),
                        "err": errno as i32);
                }
            }
        }

        // Adjust process resources to limit core dumps.
        let restrict_prlimit = !sandbox.flags.allow_unsafe_prlimit();
        if restrict_prlimit {
            match setrlimit(Resource::RLIMIT_CORE, 0, 0) {
                Ok(_) => {
                    info!("ctx": "run", "op": "set_rlimit_core",
                        "msg": "coredump generation disabled");
                }
                Err(errno) => {
                    error!("ctx": "run", "op": "set_rlimit_core",
                        "msg": format!("setrlimit error: {errno}"),
                        "err": errno as i32);
                }
            };
        }

        // Create an IPC worker if requested.
        // We do it late here to ensure syd_ipc shares namespaces,
        // however we also do it before bind mounts and Landlock
        // to give privileged access to syd_ipc which will confine
        // itself.
        env::remove_var(ENV_IPC_POLL_FD);
        env::remove_var(ENV_IPC_UNIX_FD);
        let ipc_worker = if sandbox.locked() {
            // Sandbox is locked, no need to spawn IPC.
            None
        } else if let Some(ref ipc_path) = sandbox.ipc {
            if sandbox.lock.is_none() {
                // ipc command implies lock:ipc if lock was not set explicitly.
                sandbox.lock = Some(LockState::Ipc);
            }
            let mut ipc = IpcWorker::new(
                ipc_path,
                sandbox.ipc_uid,
                sandbox.ipc_gid,
                *sandbox.flags,
                &sandbox.transit_uids,
                &sandbox.transit_gids,
            );
            ipc.setup()?;
            Some(ipc)
        } else {
            None
        };

        // Prepare the command to execute.
        // We create it early here so dynamic library loading
        // works even if we mount the owning fs noexec later.
        let mut command = match crate::unshare::Command::new(argv0) {
            Ok(command) => command,
            Err(errno) => return Ok(u8::try_from(errno as i32).unwrap_or(127)),
        };
        command.deny_tsc(sandbox.flags.deny_tsc());
        if sandbox.flags.allow_unsafe_caps() || sandbox.flags.allow_unsafe_ptrace() {
            // Keep CAP_SYS_PTRACE in the sandbox process.
            command.keep(true);
        }
        if !sandbox.flags.allow_unsafe_ptrace() {
            // Exec TOCTOU mitigation.
            command.stop(true);
        }
        if sandbox.flags.allow_unsafe_spec_exec() {
            // Disable Speculative Store Bypass mitigations
            // for seccomp(2) filters with trace/allow_unsafe_spec_exec:1
            command.ssb(true);
        }
        if !sandbox.skip_append() {
            // Enable append-only mitigations.
            command.append_only(true);
        }
        command.args(&argv);
        if let Some(ref arg0) = arg0 {
            command.arg0(arg0);
        }

        // Set mount propagation on the root filesystem for mount namespace.
        // Note, we can only mount /proc after the initial clone as we're now pid=1.
        //
        // SAFETY: Avoid following symbolic links using openat2(2).
        // O_NOFOLLOW|RESOLVE_SYMLINKS|RESOLVE_MAGICLINKS|, and more.
        //
        // SAFETY: Mount using `/proc/self/fd` indirection
        // to prevent symbolic link TOCTTOU during mount(2).
        let how = safe_open_how(OFlag::O_PATH, ResolveFlag::empty())
            // Drop RESOLVE_BENEATH which we cannot use here.
            .resolve(ResolveFlag::RESOLVE_NO_MAGICLINKS | ResolveFlag::RESOLVE_NO_SYMLINKS);
        let how_dir = how.flags(OFlag::O_PATH | OFlag::O_DIRECTORY);
        let how_new = how
            .flags(OFlag::O_RDONLY | OFlag::O_CREAT | OFlag::O_EXCL)
            .mode(Mode::S_IRUSR);
        let root_fd = if sandbox.flags.unshare_mount() {
            // Set root mount propagation to private recursively.
            mount(
                Some("none"),
                "/",
                NONE,
                MsFlags::MS_PRIVATE | MsFlags::MS_REC,
                NONE,
            )?;

            let (root_fd, root_is_tmpfs) = if let Some(ref root) = sandbox.root {
                let root_is_tmpfs = root.is_equal(b"tmpfs");
                let root = if root_is_tmpfs {
                    // Create a temporary, private root directory.
                    Cow::Owned(mkdtemp("/tmp/syd.XXXXXX").map(XPathBuf::from)?)
                } else {
                    // Use specified new-root directory.
                    Cow::Borrowed(root)
                };

                #[expect(clippy::disallowed_methods)]
                let mut fd = retry_on_eintr(|| openat2(AT_BADFD, root.as_ref(), how_dir))?;

                let mut p = XPathBuf::from("/proc/self/fd");
                p.push_fd(fd.as_raw_fd());
                #[expect(clippy::disallowed_methods)]
                if root_is_tmpfs {
                    // Mount tmpfs over root filesystem with default options.
                    // This also ensures rootfs is a mountpoint which is required
                    // by pivot_root(2).
                    mount(Some("tmpfs"), &p, Some("tmpfs"), MsFlags::MS_NOSUID, NONE)?;

                    // Reopen rootfd after rootfs mount.
                    drop(fd);
                    fd = retry_on_eintr(|| openat2(AT_BADFD, root.as_ref(), how_dir))?;

                    // Ensure safe CWD.
                    // This is important because we may recursively
                    // create directories later on in this directory.
                    fchdir(&fd)?;

                    // Create /dev and /proc directories.
                    mkdirat(&fd, "dev", Mode::S_IRWXU)?;
                    mkdirat(&fd, "proc", Mode::S_IRWXU)?;
                } else {
                    // Make new root directory a mountpoint with a self-bind.
                    // This is required by pivot_root(2).
                    mount(Some(&p), &p, NONE, MsFlags::MS_BIND | MsFlags::MS_REC, NONE)?;

                    // Reopen rootfd after rootfs mount.
                    drop(fd);
                    fd = retry_on_eintr(|| openat2(AT_BADFD, root.as_ref(), how_dir))?;

                    // Mount /dev tmpfs with default options.
                    // Do it early here so the user can populate it.
                    #[expect(clippy::disallowed_methods)]
                    let fd = retry_on_eintr(|| openat2(&fd, "dev", how_dir))?;

                    p.pop(); // /proc/self/fd
                    p.push_fd(fd.as_raw_fd());

                    mount(
                        Some("tmpfs"),
                        &p,
                        Some("tmpfs"),
                        MsFlags::MS_NOSUID | MsFlags::MS_NOEXEC,
                        NONE,
                    )?;
                }

                (Some(fd), root_is_tmpfs)
            } else {
                (None, false)
            };

            // Process bind mounts as necessary.
            if let Some(bind_mounts) = sandbox.collect_bind_mounts() {
                const NONE: Option<&XPathBuf> = None;

                for bind in bind_mounts {
                    // SAFETY: Ensure root cannot be mounted over.
                    if root_fd.is_some() && bind.dst.is_rootfs() {
                        let errno = Errno::EINVAL;
                        error!("ctx": "run", "op": "spec_mount", "mnt": &bind, "err": errno as i32,
                            "msg": format!("mount over rootfs is not permitted: {errno}"));
                        return Err(errno.into());
                    }

                    if bind.src.is_relative() {
                        // Note `bind.dst` is always an absolute path.
                        let dst_fd = if let Some(ref root_fd) = root_fd {
                            // SAFETY: Ensure no consecutive slashes exist.
                            let mut dst = bind.dst.clone();
                            dst.clean_consecutive_slashes();
                            dst.remove(0); // Turn into relative path.

                            if root_is_tmpfs && !dst.is_empty() {
                                // Create directories recursively under temporary root.
                                // SAFETY:
                                // 1. `dst' is relative.
                                // 2. `dst' has no `..` components.
                                // 3. Current working directory points to root FD.
                                create_dir_all(&dst)?;
                            }

                            #[expect(clippy::disallowed_methods)]
                            retry_on_eintr(|| openat2(root_fd, &dst, how_dir))
                        } else {
                            #[expect(clippy::disallowed_methods)]
                            retry_on_eintr(|| openat2(AT_BADFD, &bind.dst, how_dir))
                        }?;

                        let mut p = XPathBuf::from("/proc/self/fd");
                        p.push_fd(dst_fd.as_raw_fd());
                        match mount(
                            Some(&bind.src),
                            &p,
                            Some(&bind.src),
                            bind.opt,
                            bind.dat.as_ref(),
                        ) {
                            Ok(_) => {
                                info!("ctx": "run", "op": "spec_mount", "mnt": &bind,
                                    "msg": format!("special-fs mount `{bind}' succeeded"));
                            }
                            Err(errno) => {
                                error!("ctx": "run", "op": "spec_mount", "mnt": &bind, "err": errno as i32,
                                    "msg": format!("special-fs mount `{bind}' failed: {errno}"));
                                return Err(errno.into());
                            }
                        }
                    } else {
                        #[expect(clippy::disallowed_methods)]
                        let src_fd = retry_on_eintr(|| openat2(AT_BADFD, &bind.src, how))?;
                        let is_dir = file_type(&src_fd, None, false)?.is_dir();

                        // Note `bind.dst` is always an absolute path.
                        let dst_fd = if let Some(ref root_fd) = root_fd {
                            // SAFETY: Ensure no consecutive slashes exist.
                            let mut dst = bind.dst.clone();
                            dst.clean_consecutive_slashes();
                            dst.remove(0); // Turn into relative path.
                            let empty = dst.is_empty();

                            if root_is_tmpfs && !empty {
                                // Create parent directories recursively under temporary root.
                                let parent = dst.parent();
                                if *dst != *parent {
                                    // `dst' has a parent.
                                    // SAFETY:
                                    // 1. `parent' is relative.
                                    // 2. `parent' has no `..` components.
                                    // 3. Current working directory points to root FD.
                                    create_dir_all(parent)?;
                                }

                                // Create file or directory under temporary root.
                                if is_dir {
                                    mkdirat(root_fd, &dst, Mode::S_IRWXU)?;
                                } else {
                                    mknodat(root_fd, &dst, SFlag::S_IFREG, Mode::S_IRUSR, 0)?;
                                }
                            }

                            // Ensure we open current directory for `/`.
                            if dst.is_empty() {
                                dst.append_byte(b'.');
                            }

                            #[expect(clippy::disallowed_methods)]
                            retry_on_eintr(|| {
                                openat2(root_fd, &dst, if is_dir { how_dir } else { how })
                            })
                        } else {
                            #[expect(clippy::disallowed_methods)]
                            retry_on_eintr(|| {
                                openat2(AT_BADFD, &bind.dst, if is_dir { how_dir } else { how })
                            })
                        }?;

                        let mut src_p = XPathBuf::from("/proc/self/fd");
                        let mut dst_p = XPathBuf::from("/proc/self/fd");
                        src_p.push_fd(src_fd.as_raw_fd());
                        dst_p.push_fd(dst_fd.as_raw_fd());

                        let flags = bind.opt | MsFlags::MS_BIND | MsFlags::MS_REC;
                        match mount(Some(&src_p), &dst_p, NONE, flags, NONE) {
                            Ok(_) => {
                                info!("ctx": "run", "op": "bind_mount", "mnt": &bind,
                                    "msg": format!("bind mount `{bind}' succeeded"));
                            }
                            Err(errno) => {
                                error!("ctx": "run", "op": "bind_mount", "mnt": &bind, "err": errno as i32,
                                    "msg": format!("bind mount `{bind}' failed: {errno}"));
                                return Err(errno.into());
                            }
                        }
                    }
                }
            }

            root_fd
        } else {
            None
        };

        if sandbox.flags.unshare_mount() {
            // Unshare/Mount implies Unshare/PID.
            //
            // Set namespaced kernel.pid_max sysctl:
            // 1. The kernel reserves 300 pids via RESERVED_PIDS in kernel/pid.c
            // 2. Kernel adapts reserved pids based on number of CPUs on the system.
            // 3. We cannot do this after private procfs mount due to subset=pid.
            if *HAVE_NAMESPACED_PID_MAX
                && sandbox.enabled(Capability::CAP_PID)
                && sandbox.pid_max > 0
            {
                const PIDS_PER_CPU_MIN: u64 = 8;
                const RESERVED_PIDS: u64 = 301;
                let pid_max = sandbox
                    .pid_max
                    .max(RESERVED_PIDS)
                    .max(PIDS_PER_CPU_MIN * (*NPROC as u64));
                proc_pid_max(pid_max)?;
                info!("ctx": "run", "op": "set_pid_max", "max": pid_max,
                    "msg": format!("set namespaced kernel.pid_max sysctl to {pid_max}"));
            }

            // Mount private procfs.
            //
            // The target directory may be under the chroot directory.
            // Use hidepid=2 to hide pid=1.
            // As of version 3.37.2 we use hidepid=4 which is Linux>=5.8.
            // As of version 3.39.0 we use subset=pid which is Linux>=5.8.
            //
            // SAFETY: Private procfs is mounted _after_ custom bind mounts
            // to ensure they cannot interfere with this mount.
            #[expect(clippy::disallowed_methods)]
            let proc_fd = if let Some(ref root_fd) = root_fd {
                retry_on_eintr(|| openat2(root_fd, "proc", how_dir))
            } else {
                retry_on_eintr(|| openat2(AT_BADFD, "/proc", how_dir))
            }?;

            let mut p = XPathBuf::from("/proc/self/fd");
            p.push_fd(proc_fd.as_raw_fd());

            let flags = MsFlags::MS_NOSUID | MsFlags::MS_NOEXEC | MsFlags::MS_NODEV;
            let mut opts = XPathBuf::from("hidepid=4");
            if !sandbox.flags.allow_unsafe_proc_files() {
                opts.append_bytes(b",subset=pid");
            }
            mount(Some("proc"), &p, Some("proc"), flags, Some(&opts))?;
            if log_enabled!(LogLevel::Info) {
                let bind = BindMount {
                    src: XPathBuf::from("proc"),
                    dst: p,
                    opt: flags,
                    dat: Some(opts),
                };
                info!("ctx": "run", "op": "mount_procfs", "mnt": &bind,
                    "msg": format!("proc mount `{bind}' succeeded"));
            }

            if let Some(ref root_fd) = root_fd {
                // Provide /dev/null which is required by Syd.
                #[expect(clippy::disallowed_methods)]
                let src_fd = retry_on_eintr(|| openat2(AT_BADFD, "/dev/null", how))?;
                #[expect(clippy::disallowed_methods)]
                let dst_fd = retry_on_eintr(|| openat2(root_fd, "dev/null", how_new))?;

                let mut src_p = XPathBuf::from("/proc/self/fd");
                let mut dst_p = XPathBuf::from("/proc/self/fd");
                src_p.push_fd(src_fd.as_raw_fd());
                dst_p.push_fd(dst_fd.as_raw_fd());

                mount(
                    Some(&src_p),
                    &dst_p,
                    NONE,
                    MsFlags::MS_BIND | MsFlags::MS_REC,
                    NONE,
                )?;

                // Provide symbolic links for standard file descriptors.
                //
                // Note, these symbolic links are user-owned so the sandbox
                // process may remove them. Since these symbolic links are
                // only for convenience, we do not check for errors.
                //
                // /dev/stdin
                let _ = symlinkat("/proc/self/fd/0", root_fd, "dev/stdin");
                // /dev/stdout
                let _ = symlinkat("/proc/self/fd/1", root_fd, "dev/stdout");
                // /dev/stderr
                let _ = symlinkat("/proc/self/fd/2", root_fd, "dev/stderr");

                // All set, change root directory.
                // Move old mount over itself.
                fchdir(root_fd)?;
                pivot_root(".", ".")?;

                // Unmount old root directory.
                umount2(".", MntFlags::MNT_DETACH)?;

                // Ensure CWD equals root.
                chdir("/")?;
            }
        }

        // Clean up the environment as necessary.
        if !sandbox.flags.allow_unsafe_env() {
            for &var in UNSAFE_ENV {
                let var = OsStr::from_bytes(var);
                if !envp.map(|envp| envp.contains(var)).unwrap_or(false)
                    && env::var_os(var).is_some()
                {
                    env::remove_var(var);
                    if !log_enabled!(LogLevel::Info) {
                        continue;
                    }
                    let var = XPathBuf::from(var.to_os_string());
                    info!("ctx": "run", "op": "sanitize_process_environment",
                        "msg": format!("removed unsafe variable {var} from environment"),
                        "tip": format!("use `syd -e{var}=' to pass-through an unsafe environment variable"));
                }
            }
        }

        // SAFETY: Ensure randomized timer is initialized as necessary.
        let restrict_sysinfo = !sandbox.flags.allow_unsafe_sysinfo();
        if restrict_sysinfo {
            timer_init()?;
        }

        // SAFETY: Ensure the static file descriptors are open
        // before sandboxing starts but after the mounts are
        // processed. This comes with two limitations:
        // (1) `proc_init` is called before Landlock sandbox is applied.
        // (2) `proc_init` is called before sandbox process startup.
        //
        // (1) is necessary because otherwise we'd require
        //     ReadDir access to `/`, aka rootfs, from Landlock
        //     thereby rendering the ReadDir category effectively
        //     useless.
        // (2) Syd is included into the same Landlock sandbox
        //     as the sandbox process therefore to accomplish (1)
        //     we must open static file descriptors exactly here.
        //
        // The rationale behind pre-opening rootfs is to prevent
        // root escapes utilizing RESOLVE_BENEATH within the path
        // canonicalizer. See syd::fs::safe_canonicalize for more
        // information.
        if let Err(errno) = proc_init() {
            error!("ctx": "run", "op": "opendir",
                "msg": format!("failed to open /proc at startup: {errno}"),
                "err": errno as i32);
            return Err(errno.into());
        };

        // Set up the Landlock sandbox if requested. Note,
        // we set it up here before spawning the child so as to
        // include the Syd sandbox threads into the sandbox as
        // well. This is done for added security.
        // Note, Landlock errors are not fatal without default/lock:kill.
        if let Some(ref policy) = sandbox.collect_landlock() {
            let abi = crate::landlock::ABI::new_current();
            let lvl = policy.compat_level.unwrap_or(CompatLevel::BestEffort);
            let compat_level = match lvl {
                CompatLevel::HardRequirement => "hard-requirement",
                CompatLevel::SoftRequirement => "soft-requirement",
                CompatLevel::BestEffort => "best-effort",
            };
            match policy.restrict_self(abi) {
                Ok(status) => {
                    let ruleset_stat = match status.ruleset {
                        RulesetStatus::FullyEnforced => "fully enforced",
                        RulesetStatus::PartiallyEnforced => "partially enforced",
                        RulesetStatus::NotEnforced => "not enforced",
                    };

                    match (status.ruleset, lvl) {
                        // The FullyEnforced case must be tested by the developer.
                        (RulesetStatus::FullyEnforced, _) => {
                            info!("ctx": "run", "op": "apply_landlock",
                                "msg": format!("Landlock ABI {} is {ruleset_stat}", abi as i32),
                                "abi": abi as i32,
                                "cmp": compat_level);
                            // fall-through.
                        }
                        // Users should be warned that they are not protected.
                        (_, CompatLevel::BestEffort) => {
                            warn!("ctx": "run", "op": "apply_landlock",
                                "msg": format!("Landlock ABI {} is {ruleset_stat}", abi as i32),
                                "abi": abi as i32,
                                "cmp": compat_level);
                            // fall-through.
                        }
                        _ => {
                            error!("ctx": "run", "op": "apply_landlock",
                                "msg": format!("Landlock ABI {} is {ruleset_stat}", abi as i32),
                                "abi": abi as i32,
                                "cmp": compat_level);
                            return Err(Errno::EOPNOTSUPP.into());
                        }
                    };
                }
                Err(error) if lvl == CompatLevel::BestEffort => {
                    warn!("ctx": "run", "op": "apply_landlock",
                        "msg": format!("Landlock ABI {} is unsupported", abi as i32),
                        "err": &error.to_string(),
                        "abi": abi as i32, "cmp": compat_level);
                    // fall-through.
                }
                Err(error) => {
                    error!("ctx": "run", "op": "apply_landlock",
                        "msg": format!("Landlock ABI {} is unsupported", abi as i32),
                        "err": &error.to_string(),
                        "abi": abi as i32, "cmp": compat_level);
                    return Err(Errno::ENOSYS.into());
                }
            }
        }

        // Initialize sandbox supervisor.
        let supervisor = Supervisor::new(sandbox, pty_child, ipc_worker)?;

        // Start profiling if requested.
        #[cfg(feature = "prof")]
        if let Some(val) = env::var_os("SYD_PROF") {
            match val.as_bytes() {
                b"cpu" => crate::start_cpu_profile("main"),
                b"mem" => crate::start_mem_profile("main"),
                _ => {}
            }
        };

        // Spawn the program under sandbox.
        let log = if log_enabled!(LogLevel::Info) {
            let cmd = arg0
                .map(XPathBuf::from)
                .unwrap_or_else(|| XPathBuf::from(argv0.to_os_string()));
            let args = argv.into_iter().map(XPathBuf::from).collect::<Vec<_>>();
            Some((cmd, args))
        } else {
            None
        };
        let (cache, sandbox, crypt_map, crypt_handle, should_exit) = match supervisor.spawn(command)
        {
            Ok(result) => {
                if let Some((cmd, args)) = log {
                    info!("ctx": "run", "op": "run_command",
                        "msg": format!("spawned `{cmd}' with arguments {args:?}"),
                        "cmd": cmd, "argv": args);
                }
                result
            }
            Err(error) => {
                let errno = Errno::last();
                if let Some((cmd, args)) = log {
                    info!("ctx": "run", "op": "run_command",
                        "msg": format!("spawn error executing `{cmd}': {errno}"),
                        "err": errno as i32, "cmd": cmd, "argv": args);
                }
                errno.set();
                return Err(error);
            }
        };

        // Wait for the process to exit and return the same error code.
        let result = Supervisor::wait(cache, sandbox, crypt_map, crypt_handle, should_exit)
            .inspect_err(|error| {
                Errno::set_raw(error.errno().map(|e| e as i32).unwrap_or(libc::ENOSYS));
            });

        // End profiling if requested.
        #[cfg(feature = "prof")]
        if let Some(val) = env::var_os("SYD_PROF") {
            match val.as_bytes() {
                b"cpu" => crate::stop_cpu_profile(),
                b"mem" => {
                    crate::dump_mem_profile("main");
                    crate::stop_mem_profile();
                }
                _ => {}
            }
        }

        // Finally return the result to the caller.
        if ExportMode::from_env().is_some() {
            // We're exporting, not sandboxing: return 0.
            Ok(0)
        } else {
            result
        }
    }
}

/*
 * System call handlers
 */

#[expect(clippy::cognitive_complexity)]
fn handle_ptrace_sysenter(
    pid: Pid,
    info: ptrace_syscall_info,
    cache: &Arc<WorkerCache>,
    sandbox: &Arc<RwLock<Sandbox>>,
) -> Result<(), Errno> {
    #[expect(clippy::disallowed_methods)]
    let info_scmp = info.seccomp().unwrap();

    #[expect(clippy::cast_possible_truncation)]
    let scmp_trace_data = info_scmp.ret_data as u16;

    match scmp_trace_data {
        PTRACE_DATA_CHDIR => {
            // Acquire a read lock to the sandbox.
            let my_sandbox =
                SandboxGuard::Read(sandbox.read().unwrap_or_else(|err| err.into_inner()));

            if !my_sandbox.enabled(Capability::CAP_CHDIR) {
                // SAFETY: Chdir sandboxing is not enabled,
                // continue the system call without any
                // checking.
                return Err(Errno::ECANCELED);
            }

            let result = crate::kernel::chdir::sysenter_chdir(pid, &my_sandbox, info_scmp);

            drop(my_sandbox); // release the read lock.

            let path = match result {
                Ok(path) => path,
                Err(errno) => {
                    // Set system call to -1 to skip the system call.
                    // Write error value into the return register.
                    return if let Err(errno) = ptrace_skip_syscall(pid, info.arch, Some(errno)) {
                        // SAFETY: Failed to set return value, terminate the process.
                        if errno != Errno::ESRCH {
                            let _ = kill(pid, Some(Signal::SIGKILL));
                        }
                        Err(Errno::ESRCH)
                    } else if cfg!(any(
                        target_arch = "mips",
                        target_arch = "mips32r6",
                        target_arch = "mips64",
                        target_arch = "mips64r6",
                        target_arch = "s390x"
                    )) {
                        // Skip to syscall-stop to write return value.
                        cache.add_error(pid, Some(errno));
                        Ok(())
                    } else {
                        // Continue process.
                        Err(Errno::ECANCELED)
                    };
                }
            };

            // Record the chdir result.
            cache.add_chdir(pid, path);

            // Stop at syscall exit.
            Ok(())
        }
        PTRACE_DATA_EXECVE | PTRACE_DATA_EXECVEAT => {
            // Acquire a read lock to the sandbox.
            let my_sandbox =
                SandboxGuard::Read(sandbox.read().unwrap_or_else(|err| err.into_inner()));

            // Call the system call handler, and record the result.
            let result = crate::kernel::exec::sysenter_exec(pid, &my_sandbox, info);

            drop(my_sandbox); // release the read lock.

            let (file, exe) = match result {
                Ok((file, exe)) => (file, exe),
                Err(errno) => {
                    // AT_EXECVE_CHECK success is indicated by ECANCELED.
                    // See sysenter_exec.
                    let errno = if errno == Errno::ECANCELED {
                        None
                    } else {
                        Some(errno)
                    };
                    // Set system call to -1 to skip the system call.
                    // Write error value into the return register.
                    return if let Err(errno) = ptrace_skip_syscall(pid, info.arch, errno) {
                        // SAFETY: Failed to set return value, terminate the process.
                        if errno != Errno::ESRCH {
                            error!("ctx": "skip_syscall",
                                "msg": format!("skip exec syscall error: {errno}"),
                                "err": errno as i32,
                                "tip": "check with SYD_LOG=debug and/or submit a bug report");
                            let _ = kill(pid, Some(Signal::SIGKILL));
                        }
                        Err(Errno::ESRCH)
                    } else if cfg!(any(
                        target_arch = "mips",
                        target_arch = "mips32r6",
                        target_arch = "mips64",
                        target_arch = "mips64r6",
                        target_arch = "s390x"
                    )) {
                        // Skip to syscall-stop to write return value.
                        cache.add_error(pid, errno);
                        Ok(())
                    } else {
                        // Continue process.
                        Err(Errno::ECANCELED)
                    };
                }
            };

            // Read memory pointed by IP and SP for logging.
            let ip = info.instruction_pointer;
            let sp = (info.stack_pointer & !0xF).saturating_sub(16);

            let process = RemoteProcess::new(pid);
            let ip_mem = if ip >= *MMAP_MIN_ADDR {
                let mut ip_mem = [0u8; 64];
                // SAFETY: This is a ptrace hook, the PID cannot be validated.
                match unsafe { process.read_mem(&mut ip_mem, ip) } {
                    Ok(_) => Some(ip_mem),
                    Err(Errno::EFAULT) => None,
                    Err(Errno::ESRCH) => return Err(Errno::ESRCH),
                    Err(_) => {
                        // SAFETY: Process is alive, but
                        // we cannot read memory: Terminate!
                        let _ = kill(pid, Some(Signal::SIGKILL));
                        return Err(Errno::ESRCH);
                    }
                }
            } else {
                None
            };

            let sp_mem = if sp >= *MMAP_MIN_ADDR {
                let mut sp_mem = [0u8; 64];
                // SAFETY: This is a ptrace hook, the PID cannot be validated.
                match unsafe { process.read_mem(&mut sp_mem, sp) } {
                    Ok(_) => Some(sp_mem),
                    Err(Errno::EFAULT) => None,
                    Err(Errno::ESRCH) => return Err(Errno::ESRCH),
                    Err(_) => {
                        // SAFETY: Process is alive, but
                        // we cannot read memory: Terminate!
                        let _ = kill(process.pid, Some(Signal::SIGKILL));
                        return Err(Errno::ESRCH);
                    }
                }
            } else {
                None
            };

            // Save `/proc/$pid/maps`.
            let memmap = proc_maps(pid).ok();

            // Record the exec result.
            //
            // SAFETY: Terminate the process on errors.
            cache.add_exec(
                pid,
                exe,
                file,
                ip,
                sp,
                info_scmp.args,
                ip_mem,
                sp_mem,
                memmap,
            );

            // Continue process, it will stop at EVENT_EXEC.
            Err(Errno::ECANCELED)
        }
        PTRACE_DATA_SIGRETURN | PTRACE_DATA_RT_SIGRETURN => {
            // Entry to sigreturn(2) or rt_sigreturn(2).
            //
            // SAFETY: Signal handlers are per-process not per-thread!
            let status = match proc_status(pid) {
                Ok(status) => status,
                Err(_) => {
                    // SAFETY: Failed to get TGID,
                    // terminate the process.
                    let _ = kill(pid, Some(Signal::SIGKILL));
                    return Err(Errno::ESRCH);
                }
            };

            // SAFETY: Check for signal counts for SROP mitigation.
            let tgid = Pid::from_raw(status.pid);
            if cache.dec_sig_handle(tgid) {
                // Signal return has a corresponding signal.
                // All good, continue process normally.
                return Err(Errno::ECANCELED);
            }

            // !!! SIGRETURN W/O SIGNAL AKA SROP !!!
            //
            // Check sandbox verbosity.
            // Verbose logging is intended for malware analysis.
            let verbose = {
                SandboxGuard::Read(sandbox.read().unwrap_or_else(|err| err.into_inner())).verbose
            };

            // Read memory maps for logging.
            let memmap = if verbose { proc_maps(pid).ok() } else { None };

            // Read memory pointed by IP and SP.
            let ip = info.instruction_pointer;
            let sp = (info.stack_pointer & !0xF).saturating_sub(16);
            let ip_mem = if verbose { Some([0u8; 64]) } else { None };
            let sp_mem = if verbose { Some([0u8; 64]) } else { None };
            let process = RemoteProcess::new(pid);

            if let Some(mut ip_mem) = ip_mem {
                // SAFETY: This is a ptrace hook, the PID cannot be validated.
                let _ = unsafe { process.read_mem(&mut ip_mem, ip) };
            }
            if let Some(mut sp_mem) = sp_mem {
                // SAFETY: ditto.
                let _ = unsafe { process.read_mem(&mut sp_mem, sp) };
            }

            // Terminate the process.
            let _ = kill(pid, Some(Signal::SIGKILL));

            #[expect(clippy::disallowed_methods)]
            let arch = SydArch(scmp_arch(info.arch).unwrap());
            let is_realtime = scmp_trace_data == PTRACE_DATA_RT_SIGRETURN;

            // Log and return ESRCH.
            #[expect(clippy::disallowed_methods)]
            if !verbose {
                error!("ctx": "sigreturn", "op": "check_SROP",
                    "msg": "Artificial sigreturn(2) detected: assume SROP!",
                    "act": Action::Kill,
                    "pid": process.pid.as_raw(),
                    "sys": if is_realtime { "rt_sigreturn" } else { "sigreturn" },
                    "arch": arch,
                    "tgid": tgid.as_raw(),
                    "tip": "configure `trace/allow_unsafe_sigreturn:1'");
            } else {
                error!("ctx": "sigreturn", "op": "check_SROP",
                    "msg": "Artificial sigreturn(2) detected: assume SROP!",
                    "act": Action::Kill,
                    "pid": process.pid.as_raw(),
                    "sys": if is_realtime { "rt_sigreturn" } else { "sigreturn" },
                    "args": info_scmp.args,
                    "arch": arch,
                    "tgid": tgid.as_raw(),
                    "sig_caught": status.sig_caught,
                    "sig_blocked": status.sig_blocked,
                    "sig_ignored": status.sig_ignored,
                    "sig_pending_thread": status.sig_pending_thread,
                    "sig_pending_process": status.sig_pending_process,
                    "ip": ip,
                    "sp": sp,
                    "ip_mem": HEXLOWER.encode(ip_mem.as_ref().unwrap()),
                    "sp_mem": HEXLOWER.encode(sp_mem.as_ref().unwrap()),
                    "memmap": memmap,
                    "tip": "configure `trace/allow_unsafe_sigreturn:1'");
            }

            // Process is dead, Jim.
            Err(Errno::ESRCH)
        }

        data => unreachable!("BUG: invalid syscall data {data}!"),
    }
}

fn handle_ptrace_sysexit(
    pid: Pid,
    info: ptrace_syscall_info,
    cache: &Arc<WorkerCache>,
) -> Result<(), Errno> {
    // Get and remove the syscall entry from the cache,
    // and call the respective syscall handler.
    if let Some((pid, path)) = cache.get_chdir(pid) {
        crate::kernel::chdir::sysexit_chdir(pid, info, path)
    } else if let Some((pid, errno)) = cache.get_error(pid) {
        // Architectures like mips, s390x where return value has to be written twice.
        // errno is None for success.
        ptrace_set_return(pid, info.arch, errno)
    } else {
        unreachable!("BUG: Invalid syscall exit stop: {info:?}");
    }
}
