/*
Copyright 2022 The Kubernetes Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package v1alpha3

import (
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
)

// DeviceSelector must have exactly one field set.
type DeviceSelector struct {
	// CEL contains a CEL expression for selecting a device.
	//
	// +optional
	// +oneOf=SelectorType
	CEL *CELDeviceSelector `json:"cel,omitempty" protobuf:"bytes,1,opt,name=cel"`
}

// CELDeviceSelector contains a CEL expression for selecting a device.
type CELDeviceSelector struct {
	// Expression is a CEL expression which evaluates a single device. It
	// must evaluate to true when the device under consideration satisfies
	// the desired criteria, and false when it does not. Any other result
	// is an error and causes allocation of devices to abort.
	//
	// The expression's input is an object named "device", which carries
	// the following properties:
	//  - driver (string): the name of the driver which defines this device.
	//  - attributes (map[string]object): the device's attributes, grouped by prefix
	//    (e.g. device.attributes["dra.example.com"] evaluates to an object with all
	//    of the attributes which were prefixed by "dra.example.com".
	//  - capacity (map[string]object): the device's capacities, grouped by prefix.
	//
	// Example: Consider a device with driver="dra.example.com", which exposes
	// two attributes named "model" and "ext.example.com/family" and which
	// exposes one capacity named "modules". This input to this expression
	// would have the following fields:
	//
	//     device.driver
	//     device.attributes["dra.example.com"].model
	//     device.attributes["ext.example.com"].family
	//     device.capacity["dra.example.com"].modules
	//
	// The device.driver field can be used to check for a specific driver,
	// either as a high-level precondition (i.e. you only want to consider
	// devices from this driver) or as part of a multi-clause expression
	// that is meant to consider devices from different drivers.
	//
	// The value type of each attribute is defined by the device
	// definition, and users who write these expressions must consult the
	// documentation for their specific drivers. The value type of each
	// capacity is Quantity.
	//
	// If an unknown prefix is used as a lookup in either device.attributes
	// or device.capacity, an empty map will be returned. Any reference to
	// an unknown field will cause an evaluation error and allocation to
	// abort.
	//
	// A robust expression should check for the existence of attributes
	// before referencing them.
	//
	// For ease of use, the cel.bind() function is enabled, and can be used
	// to simplify expressions that access multiple attributes with the
	// same domain. For example:
	//
	//     cel.bind(dra, device.attributes["dra.example.com"], dra.someBool && dra.anotherBool)
	//
	// The length of the expression must be smaller or equal to 10 Ki. The
	// cost of evaluating it is also limited based on the estimated number
	// of logical steps.
	//
	// +required
	Expression string `json:"expression" protobuf:"bytes,1,name=expression"`
}

// CELSelectorExpressionMaxCost specifies the cost limit for a single CEL selector
// evaluation.
//
// There is no overall budget for selecting a device, so the actual time
// required for that is proportional to the number of CEL selectors and how
// often they need to be evaluated, which can vary depending on several factors
// (number of devices, cluster utilization, additional constraints).
//
// Validation against this limit and [CELSelectorExpressionMaxLength] happens
// only when setting an expression for the first time or when changing it. If
// the limits are changed in a future Kubernetes release, existing users are
// guaranteed that existing expressions will continue to be valid.
//
// However, the kube-scheduler also applies this cost limit at runtime, so it
// could happen that a valid expression fails at runtime after an up- or
// downgrade. This can also happen without version skew when the cost estimate
// underestimated the actual cost. That this might happen is the reason why
// kube-scheduler enforces the runtime limit instead of relying on validation.
//
// According to
// https://github.com/kubernetes/kubernetes/blob/4aeaf1e99e82da8334c0d6dddd848a194cd44b4f/staging/src/k8s.io/apiserver/pkg/apis/cel/config.go#L20-L22,
// this gives roughly 0.1 second for each expression evaluation.
// However, this depends on how fast the machine is.
const CELSelectorExpressionMaxCost = 1000000

// CELSelectorExpressionMaxLength is the maximum length of a CEL selector expression string.
const CELSelectorExpressionMaxLength = 10 * 1024

// The device this taint is attached to has the "effect" on
// any claim which does not tolerate the taint and, through the claim,
// to pods using the claim.
//
// +protobuf.options.(gogoproto.goproto_stringer)=false
type DeviceTaint struct {
	// The taint key to be applied to a device.
	// Must be a label name.
	//
	// +required
	Key string `json:"key" protobuf:"bytes,1,name=key"`

	// The taint value corresponding to the taint key.
	// Must be a label value.
	//
	// +optional
	Value string `json:"value,omitempty" protobuf:"bytes,2,opt,name=value"`

	// The effect of the taint on claims that do not tolerate the taint
	// and through such claims on the pods using them.
	// Valid effects are NoSchedule and NoExecute. PreferNoSchedule as used for
	// nodes is not valid here.
	//
	// +required
	Effect DeviceTaintEffect `json:"effect" protobuf:"bytes,3,name=effect,casttype=DeviceTaintEffect"`

	// ^^^^
	//
	// Implementing PreferNoSchedule would depend on a scoring solution for DRA.
	// It might get added as part of that.

	// TimeAdded represents the time at which the taint was added.
	// Added automatically during create or update if not set.
	//
	// +optional
	TimeAdded *metav1.Time `json:"timeAdded,omitempty" protobuf:"bytes,4,opt,name=timeAdded"`

	// ^^^
	//
	// This field was defined as "It is only written for NoExecute taints." for node taints.
	// But in practice, Kubernetes never did anything with it (no validation, no defaulting,
	// ignored during pod eviction in pkg/controller/tainteviction).
}

// +enum
type DeviceTaintEffect string

const (
	// Do not allow new pods to schedule which use a tainted device unless they tolerate the taint,
	// but allow all pods submitted to Kubelet without going through the scheduler
	// to start, and allow all already-running pods to continue running.
	DeviceTaintEffectNoSchedule DeviceTaintEffect = "NoSchedule"

	// Evict any already-running pods that do not tolerate the device taint.
	DeviceTaintEffectNoExecute DeviceTaintEffect = "NoExecute"
)

// +genclient
// +genclient:nonNamespaced
// +k8s:deepcopy-gen:interfaces=k8s.io/apimachinery/pkg/runtime.Object
// +k8s:prerelease-lifecycle-gen:introduced=1.33

// DeviceTaintRule adds one taint to all devices which match the selector.
// This has the same effect as if the taint was specified directly
// in the ResourceSlice by the DRA driver.
type DeviceTaintRule struct {
	metav1.TypeMeta `json:",inline"`
	// Standard object metadata
	// +optional
	metav1.ObjectMeta `json:"metadata,omitempty" protobuf:"bytes,1,opt,name=metadata"`

	// Spec specifies the selector and one taint.
	//
	// Changing the spec automatically increments the metadata.generation number.
	Spec DeviceTaintRuleSpec `json:"spec" protobuf:"bytes,2,name=spec"`

	// ^^^
	// A spec gets added because adding a status seems likely.
	// Such a status could provide feedback on applying the
	// eviction and/or statistics (number of matching devices,
	// affected allocated claims, pods remaining to be evicted,
	// etc.).
}

// DeviceTaintRuleSpec specifies the selector and one taint.
type DeviceTaintRuleSpec struct {
	// DeviceSelector defines which device(s) the taint is applied to.
	// All selector criteria must be satified for a device to
	// match. The empty selector matches all devices. Without
	// a selector, no devices are matches.
	//
	// +optional
	DeviceSelector *DeviceTaintSelector `json:"deviceSelector,omitempty" protobuf:"bytes,1,opt,name=deviceSelector"`

	// The taint that gets applied to matching devices.
	//
	// +required
	Taint DeviceTaint `json:"taint,omitempty" protobuf:"bytes,2,rep,name=taint"`
}

// DeviceTaintSelector defines which device(s) a DeviceTaintRule applies to.
// The empty selector matches all devices. Without a selector, no devices
// are matched.
type DeviceTaintSelector struct {
	// If DeviceClassName is set, the selectors defined there must be
	// satisfied by a device to be selected. This field corresponds
	// to class.metadata.name.
	//
	// +optional
	DeviceClassName *string `json:"deviceClassName,omitempty" protobuf:"bytes,1,opt,name=deviceClassName"`

	// If driver is set, only devices from that driver are selected.
	// This fields corresponds to slice.spec.driver.
	//
	// +optional
	Driver *string `json:"driver,omitempty" protobuf:"bytes,2,opt,name=driver"`

	// If pool is set, only devices in that pool are selected.
	//
	// Also setting the driver name may be useful to avoid
	// ambiguity when different drivers use the same pool name,
	// but this is not required because selecting pools from
	// different drivers may also be useful, for example when
	// drivers with node-local devices use the node name as
	// their pool name.
	//
	// +optional
	Pool *string `json:"pool,omitempty" protobuf:"bytes,3,opt,name=pool"`

	// If device is set, only devices with that name are selected.
	// This field corresponds to slice.spec.devices[].name.
	//
	// Setting also driver and pool may be required to avoid ambiguity,
	// but is not required.
	//
	// +optional
	Device *string `json:"device,omitempty" protobuf:"bytes,4,opt,name=device"`

	// Selectors contains the same selection criteria as a ResourceClaim.
	// Currently, CEL expressions are supported. All of these selectors
	// must be satisfied.
	//
	// +optional
	// +listType=atomic
	Selectors []DeviceSelector `json:"selectors,omitempty" protobuf:"bytes,5,rep,name=selectors"`
}

// +k8s:deepcopy-gen:interfaces=k8s.io/apimachinery/pkg/runtime.Object
// +k8s:prerelease-lifecycle-gen:introduced=1.33

// DeviceTaintRuleList is a collection of DeviceTaintRules.
type DeviceTaintRuleList struct {
	metav1.TypeMeta `json:",inline"`
	// Standard list metadata
	// +optional
	metav1.ListMeta `json:"metadata,omitempty" protobuf:"bytes,1,opt,name=metadata"`

	// Items is the list of DeviceTaintRules.
	Items []DeviceTaintRule `json:"items" protobuf:"bytes,2,rep,name=items"`
}
