// Copyright (C) 2016 The Qt Company Ltd.
// SPDX-License-Identifier: LicenseRef-Qt-Commercial OR LGPL-3.0-only OR GPL-2.0-only OR GPL-3.0-only
// Qt-Security score:significant reason:default

#include "qprogressbar.h"

#include <qlocale.h>
#include <qevent.h>
#include <qpainter.h>
#include <qstylepainter.h>
#include <qstyleoption.h>
#include <private/qwidget_p.h>
#if QT_CONFIG(accessibility)
#include <qaccessible.h>
#endif
#include <limits.h>

QT_BEGIN_NAMESPACE

using namespace Qt::StringLiterals;

class QProgressBarPrivate : public QWidgetPrivate
{
    Q_DECLARE_PUBLIC(QProgressBar)

public:
    QProgressBarPrivate();

    void init();
    void initDefaultFormat();
    inline void resetLayoutItemMargins();

    int minimum;
    int maximum;
    int value;
    Qt::Alignment alignment;
    uint textVisible : 1;
    uint defaultFormat: 1;
    int lastPaintedValue;
    Qt::Orientation orientation;
    bool invertedAppearance;
    QProgressBar::Direction textDirection;
    QString format;
    inline int bound(int val) const { return qMax(minimum-1, qMin(maximum, val)); }
    bool repaintRequired() const;
};

QProgressBarPrivate::QProgressBarPrivate()
    : minimum(0), maximum(100), value(-1), alignment(Qt::AlignLeft), textVisible(true),
      defaultFormat(true), lastPaintedValue(-1), orientation(Qt::Horizontal), invertedAppearance(false),
      textDirection(QProgressBar::TopToBottom)
{
    initDefaultFormat();
}

void QProgressBarPrivate::initDefaultFormat()
{
    if (defaultFormat) {
        //: %p is the percent value, % is the percent sign. When translated
        //: to the identical %p%, then the second % symbol will be replaced
        //: by the percentage-symbol from the user's locale.
        format = QProgressBar::tr("%p%");
    }

    if (format == "%p%"_L1)
        format = "%p"_L1 + locale.percent();
}

void QProgressBarPrivate::init()
{
    Q_Q(QProgressBar);
    QSizePolicy sp(QSizePolicy::Expanding, QSizePolicy::Fixed);
    if (orientation == Qt::Vertical)
        sp.transpose();
    q->setSizePolicy(sp);
    q->setAttribute(Qt::WA_WState_OwnSizePolicy, false);
    resetLayoutItemMargins();
}

void QProgressBarPrivate::resetLayoutItemMargins()
{
    Q_Q(QProgressBar);
    QStyleOptionProgressBar option;
    // ### It seems like this can be called directly from the constructor which should be avoided
    // ### if possible, since we will not call a possible re-implemented version
    q->initStyleOption(&option);
    setLayoutItemMargins(QStyle::SE_ProgressBarLayoutItem, &option);
}

/*!
    Initialize \a option with the values from this QProgressBar. This method is useful
    for subclasses when they need a QStyleOptionProgressBar,
    but don't want to fill in all the information themselves.

    \sa QStyleOption::initFrom()
*/
void QProgressBar::initStyleOption(QStyleOptionProgressBar *option) const
{
    if (!option)
        return;
    Q_D(const QProgressBar);
    option->initFrom(this);

    if (d->orientation == Qt::Horizontal)
        option->state |= QStyle::State_Horizontal;
    option->minimum = d->minimum;
    option->maximum = d->maximum;
    option->progress = d->value;
    option->textAlignment = d->alignment;
    option->textVisible = d->textVisible;
    option->text = text();
    option->invertedAppearance = d->invertedAppearance;
    option->bottomToTop = d->textDirection == QProgressBar::BottomToTop;
}

bool QProgressBarPrivate::repaintRequired() const
{
    Q_Q(const QProgressBar);
    if (value == lastPaintedValue)
        return false;

    const qint64 valueDifference = qAbs(qint64(value) - qint64(lastPaintedValue));

    if (value == minimum || value == maximum)
        return true;

    const int totalSteps = maximum - minimum;

    const int currentPercentage = totalSteps ? (value - minimum) * 100 / totalSteps : 0;
    const int lastPaintedPercentage = totalSteps ? (lastPaintedValue - minimum) * 100 / totalSteps
                                                 : 0;

    const int percentageChangeConstant = 1;
    const bool percentageChanged = (qAbs(currentPercentage - lastPaintedPercentage) >= percentageChangeConstant);

    if (textVisible) {
        if (format.contains("%v"_L1))
            return true;
        if (format.contains("%p"_L1) && percentageChanged)
            return true;
    }

    // Check if the bar needs to be repainted based on pixel-level differences
    QStyleOptionProgressBar opt;
    q->initStyleOption(&opt);
    QRect groove = q->style()->subElementRect(QStyle::SE_ProgressBarGroove, &opt, q);
    int grooveBlock = (q->orientation() == Qt::Horizontal) ? groove.width() : groove.height();
    const double pixelSize = static_cast<double>(totalSteps) / grooveBlock;
    return valueDifference > pixelSize;
}

/*!
    \class QProgressBar
    \brief The QProgressBar widget provides a horizontal or vertical progress bar.

    \ingroup basicwidgets
    \inmodule QtWidgets

    \image fusion-progressbar.png {Progress bar showing 42%}

    A progress bar is used to give the user an indication of the
    progress of an operation and to reassure them that the application
    is still running.

    The progress bar uses the concept of \e steps. You set it up by
    specifying the minimum and maximum possible step values, and it
    will display the percentage of steps that have been completed
    when you later give it the current step value. The percentage is
    calculated by dividing the progress (value() - minimum()) divided
    by maximum() - minimum().

    You can specify the minimum and maximum number of steps with
    setMinimum() and setMaximum. The current number of steps is set
    with setValue(). The progress bar can be rewound to the
    beginning with reset().

    If minimum and maximum both are set to 0, the bar shows a busy
    indicator instead of a percentage of steps. This is useful, for
    example, when using QNetworkAccessManager to download items when
    they are unable to determine the size of the item being downloaded.

    \sa QProgressDialog
*/

/*!
    \since 4.1
    \enum QProgressBar::Direction
    \brief Specifies the reading direction of the \l text for vertical progress bars.

    \value TopToBottom The text is rotated 90 degrees clockwise.
    \value BottomToTop The text is rotated 90 degrees counter-clockwise.

    Note that whether or not the text is drawn is dependent on the style.
    Currently CleanLooks and Plastique draw the text. Mac, Windows
    and WindowsVista style do not.

    \sa textDirection
*/

/*!
    \fn void QProgressBar::valueChanged(int value)

    This signal is emitted when the value shown in the progress bar changes.
    \a value is the new value shown by the progress bar.
*/

/*!
    Constructs a progress bar with the given \a parent.

    By default, the minimum step value is set to 0, and the maximum to 100.

    \sa setRange()
*/

QProgressBar::QProgressBar(QWidget *parent)
    : QWidget(*(new QProgressBarPrivate), parent, { })
{
    d_func()->init();
}

/*!
    Destructor.
*/
QProgressBar::~QProgressBar()
{
}

/*!
    Reset the progress bar. The progress bar "rewinds" and shows no
    progress.
*/

void QProgressBar::reset()
{
    Q_D(QProgressBar);
    if (d->minimum == INT_MIN)
        d->value = INT_MIN;
    else
        d->value = d->minimum - 1;
    repaint();
}

/*!
    \property QProgressBar::minimum
    \brief the progress bar's minimum value

    When setting this property, the \l maximum is adjusted if
    necessary to ensure that the range remains valid. If the
    current value falls outside the new range, the progress bar is reset
    with reset().
*/
void QProgressBar::setMinimum(int minimum)
{
    setRange(minimum, qMax(d_func()->maximum, minimum));
}

int QProgressBar::minimum() const
{
    return d_func()->minimum;
}


/*!
    \property QProgressBar::maximum
    \brief the progress bar's maximum value

    When setting this property, the \l minimum is adjusted if
    necessary to ensure that the range remains valid. If the
    current value falls outside the new range, the progress bar is reset
    with reset().
*/

void QProgressBar::setMaximum(int maximum)
{
    setRange(qMin(d_func()->minimum, maximum), maximum);
}

int QProgressBar::maximum() const
{
    return d_func()->maximum;
}

/*!
    \property QProgressBar::value
    \brief the progress bar's current value

    Attempting to change the current value to one outside
    the minimum-maximum range has no effect on the current value.
*/
void QProgressBar::setValue(int value)
{
    Q_D(QProgressBar);
    if (d->value == value
            || ((value > d->maximum || value < d->minimum)
                && (d->maximum != 0 || d->minimum != 0)))
        return;
    d->value = value;
    emit valueChanged(value);
#if QT_CONFIG(accessibility)
    if (isVisible()) {
        QAccessibleValueChangeEvent event(this, value);
        QAccessible::updateAccessibility(&event);
    }
#endif
    if (d->repaintRequired())
        repaint();
}

int QProgressBar::value() const
{
    return d_func()->value;
}

/*!
    Sets the progress bar's minimum and maximum values to \a minimum and
    \a maximum respectively.

    If \a maximum is smaller than \a minimum, \a minimum becomes the only
    legal value.

    If the current value falls outside the new range, the progress bar is reset
    with reset().

    The QProgressBar can be set to undetermined state by using setRange(0, 0).

    \sa minimum, maximum
*/
void QProgressBar::setRange(int minimum, int maximum)
{
    Q_D(QProgressBar);
    if (minimum != d->minimum || maximum != d->maximum) {
        d->minimum = minimum;
        d->maximum = qMax(minimum, maximum);

        if (d->value < qint64(d->minimum) - 1 || d->value > d->maximum)
            reset();
        else
            update();
    }
}

/*!
    \property QProgressBar::textVisible
    \brief whether the current completed percentage should be displayed

    This property may be ignored by the style (e.g., QMacStyle never draws the text).

    \sa textDirection
*/
void QProgressBar::setTextVisible(bool visible)
{
    Q_D(QProgressBar);
    if (d->textVisible != visible) {
        d->textVisible = visible;
        repaint();
    }
}

bool QProgressBar::isTextVisible() const
{
    return d_func()->textVisible;
}

/*!
    \property QProgressBar::alignment
    \brief the alignment of the progress bar
*/
void QProgressBar::setAlignment(Qt::Alignment alignment)
{
    if (d_func()->alignment != alignment) {
        d_func()->alignment = alignment;
        repaint();
    }
}

Qt::Alignment QProgressBar::alignment() const
{
    return d_func()->alignment;
}

/*!
    \reimp
*/
void QProgressBar::paintEvent(QPaintEvent *)
{
    QStylePainter paint(this);
    QStyleOptionProgressBar opt;
    initStyleOption(&opt);
    paint.drawControl(QStyle::CE_ProgressBar, opt);
    d_func()->lastPaintedValue = d_func()->value;
}

/*!
    \reimp
*/
QSize QProgressBar::sizeHint() const
{
    ensurePolished();
    QFontMetrics fm = fontMetrics();
    QStyleOptionProgressBar opt;
    initStyleOption(&opt);
    int cw = style()->pixelMetric(QStyle::PM_ProgressBarChunkWidth, &opt, this);
    QSize size = QSize(qMax(9, cw) * 7 + fm.horizontalAdvance(u'0') * 4, fm.height() + 8);
    if (!(opt.state & QStyle::State_Horizontal))
        size = size.transposed();
    return style()->sizeFromContents(QStyle::CT_ProgressBar, &opt, size, this);
}

/*!
    \reimp
*/
QSize QProgressBar::minimumSizeHint() const
{
    QSize size;
    if (orientation() == Qt::Horizontal)
        size = QSize(sizeHint().width(), fontMetrics().height() + 2);
    else
        size = QSize(fontMetrics().height() + 2, sizeHint().height());
    return size;
}

/*!
    \property QProgressBar::text
    \brief the descriptive text shown with the progress bar

    The text returned is the same as the text displayed in the center
    (or in some styles, to the left) of the progress bar.

    The progress shown in the text may be smaller than the minimum value,
    indicating that the progress bar is in the "reset" state before any
    progress is set.

    In the default implementation, the text either contains a percentage
    value that indicates the progress so far, or it is blank because the
    progress bar is in the reset state.
*/
QString QProgressBar::text() const
{
    Q_D(const QProgressBar);
    if ((d->maximum == 0 && d->minimum == 0) || d->value < d->minimum
            || (d->value == INT_MIN && d->minimum == INT_MIN))
        return QString();

    qint64 totalSteps = qint64(d->maximum) - d->minimum;

    QString result = d->format;
    QLocale locale = d->locale; // Omit group separators for compatibility with previous versions that were non-localized.
    locale.setNumberOptions(locale.numberOptions() | QLocale::OmitGroupSeparator);
    result.replace("%m"_L1, locale.toString(totalSteps));
    result.replace("%v"_L1, locale.toString(d->value));

    // If max and min are equal and we get this far, it means that the
    // progress bar has one step and that we are on that step. Return
    // 100% here in order to avoid division by zero further down.
    if (totalSteps == 0) {
        result.replace("%p"_L1, locale.toString(100));
        return result;
    }

    const auto progress = static_cast<int>((qint64(d->value) - d->minimum) * 100.0 / totalSteps);
    result.replace("%p"_L1, locale.toString(progress));
    return result;
}

/*!
    \since 4.1
    \property QProgressBar::orientation
    \brief the orientation of the progress bar

    The orientation must be \l Qt::Horizontal (the default) or \l
    Qt::Vertical.

    \sa invertedAppearance, textDirection
*/

void QProgressBar::setOrientation(Qt::Orientation orientation)
{
    Q_D(QProgressBar);
    if (d->orientation == orientation)
        return;
    d->orientation = orientation;
    if (!testAttribute(Qt::WA_WState_OwnSizePolicy)) {
        setSizePolicy(sizePolicy().transposed());
        setAttribute(Qt::WA_WState_OwnSizePolicy, false);
    }
    d->resetLayoutItemMargins();
    update();
    updateGeometry();
}

Qt::Orientation QProgressBar::orientation() const
{
    Q_D(const QProgressBar);
    return d->orientation;
}

/*!
    \since 4.1
    \property QProgressBar::invertedAppearance
    \brief whether or not a progress bar shows its progress inverted

    If this property is \c true, the progress bar grows in the other
    direction (e.g. from right to left). By default, the progress bar
    is not inverted.

    \sa orientation, layoutDirection
*/

void QProgressBar::setInvertedAppearance(bool invert)
{
    Q_D(QProgressBar);
    d->invertedAppearance = invert;
    update();
}

bool QProgressBar::invertedAppearance() const
{
    Q_D(const QProgressBar);
    return d->invertedAppearance;
}

/*!
    \since 4.1
    \property QProgressBar::textDirection
    \brief the reading direction of the \l text for vertical progress bars

    This property has no impact on horizontal progress bars.
    By default, the reading direction is QProgressBar::TopToBottom.

    \sa orientation, textVisible
*/
void QProgressBar::setTextDirection(QProgressBar::Direction textDirection)
{
    Q_D(QProgressBar);
    d->textDirection = textDirection;
    update();
}

QProgressBar::Direction QProgressBar::textDirection() const
{
    Q_D(const QProgressBar);
    return d->textDirection;
}

/*! \reimp */
bool QProgressBar::event(QEvent *e)
{
    Q_D(QProgressBar);
    switch (e->type()) {
    case QEvent::StyleChange:
#ifdef Q_OS_MAC
    case QEvent::MacSizeChange:
#endif
        d->resetLayoutItemMargins();
        break;
    case QEvent::LocaleChange:
        d->initDefaultFormat();
        break;
    default:
        break;
    }
    return QWidget::event(e);
}

/*!
    \since 4.2
    \property QProgressBar::format
    \brief the string used to generate the current text

    %p - is replaced by the percentage completed.
    %v - is replaced by the current value.
    %m - is replaced by the total number of steps.

    The default value is "%p%".

    \sa text()
*/
void QProgressBar::setFormat(const QString &format)
{
    Q_D(QProgressBar);
    if (d->format == format)
        return;
    d->format = format;
    d->defaultFormat = false;
    update();
}

void QProgressBar::resetFormat()
{
    Q_D(QProgressBar);
    d->defaultFormat = true;
    d->initDefaultFormat();
    update();
}

QString QProgressBar::format() const
{
    Q_D(const QProgressBar);
    return d->format;
}

QT_END_NAMESPACE

#include "moc_qprogressbar.cpp"
