// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_dq.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_dq__A6__F398

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_16_3_T_T_0___gfx120x)xyzw" };
#else
{ R"xyzw(amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_16_3_T_T_0___gfx120x)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊fp32@16_16_3_T_T_0)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx1201)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0x46194c3eu, 0xdb8391d7u, 0, 138 }, // 46194c3edb8391d7 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_16_3_T_T_0___gfx120x__P__16_16__CO__wave1_warp4_stg1--Arch_gfx1201
    { 0x34b38219u, 0xfa7f2af8u, 0, 80 }, // 34b38219fa7f2af8 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_16_3_T_T_0___gfx120x__P__16_16__CO__wave2_warp4_stg1--Arch_gfx1201
    { 0xa19de8e7u, 0x1ac5f6a8u, 0, 155 }, // a19de8e71ac5f6a8 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_16_3_T_T_0___gfx120x__P__16_16__CO__wave3_warp4_stg1--Arch_gfx1201
    { 0x55c92a69u, 0xff301042u, 0, 23 }, // 55c92a69ff301042 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_16_3_T_T_0___gfx120x__P__16_16__CO__wave4_warp2_stg1--Arch_gfx1201
    { 0xecfbab4au, 0xd9630c23u, 0, 103 }, // ecfbab4ad9630c23 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_16_3_T_T_0___gfx120x__P__16_16__CO__wave4_warp4_stg1--Arch_gfx1201
    { 0xd7342e08u, 0x8527cbc3u, 40, 46 }, // d7342e088527cbc3 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_16_3_T_T_0___gfx120x__P__32_16__CO__wave1_warp2_stg1--Arch_gfx1201
    { 0x5f5fb3c2u, 0xb58aebd7u, 40, 63 }, // 5f5fb3c2b58aebd7 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_16_3_T_T_0___gfx120x__P__32_16__CO__wave2_warp2_stg1--Arch_gfx1201
    { 0x89340abcu, 0x7fc6aebdu, 40, 6 }, // 89340abc7fc6aebd = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_16_3_T_T_0___gfx120x__P__32_16__CO__wave3_warp2_stg1--Arch_gfx1201
    { 0xb58216c3u, 0x9276aa5au, 40, 23 }, // b58216c39276aa5a = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_16_3_T_T_0___gfx120x__P__32_16__CO__wave4_warp2_stg1--Arch_gfx1201
    { 0xc87bb40au, 0x79af8bfbu, 40, 103 }, // c87bb40a79af8bfb = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_16_3_T_T_0___gfx120x__P__32_16__CO__wave4_warp4_stg1--Arch_gfx1201
    { 0xcdc52fefu, 0xbcd7fdd9u, 97, 138 }, // cdc52fefbcd7fdd9 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_16_3_T_T_0___gfx120x__P__32_32__CO__wave1_warp4_stg1--Arch_gfx1201
    { 0x9a6c557fu, 0x69a237e4u, 97, 80 }, // 9a6c557f69a237e4 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_16_3_T_T_0___gfx120x__P__32_32__CO__wave2_warp4_stg1--Arch_gfx1201
    { 0x0a35e1acu, 0xe8f6ab14u, 97, 155 }, // 0a35e1ace8f6ab14 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_16_3_T_T_0___gfx120x__P__32_32__CO__wave3_warp4_stg1--Arch_gfx1201
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_dq_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_dq_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx1201_mod0
{{ 0, 2, 4, 1, 4, 4, 4, 3, 3, 9},
 { 7, 8, 5, 8,11,12,12,10,12,12},
 { 8, 8, 7, 8, 6, 8, 6, 6, 7, 8},
 { 5, 8, 5, 5, 7, 6, 7, 8, 7, 7},
 { 6, 8, 5, 5, 7, 8, 8, 6, 7, 5},
 { 6, 8, 6, 6, 5, 6, 5, 7, 7, 7},
 { 8, 8, 7, 5, 8, 6, 7, 7, 8, 4},
 { 6, 5, 8, 6, 7, 6, 5, 7,12,10},
 { 7, 7, 8, 8, 6, 7, 7, 7, 6, 5},
 { 6, 6, 8, 7, 7, 7, 6, 5, 8, 8}}
// End of GPU gfx1201_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelDqContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelDqContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_dq__A6__F398 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 5;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_dq__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 5;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*fp32:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// D = "*fp32:16"
// stride_bz = 0
// stride_dbz = 0
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = "fp32"
// philox_seed_ptr = "*u64"
// philox_offset1 = "*u64"
// philox_offset2 = "u64"
// Window_left = "i32"
// BLOCK_DMODEL = 16
// CAUSAL_TYPE = 3
// ENABLE_DROPOUT = True
// PADDED_HEAD = True
// BIAS_TYPE = 0

// SELECT * FROM FLASH$bwd_kernel_dq WHERE gpu IN ('gfx1201_mod0') AND inputs$Q_dtype = 'torch.float32' AND inputs$BLOCK_DMODEL = 16 AND inputs$CAUSAL_TYPE = 3 AND inputs$ENABLE_DROPOUT = True AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 0

// vim: set fileencoding=utf-8

