// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_dq.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_dq__A3__F39

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_48_0_T_T_1___gfx11xx)xyzw" };
#else
{ R"xyzw(amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_48_0_T_T_1___gfx11xx)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊fp16@16_48_0_T_T_1)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx1102)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0xadc2b6ebu, 0xa3146e32u, 0, 46 }, // adc2b6eba3146e32 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_48_0_T_T_1___gfx11xx__P__16_16__CO__wave1_warp2_stg1--Arch_gfx1102
    { 0x2965dd7bu, 0xf353cb97u, 0, 63 }, // 2965dd7bf353cb97 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_48_0_T_T_1___gfx11xx__P__16_16__CO__wave2_warp2_stg1--Arch_gfx1102
    { 0xaf142704u, 0x248e1146u, 40, 46 }, // af142704248e1146 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_48_0_T_T_1___gfx11xx__P__32_16__CO__wave1_warp2_stg1--Arch_gfx1102
    { 0x73195a15u, 0x6f9c62d0u, 40, 63 }, // 73195a156f9c62d0 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_48_0_T_T_1___gfx11xx__P__32_16__CO__wave2_warp2_stg1--Arch_gfx1102
    { 0x3e725168u, 0x98249987u, 40, 6 }, // 3e72516898249987 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_48_0_T_T_1___gfx11xx__P__32_16__CO__wave3_warp2_stg1--Arch_gfx1102
    { 0x0bf2c3c3u, 0x24d3858au, 40, 23 }, // 0bf2c3c324d3858a = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_48_0_T_T_1___gfx11xx__P__32_16__CO__wave4_warp2_stg1--Arch_gfx1102
    { 0x4d21f262u, 0xa7e4dda7u, 97, 80 }, // 4d21f262a7e4dda7 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_48_0_T_T_1___gfx11xx__P__32_32__CO__wave2_warp4_stg1--Arch_gfx1102
    { 0x4e606acbu, 0x7ef5edb8u, 97, 155 }, // 4e606acb7ef5edb8 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_48_0_T_T_1___gfx11xx__P__32_32__CO__wave3_warp4_stg1--Arch_gfx1102
    { 0x0798022fu, 0xd652663au, 97, 103 }, // 0798022fd652663a = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_48_0_T_T_1___gfx11xx__P__32_32__CO__wave4_warp4_stg1--Arch_gfx1102
    { 0x07419456u, 0x56de1661u, 120, 138 }, // 0741945656de1661 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_48_0_T_T_1___gfx11xx__P__64_16__CO__wave1_warp4_stg1--Arch_gfx1102
    { 0xaff389beu, 0x26195082u, 120, 80 }, // aff389be26195082 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_48_0_T_T_1___gfx11xx__P__64_16__CO__wave2_warp4_stg1--Arch_gfx1102
    { 0xaa5eef16u, 0x3a748797u, 120, 155 }, // aa5eef163a748797 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_48_0_T_T_1___gfx11xx__P__64_16__CO__wave3_warp4_stg1--Arch_gfx1102
    { 0x7bf9db8fu, 0x1b4b6214u, 120, 103 }, // 7bf9db8f1b4b6214 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_48_0_T_T_1___gfx11xx__P__64_16__CO__wave4_warp4_stg1--Arch_gfx1102
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_dq_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_dq_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx1102_mod0
{{ 5, 5, 4, 0, 0, 0, 0, 0, 0, 0},
 {10,10,11, 0, 0, 5, 5, 5, 5, 5},
 { 4, 5, 5,12,12, 3, 4, 2, 5, 5},
 { 0, 0, 0, 0, 0, 3, 1, 1, 0, 5},
 { 0, 0, 5, 5,10, 3, 0, 0, 5, 5},
 { 0, 5, 5, 5, 2, 4, 0, 4, 5, 7},
 { 9, 5, 5, 5, 3, 4, 3, 2, 7, 8},
 {12, 5,10, 5, 5, 5, 4, 2, 3, 3},
 {11, 5,12, 5, 5, 5, 6, 6, 1, 0},
 { 9, 9, 5, 5, 5, 3, 1, 1, 3, 2}}
// End of GPU gfx1102_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelDqContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelDqContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_dq__A3__F39 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 3;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_dq__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 3;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*fp16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// D = "*fp32:16"
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = "fp32"
// philox_seed_ptr = "*u64"
// philox_offset1 = "*u64"
// philox_offset2 = "u64"
// Window_left = 0
// BLOCK_DMODEL = 48
// CAUSAL_TYPE = 0
// ENABLE_DROPOUT = True
// PADDED_HEAD = True
// BIAS_TYPE = 1

// SELECT * FROM FLASH$bwd_kernel_dq WHERE gpu IN ('gfx1100_mod0') AND inputs$Q_dtype = 'torch.float16' AND inputs$BLOCK_DMODEL = 48 AND inputs$CAUSAL_TYPE = 0 AND inputs$ENABLE_DROPOUT = True AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 1

// vim: set fileencoding=utf-8

