/* penrose-legacy.h: legacy Penrose tiling functions.
 *
 * Provides an interface with which to generate Penrose tilings
 * by recursive subdivision of an initial tile of choice (one of the
 * four sets of two pairs kite/dart, or thin/thick rhombus).
 *
 * You supply a callback function and a context pointer, which is
 * called with each tile in turn: you choose how many times to recurse.
 *
 * This method of generating Penrose tiling fragments is superseded by
 * the completely different algorithm in penrose.c. We keep the legacy
 * algorithm around only for interpreting Loopy game IDs generated by
 * older versions of the code.
 */

#ifndef PUZZLES_PENROSE_LEGACY_H
#define PUZZLES_PENROSE_LEGACY_H

#ifndef PHI
#define PHI 1.6180339887
#endif

typedef struct vector vector;

double penrose_legacy_vx(vector *vs, int i);
double penrose_legacy_vy(vector *vs, int i);

typedef struct penrose_legacy_state penrose_legacy_state;

/* Return non-zero to clip the tree here (i.e. not recurse
 * below this tile).
 *
 * Parameters are state, vector array, npoints, depth.
 * ctx is inside state.
 */
typedef int (*tile_callback)(penrose_legacy_state *, vector *, int, int);

struct penrose_legacy_state {
    int start_size;  /* initial side length */
    int max_depth;      /* Recursion depth */

    tile_callback new_tile;
    void *ctx;          /* for callback */
};

#ifndef PENROSE_ENUM_DEFINED
#define PENROSE_ENUM_DEFINED
enum { PENROSE_P2, PENROSE_P3 };
#endif

extern int penrose_legacy(penrose_legacy_state *state, int which, int angle);

/* Returns the side-length of a penrose tile at recursion level
 * gen, given a starting side length. */
extern double penrose_legacy_side_length(double start_size, int depth);

/* Calculate start size and recursion depth required to produce a
 * width-by-height sized patch of penrose tiles with the given tilesize */
extern void penrose_legacy_calculate_size(
    int which, int tilesize, int w, int h,
    double *required_radius, int *start_size, int *depth);

#endif
