#include "bridge.hh"
#include <algorithm>
#include <array>
#include <cassert>
#include <cstddef>
#include <cstdint>
#include <exception>
#include <initializer_list>
#include <iterator>
#include <memory>
#include <new>
#include <stdexcept>
#include <string>
#include <type_traits>
#include <utility>

namespace rust {
inline namespace cxxbridge1 {
// #include "rust/cxx.h"

#ifndef CXXBRIDGE1_PANIC
#define CXXBRIDGE1_PANIC
template <typename Exception>
void panic [[noreturn]] (const char *msg);
#endif // CXXBRIDGE1_PANIC

struct unsafe_bitcopy_t;

namespace {
template <typename T>
class impl;
} // namespace

class Opaque;

template <typename T>
::std::size_t size_of();
template <typename T>
::std::size_t align_of();

#ifndef CXXBRIDGE1_RUST_STRING
#define CXXBRIDGE1_RUST_STRING
class String final {
public:
  String() noexcept;
  String(const String &) noexcept;
  String(String &&) noexcept;
  ~String() noexcept;

  String(const std::string &);
  String(const char *);
  String(const char *, std::size_t);
  String(const char16_t *);
  String(const char16_t *, std::size_t);

  static String lossy(const std::string &) noexcept;
  static String lossy(const char *) noexcept;
  static String lossy(const char *, std::size_t) noexcept;
  static String lossy(const char16_t *) noexcept;
  static String lossy(const char16_t *, std::size_t) noexcept;

  String &operator=(const String &) &noexcept;
  String &operator=(String &&) &noexcept;

  explicit operator std::string() const;

  const char *data() const noexcept;
  std::size_t size() const noexcept;
  std::size_t length() const noexcept;
  bool empty() const noexcept;

  const char *c_str() noexcept;

  std::size_t capacity() const noexcept;
  void reserve(size_t new_cap) noexcept;

  using iterator = char *;
  iterator begin() noexcept;
  iterator end() noexcept;

  using const_iterator = const char *;
  const_iterator begin() const noexcept;
  const_iterator end() const noexcept;
  const_iterator cbegin() const noexcept;
  const_iterator cend() const noexcept;

  bool operator==(const String &) const noexcept;
  bool operator!=(const String &) const noexcept;
  bool operator<(const String &) const noexcept;
  bool operator<=(const String &) const noexcept;
  bool operator>(const String &) const noexcept;
  bool operator>=(const String &) const noexcept;

  void swap(String &) noexcept;

  String(unsafe_bitcopy_t, const String &) noexcept;

private:
  struct lossy_t;
  String(lossy_t, const char *, std::size_t) noexcept;
  String(lossy_t, const char16_t *, std::size_t) noexcept;
  friend void swap(String &lhs, String &rhs) noexcept { lhs.swap(rhs); }

  std::array<std::uintptr_t, 3> repr;
};
#endif // CXXBRIDGE1_RUST_STRING

#ifndef CXXBRIDGE1_RUST_SLICE
#define CXXBRIDGE1_RUST_SLICE
namespace detail {
template <bool>
struct copy_assignable_if {};

template <>
struct copy_assignable_if<false> {
  copy_assignable_if() noexcept = default;
  copy_assignable_if(const copy_assignable_if &) noexcept = default;
  copy_assignable_if &operator=(const copy_assignable_if &) &noexcept = delete;
  copy_assignable_if &operator=(copy_assignable_if &&) &noexcept = default;
};
} // namespace detail

template <typename T>
class Slice final
    : private detail::copy_assignable_if<std::is_const<T>::value> {
public:
  using value_type = T;

  Slice() noexcept;
  Slice(T *, std::size_t count) noexcept;

  template <typename C>
  explicit Slice(C& c) : Slice(c.data(), c.size()) {}

  Slice &operator=(const Slice<T> &) &noexcept = default;
  Slice &operator=(Slice<T> &&) &noexcept = default;

  T *data() const noexcept;
  std::size_t size() const noexcept;
  std::size_t length() const noexcept;
  bool empty() const noexcept;

  T &operator[](std::size_t n) const noexcept;
  T &at(std::size_t n) const;
  T &front() const noexcept;
  T &back() const noexcept;

  Slice(const Slice<T> &) noexcept = default;
  ~Slice() noexcept = default;

  class iterator;
  iterator begin() const noexcept;
  iterator end() const noexcept;

  void swap(Slice &) noexcept;

private:
  class uninit;
  Slice(uninit) noexcept;
  friend impl<Slice>;
  friend void sliceInit(void *, const void *, std::size_t) noexcept;
  friend void *slicePtr(const void *) noexcept;
  friend std::size_t sliceLen(const void *) noexcept;

  std::array<std::uintptr_t, 2> repr;
};

template <typename T>
class Slice<T>::iterator final {
public:
  using iterator_category = std::random_access_iterator_tag;
  using value_type = T;
  using difference_type = std::ptrdiff_t;
  using pointer = typename std::add_pointer<T>::type;
  using reference = typename std::add_lvalue_reference<T>::type;

  reference operator*() const noexcept;
  pointer operator->() const noexcept;
  reference operator[](difference_type) const noexcept;

  iterator &operator++() noexcept;
  iterator operator++(int) noexcept;
  iterator &operator--() noexcept;
  iterator operator--(int) noexcept;

  iterator &operator+=(difference_type) noexcept;
  iterator &operator-=(difference_type) noexcept;
  iterator operator+(difference_type) const noexcept;
  iterator operator-(difference_type) const noexcept;
  difference_type operator-(const iterator &) const noexcept;

  bool operator==(const iterator &) const noexcept;
  bool operator!=(const iterator &) const noexcept;
  bool operator<(const iterator &) const noexcept;
  bool operator<=(const iterator &) const noexcept;
  bool operator>(const iterator &) const noexcept;
  bool operator>=(const iterator &) const noexcept;

private:
  friend class Slice;
  void *pos;
  std::size_t stride;
};

template <typename T>
Slice<T>::Slice() noexcept {
  sliceInit(this, reinterpret_cast<void *>(align_of<T>()), 0);
}

template <typename T>
Slice<T>::Slice(T *s, std::size_t count) noexcept {
  assert(s != nullptr || count == 0);
  sliceInit(this,
            s == nullptr && count == 0
                ? reinterpret_cast<void *>(align_of<T>())
                : const_cast<typename std::remove_const<T>::type *>(s),
            count);
}

template <typename T>
T *Slice<T>::data() const noexcept {
  return reinterpret_cast<T *>(slicePtr(this));
}

template <typename T>
std::size_t Slice<T>::size() const noexcept {
  return sliceLen(this);
}

template <typename T>
std::size_t Slice<T>::length() const noexcept {
  return this->size();
}

template <typename T>
bool Slice<T>::empty() const noexcept {
  return this->size() == 0;
}

template <typename T>
T &Slice<T>::operator[](std::size_t n) const noexcept {
  assert(n < this->size());
  auto ptr = static_cast<char *>(slicePtr(this)) + size_of<T>() * n;
  return *reinterpret_cast<T *>(ptr);
}

template <typename T>
T &Slice<T>::at(std::size_t n) const {
  if (n >= this->size()) {
    panic<std::out_of_range>("rust::Slice index out of range");
  }
  return (*this)[n];
}

template <typename T>
T &Slice<T>::front() const noexcept {
  assert(!this->empty());
  return (*this)[0];
}

template <typename T>
T &Slice<T>::back() const noexcept {
  assert(!this->empty());
  return (*this)[this->size() - 1];
}

template <typename T>
typename Slice<T>::iterator::reference
Slice<T>::iterator::operator*() const noexcept {
  return *static_cast<T *>(this->pos);
}

template <typename T>
typename Slice<T>::iterator::pointer
Slice<T>::iterator::operator->() const noexcept {
  return static_cast<T *>(this->pos);
}

template <typename T>
typename Slice<T>::iterator::reference Slice<T>::iterator::operator[](
    typename Slice<T>::iterator::difference_type n) const noexcept {
  auto ptr = static_cast<char *>(this->pos) + this->stride * n;
  return *reinterpret_cast<T *>(ptr);
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator++() noexcept {
  this->pos = static_cast<char *>(this->pos) + this->stride;
  return *this;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator++(int) noexcept {
  auto ret = iterator(*this);
  this->pos = static_cast<char *>(this->pos) + this->stride;
  return ret;
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator--() noexcept {
  this->pos = static_cast<char *>(this->pos) - this->stride;
  return *this;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator--(int) noexcept {
  auto ret = iterator(*this);
  this->pos = static_cast<char *>(this->pos) - this->stride;
  return ret;
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator+=(
    typename Slice<T>::iterator::difference_type n) noexcept {
  this->pos = static_cast<char *>(this->pos) + this->stride * n;
  return *this;
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator-=(
    typename Slice<T>::iterator::difference_type n) noexcept {
  this->pos = static_cast<char *>(this->pos) - this->stride * n;
  return *this;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator+(
    typename Slice<T>::iterator::difference_type n) const noexcept {
  auto ret = iterator(*this);
  ret.pos = static_cast<char *>(this->pos) + this->stride * n;
  return ret;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator-(
    typename Slice<T>::iterator::difference_type n) const noexcept {
  auto ret = iterator(*this);
  ret.pos = static_cast<char *>(this->pos) - this->stride * n;
  return ret;
}

template <typename T>
typename Slice<T>::iterator::difference_type
Slice<T>::iterator::operator-(const iterator &other) const noexcept {
  auto diff = std::distance(static_cast<char *>(other.pos),
                            static_cast<char *>(this->pos));
  return diff / static_cast<typename Slice<T>::iterator::difference_type>(
                    this->stride);
}

template <typename T>
bool Slice<T>::iterator::operator==(const iterator &other) const noexcept {
  return this->pos == other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator!=(const iterator &other) const noexcept {
  return this->pos != other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator<(const iterator &other) const noexcept {
  return this->pos < other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator<=(const iterator &other) const noexcept {
  return this->pos <= other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator>(const iterator &other) const noexcept {
  return this->pos > other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator>=(const iterator &other) const noexcept {
  return this->pos >= other.pos;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::begin() const noexcept {
  iterator it;
  it.pos = slicePtr(this);
  it.stride = size_of<T>();
  return it;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::end() const noexcept {
  iterator it = this->begin();
  it.pos = static_cast<char *>(it.pos) + it.stride * this->size();
  return it;
}

template <typename T>
void Slice<T>::swap(Slice &rhs) noexcept {
  std::swap(*this, rhs);
}
#endif // CXXBRIDGE1_RUST_SLICE

#ifndef CXXBRIDGE1_RUST_BITCOPY_T
#define CXXBRIDGE1_RUST_BITCOPY_T
struct unsafe_bitcopy_t final {
  explicit unsafe_bitcopy_t() = default;
};
#endif // CXXBRIDGE1_RUST_BITCOPY_T

#ifndef CXXBRIDGE1_RUST_VEC
#define CXXBRIDGE1_RUST_VEC
template <typename T>
class Vec final {
public:
  using value_type = T;

  Vec() noexcept;
  Vec(std::initializer_list<T>);
  Vec(const Vec &);
  Vec(Vec &&) noexcept;
  ~Vec() noexcept;

  Vec &operator=(Vec &&) &noexcept;
  Vec &operator=(const Vec &) &;

  std::size_t size() const noexcept;
  bool empty() const noexcept;
  const T *data() const noexcept;
  T *data() noexcept;
  std::size_t capacity() const noexcept;

  const T &operator[](std::size_t n) const noexcept;
  const T &at(std::size_t n) const;
  const T &front() const noexcept;
  const T &back() const noexcept;

  T &operator[](std::size_t n) noexcept;
  T &at(std::size_t n);
  T &front() noexcept;
  T &back() noexcept;

  void reserve(std::size_t new_cap);
  void push_back(const T &value);
  void push_back(T &&value);
  template <typename... Args>
  void emplace_back(Args &&...args);
  void truncate(std::size_t len);
  void clear();

  using iterator = typename Slice<T>::iterator;
  iterator begin() noexcept;
  iterator end() noexcept;

  using const_iterator = typename Slice<const T>::iterator;
  const_iterator begin() const noexcept;
  const_iterator end() const noexcept;
  const_iterator cbegin() const noexcept;
  const_iterator cend() const noexcept;

  void swap(Vec &) noexcept;

  Vec(unsafe_bitcopy_t, const Vec &) noexcept;

private:
  void reserve_total(std::size_t new_cap) noexcept;
  void set_len(std::size_t len) noexcept;
  void drop() noexcept;

  friend void swap(Vec &lhs, Vec &rhs) noexcept { lhs.swap(rhs); }

  std::array<std::uintptr_t, 3> repr;
};

template <typename T>
Vec<T>::Vec(std::initializer_list<T> init) : Vec{} {
  this->reserve_total(init.size());
  std::move(init.begin(), init.end(), std::back_inserter(*this));
}

template <typename T>
Vec<T>::Vec(const Vec &other) : Vec() {
  this->reserve_total(other.size());
  std::copy(other.begin(), other.end(), std::back_inserter(*this));
}

template <typename T>
Vec<T>::Vec(Vec &&other) noexcept : repr(other.repr) {
  new (&other) Vec();
}

template <typename T>
Vec<T>::~Vec() noexcept {
  this->drop();
}

template <typename T>
Vec<T> &Vec<T>::operator=(Vec &&other) &noexcept {
  this->drop();
  this->repr = other.repr;
  new (&other) Vec();
  return *this;
}

template <typename T>
Vec<T> &Vec<T>::operator=(const Vec &other) & {
  if (this != &other) {
    this->drop();
    new (this) Vec(other);
  }
  return *this;
}

template <typename T>
bool Vec<T>::empty() const noexcept {
  return this->size() == 0;
}

template <typename T>
T *Vec<T>::data() noexcept {
  return const_cast<T *>(const_cast<const Vec<T> *>(this)->data());
}

template <typename T>
const T &Vec<T>::operator[](std::size_t n) const noexcept {
  assert(n < this->size());
  auto data = reinterpret_cast<const char *>(this->data());
  return *reinterpret_cast<const T *>(data + n * size_of<T>());
}

template <typename T>
const T &Vec<T>::at(std::size_t n) const {
  if (n >= this->size()) {
    panic<std::out_of_range>("rust::Vec index out of range");
  }
  return (*this)[n];
}

template <typename T>
const T &Vec<T>::front() const noexcept {
  assert(!this->empty());
  return (*this)[0];
}

template <typename T>
const T &Vec<T>::back() const noexcept {
  assert(!this->empty());
  return (*this)[this->size() - 1];
}

template <typename T>
T &Vec<T>::operator[](std::size_t n) noexcept {
  assert(n < this->size());
  auto data = reinterpret_cast<char *>(this->data());
  return *reinterpret_cast<T *>(data + n * size_of<T>());
}

template <typename T>
T &Vec<T>::at(std::size_t n) {
  if (n >= this->size()) {
    panic<std::out_of_range>("rust::Vec index out of range");
  }
  return (*this)[n];
}

template <typename T>
T &Vec<T>::front() noexcept {
  assert(!this->empty());
  return (*this)[0];
}

template <typename T>
T &Vec<T>::back() noexcept {
  assert(!this->empty());
  return (*this)[this->size() - 1];
}

template <typename T>
void Vec<T>::reserve(std::size_t new_cap) {
  this->reserve_total(new_cap);
}

template <typename T>
void Vec<T>::push_back(const T &value) {
  this->emplace_back(value);
}

template <typename T>
void Vec<T>::push_back(T &&value) {
  this->emplace_back(std::move(value));
}

template <typename T>
template <typename... Args>
void Vec<T>::emplace_back(Args &&...args) {
  auto size = this->size();
  this->reserve_total(size + 1);
  ::new (reinterpret_cast<T *>(reinterpret_cast<char *>(this->data()) +
                               size * size_of<T>()))
      T(std::forward<Args>(args)...);
  this->set_len(size + 1);
}

template <typename T>
void Vec<T>::clear() {
  this->truncate(0);
}

template <typename T>
typename Vec<T>::iterator Vec<T>::begin() noexcept {
  return Slice<T>(this->data(), this->size()).begin();
}

template <typename T>
typename Vec<T>::iterator Vec<T>::end() noexcept {
  return Slice<T>(this->data(), this->size()).end();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::begin() const noexcept {
  return this->cbegin();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::end() const noexcept {
  return this->cend();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::cbegin() const noexcept {
  return Slice<const T>(this->data(), this->size()).begin();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::cend() const noexcept {
  return Slice<const T>(this->data(), this->size()).end();
}

template <typename T>
void Vec<T>::swap(Vec &rhs) noexcept {
  using std::swap;
  swap(this->repr, rhs.repr);
}

template <typename T>
Vec<T>::Vec(unsafe_bitcopy_t, const Vec &bits) noexcept : repr(bits.repr) {}
#endif // CXXBRIDGE1_RUST_VEC

#ifndef CXXBRIDGE1_RUST_ERROR
#define CXXBRIDGE1_RUST_ERROR
class Error final : public std::exception {
public:
  Error(const Error &);
  Error(Error &&) noexcept;
  ~Error() noexcept override;

  Error &operator=(const Error &) &;
  Error &operator=(Error &&) &noexcept;

  const char *what() const noexcept override;

private:
  Error() noexcept = default;
  friend impl<Error>;
  const char *msg;
  std::size_t len;
};
#endif // CXXBRIDGE1_RUST_ERROR

#ifndef CXXBRIDGE1_IS_COMPLETE
#define CXXBRIDGE1_IS_COMPLETE
namespace detail {
namespace {
template <typename T, typename = std::size_t>
struct is_complete : std::false_type {};
template <typename T>
struct is_complete<T, decltype(sizeof(T))> : std::true_type {};
} // namespace
} // namespace detail
#endif // CXXBRIDGE1_IS_COMPLETE

#ifndef CXXBRIDGE1_LAYOUT
#define CXXBRIDGE1_LAYOUT
class layout {
  template <typename T>
  friend std::size_t size_of();
  template <typename T>
  friend std::size_t align_of();
  template <typename T>
  static typename std::enable_if<std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_size_of() {
    return T::layout::size();
  }
  template <typename T>
  static typename std::enable_if<!std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_size_of() {
    return sizeof(T);
  }
  template <typename T>
  static
      typename std::enable_if<detail::is_complete<T>::value, std::size_t>::type
      size_of() {
    return do_size_of<T>();
  }
  template <typename T>
  static typename std::enable_if<std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_align_of() {
    return T::layout::align();
  }
  template <typename T>
  static typename std::enable_if<!std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_align_of() {
    return alignof(T);
  }
  template <typename T>
  static
      typename std::enable_if<detail::is_complete<T>::value, std::size_t>::type
      align_of() {
    return do_align_of<T>();
  }
};

template <typename T>
std::size_t size_of() {
  return layout::size_of<T>();
}

template <typename T>
std::size_t align_of() {
  return layout::align_of<T>();
}
#endif // CXXBRIDGE1_LAYOUT

#ifndef CXXBRIDGE1_RELOCATABLE
#define CXXBRIDGE1_RELOCATABLE
namespace detail {
template <typename... Ts>
struct make_void {
  using type = void;
};

template <typename... Ts>
using void_t = typename make_void<Ts...>::type;

template <typename Void, template <typename...> class, typename...>
struct detect : std::false_type {};
template <template <typename...> class T, typename... A>
struct detect<void_t<T<A...>>, T, A...> : std::true_type {};

template <template <typename...> class T, typename... A>
using is_detected = detect<void, T, A...>;

template <typename T>
using detect_IsRelocatable = typename T::IsRelocatable;

template <typename T>
struct get_IsRelocatable
    : std::is_same<typename T::IsRelocatable, std::true_type> {};
} // namespace detail

template <typename T>
struct IsRelocatable
    : std::conditional<
          detail::is_detected<detail::detect_IsRelocatable, T>::value,
          detail::get_IsRelocatable<T>,
          std::integral_constant<
              bool, std::is_trivially_move_constructible<T>::value &&
                        std::is_trivially_destructible<T>::value>>::type {};
#endif // CXXBRIDGE1_RELOCATABLE

namespace repr {
struct PtrLen final {
  void *ptr;
  ::std::size_t len;
};
} // namespace repr

namespace detail {
class Fail final {
  ::rust::repr::PtrLen &throw$;
public:
  Fail(::rust::repr::PtrLen &throw$) noexcept : throw$(throw$) {}
  void operator()(char const *) noexcept;
  void operator()(std::string const &) noexcept;
};
} // namespace detail

template <typename T>
union ManuallyDrop {
  T value;
  ManuallyDrop(T &&value) : value(::std::move(value)) {}
  ~ManuallyDrop() {}
};

namespace {
template <>
class impl<Error> final {
public:
  static Error error(repr::PtrLen repr) noexcept {
    Error error;
    error.msg = static_cast<char const *>(repr.ptr);
    error.len = repr.len;
    return error;
  }
};

template <bool> struct deleter_if {
  template <typename T> void operator()(T *) {}
};

template <> struct deleter_if<true> {
  template <typename T> void operator()(T *ptr) { ptr->~T(); }
};
} // namespace
} // namespace cxxbridge1

namespace behavior {
class missing {};
missing trycatch(...);

template <typename Try, typename Fail>
static typename ::std::enable_if<
    ::std::is_same<decltype(trycatch(::std::declval<Try>(), ::std::declval<Fail>())),
                 missing>::value>::type
trycatch(Try &&func, Fail &&fail) noexcept try {
  func();
} catch (::std::exception const &e) {
  fail(e.what());
}
} // namespace behavior
} // namespace rust

namespace pdns {
  namespace rust {
    namespace web {
      namespace rec {
        using CredentialsHolder = ::pdns::rust::web::rec::CredentialsHolder;
        struct IncomingTLS;
        struct IncomingWSConfig;
        struct KeyValue;
        struct Request;
        struct Response;
      }
    }
  }
}

namespace pdns {
namespace rust {
namespace web {
namespace rec {
#ifndef CXXBRIDGE1_STRUCT_pdns$rust$web$rec$IncomingTLS
#define CXXBRIDGE1_STRUCT_pdns$rust$web$rec$IncomingTLS
struct IncomingTLS final {
  ::rust::String certificate;
  ::rust::String key;

  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$web$rec$IncomingTLS

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$web$rec$IncomingWSConfig
#define CXXBRIDGE1_STRUCT_pdns$rust$web$rec$IncomingWSConfig
struct IncomingWSConfig final {
  ::rust::Vec<::rust::String> addresses;
  ::pdns::rust::web::rec::IncomingTLS tls;

  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$web$rec$IncomingWSConfig

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$web$rec$KeyValue
#define CXXBRIDGE1_STRUCT_pdns$rust$web$rec$KeyValue
struct KeyValue final {
  ::rust::String key;
  ::rust::String value;

  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$web$rec$KeyValue

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$web$rec$Request
#define CXXBRIDGE1_STRUCT_pdns$rust$web$rec$Request
struct Request final {
  ::rust::Vec<::std::uint8_t> body;
  ::rust::String uri;
  ::rust::Vec<::pdns::rust::web::rec::KeyValue> vars;
  ::rust::Vec<::pdns::rust::web::rec::KeyValue> parameters;
  ::std::shared_ptr<::pdns::rust::misc::Logger> const &logger;

  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$web$rec$Request

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$web$rec$Response
#define CXXBRIDGE1_STRUCT_pdns$rust$web$rec$Response
struct Response final {
  ::std::uint16_t status;
  ::rust::Vec<::std::uint8_t> body;
  ::rust::Vec<::pdns::rust::web::rec::KeyValue> headers;

  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$web$rec$Response
} // namespace rec
} // namespace web
} // namespace rust
} // namespace pdns

static_assert(
    ::rust::IsRelocatable<::pdns::rust::misc::LogLevel>::value,
    "type pdns::rust::misc::LogLevel should be trivially move constructible and trivially destructible in C++ to be used as an argument of `serveweb` in Rust");

namespace pdns {
namespace rust {
namespace web {
namespace rec {
extern "C" {
::rust::repr::PtrLen pdns$rust$web$rec$cxxbridge1$serveweb(::rust::Vec<::pdns::rust::web::rec::IncomingWSConfig> const &incoming, ::pdns::rust::web::rec::CredentialsHolder *pwch, ::pdns::rust::web::rec::CredentialsHolder *apikeych, ::pdns::rust::misc::NetmaskGroup *acl, ::std::shared_ptr<::pdns::rust::misc::Logger> *logger, ::pdns::rust::misc::LogLevel *loglevel) noexcept;

bool pdns$rust$web$rec$cxxbridge1$CredentialsHolder$matches(::pdns::rust::web::rec::CredentialsHolder const &self, ::std::string const &str) noexcept {
  bool (::pdns::rust::web::rec::CredentialsHolder::*matches$)(::std::string const &) const = &::pdns::rust::web::rec::CredentialsHolder::matches;
  return (self.*matches$)(str);
}

::rust::repr::PtrLen pdns$rust$web$rec$cxxbridge1$apiDiscovery(::pdns::rust::web::rec::Request const &request, ::pdns::rust::web::rec::Response &response) noexcept {
  void (*apiDiscovery$)(::pdns::rust::web::rec::Request const &, ::pdns::rust::web::rec::Response &) = ::pdns::rust::web::rec::apiDiscovery;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        apiDiscovery$(request, response);
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen pdns$rust$web$rec$cxxbridge1$apiDiscoveryV1(::pdns::rust::web::rec::Request const &request, ::pdns::rust::web::rec::Response &response) noexcept {
  void (*apiDiscoveryV1$)(::pdns::rust::web::rec::Request const &, ::pdns::rust::web::rec::Response &) = ::pdns::rust::web::rec::apiDiscoveryV1;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        apiDiscoveryV1$(request, response);
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen pdns$rust$web$rec$cxxbridge1$apiServer(::pdns::rust::web::rec::Request const &request, ::pdns::rust::web::rec::Response &response) noexcept {
  void (*apiServer$)(::pdns::rust::web::rec::Request const &, ::pdns::rust::web::rec::Response &) = ::pdns::rust::web::rec::apiServer;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        apiServer$(request, response);
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen pdns$rust$web$rec$cxxbridge1$apiServerCacheFlush(::pdns::rust::web::rec::Request const &request, ::pdns::rust::web::rec::Response &response) noexcept {
  void (*apiServerCacheFlush$)(::pdns::rust::web::rec::Request const &, ::pdns::rust::web::rec::Response &) = ::pdns::rust::web::rec::apiServerCacheFlush;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        apiServerCacheFlush$(request, response);
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen pdns$rust$web$rec$cxxbridge1$apiServerConfig(::pdns::rust::web::rec::Request const &request, ::pdns::rust::web::rec::Response &response) noexcept {
  void (*apiServerConfig$)(::pdns::rust::web::rec::Request const &, ::pdns::rust::web::rec::Response &) = ::pdns::rust::web::rec::apiServerConfig;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        apiServerConfig$(request, response);
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen pdns$rust$web$rec$cxxbridge1$apiServerConfigAllowFromGET(::pdns::rust::web::rec::Request const &request, ::pdns::rust::web::rec::Response &response) noexcept {
  void (*apiServerConfigAllowFromGET$)(::pdns::rust::web::rec::Request const &, ::pdns::rust::web::rec::Response &) = ::pdns::rust::web::rec::apiServerConfigAllowFromGET;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        apiServerConfigAllowFromGET$(request, response);
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen pdns$rust$web$rec$cxxbridge1$apiServerConfigAllowFromPUT(::pdns::rust::web::rec::Request const &request, ::pdns::rust::web::rec::Response &response) noexcept {
  void (*apiServerConfigAllowFromPUT$)(::pdns::rust::web::rec::Request const &, ::pdns::rust::web::rec::Response &) = ::pdns::rust::web::rec::apiServerConfigAllowFromPUT;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        apiServerConfigAllowFromPUT$(request, response);
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen pdns$rust$web$rec$cxxbridge1$apiServerConfigAllowNotifyFromGET(::pdns::rust::web::rec::Request const &request, ::pdns::rust::web::rec::Response &response) noexcept {
  void (*apiServerConfigAllowNotifyFromGET$)(::pdns::rust::web::rec::Request const &, ::pdns::rust::web::rec::Response &) = ::pdns::rust::web::rec::apiServerConfigAllowNotifyFromGET;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        apiServerConfigAllowNotifyFromGET$(request, response);
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen pdns$rust$web$rec$cxxbridge1$apiServerConfigAllowNotifyFromPUT(::pdns::rust::web::rec::Request const &request, ::pdns::rust::web::rec::Response &response) noexcept {
  void (*apiServerConfigAllowNotifyFromPUT$)(::pdns::rust::web::rec::Request const &, ::pdns::rust::web::rec::Response &) = ::pdns::rust::web::rec::apiServerConfigAllowNotifyFromPUT;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        apiServerConfigAllowNotifyFromPUT$(request, response);
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen pdns$rust$web$rec$cxxbridge1$apiServerDetail(::pdns::rust::web::rec::Request const &requst, ::pdns::rust::web::rec::Response &response) noexcept {
  void (*apiServerDetail$)(::pdns::rust::web::rec::Request const &, ::pdns::rust::web::rec::Response &) = ::pdns::rust::web::rec::apiServerDetail;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        apiServerDetail$(requst, response);
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen pdns$rust$web$rec$cxxbridge1$apiServerRPZStats(::pdns::rust::web::rec::Request const &request, ::pdns::rust::web::rec::Response &response) noexcept {
  void (*apiServerRPZStats$)(::pdns::rust::web::rec::Request const &, ::pdns::rust::web::rec::Response &) = ::pdns::rust::web::rec::apiServerRPZStats;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        apiServerRPZStats$(request, response);
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen pdns$rust$web$rec$cxxbridge1$apiServerSearchData(::pdns::rust::web::rec::Request const &request, ::pdns::rust::web::rec::Response &response) noexcept {
  void (*apiServerSearchData$)(::pdns::rust::web::rec::Request const &, ::pdns::rust::web::rec::Response &) = ::pdns::rust::web::rec::apiServerSearchData;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        apiServerSearchData$(request, response);
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen pdns$rust$web$rec$cxxbridge1$apiServerStatistics(::pdns::rust::web::rec::Request const &requst, ::pdns::rust::web::rec::Response &response) noexcept {
  void (*apiServerStatistics$)(::pdns::rust::web::rec::Request const &, ::pdns::rust::web::rec::Response &) = ::pdns::rust::web::rec::apiServerStatistics;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        apiServerStatistics$(requst, response);
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen pdns$rust$web$rec$cxxbridge1$apiServerZoneDetailDELETE(::pdns::rust::web::rec::Request const &request, ::pdns::rust::web::rec::Response &response) noexcept {
  void (*apiServerZoneDetailDELETE$)(::pdns::rust::web::rec::Request const &, ::pdns::rust::web::rec::Response &) = ::pdns::rust::web::rec::apiServerZoneDetailDELETE;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        apiServerZoneDetailDELETE$(request, response);
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen pdns$rust$web$rec$cxxbridge1$apiServerZoneDetailGET(::pdns::rust::web::rec::Request const &request, ::pdns::rust::web::rec::Response &response) noexcept {
  void (*apiServerZoneDetailGET$)(::pdns::rust::web::rec::Request const &, ::pdns::rust::web::rec::Response &) = ::pdns::rust::web::rec::apiServerZoneDetailGET;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        apiServerZoneDetailGET$(request, response);
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen pdns$rust$web$rec$cxxbridge1$apiServerZoneDetailPUT(::pdns::rust::web::rec::Request const &request, ::pdns::rust::web::rec::Response &response) noexcept {
  void (*apiServerZoneDetailPUT$)(::pdns::rust::web::rec::Request const &, ::pdns::rust::web::rec::Response &) = ::pdns::rust::web::rec::apiServerZoneDetailPUT;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        apiServerZoneDetailPUT$(request, response);
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen pdns$rust$web$rec$cxxbridge1$apiServerZonesGET(::pdns::rust::web::rec::Request const &request, ::pdns::rust::web::rec::Response &response) noexcept {
  void (*apiServerZonesGET$)(::pdns::rust::web::rec::Request const &, ::pdns::rust::web::rec::Response &) = ::pdns::rust::web::rec::apiServerZonesGET;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        apiServerZonesGET$(request, response);
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen pdns$rust$web$rec$cxxbridge1$apiServerZonesPOST(::pdns::rust::web::rec::Request const &requst, ::pdns::rust::web::rec::Response &response) noexcept {
  void (*apiServerZonesPOST$)(::pdns::rust::web::rec::Request const &, ::pdns::rust::web::rec::Response &) = ::pdns::rust::web::rec::apiServerZonesPOST;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        apiServerZonesPOST$(requst, response);
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen pdns$rust$web$rec$cxxbridge1$jsonstat(::pdns::rust::web::rec::Request const &request, ::pdns::rust::web::rec::Response &response) noexcept {
  void (*jsonstat$)(::pdns::rust::web::rec::Request const &, ::pdns::rust::web::rec::Response &) = ::pdns::rust::web::rec::jsonstat;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        jsonstat$(request, response);
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen pdns$rust$web$rec$cxxbridge1$prometheusMetrics(::pdns::rust::web::rec::Request const &request, ::pdns::rust::web::rec::Response &response) noexcept {
  void (*prometheusMetrics$)(::pdns::rust::web::rec::Request const &, ::pdns::rust::web::rec::Response &) = ::pdns::rust::web::rec::prometheusMetrics;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        prometheusMetrics$(request, response);
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen pdns$rust$web$rec$cxxbridge1$serveStuff(::pdns::rust::web::rec::Request const &request, ::pdns::rust::web::rec::Response &response) noexcept {
  void (*serveStuff$)(::pdns::rust::web::rec::Request const &, ::pdns::rust::web::rec::Response &) = ::pdns::rust::web::rec::serveStuff;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        serveStuff$(request, response);
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}
} // extern "C"

void serveweb(::rust::Vec<::pdns::rust::web::rec::IncomingWSConfig> const &incoming, ::std::unique_ptr<::pdns::rust::web::rec::CredentialsHolder> pwch, ::std::unique_ptr<::pdns::rust::web::rec::CredentialsHolder> apikeych, ::std::unique_ptr<::pdns::rust::misc::NetmaskGroup> acl, ::std::shared_ptr<::pdns::rust::misc::Logger> logger, ::pdns::rust::misc::LogLevel loglevel) {
  ::rust::ManuallyDrop<::std::shared_ptr<::pdns::rust::misc::Logger>> logger$(::std::move(logger));
  ::rust::ManuallyDrop<::pdns::rust::misc::LogLevel> loglevel$(::std::move(loglevel));
  ::rust::repr::PtrLen error$ = pdns$rust$web$rec$cxxbridge1$serveweb(incoming, pwch.release(), apikeych.release(), acl.release(), &logger$.value, &loglevel$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}
} // namespace rec
} // namespace web
} // namespace rust
} // namespace pdns

extern "C" {
void cxxbridge1$rust_vec$pdns$rust$web$rec$IncomingWSConfig$new(::rust::Vec<::pdns::rust::web::rec::IncomingWSConfig> const *ptr) noexcept;
void cxxbridge1$rust_vec$pdns$rust$web$rec$IncomingWSConfig$drop(::rust::Vec<::pdns::rust::web::rec::IncomingWSConfig> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$pdns$rust$web$rec$IncomingWSConfig$len(::rust::Vec<::pdns::rust::web::rec::IncomingWSConfig> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$pdns$rust$web$rec$IncomingWSConfig$capacity(::rust::Vec<::pdns::rust::web::rec::IncomingWSConfig> const *ptr) noexcept;
::pdns::rust::web::rec::IncomingWSConfig const *cxxbridge1$rust_vec$pdns$rust$web$rec$IncomingWSConfig$data(::rust::Vec<::pdns::rust::web::rec::IncomingWSConfig> const *ptr) noexcept;
void cxxbridge1$rust_vec$pdns$rust$web$rec$IncomingWSConfig$reserve_total(::rust::Vec<::pdns::rust::web::rec::IncomingWSConfig> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$pdns$rust$web$rec$IncomingWSConfig$set_len(::rust::Vec<::pdns::rust::web::rec::IncomingWSConfig> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$pdns$rust$web$rec$IncomingWSConfig$truncate(::rust::Vec<::pdns::rust::web::rec::IncomingWSConfig> *ptr, ::std::size_t len) noexcept;

static_assert(::rust::detail::is_complete<::pdns::rust::web::rec::CredentialsHolder>::value, "definition of CredentialsHolder is required");
static_assert(sizeof(::std::unique_ptr<::pdns::rust::web::rec::CredentialsHolder>) == sizeof(void *), "");
static_assert(alignof(::std::unique_ptr<::pdns::rust::web::rec::CredentialsHolder>) == alignof(void *), "");
void cxxbridge1$unique_ptr$pdns$rust$web$rec$CredentialsHolder$null(::std::unique_ptr<::pdns::rust::web::rec::CredentialsHolder> *ptr) noexcept {
  ::new (ptr) ::std::unique_ptr<::pdns::rust::web::rec::CredentialsHolder>();
}
void cxxbridge1$unique_ptr$pdns$rust$web$rec$CredentialsHolder$raw(::std::unique_ptr<::pdns::rust::web::rec::CredentialsHolder> *ptr, ::pdns::rust::web::rec::CredentialsHolder *raw) noexcept {
  ::new (ptr) ::std::unique_ptr<::pdns::rust::web::rec::CredentialsHolder>(raw);
}
::pdns::rust::web::rec::CredentialsHolder const *cxxbridge1$unique_ptr$pdns$rust$web$rec$CredentialsHolder$get(::std::unique_ptr<::pdns::rust::web::rec::CredentialsHolder> const &ptr) noexcept {
  return ptr.get();
}
::pdns::rust::web::rec::CredentialsHolder *cxxbridge1$unique_ptr$pdns$rust$web$rec$CredentialsHolder$release(::std::unique_ptr<::pdns::rust::web::rec::CredentialsHolder> &ptr) noexcept {
  return ptr.release();
}
void cxxbridge1$unique_ptr$pdns$rust$web$rec$CredentialsHolder$drop(::std::unique_ptr<::pdns::rust::web::rec::CredentialsHolder> *ptr) noexcept {
  ::rust::deleter_if<::rust::detail::is_complete<::pdns::rust::web::rec::CredentialsHolder>::value>{}(ptr);
}

void cxxbridge1$rust_vec$pdns$rust$web$rec$KeyValue$new(::rust::Vec<::pdns::rust::web::rec::KeyValue> const *ptr) noexcept;
void cxxbridge1$rust_vec$pdns$rust$web$rec$KeyValue$drop(::rust::Vec<::pdns::rust::web::rec::KeyValue> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$pdns$rust$web$rec$KeyValue$len(::rust::Vec<::pdns::rust::web::rec::KeyValue> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$pdns$rust$web$rec$KeyValue$capacity(::rust::Vec<::pdns::rust::web::rec::KeyValue> const *ptr) noexcept;
::pdns::rust::web::rec::KeyValue const *cxxbridge1$rust_vec$pdns$rust$web$rec$KeyValue$data(::rust::Vec<::pdns::rust::web::rec::KeyValue> const *ptr) noexcept;
void cxxbridge1$rust_vec$pdns$rust$web$rec$KeyValue$reserve_total(::rust::Vec<::pdns::rust::web::rec::KeyValue> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$pdns$rust$web$rec$KeyValue$set_len(::rust::Vec<::pdns::rust::web::rec::KeyValue> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$pdns$rust$web$rec$KeyValue$truncate(::rust::Vec<::pdns::rust::web::rec::KeyValue> *ptr, ::std::size_t len) noexcept;
} // extern "C"

namespace rust {
inline namespace cxxbridge1 {
template <>
Vec<::pdns::rust::web::rec::IncomingWSConfig>::Vec() noexcept {
  cxxbridge1$rust_vec$pdns$rust$web$rec$IncomingWSConfig$new(this);
}
template <>
void Vec<::pdns::rust::web::rec::IncomingWSConfig>::drop() noexcept {
  return cxxbridge1$rust_vec$pdns$rust$web$rec$IncomingWSConfig$drop(this);
}
template <>
::std::size_t Vec<::pdns::rust::web::rec::IncomingWSConfig>::size() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$web$rec$IncomingWSConfig$len(this);
}
template <>
::std::size_t Vec<::pdns::rust::web::rec::IncomingWSConfig>::capacity() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$web$rec$IncomingWSConfig$capacity(this);
}
template <>
::pdns::rust::web::rec::IncomingWSConfig const *Vec<::pdns::rust::web::rec::IncomingWSConfig>::data() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$web$rec$IncomingWSConfig$data(this);
}
template <>
void Vec<::pdns::rust::web::rec::IncomingWSConfig>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$pdns$rust$web$rec$IncomingWSConfig$reserve_total(this, new_cap);
}
template <>
void Vec<::pdns::rust::web::rec::IncomingWSConfig>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$pdns$rust$web$rec$IncomingWSConfig$set_len(this, len);
}
template <>
void Vec<::pdns::rust::web::rec::IncomingWSConfig>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$pdns$rust$web$rec$IncomingWSConfig$truncate(this, len);
}
template <>
Vec<::pdns::rust::web::rec::KeyValue>::Vec() noexcept {
  cxxbridge1$rust_vec$pdns$rust$web$rec$KeyValue$new(this);
}
template <>
void Vec<::pdns::rust::web::rec::KeyValue>::drop() noexcept {
  return cxxbridge1$rust_vec$pdns$rust$web$rec$KeyValue$drop(this);
}
template <>
::std::size_t Vec<::pdns::rust::web::rec::KeyValue>::size() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$web$rec$KeyValue$len(this);
}
template <>
::std::size_t Vec<::pdns::rust::web::rec::KeyValue>::capacity() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$web$rec$KeyValue$capacity(this);
}
template <>
::pdns::rust::web::rec::KeyValue const *Vec<::pdns::rust::web::rec::KeyValue>::data() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$web$rec$KeyValue$data(this);
}
template <>
void Vec<::pdns::rust::web::rec::KeyValue>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$pdns$rust$web$rec$KeyValue$reserve_total(this, new_cap);
}
template <>
void Vec<::pdns::rust::web::rec::KeyValue>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$pdns$rust$web$rec$KeyValue$set_len(this, len);
}
template <>
void Vec<::pdns::rust::web::rec::KeyValue>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$pdns$rust$web$rec$KeyValue$truncate(this, len);
}
} // namespace cxxbridge1
} // namespace rust
