use crate::article_view::Webview;
use crate::enclosure::EnclosurePreview;
use crate::gobject_models::GEnclosure;
use crate::image_dialog::ImageDialog;
use crate::main_window::MainWindow;
use glib::{clone, subclass::prelude::*, subclass::*};
use gtk4::{Accessible, Buildable, CompositeTemplate, ConstraintTarget, Widget, prelude::*, subclass::prelude::*};
use libadwaita::{Bin, Carousel, prelude::*, subclass::prelude::*};

mod imp {
    use super::*;

    #[derive(Debug, Default, CompositeTemplate)]
    #[template(file = "data/resources/ui_templates/enclosures/image-carousel.blp")]
    pub struct ImageCarousel {
        #[template_child]
        pub carousel: TemplateChild<Carousel>,
    }

    #[glib::object_subclass]
    impl ObjectSubclass for ImageCarousel {
        const NAME: &'static str = "ImageCarousel";
        type Type = super::ImageCarousel;
        type ParentType = Bin;

        fn class_init(klass: &mut Self::Class) {
            klass.bind_template();
            klass.bind_template_callbacks();
        }

        fn instance_init(obj: &InitializingObject<Self>) {
            obj.init_template();
        }
    }

    impl ObjectImpl for ImageCarousel {}

    impl WidgetImpl for ImageCarousel {}

    impl BinImpl for ImageCarousel {}

    #[gtk4::template_callbacks]
    impl ImageCarousel {
        #[template_callback]
        fn on_clicked(&self) {
            let pos = self.carousel.position() as u32;
            let Ok(preview) = self.carousel.nth_page(pos).downcast::<EnclosurePreview>() else {
                tracing::warn!("not an image widget");
                return;
            };
            let enclosure = preview.enclosure();
            let dialog = ImageDialog::new_url(&enclosure.article_id().into(), enclosure.url().as_str());

            Webview::instance().set_image_dialog_visible(true);
            dialog.connect_closed(|_dialog| {
                Webview::instance().set_image_dialog_visible(false);
            });
            dialog.present(Some(&MainWindow::instance()))
        }
    }
}

glib::wrapper! {
    pub struct ImageCarousel(ObjectSubclass<imp::ImageCarousel>)
        @extends Widget, Bin,
        @implements Accessible, Buildable, ConstraintTarget;
}

impl ImageCarousel {
    pub fn set_images(&self, image_enclosures: &[GEnclosure]) {
        let imp = self.imp();

        // clear
        let mut child = imp.carousel.first_child();
        while let Some(widget) = child {
            child = widget.next_sibling();
            imp.carousel.remove(&widget);
        }

        for enclosure in image_enclosures {
            let preview = EnclosurePreview::new(enclosure);
            preview.set_hexpand(true);
            preview.connect_remove(clone!(
                #[weak]
                imp,
                move |preview| imp.carousel.remove(preview)
            ));

            imp.carousel.append(&preview);
        }
    }
}
