#ifdef GET_ASITagsList_DECL
const ASITag *lookupASITagByEncoding(uint8_t Encoding);
const ASITag *lookupASITagByAltName(StringRef AltName);
const ASITag *lookupASITagByName(StringRef Name);
#endif

#ifdef GET_ASITagsList_IMPL
constexpr ASITag ASITagsList[] = {
  { "ASI_N", "ASI_NUCLEUS", 0x4 }, // 0
  { "ASI_N_L", "ASI_NUCLEUS_LITTLE", 0xC }, // 1
  { "ASI_AIUP", "ASI_AS_IF_USER_PRIMARY", 0x10 }, // 2
  { "ASI_AIUS", "ASI_AS_IF_USER_SECONDARY", 0x11 }, // 3
  { "ASI_AIUP_L", "ASI_AS_IF_USER_PRIMARY_LITTLE", 0x18 }, // 4
  { "ASI_AIUS_L", "ASI_AS_IF_USER_SECONDARY_LITTLE", 0x19 }, // 5
  { "ASI_P", "ASI_PRIMARY", 0x80 }, // 6
  { "ASI_S", "ASI_SECONDARY", 0x81 }, // 7
  { "ASI_PNF", "ASI_PRIMARY_NOFAULT", 0x82 }, // 8
  { "ASI_SNF", "ASI_SECONDARY_NOFAULT", 0x83 }, // 9
  { "ASI_P_L", "ASI_PRIMARY_LITTLE", 0x88 }, // 10
  { "ASI_S_L", "ASI_SECONDARY_LITTLE", 0x89 }, // 11
  { "ASI_PNF_L", "ASI_PRIMARY_NOFAULT_LITTLE", 0x8A }, // 12
  { "ASI_SNF_L", "ASI_SECONDARY_NOFAULT_LITTLE", 0x8B }, // 13
 };

const ASITag *lookupASITagByEncoding(uint8_t Encoding) {
  struct KeyType {
    uint8_t Encoding;
  };
  KeyType Key = {Encoding};
  struct Comp {
    bool operator()(const ASITag &LHS, const KeyType &RHS) const {
      if (LHS.Encoding < RHS.Encoding)
        return true;
      if (LHS.Encoding > RHS.Encoding)
        return false;
      return false;
    }
  };
  auto Table = ArrayRef(ASITagsList);
  auto Idx = std::lower_bound(Table.begin(), Table.end(), Key, Comp());
  if (Idx == Table.end() ||
      Key.Encoding != Idx->Encoding)
    return nullptr;

  return &*Idx;
}

const ASITag *lookupASITagByAltName(StringRef AltName) {
  struct IndexType {
    const char * AltName;
    unsigned _index;
  };
  static const struct IndexType Index[] = {
    { "ASI_AS_IF_USER_PRIMARY", 2 },
    { "ASI_AS_IF_USER_PRIMARY_LITTLE", 4 },
    { "ASI_AS_IF_USER_SECONDARY", 3 },
    { "ASI_AS_IF_USER_SECONDARY_LITTLE", 5 },
    { "ASI_NUCLEUS", 0 },
    { "ASI_NUCLEUS_LITTLE", 1 },
    { "ASI_PRIMARY", 6 },
    { "ASI_PRIMARY_LITTLE", 10 },
    { "ASI_PRIMARY_NOFAULT", 8 },
    { "ASI_PRIMARY_NOFAULT_LITTLE", 12 },
    { "ASI_SECONDARY", 7 },
    { "ASI_SECONDARY_LITTLE", 11 },
    { "ASI_SECONDARY_NOFAULT", 9 },
    { "ASI_SECONDARY_NOFAULT_LITTLE", 13 },
  };

  struct KeyType {
    std::string AltName;
  };
  KeyType Key = {AltName.upper()};
  struct Comp {
    bool operator()(const IndexType &LHS, const KeyType &RHS) const {
      int CmpAltName = StringRef(LHS.AltName).compare(RHS.AltName);
      if (CmpAltName < 0) return true;
      if (CmpAltName > 0) return false;
      return false;
    }
  };
  auto Table = ArrayRef(Index);
  auto Idx = std::lower_bound(Table.begin(), Table.end(), Key, Comp());
  if (Idx == Table.end() ||
      Key.AltName != Idx->AltName)
    return nullptr;

  return &ASITagsList[Idx->_index];
}

const ASITag *lookupASITagByName(StringRef Name) {
  struct IndexType {
    const char * Name;
    unsigned _index;
  };
  static const struct IndexType Index[] = {
    { "ASI_AIUP", 2 },
    { "ASI_AIUP_L", 4 },
    { "ASI_AIUS", 3 },
    { "ASI_AIUS_L", 5 },
    { "ASI_N", 0 },
    { "ASI_N_L", 1 },
    { "ASI_P", 6 },
    { "ASI_PNF", 8 },
    { "ASI_PNF_L", 12 },
    { "ASI_P_L", 10 },
    { "ASI_S", 7 },
    { "ASI_SNF", 9 },
    { "ASI_SNF_L", 13 },
    { "ASI_S_L", 11 },
  };

  struct KeyType {
    std::string Name;
  };
  KeyType Key = {Name.upper()};
  struct Comp {
    bool operator()(const IndexType &LHS, const KeyType &RHS) const {
      int CmpName = StringRef(LHS.Name).compare(RHS.Name);
      if (CmpName < 0) return true;
      if (CmpName > 0) return false;
      return false;
    }
  };
  auto Table = ArrayRef(Index);
  auto Idx = std::lower_bound(Table.begin(), Table.end(), Key, Comp());
  if (Idx == Table.end() ||
      Key.Name != Idx->Name)
    return nullptr;

  return &ASITagsList[Idx->_index];
}
#endif

#ifdef GET_PrefetchTagsList_DECL
const PrefetchTag *lookupPrefetchTagByEncoding(uint8_t Encoding);
const PrefetchTag *lookupPrefetchTagByName(StringRef Name);
#endif

#ifdef GET_PrefetchTagsList_IMPL
constexpr PrefetchTag PrefetchTagsList[] = {
  { "n_reads", 0x0 }, // 0
  { "one_read", 0x1 }, // 1
  { "n_writes", 0x2 }, // 2
  { "one_write", 0x3 }, // 3
  { "page", 0x4 }, // 4
  { "unified", 0x11 }, // 5
  { "n_reads_strong", 0x14 }, // 6
  { "one_read_strong", 0x15 }, // 7
  { "n_writes_strong", 0x16 }, // 8
  { "one_write_strong", 0x17 }, // 9
 };

const PrefetchTag *lookupPrefetchTagByEncoding(uint8_t Encoding) {
  struct KeyType {
    uint8_t Encoding;
  };
  KeyType Key = {Encoding};
  struct Comp {
    bool operator()(const PrefetchTag &LHS, const KeyType &RHS) const {
      if (LHS.Encoding < RHS.Encoding)
        return true;
      if (LHS.Encoding > RHS.Encoding)
        return false;
      return false;
    }
  };
  auto Table = ArrayRef(PrefetchTagsList);
  auto Idx = std::lower_bound(Table.begin(), Table.end(), Key, Comp());
  if (Idx == Table.end() ||
      Key.Encoding != Idx->Encoding)
    return nullptr;

  return &*Idx;
}

const PrefetchTag *lookupPrefetchTagByName(StringRef Name) {
  struct IndexType {
    const char * Name;
    unsigned _index;
  };
  static const struct IndexType Index[] = {
    { "N_READS", 0 },
    { "N_READS_STRONG", 6 },
    { "N_WRITES", 2 },
    { "N_WRITES_STRONG", 8 },
    { "ONE_READ", 1 },
    { "ONE_READ_STRONG", 7 },
    { "ONE_WRITE", 3 },
    { "ONE_WRITE_STRONG", 9 },
    { "PAGE", 4 },
    { "UNIFIED", 5 },
  };

  struct KeyType {
    std::string Name;
  };
  KeyType Key = {Name.upper()};
  struct Comp {
    bool operator()(const IndexType &LHS, const KeyType &RHS) const {
      int CmpName = StringRef(LHS.Name).compare(RHS.Name);
      if (CmpName < 0) return true;
      if (CmpName > 0) return false;
      return false;
    }
  };
  auto Table = ArrayRef(Index);
  auto Idx = std::lower_bound(Table.begin(), Table.end(), Key, Comp());
  if (Idx == Table.end() ||
      Key.Name != Idx->Name)
    return nullptr;

  return &PrefetchTagsList[Idx->_index];
}
#endif

#undef GET_ASITagsList_DECL
#undef GET_ASITagsList_IMPL
#undef GET_PrefetchTagsList_DECL
#undef GET_PrefetchTagsList_IMPL
