/*
 * Copyright (C) 2014-2025 CZ.NIC
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations including
 * the two.
 */

#pragma once

#include <QDialog>
#include <QItemSelection>
#include <QMap>
#include <QPair>
#include <QTimer>

#include "src/datovka_shared/identifiers/account_id.h" /* AcntId is used in slots. Q_MOC_INCLUDE */
#include "src/datovka_shared/utility/network_speed.h"
#include "src/delegates/progress_delegate.h"
#include "src/identifiers/account_id_db.h"
#include "src/json/message_id.h" /* Json::MsgId2List is used in slots. Q_MOC_INCLUDE */
#include "src/models/message_timestamp_listing_model.h"
#include "src/models/progress_proxy_model.h"
#include "src/models/sort_filter_proxy_model.h"

class DbContainer; /* Forward declaration. */
class IsdsSessions; /* Forward declaration. */
class MainWindow; /* Forward declaration. */
class TaskArchiveIsdsDocument; /* Forward declaration. */
class TaskArchiveIsdsDocumentFile; /* Forward declaration. */
class TimestampDb; /* Forward declaration. */

namespace Ui {
	class DlgTimestampRenewal;
}

/*!
 * @brief Timestamp renewal dialogue.
 */
class DlgTimestampRenewal : public QDialog {
	Q_OBJECT
public:
	/*!
	 * @brief Tabs which this dialogue has.
	 */
	enum TabIndexes {
		TAB_TRACKED = 0,
		TAB_EXTERNAL,

		MAX_TABNUM /* Maximal number of columns (convenience value). */
	};

	/*!
	 * @brief Constructor.
	 *
	 * @param[in] acntIdDbList List of available accounts.
	 * @param[in.out] timestampDb Timestamp database.
	 * @param[in,out] msgDbs Message database container.
	 * @param[in,out] sessions Session container.
	 * @param[in] mw Main application window.
	 * @param[in] parent Parent widget.
	 * @param[in] flags Window flags.
	 */
	explicit DlgTimestampRenewal(const QList<AcntIdDb> &acntIdDbList,
	    TimestampDb *timestampDb,
	    DbContainer *msgDbs, IsdsSessions *sessions, MainWindow *mw,
	    QWidget *parent = Q_NULLPTR,
	    Qt::WindowFlags flags = Qt::WindowFlags());

	/*!
	 * @brief Destructor.
	 */
	~DlgTimestampRenewal(void);

protected:
	/*!
	 * @brief Check window geometry and set table columns width.
	 *
	 * @param[in] event Widget show event.
	 */
	virtual
	void showEvent(QShowEvent *event) Q_DECL_OVERRIDE;

Q_SIGNALS:
	/*!
	 * @brief Signals that a message was selected should be focused
	 *     elsewhere.
	 *
	 * @param[in] username Username identifying an account.
	 * @param[in] testing True if this is an testing account.
	 * @param[in] dmId Message identifier.
	 * @param[in] year Message delivery year.
	 * @param[in] msgType Message type.
	 */
	void focusSelectedMsg(AcntId acntId, qint64 dmId, QString year,
	    int msgType);

public Q_SLOTS:
	/*!
	 * @brief Check if a worker is working on the background and show
	 *     a dialogue whether the user wants to close this dialogue.
	 *
	 * @note Used instead of closeEvent().
	 *     https://stackoverflow.com/questions/13129380/show-a-dialog-before-closing-the-program-in-qt
	 *
	 * @param[in] resultCode Dialog result code.
	 */
	virtual
	void done(int resultCode) Q_DECL_OVERRIDE;

private Q_SLOTS:
	/*!
	 * @brief Renews timestamps for selected messages.
	 */
	void renewTimestamps(void);

/* Window state change. */
	/*!
	 * @brief Updates message selection.
	 *
	 * @param[in] selected Newly selected indexes.
	 * @param[in] deselect Deselected indexes.
	 */
	void updateMessageSelection(const QItemSelection &selected,
	    const QItemSelection &deselected = QItemSelection());

/* Mouse on widget clicks. */
	/*!
	 * @brief Enable timestamp tracking for selected messages.
	 *
	 * @param[in] enable Whether to enable/disable tracking.
	 */
	void toggleTimestampTracking(bool enable);

/* Mouse on view clicks. */
	/*!
	 * @brief Generates a context menu depending on the item and selection
	 *     it was invoked on.
	 *
	 * @param[in] point Invocation position.
	 */
	void viewTrackedMessagesListContextMenu(const QPoint &point);

	/*!
	 * @brief Emit ID of message entry.
	 *
	 * @param[in] index Model index.
	 */
	void messageItemDoubleClicked(const QModelIndex &index);

/* Filter-field-related actions. */
	/*!
	 * @brief Filter listed messages.
	 */
	void filterMessages(const QString &text);

	/*!
	 * @brief Show message list filter.
	 */
	void showMessageFilter(void);

	/*!
	 * @brief Hide message list filter.
	 */
	void hideMessageFilter(void);

/* Handle database signals. */
	/*!
	 * @brief Update data when timestamp tracking enabled.
	 *
	 * @param[in] msgIds Message identifiers.
	 * @param[in] acntId Account identifier.
	 */
	void watchIdentificationsInserted(const Json::MsgId2List &msgIds,
	    const AcntId &acntId);

	/*!
	 * @brief Update data when timestamp tracking disabled.
	 *
	 * @param[in] msgIds Message identifiers.
	 */
	void watchIdentificationsDeleted(const Json::MsgId2List &msgIds);

	/*!
	 * @brief Update data when timestamp data updated.
	 *
	 * @param[in] values Updated values.
	 */
	void watchMsgTimestampDataUpdated(const TstValidityHash &values);

/* External tab functions. */
	/*!
	 * @brief Choose source file.
	 */
	void selectExternalSrcZfo(void);

	/*!
	 * @brief View source file content.
	 */
	void viewExternalSrcZfo(void);

	/*!
	 * @brief Choose target file.
	 */
	void selectExternalTgtZfo(void);

	/*!
	 * @brief Renews timestamps for source file.
	 */
	void renewExternalTimestamps(void);

/* Handle worker signals. */
	/*!
	 * @brief Set upload progress.
	 */
	void watchUploadProgress(const AcntId &acntId,
	    const QString &transactId, qint64 uploadTotal,
	    qint64 uploadCurrent);

	/*!
	 * @brief Hide upload progress.
	 */
	void watchUploadProgressFinished(const AcntId &acntId,
	    const QString &transactId, int result, const QString &resultDesc,
	    const QVariant &resultVal);

	/*!
	 * @brief Set download progress.
	 */
	void watchDownloadProgress(const AcntId &acntId,
	    const QString &transactId, const MsgId &msgId,
	    qint64 downloadTotal, qint64 downloadCurrent);

	/*!
	 * @brief Set result into timestamp model.
	 */
	void watchDownloadProgressFinished(const AcntId &acntId,
	    const QString &transactId, const MsgId &msgId,
	    int result, const QString &resultDesc);

private:
	/*!
	 * @brief Initialise message search dialogue.
	 */
	void initTimestampWindow(void);

	/*!
	 * @brief Connects menu actions to appropriate slots.
	 */
	void menuConnectActions(void);

	/*!
	 * @brief Connect database container signals to appropriate slots.
	 */
	void databaseConnectActions(void);

	/*!
	 * @brief Connect worker signals to appropriate slots.
	 */
	void workerConnectActions(void);

	/*!
	 * @brief Enable actions and other controls according to status.
	 */
	void updateActionActivation(void);

	/*!
	 * @brief Used to clear message list filter line via event filter.
	 *
	 * @param[in] dlgPtr Pointer to dialogue window.
	 */
	static
	void clearFilterLineViaFilter(QObject *dlgPtr);

	/*!
	 * @brief Used to view message list filter line via event filter.
	 *
	 * @param[in] dlgPtr Pointer to dialogue window.
	 */
	static
	void showFilterLineViaFilter(QObject *dlgPtr);

	/*!
	 * @brief Used to hide message list filter line via event filter.
	 *
	 * @param[in] dlgPtr Pointer to dialogue window.
	 */
	static
	void hideFilterLineViaFilter(QObject *dlgPtr);

	Ui::DlgTimestampRenewal *m_ui; /*!< UI generated from UI file. */

	QList<int> m_modelRowSelection; /*!< Holds the message selection. */

	TimestampDb *m_timestampDb; /*!< Timestamp database. */
	DbContainer *m_msgDbs; /*!< Message databases. */
	IsdsSessions *m_sessions; /*!< ISDS session container. */
	MainWindow *m_mw; /*!< Pointer to main window. */

	QTimer m_updateTimer; /*!< Update curent time in m_timestampTableModel. */

	SortFilterProxyModel m_sortProxyModel; /*!<
	                                        * Used for timestamp
	                                        * sorting and filtering.
	                                        */

	ProgressBarProxyModel m_progressProxyModel; /*!< Generates progress-bars on top of the download model. */
	ProgressDelegate m_progressDelegate; /*!< Draws progress bars into the download table view. */

	MsgTstListingModel m_timestampTableModel; /*!< Model timestamp expiration table. */

	QString m_dfltFilerLineStyleSheet; /*!< Used to remember default line edit style. */

	QMap<QString, QPair<AcntId, MsgId> > m_trackedTransactionMap; /*!< Created transations for tracked messages. */
	QString m_externalTransaction; /*!< Single external file transaction. */
	QMap<QString, TaskArchiveIsdsDocument *> m_tasks; /*!< Created tasks. */
	TaskArchiveIsdsDocumentFile *m_externalTask; /*!< Created task. */

	QMap<QString, NetworkSpeed> m_speeds; /* Network speeds. */

	const QList<AcntIdDb> m_acntIdDbList; /*!< Available accounts.*/

	QSize m_dfltSize; /*!< Remembered dialogue default size. */
};
