//
// pandora: syd's Dump Inspector & Profile Writer
// pandora.rs: Main entry point
//
// Copyright (c) 2021, 2024, 2025, 2026 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0

#[cfg(all(feature = "rust-dns", feature = "system-dns"))]
compile_error!("features `rust-dns` and `system-dns` are mutually exclusive");

#[cfg(feature = "rust-dns")]
use std::convert::TryFrom;
#[cfg(feature = "rust-dns")]
use std::net::{SocketAddr, SocketAddrV4, SocketAddrV6, UdpSocket};
use std::{
    borrow::Cow,
    cmp::Ordering,
    env,
    ffi::OsString,
    fmt,
    fmt::Write as FmtWrite,
    fs::{metadata, File, OpenOptions},
    hash::{Hash, Hasher},
    io::{self, stderr, stdin, BufRead, BufReader, Read, Write as IoWrite},
    iter::FromIterator,
    net::IpAddr,
    os::{
        fd::{AsFd, AsRawFd},
        unix::ffi::{OsStrExt, OsStringExt},
    },
    path::{Path, PathBuf},
    process::{exit, Command, ExitCode},
    str::FromStr,
    sync::{Arc, Mutex},
    thread,
    time::Duration,
};

use btoi::btoi;
use console::style;
use crc::{Crc, CRC_32_ISO_HDLC, CRC_64_ECMA_182};
use data_encoding::{HEXLOWER, HEXLOWER_PERMISSIVE};
use indicatif::{MultiProgress, ProgressBar, ProgressStyle};
use libc::pid_t;
use memchr::arch::all::is_equal;
use nix::{
    errno::Errno,
    fcntl::{fcntl, openat2, FcntlArg, FdFlag, OFlag, OpenHow, ResolveFlag, AT_FDCWD},
    sys::{
        signal::{kill, sigprocmask, SigmaskHow, Signal},
        signalfd::SigSet,
        stat::Mode,
    },
    unistd::{pipe2, Pid},
};
use patricia_tree::StringPatriciaSet;
use rayon::{
    iter::{IntoParallelRefIterator, ParallelIterator},
    ThreadPoolBuilder,
};
use serde::{
    de::{MapAccess, SeqAccess, Visitor},
    Deserialize, Deserializer, Serialize, Serializer,
};
use sha1::Sha1;
use sha3::{Digest, Sha3_256, Sha3_384, Sha3_512};

const PKG_NAME: &str = "pandora";
const PKG_VERSION: &str = env!("CARGO_PKG_VERSION");
const PKG_DESCRIPTION: &str = env!("CARGO_PKG_DESCRIPTION");
const PKG_AUTHORS: &str = env!("CARGO_PKG_AUTHORS");
const PKG_LICENSE: &str = env!("CARGO_PKG_LICENSE");

const CAPSET_LPATH: &[&str] = &["walk", "stat", "chdir", "notify"];
const CAPSET_RPATH: &[&str] = &["read", "readdir"];
const CAPSET_WPATH: &[&str] = &["write", "truncate"];
const CAPSET_CPATH: &[&str] = &["create", "delete", "rename"];
const CAPSET_DPATH: &[&str] = &["mkbdev", "mkcdev"];
const CAPSET_SPATH: &[&str] = &["mkfifo", "symlink"];
const CAPSET_TPATH: &[&str] = &["mkdir", "rmdir"];
const CAPSET_FOWN: &[&str] = &["chown", "chgrp"];
const CAPSET_FATTR: &[&str] = &["chmod", "chattr", "utime"];
const CAPSET_NET: &[&str] = &["net/bind", "net/connect", "net/sendfd"];
const CAPSET_INET: &[&str] = &["net/bind", "net/connect"];
const CAPSET_BNET: &[&str] = &["net/bind"];
const CAPSET_CNET: &[&str] = &["net/connect"];
const CAPSET_SNET: &[&str] = &["net/sendfd"];

const CAPSETS: &[(&str, &[&str])] = &[
    ("lpath", CAPSET_LPATH),
    ("rpath", CAPSET_RPATH),
    ("wpath", CAPSET_WPATH),
    ("cpath", CAPSET_CPATH),
    ("dpath", CAPSET_DPATH),
    ("spath", CAPSET_SPATH),
    ("tpath", CAPSET_TPATH),
    ("fown", CAPSET_FOWN),
    ("fattr", CAPSET_FATTR),
    ("net", CAPSET_NET),
    ("inet", CAPSET_INET),
    ("bnet", CAPSET_BNET),
    ("cnet", CAPSET_CNET),
    ("snet", CAPSET_SNET),
];

const CAP_ORDER: &[&str] = &[
    // aliases (keep before base)
    "lpath",
    "rpath",
    "wpath",
    "cpath",
    "dpath",
    "spath",
    "tpath",
    "fown",
    "fattr",
    "net",
    "inet",
    "bnet",
    "cnet",
    "snet",
    // base capabilities
    "fs",
    "walk",
    "stat",
    "read",
    "write",
    "exec",
    "create",
    "delete",
    "rename",
    "symlink",
    "truncate",
    "chdir",
    "readdir",
    "mkdir",
    "rmdir",
    "chown",
    "chgrp",
    "chmod",
    "chattr",
    "chroot",
    "notify",
    "utime",
    "mkbdev",
    "mkcdev",
    "mkfifo",
    "mktemp",
    "net/bind",
    "net/connect",
    "net/sendfd",
];

#[expect(clippy::disallowed_types)]
type PandoraMap<K, V> = std::collections::HashMap<K, V, ahash::RandomState>;
#[expect(clippy::disallowed_types)]
type PandoraSet<K> = std::collections::HashSet<K, ahash::RandomState>;

// write! which returns Errno.
macro_rules! w {
    ($out:expr) => {
        retry_on_intr(|| write!($out).or(Err(Errno::EIO)))
    };
    ($out:expr, $($arg:tt)*) => {
        retry_on_intr(|| write!($out, $($arg)*).or(Err(Errno::EIO)))
    };
}

// writeln! which returns Errno.
macro_rules! wln {
    ($out:expr) => {
        retry_on_intr(|| writeln!($out).map_err(err2no))
    };
    ($out:expr, $($arg:tt)*) => {
        retry_on_intr(|| writeln!($out, $($arg)*).map_err(err2no))
    };
}

#[derive(Clone, Debug)]
enum Capability {
    One(String),
    Some(PandoraSet<String>),
}

impl PartialEq for Capability {
    fn eq(&self, other: &Self) -> bool {
        match (self, other) {
            (Capability::One(s1), Capability::One(s2)) => s1 == s2,
            (Capability::One(s1), Capability::Some(set2)) => set2.len() == 1 && set2.contains(s1),
            (Capability::Some(set1), Capability::One(s2)) => set1.len() == 1 && set1.contains(s2),
            (Capability::Some(set1), Capability::Some(set2)) => {
                set1.len() == set2.len() && set1.is_subset(set2)
            }
        }
    }
}

impl Eq for Capability {}

impl Hash for Capability {
    fn hash<H: Hasher>(&self, state: &mut H) {
        match self {
            Capability::One(s) => {
                s.hash(state);
            }
            Capability::Some(set) => {
                for item in set {
                    item.hash(state);
                }
            }
        }
    }
}

impl Serialize for Capability {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: Serializer,
    {
        match self {
            Self::One(s) => s.serialize(serializer),
            Self::Some(set) => set.serialize(serializer),
        }
    }
}

/// A custom visitor to handle "either a String or an array of strings."
struct CapabilityVisitor;

impl<'de> Visitor<'de> for CapabilityVisitor {
    type Value = Capability;

    /// A human-friendly description of what this visitor expects.
    fn expecting(&self, formatter: &mut fmt::Formatter) -> fmt::Result {
        formatter.write_str("either a string or an array for Capability")
    }

    /// If Serde sees a string, we interpret that as `Capability::One(...)`.
    fn visit_str<E>(self, value: &str) -> Result<Self::Value, E>
    where
        E: serde::de::Error,
    {
        Ok(Capability::One(value.to_owned()))
    }

    /// If Serde sees a sequence, we interpret that as `Capability::Some(HashSet<...>)`.
    fn visit_seq<A>(self, mut seq: A) -> Result<Self::Value, A::Error>
    where
        A: SeqAccess<'de>,
    {
        let mut set = PandoraSet::default();
        while let Some(elem) = seq.next_element::<String>()? {
            set.insert(elem);
        }
        Ok(Capability::Some(set))
    }
}

impl<'de> Deserialize<'de> for Capability {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: Deserializer<'de>,
    {
        deserializer.deserialize_any(CapabilityVisitor)
    }
}

#[derive(Clone, Debug)]
enum IoctlEntry {
    Name(String),
    Val(u64),
}

impl<'de> Deserialize<'de> for IoctlEntry {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: Deserializer<'de>,
    {
        struct IoctlEntryVisitor;

        impl<'de> Visitor<'de> for IoctlEntryVisitor {
            type Value = IoctlEntry;

            fn expecting(&self, formatter: &mut fmt::Formatter) -> fmt::Result {
                formatter.write_str("integer or string")
            }

            fn visit_u64<E>(self, value: u64) -> Result<Self::Value, E>
            where
                E: serde::de::Error,
            {
                Ok(IoctlEntry::Val(value))
            }

            fn visit_i64<E>(self, value: i64) -> Result<Self::Value, E>
            where
                E: serde::de::Error,
            {
                if value < 0 {
                    Err(E::custom("negative integer for ioctl"))
                } else {
                    Ok(IoctlEntry::Val(value as u64))
                }
            }

            fn visit_str<E>(self, value: &str) -> Result<Self::Value, E>
            where
                E: serde::de::Error,
            {
                Ok(IoctlEntry::Name(value.to_owned()))
            }

            fn visit_string<E>(self, value: String) -> Result<Self::Value, E>
            where
                E: serde::de::Error,
            {
                Ok(IoctlEntry::Name(value))
            }
        }

        deserializer.deserialize_any(IoctlEntryVisitor)
    }
}

#[derive(Clone, Debug)]
enum Access {
    Path {
        ctx: String,
        cap: Option<Capability>,
        fs: Option<String>,
        path: String,
    },
    Ioctl {
        ctx: String,
        ctl: Vec<IoctlEntry>,
    },
    InetAddr {
        ctx: String,
        cap: Option<Capability>,
        addr: String,
    },
    UnixAddr {
        ctx: String,
        cap: Option<Capability>,
        unix: String,
    },
    Run {
        cmd: String,
        argv: Vec<String>,
        time: String,
    },
    Exit {
        code: u8,
    },
    Any {
        _ctx: String,
    },
}

impl<'de> Deserialize<'de> for Access {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: Deserializer<'de>,
    {
        struct AccessVisitor;

        impl<'de> Visitor<'de> for AccessVisitor {
            type Value = Access;

            fn expecting(&self, f: &mut fmt::Formatter) -> fmt::Result {
                write!(f, "a map matching one of the Access enum variants")
            }

            fn visit_map<M>(self, mut map: M) -> Result<Access, M::Error>
            where
                M: MapAccess<'de>,
            {
                // Temporary storage for all possible fields:
                let mut ctx: Option<String> = None;
                let mut cap: Option<Option<Capability>> = None;

                let mut path: Option<String> = None;
                let mut args: Option<Vec<u64>> = None;
                let mut addr: Option<String> = None;
                let mut unix_: Option<String> = None;

                let mut ctl: Option<Vec<IoctlEntry>> = None;
                let mut fs: Option<String> = None;
                let mut cmd: Option<String> = None;
                let mut argv: Option<Vec<String>> = None;
                let mut time_: Option<String> = None;

                let mut op: Option<String> = None;
                let mut code: Option<u8> = None;

                // Read the incoming map field by field.
                while let Some(key) = map.next_key::<String>()? {
                    match key.as_str() {
                        "ctx" => {
                            if ctx.is_some() {
                                return Err(serde::de::Error::duplicate_field("ctx"));
                            }
                            ctx = map.next_value()?;
                        }
                        "cap" => {
                            if cap.is_some() {
                                return Err(serde::de::Error::duplicate_field("cap"));
                            }
                            cap = map.next_value()?;
                        }
                        "args" => {
                            if args.is_some() {
                                return Err(serde::de::Error::duplicate_field("args"));
                            }
                            args = map.next_value()?;
                        }
                        "path" => {
                            if path.is_some() {
                                return Err(serde::de::Error::duplicate_field("path"));
                            }
                            path = map.next_value()?;
                        }
                        "addr" => {
                            if addr.is_some() {
                                return Err(serde::de::Error::duplicate_field("addr"));
                            }
                            addr = map.next_value()?;
                        }
                        "unix" => {
                            if unix_.is_some() {
                                return Err(serde::de::Error::duplicate_field("unix"));
                            }
                            unix_ = map.next_value()?;
                        }
                        "ctl" => {
                            if ctl.is_some() {
                                return Err(serde::de::Error::duplicate_field("ctl"));
                            }
                            ctl = map.next_value()?;
                        }
                        "fs" => {
                            if fs.is_some() {
                                return Err(serde::de::Error::duplicate_field("fs"));
                            }
                            fs = map.next_value()?;
                        }
                        "cmd" => {
                            if cmd.is_some() {
                                return Err(serde::de::Error::duplicate_field("cmd"));
                            }
                            cmd = map.next_value()?;
                        }
                        "argv" => {
                            if argv.is_some() {
                                return Err(serde::de::Error::duplicate_field("argv"));
                            }
                            argv = map.next_value()?;
                        }
                        "time" => {
                            if time_.is_some() {
                                return Err(serde::de::Error::duplicate_field("time"));
                            }
                            time_ = map.next_value()?;
                        }
                        "op" => {
                            if op.is_some() {
                                return Err(serde::de::Error::duplicate_field("op"));
                            }
                            op = map.next_value()?;
                        }
                        "code" => {
                            if code.is_some() {
                                return Err(serde::de::Error::duplicate_field("code"));
                            }
                            code = map.next_value()?;
                        }
                        _ => {
                            // If there are unknown fields, we ignore.
                            let _ignored: serde::de::IgnoredAny = map.next_value()?;
                        }
                    }
                }

                // We need `ctx` in *every* variant, so ensure we have it
                let ctx = ctx.ok_or_else(|| serde::de::Error::missing_field("ctx"))?;

                // `cap` was stored as Some(...) or None => unwrap it
                let cap = cap.unwrap_or(None);

                // Now decide which variant to build based on which fields we have:
                if let Some(path) = path {
                    Ok(Access::Path { ctx, cap, fs, path })
                } else if let Some(ctl) = ctl {
                    Ok(Access::Ioctl { ctx, ctl })
                } else if let Some(addr) = addr {
                    Ok(Access::InetAddr { ctx, cap, addr })
                } else if let Some(unix) = unix_ {
                    Ok(Access::UnixAddr { ctx, cap, unix })
                } else if let (Some(cmd), Some(argv), Some(time)) = (cmd, argv, time_) {
                    Ok(Access::Run { cmd, argv, time })
                } else if let (Some(_op), Some(code)) = (op, code) {
                    Ok(Access::Exit { code })
                } else {
                    // If none of those fields were found,
                    // we assume it's the `Any` variant.
                    Ok(Access::Any { _ctx: ctx })
                }
            }
        }

        // Kick off the deserialization by asking for a map.
        deserializer.deserialize_map(AccessVisitor)
    }
}

fn io_to_errno(e: std::io::Error) -> Errno {
    e.raw_os_error().map(Errno::from_raw).unwrap_or(Errno::EIO)
}

// Performs a reverse DNS lookup for the given IP address,
// returning a hostname or an error.
#[expect(clippy::cast_possible_truncation)]
fn lookup_addr(addr: IpAddr) -> Result<String, Errno> {
    #[cfg(feature = "system-dns")]
    {
        dns_lookup::lookup_addr(&addr).map_err(io_to_errno)
    }

    #[cfg(feature = "rust-dns")]
    {
        // Read system DNS configuration (max 4KB).
        let f = File::open("/etc/resolv.conf").map_err(io_to_errno)?;
        let mut buf = Vec::with_capacity(4096);
        f.take(4096).read_to_end(&mut buf).map_err(io_to_errno)?;
        let conf = resolv_conf::Config::parse(&buf).map_err(|_| Errno::EINVAL)?;

        // Pick the first nameserver (IPv4/IPv6).
        let ns = conf
            .nameservers
            .iter()
            .find_map(|ns| -> Option<SocketAddr> {
                match ns {
                    resolv_conf::ScopedIp::V4(ipv4) => {
                        Some(SocketAddr::V4(SocketAddrV4::new(*ipv4, 53)))
                    }
                    resolv_conf::ScopedIp::V6(ipv6, _scope) => {
                        Some(SocketAddr::V6(SocketAddrV6::new(*ipv6, 53, 0, 0)))
                    }
                }
            })
            .ok_or(Errno::ENOENT)?;

        // Construct the reverse pointer name.
        let ptr_name = match addr {
            IpAddr::V4(ip) => {
                let octets = ip.octets();
                format!(
                    "{}.{}.{}.{}.in-addr.arpa",
                    octets[3], octets[2], octets[1], octets[0]
                )
            }
            IpAddr::V6(ip) => {
                let octets = ip.octets();
                let mut s = String::with_capacity(72);
                use std::fmt::Write;
                for octet in octets.iter().rev() {
                    let _ = write!(s, "{:x}.{:x}.", octet & 0x0f, (octet >> 4) & 0x0f);
                }
                s + "ip6.arpa"
            }
        };

        // Send Query using simple-dns.
        let qname = simple_dns::Name::new(&ptr_name).map_err(|_| Errno::EINVAL)?;
        let question = simple_dns::Question::new(
            qname.clone(),
            simple_dns::QTYPE::try_from(12).expect("QTYPE::PTR"),
            simple_dns::QCLASS::try_from(1).expect("QCLASS::IN"),
            false,
        );
        let mut packet = simple_dns::Packet::new_query(0);
        packet.questions.push(question);

        // Set Recursion Desired (RD) bit (Byte 2, Bit 0).
        let mut packet_bytes = packet.build_bytes_vec().map_err(|_| Errno::EINVAL)?;
        if packet_bytes.len() > 2 {
            packet_bytes[2] |= 1;
        }

        let socket = UdpSocket::bind("0.0.0.0:0").map_err(io_to_errno)?;
        socket
            .set_read_timeout(Some(Duration::from_secs(2)))
            .map_err(io_to_errno)?;
        socket.connect(ns).map_err(io_to_errno)?;
        socket.send(&packet_bytes).map_err(io_to_errno)?;

        let mut recv_buf = [0u8; 1024];
        let amt = socket.recv(&mut recv_buf).map_err(io_to_errno)?;

        let response = simple_dns::Packet::parse(&recv_buf[..amt]).map_err(|_| Errno::EIO)?;

        if response.answers.is_empty() {
            return Err(Errno::ENOENT);
        }

        // Extract PTR record.
        for answer in response.answers {
            match answer.rdata {
                simple_dns::rdata::RData::PTR(ptr) => return Ok(ptr.0.to_string()),
                _ => continue,
            }
        }

        Err(Errno::ENOENT)
    }
}

/// Defines hash functions supported by Syd.
///
/// Replicated from `syd::hash::HashAlgorithm` to avoid depending on Syd.
#[derive(Debug, Clone, Copy)]
enum HashAlgorithm {
    /// Crc32
    Crc32,
    /// Crc64
    Crc64,
    /// Md5
    Md5,
    /// SHA-1
    Sha1,
    /// SHA3-256
    Sha256,
    /// SHA3-384
    Sha384,
    /// SHA3-512
    Sha512,
}

impl FromStr for HashAlgorithm {
    type Err = Errno;

    fn from_str(value: &str) -> Result<Self, Self::Err> {
        Ok(match value {
            "sha3-512" => Self::Sha512,
            "sha3-384" => Self::Sha384,
            "sha3-256" => Self::Sha256,
            "sha1" => Self::Sha1,
            "md5" => Self::Md5,
            "crc64" => Self::Crc64,
            "crc32" => Self::Crc32,
            _ => return Err(Errno::EINVAL),
        })
    }
}

impl std::fmt::Display for HashAlgorithm {
    fn fmt(&self, f: &mut std::fmt::Formatter) -> std::fmt::Result {
        let name = match self {
            Self::Sha512 => "sha3-512",
            Self::Sha384 => "sha3-384",
            Self::Sha256 => "sha2-256",
            Self::Sha1 => "sha1",
            Self::Md5 => "md5",
            Self::Crc64 => "crc64",
            Self::Crc32 => "crc32",
        };
        write!(f, "{name}")
    }
}

// Define SYSLOG_ACTION_* constants.
// libc does not have to define these.
const SYSLOG_ACTION_READ_ALL: libc::c_int = 3;
const SYSLOG_ACTION_SIZE_BUFFER: libc::c_int = 10;

struct Syslog;

impl Syslog {
    fn open() -> io::Result<io::Cursor<Vec<u8>>> {
        let mut buf = vec![0u8; Self::capacity()?];
        loop {
            return match Syslog.read(&mut buf) {
                Ok(n) => {
                    buf.truncate(n);
                    Ok(io::Cursor::new(buf))
                }
                Err(ref e) if e.kind() == std::io::ErrorKind::Interrupted => continue,
                Err(e) => return Err(e),
            };
        }
    }

    fn capacity() -> io::Result<usize> {
        // Retrieve the total size of the kernel log buffer.
        // SAFETY: There's no nix interface for this.
        loop {
            return match Errno::result(unsafe {
                libc::syscall(libc::SYS_syslog, SYSLOG_ACTION_SIZE_BUFFER)
            }) {
                Ok(n) => Ok(n as usize),
                Err(Errno::EINTR) => continue,
                Err(errno) => Err(io::Error::from_raw_os_error(errno as i32)),
            };
        }
    }
}

impl Read for Syslog {
    fn read(&mut self, buf: &mut [u8]) -> io::Result<usize> {
        // Perform the syslog syscall with SYSLOG_ACTION_READ_ALL.
        Errno::result(unsafe {
            libc::syscall(
                libc::SYS_syslog,
                SYSLOG_ACTION_READ_ALL,
                buf.as_mut_ptr() as *mut libc::c_char,
                buf.len(),
            )
        })
        .map(|size| size as usize)
        .map_err(|errno| io::Error::from_raw_os_error(errno as i32))
    }
}

/// Top-level subcommands.
enum MainOpts {
    /// "profile" subcommand
    Profile(ProfileOpts),
    /// "inspect" subcommand
    Inspect(InspectOpts),
    /// Top-level help
    Help,
    /// Top-level version
    Version,
}

/// Options for `profile` subcommand.
struct ProfileOpts {
    /// Syd binary
    bin: String,
    /// Repeated -s flags
    syd: Vec<String>,
    /// Output path
    output: String,
    /// Hash algorithm
    hash: HashAlgorithm,
    /// Path limit
    limit: u8,
    /// Optional timeout
    timeout: Option<Duration>,
    /// Thread count
    threads: usize,
    /// Positional subcommand
    cmd: Vec<OsString>,
}

/// Options for `inspect` subcommand.
struct InspectOpts {
    /// Input path
    input: String,
    /// Output path
    output: String,
    /// Hash algorithm
    hash: HashAlgorithm,
    /// Path limit
    limit: u8,
    /// Threads
    threads: usize,
}

/// Internal map to track access control rules.
#[derive(Default)]
struct AccessMap {
    magic: PandoraMap<String, PandoraSet<String>>,
    force: PandoraSet<String>,
    fs_types: PandoraSet<String>,
    ioctl: PandoraSet<u64>,
    ioctl_names: StringPatriciaSet,
}

impl AccessMap {
    /// Parse each JSON line for relevant info.
    fn parse_json_line(&mut self, line: &str, path_limit: u8) -> Option<Access> {
        // SAFETY: Be permissive and skip all characters up until
        // the first '{'. This makes it easy to pipe dmesg(1) output
        // to pandora.
        let line = line.trim();
        let line = if let Some(start) = line.find('{') {
            &line[start.saturating_sub(1)..]
        } else {
            return None;
        };

        // SAFETY: Skip lines that cannot be parsed.
        // Warn about errors if PANDORA_DEBUG is set.
        let json = match serde_json::from_str(line) {
            Ok(json) => json,
            Err(err) => {
                if env::var_os("PANDORA_DEBUG").is_some() {
                    let _ = wln!(
                        stderr(),
                        "{} {}",
                        style("pandora:").bold().magenta(),
                        style("skip invalid JSON!").bold().yellow()
                    );
                    let _ = wln!(
                        stderr(),
                        "\t{} {}",
                        style("LINE:").bold().cyan(),
                        style(line).bold().red()
                    );
                    let _ = wln!(
                        stderr(),
                        "\t{} {}",
                        style("ERROR:").bold().cyan(),
                        style(err.to_string()).bold().red()
                    );
                }
                return None;
            }
        };

        match json {
            Access::Path {
                ctx, cap, fs, path, ..
            } if ctx == "access" => {
                let mut capabilities = match cap {
                    None => return None,
                    Some(Capability::One(cap)) => {
                        let mut caps = PandoraSet::default();
                        caps.insert(cap);
                        caps
                    }
                    Some(Capability::Some(caps)) => caps,
                };

                if capabilities.contains("exec") {
                    self.force.insert(path.clone());
                }

                if let Some(fs_type) = fs {
                    self.fs_types.insert(fs_type);
                    capabilities.remove("fs");
                    if capabilities.is_empty() {
                        return None;
                    }
                }

                let path = process_path(&path, path_limit).to_string();
                let pty = path == "/dev/pts/[0-9]*";
                self.magic
                    .entry(path.clone())
                    .or_default()
                    .extend(capabilities.clone());

                // Workaround for PTY listing.
                if pty {
                    let mut caps = PandoraSet::default();
                    caps.insert("readdir".to_string());
                    self.magic
                        .entry("/dev/pts".to_string())
                        .or_default()
                        .extend(caps);
                }
            }
            Access::UnixAddr { ctx, cap, unix, .. } if ctx == "access" => {
                let capabilities = match cap {
                    None => return None,
                    Some(Capability::One(cap)) => {
                        let mut caps = PandoraSet::default();
                        caps.insert(cap);
                        caps
                    }
                    Some(Capability::Some(caps)) => caps,
                };

                // We override the path limit for UNIX sockets for clarity.
                let unix = process_path(&unix, u8::MAX).to_string();
                self.magic.entry(unix).or_default().extend(capabilities);
            }
            Access::Ioctl { ctx, ctl, .. } if ctx == "access" => {
                for req in ctl {
                    match req {
                        IoctlEntry::Val(val) => {
                            self.ioctl.insert(val);
                        }
                        IoctlEntry::Name(name) => {
                            self.ioctl_names.insert(name);
                        }
                    }
                }
            }
            Access::InetAddr { ctx, cap, addr, .. } if ctx == "access" => {
                let capabilities = match cap {
                    None => return None,
                    Some(Capability::One(cap)) => {
                        let mut caps = PandoraSet::default();
                        caps.insert(cap);
                        caps
                    }
                    Some(Capability::Some(caps)) => caps,
                };
                self.magic.entry(addr).or_default().extend(capabilities);
            }
            Access::Run { .. } | Access::Exit { .. } => return Some(json),
            _ => {}
        };

        None
    }
}

fn command_profile(opts: ProfileOpts) -> Result<ExitCode, Errno> {
    if Path::new(&opts.output).exists() {
        wln!(
            stderr(),
            "{} error creating output file: `{}' already exists!",
            style("pandora:").bold().magenta(),
            style(opts.output.clone()).bold().yellow(),
        )
        .unwrap();
        return Ok(ExitCode::from(1));
    }

    let (fd_rd, fd_rw) = match pipe2(OFlag::O_CLOEXEC) {
        Ok((fd_rd, fd_rw)) => (fd_rd, fd_rw),
        Err(error) => {
            wln!(
                stderr(),
                "{} error creating pipe: {}!",
                style("pandora:").bold().magenta(),
                style(error.to_string()).bold().red()
            )
            .unwrap();
            return Ok(ExitCode::from(1));
        }
    };

    let mut syd = Command::new(opts.bin);

    // Pass write end of pipe fd with SYD_LOG_FD.
    set_cloexec(&fd_rw, false)?;
    let log_fd = fd_rw.as_raw_fd().to_string();
    syd.env("SYD_LOG_FD", &log_fd);
    if env::var_os("PANDORA_DEBUG").is_some() {
        let ino = fstatx(&fd_rw, STATX_INO).map(|stx| stx.stx_ino)?;
        let _ = wln!(
            stderr(),
            "{} syd log fd set to pipe {} with inode {}.",
            style("pandora:").bold().magenta(),
            style(&log_fd).bold().green(),
            style(&ino.to_string()).bold().cyan(),
        );
    }

    // Force line-oriented JSON with SYD_QUIET_TTY.
    syd.env("SYD_QUIET_TTY", "1");

    // Pass extra options to Syd.
    for opt in &opts.syd {
        syd.arg(format!("-{opt}"));
    }

    // Enable trace mode.
    // This is currently equivalent to -ptrace.
    syd.arg("-x");

    // Pass Command to execute.
    syd.arg("--").args(opts.cmd);

    // Spawn Syd.
    let mut child = syd.spawn().map_err(err2no)?;

    // Block SIGINT in the parent process.
    let mut mask = SigSet::empty();
    mask.add(Signal::SIGINT);
    sigprocmask(SigmaskHow::SIG_BLOCK, Some(&mask), None)?;

    if let Some(cmd_timeout) = opts.timeout {
        let pid = Pid::from_raw(child.id() as pid_t);
        thread::Builder::new()
            .name("pandora_mon".to_string())
            .spawn(move || {
                thread::sleep(cmd_timeout);
                let _ = wln!(
                    stderr(),
                    "{} {}",
                    style("pandora:").bold().magenta(),
                    style("Timeout expired, terminating process...")
                        .bold()
                        .yellow()
                );
                let _ = kill(pid, Signal::SIGKILL);
            })
            .map_err(err2no)?;
    }

    drop(fd_rw); // close the write end of the pipe.
    let input = Box::new(BufReader::new(File::from(fd_rd)));
    let result = do_inspect(
        input,
        &opts.output,
        opts.hash,
        opts.limit,
        opts.threads,
        Some(opts.syd),
    );

    // Wait for syd to exit.
    let _ = child.wait();

    // Return exit status.
    result
}

fn command_inspect(opts: InspectOpts) -> Result<ExitCode, Errno> {
    let input = open_input(&opts.input);
    do_inspect(
        input,
        &opts.output,
        opts.hash,
        opts.limit,
        opts.threads,
        None,
    )
}

/// Main function, returns `lexopt::Error` on errors.
fn main() -> Result<ExitCode, lexopt::Error> {
    // If PANDORA_NPROC isn't set, default to num_cpus.
    if env::var_os("PANDORA_NPROC").is_none() {
        env::set_var("PANDORA_NPROC", num_cpus::get().to_string());
    }

    let opts = parse_main_opts()?;

    let result = match opts {
        MainOpts::Help => {
            print_help_main();
            Ok(ExitCode::SUCCESS)
        }
        MainOpts::Version => {
            print_version();
            Ok(ExitCode::SUCCESS)
        }
        MainOpts::Profile(p) => command_profile(p),
        MainOpts::Inspect(p) => command_inspect(p),
    };

    match result {
        Ok(code) => Ok(code),
        Err(errno) => Ok(ExitCode::from(errno as i32 as u8)),
    }
}

/// Parse the top-level argument to see which subcommand (or help/version).
fn parse_main_opts() -> Result<MainOpts, lexopt::Error> {
    use lexopt::prelude::*;

    // Parse CLI options.
    //
    // Note, option parsing is POSIXly correct:
    // POSIX recommends that no more options are parsed after the first
    // positional argument. The other arguments are then all treated as
    // positional arguments.
    // See: https://pubs.opengroup.org/onlinepubs/9699919799/basedefs/V1_chap12.html#tag_12_02
    let mut parser = lexopt::Parser::from_env();
    let first_arg = parser.next()?;

    match first_arg {
        None => Ok(MainOpts::Help),
        Some(Short('h') | Long("help")) => Ok(MainOpts::Help),
        Some(Short('V') | Long("version")) => Ok(MainOpts::Version),
        Some(ref arg @ Value(ref cmd)) => match cmd.as_bytes() {
            b"profile" => {
                let prof = parse_profile_opts(parser)?;
                Ok(MainOpts::Profile(prof))
            }
            b"inspect" => {
                let insp = parse_inspect_opts(parser)?;
                Ok(MainOpts::Inspect(insp))
            }
            _ => Err(arg.clone().unexpected()),
        },
        Some(arg) => Err(arg.unexpected()),
    }
}

/// Parse "profile" subcommand options.
fn parse_profile_opts(mut parser: lexopt::Parser) -> Result<ProfileOpts, lexopt::Error> {
    use lexopt::prelude::*;

    let bin = parse_env_str("SYD_BIN", b"syd").map_err(|errno| lexopt::Error::ParsingFailed {
        value: "SYD_BIN".to_string(),
        error: Box::new(errno),
    })?;
    let output = parse_env_str("PANDORA_OUT", b"./pandora_out.syd-3").map_err(|errno| {
        lexopt::Error::ParsingFailed {
            value: "PANDORA_OUT".to_string(),
            error: Box::new(errno),
        }
    })?;
    let hash = parse_env_str("PANDORA_HASH", b"sha3-512")
        .map_err(|errno| lexopt::Error::ParsingFailed {
            value: "PANDORA_HASH".to_string(),
            error: Box::new(errno),
        })?
        .parse::<HashAlgorithm>()
        .map_err(|errno| lexopt::Error::ParsingFailed {
            value: "PANDORA_HASH".to_string(),
            error: Box::new(errno),
        })?;
    let limit = parse_env_u8("PANDORA_LIMIT", 3).map_err(|errno| lexopt::Error::ParsingFailed {
        value: "PANDORA_LIMIT".to_string(),
        error: Box::new(errno),
    })?;
    let timeout = {
        let val = env::var_os("PANDORA_TIMEOUT").unwrap_or_default();
        if val.is_empty() {
            None
        } else {
            Some(
                parse_os_u64(&val)
                    .map(Duration::from_secs)
                    .map_err(|errno| lexopt::Error::ParsingFailed {
                        value: "PANDORA_TIMEOUT".to_string(),
                        error: Box::new(errno),
                    })?,
            )
        }
    };
    let threads = parse_env_usize("PANDORA_NPROC", num_cpus::get()).map_err(|errno| {
        lexopt::Error::ParsingFailed {
            value: "PANDORA_NPROC".to_string(),
            error: Box::new(errno),
        }
    })?;

    let mut prof = ProfileOpts {
        bin,
        syd: Vec::new(),
        output,
        hash,
        limit,
        timeout,
        threads,
        cmd: vec![env::var_os("SYD_SHELL").unwrap_or(OsString::from("/bin/sh"))],
    };

    while let Some(arg) = parser.next()? {
        match arg {
            // -h => subcommand help
            Short('h') | Long("help") => {
                print_help_profile();
                std::process::exit(0);
            }
            // -V => version
            Short('V') | Long("version") => {
                print_version();
                std::process::exit(0);
            }
            // -H => hash algorithm.
            Short('H') => {
                prof.hash = parse_utf8_str(parser.value()?.as_bytes())
                    .map_err(|errno| lexopt::Error::ParsingFailed {
                        value: "-H".to_string(),
                        error: Box::new(errno),
                    })?
                    .parse::<HashAlgorithm>()
                    .map_err(|errno| lexopt::Error::ParsingFailed {
                        value: "-H".to_string(),
                        error: Box::new(errno),
                    })?;
            }
            // -x => bin
            Short('x') => {
                prof.bin = parse_utf8_str(parser.value()?.as_bytes()).map_err(|errno| {
                    lexopt::Error::ParsingFailed {
                        value: "-x".to_string(),
                        error: Box::new(errno),
                    }
                })?;
            }
            // -s => repeated Syd options
            Short('s') => {
                prof.syd
                    .push(parse_utf8_str(parser.value()?.as_bytes()).map_err(|errno| {
                        lexopt::Error::ParsingFailed {
                            value: "-s".to_string(),
                            error: Box::new(errno),
                        }
                    })?);
            }
            // -o => output
            Short('o') => {
                prof.output = parse_utf8_str(parser.value()?.as_bytes()).map_err(|errno| {
                    lexopt::Error::ParsingFailed {
                        value: "-s".to_string(),
                        error: Box::new(errno),
                    }
                })?;
            }
            // -l => limit
            Short('l') => {
                prof.limit = parse_u8(parser.value()?.as_bytes()).map_err(|errno| {
                    lexopt::Error::ParsingFailed {
                        value: "-l".to_string(),
                        error: Box::new(errno),
                    }
                })?;
            }
            // -t => timeout
            Short('t') => {
                prof.timeout = Some(
                    parse_u64(parser.value()?.as_bytes())
                        .map(Duration::from_secs)
                        .map_err(|errno| lexopt::Error::ParsingFailed {
                            value: "-t".to_string(),
                            error: Box::new(errno),
                        })?,
                );
            }
            // -T => threads
            Short('T') => {
                prof.threads = parse_usize(parser.value()?.as_bytes()).map_err(|errno| {
                    lexopt::Error::ParsingFailed {
                        value: "-T".to_string(),
                        error: Box::new(errno),
                    }
                })?;
            }
            // positional => belongs to cmd
            Value(prog) => {
                prof.cmd.clear();
                prof.cmd.push(prog);
                prof.cmd.extend(parser.raw_args()?);
            }
            _ => return Err(arg.unexpected()),
        }
    }

    Ok(prof)
}

/// Parse "inspect" subcommand options.
fn parse_inspect_opts(mut parser: lexopt::Parser) -> Result<InspectOpts, lexopt::Error> {
    use lexopt::prelude::*;

    let input =
        parse_env_str("PANDORA_IN", b"-").map_err(|errno| lexopt::Error::ParsingFailed {
            value: "PANDORA_IN".to_string(),
            error: Box::new(errno),
        })?;
    let output = parse_env_str("PANDORA_OUT", b"./pandora_out.syd-3").map_err(|errno| {
        lexopt::Error::ParsingFailed {
            value: "PANDORA_OUT".to_string(),
            error: Box::new(errno),
        }
    })?;
    let hash = parse_env_str("PANDORA_HASH", b"sha3-512")
        .map_err(|errno| lexopt::Error::ParsingFailed {
            value: "PANDORA_HASH".to_string(),
            error: Box::new(errno),
        })?
        .parse::<HashAlgorithm>()
        .map_err(|errno| lexopt::Error::ParsingFailed {
            value: "PANDORA_HASH".to_string(),
            error: Box::new(errno),
        })?;
    let limit = parse_env_u8("PANDORA_LIMIT", 3).map_err(|errno| lexopt::Error::ParsingFailed {
        value: "PANDORA_LIMIT".to_string(),
        error: Box::new(errno),
    })?;
    let threads = parse_env_usize("PANDORA_NPROC", num_cpus::get()).map_err(|errno| {
        lexopt::Error::ParsingFailed {
            value: "PANDORA_NPROC".to_string(),
            error: Box::new(errno),
        }
    })?;

    let mut io = InspectOpts {
        input,
        output,
        hash,
        limit,
        threads,
    };

    while let Some(arg) = parser.next()? {
        match arg {
            Short('h') | Long("help") => {
                print_help_inspect();
                std::process::exit(0);
            }
            Short('V') | Long("version") => {
                print_version();
                std::process::exit(0);
            }
            // -H => hash
            Short('H') => {
                io.hash = parse_utf8_str(parser.value()?.as_bytes())
                    .map_err(|errno| lexopt::Error::ParsingFailed {
                        value: "-H".to_string(),
                        error: Box::new(errno),
                    })?
                    .parse::<HashAlgorithm>()
                    .map_err(|errno| lexopt::Error::ParsingFailed {
                        value: "-H".to_string(),
                        error: Box::new(errno),
                    })?;
            }
            // -i => input
            Short('i') => {
                io.input = parse_utf8_str(parser.value()?.as_bytes()).map_err(|errno| {
                    lexopt::Error::ParsingFailed {
                        value: "-i".to_string(),
                        error: Box::new(errno),
                    }
                })?;
            }
            // -o => output
            Short('o') => {
                io.output = parse_utf8_str(parser.value()?.as_bytes()).map_err(|errno| {
                    lexopt::Error::ParsingFailed {
                        value: "-o".to_string(),
                        error: Box::new(errno),
                    }
                })?;
            }
            // -l => limit
            Short('l') => {
                io.limit = parse_u8(parser.value()?.as_bytes()).map_err(|errno| {
                    lexopt::Error::ParsingFailed {
                        value: "-l".to_string(),
                        error: Box::new(errno),
                    }
                })?;
            }
            // -T => threads
            Short('T') => {
                io.threads = parse_usize(parser.value()?.as_bytes()).map_err(|errno| {
                    lexopt::Error::ParsingFailed {
                        value: "-T".to_string(),
                        error: Box::new(errno),
                    }
                })?;
            }
            _ => return Err(arg.unexpected()),
        }
    }

    Ok(io)
}

/// Main function that reads logs, collects data, and writes the Syd profile.
fn do_inspect(
    input: Box<dyn std::io::BufRead>,
    output_path: &str,
    hash_function: HashAlgorithm,
    path_limit: u8,
    concurrency: usize,
    extra_options: Option<Vec<String>>,
) -> Result<ExitCode, Errno> {
    let mut access = AccessMap::default();
    let mut output = open_output(output_path);
    let mut program_command_line = vec![];
    let mut program_startup_time = "?".to_string();
    let mut program_invocation_name = "?".to_string();
    let mut program_exit_code: u8 = 0;

    for line in input.lines() {
        // Read line, continue on errors.
        let line = match line {
            Ok(line) => line,
            Err(_) => continue,
        };

        // Parse JSON.
        if let Some(json) = access.parse_json_line(&line, path_limit) {
            match json {
                Access::Run {
                    cmd, argv, time, ..
                } => {
                    program_invocation_name = cmd;
                    program_command_line = argv;
                    program_startup_time = time;
                }
                Access::Exit { code, .. } => {
                    program_exit_code = code;
                }
                _ => {}
            }
        }
    }

    let cmd = format!(
        "{program_invocation_name} {}",
        program_command_line.join(" ")
    );
    let cmd = cmd.trim_end();

    let m = MultiProgress::new();
    let _ = m.println(format!(
        "{} command `{}' exited with {}{}",
        style("pandora:").bold().magenta(),
        style(cmd).bold().yellow(),
        if program_exit_code == 0 {
            style("success".to_string()).bold().green()
        } else {
            style(format!("error {program_exit_code}")).bold().red()
        },
        if program_exit_code == 0 { "." } else { "!" },
    ));
    let _ = m.println(format!(
        "{} profile generation started.",
        style("pandora:").bold().magenta(),
    ));

    let mut config = Vec::new();
    if let Some(options) = extra_options {
        for option in options {
            match option.chars().next() {
                Some('m') => config.push(option[1..].to_string()),
                Some('P') => config.push(format!("include {}", &option[1..])),
                Some('p') => config.push(format!("include_profile {}", &option[1..])),
                _ => continue,
            }
        }
    }
    let config = config.join("\n");

    // Print out the magic header.
    wln!(
        &mut output,
        "#
# Syd profile generated by Pandora-{PKG_VERSION}
# PROG: {program_invocation_name}
# ARGS: {program_command_line:?}
# DATE: {program_startup_time}\n"
    )?;
    let _ = m.println(format!(
        "{} profile header written.",
        style("pandora:").bold().magenta(),
    ));

    // If user passed custom config lines, include them.
    if !config.is_empty() {
        wln!(
            &mut output,
            "###\n# User submitted options\n###\n{config}\n"
        )?;
        let _ = m.println(format!(
            "{} user submitted options written.",
            style("pandora").bold().magenta(),
        ));
    }

    wln!(&mut output, "###\n# Sandbox Rules\n###")?;

    // Print filesystem sandboxing rules.
    if !access.fs_types.is_empty() {
        let mut fs_types = access.fs_types.into_iter().collect::<Vec<_>>();
        fs_types.sort();
        wln!(&mut output, "allow/fs+{}\n", fs_types.join(","))?;
    }

    // Print out all the sandbox rules from `magic`.
    let mut list = Vec::from_iter(access.magic);
    // Alphabetical sort.
    list.sort_by_key(|(path, _)| path.to_string());
    // Sort reverse by Capability priority.
    list.sort_by_key(|(_, caps)| std::cmp::Reverse(caps.iter().map(cap2prio).sum::<usize>()));
    // Sort reverse by Capability count.
    list.sort_by_key(|(_, caps)| std::cmp::Reverse(caps.iter().count()));

    let len = list.len();
    let mut lastcap: Option<PandoraSet<String>> = None;
    for entry in &list {
        let elem = &entry.0;
        let mut caps = entry.1.clone();
        assert!(!caps.is_empty(), "Invalid rule!");

        if let Some(ref cap) = lastcap {
            if !cap.is_subset(&caps) {
                wln!(&mut output)?;
                lastcap = Some(caps.clone());
            }
        } else {
            lastcap = Some(caps.clone());
        }

        let mut done = false;
        if caps.contains("net/bind") {
            if ['/', '@', '!'].iter().any(|&c| elem.starts_with(c)) {
                // UNIX socket (domain, abstract or unnamed).
                wln!(&mut output, "allow/bnet+{}", elem)?;
            } else {
                // IPv{4,6} address
                let ip = elem.split('!').next().ok_or(Errno::EINVAL)?;
                let ip = ip.parse::<IpAddr>().or(Err(Errno::EINVAL))?;
                if let Ok(host) = lookup_addr(ip) {
                    wln!(&mut output, "# {host}")?;
                }
                wln!(&mut output, "allow/bnet+{}", elem)?;
            }
            done = true;
        }
        if caps.contains("net/connect") {
            if ['/', '@', '!'].iter().any(|&c| elem.starts_with(c)) {
                // UNIX socket (domain, abstract or unnamed).
                wln!(&mut output, "allow/cnet+{}", elem)?;
            } else {
                let ip = elem.split('!').next().ok_or(Errno::EINVAL)?;
                let ip = ip.parse::<IpAddr>().or(Err(Errno::EINVAL))?;
                if let Ok(host) = lookup_addr(ip) {
                    wln!(&mut output, "# {host}")?;
                }
                wln!(&mut output, "allow/cnet+{}", elem)?;
            }
            done = true;
        }
        if caps.contains("net/sendfd") {
            if ['/', '@', '!'].iter().any(|&c| elem.starts_with(c)) {
                // UNIX socket (domain, abstract or unnamed).
                wln!(&mut output, "allow/snet+{elem}")?;
            } else {
                unreachable!("BUG: invalid snet entry {:?}", entry);
            }
            caps.remove("net/sendfd");
            done = true;
        }

        if done {
            continue;
        }

        // Perform alias expansion.
        loop {
            let mut changed = false;

            for (alias, members) in CAPSETS {
                if caps.contains(*alias) {
                    continue;
                }
                if members.iter().all(|m| caps.contains(*m)) {
                    for m in *members {
                        changed |= caps.remove(*m);
                    }
                    changed |= caps.insert((*alias).to_string());
                }
            }

            if !changed {
                break;
            }
        }

        // Convert to vector and sort.
        let mut caps = caps.into_iter().collect::<Vec<_>>();
        caps.sort_by_key(cap2prio);

        wln!(&mut output, "allow/{}+{}", caps.join(","), elem)?;
    }

    let _ = wln!(
        stderr(),
        "{} generated {} rules.",
        style("pandora:").bold().magenta(),
        style(len.to_string()).bold().yellow(),
    );

    // Print out all ioctl requests.
    if !access.ioctl.is_empty() || !access.ioctl_names.is_empty() {
        wln!(&mut output, "\n###\n# Sandbox ioctl(2) Rules\n###")?;
        wln!(&mut output, "sandbox/ioctl:on\n")?;

        // Print unknown numeric ioctls.
        let mut nums = Vec::with_capacity(access.ioctl.len());
        for &n in access.ioctl.iter() {
            nums.push(n);
        }
        nums.sort_unstable();

        let mut line = String::new();
        for chunk in nums.chunks(5) {
            line.clear();
            line.push_str("allow/ioctl+");
            for (i, n) in chunk.iter().enumerate() {
                if i > 0 {
                    line.push(',');
                }
                w!(&mut line, "{:#x}", n)?;
            }
            wln!(&mut output, "{line}")?;
        }

        // Print named ioctls.
        if !access.ioctl_names.is_empty() {
            // Bucket names by inferred prefix without double-storing keys.
            let mut groups: PandoraMap<String, Vec<String>> = PandoraMap::default();
            for key in access.ioctl_names.iter() {
                let name = key.to_string();
                let gkey = find_shared_prefix(&access.ioctl_names, &name);
                groups.entry(gkey).or_default().push(name);
            }

            // Stable, human-oriented ordering:
            // 1. If one key is a prefix of the other, put the longer (more specific) first.
            // 2. Otherwise, plain lexicographic.
            // 3. If keys equal, bigger group first for stability.
            // 4. Names within each group are sorted as well.
            let mut grouped: Vec<(String, Vec<String>)> = groups.into_iter().collect();
            grouped.sort_by(|(ka, va), (kb, vb)| ka.cmp(kb).then_with(|| vb.len().cmp(&va.len())));
            grouped.sort_by(|(ka, va), (kb, vb)| {
                if ka == kb {
                    return vb.len().cmp(&va.len());
                }
                if kb.starts_with(ka) {
                    // ka is a prefix of kb -> kb (longer) should come first -> ka after kb
                    Ordering::Greater
                } else if ka.starts_with(kb) {
                    // kb is a prefix of ka -> ka (longer) should come first -> ka before kb
                    Ordering::Less
                } else {
                    ka.cmp(kb)
                }
            });

            let mut buf = String::new();
            for (_k, mut v) in grouped {
                v.sort_unstable();
                buf.clear();
                buf.push_str("allow/ioctl+");
                for (i, name) in v.iter().enumerate() {
                    if i > 0 {
                        buf.push(',');
                    }
                    buf.push_str(name);
                }
                wln!(&mut output, "{buf}")?;
            }
        }
    }

    // Print Force entries if available,
    // concurrency-limited parallel checksums + multiple progress bars.
    if !access.force.is_empty() {
        wln!(&mut output, "\n###\n# Executable Verification\n###")?;
        wln!(&mut output, "sandbox/force:on")?;

        let force: Vec<_> = access.force.into_iter().collect();
        let mut force: Vec<PathBuf> = force.iter().map(|s| path2dehex(s.as_str())).collect();
        force.sort_by_cached_key(|arg| (arg.as_os_str().as_bytes().len(), arg.clone()));
        let force_len = force.len();
        let force_max = force
            .iter()
            .map(|arg| arg.as_os_str().as_bytes().len())
            .max()
            .ok_or(Errno::EFAULT)?;

        let pool = ThreadPoolBuilder::new()
            .num_threads(concurrency)
            .build()
            .map_err(|_| Errno::EAGAIN)?;

        let _ = m.println(format!(
            "{} calculating {} checksums for {} executables...",
            style("pandora:").bold().magenta(),
            style(hash_function.to_string()).bold().cyan(),
            style(force_len.to_string()).bold().yellow(),
        ));

        // Prepare progress bar style.
        let prefix_width = force_max + hash_function.to_string().len() + "()".len();
        let fmt = format!(
            "{{prefix:<{prefix_width}}} {{bar:40.bold.cyan/bold.blue}} {{bytes:>7}}/{{total_bytes:7}} {{bytes_per_sec:7}} eta: {{eta}}",
        );
        let sty = ProgressStyle::with_template(&fmt)
            .map_err(|_| Errno::EINVAL)?
            .progress_chars("+~-");

        // Initialize multiple progressbar.
        let mut pbs = Vec::<(PathBuf, ProgressBar)>::with_capacity(force_len);
        for path in &force {
            let len = metadata(path).map(|md| md.len()).map_err(err2no)?;
            let pb = m.add(ProgressBar::new(len));
            pb.set_style(sty.clone());
            pb.set_prefix(format!(
                "{}({})",
                style(hash_function.to_string()).bold().blue(),
                style(path.display()).bold().yellow()
            ));
            pbs.push((path.clone(), pb));
        }

        // We'll collect final "force+path:hash" rules here.
        let rules = Arc::new(Mutex::new(PandoraMap::<PathBuf, String>::default()));

        // Spawn concurrency worker threads to do the hashing
        #[expect(clippy::disallowed_methods)]
        pool.install(|| {
            pbs.par_iter()
                .for_each(|(path, pb)| match path2force(path, hash_function, pb) {
                    Ok(rule) => {
                        let mut split = rule.splitn(2, ':');
                        split.next().unwrap();
                        let hash = split.next().unwrap();

                        pb.println(format!(
                            "{}({}) = {}",
                            style(hash_function.to_string()).bold().cyan(),
                            style(path.display()).bold().yellow(),
                            style(hash).bold().green(),
                        ));
                        pb.finish_and_clear();

                        {
                            let mut rules = rules.lock().unwrap_or_else(|err| err.into_inner());
                            rules.insert(path.clone(), rule);
                        }
                    }
                    Err(error) => {
                        pb.println(format!(
                            "{}({}) = {}",
                            style(hash_function.to_string()).bold().red(),
                            style(path.display()).bold().yellow(),
                            style(error).bold().red(),
                        ));
                        pb.finish_and_clear();
                    }
                });
        });

        drop(pool);
        let rules = rules.lock().unwrap_or_else(|err| err.into_inner());

        #[expect(clippy::disallowed_methods)]
        for path in &force {
            let rule = rules
                .get(path)
                .expect("BUG: path not found in force map, report a bug!");
            w!(&mut output, "\n{rule}")?;
        }
        wln!(&mut output)?;

        let _ = wln!(
            stderr(),
            "{} calculated {} checksums for {} executables.",
            style("pandora:").bold().magenta(),
            style(hash_function.to_string()).bold().cyan(),
            style(force_len.to_string()).bold().yellow(),
        );
    }

    let _ = wln!(
        stderr(),
        "{} profile generation completed! \\o/",
        style("pandora:").bold().magenta(),
    );

    let _ = wln!(
        stderr(),
        "{} profile has been written to `{}'.",
        style("pandora:").bold().magenta(),
        style(output_path).bold().yellow(),
    );

    let _ = wln!(
        stderr(),
        "{} To use it, do: {} -P \"{}\" -- {cmd}",
        style("pandora:").bold().magenta(),
        style("syd").bold().green(),
        style(output_path).bold().yellow(),
    );

    Ok(ExitCode::from(program_exit_code))
}

/// Used to perform path-based hashing in parallel with a progress bar.
fn path2force(path: &PathBuf, func: HashAlgorithm, pb: &ProgressBar) -> std::io::Result<String> {
    // We use CRC32 as defined in IEEE 802.3.
    let crc32 = Crc::<u32>::new(&CRC_32_ISO_HDLC);
    // We use CRC64 as defined in ECMA-182.
    let crc64 = Crc::<u64>::new(&CRC_64_ECMA_182);

    let mut hasher_state = match func {
        HashAlgorithm::Crc32 => HashState::Crc32(crc32.digest()),
        HashAlgorithm::Crc64 => HashState::Crc64(crc64.digest()),
        HashAlgorithm::Md5 => HashState::Md5(md5::Context::new()),
        HashAlgorithm::Sha1 => HashState::Sha1(Sha1::new()),
        HashAlgorithm::Sha256 => HashState::Sha3_256(Sha3_256::new()),
        HashAlgorithm::Sha384 => HashState::Sha3_384(Sha3_384::new()),
        HashAlgorithm::Sha512 => HashState::Sha3_512(Sha3_512::new()),
    };

    let open_how = safe_open_how(OFlag::O_RDONLY | OFlag::O_NOCTTY);
    #[expect(clippy::disallowed_methods)]
    let mut file = openat2(AT_FDCWD, path, open_how).map(File::from)?;

    let mut buffer = [0u8; 64 * 1024];
    loop {
        let read_count = match file.read(&mut buffer) {
            Ok(0) => break,
            Ok(n) => n,
            Err(ref e) if e.kind() == std::io::ErrorKind::Interrupted => continue,
            Err(e) => return Err(e),
        };
        match &mut hasher_state {
            HashState::Crc32(d) => d.update(&buffer[..read_count]),
            HashState::Crc64(d) => d.update(&buffer[..read_count]),
            HashState::Md5(c) => c.consume(&buffer[..read_count]),
            HashState::Sha1(s) => s.update(&buffer[..read_count]),
            HashState::Sha3_256(s) => s.update(&buffer[..read_count]),
            HashState::Sha3_384(s) => s.update(&buffer[..read_count]),
            HashState::Sha3_512(s) => s.update(&buffer[..read_count]),
        }
        pb.inc(read_count as u64);
    }

    let digest = match hasher_state {
        HashState::Crc32(d) => d.finalize().to_be_bytes().to_vec(),
        HashState::Crc64(d) => d.finalize().to_be_bytes().to_vec(),
        HashState::Md5(s) => s.finalize().to_vec(),
        HashState::Sha1(s) => s.finalize().to_vec(),
        HashState::Sha3_256(s) => s.finalize().to_vec(),
        HashState::Sha3_384(s) => s.finalize().to_vec(),
        HashState::Sha3_512(s) => s.finalize().to_vec(),
    };

    let hex = HEXLOWER.encode(&digest);
    Ok(format!("force+{}:{hex}", mask_path(path)))
}

/// Enum for incremental hashing.
enum HashState<'a> {
    Crc32(crc::Digest<'a, u32>),
    Crc64(crc::Digest<'a, u64>),
    Md5(md5::Context),
    Sha1(Sha1),
    Sha3_256(Sha3_256),
    Sha3_384(Sha3_384),
    Sha3_512(Sha3_512),
}

/// Open either stdin, syslog(2) or a file for reading.
fn open_input(input: &str) -> Box<dyn BufRead> {
    match input {
        "-" => Box::new(BufReader::new(stdin())),
        "dmesg" | "syslog" => Box::new(BufReader::new(match Syslog::open() {
            Ok(syslog) => syslog,
            Err(err) => {
                wln!(
                    stderr(),
                    "{} failed to access syslog: {}!",
                    style("pandora:").bold().magenta(),
                    style(err.to_string()).bold().red(),
                )
                .unwrap();
                exit(1);
            }
        })),
        path => Box::new(BufReader::new(
            #[expect(clippy::disallowed_methods)]
            match OpenOptions::new().read(true).open(path) {
                Ok(file) => file,
                Err(err) => {
                    wln!(
                        stderr(),
                        "{} failed to open file {}: {}!",
                        style("pandora:").bold().magenta(),
                        style(path).bold().yellow(),
                        style(err.to_string()).bold().red(),
                    )
                    .unwrap();
                    exit(1);
                }
            },
        )),
    }
}

/// Open either stdout or a file for writing (in create_new mode).
fn open_output(path_or_stdout: &str) -> Box<dyn std::io::Write> {
    match path_or_stdout {
        "-" => Box::new(std::io::BufWriter::new(std::io::stdout())),
        path => Box::new(std::io::BufWriter::new(
            #[expect(clippy::disallowed_methods)]
            match OpenOptions::new().write(true).create_new(true).open(path) {
                Ok(file) => file,
                Err(err) => {
                    wln!(
                        stderr(),
                        "{} failed to open file {}: {}!",
                        style("pandora:").bold().magenta(),
                        style(path).bold().cyan(),
                        style(err.to_string()).bold().red(),
                    )
                    .unwrap();
                    exit(1);
                }
            },
        )),
    }
}

/// Apply the path limit or special-case transformations.
fn process_path<'a>(path: &'a str, limit: u8) -> Cow<'a, str> {
    if path == "/" {
        Cow::Borrowed(path)
    } else if let Some(glob) = path2glob(path) {
        glob
    } else if matches!(path.chars().next(), Some('/')) {
        let limit = limit as usize;
        let members: Vec<&str> = path.split('/').filter(|&x| !x.is_empty()).collect();
        if limit > 0 && limit < members.len() {
            format!("/{}/***", members[0..limit].join("/"))
        } else {
            format!("/{}", members.join("/"))
        }
        .into()
    } else {
        // Abstract and unnamed UNIX sockets
        Cow::Borrowed(path)
    }
}

/// Possibly decode a hex path. If hex decode fails, return it as-is.
fn path2dehex(path: &str) -> PathBuf {
    if let Ok(path_decoded) = HEXLOWER_PERMISSIVE.decode(path.as_bytes()) {
        OsString::from_vec(path_decoded).into()
    } else {
        path.into()
    }
}

/// If the path is known to map to a standard glob, return it. Otherwise return None.
fn path2glob<'a>(path: &'a str) -> Option<Cow<'a, str>> {
    if !matches!(path.chars().next(), Some('/') | Some('@') | Some('!')) {
        // SAFETY: hex-encoded untrusted path, return as is.
        return Some(Cow::Borrowed(path));
    }
    // SAFETY: Path is valid UTF-8.
    let path = path2dehex(path);
    let path = path.to_string_lossy();
    let components: Vec<&str> = path.split('/').collect();
    let mut new_path = String::new();
    let mut handled = false;

    if path.starts_with("/proc/") {
        if components.len() >= 3 && components[2].chars().all(char::is_numeric) {
            if components.len() > 4
                && components[4].chars().all(char::is_numeric)
                && components[3] == "task"
            {
                // Handle the /proc/$pid/task/$tid/... case
                let rest_of_path = if components.len() > 5 {
                    format!("/{}", components[5..].join("/"))
                } else {
                    String::new()
                };
                new_path = format!("/proc/[0-9]*/task/[0-9]*{}", rest_of_path);
                handled = true;

                // Specifically handle the /proc/$pid/task/$tid/{fd,ns}/... cases.
                if components.len() > 5 && components[5] == "fd" {
                    let fd_rest_of_path = if components.len() > 6 {
                        format!("/{}", components[6..].join("/"))
                    } else {
                        String::new()
                    };
                    new_path = format!("/proc/[0-9]*/task/[0-9]*/fd{}", fd_rest_of_path);
                } else if components.len() > 5 && components[5] == "ns" {
                    let ns_rest_of_path = if components.len() > 6 {
                        format!("/{}", components[6..].join("/"))
                    } else {
                        String::new()
                    };
                    new_path = format!("/proc/[0-9]*/task/[0-9]*/ns{}", ns_rest_of_path);
                }
            } else {
                // Handle the general /proc/$pid/... case
                let rest_of_path = if components.len() > 3 {
                    format!("/{}", components[3..].join("/"))
                } else {
                    String::new()
                };
                new_path = format!("/proc/[0-9]*{}", rest_of_path);
                handled = true;

                // Specifically handle the /proc/$pid/{fd,ns}/... cases.
                if components.len() > 3 && components[3] == "fd" {
                    let fd_rest_of_path = if components.len() > 4 {
                        format!("/{}", components[4..].join("/"))
                    } else {
                        String::new()
                    };
                    new_path = format!("/proc/[0-9]*/fd{}", fd_rest_of_path);
                } else if components.len() > 3 && components[3] == "ns" {
                    let ns_rest_of_path = if components.len() > 4 {
                        format!("/{}", components[4..].join("/"))
                    } else {
                        String::new()
                    };
                    new_path = format!("/proc/[0-9]*/ns{}", ns_rest_of_path);
                }
            }
        }

        // Further handle /{fd,ns}/... parts.
        if new_path.contains("/fd/") || new_path.contains("/ns/") {
            let mut final_path = String::new();
            let fd_components: Vec<&str> = new_path.split('/').collect();
            for (i, component) in fd_components.iter().enumerate() {
                if i > 0 {
                    final_path.push('/');
                }
                if i == fd_components.len() - 1 && component.chars().all(char::is_numeric) {
                    // Convert numeric fd/ns component to [0-9]*.
                    final_path.push_str("[0-9]*");
                } else if component.contains(':') {
                    // Handle foo:[number] pattern
                    let parts: Vec<&str> = component.split(':').collect();
                    if parts.len() == 2 && parts[1].starts_with('[') && parts[1].ends_with(']') {
                        let inner = &parts[1][1..parts[1].len() - 1];
                        if inner.chars().all(char::is_numeric) {
                            final_path.push_str(&format!("{}:[0-9]*", parts[0]));
                            continue;
                        }
                    }
                    final_path.push_str(component);
                } else {
                    final_path.push_str(component);
                }
            }
            return Some(final_path.into());
        }
    }

    if handled {
        return Some(new_path.into());
    }

    // Handle memory file descriptors.
    if path.starts_with("!memfd:") {
        return Some(Cow::Borrowed("!memfd:**"));
    }
    if path.starts_with("!memfd-hugetlb:") {
        return Some(Cow::Borrowed("!memfd-hugetlb:**"));
    }

    // Handle /dev/pts/[number] case
    if path.starts_with("/dev/pts/") {
        if path
            .chars()
            .nth("/dev/pts/".len())
            .map(|c| c.is_numeric())
            .unwrap_or(false)
        {
            return Some(Cow::Borrowed("/dev/pts/[0-9]*"));
        } else {
            return None;
        }
    }

    // Handle /dev/tty case
    if path == "/dev/tty" {
        return Some(Cow::Borrowed("/dev/tty"));
    } else if path.starts_with("/dev/tty") {
        return Some(Cow::Borrowed("/dev/tty*"));
    }

    // Handle CUDA abstract sockets:
    //
    // e.g. @cuda-uvmfd--1-63797 -> @cuda-uvmfd--*
    if path.starts_with('@') {
        if let Some(dashdash_pos) = path.rfind("--") {
            let after = &path[dashdash_pos + 2..];
            if !after.is_empty()
                && after
                    .chars()
                    .all(|c| c.is_ascii_digit() || c.is_ascii_punctuation())
            {
                let path = format!("{}--*", &path[..dashdash_pos]);
                return Some(Cow::Owned(path));
            }
        }
    }

    // Handle Gecko pipes, e.g:
    // @gecko-crash-helper-pipe.462275 -> allow/net/bind+@gecko-crash-helper-pipe.*
    if let Some(at_pos) = path.find("@gecko-") {
        if let Some(dot_pos) = path[at_pos..].rfind('.') {
            let dot_pos = at_pos + dot_pos;
            let suffix = &path[dot_pos + 1..];
            if !suffix.is_empty() && suffix.chars().all(|c| c.is_ascii_alphanumeric()) {
                let mut out = String::with_capacity(path.len());
                out.push_str(&path[..dot_pos + 1]); // keep prefix up to the dot
                out.push('*'); // glob the PID tail
                return Some(Cow::Owned(out));
            }
        }
    }

    // Return None if no cases match.
    None
}

/// Logs an untrusted Path, escaping it as hex if it contains control
/// characters.
#[inline]
fn mask_path(path: &Path) -> String {
    let (mask, _) = log_untrusted_buf(path.as_os_str().as_bytes());
    mask
}

/// Logs an untrusted buffer, escaping it as hex if it contains control characters.
/// Returns a boolean in addition to the String which is true if String is hex-encoded.
fn log_untrusted_buf(buf: &[u8]) -> (String, bool) {
    if contains_ascii_unprintable(buf) {
        (HEXLOWER.encode(buf), true)
    } else if let Ok(s) = std::str::from_utf8(buf) {
        (s.to_string(), false)
    } else {
        (HEXLOWER.encode(buf), true)
    }
}

/// Checks if the buffer contains ASCII unprintable characters.
fn contains_ascii_unprintable(buf: &[u8]) -> bool {
    buf.iter().any(|byte| !is_ascii_printable(*byte))
}

/// Checks if the given character is ASCII printable.
fn is_ascii_printable(byte: u8) -> bool {
    (0x20..=0x7e).contains(&byte)
}

/// Converts capability to a priority number for sorting.
#[inline]
#[expect(clippy::ptr_arg)]
fn cap2prio(cap: &String) -> usize {
    let cap = cap.as_str().as_bytes();

    match CAP_ORDER.iter().position(|&s| is_equal(cap, s.as_bytes())) {
        Some(idx) => idx + 1,
        None => 0, // unknown/new capability => sort first
    }
}

#[expect(clippy::disallowed_methods)]
fn print_help_main() {
    let nproc = env::var("PANDORA_NPROC").unwrap();
    w!(
        stderr(),
        r#"{PKG_NAME} {PKG_VERSION}
{PKG_DESCRIPTION}
Copyright (c) 2023, 2024, 2025 {PKG_AUTHORS}
SPDX-License-Identifier: {PKG_LICENSE}

Usage: {PKG_NAME} [COMMAND] [OPTIONS...]

Commands:
  profile  Execute a program under inspection and write a Syd profile
  inspect  Read Syd logs from input and write a Syd profile

Options:
  -h  Print help
  -V  Print version

Environment Variables:
  SYD_BIN         Path to Syd binary [default: syd]
  PANDORA_IN      Path to Syd access violation logs, use "-" for standard input, "syslog" for syslog(2) [default: -]
  PANDORA_OUT     Path to Syd profile output, use "-" for standard output [default: ./pandora_out.syd-3]
  PANDORA_LIMIT   Maximum number of path members before trim, 0 to disable [default: 3]
  PANDORA_TIMEOUT Timeout in seconds
  PANDORA_HASH    Hash algorithm:
                  sha3-512 (default), sha3-384, sha3-256, sha1, md5, crc64, crc32
  PANDORA_NPROC   Number of concurrency threads used for parallel hashing [default: {nproc}]

Hey you, out there beyond the wall,
Breaking bottles in the hall,
Can you help me?

Send bug reports to {PKG_AUTHORS}.
Attaching poems encourages consideration tremendously.

Homepage: https://sydbox.exherbo.org
Repository: https://gitlab.exherbo.org/sydbox/
"#,
    ).unwrap();
}

#[expect(clippy::disallowed_methods)]
fn print_help_profile() {
    let nproc = env::var("PANDORA_NPROC").unwrap();
    w!(
        stderr(),
        r#"{PKG_NAME} {PKG_VERSION}
Profile subcommand

Usage: {PKG_NAME} profile [OPTIONS] <cmd>...

Options:
  -h           Print help
  -V           Print version
  -x <bin>     Path to Syd binary [default: syd, env:SYD_BIN]
  -s <option>  Pass an option to Syd during init, may be repeated
  -o <output>  Path to Syd profile output, use "-" for standard output [default: ./pandora_out.syd-3, env:PANDORA_OUT]
  -l <limit>   Maximum number of path members before trim, 0 to disable [default: 3, env:PANDORA_LIMIT]
  -t <secs>    Timeout in seconds [env:PANDORA_TIMEOUT]
  -H <hash>    Hash algorithm: [env:PANDORA_HASH]
               sha3-512 (default), sha3-384, sha3-256, sha1, md5, crc64, crc32
  -T <threads> Number of concurrency threads used for parallel hashing [default: {nproc}, env:PANDORA_NPROC]
"#,
    ).unwrap();
}

#[expect(clippy::disallowed_methods)]
fn print_help_inspect() {
    let nproc = env::var("PANDORA_NPROC").unwrap();
    w!(
        stderr(),
        r#"{PKG_NAME} {PKG_VERSION}
Inspect subcommand

Usage: {PKG_NAME} inspect [OPTIONS]

Options:
  -h           Print help
  -V           Print version
  -i <input>   Path to Syd access violation logs, use "-" for standard input, "syslog" for syslog(2) [default: -]
  -o <output>  Path to Syd profile output, use "-" for standard output [default: ./pandora_out.syd-3, env:PANDORA_OUT]
  -l <limit>   Maximum number of path members before trim, 0 to disable [default: 3, env:PANDORA_LIMIT]
  -H <hash>    Hash algorithm: [env:PANDORA_HASH]
               sha3-512 (default), sha3-384, sha3-256, sha1, md5, crc64, crc32
  -T <threads> Number of concurrency threads used for parallel hashing [default: {nproc}, env:PANDORA_NPROC]
"#,
    ).unwrap();
}

fn print_version() {
    wln!(stderr(), "{PKG_NAME}-{PKG_VERSION}").unwrap();
}

// Returns the longest prefix of `s` shared by ≥2 keys in `set`, else `s`.
#[inline]
fn find_shared_prefix(set: &StringPatriciaSet, s: &str) -> String {
    if s.is_empty() {
        return String::new();
    }

    let mut last = 0usize;

    // Walk all UTF-8 char boundaries plus the end-of-string (s.len()).
    // This lets a shorter key (e.g., "...CREATE") become the chosen prefix
    // when a longer one ("...CREATE_EXT") also exists.
    let mut boundaries = s
        .char_indices()
        .map(|(i, _)| i)
        .chain(std::iter::once(s.len()));

    // Skip the empty prefix at 0.
    boundaries.next();

    for i in boundaries {
        let mut it = set.iter_prefix(&s[..i]);
        let two_or_more = it.next().is_some() && it.next().is_some();
        if two_or_more {
            last = i;
        } else {
            break;
        }
    }

    if last > 0 {
        s[..last].to_owned()
    } else {
        s.to_owned()
    }
}

fn parse_env_str(var: &str, default: &[u8]) -> Result<String, Errno> {
    // If var is set, parse as valid UTF-8. If not set, fallback to default.
    if let Some(osv) = env::var_os(var) {
        if osv.is_empty() {
            return Err(Errno::EINVAL);
        }
        let bytes = osv.as_bytes();
        parse_utf8_str(bytes)
    } else {
        parse_utf8_str(default)
    }
}

fn parse_env_u8(var: &str, default_val: u8) -> Result<u8, Errno> {
    // If var is set, parse it as an integer, else default_val
    if let Some(osv) = env::var_os(var) {
        if osv.is_empty() {
            return Err(Errno::EINVAL);
        }
        parse_u8(osv.as_bytes())
    } else {
        Ok(default_val)
    }
}

fn parse_env_usize(var: &str, default_val: usize) -> Result<usize, Errno> {
    if let Some(osv) = env::var_os(var) {
        if osv.is_empty() {
            return Err(Errno::EINVAL);
        }
        parse_usize(osv.as_bytes())
    } else {
        Ok(default_val)
    }
}

fn parse_os_u64(osv: &std::ffi::OsString) -> Result<u64, Errno> {
    if osv.is_empty() {
        return Err(Errno::EINVAL);
    }
    parse_u64(osv.as_bytes())
}

fn parse_utf8_str(bytes: &[u8]) -> Result<String, Errno> {
    match std::str::from_utf8(bytes) {
        Ok(s) => Ok(s.to_owned()),
        Err(_) => {
            wln!(stderr(), "ERROR: invalid UTF-8 data")?;
            Err(Errno::EINVAL)
        }
    }
}

fn parse_u8(bytes: &[u8]) -> Result<u8, Errno> {
    let n = btoi::<i64>(bytes).map_err(|_| Errno::EINVAL)?;
    if n < 0 || n > u8::MAX as i64 {
        return Err(Errno::EINVAL);
    }
    Ok(n as u8)
}

fn parse_usize(bytes: &[u8]) -> Result<usize, Errno> {
    let n = btoi::<i64>(bytes).map_err(|_| Errno::EINVAL)?;
    if n < 0 {
        return Err(Errno::EINVAL);
    }
    Ok(n as usize)
}

fn parse_u64(bytes: &[u8]) -> Result<u64, Errno> {
    let n = btoi::<i64>(bytes).map_err(|_| Errno::EINVAL)?;
    if n < 0 {
        return Err(Errno::EINVAL);
    }
    Ok(n as u64)
}

// Convert a std::io::Error into a nix::Errno.
fn err2no(err: std::io::Error) -> Errno {
    err.raw_os_error()
        .map(Errno::from_raw)
        .unwrap_or(Errno::ENOSYS)
}

// Return a safe OpenHow structure.
fn safe_open_how(flags: OFlag) -> OpenHow {
    // Note we leave the caller to handle O_NOCTTY,
    // because its use is invalid with O_PATH.
    let mode = if flags.contains(OFlag::O_CREAT) || flags.contains(OFlag::O_TMPFILE) {
        Mode::from_bits_truncate(0o600)
    } else {
        Mode::empty()
    };
    OpenHow::new()
        .flags(flags | OFlag::O_CLOEXEC | OFlag::O_NOFOLLOW)
        .mode(mode)
        .resolve(ResolveFlag::RESOLVE_NO_MAGICLINKS | ResolveFlag::RESOLVE_NO_SYMLINKS)
}

// Sets or clears the close-on-exec (FD_CLOEXEC) flag on a file descriptor.
fn set_cloexec<Fd: AsFd>(fd: Fd, state: bool) -> Result<(), Errno> {
    let flags = fcntl(&fd, FcntlArg::F_GETFD)?;

    let mut new_flags = flags;
    if state {
        new_flags |= FdFlag::FD_CLOEXEC.bits();
    } else {
        new_flags &= !FdFlag::FD_CLOEXEC.bits();
    }

    fcntl(
        &fd,
        FcntlArg::F_SETFD(FdFlag::from_bits_truncate(new_flags)),
    )
    .map(drop)
}

// Keep in sync with syd/src/compat.rs
//
// This structure represents the Linux data structure `struct statx_timestamp`
#[repr(C)]
#[derive(Copy, Clone, Debug, Default)]
struct FileStatxTimestamp {
    tv_sec: i64,
    tv_nsec: u32,
    __statx_timestamp_pad1: [i32; 1],
}

impl PartialEq for FileStatxTimestamp {
    fn eq(&self, other: &Self) -> bool {
        self.tv_sec == other.tv_sec && self.tv_nsec == other.tv_nsec
    }
}

impl Eq for FileStatxTimestamp {}

impl PartialOrd for FileStatxTimestamp {
    fn partial_cmp(&self, other: &Self) -> Option<Ordering> {
        Some(self.cmp(other))
    }
}

impl Ord for FileStatxTimestamp {
    fn cmp(&self, other: &Self) -> Ordering {
        match self.tv_sec.cmp(&other.tv_sec) {
            Ordering::Equal => self.tv_nsec.cmp(&other.tv_nsec),
            ord => ord,
        }
    }
}

// This structure represents the Linux data structure `struct statx`
#[repr(C)]
#[derive(Copy, Clone, Debug)]
struct FileStatx {
    stx_mask: u32,       // What results were written [uncond]
    stx_blksize: u32,    // Preferred general I/O size [uncond]
    stx_attributes: u64, // Flags conveying information about the file [uncond]

    stx_nlink: u32, // Number of hard links
    stx_uid: u32,   // User ID of owner
    stx_gid: u32,   // Group ID of owner
    stx_mode: u16,  // File mode
    __statx_pad1: [u16; 1],

    stx_ino: u64,             // Inode number
    stx_size: u64,            // File size
    stx_blocks: u64,          // Number of 512-byte blocks allocated
    stx_attributes_mask: u64, // Mask to show what's supported in stx_attributes

    stx_atime: FileStatxTimestamp, // Last access time
    stx_btime: FileStatxTimestamp, // File creation time
    stx_ctime: FileStatxTimestamp, // Last attribute change time
    stx_mtime: FileStatxTimestamp, // Last data modification time

    stx_rdev_major: u32, // Device ID of special file [if bdev/cdev]
    stx_rdev_minor: u32,

    // Note, these are not not public on purpose
    // as they return inconsistent values on filesytems
    // such as btrfs and overlayfs. `stx_mnt_id` should
    // be used instead.
    stx_dev_major: u32, // ID of device containing file [uncond]
    stx_dev_minor: u32,

    stx_mnt_id: u64,
    stx_dio_mem_align: u32,    // Memory buffer alignment for direct I/O
    stx_dio_offset_align: u32, // File offset alignment for direct I/O

    __statx_pad2: [u64; 12], // Spare space for future expansion
}

// Safe statx() wrapper to use with a FD only.
fn fstatx<Fd: AsFd>(fd: Fd, mask: libc::c_uint) -> Result<FileStatx, Errno> {
    let fd = fd.as_fd().as_raw_fd();
    let mut dst = std::mem::MaybeUninit::uninit();

    // SAFETY: Neither nix nor libc has a wrapper for statx.
    Errno::result(unsafe {
        libc::syscall(
            libc::SYS_statx,
            fd,
            b"\0".as_ptr(),
            libc::AT_EMPTY_PATH,
            mask,
            dst.as_mut_ptr(),
        )
    })?;

    // SAFETY: statx returned success.
    Ok(unsafe { dst.assume_init() })
}

// Want/got stx_ino.
const STATX_INO: libc::c_uint = 0x00000100;

// Retries a closure on `EAGAIN` and `EINTR` errors.
//
// This function will call the provided closure, and if the closure
// returns `EAGAIN` or `EINTR` error, it will retry the operation until it
// succeeds or fails with a different error.
fn retry_on_intr<F, T>(mut f: F) -> Result<T, Errno>
where
    F: FnMut() -> Result<T, Errno>,
{
    loop {
        match f() {
            Err(Errno::EAGAIN | Errno::EINTR) => continue,
            result => return result,
        }
    }
}
