/*
 * SPDX-FileCopyrightText: Oliver Beard
 * SPDX-FileCopyrightText: David Edmundson
 *
 * SPDX-License-Identifier: LGPL-2.0-or-later
 */

import org.kde.breeze.components

import QtQuick 2.15
import QtQuick.Layouts 1.15
import QtQuick.Controls 2.15 as QQC2

import org.kde.plasma.components 3.0 as PlasmaComponents3
import org.kde.plasma.extras 2.0 as PlasmaExtras
import org.kde.kirigami 2.20 as Kirigami

import org.kde.plasma.login as PlasmaLogin

SessionManagementScreen {
    id: root
    property Item mainPasswordBox: passwordBox

    property bool showUsernamePrompt: !showUserList

    property bool loginScreenUiVisible: false

    //the y position that should be ensured visible when the on screen keyboard is visible
    property int visibleBoundary: mapFromItem(loginButton, 0, 0).y
    onHeightChanged: visibleBoundary = mapFromItem(loginButton, 0, 0).y + loginButton.height + Kirigami.Units.smallSpacing

    property real fontSize: Kirigami.Theme.defaultFont.pointSize

    signal loginRequest(string username, string password)

    onUserSelected: {
        // Don't startLogin() here, because the signal is connected to the
        // Escape key as well, for which it wouldn't make sense to trigger
        // login.
        passwordBox.clear();
        focusFirstVisibleFormControl();
    }

    QQC2.StackView.onActivating: {
        // Controls are not visible yet.
        Qt.callLater(focusFirstVisibleFormControl);
    }

    function focusFirstVisibleFormControl() {
        const nextControl = (userNameInput.visible
            ? userNameInput
            : (passwordBox.visible
                ? passwordBox
                : loginButton));
        // Using TabFocusReason, so that the loginButton gets the visual highlight.
        nextControl.forceActiveFocus(Qt.TabFocusReason);
    }

    /*
     * Login has been requested with the following username and password
     * If username field is visible, it will be taken from that, otherwise from the "name" property of the currentIndex
     */
    function startLogin() {
        const username = showUsernamePrompt ? userNameInput.text : userList.selectedUser
        const password = passwordBox.text

        footer.enabled = false
        mainStack.enabled = false
        userListComponent.userList.opacity = 0.75

        // This is partly because it looks nicer, but more importantly it
        // works round a Qt bug that can trigger if the app is closed with a
        // TextField focused.
        //
        // See https://bugreports.qt.io/browse/QTBUG-55460
        loginButton.forceActiveFocus();
        loginRequest(username, password);
    }

    PlasmaComponents3.TextField {
        id: userNameInput
        font.pointSize: fontSize + 1
        Layout.fillWidth: true

        text: ""
        visible: showUsernamePrompt
        focus: showUsernamePrompt
        placeholderText: i18nd("plasma_login", "Username")

        onAccepted: {
            if (root.loginScreenUiVisible) {
                passwordBox.forceActiveFocus()
            }
        }
    }

    RowLayout {
        Layout.fillWidth: true

        PlasmaExtras.PasswordField {
            id: passwordBox
            font.pointSize: fontSize + 1
            Layout.fillWidth: true

            placeholderText: i18nd("plasma_login", "Password")
            focus: !showUsernamePrompt

            onAccepted: {
                if (root.loginScreenUiVisible) {
                    startLogin();
                }
            }

            visible: root.showUsernamePrompt || userList.currentItem.needsPassword

            Keys.onEscapePressed: {
                mainStack.currentItem.forceActiveFocus();
            }

            //if empty and left or right is pressed change selection in user switch
            //this cannot be in keys.onLeftPressed as then it doesn't reach the password box
            Keys.onPressed: event => {
                if (event.key === Qt.Key_Left && !text) {
                    userList.decrementCurrentIndex();
                    event.accepted = true
                }
                if (event.key === Qt.Key_Right && !text) {
                    userList.incrementCurrentIndex();
                    event.accepted = true
                }
            }

            Connections {
                target: PlasmaLogin.Authenticator

                function onLoginFailed() {
                    passwordBox.selectAll()
                    passwordBox.forceActiveFocus()
                }
            }
        }

        PlasmaComponents3.Button {
            id: loginButton
            Accessible.name: i18nd("plasma_login", "Log In")
            Layout.preferredHeight: passwordBox.implicitHeight
            Layout.preferredWidth: text.length === 0 ? loginButton.Layout.preferredHeight : -1

            icon.name: text.length === 0 ? (root.LayoutMirroring.enabled ? "go-previous" : "go-next") : ""

            text: root.showUsernamePrompt || userList.currentItem.needsPassword ? "" : i18n("Log In")
            onClicked: startLogin()
            Keys.onEnterPressed: clicked()
            Keys.onReturnPressed: clicked()
        }
    }

    // Synchronise state
    Item {
        id: sync

        readonly property bool isUserList: root.showUserList && !root.showUsernamePrompt

        // Login initial state
        Component.onCompleted: {
            if (sync.isUserList) {
                root.userList.currentIndex = PlasmaLogin.GreeterState.userListIndex;
                passwordBox.text = PlasmaLogin.GreeterState.userListPassword;
            } else {
                userNameInput.text = PlasmaLogin.GreeterState.userPromptUsername;
                passwordBox.text = PlasmaLogin.GreeterState.userPromptPassword;
            }

            passwordBox.showPassword = PlasmaLogin.GreeterState.showPassword;
        }

        // Login -> GreeterState
        Connections {
            target: root.userList

            function onCurrentIndexChanged() {
                if (!sync.isUserList) {
                    return;
                }

                if (PlasmaLogin.GreeterState.userListIndex != root.userList.currentIndex) {
                    PlasmaLogin.GreeterState.userListIndex = root.userList.currentIndex;
                }
            }
        }

        Connections {
            target: userNameInput

            function onTextChanged() {
                if (!sync.isUserList) {
                    if (PlasmaLogin.GreeterState.userPromptUsername != userNameInput.text) {
                        PlasmaLogin.GreeterState.userPromptUsername = userNameInput.text;
                    }
                }
            }
        }

        Connections {
            target: passwordBox

            function onTextChanged() {
                if (sync.isUserList) {
                    if (PlasmaLogin.GreeterState.userListPassword != passwordBox.text) {
                        PlasmaLogin.GreeterState.userListPassword = passwordBox.text;
                    }
                } else {
                    if (PlasmaLogin.GreeterState.userPromptPassword != passwordBox.text) {
                        PlasmaLogin.GreeterState.userPromptPassword = passwordBox.text;
                    }
                }
            }

            function onShowPasswordChanged() {
                if (PlasmaLogin.GreeterState.showPassword != passwordBox.showPassword) {
                    PlasmaLogin.GreeterState.showPassword = passwordBox.showPassword;
                }
            }
        }

        // GreeterState -> Login
        Connections {
            target: PlasmaLogin.GreeterState

            function onUserListIndexChanged() {
                if (!sync.isUserList) {
                    return;
                }

                if (root.userList.currentIndex != PlasmaLogin.GreeterState.userListIndex) {
                    root.userList.currentIndex = PlasmaLogin.GreeterState.userListIndex;
                }
            }

            function onUserListPasswordChanged() {
                if (!sync.isUserList) {
                    return;
                }

                if (passwordBox.text != PlasmaLogin.GreeterState.userListPassword) {
                    passwordBox.text = PlasmaLogin.GreeterState.userListPassword;
                }
            }

            function onUserPromptUsernameChanged() {
                if (sync.isUserList) {
                    return;
                }

                if (userNameInput.text != PlasmaLogin.GreeterState.userPromptUsername) {
                    userNameInput.text = PlasmaLogin.GreeterState.userPromptUsername;
                }
            }

            function onUserPromptPasswordChanged() {
                if (sync.isUserList) {
                    return;
                }

                if (passwordBox.text != PlasmaLogin.GreeterState.userPromptPassword) {
                    passwordBox.text = PlasmaLogin.GreeterState.userPromptPassword;
                }
            }

            function onShowPasswordChanged() {
                if (passwordBox.showPassword != PlasmaLogin.GreeterState.showPassword) {
                    passwordBox.showPassword = PlasmaLogin.GreeterState.showPassword;
                }
            }
        }
    }
}
