//! @file prelude-gsl.c
//! @author J. Marcel van der Veer

//! @section Copyright
//!
//! This file is part of Algol68G - an Algol 68 compiler-interpreter.
//! Copyright 2001-2025 J. Marcel van der Veer [algol68g@xs4all.nl].

//! @section License
//!
//! This program is free software; you can redistribute it and/or modify it 
//! under the terms of the GNU General Public License as published by the 
//! Free Software Foundation; either version 3 of the License, or 
//! (at your option) any later version.
//!
//! This program is distributed in the hope that it will be useful, but 
//! WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
//! or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for 
//! more details. You should have received a copy of the GNU General Public 
//! License along with this program. If not, see [http://www.gnu.org/licenses/].

//! @section Synopsis
//!
//! Standard prelude definitions from GSL.

#include "a68g.h"
#include "a68g-prelude.h"
#include "a68g-prelude-gsl.h"

#if defined (HAVE_GSL)

void stand_gsl_sf (void)
{
  a68g_idf (A68G_EXT, "airyai", A68G_MCACHE (proc_real_real), genie_airy_ai_real);
  a68g_idf (A68G_EXT, "airyaiscaled", A68G_MCACHE (proc_real_real), genie_airy_ai_scaled_real);
  a68g_idf (A68G_EXT, "airybi", A68G_MCACHE (proc_real_real), genie_airy_bi_real);
  a68g_idf (A68G_EXT, "airybiscaled", A68G_MCACHE (proc_real_real), genie_airy_bi_scaled_real);
  a68g_idf (A68G_EXT, "besselin0", A68G_MCACHE (proc_real_real), genie_bessel_in0_real);
  a68g_idf (A68G_EXT, "besselin1", A68G_MCACHE (proc_real_real), genie_bessel_in1_real);
  a68g_idf (A68G_EXT, "besselin0scaled", A68G_MCACHE (proc_real_real), genie_bessel_in0_scaled_real);
  a68g_idf (A68G_EXT, "besselin1scaled", A68G_MCACHE (proc_real_real), genie_bessel_in1_scaled_real);
  a68g_idf (A68G_EXT, "besseljn0", A68G_MCACHE (proc_real_real), genie_bessel_jn0_real);
  a68g_idf (A68G_EXT, "besseljn1", A68G_MCACHE (proc_real_real), genie_bessel_jn1_real);
  a68g_idf (A68G_EXT, "besselkn0", A68G_MCACHE (proc_real_real), genie_bessel_kn0_real);
  a68g_idf (A68G_EXT, "besselkn1", A68G_MCACHE (proc_real_real), genie_bessel_kn1_real);
  a68g_idf (A68G_EXT, "besselkn0scaled", A68G_MCACHE (proc_real_real), genie_bessel_kn0_scaled_real);
  a68g_idf (A68G_EXT, "besselkn1scaled", A68G_MCACHE (proc_real_real), genie_bessel_kn1_scaled_real);
  a68g_idf (A68G_EXT, "besselyn0", A68G_MCACHE (proc_real_real), genie_bessel_yn0_real);
  a68g_idf (A68G_EXT, "besselyn1", A68G_MCACHE (proc_real_real), genie_bessel_yn1_real);
  a68g_idf (A68G_EXT, "expinte1", A68G_MCACHE (proc_real_real), genie_expint_e1_real);
  a68g_idf (A68G_EXT, "expintei", A68G_MCACHE (proc_real_real), genie_expint_ei_real);
  a68g_idf (A68G_EXT, "dawson", A68G_MCACHE (proc_real_real), genie_dawson_real);
  a68g_idf (A68G_EXT, "exprel", A68G_MCACHE (proc_real_real), genie_exprel_real);
  a68g_idf (A68G_EXT, "betaincgsl", A68G_MCACHE (proc_real_real_real_real), genie_beta_inc_real);
  a68g_idf (A68G_EXT, "poch", A68G_MCACHE (proc_real_real_real), genie_poch_real);
  a68g_idf (A68G_EXT, "digamma", A68G_MCACHE (proc_real_real), genie_digamma_real);
  a68g_idf (A68G_EXT, "airyaiderivative", A68G_MCACHE (proc_real_real), genie_airy_ai_deriv_real);
  a68g_idf (A68G_EXT, "airyaideriv", A68G_MCACHE (proc_real_real), genie_airy_ai_deriv_real);
  a68g_idf (A68G_EXT, "airyaiderivscaled", A68G_MCACHE (proc_real_real), genie_airy_ai_deriv_scaled_real);
  a68g_idf (A68G_EXT, "airybiderivative", A68G_MCACHE (proc_real_real), genie_airy_bi_deriv_real);
  a68g_idf (A68G_EXT, "airybideriv", A68G_MCACHE (proc_real_real), genie_airy_bi_deriv_real);
  a68g_idf (A68G_EXT, "airybiderivscaled", A68G_MCACHE (proc_real_real), genie_airy_bi_deriv_scaled_real);
  a68g_idf (A68G_EXT, "airyzeroaideriv", A68G_MCACHE (proc_int_real), genie_airy_zero_ai_deriv_real);
  a68g_idf (A68G_EXT, "airyzeroai", A68G_MCACHE (proc_int_real), genie_airy_zero_ai_real);
  a68g_idf (A68G_EXT, "airyzerobideriv", A68G_MCACHE (proc_int_real), genie_airy_zero_bi_deriv_real);
  a68g_idf (A68G_EXT, "airyzerobi", A68G_MCACHE (proc_int_real), genie_airy_zero_bi_real);
  a68g_idf (A68G_EXT, "anglerestrictpos", A68G_MCACHE (proc_real_real), genie_angle_restrict_pos_real);
  a68g_idf (A68G_EXT, "anglerestrictsymm", A68G_MCACHE (proc_real_real), genie_angle_restrict_symm_real);
  a68g_idf (A68G_EXT, "atanint", A68G_MCACHE (proc_real_real), genie_atanint_real);
  a68g_idf (A68G_EXT, "besselil0scaled", A68G_MCACHE (proc_real_real), genie_bessel_il0_scaled_real);
  a68g_idf (A68G_EXT, "besselil1scaled", A68G_MCACHE (proc_real_real), genie_bessel_il1_scaled_real);
  a68g_idf (A68G_EXT, "besselil2scaled", A68G_MCACHE (proc_real_real), genie_bessel_il2_scaled_real);
  a68g_idf (A68G_EXT, "besselilscaled", A68G_MCACHE (proc_int_real_real), genie_bessel_il_scaled_real);
  a68g_idf (A68G_EXT, "besselilscaled", A68G_MCACHE (proc_real_real), genie_bessel_il_scaled_real);
  a68g_idf (A68G_EXT, "besselin", A68G_MCACHE (proc_int_real_real), genie_bessel_in_real);
  a68g_idf (A68G_EXT, "besselinscaled", A68G_MCACHE (proc_int_real_real), genie_bessel_in_scaled_real);
  a68g_idf (A68G_EXT, "besselinscaled", A68G_MCACHE (proc_real_real), genie_bessel_in_scaled_real);
  a68g_idf (A68G_EXT, "besselinu", A68G_MCACHE (proc_real_real_real), genie_bessel_inu_real);
  a68g_idf (A68G_EXT, "besselinuscaled", A68G_MCACHE (proc_real_real_real), genie_bessel_inu_scaled_real);
  a68g_idf (A68G_EXT, "besseljl0", A68G_MCACHE (proc_real_real), genie_bessel_jl0_real);
  a68g_idf (A68G_EXT, "besseljl1", A68G_MCACHE (proc_real_real), genie_bessel_jl1_real);
  a68g_idf (A68G_EXT, "besseljl2", A68G_MCACHE (proc_real_real), genie_bessel_jl2_real);
  a68g_idf (A68G_EXT, "besseljl", A68G_MCACHE (proc_int_real_real), genie_bessel_jl_real);
  a68g_idf (A68G_EXT, "besseljn", A68G_MCACHE (proc_int_real_real), genie_bessel_jn_real);
  a68g_idf (A68G_EXT, "besselkl0scaled", A68G_MCACHE (proc_real_real), genie_bessel_kl0_scaled_real);
  a68g_idf (A68G_EXT, "besselkl1scaled", A68G_MCACHE (proc_real_real), genie_bessel_kl1_scaled_real);
  a68g_idf (A68G_EXT, "besselkl2scaled", A68G_MCACHE (proc_real_real), genie_bessel_kl2_scaled_real);
  a68g_idf (A68G_EXT, "besselklscaled", A68G_MCACHE (proc_int_real_real), genie_bessel_kl_scaled_real);
  a68g_idf (A68G_EXT, "besselklscaled", A68G_MCACHE (proc_real_real), genie_bessel_kl_scaled_real);
  a68g_idf (A68G_EXT, "besselkn", A68G_MCACHE (proc_int_real_real), genie_bessel_kn_real);
  a68g_idf (A68G_EXT, "besselknscaled", A68G_MCACHE (proc_int_real_real), genie_bessel_kn_scaled_real);
  a68g_idf (A68G_EXT, "besselkn_scaled", A68G_MCACHE (proc_real_real), genie_bessel_kn_scaled_real);
  a68g_idf (A68G_EXT, "besselknu", A68G_MCACHE (proc_real_real_real), genie_bessel_knu_real);
  a68g_idf (A68G_EXT, "besselknuscaled", A68G_MCACHE (proc_real_real), genie_bessel_knu_scaled_real);
  a68g_idf (A68G_EXT, "besselknuscaled", A68G_MCACHE (proc_real_real_real), genie_bessel_knu_scaled_real);
  a68g_idf (A68G_EXT, "bessellnknu", A68G_MCACHE (proc_real_real), genie_bessel_ln_knu_real);
  a68g_idf (A68G_EXT, "bessellnknu", A68G_MCACHE (proc_real_real_real), genie_bessel_ln_knu_real);
  a68g_idf (A68G_EXT, "besselyl0", A68G_MCACHE (proc_real_real), genie_bessel_yl0_real);
  a68g_idf (A68G_EXT, "besselyl1", A68G_MCACHE (proc_real_real), genie_bessel_yl1_real);
  a68g_idf (A68G_EXT, "besselyl2", A68G_MCACHE (proc_real_real), genie_bessel_yl2_real);
  a68g_idf (A68G_EXT, "besselyl", A68G_MCACHE (proc_int_real_real), genie_bessel_yl_real);
  a68g_idf (A68G_EXT, "besselyn", A68G_MCACHE (proc_int_real_real), genie_bessel_yn_real);
  a68g_idf (A68G_EXT, "besselynu", A68G_MCACHE (proc_real_real_real), genie_bessel_ynu_real);
  a68g_idf (A68G_EXT, "besselzeroj0", A68G_MCACHE (proc_int_real), genie_bessel_zero_jnu0_real);
  a68g_idf (A68G_EXT, "besselzeroj1", A68G_MCACHE (proc_int_real), genie_bessel_zero_jnu1_real);
  a68g_idf (A68G_EXT, "besselzerojnu", A68G_MCACHE (proc_int_real_real), genie_bessel_zero_jnu_real);
  a68g_idf (A68G_EXT, "chi", A68G_MCACHE (proc_real_real), genie_chi_real);
  a68g_idf (A68G_EXT, "ci", A68G_MCACHE (proc_real_real), genie_ci_real);
  a68g_idf (A68G_EXT, "clausen", A68G_MCACHE (proc_real_real), genie_clausen_real);
  a68g_idf (A68G_EXT, "conicalp0", A68G_MCACHE (proc_real_real_real), genie_conicalp_0_real);
  a68g_idf (A68G_EXT, "conicalp1", A68G_MCACHE (proc_real_real_real), genie_conicalp_1_real);
  a68g_idf (A68G_EXT, "conicalpcylreg", A68G_MCACHE (proc_int_real_real_real), genie_conicalp_cyl_reg_real);
  a68g_idf (A68G_EXT, "conicalphalf", A68G_MCACHE (proc_real_real_real), genie_conicalp_half_real);
  a68g_idf (A68G_EXT, "conicalpmhalf", A68G_MCACHE (proc_real_real_real), genie_conicalp_mhalf_real);
  a68g_idf (A68G_EXT, "conicalpsphreg", A68G_MCACHE (proc_int_real_real_real), genie_conicalp_sph_reg_real);
  a68g_idf (A68G_EXT, "debye1", A68G_MCACHE (proc_real_real), genie_debye_1_real);
  a68g_idf (A68G_EXT, "debye2", A68G_MCACHE (proc_real_real), genie_debye_2_real);
  a68g_idf (A68G_EXT, "debye3", A68G_MCACHE (proc_real_real), genie_debye_3_real);
  a68g_idf (A68G_EXT, "debye4", A68G_MCACHE (proc_real_real), genie_debye_4_real);
  a68g_idf (A68G_EXT, "debye5", A68G_MCACHE (proc_real_real), genie_debye_5_real);
  a68g_idf (A68G_EXT, "debye6", A68G_MCACHE (proc_real_real), genie_debye_6_real);
  a68g_idf (A68G_EXT, "dilog", A68G_MCACHE (proc_real_real), genie_dilog_real);
  a68g_idf (A68G_EXT, "doublefact", A68G_MCACHE (proc_int_real), genie_doublefact_real);
  a68g_idf (A68G_EXT, "ellintd", A68G_MCACHE (proc_real_real_real), genie_ellint_d_real);
  a68g_idf (A68G_EXT, "ellintecomp", A68G_MCACHE (proc_real_real), genie_ellint_e_comp_real);
  a68g_idf (A68G_EXT, "ellinte", A68G_MCACHE (proc_real_real_real), genie_ellint_e_real);
  a68g_idf (A68G_EXT, "ellintf", A68G_MCACHE (proc_real_real_real), genie_ellint_f_real);
  a68g_idf (A68G_EXT, "ellintkcomp", A68G_MCACHE (proc_real_real), genie_ellint_k_comp_real);
  a68g_idf (A68G_EXT, "ellintpcomp", A68G_MCACHE (proc_real_real_real), genie_ellint_p_comp_real);
  a68g_idf (A68G_EXT, "ellintp", A68G_MCACHE (proc_real_real_real_real), genie_ellint_p_real);
  a68g_idf (A68G_EXT, "ellintrc", A68G_MCACHE (proc_real_real_real), genie_ellint_rc_real);
  a68g_idf (A68G_EXT, "ellintrd", A68G_MCACHE (proc_real_real_real_real), genie_ellint_rd_real);
  a68g_idf (A68G_EXT, "ellintrf", A68G_MCACHE (proc_real_real_real_real), genie_ellint_rf_real);
  a68g_idf (A68G_EXT, "ellintrj", A68G_MCACHE (proc_real_real_real_real_real), genie_ellint_rj_real);
  a68g_idf (A68G_EXT, "etaint", A68G_MCACHE (proc_int_real), genie_etaint_real);
  a68g_idf (A68G_EXT, "eta", A68G_MCACHE (proc_real_real), genie_eta_real);
  a68g_idf (A68G_EXT, "expint3", A68G_MCACHE (proc_real_real), genie_expint_3_real);
  a68g_idf (A68G_EXT, "expinte2", A68G_MCACHE (proc_real_real), genie_expint_e2_real);
  a68g_idf (A68G_EXT, "expinten", A68G_MCACHE (proc_int_real_real), genie_expint_en_real);
  a68g_idf (A68G_EXT, "expm1", A68G_MCACHE (proc_real_real), genie_expm1_real);
  a68g_idf (A68G_EXT, "exprel2", A68G_MCACHE (proc_real_real), genie_exprel_2_real);
  a68g_idf (A68G_EXT, "expreln", A68G_MCACHE (proc_int_real_real), genie_exprel_n_real);
  a68g_idf (A68G_EXT, "fermidirac0", A68G_MCACHE (proc_real_real), genie_fermi_dirac_0_real);
  a68g_idf (A68G_EXT, "fermidirac1", A68G_MCACHE (proc_real_real), genie_fermi_dirac_1_real);
  a68g_idf (A68G_EXT, "fermidirac2", A68G_MCACHE (proc_real_real), genie_fermi_dirac_2_real);
  a68g_idf (A68G_EXT, "fermidirac3half", A68G_MCACHE (proc_real_real), genie_fermi_dirac_3half_real);
  a68g_idf (A68G_EXT, "fermidirachalf", A68G_MCACHE (proc_real_real), genie_fermi_dirac_half_real);
  a68g_idf (A68G_EXT, "fermidiracinc0", A68G_MCACHE (proc_real_real_real), genie_fermi_dirac_inc_0_real);
  a68g_idf (A68G_EXT, "fermidiracint", A68G_MCACHE (proc_int_real_real), genie_fermi_dirac_int_real);
  a68g_idf (A68G_EXT, "fermidiracm1", A68G_MCACHE (proc_real_real), genie_fermi_dirac_m1_real);
  a68g_idf (A68G_EXT, "fermidiracmhalf", A68G_MCACHE (proc_real_real), genie_fermi_dirac_mhalf_real);
  a68g_idf (A68G_EXT, "gammaincgsl", A68G_MCACHE (proc_real_real_real), genie_gamma_inc_real);
  a68g_idf (A68G_EXT, "gammaincp", A68G_MCACHE (proc_real_real_real), genie_gamma_inc_p_real);
  a68g_idf (A68G_EXT, "gammaincq", A68G_MCACHE (proc_real_real_real), genie_gamma_inc_q_real);
  a68g_idf (A68G_EXT, "gammainv", A68G_MCACHE (proc_real_real), genie_gammainv_real);
  a68g_idf (A68G_EXT, "gammastar", A68G_MCACHE (proc_real_real), genie_gammastar_real);
  a68g_idf (A68G_EXT, "gegenpoly1real", A68G_MCACHE (proc_real_real_real), genie_gegenpoly_1_real);
  a68g_idf (A68G_EXT, "gegenpoly2real", A68G_MCACHE (proc_real_real_real), genie_gegenpoly_2_real);
  a68g_idf (A68G_EXT, "gegenpoly3real", A68G_MCACHE (proc_real_real_real), genie_gegenpoly_3_real);
  a68g_idf (A68G_EXT, "gegenpolynreal", A68G_MCACHE (proc_real_real_real), genie_gegenpoly_n_real);
  a68g_idf (A68G_EXT, "hermitefunc", A68G_MCACHE (proc_int_real_real), genie_hermite_func_real);
  a68g_idf (A68G_EXT, "hypot", A68G_MCACHE (proc_real_real_real), genie_hypot_real);
  a68g_idf (A68G_EXT, "hzeta", A68G_MCACHE (proc_real_real_real), genie_hzeta_real);
  a68g_idf (A68G_EXT, "laguerre1real", A68G_MCACHE (proc_real_real_real), genie_laguerre_1_real);
  a68g_idf (A68G_EXT, "laguerre2real", A68G_MCACHE (proc_real_real_real), genie_laguerre_2_real);
  a68g_idf (A68G_EXT, "laguerre3real", A68G_MCACHE (proc_real_real_real), genie_laguerre_3_real);
  a68g_idf (A68G_EXT, "laguerrenreal", A68G_MCACHE (proc_real_real_real), genie_laguerre_n_real);
  a68g_idf (A68G_EXT, "lambertw0", A68G_MCACHE (proc_real_real), genie_lambert_w0_real);
  a68g_idf (A68G_EXT, "lambertwm1", A68G_MCACHE (proc_real_real), genie_lambert_wm1_real);
  a68g_idf (A68G_EXT, "legendreh3d0", A68G_MCACHE (proc_real_real_real), genie_legendre_h3d_0_real);
  a68g_idf (A68G_EXT, "legendreh3d1", A68G_MCACHE (proc_real_real_real), genie_legendre_h3d_1_real);
  a68g_idf (A68G_EXT, "legendreh3d", A68G_MCACHE (proc_int_real_real_real), genie_legendre_H3d_real);
  a68g_idf (A68G_EXT, "legendrep1", A68G_MCACHE (proc_real_real), genie_legendre_p1_real);
  a68g_idf (A68G_EXT, "legendrep2", A68G_MCACHE (proc_real_real), genie_legendre_p2_real);
  a68g_idf (A68G_EXT, "legendrep3", A68G_MCACHE (proc_real_real), genie_legendre_p3_real);
  a68g_idf (A68G_EXT, "legendrepl", A68G_MCACHE (proc_int_real_real), genie_legendre_pl_real);
  a68g_idf (A68G_EXT, "legendreq0", A68G_MCACHE (proc_real_real), genie_legendre_q0_real);
  a68g_idf (A68G_EXT, "legendreq1", A68G_MCACHE (proc_real_real), genie_legendre_q1_real);
  a68g_idf (A68G_EXT, "legendreql", A68G_MCACHE (proc_int_real_real), genie_legendre_ql_real);
  a68g_idf (A68G_EXT, "lncosh", A68G_MCACHE (proc_real_real), genie_lncosh_real);
  a68g_idf (A68G_EXT, "lndoublefact", A68G_MCACHE (proc_int_real), genie_lndoublefact_real);
  a68g_idf (A68G_EXT, "lnpoch", A68G_MCACHE (proc_real_real_real), genie_lnpoch_real);
  a68g_idf (A68G_EXT, "lnsinh", A68G_MCACHE (proc_real_real), genie_lnsinh_real);
  a68g_idf (A68G_EXT, "ln1plusxmx", A68G_MCACHE (proc_real_real), genie_log_1plusx_mx_real);
  a68g_idf (A68G_EXT, "ln1plusx", A68G_MCACHE (proc_real_real), genie_log_1plusx_real);
  a68g_idf (A68G_EXT, "lnabs", A68G_MCACHE (proc_real_real), genie_log_abs_real);
  a68g_idf (A68G_EXT, "pochrel", A68G_MCACHE (proc_real_real_real), genie_pochrel_real);
  a68g_idf (A68G_EXT, "psi1int", A68G_MCACHE (proc_int_real), genie_psi_1_int_real);
  a68g_idf (A68G_EXT, "psi1piy", A68G_MCACHE (proc_real_real), genie_psi_1piy_real);
  a68g_idf (A68G_EXT, "psi1", A68G_MCACHE (proc_real_real), genie_psi_1_real);
  a68g_idf (A68G_EXT, "psiint", A68G_MCACHE (proc_int_real), genie_psi_int_real);
  a68g_idf (A68G_EXT, "psin", A68G_MCACHE (proc_int_real_real), genie_psi_n_real);
  a68g_idf (A68G_EXT, "psi", A68G_MCACHE (proc_real_real), genie_psi_real);
  a68g_idf (A68G_EXT, "shi", A68G_MCACHE (proc_real_real), genie_shi_real);
  a68g_idf (A68G_EXT, "sinc", A68G_MCACHE (proc_real_real), genie_sinc_real);
  a68g_idf (A68G_EXT, "si", A68G_MCACHE (proc_real_real), genie_si_real);
  a68g_idf (A68G_EXT, "synchrotron1", A68G_MCACHE (proc_real_real), genie_synchrotron_1_real);
  a68g_idf (A68G_EXT, "synchrotron2", A68G_MCACHE (proc_real_real), genie_synchrotron_2_real);
  a68g_idf (A68G_EXT, "taylorcoeff", A68G_MCACHE (proc_int_real_real), genie_taylorcoeff_real);
  a68g_idf (A68G_EXT, "transport2", A68G_MCACHE (proc_real_real), genie_transport_2_real);
  a68g_idf (A68G_EXT, "transport3", A68G_MCACHE (proc_real_real), genie_transport_3_real);
  a68g_idf (A68G_EXT, "transport4", A68G_MCACHE (proc_real_real), genie_transport_4_real);
  a68g_idf (A68G_EXT, "transport5", A68G_MCACHE (proc_real_real), genie_transport_5_real);
  a68g_idf (A68G_EXT, "zetaint", A68G_MCACHE (proc_int_real), genie_zeta_int_real);
  a68g_idf (A68G_EXT, "zetam1int", A68G_MCACHE (proc_int_real), genie_zetam1_int_real);
  a68g_idf (A68G_EXT, "zetam1", A68G_MCACHE (proc_real_real), genie_zetam1_real);
  a68g_idf (A68G_EXT, "zeta", A68G_MCACHE (proc_real_real), genie_zeta_real);
}

void stand_gsl_linear_algebra (void)
{
// Vector and matrix pretty print.
  MOID_T *m = a68g_proc (M_VOID, M_ROW_REAL, M_INT, NO_MOID);
  a68g_idf (A68G_EXT, "printvector", m, genie_print_vector);
  m = a68g_proc (M_VOID, M_ROW_ROW_REAL, M_INT, NO_MOID);
  a68g_idf (A68G_EXT, "printmatrix", m, genie_print_matrix);
// Vector and matrix monadic.
  m = a68g_proc (M_ROW_REAL, M_ROW_REAL, NO_MOID);
  a68g_op (A68G_EXT, "+", m, genie_idle);
  a68g_op (A68G_EXT, "-", m, genie_vector_minus);
  m = a68g_proc (M_ROW_ROW_REAL, M_ROW_REAL, NO_MOID);
  a68g_op (A68G_EXT, "CV", m, genie_vector_col);
  a68g_op (A68G_EXT, "RV", m, genie_vector_row);
  m = a68g_proc (M_ROW_ROW_REAL, M_ROW_ROW_REAL, NO_MOID);
  a68g_op (A68G_EXT, "+", m, genie_idle);
  a68g_op (A68G_EXT, "-", m, genie_matrix_minus);
  a68g_op (A68G_EXT, "T", m, genie_matrix_transpose);
  a68g_op (A68G_EXT, "INV", m, genie_matrix_inv);
  a68g_op (A68G_EXT, "PINV", m, genie_matrix_pinv);
  a68g_op (A68G_EXT, "MEAN", m, genie_matrix_column_mean);
  m = a68g_proc (M_REAL, M_ROW_ROW_REAL, NO_MOID);
  a68g_op (A68G_EXT, "DET", m, genie_matrix_det);
  a68g_op (A68G_EXT, "TRACE", m, genie_matrix_trace);
  m = a68g_proc (M_ROW_COMPLEX, M_ROW_COMPLEX, NO_MOID);
  a68g_op (A68G_EXT, "+", m, genie_idle);
  a68g_op (A68G_EXT, "-", m, genie_vector_complex_minus);
  m = a68g_proc (M_ROW_ROW_COMPLEX, M_ROW_ROW_COMPLEX, NO_MOID);
  a68g_op (A68G_EXT, "+", m, genie_idle);
  a68g_op (A68G_EXT, "-", m, genie_matrix_complex_minus);
  a68g_op (A68G_EXT, "T", m, genie_matrix_complex_transpose);
  a68g_op (A68G_EXT, "INV", m, genie_matrix_complex_inv);
  m = a68g_proc (M_COMPLEX, M_ROW_ROW_COMPLEX, NO_MOID);
  a68g_op (A68G_EXT, "DET", m, genie_matrix_complex_det);
  a68g_op (A68G_EXT, "TRACE", m, genie_matrix_complex_trace);
// Vector and matrix dyadic.
  m = a68g_proc (M_BOOL, M_ROW_REAL, M_ROW_REAL, NO_MOID);
  a68g_op (A68G_EXT, "=", m, genie_vector_eq);
  a68g_op (A68G_EXT, "/=", m, genie_vector_ne);
  m = a68g_proc (M_ROW_REAL, M_ROW_REAL, M_ROW_REAL, NO_MOID);
  a68g_op (A68G_EXT, "+", m, genie_vector_add);
  a68g_op (A68G_EXT, "-", m, genie_vector_sub);
  m = a68g_proc (M_REF_ROW_REAL, M_REF_ROW_REAL, M_ROW_REAL, NO_MOID);
  a68g_op (A68G_EXT, "+:=", m, genie_vector_plusab);
  a68g_op (A68G_EXT, "PLUSAB", m, genie_vector_plusab);
  a68g_op (A68G_EXT, "-:=", m, genie_vector_minusab);
  a68g_op (A68G_EXT, "MINUSAB", m, genie_vector_minusab);
  m = a68g_proc (M_BOOL, M_ROW_ROW_REAL, M_ROW_ROW_REAL, NO_MOID);
  a68g_op (A68G_EXT, "=", m, genie_matrix_eq);
  a68g_op (A68G_EXT, "/-", m, genie_matrix_ne);
  m = a68g_proc (M_ROW_ROW_REAL, M_ROW_ROW_REAL, M_ROW_ROW_REAL, NO_MOID);
  a68g_op (A68G_EXT, "+", m, genie_matrix_add);
  a68g_op (A68G_EXT, "-", m, genie_matrix_sub);
  a68g_op (A68G_EXT, "BEFORE", m, genie_matrix_hcat);
  a68g_op (A68G_EXT, "ABOVE", m, genie_matrix_vcat);
  a68g_prio ("BEFORE", 3);
  a68g_prio ("ABOVE", 3);
  m = a68g_proc (M_REF_ROW_ROW_REAL, M_REF_ROW_ROW_REAL, M_ROW_ROW_REAL, NO_MOID);
  a68g_op (A68G_EXT, "+:=", m, genie_matrix_plusab);
  a68g_op (A68G_EXT, "PLUSAB", m, genie_matrix_plusab);
  a68g_op (A68G_EXT, "-:=", m, genie_matrix_minusab);
  a68g_op (A68G_EXT, "MINUSAB", m, genie_matrix_minusab);
  m = a68g_proc (M_BOOL, M_ROW_COMPLEX, M_ROW_COMPLEX, NO_MOID);
  a68g_op (A68G_EXT, "=", m, genie_vector_complex_eq);
  a68g_op (A68G_EXT, "/=", m, genie_vector_complex_ne);
  m = a68g_proc (M_ROW_COMPLEX, M_ROW_COMPLEX, M_ROW_COMPLEX, NO_MOID);
  a68g_op (A68G_EXT, "+", m, genie_vector_complex_add);
  a68g_op (A68G_EXT, "-", m, genie_vector_complex_sub);
  m = a68g_proc (M_REF_ROW_COMPLEX, M_REF_ROW_COMPLEX, M_ROW_COMPLEX, NO_MOID);
  a68g_op (A68G_EXT, "+:=", m, genie_vector_complex_plusab);
  a68g_op (A68G_EXT, "PLUSAB", m, genie_vector_complex_plusab);
  a68g_op (A68G_EXT, "-:=", m, genie_vector_complex_minusab);
  a68g_op (A68G_EXT, "MINUSAB", m, genie_vector_complex_minusab);
  m = a68g_proc (M_BOOL, M_ROW_ROW_COMPLEX, M_ROW_ROW_COMPLEX, NO_MOID);
  a68g_op (A68G_EXT, "=", m, genie_matrix_complex_eq);
  a68g_op (A68G_EXT, "/=", m, genie_matrix_complex_ne);
  m = a68g_proc (M_ROW_ROW_COMPLEX, M_ROW_ROW_COMPLEX, M_ROW_ROW_COMPLEX, NO_MOID);
  a68g_op (A68G_EXT, "+", m, genie_matrix_complex_add);
  a68g_op (A68G_EXT, "-", m, genie_matrix_complex_sub);
  m = a68g_proc (M_REF_ROW_ROW_COMPLEX, M_REF_ROW_ROW_COMPLEX, M_ROW_ROW_COMPLEX, NO_MOID);
  a68g_op (A68G_EXT, "+:=", m, genie_matrix_complex_plusab);
  a68g_op (A68G_EXT, "PLUSAB", m, genie_matrix_complex_plusab);
  a68g_op (A68G_EXT, "-:=", m, genie_matrix_complex_minusab);
  a68g_op (A68G_EXT, "MINUSAB", m, genie_matrix_complex_minusab);
// Vector and matrix scaling.
  m = a68g_proc (M_ROW_REAL, M_REAL, M_ROW_REAL, NO_MOID);
  a68g_op (A68G_EXT, "*", m, genie_real_scale_vector);
  m = a68g_proc (M_ROW_REAL, M_ROW_REAL, M_REAL, NO_MOID);
  a68g_op (A68G_EXT, "*", m, genie_vector_scale_real);
  a68g_op (A68G_EXT, "/", m, genie_vector_div_real);
  m = a68g_proc (M_ROW_ROW_REAL, M_REAL, M_ROW_ROW_REAL, NO_MOID);
  a68g_op (A68G_EXT, "*", m, genie_real_scale_matrix);
  m = a68g_proc (M_ROW_ROW_REAL, M_ROW_ROW_REAL, M_REAL, NO_MOID);
  a68g_op (A68G_EXT, "*", m, genie_matrix_scale_real);
  a68g_op (A68G_EXT, "/", m, genie_matrix_div_real);
  m = a68g_proc (M_ROW_COMPLEX, M_COMPLEX, M_ROW_COMPLEX, NO_MOID);
  a68g_op (A68G_EXT, "*", m, genie_complex_scale_vector_complex);
  m = a68g_proc (M_ROW_COMPLEX, M_ROW_COMPLEX, M_COMPLEX, NO_MOID);
  a68g_op (A68G_EXT, "*", m, genie_vector_complex_scale_complex);
  a68g_op (A68G_EXT, "/", m, genie_vector_complex_div_complex);
  m = a68g_proc (M_ROW_ROW_COMPLEX, M_COMPLEX, M_ROW_ROW_COMPLEX, NO_MOID);
  a68g_op (A68G_EXT, "*", m, genie_complex_scale_matrix_complex);
  m = a68g_proc (M_ROW_ROW_COMPLEX, M_ROW_ROW_COMPLEX, M_COMPLEX, NO_MOID);
  a68g_op (A68G_EXT, "*", m, genie_matrix_complex_scale_complex);
  a68g_op (A68G_EXT, "/", m, genie_matrix_complex_div_complex);
  m = a68g_proc (M_REF_ROW_REAL, M_REF_ROW_REAL, M_REAL, NO_MOID);
  a68g_op (A68G_EXT, "*:=", m, genie_vector_scale_real_ab);
  a68g_op (A68G_EXT, "/:=", m, genie_vector_div_real_ab);
  m = a68g_proc (M_REF_ROW_ROW_REAL, M_REF_ROW_ROW_REAL, M_REAL, NO_MOID);
  a68g_op (A68G_EXT, "*:=", m, genie_matrix_scale_real_ab);
  a68g_op (A68G_EXT, "/:=", m, genie_matrix_div_real_ab);
  m = a68g_proc (M_REF_ROW_COMPLEX, M_REF_ROW_COMPLEX, M_COMPLEX, NO_MOID);
  a68g_op (A68G_EXT, "*:=", m, genie_vector_complex_scale_complex_ab);
  a68g_op (A68G_EXT, "/:=", m, genie_vector_complex_div_complex_ab);
  m = a68g_proc (M_REF_ROW_ROW_COMPLEX, M_REF_ROW_ROW_COMPLEX, M_COMPLEX, NO_MOID);
  a68g_op (A68G_EXT, "*:=", m, genie_matrix_complex_scale_complex_ab);
  a68g_op (A68G_EXT, "/:=", m, genie_matrix_complex_div_complex_ab);
  m = a68g_proc (M_ROW_REAL, M_ROW_REAL, M_ROW_ROW_REAL, NO_MOID);
  a68g_op (A68G_EXT, "*", m, genie_vector_times_matrix);
  m = a68g_proc (M_ROW_COMPLEX, M_ROW_COMPLEX, M_ROW_ROW_COMPLEX, NO_MOID);
  a68g_op (A68G_EXT, "*", m, genie_vector_complex_times_matrix);
// Matrix times vector or matrix.
  m = a68g_proc (M_ROW_REAL, M_ROW_ROW_REAL, M_ROW_REAL, NO_MOID);
  a68g_op (A68G_EXT, "*", m, genie_matrix_times_vector);
  m = a68g_proc (M_ROW_ROW_REAL, M_ROW_ROW_REAL, M_ROW_ROW_REAL, NO_MOID);
  a68g_op (A68G_EXT, "*", m, genie_matrix_times_matrix);
  m = a68g_proc (M_ROW_COMPLEX, M_ROW_ROW_COMPLEX, M_ROW_COMPLEX, NO_MOID);
  a68g_op (A68G_EXT, "*", m, genie_matrix_complex_times_vector);
  m = a68g_proc (M_ROW_ROW_COMPLEX, M_ROW_ROW_COMPLEX, M_ROW_ROW_COMPLEX, NO_MOID);
  a68g_op (A68G_EXT, "*", m, genie_matrix_complex_times_matrix);
// Vector and matrix miscellaneous.
  m = a68g_proc (M_ROW_REAL, M_ROW_REAL, NO_MOID);
  a68g_idf (A68G_EXT, "vectorecho", m, genie_vector_echo);
  m = a68g_proc (M_ROW_ROW_REAL, M_ROW_ROW_REAL, NO_MOID);
  a68g_idf (A68G_EXT, "matrixecho", m, genie_matrix_echo);
  m = a68g_proc (M_ROW_COMPLEX, M_ROW_COMPLEX, NO_MOID);
  a68g_idf (A68G_EXT, "complvectorecho", m, genie_vector_complex_echo);
  m = a68g_proc (M_ROW_ROW_COMPLEX, M_ROW_ROW_COMPLEX, NO_MOID);
  a68g_idf (A68G_EXT, "complmatrixecho", m, genie_matrix_complex_echo);
  m = a68g_proc (M_REAL, M_ROW_REAL, M_ROW_REAL, NO_MOID);
  a68g_op (A68G_EXT, "*", m, genie_vector_dot);
  m = a68g_proc (M_COMPLEX, M_ROW_COMPLEX, M_ROW_COMPLEX, NO_MOID);
  a68g_op (A68G_EXT, "*", m, genie_vector_complex_dot);
  m = a68g_proc (M_REAL, M_ROW_REAL, NO_MOID);
  a68g_op (A68G_EXT, "NORM", m, genie_vector_norm);
  m = a68g_proc (M_REAL, M_ROW_ROW_REAL, NO_MOID);
  a68g_op (A68G_EXT, "NORM", m, genie_matrix_norm);
  m = a68g_proc (M_REAL, M_ROW_COMPLEX, NO_MOID);
  a68g_op (A68G_EXT, "NORM", m, genie_vector_complex_norm);
  m = a68g_proc (M_ROW_ROW_REAL, M_ROW_REAL, M_ROW_REAL, NO_MOID);
  a68g_op (A68G_EXT, "DYAD", m, genie_vector_dyad);
  m = a68g_proc (M_ROW_ROW_COMPLEX, M_ROW_COMPLEX, M_ROW_COMPLEX, NO_MOID);
  a68g_op (A68G_EXT, "DYAD", m, genie_vector_complex_dyad);
  a68g_prio ("DYAD", 3);
// Principle Component Analysis.
  m = a68g_proc (M_ROW_ROW_REAL, M_ROW_ROW_REAL, M_REF_ROW_REAL, NO_MOID);
  a68g_idf (A68G_EXT, "pcacv", m, genie_matrix_pca_cv);
  a68g_idf (A68G_EXT, "pcasvd", m, genie_matrix_pca_svd);
// Total Least Square regression.
  m = a68g_proc (M_ROW_ROW_REAL, M_ROW_ROW_REAL, M_ROW_ROW_REAL, M_REF_ROW_REAL, NO_MOID);
  a68g_idf (A68G_EXT, "ols", m, genie_matrix_ols);
  a68g_idf (A68G_EXT, "tls", m, genie_matrix_tls);
// Partial Least Squares regression.
  m = a68g_proc (M_ROW_ROW_REAL, M_ROW_ROW_REAL, M_ROW_ROW_REAL, M_REF_ROW_REAL, M_INT, M_REAL, NO_MOID);
  a68g_idf (A68G_EXT, "pcr", m, genie_matrix_pcr);
  a68g_idf (A68G_EXT, "pls1", m, genie_matrix_pls1);
  a68g_idf (A68G_EXT, "pls2", m, genie_matrix_pls2);
// Routine left columns, a GSL alternative to trimming columns.
  m = a68g_proc (M_ROW_ROW_REAL, M_ROW_ROW_REAL, M_INT, NO_MOID);
  a68g_idf (A68G_EXT, "leftcolumns", m, genie_left_columns);
// Moore-Penrose pseudo inverse.
  m = a68g_proc (M_ROW_ROW_REAL, M_ROW_ROW_REAL, M_REAL, NO_MOID);
  a68g_idf (A68G_EXT, "pseudoinv", m, genie_matrix_pinv_lim);
// LU decomposition.
  m = a68g_proc (M_ROW_ROW_REAL, M_ROW_ROW_REAL, M_REF_ROW_INT, M_REF_INT, NO_MOID);
  a68g_idf (A68G_EXT, "ludecomp", m, genie_matrix_lu);
  m = a68g_proc (M_REAL, M_ROW_ROW_REAL, M_INT, NO_MOID);
  a68g_idf (A68G_EXT, "ludet", m, genie_matrix_lu_det);
  m = a68g_proc (M_ROW_ROW_REAL, M_ROW_ROW_REAL, M_ROW_INT, NO_MOID);
  a68g_idf (A68G_EXT, "luinv", m, genie_matrix_lu_inv);
  m = a68g_proc (M_ROW_REAL, M_ROW_ROW_REAL, M_ROW_ROW_REAL, M_ROW_INT, M_ROW_REAL, NO_MOID);
  a68g_idf (A68G_EXT, "lusolve", m, genie_matrix_lu_solve);
  m = a68g_proc (M_ROW_ROW_COMPLEX, M_ROW_ROW_COMPLEX, M_REF_ROW_INT, M_REF_INT, NO_MOID);
  a68g_idf (A68G_EXT, "complexludecomp", m, genie_matrix_complex_lu);
  m = a68g_proc (M_COMPLEX, M_ROW_ROW_COMPLEX, M_INT, NO_MOID);
  a68g_idf (A68G_EXT, "complexludet", m, genie_matrix_complex_lu_det);
  m = a68g_proc (M_ROW_ROW_COMPLEX, M_ROW_ROW_COMPLEX, M_ROW_INT, NO_MOID);
  a68g_idf (A68G_EXT, "complexluinv", m, genie_matrix_complex_lu_inv);
  m = a68g_proc (M_ROW_COMPLEX, M_ROW_ROW_COMPLEX, M_ROW_ROW_COMPLEX, M_ROW_INT, M_ROW_COMPLEX, NO_MOID);
  a68g_idf (A68G_EXT, "complexlusolve", m, genie_matrix_complex_lu_solve);
// SVD decomposition.
  m = a68g_proc (M_VOID, M_ROW_ROW_REAL, M_REF_ROW_ROW_REAL, M_REF_ROW_REAL, M_REF_ROW_ROW_REAL, NO_MOID);
  a68g_idf (A68G_EXT, "svddecomp", m, genie_matrix_svd);
  m = a68g_proc (M_ROW_REAL, M_ROW_ROW_REAL, M_ROW_REAL, M_ROW_ROW_REAL, M_ROW_REAL, NO_MOID);
  a68g_idf (A68G_EXT, "svdsolve", m, genie_matrix_svd_solve);
// QR decomposition.
  m = a68g_proc (M_ROW_ROW_REAL, M_ROW_ROW_REAL, M_REF_ROW_REAL, NO_MOID);
  a68g_idf (A68G_EXT, "qrdecomp", m, genie_matrix_qr);
  m = a68g_proc (M_ROW_REAL, M_ROW_ROW_REAL, M_ROW_REAL, M_ROW_REAL, NO_MOID);
  a68g_idf (A68G_EXT, "qrsolve", m, genie_matrix_qr_solve);
  a68g_idf (A68G_EXT, "qrlssolve", m, genie_matrix_qr_ls_solve);
// Cholesky decomposition.
  m = a68g_proc (M_ROW_ROW_REAL, M_ROW_ROW_REAL, NO_MOID);
  a68g_idf (A68G_EXT, "choleskydecomp", m, genie_matrix_ch);
  m = a68g_proc (M_ROW_REAL, M_ROW_ROW_REAL, M_ROW_REAL, NO_MOID);
  a68g_idf (A68G_EXT, "choleskysolve", m, genie_matrix_ch_solve);
}

void stand_gsl_constants (void)
{
// Constants ex GSL.
  a68g_idf (A68G_EXT, "cgsspeedoflight", M_REAL, genie_cgs_speed_of_light);
  a68g_idf (A68G_EXT, "cgsgravitationalconstant", M_REAL, genie_cgs_gravitational_constant);
  a68g_idf (A68G_EXT, "cgsplanckconstant", M_REAL, genie_cgs_planck_constant_h);
  a68g_idf (A68G_EXT, "cgsplanckconstantbar", M_REAL, genie_cgs_planck_constant_hbar);
  a68g_idf (A68G_EXT, "cgsastronomicalunit", M_REAL, genie_cgs_astronomical_unit);
  a68g_idf (A68G_EXT, "cgslightyear", M_REAL, genie_cgs_light_year);
  a68g_idf (A68G_EXT, "cgsparsec", M_REAL, genie_cgs_parsec);
  a68g_idf (A68G_EXT, "cgsgravaccel", M_REAL, genie_cgs_grav_accel);
  a68g_idf (A68G_EXT, "cgselectronvolt", M_REAL, genie_cgs_electron_volt);
  a68g_idf (A68G_EXT, "cgsmasselectron", M_REAL, genie_cgs_mass_electron);
  a68g_idf (A68G_EXT, "cgsmassmuon", M_REAL, genie_cgs_mass_muon);
  a68g_idf (A68G_EXT, "cgsmassproton", M_REAL, genie_cgs_mass_proton);
  a68g_idf (A68G_EXT, "cgsmassneutron", M_REAL, genie_cgs_mass_neutron);
  a68g_idf (A68G_EXT, "cgsrydberg", M_REAL, genie_cgs_rydberg);
  a68g_idf (A68G_EXT, "cgsboltzmann", M_REAL, genie_cgs_boltzmann);
  a68g_idf (A68G_EXT, "cgsbohrmagneton", M_REAL, genie_cgs_bohr_magneton);
  a68g_idf (A68G_EXT, "cgsnuclearmagneton", M_REAL, genie_cgs_nuclear_magneton);
  a68g_idf (A68G_EXT, "cgselectronmagneticmoment", M_REAL, genie_cgs_electron_magnetic_moment);
  a68g_idf (A68G_EXT, "cgsprotonmagneticmoment", M_REAL, genie_cgs_proton_magnetic_moment);
  a68g_idf (A68G_EXT, "cgsmolargas", M_REAL, genie_cgs_molar_gas);
  a68g_idf (A68G_EXT, "cgsstandardgasvolume", M_REAL, genie_cgs_standard_gas_volume);
  a68g_idf (A68G_EXT, "cgsminute", M_REAL, genie_cgs_minute);
  a68g_idf (A68G_EXT, "cgshour", M_REAL, genie_cgs_hour);
  a68g_idf (A68G_EXT, "cgsday", M_REAL, genie_cgs_day);
  a68g_idf (A68G_EXT, "cgsweek", M_REAL, genie_cgs_week);
  a68g_idf (A68G_EXT, "cgsinch", M_REAL, genie_cgs_inch);
  a68g_idf (A68G_EXT, "cgsfoot", M_REAL, genie_cgs_foot);
  a68g_idf (A68G_EXT, "cgsyard", M_REAL, genie_cgs_yard);
  a68g_idf (A68G_EXT, "cgsmile", M_REAL, genie_cgs_mile);
  a68g_idf (A68G_EXT, "cgsnauticalmile", M_REAL, genie_cgs_nautical_mile);
  a68g_idf (A68G_EXT, "cgsfathom", M_REAL, genie_cgs_fathom);
  a68g_idf (A68G_EXT, "cgsmil", M_REAL, genie_cgs_mil);
  a68g_idf (A68G_EXT, "cgspoint", M_REAL, genie_cgs_point);
  a68g_idf (A68G_EXT, "cgstexpoint", M_REAL, genie_cgs_texpoint);
  a68g_idf (A68G_EXT, "cgsmicron", M_REAL, genie_cgs_micron);
  a68g_idf (A68G_EXT, "cgsangstrom", M_REAL, genie_cgs_angstrom);
  a68g_idf (A68G_EXT, "cgshectare", M_REAL, genie_cgs_hectare);
  a68g_idf (A68G_EXT, "cgsacre", M_REAL, genie_cgs_acre);
  a68g_idf (A68G_EXT, "cgsbarn", M_REAL, genie_cgs_barn);
  a68g_idf (A68G_EXT, "cgsliter", M_REAL, genie_cgs_liter);
  a68g_idf (A68G_EXT, "cgsusgallon", M_REAL, genie_cgs_us_gallon);
  a68g_idf (A68G_EXT, "cgsquart", M_REAL, genie_cgs_quart);
  a68g_idf (A68G_EXT, "cgspint", M_REAL, genie_cgs_pint);
  a68g_idf (A68G_EXT, "cgscup", M_REAL, genie_cgs_cup);
  a68g_idf (A68G_EXT, "cgsfluidounce", M_REAL, genie_cgs_fluid_ounce);
  a68g_idf (A68G_EXT, "cgstablespoon", M_REAL, genie_cgs_tablespoon);
  a68g_idf (A68G_EXT, "cgsteaspoon", M_REAL, genie_cgs_teaspoon);
  a68g_idf (A68G_EXT, "cgscanadiangallon", M_REAL, genie_cgs_canadian_gallon);
  a68g_idf (A68G_EXT, "cgsukgallon", M_REAL, genie_cgs_uk_gallon);
  a68g_idf (A68G_EXT, "cgsmilesperhour", M_REAL, genie_cgs_miles_per_hour);
  a68g_idf (A68G_EXT, "cgskilometersperhour", M_REAL, genie_cgs_kilometers_per_hour);
  a68g_idf (A68G_EXT, "cgsknot", M_REAL, genie_cgs_knot);
  a68g_idf (A68G_EXT, "cgspoundmass", M_REAL, genie_cgs_pound_mass);
  a68g_idf (A68G_EXT, "cgsouncemass", M_REAL, genie_cgs_ounce_mass);
  a68g_idf (A68G_EXT, "cgston", M_REAL, genie_cgs_ton);
  a68g_idf (A68G_EXT, "cgsmetricton", M_REAL, genie_cgs_metric_ton);
  a68g_idf (A68G_EXT, "cgsukton", M_REAL, genie_cgs_uk_ton);
  a68g_idf (A68G_EXT, "cgstroyounce", M_REAL, genie_cgs_troy_ounce);
  a68g_idf (A68G_EXT, "cgscarat", M_REAL, genie_cgs_carat);
  a68g_idf (A68G_EXT, "cgsunifiedatomicmass", M_REAL, genie_cgs_unified_atomic_mass);
  a68g_idf (A68G_EXT, "cgsgramforce", M_REAL, genie_cgs_gram_force);
  a68g_idf (A68G_EXT, "cgspoundforce", M_REAL, genie_cgs_pound_force);
  a68g_idf (A68G_EXT, "cgskilopoundforce", M_REAL, genie_cgs_kilopound_force);
  a68g_idf (A68G_EXT, "cgspoundal", M_REAL, genie_cgs_poundal);
  a68g_idf (A68G_EXT, "cgscalorie", M_REAL, genie_cgs_calorie);
  a68g_idf (A68G_EXT, "cgsbtu", M_REAL, genie_cgs_btu);
  a68g_idf (A68G_EXT, "cgstherm", M_REAL, genie_cgs_therm);
  a68g_idf (A68G_EXT, "cgshorsepower", M_REAL, genie_cgs_horsepower);
  a68g_idf (A68G_EXT, "cgsbar", M_REAL, genie_cgs_bar);
  a68g_idf (A68G_EXT, "cgsstdatmosphere", M_REAL, genie_cgs_std_atmosphere);
  a68g_idf (A68G_EXT, "cgstorr", M_REAL, genie_cgs_torr);
  a68g_idf (A68G_EXT, "cgsmeterofmercury", M_REAL, genie_cgs_meter_of_mercury);
  a68g_idf (A68G_EXT, "cgsinchofmercury", M_REAL, genie_cgs_inch_of_mercury);
  a68g_idf (A68G_EXT, "cgsinchofwater", M_REAL, genie_cgs_inch_of_water);
  a68g_idf (A68G_EXT, "cgspsi", M_REAL, genie_cgs_psi);
  a68g_idf (A68G_EXT, "cgspoise", M_REAL, genie_cgs_poise);
  a68g_idf (A68G_EXT, "cgsstokes", M_REAL, genie_cgs_stokes);
  a68g_idf (A68G_EXT, "cgsfaraday", M_REAL, genie_cgs_faraday);
  a68g_idf (A68G_EXT, "cgselectroncharge", M_REAL, genie_cgs_electron_charge);
  a68g_idf (A68G_EXT, "cgsgauss", M_REAL, genie_cgs_gauss);
  a68g_idf (A68G_EXT, "cgsstilb", M_REAL, genie_cgs_stilb);
  a68g_idf (A68G_EXT, "cgslumen", M_REAL, genie_cgs_lumen);
  a68g_idf (A68G_EXT, "cgslux", M_REAL, genie_cgs_lux);
  a68g_idf (A68G_EXT, "cgsphot", M_REAL, genie_cgs_phot);
  a68g_idf (A68G_EXT, "cgsfootcandle", M_REAL, genie_cgs_footcandle);
  a68g_idf (A68G_EXT, "cgslambert", M_REAL, genie_cgs_lambert);
  a68g_idf (A68G_EXT, "cgsfootlambert", M_REAL, genie_cgs_footlambert);
  a68g_idf (A68G_EXT, "cgscurie", M_REAL, genie_cgs_curie);
  a68g_idf (A68G_EXT, "cgsroentgen", M_REAL, genie_cgs_roentgen);
  a68g_idf (A68G_EXT, "cgsrad", M_REAL, genie_cgs_rad);
  a68g_idf (A68G_EXT, "cgssolarmass", M_REAL, genie_cgs_solar_mass);
  a68g_idf (A68G_EXT, "cgsbohrradius", M_REAL, genie_cgs_bohr_radius);
  a68g_idf (A68G_EXT, "cgsnewton", M_REAL, genie_cgs_newton);
  a68g_idf (A68G_EXT, "cgsdyne", M_REAL, genie_cgs_dyne);
  a68g_idf (A68G_EXT, "cgsjoule", M_REAL, genie_cgs_joule);
  a68g_idf (A68G_EXT, "cgserg", M_REAL, genie_cgs_erg);
  a68g_idf (A68G_EXT, "mksaspeedoflight", M_REAL, genie_mks_speed_of_light);
  a68g_idf (A68G_EXT, "mksagravitationalconstant", M_REAL, genie_mks_gravitational_constant);
  a68g_idf (A68G_EXT, "mksaplanckconstant", M_REAL, genie_mks_planck_constant_h);
  a68g_idf (A68G_EXT, "mksaplanckconstantbar", M_REAL, genie_mks_planck_constant_hbar);
  a68g_idf (A68G_EXT, "mksavacuumpermeability", M_REAL, genie_mks_vacuum_permeability);
  a68g_idf (A68G_EXT, "mksaastronomicalunit", M_REAL, genie_mks_astronomical_unit);
  a68g_idf (A68G_EXT, "mksalightyear", M_REAL, genie_mks_light_year);
  a68g_idf (A68G_EXT, "mksaparsec", M_REAL, genie_mks_parsec);
  a68g_idf (A68G_EXT, "mksagravaccel", M_REAL, genie_mks_grav_accel);
  a68g_idf (A68G_EXT, "mksaelectronvolt", M_REAL, genie_mks_electron_volt);
  a68g_idf (A68G_EXT, "mksamasselectron", M_REAL, genie_mks_mass_electron);
  a68g_idf (A68G_EXT, "mksamassmuon", M_REAL, genie_mks_mass_muon);
  a68g_idf (A68G_EXT, "mksamassproton", M_REAL, genie_mks_mass_proton);
  a68g_idf (A68G_EXT, "mksamassneutron", M_REAL, genie_mks_mass_neutron);
  a68g_idf (A68G_EXT, "mksarydberg", M_REAL, genie_mks_rydberg);
  a68g_idf (A68G_EXT, "mksaboltzmann", M_REAL, genie_mks_boltzmann);
  a68g_idf (A68G_EXT, "mksabohrmagneton", M_REAL, genie_mks_bohr_magneton);
  a68g_idf (A68G_EXT, "mksanuclearmagneton", M_REAL, genie_mks_nuclear_magneton);
  a68g_idf (A68G_EXT, "mksaelectronmagneticmoment", M_REAL, genie_mks_electron_magnetic_moment);
  a68g_idf (A68G_EXT, "mksaprotonmagneticmoment", M_REAL, genie_mks_proton_magnetic_moment);
  a68g_idf (A68G_EXT, "mksamolargas", M_REAL, genie_mks_molar_gas);
  a68g_idf (A68G_EXT, "mksastandardgasvolume", M_REAL, genie_mks_standard_gas_volume);
  a68g_idf (A68G_EXT, "mksaminute", M_REAL, genie_mks_minute);
  a68g_idf (A68G_EXT, "mksahour", M_REAL, genie_mks_hour);
  a68g_idf (A68G_EXT, "mksaday", M_REAL, genie_mks_day);
  a68g_idf (A68G_EXT, "mksaweek", M_REAL, genie_mks_week);
  a68g_idf (A68G_EXT, "mksainch", M_REAL, genie_mks_inch);
  a68g_idf (A68G_EXT, "mksafoot", M_REAL, genie_mks_foot);
  a68g_idf (A68G_EXT, "mksayard", M_REAL, genie_mks_yard);
  a68g_idf (A68G_EXT, "mksamile", M_REAL, genie_mks_mile);
  a68g_idf (A68G_EXT, "mksanauticalmile", M_REAL, genie_mks_nautical_mile);
  a68g_idf (A68G_EXT, "mksafathom", M_REAL, genie_mks_fathom);
  a68g_idf (A68G_EXT, "mksamil", M_REAL, genie_mks_mil);
  a68g_idf (A68G_EXT, "mksapoint", M_REAL, genie_mks_point);
  a68g_idf (A68G_EXT, "mksatexpoint", M_REAL, genie_mks_texpoint);
  a68g_idf (A68G_EXT, "mksamicron", M_REAL, genie_mks_micron);
  a68g_idf (A68G_EXT, "mksaangstrom", M_REAL, genie_mks_angstrom);
  a68g_idf (A68G_EXT, "mksahectare", M_REAL, genie_mks_hectare);
  a68g_idf (A68G_EXT, "mksaacre", M_REAL, genie_mks_acre);
  a68g_idf (A68G_EXT, "mksabarn", M_REAL, genie_mks_barn);
  a68g_idf (A68G_EXT, "mksaliter", M_REAL, genie_mks_liter);
  a68g_idf (A68G_EXT, "mksausgallon", M_REAL, genie_mks_us_gallon);
  a68g_idf (A68G_EXT, "mksaquart", M_REAL, genie_mks_quart);
  a68g_idf (A68G_EXT, "mksapint", M_REAL, genie_mks_pint);
  a68g_idf (A68G_EXT, "mksacup", M_REAL, genie_mks_cup);
  a68g_idf (A68G_EXT, "mksafluidounce", M_REAL, genie_mks_fluid_ounce);
  a68g_idf (A68G_EXT, "mksatablespoon", M_REAL, genie_mks_tablespoon);
  a68g_idf (A68G_EXT, "mksateaspoon", M_REAL, genie_mks_teaspoon);
  a68g_idf (A68G_EXT, "mksacanadiangallon", M_REAL, genie_mks_canadian_gallon);
  a68g_idf (A68G_EXT, "mksaukgallon", M_REAL, genie_mks_uk_gallon);
  a68g_idf (A68G_EXT, "mksamilesperhour", M_REAL, genie_mks_miles_per_hour);
  a68g_idf (A68G_EXT, "mksakilometersperhour", M_REAL, genie_mks_kilometers_per_hour);
  a68g_idf (A68G_EXT, "mksaknot", M_REAL, genie_mks_knot);
  a68g_idf (A68G_EXT, "mksapoundmass", M_REAL, genie_mks_pound_mass);
  a68g_idf (A68G_EXT, "mksaouncemass", M_REAL, genie_mks_ounce_mass);
  a68g_idf (A68G_EXT, "mksaton", M_REAL, genie_mks_ton);
  a68g_idf (A68G_EXT, "mksametricton", M_REAL, genie_mks_metric_ton);
  a68g_idf (A68G_EXT, "mksaukton", M_REAL, genie_mks_uk_ton);
  a68g_idf (A68G_EXT, "mksatroyounce", M_REAL, genie_mks_troy_ounce);
  a68g_idf (A68G_EXT, "mksacarat", M_REAL, genie_mks_carat);
  a68g_idf (A68G_EXT, "mksaunifiedatomicmass", M_REAL, genie_mks_unified_atomic_mass);
  a68g_idf (A68G_EXT, "mksagramforce", M_REAL, genie_mks_gram_force);
  a68g_idf (A68G_EXT, "mksapoundforce", M_REAL, genie_mks_pound_force);
  a68g_idf (A68G_EXT, "mksakilopoundforce", M_REAL, genie_mks_kilopound_force);
  a68g_idf (A68G_EXT, "mksapoundal", M_REAL, genie_mks_poundal);
  a68g_idf (A68G_EXT, "mksacalorie", M_REAL, genie_mks_calorie);
  a68g_idf (A68G_EXT, "mksabtu", M_REAL, genie_mks_btu);
  a68g_idf (A68G_EXT, "mksatherm", M_REAL, genie_mks_therm);
  a68g_idf (A68G_EXT, "mksahorsepower", M_REAL, genie_mks_horsepower);
  a68g_idf (A68G_EXT, "mksabar", M_REAL, genie_mks_bar);
  a68g_idf (A68G_EXT, "mksastdatmosphere", M_REAL, genie_mks_std_atmosphere);
  a68g_idf (A68G_EXT, "mksatorr", M_REAL, genie_mks_torr);
  a68g_idf (A68G_EXT, "mksameterofmercury", M_REAL, genie_mks_meter_of_mercury);
  a68g_idf (A68G_EXT, "mksainchofmercury", M_REAL, genie_mks_inch_of_mercury);
  a68g_idf (A68G_EXT, "mksainchofwater", M_REAL, genie_mks_inch_of_water);
  a68g_idf (A68G_EXT, "mksapsi", M_REAL, genie_mks_psi);
  a68g_idf (A68G_EXT, "mksapoise", M_REAL, genie_mks_poise);
  a68g_idf (A68G_EXT, "mksastokes", M_REAL, genie_mks_stokes);
  a68g_idf (A68G_EXT, "mksafaraday", M_REAL, genie_mks_faraday);
  a68g_idf (A68G_EXT, "mksaelectroncharge", M_REAL, genie_mks_electron_charge);
  a68g_idf (A68G_EXT, "mksagauss", M_REAL, genie_mks_gauss);
  a68g_idf (A68G_EXT, "mksastilb", M_REAL, genie_mks_stilb);
  a68g_idf (A68G_EXT, "mksalumen", M_REAL, genie_mks_lumen);
  a68g_idf (A68G_EXT, "mksalux", M_REAL, genie_mks_lux);
  a68g_idf (A68G_EXT, "mksaphot", M_REAL, genie_mks_phot);
  a68g_idf (A68G_EXT, "mksafootcandle", M_REAL, genie_mks_footcandle);
  a68g_idf (A68G_EXT, "mksalambert", M_REAL, genie_mks_lambert);
  a68g_idf (A68G_EXT, "mksafootlambert", M_REAL, genie_mks_footlambert);
  a68g_idf (A68G_EXT, "mksacurie", M_REAL, genie_mks_curie);
  a68g_idf (A68G_EXT, "mksaroentgen", M_REAL, genie_mks_roentgen);
  a68g_idf (A68G_EXT, "mksarad", M_REAL, genie_mks_rad);
  a68g_idf (A68G_EXT, "mksasolarmass", M_REAL, genie_mks_solar_mass);
  a68g_idf (A68G_EXT, "mksabohrradius", M_REAL, genie_mks_bohr_radius);
  a68g_idf (A68G_EXT, "mksavacuumpermittivity", M_REAL, genie_mks_vacuum_permittivity);
  a68g_idf (A68G_EXT, "mksanewton", M_REAL, genie_mks_newton);
  a68g_idf (A68G_EXT, "mksadyne", M_REAL, genie_mks_dyne);
  a68g_idf (A68G_EXT, "mksajoule", M_REAL, genie_mks_joule);
  a68g_idf (A68G_EXT, "mksaerg", M_REAL, genie_mks_erg);
  a68g_idf (A68G_EXT, "numfinestructure", M_REAL, genie_num_fine_structure);
  a68g_idf (A68G_EXT, "numavogadro", M_REAL, genie_num_avogadro);
  a68g_idf (A68G_EXT, "numyotta", M_REAL, genie_num_yotta);
  a68g_idf (A68G_EXT, "numzetta", M_REAL, genie_num_zetta);
  a68g_idf (A68G_EXT, "numexa", M_REAL, genie_num_exa);
  a68g_idf (A68G_EXT, "numpeta", M_REAL, genie_num_peta);
  a68g_idf (A68G_EXT, "numtera", M_REAL, genie_num_tera);
  a68g_idf (A68G_EXT, "numgiga", M_REAL, genie_num_giga);
  a68g_idf (A68G_EXT, "nummega", M_REAL, genie_num_mega);
  a68g_idf (A68G_EXT, "numkilo", M_REAL, genie_num_kilo);
  a68g_idf (A68G_EXT, "nummilli", M_REAL, genie_num_milli);
  a68g_idf (A68G_EXT, "nummicro", M_REAL, genie_num_micro);
  a68g_idf (A68G_EXT, "numnano", M_REAL, genie_num_nano);
  a68g_idf (A68G_EXT, "numpico", M_REAL, genie_num_pico);
  a68g_idf (A68G_EXT, "numfemto", M_REAL, genie_num_femto);
  a68g_idf (A68G_EXT, "numatto", M_REAL, genie_num_atto);
  a68g_idf (A68G_EXT, "numzepto", M_REAL, genie_num_zepto);
  a68g_idf (A68G_EXT, "numyocto", M_REAL, genie_num_yocto);
}

void stand_gsl_fft_laplace (void)
{
// FFT.
  MOID_T *m = a68g_proc (M_ROW_INT, M_INT, NO_MOID);
  a68g_idf (A68G_EXT, "primefactors", m, genie_prime_factors);
  m = a68g_proc (M_ROW_COMPLEX, M_ROW_COMPLEX, NO_MOID);
  a68g_idf (A68G_EXT, "fftcomplexforward", m, genie_fft_complex_forward);
  a68g_idf (A68G_EXT, "fftcomplexbackward", m, genie_fft_complex_backward);
  a68g_idf (A68G_EXT, "fftcomplexinverse", m, genie_fft_complex_inverse);
  m = a68g_proc (M_ROW_COMPLEX, M_ROW_REAL, NO_MOID);
  a68g_idf (A68G_EXT, "fftforward", m, genie_fft_forward);
  m = a68g_proc (M_ROW_REAL, M_ROW_COMPLEX, NO_MOID);
  a68g_idf (A68G_EXT, "fftbackward", m, genie_fft_backward);
  a68g_idf (A68G_EXT, "fftinverse", m, genie_fft_inverse);
// Laplace.
  m = a68g_proc (M_REAL, A68G_MCACHE (proc_real_real), M_REAL, M_REF_REAL, NO_MOID);
  a68g_idf (A68G_EXT, "laplace", m, genie_laplace);
}

void stand_gsl (void)
{
  stand_gsl_sf ();
  stand_gsl_linear_algebra ();
  stand_gsl_constants ();
  stand_gsl_fft_laplace ();
}

#endif
