/*
 * Decompiled with CFR 0.152.
 */
package org.jkiss.dbeaver.registry.storage;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.sql.Connection;
import java.sql.Driver;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.Properties;
import org.jkiss.code.NotNull;
import org.jkiss.dbeaver.DBException;
import org.jkiss.dbeaver.Log;
import org.jkiss.dbeaver.model.connection.DBPDriver;
import org.jkiss.dbeaver.model.connection.InternalDatabaseConfig;
import org.jkiss.dbeaver.model.runtime.DBRProgressMonitor;
import org.jkiss.dbeaver.registry.DataSourceProviderRegistry;
import org.jkiss.utils.CommonUtils;

public class H2Migrator {
    private static final Log log = Log.getLog(H2Migrator.class);
    public static final String V1_DRIVER_NAME = "h2_embedded";
    public static final String V2_DRIVER_NAME = "h2_embedded_v2";
    private static final String EXPORT_SCRIPT = "SCRIPT TO ?";
    private static final String EXPORT_SCRIPT_USING_PASSWORD = "SCRIPT TO ? COMPRESSION DEFLATE CIPHER AES PASSWORD ? CHARSET 'UTF-8'";
    private static final String IMPORT_SCRIPT = "RUNSCRIPT FROM ? FROM_1X";
    private static final String IMPORT_SCRIPT_USING_PASSWORD = "RUNSCRIPT FROM ? COMPRESSION DEFLATE CIPHER AES PASSWORD ? CHARSET 'UTF-8' FROM_1X";
    @NotNull
    private final DBRProgressMonitor monitor;
    @NotNull
    private final DataSourceProviderRegistry dataSourceProviderRegistry;
    @NotNull
    private final InternalDatabaseConfig databaseConfiguration;
    @NotNull
    private final Properties dbProperties;

    public H2Migrator(@NotNull DBRProgressMonitor monitor, @NotNull DataSourceProviderRegistry dataSourceProviderRegistry, @NotNull InternalDatabaseConfig databaseConfiguration, @NotNull Properties dbProperties) {
        this.monitor = monitor;
        this.dataSourceProviderRegistry = dataSourceProviderRegistry;
        this.databaseConfiguration = databaseConfiguration;
        this.dbProperties = dbProperties;
    }

    public void migrateDatabaseIfNeeded(@NotNull String dbNameV1, @NotNull String dbNameV2) {
        String resolvedDbUrl = this.databaseConfiguration.getResolvedUrl();
        if (!resolvedDbUrl.endsWith(dbNameV1) || !V1_DRIVER_NAME.equals(this.databaseConfiguration.getDriver()) || resolvedDbUrl.startsWith("jdbc:h2:mem:")) {
            log.trace((Object)"No migration needed");
            return;
        }
        WorkspacePaths workspacePaths = new WorkspacePaths(resolvedDbUrl, dbNameV1, dbNameV2);
        if (workspacePaths.v2Paths.dbDataFile.toFile().exists() && (resolvedDbUrl.endsWith(dbNameV1) || V1_DRIVER_NAME.equals(this.databaseConfiguration.getDriver()))) {
            this.updateConfig(workspacePaths);
            return;
        }
        String oldUrl = this.databaseConfiguration.getUrl();
        String oldDriver = this.databaseConfiguration.getDriver();
        try {
            this.migrateDatabase(workspacePaths);
            log.debug((Object)"H2 v1->v2 migration was successful");
        }
        catch (Exception e) {
            log.error((Object)"Migration H2 v1->v2 failed", (Throwable)e);
            this.rollback(workspacePaths, oldUrl, oldDriver);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void migrateDatabase(@NotNull WorkspacePaths workspacePaths) throws DBException, SQLException, IOException {
        try {
            this.monitor.beginTask("H2 database v1 -> v2 migration started", 3);
            Driver v1Driver = this.getDriver(V1_DRIVER_NAME);
            Driver v2Driver = this.getDriver(V2_DRIVER_NAME);
            String exportFilePath = workspacePaths.exportFilePath.toString();
            this.monitor.subTask("Exporting v1 database");
            if (this.dbProperties.getProperty("password") == null) {
                this.executeScript(v1Driver, EXPORT_SCRIPT, exportFilePath);
            } else {
                this.executeScript(v1Driver, EXPORT_SCRIPT_USING_PASSWORD, exportFilePath);
            }
            this.monitor.worked(1);
            this.monitor.subTask("Creating v1 database backup '" + String.valueOf(workspacePaths.v1DataBackupPath) + "'");
            Files.move(workspacePaths.v1Paths.dbDataFile, workspacePaths.v1DataBackupPath, StandardCopyOption.REPLACE_EXISTING);
            if (workspacePaths.v1Paths.dbTraceFile.toFile().exists()) {
                Files.move(workspacePaths.v1Paths.dbTraceFile, workspacePaths.v1TraceBackupPath, StandardCopyOption.REPLACE_EXISTING);
            }
            this.updateConfig(workspacePaths);
            this.monitor.worked(1);
            this.monitor.subTask("Importing data to new v2 database");
            if (this.dbProperties.getProperty("password") == null) {
                this.executeScript(v2Driver, IMPORT_SCRIPT, exportFilePath);
            } else {
                this.executeScript(v2Driver, IMPORT_SCRIPT_USING_PASSWORD, exportFilePath);
            }
            this.removeExportFile(workspacePaths);
            log.debug((Object)("Export file removed '" + String.valueOf(workspacePaths.exportFilePath) + "'"));
            this.monitor.worked(1);
        }
        finally {
            this.monitor.done();
        }
    }

    @NotNull
    private Driver getDriver(@NotNull String driverName) throws DBException {
        DBPDriver driverDescriptor = this.dataSourceProviderRegistry.findDriver(driverName);
        if (driverDescriptor == null) {
            throw new DBException("Driver '" + driverName + "' couldn't be resolved");
        }
        return (Driver)driverDescriptor.getDefaultDriverLoader().getDriverInstance(this.monitor);
    }

    private void executeScript(@NotNull Driver driver, @NotNull String script, @NotNull String filePath) throws SQLException {
        try (Connection connection = driver.connect(this.databaseConfiguration.getResolvedUrl(), this.dbProperties);
             PreparedStatement statement = connection.prepareStatement(script);){
            statement.setString(1, filePath);
            String password = this.dbProperties.getProperty("password");
            if (password != null) {
                statement.setString(2, password);
            }
            statement.execute();
        }
    }

    private void updateConfig(@NotNull WorkspacePaths workspacePaths) {
        String updatedDbUrl;
        if (!V2_DRIVER_NAME.equals(this.databaseConfiguration.getDriver())) {
            log.debug((Object)"Using database driver 'h2_embedded_v2' instead of 'h2_embedded' from config");
            this.databaseConfiguration.setDriver(V2_DRIVER_NAME);
        }
        if (!(updatedDbUrl = CommonUtils.replaceLast((String)this.databaseConfiguration.getUrl(), (String)workspacePaths.v1Paths.dbName, (String)workspacePaths.v2Paths.dbName)).equals(this.databaseConfiguration.getUrl())) {
            log.debug((Object)("Using database file '" + String.valueOf(workspacePaths.v2Paths.dbDataFile) + "' instead of '" + String.valueOf(workspacePaths.v1Paths.dbDataFile) + "' from config"));
            this.databaseConfiguration.setUrl(updatedDbUrl);
        }
    }

    private void removeExportFile(@NotNull WorkspacePaths workspacePaths) {
        File exportFile = workspacePaths.exportFilePath.toFile();
        if (exportFile.exists() && !exportFile.delete()) {
            log.error((Object)"Unable to remove H2 v1 export script file");
        }
    }

    private void rollback(@NotNull WorkspacePaths workspacePaths, @NotNull String oldUrl, @NotNull String oldDriver) {
        this.removeExportFile(workspacePaths);
        try {
            Files.move(workspacePaths.v1DataBackupPath, workspacePaths.v1Paths.dbDataFile, StandardCopyOption.REPLACE_EXISTING);
            if (workspacePaths.v1TraceBackupPath.toFile().exists()) {
                Files.move(workspacePaths.v1TraceBackupPath, workspacePaths.v1Paths.dbTraceFile, StandardCopyOption.REPLACE_EXISTING);
            }
            log.debug((Object)"v1 files restored");
        }
        catch (IOException e) {
            log.error((Object)("Unable to restore old database file '" + String.valueOf(workspacePaths.v1Paths.dbDataFile) + "'"), (Throwable)e);
        }
        this.databaseConfiguration.setUrl(oldUrl);
        this.databaseConfiguration.setDriver(oldDriver);
    }

    public static boolean isH2Database(InternalDatabaseConfig databaseConfiguration) {
        return databaseConfiguration.getUrl().startsWith("jdbc:h2");
    }

    private static class WorkspacePaths {
        private static final String V1_DATA_BACKUP_FILE_NAME = "h2db_v1_backup";
        private static final String V1_TRACE_BACKUP_FILE_NAME = "h2db_trace_v1_backup";
        private static final String EXPORT_SCRIPT_FILE_NAME = "H2v1ExportScript";
        @NotNull
        private final H2FilesPaths v1Paths;
        @NotNull
        private final H2FilesPaths v2Paths;
        @NotNull
        private final Path v1DataBackupPath;
        @NotNull
        private final Path v1TraceBackupPath;
        @NotNull
        private final Path exportFilePath;

        private WorkspacePaths(@NotNull String resolvedDbUrl, @NotNull String dbNameV1, @NotNull String dbNameV2) {
            Path dbFolderPath = WorkspacePaths.getFolderPath(resolvedDbUrl);
            this.v1Paths = new H2FilesPaths(dbFolderPath, dbNameV1);
            this.v2Paths = new H2FilesPaths(dbFolderPath, dbNameV2);
            this.v1DataBackupPath = dbFolderPath.resolve(this.createBackupFileName(this.v1Paths.dbDataFile));
            this.v1TraceBackupPath = dbFolderPath.resolve(this.createBackupFileName(this.v1Paths.dbTraceFile));
            this.exportFilePath = dbFolderPath.resolve(EXPORT_SCRIPT_FILE_NAME);
        }

        private String createBackupFileName(Path file) {
            String backupFileName = file.getFileName().toString() + ".backup";
            if (!backupFileName.startsWith(".")) {
                backupFileName = "." + backupFileName;
            }
            return backupFileName;
        }

        @NotNull
        private static Path getFolderPath(@NotNull String resolvedDbUrl) {
            Path filePath = Paths.get(resolvedDbUrl.substring("jdbc:h2:".length()), new String[0]);
            return filePath.getParent();
        }
    }

    private static class H2FilesPaths {
        @NotNull
        private final String dbName;
        @NotNull
        private final Path dbDataFile;
        @NotNull
        private final Path dbTraceFile;

        private H2FilesPaths(@NotNull Path folderPath, @NotNull String dbName) {
            this.dbName = dbName;
            this.dbDataFile = folderPath.resolve(dbName + ".mv.db");
            this.dbTraceFile = folderPath.resolve(dbName + ".trace.db");
        }
    }
}

