/*
 * Copyright (C) 2023 - 2025, Stephan Mueller <smueller@chronox.de>
 *
 * License: see LICENSE file in root directory
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, ALL OF
 * WHICH ARE HEREBY DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF NOT ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */

#include "ext_headers_internal.h"

#include "compare.h"
#include "kyber_x25519_internal.h"
#include "ret_checkers.h"
#include "small_stack_support.h"
#include "selftest_rng.h"
#include "test_helper_common.h"
#include "visibility.h"

#if LC_KYBER_K == 2
static const struct lc_kyber_x25519_pk kyber_pk_exp = {
	.pk.pk = { 0xc2, 0x9a, 0xc6, 0x6c, 0x84, 0xbe, 0xe3, 0xf1, 0x29, 0x50,
		   0x8c, 0x2b, 0x8c, 0x79, 0x0c, 0x99, 0xa5, 0xca, 0x41, 0xe5,
		   0x70, 0x7e, 0x9b, 0x8c, 0x75, 0xc0, 0x4d, 0x7e, 0xa8, 0xa4,
		   0x81, 0x98, 0x0a, 0x35, 0x8b, 0x06, 0x6a, 0x4a, 0x7e, 0x28,
		   0xd1, 0x5d, 0x10, 0x37, 0x4f, 0x75, 0xc3, 0x3d, 0xa6, 0x02,
		   0x9c, 0xd4, 0x90, 0x74, 0x6f, 0xb5, 0x5f, 0x5f, 0xea, 0xb2,
		   0xce, 0x82, 0x3d, 0xec, 0x18, 0x30, 0xc0, 0xc1, 0x8e, 0xf8,
		   0x9b, 0xa3, 0xcc, 0x3b, 0xce, 0x4d, 0x25, 0x2a, 0x07, 0xa4,
		   0x0d, 0x40, 0x1a, 0x4c, 0x8d, 0x27, 0x36, 0x18, 0xb5, 0x95,
		   0xdb, 0x21, 0xca, 0x4a, 0x95, 0x9e, 0xb4, 0xd1, 0x33, 0x55,
		   0x68, 0x01, 0xb2, 0xb7, 0x82, 0x54, 0xb2, 0xb5, 0x95, 0x5c,
		   0x04, 0x00, 0x96, 0x2d, 0xbb, 0x5a, 0x48, 0x7a, 0xaa, 0x24,
		   0xa4, 0x30, 0xb6, 0x14, 0xe2, 0xaf, 0x93, 0x38, 0xaf, 0xd4,
		   0xb0, 0x33, 0x9b, 0xd8, 0x30, 0xa9, 0xcb, 0x76, 0x1d, 0xb1,
		   0xa8, 0x32, 0x20, 0x35, 0x2d, 0xd5, 0x23, 0xb8, 0xa5, 0x83,
		   0x02, 0x2c, 0xa1, 0x3e, 0x24, 0x65, 0x06, 0x54, 0x8c, 0x57,
		   0xba, 0x38, 0x50, 0xac, 0x5c, 0x50, 0xd8, 0x64, 0xe2, 0xd4,
		   0x8b, 0x89, 0x69, 0x42, 0x77, 0xcc, 0x7a, 0x81, 0x74, 0xad,
		   0x7b, 0x1b, 0xa2, 0xee, 0x62, 0x1d, 0x57, 0xd2, 0x9b, 0x23,
		   0xc2, 0x2d, 0xf8, 0x31, 0x87, 0x20, 0x51, 0x14, 0x5d, 0x25,
		   0x35, 0xdb, 0xb0, 0x25, 0xe2, 0x9c, 0x6d, 0xa3, 0x8c, 0xb4,
		   0x75, 0x95, 0x8e, 0x2a, 0x88, 0x08, 0xa4, 0x31, 0xbe, 0xd5,
		   0xb4, 0x86, 0x82, 0x1a, 0x15, 0x13, 0xa7, 0x29, 0xcd, 0x97,
		   0x9b, 0x6c, 0x8b, 0x38, 0x25, 0x03, 0xcf, 0x53, 0x33, 0x76,
		   0x16, 0xe1, 0x05, 0x9d, 0xdc, 0x59, 0x97, 0x72, 0x19, 0xd2,
		   0xf1, 0x7b, 0x32, 0xac, 0xab, 0xbe, 0x86, 0xc4, 0x68, 0xc1,
		   0x26, 0x7b, 0x2b, 0x86, 0x2a, 0xc5, 0x65, 0xa7, 0xe2, 0x32,
		   0x26, 0x6c, 0x83, 0xfc, 0x70, 0x0c, 0xda, 0x19, 0xb6, 0x16,
		   0x37, 0x73, 0x89, 0x42, 0x8e, 0x62, 0x61, 0x4c, 0xd6, 0xac,
		   0x9f, 0x08, 0xfb, 0x12, 0xd2, 0x18, 0x27, 0xfe, 0xb3, 0x24,
		   0xb1, 0x4b, 0xc1, 0x6c, 0xc0, 0xc2, 0x09, 0x04, 0xb3, 0x97,
		   0x3b, 0x9d, 0xfe, 0x60, 0xaf, 0x57, 0x77, 0x76, 0x44, 0x9a,
		   0xc3, 0x4e, 0xb0, 0x09, 0x89, 0xe5, 0xa8, 0x76, 0x14, 0x3b,
		   0xc9, 0xb9, 0x30, 0xa3, 0xc2, 0xa5, 0xbb, 0x86, 0x12, 0x08,
		   0xf2, 0x25, 0xdd, 0x97, 0x62, 0x5b, 0xb3, 0x6b, 0x2d, 0x88,
		   0x10, 0xd0, 0x45, 0x2e, 0x45, 0x89, 0x68, 0xb9, 0x0c, 0x6a,
		   0x6d, 0x97, 0xc2, 0xdf, 0xca, 0x73, 0x41, 0xd5, 0x3c, 0x74,
		   0x1a, 0x60, 0xe2, 0xb9, 0x13, 0x85, 0xc8, 0x6b, 0xb3, 0xb1,
		   0xbe, 0x5f, 0x10, 0x70, 0x8c, 0x54, 0x6a, 0xe6, 0x31, 0x82,
		   0x09, 0xba, 0x26, 0x8d, 0xd2, 0xb4, 0xe3, 0xe4, 0x72, 0x27,
		   0x61, 0x49, 0x1a, 0xe2, 0x15, 0x60, 0x02, 0x48, 0xe7, 0x54,
		   0x1e, 0x60, 0x80, 0xb8, 0x65, 0xf4, 0x4a, 0x16, 0x9c, 0x02,
		   0xa5, 0xb3, 0x74, 0xf1, 0xfa, 0x55, 0x02, 0x33, 0x37, 0x53,
		   0x97, 0x9c, 0x9d, 0xa7, 0x54, 0x52, 0xc4, 0x81, 0x4a, 0x8b,
		   0x8f, 0xd6, 0x82, 0xb8, 0xe0, 0x37, 0x06, 0x99, 0x29, 0x6b,
		   0xc2, 0xf4, 0x2d, 0x38, 0xe5, 0xa2, 0xde, 0x92, 0xc5, 0xf5,
		   0xf6, 0x1f, 0x24, 0xb6, 0x5e, 0x1b, 0x9b, 0x75, 0x24, 0x9b,
		   0x87, 0x59, 0x70, 0x87, 0x06, 0x64, 0x00, 0x00, 0xb4, 0x90,
		   0x93, 0x76, 0x83, 0xe1, 0xcc, 0xbb, 0xf3, 0x9a, 0x0b, 0xae,
		   0xcb, 0x1e, 0xdd, 0x62, 0x82, 0x67, 0x21, 0x52, 0x4b, 0xcc,
		   0x4d, 0x3b, 0xf0, 0x6f, 0x8e, 0x5a, 0xb3, 0x68, 0xaa, 0x42,
		   0x6e, 0xb4, 0x2c, 0x70, 0x18, 0x39, 0xe8, 0xbb, 0xcc, 0x69,
		   0xdb, 0xcc, 0xd2, 0x75, 0x7f, 0x84, 0xc6, 0x0a, 0x40, 0x9b,
		   0x66, 0x29, 0xd9, 0x8d, 0xc9, 0x29, 0x47, 0x36, 0x7c, 0x52,
		   0xf0, 0x69, 0x8c, 0x3c, 0x41, 0xc0, 0x11, 0x10, 0xba, 0xfb,
		   0xeb, 0x06, 0x21, 0xcc, 0x17, 0x51, 0x70, 0x37, 0x5f, 0x82,
		   0x50, 0xc0, 0xf6, 0x24, 0x06, 0xe3, 0x2e, 0x10, 0x72, 0x5b,
		   0x47, 0xe3, 0xc1, 0x65, 0x84, 0x5a, 0x7a, 0xda, 0x8f, 0xb2,
		   0xc5, 0x55, 0x55, 0xf5, 0xcc, 0x86, 0x02, 0x24, 0x6f, 0xe0,
		   0x6f, 0x87, 0x5a, 0x6e, 0xb5, 0xc9, 0x45, 0x9d, 0xd1, 0x92,
		   0x92, 0x42, 0x84, 0x92, 0xa7, 0x67, 0x38, 0xb8, 0x05, 0x7e,
		   0x84, 0xc8, 0xe1, 0x14, 0x36, 0xc4, 0xeb, 0x78, 0x8e, 0x49,
		   0x4e, 0x39, 0x52, 0x65, 0x7e, 0xd3, 0x31, 0xa2, 0xe4, 0xa4,
		   0x7d, 0x75, 0x49, 0x23, 0x65, 0xc4, 0x73, 0xf4, 0x84, 0x20,
		   0x05, 0x02, 0x86, 0x8c, 0x73, 0x3b, 0xb9, 0xb2, 0xb3, 0xe4,
		   0x4a, 0xf2, 0xc9, 0x8a, 0xdf, 0xb9, 0x1a, 0x4e, 0x1a, 0x14,
		   0x4e, 0xec, 0x5a, 0x32, 0xd2, 0x4f, 0x13, 0xc7, 0x87, 0x75,
		   0xf2, 0x06, 0xd7, 0xa9, 0x4d, 0xee, 0x12, 0x00, 0x7f, 0x77,
		   0x8e, 0x48, 0x4a, 0x14, 0x30, 0x66, 0x6b, 0xfe, 0xda, 0x6d,
		   0xbb, 0x31, 0xb7, 0x57, 0xd7, 0x2b, 0x5c, 0x79, 0xa4, 0x69,
		   0x52, 0x2f, 0xe6, 0x59, 0x9a, 0xbb, 0x45, 0xb5, 0x2c, 0xc6,
		   0x40, 0x3c, 0x49, 0x70, 0xf6, 0x56, 0x84, 0xbb, 0x77, 0x0c,
		   0xa7, 0x88, 0x93, 0xba, 0x43, 0x90, 0x57, 0x01, 0x36, 0x55,
		   0x30, 0x9c, 0xd8, 0x4c, 0x52, 0x4c, 0x9c, 0xa5, 0x58, 0xe3,
		   0x61, 0x20, 0x41, 0xb7, 0xb2, 0x02, 0x6e, 0x21, 0x87, 0x59,
		   0x8a, 0xfb, 0x46, 0xf1, 0xd4, 0xca, 0x85, 0x09, 0x6d, 0xbc,
		   0x9b, 0xcc, 0x1c, 0x25, 0x77, 0x9d, 0xfb, 0x60, 0x70, 0x52,
		   0xe1, 0x16, 0x49, 0xbb, 0x7f, 0x5f, 0x72, 0x68, 0xf9, 0x79,
		   0xc4, 0xd8, 0x14, 0x0a, 0xfe, 0x6c, 0xe5, 0x38, 0x30, 0xf3,
		   0x86, 0x02, 0x29, 0x0d, 0x75, 0x14, 0x27, 0xf0, 0x7b, 0x27 },
	.pk_x25519.pk = { 0xe5, 0x6f, 0x17, 0x57, 0x67, 0x40, 0xce, 0x2a,
			  0x32, 0xfc, 0x51, 0x45, 0x03, 0x01, 0x45, 0xcf,
			  0xb9, 0x7e, 0x63, 0xe0, 0xe4, 0x1d, 0x35, 0x42,
			  0x74, 0xa0, 0x79, 0xd3, 0xe6, 0xfb, 0x2e, 0x15 }
};
static const struct lc_kyber_x25519_sk kyber_sk_exp = {
	.sk.sk = { 0x7c, 0xf1, 0x08, 0xc7, 0x5a, 0x4d, 0x35, 0x92, 0x05, 0x3d,
		   0x0c, 0xa7, 0x9c, 0xe5, 0x27, 0xec, 0x17, 0x34, 0xf7, 0x02,
		   0x36, 0x56, 0xe2, 0x62, 0x53, 0xe2, 0xbf, 0xc6, 0x89, 0x60,
		   0xde, 0xa7, 0x3d, 0x28, 0xd7, 0xa9, 0x82, 0x15, 0x97, 0xb4,
		   0x8b, 0x45, 0x04, 0x83, 0x7e, 0x27, 0x13, 0x2c, 0x8e, 0xbc,
		   0x48, 0x50, 0x53, 0x03, 0xae, 0xb5, 0x68, 0xf9, 0xd1, 0x92,
		   0x8f, 0x72, 0x44, 0xcf, 0x98, 0xb4, 0xa8, 0x88, 0x43, 0xc5,
		   0xdb, 0x69, 0x84, 0x5a, 0xbf, 0xc4, 0xe4, 0x06, 0x83, 0xdc,
		   0xa3, 0xbd, 0xe6, 0x94, 0x44, 0x5c, 0x63, 0xcc, 0x51, 0x2b,
		   0xd2, 0xe4, 0x60, 0x8e, 0x91, 0xa5, 0x69, 0x77, 0x38, 0xbb,
		   0xca, 0x3b, 0x09, 0xa9, 0x8d, 0xa4, 0x97, 0x57, 0xad, 0x65,
		   0xc0, 0x85, 0x12, 0x5d, 0x11, 0xd0, 0x11, 0x78, 0xd1, 0x5a,
		   0xcb, 0x25, 0x1f, 0x41, 0x7c, 0xbe, 0xff, 0x2a, 0x2a, 0xc1,
		   0xb0, 0x5d, 0xc7, 0x08, 0x39, 0xff, 0xb2, 0x97, 0xbe, 0x20,
		   0x51, 0x52, 0xe9, 0x83, 0x71, 0x74, 0x5d, 0xe7, 0x62, 0x43,
		   0xd3, 0xa3, 0x02, 0xdf, 0xda, 0x7f, 0x16, 0xf5, 0xa8, 0xf0,
		   0xd2, 0xb7, 0x30, 0x21, 0xa7, 0xf5, 0xc0, 0x49, 0x0f, 0xf6,
		   0xb5, 0xa4, 0x12, 0x58, 0x43, 0x88, 0x93, 0x64, 0x28, 0x8b,
		   0xed, 0xe4, 0x30, 0x83, 0x5c, 0xce, 0x5b, 0xf8, 0xb8, 0x63,
		   0x77, 0x13, 0x2a, 0xea, 0x3c, 0xed, 0x91, 0x13, 0x11, 0xa4,
		   0x3b, 0x87, 0xe1, 0x0a, 0xe4, 0x71, 0xb4, 0x9f, 0x1a, 0x31,
		   0xb6, 0x36, 0xce, 0x5d, 0x7c, 0x41, 0x5d, 0x7c, 0x17, 0x53,
		   0x95, 0x75, 0x72, 0x4c, 0x31, 0x66, 0xab, 0x23, 0xca, 0x18,
		   0x3f, 0x99, 0x62, 0xa0, 0xb1, 0x71, 0x90, 0xf3, 0xdc, 0x20,
		   0x4d, 0xc2, 0x55, 0x1e, 0xec, 0x23, 0x4b, 0x7c, 0x4e, 0x29,
		   0x8a, 0x4e, 0x60, 0xf1, 0x5d, 0x4e, 0x40, 0xbc, 0xaf, 0x1b,
		   0xa0, 0xea, 0xa8, 0x06, 0xe2, 0x14, 0xbd, 0xa0, 0xa0, 0xb0,
		   0xe0, 0xf5, 0x1b, 0x34, 0x74, 0x5e, 0x14, 0x82, 0x90, 0xe2,
		   0x22, 0xa4, 0x49, 0x6c, 0x94, 0x67, 0x66, 0x02, 0x9c, 0xd8,
		   0x29, 0x64, 0x42, 0x9f, 0xd8, 0x5a, 0x30, 0xee, 0x3c, 0x0c,
		   0x6d, 0xea, 0x4e, 0x6b, 0xc8, 0x40, 0x23, 0x7c, 0x38, 0xd8,
		   0xe9, 0xb6, 0xa8, 0x93, 0x70, 0x24, 0x44, 0xbf, 0xee, 0xf7,
		   0x83, 0xa1, 0x04, 0xa8, 0xb4, 0x6b, 0x83, 0x09, 0xa9, 0x14,
		   0xcb, 0xaa, 0xa8, 0xce, 0xc9, 0xc0, 0x8b, 0xb5, 0x4a, 0x3c,
		   0xfc, 0x1a, 0x22, 0xc5, 0xa9, 0xd1, 0xb8, 0x76, 0x85, 0xa3,
		   0x04, 0x17, 0xd8, 0x46, 0xd5, 0xe2, 0x22, 0x42, 0xe8, 0xc0,
		   0xc8, 0x31, 0x6c, 0xb7, 0xf1, 0x58, 0x32, 0xb7, 0x0b, 0x4b,
		   0xbc, 0x45, 0x3b, 0xb6, 0x74, 0x85, 0xe3, 0x4c, 0x87, 0xd9,
		   0xa2, 0xd4, 0xb2, 0x94, 0xd2, 0x57, 0xa9, 0x8d, 0x86, 0x4a,
		   0xfa, 0xb4, 0x45, 0xae, 0xa6, 0xbf, 0x48, 0x30, 0x40, 0x4d,
		   0x17, 0x0c, 0xa7, 0x48, 0x4c, 0xc7, 0x99, 0xb7, 0x3a, 0xe3,
		   0xce, 0x37, 0x5c, 0x3a, 0x13, 0xd1, 0x31, 0x8b, 0xd2, 0x91,
		   0x58, 0x62, 0x03, 0x28, 0xc8, 0xbc, 0x22, 0xd8, 0x12, 0xd2,
		   0x69, 0xc8, 0xec, 0x13, 0x3c, 0xe4, 0xf9, 0x77, 0xbd, 0x95,
		   0x30, 0x00, 0x54, 0xb5, 0xef, 0x24, 0x6b, 0xb0, 0x86, 0xa6,
		   0xd3, 0x34, 0x05, 0x75, 0x46, 0x41, 0x83, 0xe1, 0x0d, 0xb1,
		   0xa1, 0x26, 0xf8, 0x1a, 0x43, 0xd2, 0xf8, 0x00, 0xe9, 0x62,
		   0x56, 0x8d, 0xe3, 0x41, 0x15, 0x15, 0xb9, 0x0d, 0xc1, 0x25,
		   0x64, 0x46, 0xcf, 0x8f, 0xc3, 0x31, 0x49, 0xe6, 0x6a, 0x86,
		   0x57, 0xc2, 0x21, 0x33, 0x5e, 0xb5, 0x7a, 0x8c, 0xa6, 0x26,
		   0x2b, 0x06, 0x98, 0x1a, 0x03, 0x28, 0x23, 0x22, 0x2c, 0x9d,
		   0xa2, 0x28, 0x35, 0x27, 0x10, 0x33, 0x21, 0x2b, 0x4e, 0x9d,
		   0x51, 0x61, 0x32, 0xa4, 0x28, 0x70, 0x9b, 0x58, 0x82, 0xab,
		   0x87, 0x64, 0x77, 0x18, 0x32, 0x96, 0x8d, 0xc7, 0x3c, 0x4c,
		   0xd2, 0x41, 0xad, 0xb1, 0x45, 0xb5, 0xe6, 0x54, 0xb3, 0x8b,
		   0xab, 0x03, 0xe3, 0x52, 0x4f, 0x11, 0x35, 0x25, 0xdd, 0x0a,
		   0x63, 0xe8, 0xf1, 0x95, 0xfc, 0xeb, 0x26, 0xb8, 0xb0, 0x30,
		   0x63, 0xc4, 0x86, 0x34, 0x59, 0xcc, 0x10, 0xc5, 0x88, 0x4c,
		   0xac, 0x01, 0x6e, 0xeb, 0xb7, 0xaf, 0xa2, 0x51, 0x9b, 0x27,
		   0x29, 0x3b, 0x84, 0x5e, 0x7e, 0x89, 0x3e, 0x70, 0xbb, 0x21,
		   0xd0, 0xe0, 0x04, 0xf1, 0xd0, 0x50, 0x15, 0xdc, 0x71, 0xfc,
		   0xf6, 0xb1, 0x50, 0xc7, 0x50, 0xaa, 0x98, 0xc9, 0x6f, 0x89,
		   0xc8, 0xda, 0x19, 0x9e, 0xef, 0xc1, 0x71, 0x2c, 0x40, 0x9b,
		   0xc8, 0x33, 0xa8, 0x4d, 0x40, 0x4a, 0xd7, 0x2c, 0x87, 0x3b,
		   0xf9, 0x10, 0xc9, 0x19, 0x06, 0x45, 0xb9, 0x4d, 0xd2, 0xa3,
		   0x13, 0x47, 0x59, 0x60, 0x29, 0xd2, 0x2a, 0xbd, 0x07, 0x1f,
		   0x53, 0x87, 0x58, 0x8a, 0x99, 0x17, 0xfe, 0xe4, 0xb3, 0x3d,
		   0x1b, 0xbc, 0xfa, 0x18, 0x58, 0x39, 0x06, 0x62, 0xb6, 0x56,
		   0xc0, 0x9f, 0xf8, 0x24, 0xed, 0x17, 0x7a, 0xbf, 0x44, 0xa1,
		   0x18, 0xb7, 0xc7, 0x8b, 0xba, 0x96, 0xb9, 0xf9, 0xb5, 0xa3,
		   0xa5, 0x4e, 0x59, 0x78, 0x29, 0xbe, 0x68, 0x68, 0x68, 0xe5,
		   0x46, 0x3f, 0x99, 0xbe, 0x58, 0xda, 0x0b, 0xe7, 0x88, 0x65,
		   0x01, 0x73, 0x0e, 0x01, 0xe2, 0x6a, 0xad, 0x05, 0x35, 0x89,
		   0x66, 0x16, 0xd9, 0x81, 0x25, 0xc3, 0x8b, 0x3e, 0xb7, 0x78,
		   0x11, 0x41, 0x76, 0xb1, 0xbd, 0x49, 0x8e, 0xbd, 0x92, 0xb5,
		   0xbd, 0x0a, 0x98, 0xa0, 0xa3, 0x9f, 0x77, 0xb5, 0x99, 0xc3,
		   0xe6, 0x3e, 0x66, 0xfb, 0x62, 0x16, 0x7b, 0x06, 0xc2, 0x9a,
		   0xc6, 0x6c, 0x84, 0xbe, 0xe3, 0xf1, 0x29, 0x50, 0x8c, 0x2b,
		   0x8c, 0x79, 0x0c, 0x99, 0xa5, 0xca, 0x41, 0xe5, 0x70, 0x7e,
		   0x9b, 0x8c, 0x75, 0xc0, 0x4d, 0x7e, 0xa8, 0xa4, 0x81, 0x98,
		   0x0a, 0x35, 0x8b, 0x06, 0x6a, 0x4a, 0x7e, 0x28, 0xd1, 0x5d,
		   0x10, 0x37, 0x4f, 0x75, 0xc3, 0x3d, 0xa6, 0x02, 0x9c, 0xd4,
		   0x90, 0x74, 0x6f, 0xb5, 0x5f, 0x5f, 0xea, 0xb2, 0xce, 0x82,
		   0x3d, 0xec, 0x18, 0x30, 0xc0, 0xc1, 0x8e, 0xf8, 0x9b, 0xa3,
		   0xcc, 0x3b, 0xce, 0x4d, 0x25, 0x2a, 0x07, 0xa4, 0x0d, 0x40,
		   0x1a, 0x4c, 0x8d, 0x27, 0x36, 0x18, 0xb5, 0x95, 0xdb, 0x21,
		   0xca, 0x4a, 0x95, 0x9e, 0xb4, 0xd1, 0x33, 0x55, 0x68, 0x01,
		   0xb2, 0xb7, 0x82, 0x54, 0xb2, 0xb5, 0x95, 0x5c, 0x04, 0x00,
		   0x96, 0x2d, 0xbb, 0x5a, 0x48, 0x7a, 0xaa, 0x24, 0xa4, 0x30,
		   0xb6, 0x14, 0xe2, 0xaf, 0x93, 0x38, 0xaf, 0xd4, 0xb0, 0x33,
		   0x9b, 0xd8, 0x30, 0xa9, 0xcb, 0x76, 0x1d, 0xb1, 0xa8, 0x32,
		   0x20, 0x35, 0x2d, 0xd5, 0x23, 0xb8, 0xa5, 0x83, 0x02, 0x2c,
		   0xa1, 0x3e, 0x24, 0x65, 0x06, 0x54, 0x8c, 0x57, 0xba, 0x38,
		   0x50, 0xac, 0x5c, 0x50, 0xd8, 0x64, 0xe2, 0xd4, 0x8b, 0x89,
		   0x69, 0x42, 0x77, 0xcc, 0x7a, 0x81, 0x74, 0xad, 0x7b, 0x1b,
		   0xa2, 0xee, 0x62, 0x1d, 0x57, 0xd2, 0x9b, 0x23, 0xc2, 0x2d,
		   0xf8, 0x31, 0x87, 0x20, 0x51, 0x14, 0x5d, 0x25, 0x35, 0xdb,
		   0xb0, 0x25, 0xe2, 0x9c, 0x6d, 0xa3, 0x8c, 0xb4, 0x75, 0x95,
		   0x8e, 0x2a, 0x88, 0x08, 0xa4, 0x31, 0xbe, 0xd5, 0xb4, 0x86,
		   0x82, 0x1a, 0x15, 0x13, 0xa7, 0x29, 0xcd, 0x97, 0x9b, 0x6c,
		   0x8b, 0x38, 0x25, 0x03, 0xcf, 0x53, 0x33, 0x76, 0x16, 0xe1,
		   0x05, 0x9d, 0xdc, 0x59, 0x97, 0x72, 0x19, 0xd2, 0xf1, 0x7b,
		   0x32, 0xac, 0xab, 0xbe, 0x86, 0xc4, 0x68, 0xc1, 0x26, 0x7b,
		   0x2b, 0x86, 0x2a, 0xc5, 0x65, 0xa7, 0xe2, 0x32, 0x26, 0x6c,
		   0x83, 0xfc, 0x70, 0x0c, 0xda, 0x19, 0xb6, 0x16, 0x37, 0x73,
		   0x89, 0x42, 0x8e, 0x62, 0x61, 0x4c, 0xd6, 0xac, 0x9f, 0x08,
		   0xfb, 0x12, 0xd2, 0x18, 0x27, 0xfe, 0xb3, 0x24, 0xb1, 0x4b,
		   0xc1, 0x6c, 0xc0, 0xc2, 0x09, 0x04, 0xb3, 0x97, 0x3b, 0x9d,
		   0xfe, 0x60, 0xaf, 0x57, 0x77, 0x76, 0x44, 0x9a, 0xc3, 0x4e,
		   0xb0, 0x09, 0x89, 0xe5, 0xa8, 0x76, 0x14, 0x3b, 0xc9, 0xb9,
		   0x30, 0xa3, 0xc2, 0xa5, 0xbb, 0x86, 0x12, 0x08, 0xf2, 0x25,
		   0xdd, 0x97, 0x62, 0x5b, 0xb3, 0x6b, 0x2d, 0x88, 0x10, 0xd0,
		   0x45, 0x2e, 0x45, 0x89, 0x68, 0xb9, 0x0c, 0x6a, 0x6d, 0x97,
		   0xc2, 0xdf, 0xca, 0x73, 0x41, 0xd5, 0x3c, 0x74, 0x1a, 0x60,
		   0xe2, 0xb9, 0x13, 0x85, 0xc8, 0x6b, 0xb3, 0xb1, 0xbe, 0x5f,
		   0x10, 0x70, 0x8c, 0x54, 0x6a, 0xe6, 0x31, 0x82, 0x09, 0xba,
		   0x26, 0x8d, 0xd2, 0xb4, 0xe3, 0xe4, 0x72, 0x27, 0x61, 0x49,
		   0x1a, 0xe2, 0x15, 0x60, 0x02, 0x48, 0xe7, 0x54, 0x1e, 0x60,
		   0x80, 0xb8, 0x65, 0xf4, 0x4a, 0x16, 0x9c, 0x02, 0xa5, 0xb3,
		   0x74, 0xf1, 0xfa, 0x55, 0x02, 0x33, 0x37, 0x53, 0x97, 0x9c,
		   0x9d, 0xa7, 0x54, 0x52, 0xc4, 0x81, 0x4a, 0x8b, 0x8f, 0xd6,
		   0x82, 0xb8, 0xe0, 0x37, 0x06, 0x99, 0x29, 0x6b, 0xc2, 0xf4,
		   0x2d, 0x38, 0xe5, 0xa2, 0xde, 0x92, 0xc5, 0xf5, 0xf6, 0x1f,
		   0x24, 0xb6, 0x5e, 0x1b, 0x9b, 0x75, 0x24, 0x9b, 0x87, 0x59,
		   0x70, 0x87, 0x06, 0x64, 0x00, 0x00, 0xb4, 0x90, 0x93, 0x76,
		   0x83, 0xe1, 0xcc, 0xbb, 0xf3, 0x9a, 0x0b, 0xae, 0xcb, 0x1e,
		   0xdd, 0x62, 0x82, 0x67, 0x21, 0x52, 0x4b, 0xcc, 0x4d, 0x3b,
		   0xf0, 0x6f, 0x8e, 0x5a, 0xb3, 0x68, 0xaa, 0x42, 0x6e, 0xb4,
		   0x2c, 0x70, 0x18, 0x39, 0xe8, 0xbb, 0xcc, 0x69, 0xdb, 0xcc,
		   0xd2, 0x75, 0x7f, 0x84, 0xc6, 0x0a, 0x40, 0x9b, 0x66, 0x29,
		   0xd9, 0x8d, 0xc9, 0x29, 0x47, 0x36, 0x7c, 0x52, 0xf0, 0x69,
		   0x8c, 0x3c, 0x41, 0xc0, 0x11, 0x10, 0xba, 0xfb, 0xeb, 0x06,
		   0x21, 0xcc, 0x17, 0x51, 0x70, 0x37, 0x5f, 0x82, 0x50, 0xc0,
		   0xf6, 0x24, 0x06, 0xe3, 0x2e, 0x10, 0x72, 0x5b, 0x47, 0xe3,
		   0xc1, 0x65, 0x84, 0x5a, 0x7a, 0xda, 0x8f, 0xb2, 0xc5, 0x55,
		   0x55, 0xf5, 0xcc, 0x86, 0x02, 0x24, 0x6f, 0xe0, 0x6f, 0x87,
		   0x5a, 0x6e, 0xb5, 0xc9, 0x45, 0x9d, 0xd1, 0x92, 0x92, 0x42,
		   0x84, 0x92, 0xa7, 0x67, 0x38, 0xb8, 0x05, 0x7e, 0x84, 0xc8,
		   0xe1, 0x14, 0x36, 0xc4, 0xeb, 0x78, 0x8e, 0x49, 0x4e, 0x39,
		   0x52, 0x65, 0x7e, 0xd3, 0x31, 0xa2, 0xe4, 0xa4, 0x7d, 0x75,
		   0x49, 0x23, 0x65, 0xc4, 0x73, 0xf4, 0x84, 0x20, 0x05, 0x02,
		   0x86, 0x8c, 0x73, 0x3b, 0xb9, 0xb2, 0xb3, 0xe4, 0x4a, 0xf2,
		   0xc9, 0x8a, 0xdf, 0xb9, 0x1a, 0x4e, 0x1a, 0x14, 0x4e, 0xec,
		   0x5a, 0x32, 0xd2, 0x4f, 0x13, 0xc7, 0x87, 0x75, 0xf2, 0x06,
		   0xd7, 0xa9, 0x4d, 0xee, 0x12, 0x00, 0x7f, 0x77, 0x8e, 0x48,
		   0x4a, 0x14, 0x30, 0x66, 0x6b, 0xfe, 0xda, 0x6d, 0xbb, 0x31,
		   0xb7, 0x57, 0xd7, 0x2b, 0x5c, 0x79, 0xa4, 0x69, 0x52, 0x2f,
		   0xe6, 0x59, 0x9a, 0xbb, 0x45, 0xb5, 0x2c, 0xc6, 0x40, 0x3c,
		   0x49, 0x70, 0xf6, 0x56, 0x84, 0xbb, 0x77, 0x0c, 0xa7, 0x88,
		   0x93, 0xba, 0x43, 0x90, 0x57, 0x01, 0x36, 0x55, 0x30, 0x9c,
		   0xd8, 0x4c, 0x52, 0x4c, 0x9c, 0xa5, 0x58, 0xe3, 0x61, 0x20,
		   0x41, 0xb7, 0xb2, 0x02, 0x6e, 0x21, 0x87, 0x59, 0x8a, 0xfb,
		   0x46, 0xf1, 0xd4, 0xca, 0x85, 0x09, 0x6d, 0xbc, 0x9b, 0xcc,
		   0x1c, 0x25, 0x77, 0x9d, 0xfb, 0x60, 0x70, 0x52, 0xe1, 0x16,
		   0x49, 0xbb, 0x7f, 0x5f, 0x72, 0x68, 0xf9, 0x79, 0xc4, 0xd8,
		   0x14, 0x0a, 0xfe, 0x6c, 0xe5, 0x38, 0x30, 0xf3, 0x86, 0x02,
		   0x29, 0x0d, 0x75, 0x14, 0x27, 0xf0, 0x7b, 0x27, 0xcd, 0xa9,
		   0x3d, 0xec, 0x4c, 0x4d, 0xc4, 0xd8, 0x48, 0x44, 0x57, 0xfd,
		   0x88, 0x23, 0x99, 0xc4, 0xb9, 0x18, 0xc4, 0x9f, 0xa8, 0x38,
		   0x9a, 0x1d, 0xfa, 0x8c, 0x9f, 0x92, 0xf3, 0x9b, 0x00, 0xcf,
		   0x3c, 0xb1, 0xee, 0xa9, 0x88, 0x00, 0x4b, 0x93, 0x10, 0x3c,
		   0xfb, 0x0a, 0xee, 0xfd, 0x2a, 0x68, 0x6e, 0x01, 0xfa, 0x4a,
		   0x58, 0xe8, 0xa3, 0x63, 0x9c, 0xa8, 0xa1, 0xe3, 0xf9, 0xae,
		   0x57, 0xe2 },
	.sk_x25519.sk = { 0x35, 0xb8, 0xcc, 0x87, 0x3c, 0x23, 0xdc, 0x62,
			  0xb8, 0xd2, 0x60, 0x16, 0x9a, 0xfa, 0x2f, 0x75,
			  0xab, 0x91, 0x6a, 0x58, 0xd9, 0x74, 0x91, 0x88,
			  0x35, 0xd2, 0x5e, 0x6a, 0x43, 0x50, 0x85, 0xb2 }
};
static const uint8_t ss_exp[] = {
	0xbc, 0xc9, 0xa3, 0x66, 0x1e, 0xfa, 0x7e, 0xc4, 0x9b, 0x09, 0x7a, 0x5b,
	0xc7, 0x6d, 0x43, 0x73, 0x5b, 0x0d, 0x29, 0x3f, 0x9d, 0x90, 0xa0, 0x08,
	0x0c, 0xc4, 0x9f, 0x50, 0x56, 0xa4, 0xdb, 0x6c, 0xbf, 0x70, 0x90, 0x0c,
	0x05, 0x10, 0xc3, 0x96, 0xd3, 0x58, 0x3f, 0xe0, 0xc4, 0x29, 0x80, 0xfa,
	0xd3, 0x27, 0x0b, 0x1f, 0x5b, 0x50, 0xf2, 0xc7, 0x19, 0xe5, 0x50, 0xd0,
	0x98, 0x2b, 0x04, 0x18, 0x4b, 0x54, 0xf7
};
#elif LC_KYBER_K == 3
static const struct lc_kyber_x25519_pk kyber_pk_exp = {
	.pk.pk = { 0x78, 0x20, 0x32, 0x02, 0x30, 0x23, 0x8e, 0x44, 0x7a, 0xcf,
		   0xa9, 0x9b, 0x63, 0x32, 0xb7, 0x53, 0x1c, 0x7c, 0xe5, 0x42,
		   0x03, 0x1b, 0x93, 0xca, 0x14, 0x25, 0x8f, 0x5f, 0x98, 0xb3,
		   0x0c, 0x87, 0x3f, 0x6d, 0x01, 0xb2, 0xc5, 0x85, 0x3f, 0x34,
		   0x02, 0x69, 0x18, 0x04, 0x80, 0x47, 0x08, 0xa7, 0xec, 0xfa,
		   0x5b, 0xa5, 0x98, 0x0f, 0xef, 0x88, 0x36, 0xce, 0xf5, 0xa8,
		   0x2d, 0xac, 0x83, 0x75, 0xe0, 0x62, 0x34, 0xcb, 0xbe, 0xd6,
		   0x4c, 0xc5, 0xa4, 0xb2, 0xac, 0x06, 0x84, 0xa0, 0xb4, 0xd9,
		   0x86, 0x3e, 0x66, 0x44, 0xa8, 0x33, 0x48, 0x49, 0xc0, 0xcf,
		   0xdb, 0xfb, 0x85, 0x1b, 0x84, 0xb0, 0xe4, 0x8c, 0x6f, 0xb0,
		   0x48, 0x01, 0xd0, 0x52, 0x2f, 0xd5, 0x76, 0x68, 0x9b, 0x16,
		   0x44, 0x67, 0x60, 0x4a, 0xb4, 0xda, 0x2e, 0x34, 0xab, 0xcf,
		   0x76, 0xdc, 0x92, 0x44, 0x67, 0x4c, 0x4e, 0x57, 0x30, 0xf5,
		   0x78, 0x7c, 0x67, 0x90, 0x21, 0x42, 0x4c, 0x13, 0x40, 0x15,
		   0x4d, 0x24, 0x49, 0x33, 0x0d, 0xbc, 0x94, 0x0e, 0xba, 0x7b,
		   0xe1, 0x03, 0x99, 0xec, 0x94, 0x0a, 0x6b, 0x35, 0x40, 0x60,
		   0x14, 0x99, 0x66, 0x50, 0xb1, 0x48, 0xa1, 0x89, 0x43, 0x47,
		   0x7f, 0x5e, 0x95, 0x0f, 0x41, 0xa0, 0x1a, 0xb6, 0x34, 0x88,
		   0xfa, 0xf2, 0x06, 0xf0, 0xf2, 0x25, 0xa7, 0x31, 0x09, 0x67,
		   0x77, 0x84, 0x79, 0x1b, 0x6d, 0xec, 0x98, 0x0e, 0x60, 0xeb,
		   0x8f, 0x96, 0xbb, 0xbf, 0x47, 0x22, 0x22, 0xb0, 0x24, 0x8a,
		   0x12, 0xf9, 0x61, 0x88, 0xa1, 0x7d, 0x68, 0xb8, 0x3e, 0x52,
		   0xc9, 0x82, 0x7b, 0xa2, 0xc6, 0xbc, 0x82, 0x75, 0x0b, 0x31,
		   0xae, 0x46, 0xb9, 0x82, 0x7a, 0x28, 0x77, 0xfc, 0xb3, 0x38,
		   0x00, 0x3a, 0x68, 0xb8, 0xc1, 0xaa, 0x9c, 0x96, 0x6b, 0x55,
		   0xd2, 0x03, 0x78, 0x74, 0x11, 0x57, 0x77, 0x05, 0xf6, 0xdb,
		   0x15, 0x59, 0xf9, 0xb4, 0x25, 0x8a, 0x57, 0x18, 0x20, 0x50,
		   0x5d, 0x83, 0xc1, 0x00, 0x9a, 0xa2, 0xb2, 0x84, 0xb3, 0x75,
		   0xa7, 0xac, 0x61, 0x6a, 0x1b, 0x29, 0x42, 0xa2, 0xb4, 0x94,
		   0xc4, 0x3e, 0xb3, 0x70, 0xc1, 0xb9, 0x4a, 0xcd, 0x52, 0x2b,
		   0x20, 0x8c, 0x17, 0xc7, 0x35, 0x07, 0x7e, 0x60, 0x2f, 0x41,
		   0x41, 0x0c, 0xfb, 0x1b, 0x3e, 0xa1, 0x09, 0x21, 0x58, 0xc3,
		   0xb6, 0xea, 0x43, 0xcc, 0x7a, 0xa7, 0x03, 0x09, 0xa6, 0x0f,
		   0xa9, 0x64, 0x47, 0x21, 0x33, 0x2a, 0x0d, 0xc4, 0x55, 0xbd,
		   0xb5, 0x63, 0xd5, 0xe6, 0x6e, 0xe2, 0x12, 0x10, 0x87, 0x2a,
		   0x59, 0x8d, 0x13, 0xa9, 0x51, 0x83, 0x11, 0x9a, 0x50, 0x7a,
		   0x76, 0x34, 0x3b, 0x45, 0x4b, 0xb3, 0x20, 0x4b, 0xad, 0x83,
		   0x46, 0x9b, 0x56, 0x27, 0x52, 0xa7, 0x3a, 0x00, 0xe5, 0xe8,
		   0xc5, 0xa6, 0xd9, 0x87, 0x0a, 0xca, 0x93, 0x78, 0x69, 0x05,
		   0xc2, 0x0b, 0x85, 0x02, 0xf4, 0x5d, 0x91, 0x14, 0xa8, 0xd2,
		   0xc7, 0xbd, 0x2e, 0xe0, 0x08, 0x7f, 0xb7, 0x80, 0xfa, 0x50,
		   0x78, 0x58, 0x8b, 0x02, 0x6f, 0x14, 0x19, 0x6c, 0xc0, 0x68,
		   0xf6, 0xa7, 0x5e, 0x7b, 0x4a, 0x44, 0x3a, 0xea, 0x41, 0x42,
		   0x5c, 0xaf, 0x2d, 0xa2, 0x12, 0xa3, 0xe6, 0xad, 0xf6, 0x44,
		   0x51, 0x64, 0xc2, 0x90, 0x8f, 0x40, 0x9d, 0x43, 0x96, 0x5e,
		   0xe4, 0x0b, 0x89, 0xd6, 0x54, 0xc8, 0x41, 0x30, 0x97, 0x48,
		   0x6a, 0x9b, 0x90, 0xe2, 0xb0, 0x44, 0x74, 0x38, 0x91, 0x18,
		   0x36, 0xb7, 0xca, 0x84, 0xe7, 0x6b, 0x7a, 0x8e, 0xe6, 0x00,
		   0xa1, 0x3b, 0x55, 0xa8, 0xb8, 0x3d, 0xa0, 0xb2, 0x08, 0x25,
		   0x38, 0x2b, 0xda, 0xc3, 0xbe, 0x25, 0x04, 0x60, 0x24, 0xc7,
		   0x88, 0x76, 0x40, 0x53, 0xf1, 0x92, 0x2c, 0xfa, 0xa2, 0xa1,
		   0xb8, 0x16, 0x22, 0x30, 0xa3, 0x6d, 0xbb, 0x36, 0xcf, 0x8e,
		   0x62, 0xc8, 0xa1, 0xb1, 0x5f, 0x9d, 0x51, 0x90, 0x07, 0x0a,
		   0xb4, 0xf5, 0x70, 0xa3, 0xba, 0xf8, 0x63, 0x7c, 0x04, 0xad,
		   0x19, 0x51, 0xa4, 0xb4, 0x61, 0x0b, 0x5b, 0x7a, 0xb2, 0x33,
		   0x56, 0xc3, 0x9e, 0x36, 0xa8, 0x12, 0x85, 0x58, 0xa2, 0x59,
		   0x14, 0xe1, 0xa8, 0xaf, 0x76, 0xe4, 0x77, 0x12, 0xf8, 0x54,
		   0x38, 0xb1, 0x9e, 0x8d, 0x04, 0xce, 0x92, 0x35, 0x20, 0xd3,
		   0xab, 0x0a, 0xc4, 0xe4, 0x08, 0x58, 0xf2, 0x36, 0x2d, 0xd4,
		   0x9d, 0xb4, 0x46, 0xb7, 0xed, 0xf7, 0xa0, 0xdd, 0x4c, 0x83,
		   0xaa, 0x30, 0x7e, 0xfa, 0x87, 0x76, 0xb3, 0xb8, 0xc7, 0xa1,
		   0x0a, 0x1f, 0xdf, 0xe6, 0x8e, 0xda, 0x1b, 0x9c, 0xd4, 0x2a,
		   0xc0, 0xce, 0xc4, 0x76, 0x0c, 0x25, 0x93, 0x18, 0x08, 0x2c,
		   0x18, 0x41, 0x5d, 0x2c, 0xc2, 0x93, 0x67, 0xf6, 0x06, 0xee,
		   0xea, 0x1c, 0x01, 0x45, 0x38, 0x9e, 0xa2, 0x88, 0xf3, 0x83,
		   0xa6, 0x8d, 0xd6, 0x20, 0x12, 0x7a, 0xac, 0xa8, 0x8b, 0xcb,
		   0x6b, 0x42, 0x45, 0xa6, 0x64, 0x4f, 0x05, 0xa6, 0xb9, 0xf1,
		   0x16, 0x3d, 0x38, 0xe1, 0x7e, 0x57, 0xcb, 0x27, 0xcc, 0x5c,
		   0x46, 0x7b, 0x2c, 0xca, 0x94, 0xf9, 0x74, 0xcf, 0xd8, 0x15,
		   0x0d, 0x52, 0x94, 0xe1, 0x52, 0x20, 0x0b, 0x39, 0xba, 0xfa,
		   0x46, 0x9b, 0x9c, 0xfa, 0xc3, 0xac, 0xeb, 0x7b, 0x56, 0xb4,
		   0x7b, 0x13, 0xf1, 0x71, 0x81, 0xea, 0x95, 0x87, 0x78, 0x10,
		   0xa7, 0x77, 0xae, 0x9d, 0x15, 0x60, 0xed, 0x51, 0x19, 0xab,
		   0x09, 0xb7, 0x6a, 0xb1, 0x20, 0xe3, 0xc6, 0x19, 0x61, 0xd5,
		   0x8c, 0x3a, 0x93, 0x16, 0x69, 0xa0, 0x5f, 0xd2, 0x88, 0x2f,
		   0x51, 0x97, 0xb7, 0xc3, 0xb0, 0x47, 0xa1, 0x97, 0xaa, 0x82,
		   0x0c, 0x20, 0x8a, 0x5c, 0xb5, 0xbc, 0x40, 0xbe, 0xb8, 0x25,
		   0x88, 0xf5, 0x61, 0x0b, 0x0b, 0x35, 0x4e, 0xb8, 0x50, 0x39,
		   0xb0, 0xa4, 0x1d, 0xfd, 0x39, 0x24, 0x8f, 0x16, 0x5a, 0xbb,
		   0x02, 0x7a, 0xfd, 0x68, 0x17, 0xe0, 0x0c, 0x8d, 0xad, 0xaa,
		   0xcb, 0x29, 0xcb, 0x25, 0xe4, 0x24, 0x75, 0x09, 0x32, 0xae,
		   0x27, 0x75, 0x6a, 0x98, 0xb7, 0xb7, 0x49, 0x28, 0x53, 0x90,
		   0x9c, 0xa9, 0x5a, 0x56, 0x92, 0x2a, 0x13, 0x36, 0x90, 0x6c,
		   0x02, 0xb8, 0x12, 0x05, 0xbb, 0x58, 0x69, 0x4d, 0xe8, 0xc0,
		   0x2a, 0x00, 0x5c, 0x23, 0x26, 0x09, 0xc8, 0x79, 0x97, 0xb1,
		   0x6a, 0xb6, 0xbc, 0x79, 0x0d, 0xd3, 0x29, 0x30, 0x03, 0xb9,
		   0xb4, 0x86, 0x7b, 0x2b, 0x6e, 0x04, 0xc4, 0x56, 0x18, 0x50,
		   0x35, 0x94, 0xaa, 0x26, 0x9c, 0x9d, 0x4f, 0xc3, 0x9a, 0xa1,
		   0xec, 0x36, 0x3f, 0xbc, 0x57, 0xc7, 0x02, 0x41, 0x31, 0x0b,
		   0x1d, 0x6b, 0x44, 0x98, 0xe8, 0xa5, 0x56, 0xc4, 0x87, 0xb1,
		   0x6e, 0x45, 0xaa, 0x9b, 0xb9, 0xb1, 0xa6, 0xa9, 0xbb, 0x2a,
		   0x44, 0x6f, 0x6e, 0xe3, 0x09, 0xed, 0xec, 0x1d, 0x55, 0x32,
		   0x99, 0x7a, 0x86, 0x94, 0xdd, 0xb9, 0x91, 0xe3, 0xa6, 0x77,
		   0xf8, 0x95, 0x88, 0x5f, 0xa9, 0x34, 0x81, 0xea, 0x21, 0x7e,
		   0xba, 0x1c, 0x47, 0x4b, 0xc4, 0x81, 0x88, 0x14, 0x7f, 0xdc,
		   0x03, 0x22, 0x81, 0xcc, 0xa5, 0xc3, 0x0d, 0x5b, 0x1c, 0x3e,
		   0xd9, 0x09, 0x32, 0xbd, 0xe1, 0x48, 0x55, 0x92, 0x9f, 0x5e,
		   0x22, 0x4f, 0x27, 0x38, 0x75, 0x36, 0x4b, 0x60, 0x0c, 0x7c,
		   0x91, 0x6b, 0xd3, 0x1f, 0x66, 0xd7, 0xbb, 0xc4, 0x70, 0xc5,
		   0x36, 0xe6, 0x00, 0xe8, 0xda, 0x7f, 0x31, 0xfb, 0x3b, 0x99,
		   0x45, 0x73, 0xe7, 0x77, 0x96, 0xc5, 0x39, 0x69, 0x37, 0x1a,
		   0x5e, 0x30, 0xc3, 0x71, 0xa0, 0x32, 0x50, 0xe4, 0x26, 0xc2,
		   0xdb, 0xda, 0x9e, 0xee, 0xd3, 0x9b, 0x6b, 0x65, 0x03, 0x38,
		   0x80, 0x66, 0x05, 0xd3, 0x5e, 0x64, 0xd5, 0x59, 0xd0, 0x91,
		   0x0d, 0x2e, 0xf7, 0x51, 0xaf, 0xa8, 0x33, 0x00, 0x2c, 0x90,
		   0x5d, 0xb5, 0x5e, 0xe0, 0x09, 0x81, 0x75, 0xf2, 0x11, 0x11,
		   0x82, 0x2f, 0x6a, 0x9c, 0x1a, 0xab, 0xcb, 0x97, 0x71, 0xcc,
		   0x15, 0x3b, 0xb4, 0xb6, 0x19, 0xd7, 0x35, 0xfa, 0x82, 0x23,
		   0xd5, 0xf0, 0x6a, 0x8b, 0x2b, 0x79, 0x08, 0xe3, 0x2d, 0x8f,
		   0x7b, 0x82, 0x9d, 0xac, 0x88, 0xc9, 0xca, 0xa4, 0x74, 0x2a,
		   0x12, 0xa2, 0xa4, 0x4b, 0xc6, 0x14, 0x20, 0xf7, 0xd2, 0x6b,
		   0x87, 0xc3, 0xc8, 0x33, 0xc5, 0x6c, 0xb5, 0x1a, 0xcc, 0x1b,
		   0xd4, 0x8c, 0x9a, 0x64, 0x8b, 0x4f, 0x60, 0x8e, 0x4f, 0xeb,
		   0x37, 0x8e, 0x65, 0x54, 0xb7, 0x3c, 0xa8, 0xe3, 0x7b, 0x9d,
		   0x5f, 0xa7, 0x6b, 0x44, 0x1c, 0x03, 0x97, 0xf8, 0xc0, 0xaa,
		   0x39, 0xb2, 0x34, 0x8a, 0xbc, 0x3c, 0x6d, 0x0d, 0x13, 0xa6,
		   0xbd, 0x91, 0xd6, 0x2a, 0x43, 0xc6, 0x51, 0x60, 0x63, 0xdf,
		   0xd5, 0xbf, 0x77, 0xf0, 0xf1, 0x08, 0x42, 0x24, 0x43, 0x7b,
		   0x47, 0xa8, 0x2b, 0xa2 },
	.pk_x25519.pk = { 0xe5, 0x6f, 0x17, 0x57, 0x67, 0x40, 0xce, 0x2a,
			  0x32, 0xfc, 0x51, 0x45, 0x03, 0x01, 0x45, 0xcf,
			  0xb9, 0x7e, 0x63, 0xe0, 0xe4, 0x1d, 0x35, 0x42,
			  0x74, 0xa0, 0x79, 0xd3, 0xe6, 0xfb, 0x2e, 0x15 }
};
static const struct lc_kyber_x25519_sk kyber_sk_exp = {
	.sk.sk = { 0xe7, 0xfa, 0x46, 0xcd, 0x07, 0xc1, 0xb4, 0xda, 0x48, 0xcb,
		   0x48, 0xa3, 0xaf, 0xe1, 0x47, 0x9f, 0x96, 0x71, 0x1b, 0xac,
		   0x5c, 0xd2, 0x62, 0xc4, 0x3d, 0xaa, 0xc5, 0xbc, 0x68, 0x66,
		   0x2d, 0x5c, 0x5d, 0x0b, 0x83, 0x55, 0x1b, 0xda, 0xb9, 0xd1,
		   0x48, 0xa4, 0xd7, 0xe4, 0x3d, 0xc6, 0x16, 0x56, 0x1f, 0xb2,
		   0x7f, 0x58, 0x42, 0x83, 0x8c, 0x71, 0x47, 0x22, 0x45, 0x6c,
		   0x4c, 0x7c, 0x44, 0xb2, 0x48, 0xa8, 0xe0, 0x66, 0x37, 0xad,
		   0x40, 0x30, 0xe5, 0x39, 0x5b, 0xd2, 0xa7, 0x81, 0x36, 0x1a,
		   0x9e, 0x23, 0x61, 0xc1, 0x19, 0x38, 0x39, 0x04, 0x87, 0x81,
		   0x2e, 0xdb, 0x21, 0xbd, 0x04, 0x8c, 0x45, 0xe9, 0x83, 0xb2,
		   0x33, 0x94, 0x98, 0x5c, 0x5a, 0x2b, 0x60, 0x7e, 0xfc, 0x97,
		   0x50, 0x57, 0x3c, 0x7e, 0x9f, 0x74, 0x90, 0xce, 0x42, 0x66,
		   0x79, 0x59, 0x92, 0x97, 0xc5, 0x8c, 0xe6, 0x42, 0x0f, 0x2a,
		   0x67, 0x33, 0xb9, 0x01, 0x81, 0x9a, 0xe2, 0x59, 0x9e, 0x50,
		   0x70, 0xf9, 0xf0, 0x77, 0xb9, 0xe0, 0xb9, 0x94, 0x73, 0x80,
		   0x55, 0x2c, 0xc4, 0xb6, 0x24, 0x2b, 0x30, 0xf2, 0x88, 0x46,
		   0xd5, 0x0e, 0x62, 0x38, 0xbd, 0xcb, 0x0b, 0x74, 0x2f, 0x17,
		   0x5c, 0xfe, 0xbb, 0xab, 0xc6, 0x47, 0xa2, 0xf0, 0x32, 0x93,
		   0x99, 0x8c, 0x7d, 0x96, 0x41, 0x13, 0xdd, 0x5a, 0x79, 0xe4,
		   0x7b, 0x40, 0x7c, 0x6b, 0x9c, 0x23, 0x25, 0x2d, 0xa4, 0xf3,
		   0x40, 0x1e, 0xb5, 0x02, 0x1f, 0x3c, 0xc1, 0xfd, 0x43, 0xc7,
		   0x87, 0x01, 0xb9, 0x72, 0x87, 0x71, 0x00, 0x12, 0xb9, 0x85,
		   0x44, 0xac, 0x0f, 0x35, 0xc0, 0x78, 0x8b, 0x7a, 0xa5, 0x14,
		   0x94, 0x5d, 0x41, 0x80, 0xce, 0x44, 0x9d, 0x12, 0x28, 0x01,
		   0x77, 0x79, 0xb7, 0xf3, 0xb5, 0x16, 0xf1, 0x97, 0x58, 0x5a,
		   0x75, 0x29, 0x90, 0x2a, 0x57, 0xd3, 0x08, 0xc0, 0x00, 0xc1,
		   0x42, 0xe3, 0x96, 0x08, 0xd3, 0xba, 0x70, 0x81, 0x21, 0x8b,
		   0x3a, 0x1a, 0x29, 0xe4, 0xe3, 0x9e, 0x4f, 0x21, 0x0c, 0x27,
		   0xa9, 0x2c, 0x30, 0xd0, 0xbf, 0x61, 0x1a, 0x28, 0x03, 0x3a,
		   0x44, 0xc7, 0x13, 0xa1, 0x6a, 0xb2, 0x0a, 0x59, 0x71, 0x7d,
		   0xa1, 0xe7, 0x1a, 0xae, 0x78, 0x67, 0x5b, 0xfc, 0x66, 0xa3,
		   0x97, 0x6d, 0xfb, 0xac, 0x7e, 0xab, 0xfc, 0x89, 0x61, 0x4a,
		   0x24, 0xe7, 0xe0, 0x9e, 0xf7, 0xe4, 0x48, 0xef, 0xc5, 0x28,
		   0xa8, 0x23, 0x30, 0xc2, 0xcb, 0xa2, 0x12, 0x8b, 0xc8, 0x17,
		   0x2a, 0xb6, 0x97, 0x88, 0xba, 0xee, 0x2c, 0x8f, 0x72, 0x1a,
		   0x30, 0x81, 0xb3, 0xb7, 0x83, 0x5c, 0x9d, 0xfd, 0xf9, 0xc8,
		   0x37, 0x25, 0xa9, 0x42, 0xfc, 0xb0, 0xe4, 0x06, 0x45, 0xeb,
		   0x4b, 0x25, 0x8d, 0x62, 0x2c, 0x7c, 0xd5, 0x78, 0x0b, 0x27,
		   0x09, 0x14, 0x42, 0x69, 0xeb, 0x82, 0x3f, 0xa8, 0xa8, 0x55,
		   0x40, 0xa2, 0x74, 0x9a, 0xa1, 0x71, 0x31, 0x85, 0x11, 0x94,
		   0xbc, 0x1d, 0xd9, 0x07, 0x08, 0xf8, 0x09, 0xc3, 0x0d, 0xd8,
		   0x99, 0x66, 0x82, 0x23, 0xdd, 0x20, 0x06, 0xaf, 0x02, 0x42,
		   0x39, 0x50, 0x0c, 0x7d, 0x85, 0x6f, 0xef, 0xb7, 0x16, 0xcd,
		   0x98, 0x8a, 0x2d, 0xea, 0x8a, 0x2f, 0xc1, 0x03, 0x6a, 0x22,
		   0x3e, 0x24, 0x01, 0x80, 0x99, 0x91, 0x22, 0x65, 0x5a, 0xcc,
		   0xee, 0xa5, 0x12, 0xc1, 0x42, 0xb2, 0xf9, 0xd7, 0xc1, 0xd3,
		   0x92, 0x1a, 0x34, 0xe9, 0x49, 0x04, 0xf0, 0x82, 0x97, 0xd8,
		   0x9d, 0xd9, 0xf6, 0x3f, 0xd3, 0x6b, 0xa8, 0x03, 0xf9, 0x9b,
		   0x1a, 0xf5, 0x19, 0x71, 0xf5, 0xaf, 0xde, 0xf3, 0x09, 0x61,
		   0x8a, 0x8b, 0x9c, 0xc2, 0x5f, 0xf4, 0x10, 0x90, 0x0a, 0x43,
		   0x71, 0x8a, 0xb1, 0xbc, 0x4a, 0xc2, 0xac, 0x9a, 0x57, 0x61,
		   0x7f, 0x15, 0x0e, 0x2a, 0xc8, 0x3a, 0x06, 0x03, 0x2e, 0x37,
		   0x3a, 0x92, 0xf1, 0x5a, 0xc4, 0xaa, 0xa8, 0x70, 0x76, 0xe7,
		   0x2c, 0xa5, 0x21, 0x12, 0xda, 0x95, 0xc0, 0x0c, 0x93, 0x28,
		   0xea, 0xfa, 0x70, 0x89, 0x01, 0x32, 0x4a, 0xa3, 0x4d, 0xe9,
		   0xc3, 0x92, 0x78, 0x83, 0x49, 0x2d, 0x79, 0x1a, 0x97, 0x45,
		   0x2d, 0x9d, 0x18, 0x3c, 0x37, 0xb4, 0x45, 0x6c, 0x84, 0x21,
		   0x13, 0xac, 0x6c, 0xfe, 0x59, 0x7e, 0x51, 0xe4, 0x82, 0x34,
		   0xe0, 0x99, 0x7e, 0xe9, 0xcf, 0x45, 0x6a, 0x5f, 0xa3, 0x26,
		   0x40, 0xe1, 0xfb, 0x1e, 0xa6, 0x49, 0x14, 0xff, 0x62, 0x68,
		   0xe1, 0x2b, 0x65, 0x75, 0xd5, 0xb8, 0x2e, 0x2b, 0x81, 0x7a,
		   0x50, 0x1d, 0x2b, 0x07, 0x4f, 0x91, 0xfb, 0x3c, 0xa2, 0xac,
		   0x6e, 0x2a, 0x39, 0x3e, 0xa6, 0xc1, 0x52, 0x7f, 0x95, 0x0c,
		   0xdd, 0xa9, 0x2d, 0xe8, 0xa1, 0xa7, 0x60, 0x10, 0x36, 0xa7,
		   0x33, 0xb5, 0x6d, 0x4b, 0x51, 0xfc, 0xac, 0xc7, 0x03, 0xd1,
		   0x25, 0xdc, 0x87, 0x12, 0x68, 0xc5, 0x93, 0xb1, 0xb7, 0x97,
		   0x47, 0x00, 0x15, 0xe7, 0xba, 0x5c, 0x6d, 0x07, 0x5a, 0x0d,
		   0x93, 0x56, 0xac, 0xb0, 0x5a, 0x1a, 0x80, 0x62, 0x5e, 0xb7,
		   0xbb, 0x0e, 0x6b, 0xc4, 0xdb, 0xe0, 0xb4, 0x6e, 0x2c, 0x81,
		   0x53, 0x5c, 0xcd, 0x9b, 0xb2, 0xa5, 0x4f, 0x02, 0x9d, 0xb1,
		   0x8b, 0x3c, 0xf1, 0x6c, 0xc3, 0x03, 0x00, 0xbe, 0x16, 0x73,
		   0x0e, 0x39, 0x49, 0x57, 0xcf, 0xe7, 0xa3, 0x88, 0xfc, 0x07,
		   0xf8, 0x58, 0x3e, 0xac, 0xb0, 0x1f, 0x22, 0x91, 0xc0, 0xf3,
		   0xf9, 0x13, 0xb3, 0x6a, 0x0e, 0x3f, 0x85, 0x43, 0x6d, 0x78,
		   0x43, 0xcc, 0x49, 0x23, 0x1e, 0x09, 0x16, 0x70, 0x3b, 0x79,
		   0x1c, 0x72, 0x0e, 0xed, 0xa5, 0x5b, 0x47, 0xaa, 0xbb, 0xcc,
		   0x4c, 0x76, 0x72, 0x82, 0x84, 0xee, 0x39, 0x3b, 0xb7, 0x81,
		   0xab, 0x3e, 0xbc, 0x59, 0x60, 0x94, 0x96, 0xac, 0xbc, 0x44,
		   0x22, 0x9a, 0x61, 0xf5, 0xa1, 0x6c, 0x9d, 0xec, 0x6d, 0x0f,
		   0x5c, 0x81, 0xb9, 0x9c, 0x0f, 0xbe, 0xab, 0x47, 0xe6, 0xdc,
		   0x92, 0xb7, 0xa7, 0x92, 0xa7, 0xa1, 0x15, 0xc5, 0xf3, 0x8b,
		   0x22, 0x3b, 0x3e, 0x3e, 0xa1, 0x43, 0xd2, 0x09, 0x83, 0xff,
		   0x1b, 0x8b, 0xd3, 0x75, 0x1e, 0x27, 0x6a, 0x35, 0x95, 0xc3,
		   0x36, 0xfc, 0x38, 0x9d, 0x46, 0x87, 0xb1, 0xf8, 0x03, 0x24,
		   0x37, 0x98, 0xca, 0x37, 0x35, 0x80, 0x9f, 0x06, 0x14, 0xe1,
		   0xd8, 0x6d, 0x1d, 0xab, 0x06, 0xce, 0xe8, 0x95, 0xaa, 0xa7,
		   0x46, 0xeb, 0xb5, 0x96, 0xce, 0x12, 0x68, 0xa5, 0xc3, 0xcd,
		   0xbf, 0xb2, 0x57, 0xe9, 0xfc, 0xa7, 0x52, 0x13, 0x38, 0x70,
		   0x33, 0x0c, 0x8e, 0x7c, 0x22, 0x29, 0xe4, 0x21, 0x29, 0xe1,
		   0x58, 0x60, 0xb5, 0xc7, 0xe9, 0x17, 0xaf, 0x4b, 0x30, 0x42,
		   0x9c, 0xdc, 0xb6, 0x86, 0x57, 0x54, 0x6c, 0x06, 0x73, 0xe5,
		   0xa8, 0x46, 0xec, 0xa7, 0x64, 0xcc, 0x1c, 0xb1, 0x77, 0xb1,
		   0x3b, 0x9d, 0x33, 0x21, 0x3e, 0x20, 0x39, 0xe8, 0xf1, 0x8b,
		   0x80, 0xd2, 0x8e, 0xc0, 0x73, 0x7c, 0xa0, 0x5c, 0x10, 0x14,
		   0x5b, 0x00, 0xd0, 0x44, 0x6e, 0x49, 0xf3, 0x0e, 0xfa, 0x23,
		   0x9e, 0xef, 0x00, 0x7f, 0x37, 0x82, 0x85, 0x92, 0x75, 0x04,
		   0xa3, 0x25, 0x89, 0xf4, 0x3a, 0x77, 0x72, 0xe3, 0x6c, 0x52,
		   0x05, 0x9c, 0xad, 0xe9, 0x54, 0x08, 0xf8, 0x17, 0x9f, 0x89,
		   0x4e, 0xc2, 0xf5, 0xc7, 0xef, 0xda, 0x05, 0x14, 0x37, 0x58,
		   0x09, 0xf6, 0x2d, 0x93, 0x11, 0xa7, 0x79, 0xbb, 0xb1, 0xec,
		   0x50, 0x35, 0x5a, 0x87, 0x75, 0xa6, 0xc7, 0x8b, 0xe5, 0x63,
		   0xc5, 0x3f, 0xfb, 0x51, 0x9b, 0xa4, 0x9f, 0x60, 0x21, 0xc5,
		   0xd7, 0xca, 0x5f, 0x94, 0x13, 0x70, 0x00, 0x48, 0x40, 0x58,
		   0x4a, 0x7b, 0x58, 0x78, 0xa3, 0xfc, 0x38, 0x49, 0x3a, 0xd0,
		   0x72, 0x99, 0xb5, 0x58, 0x81, 0x62, 0x83, 0xd3, 0x7a, 0x03,
		   0x08, 0x76, 0xb3, 0xcf, 0x27, 0xa9, 0x88, 0xc8, 0xae, 0x35,
		   0xdb, 0x0d, 0x50, 0x6a, 0xa8, 0xa2, 0x31, 0x4f, 0x3a, 0x6b,
		   0x5f, 0x04, 0x37, 0x54, 0xbe, 0x51, 0xb8, 0x92, 0xb7, 0x02,
		   0x5a, 0x06, 0x50, 0xee, 0x40, 0xa5, 0xff, 0x76, 0x6c, 0x63,
		   0x09, 0xc9, 0x26, 0x5c, 0x16, 0x61, 0xca, 0x5a, 0xe5, 0xb1,
		   0xaa, 0xe1, 0xf3, 0x84, 0xd3, 0x29, 0x25, 0x9a, 0xf7, 0x4c,
		   0xed, 0xc6, 0x0f, 0x29, 0xfc, 0x04, 0xb2, 0x28, 0x24, 0xe3,
		   0x78, 0x86, 0xed, 0x08, 0x49, 0x5c, 0xd6, 0xb5, 0x50, 0x4c,
		   0xbe, 0x0e, 0x13, 0xcc, 0x94, 0x71, 0xcf, 0x99, 0x04, 0x3e,
		   0x6e, 0x75, 0x07, 0x83, 0x2c, 0x1b, 0x47, 0xcb, 0xac, 0x22,
		   0x68, 0xa8, 0x78, 0x20, 0x32, 0x02, 0x30, 0x23, 0x8e, 0x44,
		   0x7a, 0xcf, 0xa9, 0x9b, 0x63, 0x32, 0xb7, 0x53, 0x1c, 0x7c,
		   0xe5, 0x42, 0x03, 0x1b, 0x93, 0xca, 0x14, 0x25, 0x8f, 0x5f,
		   0x98, 0xb3, 0x0c, 0x87, 0x3f, 0x6d, 0x01, 0xb2, 0xc5, 0x85,
		   0x3f, 0x34, 0x02, 0x69, 0x18, 0x04, 0x80, 0x47, 0x08, 0xa7,
		   0xec, 0xfa, 0x5b, 0xa5, 0x98, 0x0f, 0xef, 0x88, 0x36, 0xce,
		   0xf5, 0xa8, 0x2d, 0xac, 0x83, 0x75, 0xe0, 0x62, 0x34, 0xcb,
		   0xbe, 0xd6, 0x4c, 0xc5, 0xa4, 0xb2, 0xac, 0x06, 0x84, 0xa0,
		   0xb4, 0xd9, 0x86, 0x3e, 0x66, 0x44, 0xa8, 0x33, 0x48, 0x49,
		   0xc0, 0xcf, 0xdb, 0xfb, 0x85, 0x1b, 0x84, 0xb0, 0xe4, 0x8c,
		   0x6f, 0xb0, 0x48, 0x01, 0xd0, 0x52, 0x2f, 0xd5, 0x76, 0x68,
		   0x9b, 0x16, 0x44, 0x67, 0x60, 0x4a, 0xb4, 0xda, 0x2e, 0x34,
		   0xab, 0xcf, 0x76, 0xdc, 0x92, 0x44, 0x67, 0x4c, 0x4e, 0x57,
		   0x30, 0xf5, 0x78, 0x7c, 0x67, 0x90, 0x21, 0x42, 0x4c, 0x13,
		   0x40, 0x15, 0x4d, 0x24, 0x49, 0x33, 0x0d, 0xbc, 0x94, 0x0e,
		   0xba, 0x7b, 0xe1, 0x03, 0x99, 0xec, 0x94, 0x0a, 0x6b, 0x35,
		   0x40, 0x60, 0x14, 0x99, 0x66, 0x50, 0xb1, 0x48, 0xa1, 0x89,
		   0x43, 0x47, 0x7f, 0x5e, 0x95, 0x0f, 0x41, 0xa0, 0x1a, 0xb6,
		   0x34, 0x88, 0xfa, 0xf2, 0x06, 0xf0, 0xf2, 0x25, 0xa7, 0x31,
		   0x09, 0x67, 0x77, 0x84, 0x79, 0x1b, 0x6d, 0xec, 0x98, 0x0e,
		   0x60, 0xeb, 0x8f, 0x96, 0xbb, 0xbf, 0x47, 0x22, 0x22, 0xb0,
		   0x24, 0x8a, 0x12, 0xf9, 0x61, 0x88, 0xa1, 0x7d, 0x68, 0xb8,
		   0x3e, 0x52, 0xc9, 0x82, 0x7b, 0xa2, 0xc6, 0xbc, 0x82, 0x75,
		   0x0b, 0x31, 0xae, 0x46, 0xb9, 0x82, 0x7a, 0x28, 0x77, 0xfc,
		   0xb3, 0x38, 0x00, 0x3a, 0x68, 0xb8, 0xc1, 0xaa, 0x9c, 0x96,
		   0x6b, 0x55, 0xd2, 0x03, 0x78, 0x74, 0x11, 0x57, 0x77, 0x05,
		   0xf6, 0xdb, 0x15, 0x59, 0xf9, 0xb4, 0x25, 0x8a, 0x57, 0x18,
		   0x20, 0x50, 0x5d, 0x83, 0xc1, 0x00, 0x9a, 0xa2, 0xb2, 0x84,
		   0xb3, 0x75, 0xa7, 0xac, 0x61, 0x6a, 0x1b, 0x29, 0x42, 0xa2,
		   0xb4, 0x94, 0xc4, 0x3e, 0xb3, 0x70, 0xc1, 0xb9, 0x4a, 0xcd,
		   0x52, 0x2b, 0x20, 0x8c, 0x17, 0xc7, 0x35, 0x07, 0x7e, 0x60,
		   0x2f, 0x41, 0x41, 0x0c, 0xfb, 0x1b, 0x3e, 0xa1, 0x09, 0x21,
		   0x58, 0xc3, 0xb6, 0xea, 0x43, 0xcc, 0x7a, 0xa7, 0x03, 0x09,
		   0xa6, 0x0f, 0xa9, 0x64, 0x47, 0x21, 0x33, 0x2a, 0x0d, 0xc4,
		   0x55, 0xbd, 0xb5, 0x63, 0xd5, 0xe6, 0x6e, 0xe2, 0x12, 0x10,
		   0x87, 0x2a, 0x59, 0x8d, 0x13, 0xa9, 0x51, 0x83, 0x11, 0x9a,
		   0x50, 0x7a, 0x76, 0x34, 0x3b, 0x45, 0x4b, 0xb3, 0x20, 0x4b,
		   0xad, 0x83, 0x46, 0x9b, 0x56, 0x27, 0x52, 0xa7, 0x3a, 0x00,
		   0xe5, 0xe8, 0xc5, 0xa6, 0xd9, 0x87, 0x0a, 0xca, 0x93, 0x78,
		   0x69, 0x05, 0xc2, 0x0b, 0x85, 0x02, 0xf4, 0x5d, 0x91, 0x14,
		   0xa8, 0xd2, 0xc7, 0xbd, 0x2e, 0xe0, 0x08, 0x7f, 0xb7, 0x80,
		   0xfa, 0x50, 0x78, 0x58, 0x8b, 0x02, 0x6f, 0x14, 0x19, 0x6c,
		   0xc0, 0x68, 0xf6, 0xa7, 0x5e, 0x7b, 0x4a, 0x44, 0x3a, 0xea,
		   0x41, 0x42, 0x5c, 0xaf, 0x2d, 0xa2, 0x12, 0xa3, 0xe6, 0xad,
		   0xf6, 0x44, 0x51, 0x64, 0xc2, 0x90, 0x8f, 0x40, 0x9d, 0x43,
		   0x96, 0x5e, 0xe4, 0x0b, 0x89, 0xd6, 0x54, 0xc8, 0x41, 0x30,
		   0x97, 0x48, 0x6a, 0x9b, 0x90, 0xe2, 0xb0, 0x44, 0x74, 0x38,
		   0x91, 0x18, 0x36, 0xb7, 0xca, 0x84, 0xe7, 0x6b, 0x7a, 0x8e,
		   0xe6, 0x00, 0xa1, 0x3b, 0x55, 0xa8, 0xb8, 0x3d, 0xa0, 0xb2,
		   0x08, 0x25, 0x38, 0x2b, 0xda, 0xc3, 0xbe, 0x25, 0x04, 0x60,
		   0x24, 0xc7, 0x88, 0x76, 0x40, 0x53, 0xf1, 0x92, 0x2c, 0xfa,
		   0xa2, 0xa1, 0xb8, 0x16, 0x22, 0x30, 0xa3, 0x6d, 0xbb, 0x36,
		   0xcf, 0x8e, 0x62, 0xc8, 0xa1, 0xb1, 0x5f, 0x9d, 0x51, 0x90,
		   0x07, 0x0a, 0xb4, 0xf5, 0x70, 0xa3, 0xba, 0xf8, 0x63, 0x7c,
		   0x04, 0xad, 0x19, 0x51, 0xa4, 0xb4, 0x61, 0x0b, 0x5b, 0x7a,
		   0xb2, 0x33, 0x56, 0xc3, 0x9e, 0x36, 0xa8, 0x12, 0x85, 0x58,
		   0xa2, 0x59, 0x14, 0xe1, 0xa8, 0xaf, 0x76, 0xe4, 0x77, 0x12,
		   0xf8, 0x54, 0x38, 0xb1, 0x9e, 0x8d, 0x04, 0xce, 0x92, 0x35,
		   0x20, 0xd3, 0xab, 0x0a, 0xc4, 0xe4, 0x08, 0x58, 0xf2, 0x36,
		   0x2d, 0xd4, 0x9d, 0xb4, 0x46, 0xb7, 0xed, 0xf7, 0xa0, 0xdd,
		   0x4c, 0x83, 0xaa, 0x30, 0x7e, 0xfa, 0x87, 0x76, 0xb3, 0xb8,
		   0xc7, 0xa1, 0x0a, 0x1f, 0xdf, 0xe6, 0x8e, 0xda, 0x1b, 0x9c,
		   0xd4, 0x2a, 0xc0, 0xce, 0xc4, 0x76, 0x0c, 0x25, 0x93, 0x18,
		   0x08, 0x2c, 0x18, 0x41, 0x5d, 0x2c, 0xc2, 0x93, 0x67, 0xf6,
		   0x06, 0xee, 0xea, 0x1c, 0x01, 0x45, 0x38, 0x9e, 0xa2, 0x88,
		   0xf3, 0x83, 0xa6, 0x8d, 0xd6, 0x20, 0x12, 0x7a, 0xac, 0xa8,
		   0x8b, 0xcb, 0x6b, 0x42, 0x45, 0xa6, 0x64, 0x4f, 0x05, 0xa6,
		   0xb9, 0xf1, 0x16, 0x3d, 0x38, 0xe1, 0x7e, 0x57, 0xcb, 0x27,
		   0xcc, 0x5c, 0x46, 0x7b, 0x2c, 0xca, 0x94, 0xf9, 0x74, 0xcf,
		   0xd8, 0x15, 0x0d, 0x52, 0x94, 0xe1, 0x52, 0x20, 0x0b, 0x39,
		   0xba, 0xfa, 0x46, 0x9b, 0x9c, 0xfa, 0xc3, 0xac, 0xeb, 0x7b,
		   0x56, 0xb4, 0x7b, 0x13, 0xf1, 0x71, 0x81, 0xea, 0x95, 0x87,
		   0x78, 0x10, 0xa7, 0x77, 0xae, 0x9d, 0x15, 0x60, 0xed, 0x51,
		   0x19, 0xab, 0x09, 0xb7, 0x6a, 0xb1, 0x20, 0xe3, 0xc6, 0x19,
		   0x61, 0xd5, 0x8c, 0x3a, 0x93, 0x16, 0x69, 0xa0, 0x5f, 0xd2,
		   0x88, 0x2f, 0x51, 0x97, 0xb7, 0xc3, 0xb0, 0x47, 0xa1, 0x97,
		   0xaa, 0x82, 0x0c, 0x20, 0x8a, 0x5c, 0xb5, 0xbc, 0x40, 0xbe,
		   0xb8, 0x25, 0x88, 0xf5, 0x61, 0x0b, 0x0b, 0x35, 0x4e, 0xb8,
		   0x50, 0x39, 0xb0, 0xa4, 0x1d, 0xfd, 0x39, 0x24, 0x8f, 0x16,
		   0x5a, 0xbb, 0x02, 0x7a, 0xfd, 0x68, 0x17, 0xe0, 0x0c, 0x8d,
		   0xad, 0xaa, 0xcb, 0x29, 0xcb, 0x25, 0xe4, 0x24, 0x75, 0x09,
		   0x32, 0xae, 0x27, 0x75, 0x6a, 0x98, 0xb7, 0xb7, 0x49, 0x28,
		   0x53, 0x90, 0x9c, 0xa9, 0x5a, 0x56, 0x92, 0x2a, 0x13, 0x36,
		   0x90, 0x6c, 0x02, 0xb8, 0x12, 0x05, 0xbb, 0x58, 0x69, 0x4d,
		   0xe8, 0xc0, 0x2a, 0x00, 0x5c, 0x23, 0x26, 0x09, 0xc8, 0x79,
		   0x97, 0xb1, 0x6a, 0xb6, 0xbc, 0x79, 0x0d, 0xd3, 0x29, 0x30,
		   0x03, 0xb9, 0xb4, 0x86, 0x7b, 0x2b, 0x6e, 0x04, 0xc4, 0x56,
		   0x18, 0x50, 0x35, 0x94, 0xaa, 0x26, 0x9c, 0x9d, 0x4f, 0xc3,
		   0x9a, 0xa1, 0xec, 0x36, 0x3f, 0xbc, 0x57, 0xc7, 0x02, 0x41,
		   0x31, 0x0b, 0x1d, 0x6b, 0x44, 0x98, 0xe8, 0xa5, 0x56, 0xc4,
		   0x87, 0xb1, 0x6e, 0x45, 0xaa, 0x9b, 0xb9, 0xb1, 0xa6, 0xa9,
		   0xbb, 0x2a, 0x44, 0x6f, 0x6e, 0xe3, 0x09, 0xed, 0xec, 0x1d,
		   0x55, 0x32, 0x99, 0x7a, 0x86, 0x94, 0xdd, 0xb9, 0x91, 0xe3,
		   0xa6, 0x77, 0xf8, 0x95, 0x88, 0x5f, 0xa9, 0x34, 0x81, 0xea,
		   0x21, 0x7e, 0xba, 0x1c, 0x47, 0x4b, 0xc4, 0x81, 0x88, 0x14,
		   0x7f, 0xdc, 0x03, 0x22, 0x81, 0xcc, 0xa5, 0xc3, 0x0d, 0x5b,
		   0x1c, 0x3e, 0xd9, 0x09, 0x32, 0xbd, 0xe1, 0x48, 0x55, 0x92,
		   0x9f, 0x5e, 0x22, 0x4f, 0x27, 0x38, 0x75, 0x36, 0x4b, 0x60,
		   0x0c, 0x7c, 0x91, 0x6b, 0xd3, 0x1f, 0x66, 0xd7, 0xbb, 0xc4,
		   0x70, 0xc5, 0x36, 0xe6, 0x00, 0xe8, 0xda, 0x7f, 0x31, 0xfb,
		   0x3b, 0x99, 0x45, 0x73, 0xe7, 0x77, 0x96, 0xc5, 0x39, 0x69,
		   0x37, 0x1a, 0x5e, 0x30, 0xc3, 0x71, 0xa0, 0x32, 0x50, 0xe4,
		   0x26, 0xc2, 0xdb, 0xda, 0x9e, 0xee, 0xd3, 0x9b, 0x6b, 0x65,
		   0x03, 0x38, 0x80, 0x66, 0x05, 0xd3, 0x5e, 0x64, 0xd5, 0x59,
		   0xd0, 0x91, 0x0d, 0x2e, 0xf7, 0x51, 0xaf, 0xa8, 0x33, 0x00,
		   0x2c, 0x90, 0x5d, 0xb5, 0x5e, 0xe0, 0x09, 0x81, 0x75, 0xf2,
		   0x11, 0x11, 0x82, 0x2f, 0x6a, 0x9c, 0x1a, 0xab, 0xcb, 0x97,
		   0x71, 0xcc, 0x15, 0x3b, 0xb4, 0xb6, 0x19, 0xd7, 0x35, 0xfa,
		   0x82, 0x23, 0xd5, 0xf0, 0x6a, 0x8b, 0x2b, 0x79, 0x08, 0xe3,
		   0x2d, 0x8f, 0x7b, 0x82, 0x9d, 0xac, 0x88, 0xc9, 0xca, 0xa4,
		   0x74, 0x2a, 0x12, 0xa2, 0xa4, 0x4b, 0xc6, 0x14, 0x20, 0xf7,
		   0xd2, 0x6b, 0x87, 0xc3, 0xc8, 0x33, 0xc5, 0x6c, 0xb5, 0x1a,
		   0xcc, 0x1b, 0xd4, 0x8c, 0x9a, 0x64, 0x8b, 0x4f, 0x60, 0x8e,
		   0x4f, 0xeb, 0x37, 0x8e, 0x65, 0x54, 0xb7, 0x3c, 0xa8, 0xe3,
		   0x7b, 0x9d, 0x5f, 0xa7, 0x6b, 0x44, 0x1c, 0x03, 0x97, 0xf8,
		   0xc0, 0xaa, 0x39, 0xb2, 0x34, 0x8a, 0xbc, 0x3c, 0x6d, 0x0d,
		   0x13, 0xa6, 0xbd, 0x91, 0xd6, 0x2a, 0x43, 0xc6, 0x51, 0x60,
		   0x63, 0xdf, 0xd5, 0xbf, 0x77, 0xf0, 0xf1, 0x08, 0x42, 0x24,
		   0x43, 0x7b, 0x47, 0xa8, 0x2b, 0xa2, 0x28, 0xb8, 0x74, 0x69,
		   0xd4, 0xee, 0x89, 0x06, 0xec, 0x34, 0xdb, 0xa7, 0x6c, 0x68,
		   0xd8, 0xa8, 0x22, 0x8d, 0xf3, 0x3c, 0xcf, 0x3a, 0x80, 0xbf,
		   0x15, 0x6b, 0x29, 0x53, 0xa5, 0x31, 0x26, 0x9f, 0x3c, 0xb1,
		   0xee, 0xa9, 0x88, 0x00, 0x4b, 0x93, 0x10, 0x3c, 0xfb, 0x0a,
		   0xee, 0xfd, 0x2a, 0x68, 0x6e, 0x01, 0xfa, 0x4a, 0x58, 0xe8,
		   0xa3, 0x63, 0x9c, 0xa8, 0xa1, 0xe3, 0xf9, 0xae, 0x57, 0xe2 },
	.sk_x25519.sk = { 0x35, 0xb8, 0xcc, 0x87, 0x3c, 0x23, 0xdc, 0x62,
			  0xb8, 0xd2, 0x60, 0x16, 0x9a, 0xfa, 0x2f, 0x75,
			  0xab, 0x91, 0x6a, 0x58, 0xd9, 0x74, 0x91, 0x88,
			  0x35, 0xd2, 0x5e, 0x6a, 0x43, 0x50, 0x85, 0xb2 }
};
static const uint8_t ss_exp[] = {
	0xc9, 0xdb, 0xe5, 0x39, 0x8d, 0x1f, 0xe8, 0x50, 0x86, 0x69, 0xd8, 0x0f,
	0x69, 0x5c, 0xe0, 0xb7, 0xd5, 0xc2, 0x34, 0x25, 0x34, 0x5e, 0xba, 0xf0,
	0x17, 0x5a, 0x32, 0x2e, 0xed, 0xec, 0xa2, 0x34, 0x1f, 0x94, 0x7c, 0x10,
	0x16, 0x58, 0xa1, 0x05, 0xf7, 0x44, 0x91, 0x20, 0x70, 0x57, 0x82, 0x04,
	0x90, 0xf7, 0xd2, 0xa4, 0xdb, 0x7f, 0xdd, 0xcf, 0x02, 0x51, 0x29, 0x74,
	0x46, 0x31, 0x60, 0xd4, 0x1f, 0xa9, 0x20
};
#elif LC_KYBER_K == 4
static const struct lc_kyber_x25519_pk kyber_pk_exp = {
	.pk.pk = { 0x28, 0x86, 0x74, 0x55, 0x84, 0x74, 0xb6, 0xea, 0x3b, 0xb3,
		   0xd1, 0x17, 0x02, 0xa7, 0x12, 0xd1, 0x9b, 0x64, 0x40, 0xf6,
		   0x8c, 0x66, 0x77, 0x5c, 0x61, 0xa1, 0x68, 0x65, 0x66, 0x9b,
		   0xf2, 0x34, 0x0e, 0xa9, 0xf1, 0x8c, 0x40, 0xb2, 0xb3, 0xe4,
		   0x98, 0x8d, 0xf4, 0x7c, 0x31, 0x7c, 0xb0, 0x3f, 0x73, 0x6a,
		   0xaf, 0x2c, 0x1a, 0x5a, 0xd4, 0x23, 0x45, 0x2f, 0x8b, 0xc1,
		   0xb8, 0x1a, 0x74, 0x95, 0x3c, 0xb8, 0x61, 0x11, 0x38, 0x7d,
		   0xf6, 0x7c, 0x47, 0xc7, 0x90, 0x26, 0x38, 0x6e, 0x2a, 0xd2,
		   0xc5, 0xe0, 0x33, 0x25, 0x6f, 0x61, 0x74, 0xd0, 0xd7, 0x85,
		   0xc8, 0x6b, 0x71, 0x5e, 0x25, 0x71, 0x56, 0xab, 0x24, 0xae,
		   0x68, 0x20, 0xaa, 0xb5, 0xa0, 0xb4, 0x3c, 0x5d, 0x15, 0xab,
		   0x25, 0xe2, 0x79, 0x51, 0x18, 0x95, 0x1d, 0xd0, 0x36, 0x0b,
		   0x47, 0xd1, 0x3c, 0xca, 0x45, 0xbd, 0x01, 0x6a, 0x0c, 0xb4,
		   0xc9, 0x7f, 0x3f, 0x66, 0xb1, 0xfc, 0x61, 0x2e, 0x97, 0x41,
		   0x4d, 0xb3, 0x9b, 0x8f, 0xd8, 0x2a, 0x4b, 0xc0, 0x9a, 0x7d,
		   0x92, 0xf3, 0x82, 0x42, 0xe9, 0xa7, 0x19, 0x53, 0x3f, 0xa8,
		   0x40, 0x6b, 0xb9, 0x17, 0x84, 0xfc, 0x15, 0x86, 0xcf, 0x52,
		   0x69, 0x9f, 0xd9, 0x2d, 0x53, 0x03, 0x09, 0xfa, 0xc5, 0x0d,
		   0xe1, 0xfc, 0xc4, 0x5c, 0x1c, 0x9a, 0xa0, 0xdb, 0x9b, 0xcc,
		   0x38, 0x04, 0x34, 0x09, 0x6e, 0x9c, 0x92, 0x17, 0xca, 0xd0,
		   0x36, 0x93, 0x34, 0x73, 0x2f, 0x83, 0x3d, 0x47, 0x98, 0x21,
		   0x16, 0xbc, 0x95, 0x1a, 0x1c, 0x3d, 0x35, 0xd0, 0x88, 0x4a,
		   0x61, 0x42, 0x74, 0xf5, 0xc8, 0x70, 0xa6, 0x7f, 0x05, 0xf5,
		   0x26, 0x25, 0xe7, 0x71, 0xc9, 0xd0, 0x86, 0x5d, 0x71, 0xa4,
		   0xa5, 0x79, 0xa3, 0x0b, 0x1c, 0x0f, 0xad, 0x22, 0x83, 0x62,
		   0xfb, 0x8c, 0x0e, 0xb1, 0xca, 0xfd, 0xc5, 0x81, 0xb2, 0xb8,
		   0x7a, 0xb1, 0xd6, 0xb2, 0xc6, 0x47, 0x92, 0x55, 0x98, 0xa7,
		   0xcf, 0x23, 0x9d, 0x74, 0xe5, 0x6a, 0x4a, 0xb6, 0x00, 0x25,
		   0x59, 0x7e, 0xaf, 0x4c, 0x32, 0xe5, 0x7a, 0x41, 0x25, 0xb4,
		   0x9d, 0x57, 0x31, 0x2d, 0x23, 0x82, 0x5a, 0x32, 0xc1, 0x29,
		   0x28, 0x9b, 0x40, 0xc5, 0xeb, 0x68, 0x8a, 0x7a, 0xaf, 0xbf,
		   0x67, 0x5b, 0x9e, 0x46, 0xa6, 0xb4, 0x66, 0x20, 0x68, 0xe3,
		   0x75, 0x14, 0x5a, 0x2e, 0xb9, 0x67, 0x50, 0x5d, 0x33, 0xca,
		   0x57, 0x19, 0xbd, 0x03, 0xf0, 0x00, 0x4b, 0x58, 0xcf, 0x70,
		   0xf1, 0x0a, 0x4f, 0xeb, 0x59, 0xf8, 0xa9, 0x9c, 0x25, 0x71,
		   0x1c, 0x49, 0x43, 0x0c, 0xef, 0x86, 0x53, 0xa4, 0x1c, 0xb7,
		   0x36, 0x10, 0x0e, 0x00, 0x34, 0xb1, 0x74, 0x32, 0x46, 0x75,
		   0x23, 0x57, 0x80, 0x60, 0x28, 0xdc, 0xf8, 0x53, 0x2b, 0xcc,
		   0xcf, 0x5a, 0x91, 0x01, 0x9a, 0x12, 0xa6, 0xe8, 0x69, 0x7f,
		   0x91, 0x8a, 0x7f, 0x2b, 0x50, 0x48, 0x4f, 0x26, 0x5a, 0xa1,
		   0x39, 0x76, 0x97, 0xe2, 0x5e, 0xd8, 0x67, 0x47, 0x23, 0x38,
		   0x8a, 0x4e, 0x51, 0x96, 0xfb, 0x35, 0x95, 0xf5, 0x6c, 0x77,
		   0xdc, 0x3a, 0x70, 0x34, 0x24, 0x53, 0x82, 0x2c, 0x7a, 0xdc,
		   0x87, 0xb9, 0xd1, 0xb3, 0x37, 0x3d, 0xba, 0x42, 0x70, 0x45,
		   0x5b, 0xe4, 0xc4, 0xb3, 0xb3, 0xe6, 0x47, 0x6a, 0x08, 0xa2,
		   0xf4, 0xe2, 0xcb, 0x81, 0xb8, 0x5a, 0xd1, 0x95, 0x3e, 0xcc,
		   0xb6, 0x7b, 0xa0, 0xf5, 0x1d, 0x11, 0x0c, 0xbb, 0x3c, 0x83,
		   0xc4, 0x80, 0x91, 0x8d, 0x10, 0xdb, 0x0d, 0xb9, 0x59, 0x56,
		   0x2a, 0x09, 0x11, 0x5a, 0x29, 0xbe, 0xa5, 0x8b, 0x31, 0xe6,
		   0x45, 0xb3, 0x13, 0x74, 0x96, 0xe4, 0x9b, 0xc4, 0xb6, 0xd2,
		   0x2b, 0x3d, 0x2c, 0xbb, 0x7c, 0xd8, 0x66, 0x52, 0xab, 0x09,
		   0xd9, 0xf5, 0x22, 0x4d, 0x39, 0x46, 0xf5, 0x4a, 0x3a, 0x8e,
		   0xe3, 0x06, 0xf2, 0x04, 0x89, 0x77, 0x54, 0x7f, 0xaf, 0xa4,
		   0xa5, 0xf9, 0x09, 0x60, 0xd1, 0x82, 0xc1, 0xce, 0xc5, 0x9c,
		   0xb5, 0xe6, 0x64, 0x45, 0xcc, 0x9d, 0x38, 0x97, 0xbc, 0x29,
		   0xd0, 0x74, 0x00, 0x4a, 0xcc, 0x51, 0xf3, 0xb3, 0x06, 0xfa,
		   0x5a, 0xfb, 0x03, 0x4a, 0xa6, 0x50, 0xaa, 0x3d, 0xcc, 0x29,
		   0x25, 0xbc, 0x0c, 0xda, 0x35, 0xba, 0x9d, 0x35, 0x43, 0xdb,
		   0x27, 0xa3, 0x74, 0xf4, 0x3d, 0x10, 0xc2, 0x5b, 0xbd, 0x06,
		   0x36, 0x68, 0xd1, 0x30, 0x20, 0x75, 0xcc, 0x5a, 0xa2, 0x6e,
		   0x19, 0xc2, 0xa0, 0x6f, 0xdb, 0xc7, 0x25, 0x8b, 0x53, 0xc6,
		   0x87, 0xb4, 0x72, 0x71, 0xc7, 0xe3, 0xe8, 0xa2, 0x50, 0x44,
		   0x2b, 0xca, 0x63, 0x2f, 0x85, 0x58, 0x7b, 0xdf, 0x02, 0xa9,
		   0x83, 0xea, 0x7d, 0x2b, 0x14, 0x08, 0x5a, 0xe0, 0x76, 0xa7,
		   0xa9, 0xb8, 0x41, 0xc6, 0x8e, 0x0f, 0xd1, 0x69, 0x22, 0xa4,
		   0x8f, 0x08, 0x29, 0xc8, 0x14, 0xa8, 0x17, 0xd8, 0xe2, 0x85,
		   0xef, 0x08, 0x75, 0x8c, 0x44, 0xc5, 0x99, 0xd8, 0x30, 0x2a,
		   0x7c, 0xb0, 0x0f, 0xc9, 0x27, 0xcf, 0x49, 0x9f, 0x05, 0x88,
		   0x44, 0x7c, 0xf2, 0x8d, 0x4e, 0x77, 0x8d, 0x03, 0x9c, 0x3e,
		   0x6f, 0xc5, 0x19, 0x2a, 0xf1, 0x1c, 0x1d, 0xfb, 0x8a, 0xb8,
		   0x5a, 0x06, 0xf5, 0xa4, 0xaa, 0x92, 0x43, 0x74, 0x79, 0xe9,
		   0xb9, 0x18, 0xc6, 0xcc, 0x68, 0x9a, 0x45, 0xc7, 0xe8, 0xa9,
		   0x48, 0x97, 0x07, 0x5e, 0xfb, 0x74, 0x82, 0xb1, 0x44, 0x17,
		   0x33, 0x95, 0xb0, 0x4a, 0x2d, 0xa9, 0x88, 0xc8, 0x11, 0xf8,
		   0x78, 0xe9, 0xb5, 0x65, 0xbe, 0xca, 0x33, 0xf2, 0x64, 0x3c,
		   0x6a, 0xd0, 0xb2, 0xca, 0xf2, 0x31, 0x5b, 0xb0, 0xcb, 0xd3,
		   0xf1, 0x84, 0xcd, 0x98, 0x7f, 0xbd, 0x43, 0x71, 0x78, 0x19,
		   0x18, 0x3e, 0x05, 0xa6, 0x69, 0x57, 0x77, 0x1c, 0x89, 0xaa,
		   0x83, 0x99, 0x60, 0xc6, 0xa8, 0x59, 0x56, 0x00, 0xcc, 0xad,
		   0x23, 0x55, 0x96, 0x57, 0x2c, 0x2f, 0x52, 0x06, 0x0b, 0x3a,
		   0x0b, 0x08, 0x75, 0xc8, 0x4a, 0x02, 0x9e, 0x0b, 0x3a, 0x28,
		   0x84, 0x74, 0x35, 0xee, 0xe5, 0x3e, 0x72, 0xa9, 0x88, 0x06,
		   0xc3, 0x37, 0x00, 0xa1, 0x3e, 0xe8, 0x19, 0x0b, 0xc3, 0x72,
		   0x2d, 0xc6, 0x34, 0x66, 0xea, 0x85, 0x97, 0xde, 0x7a, 0x8e,
		   0x51, 0x97, 0x2e, 0x99, 0x44, 0x1f, 0x83, 0x60, 0xce, 0x88,
		   0xf9, 0x56, 0x97, 0x44, 0x52, 0x9d, 0xe0, 0x83, 0xa7, 0xf7,
		   0xc5, 0x04, 0x42, 0xa3, 0xd7, 0xf5, 0xc4, 0x53, 0x69, 0x61,
		   0xb8, 0x95, 0xae, 0x90, 0xe2, 0x29, 0x1e, 0x0b, 0x99, 0xb3,
		   0xe1, 0xcc, 0x01, 0xa8, 0xa3, 0xa4, 0x5a, 0xba, 0x72, 0xc4,
		   0x06, 0xba, 0xa3, 0x0e, 0xd5, 0xf3, 0x81, 0x86, 0x00, 0xb7,
		   0x7f, 0x86, 0x5a, 0x5d, 0x79, 0x6e, 0xe8, 0x23, 0xc9, 0x8d,
		   0xd6, 0x0a, 0x28, 0x5b, 0x61, 0xe2, 0xe7, 0x9e, 0x84, 0x2c,
		   0xc6, 0x3e, 0x62, 0x1e, 0x3b, 0x36, 0x15, 0x14, 0x44, 0xb8,
		   0x76, 0x41, 0x65, 0x8c, 0x64, 0xc3, 0x71, 0x1b, 0xa7, 0x3e,
		   0x0a, 0x97, 0xaa, 0x31, 0x30, 0x07, 0xc9, 0xc9, 0xc9, 0xbc,
		   0x77, 0x16, 0xf7, 0x71, 0x8f, 0xc2, 0x7c, 0xbf, 0x32, 0xa0,
		   0x5b, 0xfc, 0x00, 0xd7, 0x19, 0x9a, 0x43, 0x39, 0x45, 0x9a,
		   0x30, 0x79, 0x93, 0x3a, 0x5f, 0x4f, 0xc8, 0x3b, 0x15, 0xe1,
		   0x10, 0x3d, 0x95, 0x1a, 0x68, 0x05, 0x1a, 0x98, 0xe7, 0x3e,
		   0xe3, 0xb9, 0x90, 0xc5, 0x4a, 0x07, 0x9d, 0x92, 0x24, 0x5d,
		   0x8c, 0x8f, 0xac, 0xd4, 0xa0, 0x9b, 0x12, 0x89, 0xe4, 0x6c,
		   0x8e, 0x22, 0x54, 0x11, 0x2e, 0x41, 0x7d, 0xb4, 0x54, 0xb9,
		   0xf3, 0x2c, 0x45, 0xae, 0x69, 0xad, 0xe0, 0x26, 0x02, 0x69,
		   0x93, 0x0a, 0x40, 0xd8, 0x15, 0xae, 0x60, 0xc4, 0x75, 0x1b,
		   0x06, 0x75, 0xc2, 0x1c, 0x22, 0x5b, 0x70, 0x00, 0x97, 0x9f,
		   0x25, 0x6b, 0xbd, 0xf4, 0x45, 0x2b, 0xa9, 0x8c, 0xab, 0xd0,
		   0xb3, 0x30, 0xd5, 0xc3, 0x59, 0x2e, 0x58, 0x99, 0x42, 0xe0,
		   0x9f, 0xbd, 0x87, 0x89, 0xb4, 0xc1, 0x17, 0x29, 0x84, 0x42,
		   0x29, 0xe2, 0xcb, 0x96, 0x7c, 0xb1, 0xcb, 0xe5, 0xc8, 0xe9,
		   0xe0, 0x8b, 0xf6, 0x94, 0xb4, 0x52, 0xbc, 0x6c, 0x97, 0x9c,
		   0x10, 0x01, 0x17, 0x45, 0xb0, 0xe6, 0x3e, 0xf5, 0xaa, 0x4e,
		   0x91, 0x0a, 0x29, 0x3c, 0xa5, 0xcd, 0x1d, 0xb1, 0x2c, 0x09,
		   0x1a, 0x5a, 0xfc, 0x79, 0x4e, 0x18, 0x22, 0x43, 0x87, 0x6b,
		   0x75, 0x8b, 0x1b, 0x0e, 0xb6, 0xca, 0x3d, 0xb3, 0x82, 0xb7,
		   0xc6, 0x51, 0xc4, 0x0f, 0x53, 0x2f, 0x6c, 0xe2, 0xc1, 0x8b,
		   0x9c, 0xa5, 0x6b, 0xb5, 0x7e, 0x1f, 0x9a, 0xa5, 0x65, 0xba,
		   0xc4, 0x68, 0x36, 0x99, 0x79, 0x38, 0x63, 0x7a, 0x92, 0xb7,
		   0x0f, 0xaa, 0xae, 0x43, 0xd2, 0x26, 0xe8, 0x72, 0x5d, 0x3f,
		   0xc7, 0x35, 0xbd, 0x5b, 0x25, 0x3f, 0x32, 0xbb, 0x34, 0xa8,
		   0x0c, 0xe0, 0x1a, 0x23, 0x7a, 0x63, 0x5a, 0x4d, 0x79, 0xc2,
		   0x22, 0xb9, 0x88, 0xef, 0x50, 0x75, 0x4d, 0x23, 0x40, 0x45,
		   0xb9, 0xc3, 0xf3, 0x61, 0xa5, 0xc7, 0xc1, 0x5b, 0xd5, 0x57,
		   0xab, 0xe1, 0xb7, 0xb7, 0xe0, 0xdc, 0x0c, 0x95, 0x6b, 0x8a,
		   0x64, 0x13, 0x8e, 0xdc, 0xdb, 0xb2, 0x68, 0x4c, 0x2c, 0x4c,
		   0x69, 0x4b, 0xf7, 0x06, 0x24, 0x7e, 0x01, 0x06, 0x38, 0x35,
		   0xa6, 0x39, 0xd2, 0x08, 0xe1, 0xe3, 0x11, 0xb9, 0xe9, 0x11,
		   0x63, 0x84, 0x12, 0x90, 0x53, 0xc0, 0xc2, 0x85, 0x43, 0x6c,
		   0x25, 0x87, 0x05, 0x3c, 0x1b, 0xc4, 0x83, 0xa0, 0xd8, 0xda,
		   0x38, 0x7d, 0x4b, 0xbf, 0xea, 0xe5, 0x12, 0xea, 0x31, 0xc2,
		   0xcf, 0x07, 0x03, 0xdd, 0x77, 0x1d, 0x4a, 0x06, 0x7e, 0x67,
		   0x18, 0x9b, 0x79, 0x80, 0xca, 0x98, 0xbb, 0xcc, 0xda, 0x38,
		   0x4c, 0x96, 0xe8, 0x27, 0x2e, 0x0c, 0x70, 0x59, 0x58, 0x44,
		   0xa4, 0x65, 0x49, 0x7f, 0xac, 0x3a, 0x58, 0xa7, 0x4f, 0xf2,
		   0x86, 0x9f, 0xb2, 0x53, 0x7e, 0x53, 0xb6, 0x53, 0xdb, 0x58,
		   0xa4, 0x81, 0x48, 0x67, 0xd3, 0x0a, 0x6b, 0xcb, 0x5a, 0x1f,
		   0x80, 0x46, 0xce, 0x3a, 0xc2, 0x0b, 0x11, 0x33, 0x84, 0x87,
		   0xca, 0x87, 0xa9, 0x8c, 0xaf, 0xb3, 0x13, 0x77, 0x3e, 0x81,
		   0x25, 0x71, 0x72, 0x18, 0xdb, 0x4c, 0x23, 0x8e, 0x5c, 0x09,
		   0xfd, 0x2c, 0xa0, 0x1a, 0xf4, 0x9c, 0xb7, 0xe5, 0x81, 0x76,
		   0x20, 0xab, 0x46, 0x6a, 0x57, 0x4d, 0x1a, 0xaa, 0xbd, 0x53,
		   0xb7, 0xd0, 0xd6, 0x30, 0x91, 0xfc, 0x17, 0x9b, 0x54, 0x9f,
		   0x41, 0xdc, 0x04, 0x8f, 0xb8, 0x8f, 0xa5, 0x48, 0x67, 0xb0,
		   0x63, 0x56, 0xb1, 0xdb, 0x8a, 0xc4, 0x41, 0x06, 0x47, 0xd8,
		   0x4f, 0xe1, 0xb7, 0xbf, 0x44, 0x83, 0x89, 0x9e, 0x63, 0x65,
		   0x5e, 0xf5, 0xbd, 0xe8, 0xe8, 0x24, 0xbf, 0x95, 0x66, 0x85,
		   0x82, 0x29, 0xa6, 0x50, 0x58, 0x09, 0xe3, 0x37, 0x67, 0x4a,
		   0x1f, 0xc0, 0x27, 0x0c, 0xa6, 0x9c, 0x2a, 0xbc, 0xb3, 0xca,
		   0x05, 0x4c, 0x2b, 0x55, 0x70, 0x0f, 0xde, 0xb1, 0x69, 0x02,
		   0x2c, 0x7e, 0x37, 0xe3, 0x62, 0xd2, 0xfc, 0xb3, 0x71, 0xa2,
		   0x76, 0xb8, 0x54, 0x90, 0xeb, 0x23, 0x55, 0xd6, 0x42, 0x11,
		   0x87, 0x52, 0xbc, 0x16, 0xf4, 0x74, 0x4f, 0xe9, 0x11, 0x57,
		   0x12, 0x49, 0x08, 0x11, 0x62, 0xa4, 0x11, 0x84, 0xf7, 0x19,
		   0xa1, 0xe1, 0x30, 0x29, 0xc9, 0x53, 0x96, 0x9b, 0xc1, 0x59,
		   0xfd, 0x55, 0x1b, 0x15, 0x5a, 0x7a, 0x68, 0x0a, 0x82, 0x36,
		   0xe7, 0xdd, 0x9a, 0x1e, 0x65, 0x52, 0xdd, 0xb9, 0x53, 0xe5,
		   0xc8, 0x19, 0x4c, 0xc3, 0x38, 0xcc, 0xb8, 0x8b, 0xd4, 0xda,
		   0x4c, 0x1a, 0x4c, 0xe8, 0xaa, 0xe1, 0x59, 0x39 },
	.pk_x25519.pk = { 0xe5, 0x6f, 0x17, 0x57, 0x67, 0x40, 0xce, 0x2a,
			  0x32, 0xfc, 0x51, 0x45, 0x03, 0x01, 0x45, 0xcf,
			  0xb9, 0x7e, 0x63, 0xe0, 0xe4, 0x1d, 0x35, 0x42,
			  0x74, 0xa0, 0x79, 0xd3, 0xe6, 0xfb, 0x2e, 0x15 }
};
static const struct lc_kyber_x25519_sk kyber_sk_exp = {
	.sk.sk = { 0x5b, 0x80, 0x95, 0xd5, 0xa6, 0x9e, 0xcb, 0xe9, 0xab, 0x66,
		   0xc6, 0x69, 0x94, 0x98, 0x52, 0x5e, 0x93, 0xc3, 0x3c, 0xfc,
		   0x45, 0x82, 0x03, 0x44, 0x8e, 0xe3, 0x38, 0x20, 0x76, 0x95,
		   0x44, 0x1b, 0x33, 0x29, 0xa8, 0xb1, 0x94, 0x3c, 0x7d, 0xc2,
		   0x12, 0xad, 0x93, 0xa6, 0x3d, 0xbe, 0x91, 0x39, 0x95, 0xac,
		   0x08, 0x2b, 0x25, 0xa6, 0x89, 0x61, 0xc5, 0xb2, 0x18, 0xc5,
		   0xf3, 0xd3, 0x3a, 0x8f, 0x36, 0x3b, 0x54, 0x8a, 0x9b, 0x42,
		   0x48, 0x2e, 0xf3, 0x5b, 0xb1, 0x83, 0x27, 0x20, 0xf0, 0xdb,
		   0xc0, 0x27, 0x19, 0xab, 0x55, 0x53, 0x8e, 0xf0, 0x91, 0x09,
		   0x3a, 0xd0, 0x6c, 0xc8, 0x88, 0xcd, 0x08, 0xe6, 0x05, 0x98,
		   0xda, 0x6a, 0x58, 0x2c, 0xbe, 0xd1, 0x69, 0x5e, 0x6d, 0x01,
		   0x97, 0xff, 0xd1, 0x29, 0xcd, 0xb1, 0x35, 0xa1, 0xcb, 0x77,
		   0xeb, 0x45, 0x54, 0x1e, 0x8b, 0xa3, 0xbd, 0xb9, 0x67, 0xb0,
		   0x96, 0x1f, 0x6b, 0xe8, 0x57, 0xc8, 0x10, 0x02, 0xd8, 0xd1,
		   0x51, 0x91, 0x85, 0x70, 0x80, 0xd0, 0x61, 0x3a, 0xd9, 0xa0,
		   0x9a, 0x35, 0x37, 0xd0, 0x81, 0x92, 0xbb, 0x22, 0xa1, 0x64,
		   0x2c, 0x06, 0x40, 0x9c, 0xcb, 0x7e, 0x48, 0x5c, 0xba, 0xac,
		   0x71, 0xea, 0xd3, 0x2d, 0x3d, 0x8c, 0x8c, 0xfa, 0x5c, 0x36,
		   0x81, 0xa7, 0xae, 0xb4, 0x72, 0x69, 0xd4, 0x26, 0x5f, 0xef,
		   0xf5, 0xc7, 0xdc, 0x20, 0x3a, 0x7a, 0x3a, 0x31, 0x4a, 0x8c,
		   0xcb, 0x89, 0x15, 0xcb, 0x54, 0x47, 0x97, 0x7d, 0x76, 0xaa,
		   0x4a, 0xa0, 0xa6, 0x0e, 0x8a, 0x80, 0x07, 0x29, 0x9f, 0x43,
		   0x42, 0xc1, 0x36, 0x1b, 0x7c, 0x15, 0xc9, 0x78, 0xe8, 0x06,
		   0xb7, 0x73, 0x98, 0x7b, 0xde, 0xe7, 0x5c, 0x4a, 0x37, 0x04,
		   0x95, 0xa3, 0x4f, 0x6c, 0x11, 0x08, 0x6c, 0x24, 0xad, 0x6f,
		   0x1a, 0x33, 0x8b, 0x78, 0x24, 0x79, 0xa3, 0x87, 0xa8, 0x12,
		   0x70, 0x18, 0xe6, 0x47, 0x72, 0xd9, 0x66, 0x62, 0xf3, 0x3c,
		   0xd2, 0x34, 0x4c, 0x30, 0x85, 0x1c, 0xfe, 0x14, 0x8e, 0xae,
		   0xa2, 0xcc, 0x2c, 0x44, 0x46, 0x86, 0x54, 0x80, 0x32, 0x98,
		   0x5d, 0x77, 0xc0, 0x1d, 0x10, 0xfb, 0x38, 0xbe, 0x02, 0x19,
		   0xe7, 0xc2, 0x84, 0xfb, 0x78, 0x8b, 0x04, 0x8b, 0xbb, 0x88,
		   0x23, 0x0d, 0xb4, 0xc5, 0xb2, 0x93, 0x69, 0x00, 0xe4, 0x32,
		   0x3c, 0x09, 0xac, 0xcc, 0x25, 0xf1, 0x6d, 0xe8, 0xf2, 0x49,
		   0x2b, 0x59, 0xc3, 0x81, 0xf3, 0xc9, 0xbc, 0x84, 0x0f, 0xfd,
		   0xa6, 0xc5, 0xc3, 0x3a, 0x88, 0x9d, 0xf7, 0xa6, 0xd4, 0xf6,
		   0x66, 0x2a, 0x35, 0x79, 0x6d, 0xc4, 0x92, 0xb9, 0xf9, 0x90,
		   0x8f, 0x70, 0x89, 0x92, 0x77, 0x51, 0xff, 0x24, 0x0d, 0x8a,
		   0xe5, 0x3c, 0x47, 0x6b, 0x1d, 0x14, 0xe1, 0x4b, 0x79, 0x95,
		   0x4d, 0xb2, 0x29, 0x22, 0xb1, 0x53, 0x00, 0x37, 0x63, 0x65,
		   0x84, 0xfc, 0x33, 0x33, 0xab, 0x86, 0x79, 0x69, 0x5e, 0x3e,
		   0x5b, 0x01, 0xec, 0x13, 0x18, 0x69, 0xb0, 0x23, 0x5d, 0x66,
		   0x40, 0xfc, 0x8c, 0x68, 0xf3, 0xfb, 0x81, 0xe6, 0x2c, 0xa4,
		   0x8a, 0x18, 0x0d, 0x4f, 0xa6, 0x73, 0x26, 0x19, 0xba, 0x99,
		   0xd2, 0x37, 0x35, 0xa4, 0x2a, 0xd8, 0x53, 0xa9, 0x2e, 0x53,
		   0x5e, 0x00, 0x6d, 0xc8, 0xf2, 0x39, 0x0b, 0x8e, 0x78, 0x93,
		   0x5c, 0x8a, 0x76, 0x51, 0x95, 0x33, 0xab, 0x63, 0xb2, 0x1c,
		   0x33, 0x09, 0x17, 0x4b, 0x47, 0x7b, 0x03, 0xb4, 0xef, 0x10,
		   0x17, 0x2e, 0xc0, 0xca, 0xe6, 0x44, 0xcc, 0xa6, 0x23, 0x67,
		   0xe8, 0xda, 0x82, 0x36, 0x06, 0x25, 0x66, 0xda, 0x4d, 0x69,
		   0x29, 0x33, 0x76, 0x5b, 0x9f, 0xf1, 0x1a, 0x40, 0x37, 0xd7,
		   0x81, 0x43, 0xd2, 0xb4, 0x8f, 0xa6, 0x5a, 0x59, 0x39, 0xa9,
		   0xac, 0x17, 0x86, 0x92, 0x97, 0x8d, 0xdf, 0x52, 0xba, 0x42,
		   0xa7, 0x0b, 0xc1, 0xb7, 0x4a, 0x45, 0x80, 0x0a, 0x8e, 0xa1,
		   0x4e, 0x91, 0x25, 0xb0, 0x7d, 0x42, 0x0c, 0x6e, 0x61, 0xc5,
		   0x93, 0xa8, 0x7e, 0xb8, 0x6b, 0x89, 0x0e, 0x5c, 0x87, 0x24,
		   0x80, 0xaf, 0x42, 0xcc, 0x92, 0x07, 0xe2, 0x5a, 0xd3, 0x43,
		   0x8a, 0x01, 0x91, 0x9e, 0x13, 0xaa, 0x52, 0xf0, 0x6a, 0xc9,
		   0x34, 0x0a, 0x9c, 0x4c, 0xa1, 0x7a, 0x17, 0xac, 0x4c, 0x0e,
		   0x0c, 0x45, 0xc1, 0x77, 0x8c, 0x80, 0xfc, 0x7f, 0x39, 0x71,
		   0x65, 0x4c, 0x37, 0x6e, 0x1e, 0xc1, 0x4d, 0x3c, 0x4b, 0xb9,
		   0xa4, 0xba, 0x4e, 0x15, 0x97, 0xad, 0xff, 0x50, 0x35, 0x75,
		   0xa4, 0x4d, 0xfa, 0x41, 0x2d, 0x4b, 0xf9, 0xbd, 0xd8, 0xd0,
		   0x40, 0x1d, 0x9b, 0x11, 0x9a, 0xeb, 0xba, 0x19, 0x6a, 0x4c,
		   0xb2, 0x16, 0xa1, 0x3d, 0x2a, 0x2f, 0xde, 0xf7, 0x80, 0x2c,
		   0xec, 0x25, 0x5c, 0x91, 0x4b, 0xa3, 0x19, 0x97, 0x5f, 0x1c,
		   0x43, 0x64, 0xf7, 0x18, 0x8a, 0x1b, 0xa1, 0x17, 0x71, 0x09,
		   0x7c, 0x61, 0x68, 0x49, 0x82, 0x8f, 0x75, 0xa2, 0xc1, 0x1d,
		   0x24, 0x03, 0x46, 0xd6, 0xb6, 0xcc, 0xa8, 0x6c, 0xc0, 0x44,
		   0x8f, 0xec, 0x17, 0x14, 0xbf, 0x3a, 0xcd, 0x5d, 0x44, 0x39,
		   0x66, 0x8b, 0x0e, 0xd4, 0xf3, 0xaf, 0xd2, 0x22, 0xaf, 0x7d,
		   0x09, 0x73, 0xe9, 0x14, 0x08, 0xe7, 0x40, 0xcb, 0x03, 0xb3,
		   0x5c, 0xc8, 0x49, 0x06, 0x52, 0x2b, 0x62, 0xa5, 0x6c, 0xb8,
		   0xdf, 0x10, 0x82, 0x68, 0x81, 0xc1, 0xd0, 0x3c, 0x1d, 0x7f,
		   0x41, 0x95, 0xd7, 0xe7, 0xcf, 0xeb, 0x85, 0x6c, 0xe8, 0x37,
		   0x0c, 0xc8, 0x96, 0x3a, 0x00, 0x84, 0x9a, 0x55, 0x54, 0x8c,
		   0x37, 0x6c, 0x84, 0x7b, 0x25, 0x69, 0x47, 0x00, 0x4a, 0xeb,
		   0x34, 0x30, 0xd6, 0xe5, 0x84, 0x03, 0xaa, 0x21, 0x04, 0x3a,
		   0xa5, 0x83, 0x35, 0xba, 0x01, 0x20, 0xa8, 0x8f, 0xe4, 0x95,
		   0xe2, 0x80, 0x9f, 0xe2, 0x40, 0xb1, 0x62, 0xe1, 0x99, 0xf8,
		   0x69, 0x73, 0x1c, 0xe2, 0x17, 0x94, 0x25, 0xbd, 0x4d, 0xe8,
		   0xa6, 0x8a, 0xd7, 0x69, 0x7d, 0x18, 0xb6, 0xf1, 0x70, 0x81,
		   0x77, 0x5c, 0x1b, 0x8d, 0x75, 0xb1, 0xac, 0x71, 0xca, 0x7d,
		   0x85, 0x20, 0xad, 0x50, 0x43, 0x31, 0xeb, 0x6e, 0x3d, 0x82,
		   0x2b, 0x4d, 0x26, 0x77, 0x7b, 0xfc, 0x40, 0xc5, 0xe7, 0x5c,
		   0x0c, 0x35, 0x2e, 0xdd, 0x27, 0x0f, 0x7e, 0x82, 0x67, 0xd1,
		   0x93, 0x28, 0x06, 0x58, 0x00, 0x5d, 0x39, 0x2b, 0x2c, 0x64,
		   0x87, 0xce, 0xf9, 0x0a, 0x4c, 0x6c, 0xb1, 0xc1, 0xc7, 0x0c,
		   0xb8, 0x2a, 0xa0, 0xd8, 0xea, 0x98, 0x50, 0x15, 0x1b, 0x8a,
		   0x06, 0x44, 0x52, 0x99, 0x76, 0x7d, 0xe7, 0x76, 0xb0, 0x19,
		   0x60, 0xc1, 0x0c, 0x41, 0x64, 0x07, 0x08, 0x3e, 0xda, 0x18,
		   0x90, 0x6b, 0x9b, 0x5b, 0x96, 0x7b, 0xc8, 0x88, 0x7f, 0x8c,
		   0x7b, 0xc5, 0x80, 0x6a, 0x1c, 0x12, 0x9c, 0x78, 0x05, 0x27,
		   0x0d, 0x76, 0x84, 0x5d, 0x14, 0x1c, 0x85, 0xc8, 0xe6, 0xc5,
		   0xd1, 0x84, 0x0e, 0xbc, 0xb5, 0x21, 0xc1, 0xdc, 0x85, 0x6e,
		   0x25, 0x9e, 0x8c, 0x21, 0x25, 0x1f, 0x09, 0x53, 0x87, 0xf7,
		   0x7d, 0xc0, 0x3b, 0x5f, 0x21, 0xca, 0x53, 0x7d, 0xf8, 0x04,
		   0xbf, 0x17, 0xca, 0x8a, 0xa0, 0xae, 0x3e, 0xbb, 0x2c, 0xcb,
		   0xe2, 0x14, 0x4d, 0x1a, 0x6c, 0x6b, 0x19, 0xad, 0x9d, 0x20,
		   0x04, 0x44, 0x14, 0x69, 0x8f, 0x0c, 0xd0, 0x1a, 0xb6, 0x43,
		   0x71, 0xe6, 0x35, 0x98, 0x8b, 0x5b, 0x7a, 0xb3, 0x30, 0x84,
		   0x06, 0x23, 0x74, 0xd2, 0x8e, 0x9d, 0x79, 0xa6, 0xbf, 0x8c,
		   0x2d, 0x50, 0x90, 0x62, 0x70, 0x7c, 0x0f, 0x22, 0x9c, 0x21,
		   0xe8, 0x11, 0xa8, 0x0a, 0xf6, 0x75, 0xb5, 0x34, 0x70, 0x05,
		   0x05, 0x95, 0xb3, 0xb6, 0x2f, 0x4e, 0x2c, 0xa6, 0xd8, 0x18,
		   0x29, 0x2d, 0x27, 0x00, 0x15, 0xb1, 0x64, 0xb8, 0x38, 0x55,
		   0x0d, 0xf9, 0x8f, 0x1a, 0x4b, 0x44, 0xb3, 0x60, 0x3e, 0xc9,
		   0x94, 0xa2, 0xf8, 0x91, 0x7f, 0x76, 0xc2, 0x7c, 0x2a, 0x16,
		   0x9d, 0x3b, 0x12, 0x11, 0xc2, 0xca, 0x78, 0x51, 0x02, 0x1a,
		   0xd8, 0x77, 0x98, 0x34, 0xf0, 0x26, 0xff, 0xe6, 0x43, 0xeb,
		   0x82, 0x62, 0x59, 0xf5, 0x6a, 0x6c, 0xb5, 0x34, 0x0e, 0x67,
		   0x95, 0x76, 0xc9, 0xb2, 0x6b, 0x45, 0x6a, 0x8b, 0x02, 0x45,
		   0xce, 0xe6, 0x1e, 0x7c, 0x28, 0xc5, 0x34, 0x7c, 0x83, 0x5b,
		   0x88, 0xc1, 0x48, 0xd7, 0x5c, 0x0e, 0x86, 0x07, 0x32, 0x92,
		   0x68, 0x35, 0xd1, 0x12, 0x64, 0x7b, 0xcc, 0xec, 0xd3, 0xaf,
		   0x2e, 0x98, 0x8d, 0x3a, 0xe8, 0x35, 0x89, 0xd5, 0xca, 0x81,
		   0x09, 0x52, 0x0b, 0x19, 0xa9, 0xb2, 0xa6, 0x61, 0x5c, 0xa1,
		   0x86, 0xf1, 0xe0, 0x38, 0x6f, 0xf1, 0xbf, 0x11, 0x48, 0x8f,
		   0xbe, 0xc3, 0x4c, 0x18, 0x55, 0x2a, 0x9d, 0x73, 0x92, 0x19,
		   0xb8, 0x57, 0x6e, 0x52, 0x7c, 0xe8, 0xf3, 0x2d, 0x06, 0xa3,
		   0x68, 0x6c, 0xb5, 0xc8, 0x25, 0x49, 0x9f, 0x20, 0x6c, 0x42,
		   0x50, 0x46, 0x98, 0xd9, 0x36, 0x99, 0xf0, 0xec, 0x04, 0xd7,
		   0x97, 0x95, 0x36, 0x49, 0x66, 0x95, 0x04, 0x4a, 0x10, 0xb3,
		   0x77, 0xb5, 0x13, 0x04, 0x36, 0x49, 0x2d, 0x3f, 0x34, 0xac,
		   0x07, 0xb5, 0x00, 0x6e, 0x4b, 0x81, 0x26, 0xf5, 0xa6, 0x41,
		   0x05, 0x62, 0xe0, 0xcc, 0x9e, 0x82, 0x45, 0x4d, 0x52, 0x03,
		   0x01, 0x11, 0xd3, 0xa5, 0x79, 0xb6, 0x32, 0x4c, 0x0c, 0x7b,
		   0x0f, 0x09, 0xa5, 0x63, 0x21, 0xb1, 0x97, 0x0c, 0x30, 0x2c,
		   0xc7, 0x8d, 0xd4, 0xe0, 0x56, 0x72, 0x26, 0xba, 0x79, 0x87,
		   0x27, 0x17, 0x2a, 0xc5, 0x3b, 0x38, 0x70, 0x80, 0xb5, 0x29,
		   0x2b, 0xa2, 0xab, 0x27, 0x01, 0x18, 0x8a, 0x5c, 0x02, 0x18,
		   0xe7, 0xc6, 0x62, 0x3c, 0x6f, 0x95, 0x2c, 0x80, 0xa3, 0x45,
		   0x1c, 0x7a, 0x19, 0x59, 0x9b, 0x54, 0xa8, 0x72, 0xc2, 0x80,
		   0xf5, 0x7b, 0xbe, 0xe0, 0x01, 0x1c, 0x64, 0x05, 0xc6, 0x2f,
		   0xcb, 0x82, 0x66, 0x13, 0x8a, 0xd5, 0x04, 0xa2, 0x24, 0x8a,
		   0x19, 0xc0, 0x15, 0xa3, 0x05, 0x76, 0x59, 0x49, 0x1a, 0x7b,
		   0xfe, 0x55, 0x78, 0x10, 0x50, 0x49, 0x93, 0x60, 0x59, 0xb5,
		   0x65, 0xbb, 0x6e, 0xc4, 0x9b, 0xf4, 0xa7, 0xcd, 0x88, 0x21,
		   0x52, 0xd6, 0x9c, 0x4a, 0x1a, 0x29, 0x21, 0x3e, 0x5a, 0x06,
		   0x71, 0x87, 0x07, 0xc5, 0x7a, 0x3f, 0x8a, 0x1b, 0xcd, 0xdc,
		   0xf0, 0x4a, 0x0c, 0x32, 0xc1, 0xfc, 0x2c, 0x01, 0x9f, 0x10,
		   0x2f, 0x05, 0xc1, 0x2d, 0xbf, 0x2a, 0x6f, 0xa1, 0x0a, 0x07,
		   0x5b, 0x59, 0x78, 0xb2, 0x12, 0x79, 0x1a, 0xb3, 0xa3, 0x7c,
		   0x47, 0x81, 0x37, 0xca, 0x23, 0xbd, 0xf1, 0x2c, 0xcd, 0xc2,
		   0x26, 0x89, 0x08, 0x6c, 0x7c, 0x86, 0x9b, 0xbc, 0x29, 0x31,
		   0x0b, 0xaa, 0x94, 0x6a, 0x3c, 0x3f, 0xe0, 0x50, 0x3f, 0xe4,
		   0x57, 0x11, 0x00, 0x02, 0xb2, 0x35, 0xc3, 0x9b, 0x85, 0x55,
		   0xa7, 0x9c, 0xcc, 0x4d, 0xa5, 0x37, 0x99, 0xe3, 0x67, 0x47,
		   0x62, 0x03, 0xb1, 0x2a, 0x4b, 0x6d, 0x5f, 0x10, 0xac, 0x63,
		   0x87, 0x50, 0x86, 0x74, 0xa0, 0x63, 0xb1, 0x5f, 0x7f, 0x25,
		   0x36, 0x36, 0xf6, 0x67, 0x64, 0x77, 0x89, 0xad, 0xd7, 0xba,
		   0xed, 0xd9, 0x6c, 0xba, 0x96, 0x39, 0x0e, 0xd3, 0x18, 0x6b,
		   0x65, 0x40, 0xcb, 0xe5, 0x92, 0x93, 0xca, 0x9b, 0x63, 0x4a,
		   0xbc, 0x20, 0x81, 0x29, 0x4d, 0xc3, 0x94, 0x11, 0x59, 0xb5,
		   0xe1, 0xb2, 0x28, 0x02, 0x54, 0xa7, 0x28, 0x86, 0x74, 0x55,
		   0x84, 0x74, 0xb6, 0xea, 0x3b, 0xb3, 0xd1, 0x17, 0x02, 0xa7,
		   0x12, 0xd1, 0x9b, 0x64, 0x40, 0xf6, 0x8c, 0x66, 0x77, 0x5c,
		   0x61, 0xa1, 0x68, 0x65, 0x66, 0x9b, 0xf2, 0x34, 0x0e, 0xa9,
		   0xf1, 0x8c, 0x40, 0xb2, 0xb3, 0xe4, 0x98, 0x8d, 0xf4, 0x7c,
		   0x31, 0x7c, 0xb0, 0x3f, 0x73, 0x6a, 0xaf, 0x2c, 0x1a, 0x5a,
		   0xd4, 0x23, 0x45, 0x2f, 0x8b, 0xc1, 0xb8, 0x1a, 0x74, 0x95,
		   0x3c, 0xb8, 0x61, 0x11, 0x38, 0x7d, 0xf6, 0x7c, 0x47, 0xc7,
		   0x90, 0x26, 0x38, 0x6e, 0x2a, 0xd2, 0xc5, 0xe0, 0x33, 0x25,
		   0x6f, 0x61, 0x74, 0xd0, 0xd7, 0x85, 0xc8, 0x6b, 0x71, 0x5e,
		   0x25, 0x71, 0x56, 0xab, 0x24, 0xae, 0x68, 0x20, 0xaa, 0xb5,
		   0xa0, 0xb4, 0x3c, 0x5d, 0x15, 0xab, 0x25, 0xe2, 0x79, 0x51,
		   0x18, 0x95, 0x1d, 0xd0, 0x36, 0x0b, 0x47, 0xd1, 0x3c, 0xca,
		   0x45, 0xbd, 0x01, 0x6a, 0x0c, 0xb4, 0xc9, 0x7f, 0x3f, 0x66,
		   0xb1, 0xfc, 0x61, 0x2e, 0x97, 0x41, 0x4d, 0xb3, 0x9b, 0x8f,
		   0xd8, 0x2a, 0x4b, 0xc0, 0x9a, 0x7d, 0x92, 0xf3, 0x82, 0x42,
		   0xe9, 0xa7, 0x19, 0x53, 0x3f, 0xa8, 0x40, 0x6b, 0xb9, 0x17,
		   0x84, 0xfc, 0x15, 0x86, 0xcf, 0x52, 0x69, 0x9f, 0xd9, 0x2d,
		   0x53, 0x03, 0x09, 0xfa, 0xc5, 0x0d, 0xe1, 0xfc, 0xc4, 0x5c,
		   0x1c, 0x9a, 0xa0, 0xdb, 0x9b, 0xcc, 0x38, 0x04, 0x34, 0x09,
		   0x6e, 0x9c, 0x92, 0x17, 0xca, 0xd0, 0x36, 0x93, 0x34, 0x73,
		   0x2f, 0x83, 0x3d, 0x47, 0x98, 0x21, 0x16, 0xbc, 0x95, 0x1a,
		   0x1c, 0x3d, 0x35, 0xd0, 0x88, 0x4a, 0x61, 0x42, 0x74, 0xf5,
		   0xc8, 0x70, 0xa6, 0x7f, 0x05, 0xf5, 0x26, 0x25, 0xe7, 0x71,
		   0xc9, 0xd0, 0x86, 0x5d, 0x71, 0xa4, 0xa5, 0x79, 0xa3, 0x0b,
		   0x1c, 0x0f, 0xad, 0x22, 0x83, 0x62, 0xfb, 0x8c, 0x0e, 0xb1,
		   0xca, 0xfd, 0xc5, 0x81, 0xb2, 0xb8, 0x7a, 0xb1, 0xd6, 0xb2,
		   0xc6, 0x47, 0x92, 0x55, 0x98, 0xa7, 0xcf, 0x23, 0x9d, 0x74,
		   0xe5, 0x6a, 0x4a, 0xb6, 0x00, 0x25, 0x59, 0x7e, 0xaf, 0x4c,
		   0x32, 0xe5, 0x7a, 0x41, 0x25, 0xb4, 0x9d, 0x57, 0x31, 0x2d,
		   0x23, 0x82, 0x5a, 0x32, 0xc1, 0x29, 0x28, 0x9b, 0x40, 0xc5,
		   0xeb, 0x68, 0x8a, 0x7a, 0xaf, 0xbf, 0x67, 0x5b, 0x9e, 0x46,
		   0xa6, 0xb4, 0x66, 0x20, 0x68, 0xe3, 0x75, 0x14, 0x5a, 0x2e,
		   0xb9, 0x67, 0x50, 0x5d, 0x33, 0xca, 0x57, 0x19, 0xbd, 0x03,
		   0xf0, 0x00, 0x4b, 0x58, 0xcf, 0x70, 0xf1, 0x0a, 0x4f, 0xeb,
		   0x59, 0xf8, 0xa9, 0x9c, 0x25, 0x71, 0x1c, 0x49, 0x43, 0x0c,
		   0xef, 0x86, 0x53, 0xa4, 0x1c, 0xb7, 0x36, 0x10, 0x0e, 0x00,
		   0x34, 0xb1, 0x74, 0x32, 0x46, 0x75, 0x23, 0x57, 0x80, 0x60,
		   0x28, 0xdc, 0xf8, 0x53, 0x2b, 0xcc, 0xcf, 0x5a, 0x91, 0x01,
		   0x9a, 0x12, 0xa6, 0xe8, 0x69, 0x7f, 0x91, 0x8a, 0x7f, 0x2b,
		   0x50, 0x48, 0x4f, 0x26, 0x5a, 0xa1, 0x39, 0x76, 0x97, 0xe2,
		   0x5e, 0xd8, 0x67, 0x47, 0x23, 0x38, 0x8a, 0x4e, 0x51, 0x96,
		   0xfb, 0x35, 0x95, 0xf5, 0x6c, 0x77, 0xdc, 0x3a, 0x70, 0x34,
		   0x24, 0x53, 0x82, 0x2c, 0x7a, 0xdc, 0x87, 0xb9, 0xd1, 0xb3,
		   0x37, 0x3d, 0xba, 0x42, 0x70, 0x45, 0x5b, 0xe4, 0xc4, 0xb3,
		   0xb3, 0xe6, 0x47, 0x6a, 0x08, 0xa2, 0xf4, 0xe2, 0xcb, 0x81,
		   0xb8, 0x5a, 0xd1, 0x95, 0x3e, 0xcc, 0xb6, 0x7b, 0xa0, 0xf5,
		   0x1d, 0x11, 0x0c, 0xbb, 0x3c, 0x83, 0xc4, 0x80, 0x91, 0x8d,
		   0x10, 0xdb, 0x0d, 0xb9, 0x59, 0x56, 0x2a, 0x09, 0x11, 0x5a,
		   0x29, 0xbe, 0xa5, 0x8b, 0x31, 0xe6, 0x45, 0xb3, 0x13, 0x74,
		   0x96, 0xe4, 0x9b, 0xc4, 0xb6, 0xd2, 0x2b, 0x3d, 0x2c, 0xbb,
		   0x7c, 0xd8, 0x66, 0x52, 0xab, 0x09, 0xd9, 0xf5, 0x22, 0x4d,
		   0x39, 0x46, 0xf5, 0x4a, 0x3a, 0x8e, 0xe3, 0x06, 0xf2, 0x04,
		   0x89, 0x77, 0x54, 0x7f, 0xaf, 0xa4, 0xa5, 0xf9, 0x09, 0x60,
		   0xd1, 0x82, 0xc1, 0xce, 0xc5, 0x9c, 0xb5, 0xe6, 0x64, 0x45,
		   0xcc, 0x9d, 0x38, 0x97, 0xbc, 0x29, 0xd0, 0x74, 0x00, 0x4a,
		   0xcc, 0x51, 0xf3, 0xb3, 0x06, 0xfa, 0x5a, 0xfb, 0x03, 0x4a,
		   0xa6, 0x50, 0xaa, 0x3d, 0xcc, 0x29, 0x25, 0xbc, 0x0c, 0xda,
		   0x35, 0xba, 0x9d, 0x35, 0x43, 0xdb, 0x27, 0xa3, 0x74, 0xf4,
		   0x3d, 0x10, 0xc2, 0x5b, 0xbd, 0x06, 0x36, 0x68, 0xd1, 0x30,
		   0x20, 0x75, 0xcc, 0x5a, 0xa2, 0x6e, 0x19, 0xc2, 0xa0, 0x6f,
		   0xdb, 0xc7, 0x25, 0x8b, 0x53, 0xc6, 0x87, 0xb4, 0x72, 0x71,
		   0xc7, 0xe3, 0xe8, 0xa2, 0x50, 0x44, 0x2b, 0xca, 0x63, 0x2f,
		   0x85, 0x58, 0x7b, 0xdf, 0x02, 0xa9, 0x83, 0xea, 0x7d, 0x2b,
		   0x14, 0x08, 0x5a, 0xe0, 0x76, 0xa7, 0xa9, 0xb8, 0x41, 0xc6,
		   0x8e, 0x0f, 0xd1, 0x69, 0x22, 0xa4, 0x8f, 0x08, 0x29, 0xc8,
		   0x14, 0xa8, 0x17, 0xd8, 0xe2, 0x85, 0xef, 0x08, 0x75, 0x8c,
		   0x44, 0xc5, 0x99, 0xd8, 0x30, 0x2a, 0x7c, 0xb0, 0x0f, 0xc9,
		   0x27, 0xcf, 0x49, 0x9f, 0x05, 0x88, 0x44, 0x7c, 0xf2, 0x8d,
		   0x4e, 0x77, 0x8d, 0x03, 0x9c, 0x3e, 0x6f, 0xc5, 0x19, 0x2a,
		   0xf1, 0x1c, 0x1d, 0xfb, 0x8a, 0xb8, 0x5a, 0x06, 0xf5, 0xa4,
		   0xaa, 0x92, 0x43, 0x74, 0x79, 0xe9, 0xb9, 0x18, 0xc6, 0xcc,
		   0x68, 0x9a, 0x45, 0xc7, 0xe8, 0xa9, 0x48, 0x97, 0x07, 0x5e,
		   0xfb, 0x74, 0x82, 0xb1, 0x44, 0x17, 0x33, 0x95, 0xb0, 0x4a,
		   0x2d, 0xa9, 0x88, 0xc8, 0x11, 0xf8, 0x78, 0xe9, 0xb5, 0x65,
		   0xbe, 0xca, 0x33, 0xf2, 0x64, 0x3c, 0x6a, 0xd0, 0xb2, 0xca,
		   0xf2, 0x31, 0x5b, 0xb0, 0xcb, 0xd3, 0xf1, 0x84, 0xcd, 0x98,
		   0x7f, 0xbd, 0x43, 0x71, 0x78, 0x19, 0x18, 0x3e, 0x05, 0xa6,
		   0x69, 0x57, 0x77, 0x1c, 0x89, 0xaa, 0x83, 0x99, 0x60, 0xc6,
		   0xa8, 0x59, 0x56, 0x00, 0xcc, 0xad, 0x23, 0x55, 0x96, 0x57,
		   0x2c, 0x2f, 0x52, 0x06, 0x0b, 0x3a, 0x0b, 0x08, 0x75, 0xc8,
		   0x4a, 0x02, 0x9e, 0x0b, 0x3a, 0x28, 0x84, 0x74, 0x35, 0xee,
		   0xe5, 0x3e, 0x72, 0xa9, 0x88, 0x06, 0xc3, 0x37, 0x00, 0xa1,
		   0x3e, 0xe8, 0x19, 0x0b, 0xc3, 0x72, 0x2d, 0xc6, 0x34, 0x66,
		   0xea, 0x85, 0x97, 0xde, 0x7a, 0x8e, 0x51, 0x97, 0x2e, 0x99,
		   0x44, 0x1f, 0x83, 0x60, 0xce, 0x88, 0xf9, 0x56, 0x97, 0x44,
		   0x52, 0x9d, 0xe0, 0x83, 0xa7, 0xf7, 0xc5, 0x04, 0x42, 0xa3,
		   0xd7, 0xf5, 0xc4, 0x53, 0x69, 0x61, 0xb8, 0x95, 0xae, 0x90,
		   0xe2, 0x29, 0x1e, 0x0b, 0x99, 0xb3, 0xe1, 0xcc, 0x01, 0xa8,
		   0xa3, 0xa4, 0x5a, 0xba, 0x72, 0xc4, 0x06, 0xba, 0xa3, 0x0e,
		   0xd5, 0xf3, 0x81, 0x86, 0x00, 0xb7, 0x7f, 0x86, 0x5a, 0x5d,
		   0x79, 0x6e, 0xe8, 0x23, 0xc9, 0x8d, 0xd6, 0x0a, 0x28, 0x5b,
		   0x61, 0xe2, 0xe7, 0x9e, 0x84, 0x2c, 0xc6, 0x3e, 0x62, 0x1e,
		   0x3b, 0x36, 0x15, 0x14, 0x44, 0xb8, 0x76, 0x41, 0x65, 0x8c,
		   0x64, 0xc3, 0x71, 0x1b, 0xa7, 0x3e, 0x0a, 0x97, 0xaa, 0x31,
		   0x30, 0x07, 0xc9, 0xc9, 0xc9, 0xbc, 0x77, 0x16, 0xf7, 0x71,
		   0x8f, 0xc2, 0x7c, 0xbf, 0x32, 0xa0, 0x5b, 0xfc, 0x00, 0xd7,
		   0x19, 0x9a, 0x43, 0x39, 0x45, 0x9a, 0x30, 0x79, 0x93, 0x3a,
		   0x5f, 0x4f, 0xc8, 0x3b, 0x15, 0xe1, 0x10, 0x3d, 0x95, 0x1a,
		   0x68, 0x05, 0x1a, 0x98, 0xe7, 0x3e, 0xe3, 0xb9, 0x90, 0xc5,
		   0x4a, 0x07, 0x9d, 0x92, 0x24, 0x5d, 0x8c, 0x8f, 0xac, 0xd4,
		   0xa0, 0x9b, 0x12, 0x89, 0xe4, 0x6c, 0x8e, 0x22, 0x54, 0x11,
		   0x2e, 0x41, 0x7d, 0xb4, 0x54, 0xb9, 0xf3, 0x2c, 0x45, 0xae,
		   0x69, 0xad, 0xe0, 0x26, 0x02, 0x69, 0x93, 0x0a, 0x40, 0xd8,
		   0x15, 0xae, 0x60, 0xc4, 0x75, 0x1b, 0x06, 0x75, 0xc2, 0x1c,
		   0x22, 0x5b, 0x70, 0x00, 0x97, 0x9f, 0x25, 0x6b, 0xbd, 0xf4,
		   0x45, 0x2b, 0xa9, 0x8c, 0xab, 0xd0, 0xb3, 0x30, 0xd5, 0xc3,
		   0x59, 0x2e, 0x58, 0x99, 0x42, 0xe0, 0x9f, 0xbd, 0x87, 0x89,
		   0xb4, 0xc1, 0x17, 0x29, 0x84, 0x42, 0x29, 0xe2, 0xcb, 0x96,
		   0x7c, 0xb1, 0xcb, 0xe5, 0xc8, 0xe9, 0xe0, 0x8b, 0xf6, 0x94,
		   0xb4, 0x52, 0xbc, 0x6c, 0x97, 0x9c, 0x10, 0x01, 0x17, 0x45,
		   0xb0, 0xe6, 0x3e, 0xf5, 0xaa, 0x4e, 0x91, 0x0a, 0x29, 0x3c,
		   0xa5, 0xcd, 0x1d, 0xb1, 0x2c, 0x09, 0x1a, 0x5a, 0xfc, 0x79,
		   0x4e, 0x18, 0x22, 0x43, 0x87, 0x6b, 0x75, 0x8b, 0x1b, 0x0e,
		   0xb6, 0xca, 0x3d, 0xb3, 0x82, 0xb7, 0xc6, 0x51, 0xc4, 0x0f,
		   0x53, 0x2f, 0x6c, 0xe2, 0xc1, 0x8b, 0x9c, 0xa5, 0x6b, 0xb5,
		   0x7e, 0x1f, 0x9a, 0xa5, 0x65, 0xba, 0xc4, 0x68, 0x36, 0x99,
		   0x79, 0x38, 0x63, 0x7a, 0x92, 0xb7, 0x0f, 0xaa, 0xae, 0x43,
		   0xd2, 0x26, 0xe8, 0x72, 0x5d, 0x3f, 0xc7, 0x35, 0xbd, 0x5b,
		   0x25, 0x3f, 0x32, 0xbb, 0x34, 0xa8, 0x0c, 0xe0, 0x1a, 0x23,
		   0x7a, 0x63, 0x5a, 0x4d, 0x79, 0xc2, 0x22, 0xb9, 0x88, 0xef,
		   0x50, 0x75, 0x4d, 0x23, 0x40, 0x45, 0xb9, 0xc3, 0xf3, 0x61,
		   0xa5, 0xc7, 0xc1, 0x5b, 0xd5, 0x57, 0xab, 0xe1, 0xb7, 0xb7,
		   0xe0, 0xdc, 0x0c, 0x95, 0x6b, 0x8a, 0x64, 0x13, 0x8e, 0xdc,
		   0xdb, 0xb2, 0x68, 0x4c, 0x2c, 0x4c, 0x69, 0x4b, 0xf7, 0x06,
		   0x24, 0x7e, 0x01, 0x06, 0x38, 0x35, 0xa6, 0x39, 0xd2, 0x08,
		   0xe1, 0xe3, 0x11, 0xb9, 0xe9, 0x11, 0x63, 0x84, 0x12, 0x90,
		   0x53, 0xc0, 0xc2, 0x85, 0x43, 0x6c, 0x25, 0x87, 0x05, 0x3c,
		   0x1b, 0xc4, 0x83, 0xa0, 0xd8, 0xda, 0x38, 0x7d, 0x4b, 0xbf,
		   0xea, 0xe5, 0x12, 0xea, 0x31, 0xc2, 0xcf, 0x07, 0x03, 0xdd,
		   0x77, 0x1d, 0x4a, 0x06, 0x7e, 0x67, 0x18, 0x9b, 0x79, 0x80,
		   0xca, 0x98, 0xbb, 0xcc, 0xda, 0x38, 0x4c, 0x96, 0xe8, 0x27,
		   0x2e, 0x0c, 0x70, 0x59, 0x58, 0x44, 0xa4, 0x65, 0x49, 0x7f,
		   0xac, 0x3a, 0x58, 0xa7, 0x4f, 0xf2, 0x86, 0x9f, 0xb2, 0x53,
		   0x7e, 0x53, 0xb6, 0x53, 0xdb, 0x58, 0xa4, 0x81, 0x48, 0x67,
		   0xd3, 0x0a, 0x6b, 0xcb, 0x5a, 0x1f, 0x80, 0x46, 0xce, 0x3a,
		   0xc2, 0x0b, 0x11, 0x33, 0x84, 0x87, 0xca, 0x87, 0xa9, 0x8c,
		   0xaf, 0xb3, 0x13, 0x77, 0x3e, 0x81, 0x25, 0x71, 0x72, 0x18,
		   0xdb, 0x4c, 0x23, 0x8e, 0x5c, 0x09, 0xfd, 0x2c, 0xa0, 0x1a,
		   0xf4, 0x9c, 0xb7, 0xe5, 0x81, 0x76, 0x20, 0xab, 0x46, 0x6a,
		   0x57, 0x4d, 0x1a, 0xaa, 0xbd, 0x53, 0xb7, 0xd0, 0xd6, 0x30,
		   0x91, 0xfc, 0x17, 0x9b, 0x54, 0x9f, 0x41, 0xdc, 0x04, 0x8f,
		   0xb8, 0x8f, 0xa5, 0x48, 0x67, 0xb0, 0x63, 0x56, 0xb1, 0xdb,
		   0x8a, 0xc4, 0x41, 0x06, 0x47, 0xd8, 0x4f, 0xe1, 0xb7, 0xbf,
		   0x44, 0x83, 0x89, 0x9e, 0x63, 0x65, 0x5e, 0xf5, 0xbd, 0xe8,
		   0xe8, 0x24, 0xbf, 0x95, 0x66, 0x85, 0x82, 0x29, 0xa6, 0x50,
		   0x58, 0x09, 0xe3, 0x37, 0x67, 0x4a, 0x1f, 0xc0, 0x27, 0x0c,
		   0xa6, 0x9c, 0x2a, 0xbc, 0xb3, 0xca, 0x05, 0x4c, 0x2b, 0x55,
		   0x70, 0x0f, 0xde, 0xb1, 0x69, 0x02, 0x2c, 0x7e, 0x37, 0xe3,
		   0x62, 0xd2, 0xfc, 0xb3, 0x71, 0xa2, 0x76, 0xb8, 0x54, 0x90,
		   0xeb, 0x23, 0x55, 0xd6, 0x42, 0x11, 0x87, 0x52, 0xbc, 0x16,
		   0xf4, 0x74, 0x4f, 0xe9, 0x11, 0x57, 0x12, 0x49, 0x08, 0x11,
		   0x62, 0xa4, 0x11, 0x84, 0xf7, 0x19, 0xa1, 0xe1, 0x30, 0x29,
		   0xc9, 0x53, 0x96, 0x9b, 0xc1, 0x59, 0xfd, 0x55, 0x1b, 0x15,
		   0x5a, 0x7a, 0x68, 0x0a, 0x82, 0x36, 0xe7, 0xdd, 0x9a, 0x1e,
		   0x65, 0x52, 0xdd, 0xb9, 0x53, 0xe5, 0xc8, 0x19, 0x4c, 0xc3,
		   0x38, 0xcc, 0xb8, 0x8b, 0xd4, 0xda, 0x4c, 0x1a, 0x4c, 0xe8,
		   0xaa, 0xe1, 0x59, 0x39, 0x06, 0x81, 0x5e, 0xd5, 0x7d, 0x48,
		   0xce, 0xd9, 0x74, 0xe2, 0xd3, 0x04, 0x22, 0xba, 0x60, 0xed,
		   0xcf, 0x86, 0x11, 0x50, 0xb9, 0x3e, 0x96, 0x85, 0x00, 0xca,
		   0x42, 0x1d, 0x90, 0xdd, 0xd4, 0xa1, 0x3c, 0xb1, 0xee, 0xa9,
		   0x88, 0x00, 0x4b, 0x93, 0x10, 0x3c, 0xfb, 0x0a, 0xee, 0xfd,
		   0x2a, 0x68, 0x6e, 0x01, 0xfa, 0x4a, 0x58, 0xe8, 0xa3, 0x63,
		   0x9c, 0xa8, 0xa1, 0xe3, 0xf9, 0xae, 0x57, 0xe2 },
	.sk_x25519.sk = { 0x35, 0xb8, 0xcc, 0x87, 0x3c, 0x23, 0xdc, 0x62,
			  0xb8, 0xd2, 0x60, 0x16, 0x9a, 0xfa, 0x2f, 0x75,
			  0xab, 0x91, 0x6a, 0x58, 0xd9, 0x74, 0x91, 0x88,
			  0x35, 0xd2, 0x5e, 0x6a, 0x43, 0x50, 0x85, 0xb2 }
};
static const uint8_t ss_exp[] = {
	0x9a, 0x58, 0xf5, 0x6e, 0xb9, 0x70, 0xf9, 0x32, 0x91, 0xaf, 0x0a, 0x93,
	0x83, 0xb9, 0x38, 0xa2, 0xb2, 0xb2, 0xf6, 0xa8, 0x35, 0xa9, 0x35, 0xeb,
	0xfa, 0xed, 0x9e, 0x70, 0x2d, 0x47, 0xa4, 0xb0, 0x97, 0xfb, 0xb9, 0x55,
	0x52, 0x82, 0x55, 0x32, 0xdb, 0x7d, 0xe1, 0x8c, 0xcf, 0x07, 0xb9, 0xbf,
	0x4d, 0x0e, 0xf4, 0xa6, 0x52, 0x32, 0x6a, 0xa8, 0x7b, 0xa1, 0x03, 0x3e,
	0x9d, 0xe0, 0x20, 0x6e, 0xc2, 0xbf, 0x19
};
#endif

static int kyber_kem_double_tester(int check)
{
	struct workspace {
		struct lc_kyber_x25519_pk pk;
		struct lc_kyber_x25519_sk sk;
		struct lc_kyber_x25519_ct ct;
		uint8_t ss1[sizeof(ss_exp)], ss2[sizeof(ss_exp)];
	};
	int ret, rc = 0;
	LC_DECLARE_MEM(ws, struct workspace, sizeof(uint64_t));
	LC_SELFTEST_DRNG_CTX_ON_STACK(selftest_rng);

	ret = lc_kyber_x25519_keypair(&ws->pk, &ws->sk, selftest_rng);

	if (check) {
		rc += lc_compare(ws->pk.pk.pk, kyber_pk_exp.pk.pk,
				 LC_KYBER_PUBLICKEYBYTES, "Kyber pk keygen\n");
		rc += lc_compare(ws->sk.sk.sk, kyber_sk_exp.sk.sk,
				 LC_KYBER_SECRETKEYBYTES, "Kyber sk keygen\n");
		rc += lc_compare(ws->pk.pk_x25519.pk, kyber_pk_exp.pk_x25519.pk,
				 LC_X25519_PUBLICKEYBYTES,
				 "X25519 pk keygen\n");
		rc += lc_compare(ws->sk.sk_x25519.sk, kyber_sk_exp.sk_x25519.sk,
				 LC_X25519_SECRETKEYBYTES,
				 "X25519 sk keygen\n");
	}

	ret |= lc_kyber_x25519_enc_kdf_internal(
		&ws->ct, ws->ss1, sizeof(ws->ss1), &ws->pk, selftest_rng);

	ret += lc_kyber_x25519_dec_kdf(ws->ss2, sizeof(ws->ss2), &ws->ct,
				       &ws->sk);

	if (ret)
		goto out;

	if (check) {
		rc += lc_compare(ws->ss1, ss_exp, sizeof(ss_exp),
				 "Kyber X25519 SS generation\n");
		rc += lc_compare(ws->ss1, ws->ss2, sizeof(ws->ss2),
				 "Kyber X25519 SS comparison\n");
	}

out:
	LC_RELEASE_MEM(ws);
	return ret ? !!ret : rc;
}

static int kyber_kem_double_enc_tester(void)
{
	struct workspace {
		uint8_t ss1[67];
		struct lc_kyber_x25519_ct ct;
	};
	int ret = 0;
	LC_DECLARE_MEM(ws, struct workspace, sizeof(uint64_t));
	LC_SELFTEST_DRNG_CTX_ON_STACK(selftest_rng);

	lc_disable_selftest();

	// Encapsulation
	CKINT(lc_kyber_x25519_enc_kdf_internal(
		&ws->ct, ws->ss1, sizeof(ws->ss1),
		(const struct lc_kyber_x25519_pk *)&kyber_pk_exp,
		selftest_rng));

out:
	LC_RELEASE_MEM(ws);
	return ret;
}

static int kyber_kem_double_dec_tester(void)
{
	struct workspace {
		uint8_t ss1[67];
		struct lc_kyber_x25519_ct ct;
	};
	int ret = 0;
	LC_DECLARE_MEM(ws, struct workspace, sizeof(uint64_t));
	LC_SELFTEST_DRNG_CTX_ON_STACK(selftest_rng);

	lc_disable_selftest();

	// Decapsulation
	CKINT(lc_kyber_x25519_dec_kdf(
		ws->ss1, sizeof(ws->ss1), &ws->ct,
		(const struct lc_kyber_x25519_sk *)&kyber_sk_exp));

out:
	LC_RELEASE_MEM(ws);
	return ret;
}

static int kyber_kem_double_keygen_tester(void)
{
	struct workspace {
		struct lc_kyber_x25519_pk pk;
		struct lc_kyber_x25519_sk sk;
	};
	int ret = 0;
	LC_DECLARE_MEM(ws, struct workspace, sizeof(uint64_t));
	LC_SELFTEST_DRNG_CTX_ON_STACK(selftest_rng);

	lc_disable_selftest();
	CKINT(lc_kyber_x25519_keypair(&ws->pk, &ws->sk, selftest_rng));

out:
	LC_RELEASE_MEM(ws);
	return ret;
}
LC_TEST_FUNC(int, main, int argc, char *argv[])
{
	int ret = 0;

	(void)argv;

#ifdef LC_FIPS140_DEBUG
	/*
	 * Both algos are used for the random number generation as part of
	 * the key generation. Thus we need to enable them for executing the
	 * test.
	 */
	alg_status_set_result(lc_alg_status_result_passed, LC_ALG_STATUS_SHAKE);
	alg_status_set_result(lc_alg_status_result_passed, LC_ALG_STATUS_SHA3);
#endif

	if (argc != 2)
		ret = kyber_kem_double_tester(1);
	else if (argv[1][0] == 'e')
		ret = kyber_kem_double_enc_tester();
	else if (argv[1][0] == 'd')
		ret = kyber_kem_double_dec_tester();
	else if (argv[1][0] == 'k')
		ret = kyber_kem_double_keygen_tester();
	else {
		unsigned int count;

		for (count = 0; count < 50000; count++)
			ret += kyber_kem_double_tester(0);
	}

	/*
	 * Only verify kyber_kem_tester_common because the other tests
	 * disable the self tests.
	 */
	if (argc != 2) {
		ret = test_validate_status(ret, LC_ALG_STATUS_MLKEM_KEYGEN, 1);
		ret = test_validate_status(ret, LC_ALG_STATUS_MLKEM_ENC, 1);
		ret = test_validate_status(ret, LC_ALG_STATUS_MLKEM_DEC, 1);
#ifndef LC_FIPS140_DEBUG
		ret = test_validate_status(ret, LC_ALG_STATUS_X25519_KEYGEN, 0);
		ret = test_validate_status(ret, LC_ALG_STATUS_X25519_SS, 0);
		ret = test_validate_status(ret, LC_ALG_STATUS_SHAKE, 1);
#endif
		ret += test_print_status();
	}

	return ret;
}
