/* api.c API unit tests
 *
 * Copyright (C) 2006-2025 wolfSSL Inc.
 *
 * This file is part of wolfSSL.
 *
 * wolfSSL is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * wolfSSL is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1335, USA
 */

/* For AES-CBC, input lengths can optionally be validated to be a
 * multiple of the block size, by defining WOLFSSL_AES_CBC_LENGTH_CHECKS,
 * also available via the configure option --enable-aescbc-length-checks.
 */


/*----------------------------------------------------------------------------*
 | Includes
 *----------------------------------------------------------------------------*/

#include <tests/unit.h>

#include <wolfssl/wolfcrypt/logging.h>
#include <wolfssl/wolfcrypt/hash.h>

#if defined(WOLFSSL_STATIC_MEMORY)
    #include <wolfssl/wolfcrypt/memory.h>
#endif
#ifdef WOLFSSL_ASNC_CRYPT
    #include <wolfssl/wolfcrypt/async.h>
#endif
#ifdef HAVE_ECC
    #include <wolfssl/wolfcrypt/ecc.h>   /* wc_ecc_fp_free */
    #ifdef WOLFSSL_SM2
        #include <wolfssl/wolfcrypt/sm2.h>
    #endif
#endif
#ifndef NO_ASN
    #include <wolfssl/wolfcrypt/asn_public.h>
#endif

#include <stdlib.h>

#ifdef __linux__
#include <unistd.h>
#include <sys/wait.h>
#endif

#include <wolfssl/ssl.h>  /* compatibility layer */
#include <wolfssl/error-ssl.h>

#include <wolfssl/test.h>
#include <tests/utils.h>
#include <testsuite/utils.h>

/* for testing compatibility layer callbacks */
#include "examples/server/server.h"

#ifndef NO_SIG_WRAPPER
    #include <wolfssl/wolfcrypt/signature.h>
#endif

#ifdef WOLFSSL_SMALL_CERT_VERIFY
    #include <wolfssl/wolfcrypt/asn.h>
#endif

#ifndef NO_DSA
    #include <wolfssl/wolfcrypt/dsa.h>
#endif

#if defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL) || \
    defined(OPENSSL_ALL)
    #include <wolfssl/openssl/ssl.h>
    #ifndef NO_ASN
        /* for ASN_COMMON_NAME DN_tags enum */
        #include <wolfssl/wolfcrypt/asn.h>
    #endif
    #ifdef HAVE_OCSP
        #include <wolfssl/openssl/ocsp.h>
    #endif
#endif
#ifdef OPENSSL_EXTRA
    #include <wolfssl/openssl/cmac.h>
    #include <wolfssl/openssl/x509v3.h>
    #include <wolfssl/openssl/asn1.h>
    #include <wolfssl/openssl/crypto.h>
    #include <wolfssl/openssl/pkcs12.h>
    #include <wolfssl/openssl/evp.h>
    #include <wolfssl/openssl/dh.h>
    #include <wolfssl/openssl/bn.h>
    #include <wolfssl/openssl/buffer.h>
    #include <wolfssl/openssl/pem.h>
    #include <wolfssl/openssl/ec.h>
    #include <wolfssl/openssl/ecdh.h>
    #include <wolfssl/openssl/engine.h>
    #include <wolfssl/openssl/hmac.h>
    #include <wolfssl/openssl/objects.h>
    #include <wolfssl/openssl/rand.h>
    #include <wolfssl/openssl/modes.h>
    #include <wolfssl/openssl/fips_rand.h>
    #include <wolfssl/openssl/kdf.h>
    #include <wolfssl/openssl/x509_vfy.h>
#ifdef OPENSSL_ALL
    #include <wolfssl/openssl/txt_db.h>
    #include <wolfssl/openssl/lhash.h>
#endif
#ifndef NO_AES
    #include <wolfssl/openssl/aes.h>
#endif
#ifndef NO_DES3
    #include <wolfssl/openssl/des.h>
#endif
#ifndef NO_RC4
    #include <wolfssl/openssl/rc4.h>
#endif
#ifdef HAVE_ECC
    #include <wolfssl/openssl/ecdsa.h>
#endif
#ifdef HAVE_CURVE25519
    #include <wolfssl/openssl/ec25519.h>
#endif
#ifdef HAVE_ED25519
    #include <wolfssl/openssl/ed25519.h>
#endif
#ifdef HAVE_CURVE448
    #include <wolfssl/openssl/ec448.h>
#endif
#ifdef HAVE_ED448
    #include <wolfssl/openssl/ed448.h>
#endif
#endif /* OPENSSL_EXTRA */

#if defined(OPENSSL_EXTRA) && defined(WOLFCRYPT_HAVE_SRP) && \
    !defined(NO_SHA256) && !defined(RC_NO_RNG)
        #include <wolfssl/wolfcrypt/srp.h>
#endif

#if (defined(SESSION_CERTS) && defined(TEST_PEER_CERT_CHAIN)) || \
    defined(HAVE_SESSION_TICKET) || (defined(OPENSSL_EXTRA) && \
    defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_CERT_GEN)) || \
    defined(WOLFSSL_TEST_STATIC_BUILD) || defined(WOLFSSL_DTLS) || \
    defined(HAVE_ECH) || defined(HAVE_EX_DATA) || !defined(NO_SESSION_CACHE) \
    || !defined(WOLFSSL_NO_TLS12) || defined(WOLFSSL_TLS13)
    /* for testing SSL_get_peer_cert_chain, or SESSION_TICKET_HINT_DEFAULT,
     * for setting authKeyIdSrc in WOLFSSL_X509, or testing DTLS sequence
     * number tracking */
    #include "wolfssl/internal.h"
#endif

/* include misc.c here regardless of NO_INLINE, because misc.c implementations
 * have default (hidden) visibility, and in the absence of visibility, it's
 * benign to mask out the library implementation.
 */
#define WOLFSSL_MISC_INCLUDED
#include <wolfcrypt/src/misc.c>

#include <tests/api/api.h>
#include <tests/api/api_decl.h>

/* Gather test declarations to include them in the testCases array */
#include <tests/api/test_md2.h>
#include <tests/api/test_md4.h>
#include <tests/api/test_md5.h>
#include <tests/api/test_sha.h>
#include <tests/api/test_sha256.h>
#include <tests/api/test_sha512.h>
#include <tests/api/test_sha3.h>
#include <tests/api/test_blake2.h>
#include <tests/api/test_sm3.h>
#include <tests/api/test_ripemd.h>
#include <tests/api/test_hash.h>
#include <tests/api/test_hmac.h>
#include <tests/api/test_cmac.h>
#include <tests/api/test_des3.h>
#include <tests/api/test_chacha.h>
#include <tests/api/test_poly1305.h>
#include <tests/api/test_chacha20_poly1305.h>
#include <tests/api/test_camellia.h>
#include <tests/api/test_arc4.h>
#include <tests/api/test_rc2.h>
#include <tests/api/test_aes.h>
#include <tests/api/test_ascon.h>
#include <tests/api/test_sm4.h>
#include <tests/api/test_wc_encrypt.h>
#include <tests/api/test_random.h>
#include <tests/api/test_wolfmath.h>
#include <tests/api/test_rsa.h>
#include <tests/api/test_dsa.h>
#include <tests/api/test_dh.h>
#include <tests/api/test_ecc.h>
#include <tests/api/test_sm2.h>
#include <tests/api/test_curve25519.h>
#include <tests/api/test_ed25519.h>
#include <tests/api/test_curve448.h>
#include <tests/api/test_ed448.h>
#include <tests/api/test_mlkem.h>
#include <tests/api/test_mldsa.h>
#include <tests/api/test_signature.h>
#include <tests/api/test_dtls.h>
#include <tests/api/test_ocsp.h>
#include <tests/api/test_evp.h>
#include <tests/api/test_tls_ext.h>
#include <tests/api/test_tls.h>
#include <tests/api/test_x509.h>
#include <tests/api/test_asn.h>
#include <tests/api/test_pkcs7.h>
#include <tests/api/test_pkcs12.h>
#include <tests/api/test_ossl_asn1.h>
#include <tests/api/test_ossl_bn.h>
#include <tests/api/test_ossl_bio.h>
#include <tests/api/test_ossl_dgst.h>
#include <tests/api/test_ossl_mac.h>
#include <tests/api/test_ossl_cipher.h>
#include <tests/api/test_ossl_rsa.h>
#include <tests/api/test_ossl_dh.h>
#include <tests/api/test_ossl_ec.h>
#include <tests/api/test_ossl_ecx.h>
#include <tests/api/test_ossl_dsa.h>
#include <tests/api/test_ossl_sk.h>
#include <tests/api/test_tls13.h>

#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    !defined(NO_RSA) && \
    !defined(NO_WOLFSSL_SERVER) && !defined(NO_WOLFSSL_CLIENT) && \
    !defined(WOLFSSL_TIRTOS)
    #define HAVE_SSL_MEMIO_TESTS_DEPENDENCIES
#endif

#if !defined(NO_RSA) && !defined(NO_SHA) && !defined(NO_FILESYSTEM) && \
    !defined(NO_CERTS) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(WOLFSSL_NO_CLIENT_AUTH))
    #define HAVE_CERT_CHAIN_VALIDATION
#endif

#if defined(WOLFSSL_STATIC_MEMORY) && !defined(WOLFCRYPT_ONLY)
    #if (defined(HAVE_ECC) && !defined(ALT_ECC_SIZE)) || defined(SESSION_CERTS)
        #ifdef OPENSSL_EXTRA
            #define TEST_TLS_STATIC_MEMSZ (400000)
        #else
            #define TEST_TLS_STATIC_MEMSZ (320000)
        #endif
    #else
            #define TEST_TLS_STATIC_MEMSZ (80000)
    #endif
#endif

#ifdef WOLFSSL_DUMP_MEMIO_STREAM
const char* currentTestName;
char tmpDirName[16];
int tmpDirNameSet = 0;
#endif

/*----------------------------------------------------------------------------*
 | Constants
 *----------------------------------------------------------------------------*/

#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    (!defined(NO_WOLFSSL_SERVER) || !defined(NO_WOLFSSL_CLIENT))
    static const char* bogusFile  =
    #ifdef _WIN32
        "NUL"
    #else
        "/dev/null"
    #endif
    ;
#endif /* !NO_FILESYSTEM && !NO_CERTS && (!NO_WOLFSSL_SERVER || !NO_WOLFSSL_CLIENT) */

enum {
    TESTING_RSA = 1,
    TESTING_ECC = 2
};

#ifdef WOLFSSL_QNX_CAAM
#include <wolfssl/wolfcrypt/port/caam/wolfcaam.h>
int testDevId = WOLFSSL_CAAM_DEVID;
#else
int testDevId = INVALID_DEVID;
#endif

#ifdef USE_WINDOWS_API
    #define MESSAGE_TYPE_CAST char*
#else
    #define MESSAGE_TYPE_CAST void*
#endif

/*----------------------------------------------------------------------------*
 | BIO with fixed read/write size
 *----------------------------------------------------------------------------*/

#if defined(OPENSSL_EXTRA) && !defined(NO_BIO)

static int wolfssl_bio_s_fixed_mem_write(WOLFSSL_BIO* bio, const char* data,
    int len)
{
    if ((bio == NULL) || (bio->ptr.mem_buf_data == NULL) || (data == NULL)) {
        len = 0;
    }
    else {
        if (bio->wrSz - bio->wrIdx < len) {
            len = bio->wrSz - bio->wrIdx;
        }
        XMEMCPY(bio->ptr.mem_buf_data + bio->wrIdx, data, (size_t)len);
        bio->wrIdx += len;
    }

    return len;
}

static int wolfssl_bio_s_fixed_mem_read(WOLFSSL_BIO* bio, char* data, int len)
{
    if ((bio == NULL) || (bio->ptr.mem_buf_data == NULL) || (data == NULL)) {
        len = 0;
    }
    else {
        if (bio->wrSz - bio->rdIdx < len) {
            len = bio->wrSz - bio->rdIdx;
        }
        XMEMCPY(data, bio->ptr.mem_buf_data + bio->rdIdx, (size_t)len);
        bio->rdIdx += len;
    }

    return len;
}

WOLFSSL_BIO_METHOD* wolfSSL_BIO_s_fixed_mem(void)
{
    static WOLFSSL_BIO_METHOD meth;

    meth.type = WOLFSSL_BIO_BIO;
    XMEMCPY(meth.name, "Fixed Memory Size", 18);
    meth.writeCb = wolfssl_bio_s_fixed_mem_write;
    meth.readCb = wolfssl_bio_s_fixed_mem_read;

    return &meth;
}

#endif

/*----------------------------------------------------------------------------*
 | Setup
 *----------------------------------------------------------------------------*/

static int test_wolfSSL_Init(void)
{
    EXPECT_DECLS;
    ExpectIntEQ(wolfSSL_Init(), WOLFSSL_SUCCESS);
    return EXPECT_RESULT();
}


static int test_wolfSSL_Cleanup(void)
{
    EXPECT_DECLS;
    ExpectIntEQ(wolfSSL_Cleanup(), WOLFSSL_SUCCESS);
    return EXPECT_RESULT();
}


/*  Initialize the wolfCrypt state.
 *  POST: 0 success.
 */
static int test_wolfCrypt_Init(void)
{
    EXPECT_DECLS;
    ExpectIntEQ(wolfCrypt_Init(), 0);
    return EXPECT_RESULT();

} /* END test_wolfCrypt_Init */

static int test_wolfCrypt_Cleanup(void)
{
    EXPECT_DECLS;
    ExpectIntEQ(wolfCrypt_Cleanup(), 0);
    return EXPECT_RESULT();
}


#ifdef WOLFSSL_STATIC_MEMORY
    #define TEST_LSM_STATIC_SIZE 440000
    /* Create new bucket list, using the default list, adding
     * one dang large buffer size. */
    #define TEST_LSM_DEF_BUCKETS (WOLFMEM_DEF_BUCKETS+1)
    #define TEST_LSM_BUCKETS WOLFMEM_BUCKETS,(LARGEST_MEM_BUCKET*2)
    #define TEST_LSM_DIST WOLFMEM_DIST,1
#endif

static int test_wc_LoadStaticMemory_ex(void)
{
    EXPECT_DECLS;
#ifdef WOLFSSL_STATIC_MEMORY
    byte staticMemory[TEST_LSM_STATIC_SIZE];
    word32 sizeList[TEST_LSM_DEF_BUCKETS] = { TEST_LSM_BUCKETS };
    word32 distList[TEST_LSM_DEF_BUCKETS] = { TEST_LSM_DIST };
    WOLFSSL_HEAP_HINT* heap;

    /* For this test, the size and dist lists will be the ones configured
     * for the build, or default. The value of WOLFMEM_DEF_BUCKETS is 9,
     * so these lists are 10 long. For most tests, the value of
     * WOLFMEM_DEF_BUCKETS is used. There's a test case where one is added
     * to that, to make sure the list size is larger than
     * WOLFMEM_MAX_BUCKETS. */

    /* Pass in zero everything. */
    ExpectIntEQ(wc_LoadStaticMemory_ex(NULL, 0, NULL, NULL, NULL, 0, 0, 0),
            WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    /* Set the heap pointer to NULL. */
    ExpectIntEQ(wc_LoadStaticMemory_ex(NULL,
                WOLFMEM_DEF_BUCKETS, sizeList, distList,
                staticMemory, (word32)sizeof(staticMemory),
                0, 1),
            WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    /* Set other pointer values to NULL one at a time. */
    heap = NULL;
    ExpectIntEQ(wc_LoadStaticMemory_ex(&heap,
                WOLFMEM_DEF_BUCKETS, NULL, distList,
                staticMemory, (word32)sizeof(staticMemory),
                0, 1),
            WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    heap = NULL;
    ExpectIntEQ(wc_LoadStaticMemory_ex(&heap,
                WOLFMEM_DEF_BUCKETS, sizeList, NULL,
                staticMemory, (word32)sizeof(staticMemory),
                0, 1),
            WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    heap = NULL;
    ExpectIntEQ(wc_LoadStaticMemory_ex(&heap,
                WOLFMEM_DEF_BUCKETS, sizeList, distList,
                NULL, (word32)sizeof(staticMemory),
                0, 1),
            WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    /* Set the size of the static buffer to 0. */
    heap = NULL;
    ExpectIntEQ(wc_LoadStaticMemory_ex(&heap,
                WOLFMEM_DEF_BUCKETS, sizeList, distList,
                staticMemory, 0,
                0, 1),
            WC_NO_ERR_TRACE(BUFFER_E));

    /* Set the size of the static buffer to one less than minimum allowed. */
    heap = NULL;
    ExpectIntEQ(wc_LoadStaticMemory_ex(&heap,
                WOLFMEM_DEF_BUCKETS, sizeList, distList,
                staticMemory,
                (word32)(sizeof(WOLFSSL_HEAP) + sizeof(WOLFSSL_HEAP_HINT)) - 1,
                0, 1),
            WC_NO_ERR_TRACE(BUFFER_E));

    /* Set the size of the static buffer to exactly the minimum size. */
    heap = NULL;
    ExpectIntEQ(wc_LoadStaticMemory_ex(&heap,
                WOLFMEM_DEF_BUCKETS, sizeList, distList,
                staticMemory,
                (word32)(sizeof(WOLFSSL_HEAP) + sizeof(WOLFSSL_HEAP_HINT)),
                0, 1),
            0);
    wc_UnloadStaticMemory(heap);

    /* Use more buckets than able. Success case. */
    heap = NULL;
    ExpectIntEQ(wc_LoadStaticMemory_ex(&heap,
                WOLFMEM_DEF_BUCKETS*2, sizeList, distList,
                staticMemory, (word32)sizeof(staticMemory),
                0, 1),
            0);
    wc_UnloadStaticMemory(heap);

    /* Success case. */
    heap = NULL;
    ExpectIntEQ(wc_LoadStaticMemory_ex(&heap,
                WOLFMEM_DEF_BUCKETS, sizeList, distList,
                staticMemory, (word32)sizeof(staticMemory),
                0, 1),
            0);
    wc_UnloadStaticMemory(heap);
#endif /* WOLFSSL_STATIC_MEMORY */
    return EXPECT_RESULT();
}


static int test_wc_LoadStaticMemory_CTX(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_STATIC_MEMORY) && !defined(NO_WOLFSSL_CLIENT)
    byte staticMemory[TEST_LSM_STATIC_SIZE];
    word32 sizeList[TEST_LSM_DEF_BUCKETS] = { TEST_LSM_BUCKETS };
    word32 distList[TEST_LSM_DEF_BUCKETS] = { TEST_LSM_DIST };
    WOLFSSL_HEAP_HINT* heap;
    WOLFSSL_CTX *ctx1 = NULL, *ctx2 = NULL;


    /* Set the size of the static buffer to exactly the minimum size. */
    heap = NULL;
    ExpectIntEQ(wc_LoadStaticMemory_ex(&heap,
                WOLFMEM_DEF_BUCKETS, sizeList, distList,
                staticMemory, sizeof(staticMemory), 0, 1),
            0);

    /* Creating two WOLFSSL_CTX objects from the same heap hint and free'ing
     * them should not cause issues. */
    ExpectNotNull((ctx1 = wolfSSL_CTX_new_ex(wolfSSLv23_client_method_ex(heap),
        heap)));
    wolfSSL_CTX_free(ctx1);
    ExpectNotNull((ctx2 = wolfSSL_CTX_new_ex(wolfSSLv23_client_method_ex(heap),
        heap)));
    wolfSSL_CTX_free(ctx2);

    /* two CTX's at once */
    ExpectNotNull((ctx1 = wolfSSL_CTX_new_ex(wolfSSLv23_client_method_ex(heap),
        heap)));
    ExpectNotNull((ctx2 = wolfSSL_CTX_new_ex(wolfSSLv23_client_method_ex(heap),
        heap)));
    wolfSSL_CTX_free(ctx1);
    wolfSSL_CTX_free(ctx2);

    wc_UnloadStaticMemory(heap);
#endif /* WOLFSSL_STATIC_MEMORY */
    return EXPECT_RESULT();
}


/*----------------------------------------------------------------------------*
 | Platform dependent function test
 *----------------------------------------------------------------------------*/
static int test_fileAccess(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_TEST_PLATFORMDEPEND) && !defined(NO_FILESYSTEM)
    const char *fname[] = {
        svrCertFile, svrKeyFile, caCertFile,
        eccCertFile, eccKeyFile, eccRsaCertFile,
        cliCertFile, cliCertDerFile, cliKeyFile,
        dhParamFile,
        cliEccKeyFile, cliEccCertFile, caEccCertFile, edCertFile, edKeyFile,
        cliEdCertFile, cliEdKeyFile, caEdCertFile,
        NULL
    };
    const char derfile[] = "./certs/server-cert.der";
    XFILE f = XBADFILE;
    size_t sz;
    byte *buff = NULL;
    int i;

    ExpectTrue(XFOPEN("badfilename", "rb") == XBADFILE);
    for (i=0; EXPECT_SUCCESS() && fname[i] != NULL ; i++) {
        ExpectTrue((f = XFOPEN(fname[i], "rb")) != XBADFILE);
        XFCLOSE(f);
    }

    ExpectTrue((f = XFOPEN(derfile, "rb")) != XBADFILE);
    ExpectTrue(XFSEEK(f, 0, XSEEK_END) == 0);
    ExpectIntGE(sz = (size_t) XFTELL(f), sizeof_server_cert_der_2048);
    ExpectTrue(XFSEEK(f, 0, XSEEK_SET) == 0);
    ExpectTrue((buff = (byte*)XMALLOC(sz, NULL, DYNAMIC_TYPE_FILE)) != NULL);
    ExpectTrue(XFREAD(buff, 1, sz, f) == sz);
    ExpectIntEQ(XMEMCMP(server_cert_der_2048, buff, sz), 0);
    XFREE(buff, NULL, DYNAMIC_TYPE_FILE);
    XFCLOSE(f);
#endif
    return EXPECT_RESULT();
}
static int test_wc_FreeCertList(void)
{
    EXPECT_DECLS;
#if defined(HAVE_PKCS12) && !defined(NO_ASN) && \
!defined(NO_PWDBASED) && !defined(NO_HMAC) && !defined(NO_CERTS) && \
defined(USE_CERT_BUFFERS_2048)
    WC_DerCertList* list = NULL;
    void* heap = NULL;
    /* Test freeing a list with a single node */
    list = (WC_DerCertList*)XMALLOC(sizeof(WC_DerCertList),
            heap, DYNAMIC_TYPE_PKCS);
    ExpectNotNull(list);
    if (list != NULL) {
        list->buffer = (byte*)XMALLOC(10, heap, DYNAMIC_TYPE_PKCS);
        ExpectNotNull(list->buffer);
        if (list->buffer != NULL) {
            list->bufferSz = 10;
            list->next = NULL;
            wc_FreeCertList(list, heap);
        }
        else {
            XFREE(list, heap, DYNAMIC_TYPE_PKCS);
            list = NULL;
        }
    }
#endif
    return EXPECT_RESULT();
}

/*----------------------------------------------------------------------------*
 | Method Allocators
 *----------------------------------------------------------------------------*/

static int test_wolfSSL_Method_Allocators(void)
{
    EXPECT_DECLS;

    #define TEST_METHOD_ALLOCATOR(allocator, condition) \
        do {                                            \
            WOLFSSL_METHOD *method = NULL;              \
            condition(method = allocator());            \
            XFREE(method, 0, DYNAMIC_TYPE_METHOD);      \
        } while (0)

    #define TEST_VALID_METHOD_ALLOCATOR(a) \
            TEST_METHOD_ALLOCATOR(a, ExpectNotNull)

    #define TEST_INVALID_METHOD_ALLOCATOR(a) \
            TEST_METHOD_ALLOCATOR(a, ExpectNull)

#ifndef NO_TLS
#ifndef NO_OLD_TLS
    #ifdef WOLFSSL_ALLOW_SSLV3
        #ifndef NO_WOLFSSL_SERVER
        TEST_VALID_METHOD_ALLOCATOR(wolfSSLv3_server_method);
        #endif
        #ifndef NO_WOLFSSL_CLIENT
        TEST_VALID_METHOD_ALLOCATOR(wolfSSLv3_client_method);
        #endif
    #endif
    #ifdef WOLFSSL_ALLOW_TLSV10
        #ifndef NO_WOLFSSL_SERVER
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_server_method);
        #endif
        #ifndef NO_WOLFSSL_CLIENT
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_client_method);
        #endif
    #endif
    #ifndef NO_WOLFSSL_SERVER
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_1_server_method);
    #endif
    #ifndef NO_WOLFSSL_CLIENT
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_1_client_method);
    #endif
#endif /* !NO_OLD_TLS */

#ifndef WOLFSSL_NO_TLS12
    #ifndef NO_WOLFSSL_SERVER
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_2_server_method);
    #endif
    #ifndef NO_WOLFSSL_CLIENT
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_2_client_method);
    #endif
#endif /* !WOLFSSL_NO_TLS12 */

#ifdef WOLFSSL_TLS13
    #ifndef NO_WOLFSSL_SERVER
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_3_server_method);
    #endif
    #ifndef NO_WOLFSSL_CLIENT
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_3_client_method);
    #endif
#endif /* WOLFSSL_TLS13 */

#ifndef NO_WOLFSSL_SERVER
    TEST_VALID_METHOD_ALLOCATOR(wolfSSLv23_server_method);
#endif
#ifndef NO_WOLFSSL_CLIENT
    TEST_VALID_METHOD_ALLOCATOR(wolfSSLv23_client_method);
#endif

#ifdef WOLFSSL_DTLS
    #ifndef NO_OLD_TLS
        #ifndef NO_WOLFSSL_SERVER
        TEST_VALID_METHOD_ALLOCATOR(wolfDTLSv1_server_method);
        #endif
        #ifndef NO_WOLFSSL_CLIENT
        TEST_VALID_METHOD_ALLOCATOR(wolfDTLSv1_client_method);
        #endif
    #endif
    #ifndef WOLFSSL_NO_TLS12
        #ifndef NO_WOLFSSL_SERVER
        TEST_VALID_METHOD_ALLOCATOR(wolfDTLSv1_2_server_method);
        #endif
        #ifndef NO_WOLFSSL_CLIENT
        TEST_VALID_METHOD_ALLOCATOR(wolfDTLSv1_2_client_method);
        #endif
    #endif
#endif /* WOLFSSL_DTLS */

#if !defined(NO_OLD_TLS) && defined(OPENSSL_EXTRA)
    /* Stubs */
    #ifndef NO_WOLFSSL_SERVER
        TEST_INVALID_METHOD_ALLOCATOR(wolfSSLv2_server_method);
    #endif
    #ifndef NO_WOLFSSL_CLIENT
        TEST_INVALID_METHOD_ALLOCATOR(wolfSSLv2_client_method);
    #endif
#endif

    /* Test Either Method (client or server) */
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    TEST_VALID_METHOD_ALLOCATOR(wolfSSLv23_method);
    #ifndef NO_OLD_TLS
        #ifdef WOLFSSL_ALLOW_TLSV10
            TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_method);
        #endif
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_1_method);
    #endif /* !NO_OLD_TLS */
    #ifndef WOLFSSL_NO_TLS12
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_2_method);
    #endif /* !WOLFSSL_NO_TLS12 */
    #ifdef WOLFSSL_TLS13
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_3_method);
    #endif /* WOLFSSL_TLS13 */
    #ifdef WOLFSSL_DTLS
        TEST_VALID_METHOD_ALLOCATOR(wolfDTLS_method);
        #ifndef NO_OLD_TLS
            TEST_VALID_METHOD_ALLOCATOR(wolfDTLSv1_method);
        #endif /* !NO_OLD_TLS */
        #ifndef WOLFSSL_NO_TLS12
            TEST_VALID_METHOD_ALLOCATOR(wolfDTLSv1_2_method);
        #endif /* !WOLFSSL_NO_TLS12 */
        #ifdef WOLFSSL_DTLS13
            TEST_VALID_METHOD_ALLOCATOR(wolfDTLSv1_3_method);
        #endif /* WOLFSSL_DTLS13 */
    #endif /* WOLFSSL_DTLS */
#endif /* OPENSSL_EXTRA || WOLFSSL_EITHER_SIDE */
#endif /* !NO_TLS */
    return EXPECT_RESULT();
}

#if defined(WOLFSSL_DUAL_ALG_CERTS) && !defined(NO_FILESYSTEM)
/*----------------------------------------------------------------------------*
 | Dual algorithm Certificate Tests
 *----------------------------------------------------------------------------*/
#define LARGE_TEMP_SZ 4096

/* To better understand this, please see the X9.146 example in wolfssl-examples
 * repo. */
static int do_dual_alg_root_certgen(byte **out, char *caKeyFile,
                                    char *sapkiFile, char *altPrivFile)
{
    EXPECT_DECLS;
    FILE* file = NULL;
    Cert newCert;
    DecodedCert preTBS;

    byte caKeyBuf[LARGE_TEMP_SZ];
    word32 caKeySz = LARGE_TEMP_SZ;
    byte sapkiBuf[LARGE_TEMP_SZ];
    word32 sapkiSz = LARGE_TEMP_SZ;
    byte altPrivBuf[LARGE_TEMP_SZ];
    word32 altPrivSz = LARGE_TEMP_SZ;
    byte altSigAlgBuf[LARGE_TEMP_SZ];
    word32 altSigAlgSz = LARGE_TEMP_SZ;
    byte scratchBuf[LARGE_TEMP_SZ];
    word32 scratchSz = LARGE_TEMP_SZ;
    byte preTbsBuf[LARGE_TEMP_SZ];
    word32 preTbsSz = LARGE_TEMP_SZ;
    byte altSigValBuf[LARGE_TEMP_SZ];
    word32 altSigValSz = LARGE_TEMP_SZ;
    byte *outBuf = NULL;
    word32 outSz = LARGE_TEMP_SZ;
    WC_RNG rng;
    RsaKey caKey;
    ecc_key altCaKey;
    word32 idx = 0;

    XMEMSET(&rng, 0, sizeof(WC_RNG));
    XMEMSET(&caKey, 0, sizeof(RsaKey));
    XMEMSET(&altCaKey, 0, sizeof(ecc_key));

    ExpectNotNull(outBuf = (byte*)XMALLOC(outSz, NULL,
                  DYNAMIC_TYPE_TMP_BUFFER));
    ExpectIntEQ(wc_InitRng(&rng), 0);
    XMEMSET(caKeyBuf, 0, caKeySz);
    ExpectNotNull(file = fopen(caKeyFile, "rb"));
    ExpectIntGT(caKeySz = (word32)fread(caKeyBuf, 1, caKeySz, file), 0);
    if (file) {
        fclose(file);
        file = NULL;
    }
    ExpectIntEQ(wc_InitRsaKey_ex(&caKey, NULL, INVALID_DEVID), 0);
    idx = 0;
    ExpectIntEQ(wc_RsaPrivateKeyDecode(caKeyBuf, &idx, &caKey, caKeySz),
                0);
    XMEMSET(sapkiBuf, 0, sapkiSz);
    ExpectNotNull(file = fopen(sapkiFile, "rb"));
    ExpectIntGT(sapkiSz = (word32)fread(sapkiBuf, 1, sapkiSz, file), 0);
    if (file) {
        fclose(file);
        file = NULL;
    }
    XMEMSET(altPrivBuf, 0, altPrivSz);
    ExpectNotNull(file = fopen(altPrivFile, "rb"));
    ExpectIntGT(altPrivSz = (word32)fread(altPrivBuf, 1, altPrivSz, file), 0);
    if (file) {
        fclose(file);
        file = NULL;
    }
    wc_ecc_init(&altCaKey);
    idx = 0;
    ExpectIntEQ(wc_EccPrivateKeyDecode(altPrivBuf, &idx, &altCaKey,
                                       (word32)altPrivSz), 0);
    XMEMSET(altSigAlgBuf, 0, altSigAlgSz);
    ExpectIntGT(altSigAlgSz = SetAlgoID(CTC_SHA256wECDSA, altSigAlgBuf,
                                         oidSigType, 0), 0);
    wc_InitCert(&newCert);
    strncpy(newCert.subject.country, "US", CTC_NAME_SIZE);
    strncpy(newCert.subject.state, "MT", CTC_NAME_SIZE);
    strncpy(newCert.subject.locality, "Bozeman", CTC_NAME_SIZE);
    strncpy(newCert.subject.org, "wolfSSL", CTC_NAME_SIZE);
    strncpy(newCert.subject.unit, "Engineering", CTC_NAME_SIZE);
    strncpy(newCert.subject.commonName, "www.wolfssl.com", CTC_NAME_SIZE);
    strncpy(newCert.subject.email, "root@wolfssl.com", CTC_NAME_SIZE);
    strncpy((char*)newCert.beforeDate, "\x18\x0f""20250101000000Z",
        CTC_DATE_SIZE);
    newCert.beforeDateSz = 17;
    strncpy((char*)newCert.afterDate, "\x18\x0f""20493112115959Z",
        CTC_DATE_SIZE);
    newCert.afterDateSz = 17;
    newCert.sigType = CTC_SHA256wRSA;
    newCert.isCA    = 1;

    ExpectIntEQ(wc_SetCustomExtension(&newCert, 0, "1.2.3.4.5",
                (const byte *)"This is NOT a critical extension", 32), 0);
    ExpectIntEQ(wc_SetCustomExtension(&newCert, 0, "2.5.29.72", sapkiBuf,
                sapkiSz), 0);
    ExpectIntEQ(wc_SetCustomExtension(&newCert, 0, "2.5.29.73", altSigAlgBuf,
                                altSigAlgSz), 0);

    XMEMSET(scratchBuf, 0, scratchSz);
    ExpectIntGT(scratchSz = wc_MakeSelfCert(&newCert, scratchBuf, scratchSz,
                &caKey, &rng), 0);

    wc_InitDecodedCert(&preTBS, scratchBuf, scratchSz, 0);
    ExpectIntEQ(wc_ParseCert(&preTBS, CERT_TYPE, NO_VERIFY, NULL), 0);

    XMEMSET(preTbsBuf, 0, preTbsSz);
    ExpectIntGT(preTbsSz = wc_GeneratePreTBS(&preTBS, preTbsBuf, preTbsSz), 0);
    XMEMSET(altSigValBuf, 0, altSigValSz);
    ExpectIntGT(altSigValSz = wc_MakeSigWithBitStr(altSigValBuf, altSigValSz,
                CTC_SHA256wECDSA, preTbsBuf, preTbsSz, ECC_TYPE, &altCaKey,
                &rng), 0);
    ExpectIntEQ(wc_SetCustomExtension(&newCert, 0, "2.5.29.74", altSigValBuf,
                altSigValSz), 0);

    /* Finally, generate the new certificate. */
    if (outBuf != NULL) {
        XMEMSET(outBuf, 0, outSz);
    }
    ExpectIntGT(outSz = wc_MakeSelfCert(&newCert, outBuf, outSz, &caKey, &rng),
                0);
    *out = outBuf;

    wc_FreeRsaKey(&caKey);
    wc_FreeRng(&rng);
    wc_FreeDecodedCert(&preTBS);
    return outSz;
}

static int do_dual_alg_server_certgen(byte **out, char *caKeyFile,
                                      char *sapkiFile, char *altPrivFile,
                                      char *serverKeyFile,
                                      byte *caCertBuf, int caCertSz)
{
    EXPECT_DECLS;
    FILE* file = NULL;
    Cert newCert;
    DecodedCert preTBS;

    byte serverKeyBuf[LARGE_TEMP_SZ];
    word32 serverKeySz = LARGE_TEMP_SZ;
    byte caKeyBuf[LARGE_TEMP_SZ];
    word32 caKeySz = LARGE_TEMP_SZ;
    byte sapkiBuf[LARGE_TEMP_SZ];
    word32 sapkiSz = LARGE_TEMP_SZ;
    byte altPrivBuf[LARGE_TEMP_SZ];
    word32 altPrivSz = LARGE_TEMP_SZ;
    byte altSigAlgBuf[LARGE_TEMP_SZ];
    word32 altSigAlgSz = LARGE_TEMP_SZ;
    byte scratchBuf[LARGE_TEMP_SZ];
    word32 scratchSz = LARGE_TEMP_SZ;
    byte preTbsBuf[LARGE_TEMP_SZ];
    word32 preTbsSz = LARGE_TEMP_SZ;
    byte altSigValBuf[LARGE_TEMP_SZ];
    word32 altSigValSz = LARGE_TEMP_SZ;
    byte *outBuf = NULL;
    word32 outSz = LARGE_TEMP_SZ;
    WC_RNG rng;
    RsaKey caKey;
    RsaKey serverKey;
    ecc_key altCaKey;
    word32 idx = 0;

    XMEMSET(&rng, 0, sizeof(WC_RNG));
    XMEMSET(&caKey, 0, sizeof(RsaKey));
    XMEMSET(&serverKey, 0, sizeof(RsaKey));
    XMEMSET(&altCaKey, 0, sizeof(ecc_key));

    ExpectNotNull(outBuf = (byte*)XMALLOC(outSz, NULL,
                  DYNAMIC_TYPE_TMP_BUFFER));
    ExpectIntEQ(wc_InitRng(&rng), 0);
    XMEMSET(serverKeyBuf, 0, serverKeySz);
    ExpectNotNull(file = fopen(serverKeyFile, "rb"));
    ExpectIntGT(serverKeySz = (word32)fread(serverKeyBuf, 1, serverKeySz, file),
                0);
    if (file) {
        fclose(file);
        file = NULL;
    }
    ExpectIntEQ(wc_InitRsaKey_ex(&serverKey, NULL, INVALID_DEVID), 0);
    idx = 0;
    ExpectIntEQ(wc_RsaPrivateKeyDecode(serverKeyBuf, &idx, &serverKey,
                (word32)serverKeySz), 0);
    XMEMSET(caKeyBuf, 0, caKeySz);
    ExpectNotNull(file = fopen(caKeyFile, "rb"));
    ExpectIntGT(caKeySz = (word32)fread(caKeyBuf, 1, caKeySz, file), 0);
    if (file) {
        fclose(file);
        file = NULL;
    }
    ExpectIntEQ(wc_InitRsaKey_ex(&caKey, NULL, INVALID_DEVID), 0);
    idx = 0;
    ExpectIntEQ(wc_RsaPrivateKeyDecode(caKeyBuf, &idx, &caKey,
                (word32)caKeySz), 0);
    XMEMSET(sapkiBuf, 0, sapkiSz);
    ExpectNotNull(file = fopen(sapkiFile, "rb"));
    ExpectIntGT(sapkiSz = (word32)fread(sapkiBuf, 1, sapkiSz, file), 0);
    if (file) {
        fclose(file);
        file = NULL;
    }
    XMEMSET(altPrivBuf, 0, altPrivSz);
    ExpectNotNull(file = fopen(altPrivFile, "rb"));
    ExpectIntGT(altPrivSz = (word32)fread(altPrivBuf, 1, altPrivSz, file), 0);
    if (file) {
        fclose(file);
        file = NULL;
    }
    wc_ecc_init(&altCaKey);
    idx = 0;
    ExpectIntEQ(wc_EccPrivateKeyDecode(altPrivBuf, &idx, &altCaKey,
                (word32)altPrivSz), 0);
    XMEMSET(altSigAlgBuf, 0, altSigAlgSz);
    ExpectIntGT(altSigAlgSz = SetAlgoID(CTC_SHA256wECDSA, altSigAlgBuf,
                oidSigType, 0), 0);
    wc_InitCert(&newCert);
    strncpy(newCert.subject.country, "US", CTC_NAME_SIZE);
    strncpy(newCert.subject.state, "MT", CTC_NAME_SIZE);
    strncpy(newCert.subject.locality, "Bozeman", CTC_NAME_SIZE);
    strncpy(newCert.subject.org, "wolfSSL", CTC_NAME_SIZE);
    strncpy(newCert.subject.unit, "Engineering", CTC_NAME_SIZE);
    strncpy(newCert.subject.commonName, "www.wolfssl.com", CTC_NAME_SIZE);
    strncpy(newCert.subject.email, "server@wolfssl.com", CTC_NAME_SIZE);
    strncpy((char*)newCert.beforeDate, "\x18\x0f""20250101000000Z",
        CTC_DATE_SIZE);
    newCert.beforeDateSz = 17;
    strncpy((char*)newCert.afterDate, "\x18\x0f""20493112115959Z",
        CTC_DATE_SIZE);
    newCert.afterDateSz = 17;

    newCert.sigType = CTC_SHA256wRSA;
    newCert.isCA    = 0;
    ExpectIntEQ(wc_SetIssuerBuffer(&newCert, caCertBuf, caCertSz), 0);
    ExpectIntEQ(wc_SetCustomExtension(&newCert, 0, "2.5.29.72", sapkiBuf,
                sapkiSz), 0);
    ExpectIntEQ(wc_SetCustomExtension(&newCert, 0, "2.5.29.73", altSigAlgBuf,
                altSigAlgSz), 0);
    XMEMSET(scratchBuf, 0, scratchSz);
    ExpectIntGT(wc_MakeCert(&newCert, scratchBuf, scratchSz, &serverKey, NULL,
                &rng), 0);
    ExpectIntGT(scratchSz = wc_SignCert(newCert.bodySz, newCert.sigType,
                scratchBuf, scratchSz, &caKey, NULL, &rng), 0);
    wc_InitDecodedCert(&preTBS, scratchBuf, scratchSz, 0);
    ExpectIntEQ(wc_ParseCert(&preTBS, CERT_TYPE, NO_VERIFY, NULL), 0);
    XMEMSET(preTbsBuf, 0, preTbsSz);
    ExpectIntGT(preTbsSz = wc_GeneratePreTBS(&preTBS, preTbsBuf, preTbsSz), 0);
    XMEMSET(altSigValBuf, 0, altSigValSz);
    ExpectIntGT(altSigValSz = wc_MakeSigWithBitStr(altSigValBuf, altSigValSz,
                CTC_SHA256wECDSA, preTbsBuf, preTbsSz, ECC_TYPE, &altCaKey,
                &rng), 0);
    ExpectIntEQ(wc_SetCustomExtension(&newCert, 0, "2.5.29.74",
                 altSigValBuf, altSigValSz), 0);
    /* Finally, generate the new certificate. */
    if (outBuf != NULL) {
        XMEMSET(outBuf, 0, outSz);
    }
    ExpectIntGT(wc_MakeCert(&newCert, outBuf, outSz, &serverKey, NULL, &rng),
                0);
    ExpectIntGT(outSz = wc_SignCert(newCert.bodySz, newCert.sigType, outBuf,
                outSz, &caKey, NULL, &rng), 0);
    *out = outBuf;
    wc_FreeRsaKey(&caKey);
    wc_FreeRsaKey(&serverKey);
    wc_FreeRng(&rng);
    wc_FreeDecodedCert(&preTBS);
    return outSz;
}

static int do_dual_alg_tls13_connection(byte *caCert, word32 caCertSz,
                                        byte *serverCert, word32 serverCertSz,
                                        byte *serverKey, word32 serverKeySz,
                                        int negative_test)
{
    EXPECT_DECLS;
    WOLFSSL_CTX *ctx_c = NULL;
    WOLFSSL_CTX *ctx_s = NULL;
    WOLFSSL *ssl_c = NULL;
    WOLFSSL *ssl_s = NULL;
    struct test_memio_ctx test_ctx;

    XMEMSET(&test_ctx, 0, sizeof(test_ctx));
    ExpectIntEQ(test_memio_setup_ex(&test_ctx, &ctx_c, &ctx_s, &ssl_c, &ssl_s,
                wolfTLSv1_3_client_method, wolfTLSv1_3_server_method,
                caCert, caCertSz, serverCert, serverCertSz,
                serverKey, serverKeySz), 0);
    if (negative_test) {
        ExpectTrue(test_memio_do_handshake(ssl_c, ssl_s, 10, NULL) != 0);
    }
    else {
        ExpectIntEQ(test_memio_do_handshake(ssl_c, ssl_s, 10, NULL), 0);
    }
    wolfSSL_free(ssl_c);
    wolfSSL_free(ssl_s);
    wolfSSL_CTX_free(ctx_c);
    wolfSSL_CTX_free(ctx_s);
    return EXPECT_RESULT();
}

/**
 * Function to generate a root certificate with dual algorithm support and
 * configurable criticality for extensions and path length constraints.
 *
 * @param out         [out] Pointer to store the generated certificate
 * @param caKeyFile   [in]  Path to the CA key file
 * @param sapkiFile   [in]  Path to the subject alternative public key info file
 * @param altPrivFile [in]  Path to the alternative private key file
 * @param setCrit     [in]  Flag to set criticality of extensions (1=critical, 0=non-critical)
 * @param setPathLen  [in]  Flag to set path length constraint (1=set, 0=don't set)
 * @param pathLen     [in]  Path length value (only used if setPathLen=1)
 * @return            Size of the generated certificate or negative on error
 */
static int do_dual_alg_root_certgen_crit(byte **out, char *caKeyFile,
                                      char *sapkiFile, char *altPrivFile,
                                      int setCrit, int setPathLen, int pathLen)
{
    EXPECT_DECLS;
    FILE* file = NULL;
    Cert newCert;
    DecodedCert preTBS;

    byte caKeyBuf[LARGE_TEMP_SZ];
    word32 caKeySz = LARGE_TEMP_SZ;
    byte sapkiBuf[LARGE_TEMP_SZ];
    word32 sapkiSz = LARGE_TEMP_SZ;
    byte altPrivBuf[LARGE_TEMP_SZ];
    word32 altPrivSz = LARGE_TEMP_SZ;
    byte altSigAlgBuf[LARGE_TEMP_SZ];
    word32 altSigAlgSz = LARGE_TEMP_SZ;
    byte scratchBuf[LARGE_TEMP_SZ];
    word32 scratchSz = LARGE_TEMP_SZ;
    byte preTbsBuf[LARGE_TEMP_SZ];
    word32 preTbsSz = LARGE_TEMP_SZ;
    byte altSigValBuf[LARGE_TEMP_SZ];
    word32 altSigValSz = LARGE_TEMP_SZ;
    byte *outBuf = NULL;
    word32 outSz = LARGE_TEMP_SZ;
    WC_RNG rng;
    RsaKey caKey;
    ecc_key altCaKey;
    word32 idx = 0;

    XMEMSET(&rng, 0, sizeof(WC_RNG));
    XMEMSET(&caKey, 0, sizeof(RsaKey));
    XMEMSET(&altCaKey, 0, sizeof(ecc_key));

    ExpectNotNull(outBuf = (byte*)XMALLOC(outSz, NULL,
                  DYNAMIC_TYPE_TMP_BUFFER));
    ExpectIntEQ(wc_InitRng(&rng), 0);
    XMEMSET(caKeyBuf, 0, caKeySz);
    ExpectNotNull(file = fopen(caKeyFile, "rb"));
    ExpectIntGT(caKeySz = (word32)fread(caKeyBuf, 1, caKeySz, file), 0);
    if (file) {
        fclose(file);
        file = NULL;
    }
    ExpectIntEQ(wc_InitRsaKey_ex(&caKey, NULL, INVALID_DEVID), 0);
    idx = 0;
    ExpectIntEQ(wc_RsaPrivateKeyDecode(caKeyBuf, &idx, &caKey, caKeySz),
                0);
    XMEMSET(sapkiBuf, 0, sapkiSz);
    ExpectNotNull(file = fopen(sapkiFile, "rb"));
    ExpectIntGT(sapkiSz = (word32)fread(sapkiBuf, 1, sapkiSz, file), 0);
    if (file) {
        fclose(file);
        file = NULL;
    }
    XMEMSET(altPrivBuf, 0, altPrivSz);
    ExpectNotNull(file = fopen(altPrivFile, "rb"));
    ExpectIntGT(altPrivSz = (word32)fread(altPrivBuf, 1, altPrivSz, file), 0);
    if (file) {
        fclose(file);
        file = NULL;
    }
    wc_ecc_init(&altCaKey);
    idx = 0;
    ExpectIntEQ(wc_EccPrivateKeyDecode(altPrivBuf, &idx, &altCaKey,
                                       (word32)altPrivSz), 0);
    XMEMSET(altSigAlgBuf, 0, altSigAlgSz);
    ExpectIntGT(altSigAlgSz = SetAlgoID(CTC_SHA256wECDSA, altSigAlgBuf,
                                         oidSigType, 0), 0);
    wc_InitCert(&newCert);
    strncpy(newCert.subject.country, "US", CTC_NAME_SIZE);
    strncpy(newCert.subject.state, "MT", CTC_NAME_SIZE);
    strncpy(newCert.subject.locality, "Bozeman", CTC_NAME_SIZE);
    strncpy(newCert.subject.org, "wolfSSL", CTC_NAME_SIZE);
    strncpy(newCert.subject.unit, "Engineering", CTC_NAME_SIZE);
    strncpy(newCert.subject.commonName, "www.wolfssl.com", CTC_NAME_SIZE);
    strncpy(newCert.subject.email, "root@wolfssl.com", CTC_NAME_SIZE);
    strncpy((char*)newCert.beforeDate, "\x18\x0f""20250101000000Z",
        CTC_DATE_SIZE);
    newCert.beforeDateSz = 17;
    strncpy((char*)newCert.afterDate, "\x18\x0f""20493112115959Z",
        CTC_DATE_SIZE);
    newCert.afterDateSz = 17;
    newCert.sigType = CTC_SHA256wRSA;
    newCert.isCA    = 1;

    /* Set criticality of basic constraint extension if requested */
    if (setCrit) {
        newCert.basicConstCrit = 1;
    }

    /* Set pathlen if requested */
    if (setPathLen) {
        newCert.pathLen = pathLen;
        newCert.pathLenSet = 1;
    }

    ExpectIntEQ(wc_SetCustomExtension(&newCert, 0, "1.2.3.4.5",
                (const byte *)"This is NOT a critical extension", 32), 0);
    ExpectIntEQ(wc_SetCustomExtension(&newCert, setCrit, "2.5.29.72", sapkiBuf,
                sapkiSz), 0);
    ExpectIntEQ(wc_SetCustomExtension(&newCert, setCrit, "2.5.29.73",
                                      altSigAlgBuf, altSigAlgSz), 0);

    XMEMSET(scratchBuf, 0, scratchSz);
    ExpectIntGT(scratchSz = wc_MakeSelfCert(&newCert, scratchBuf, scratchSz,
                &caKey, &rng), 0);

    wc_InitDecodedCert(&preTBS, scratchBuf, scratchSz, 0);
    ExpectIntEQ(wc_ParseCert(&preTBS, CERT_TYPE, NO_VERIFY, NULL), 0);

    XMEMSET(preTbsBuf, 0, preTbsSz);
    ExpectIntGT(preTbsSz = wc_GeneratePreTBS(&preTBS, preTbsBuf, preTbsSz), 0);
    XMEMSET(altSigValBuf, 0, altSigValSz);
    ExpectIntGT(altSigValSz = wc_MakeSigWithBitStr(altSigValBuf, altSigValSz,
                CTC_SHA256wECDSA, preTbsBuf, preTbsSz, ECC_TYPE, &altCaKey,
                &rng), 0);
    ExpectIntEQ(wc_SetCustomExtension(&newCert, setCrit, "2.5.29.74",
                                      altSigValBuf, altSigValSz), 0);

    /* Finally, generate the new certificate. */
    if (outBuf != NULL) {
        XMEMSET(outBuf, 0, outSz);
        ExpectIntGT(outSz = wc_MakeSelfCert(&newCert, outBuf, outSz, &caKey,
                    &rng), 0);
        *out = outBuf;
    }
    else {
        outSz = 0;
    }

    wc_FreeDecodedCert(&preTBS);
    wc_ecc_free(&altCaKey);
    wc_FreeRsaKey(&caKey);
    wc_FreeRng(&rng);

    return (int)outSz;
}

/**
 * Function to generate a server certificate with dual algorithm support and
 * configurable criticality for extensions and path length constraints.
 *
 * @param out           [out] Pointer to store the generated certificate
 * @param caKeyFile     [in]  Path to the CA key file
 * @param sapkiFile     [in]  Path to the subject alternative public key info file
 * @param altPrivFile   [in]  Path to the alternative private key file
 * @param serverKeyFile [in]  Path to the server key file
 * @param caCertBuf     [in]  Buffer containing the CA certificate
 * @param caCertSz      [in]  Size of the CA certificate buffer
 * @param setCrit       [in]  Flag to set criticality of extensions (1=critical, 0=non-critical)
 * @param setPathLen    [in]  Flag to set path length constraint (1=set, 0=don't set)
 * @param pathLen       [in]  Path length value (only used if setPathLen=1)
 * @return              Size of the generated certificate or negative on error
 */
static int do_dual_alg_server_certgen_crit(byte **out, char *caKeyFile,
                                        char *sapkiFile, char *altPrivFile,
                                        char *serverKeyFile,
                                        byte *caCertBuf, int caCertSz,
                                        int setCrit)
{
    EXPECT_DECLS;
    FILE* file = NULL;
    Cert newCert;
    DecodedCert preTBS;

    byte serverKeyBuf[LARGE_TEMP_SZ];
    word32 serverKeySz = LARGE_TEMP_SZ;
    byte caKeyBuf[LARGE_TEMP_SZ];
    word32 caKeySz = LARGE_TEMP_SZ;
    byte sapkiBuf[LARGE_TEMP_SZ];
    word32 sapkiSz = LARGE_TEMP_SZ;
    byte altPrivBuf[LARGE_TEMP_SZ];
    word32 altPrivSz = LARGE_TEMP_SZ;
    byte altSigAlgBuf[LARGE_TEMP_SZ];
    word32 altSigAlgSz = LARGE_TEMP_SZ;
    byte scratchBuf[LARGE_TEMP_SZ];
    word32 scratchSz = LARGE_TEMP_SZ;
    byte preTbsBuf[LARGE_TEMP_SZ];
    word32 preTbsSz = LARGE_TEMP_SZ;
    byte altSigValBuf[LARGE_TEMP_SZ];
    word32 altSigValSz = LARGE_TEMP_SZ;
    byte *outBuf = NULL;
    word32 outSz = LARGE_TEMP_SZ;
    WC_RNG rng;
    RsaKey caKey;
    RsaKey serverKey;
    ecc_key altCaKey;
    word32 idx = 0;

    XMEMSET(&rng, 0, sizeof(WC_RNG));
    XMEMSET(&caKey, 0, sizeof(RsaKey));
    XMEMSET(&serverKey, 0, sizeof(RsaKey));
    XMEMSET(&altCaKey, 0, sizeof(ecc_key));

    ExpectNotNull(outBuf = (byte*)XMALLOC(outSz, NULL,
                  DYNAMIC_TYPE_TMP_BUFFER));
    ExpectIntEQ(wc_InitRng(&rng), 0);
    XMEMSET(serverKeyBuf, 0, serverKeySz);
    ExpectNotNull(file = fopen(serverKeyFile, "rb"));
    ExpectIntGT(serverKeySz = (word32)fread(serverKeyBuf, 1, serverKeySz, file),
                0);
    if (file) {
        fclose(file);
        file = NULL;
    }
    ExpectIntEQ(wc_InitRsaKey_ex(&serverKey, NULL, INVALID_DEVID), 0);
    idx = 0;
    ExpectIntEQ(wc_RsaPrivateKeyDecode(serverKeyBuf, &idx, &serverKey,
                                       serverKeySz), 0);
    XMEMSET(caKeyBuf, 0, caKeySz);
    ExpectNotNull(file = fopen(caKeyFile, "rb"));
    ExpectIntGT(caKeySz = (word32)fread(caKeyBuf, 1, caKeySz, file), 0);
    if (file) {
        fclose(file);
        file = NULL;
    }
    ExpectIntEQ(wc_InitRsaKey_ex(&caKey, NULL, INVALID_DEVID), 0);
    idx = 0;
    ExpectIntEQ(wc_RsaPrivateKeyDecode(caKeyBuf, &idx, &caKey, caKeySz), 0);
    XMEMSET(sapkiBuf, 0, sapkiSz);
    ExpectNotNull(file = fopen(sapkiFile, "rb"));
    ExpectIntGT(sapkiSz = (word32)fread(sapkiBuf, 1, sapkiSz, file), 0);
    if (file) {
        fclose(file);
        file = NULL;
    }
    XMEMSET(altPrivBuf, 0, altPrivSz);
    ExpectNotNull(file = fopen(altPrivFile, "rb"));
    ExpectIntGT(altPrivSz = (word32)fread(altPrivBuf, 1, altPrivSz, file), 0);
    if (file) {
        fclose(file);
        file = NULL;
    }
    wc_ecc_init(&altCaKey);
    idx = 0;
    ExpectIntEQ(wc_EccPrivateKeyDecode(altPrivBuf, &idx, &altCaKey,
                                       (word32)altPrivSz), 0);
    XMEMSET(altSigAlgBuf, 0, altSigAlgSz);
    ExpectIntGT(altSigAlgSz = SetAlgoID(CTC_SHA256wECDSA, altSigAlgBuf,
                                         oidSigType, 0), 0);
    wc_InitCert(&newCert);
    strncpy(newCert.subject.country, "US", CTC_NAME_SIZE);
    strncpy(newCert.subject.state, "MT", CTC_NAME_SIZE);
    strncpy(newCert.subject.locality, "Bozeman", CTC_NAME_SIZE);
    strncpy(newCert.subject.org, "wolfSSL", CTC_NAME_SIZE);
    strncpy(newCert.subject.unit, "Engineering", CTC_NAME_SIZE);
    strncpy(newCert.subject.commonName, "www.wolfssl.com", CTC_NAME_SIZE);
    strncpy(newCert.subject.email, "server@wolfssl.com", CTC_NAME_SIZE);
    strncpy((char*)newCert.beforeDate, "\x18\x0f""20250101000000Z",
        CTC_DATE_SIZE);
    newCert.beforeDateSz = 17;
    strncpy((char*)newCert.afterDate, "\x18\x0f""20493112115959Z",
        CTC_DATE_SIZE);
    newCert.afterDateSz = 17;

    newCert.sigType = CTC_SHA256wRSA;
    newCert.isCA    = 0;
    ExpectIntEQ(wc_SetIssuerBuffer(&newCert, caCertBuf, caCertSz), 0);
    ExpectIntEQ(wc_SetCustomExtension(&newCert, setCrit, "2.5.29.72", sapkiBuf,
                                      sapkiSz), 0);
    ExpectIntEQ(wc_SetCustomExtension(&newCert, setCrit, "2.5.29.73",
                                      altSigAlgBuf, altSigAlgSz), 0);
    XMEMSET(scratchBuf, 0, scratchSz);
    ExpectIntGT(wc_MakeCert(&newCert, scratchBuf, scratchSz, &serverKey, NULL,
                &rng), 0);
    ExpectIntGT(scratchSz = wc_SignCert(newCert.bodySz, newCert.sigType,
                scratchBuf, scratchSz, &caKey, NULL, &rng), 0);
    wc_InitDecodedCert(&preTBS, scratchBuf, scratchSz, 0);
    ExpectIntEQ(wc_ParseCert(&preTBS, CERT_TYPE, NO_VERIFY, NULL), 0);
    XMEMSET(preTbsBuf, 0, preTbsSz);
    ExpectIntGT(preTbsSz = wc_GeneratePreTBS(&preTBS, preTbsBuf, preTbsSz), 0);
    XMEMSET(altSigValBuf, 0, altSigValSz);
    ExpectIntGT(altSigValSz = wc_MakeSigWithBitStr(altSigValBuf, altSigValSz,
                CTC_SHA256wECDSA, preTbsBuf, preTbsSz, ECC_TYPE, &altCaKey,
                &rng), 0);
    ExpectIntEQ(wc_SetCustomExtension(&newCert, setCrit, "2.5.29.74",
                                      altSigValBuf, altSigValSz), 0);

    /* Finally, generate the new certificate. */
    if (outBuf != NULL) {
        XMEMSET(outBuf, 0, outSz);
        ExpectIntGT(outSz = wc_SignCert(newCert.bodySz, newCert.sigType, scratchBuf,
                    outSz, &caKey, NULL, &rng), 0);
        *out = outBuf;
    }
    else {
        outSz = 0;
    }

    wc_FreeDecodedCert(&preTBS);
    wc_ecc_free(&altCaKey);
    wc_FreeRsaKey(&serverKey);
    wc_FreeRsaKey(&caKey);
    wc_FreeRng(&rng);

    return (int)outSz;
}

/**
 * Test dual-alg ECDSA + ML-DSA with critical extensions and path length
 * constraints:
 *  - keygen + certgen
 *
 * TLS tests not designed to pass with these extensions marked critical. No
 * TLS connection.
 * */
static int test_dual_alg_crit_ext_support(void)
{
    EXPECT_DECLS;
    /* Root CA and server keys will be the same. This is only appropriate for
     * testing. */
    char keyFile[] = "./certs/ca-key.der";
    char sapkiFile[] = "./certs/ecc-keyPub.der";
    char altPrivFile[] = "./certs/ecc-key.der";
    byte *serverKey = NULL;
    size_t serverKeySz = 0;
    byte *root = NULL;
    int rootSz = 0;
    byte *server = NULL;
    int serverSz = 0;

    ExpectIntEQ(load_file(keyFile, &serverKey, &serverKeySz), 0);

    /* Test with critical extensions and pathlen set to 1 */
    if (EXPECT_SUCCESS()) {
        rootSz = do_dual_alg_root_certgen_crit(&root, keyFile, sapkiFile,
            altPrivFile, 1, 1, 1);
    }
    ExpectNotNull(root);
    ExpectIntGT(rootSz, 0);
    if (EXPECT_SUCCESS()) {
        serverSz = do_dual_alg_server_certgen_crit(&server, keyFile, sapkiFile,
            altPrivFile, keyFile, root, rootSz, 1);
    }
    ExpectNotNull(server);
    ExpectIntGT(serverSz, 0);
    XFREE(root, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    root = NULL;
    XFREE(server, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    server = NULL;

    /* Test with critical extensions and pathlen set to 0 */
    if (EXPECT_SUCCESS()) {
        rootSz = do_dual_alg_root_certgen_crit(&root, keyFile, sapkiFile,
            altPrivFile, 1, 1, 0);
    }
    ExpectNotNull(root);
    ExpectIntGT(rootSz, 0);
    if (EXPECT_SUCCESS()) {
        serverSz = do_dual_alg_server_certgen_crit(&server, keyFile, sapkiFile,
            altPrivFile, keyFile, root, rootSz, 1);
    }
    ExpectNotNull(server);
    ExpectIntGT(serverSz, 0);
    XFREE(root, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    root = NULL;
    XFREE(server, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    server = NULL;

    /* Test with critical alt extensions and no pathlen set */
    if (EXPECT_SUCCESS()) {
        rootSz = do_dual_alg_root_certgen_crit(&root, keyFile, sapkiFile,
            altPrivFile, 1, 0, 0);
    }
    ExpectNotNull(root);
    ExpectIntGT(rootSz, 0);
    if (EXPECT_SUCCESS()) {
        serverSz = do_dual_alg_server_certgen_crit(&server, keyFile, sapkiFile,
            altPrivFile, keyFile, root, rootSz, 0);
    }
    ExpectNotNull(server);
    ExpectIntGT(serverSz, 0);
    XFREE(root, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    XFREE(server, NULL, DYNAMIC_TYPE_TMP_BUFFER);

    free(serverKey);

    return EXPECT_RESULT();
}

static int test_dual_alg_support(void)
{
    EXPECT_DECLS;
    /* Root CA and server keys will be the same. This is only appropriate for
     * testing. */
    char keyFile[] = "./certs/ca-key.der";
    char sapkiFile[] = "./certs/ecc-keyPub.der";
    char altPrivFile[] = "./certs/ecc-key.der";
    char wrongPrivFile[] = "./certs/ecc-client-key.der";
    byte *serverKey = NULL;
    size_t serverKeySz = 0;
    byte *root = NULL;
    int rootSz = 0;
    byte *server = NULL;
    int serverSz = 0;

    ExpectIntEQ(load_file(keyFile, &serverKey, &serverKeySz), 0);

    /* Base normal case. */
    if (EXPECT_SUCCESS()) {
        rootSz = do_dual_alg_root_certgen(&root, keyFile, sapkiFile,
            altPrivFile);
    }
    ExpectNotNull(root);
    ExpectIntGT(rootSz, 0);
    if (EXPECT_SUCCESS()) {
        serverSz = do_dual_alg_server_certgen(&server, keyFile, sapkiFile,
            altPrivFile, keyFile, root, rootSz);
    }
    ExpectNotNull(server);
    ExpectIntGT(serverSz, 0);
    ExpectIntEQ(do_dual_alg_tls13_connection(root, rootSz,
                server, serverSz, serverKey, (word32)serverKeySz, 0),
                TEST_SUCCESS);
    XFREE(root, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    root = NULL;
    XFREE(server, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    server = NULL;

    /* Now we try a negative case. Note that we use wrongPrivFile to generate
     * the alternative signature and then set negative_test to true for the
     * call to do_dual_alg_tls13_connection(). Its expecting a failed connection
     * because the signature won't verify. The exception is if
     * WOLFSSL_TRUST_PEER_CERT is defined. In that case, no verification happens
     * and this is no longer a negative test. */
    if (EXPECT_SUCCESS()) {
        rootSz = do_dual_alg_root_certgen(&root, keyFile, sapkiFile,
            wrongPrivFile);
    }
    ExpectNotNull(root);
    ExpectIntGT(rootSz, 0);
    if (EXPECT_SUCCESS()) {
        serverSz = do_dual_alg_server_certgen(&server, keyFile, sapkiFile,
            wrongPrivFile, keyFile, root, rootSz);
    }
    ExpectNotNull(server);
    ExpectIntGT(serverSz, 0);
#ifdef WOLFSSL_TRUST_PEER_CERT
    ExpectIntEQ(do_dual_alg_tls13_connection(root, rootSz,
                server, serverSz, serverKey, (word32)serverKeySz, 0),
                TEST_SUCCESS);
#else
    ExpectIntEQ(do_dual_alg_tls13_connection(root, rootSz,
                server, serverSz, serverKey, (word32)serverKeySz, 1),
                TEST_SUCCESS);
#endif

    XFREE(root, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    XFREE(server, NULL, DYNAMIC_TYPE_TMP_BUFFER);

    free(serverKey);

    return EXPECT_RESULT();
}
#else
static int test_dual_alg_support(void)
{
    return TEST_SKIPPED;
}

static int test_dual_alg_crit_ext_support(void)
{
    return TEST_SKIPPED;
}

#endif /* WOLFSSL_DUAL_ALG_CERTS && !NO_FILESYSTEM */

/**
 * Test dual-alg ECDSA + ML-DSA:
 *  - keygen + certgen + cert manager load
 * */
static int test_dual_alg_ecdsa_mldsa(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_DUAL_ALG_CERTS) && defined(HAVE_DILITHIUM) && \
    defined(HAVE_ECC) && !defined(WC_NO_RNG) && \
    defined(WOLFSSL_WC_DILITHIUM) && \
    !defined(WOLFSSL_DILITHIUM_NO_MAKE_KEY) && \
    !defined(WOLFSSL_DILITHIUM_NO_SIGN) && \
    !defined(WOLFSSL_DILITHIUM_NO_VERIFY) && !defined(WOLFSSL_SMALL_STACK)
    WOLFSSL_CERT_MANAGER * cm = NULL;
    MlDsaKey    alt_ca_key;
    ecc_key     ca_key;
    WC_RNG      rng;
    int         ret = 0;
    DecodedCert d_cert;
    Cert        new_cert;
    /* various tmp buffs.  */
    byte        alt_pub_der[LARGE_TEMP_SZ];
    word32      alt_pub_sz = LARGE_TEMP_SZ;
    byte        alt_sig_alg[LARGE_TEMP_SZ];
    word32      alt_sig_alg_sz = LARGE_TEMP_SZ;
    byte        tbs_der[LARGE_TEMP_SZ];
    word32      tbs_der_sz = LARGE_TEMP_SZ;
    byte        alt_sig[LARGE_TEMP_SZ];
    word32      alt_sig_sz = LARGE_TEMP_SZ;
    /* Intermediate der. */
    byte        der[LARGE_TEMP_SZ];
    word32      der_sz = LARGE_TEMP_SZ;
    /* The final der will be large because of ML-DSA signature. */
    byte        final_der[2 * LARGE_TEMP_SZ];
    word32      final_der_sz = 2 * LARGE_TEMP_SZ;

    XMEMSET(alt_pub_der, 0, alt_pub_sz);
    XMEMSET(alt_sig_alg, 0, alt_sig_alg_sz);
    XMEMSET(tbs_der, 0, tbs_der_sz);
    XMEMSET(alt_sig, 0, alt_sig_sz);
    XMEMSET(der, 0, der_sz);
    XMEMSET(final_der, 0, final_der_sz);

    ExpectIntEQ(wc_InitRng(&rng), 0);

    /**
     * ML-DSA key gen.
     * */
    ret = wc_MlDsaKey_Init(&alt_ca_key, NULL, INVALID_DEVID);
    ExpectIntEQ(ret, 0);
    ret = wc_MlDsaKey_SetParams(&alt_ca_key, WC_ML_DSA_44);
    ExpectIntEQ(ret, 0);
    ret = wc_MlDsaKey_MakeKey(&alt_ca_key, &rng);
    ExpectIntEQ(ret, 0);
    alt_pub_sz = wc_MlDsaKey_PublicKeyToDer(&alt_ca_key, alt_pub_der,
                                            alt_pub_sz, 1);
    ExpectIntGT(alt_pub_sz, 0);

    alt_sig_alg_sz = SetAlgoID(CTC_SHA256wECDSA, alt_sig_alg, oidSigType, 0);
    ExpectIntGT(alt_sig_alg_sz, 0);

    /**
     * ECC key gen.
     * */
    ret = wc_ecc_init(&ca_key);
    ExpectIntEQ(ret, 0);
    ret = wc_ecc_make_key(&rng, KEY32, &ca_key);
    ExpectIntEQ(ret, 0);

    /**
     * Cert gen.
     * */
    wc_InitCert(&new_cert);
    strncpy(new_cert.subject.country, "US", CTC_NAME_SIZE);
    strncpy(new_cert.subject.state, "MT", CTC_NAME_SIZE);
    strncpy(new_cert.subject.locality, "Bozeman", CTC_NAME_SIZE);
    strncpy(new_cert.subject.org, "wolfSSL", CTC_NAME_SIZE);
    strncpy(new_cert.subject.unit, "Engineering", CTC_NAME_SIZE);
    strncpy(new_cert.subject.commonName, "www.wolfssl.com", CTC_NAME_SIZE);
    strncpy(new_cert.subject.email, "root@wolfssl.com", CTC_NAME_SIZE);
    new_cert.sigType = CTC_SHA256wECDSA;
    new_cert.isCA    = 1;

    ret = wc_SetCustomExtension(&new_cert, 0, "1.2.3.4.5",
                         (const byte *)"This is NOT a critical extension", 32);
    ExpectIntEQ(ret, 0);

    ExpectIntEQ(wc_SetCustomExtension(&new_cert, 0, "2.5.29.72", alt_pub_der,
                alt_pub_sz), 0);
    ExpectIntEQ(wc_SetCustomExtension(&new_cert, 0, "2.5.29.73", alt_sig_alg,
                alt_sig_alg_sz), 0);

    ret = wc_MakeCert_ex(&new_cert, der, der_sz, ECC_TYPE, &ca_key, &rng);
    ExpectIntGT(ret, 0);

    der_sz = wc_SignCert_ex(new_cert.bodySz, new_cert.sigType, der, der_sz,
                            ECC_TYPE, &ca_key, &rng);
    ExpectIntGT(der_sz, 0);

    wc_InitDecodedCert(&d_cert, der, der_sz, 0);
    ret = wc_ParseCert(&d_cert, CERT_TYPE, NO_VERIFY, NULL);
    ExpectIntEQ(ret, 0);

    tbs_der_sz = wc_GeneratePreTBS(&d_cert, tbs_der, tbs_der_sz);
    ExpectIntGT(tbs_der_sz, 0);

    alt_sig_sz = wc_MakeSigWithBitStr(alt_sig, alt_sig_sz,
                                      CTC_ML_DSA_LEVEL2, tbs_der, tbs_der_sz,
                                      ML_DSA_LEVEL2_TYPE, &alt_ca_key, &rng);
    ExpectIntGT(alt_sig_sz, 0);

    ret = wc_SetCustomExtension(&new_cert, 0, "2.5.29.74", alt_sig, alt_sig_sz);
    ExpectIntEQ(ret, 0);

    /* Finally generate the new certificate. */
    ret = wc_MakeCert_ex(&new_cert, final_der, final_der_sz, ECC_TYPE, &ca_key,
                         &rng);
    ExpectIntGT(ret, 0);

    final_der_sz = wc_SignCert_ex(new_cert.bodySz, new_cert.sigType, final_der,
                                  final_der_sz, ECC_TYPE, &ca_key, &rng);
    ExpectIntGT(final_der_sz, 0);

    cm = wolfSSL_CertManagerNew();
    ExpectNotNull(cm);

    /* Load the certificate into CertManager. */
    if (cm != NULL && final_der_sz > 0) {
        ret = wolfSSL_CertManagerLoadCABuffer(cm, final_der, final_der_sz,
                                              WOLFSSL_FILETYPE_ASN1);
        ExpectIntEQ(ret, WOLFSSL_SUCCESS);
    }

    if (cm != NULL) {
        wolfSSL_CertManagerFree(cm);
        cm = NULL;
    }

    wc_FreeDecodedCert(&d_cert);
    wc_ecc_free(&ca_key);
    wc_MlDsaKey_Free(&alt_ca_key);
    wc_FreeRng(&rng);

#endif /* WOLFSSL_DUAL_ALG_CERTS && DILITHIUM and more */
    return EXPECT_RESULT();
}


/*----------------------------------------------------------------------------*
 | Context
 *----------------------------------------------------------------------------*/
#if !defined(NO_WOLFSSL_SERVER) && !defined(NO_TLS)
static int test_wolfSSL_CTX_new(void)
{
    EXPECT_DECLS;
    WOLFSSL_CTX *ctx;
    WOLFSSL_METHOD* method = NULL;

    ExpectNull(ctx = wolfSSL_CTX_new(NULL));
    ExpectNotNull(method = wolfSSLv23_server_method());
    if (method != NULL)
        ExpectNotNull(ctx = wolfSSL_CTX_new(method));

    wolfSSL_CTX_free(ctx);

    return EXPECT_RESULT();
}
#endif

#if (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)) && \
    !defined(NO_TLS) && \
    (!defined(NO_RSA) || defined(HAVE_ECC)) && !defined(NO_FILESYSTEM)
static int test_for_double_Free(void)
{
    EXPECT_DECLS;
    WOLFSSL_CTX* ctx = NULL;
    WOLFSSL*     ssl = NULL;
    int skipTest = 0;
    const char* testCertFile;
    const char* testKeyFile;
    char optionsCiphers[] = "RC4-SHA:RC4-MD5:DES-CBC3-SHA:AES128-SHA:AES256-SHA"
":NULL-SHA:NULL-SHA256:DHE-RSA-AES128-SHA:DHE-RSA-AES256-SHA:DHE-PSK-AES256-GCM"
"-SHA384:DHE-PSK-AES128-GCM-SHA256:PSK-AES256-GCM-SHA384:PSK-AES128-GCM-SHA256:"
"DHE-PSK-AES256-CBC-SHA384:DHE-PSK-AES128-CBC-SHA256:PSK-AES256-CBC-SHA384:PSK-"
"AES128-CBC-SHA256:PSK-AES128-CBC-SHA:PSK-AES256-CBC-SHA:DHE-PSK-AES128-CCM:DHE"
"-PSK-AES256-CCM:PSK-AES128-CCM:PSK-AES256-CCM:PSK-AES128-CCM-8:PSK-AES256-CCM-"
"8:DHE-PSK-NULL-SHA384:DHE-PSK-NULL-SHA256:PSK-NULL-SHA384:PSK-NULL-SHA256:PSK-"
"NULL-SHA:AES128-CCM-8:AES256-CCM-8:ECDHE-ECDSA-"
"AES128-CCM:ECDHE-ECDSA-AES128-CCM-8:ECDHE-ECDSA-AES256-CCM-8:ECDHE-RSA-AES128-"
"SHA:ECDHE-RSA-AES256-SHA:ECDHE-ECDSA-AES128-SHA:ECDHE-ECDSA-AES256-SHA:ECDHE-R"
"SA-RC4-SHA:ECDHE-RSA-DES-CBC3-SHA:ECDHE-ECDSA-RC4-SHA:ECDHE-ECDSA-DES-CBC3-SHA"
":AES128-SHA256:AES256-SHA256:DHE-RSA-AES128-SHA256:DHE-RSA-AES256-SHA256:ECDH-"
"RSA-AES128-SHA:ECDH-RSA-AES256-SHA:ECDH-ECDSA-AES128-SHA:ECDH-ECDSA-AES256-SHA"
":ECDH-RSA-RC4-SHA:ECDH-RSA-DES-CBC3-SHA:ECDH-ECDSA-RC4-SHA:ECDH-ECDSA-DES-CBC3"
"-SHA:AES128-GCM-SHA256:AES256-GCM-SHA384:DHE-RSA-AES128-GCM-SHA256:DHE-RSA-AES"
"256-GCM-SHA384:ECDHE-RSA-AES128-GCM-SHA256:ECDHE-RSA-AES256-GCM-SHA384:ECDHE-E"
"CDSA-AES128-GCM-SHA256:ECDHE-ECDSA-AES256-GCM-SHA384:ECDH-RSA-AES128-GCM-SHA25"
"6:ECDH-RSA-AES256-GCM-SHA384:ECDH-ECDSA-AES128-GCM-SHA256:ECDH-ECDSA-AES256-GC"
"M-SHA384:CAMELLIA128-SHA:DHE-RSA-CAMELLIA128-SHA:CAMELLIA256-SHA:DHE-RSA-CAMEL"
"LIA256-SHA:CAMELLIA128-SHA256:DHE-RSA-CAMELLIA128-SHA256:CAMELLIA256-SHA256:DH"
"E-RSA-CAMELLIA256-SHA256:ECDHE-RSA-AES128-SHA256:ECDHE-ECDSA-AES128-SHA256:ECD"
"H-RSA-AES128-SHA256:ECDH-ECDSA-AES128-SHA256:ECDHE-RSA-AES256-SHA384:ECDHE-ECD"
"SA-AES256-SHA384:ECDH-RSA-AES256-SHA384:ECDH-ECDSA-AES256-SHA384:ECDHE-RSA-CHA"
"CHA20-POLY1305:ECDHE-ECDSA-CHACHA20-POLY1305:DHE-RSA-CHACHA20-POLY1305:ECDHE-R"
"SA-CHACHA20-POLY1305-OLD:ECDHE-ECDSA-CHACHA20-POLY1305-OLD:DHE-RSA-CHACHA20-PO"
"LY1305-OLD:ECDHE-ECDSA-NULL-SHA:ECDHE-PSK-NULL-SHA256:ECDHE-PSK-A"
"ES128-CBC-SHA256:PSK-CHACHA20-POLY1305:ECDHE-PSK-CHACHA20-POLY1305:DHE-PSK-CHA"
"CHA20-POLY1305:EDH-RSA-DES-CBC3-SHA:TLS13-AES128-GCM-SHA256:TLS13-AES256-GCM-S"
"HA384:TLS13-CHACHA20-POLY1305-SHA256:TLS13-AES128-CCM-SHA256:TLS13-AES128-CCM-"
"8-SHA256:TLS13-SHA256-SHA256:TLS13-SHA384-SHA384";
    /* OpenVPN uses a "blacklist" method to specify which ciphers NOT to use */
#ifdef OPENSSL_EXTRA
    char openvpnCiphers[] = "DEFAULT:!EXP:!LOW:!MEDIUM:!kDH:!kECDH:!DSS:!PSK:"
                            "!SRP:!kRSA:!aNULL:!eNULL";
#endif

#ifndef NO_RSA
    testCertFile = svrCertFile;
    testKeyFile = svrKeyFile;
#elif defined(HAVE_ECC)
    testCertFile = eccCertFile;
    testKeyFile = eccKeyFile;
#else
    skipTest = 1;
#endif

    if (skipTest != 1) {
#ifndef NO_WOLFSSL_SERVER
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#else
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
#endif
        ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, testCertFile,
            CERT_FILETYPE));
        ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, testKeyFile,
            CERT_FILETYPE));
        ExpectNotNull(ssl = wolfSSL_new(ctx));

        /* First test freeing SSL, then CTX */
        wolfSSL_free(ssl);
        ssl = NULL;
        wolfSSL_CTX_free(ctx);
        ctx = NULL;

#ifndef NO_WOLFSSL_CLIENT
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
#else
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#endif
        ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, testCertFile,
            CERT_FILETYPE));
        ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, testKeyFile,
            CERT_FILETYPE));
        ExpectNotNull(ssl = wolfSSL_new(ctx));

        /* Next test freeing CTX then SSL */
        wolfSSL_CTX_free(ctx);
        ctx = NULL;
        wolfSSL_free(ssl);
        ssl = NULL;

#ifndef NO_WOLFSSL_SERVER
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#else
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
#endif
        /* Test setting ciphers at ctx level */
        ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, testCertFile,
            CERT_FILETYPE));
        ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, testKeyFile,
            CERT_FILETYPE));
        ExpectTrue(wolfSSL_CTX_set_cipher_list(ctx, optionsCiphers));
#if defined(OPENSSL_EXTRA) && defined(WOLFSSL_TLS13) && defined(HAVE_AESGCM) && \
        defined(WOLFSSL_SHA384) && defined(WOLFSSL_AES_256)
        /* only update TLSv13 suites */
        ExpectTrue(wolfSSL_CTX_set_cipher_list(ctx, "TLS13-AES256-GCM-SHA384"));
#endif
#if defined(OPENSSL_EXTRA) && defined(HAVE_ECC) && defined(HAVE_AESGCM) && \
    !defined(NO_SHA256) && !defined(WOLFSSL_NO_TLS12) && \
    defined(WOLFSSL_AES_128) && !defined(NO_RSA)
        /* only update pre-TLSv13 suites */
        ExpectTrue(wolfSSL_CTX_set_cipher_list(ctx,
            "ECDHE-RSA-AES128-GCM-SHA256"));
#endif
#ifdef OPENSSL_EXTRA
        ExpectTrue(wolfSSL_CTX_set_cipher_list(ctx, openvpnCiphers));
#endif
        ExpectNotNull(ssl = wolfSSL_new(ctx));
        wolfSSL_CTX_free(ctx);
        ctx = NULL;
        wolfSSL_free(ssl);
        ssl = NULL;

#ifndef NO_WOLFSSL_CLIENT
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
#else
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#endif
        ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, testCertFile,
            CERT_FILETYPE));
        ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, testKeyFile,
            CERT_FILETYPE));
        ExpectNotNull(ssl = wolfSSL_new(ctx));
        /* test setting ciphers at SSL level */
        ExpectTrue(wolfSSL_set_cipher_list(ssl, optionsCiphers));
#if defined(OPENSSL_EXTRA) && defined(WOLFSSL_TLS13) && defined(HAVE_AESGCM) && \
        defined(WOLFSSL_SHA384) && defined(WOLFSSL_AES_256)
        /* only update TLSv13 suites */
        ExpectTrue(wolfSSL_set_cipher_list(ssl, "TLS13-AES256-GCM-SHA384"));
#endif
#if defined(OPENSSL_EXTRA) && defined(HAVE_ECC) && defined(HAVE_AESGCM) && \
    !defined(NO_SHA256) && !defined(WOLFSSL_NO_TLS12) && \
    defined(WOLFSSL_AES_128) && !defined(NO_RSA)
        /* only update pre-TLSv13 suites */
        ExpectTrue(wolfSSL_set_cipher_list(ssl, "ECDHE-RSA-AES128-GCM-SHA256"));
#endif
        wolfSSL_CTX_free(ctx);
        ctx = NULL;
        wolfSSL_free(ssl);
        ssl = NULL;
    }

    return EXPECT_RESULT();
}
#endif


static int test_wolfSSL_CTX_set_cipher_list_bytes(void)
{
    EXPECT_DECLS;
#if (defined(OPENSSL_EXTRA) || defined(WOLFSSL_SET_CIPHER_BYTES)) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)) && \
    (!defined(NO_RSA) || defined(HAVE_ECC)) && !defined(NO_FILESYSTEM) && \
    !defined(NO_TLS)
    const char* testCertFile;
    const char* testKeyFile;
    WOLFSSL_CTX* ctx = NULL;
    WOLFSSL* ssl = NULL;

    const byte cipherList[] =
    {
        /* TLS_DHE_RSA_WITH_3DES_EDE_CBC_SHA */ 0x00, 0x16,
        /* TLS_DHE_RSA_WITH_AES_256_CBC_SHA  */ 0x00, 0x39,
        /* TLS_DHE_RSA_WITH_AES_128_CBC_SHA  */ 0x00, 0x33,
        /* TLS_DH_anon_WITH_AES_128_CBC_SHA  */ 0x00, 0x34,
        /* TLS_RSA_WITH_AES_256_CBC_SHA      */ 0x00, 0x35,
        /* TLS_RSA_WITH_AES_128_CBC_SHA      */ 0x00, 0x2F,
        /* TLS_RSA_WITH_NULL_MD5             */ 0x00, 0x01,
        /* TLS_RSA_WITH_NULL_SHA             */ 0x00, 0x02,
        /* TLS_PSK_WITH_AES_256_CBC_SHA      */ 0x00, 0x8d,
        /* TLS_PSK_WITH_AES_128_CBC_SHA256   */ 0x00, 0xae,
        /* TLS_PSK_WITH_AES_256_CBC_SHA384   */ 0x00, 0xaf,
        /* TLS_PSK_WITH_AES_128_CBC_SHA      */ 0x00, 0x8c,
        /* TLS_PSK_WITH_NULL_SHA256          */ 0x00, 0xb0,
        /* TLS_PSK_WITH_NULL_SHA384          */ 0x00, 0xb1,
        /* TLS_PSK_WITH_NULL_SHA             */ 0x00, 0x2c,
        /* SSL_RSA_WITH_RC4_128_SHA          */ 0x00, 0x05,
        /* SSL_RSA_WITH_RC4_128_MD5          */ 0x00, 0x04,
        /* SSL_RSA_WITH_3DES_EDE_CBC_SHA     */ 0x00, 0x0A,

        /* ECC suites, first byte is 0xC0 (ECC_BYTE) */
        /* TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA     */ 0xC0, 0x14,
        /* TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA     */ 0xC0, 0x13,
        /* TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA   */ 0xC0, 0x0A,
        /* TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA   */ 0xC0, 0x09,
        /* TLS_ECDHE_RSA_WITH_RC4_128_SHA         */ 0xC0, 0x11,
        /* TLS_ECDHE_ECDSA_WITH_RC4_128_SHA       */ 0xC0, 0x07,
        /* TLS_ECDHE_RSA_WITH_3DES_EDE_CBC_SHA    */ 0xC0, 0x12,
        /* TLS_ECDHE_ECDSA_WITH_3DES_EDE_CBC_SHA  */ 0xC0, 0x08,
        /* TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256  */ 0xC0, 0x27,
        /* TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256*/ 0xC0, 0x23,
        /* TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384  */ 0xC0, 0x28,
        /* TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384*/ 0xC0, 0x24,
        /* TLS_ECDHE_ECDSA_WITH_NULL_SHA          */ 0xC0, 0x06,
        /* TLS_ECDHE_PSK_WITH_NULL_SHA256         */ 0xC0, 0x3a,
        /* TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA256  */ 0xC0, 0x37,

        /* static ECDH, first byte is 0xC0 (ECC_BYTE) */
        /* TLS_ECDH_RSA_WITH_AES_256_CBC_SHA      */ 0xC0, 0x0F,
        /* TLS_ECDH_RSA_WITH_AES_128_CBC_SHA      */ 0xC0, 0x0E,
        /* TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA    */ 0xC0, 0x05,
        /* TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA    */ 0xC0, 0x04,
        /* TLS_ECDH_RSA_WITH_RC4_128_SHA          */ 0xC0, 0x0C,
        /* TLS_ECDH_ECDSA_WITH_RC4_128_SHA        */ 0xC0, 0x02,
        /* TLS_ECDH_RSA_WITH_3DES_EDE_CBC_SHA     */ 0xC0, 0x0D,
        /* TLS_ECDH_ECDSA_WITH_3DES_EDE_CBC_SHA   */ 0xC0, 0x03,
        /* TLS_ECDH_RSA_WITH_AES_128_CBC_SHA256   */ 0xC0, 0x29,
        /* TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA256 */ 0xC0, 0x25,
        /* TLS_ECDH_RSA_WITH_AES_256_CBC_SHA384   */ 0xC0, 0x2A,
        /* TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA384 */ 0xC0, 0x26,

        /* WDM_WITH_NULL_SHA256 */ 0x00, 0xFE, /* wolfSSL DTLS Multicast */

        /* SHA256 */
        /* TLS_DHE_RSA_WITH_AES_256_CBC_SHA256 */ 0x00, 0x6b,
        /* TLS_DHE_RSA_WITH_AES_128_CBC_SHA256 */ 0x00, 0x67,
        /* TLS_RSA_WITH_AES_256_CBC_SHA256     */ 0x00, 0x3d,
        /* TLS_RSA_WITH_AES_128_CBC_SHA256     */ 0x00, 0x3c,
        /* TLS_RSA_WITH_NULL_SHA256            */ 0x00, 0x3b,
        /* TLS_DHE_PSK_WITH_AES_128_CBC_SHA256 */ 0x00, 0xb2,
        /* TLS_DHE_PSK_WITH_NULL_SHA256        */ 0x00, 0xb4,

        /* SHA384 */
        /* TLS_DHE_PSK_WITH_AES_256_CBC_SHA384 */ 0x00, 0xb3,
        /* TLS_DHE_PSK_WITH_NULL_SHA384        */ 0x00, 0xb5,

        /* AES-GCM */
        /* TLS_RSA_WITH_AES_128_GCM_SHA256      */ 0x00, 0x9c,
        /* TLS_RSA_WITH_AES_256_GCM_SHA384      */ 0x00, 0x9d,
        /* TLS_DHE_RSA_WITH_AES_128_GCM_SHA256  */ 0x00, 0x9e,
        /* TLS_DHE_RSA_WITH_AES_256_GCM_SHA384  */ 0x00, 0x9f,
        /* TLS_DH_anon_WITH_AES_256_GCM_SHA384  */ 0x00, 0xa7,
        /* TLS_PSK_WITH_AES_128_GCM_SHA256      */ 0x00, 0xa8,
        /* TLS_PSK_WITH_AES_256_GCM_SHA384      */ 0x00, 0xa9,
        /* TLS_DHE_PSK_WITH_AES_128_GCM_SHA256  */ 0x00, 0xaa,
        /* TLS_DHE_PSK_WITH_AES_256_GCM_SHA384  */ 0x00, 0xab,

        /* ECC AES-GCM, first byte is 0xC0 (ECC_BYTE) */
        /* TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256 */ 0xC0, 0x2b,
        /* TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384 */ 0xC0, 0x2c,
        /* TLS_ECDH_ECDSA_WITH_AES_128_GCM_SHA256  */ 0xC0, 0x2d,
        /* TLS_ECDH_ECDSA_WITH_AES_256_GCM_SHA384  */ 0xC0, 0x2e,
        /* TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256   */ 0xC0, 0x2f,
        /* TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384   */ 0xC0, 0x30,
        /* TLS_ECDH_RSA_WITH_AES_128_GCM_SHA256    */ 0xC0, 0x31,
        /* TLS_ECDH_RSA_WITH_AES_256_GCM_SHA384    */ 0xC0, 0x32,

        /* AES-CCM, first byte is 0xC0 but isn't ECC,
         * also, in some of the other AES-CCM suites
         * there will be second byte number conflicts
         * with non-ECC AES-GCM */
        /* TLS_RSA_WITH_AES_128_CCM_8         */ 0xC0, 0xa0,
        /* TLS_RSA_WITH_AES_256_CCM_8         */ 0xC0, 0xa1,
        /* TLS_ECDHE_ECDSA_WITH_AES_128_CCM   */ 0xC0, 0xac,
        /* TLS_ECDHE_ECDSA_WITH_AES_128_CCM_8 */ 0xC0, 0xae,
        /* TLS_ECDHE_ECDSA_WITH_AES_256_CCM_8 */ 0xC0, 0xaf,
        /* TLS_PSK_WITH_AES_128_CCM           */ 0xC0, 0xa4,
        /* TLS_PSK_WITH_AES_256_CCM           */ 0xC0, 0xa5,
        /* TLS_PSK_WITH_AES_128_CCM_8         */ 0xC0, 0xa8,
        /* TLS_PSK_WITH_AES_256_CCM_8         */ 0xC0, 0xa9,
        /* TLS_DHE_PSK_WITH_AES_128_CCM       */ 0xC0, 0xa6,
        /* TLS_DHE_PSK_WITH_AES_256_CCM       */ 0xC0, 0xa7,

        /* Camellia */
        /* TLS_RSA_WITH_CAMELLIA_128_CBC_SHA        */ 0x00, 0x41,
        /* TLS_RSA_WITH_CAMELLIA_256_CBC_SHA        */ 0x00, 0x84,
        /* TLS_RSA_WITH_CAMELLIA_128_CBC_SHA256     */ 0x00, 0xba,
        /* TLS_RSA_WITH_CAMELLIA_256_CBC_SHA256     */ 0x00, 0xc0,
        /* TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA    */ 0x00, 0x45,
        /* TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA    */ 0x00, 0x88,
        /* TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA256 */ 0x00, 0xbe,
        /* TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA256 */ 0x00, 0xc4,

        /* chacha20-poly1305 suites first byte is 0xCC (CHACHA_BYTE) */
        /* TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256   */ 0xCC, 0xa8,
        /* TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256 */ 0xCC, 0xa9,
        /* TLS_DHE_RSA_WITH_CHACHA20_POLY1305_SHA256     */ 0xCC, 0xaa,
        /* TLS_ECDHE_PSK_WITH_CHACHA20_POLY1305_SHA256   */ 0xCC, 0xac,
        /* TLS_PSK_WITH_CHACHA20_POLY1305_SHA256         */ 0xCC, 0xab,
        /* TLS_DHE_PSK_WITH_CHACHA20_POLY1305_SHA256     */ 0xCC, 0xad,

        /* chacha20-poly1305 earlier version of nonce and padding (CHACHA_BYTE) */
        /* TLS_ECDHE_RSA_WITH_CHACHA20_OLD_POLY1305_SHA256   */ 0xCC, 0x13,
        /* TLS_ECDHE_ECDSA_WITH_CHACHA20_OLD_POLY1305_SHA256 */ 0xCC, 0x14,
        /* TLS_DHE_RSA_WITH_CHACHA20_OLD_POLY1305_SHA256     */ 0xCC, 0x15,

        /* ECDHE_PSK RFC8442, first byte is 0xD0 (ECDHE_PSK_BYTE) */
        /* TLS_ECDHE_PSK_WITH_AES_128_GCM_SHA256 */ 0xD0, 0x01,

        /* TLS v1.3 cipher suites */
        /* TLS_AES_128_GCM_SHA256       */ 0x13, 0x01,
        /* TLS_AES_256_GCM_SHA384       */ 0x13, 0x02,
        /* TLS_CHACHA20_POLY1305_SHA256 */ 0x13, 0x03,
        /* TLS_AES_128_CCM_SHA256       */ 0x13, 0x04,
        /* TLS_AES_128_CCM_8_SHA256     */ 0x13, 0x05,

        /* TLS v1.3 Integrity only cipher suites - 0xC0 (ECC) first byte */
        /* TLS_SHA256_SHA256 */ 0xC0, 0xB4,
        /* TLS_SHA384_SHA384 */ 0xC0, 0xB5
    };

#ifndef NO_RSA
    testCertFile = svrCertFile;
    testKeyFile = svrKeyFile;
#elif defined(HAVE_ECC)
    testCertFile = eccCertFile;
    testKeyFile = eccKeyFile;
#endif

#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#else
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
#endif

    ExpectTrue(wolfSSL_CTX_set_cipher_list_bytes(ctx, &cipherList[0U],
                                                           sizeof(cipherList)));

    wolfSSL_CTX_free(ctx);
    ctx = NULL;

#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#else
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
#endif

    ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, testCertFile,
                                                         CERT_FILETYPE));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, testKeyFile,
                                                         CERT_FILETYPE));

    ExpectNotNull(ssl = wolfSSL_new(ctx));

    ExpectTrue(wolfSSL_set_cipher_list_bytes(ssl, &cipherList[0U],
                                                           sizeof(cipherList)));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif /* (OPENSSL_EXTRA || WOLFSSL_SET_CIPHER_BYTES) &&
    (!NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER) && (!NO_RSA || HAVE_ECC) */

    return EXPECT_RESULT();
}


static int test_wolfSSL_CTX_use_certificate(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) || defined(HAVE_LIGHTY) || \
    defined(WOLFSSL_MYSQL_COMPATIBLE) || defined(HAVE_STUNNEL) || \
    defined(WOLFSSL_NGINX) || defined(HAVE_POCO_LIB) || \
    defined(WOLFSSL_HAPROXY)
#if !defined(NO_TLS) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
    WOLFSSL_CTX* ctx = NULL;
    X509* x509 = NULL;

#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#else
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
#endif

    ExpectNotNull(x509 = wolfSSL_X509_new());

    /* Negative tests. */
    ExpectIntEQ(SSL_CTX_use_certificate(NULL, NULL), 0);
    ExpectIntEQ(SSL_CTX_use_certificate(ctx, NULL), 0);
    ExpectIntEQ(SSL_CTX_use_certificate(NULL, x509), 0);
    /* Empty certificate */
    ExpectIntEQ(SSL_CTX_use_certificate(ctx, x509), 0);

    wolfSSL_X509_free(x509);
    wolfSSL_CTX_free(ctx);
#endif /* !NO_TLS && (!NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER) */
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_use_certificate_file(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    !defined(NO_WOLFSSL_SERVER)
    WOLFSSL_CTX *ctx = NULL;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));

    /* invalid context */
    ExpectFalse(wolfSSL_CTX_use_certificate_file(NULL, svrCertFile,
                                                         CERT_FILETYPE));
    /* invalid cert file */
    ExpectFalse(wolfSSL_CTX_use_certificate_file(ctx, bogusFile,
                                                         CERT_FILETYPE));
    /* invalid cert type */
    ExpectFalse(wolfSSL_CTX_use_certificate_file(ctx, svrCertFile, 9999));

#ifdef NO_RSA
    /* rsa needed */
    ExpectFalse(wolfSSL_CTX_use_certificate_file(ctx, svrCertFile,
                                                         CERT_FILETYPE));
#else
    /* success */
    ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, svrCertFile,
                                                         CERT_FILETYPE));
#endif

    wolfSSL_CTX_free(ctx);
#endif

    return EXPECT_RESULT();
}

#if (defined(OPENSSL_ALL) || defined(WOLFSSL_ASIO)) && !defined(NO_RSA)
static int test_wolfSSL_CTX_use_certificate_ASN1(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && !defined(NO_WOLFSSL_SERVER) && !defined(NO_ASN) && \
    !defined(NO_TLS)
    WOLFSSL_CTX* ctx = NULL;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));

    /* Failure cases. */
    ExpectIntEQ(SSL_CTX_use_certificate_ASN1(NULL, 0, NULL                ),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(SSL_CTX_use_certificate_ASN1(ctx , 0, NULL                ),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(SSL_CTX_use_certificate_ASN1(NULL, 0, server_cert_der_2048),
       WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(SSL_CTX_use_certificate_ASN1(ctx , 0, server_cert_der_2048),
       WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

    ExpectIntEQ(SSL_CTX_use_certificate_ASN1(ctx, sizeof_server_cert_der_2048,
        server_cert_der_2048), WOLFSSL_SUCCESS);

    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}
#endif /* (OPENSSL_ALL || WOLFSSL_ASIO) && !NO_RSA */

/*  Test function for wolfSSL_CTX_use_certificate_buffer. Load cert into
 *  context using buffer.
 *  PRE: NO_CERTS not defined; USE_CERT_BUFFERS_2048 defined; compile with
 *  --enable-testcert flag.
 */
static int test_wolfSSL_CTX_use_certificate_buffer(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && !defined(NO_TLS) && !defined(NO_WOLFSSL_SERVER) && \
    defined(USE_CERT_BUFFERS_2048) && !defined(NO_RSA)
    WOLFSSL_CTX* ctx = NULL;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));

    /* Invalid parameters. */
    ExpectIntEQ(wolfSSL_CTX_use_certificate_buffer(NULL, NULL, 0,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_use_certificate_buffer(ctx, NULL, 0,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(ASN_PARSE_E));
    ExpectIntEQ(wolfSSL_CTX_use_certificate_buffer(NULL, server_cert_der_2048,
        0, WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_use_certificate_buffer(ctx, server_cert_der_2048, 0,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(ASN_PARSE_E));

    ExpectIntEQ(wolfSSL_CTX_use_certificate_buffer(ctx,
        server_cert_der_2048, sizeof_server_cert_der_2048,
        WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();

} /* END test_wolfSSL_CTX_use_certificate_buffer */

static int test_wolfSSL_use_certificate_buffer(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && !defined(NO_TLS) && !defined(NO_WOLFSSL_CLIENT) && \
    defined(USE_CERT_BUFFERS_2048) && !defined(NO_RSA)
    WOLFSSL_CTX* ctx = NULL;
    WOLFSSL*     ssl = NULL;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
    ExpectNotNull(ssl = wolfSSL_new(ctx));

    /* Invalid parameters. */
    ExpectIntEQ(wolfSSL_use_certificate_buffer(NULL, NULL, 0,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_use_certificate_buffer(ssl, NULL, 0,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(ASN_PARSE_E));
    ExpectIntEQ(wolfSSL_use_certificate_buffer(NULL, client_cert_der_2048, 0,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_use_certificate_buffer(ssl, client_cert_der_2048, 0,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(ASN_PARSE_E));

    ExpectIntEQ(wolfSSL_use_certificate_buffer(ssl,
        client_cert_der_2048, sizeof_client_cert_der_2048,
        WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_use_PrivateKey_file(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    !defined(NO_WOLFSSL_SERVER)
    WOLFSSL_CTX *ctx = NULL;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));

    /* invalid context */
    ExpectFalse(wolfSSL_CTX_use_PrivateKey_file(NULL, svrKeyFile,
                                                         CERT_FILETYPE));
    /* invalid key file */
    ExpectFalse(wolfSSL_CTX_use_PrivateKey_file(ctx, bogusFile,
                                                         CERT_FILETYPE));
    /* invalid key type */
    ExpectFalse(wolfSSL_CTX_use_PrivateKey_file(ctx, svrKeyFile, 9999));

    /* invalid key format */
#ifdef WOLFSSL_PEM_TO_DER
    ExpectFalse(wolfSSL_CTX_use_PrivateKey_file(ctx, "./certs/dh-priv-2048.pem",
                                                         WOLFSSL_FILETYPE_PEM));
#else
    ExpectFalse(wolfSSL_CTX_use_PrivateKey_file(ctx, "./certs/dh-priv-2048.der",
                                                         WOLFSSL_FILETYPE_ASN1));
#endif

    /* success */
#ifdef NO_RSA
    /* rsa needed */
    ExpectFalse(wolfSSL_CTX_use_PrivateKey_file(ctx, svrKeyFile,
                                                         CERT_FILETYPE));
#else
    /* success */
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, svrKeyFile,
                                                         CERT_FILETYPE));
#endif

    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_use_RSAPrivateKey_file(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    !defined(NO_WOLFSSL_SERVER) && defined(OPENSSL_EXTRA)
    WOLFSSL_CTX *ctx = NULL;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));

    /* invalid context */
    ExpectIntEQ(wolfSSL_CTX_use_RSAPrivateKey_file(NULL, svrKeyFile,
        WOLFSSL_FILETYPE_PEM), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    /* invalid key file */
    ExpectIntEQ(wolfSSL_CTX_use_RSAPrivateKey_file(ctx, bogusFile,
        WOLFSSL_FILETYPE_PEM), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    /* invalid key type */
    ExpectIntEQ(wolfSSL_CTX_use_RSAPrivateKey_file(ctx, svrKeyFile, 9999),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

    /* success */
#ifdef NO_RSA
    /* rsa needed */
    ExpectIntEQ(wolfSSL_CTX_use_RSAPrivateKey_file(ctx, svrKeyFile,
        WOLFSSL_FILETYPE_PEM), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
#else
    /* success */
    ExpectIntEQ(wolfSSL_CTX_use_RSAPrivateKey_file(ctx, svrKeyFile,
        WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);
#endif

    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_use_RSAPrivateKey_file(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    !defined(NO_WOLFSSL_CLIENT) && defined(OPENSSL_EXTRA)
    WOLFSSL_CTX* ctx = NULL;
    WOLFSSL*     ssl = NULL;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
    ExpectNotNull(ssl = SSL_new(ctx));

    /* invalid context */
    ExpectIntEQ(wolfSSL_use_RSAPrivateKey_file(NULL, svrKeyFile,
        WOLFSSL_FILETYPE_PEM), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    /* invalid key file */
    ExpectIntEQ(wolfSSL_use_RSAPrivateKey_file(ssl, bogusFile,
        WOLFSSL_FILETYPE_PEM), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    /* invalid key type */
    ExpectIntEQ(wolfSSL_use_RSAPrivateKey_file(ssl, svrKeyFile, 9999),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

    /* success */
#ifdef NO_RSA
    /* rsa needed */
    ExpectIntEQ(wolfSSL_use_RSAPrivateKey_file(ssl, svrKeyFile,
        WOLFSSL_FILETYPE_PEM), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
#else
    /* success */
    ExpectIntEQ(wolfSSL_use_RSAPrivateKey_file(ssl, svrKeyFile,
        WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);
#endif

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_use_PrivateKey(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    !defined(NO_WOLFSSL_SERVER) && defined(OPENSSL_EXTRA)
    WOLFSSL_CTX *ctx = NULL;
    WOLFSSL_EVP_PKEY* pkey = NULL;
    const unsigned char* p;

    (void)p;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));

    ExpectNotNull(pkey = wolfSSL_EVP_PKEY_new());
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey(NULL, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey(ctx, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey(NULL, pkey), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    /* No data. */
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey(ctx, pkey), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    wolfSSL_EVP_PKEY_free(pkey);
    pkey = NULL;

#if defined(USE_CERT_BUFFERS_2048)
#if !defined(NO_RSA)
    p = client_key_der_2048;
    ExpectNotNull(pkey = d2i_PrivateKey(EVP_PKEY_RSA, NULL, &p,
        sizeof_client_key_der_2048));
#if defined(WOLFSSL_KEY_GEN)
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey(ctx, pkey), WOLFSSL_SUCCESS);
#else
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey(ctx, pkey), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
#endif
    wolfSSL_EVP_PKEY_free(pkey);
    pkey = NULL;
#endif
#if defined(WOLFSSL_QT) || defined(OPENSSL_ALL) || defined(WOLFSSL_OPENSSH)
#ifndef NO_DSA
    p = dsa_key_der_2048;
    ExpectNotNull(pkey = d2i_PrivateKey(EVP_PKEY_DSA, NULL, &p,
        sizeof_dsa_key_der_2048));
#if !defined(HAVE_SELFTEST) && (defined(WOLFSSL_KEY_GEN) || \
    defined(WOLFSSL_CERT_GEN))
    /* Not supported in ProcessBuffer. */
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey(ctx, pkey), WC_NO_ERR_TRACE(WOLFSSL_BAD_FILE));
#else
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey(ctx, pkey), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
#endif
    wolfSSL_EVP_PKEY_free(pkey);
    pkey = NULL;
#endif
#endif /* WOLFSSL_QT || OPENSSL_ALL || WOLFSSL_OPENSSH */
#if !defined(NO_DH) && defined(OPENSSL_ALL) && \
    (!defined(HAVE_FIPS) || FIPS_VERSION_GT(2,0))
    p = dh_ffdhe_statickey_der_2048;
    ExpectNotNull(pkey = d2i_PrivateKey(EVP_PKEY_DH, NULL, &p,
        sizeof_dh_ffdhe_statickey_der_2048));
    /* Not supported. */
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey(ctx, pkey), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    wolfSSL_EVP_PKEY_free(pkey);
    pkey = NULL;
#endif
#endif /* USE_CERT_BUFFERS_2048 */
#if defined(HAVE_ECC) && defined(USE_CERT_BUFFERS_256)
    p = ecc_clikey_der_256;
    ExpectNotNull(pkey = d2i_PrivateKey(EVP_PKEY_EC, NULL, &p,
        sizeof_ecc_clikey_der_256));
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey(ctx, pkey), WOLFSSL_SUCCESS);
    wolfSSL_EVP_PKEY_free(pkey);
    pkey = NULL;
#endif
    ExpectNotNull(pkey = wolfSSL_EVP_PKEY_new_mac_key(EVP_PKEY_HMAC, NULL,
        (unsigned char*)"01234567012345670123456701234567", 32));
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey(ctx, pkey), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    wolfSSL_EVP_PKEY_free(pkey);
    pkey = NULL;

    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

/* test both file and buffer versions along with unloading trusted peer certs */
static int test_wolfSSL_CTX_trust_peer_cert(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && defined(WOLFSSL_TRUST_PEER_CERT) && \
    !defined(NO_TLS) && !defined(NO_WOLFSSL_CLIENT) && !defined(NO_RSA)
    WOLFSSL_CTX *ctx = NULL;
    WOLFSSL* ssl = NULL;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
    ExpectNotNull(ssl = wolfSSL_new(ctx));

#if !defined(NO_FILESYSTEM)
    /* invalid file */
    ExpectIntNE(wolfSSL_CTX_trust_peer_cert(ctx, NULL,
                                     WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);
    ExpectIntNE(wolfSSL_CTX_trust_peer_cert(ctx, bogusFile,
                                     WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);
    ExpectIntNE(wolfSSL_CTX_trust_peer_cert(ctx, cliCertFile,
                                     WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* success */
    ExpectIntEQ(wolfSSL_CTX_trust_peer_cert(ctx, cliCertFile,
                                     WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);

    /* unload cert */
    ExpectIntNE(wolfSSL_CTX_Unload_trust_peers(NULL), WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_CTX_Unload_trust_peers(ctx), WOLFSSL_SUCCESS);

    /* invalid file */
    ExpectIntNE(wolfSSL_trust_peer_cert(ssl, NULL,
                                     WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);
    ExpectIntNE(wolfSSL_trust_peer_cert(ssl, bogusFile,
                                     WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);
    ExpectIntNE(wolfSSL_trust_peer_cert(ssl, cliCertFile,
                                     WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* success */
    ExpectIntEQ(wolfSSL_trust_peer_cert(ssl, cliCertFile,
                                     WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);

    #ifdef WOLFSSL_LOCAL_X509_STORE
    /* unload cert */
    ExpectIntNE(wolfSSL_Unload_trust_peers(NULL), WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_Unload_trust_peers(ssl), WOLFSSL_SUCCESS);
    #endif
#endif

    /* Test of loading certs from buffers */

    /* invalid buffer */
    ExpectIntNE(wolfSSL_CTX_trust_peer_buffer(ctx, NULL, -1,
                                     WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* success */
#ifdef USE_CERT_BUFFERS_1024
    ExpectIntEQ(wolfSSL_CTX_trust_peer_buffer(ctx, client_cert_der_1024,
          sizeof_client_cert_der_1024, WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
#endif
#ifdef USE_CERT_BUFFERS_2048
    ExpectIntEQ(wolfSSL_CTX_trust_peer_buffer(ctx, client_cert_der_2048,
          sizeof_client_cert_der_2048, WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
#endif

    /* unload cert */
    ExpectIntNE(wolfSSL_CTX_Unload_trust_peers(NULL), WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_CTX_Unload_trust_peers(ctx), WOLFSSL_SUCCESS);

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_load_verify_locations(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    !defined(NO_WOLFSSL_CLIENT)
    WOLFSSL_CTX *ctx = NULL;
#ifndef NO_RSA
    WOLFSSL_CERT_MANAGER* cm = NULL;
#ifdef PERSIST_CERT_CACHE
    int cacheSz = 0;
    unsigned char* cache = NULL;
    int used = 0;
#ifndef NO_FILESYSTEM
    const char* cacheFile = "./tests/cert_cache.tmp";
#endif
    int i;
    int t;
    int* p;
#endif
#endif
#if !defined(NO_WOLFSSL_DIR) && !defined(WOLFSSL_TIRTOS) && \
    defined(WOLFSSL_PEM_TO_DER)
    const char* load_certs_path = "./certs/external";
    const char* load_no_certs_path = "./examples";
    const char* load_expired_path = "./certs/test/expired";
#endif

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));

    /* invalid arguments */
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations(NULL, caCertFile, NULL),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations(ctx, NULL, NULL),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

    /* invalid ca file */
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations(ctx, bogusFile, NULL),
        WS_RETURN_CODE(WC_NO_ERR_TRACE(WOLFSSL_BAD_FILE),
                       WC_NO_ERR_TRACE(WOLFSSL_FAILURE)));


#if !defined(NO_WOLFSSL_DIR) && !defined(WOLFSSL_TIRTOS) && \
  ((defined(WOLFSSL_QT) || defined(WOLFSSL_IGNORE_BAD_CERT_PATH)) && \
  !(WOLFSSL_LOAD_VERIFY_DEFAULT_FLAGS & WOLFSSL_LOAD_FLAG_IGNORE_BAD_PATH_ERR))
    /* invalid path */
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations(ctx, NULL, bogusFile),
        WS_RETURN_CODE(WC_NO_ERR_TRACE(BAD_PATH_ERROR),WC_NO_ERR_TRACE(WOLFSSL_FAILURE)));
#endif
#if defined(WOLFSSL_QT) || defined(WOLFSSL_IGNORE_BAD_CERT_PATH)
    /* test ignoring the invalid path */
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations_ex(ctx, NULL, bogusFile,
        WOLFSSL_LOAD_FLAG_IGNORE_BAD_PATH_ERR), WOLFSSL_SUCCESS);
#endif

    /* load ca cert */
#ifdef NO_RSA
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations(ctx, caCertFile, NULL),
        WS_RETURN_CODE(WC_NO_ERR_TRACE(ASN_UNKNOWN_OID_E),WC_NO_ERR_TRACE(WOLFSSL_FAILURE)));
#else /* Skip the following test without RSA certs. */
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations(ctx, caCertFile, NULL),
        WOLFSSL_SUCCESS);

#ifdef PERSIST_CERT_CACHE
    /* Get cert cache size */
    ExpectIntGT(cacheSz = wolfSSL_CTX_get_cert_cache_memsize(ctx), 0);

    ExpectNotNull(cache = (byte*)XMALLOC((size_t)cacheSz, NULL,
                            DYNAMIC_TYPE_TMP_BUFFER));

    ExpectIntEQ(wolfSSL_CTX_memsave_cert_cache(NULL, NULL, -1, NULL),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_memsave_cert_cache(ctx, NULL, -1, NULL),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_memsave_cert_cache(NULL, cache, -1, NULL),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_memsave_cert_cache(NULL, NULL, cacheSz, NULL),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_memsave_cert_cache(NULL, NULL, -1, &used),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_memsave_cert_cache(NULL, cache, cacheSz, &used),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_memsave_cert_cache(ctx, NULL, cacheSz, &used),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_memsave_cert_cache(ctx, cache, -1, &used),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_memsave_cert_cache(ctx, cache, cacheSz, NULL),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_memsave_cert_cache(ctx, cache, cacheSz - 10, &used),
        WC_NO_ERR_TRACE(BUFFER_E));
    ExpectIntEQ(wolfSSL_CTX_memsave_cert_cache(ctx, cache, cacheSz, &used), 1);
    ExpectIntEQ(cacheSz, used);

    ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(NULL, NULL, -1),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(ctx, NULL, -1),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(NULL, cache, -1),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(NULL, NULL, cacheSz),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(NULL, cache, cacheSz),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(ctx, NULL, cacheSz),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(ctx, cache, -1),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    /* Smaller than header. */
    ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(ctx, cache, 1), WC_NO_ERR_TRACE(BUFFER_E));
    for (i = 1; i < cacheSz; i++) {
        ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(ctx, cache, cacheSz - i),
            WC_NO_ERR_TRACE(BUFFER_E));
    }
    if (EXPECT_SUCCESS()) {
        /* Modify header for bad results! */
        p = (int*)cache;
        /* version */
        t = p[0]; p[0] = 0xff;
        ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(ctx, cache, cacheSz),
            WC_NO_ERR_TRACE(CACHE_MATCH_ERROR));
        p[0] = t; p++;
        /* rows */
        t = p[0]; p[0] = 0xff;
        ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(ctx, cache, cacheSz),
            WC_NO_ERR_TRACE(CACHE_MATCH_ERROR));
        p[0] = t; p++;
        /* columns[0] */
        t = p[0]; p[0] = -1;
        ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(ctx, cache, cacheSz),
            WC_NO_ERR_TRACE(PARSE_ERROR));
        p[0] = t; p += CA_TABLE_SIZE;
        /* signerSz*/
        t = p[0]; p[0] = 0xff;
        ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(ctx, cache, cacheSz),
            WC_NO_ERR_TRACE(CACHE_MATCH_ERROR));
        p[0] = t;
    }

    ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(ctx, cache, cacheSz), 1);
    ExpectIntEQ(cacheSz = wolfSSL_CTX_get_cert_cache_memsize(ctx), used);

#ifndef NO_FILESYSTEM
    ExpectIntEQ(wolfSSL_CTX_save_cert_cache(NULL, NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_save_cert_cache(ctx, NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_save_cert_cache(NULL, cacheFile), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_save_cert_cache(ctx, cacheFile), 1);

    ExpectIntEQ(wolfSSL_CTX_restore_cert_cache(NULL, NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_restore_cert_cache(ctx, NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_restore_cert_cache(NULL, cacheFile), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_restore_cert_cache(ctx, "no-file"),
        WC_NO_ERR_TRACE(WOLFSSL_BAD_FILE));
    ExpectIntEQ(wolfSSL_CTX_restore_cert_cache(ctx, cacheFile), 1);
    /* File contents is not a cache. */
    ExpectIntEQ(wolfSSL_CTX_restore_cert_cache(ctx, "./certs/ca-cert.pem"),
        WC_NO_ERR_TRACE(CACHE_MATCH_ERROR));
#endif

    XFREE(cache, NULL, DYNAMIC_TYPE_TMP_BUFFER);
#endif
    /* Test unloading CA's */
    ExpectIntEQ(wolfSSL_CTX_UnloadCAs(ctx), WOLFSSL_SUCCESS);

#ifdef PERSIST_CERT_CACHE
    /* Verify no certs (result is less than cacheSz) */
    ExpectIntGT(cacheSz, wolfSSL_CTX_get_cert_cache_memsize(ctx));
#endif

    /* load ca cert again */
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations(ctx, caCertFile, NULL),
        WOLFSSL_SUCCESS);

    /* Test getting CERT_MANAGER */
    ExpectNotNull(cm = wolfSSL_CTX_GetCertManager(ctx));

    /* Test unloading CA's using CM */
    ExpectIntEQ(wolfSSL_CertManagerUnloadCAs(cm), WOLFSSL_SUCCESS);

#ifdef PERSIST_CERT_CACHE
    /* Verify no certs (result is less than cacheSz) */
    ExpectIntGT(cacheSz, wolfSSL_CTX_get_cert_cache_memsize(ctx));
#endif
#endif

#if !defined(NO_WOLFSSL_DIR) && !defined(WOLFSSL_TIRTOS) && \
    defined(WOLFSSL_PEM_TO_DER)
    /* Test loading CA certificates using a path */
    #ifdef NO_RSA
    /* failure here okay since certs in external directory are RSA */
    ExpectIntNE(wolfSSL_CTX_load_verify_locations_ex(ctx, NULL, load_certs_path,
        WOLFSSL_LOAD_FLAG_PEM_CA_ONLY), WOLFSSL_SUCCESS);
    #else
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations_ex(ctx, NULL, load_certs_path,
        WOLFSSL_LOAD_FLAG_PEM_CA_ONLY), WOLFSSL_SUCCESS);
    #endif

    /* Test loading path with no files */
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations_ex(ctx, NULL,
        load_no_certs_path, WOLFSSL_LOAD_FLAG_PEM_CA_ONLY),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

    /* Test loading expired CA certificates */
    #ifdef NO_RSA
    ExpectIntNE(wolfSSL_CTX_load_verify_locations_ex(ctx, NULL,
        load_expired_path,
        WOLFSSL_LOAD_FLAG_DATE_ERR_OKAY | WOLFSSL_LOAD_FLAG_PEM_CA_ONLY),
        WOLFSSL_SUCCESS);
    #else
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations_ex(ctx, NULL,
        load_expired_path,
        WOLFSSL_LOAD_FLAG_DATE_ERR_OKAY | WOLFSSL_LOAD_FLAG_PEM_CA_ONLY),
        WOLFSSL_SUCCESS);
    #endif

    /* Test loading CA certificates and ignoring all errors */
    #ifdef NO_RSA
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations_ex(ctx, NULL, load_certs_path,
        WOLFSSL_LOAD_FLAG_IGNORE_ERR), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    #else
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations_ex(ctx, NULL, load_certs_path,
        WOLFSSL_LOAD_FLAG_IGNORE_ERR), WOLFSSL_SUCCESS);
    #endif
#endif

    wolfSSL_CTX_free(ctx);
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_load_system_CA_certs(void)
{
    int res = TEST_SKIPPED;
#if defined(WOLFSSL_SYS_CA_CERTS) && !defined(NO_WOLFSSL_CLIENT) && \
    !defined(NO_TLS) && (!defined(NO_RSA) || defined(HAVE_ECC)) && \
    defined(WOLFSSL_PEM_TO_DER)
    WOLFSSL_CTX* ctx;
    byte dirValid = 0;
    int ret = 0;

    ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
    if (ctx == NULL) {
        fprintf(stderr, "wolfSSL_CTX_new failed.\n");
        ret = -1;
    }
    if (ret == 0) {
    #if defined(USE_WINDOWS_API) || defined(__APPLE__) || \
        defined(NO_WOLFSSL_DIR)
        dirValid = 1;
    #else
        word32 numDirs;
        const char** caDirs = wolfSSL_get_system_CA_dirs(&numDirs);

        if (caDirs == NULL || numDirs == 0) {
            fprintf(stderr, "wolfSSL_get_system_CA_dirs failed.\n");
            ret = -1;
        }
        else {
            ReadDirCtx dirCtx;
            word32 i;

            for (i = 0; i < numDirs; ++i) {
                if (wc_ReadDirFirst(&dirCtx, caDirs[i], NULL) == 0) {
                    /* Directory isn't empty. */
                    dirValid = 1;
                    wc_ReadDirClose(&dirCtx);
                    break;
                }
            }
        }
    #endif
    }
    /*
     * If the directory isn't empty, we should be able to load CA
     * certs from it. On Windows/Mac, we assume the CA cert stores are
     * usable.
     */
    if (ret == 0 && dirValid && wolfSSL_CTX_load_system_CA_certs(ctx) !=
        WOLFSSL_SUCCESS) {
        fprintf(stderr, "wolfSSL_CTX_load_system_CA_certs failed.\n");
        ret = -1;
    }
#ifdef OPENSSL_EXTRA
    if (ret == 0 &&
        wolfSSL_CTX_set_default_verify_paths(ctx) != WOLFSSL_SUCCESS) {
        fprintf(stderr, "wolfSSL_CTX_set_default_verify_paths failed.\n");
        ret = -1;
    }
#endif /* OPENSSL_EXTRA */

    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(ret == 0);
#endif /* WOLFSSL_SYS_CA_CERTS && !NO_WOLFSSL_CLIENT */

    return res;
}

#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS)
static int test_cm_load_ca_buffer(const byte* cert_buf, size_t cert_sz,
    int file_type)
{
    int ret;
    WOLFSSL_CERT_MANAGER* cm;

    cm = wolfSSL_CertManagerNew();
    if (cm == NULL) {
        fprintf(stderr, "test_cm_load_ca failed\n");
        return -1;
    }

    ret = wolfSSL_CertManagerLoadCABuffer(cm, cert_buf, (sword32)cert_sz, file_type);

    wolfSSL_CertManagerFree(cm);

    return ret;
}

static int test_cm_load_ca_file(const char* ca_cert_file)
{
    int ret = 0;
    byte* cert_buf = NULL;
    size_t cert_sz = 0;
#if defined(WOLFSSL_PEM_TO_DER)
    DerBuffer* pDer = NULL;
#endif

    ret = load_file(ca_cert_file, &cert_buf, &cert_sz);
    if (ret == 0) {
        /* normal test */
        ret = test_cm_load_ca_buffer(cert_buf, cert_sz, CERT_FILETYPE);

        if (ret == WOLFSSL_SUCCESS) {
            /* test including null terminator in length */
            byte* tmp = (byte*)realloc(cert_buf, cert_sz+1);
            if (tmp == NULL) {
                ret = MEMORY_E;
            }
            else {
                cert_buf = tmp;
                cert_buf[cert_sz] = '\0';
                ret = test_cm_load_ca_buffer(cert_buf, cert_sz+1,
                        CERT_FILETYPE);
            }

        }

    #if defined(WOLFSSL_PEM_TO_DER)
        if (ret == WOLFSSL_SUCCESS) {
            /* test loading DER */
            ret = wc_PemToDer(cert_buf, (sword32)cert_sz, CA_TYPE, &pDer,
                    NULL, NULL, NULL);
            if (ret == 0 && pDer != NULL) {
                ret = test_cm_load_ca_buffer(pDer->buffer, pDer->length,
                    WOLFSSL_FILETYPE_ASN1);

                wc_FreeDer(&pDer);
            }
        }
    #endif

    }
    free(cert_buf);

    return ret;
}

static int test_cm_load_ca_buffer_ex(const byte* cert_buf, size_t cert_sz,
                                     int file_type, word32 flags)
{
    int ret;
    WOLFSSL_CERT_MANAGER* cm;

    cm = wolfSSL_CertManagerNew();
    if (cm == NULL) {
        fprintf(stderr, "test_cm_load_ca failed\n");
        return -1;
    }

    ret = wolfSSL_CertManagerLoadCABuffer_ex(cm, cert_buf, (sword32)cert_sz, file_type,
                                             0, flags);

    wolfSSL_CertManagerFree(cm);

    return ret;
}

static int test_cm_load_ca_file_ex(const char* ca_cert_file, word32 flags)
{
    int ret = 0;
    byte* cert_buf = NULL;
    size_t cert_sz = 0;
#if defined(WOLFSSL_PEM_TO_DER)
    DerBuffer* pDer = NULL;
#endif

    ret = load_file(ca_cert_file, &cert_buf, &cert_sz);
    if (ret == 0) {
        /* normal test */
        ret = test_cm_load_ca_buffer_ex(cert_buf, cert_sz,
                                        CERT_FILETYPE, flags);

        if (ret == WOLFSSL_SUCCESS) {
            /* test including null terminator in length */
            byte* tmp = (byte*)realloc(cert_buf, cert_sz+1);
            if (tmp == NULL) {
                ret = MEMORY_E;
            }
            else {
                cert_buf = tmp;
                cert_buf[cert_sz] = '\0';
                ret = test_cm_load_ca_buffer_ex(cert_buf, cert_sz+1,
                        CERT_FILETYPE, flags);
            }

        }

    #if defined(WOLFSSL_PEM_TO_DER)
        if (ret == WOLFSSL_SUCCESS) {
            /* test loading DER */
            ret = wc_PemToDer(cert_buf, (sword32)cert_sz, CA_TYPE, &pDer,
                              NULL, NULL, NULL);
            if (ret == 0 && pDer != NULL) {
                ret = test_cm_load_ca_buffer_ex(pDer->buffer, pDer->length,
                    WOLFSSL_FILETYPE_ASN1, flags);

                wc_FreeDer(&pDer);
            }
        }
    #endif

    }
    free(cert_buf);

    return ret;
}

#endif /* !NO_FILESYSTEM && !NO_CERTS */

static int test_wolfSSL_CertManagerAPI(void)
{
    EXPECT_DECLS;
#ifndef NO_CERTS
    WOLFSSL_CERT_MANAGER* cm = NULL;
    unsigned char c = 0;

    ExpectNotNull(cm = wolfSSL_CertManagerNew_ex(NULL));

    wolfSSL_CertManagerFree(NULL);
    ExpectIntEQ(wolfSSL_CertManager_up_ref(NULL), 0);
    ExpectIntEQ(wolfSSL_CertManagerUnloadCAs(NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
#ifdef WOLFSSL_TRUST_PEER_CERT
    ExpectIntEQ(wolfSSL_CertManagerUnload_trust_peers(NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
#endif

    ExpectIntEQ(wolfSSL_CertManagerLoadCABuffer_ex(NULL, &c, 1,
        WOLFSSL_FILETYPE_ASN1, 0, 0), WC_NO_ERR_TRACE(WOLFSSL_FATAL_ERROR));

#if !defined(NO_WOLFSSL_CLIENT) || !defined(WOLFSSL_NO_CLIENT_AUTH)
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(NULL, NULL, -1,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, NULL, -1,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(NULL, &c, -1,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(NULL, NULL, 1,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(NULL, &c, 1,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, NULL, 1,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, &c, -1,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, &c, 1, -1),
        WC_NO_ERR_TRACE(WOLFSSL_BAD_FILETYPE));
#endif

#if !defined(NO_FILESYSTEM)
    {
    #ifdef WOLFSSL_PEM_TO_DER
        const char* ca_cert = "./certs/ca-cert.pem";
    #if !defined(NO_WOLFSSL_CLIENT) || !defined(WOLFSSL_NO_CLIENT_AUTH)
        const char* ca_cert_der = "./certs/ca-cert.der";
    #endif
    #else
        const char* ca_cert = "./certs/ca-cert.der";
    #endif
        const char* ca_path = "./certs";

    #if !defined(NO_WOLFSSL_CLIENT) || !defined(WOLFSSL_NO_CLIENT_AUTH)
        ExpectIntEQ(wolfSSL_CertManagerVerify(NULL, NULL, -1),
            WC_NO_ERR_TRACE(BAD_FUNC_ARG));
        ExpectIntEQ(wolfSSL_CertManagerVerify(cm, NULL, WOLFSSL_FILETYPE_ASN1),
            WC_NO_ERR_TRACE(BAD_FUNC_ARG));
        ExpectIntEQ(wolfSSL_CertManagerVerify(NULL, ca_cert,
            WOLFSSL_FILETYPE_PEM), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
        ExpectIntEQ(wolfSSL_CertManagerVerify(cm, ca_cert, -1),
            WC_NO_ERR_TRACE(WOLFSSL_BAD_FILETYPE));
#ifdef WOLFSSL_PEM_TO_DER
        ExpectIntEQ(wolfSSL_CertManagerVerify(cm, ca_cert_der,
            WOLFSSL_FILETYPE_PEM), WC_NO_ERR_TRACE(ASN_NO_PEM_HEADER));
#endif
        ExpectIntEQ(wolfSSL_CertManagerVerify(cm, "no-file",
            WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(WOLFSSL_BAD_FILE));
    #endif

        ExpectIntEQ(wolfSSL_CertManagerLoadCA(NULL, NULL, NULL),
            WC_NO_ERR_TRACE(WOLFSSL_FATAL_ERROR));
        ExpectIntEQ(wolfSSL_CertManagerLoadCA(NULL, ca_cert, NULL),
            WC_NO_ERR_TRACE(WOLFSSL_FATAL_ERROR));
        ExpectIntEQ(wolfSSL_CertManagerLoadCA(NULL, NULL, ca_path),
            WC_NO_ERR_TRACE(WOLFSSL_FATAL_ERROR));
        ExpectIntEQ(wolfSSL_CertManagerLoadCA(NULL, ca_cert, ca_path),
            WC_NO_ERR_TRACE(WOLFSSL_FATAL_ERROR));
    }
#endif

#ifdef OPENSSL_COMPATIBLE_DEFAULTS
    ExpectIntEQ(wolfSSL_CertManagerEnableCRL(cm, 0), 1);
#elif !defined(HAVE_CRL)
    ExpectIntEQ(wolfSSL_CertManagerEnableCRL(cm, 0), WC_NO_ERR_TRACE(NOT_COMPILED_IN));
#endif

    ExpectIntEQ(wolfSSL_CertManagerDisableCRL(NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerDisableCRL(cm), 1);
#ifdef HAVE_CRL
    /* Test APIs when CRL is disabled. */
#ifdef HAVE_CRL_IO
    ExpectIntEQ(wolfSSL_CertManagerSetCRL_IOCb(cm, NULL), 1);
#endif
    ExpectIntEQ(wolfSSL_CertManagerCheckCRL(cm, server_cert_der_2048,
        sizeof_server_cert_der_2048), 1);
    ExpectIntEQ(wolfSSL_CertManagerFreeCRL(cm), 1);
#endif

    /* OCSP */
    ExpectIntEQ(wolfSSL_CertManagerEnableOCSP(NULL, 0), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerDisableOCSP(NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerEnableOCSPStapling(NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerDisableOCSPStapling(NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerEnableOCSPMustStaple(NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerDisableOCSPMustStaple(NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
#if !defined(HAVE_CERTIFICATE_STATUS_REQUEST) && \
    !defined(HAVE_CERTIFICATE_STATUS_REQUEST_V2)
    ExpectIntEQ(wolfSSL_CertManagerDisableOCSPStapling(cm), WC_NO_ERR_TRACE(NOT_COMPILED_IN));
    ExpectIntEQ(wolfSSL_CertManagerEnableOCSPMustStaple(cm), WC_NO_ERR_TRACE(NOT_COMPILED_IN));
    ExpectIntEQ(wolfSSL_CertManagerDisableOCSPMustStaple(cm), WC_NO_ERR_TRACE(NOT_COMPILED_IN));
#endif

#ifdef HAVE_OCSP
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSP(NULL, NULL, -1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSP(cm, NULL, -1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSP(NULL, &c, -1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSP(NULL, NULL, 1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSP(NULL, &c, 1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSP(cm, NULL, 1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSP(cm, &c, -1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    ExpectIntEQ(wolfSSL_CertManagerCheckOCSPResponse(NULL, NULL, 0,
        NULL, NULL, NULL, NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSPResponse(cm, NULL, 1,
        NULL, NULL, NULL, NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSPResponse(NULL, &c, 1,
        NULL, NULL, NULL, NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    ExpectIntEQ(wolfSSL_CertManagerSetOCSPOverrideURL(NULL, NULL),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerSetOCSPOverrideURL(NULL, ""),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerSetOCSPOverrideURL(cm, NULL), 1);

    ExpectIntEQ(wolfSSL_CertManagerSetOCSP_Cb(NULL, NULL, NULL, NULL),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerSetOCSP_Cb(cm, NULL, NULL, NULL), 1);

    ExpectIntEQ(wolfSSL_CertManagerDisableOCSP(cm), 1);
    /* Test APIs when OCSP is disabled. */
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSPResponse(cm, &c, 1,
        NULL, NULL, NULL, NULL), 1);
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSP(cm, &c, 1), 1);

#endif

    ExpectIntEQ(wolfSSL_CertManager_up_ref(cm), 1);
    if (EXPECT_SUCCESS()) {
        wolfSSL_CertManagerFree(cm);
    }
    wolfSSL_CertManagerFree(cm);
    cm = NULL;

    ExpectNotNull(cm = wolfSSL_CertManagerNew_ex(NULL));

#ifdef HAVE_OCSP
    ExpectIntEQ(wolfSSL_CertManagerEnableOCSP(cm, WOLFSSL_OCSP_URL_OVERRIDE |
         WOLFSSL_OCSP_CHECKALL), 1);
#if defined(HAVE_CERTIFICATE_STATUS_REQUEST) || \
    defined(HAVE_CERTIFICATE_STATUS_REQUEST_V2)
    ExpectIntEQ(wolfSSL_CertManagerEnableOCSPStapling(cm), 1);
    ExpectIntEQ(wolfSSL_CertManagerEnableOCSPStapling(cm), 1);
    ExpectIntEQ(wolfSSL_CertManagerDisableOCSPStapling(cm), 1);
    ExpectIntEQ(wolfSSL_CertManagerEnableOCSPStapling(cm), 1);
    ExpectIntEQ(wolfSSL_CertManagerEnableOCSPMustStaple(cm), 1);
    ExpectIntEQ(wolfSSL_CertManagerDisableOCSPMustStaple(cm), 1);
#endif

    ExpectIntEQ(wolfSSL_CertManagerSetOCSPOverrideURL(cm, ""), 1);
    ExpectIntEQ(wolfSSL_CertManagerSetOCSPOverrideURL(cm, ""), 1);
#endif

#ifdef WOLFSSL_TRUST_PEER_CERT
    ExpectIntEQ(wolfSSL_CertManagerUnload_trust_peers(cm), 1);
#endif
    wolfSSL_CertManagerFree(cm);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_CertManagerLoadCABuffer(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS)
#if defined(WOLFSSL_PEM_TO_DER)
    const char* ca_cert = "./certs/ca-cert.pem";
    const char* ca_expired_cert = "./certs/test/expired/expired-ca.pem";
#else
    const char* ca_cert = "./certs/ca-cert.der";
    const char* ca_expired_cert = "./certs/test/expired/expired-ca.der";
#endif
    int ret;

    ExpectIntLE(ret = test_cm_load_ca_file(ca_cert), 1);
#if defined(NO_WOLFSSL_CLIENT) && defined(NO_WOLFSSL_SERVER)
    ExpectIntEQ(ret, WC_NO_ERR_TRACE(WOLFSSL_FATAL_ERROR));
#elif defined(NO_RSA)
    ExpectIntEQ(ret, WC_NO_ERR_TRACE(ASN_UNKNOWN_OID_E));
#else
    ExpectIntEQ(ret, WOLFSSL_SUCCESS);
#endif

    ExpectIntLE(ret = test_cm_load_ca_file(ca_expired_cert), 1);
#if defined(NO_WOLFSSL_CLIENT) && defined(NO_WOLFSSL_SERVER)
    ExpectIntEQ(ret, WC_NO_ERR_TRACE(WOLFSSL_FATAL_ERROR));
#elif defined(NO_RSA)
    ExpectIntEQ(ret, WC_NO_ERR_TRACE(ASN_UNKNOWN_OID_E));
#elif !(WOLFSSL_LOAD_VERIFY_DEFAULT_FLAGS & WOLFSSL_LOAD_FLAG_DATE_ERR_OKAY) && \
      !defined(NO_ASN_TIME)
    ExpectIntEQ(ret, WC_NO_ERR_TRACE(ASN_AFTER_DATE_E));
#else
    ExpectIntEQ(ret, WOLFSSL_SUCCESS);
#endif
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_CertManagerLoadCABuffer_ex(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS)
#if defined(WOLFSSL_PEM_TO_DER)
    const char* ca_cert = "./certs/ca-cert.pem";
    const char* ca_expired_cert = "./certs/test/expired/expired-ca.pem";
#else
    const char* ca_cert = "./certs/ca-cert.der";
    const char* ca_expired_cert = "./certs/test/expired/expired-ca.der";
#endif
    int ret;

    ExpectIntLE(ret = test_cm_load_ca_file_ex(ca_cert, WOLFSSL_LOAD_FLAG_NONE),
        1);
#if defined(NO_WOLFSSL_CLIENT) && defined(NO_WOLFSSL_SERVER)
    ExpectIntEQ(ret, WC_NO_ERR_TRACE(WOLFSSL_FATAL_ERROR));
#elif defined(NO_RSA)
    ExpectIntEQ(ret, WC_NO_ERR_TRACE(ASN_UNKNOWN_OID_E));
#else
    ExpectIntEQ(ret, WOLFSSL_SUCCESS);
#endif

    ExpectIntLE(ret = test_cm_load_ca_file_ex(ca_expired_cert,
        WOLFSSL_LOAD_FLAG_DATE_ERR_OKAY), 1);
#if defined(NO_WOLFSSL_CLIENT) && defined(NO_WOLFSSL_SERVER)
    ExpectIntEQ(ret, WC_NO_ERR_TRACE(WOLFSSL_FATAL_ERROR));
#elif defined(NO_RSA)
    ExpectIntEQ(ret, WC_NO_ERR_TRACE(ASN_UNKNOWN_OID_E));
#elif !(WOLFSSL_LOAD_VERIFY_DEFAULT_FLAGS & WOLFSSL_LOAD_FLAG_DATE_ERR_OKAY) && \
      !defined(NO_ASN_TIME) && defined(WOLFSSL_TRUST_PEER_CERT) && \
      defined(OPENSSL_COMPATIBLE_DEFAULTS)
    ExpectIntEQ(ret, WC_NO_ERR_TRACE(ASN_AFTER_DATE_E));
#else
    ExpectIntEQ(ret, WOLFSSL_SUCCESS);
#endif

#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_CertManagerLoadCABufferType(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    !defined(NO_RSA) && !defined(NO_SHA256) && \
    !defined(WOLFSSL_TEST_APPLE_NATIVE_CERT_VALIDATION)
#if defined(WOLFSSL_PEM_TO_DER)
    const char* ca_cert = "./certs/ca-cert.pem";
    const char* int1_cert = "./certs/intermediate/ca-int-cert.pem";
    const char* int2_cert = "./certs/intermediate/ca-int2-cert.pem";
    const char* client_cert = "./certs/intermediate/client-int-cert.pem";
#else
    const char* ca_cert = "./certs/ca-cert.der";
    const char* int1_cert = "./certs/intermediate/ca-int-cert.der";
    const char* int2_cert = "./certs/intermediate/ca-int2-cert.der";
    const char* client_cert = "./certs/intermediate/client-int-cert.der";
#endif
    byte* ca_cert_buf = NULL;
    byte* int1_cert_buf = NULL;
    byte* int2_cert_buf = NULL;
    byte* client_cert_buf = NULL;
    size_t ca_cert_sz = 0;
    size_t int1_cert_sz = 0;
    size_t int2_cert_sz = 0;
    size_t client_cert_sz = 0;
    WOLFSSL_CERT_MANAGER* cm = NULL;

    ExpectNotNull(cm = wolfSSL_CertManagerNew());
    ExpectIntEQ(load_file(ca_cert, &ca_cert_buf, &ca_cert_sz), 0);
    ExpectIntEQ(load_file(int1_cert, &int1_cert_buf, &int1_cert_sz), 0);
    ExpectIntEQ(load_file(int2_cert, &int2_cert_buf, &int2_cert_sz), 0);
    ExpectIntEQ(load_file(client_cert, &client_cert_buf, &client_cert_sz), 0);

    ExpectIntNE(wolfSSL_CertManagerLoadCABufferType(cm, ca_cert_buf,
        (sword32)ca_cert_sz, CERT_FILETYPE, 0,
        WOLFSSL_LOAD_VERIFY_DEFAULT_FLAGS, 0), WOLFSSL_SUCCESS);
    ExpectIntNE(wolfSSL_CertManagerLoadCABufferType(cm, ca_cert_buf,
        (sword32)ca_cert_sz, CERT_FILETYPE, 0,
        WOLFSSL_LOAD_VERIFY_DEFAULT_FLAGS, 5), WOLFSSL_SUCCESS);

    ExpectIntEQ(wolfSSL_CertManagerLoadCABufferType(cm, ca_cert_buf,
        (sword32)ca_cert_sz, CERT_FILETYPE, 0,
        WOLFSSL_LOAD_VERIFY_DEFAULT_FLAGS, WOLFSSL_USER_CA),
        WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, int1_cert_buf,
        int1_cert_sz, CERT_FILETYPE), WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_CertManagerLoadCABufferType(cm, int1_cert_buf,
        (sword32)int1_cert_sz, CERT_FILETYPE, 0,
        WOLFSSL_LOAD_VERIFY_DEFAULT_FLAGS, WOLFSSL_USER_INTER),
        WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, int2_cert_buf,
        int2_cert_sz, CERT_FILETYPE), WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_CertManagerLoadCABufferType(cm, int2_cert_buf,
        (sword32)int2_cert_sz, CERT_FILETYPE, 0,
        WOLFSSL_LOAD_VERIFY_DEFAULT_FLAGS, WOLFSSL_USER_INTER),
        WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, client_cert_buf,
        client_cert_sz, CERT_FILETYPE), WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_CertManagerLoadCABufferType(cm, client_cert_buf,
        (sword32)client_cert_sz, CERT_FILETYPE, 0,
        WOLFSSL_LOAD_VERIFY_DEFAULT_FLAGS, WOLFSSL_USER_INTER),
        WOLFSSL_SUCCESS);

    ExpectIntEQ(wolfSSL_CertManagerUnloadTypeCerts(cm, WOLFSSL_USER_INTER),
        WOLFSSL_SUCCESS);

    /* Intermediate certs have been unloaded, but CA cert is still
       loaded.  Expect first level intermediate to verify, rest to fail. */
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, int1_cert_buf,
        int1_cert_sz, CERT_FILETYPE), WOLFSSL_SUCCESS);
    ExpectIntNE(wolfSSL_CertManagerVerifyBuffer(cm, int2_cert_buf,
        int2_cert_sz, CERT_FILETYPE), WOLFSSL_SUCCESS);
    ExpectIntNE(wolfSSL_CertManagerVerifyBuffer(cm, client_cert_buf,
        client_cert_sz, CERT_FILETYPE), WOLFSSL_SUCCESS);

    ExpectIntEQ(wolfSSL_CertManagerLoadCABufferType(cm, int1_cert_buf,
        (sword32)int1_cert_sz, CERT_FILETYPE, 0,
        WOLFSSL_LOAD_VERIFY_DEFAULT_FLAGS, WOLFSSL_TEMP_CA),
        WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, int2_cert_buf,
        int2_cert_sz, CERT_FILETYPE), WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_CertManagerLoadCABufferType(cm, int2_cert_buf,
        (sword32)int2_cert_sz, CERT_FILETYPE, 0,
        WOLFSSL_LOAD_VERIFY_DEFAULT_FLAGS, WOLFSSL_CHAIN_CA),
        WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, client_cert_buf,
        client_cert_sz, CERT_FILETYPE), WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_CertManagerLoadCABufferType(cm, client_cert_buf,
        (sword32)client_cert_sz, CERT_FILETYPE, 0,
        WOLFSSL_LOAD_VERIFY_DEFAULT_FLAGS, WOLFSSL_USER_INTER),
        WOLFSSL_SUCCESS);

    ExpectIntEQ(wolfSSL_CertManagerUnloadTypeCerts(cm, WOLFSSL_USER_INTER),
        WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, int1_cert_buf,
        int1_cert_sz, CERT_FILETYPE), WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, int2_cert_buf,
        int2_cert_sz, CERT_FILETYPE), WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, client_cert_buf,
        client_cert_sz, CERT_FILETYPE), WOLFSSL_SUCCESS);

    ExpectIntEQ(wolfSSL_CertManagerUnloadTypeCerts(cm, WOLFSSL_CHAIN_CA),
        WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, int1_cert_buf,
        int1_cert_sz, CERT_FILETYPE), WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, int2_cert_buf,
        int2_cert_sz, CERT_FILETYPE), WOLFSSL_SUCCESS);
    ExpectIntNE(wolfSSL_CertManagerVerifyBuffer(cm, client_cert_buf,
        client_cert_sz, CERT_FILETYPE), WOLFSSL_SUCCESS);

    ExpectIntEQ(wolfSSL_CertManagerUnloadTypeCerts(cm, WOLFSSL_TEMP_CA),
        WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, int1_cert_buf,
        int1_cert_sz, CERT_FILETYPE), WOLFSSL_SUCCESS);
    ExpectIntNE(wolfSSL_CertManagerVerifyBuffer(cm, int2_cert_buf,
        int2_cert_sz, CERT_FILETYPE), WOLFSSL_SUCCESS);
    ExpectIntNE(wolfSSL_CertManagerVerifyBuffer(cm, client_cert_buf,
        client_cert_sz, CERT_FILETYPE), WOLFSSL_SUCCESS);

    ExpectIntEQ(wolfSSL_CertManagerUnloadTypeCerts(cm, WOLFSSL_USER_CA),
        WOLFSSL_SUCCESS);
    ExpectIntNE(wolfSSL_CertManagerVerifyBuffer(cm, int1_cert_buf,
        int1_cert_sz, CERT_FILETYPE), WOLFSSL_SUCCESS);
    ExpectIntNE(wolfSSL_CertManagerVerifyBuffer(cm, int2_cert_buf,
        int2_cert_sz, CERT_FILETYPE), WOLFSSL_SUCCESS);
    ExpectIntNE(wolfSSL_CertManagerVerifyBuffer(cm, client_cert_buf,
        client_cert_sz, CERT_FILETYPE), WOLFSSL_SUCCESS);

    if (cm)
        wolfSSL_CertManagerFree(cm);
    if (ca_cert_buf)
        free(ca_cert_buf);
    if (int1_cert_buf)
        free(int1_cert_buf);
    if (int2_cert_buf)
        free(int2_cert_buf);
    if (client_cert_buf)
        free(client_cert_buf);
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_CertManagerGetCerts(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_ALL) && !defined(NO_CERTS) && \
    !defined(NO_FILESYSTEM) && !defined(NO_RSA) && \
     defined(WOLFSSL_SIGNER_DER_CERT)
    WOLFSSL_CERT_MANAGER* cm = NULL;
    WOLFSSL_STACK* sk = NULL;
    X509* x509 = NULL;
    X509* cert1 = NULL;
    FILE* file1 = NULL;
#ifdef DEBUG_WOLFSSL_VERBOSE
    WOLFSSL_BIO* bio = NULL;
#endif
    int i = 0;
    int ret = 0;
    const byte* der = NULL;
    int derSz = 0;

    ExpectNotNull(file1 = fopen("./certs/ca-cert.pem", "rb"));

    ExpectNotNull(cert1 = wolfSSL_PEM_read_X509(file1, NULL, NULL, NULL));
    if (file1 != NULL) {
        fclose(file1);
    }

    ExpectNull(sk = wolfSSL_CertManagerGetCerts(NULL));
    ExpectNotNull(cm = wolfSSL_CertManagerNew_ex(NULL));
    ExpectNull(sk = wolfSSL_CertManagerGetCerts(cm));

    ExpectNotNull(der = wolfSSL_X509_get_der(cert1, &derSz));
#if defined(OPENSSL_ALL) || defined(WOLFSSL_QT)
    /* Check that ASN_SELF_SIGNED_E is returned for a self-signed cert for QT
     * and full OpenSSL compatibility */
    ExpectIntEQ(ret = wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(ASN_SELF_SIGNED_E));
#else
    ExpectIntEQ(ret = wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(ASN_NO_SIGNER_E));
#endif

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CertManagerLoadCA(cm,
                "./certs/ca-cert.pem", NULL));

    ExpectNotNull(sk = wolfSSL_CertManagerGetCerts(cm));

    for (i = 0; EXPECT_SUCCESS() && i < sk_X509_num(sk); i++) {
        ExpectNotNull(x509 = sk_X509_value(sk, i));
        ExpectIntEQ(0, wolfSSL_X509_cmp(x509, cert1));

#ifdef DEBUG_WOLFSSL_VERBOSE
        bio = BIO_new(wolfSSL_BIO_s_file());
        if (bio != NULL) {
            BIO_set_fp(bio, stderr, BIO_NOCLOSE);
            X509_print(bio, x509);
            BIO_free(bio);
        }
#endif /* DEBUG_WOLFSSL_VERBOSE */
    }
    wolfSSL_X509_free(cert1);
    sk_X509_pop_free(sk, NULL);
    wolfSSL_CertManagerFree(cm);
#endif /* defined(OPENSSL_ALL) && !defined(NO_CERTS) && \
          !defined(NO_FILESYSTEM) && !defined(NO_RSA) && \
          defined(WOLFSSL_SIGNER_DER_CERT) */

    return EXPECT_RESULT();
}

static int test_wolfSSL_CertManagerSetVerify(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    !defined(NO_WOLFSSL_CM_VERIFY) && !defined(NO_RSA) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(WOLFSSL_NO_CLIENT_AUTH))
    WOLFSSL_CERT_MANAGER* cm = NULL;
    int tmp = myVerifyAction;
#ifdef WOLFSSL_PEM_TO_DER
    const char* ca_cert = "./certs/ca-cert.pem";
    const char* expiredCert = "./certs/test/expired/expired-cert.pem";
#else
    const char* ca_cert = "./certs/ca-cert.der";
    const char* expiredCert = "./certs/test/expired/expired-cert.der";
#endif

    wolfSSL_CertManagerSetVerify(NULL, NULL);
    wolfSSL_CertManagerSetVerify(NULL, myVerify);

    ExpectNotNull(cm = wolfSSL_CertManagerNew());

    wolfSSL_CertManagerSetVerify(cm, myVerify);

#if defined(NO_WOLFSSL_CLIENT) && defined(NO_WOLFSSL_SERVER)
    ExpectIntEQ(wolfSSL_CertManagerLoadCA(cm, ca_cert, NULL), -1);
#else
    ExpectIntEQ(wolfSSL_CertManagerLoadCA(cm, ca_cert, NULL),
                WOLFSSL_SUCCESS);
#endif
    /* Use the test CB that always accepts certs */
    myVerifyAction = VERIFY_OVERRIDE_ERROR;

    ExpectIntEQ(wolfSSL_CertManagerVerify(cm, expiredCert,
        CERT_FILETYPE), WOLFSSL_SUCCESS);

#ifdef WOLFSSL_ALWAYS_VERIFY_CB
    {
        const char* verifyCert = "./certs/server-cert.der";
        /* Use the test CB that always fails certs */
        myVerifyAction = VERIFY_FORCE_FAIL;

        ExpectIntEQ(wolfSSL_CertManagerVerify(cm, verifyCert,
            WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(VERIFY_CERT_ERROR));
    }
#endif

    wolfSSL_CertManagerFree(cm);
    myVerifyAction = tmp;
#endif

    return EXPECT_RESULT();
}

#if !defined(NO_FILESYSTEM) && defined(OPENSSL_EXTRA) && \
    defined(DEBUG_UNIT_TEST_CERTS)
/* Used when debugging name constraint tests. Not static to allow use in
 * multiple locations with complex define guards. */
void DEBUG_WRITE_CERT_X509(WOLFSSL_X509* x509, const char* fileName)
{
    BIO* out = BIO_new_file(fileName, "wb");
    if (out != NULL) {
        PEM_write_bio_X509(out, x509);
        BIO_free(out);
    }
}
void DEBUG_WRITE_DER(const byte* der, int derSz, const char* fileName)
{
    BIO* out = BIO_new_file(fileName, "wb");
    if (out != NULL) {
        BIO_write(out, der, derSz);
        BIO_free(out);
    }
}
#else
#define DEBUG_WRITE_CERT_X509(x509, fileName) WC_DO_NOTHING
#define DEBUG_WRITE_DER(der, derSz, fileName) WC_DO_NOTHING
#endif


static int test_wolfSSL_CertManagerNameConstraint(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && \
    !defined(NO_WOLFSSL_CM_VERIFY) && !defined(NO_RSA) && \
    defined(OPENSSL_EXTRA) && defined(WOLFSSL_CERT_GEN) && \
    defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_ALT_NAMES) && \
    !defined(NO_SHA256)
    WOLFSSL_CERT_MANAGER* cm = NULL;
    WOLFSSL_EVP_PKEY *priv = NULL;
    WOLFSSL_X509_NAME* name = NULL;
    const char* ca_cert = "./certs/test/cert-ext-nc.der";
    const char* server_cert = "./certs/test/server-goodcn.pem";
    int i = 0;
    static const byte extNameConsOid[] = {85, 29, 30};

    RsaKey  key;
    WC_RNG  rng;
    byte    *der = NULL;
    int     derSz = 0;
    word32  idx = 0;
    byte    *pt;
    WOLFSSL_X509 *x509 = NULL;
    WOLFSSL_X509 *ca = NULL;

    wc_InitRng(&rng);

    /* load in CA private key for signing */
    ExpectIntEQ(wc_InitRsaKey_ex(&key, HEAP_HINT, testDevId), 0);
    ExpectIntEQ(wc_RsaPrivateKeyDecode(server_key_der_2048, &idx, &key,
                sizeof_server_key_der_2048), 0);

    /* get ca certificate then alter it */
    ExpectNotNull(der =
            (byte*)XMALLOC(FOURK_BUF, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER));
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(ca_cert,
                WOLFSSL_FILETYPE_ASN1));
    ExpectNotNull(pt = (byte*)wolfSSL_X509_get_tbs(x509, &derSz));
    if (EXPECT_SUCCESS() && (der != NULL)) {
        XMEMCPY(der, pt, (size_t)derSz);

        /* find the name constraint extension and alter it */
        pt = der;
        for (i = 0; i < derSz - 3; i++) {
            if (XMEMCMP(pt, extNameConsOid, 3) == 0) {
                pt += 3;
                break;
            }
            pt++;
        }
        ExpectIntNE(i, derSz - 3); /* did not find OID if this case is hit */

        /* go to the length value and set it to 0 */
        while (i < derSz && *pt != 0x81) {
            pt++;
            i++;
        }
        ExpectIntNE(i, derSz); /* did not place to alter */
        pt++;
        *pt = 0x00;
    }

    /* resign the altered certificate */
    ExpectIntGT((derSz = wc_SignCert(derSz, CTC_SHA256wRSA, der,
                             FOURK_BUF, &key, NULL, &rng)), 0);

    ExpectNotNull(cm = wolfSSL_CertManagerNew());
    ExpectIntEQ(wolfSSL_CertManagerLoadCABuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(ASN_PARSE_E));
    wolfSSL_CertManagerFree(cm);

    XFREE(der, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    wolfSSL_X509_free(x509);
    wc_FreeRsaKey(&key);
    wc_FreeRng(&rng);

    /* add email alt name to satisfy constraint */
    pt = (byte*)server_key_der_2048;
    ExpectNotNull(priv = wolfSSL_d2i_PrivateKey(EVP_PKEY_RSA, NULL,
                (const unsigned char**)&pt, sizeof_server_key_der_2048));

    ExpectNotNull(cm = wolfSSL_CertManagerNew());
    ExpectNotNull(ca = wolfSSL_X509_load_certificate_file(ca_cert,
                WOLFSSL_FILETYPE_ASN1));

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(ca, &derSz)));
    DEBUG_WRITE_DER(der, derSz, "ca.der");

    ExpectIntEQ(wolfSSL_CertManagerLoadCABuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* Good cert test with proper alt email name */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"wolfssl.com", 11, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "emailAddress", MBSTRING_UTF8,
                    (byte*)"support@info.wolfssl.com", 24, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);
    name = NULL;

    wolfSSL_X509_add_altname(x509, "wolfssl@info.wolfssl.com", ASN_RFC822_TYPE);

    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "good-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    wolfSSL_X509_free(x509);
    x509 = NULL;


    /* Cert with bad alt name list */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"wolfssl.com", 11, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "emailAddress", MBSTRING_UTF8,
                    (byte*)"support@info.wolfssl.com", 24, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);

    wolfSSL_X509_add_altname(x509, "wolfssl@info.com", ASN_RFC822_TYPE);
    wolfSSL_X509_add_altname(x509, "wolfssl@info.wolfssl.com", ASN_RFC822_TYPE);

    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "bad-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(ASN_NAME_INVALID_E));

    wolfSSL_CertManagerFree(cm);
    wolfSSL_X509_free(x509);
    wolfSSL_X509_free(ca);
    wolfSSL_EVP_PKEY_free(priv);
#endif

    return EXPECT_RESULT();
}


static int test_wolfSSL_CertManagerNameConstraint2(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && \
    !defined(NO_WOLFSSL_CM_VERIFY) && !defined(NO_RSA) && \
    defined(OPENSSL_EXTRA) && defined(WOLFSSL_CERT_GEN) && \
    defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_ALT_NAMES)
    const char* ca_cert  = "./certs/test/cert-ext-ndir.der";
    const char* ca_cert2 = "./certs/test/cert-ext-ndir-exc.der";
    const char* server_cert = "./certs/server-cert.pem";
    WOLFSSL_CERT_MANAGER* cm = NULL;
    WOLFSSL_X509 *x509 = NULL;
    WOLFSSL_X509 *ca = NULL;

    const unsigned char *der = NULL;
    const unsigned char *pt;
    WOLFSSL_EVP_PKEY *priv = NULL;
    WOLFSSL_X509_NAME* name = NULL;
    int   derSz = 0;

    /* C=US*/
    char altName[] = {
        0x30, 0x0D, 0x31, 0x0B, 0x30, 0x09,
        0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53
    };

    /* C=ID */
    char altNameFail[] = {
        0x30, 0x0D, 0x31, 0x0B, 0x30, 0x09,
        0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x49, 0x44
    };

    /* C=US ST=California*/
    char altNameExc[] = {
        0x30, 0x22,
        0x31, 0x0B,
        0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53,
        0x31, 0x13,
        0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0C, 0x0A,
        0x43, 0x61, 0x6c, 0x69, 0x66, 0x6f, 0x72, 0x6e, 0x69, 0x61
    };
    /* load in CA private key for signing */
    pt = ca_key_der_2048;
    ExpectNotNull(priv = wolfSSL_d2i_PrivateKey(EVP_PKEY_RSA, NULL, &pt,
                                                sizeof_ca_key_der_2048));

    ExpectNotNull(cm = wolfSSL_CertManagerNew());
    ExpectNotNull(ca = wolfSSL_X509_load_certificate_file(ca_cert,
                WOLFSSL_FILETYPE_ASN1));
    ExpectNotNull((der = wolfSSL_X509_get_der(ca, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerLoadCABuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_NOSHA3_256)
    wolfSSL_X509_sign(x509, priv, EVP_sha3_256());
#else
    wolfSSL_X509_sign(x509, priv, EVP_sha256());
#endif
    ExpectNotNull((der = wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* Test no name case. */
    ExpectIntEQ(wolfSSL_X509_add_altname_ex(x509, NULL, 0, ASN_DIR_TYPE),
        WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_add_altname(x509, "", ASN_DIR_TYPE),
        WOLFSSL_SUCCESS);
    /* IP not supported. */
    ExpectIntEQ(wolfSSL_X509_add_altname(x509, "127.0.0.1", ASN_IP_TYPE),
        WOLFSSL_FAILURE);

    /* add in matching DIR alt name and resign */
    wolfSSL_X509_add_altname_ex(x509, altName, sizeof(altName), ASN_DIR_TYPE);
#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_NOSHA3_256)
    wolfSSL_X509_sign(x509, priv, EVP_sha3_256());
#else
    wolfSSL_X509_sign(x509, priv, EVP_sha256());
#endif

    ExpectNotNull((der = wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    wolfSSL_X509_free(x509);
    x509 = NULL;

    /* check verify fail */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);

    /* add in miss matching DIR alt name and resign */
    wolfSSL_X509_add_altname_ex(x509, altNameFail, sizeof(altNameFail),
            ASN_DIR_TYPE);

#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_NOSHA3_256)
    wolfSSL_X509_sign(x509, priv, EVP_sha3_256());
#else
    wolfSSL_X509_sign(x509, priv, EVP_sha256());
#endif
    ExpectNotNull((der = wolfSSL_X509_get_der(x509, &derSz)));
#ifndef WOLFSSL_NO_ASN_STRICT
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(ASN_NAME_INVALID_E));
#else
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
#endif

    /* check that it still fails if one bad altname and one good altname is in
     * the certificate */
    wolfSSL_X509_free(x509);
    x509 = NULL;
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    wolfSSL_X509_add_altname_ex(x509, altName, sizeof(altName), ASN_DIR_TYPE);
    wolfSSL_X509_add_altname_ex(x509, altNameFail, sizeof(altNameFail),
            ASN_DIR_TYPE);

#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_NOSHA3_256)
    wolfSSL_X509_sign(x509, priv, EVP_sha3_256());
#else
    wolfSSL_X509_sign(x509, priv, EVP_sha256());
#endif
    ExpectNotNull((der = wolfSSL_X509_get_der(x509, &derSz)));
#ifndef WOLFSSL_NO_ASN_STRICT
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(ASN_NAME_INVALID_E));
#else
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
#endif

    /* check it fails with switching position of bad altname */
    wolfSSL_X509_free(x509);
    x509 = NULL;
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    wolfSSL_X509_add_altname_ex(x509, altNameFail, sizeof(altNameFail),
            ASN_DIR_TYPE);
    wolfSSL_X509_add_altname_ex(x509, altName, sizeof(altName), ASN_DIR_TYPE);

#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_NOSHA3_256)
    wolfSSL_X509_sign(x509, priv, EVP_sha3_256());
#else
    wolfSSL_X509_sign(x509, priv, EVP_sha256());
#endif
    ExpectNotNull((der = wolfSSL_X509_get_der(x509, &derSz)));
#ifndef WOLFSSL_NO_ASN_STRICT
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(ASN_NAME_INVALID_E));
#else
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
#endif
    wolfSSL_CertManagerFree(cm);

    wolfSSL_X509_free(x509);
    x509 = NULL;
    wolfSSL_X509_free(ca);
    ca = NULL;

    /* now test with excluded name constraint */
    ExpectNotNull(cm = wolfSSL_CertManagerNew());
    ExpectNotNull(ca = wolfSSL_X509_load_certificate_file(ca_cert2,
                WOLFSSL_FILETYPE_ASN1));
    ExpectNotNull((der = wolfSSL_X509_get_der(ca, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerLoadCABuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    wolfSSL_X509_add_altname_ex(x509, altNameExc, sizeof(altNameExc),
            ASN_DIR_TYPE);
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);

#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_NOSHA3_256)
    wolfSSL_X509_sign(x509, priv, EVP_sha3_256());
#else
    wolfSSL_X509_sign(x509, priv, EVP_sha256());
#endif
    ExpectNotNull((der = wolfSSL_X509_get_der(x509, &derSz)));
#ifndef WOLFSSL_NO_ASN_STRICT
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(ASN_NAME_INVALID_E));
#else
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
#endif
    wolfSSL_CertManagerFree(cm);
    wolfSSL_X509_free(x509);
    wolfSSL_X509_free(ca);
    wolfSSL_EVP_PKEY_free(priv);
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_CertManagerNameConstraint3(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && \
    !defined(NO_WOLFSSL_CM_VERIFY) && !defined(NO_RSA) && \
    defined(OPENSSL_EXTRA) && defined(WOLFSSL_CERT_GEN) && \
    defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_ALT_NAMES) && \
    !defined(NO_SHA256)
    WOLFSSL_CERT_MANAGER* cm = NULL;
    WOLFSSL_EVP_PKEY *priv = NULL;
    WOLFSSL_X509_NAME* name = NULL;
    const char* ca_cert = "./certs/test/cert-ext-mnc.der";
    const char* server_cert = "./certs/test/server-goodcn.pem";

    byte    *der = NULL;
    int     derSz = 0;
    byte    *pt;
    WOLFSSL_X509 *x509 = NULL;
    WOLFSSL_X509 *ca = NULL;

    pt = (byte*)server_key_der_2048;
    ExpectNotNull(priv = wolfSSL_d2i_PrivateKey(EVP_PKEY_RSA, NULL,
                (const unsigned char**)&pt, sizeof_server_key_der_2048));

    ExpectNotNull(cm = wolfSSL_CertManagerNew());
    ExpectNotNull(ca = wolfSSL_X509_load_certificate_file(ca_cert,
                WOLFSSL_FILETYPE_ASN1));
    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(ca, &derSz)));
    DEBUG_WRITE_DER(der, derSz, "ca.der");

    ExpectIntEQ(wolfSSL_CertManagerLoadCABuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* check satisfying .wolfssl.com constraint passes */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"wolfssl.com", 11, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "emailAddress", MBSTRING_UTF8,
                    (byte*)"support@info.wolfssl.com", 24, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);
    name = NULL;

    wolfSSL_X509_add_altname(x509, "wolfssl@info.wolfssl.com", ASN_RFC822_TYPE);

    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "good-1st-constraint-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    wolfSSL_X509_free(x509);
    x509 = NULL;

    /* check satisfying .random.com constraint passes */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"wolfssl.com", 11, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "emailAddress", MBSTRING_UTF8,
                    (byte*)"support@info.example.com", 24, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);
    name = NULL;

    wolfSSL_X509_add_altname(x509, "wolfssl@info.example.com", ASN_RFC822_TYPE);

    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "good-2nd-constraint-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    wolfSSL_X509_free(x509);
    x509 = NULL;

    /* check fail case when neither constraint is matched */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"wolfssl.com", 11, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "emailAddress", MBSTRING_UTF8,
                     (byte*)"support@info.com", 16, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);

    wolfSSL_X509_add_altname(x509, "wolfssl@info.com", ASN_RFC822_TYPE);

    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "bad-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(ASN_NAME_INVALID_E));

    wolfSSL_CertManagerFree(cm);
    wolfSSL_X509_free(x509);
    wolfSSL_X509_free(ca);
    wolfSSL_EVP_PKEY_free(priv);
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_CertManagerNameConstraint4(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && \
    !defined(NO_WOLFSSL_CM_VERIFY) && !defined(NO_RSA) && \
    defined(OPENSSL_EXTRA) && defined(WOLFSSL_CERT_GEN) && \
    defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_ALT_NAMES) && \
    !defined(NO_SHA256)
    WOLFSSL_CERT_MANAGER* cm = NULL;
    WOLFSSL_EVP_PKEY *priv = NULL;
    WOLFSSL_X509_NAME* name = NULL;
    const char* ca_cert = "./certs/test/cert-ext-ncdns.der";
    const char* server_cert = "./certs/test/server-goodcn.pem";

    byte    *der = NULL;
    int     derSz;
    byte    *pt;
    WOLFSSL_X509 *x509 = NULL;
    WOLFSSL_X509 *ca = NULL;

    pt = (byte*)server_key_der_2048;
    ExpectNotNull(priv = wolfSSL_d2i_PrivateKey(EVP_PKEY_RSA, NULL,
                (const unsigned char**)&pt, sizeof_server_key_der_2048));

    ExpectNotNull(cm = wolfSSL_CertManagerNew());
    ExpectNotNull(ca = wolfSSL_X509_load_certificate_file(ca_cert,
                WOLFSSL_FILETYPE_ASN1));
    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(ca, &derSz)));
    DEBUG_WRITE_DER(der, derSz, "ca.der");

    ExpectIntEQ(wolfSSL_CertManagerLoadCABuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* check satisfying wolfssl.com constraint passes */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"wolfssl.com", 11, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);
    name = NULL;

    wolfSSL_X509_add_altname(x509, "www.wolfssl.com", ASN_DNS_TYPE);
    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "good-1st-constraint-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    wolfSSL_X509_free(x509);
    x509 = NULL;

    /* check satisfying example.com constraint passes */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"example.com", 11, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);
    name = NULL;

    wolfSSL_X509_add_altname(x509, "www.example.com", ASN_DNS_TYPE);
    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "good-2nd-constraint-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    wolfSSL_X509_free(x509);
    x509 = NULL;

    /* check satisfying wolfssl.com constraint passes with list of DNS's */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"wolfssl.com", 11, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);
    name = NULL;

    wolfSSL_X509_add_altname(x509, "www.wolfssl.com", ASN_DNS_TYPE);
    wolfSSL_X509_add_altname(x509, "www.info.wolfssl.com", ASN_DNS_TYPE);
    wolfSSL_X509_add_altname(x509, "extra.wolfssl.com", ASN_DNS_TYPE);
    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "good-multiple-constraint-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    wolfSSL_X509_free(x509);
    x509 = NULL;

    /* check fail when one DNS in the list is bad */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"wolfssl.com", 11, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);
    name = NULL;

    wolfSSL_X509_add_altname(x509, "www.wolfssl.com", ASN_DNS_TYPE);
    wolfSSL_X509_add_altname(x509, "www.nomatch.com", ASN_DNS_TYPE);
    wolfSSL_X509_add_altname(x509, "www.info.wolfssl.com", ASN_DNS_TYPE);
    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "bad-multiple-constraint-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(ASN_NAME_INVALID_E));
    wolfSSL_X509_free(x509);
    x509 = NULL;

    /* check fail case when neither constraint is matched */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"common", 6, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);

    wolfSSL_X509_add_altname(x509, "www.random.com", ASN_DNS_TYPE);
    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "bad-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(ASN_NAME_INVALID_E));

    wolfSSL_CertManagerFree(cm);
    wolfSSL_X509_free(x509);
    wolfSSL_X509_free(ca);
    wolfSSL_EVP_PKEY_free(priv);
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_CertManagerNameConstraint5(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && \
    !defined(NO_WOLFSSL_CM_VERIFY) && !defined(NO_RSA) && \
    defined(OPENSSL_EXTRA) && defined(WOLFSSL_CERT_GEN) && \
    defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_ALT_NAMES) && \
    !defined(NO_SHA256)
    WOLFSSL_CERT_MANAGER* cm = NULL;
    WOLFSSL_EVP_PKEY *priv = NULL;
    WOLFSSL_X509_NAME* name = NULL;
    const char* ca_cert = "./certs/test/cert-ext-ncmixed.der";
    const char* server_cert = "./certs/test/server-goodcn.pem";

    byte    *der = NULL;
    int     derSz;
    byte    *pt;
    WOLFSSL_X509 *x509 = NULL;
    WOLFSSL_X509 *ca = NULL;

    pt = (byte*)server_key_der_2048;
    ExpectNotNull(priv = wolfSSL_d2i_PrivateKey(EVP_PKEY_RSA, NULL,
                (const unsigned char**)&pt, sizeof_server_key_der_2048));

    ExpectNotNull(cm = wolfSSL_CertManagerNew());
    ExpectNotNull(ca = wolfSSL_X509_load_certificate_file(ca_cert,
                WOLFSSL_FILETYPE_ASN1));
    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(ca, &derSz)));
    DEBUG_WRITE_DER(der, derSz, "ca.der");

    ExpectIntEQ(wolfSSL_CertManagerLoadCABuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* check satisfying wolfssl.com constraint passes */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"example", 7, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);
    name = NULL;

    wolfSSL_X509_add_altname(x509, "good.example", ASN_DNS_TYPE);
    wolfSSL_X509_add_altname(x509, "facts@into.wolfssl.com", ASN_RFC822_TYPE);
    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "good-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    wolfSSL_X509_free(x509);
    x509 = NULL;

    /* fail with DNS check because of common name */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"wolfssl.com", 11, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);
    name = NULL;

    wolfSSL_X509_add_altname(x509, "example", ASN_DNS_TYPE);
    wolfSSL_X509_add_altname(x509, "facts@wolfssl.com", ASN_RFC822_TYPE);
    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "bad-cn-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(ASN_NAME_INVALID_E));
    wolfSSL_X509_free(x509);
    x509 = NULL;

    /* fail on permitted DNS name constraint */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);
    name = NULL;

    wolfSSL_X509_add_altname(x509, "www.example", ASN_DNS_TYPE);
    wolfSSL_X509_add_altname(x509, "www.wolfssl", ASN_DNS_TYPE);
    wolfSSL_X509_add_altname(x509, "info@wolfssl.com", ASN_RFC822_TYPE);
    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "bad-1st-constraint-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(ASN_NAME_INVALID_E));
    wolfSSL_X509_free(x509);
    x509 = NULL;

    /* fail on permitted email name constraint */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);
    name = NULL;

    wolfSSL_X509_add_altname(x509, "example", ASN_DNS_TYPE);
    wolfSSL_X509_add_altname(x509, "info@wolfssl.com", ASN_RFC822_TYPE);
    wolfSSL_X509_add_altname(x509, "info@example.com", ASN_RFC822_TYPE);
    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "bad-2nd-constraint-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(ASN_NAME_INVALID_E));
    wolfSSL_X509_free(x509);
    x509 = NULL;

    /* success with empty email name */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);

    wolfSSL_X509_add_altname(x509, "example", ASN_DNS_TYPE);
    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "good-missing-constraint-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    wolfSSL_X509_free(x509);

    wolfSSL_CertManagerFree(cm);
    wolfSSL_X509_free(ca);
    wolfSSL_EVP_PKEY_free(priv);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_CRL_duplicate_extensions(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_ASN_TEMPLATE) && !defined(NO_CERTS) && \
    defined(HAVE_CRL) && !defined(NO_RSA) && !defined(WOLFSSL_NO_ASN_STRICT) && \
    (defined(WC_ASN_RUNTIME_DATE_CHECK_CONTROL) || defined(NO_ASN_TIME_CHECK))
    const unsigned char crl_duplicate_akd[] =
        "-----BEGIN X509 CRL-----\n"
        "MIICCDCB8QIBATANBgkqhkiG9w0BAQsFADB5MQswCQYDVQQGEwJVUzETMBEGA1UE\n"
        "CAwKQ2FsaWZvcm5pYTEWMBQGA1UEBwwNU2FuIEZyYW5jaXNjbzETMBEGA1UECgwK\n"
        "TXkgQ29tcGFueTETMBEGA1UEAwwKTXkgUm9vdCBDQTETMBEGA1UECwwKTXkgUm9v\n"
        "dCBDQRcNMjQwOTAxMDAwMDAwWhcNMjUxMjAxMDAwMDAwWqBEMEIwHwYDVR0jBBgw\n"
        "FoAU72ng99Ud5pns3G3Q9+K5XGRxgzUwHwYDVR0jBBgwFoAU72ng99Ud5pns3G3Q\n"
        "9+K5XGRxgzUwDQYJKoZIhvcNAQELBQADggEBAIFVw4jrS4taSXR/9gPzqGrqFeHr\n"
        "IXCnFtHJTLxqa8vUOAqSwqysvNpepVKioMVoGrLjFMjANjWQqTEiMROAnLfJ/+L8\n"
        "FHZkV/mZwOKAXMhIC9MrJzifxBICwmvD028qnwQm09EP8z4ICZptD6wPdRTDzduc\n"
        "KBuAX+zn8pNrJgyrheRKpPgno9KsbCzK4D/RIt1sTK2M3vVOtY+vpsN70QYUXvQ4\n"
        "r2RZac3omlT43x5lddPxIlcouQpwWcVvr/K+Va770MRrjn88PBrJmvsEw/QYVBXp\n"
        "Gxv2b78HFDacba80sMIm8ltRdqUCa5qIc6OATsz7izCQXEbkTEeESrcK1MA=\n"
        "-----END X509 CRL-----\n";

    WOLFSSL_CERT_MANAGER* cm = NULL;
    int ret;

    (void)wc_AsnSetSkipDateCheck(1);

    cm = wolfSSL_CertManagerNew();
    ExpectNotNull(cm);

    /* Test loading CRL with duplicate extensions */
    WOLFSSL_MSG("Testing CRL with duplicate Authority Key Identifier extensions");
    ret = wolfSSL_CertManagerLoadCRLBuffer(cm, crl_duplicate_akd,
                                           sizeof(crl_duplicate_akd),
                                           WOLFSSL_FILETYPE_PEM);
    ExpectIntEQ(ret, ASN_PARSE_E);

    wolfSSL_CertManagerFree(cm);

    (void)wc_AsnSetSkipDateCheck(0);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_CertManagerCRL(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && defined(HAVE_CRL) && \
    !defined(NO_RSA)
    const char* ca_cert = "./certs/ca-cert.pem";
    const char* crl1     = "./certs/crl/crl.pem";
    const char* crl2     = "./certs/crl/crl2.pem";
#ifdef WC_RSA_PSS
    const char* crl_rsapss = "./certs/crl/crl_rsapss.pem";
    const char* ca_rsapss  = "./certs/rsapss/ca-rsapss.pem";
#endif
    /* ./certs/crl/crl.der */
    const unsigned char crl_buff[] = {
        0x30, 0x82, 0x02, 0x04, 0x30, 0x81, 0xED, 0x02,
        0x01, 0x01, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86,
        0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x0B, 0x05,
        0x00, 0x30, 0x81, 0x94, 0x31, 0x0B, 0x30, 0x09,
        0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55,
        0x53, 0x31, 0x10, 0x30, 0x0E, 0x06, 0x03, 0x55,
        0x04, 0x08, 0x0C, 0x07, 0x4D, 0x6F, 0x6E, 0x74,
        0x61, 0x6E, 0x61, 0x31, 0x10, 0x30, 0x0E, 0x06,
        0x03, 0x55, 0x04, 0x07, 0x0C, 0x07, 0x42, 0x6F,
        0x7A, 0x65, 0x6D, 0x61, 0x6E, 0x31, 0x11, 0x30,
        0x0F, 0x06, 0x03, 0x55, 0x04, 0x0A, 0x0C, 0x08,
        0x53, 0x61, 0x77, 0x74, 0x6F, 0x6F, 0x74, 0x68,
        0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04,
        0x0B, 0x0C, 0x0A, 0x43, 0x6F, 0x6E, 0x73, 0x75,
        0x6C, 0x74, 0x69, 0x6E, 0x67, 0x31, 0x18, 0x30,
        0x16, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0C, 0x0F,
        0x77, 0x77, 0x77, 0x2E, 0x77, 0x6F, 0x6C, 0x66,
        0x73, 0x73, 0x6C, 0x2E, 0x63, 0x6F, 0x6D, 0x31,
        0x1F, 0x30, 0x1D, 0x06, 0x09, 0x2A, 0x86, 0x48,
        0x86, 0xF7, 0x0D, 0x01, 0x09, 0x01, 0x16, 0x10,
        0x69, 0x6E, 0x66, 0x6F, 0x40, 0x77, 0x6F, 0x6C,
        0x66, 0x73, 0x73, 0x6C, 0x2E, 0x63, 0x6F, 0x6D,
        0x17, 0x0D, 0x32, 0x34, 0x30, 0x31, 0x30, 0x39,
        0x30, 0x30, 0x33, 0x34, 0x33, 0x30, 0x5A, 0x17,
        0x0D, 0x32, 0x36, 0x31, 0x30, 0x30, 0x35, 0x30,
        0x30, 0x33, 0x34, 0x33, 0x30, 0x5A, 0x30, 0x14,
        0x30, 0x12, 0x02, 0x01, 0x02, 0x17, 0x0D, 0x32,
        0x34, 0x30, 0x31, 0x30, 0x39, 0x30, 0x30, 0x33,
        0x34, 0x33, 0x30, 0x5A, 0xA0, 0x0E, 0x30, 0x0C,
        0x30, 0x0A, 0x06, 0x03, 0x55, 0x1D, 0x14, 0x04,
        0x03, 0x02, 0x01, 0x02, 0x30, 0x0D, 0x06, 0x09,
        0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01,
        0x0B, 0x05, 0x00, 0x03, 0x82, 0x01, 0x01, 0x00,
        0xB3, 0x6F, 0xED, 0x72, 0xD2, 0x73, 0x6A, 0x77,
        0xBF, 0x3A, 0x55, 0xBC, 0x54, 0x18, 0x6A, 0x71,
        0xBC, 0x6A, 0xCC, 0xCD, 0x5D, 0x90, 0xF5, 0x64,
        0x8D, 0x1B, 0xF0, 0xE0, 0x48, 0x7B, 0xF2, 0x7B,
        0x06, 0x86, 0x53, 0x63, 0x9B, 0xD8, 0x24, 0x15,
        0x10, 0xB1, 0x19, 0x96, 0x9B, 0xD2, 0x75, 0xA8,
        0x25, 0xA2, 0x35, 0xA9, 0x14, 0xD6, 0xD5, 0x5E,
        0x53, 0xE3, 0x34, 0x9D, 0xF2, 0x8B, 0x07, 0x19,
        0x9B, 0x1F, 0xF1, 0x02, 0x0F, 0x04, 0x46, 0xE8,
        0xB8, 0xB6, 0xF2, 0x8D, 0xC7, 0xC0, 0x15, 0x3E,
        0x3E, 0x8E, 0x96, 0x73, 0x15, 0x1E, 0x62, 0xF6,
        0x4E, 0x2A, 0xF7, 0xAA, 0xA0, 0x91, 0x80, 0x12,
        0x7F, 0x81, 0x0C, 0x65, 0xCC, 0x38, 0xBE, 0x58,
        0x6C, 0x14, 0xA5, 0x21, 0xA1, 0x8D, 0xF7, 0x8A,
        0xB9, 0x24, 0xF4, 0x2D, 0xCA, 0xC0, 0x67, 0x43,
        0x0B, 0xC8, 0x1C, 0xB4, 0x7D, 0x12, 0x7F, 0xA2,
        0x1B, 0x19, 0x0E, 0x94, 0xCF, 0x7B, 0x9F, 0x75,
        0xA0, 0x08, 0x9A, 0x67, 0x3F, 0x87, 0x89, 0x3E,
        0xF8, 0x58, 0xA5, 0x8A, 0x1B, 0x2D, 0xDA, 0x9B,
        0xD0, 0x1B, 0x18, 0x92, 0xC3, 0xD2, 0x6A, 0xD7,
        0x1C, 0xFC, 0x45, 0x69, 0x77, 0xC3, 0x57, 0x65,
        0x75, 0x99, 0x9E, 0x47, 0x2A, 0x20, 0x25, 0xEF,
        0x90, 0xF2, 0x5F, 0x3B, 0x7D, 0x9C, 0x7D, 0x00,
        0xEA, 0x92, 0x54, 0xEB, 0x0B, 0xE7, 0x17, 0xAF,
        0x24, 0x1A, 0xF9, 0x7C, 0x83, 0x50, 0x68, 0x1D,
        0xDC, 0x5B, 0x60, 0x12, 0xA7, 0x52, 0x78, 0xD9,
        0xA9, 0xB0, 0x1F, 0x59, 0x48, 0x36, 0xC7, 0xA6,
        0x97, 0x34, 0xC7, 0x87, 0x3F, 0xAE, 0xFD, 0xA9,
        0x56, 0x5D, 0x48, 0xCC, 0x89, 0x7A, 0x79, 0x60,
        0x8F, 0x9B, 0x2B, 0x63, 0x3C, 0xB3, 0x04, 0x1D,
        0x5F, 0xF7, 0x20, 0xD2, 0xFD, 0xF2, 0x51, 0xB1,
        0x96, 0x93, 0x13, 0x5B, 0xAB, 0x74, 0x82, 0x8B
    };

    WOLFSSL_CERT_MANAGER* cm = NULL;

    ExpectNotNull(cm = wolfSSL_CertManagerNew());

    ExpectIntEQ(wolfSSL_CertManagerEnableCRL(NULL, 0), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerEnableCRL(cm, WOLFSSL_CRL_CHECKALL), 1);
    ExpectIntEQ(wolfSSL_CertManagerEnableCRL(cm, WOLFSSL_CRL_CHECK), 1);
    ExpectIntEQ(wolfSSL_CertManagerEnableCRL(cm,
        WOLFSSL_CRL_CHECK | WOLFSSL_CRL_CHECKALL), 1);
    ExpectIntEQ(wolfSSL_CertManagerEnableCRL(cm, 16), 1);
    ExpectIntEQ(wolfSSL_CertManagerEnableCRL(cm, WOLFSSL_CRL_CHECKALL), 1);

    ExpectIntEQ(wolfSSL_CertManagerCheckCRL(NULL, NULL, -1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerCheckCRL(cm, NULL, -1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerCheckCRL(NULL, server_cert_der_2048, -1),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerCheckCRL(NULL, NULL, 1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerCheckCRL(NULL, server_cert_der_2048, 1),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerCheckCRL(cm, NULL, 1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerCheckCRL(cm, server_cert_der_2048, -1),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerCheckCRL(cm, server_cert_der_2048,
        sizeof_server_cert_der_2048), WC_NO_ERR_TRACE(ASN_NO_SIGNER_E));

    ExpectIntEQ(wolfSSL_CertManagerSetCRL_Cb(NULL, NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerSetCRL_Cb(cm, NULL), 1);
#ifdef HAVE_CRL_IO
    ExpectIntEQ(wolfSSL_CertManagerSetCRL_IOCb(NULL, NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerSetCRL_IOCb(cm, NULL), 1);
#endif

#ifndef NO_FILESYSTEM
    ExpectIntEQ(wolfSSL_CertManagerLoadCRL(NULL, NULL, WOLFSSL_FILETYPE_ASN1,
        0), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerLoadCRL(cm, NULL, WOLFSSL_FILETYPE_ASN1,
        0), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    /* -1 seen as !WOLFSSL_FILETYPE_PEM */
    ExpectIntEQ(wolfSSL_CertManagerLoadCRL(cm, "./certs/crl", -1, 0), 1);

    ExpectIntEQ(wolfSSL_CertManagerLoadCRLFile(NULL, NULL,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerLoadCRLFile(cm, NULL, WOLFSSL_FILETYPE_ASN1),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    /* -1 seen as !WOLFSSL_FILETYPE_PEM */
    ExpectIntEQ(wolfSSL_CertManagerLoadCRLFile(cm, "./certs/crl/crl.pem", -1),
        WC_NO_ERR_TRACE(ASN_PARSE_E));
#endif

    ExpectIntEQ(wolfSSL_CertManagerLoadCRLBuffer(NULL, NULL, -1,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerLoadCRLBuffer(cm, NULL, -1,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerLoadCRLBuffer(NULL, crl_buff, -1,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerLoadCRLBuffer(NULL, NULL, 1,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerLoadCRLBuffer(NULL, crl_buff, 1,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerLoadCRLBuffer(cm, NULL, 1,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerLoadCRLBuffer(cm, crl_buff, -1,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    ExpectIntEQ(wolfSSL_CertManagerFreeCRL(NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    DoExpectIntEQ(wolfSSL_CertManagerFreeCRL(cm), 1);

    ExpectIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CertManagerLoadCA(cm, ca_cert, NULL));
    ExpectIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CertManagerLoadCRL(cm, crl1, WOLFSSL_FILETYPE_PEM, 0));
    ExpectIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CertManagerLoadCRL(cm, crl2, WOLFSSL_FILETYPE_PEM, 0));
    wolfSSL_CertManagerFreeCRL(cm);

#ifndef WOLFSSL_CRL_ALLOW_MISSING_CDP
    ExpectIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CertManagerLoadCRL(cm, crl1, WOLFSSL_FILETYPE_PEM, 0));
    ExpectIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CertManagerLoadCA(cm, ca_cert, NULL));
    ExpectIntEQ(wolfSSL_CertManagerCheckCRL(cm, server_cert_der_2048,
        sizeof_server_cert_der_2048), WC_NO_ERR_TRACE(CRL_MISSING));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, server_cert_der_2048,
        sizeof_server_cert_der_2048, WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(CRL_MISSING));
#endif /* !WOLFSSL_CRL_ALLOW_MISSING_CDP */

    ExpectIntEQ(wolfSSL_CertManagerLoadCRLBuffer(cm, crl_buff, sizeof(crl_buff),
        WOLFSSL_FILETYPE_ASN1), 1);

#if !defined(NO_FILESYSTEM) && defined(WC_RSA_PSS)
    /* loading should fail without the CA set */
    ExpectIntEQ(wolfSSL_CertManagerLoadCRLFile(cm, crl_rsapss,
        WOLFSSL_FILETYPE_PEM), WC_NO_ERR_TRACE(ASN_CRL_NO_SIGNER_E));

    /* now successfully load the RSA-PSS crl once loading in it's CA */
    ExpectIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CertManagerLoadCA(cm, ca_rsapss, NULL));
    ExpectIntEQ(wolfSSL_CertManagerLoadCRLFile(cm, crl_rsapss,
        WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);
#endif

    wolfSSL_CertManagerFree(cm);
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_CertManagerCheckOCSPResponse(void)
{
    EXPECT_DECLS;
#if defined(HAVE_OCSP) && !defined(NO_RSA) && !defined(NO_SHA)
/* Need one of these for wolfSSL_OCSP_REQUEST_new. */
#if defined(OPENSSL_ALL) || defined(WOLFSSL_NGINX) || \
    defined(WOLFSSL_HAPROXY) || defined(WOLFSSL_APACHE_HTTPD) || \
    defined(HAVE_LIGHTY)
    WOLFSSL_CERT_MANAGER* cm = NULL;
    /* Raw OCSP response bytes captured using the following setup:
     * - Run responder with
     *      openssl ocsp -port 9999 -ndays 9999
     *      -index certs/ocsp/index-intermediate1-ca-issued-certs.txt
     *      -rsigner certs/ocsp/ocsp-responder-cert.pem
     *      -rkey certs/ocsp/ocsp-responder-key.pem
     *      -CA certs/ocsp/intermediate1-ca-cert.pem
     * - Run client with
     *      openssl ocsp -host 127.0.0.1:9999 -respout resp.out
     *      -issuer certs/ocsp/intermediate1-ca-cert.pem
     *      -cert certs/ocsp/server1-cert.pem
     *      -CAfile certs/ocsp/root-ca-cert.pem -noverify
     * - Select the response packet in Wireshark, and export it using
     *   "File->Export Packet Dissection->As "C" Arrays".  Select "Selected
     *   packets only".  After importing into the editor, remove the initial
     *   ~148 bytes of header, ending with the Content-Length and the \r\n\r\n.
     */
    static const byte response[] = {
        0x30, 0x82, 0x07, 0x40, /* ....0..@ */
        0x0a, 0x01, 0x00, 0xa0, 0x82, 0x07, 0x39, 0x30, /* ......90 */
        0x82, 0x07, 0x35, 0x06, 0x09, 0x2b, 0x06, 0x01, /* ..5..+.. */
        0x05, 0x05, 0x07, 0x30, 0x01, 0x01, 0x04, 0x82, /* ...0.... */
        0x07, 0x26, 0x30, 0x82, 0x07, 0x22, 0x30, 0x82, /* .&0.."0. */
        0x01, 0x40, 0xa1, 0x81, 0xa1, 0x30, 0x81, 0x9e, /* .@...0.. */
        0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, /* 1.0...U. */
        0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x13, 0x30, /* ...US1.0 */
        0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c, 0x0a, /* ...U.... */
        0x57, 0x61, 0x73, 0x68, 0x69, 0x6e, 0x67, 0x74, /* Washingt */
        0x6f, 0x6e, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, /* on1.0... */
        0x55, 0x04, 0x07, 0x0c, 0x07, 0x53, 0x65, 0x61, /* U....Sea */
        0x74, 0x74, 0x6c, 0x65, 0x31, 0x10, 0x30, 0x0e, /* ttle1.0. */
        0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c, 0x07, 0x77, /* ..U....w */
        0x6f, 0x6c, 0x66, 0x53, 0x53, 0x4c, 0x31, 0x14, /* olfSSL1. */
        0x30, 0x12, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x0c, /* 0...U... */
        0x0b, 0x45, 0x6e, 0x67, 0x69, 0x6e, 0x65, 0x65, /* .Enginee */
        0x72, 0x69, 0x6e, 0x67, 0x31, 0x1f, 0x30, 0x1d, /* ring1.0. */
        0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x16, 0x77, /* ..U....w */
        0x6f, 0x6c, 0x66, 0x53, 0x53, 0x4c, 0x20, 0x4f, /* olfSSL O */
        0x43, 0x53, 0x50, 0x20, 0x52, 0x65, 0x73, 0x70, /* CSP Resp */
        0x6f, 0x6e, 0x64, 0x65, 0x72, 0x31, 0x1f, 0x30, /* onder1.0 */
        0x1d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, /* ...*.H.. */
        0x0d, 0x01, 0x09, 0x01, 0x16, 0x10, 0x69, 0x6e, /* ......in */
        0x66, 0x6f, 0x40, 0x77, 0x6f, 0x6c, 0x66, 0x73, /* fo@wolfs */
        0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x18, 0x0f, /* sl.com.. */
        0x32, 0x30, 0x32, 0x34, 0x31, 0x32, 0x32, 0x30, /* 20241220 */
        0x31, 0x37, 0x30, 0x37, 0x30, 0x34, 0x5a, 0x30, /* 170704Z0 */
        0x64, 0x30, 0x62, 0x30, 0x3a, 0x30, 0x09, 0x06, /* d0b0:0.. */
        0x05, 0x2b, 0x0e, 0x03, 0x02, 0x1a, 0x05, 0x00, /* .+...... */
        0x04, 0x14, 0x71, 0x4d, 0x82, 0x23, 0x40, 0x59, /* ..qM.#@Y */
        0xc0, 0x96, 0xa1, 0x37, 0x43, 0xfa, 0x31, 0xdb, /* ...7C.1. */
        0xba, 0xb1, 0x43, 0x18, 0xda, 0x04, 0x04, 0x14, /* ..C..... */
        0x83, 0xc6, 0x3a, 0x89, 0x2c, 0x81, 0xf4, 0x02, /* ..:.,... */
        0xd7, 0x9d, 0x4c, 0xe2, 0x2a, 0xc0, 0x71, 0x82, /* ..L.*.q. */
        0x64, 0x44, 0xda, 0x0e, 0x02, 0x01, 0x05, 0x80, /* dD...... */
        0x00, 0x18, 0x0f, 0x32, 0x30, 0x32, 0x34, 0x31, /* ...20241 */
        0x32, 0x32, 0x30, 0x31, 0x37, 0x30, 0x37, 0x30, /* 22017070 */
        0x34, 0x5a, 0xa0, 0x11, 0x18, 0x0f, 0x32, 0x30, /* 4Z....20 */
        0x35, 0x32, 0x30, 0x35, 0x30, 0x36, 0x31, 0x37, /* 52050617 */
        0x30, 0x37, 0x30, 0x34, 0x5a, 0xa1, 0x23, 0x30, /* 0704Z.#0 */
        0x21, 0x30, 0x1f, 0x06, 0x09, 0x2b, 0x06, 0x01, /* !0...+.. */
        0x05, 0x05, 0x07, 0x30, 0x01, 0x02, 0x04, 0x12, /* ...0.... */
        0x04, 0x10, 0x12, 0x7c, 0x27, 0xbd, 0x22, 0x28, /* ...|'."( */
        0x5e, 0x62, 0x81, 0xed, 0x6d, 0x2c, 0x2d, 0x59, /* ^b..m,-Y */
        0x42, 0xd7, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, /* B.0...*. */
        0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, /* H....... */
        0x00, 0x03, 0x82, 0x01, 0x01, 0x00, 0x6c, 0xce, /* ......l. */
        0xa8, 0xe8, 0xfe, 0xaf, 0x33, 0xe2, 0xce, 0x4e, /* ....3..N */
        0x63, 0x8d, 0x61, 0x16, 0x0f, 0x70, 0xb2, 0x0c, /* c.a..p.. */
        0x9a, 0xe3, 0x01, 0xd5, 0xca, 0xe5, 0x9b, 0x70, /* .......p */
        0x81, 0x6f, 0x94, 0x09, 0xe8, 0x88, 0x98, 0x1a, /* .o...... */
        0x67, 0xa0, 0xc2, 0xe7, 0x8f, 0x9b, 0x5f, 0x13, /* g....._. */
        0x17, 0x8d, 0x93, 0x8c, 0x31, 0x61, 0x7d, 0x72, /* ....1a}r */
        0x34, 0xbd, 0x21, 0x48, 0xca, 0xb2, 0xc9, 0xae, /* 4.!H.... */
        0x28, 0x5f, 0x97, 0x19, 0xcb, 0xdf, 0xed, 0xd4, /* (_...... */
        0x6e, 0x89, 0x30, 0x89, 0x11, 0xd1, 0x05, 0x08, /* n.0..... */
        0x81, 0xe9, 0xa7, 0xba, 0xf7, 0x16, 0x0c, 0xbe, /* ........ */
        0x48, 0x2e, 0xc0, 0x05, 0xac, 0x90, 0xc2, 0x35, /* H......5 */
        0xce, 0x6c, 0x94, 0x5d, 0x2b, 0xad, 0x4f, 0x19, /* .l.]+.O. */
        0xea, 0x7b, 0xd9, 0x4f, 0x49, 0x20, 0x8d, 0x98, /* .{.OI .. */
        0xa9, 0xe4, 0x53, 0x6d, 0xca, 0x34, 0xdb, 0x4a, /* ..Sm.4.J */
        0x28, 0xb3, 0x33, 0xfb, 0xfd, 0xcc, 0x4b, 0xfa, /* (.3...K. */
        0xdb, 0x70, 0xe1, 0x96, 0xc8, 0xd4, 0xf1, 0x85, /* .p...... */
        0x99, 0xaf, 0x06, 0xeb, 0xfd, 0x96, 0x21, 0x86, /* ......!. */
        0x81, 0xee, 0xcf, 0xd2, 0xf4, 0x83, 0xc9, 0x1d, /* ........ */
        0x8f, 0x42, 0xd1, 0xc1, 0xbc, 0x50, 0x0a, 0xfb, /* .B...P.. */
        0x95, 0x39, 0x4c, 0x36, 0xa8, 0xfe, 0x2b, 0x8e, /* .9L6..+. */
        0xc5, 0xb5, 0xe0, 0xab, 0xdb, 0xc0, 0xbf, 0x1d, /* ........ */
        0x35, 0x4d, 0xc0, 0x52, 0xfb, 0x08, 0x04, 0x4c, /* 5M.R...L */
        0x98, 0xf0, 0xb5, 0x5b, 0xff, 0x99, 0x74, 0xce, /* ...[..t. */
        0xb7, 0xc9, 0xe3, 0xe5, 0x70, 0x2e, 0xd3, 0x1d, /* ....p... */
        0x46, 0x38, 0xf9, 0x51, 0x17, 0x73, 0xd1, 0x08, /* F8.Q.s.. */
        0x8d, 0x3d, 0x12, 0x47, 0xd0, 0x66, 0x77, 0xaf, /* .=.G.fw. */
        0xfd, 0x4c, 0x75, 0x1f, 0xe9, 0x6c, 0xf4, 0x5a, /* .Lu..l.Z */
        0xde, 0xec, 0x37, 0xc7, 0xc4, 0x0a, 0xbe, 0x91, /* ..7..... */
        0xbc, 0x05, 0x08, 0x86, 0x47, 0x30, 0x2a, 0xc6, /* ....G0*. */
        0x85, 0x4b, 0x55, 0x6c, 0xef, 0xdf, 0x2d, 0x5a, /* .KUl..-Z */
        0xf7, 0x5b, 0xb5, 0xba, 0xed, 0x38, 0xb0, 0xcb, /* .[...8.. */
        0xeb, 0x7e, 0x84, 0x3a, 0x69, 0x2c, 0xa0, 0x82, /* .~.:i,.. */
        0x04, 0xc6, 0x30, 0x82, 0x04, 0xc2, 0x30, 0x82, /* ..0...0. */
        0x04, 0xbe, 0x30, 0x82, 0x03, 0xa6, 0xa0, 0x03, /* ..0..... */
        0x02, 0x01, 0x02, 0x02, 0x01, 0x04, 0x30, 0x0d, /* ......0. */
        0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, /* ..*.H... */
        0x01, 0x01, 0x0b, 0x05, 0x00, 0x30, 0x81, 0x97, /* .....0.. */
        0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, /* 1.0...U. */
        0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x13, 0x30, /* ...US1.0 */
        0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c, 0x0a, /* ...U.... */
        0x57, 0x61, 0x73, 0x68, 0x69, 0x6e, 0x67, 0x74, /* Washingt */
        0x6f, 0x6e, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, /* on1.0... */
        0x55, 0x04, 0x07, 0x0c, 0x07, 0x53, 0x65, 0x61, /* U....Sea */
        0x74, 0x74, 0x6c, 0x65, 0x31, 0x10, 0x30, 0x0e, /* ttle1.0. */
        0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c, 0x07, 0x77, /* ..U....w */
        0x6f, 0x6c, 0x66, 0x53, 0x53, 0x4c, 0x31, 0x14, /* olfSSL1. */
        0x30, 0x12, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x0c, /* 0...U... */
        0x0b, 0x45, 0x6e, 0x67, 0x69, 0x6e, 0x65, 0x65, /* .Enginee */
        0x72, 0x69, 0x6e, 0x67, 0x31, 0x18, 0x30, 0x16, /* ring1.0. */
        0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x77, /* ..U....w */
        0x6f, 0x6c, 0x66, 0x53, 0x53, 0x4c, 0x20, 0x72, /* olfSSL r */
        0x6f, 0x6f, 0x74, 0x20, 0x43, 0x41, 0x31, 0x1f, /* oot CA1. */
        0x30, 0x1d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, /* 0...*.H. */
        0xf7, 0x0d, 0x01, 0x09, 0x01, 0x16, 0x10, 0x69, /* .......i */
        0x6e, 0x66, 0x6f, 0x40, 0x77, 0x6f, 0x6c, 0x66, /* nfo@wolf */
        0x73, 0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x30, /* ssl.com0 */
        0x1e, 0x17, 0x0d, 0x32, 0x34, 0x31, 0x32, 0x31, /* ...24121 */
        0x38, 0x32, 0x31, 0x32, 0x35, 0x33, 0x31, 0x5a, /* 8212531Z */
        0x17, 0x0d, 0x32, 0x37, 0x30, 0x39, 0x31, 0x34, /* ..270914 */
        0x32, 0x31, 0x32, 0x35, 0x33, 0x31, 0x5a, 0x30, /* 212531Z0 */
        0x81, 0x9e, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, /* ..1.0... */
        0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, /* U....US1 */
        0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, /* .0...U.. */
        0x0c, 0x0a, 0x57, 0x61, 0x73, 0x68, 0x69, 0x6e, /* ..Washin */
        0x67, 0x74, 0x6f, 0x6e, 0x31, 0x10, 0x30, 0x0e, /* gton1.0. */
        0x06, 0x03, 0x55, 0x04, 0x07, 0x0c, 0x07, 0x53, /* ..U....S */
        0x65, 0x61, 0x74, 0x74, 0x6c, 0x65, 0x31, 0x10, /* eattle1. */
        0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c, /* 0...U... */
        0x07, 0x77, 0x6f, 0x6c, 0x66, 0x53, 0x53, 0x4c, /* .wolfSSL */
        0x31, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55, 0x04, /* 1.0...U. */
        0x0b, 0x0c, 0x0b, 0x45, 0x6e, 0x67, 0x69, 0x6e, /* ...Engin */
        0x65, 0x65, 0x72, 0x69, 0x6e, 0x67, 0x31, 0x1f, /* eering1. */
        0x30, 0x1d, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, /* 0...U... */
        0x16, 0x77, 0x6f, 0x6c, 0x66, 0x53, 0x53, 0x4c, /* .wolfSSL */
        0x20, 0x4f, 0x43, 0x53, 0x50, 0x20, 0x52, 0x65, /*  OCSP Re */
        0x73, 0x70, 0x6f, 0x6e, 0x64, 0x65, 0x72, 0x31, /* sponder1 */
        0x1f, 0x30, 0x1d, 0x06, 0x09, 0x2a, 0x86, 0x48, /* .0...*.H */
        0x86, 0xf7, 0x0d, 0x01, 0x09, 0x01, 0x16, 0x10, /* ........ */
        0x69, 0x6e, 0x66, 0x6f, 0x40, 0x77, 0x6f, 0x6c, /* info@wol */
        0x66, 0x73, 0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, /* fssl.com */
        0x30, 0x82, 0x01, 0x22, 0x30, 0x0d, 0x06, 0x09, /* 0.."0... */
        0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, /* *.H..... */
        0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0f, 0x00, /* ........ */
        0x30, 0x82, 0x01, 0x0a, 0x02, 0x82, 0x01, 0x01, /* 0....... */
        0x00, 0xb8, 0xba, 0x23, 0xb4, 0xf6, 0xc3, 0x7b, /* ...#...{ */
        0x14, 0xc3, 0xa4, 0xf5, 0x1d, 0x61, 0xa1, 0xf5, /* .....a.. */
        0x1e, 0x63, 0xb9, 0x85, 0x23, 0x34, 0x50, 0x6d, /* .c..#4Pm */
        0xf8, 0x7c, 0xa2, 0x8a, 0x04, 0x8b, 0xd5, 0x75, /* .|.....u */
        0x5c, 0x2d, 0xf7, 0x63, 0x88, 0xd1, 0x07, 0x7a, /* \-.c...z */
        0xea, 0x0b, 0x45, 0x35, 0x2b, 0xeb, 0x1f, 0xb1, /* ..E5+... */
        0x22, 0xb4, 0x94, 0x41, 0x38, 0xe2, 0x9d, 0x74, /* "..A8..t */
        0xd6, 0x8b, 0x30, 0x22, 0x10, 0x51, 0xc5, 0xdb, /* ..0".Q.. */
        0xca, 0x3f, 0x46, 0x2b, 0xfe, 0xe5, 0x5a, 0x3f, /* .?F+..Z? */
        0x41, 0x74, 0x67, 0x75, 0x95, 0xa9, 0x94, 0xd5, /* Atgu.... */
        0xc3, 0xee, 0x42, 0xf8, 0x8d, 0xeb, 0x92, 0x95, /* ..B..... */
        0xe1, 0xd9, 0x65, 0xb7, 0x43, 0xc4, 0x18, 0xde, /* ..e.C... */
        0x16, 0x80, 0x90, 0xce, 0x24, 0x35, 0x21, 0xc4, /* ....$5!. */
        0x55, 0xac, 0x5a, 0x51, 0xe0, 0x2e, 0x2d, 0xb3, /* U.ZQ..-. */
        0x0a, 0x5a, 0x4f, 0x4a, 0x73, 0x31, 0x50, 0xee, /* .ZOJs1P. */
        0x4a, 0x16, 0xbd, 0x39, 0x8b, 0xad, 0x05, 0x48, /* J..9...H */
        0x87, 0xb1, 0x99, 0xe2, 0x10, 0xa7, 0x06, 0x72, /* .......r */
        0x67, 0xca, 0x5c, 0xd1, 0x97, 0xbd, 0xc8, 0xf1, /* g.\..... */
        0x76, 0xf8, 0xe0, 0x4a, 0xec, 0xbc, 0x93, 0xf4, /* v..J.... */
        0x66, 0x4c, 0x28, 0x71, 0xd1, 0xd8, 0x66, 0x03, /* fL(q..f. */
        0xb4, 0x90, 0x30, 0xbb, 0x17, 0xb0, 0xfe, 0x97, /* ..0..... */
        0xf5, 0x1e, 0xe8, 0xc7, 0x5d, 0x9b, 0x8b, 0x11, /* ....]... */
        0x19, 0x12, 0x3c, 0xab, 0x82, 0x71, 0x78, 0xff, /* ..<..qx. */
        0xae, 0x3f, 0x32, 0xb2, 0x08, 0x71, 0xb2, 0x1b, /* .?2..q.. */
        0x8c, 0x27, 0xac, 0x11, 0xb8, 0xd8, 0x43, 0x49, /* .'....CI */
        0xcf, 0xb0, 0x70, 0xb1, 0xf0, 0x8c, 0xae, 0xda, /* ..p..... */
        0x24, 0x87, 0x17, 0x3b, 0xd8, 0x04, 0x65, 0x6c, /* $..;..el */
        0x00, 0x76, 0x50, 0xef, 0x15, 0x08, 0xd7, 0xb4, /* .vP..... */
        0x73, 0x68, 0x26, 0x14, 0x87, 0x95, 0xc3, 0x5f, /* sh&...._ */
        0x6e, 0x61, 0xb8, 0x87, 0x84, 0xfa, 0x80, 0x1a, /* na...... */
        0x0a, 0x8b, 0x98, 0xf3, 0xe3, 0xff, 0x4e, 0x44, /* ......ND */
        0x1c, 0x65, 0x74, 0x7c, 0x71, 0x54, 0x65, 0xe5, /* .et|qTe. */
        0x39, 0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x82, /* 9....... */
        0x01, 0x0a, 0x30, 0x82, 0x01, 0x06, 0x30, 0x09, /* ..0...0. */
        0x06, 0x03, 0x55, 0x1d, 0x13, 0x04, 0x02, 0x30, /* ..U....0 */
        0x00, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, /* .0...U.. */
        0x04, 0x16, 0x04, 0x14, 0x32, 0x67, 0xe1, 0xb1, /* ....2g.. */
        0x79, 0xd2, 0x81, 0xfc, 0x9f, 0x23, 0x0c, 0x70, /* y....#.p */
        0x40, 0x50, 0xb5, 0x46, 0x56, 0xb8, 0x30, 0x36, /* @P.FV.06 */
        0x30, 0x81, 0xc4, 0x06, 0x03, 0x55, 0x1d, 0x23, /* 0....U.# */
        0x04, 0x81, 0xbc, 0x30, 0x81, 0xb9, 0x80, 0x14, /* ...0.... */
        0x73, 0xb0, 0x1c, 0xa4, 0x2f, 0x82, 0xcb, 0xcf, /* s.../... */
        0x47, 0xa5, 0x38, 0xd7, 0xb0, 0x04, 0x82, 0x3a, /* G.8....: */
        0x7e, 0x72, 0x15, 0x21, 0xa1, 0x81, 0x9d, 0xa4, /* ~r.!.... */
        0x81, 0x9a, 0x30, 0x81, 0x97, 0x31, 0x0b, 0x30, /* ..0..1.0 */
        0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, /* ...U.... */
        0x55, 0x53, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, /* US1.0... */
        0x55, 0x04, 0x08, 0x0c, 0x0a, 0x57, 0x61, 0x73, /* U....Was */
        0x68, 0x69, 0x6e, 0x67, 0x74, 0x6f, 0x6e, 0x31, /* hington1 */
        0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x07, /* .0...U.. */
        0x0c, 0x07, 0x53, 0x65, 0x61, 0x74, 0x74, 0x6c, /* ..Seattl */
        0x65, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, /* e1.0...U */
        0x04, 0x0a, 0x0c, 0x07, 0x77, 0x6f, 0x6c, 0x66, /* ....wolf */
        0x53, 0x53, 0x4c, 0x31, 0x14, 0x30, 0x12, 0x06, /* SSL1.0.. */
        0x03, 0x55, 0x04, 0x0b, 0x0c, 0x0b, 0x45, 0x6e, /* .U....En */
        0x67, 0x69, 0x6e, 0x65, 0x65, 0x72, 0x69, 0x6e, /* gineerin */
        0x67, 0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55, /* g1.0...U */
        0x04, 0x03, 0x0c, 0x0f, 0x77, 0x6f, 0x6c, 0x66, /* ....wolf */
        0x53, 0x53, 0x4c, 0x20, 0x72, 0x6f, 0x6f, 0x74, /* SSL root */
        0x20, 0x43, 0x41, 0x31, 0x1f, 0x30, 0x1d, 0x06, /*  CA1.0.. */
        0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, /* .*.H.... */
        0x09, 0x01, 0x16, 0x10, 0x69, 0x6e, 0x66, 0x6f, /* ....info */
        0x40, 0x77, 0x6f, 0x6c, 0x66, 0x73, 0x73, 0x6c, /* @wolfssl */
        0x2e, 0x63, 0x6f, 0x6d, 0x82, 0x01, 0x63, 0x30, /* .com..c0 */
        0x13, 0x06, 0x03, 0x55, 0x1d, 0x25, 0x04, 0x0c, /* ...U.%.. */
        0x30, 0x0a, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, /* 0...+... */
        0x05, 0x07, 0x03, 0x09, 0x30, 0x0d, 0x06, 0x09, /* ....0... */
        0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, /* *.H..... */
        0x0b, 0x05, 0x00, 0x03, 0x82, 0x01, 0x01, 0x00, /* ........ */
        0x4d, 0xa2, 0xd8, 0x55, 0xe0, 0x2b, 0xf4, 0xad, /* M..U.+.. */
        0x65, 0xe2, 0x92, 0x35, 0xcb, 0x60, 0xa0, 0xa2, /* e..5.`.. */
        0x6b, 0xa6, 0x88, 0xc1, 0x86, 0x58, 0x57, 0x37, /* k....XW7 */
        0xbd, 0x2e, 0x28, 0x6e, 0x1c, 0x56, 0x2a, 0x35, /* ..(n.V*5 */
        0xde, 0xff, 0x3e, 0x8e, 0x3d, 0x47, 0x21, 0x1a, /* ..>.=G!. */
        0xe9, 0xd3, 0xc6, 0xb4, 0xe2, 0xcb, 0x3e, 0xc6, /* ......>. */
        0xaf, 0x9b, 0xef, 0x23, 0x88, 0x56, 0x95, 0x73, /* ...#.V.s */
        0x2e, 0xb3, 0xed, 0xc5, 0x11, 0x4b, 0x69, 0xf7, /* .....Ki. */
        0x13, 0x3a, 0x05, 0xe1, 0xaf, 0xba, 0xc9, 0x59, /* .:.....Y */
        0xfd, 0xe2, 0xa0, 0x81, 0xa0, 0x4c, 0x0c, 0x2c, /* .....L., */
        0xcb, 0x57, 0xad, 0x96, 0x3a, 0x8c, 0x32, 0xa6, /* .W..:.2. */
        0x4a, 0xf8, 0x72, 0xb8, 0xec, 0xb3, 0x26, 0x69, /* J.r...&i */
        0xd6, 0x6a, 0x4c, 0x4c, 0x78, 0x18, 0x3c, 0xca, /* .jLLx.<. */
        0x19, 0xf1, 0xb5, 0x8e, 0x23, 0x81, 0x5b, 0x27, /* ....#.[' */
        0x90, 0xe0, 0x5c, 0x2b, 0x17, 0x4d, 0x78, 0x99, /* ..\+.Mx. */
        0x6b, 0x25, 0xbd, 0x2f, 0xae, 0x1b, 0xaa, 0xce, /* k%./.... */
        0x84, 0xb9, 0x44, 0x21, 0x46, 0xc0, 0x34, 0x6b, /* ..D!F.4k */
        0x5b, 0xb9, 0x1b, 0xca, 0x5c, 0x60, 0xf1, 0xef, /* [...\`.. */
        0xe6, 0x66, 0xbc, 0x84, 0x63, 0x56, 0x50, 0x7d, /* .f..cVP} */
        0xbb, 0x2c, 0x2f, 0x7b, 0x47, 0xb4, 0xfd, 0x58, /* .,/{G..X */
        0x77, 0x87, 0xee, 0x27, 0x20, 0x96, 0x72, 0x8e, /* w..' .r. */
        0x4c, 0x7e, 0x4f, 0x93, 0xeb, 0x5f, 0x8f, 0x9c, /* L~O.._.. */
        0x1e, 0x59, 0x7a, 0x96, 0xaa, 0x53, 0x77, 0x22, /* .Yz..Sw" */
        0x41, 0xd8, 0xd3, 0xf9, 0x89, 0x8f, 0xe8, 0x9d, /* A....... */
        0x65, 0xbd, 0x0c, 0x71, 0x3c, 0xbb, 0xa3, 0x07, /* e..q<... */
        0xbf, 0xfb, 0xa8, 0xd1, 0x18, 0x0a, 0xb4, 0xc4, /* ........ */
        0xf7, 0x83, 0xb3, 0x86, 0x2b, 0xf0, 0x5b, 0x05, /* ....+.[. */
        0x28, 0xc1, 0x01, 0x31, 0x73, 0x5c, 0x2b, 0xbd, /* (..1s\+. */
        0x60, 0x97, 0xa3, 0x36, 0x82, 0x96, 0xd7, 0x83, /* `..6.... */
        0xdf, 0x75, 0xee, 0x29, 0x42, 0x97, 0x86, 0x41, /* .u.)B..A */
        0x55, 0xb9, 0x70, 0x87, 0xd5, 0x02, 0x85, 0x13, /* U.p..... */
        0x41, 0xf8, 0x25, 0x05, 0xab, 0x6a, 0xaa, 0x57  /* A.%..j.W */
    };
    OcspEntry entry[1];
    CertStatus status[1];
    OcspRequest* request = NULL;
#ifndef NO_FILESYSTEM
    const char* ca_cert = "./certs/ca-cert.pem";
#endif

    byte serial[] = {0x05};
    byte issuerHash[] = {0x71, 0x4d, 0x82, 0x23, 0x40, 0x59, 0xc0, 0x96, 0xa1, 0x37, 0x43, 0xfa, 0x31, 0xdb, 0xba, 0xb1, 0x43, 0x18, 0xda, 0x04};
    byte issuerKeyHash[] = {0x83, 0xc6, 0x3a, 0x89, 0x2c, 0x81, 0xf4, 0x02, 0xd7, 0x9d, 0x4c, 0xe2, 0x2a, 0xc0, 0x71, 0x82, 0x64, 0x44, 0xda, 0x0e};


    XMEMSET(entry, 0, sizeof(OcspEntry));
    XMEMSET(status, 0, sizeof(CertStatus));

    ExpectNotNull(request = wolfSSL_OCSP_REQUEST_new());
    ExpectNotNull(request->serial = (byte*)XMALLOC(sizeof(serial), NULL,
                                                   DYNAMIC_TYPE_OCSP_REQUEST));

    if ((request != NULL) && (request->serial != NULL)) {
        request->serialSz = sizeof(serial);
        XMEMCPY(request->serial, serial, sizeof(serial));
        XMEMCPY(request->issuerHash, issuerHash, sizeof(issuerHash));
        XMEMCPY(request->issuerKeyHash, issuerKeyHash, sizeof(issuerKeyHash));
    }

    ExpectNotNull(cm = wolfSSL_CertManagerNew_ex(NULL));
    ExpectIntEQ(wolfSSL_CertManagerEnableOCSP(cm, 0), WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_CertManagerLoadCA(cm,
        "./certs/ocsp/intermediate1-ca-cert.pem", NULL), WOLFSSL_SUCCESS);

    /* Response should be valid. */
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSPResponse(cm, (byte *)response,
        sizeof(response), NULL, status, entry, request), WOLFSSL_SUCCESS);

    /* Flip a byte in the request serial number, response should be invalid
     * now. */
    if ((request != NULL) && (request->serial != NULL))
        request->serial[0] ^= request->serial[0];
    ExpectIntNE(wolfSSL_CertManagerCheckOCSPResponse(cm, (byte *)response,
        sizeof(response), NULL, status, entry, request), WOLFSSL_SUCCESS);

#ifndef NO_FILESYSTEM
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSP(cm, server_cert_der_2048,
        sizeof(server_cert_der_2048)), WC_NO_ERR_TRACE(ASN_NO_SIGNER_E));
    ExpectIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CertManagerLoadCA(cm, ca_cert, NULL));
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSP(cm, server_cert_der_2048,
        sizeof(server_cert_der_2048)), 1);
#endif

    wolfSSL_OCSP_REQUEST_free(request);
    wolfSSL_CertManagerFree(cm);
#endif /* OPENSSL_ALL || WOLFSSL_NGINX ||  WOLFSSL_HAPROXY ||
        * WOLFSSL_APACHE_HTTPD || HAVE_LIGHTY */
#endif /* HAVE_OCSP */
    return EXPECT_RESULT();
}

static int test_wolfSSL_CheckOCSPResponse(void)
{
    EXPECT_DECLS;
#if defined(HAVE_OCSP) && defined(OPENSSL_EXTRA) && \
    !defined(NO_RSA) && !defined(NO_SHA)
    const char* responseFile = "./certs/ocsp/test-response.der";
    const char* responseMultiFile = "./certs/ocsp/test-multi-response.der";
    const char* responseNoInternFile =
        "./certs/ocsp/test-response-nointern.der";
    const char* caFile = "./certs/ocsp/root-ca-cert.pem";
    OcspResponse* res = NULL;
    byte data[4096];
    const unsigned char* pt;
    int  dataSz = 0; /* initialize to mitigate spurious maybe-uninitialized from
                      * gcc sanitizer with --enable-heapmath.
                      */
    XFILE f = XBADFILE;
    WOLFSSL_OCSP_BASICRESP* bs = NULL;
    WOLFSSL_X509_STORE* st = NULL;
    WOLFSSL_X509* issuer = NULL;


    ExpectTrue((f = XFOPEN(responseFile, "rb")) != XBADFILE);
    ExpectIntGT(dataSz = (word32)XFREAD(data, 1, sizeof(data), f), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }

    pt = data;
    ExpectNotNull(res = wolfSSL_d2i_OCSP_RESPONSE(NULL, &pt, dataSz));
    ExpectNotNull(issuer = wolfSSL_X509_load_certificate_file(caFile,
        SSL_FILETYPE_PEM));
    ExpectNotNull(st = wolfSSL_X509_STORE_new());
    ExpectIntEQ(wolfSSL_X509_STORE_add_cert(st, issuer), WOLFSSL_SUCCESS);
    ExpectNotNull(bs = wolfSSL_OCSP_response_get1_basic(res));
    ExpectIntEQ(wolfSSL_OCSP_basic_verify(bs, NULL, st, 0), WOLFSSL_SUCCESS);
    wolfSSL_OCSP_BASICRESP_free(bs);
    bs = NULL;
    wolfSSL_OCSP_RESPONSE_free(res);
    res = NULL;
    wolfSSL_X509_STORE_free(st);
    st = NULL;
    wolfSSL_X509_free(issuer);
    issuer = NULL;

    /* check loading a response with optional certs */
    ExpectTrue((f = XFOPEN(responseNoInternFile, "rb")) != XBADFILE);
    ExpectIntGT(dataSz = (word32)XFREAD(data, 1, sizeof(data), f), 0);
    if (f != XBADFILE)
        XFCLOSE(f);
    f = XBADFILE;

    pt = data;
    ExpectNotNull(res = wolfSSL_d2i_OCSP_RESPONSE(NULL, &pt, dataSz));
    wolfSSL_OCSP_RESPONSE_free(res);
    res = NULL;

    /* check loading a response with multiple certs */
    {
        WOLFSSL_CERT_MANAGER* cm = NULL;
        OcspEntry *entry = NULL;
        CertStatus* status = NULL;
        OcspRequest* request = NULL;

        byte serial1[] = {0x01};
        byte serial[] = {0x02};

        byte issuerHash[] = {
            0x44, 0xA8, 0xDB, 0xD1, 0xBC, 0x97, 0x0A, 0x83,
            0x3B, 0x5B, 0x31, 0x9A, 0x4C, 0xB8, 0xD2, 0x52,
            0x37, 0x15, 0x8A, 0x88
        };
        byte issuerKeyHash[] = {
            0x73, 0xB0, 0x1C, 0xA4, 0x2F, 0x82, 0xCB, 0xCF,
            0x47, 0xA5, 0x38, 0xD7, 0xB0, 0x04, 0x82, 0x3A,
            0x7E, 0x72, 0x15, 0x21
        };

        ExpectNotNull(entry = (OcspEntry*)XMALLOC(sizeof(OcspEntry), NULL,
             DYNAMIC_TYPE_OPENSSL));

        ExpectNotNull(status = (CertStatus*)XMALLOC(sizeof(CertStatus), NULL,
             DYNAMIC_TYPE_OPENSSL));

        if (entry != NULL)
            XMEMSET(entry, 0, sizeof(OcspEntry));
        if (status != NULL)
            XMEMSET(status, 0, sizeof(CertStatus));

        ExpectNotNull(request = wolfSSL_OCSP_REQUEST_new());
        ExpectNotNull(request->serial = (byte*)XMALLOC(sizeof(serial), NULL,
                                                    DYNAMIC_TYPE_OCSP_REQUEST));

        if (request != NULL && request->serial != NULL) {
             request->serialSz = sizeof(serial);
             XMEMCPY(request->serial, serial, sizeof(serial));
             XMEMCPY(request->issuerHash, issuerHash, sizeof(issuerHash));
             XMEMCPY(request->issuerKeyHash, issuerKeyHash,
                 sizeof(issuerKeyHash));
        }

        ExpectNotNull(cm = wolfSSL_CertManagerNew_ex(NULL));
        ExpectIntEQ(wolfSSL_CertManagerEnableOCSP(cm, 0), WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_CertManagerLoadCA(cm, caFile, NULL),
            WOLFSSL_SUCCESS);

        ExpectTrue((f = XFOPEN(responseMultiFile, "rb")) != XBADFILE);
        ExpectIntGT(dataSz = (word32)XFREAD(data, 1, sizeof(data), f), 0);
        if (f != XBADFILE)
            XFCLOSE(f);
        f = XBADFILE;

        ExpectIntEQ(wolfSSL_CertManagerCheckOCSPResponse(cm, data,
            dataSz, NULL, status, entry, request), WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_CertManagerCheckOCSPResponse(cm, data,
            dataSz, NULL, entry->status, entry, request), WOLFSSL_SUCCESS);
        ExpectNotNull(entry->status);

        if (request != NULL && request->serial != NULL)
            XMEMCPY(request->serial, serial1, sizeof(serial1));
        ExpectIntEQ(wolfSSL_CertManagerCheckOCSPResponse(cm, data,
            dataSz, NULL, status, entry, request), WOLFSSL_SUCCESS);

        /* store both status's in the entry to check that "next" is not
         * overwritten */
        if (EXPECT_SUCCESS() && status != NULL && entry != NULL) {
            status->next  = entry->status;
            entry->status = status;
        }

        if (request != NULL && request->serial != NULL)
            XMEMCPY(request->serial, serial, sizeof(serial));
        ExpectIntEQ(wolfSSL_CertManagerCheckOCSPResponse(cm, data,
            dataSz, NULL, entry->status, entry, request), WOLFSSL_SUCCESS);
        ExpectNotNull(entry->status->next);

        /* compare the status found */
        ExpectIntEQ(status->serialSz, entry->status->serialSz);
        ExpectIntEQ(XMEMCMP(status->serial, entry->status->serial,
            status->serialSz), 0);

        if (status != NULL && entry != NULL && entry->status != status) {
            XFREE(status, NULL, DYNAMIC_TYPE_OPENSSL);
        }
        wolfSSL_OCSP_CERTID_free(entry);
        wolfSSL_OCSP_REQUEST_free(request);
        wolfSSL_CertManagerFree(cm);
    }

/* FIPS v2 and below don't support long salts. */
#if defined(WC_RSA_PSS) && \
    (!defined(HAVE_FIPS) || (defined(HAVE_FIPS_VERSION) && \
     (HAVE_FIPS_VERSION > 2))) && (!defined(HAVE_SELFTEST) || \
     (defined(HAVE_SELFTEST_VERSION) && (HAVE_SELFTEST_VERSION > 2)))
    {
        const char* responsePssFile = "./certs/ocsp/test-response-rsapss.der";

        /* check loading a response with RSA-PSS signature */
        ExpectTrue((f = XFOPEN(responsePssFile, "rb")) != XBADFILE);
        ExpectIntGT(dataSz = (word32)XFREAD(data, 1, sizeof(data), f), 0);
        if (f != XBADFILE)
            XFCLOSE(f);

        pt = data;
        ExpectNotNull(res = wolfSSL_d2i_OCSP_RESPONSE(NULL, &pt, dataSz));

        /* try to verify the response */
        ExpectNotNull(issuer = wolfSSL_X509_load_certificate_file(caFile,
            SSL_FILETYPE_PEM));
        ExpectNotNull(st = wolfSSL_X509_STORE_new());
        ExpectIntEQ(wolfSSL_X509_STORE_add_cert(st, issuer), WOLFSSL_SUCCESS);
        ExpectNotNull(bs = wolfSSL_OCSP_response_get1_basic(res));
        ExpectIntEQ(wolfSSL_OCSP_basic_verify(bs, NULL, st, 0),
            WOLFSSL_SUCCESS);
        wolfSSL_OCSP_BASICRESP_free(bs);
        wolfSSL_OCSP_RESPONSE_free(res);
        wolfSSL_X509_STORE_free(st);
        wolfSSL_X509_free(issuer);
    }
#endif
#endif /* HAVE_OCSP */
    return EXPECT_RESULT();
}

static int test_wolfSSL_FPKI(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_FPKI) && !defined(NO_RSA) && !defined(NO_FILESYSTEM)
    XFILE f = XBADFILE;
    const char* fpkiCert = "./certs/fpki-cert.der";
    const char* fpkiCertPolCert = "./certs/fpki-certpol-cert.der";
    DecodedCert cert;
    byte buf[4096];
    byte* uuid = NULL;
    byte* fascn = NULL;
    word32 fascnSz;
    word32 uuidSz;
    int bytes = 0;

    ExpectTrue((f = XFOPEN(fpkiCert, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buf, 1, sizeof(buf), f), 0);
    if (f != XBADFILE)
        XFCLOSE(f);

    wc_InitDecodedCert(&cert, buf, (word32)bytes, NULL);
    ExpectIntEQ(wc_ParseCert(&cert, CERT_TYPE, 0, NULL), 0);
    ExpectIntEQ(wc_GetFASCNFromCert(&cert, NULL, &fascnSz), WC_NO_ERR_TRACE(LENGTH_ONLY_E));
    ExpectNotNull(fascn = (byte*)XMALLOC(fascnSz, NULL,
        DYNAMIC_TYPE_TMP_BUFFER));
    ExpectIntEQ(wc_GetFASCNFromCert(&cert, fascn, &fascnSz), 0);
    XFREE(fascn, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    fascn = NULL;

    ExpectIntEQ(wc_GetUUIDFromCert(&cert, NULL, &uuidSz), WC_NO_ERR_TRACE(LENGTH_ONLY_E));
    ExpectNotNull(uuid = (byte*)XMALLOC(uuidSz, NULL, DYNAMIC_TYPE_TMP_BUFFER));
    ExpectIntEQ(wc_GetUUIDFromCert(&cert, uuid, &uuidSz), 0);
    XFREE(uuid, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    uuid = NULL;
    wc_FreeDecodedCert(&cert);

    XMEMSET(buf, 0, 4096);
    fascnSz = uuidSz = bytes = 0;
    f = XBADFILE;

    ExpectTrue((f = XFOPEN(fpkiCertPolCert, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buf, 1, sizeof(buf), f), 0);
    if (f != XBADFILE)
        XFCLOSE(f);

    wc_InitDecodedCert(&cert, buf, (word32)bytes, NULL);
    ExpectIntEQ(wc_ParseCert(&cert, CERT_TYPE, 0, NULL), 0);
    ExpectIntEQ(wc_GetFASCNFromCert(&cert, NULL, &fascnSz), WC_NO_ERR_TRACE(LENGTH_ONLY_E));
    ExpectNotNull(fascn = (byte*)XMALLOC(fascnSz, NULL,
        DYNAMIC_TYPE_TMP_BUFFER));
    ExpectIntEQ(wc_GetFASCNFromCert(&cert, fascn, &fascnSz), 0);
    XFREE(fascn, NULL, DYNAMIC_TYPE_TMP_BUFFER);

    ExpectIntEQ(wc_GetUUIDFromCert(&cert, NULL, &uuidSz), WC_NO_ERR_TRACE(LENGTH_ONLY_E));
    ExpectNotNull(uuid = (byte*)XMALLOC(uuidSz, NULL, DYNAMIC_TYPE_TMP_BUFFER));
    ExpectIntEQ(wc_GetUUIDFromCert(&cert, uuid, &uuidSz), 0);
    XFREE(uuid, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    wc_FreeDecodedCert(&cert);
#endif

    return EXPECT_RESULT();
}

/* use RID in confuncture with other names to test parsing of unknown other
 * names */
static int test_wolfSSL_OtherName(void)
{
    EXPECT_DECLS;
#if !defined(NO_RSA) && !defined(NO_FILESYSTEM)
    XFILE f = XBADFILE;
    const char* ridCert = "./certs/rid-cert.der";
    DecodedCert cert;
    byte buf[4096];
    int bytes = 0;

    ExpectTrue((f = XFOPEN(ridCert, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buf, 1, sizeof(buf), f), 0);
    if (f != XBADFILE)
        XFCLOSE(f);

    wc_InitDecodedCert(&cert, buf, (word32)bytes, NULL);
    ExpectIntEQ(wc_ParseCert(&cert, CERT_TYPE, 0, NULL), 0);
    wc_FreeDecodedCert(&cert);
#endif

    return EXPECT_RESULT();
}

#ifdef HAVE_CERT_CHAIN_VALIDATION
#ifndef WOLFSSL_TEST_APPLE_NATIVE_CERT_VALIDATION
static int test_wolfSSL_CertRsaPss(void)
{
    EXPECT_DECLS;
/* FIPS v2 and below don't support long salts. */
#if !defined(NO_RSA) && defined(WC_RSA_PSS) && !defined(NO_FILESYSTEM) && \
    (!defined(HAVE_FIPS) || (defined(HAVE_FIPS_VERSION) && \
     (HAVE_FIPS_VERSION > 2))) && (!defined(HAVE_SELFTEST) || \
     (defined(HAVE_SELFTEST_VERSION) && (HAVE_SELFTEST_VERSION > 2)))
    XFILE f = XBADFILE;
#ifndef NO_SHA256
    const char* rsaPssSha256Cert = "./certs/rsapss/ca-rsapss.der";
#ifdef WOLFSSL_PEM_TO_DER
    const char* rsaPssRootSha256Cert = "./certs/rsapss/root-rsapss.pem";
#else
    const char* rsaPssRootSha256Cert = "./certs/rsapss/root-rsapss.der";
#endif
#endif
#if defined(WOLFSSL_SHA384) && defined(WOLFSSL_PSS_LONG_SALT) && \
    RSA_MAX_SIZE >= 3072
    const char* rsaPssSha384Cert = "./certs/rsapss/ca-3072-rsapss.der";
#endif
#if defined(WOLFSSL_SHA384) && RSA_MAX_SIZE >= 3072
#ifdef WOLFSSL_PEM_TO_DER
    const char* rsaPssRootSha384Cert = "./certs/rsapss/root-3072-rsapss.pem";
#else
    const char* rsaPssRootSha384Cert = "./certs/rsapss/root-3072-rsapss.der";
#endif
#endif
    DecodedCert cert;
    byte buf[4096];
    int bytes = 0;
    WOLFSSL_CERT_MANAGER* cm = NULL;

    ExpectNotNull(cm = wolfSSL_CertManagerNew());
#ifndef NO_SHA256
    ExpectIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CertManagerLoadCA(cm, rsaPssRootSha256Cert, NULL));
#endif
#if defined(WOLFSSL_SHA384) && RSA_MAX_SIZE >= 3072
    ExpectIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CertManagerLoadCA(cm, rsaPssRootSha384Cert, NULL));
#endif

#ifndef NO_SHA256
    ExpectTrue((f = XFOPEN(rsaPssSha256Cert, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buf, 1, sizeof(buf), f), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }
    wc_InitDecodedCert(&cert, buf, (word32)bytes, NULL);
    ExpectIntEQ(wc_ParseCert(&cert, CERT_TYPE, VERIFY, cm), 0);
    wc_FreeDecodedCert(&cert);
#endif

#if defined(WOLFSSL_SHA384) && defined(WOLFSSL_PSS_LONG_SALT) && \
    RSA_MAX_SIZE >= 3072
    ExpectTrue((f = XFOPEN(rsaPssSha384Cert, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buf, 1, sizeof(buf), f), 0);
    if (f != XBADFILE)
        XFCLOSE(f);
    wc_InitDecodedCert(&cert, buf, (word32)bytes, NULL);
    ExpectIntEQ(wc_ParseCert(&cert, CERT_TYPE, VERIFY, cm), 0);
    wc_FreeDecodedCert(&cert);
#endif

    wolfSSL_CertManagerFree(cm);

    (void)buf;
    (void)bytes;
#endif

    return EXPECT_RESULT();
}
#endif /* WOLFSSL_TEST_APPLE_NATIVE_CERT_VALIDATION */
#endif /* HAVE_CERT_CHAIN_VALIDATION */


/* Test RSA-PSS digital signature creation and verification */
static int test_wc_RsaPSS_DigitalSignVerify(void)
{
    EXPECT_DECLS;

    /* Early FIPS did not support PSS. */
#if (!defined(HAVE_FIPS) || (defined(HAVE_FIPS_VERSION) && \
    (HAVE_FIPS_VERSION > 2))) && \
    (!defined(HAVE_SELFTEST) || (defined(HAVE_SELFTEST_VERSION) && \
    (HAVE_SELFTEST_VERSION > 2))) && \
    !defined(NO_RSA) && defined(WC_RSA_PSS) && defined(OPENSSL_EXTRA) && \
    defined(WOLFSSL_KEY_GEN) && defined(WC_RSA_NO_PADDING) && \
    !defined(NO_SHA256)

    /* Test digest */
    const unsigned char test_digest[32] = {
        0x08, 0x09, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
        0x06, 0x07, 0x08, 0x09, 0x00, 0x01, 0x02, 0x03,
        0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x00, 0x01,
        0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09
    };
    const unsigned int digest_len = sizeof(test_digest);

    /* Variables for RSA key generation and signature operations */
    EVP_PKEY_CTX *pkctx = NULL;
    EVP_PKEY *pkey = NULL;
    EVP_PKEY_CTX *sign_ctx = NULL;
    EVP_PKEY_CTX *verify_ctx = NULL;
    unsigned char signature[256+MAX_DER_DIGEST_ASN_SZ] = {0};
    size_t signature_len = sizeof(signature);
    int modulus_bits = 2048;

    /* Generate RSA key pair to avoid file dependencies */
    ExpectNotNull(pkctx = EVP_PKEY_CTX_new_id(EVP_PKEY_RSA, NULL));
    ExpectIntEQ(EVP_PKEY_keygen_init(pkctx), 1);
    ExpectIntEQ(EVP_PKEY_CTX_set_rsa_keygen_bits(pkctx, modulus_bits), 1);
    ExpectIntEQ(EVP_PKEY_keygen(pkctx, &pkey), 1);

    /* Create signing context */
    ExpectNotNull(sign_ctx = EVP_PKEY_CTX_new(pkey, NULL));
    ExpectIntEQ(EVP_PKEY_sign_init(sign_ctx), 1);

    /* Configure RSA-PSS parameters for signing. */
    ExpectIntEQ(EVP_PKEY_CTX_set_rsa_padding(sign_ctx, RSA_PKCS1_PSS_PADDING),
        1);
    /* Default salt length matched hash so use 32 for SHA256 */
    ExpectIntEQ(EVP_PKEY_CTX_set_rsa_pss_saltlen(sign_ctx, 32), 1);
    ExpectIntEQ(EVP_PKEY_CTX_set_rsa_mgf1_md(sign_ctx, EVP_sha256()), 1);
    ExpectIntEQ(EVP_PKEY_CTX_set_signature_md(sign_ctx, EVP_sha256()), 1);

    /* Create the digital signature */
    ExpectIntEQ(EVP_PKEY_sign(sign_ctx, signature, &signature_len, test_digest,
                              digest_len), 1);
    ExpectIntGT((int)signature_len, 0);

    /* Create verification context */
    ExpectNotNull(verify_ctx = EVP_PKEY_CTX_new(pkey, NULL));
    ExpectIntEQ(EVP_PKEY_verify_init(verify_ctx), 1);

    /* Configure RSA-PSS parameters for verification */
    ExpectIntEQ(EVP_PKEY_CTX_set_rsa_padding(verify_ctx, RSA_PKCS1_PSS_PADDING),
        1);
    ExpectIntEQ(EVP_PKEY_CTX_set_rsa_pss_saltlen(verify_ctx, 32), 1);
    ExpectIntEQ(EVP_PKEY_CTX_set_rsa_mgf1_md(verify_ctx, EVP_sha256()), 1);
    ExpectIntEQ(EVP_PKEY_CTX_set_signature_md(verify_ctx, EVP_sha256()), 1);

    /* Verify the digital signature */
    ExpectIntEQ(EVP_PKEY_verify(verify_ctx, signature, signature_len,
                                test_digest, digest_len), 1);

    /* Test with wrong digest to ensure verification fails (negative test) */
    {
        const unsigned char wrong_digest[32] = {
            0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
            0x09, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
            0x07, 0x08, 0x09, 0x00, 0x01, 0x02, 0x03, 0x04,
            0x05, 0x06, 0x07, 0x08, 0x09, 0x00, 0x01, 0x02
        };
        ExpectIntNE(EVP_PKEY_verify(verify_ctx, signature, signature_len,
                    wrong_digest, digest_len), 1);
    }

    /* Clean up */
    if (verify_ctx)
        EVP_PKEY_CTX_free(verify_ctx);
    if (sign_ctx)
        EVP_PKEY_CTX_free(sign_ctx);
    if (pkey)
        EVP_PKEY_free(pkey);
    if (pkctx)
        EVP_PKEY_CTX_free(pkctx);

#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_load_verify_locations_ex(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    !defined(NO_WOLFSSL_CLIENT) && !defined(NO_RSA)
    WOLFSSL_CTX* ctx = NULL;
#ifdef WOLFSSL_PEM_TO_DER
    const char* ca_cert = "./certs/ca-cert.pem";
    const char* ca_expired_cert = "./certs/test/expired/expired-ca.pem";
#else
    const char* ca_cert = "./certs/ca-cert.der";
    const char* ca_expired_cert = "./certs/test/expired/expired-ca.der";
#endif

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));

    /* test good CA */
    ExpectTrue(WOLFSSL_SUCCESS ==
        wolfSSL_CTX_load_verify_locations_ex(ctx, ca_cert, NULL,
            WOLFSSL_LOAD_FLAG_NONE));

    /* test expired CA */
#if !defined(OPENSSL_COMPATIBLE_DEFAULTS) && !defined(NO_ASN_TIME)
    ExpectIntNE(wolfSSL_CTX_load_verify_locations_ex(ctx, ca_expired_cert, NULL,
            WOLFSSL_LOAD_FLAG_NONE), WOLFSSL_SUCCESS);
#else
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations_ex(ctx, ca_expired_cert, NULL,
            WOLFSSL_LOAD_FLAG_NONE), WOLFSSL_SUCCESS);
#endif
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations_ex(ctx, ca_expired_cert, NULL,
            WOLFSSL_LOAD_FLAG_DATE_ERR_OKAY), WOLFSSL_SUCCESS);

    wolfSSL_CTX_free(ctx);
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_load_verify_buffer_ex(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    !defined(NO_RSA) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
    WOLFSSL_CTX* ctx;
    const char* ca_expired_cert_file = "./certs/test/expired/expired-ca.der";
    byte ca_expired_cert[TWOK_BUF];
    word32 sizeof_ca_expired_cert = 0;
    XFILE fp = XBADFILE;

#ifndef NO_WOLFSSL_CLIENT
    ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
#else
    ctx = wolfSSL_CTX_new(wolfSSLv23_server_method());
#endif
    ExpectNotNull(ctx);

#if defined(USE_CERT_BUFFERS_2048)
    /* test good CA */
    ExpectTrue(WOLFSSL_SUCCESS ==
        wolfSSL_CTX_load_verify_buffer_ex(ctx, ca_cert_der_2048,
            sizeof_ca_cert_der_2048, WOLFSSL_FILETYPE_ASN1, 0,
            WOLFSSL_LOAD_FLAG_NONE));
#endif

    /* load expired CA */
    XMEMSET(ca_expired_cert, 0, sizeof(ca_expired_cert));
    ExpectTrue((fp = XFOPEN(ca_expired_cert_file, "rb")) != XBADFILE);
    ExpectIntGT(sizeof_ca_expired_cert = (word32)XFREAD(ca_expired_cert, 1,
        sizeof(ca_expired_cert), fp), 0);
    if (fp != XBADFILE)
        XFCLOSE(fp);

    /* test expired CA failure */


#if !defined(OPENSSL_COMPATIBLE_DEFAULTS) && !defined(NO_ASN_TIME)
    ExpectIntNE(wolfSSL_CTX_load_verify_buffer_ex(ctx, ca_expired_cert,
            sizeof_ca_expired_cert, WOLFSSL_FILETYPE_ASN1, 0,
            WOLFSSL_LOAD_FLAG_NONE), WOLFSSL_SUCCESS);
#else
    ExpectIntEQ(wolfSSL_CTX_load_verify_buffer_ex(ctx, ca_expired_cert,
            sizeof_ca_expired_cert, WOLFSSL_FILETYPE_ASN1, 0,
            WOLFSSL_LOAD_FLAG_NONE), WOLFSSL_SUCCESS);
#endif
    /* test expired CA success */
    ExpectIntEQ(wolfSSL_CTX_load_verify_buffer_ex(ctx, ca_expired_cert,
            sizeof_ca_expired_cert, WOLFSSL_FILETYPE_ASN1, 0,
            WOLFSSL_LOAD_FLAG_DATE_ERR_OKAY), WOLFSSL_SUCCESS);

    /* Fail when ctx is NULL. */
    ExpectIntEQ(wolfSSL_CTX_load_verify_buffer_ex(NULL, ca_expired_cert,
            sizeof_ca_expired_cert, WOLFSSL_FILETYPE_ASN1, 0,
            WOLFSSL_LOAD_FLAG_DATE_ERR_OKAY), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    /* Load as modified cert - bad initial length. */
    ca_expired_cert[2] = 0x7f;
    ExpectIntEQ(wolfSSL_CTX_load_verify_buffer_ex(ctx, ca_expired_cert,
            sizeof_ca_expired_cert, WOLFSSL_FILETYPE_ASN1, 1,
            WOLFSSL_LOAD_FLAG_DATE_ERR_OKAY), WC_NO_ERR_TRACE(ASN_PARSE_E));

    wolfSSL_CTX_free(ctx);
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_load_verify_chain_buffer_format(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && !defined(NO_RSA) && defined(OPENSSL_EXTRA) && \
    defined(USE_CERT_BUFFERS_2048) && (WOLFSSL_MIN_RSA_BITS <= 1024) && \
    !defined(NO_TLS) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
    WOLFSSL_CTX* ctx = NULL;

  #ifndef NO_WOLFSSL_CLIENT
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
  #else
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
  #endif

    /* Public key 140 bytes??? */
    ExpectIntEQ(wolfSSL_CTX_load_verify_chain_buffer_format(ctx,
        ca_cert_chain_der, sizeof_ca_cert_chain_der, WOLFSSL_FILETYPE_ASN1),
        WOLFSSL_SUCCESS);

    wolfSSL_CTX_free(ctx);
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_add1_chain_cert(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && defined(OPENSSL_EXTRA) && \
    defined(KEEP_OUR_CERT) && !defined(NO_RSA) && !defined(NO_TLS) && \
    !defined(NO_WOLFSSL_CLIENT)
    WOLFSSL_CTX*        ctx;
    WOLFSSL*            ssl = NULL;
    const char *certChain[] = {
            "./certs/intermediate/client-int-cert.pem",
            "./certs/intermediate/ca-int2-cert.pem",
            "./certs/intermediate/ca-int-cert.pem",
            "./certs/ca-cert.pem",
            NULL
    };
    const char** cert;
    WOLFSSL_X509* x509 = NULL;
    WOLF_STACK_OF(X509)* chain = NULL;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
    ExpectNotNull(ssl = wolfSSL_new(ctx));

    ExpectNotNull(x509 = wolfSSL_X509_new());
    ExpectIntEQ(SSL_CTX_add1_chain_cert(ctx, x509), 0);
    ExpectIntEQ(SSL_CTX_add0_chain_cert(ctx, x509), 0);
    ExpectIntEQ(SSL_add1_chain_cert(ssl, x509), 0);
    ExpectIntEQ(SSL_add0_chain_cert(ssl, x509), 0);
    wolfSSL_X509_free(x509);
    x509 = NULL;

    for (cert = certChain; EXPECT_SUCCESS() && *cert != NULL; cert++) {
        ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(*cert,
            WOLFSSL_FILETYPE_PEM));

        /* Do negative tests once */
        if (cert == certChain) {
            /* Negative tests. */
            ExpectIntEQ(SSL_CTX_add1_chain_cert(NULL, NULL), 0);
            ExpectIntEQ(SSL_CTX_add1_chain_cert(ctx, NULL), 0);
            ExpectIntEQ(SSL_CTX_add1_chain_cert(NULL, x509), 0);
            ExpectIntEQ(SSL_CTX_add0_chain_cert(NULL, NULL), 0);
            ExpectIntEQ(SSL_CTX_add0_chain_cert(ctx, NULL), 0);
            ExpectIntEQ(SSL_CTX_add0_chain_cert(NULL, x509), 0);
        }

        ExpectIntEQ(SSL_CTX_add1_chain_cert(ctx, x509), 1);
        X509_free(x509);
        x509 = NULL;
    }
    for (cert = certChain; EXPECT_SUCCESS() && *cert != NULL; cert++) {
        ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(*cert,
            WOLFSSL_FILETYPE_PEM));

        /* Do negative tests once */
        if (cert == certChain) {
            /* Negative tests. */
            ExpectIntEQ(SSL_add1_chain_cert(NULL, NULL), 0);
            ExpectIntEQ(SSL_add1_chain_cert(ssl, NULL), 0);
            ExpectIntEQ(SSL_add1_chain_cert(NULL, x509), 0);
            ExpectIntEQ(SSL_add0_chain_cert(NULL, NULL), 0);
            ExpectIntEQ(SSL_add0_chain_cert(ssl, NULL), 0);
            ExpectIntEQ(SSL_add0_chain_cert(NULL, x509), 0);
        }

        ExpectIntEQ(SSL_add1_chain_cert(ssl, x509), 1);
        X509_free(x509);
        x509 = NULL;
    }

    ExpectIntEQ(SSL_CTX_get0_chain_certs(ctx, &chain), 1);
    ExpectIntEQ(sk_X509_num(chain), 3);
    ExpectIntEQ(SSL_get0_chain_certs(ssl, &chain), 1);
    ExpectIntEQ(sk_X509_num(chain), 3);

    SSL_free(ssl);
    SSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_use_certificate_chain_buffer_format(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && !defined(NO_TLS) && \
    !defined(NO_WOLFSSL_CLIENT) && !defined(NO_RSA) && \
    (!defined(NO_FILESYSTEM) || defined(USE_CERT_BUFFERS_2048))
    WOLFSSL_CTX* ctx = NULL;
    WOLFSSL* ssl = NULL;
#ifndef NO_FILESYSTEM
    const char* cert = svrCertFile;
    unsigned char* buf = NULL;
    size_t len = 0;

    ExpectIntEQ(load_file(cert, &buf, &len), 0);
#endif

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
    ExpectNotNull(ssl = wolfSSL_new(ctx));

    /* Invalid parameters. */
#ifndef NO_FILESYSTEM
    ExpectIntEQ(wolfSSL_CTX_use_certificate_chain_buffer_format(NULL,
        NULL, 0, WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_use_certificate_chain_buffer_format(ctx,
        NULL, 0, WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(ASN_PARSE_E));
    ExpectIntEQ(wolfSSL_CTX_use_certificate_chain_buffer(NULL, NULL, 0),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
#ifdef WOLFSSL_PEM_TO_DER
    ExpectIntEQ(wolfSSL_CTX_use_certificate_chain_buffer(ctx, NULL, 0),
        WC_NO_ERR_TRACE(ASN_NO_PEM_HEADER));
#endif
    ExpectIntEQ(wolfSSL_CTX_use_certificate_chain_buffer(NULL, buf,
        (sword32)len), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_use_certificate_chain_buffer(NULL, NULL, 0),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
#ifdef WOLFSSL_PEM_TO_DER
    ExpectIntEQ(wolfSSL_use_certificate_chain_buffer(ssl, NULL, 0),
        WC_NO_ERR_TRACE(ASN_NO_PEM_HEADER));
#endif
    ExpectIntEQ(wolfSSL_use_certificate_chain_buffer(NULL, buf, (sword32)len),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    ExpectIntEQ(wolfSSL_CTX_use_certificate_chain_buffer_format(ctx, buf,
        (sword32)len, CERT_FILETYPE), WOLFSSL_SUCCESS);

    ExpectIntEQ(wolfSSL_CTX_use_certificate_chain_buffer(ctx, buf, (sword32)len),
        WOLFSSL_SUCCESS);

    ExpectIntEQ(wolfSSL_use_certificate_chain_buffer(ssl, buf, (sword32)len),
        WOLFSSL_SUCCESS);
#endif /* !NO_FILESYSTEM */

    ExpectIntEQ(wolfSSL_CTX_use_certificate_chain_buffer_format(NULL,
        server_cert_der_2048, sizeof_server_cert_der_2048,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    ExpectIntEQ(wolfSSL_CTX_use_certificate_chain_buffer_format(ctx,
        server_cert_der_2048, sizeof_server_cert_der_2048,
        WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

#ifdef WOLFSSL_PEM_TO_DER
    ExpectIntEQ(wolfSSL_CTX_use_certificate_chain_buffer(ctx,
        server_cert_der_2048, sizeof_server_cert_der_2048),
        WC_NO_ERR_TRACE(ASN_NO_PEM_HEADER));

    ExpectIntEQ(wolfSSL_use_certificate_chain_buffer(ssl, server_cert_der_2048,
        sizeof_server_cert_der_2048), WC_NO_ERR_TRACE(ASN_NO_PEM_HEADER));
#endif

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#ifndef NO_FILESYSTEM
    if (buf != NULL) {
        free(buf);
    }
#endif
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_use_certificate_chain_file_format(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    !defined(NO_RSA) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
    const char* server_chain_der = "./certs/server-cert-chain.der";
    const char* client_single_pem = cliCertFile;
    WOLFSSL_CTX* ctx = NULL;

    (void)server_chain_der;
    (void)client_single_pem;
    (void)ctx;

  #ifndef NO_WOLFSSL_CLIENT
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
  #else
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
  #endif

    ExpectIntEQ(wolfSSL_CTX_use_certificate_chain_file_format(ctx,
        server_chain_der, WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_CTX_use_certificate_chain_file_format(ctx,
        client_single_pem, CERT_FILETYPE), WOLFSSL_SUCCESS);

    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_use_certificate_chain_file(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    !defined(NO_WOLFSSL_CLIENT) && !defined(NO_RSA)
    const char* server_chain_der = "./certs/server-cert-chain.der";
    const char* client_single_pem = cliCertFile;
    WOLFSSL_CTX* ctx = NULL;
    WOLFSSL* ssl = NULL;

    (void)server_chain_der;
    (void)client_single_pem;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
    ExpectNotNull(ssl = wolfSSL_new(ctx));

    /* Invalid parameters. */
    ExpectIntEQ(wolfSSL_use_certificate_chain_file_format(NULL, NULL,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_use_certificate_chain_file_format(ssl, NULL,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(wolfSSL_use_certificate_chain_file_format(NULL,
       server_chain_der, WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_use_certificate_chain_file(NULL, NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_use_certificate_chain_file(ssl, NULL),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(wolfSSL_use_certificate_chain_file(NULL, client_single_pem),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
#ifdef WOLFSSL_PEM_TO_DER
    ExpectIntEQ(wolfSSL_use_certificate_chain_file(ssl, server_chain_der),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
#endif

    ExpectIntEQ(wolfSSL_use_certificate_chain_file_format(ssl,
        server_chain_der, WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_use_certificate_chain_file_format(ssl,
        client_single_pem, CERT_FILETYPE), WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_use_certificate_chain_file(ssl, client_single_pem),
        WOLFSSL_SUCCESS);

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_SetTmpDH_file(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    !defined(NO_DH) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
    WOLFSSL_CTX *ctx = NULL;
#if defined(WOLFSSL_WPAS) && !defined(NO_DSA)
#if defined(WOLFSSL_PEM_TO_DER)
    const char* dsaParamFile = "./certs/dsaparams.pem";
#else
    const char* dsaParamFile = "./certs/dsaparams.der";
#endif
#endif

    (void)ctx;

  #ifndef NO_WOLFSSL_CLIENT
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
  #else
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
  #endif

    /* invalid context */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_file(NULL,
                dhParamFile, CERT_FILETYPE));

    /* invalid dhParamFile file */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_file(ctx,
                NULL, CERT_FILETYPE));

    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_file(ctx,
                bogusFile, CERT_FILETYPE));

    /* success */
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_file(ctx, dhParamFile,
                CERT_FILETYPE));
#if defined(WOLFSSL_WPAS) && !defined(NO_DSA)
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_file(ctx, dsaParamFile,
                CERT_FILETYPE));
#endif

    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_SetTmpDH_buffer(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && !defined(NO_TLS) && !defined(NO_DH) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
    WOLFSSL_CTX *ctx = NULL;

  #ifndef NO_WOLFSSL_CLIENT
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
  #else
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
  #endif

    /* invalid context */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_buffer(NULL,
                dh_key_der_2048, sizeof_dh_key_der_2048,
                WOLFSSL_FILETYPE_ASN1));

    /* invalid dhParamFile file */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_buffer(NULL, NULL,
                0, WOLFSSL_FILETYPE_ASN1));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_buffer(ctx, NULL,
                0, WOLFSSL_FILETYPE_ASN1));

    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_buffer(ctx,
                dsa_key_der_2048, sizeof_dsa_key_der_2048,
                WOLFSSL_FILETYPE_ASN1));

    /* invalid file format */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_buffer(ctx,
                dh_key_der_2048, sizeof_dh_key_der_2048, -1));

    /* success */
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_buffer(ctx,
                dh_key_der_2048, sizeof_dh_key_der_2048,
                WOLFSSL_FILETYPE_ASN1));

    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

static int test_wc_DhSetNamedKey(void)
{
    EXPECT_DECLS;
#if !defined(HAVE_SELFTEST) && !defined(NO_DH) && \
    !defined(WOLFSSL_NO_MALLOC) && defined(HAVE_FFDHE) && \
    (!defined(HAVE_FIPS) || FIPS_VERSION_GT(2,0))
    DhKey *key = NULL;
    key = (DhKey*)XMALLOC(sizeof(DhKey), HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    ExpectNotNull(key);
    if (key != NULL){
        #ifdef HAVE_FFDHE_2048
        if (wc_InitDhKey_ex(key, HEAP_HINT, INVALID_DEVID) == 0){
            ExpectIntEQ(wc_DhSetNamedKey(key, WC_FFDHE_2048), 0);
            ExpectIntEQ(wc_DhGetNamedKeyMinSize(WC_FFDHE_2048), 29);
            wc_FreeDhKey(key);
        }
        #endif
        #ifdef HAVE_FFDHE_3072
        if (wc_InitDhKey_ex(key, HEAP_HINT, INVALID_DEVID) == 0){
            ExpectIntEQ(wc_DhSetNamedKey(key, WC_FFDHE_3072), 0);
            ExpectIntEQ(wc_DhGetNamedKeyMinSize(WC_FFDHE_3072), 34);
            wc_FreeDhKey(key);
        }
        #endif
        #ifdef HAVE_FFDHE_4096
        if (wc_InitDhKey_ex(key, HEAP_HINT, INVALID_DEVID) == 0){
            ExpectIntEQ(wc_DhSetNamedKey(key, WC_FFDHE_4096), 0);
            ExpectIntEQ(wc_DhGetNamedKeyMinSize(WC_FFDHE_4096), 39);
            wc_FreeDhKey(key);
        }
        #endif
        #ifdef HAVE_FFDHE_6144
        if (wc_InitDhKey_ex(key, HEAP_HINT, INVALID_DEVID) == 0){
            ExpectIntEQ(wc_DhSetNamedKey(key, WC_FFDHE_6144), 0);
            ExpectIntEQ(wc_DhGetNamedKeyMinSize(WC_FFDHE_6144), 46);
            wc_FreeDhKey(key);
        }
        #endif
        #ifdef HAVE_FFDHE_8192
        if (wc_InitDhKey_ex(key, HEAP_HINT, INVALID_DEVID) == 0){
            ExpectIntEQ(wc_DhSetNamedKey(key, WC_FFDHE_8192), 0);
            ExpectIntEQ(wc_DhGetNamedKeyMinSize(WC_FFDHE_8192), 52);
            wc_FreeDhKey(key);
        }
        #endif
        XFREE(key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_SetMinMaxDhKey_Sz(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && !defined(NO_TLS) && !defined(NO_DH) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
    WOLFSSL_CTX *ctx;

    (void)ctx;

  #ifndef NO_WOLFSSL_CLIENT
    ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
  #else
    ctx = wolfSSL_CTX_new(wolfSSLv23_server_method());
  #endif
    ExpectNotNull(ctx);

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetMinDhKey_Sz(ctx, 3072));

    ExpectIntEQ(WC_NO_ERR_TRACE(DH_KEY_SIZE_E), wolfSSL_CTX_SetTmpDH_buffer(ctx, dh_key_der_2048,
                sizeof_dh_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetMinDhKey_Sz(ctx, 2048));

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_buffer(ctx,
                dh_key_der_2048, sizeof_dh_key_der_2048,
                WOLFSSL_FILETYPE_ASN1));

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetMaxDhKey_Sz(ctx, 1024));

    ExpectIntEQ(WC_NO_ERR_TRACE(DH_KEY_SIZE_E), wolfSSL_CTX_SetTmpDH_buffer(ctx,
                dh_key_der_2048, sizeof_dh_key_der_2048,
                WOLFSSL_FILETYPE_ASN1));

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetMaxDhKey_Sz(ctx, 2048));

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_buffer(ctx,
                dh_key_der_2048, sizeof_dh_key_der_2048,
                WOLFSSL_FILETYPE_ASN1));

    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_der_load_verify_locations(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && defined(WOLFSSL_DER_LOAD) && \
    !defined(NO_TLS) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
    WOLFSSL_CTX* ctx = NULL;
    const char* derCert = "./certs/server-cert.der";
    const char* nullPath = NULL;
    const char* invalidPath = "./certs/this-cert-does-not-exist.der";
    const char* emptyPath = "";

    /* der load Case 1 ctx NULL */
    ExpectIntEQ(wolfSSL_CTX_der_load_verify_locations(ctx, derCert,
                WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

  #ifndef NO_WOLFSSL_CLIENT
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
  #else
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
  #endif

    /* Case 2 filePath NULL */
    ExpectIntEQ(wolfSSL_CTX_der_load_verify_locations(ctx, nullPath,
                WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    /* Case 3 invalid format */
    ExpectIntEQ(wolfSSL_CTX_der_load_verify_locations(ctx, derCert,
                WOLFSSL_FILETYPE_PEM), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    /* Case 4 filePath not valid */
    ExpectIntEQ(wolfSSL_CTX_der_load_verify_locations(ctx, invalidPath,
                WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    /* Case 5 filePath empty */
    ExpectIntEQ(wolfSSL_CTX_der_load_verify_locations(ctx, emptyPath,
                WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
#ifndef NO_RSA
    /* Case 6 success case */
    ExpectIntEQ(wolfSSL_CTX_der_load_verify_locations(ctx, derCert,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
#endif

    wolfSSL_CTX_free(ctx);
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_enable_disable(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && !defined(NO_TLS)
    WOLFSSL_CTX* ctx = NULL;

  #ifdef HAVE_CRL
    ExpectIntEQ(wolfSSL_CTX_DisableCRL(ctx), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_EnableCRL(ctx, 0), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
  #endif

  #ifdef HAVE_OCSP
    ExpectIntEQ(wolfSSL_CTX_DisableOCSP(ctx), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_EnableOCSP(ctx, 0), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
  #endif

  #if defined(HAVE_CERTIFICATE_STATUS_REQUEST) || \
      defined(HAVE_CERTIFICATE_STATUS_REQUEST_V2)
    ExpectIntEQ(wolfSSL_CTX_DisableOCSPStapling(ctx), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_EnableOCSPStapling(ctx), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_DisableOCSPMustStaple(ctx), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_EnableOCSPMustStaple(ctx), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
  #endif

  #ifndef NO_WOLFSSL_CLIENT

    #ifdef HAVE_EXTENDED_MASTER
    ExpectIntEQ(wolfSSL_CTX_DisableExtendedMasterSecret(ctx), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    #endif
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));

    #ifdef HAVE_EXTENDED_MASTER
    ExpectIntEQ(wolfSSL_CTX_DisableExtendedMasterSecret(ctx), WOLFSSL_SUCCESS);
    #endif

  #elif !defined(NO_WOLFSSL_SERVER)
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
  #endif

  #if !defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)

    #ifdef HAVE_CRL
        ExpectIntEQ(wolfSSL_CTX_DisableCRL(ctx), WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_CTX_EnableCRL(ctx, 0), WOLFSSL_SUCCESS);
    #endif

    #ifdef HAVE_OCSP
        ExpectIntEQ(wolfSSL_CTX_DisableOCSP(ctx), WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_CTX_EnableOCSP(ctx, WOLFSSL_OCSP_URL_OVERRIDE),
                    WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_CTX_EnableOCSP(ctx, WOLFSSL_OCSP_NO_NONCE),
                    WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_CTX_EnableOCSP(ctx, WOLFSSL_OCSP_CHECKALL),
                    WOLFSSL_SUCCESS);
    #endif

    #if defined(HAVE_CERTIFICATE_STATUS_REQUEST) || \
        defined(HAVE_CERTIFICATE_STATUS_REQUEST_V2)
        ExpectIntEQ(wolfSSL_CTX_DisableOCSPStapling(ctx), WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_CTX_EnableOCSPStapling(ctx), WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_CTX_DisableOCSPMustStaple(ctx), WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_CTX_DisableOCSPMustStaple(ctx), WOLFSSL_SUCCESS);
    #endif
        wolfSSL_CTX_free(ctx);
  #endif /* !NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER */
#endif /* !NO_CERTS && !NO_CERTS */

    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_ticket_API(void)
{
    EXPECT_DECLS;
#if defined(HAVE_SESSION_TICKET) && !defined(NO_WOLFSSL_SERVER) && \
    !defined(NO_TLS)
    WOLFSSL_CTX* ctx = NULL;
    void *userCtx = (void*)"this is my ctx";

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_set_TicketEncCtx(ctx, userCtx));
    ExpectTrue(userCtx == wolfSSL_CTX_get_TicketEncCtx(ctx));

    wolfSSL_CTX_free(ctx);

    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_set_TicketEncCtx(NULL, userCtx));
    ExpectNull(wolfSSL_CTX_get_TicketEncCtx(NULL));
#endif /* HAVE_SESSION_TICKET && !NO_WOLFSSL_SERVER && !NO_TLS */
    return EXPECT_RESULT();
}

static int test_wolfSSL_set_minmax_proto_version(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_TLS)
    WOLFSSL_CTX *ctx = NULL;
    WOLFSSL *ssl = NULL;

    (void)ssl;

#ifndef NO_WOLFSSL_CLIENT
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
    ExpectNotNull(ssl = wolfSSL_new(ctx));

    ExpectIntEQ(wolfSSL_CTX_set_min_proto_version(NULL, 0), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(wolfSSL_CTX_set_max_proto_version(NULL, 0), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(wolfSSL_CTX_set_min_proto_version(ctx, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_CTX_set_max_proto_version(ctx, 0), SSL_SUCCESS);

    ExpectIntEQ(wolfSSL_set_min_proto_version(NULL, 0), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(wolfSSL_set_min_proto_version(ssl, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_set_max_proto_version(NULL, 0), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(wolfSSL_set_max_proto_version(ssl, 0), SSL_SUCCESS);

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
    ctx = NULL;
#endif
#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));

    ExpectIntEQ(wolfSSL_CTX_set_min_proto_version(NULL, 0), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(wolfSSL_CTX_set_max_proto_version(NULL, 0), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(wolfSSL_CTX_set_min_proto_version(ctx, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_CTX_set_max_proto_version(ctx, 0), SSL_SUCCESS);

    wolfSSL_CTX_free(ctx);
#endif
#endif

    return EXPECT_RESULT();
}

#if defined(WOLFSSL_TLS13) && !defined(WOLFSSL_NO_TLS12) && \
    defined(OPENSSL_EXTRA) && defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES)
static int test_wolfSSL_CTX_set_max_proto_version_on_result(WOLFSSL* ssl)
{
    EXPECT_DECLS;
    ExpectStrEQ(wolfSSL_get_version(ssl), "TLSv1.2");
    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_set_max_proto_version_ctx_ready(WOLFSSL_CTX* ctx)
{
    EXPECT_DECLS;
    /* Set TLS 1.2 */
    ExpectIntEQ(wolfSSL_CTX_set_max_proto_version(ctx, TLS1_2_VERSION),
            WOLFSSL_SUCCESS);
    return EXPECT_RESULT();
}

/* Test using wolfSSL_CTX_set_max_proto_version to limit the version below
 * what was set at ctx creation. */
static int test_wolfSSL_CTX_set_max_proto_version(void)
{
    EXPECT_DECLS;
    test_ssl_cbf client_cbs;
    test_ssl_cbf server_cbs;

    XMEMSET(&client_cbs, 0, sizeof(client_cbs));
    XMEMSET(&server_cbs, 0, sizeof(server_cbs));

    client_cbs.method = wolfTLS_client_method;
    server_cbs.method = wolfTLS_server_method;

    server_cbs.ctx_ready = test_wolfSSL_CTX_set_max_proto_version_ctx_ready;

    client_cbs.on_result = test_wolfSSL_CTX_set_max_proto_version_on_result;
    server_cbs.on_result = test_wolfSSL_CTX_set_max_proto_version_on_result;

    ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&client_cbs,
        &server_cbs, NULL), TEST_SUCCESS);

    return EXPECT_RESULT();
}
#else
static int test_wolfSSL_CTX_set_max_proto_version(void)
{
    return TEST_SKIPPED;
}
#endif

/*----------------------------------------------------------------------------*
 | SSL
 *----------------------------------------------------------------------------*/

static int test_server_wolfSSL_new(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    !defined(NO_WOLFSSL_SERVER) && !defined(NO_RSA)

    WOLFSSL_CTX *ctx = NULL;
    WOLFSSL_CTX *ctx_nocert = NULL;
    WOLFSSL *ssl = NULL;

    ExpectNotNull(ctx_nocert = wolfSSL_CTX_new(wolfSSLv23_server_method()));
    ExpectNotNull(ctx        = wolfSSL_CTX_new(wolfSSLv23_server_method()));

    ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, svrCertFile,
        CERT_FILETYPE));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, svrKeyFile,
        CERT_FILETYPE));

    /* invalid context */
    ExpectNull(ssl = wolfSSL_new(NULL));
#if !defined(WOLFSSL_SESSION_EXPORT) && !defined(WOLFSSL_QT) && \
        !defined(OPENSSL_EXTRA) && !defined(WOLFSSL_NO_INIT_CTX_KEY)
    ExpectNull(ssl = wolfSSL_new(ctx_nocert));
#endif

    /* success */
    ExpectNotNull(ssl = wolfSSL_new(ctx));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
    wolfSSL_CTX_free(ctx_nocert);
#endif

    return EXPECT_RESULT();
}


static int test_client_wolfSSL_new(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    !defined(NO_WOLFSSL_CLIENT) && !defined(NO_RSA)

    WOLFSSL_CTX *ctx = NULL;
    WOLFSSL_CTX *ctx_nocert = NULL;
    WOLFSSL *ssl = NULL;

    ExpectNotNull(ctx_nocert = wolfSSL_CTX_new(wolfSSLv23_client_method()));
    ExpectNotNull(ctx        = wolfSSL_CTX_new(wolfSSLv23_client_method()));

    ExpectTrue(wolfSSL_CTX_load_verify_locations(ctx, caCertFile, 0));

    /* invalid context */
    ExpectNull(ssl = wolfSSL_new(NULL));

    /* success */
    ExpectNotNull(ssl = wolfSSL_new(ctx_nocert));
    wolfSSL_free(ssl);
    ssl = NULL;

    /* success */
    ExpectNotNull(ssl = wolfSSL_new(ctx));
    wolfSSL_free(ssl);

    wolfSSL_CTX_free(ctx);
    wolfSSL_CTX_free(ctx_nocert);
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_SetTmpDH_file(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    !defined(NO_WOLFSSL_SERVER) && !defined(NO_DH)

    WOLFSSL_CTX *ctx = NULL;
    WOLFSSL *ssl = NULL;
#ifdef WOLFSSL_PEM_TO_DER
    const char* dhX942ParamFile = "./certs/x942dh2048.pem";
#if defined(WOLFSSL_WPAS) && !defined(NO_DSA)
    const char* dsaParamFile = "./certs/dsaparams.pem";
#endif
#else
    const char* dhX942ParamFile = "./certs/x942dh2048.der";
#if defined(WOLFSSL_WPAS) && !defined(NO_DSA)
    const char* dsaParamFile = "./certs/dsaparams.der";
#endif
#endif

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#ifndef NO_RSA
    ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, svrCertFile,
               CERT_FILETYPE));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, svrKeyFile,
               CERT_FILETYPE));
#elif defined(HAVE_ECC)
    ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, eccCertFile,
               CERT_FILETYPE));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, eccKeyFile,
               CERT_FILETYPE));
#elif defined(HAVE_ED25519)
    ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, edCertFile,
               CERT_FILETYPE));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, edKeyFile,
               CERT_FILETYPE));
#elif defined(HAVE_ED448)
    ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, ed448CertFile,
               CERT_FILETYPE));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, ed448KeyFile,
               CERT_FILETYPE));
#endif
    ExpectNotNull(ssl = wolfSSL_new(ctx));

    /* invalid ssl */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_file(NULL,
                dhParamFile, CERT_FILETYPE));

    /* invalid dhParamFile file */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_file(ssl,
                NULL, CERT_FILETYPE));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_file(ssl,
                bogusFile, CERT_FILETYPE));

    /* success */
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_file(ssl, dhParamFile,
                CERT_FILETYPE));
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_file(ssl, dhX942ParamFile,
                CERT_FILETYPE));
#if defined(WOLFSSL_WPAS) && !defined(NO_DSA)
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_file(ctx, dsaParamFile,
                CERT_FILETYPE));
#endif

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_SetTmpDH_buffer(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && !defined(NO_TLS) && !defined(NO_WOLFSSL_SERVER) && \
    !defined(NO_DH)
    WOLFSSL_CTX *ctx = NULL;
    WOLFSSL *ssl = NULL;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
    ExpectTrue(wolfSSL_CTX_use_certificate_buffer(ctx, server_cert_der_2048,
               sizeof_server_cert_der_2048, WOLFSSL_FILETYPE_ASN1));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_buffer(ctx, server_key_der_2048,
               sizeof_server_key_der_2048, WOLFSSL_FILETYPE_ASN1));
    ExpectNotNull(ssl = wolfSSL_new(ctx));

    /* invalid ssl */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_buffer(NULL, dh_key_der_2048,
                sizeof_dh_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    /* invalid dhParamFile file */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_buffer(NULL, NULL,
                0, WOLFSSL_FILETYPE_ASN1));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_buffer(ssl, NULL, 0,
                WOLFSSL_FILETYPE_ASN1));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_buffer(ssl, dsa_key_der_2048,
                sizeof_dsa_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    /* success */
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_buffer(ssl, dh_key_der_2048,
                sizeof_dh_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_SetMinMaxDhKey_Sz(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && !defined(NO_TLS) && !defined(NO_WOLFSSL_SERVER) && \
    !defined(NO_DH)
    WOLFSSL_CTX *ctx = NULL;
    WOLFSSL_CTX *ctx2 = NULL;
    WOLFSSL *ssl = NULL;
    WOLFSSL *ssl2 = NULL;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
    ExpectTrue(wolfSSL_CTX_use_certificate_buffer(ctx, server_cert_der_2048,
                sizeof_server_cert_der_2048, WOLFSSL_FILETYPE_ASN1));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_buffer(ctx, server_key_der_2048,
                sizeof_server_key_der_2048, WOLFSSL_FILETYPE_ASN1));
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetMinDhKey_Sz(ctx, 3072));
    ExpectNotNull(ssl = wolfSSL_new(ctx));
    ExpectNotNull(ctx2 = wolfSSL_CTX_new(wolfSSLv23_server_method()));
    ExpectTrue(wolfSSL_CTX_use_certificate_buffer(ctx2, server_cert_der_2048,
                sizeof_server_cert_der_2048, WOLFSSL_FILETYPE_ASN1));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_buffer(ctx2, server_key_der_2048,
                sizeof_server_key_der_2048, WOLFSSL_FILETYPE_ASN1));
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetMaxDhKey_Sz(ctx, 1024));
    ExpectNotNull(ssl2 = wolfSSL_new(ctx2));

    ExpectIntEQ(WC_NO_ERR_TRACE(DH_KEY_SIZE_E), wolfSSL_SetTmpDH_buffer(ssl, dh_key_der_2048,
                sizeof_dh_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_SetMinDhKey_Sz(ssl, 2048));
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_buffer(ssl, dh_key_der_2048,
                sizeof_dh_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_SetMinDhKey_Sz(ssl, 3072));
    ExpectIntEQ(WC_NO_ERR_TRACE(DH_KEY_SIZE_E), wolfSSL_SetTmpDH_buffer(ssl, dh_key_der_2048,
                sizeof_dh_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_buffer(ssl2, dh_key_der_2048,
                sizeof_dh_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_SetMaxDhKey_Sz(ssl2, 2048));
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_buffer(ssl2, dh_key_der_2048,
                sizeof_dh_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_SetMaxDhKey_Sz(ssl2, 1024));
    ExpectIntEQ(WC_NO_ERR_TRACE(DH_KEY_SIZE_E), wolfSSL_SetTmpDH_buffer(ssl, dh_key_der_2048,
                sizeof_dh_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    wolfSSL_free(ssl2);
    wolfSSL_CTX_free(ctx2);
    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}


/* Test function for wolfSSL_SetMinVersion. Sets the minimum downgrade version
 * allowed.
 * POST: return 1 on success.
 */
static int test_wolfSSL_SetMinVersion(void)
{
    int                 res = TEST_SKIPPED;
#if !defined(NO_WOLFSSL_CLIENT) && !defined(NO_TLS)
    int                 failFlag = WOLFSSL_SUCCESS;
    WOLFSSL_CTX*        ctx = NULL;
    WOLFSSL*            ssl = NULL;
    int                 itr;

    #ifndef NO_OLD_TLS
        const int versions[]  =  {
                            #ifdef WOLFSSL_ALLOW_TLSV10
                                   WOLFSSL_TLSV1,
                            #endif
                                   WOLFSSL_TLSV1_1,
                                   WOLFSSL_TLSV1_2};
    #elif !defined(WOLFSSL_NO_TLS12)
        const int versions[]  =  { WOLFSSL_TLSV1_2 };
    #else
        const int versions[]  =  { WOLFSSL_TLSV1_3 };
    #endif

    ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
    ssl = wolfSSL_new(ctx);

    for (itr = 0; itr < (int)(sizeof(versions)/sizeof(int)); itr++) {
       if (wolfSSL_SetMinVersion(ssl, *(versions + itr)) != WOLFSSL_SUCCESS) {
            failFlag = WOLFSSL_FAILURE;
        }
    }

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(failFlag == WOLFSSL_SUCCESS);
#endif
    return res;

} /* END test_wolfSSL_SetMinVersion */


#include <wolfssl/openssl/pem.h>
/*----------------------------------------------------------------------------*
 | EVP
 *----------------------------------------------------------------------------*/

static int test_wolfSSL_EVP_PKEY_print_public(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_BIO)
    WOLFSSL_BIO* rbio = NULL;
    WOLFSSL_BIO* wbio = NULL;
    WOLFSSL_EVP_PKEY* pkey = NULL;
    char line[256] = { 0 };
    char line1[256] = { 0 };
    int i = 0;

    /* test error cases */
    ExpectIntEQ( EVP_PKEY_print_public(NULL,NULL,0,NULL),0L);

    /*
     *  test RSA public key print
     *  in this test, pass '3' for indent
     */
#if !defined(NO_RSA) && defined(USE_CERT_BUFFERS_1024)

    ExpectNotNull(rbio = BIO_new_mem_buf( client_keypub_der_1024,
        sizeof_client_keypub_der_1024));

    ExpectNotNull(wolfSSL_d2i_PUBKEY_bio(rbio, &pkey));

    ExpectNotNull(wbio = BIO_new(BIO_s_mem()));

    ExpectIntEQ(EVP_PKEY_print_public(wbio, pkey,3,NULL),1);

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "   RSA Public-Key: (1024 bit)\n");
    ExpectIntEQ(XSTRNCMP(line, line1, XSTRLEN(line1)), 0);

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "   Modulus:\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "       00:bc:73:0e:a8:49:f3:74:a2:a9:ef:18:a5:da:55:\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);


    /* skip to the end of modulus element*/
    for (i = 0; i < 8 ;i++) {
        ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    }

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "   Exponent: 65537 (0x010001)\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);


    /* should reach EOF */
    ExpectIntLE(BIO_gets(wbio, line, sizeof(line)), 0);

    EVP_PKEY_free(pkey);
    pkey = NULL;
    BIO_free(rbio);
    BIO_free(wbio);
    rbio = NULL;
    wbio = NULL;

#endif  /* !NO_RSA && USE_CERT_BUFFERS_1024*/

    /*
     *  test DSA public key print
     */
#if !defined(NO_DSA) && defined(USE_CERT_BUFFERS_2048)
    ExpectNotNull(rbio = BIO_new_mem_buf( dsa_pub_key_der_2048,
        sizeof_dsa_pub_key_der_2048));

    ExpectNotNull(wolfSSL_d2i_PUBKEY_bio(rbio, &pkey));

    ExpectNotNull(wbio = BIO_new(BIO_s_mem()));

    ExpectIntEQ(EVP_PKEY_print_public(wbio, pkey,0,NULL),1);

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "DSA Public-Key: (2048 bit)\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "pub:\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1,
        "    00:C2:35:2D:EC:83:83:6C:73:13:9E:52:7C:74:C8:\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    /* skip to the end of pub element*/
    for (i = 0; i < 17 ;i++) {
        ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    }

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "P:\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    /* skip to the end of P element*/
    for (i = 0; i < 18 ;i++) {
        ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    }

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "Q:\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    /* skip to the end of Q element*/
    for (i = 0; i < 3 ;i++) {
        ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    }
    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "G:\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    /* skip to the end of G element*/
    for (i = 0; i < 18 ;i++) {
        ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    }
    /* should reach EOF */
    ExpectIntLE(BIO_gets(wbio, line, sizeof(line)), 0);

    EVP_PKEY_free(pkey);
    pkey = NULL;
    BIO_free(rbio);
    BIO_free(wbio);
    rbio = NULL;
    wbio = NULL;

#endif /* !NO_DSA && USE_CERT_BUFFERS_2048 */

    /*
     *  test ECC public key print
     */
#if defined(HAVE_ECC) && defined(USE_CERT_BUFFERS_256)

    ExpectNotNull(rbio = BIO_new_mem_buf( ecc_clikeypub_der_256,
        sizeof_ecc_clikeypub_der_256));

    ExpectNotNull(wolfSSL_d2i_PUBKEY_bio(rbio, &pkey));

    ExpectNotNull(wbio = BIO_new(BIO_s_mem()));

    ExpectIntEQ(EVP_PKEY_print_public(wbio, pkey,0,NULL),1);

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    ExpectStrEQ(line, "Public-Key: (256 bit)\n");

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "pub:\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1,
            "    04:55:BF:F4:0F:44:50:9A:3D:CE:9B:B7:F0:C5:4D:\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    /* skip to the end of pub element*/
    for (i = 0; i < 4 ;i++) {
        ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    }

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "ASN1 OID: prime256v1\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "NIST CURVE: P-256\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);


    /* should reach EOF */
    ExpectIntLE(BIO_gets(wbio, line, sizeof(line)), 0);

    EVP_PKEY_free(pkey);
    pkey = NULL;
    BIO_free(rbio);
    BIO_free(wbio);
    rbio = NULL;
    wbio = NULL;

#endif /* HAVE_ECC && USE_CERT_BUFFERS_256 */

    /*
     *  test DH public key print
     */
#if defined(WOLFSSL_DH_EXTRA) && defined(USE_CERT_BUFFERS_2048)

    ExpectNotNull(rbio = BIO_new_mem_buf( dh_pub_key_der_2048,
        sizeof_dh_pub_key_der_2048));

    ExpectNotNull(wolfSSL_d2i_PUBKEY_bio(rbio, &pkey));

    ExpectNotNull(wbio = BIO_new(BIO_s_mem()));

    ExpectIntEQ(EVP_PKEY_print_public(wbio, pkey,0,NULL), 1);

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "DH Public-Key: (2048 bit)\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "public-key:\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1,
        "    34:41:BF:E9:F2:11:BF:05:DB:B2:72:A8:29:CC:BD:\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    /* skip to the end of public-key element*/
    for (i = 0; i < 17 ;i++) {
        ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    }

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "prime:\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1,
        "    00:D3:B2:99:84:5C:0A:4C:E7:37:CC:FC:18:37:01:\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    /* skip to the end of prime element*/
    for (i = 0; i < 17 ;i++) {
        ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    }

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "generator: 2 (0x02)\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    /* should reach EOF */
    ExpectIntLE(BIO_gets(wbio, line, sizeof(line)), 0);

    EVP_PKEY_free(pkey);
    pkey = NULL;
    BIO_free(rbio);
    BIO_free(wbio);
    rbio = NULL;
    wbio = NULL;

#endif /* WOLFSSL_DH_EXTRA && USE_CERT_BUFFERS_2048 */

    /* to prevent "unused variable" warning */
    (void)pkey;
    (void)wbio;
    (void)rbio;
    (void)line;
    (void)line1;
    (void)i;
#endif /* OPENSSL_EXTRA */
    return EXPECT_RESULT();
}
/* Test functions for base64 encode/decode */
static int test_wolfSSL_EVP_ENCODE_CTX_new(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && \
( defined(WOLFSSL_BASE64_ENCODE) || defined(WOLFSSL_BASE64_DECODE))
    EVP_ENCODE_CTX* ctx = NULL;

    ExpectNotNull(ctx = EVP_ENCODE_CTX_new());
    ExpectIntEQ(ctx->remaining,0);
    ExpectIntEQ(ctx->data[0],0);
    ExpectIntEQ(ctx->data[sizeof(ctx->data) -1],0);
    EVP_ENCODE_CTX_free(ctx);
#endif /* OPENSSL_EXTRA && (WOLFSSL_BASE64_ENCODE || WOLFSSL_BASE64_DECODE) */
    return EXPECT_RESULT();
}
static int test_wolfSSL_EVP_ENCODE_CTX_free(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && \
( defined(WOLFSSL_BASE64_ENCODE) || defined(WOLFSSL_BASE64_DECODE))
    EVP_ENCODE_CTX* ctx = NULL;

    ExpectNotNull(ctx = EVP_ENCODE_CTX_new());
    EVP_ENCODE_CTX_free(ctx);
#endif /* OPENSSL_EXTRA && (WOLFSSL_BASE64_ENCODE || WOLFSSL_BASE64_DECODE) */
    return EXPECT_RESULT();
}

static int test_wolfSSL_EVP_EncodeInit(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && defined(WOLFSSL_BASE64_ENCODE)
    EVP_ENCODE_CTX* ctx = NULL;

    ExpectNotNull(ctx = EVP_ENCODE_CTX_new());
    ExpectIntEQ(ctx->remaining, 0);
    ExpectIntEQ(ctx->data[0], 0);
    ExpectIntEQ(ctx->data[sizeof(ctx->data) -1], 0);

    if (ctx != NULL) {
        /* make ctx dirty */
        ctx->remaining = 10;
        XMEMSET(ctx->data, 0x77, sizeof(ctx->data));
    }

    EVP_EncodeInit(ctx);

    ExpectIntEQ(ctx->remaining, 0);
    ExpectIntEQ(ctx->data[0], 0);
    ExpectIntEQ(ctx->data[sizeof(ctx->data) -1], 0);

    EVP_ENCODE_CTX_free(ctx);
#endif /* OPENSSL_EXTRA && WOLFSSL_BASE64_ENCODE*/
    return EXPECT_RESULT();
}
static int test_wolfSSL_EVP_EncodeUpdate(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && defined(WOLFSSL_BASE64_ENCODE)
    int outl;
    int total;

    const unsigned char plain0[] = {"Th"};
    const unsigned char plain1[] = {"This is a base64 encodeing test."};
    const unsigned char plain2[] = {"This is additional data."};

    const unsigned char encBlock0[] = {"VGg="};
    const unsigned char enc0[]   = {"VGg=\n"};
    /* expected encoded result for the first output 64 chars plus trailing LF*/
    const unsigned char enc1[]   = {"VGhpcyBpcyBhIGJhc2U2NCBlbmNvZGVpbmcgdGVzdC5UaGlzIGlzIGFkZGl0aW9u\n"};

    const unsigned char enc2[]   =
    {"VGhpcyBpcyBhIGJhc2U2NCBlbmNvZGVpbmcgdGVzdC5UaGlzIGlzIGFkZGl0aW9u\nYWwgZGF0YS4=\n"};

    unsigned char encOutBuff[300];

    EVP_ENCODE_CTX* ctx = NULL;

    ExpectNotNull(ctx = EVP_ENCODE_CTX_new());

    EVP_EncodeInit(ctx);

    /* illegal parameter test */
    ExpectIntEQ(
        EVP_EncodeUpdate(
            NULL,            /* pass NULL as ctx */
            encOutBuff,
            &outl,
            plain1,
            sizeof(plain1)-1),
        0                    /* expected result code 0: fail */
    );

    ExpectIntEQ(
        EVP_EncodeUpdate(
            ctx,
            NULL,           /* pass NULL as out buff */
            &outl,
            plain1,
            sizeof(plain1)-1),
        0                    /* expected result code 0: fail */
    );

    ExpectIntEQ(
        EVP_EncodeUpdate(
            ctx,
            encOutBuff,
            NULL,            /* pass NULL as outl */
            plain1,
            sizeof(plain1)-1),
        0                    /* expected result code 0: fail */
    );

    ExpectIntEQ(
        EVP_EncodeUpdate(
            ctx,
            encOutBuff,
            &outl,
            NULL,            /* pass NULL as in */
            sizeof(plain1)-1),
        0                    /* expected result code 0: fail */
    );

    ExpectIntEQ(EVP_EncodeBlock(NULL, NULL, 0), -1);

    /* meaningless parameter test */

    ExpectIntEQ(
        EVP_EncodeUpdate(
            ctx,
            encOutBuff,
            &outl,
            plain1,
            0),              /* pass zero input */
        1                    /* expected result code 1: success */
    );

    /* very small data encoding test */

    EVP_EncodeInit(ctx);

    ExpectIntEQ(
        EVP_EncodeUpdate(
            ctx,
            encOutBuff,
            &outl,
            plain0,
            sizeof(plain0)-1),
        1                    /* expected result code 1: success */
    );
    ExpectIntEQ(outl,0);

    if (EXPECT_SUCCESS()) {
        EVP_EncodeFinal(
            ctx,
            encOutBuff + outl,
            &outl);
    }

    ExpectIntEQ( outl, sizeof(enc0)-1);
    ExpectIntEQ(
        XSTRNCMP(
            (const char*)encOutBuff,
            (const char*)enc0,sizeof(enc0) ),
    0);

    XMEMSET( encOutBuff,0, sizeof(encOutBuff));
    ExpectIntEQ(EVP_EncodeBlock(encOutBuff, plain0, sizeof(plain0)-1),
                sizeof(encBlock0)-1);
    ExpectStrEQ(encOutBuff, encBlock0);

    /* pass small size( < 48bytes ) input, then make sure they are not
     * encoded  and just stored in ctx
     */

    EVP_EncodeInit(ctx);

    total = 0;
    outl = 0;
    XMEMSET( encOutBuff,0, sizeof(encOutBuff));

    ExpectIntEQ(
    EVP_EncodeUpdate(
        ctx,
        encOutBuff,         /* buffer for output */
        &outl,              /* size of output */
        plain1,             /* input */
        sizeof(plain1)-1),  /* size of input */
        1);                 /* expected result code 1:success */

    total += outl;

    ExpectIntEQ(outl, 0);  /* no output expected */
    ExpectIntEQ(ctx->remaining, sizeof(plain1) -1);
    ExpectTrue(
        XSTRNCMP((const char*)(ctx->data),
                 (const char*)plain1,
                 ctx->remaining) ==0 );
    ExpectTrue(encOutBuff[0] == 0);

    /* call wolfSSL_EVP_EncodeUpdate again to make it encode
     * the stored data and the new input together
     */
    ExpectIntEQ(
    EVP_EncodeUpdate(
        ctx,
        encOutBuff + outl,  /* buffer for output */
        &outl,              /* size of output */
        plain2,             /* additional input */
        sizeof(plain2) -1), /* size of additional input */
        1);                 /* expected result code 1:success */

    total += outl;

    ExpectIntNE(outl, 0);   /* some output is expected this time*/
    ExpectIntEQ(outl, BASE64_ENCODE_RESULT_BLOCK_SIZE +1); /* 64 bytes and LF */
    ExpectIntEQ(
        XSTRNCMP((const char*)encOutBuff,(const char*)enc1,sizeof(enc1) ),0);

    /* call wolfSSL_EVP_EncodeFinal to flush all the unprocessed input */
    EVP_EncodeFinal(
        ctx,
        encOutBuff + outl,
        &outl);

    total += outl;

    ExpectIntNE(total,0);
    ExpectIntNE(outl,0);
    ExpectIntEQ(XSTRNCMP(
        (const char*)encOutBuff,(const char*)enc2,sizeof(enc2) ),0);

    /* test with illeagal parameters */
    outl = 1;
    EVP_EncodeFinal(NULL, encOutBuff + outl, &outl);
    ExpectIntEQ(outl, 0);
    outl = 1;
    EVP_EncodeFinal(ctx, NULL, &outl);
    ExpectIntEQ(outl, 0);
    EVP_EncodeFinal(ctx, encOutBuff + outl, NULL);
    EVP_EncodeFinal(NULL, NULL, NULL);

    EVP_ENCODE_CTX_free(ctx);
#endif /* OPENSSL_EXTRA && WOLFSSL_BASE64_ENCODE*/
    return EXPECT_RESULT();
}
static int test_wolfSSL_EVP_EncodeFinal(void)
{
    int res = TEST_SKIPPED;
#if defined(OPENSSL_EXTRA) && defined(WOLFSSL_BASE64_ENCODE)
    /* tests for wolfSSL_EVP_EncodeFinal are included in
     * test_wolfSSL_EVP_EncodeUpdate
     */
    res = TEST_SUCCESS;
#endif /* OPENSSL_EXTRA && WOLFSSL_BASE64_ENCODE*/
    return res;
}


static int test_wolfSSL_EVP_DecodeInit(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && defined(WOLFSSL_BASE64_DECODE)
    EVP_ENCODE_CTX* ctx = NULL;

    ExpectNotNull( ctx = EVP_ENCODE_CTX_new());
    ExpectIntEQ( ctx->remaining,0);
    ExpectIntEQ( ctx->data[0],0);
    ExpectIntEQ( ctx->data[sizeof(ctx->data) -1],0);

    if (ctx != NULL) {
        /* make ctx dirty */
        ctx->remaining = 10;
        XMEMSET( ctx->data, 0x77, sizeof(ctx->data));
    }

    EVP_DecodeInit(ctx);

    ExpectIntEQ( ctx->remaining,0);
    ExpectIntEQ( ctx->data[0],0);
    ExpectIntEQ( ctx->data[sizeof(ctx->data) -1],0);

    EVP_ENCODE_CTX_free(ctx);
#endif /* OPENSSL && WOLFSSL_BASE_DECODE */
    return EXPECT_RESULT();
}
static int test_wolfSSL_EVP_DecodeUpdate(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && defined(WOLFSSL_BASE64_DECODE)
    int outl;
    unsigned char decOutBuff[300];

    EVP_ENCODE_CTX* ctx = NULL;

    static const unsigned char enc1[]   =
            {"VGhpcyBpcyBhIGJhc2U2NCBkZWNvZGluZyB0ZXN0Lg==\n"};
/*    const unsigned char plain1[] =
    {"This is a base64 decoding test."} */

    ExpectNotNull(ctx = EVP_ENCODE_CTX_new());

    EVP_DecodeInit(ctx);

    /* illegal parameter tests */

    /* pass NULL as ctx */
    ExpectIntEQ(
        EVP_DecodeUpdate(
            NULL,            /* pass NULL as ctx */
            decOutBuff,
            &outl,
            enc1,
            sizeof(enc1)-1),
        -1                    /* expected result code -1: fail */
    );
    ExpectIntEQ( outl, 0);

    /* pass NULL as output */
    ExpectIntEQ(
        EVP_DecodeUpdate(
            ctx,
            NULL,           /* pass NULL as out buff */
            &outl,
            enc1,
            sizeof(enc1)-1),
        -1                    /* expected result code -1: fail */
    );
    ExpectIntEQ( outl, 0);

    /* pass NULL as outl */
    ExpectIntEQ(
        EVP_DecodeUpdate(
            ctx,
            decOutBuff,
            NULL,            /* pass NULL as outl */
            enc1,
            sizeof(enc1)-1),
        -1                   /* expected result code -1: fail */
    );

    /* pass NULL as input */
    ExpectIntEQ(
        EVP_DecodeUpdate(
            ctx,
            decOutBuff,
            &outl,
            NULL,            /* pass NULL as in */
            sizeof(enc1)-1),
        -1                    /* expected result code -1: fail */
    );
    ExpectIntEQ( outl, 0);

    ExpectIntEQ(EVP_DecodeBlock(NULL, NULL, 0), -1);

    /* pass zero length input */

    ExpectIntEQ(
        EVP_DecodeUpdate(
            ctx,
            decOutBuff,
            &outl,
            enc1,
            0),              /* pass zero as input len */
        1                    /* expected result code 1: success */
    );

    /* decode correct base64 string */

    {
        static const unsigned char enc2[]   =
        {"VGhpcyBpcyBhIGJhc2U2NCBkZWNvZGluZyB0ZXN0Lg==\n"};
        static const unsigned char plain2[] =
        {"This is a base64 decoding test."};

        EVP_EncodeInit(ctx);

        ExpectIntEQ(
            EVP_DecodeUpdate(
                ctx,
                decOutBuff,
                &outl,
                enc2,
                sizeof(enc2)-1),
            0                    /* expected result code 0: success */
            );

        ExpectIntEQ(outl,sizeof(plain2) -1);

        ExpectIntEQ(
            EVP_DecodeFinal(
                ctx,
                decOutBuff + outl,
                &outl),
            1                    /* expected result code 1: success */
            );
        ExpectIntEQ(outl, 0);   /* expected DecodeFinal output no data */

        ExpectIntEQ(XSTRNCMP( (const char*)plain2,(const char*)decOutBuff,
                              sizeof(plain2) -1 ),0);
        ExpectIntEQ(EVP_DecodeBlock(decOutBuff, enc2, sizeof(enc2)),
                    sizeof(plain2)-1);
        ExpectIntEQ(XSTRNCMP( (const char*)plain2,(const char*)decOutBuff,
                              sizeof(plain2) -1 ),0);
    }

    /* decode correct base64 string which does not have '\n' in its last*/

    {
        static const unsigned char enc3[]   =
        {"VGhpcyBpcyBhIGJhc2U2NCBkZWNvZGluZyB0ZXN0Lg=="}; /* 44 chars */
        static const unsigned char plain3[] =
        {"This is a base64 decoding test."}; /* 31 chars */

        EVP_EncodeInit(ctx);

        ExpectIntEQ(
            EVP_DecodeUpdate(
                ctx,
                decOutBuff,
                &outl,
                enc3,
                sizeof(enc3)-1),
            0                    /* expected result code 0: success */
            );

        ExpectIntEQ(outl,sizeof(plain3)-1);   /* 31 chars should be output */

        ExpectIntEQ(XSTRNCMP( (const char*)plain3,(const char*)decOutBuff,
                              sizeof(plain3) -1 ),0);

        ExpectIntEQ(
            EVP_DecodeFinal(
                ctx,
                decOutBuff + outl,
                &outl),
            1                    /* expected result code 1: success */
            );

        ExpectIntEQ(outl,0 );

        ExpectIntEQ(EVP_DecodeBlock(decOutBuff, enc3, sizeof(enc3)-1),
                    sizeof(plain3)-1);
        ExpectIntEQ(XSTRNCMP( (const char*)plain3,(const char*)decOutBuff,
                              sizeof(plain3) -1 ),0);
    }

    /* decode string which has a padding char ('=') in the illegal position*/

    {
        static const unsigned char enc4[]   =
            {"VGhpcyBpcyBhIGJhc2U2N=CBkZWNvZGluZyB0ZXN0Lg==\n"};

        EVP_EncodeInit(ctx);

        ExpectIntEQ(
            EVP_DecodeUpdate(
                ctx,
                decOutBuff,
                &outl,
                enc4,
                sizeof(enc4)-1),
            -1                    /* expected result code -1: error */
            );
        ExpectIntEQ(outl,0);
        ExpectIntEQ(EVP_DecodeBlock(decOutBuff, enc4, sizeof(enc4)-1), -1);
    }

    /* small data decode test */

    {
        static const unsigned char enc00[]   = {"VG"};
        static const unsigned char enc01[]   = {"g=\n"};
        static const unsigned char plain4[]  = {"Th"};

        EVP_EncodeInit(ctx);

        ExpectIntEQ(
            EVP_DecodeUpdate(
                ctx,
                decOutBuff,
                &outl,
                enc00,
                sizeof(enc00)-1),
            1                    /* expected result code 1: success */
            );
        ExpectIntEQ(outl,0);

        ExpectIntEQ(
            EVP_DecodeUpdate(
                ctx,
                decOutBuff + outl,
                &outl,
                enc01,
                sizeof(enc01)-1),
            0                    /* expected result code 0: success */
            );

        ExpectIntEQ(outl,sizeof(plain4)-1);

        /* test with illegal parameters */
        ExpectIntEQ(EVP_DecodeFinal(NULL,decOutBuff + outl,&outl), -1);
        ExpectIntEQ(EVP_DecodeFinal(ctx,NULL,&outl), -1);
        ExpectIntEQ(EVP_DecodeFinal(ctx,decOutBuff + outl, NULL), -1);
        ExpectIntEQ(EVP_DecodeFinal(NULL,NULL, NULL), -1);

        if (EXPECT_SUCCESS()) {
            EVP_DecodeFinal(
                ctx,
                decOutBuff + outl,
                &outl);
        }

        ExpectIntEQ( outl, 0);
        ExpectIntEQ(
            XSTRNCMP(
                (const char*)decOutBuff,
                (const char*)plain4,sizeof(plain4)-1 ),
            0);
    }

    EVP_ENCODE_CTX_free(ctx);
#endif /* OPENSSL && WOLFSSL_BASE_DECODE */
    return EXPECT_RESULT();
}
static int test_wolfSSL_EVP_DecodeFinal(void)
{
    int res = TEST_SKIPPED;
#if defined(OPENSSL_EXTRA) && defined(WOLFSSL_BASE64_DECODE)
    /* tests for wolfSSL_EVP_DecodeFinal are included in
     * test_wolfSSL_EVP_DecodeUpdate
     */
    res = TEST_SUCCESS;
#endif /* OPENSSL && WOLFSSL_BASE_DECODE */
    return res;
}

/* Test function for wolfSSL_EVP_get_cipherbynid.
 */

#ifdef OPENSSL_EXTRA
static int test_wolfSSL_EVP_get_cipherbynid(void)
{
    EXPECT_DECLS;
#ifndef NO_AES
    const WOLFSSL_EVP_CIPHER* c;

    c = wolfSSL_EVP_get_cipherbynid(419);
    #if (defined(HAVE_AES_CBC) || defined(WOLFSSL_AES_DIRECT)) && \
         defined(WOLFSSL_AES_128)
        ExpectNotNull(c);
        ExpectNotNull(XSTRCMP("EVP_AES_128_CBC", c));
    #else
        ExpectNull(c);
    #endif

    c = wolfSSL_EVP_get_cipherbynid(423);
    #if (defined(HAVE_AES_CBC) || defined(WOLFSSL_AES_DIRECT)) && \
         defined(WOLFSSL_AES_192)
        ExpectNotNull(c);
        ExpectNotNull(XSTRCMP("EVP_AES_192_CBC", c));
    #else
        ExpectNull(c);
    #endif

    c = wolfSSL_EVP_get_cipherbynid(427);
    #if (defined(HAVE_AES_CBC) || defined(WOLFSSL_AES_DIRECT)) && \
         defined(WOLFSSL_AES_256)
        ExpectNotNull(c);
        ExpectNotNull(XSTRCMP("EVP_AES_256_CBC", c));
    #else
        ExpectNull(c);
    #endif

    c = wolfSSL_EVP_get_cipherbynid(904);
    #if defined(WOLFSSL_AES_COUNTER) && defined(WOLFSSL_AES_128)
        ExpectNotNull(c);
        ExpectNotNull(XSTRCMP("EVP_AES_128_CTR", c));
    #else
        ExpectNull(c);
    #endif

    c = wolfSSL_EVP_get_cipherbynid(905);
    #if defined(WOLFSSL_AES_COUNTER) && defined(WOLFSSL_AES_192)
        ExpectNotNull(c);
        ExpectNotNull(XSTRCMP("EVP_AES_192_CTR", c));
    #else
        ExpectNull(c);
    #endif

    c = wolfSSL_EVP_get_cipherbynid(906);
    #if defined(WOLFSSL_AES_COUNTER) && defined(WOLFSSL_AES_256)
        ExpectNotNull(c);
        ExpectNotNull(XSTRCMP("EVP_AES_256_CTR", c));
    #else
        ExpectNull(c);
    #endif

    c = wolfSSL_EVP_get_cipherbynid(418);
    #if defined(HAVE_AES_ECB) && defined(WOLFSSL_AES_128)
        ExpectNotNull(c);
        ExpectNotNull(XSTRCMP("EVP_AES_128_ECB", c));
    #else
        ExpectNull(c);
    #endif

    c = wolfSSL_EVP_get_cipherbynid(422);
    #if defined(HAVE_AES_ECB) && defined(WOLFSSL_AES_192)
        ExpectNotNull(c);
        ExpectNotNull(XSTRCMP("EVP_AES_192_ECB", c));
    #else
        ExpectNull(c);
    #endif

    c = wolfSSL_EVP_get_cipherbynid(426);
    #if defined(HAVE_AES_ECB) && defined(WOLFSSL_AES_256)
        ExpectNotNull(c);
        ExpectNotNull(XSTRCMP("EVP_AES_256_ECB", c));
    #else
        ExpectNull(c);
    #endif
#endif /* !NO_AES */

#ifndef NO_DES3
    ExpectNotNull(XSTRCMP("EVP_DES_CBC", wolfSSL_EVP_get_cipherbynid(31)));
#ifdef WOLFSSL_DES_ECB
    ExpectNotNull(XSTRCMP("EVP_DES_ECB", wolfSSL_EVP_get_cipherbynid(29)));
#endif
    ExpectNotNull(XSTRCMP("EVP_DES_EDE3_CBC", wolfSSL_EVP_get_cipherbynid(44)));
#ifdef WOLFSSL_DES_ECB
    ExpectNotNull(XSTRCMP("EVP_DES_EDE3_ECB", wolfSSL_EVP_get_cipherbynid(33)));
#endif
#endif /* !NO_DES3 */

#if defined(HAVE_CHACHA) && defined(HAVE_POLY1305)
    ExpectNotNull(XSTRCMP("EVP_CHACHA20_POLY13O5", EVP_get_cipherbynid(1018)));
#endif

    /* test for nid is out of range */
    ExpectNull(wolfSSL_EVP_get_cipherbynid(1));

    return EXPECT_RESULT();
}

static int test_wolfSSL_EVP_CIPHER_CTX(void)
{
    EXPECT_DECLS;
#if !defined(NO_AES) && defined(HAVE_AES_CBC) && defined(WOLFSSL_AES_128)
    EVP_CIPHER_CTX *ctx = EVP_CIPHER_CTX_new();
    const EVP_CIPHER *init = EVP_aes_128_cbc();
    const EVP_CIPHER *test;
    byte key[AES_BLOCK_SIZE] = {0};
    byte iv[AES_BLOCK_SIZE] = {0};

    ExpectNotNull(ctx);
    wolfSSL_EVP_CIPHER_CTX_init(ctx);
    ExpectIntEQ(EVP_CipherInit(ctx, init, key, iv, 1), WOLFSSL_SUCCESS);
    test = EVP_CIPHER_CTX_cipher(ctx);
    ExpectTrue(init == test);
    ExpectIntEQ(EVP_CIPHER_nid(test), NID_aes_128_cbc);

    ExpectIntEQ(EVP_CIPHER_CTX_reset(ctx), WOLFSSL_SUCCESS);
    ExpectIntEQ(EVP_CIPHER_CTX_reset(NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

    EVP_CIPHER_CTX_free(ctx);
    /* test EVP_CIPHER_CTX_cleanup with NULL */
    ExpectIntEQ(EVP_CIPHER_CTX_cleanup(NULL), WOLFSSL_SUCCESS);
#endif /* !NO_AES && HAVE_AES_CBC && WOLFSSL_AES_128 */
    return EXPECT_RESULT();
}
#endif /* OPENSSL_EXTRA */

/*----------------------------------------------------------------------------*
 | IO
 *----------------------------------------------------------------------------*/

#if defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES) || \
    defined(HAVE_IO_TESTS_DEPENDENCIES)
#ifdef WOLFSSL_HAVE_TLS_UNIQUE
    byte server_side_msg1[WC_MAX_DIGEST_SIZE]; /* msg sent by server */
    byte server_side_msg2[WC_MAX_DIGEST_SIZE]; /* msg received from client */
    byte client_side_msg1[WC_MAX_DIGEST_SIZE]; /* msg sent by client */
    byte client_side_msg2[WC_MAX_DIGEST_SIZE]; /* msg received from server */
#endif /* WOLFSSL_HAVE_TLS_UNIQUE */

/* TODO: Expand and enable this when EVP_chacha20_poly1305 is supported */
#if defined(HAVE_SESSION_TICKET) && defined(OPENSSL_EXTRA) && \
    defined(HAVE_AES_CBC) && defined(WOLFSSL_AES_256)

    typedef struct openssl_key_ctx {
        byte name[WOLFSSL_TICKET_NAME_SZ]; /* server name */
        byte key[WOLFSSL_TICKET_KEY_SZ]; /* cipher key */
        byte hmacKey[WOLFSSL_TICKET_NAME_SZ]; /* hmac key */
        byte iv[WOLFSSL_TICKET_IV_SZ]; /* cipher iv */
    } openssl_key_ctx;

    static THREAD_LS_T openssl_key_ctx myOpenSSLKey_ctx;
    static THREAD_LS_T WC_RNG myOpenSSLKey_rng;

    static WC_INLINE int OpenSSLTicketInit(void)
    {
        int ret = wc_InitRng(&myOpenSSLKey_rng);
        if (ret != 0) return ret;

        ret = wc_RNG_GenerateBlock(&myOpenSSLKey_rng, myOpenSSLKey_ctx.name,
                sizeof(myOpenSSLKey_ctx.name));
        if (ret != 0) return ret;

        ret = wc_RNG_GenerateBlock(&myOpenSSLKey_rng, myOpenSSLKey_ctx.key,
                sizeof(myOpenSSLKey_ctx.key));
        if (ret != 0) return ret;

        ret = wc_RNG_GenerateBlock(&myOpenSSLKey_rng, myOpenSSLKey_ctx.hmacKey,
                sizeof(myOpenSSLKey_ctx.hmacKey));
        if (ret != 0) return ret;

        ret = wc_RNG_GenerateBlock(&myOpenSSLKey_rng, myOpenSSLKey_ctx.iv,
                sizeof(myOpenSSLKey_ctx.iv));
        if (ret != 0) return ret;

        return 0;
    }

    static int myTicketEncCbOpenSSL(WOLFSSL* ssl,
                             byte name[WOLFSSL_TICKET_NAME_SZ],
                             byte iv[WOLFSSL_TICKET_IV_SZ],
                             WOLFSSL_EVP_CIPHER_CTX *ectx,
                             WOLFSSL_HMAC_CTX *hctx, int enc) {
        (void)ssl;
        if (enc) {
            XMEMCPY(name, myOpenSSLKey_ctx.name, sizeof(myOpenSSLKey_ctx.name));
            XMEMCPY(iv, myOpenSSLKey_ctx.iv, sizeof(myOpenSSLKey_ctx.iv));
        }
        else if (XMEMCMP(name, myOpenSSLKey_ctx.name,
                            sizeof(myOpenSSLKey_ctx.name)) != 0 ||
                 XMEMCMP(iv, myOpenSSLKey_ctx.iv,
                            sizeof(myOpenSSLKey_ctx.iv)) != 0) {
            return 0;
        }
        HMAC_Init_ex(hctx, myOpenSSLKey_ctx.hmacKey, WOLFSSL_TICKET_NAME_SZ, EVP_sha256(), NULL);
        if (enc)
            EVP_EncryptInit_ex(ectx, EVP_aes_256_cbc(), NULL, myOpenSSLKey_ctx.key, iv);
        else
            EVP_DecryptInit_ex(ectx, EVP_aes_256_cbc(), NULL, myOpenSSLKey_ctx.key, iv);
        return 1;
    }

    static WC_INLINE void OpenSSLTicketCleanup(void)
    {
        wc_FreeRng(&myOpenSSLKey_rng);
    }
#endif
#endif

/* helper functions */
#ifdef HAVE_SSL_MEMIO_TESTS_DEPENDENCIES
static WC_INLINE int test_ssl_memio_write_cb(WOLFSSL *ssl, char *data, int sz,
    void *ctx)
{
    struct test_ssl_memio_ctx *test_ctx;
    byte *buf;
    int *len;
    int *msg_sizes;
    int *msg_count;

    test_ctx = (struct test_ssl_memio_ctx*)ctx;

    if (wolfSSL_GetSide(ssl) == WOLFSSL_SERVER_END) {
        buf = test_ctx->c_buff;
        len = &test_ctx->c_len;
        msg_sizes = test_ctx->c_msg_sizes;
        msg_count = &test_ctx->c_msg_count;
    }
    else {
        buf = test_ctx->s_buff;
        len = &test_ctx->s_len;
        msg_sizes = test_ctx->s_msg_sizes;
        msg_count = &test_ctx->s_msg_count;
    }

    if ((unsigned)(*len + sz) > TEST_SSL_MEMIO_BUF_SZ)
        return WOLFSSL_CBIO_ERR_WANT_WRITE;

    if (*msg_count >= TEST_MEMIO_MAX_MSGS)
        return WOLFSSL_CBIO_ERR_WANT_WRITE;

    XMEMCPY(buf + *len, data, sz);
    msg_sizes[*msg_count] = sz;
    (*msg_count)++;
    *len += sz;

#ifdef WOLFSSL_DUMP_MEMIO_STREAM
    {
        /* This can be imported into Wireshark by transforming the file with
         *   od -Ax -tx1 -v test_output.dump > test_output.dump.hex
         * And then loading test_output.dump.hex into Wireshark using the
         * "Import from Hex Dump..." option ion and selecting the TCP
         * encapsulation option. */
        char dump_file_name[64];
        WOLFSSL_BIO *dump_file;
        sprintf(dump_file_name, "%s/%s.dump", tmpDirName, currentTestName);
        dump_file = wolfSSL_BIO_new_file(dump_file_name, "a");
        if (dump_file != NULL) {
            (void)wolfSSL_BIO_write(dump_file, data, sz);
            wolfSSL_BIO_free(dump_file);
        }
    }
#endif

    return sz;
}

static WC_INLINE int test_ssl_memio_read_cb(WOLFSSL *ssl, char *data, int sz,
    void *ctx)
{
    struct test_ssl_memio_ctx *test_ctx;
    int read_sz;
    byte *buf;
    int *len;
    int *msg_sizes;
    int *msg_count;
    int *msg_pos;
    int is_dtls;

    test_ctx = (struct test_ssl_memio_ctx*)ctx;
    is_dtls = wolfSSL_dtls(ssl);

    if (wolfSSL_GetSide(ssl) == WOLFSSL_SERVER_END) {
        buf = test_ctx->s_buff;
        len = &test_ctx->s_len;
        msg_sizes = test_ctx->s_msg_sizes;
        msg_count = &test_ctx->s_msg_count;
        msg_pos = &test_ctx->s_msg_pos;
    }
    else {
        buf = test_ctx->c_buff;
        len = &test_ctx->c_len;
        msg_sizes = test_ctx->c_msg_sizes;
        msg_count = &test_ctx->c_msg_count;
        msg_pos = &test_ctx->c_msg_pos;
    }

    if (*len == 0 || *msg_pos >= *msg_count)
        return WOLFSSL_CBIO_ERR_WANT_READ;

    /* Calculate how much we can read from current message */
    read_sz = msg_sizes[*msg_pos];
    if (read_sz > sz)
        read_sz = sz;

    if (read_sz > *len)
        return WOLFSSL_CBIO_ERR_GENERAL;

    /* Copy data from current message */
    XMEMCPY(data, buf, (size_t)read_sz);
    /* remove the read data from the buffer */
    XMEMMOVE(buf, buf + read_sz, (size_t)(*len - read_sz));
    *len -= read_sz;
    msg_sizes[*msg_pos] -= read_sz;

    /* if we are on dtls, discard the rest of the message */
    if (is_dtls && msg_sizes[*msg_pos] > 0) {
        XMEMMOVE(buf, buf + msg_sizes[*msg_pos], (size_t)(*len - msg_sizes[*msg_pos]));
        *len -= msg_sizes[*msg_pos];
        msg_sizes[*msg_pos] = 0;
    }

    /* If we've read the entire message */
    if (msg_sizes[*msg_pos] == 0) {
        /* Move to next message */
        (*msg_pos)++;
        if (*msg_pos >= *msg_count) {
            *msg_pos = 0;
            *msg_count = 0;
        }
    }

    return read_sz;
}

int test_ssl_memio_setup(test_ssl_memio_ctx *ctx)
{
    EXPECT_DECLS_NO_MSGS(-2000);
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    int c_sharedCtx = 0;
    int s_sharedCtx = 0;
#endif
    const char* clientCertFile = cliCertFile;
    const char* clientKeyFile = cliKeyFile;
    const char* serverCertFile = svrCertFile;
    const char* serverKeyFile = svrKeyFile;

    /********************************
     * Create WOLFSSL_CTX for client.
     ********************************/
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (ctx->c_ctx != NULL) {
        c_sharedCtx = ctx->c_cb.isSharedCtx;
    }
    else
#endif
    {
        WOLFSSL_METHOD* method  = NULL;
        if (ctx->c_cb.method != NULL) {
            method = ctx->c_cb.method();
        }
        else {
            method = wolfSSLv23_client_method();
        }
        ExpectNotNull(ctx->c_ctx = wolfSSL_CTX_new(method));
    }
    wolfSSL_SetIORecv(ctx->c_ctx, test_ssl_memio_read_cb);
    wolfSSL_SetIOSend(ctx->c_ctx, test_ssl_memio_write_cb);
#ifdef WOLFSSL_ENCRYPTED_KEYS
    wolfSSL_CTX_set_default_passwd_cb(ctx->c_ctx, PasswordCallBack);
#endif
    if (ctx->c_cb.caPemFile == NULL)
        ExpectIntEQ(wolfSSL_CTX_load_verify_locations(ctx->c_ctx,
                caCertFile, 0), WOLFSSL_SUCCESS);
    else if (*ctx->c_cb.caPemFile != '\0')
        ExpectIntEQ(wolfSSL_CTX_load_verify_locations(ctx->c_ctx,
                ctx->c_cb.caPemFile, 0), WOLFSSL_SUCCESS);
    if (ctx->c_cb.certPemFile != NULL) {
        clientCertFile = ctx->c_cb.certPemFile;
    }
    if (ctx->c_cb.keyPemFile != NULL) {
        clientKeyFile = ctx->c_cb.keyPemFile;
    }
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (!c_sharedCtx)
#endif
    {
        if (*clientCertFile != '\0') {
            ExpectIntEQ(wolfSSL_CTX_use_certificate_chain_file(ctx->c_ctx,
                clientCertFile), WOLFSSL_SUCCESS);
        }
        if (*clientKeyFile != '\0') {
            ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_file(ctx->c_ctx, clientKeyFile,
                CERT_FILETYPE), WOLFSSL_SUCCESS);
        }
    }
#ifdef HAVE_CRL
    if (ctx->c_cb.crlPemFile != NULL) {
        ExpectIntEQ(wolfSSL_CTX_EnableCRL(ctx->c_ctx, WOLFSSL_CRL_CHECKALL),
            WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_CTX_LoadCRLFile(ctx->c_ctx, ctx->c_cb.crlPemFile,
            CERT_FILETYPE), WOLFSSL_SUCCESS);
    }
#endif
    if (ctx->c_ciphers != NULL) {
        ExpectIntEQ(wolfSSL_CTX_set_cipher_list(ctx->c_ctx, ctx->c_ciphers),
            WOLFSSL_SUCCESS);
    }
    if (ctx->c_cb.ctx_ready != NULL) {
        ExpectIntEQ(ctx->c_cb.ctx_ready(ctx->c_ctx), TEST_SUCCESS);
    }


    /********************************
     * Create WOLFSSL_CTX for server.
     ********************************/
    if (ctx->s_ctx != NULL) {
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
        s_sharedCtx = 1;
#endif
        ctx->s_cb.isSharedCtx = 1;
    }
    else
    {
        WOLFSSL_METHOD* method  = NULL;
        if (ctx->s_cb.method != NULL) {
            method = ctx->s_cb.method();
        }
        else {
            method = wolfSSLv23_server_method();
        }
        ExpectNotNull(ctx->s_ctx = wolfSSL_CTX_new(method));
        ctx->s_cb.isSharedCtx = 0;
    }
    if (!ctx->s_cb.ticNoInit && (ctx->s_ctx != NULL)) {
#if defined(HAVE_SESSION_TICKET) && \
    ((defined(HAVE_CHACHA) && defined(HAVE_POLY1305)) || defined(HAVE_AESGCM))
#if defined(OPENSSL_EXTRA) && defined(HAVE_AES_CBC) && defined(WOLFSSL_AES_256)
        OpenSSLTicketInit();
        wolfSSL_CTX_set_tlsext_ticket_key_cb(ctx->s_ctx, myTicketEncCbOpenSSL);
#elif defined(WOLFSSL_NO_DEF_TICKET_ENC_CB)
        TicketInit();
        wolfSSL_CTX_set_TicketEncCb(ctx->s_ctx, myTicketEncCb);
#endif
#endif
    }
    wolfSSL_SetIORecv(ctx->s_ctx, test_ssl_memio_read_cb);
    wolfSSL_SetIOSend(ctx->s_ctx, test_ssl_memio_write_cb);
    wolfSSL_CTX_set_verify(ctx->s_ctx, WOLFSSL_VERIFY_PEER |
        WOLFSSL_VERIFY_FAIL_IF_NO_PEER_CERT, 0);
    if (ctx->s_cb.caPemFile == NULL)
        ExpectIntEQ(wolfSSL_CTX_load_verify_locations(ctx->s_ctx,
                cliCertFile, 0), WOLFSSL_SUCCESS);
    else if (*ctx->s_cb.caPemFile != '\0')
        ExpectIntEQ(wolfSSL_CTX_load_verify_locations(ctx->s_ctx,
                ctx->s_cb.caPemFile, 0), WOLFSSL_SUCCESS);
#ifdef WOLFSSL_ENCRYPTED_KEYS
    wolfSSL_CTX_set_default_passwd_cb(ctx->s_ctx, PasswordCallBack);
#endif
    if (ctx->s_cb.certPemFile != NULL) {
        serverCertFile = ctx->s_cb.certPemFile;
    }
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (!s_sharedCtx)
#endif
    {
        if (*serverCertFile != '\0') {
            ExpectIntEQ(wolfSSL_CTX_use_certificate_chain_file(ctx->s_ctx,
                serverCertFile), WOLFSSL_SUCCESS);
        }
    }
    if (ctx->s_cb.keyPemFile != NULL) {
        serverKeyFile = ctx->s_cb.keyPemFile;
    }
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (!s_sharedCtx)
#endif
    {
        if (*serverKeyFile != '\0') {
            ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_file(ctx->s_ctx, serverKeyFile,
                CERT_FILETYPE), WOLFSSL_SUCCESS);
        }
    }
    if (ctx->s_ciphers != NULL) {
        ExpectIntEQ(wolfSSL_CTX_set_cipher_list(ctx->s_ctx, ctx->s_ciphers),
            WOLFSSL_SUCCESS);
    }
    if (ctx->s_cb.ctx_ready != NULL) {
        ExpectIntEQ(ctx->s_cb.ctx_ready(ctx->s_ctx), TEST_SUCCESS);
    }


    /****************************
     * Create WOLFSSL for client.
     ****************************/
    ExpectNotNull(ctx->c_ssl = wolfSSL_new(ctx->c_ctx));
    wolfSSL_SetIOWriteCtx(ctx->c_ssl, ctx);
    wolfSSL_SetIOReadCtx(ctx->c_ssl, ctx);
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (c_sharedCtx) {
        if (*clientCertFile != '\0') {
            ExpectIntEQ(wolfSSL_use_certificate_chain_file(ctx->c_ssl,
                    clientCertFile), WOLFSSL_SUCCESS);
        }
        if (*clientKeyFile != '\0') {
            ExpectIntEQ(wolfSSL_use_PrivateKey_file(ctx->c_ssl, clientKeyFile,
                CERT_FILETYPE), WOLFSSL_SUCCESS);
        }
    }
#endif
    if (ctx->c_cb.ssl_ready != NULL) {
        ExpectIntEQ(ctx->c_cb.ssl_ready(ctx->c_ssl), TEST_SUCCESS);
    }

    /****************************
     * Create WOLFSSL for server.
     ****************************/
    ExpectNotNull(ctx->s_ssl = wolfSSL_new(ctx->s_ctx));
    wolfSSL_SetIOWriteCtx(ctx->s_ssl, ctx);
    wolfSSL_SetIOReadCtx(ctx->s_ssl, ctx);
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (s_sharedCtx) {
        if (*serverCertFile != '\0') {
            ExpectIntEQ(wolfSSL_use_certificate_chain_file(ctx->s_ssl,
                    serverCertFile), WOLFSSL_SUCCESS);
        }
        if (*serverKeyFile != '\0') {
            ExpectIntEQ(wolfSSL_use_PrivateKey_file(ctx->s_ssl, serverKeyFile,
                CERT_FILETYPE), WOLFSSL_SUCCESS);
        }
    }
#endif
#if !defined(NO_FILESYSTEM) && !defined(NO_DH)
    wolfSSL_SetTmpDH_file(ctx->s_ssl, dhParamFile, CERT_FILETYPE);
#elif !defined(NO_DH)
    /* will repick suites with DHE, higher priority than PSK */
    SetDH(ctx->s_ssl);
#endif
    if (ctx->s_cb.ssl_ready != NULL) {
        ExpectIntEQ(ctx->s_cb.ssl_ready(ctx->s_ssl), TEST_SUCCESS);
    }

    return EXPECT_RESULT();
}

int test_ssl_memio_do_handshake(test_ssl_memio_ctx* ctx, int max_rounds,
    int* rounds)
{
    int handshake_complete = 0;
    int hs_c = 0;
    int hs_s = 0;
    int failing_s = 0;
    int failing_c = 0;
    int ret;
    int err;

    if (rounds != NULL) {
        *rounds = 0;
    }
    while ((!handshake_complete) && (max_rounds > 0)) {
        if (!hs_c) {
            wolfSSL_SetLoggingPrefix("client");
            ret = wolfSSL_connect(ctx->c_ssl);
            wolfSSL_SetLoggingPrefix(NULL);
            if (ret == WOLFSSL_SUCCESS) {
                hs_c = 1;
            }
            else {
                err = wolfSSL_get_error(ctx->c_ssl, ret);
                if (err != WOLFSSL_ERROR_WANT_READ &&
                    err != WOLFSSL_ERROR_WANT_WRITE) {
                    char buff[WOLFSSL_MAX_ERROR_SZ];
                    fprintf(stderr, "error = %d, %s\n", err,
                        wolfSSL_ERR_error_string((word32)err, buff));
                    failing_c = 1;
                    hs_c = 1;
                    if (failing_c && failing_s) {
                        break;
                    }
                }
            }
        }
        if (!hs_s) {
            wolfSSL_SetLoggingPrefix("server");
            ret = wolfSSL_accept(ctx->s_ssl);
            wolfSSL_SetLoggingPrefix(NULL);
            if (ret == WOLFSSL_SUCCESS) {
                hs_s = 1;
            }
            else {
                err = wolfSSL_get_error(ctx->s_ssl, ret);
                if (err != WOLFSSL_ERROR_WANT_READ &&
                    err != WOLFSSL_ERROR_WANT_WRITE) {
                    char buff[WOLFSSL_MAX_ERROR_SZ];
                    fprintf(stderr, "error = %d, %s\n", err,
                        wolfSSL_ERR_error_string((word32)err, buff));
                    failing_s = 1;
                    hs_s = 1;
                    if (failing_c && failing_s) {
                        break;
                    }
                }
            }
        }
        handshake_complete = hs_c && hs_s;
        max_rounds--;
        if (rounds != NULL) {
            *rounds += 1;
        }
    }

    if (!handshake_complete || failing_c || failing_s) {
        return TEST_FAIL;
    }

    return TEST_SUCCESS;
}

static int test_ssl_memio_read_write(test_ssl_memio_ctx* ctx)
{
    EXPECT_DECLS_NO_MSGS(-3000);
    char input[1024];
    int idx = 0;
    const char* msg_c = "hello wolfssl!";
    int msglen_c = (int)XSTRLEN(msg_c);
    const char* msg_s = "I hear you fa shizzle!";
    int msglen_s = (int)XSTRLEN(msg_s);

    if (ctx->c_msg != NULL) {
        msg_c = ctx->c_msg;
        msglen_c = ctx->c_msglen;
    }
    if (ctx->s_msg != NULL) {
        msg_s = ctx->s_msg;
        msglen_s = ctx->s_msglen;
    }

    wolfSSL_SetLoggingPrefix("client");
    ExpectIntEQ(wolfSSL_write(ctx->c_ssl, msg_c, msglen_c), msglen_c);
    wolfSSL_SetLoggingPrefix("server");
    ExpectIntGT(idx = wolfSSL_read(ctx->s_ssl, input, sizeof(input) - 1), 0);
    if (idx >= 0) {
        input[idx] = '\0';
    }
    ExpectIntGT(fprintf(stderr, "Client message: %s\n", input), 0);
    ExpectIntEQ(wolfSSL_write(ctx->s_ssl, msg_s, msglen_s), msglen_s);
    ctx->s_cb.return_code = EXPECT_RESULT();
    wolfSSL_SetLoggingPrefix("client");
    ExpectIntGT(idx = wolfSSL_read(ctx->c_ssl, input, sizeof(input) - 1), 0);
    wolfSSL_SetLoggingPrefix(NULL);
    if (idx >= 0) {
        input[idx] = '\0';
    }
    ExpectIntGT(fprintf(stderr, "Server response: %s\n", input), 0);
    ctx->c_cb.return_code = EXPECT_RESULT();
    if (ctx->c_cb.on_result != NULL) {
        ExpectIntEQ(ctx->c_cb.on_result(ctx->c_ssl), TEST_SUCCESS);
    }
    if (ctx->s_cb.on_result != NULL) {
        ExpectIntEQ(ctx->s_cb.on_result(ctx->s_ssl), TEST_SUCCESS);
    }

    return EXPECT_RESULT();
}

void test_ssl_memio_cleanup(test_ssl_memio_ctx* ctx)
{
    ctx->c_cb.last_err = wolfSSL_get_error(ctx->c_ssl, 0);
    ctx->s_cb.last_err = wolfSSL_get_error(ctx->s_ssl, 0);
    if (ctx->c_cb.on_cleanup != NULL) {
        ctx->c_cb.on_cleanup(ctx->c_ssl);
    }
    if (ctx->s_cb.on_cleanup != NULL) {
        ctx->s_cb.on_cleanup(ctx->s_ssl);
    }
    wolfSSL_shutdown(ctx->s_ssl);
    wolfSSL_shutdown(ctx->c_ssl);
    wolfSSL_free(ctx->s_ssl);
    wolfSSL_free(ctx->c_ssl);
    if (ctx->c_cb.on_ctx_cleanup != NULL) {
        ctx->c_cb.on_ctx_cleanup(ctx->c_ctx);
    }
    if (!ctx->c_cb.isSharedCtx) {
        wolfSSL_CTX_free(ctx->c_ctx);
        ctx->c_ctx = NULL;
    }
    if (ctx->s_cb.on_ctx_cleanup != NULL) {
        ctx->s_cb.on_ctx_cleanup(ctx->s_ctx);
    }
    if (!ctx->s_cb.isSharedCtx) {
        wolfSSL_CTX_free(ctx->s_ctx);
        ctx->s_ctx = NULL;
    }

    if (!ctx->s_cb.ticNoInit) {
#if defined(HAVE_SESSION_TICKET) && \
    ((defined(HAVE_CHACHA) && defined(HAVE_POLY1305)) || defined(HAVE_AESGCM))
#if defined(OPENSSL_EXTRA) && defined(HAVE_AES_CBC) && defined(WOLFSSL_AES_256)
        OpenSSLTicketCleanup();
#elif defined(WOLFSSL_NO_DEF_TICKET_ENC_CB)
        TicketCleanup();
#endif
#endif
    }
}

static int test_wolfSSL_client_server_nofail_memio_ex(test_ssl_cbf* client_cb,
    test_ssl_cbf* server_cb, cbType client_on_handshake,
    cbType server_on_handshake)
{
    /* We use EXPECT_DECLS_NO_MSGS() here because this helper routine is used
     * for numerous but varied expected-to-fail scenarios that should not emit
     * error messages on the expected failures.  Instead, we return a distinct
     * code for each failure point, allowing the caller to assert on a
     * particular mode of expected failure.  On success, the usual TEST_SUCCESS
     * is returned.
     */
    EXPECT_DECLS_NO_MSGS(-1000);
    struct test_ssl_memio_ctx test_ctx;
#ifdef WOLFSSL_HAVE_TLS_UNIQUE
    size_t msg_len;
#endif /* WOLFSSL_HAVE_TLS_UNIQUE */

    XMEMSET(&test_ctx, 0, sizeof(test_ctx));
    XMEMCPY(&test_ctx.c_cb, client_cb, sizeof(test_ssl_cbf));
    XMEMCPY(&test_ctx.s_cb, server_cb, sizeof(test_ssl_cbf));

    test_ctx.c_ctx = client_cb->ctx;
    test_ctx.s_ctx = server_cb->ctx;
    test_ctx.c_cb.return_code = EXPECT_FAILURE_CODEPOINT_ID;
    test_ctx.s_cb.return_code = EXPECT_FAILURE_CODEPOINT_ID;

    ExpectIntEQ(test_ssl_memio_setup(&test_ctx), TEST_SUCCESS);
    ExpectIntEQ(test_ssl_memio_do_handshake(&test_ctx, 10, NULL), TEST_SUCCESS);

    if (client_on_handshake != NULL) {
        ExpectIntEQ(client_on_handshake(test_ctx.c_ctx, test_ctx.c_ssl),
            TEST_SUCCESS);
    }
    if (server_on_handshake != NULL) {
        ExpectIntEQ(server_on_handshake(test_ctx.s_ctx, test_ctx.s_ssl),
            TEST_SUCCESS);
    }
    if (client_cb->on_handshake != NULL) {
        ExpectIntEQ(client_cb->on_handshake(&test_ctx.c_ctx, &test_ctx.c_ssl),
            TEST_SUCCESS);
    }
    if (server_cb->on_handshake != NULL) {
        ExpectIntEQ(server_cb->on_handshake(&test_ctx.s_ctx, &test_ctx.s_ssl),
            TEST_SUCCESS);
    }
#ifdef WOLFSSL_HAVE_TLS_UNIQUE
    XMEMSET(server_side_msg2, 0, WC_MAX_DIGEST_SIZE);
    msg_len = wolfSSL_get_peer_finished(test_ctx.s_ssl, server_side_msg2,
        WC_MAX_DIGEST_SIZE);
    ExpectIntGE(msg_len, 0);

    XMEMSET(server_side_msg1, 0, WC_MAX_DIGEST_SIZE);
    msg_len = wolfSSL_get_finished(test_ctx.s_ssl, server_side_msg1,
        WC_MAX_DIGEST_SIZE);
    ExpectIntGE(msg_len, 0);
#endif /* WOLFSSL_HAVE_TLS_UNIQUE */

    ExpectIntEQ(test_ssl_memio_read_write(&test_ctx), TEST_SUCCESS);
    test_ssl_memio_cleanup(&test_ctx);

    client_cb->return_code = test_ctx.c_cb.return_code;
    client_cb->last_err = test_ctx.c_cb.last_err;
    server_cb->return_code = test_ctx.s_cb.return_code;
    server_cb->last_err = test_ctx.s_cb.last_err;

    return EXPECT_RESULT();
}

int test_wolfSSL_client_server_nofail_memio(test_ssl_cbf* client_cb,
    test_ssl_cbf* server_cb, cbType client_on_handshake)
{
    return (test_wolfSSL_client_server_nofail_memio_ex(client_cb, server_cb,
        client_on_handshake, NULL));
}
#endif

#ifdef HAVE_IO_TESTS_DEPENDENCIES

#ifdef WOLFSSL_SESSION_EXPORT
#ifdef WOLFSSL_DTLS
/* set up function for sending session information */
static int test_export(WOLFSSL* inSsl, byte* buf, word32 sz, void* userCtx)
{
    WOLFSSL_CTX* ctx = NULL;
    WOLFSSL*     ssl = NULL;

    AssertNotNull(inSsl);
    AssertNotNull(buf);
    AssertIntNE(0, sz);

    /* Set ctx to DTLS 1.2 */
    ctx = wolfSSL_CTX_new(wolfDTLSv1_2_server_method());
    AssertNotNull(ctx);

    ssl = wolfSSL_new(ctx);
    AssertNotNull(ssl);

    AssertIntGE(wolfSSL_dtls_import(ssl, buf, sz), 0);

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
    (void)userCtx;
    return 0;
}
#endif

/* returns negative value on fail and positive (including 0) on success */
static int nonblocking_accept_read(void* args, WOLFSSL* ssl, SOCKET_T* sockfd)
{
    int ret, err, loop_count, count, timeout = 10;
    char msg[] = "I hear you fa shizzle!";
    char input[1024];

    loop_count = ((func_args*)args)->argc;

    #ifdef WOLFSSL_ASYNC_CRYPT
    err = 0; /* Reset error */
    #endif
    do {
    #ifdef WOLFSSL_ASYNC_CRYPT
        if (err == WC_NO_ERR_TRACE(WC_PENDING_E)) {
            ret = wolfSSL_AsyncPoll(ssl, WOLF_POLL_FLAG_CHECK_HW);
            if (ret < 0) { break; } else if (ret == 0) { continue; }
        }
    #endif
        ret = wolfSSL_accept(ssl);
        err = wolfSSL_get_error(ssl, 0);

        if (err == WOLFSSL_ERROR_WANT_READ ||
            err == WOLFSSL_ERROR_WANT_WRITE) {
            int select_ret;

            err = WC_PENDING_E;
            select_ret = tcp_select(*sockfd, timeout);
            if (select_ret == TEST_TIMEOUT) {
                return WOLFSSL_FATAL_ERROR;
            }
        }
    } while (err == WC_NO_ERR_TRACE(WC_PENDING_E));
    if (ret != WOLFSSL_SUCCESS) {
        char buff[WOLFSSL_MAX_ERROR_SZ];
        fprintf(stderr, "error = %d, %s\n", err,
            wolfSSL_ERR_error_string(err, buff));
        return ret;
    }

    for (count = 0; count < loop_count; count++) {
        int select_ret;

        select_ret = tcp_select(*sockfd, timeout);
        if (select_ret == TEST_TIMEOUT) {
            ret = WOLFSSL_FATAL_ERROR;
            break;
        }

        do {
            ret = wolfSSL_read(ssl, input, sizeof(input)-1);
            if (ret > 0) {
                input[ret] = '\0';
                fprintf(stderr, "Client message: %s\n", input);
            }
        } while (err == WOLFSSL_ERROR_WANT_READ && ret != WOLFSSL_SUCCESS);

        do {
            if ((ret = wolfSSL_write(ssl, msg, sizeof(msg))) != sizeof(msg)) {
                return WOLFSSL_FATAL_ERROR;
            }
            err = wolfSSL_get_error(ssl, ret);
        } while (err == WOLFSSL_ERROR_WANT_READ && ret != WOLFSSL_SUCCESS);
    }
    return ret;
}
#endif /* WOLFSSL_SESSION_EXPORT */

THREAD_RETURN WOLFSSL_THREAD test_server_nofail(void* args)
{
    SOCKET_T sockfd = 0;
    SOCKET_T clientfd = 0;
    word16 port;

    callback_functions* cbf;
    WOLFSSL_CTX* ctx = NULL;
    WOLFSSL* ssl = NULL;
    func_args* opts = (func_args*)args;

    char msg[] = "I hear you fa shizzle!";
    char input[1024];
    int idx;
    int ret, err = 0;
    int sharedCtx = 0;
    int doUdp = 0;
    SOCKADDR_IN_T cliAddr;
    socklen_t     cliLen;
    const char* certFile = svrCertFile;
    const char* keyFile = svrKeyFile;

#ifdef WOLFSSL_HAVE_TLS_UNIQUE
    size_t msg_len = 0;
#endif

    wolfSSL_SetLoggingPrefix("server");

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    opts->return_code = TEST_FAIL;
    cbf = opts->callbacks;

    if (cbf != NULL && cbf->ctx) {
        ctx = cbf->ctx;
        sharedCtx = 1;
    }
    else
    {
        WOLFSSL_METHOD* method = NULL;
        if (cbf != NULL && cbf->method != NULL) {
            method = cbf->method();

        }
        else {
            method = wolfSSLv23_server_method();
        }
        ctx = wolfSSL_CTX_new(method);
    }
    if (ctx == NULL) {
        /* Release the wait for TCP ready. */
        signal_ready(opts->signal);
        goto done;
    }

    if (cbf == NULL || !cbf->ticNoInit) {
#if defined(HAVE_SESSION_TICKET) && \
    ((defined(HAVE_CHACHA) && defined(HAVE_POLY1305)) || defined(HAVE_AESGCM))
#if defined(OPENSSL_EXTRA) && defined(HAVE_AES_CBC) && defined(WOLFSSL_AES_256)
        OpenSSLTicketInit();
        wolfSSL_CTX_set_tlsext_ticket_key_cb(ctx, myTicketEncCbOpenSSL);
#elif defined(WOLFSSL_NO_DEF_TICKET_ENC_CB)
        TicketInit();
        wolfSSL_CTX_set_TicketEncCb(ctx, myTicketEncCb);
#endif
#endif
    }

#if defined(USE_WINDOWS_API)
    port = opts->signal->port;
#elif defined(NO_MAIN_DRIVER) && !defined(WOLFSSL_SNIFFER) && \
     !defined(WOLFSSL_MDK_SHELL) && !defined(WOLFSSL_TIRTOS)
    /* Let tcp_listen assign port */
    port = 0;
#else
    /* Use default port */
    port = wolfSSLPort;
#endif

    if (cbf != NULL)
        doUdp = cbf->doUdp;

    /* do it here to detect failure */
    tcp_accept(
        &sockfd, &clientfd, opts, port, 0, doUdp, 0, 0, 1, 0, 0);

    if (doUdp) {
        cliLen = sizeof(cliAddr);

        idx = (int)recvfrom(sockfd, input, sizeof(input), MSG_PEEK,
                  (struct sockaddr*)&cliAddr, &cliLen);

        AssertIntGT(idx, 0);
    }
    else {
        CloseSocket(sockfd);
    }

    wolfSSL_CTX_set_verify(ctx,
                  WOLFSSL_VERIFY_PEER | WOLFSSL_VERIFY_FAIL_IF_NO_PEER_CERT, 0);

#ifdef WOLFSSL_ENCRYPTED_KEYS
    wolfSSL_CTX_set_default_passwd_cb(ctx, PasswordCallBack);
#endif

    if (wolfSSL_CTX_load_verify_locations(ctx, cliCertFile, 0)
                                                           != WOLFSSL_SUCCESS) {
        /*err_sys("can't load ca file, Please run from wolfSSL home dir");*/
        goto done;
    }

    if (cbf != NULL && cbf->certPemFile != NULL)
        certFile = cbf->certPemFile;
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (!sharedCtx && wolfSSL_CTX_use_certificate_file(ctx, certFile,
                                     CERT_FILETYPE) != WOLFSSL_SUCCESS) {
#else
    if (wolfSSL_CTX_use_certificate_file(ctx, certFile,
                                     CERT_FILETYPE) != WOLFSSL_SUCCESS) {
#endif
        /*err_sys("can't load server cert chain file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }

    if (cbf != NULL && cbf->keyPemFile != NULL)
        keyFile = cbf->keyPemFile;
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (!sharedCtx && wolfSSL_CTX_use_PrivateKey_file(ctx, keyFile,
                                     CERT_FILETYPE) != WOLFSSL_SUCCESS) {
#else
    if (wolfSSL_CTX_use_PrivateKey_file(ctx, keyFile,
                                     CERT_FILETYPE) != WOLFSSL_SUCCESS) {
#endif
        /*err_sys("can't load server key file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }

#ifdef HAVE_CRL
    if (cbf != NULL && cbf->crlPemFile != NULL) {
        if (wolfSSL_CTX_EnableCRL(ctx, WOLFSSL_CRL_CHECKALL) != WOLFSSL_SUCCESS)
            goto done;
        if (wolfSSL_CTX_LoadCRLFile(ctx, cbf->crlPemFile, CERT_FILETYPE)
                != WOLFSSL_SUCCESS)
            goto done;
    }
#endif

    /* call ctx setup callback */
    if (cbf != NULL && cbf->ctx_ready != NULL) {
        cbf->ctx_ready(ctx);
    }

    ssl = wolfSSL_new(ctx);
    if (ssl == NULL) {
        goto done;
    }

    if (doUdp) {
        err = wolfSSL_dtls_set_peer(ssl, &cliAddr, cliLen);
        if (err != WOLFSSL_SUCCESS)
            goto done;
    }

#ifdef WOLFSSL_SESSION_EXPORT
    /* only add in more complex nonblocking case with session export tests */
    if (args && opts->argc > 0) {
        /* set as nonblock and time out for waiting on read/write */
        tcp_set_nonblocking(&clientfd);
        wolfSSL_dtls_set_using_nonblock(ssl, 1);
    }
#endif
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (sharedCtx && wolfSSL_use_certificate_file(ssl, certFile,
                                     CERT_FILETYPE) != WOLFSSL_SUCCESS) {
#else
    if (wolfSSL_use_certificate_file(ssl, certFile,
                                     CERT_FILETYPE) != WOLFSSL_SUCCESS) {
#endif
        /*err_sys("can't load server cert chain file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (sharedCtx && wolfSSL_use_PrivateKey_file(ssl, keyFile,
                                     CERT_FILETYPE) != WOLFSSL_SUCCESS) {
#else
    if (wolfSSL_use_PrivateKey_file(ssl, keyFile,
                                     CERT_FILETYPE) != WOLFSSL_SUCCESS) {
#endif
        /*err_sys("can't load server key file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }

    if (wolfSSL_set_fd(ssl, clientfd) != WOLFSSL_SUCCESS) {
        /*err_sys("SSL_set_fd failed");*/
        goto done;
    }

#if !defined(NO_FILESYSTEM) && !defined(NO_DH)
    wolfSSL_SetTmpDH_file(ssl, dhParamFile, CERT_FILETYPE);
#elif !defined(NO_DH)
    SetDH(ssl);  /* will repick suites with DHE, higher priority than PSK */
#endif

    /* call ssl setup callback */
    if (cbf != NULL && cbf->ssl_ready != NULL) {
        cbf->ssl_ready(ssl);
    }

#ifdef WOLFSSL_SESSION_EXPORT
    /* only add in more complex nonblocking case with session export tests */
    if (opts->argc > 0) {
        ret = nonblocking_accept_read(args, ssl, &clientfd);
        if (ret >= 0) {
            opts->return_code = TEST_SUCCESS;
        }
    #ifdef WOLFSSL_TIRTOS
        Task_yield();
    #endif
        goto done;
    }
#endif

    WOLFSSL_ASYNC_WHILE_PENDING(ret = wolfSSL_negotiate(ssl),
                                ret != WOLFSSL_SUCCESS);
    if (ret != WOLFSSL_SUCCESS) {
        char buff[WOLFSSL_MAX_ERROR_SZ];
        fprintf(stderr, "error = %d, %s\n", err,
            wolfSSL_ERR_error_string((word32)err, buff));
        /*err_sys("SSL_accept failed");*/
        goto done;
    }

#ifdef WOLFSSL_HAVE_TLS_UNIQUE
    XMEMSET(server_side_msg2, 0, WC_MAX_DIGEST_SIZE);
    msg_len = wolfSSL_get_peer_finished(ssl, server_side_msg2, WC_MAX_DIGEST_SIZE);
    AssertIntGE(msg_len, 0);

    XMEMSET(server_side_msg1, 0, WC_MAX_DIGEST_SIZE);
    msg_len = wolfSSL_get_finished(ssl, server_side_msg1, WC_MAX_DIGEST_SIZE);
    AssertIntGE(msg_len, 0);
#endif /* WOLFSSL_HAVE_TLS_UNIQUE */

    idx = wolfSSL_read(ssl, input, sizeof(input)-1);
    if (idx > 0) {
        input[idx] = '\0';
        fprintf(stderr, "Client message: %s\n", input);
    }
    else if (idx < 0) {
        goto done;
    }

    if (wolfSSL_write(ssl, msg, sizeof(msg)) != sizeof(msg)) {
        /*err_sys("SSL_write failed");*/
        goto done;
    }

    if (cbf != NULL && cbf->on_result != NULL)
        cbf->on_result(ssl);

#ifdef WOLFSSL_TIRTOS
    Task_yield();
#endif

    opts->return_code = TEST_SUCCESS;

done:
    if (cbf != NULL)
        cbf->last_err = err;
    if (cbf != NULL && cbf->on_cleanup != NULL)
        cbf->on_cleanup(ssl);

    wolfSSL_shutdown(ssl);
    wolfSSL_free(ssl);
    if (!sharedCtx)
        wolfSSL_CTX_free(ctx);

    CloseSocket(clientfd);

#ifdef WOLFSSL_TIRTOS
    fdCloseSession(Task_self());
#endif

#if defined(NO_MAIN_DRIVER) && defined(HAVE_ECC) && defined(FP_ECC) \
                            && defined(HAVE_THREAD_LS)
    wc_ecc_fp_free();  /* free per thread cache */
#endif

    if (cbf == NULL || !cbf->ticNoInit) {
#if defined(HAVE_SESSION_TICKET) && \
    ((defined(HAVE_CHACHA) && defined(HAVE_POLY1305)) || defined(HAVE_AESGCM))
#if defined(OPENSSL_EXTRA) && defined(HAVE_AES_CBC) && defined(WOLFSSL_AES_256)
        OpenSSLTicketCleanup();
#elif defined(WOLFSSL_NO_DEF_TICKET_ENC_CB)
        TicketCleanup();
#endif
#endif
    }

    wolfSSL_SetLoggingPrefix(NULL);

    WOLFSSL_RETURN_FROM_THREAD(0);
}

#if defined(OPENSSL_EXTRA) && !defined(NO_SESSION_CACHE) && \
    !defined(WOLFSSL_NO_TLS12)
static THREAD_RETURN WOLFSSL_THREAD test_server_loop(void* args)
{
    SOCKET_T sockfd;
    SOCKET_T clientfd = -1;
    word16 port;

    callback_functions* cbf;
    WOLFSSL_CTX* ctx = 0;
    WOLFSSL* ssl = 0;

    char msg[] = "I hear you fa shizzle!";
    char input[1024];
    int idx;
    int ret, err = 0;
    int sharedCtx = 0;
    func_args* opts = (func_args*)args;
    int loop_count = opts->argc;
    int count = 0;

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    opts->return_code = TEST_FAIL;
    cbf = opts->callbacks;

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (cbf != NULL && cbf->ctx) {
        ctx = cbf->ctx;
        sharedCtx = 1;
    }
    else
#endif
    {
        WOLFSSL_METHOD* method = NULL;
        if (cbf != NULL && cbf->method != NULL) {
            method = cbf->method();
        }
        else {
            method = wolfSSLv23_server_method();
        }
        ctx = wolfSSL_CTX_new(method);
    }

#if defined(USE_WINDOWS_API)
    port = opts->signal->port;
#elif defined(NO_MAIN_DRIVER) && !defined(WOLFSSL_SNIFFER) && \
     !defined(WOLFSSL_MDK_SHELL) && !defined(WOLFSSL_TIRTOS)
    /* Let tcp_listen assign port */
    port = 0;
#else
    /* Use default port */
    port = wolfSSLPort;
#endif

    wolfSSL_CTX_set_verify(ctx,
                  WOLFSSL_VERIFY_PEER | WOLFSSL_VERIFY_FAIL_IF_NO_PEER_CERT, 0);

#ifdef WOLFSSL_ENCRYPTED_KEYS
    wolfSSL_CTX_set_default_passwd_cb(ctx, PasswordCallBack);
#endif

    if (wolfSSL_CTX_load_verify_locations(ctx, cliCertFile, 0)
                                                           != WOLFSSL_SUCCESS) {
        /*err_sys("can't load ca file, Please run from wolfSSL home dir");*/
        /* Release the wait for TCP ready. */
        signal_ready(opts->signal);
        goto done;
    }
    if (!sharedCtx && wolfSSL_CTX_use_certificate_file(ctx, svrCertFile,
                                     CERT_FILETYPE) != WOLFSSL_SUCCESS) {
        /*err_sys("can't load server cert chain file, "
                "Please run from wolfSSL home dir");*/
        /* Release the wait for TCP ready. */
        signal_ready(opts->signal);
        goto done;
    }
    if (!sharedCtx && wolfSSL_CTX_use_PrivateKey_file(ctx, svrKeyFile,
                                     CERT_FILETYPE) != WOLFSSL_SUCCESS) {
        /*err_sys("can't load server key file, "
                "Please run from wolfSSL home dir");*/
        /* Release the wait for TCP ready. */
        signal_ready(opts->signal);
        goto done;
    }
    /* call ctx setup callback */
    if (cbf != NULL && cbf->ctx_ready != NULL) {
        cbf->ctx_ready(ctx);
    }

    while (count != loop_count) {
        ssl = wolfSSL_new(ctx);
        if (ssl == NULL) {
            signal_ready(opts->signal);
            goto done;
        }
        if (sharedCtx && wolfSSL_use_certificate_file(ssl, svrCertFile,
                                     CERT_FILETYPE) != WOLFSSL_SUCCESS) {
            /*err_sys("can't load server cert chain file, "
                    "Please run from wolfSSL home dir");*/
            /* Release the wait for TCP ready. */
            signal_ready(opts->signal);
            goto done;
        }
        if (sharedCtx && wolfSSL_use_PrivateKey_file(ssl, svrKeyFile,
                                     CERT_FILETYPE) != WOLFSSL_SUCCESS) {
            /*err_sys("can't load server key file, "
                    "Please run from wolfSSL home dir");*/
            /* Release the wait for TCP ready. */
            signal_ready(opts->signal);
            goto done;
        }

#if !defined(NO_FILESYSTEM) && !defined(NO_DH)
        wolfSSL_SetTmpDH_file(ssl, dhParamFile, CERT_FILETYPE);
#elif !defined(NO_DH)
        SetDH(ssl);  /* will repick suites with DHE, higher priority than PSK */
#endif
        /* call ssl setup callback */
        if (cbf != NULL && cbf->ssl_ready != NULL) {
            cbf->ssl_ready(ssl);
        }
        /* do it here to detect failure */
        tcp_accept(&sockfd, &clientfd, (func_args*)args, port, 0, 0, 0, 0, 1, 0,
            0);
        CloseSocket(sockfd);
        if (wolfSSL_set_fd(ssl, clientfd) != WOLFSSL_SUCCESS) {
            /*err_sys("SSL_set_fd failed");*/
            goto done;
        }

        WOLFSSL_ASYNC_WHILE_PENDING(ret = wolfSSL_accept(ssl),
                                    ret != WOLFSSL_SUCCESS);
        if (ret != WOLFSSL_SUCCESS) {
            char buff[WOLFSSL_MAX_ERROR_SZ];
            fprintf(stderr, "error = %d, %s\n", err,
                wolfSSL_ERR_error_string(err, buff));
            /*err_sys("SSL_accept failed");*/
            goto done;
        }

        idx = wolfSSL_read(ssl, input, sizeof(input)-1);
        if (idx > 0) {
            input[idx] = '\0';
            fprintf(stderr, "Client message: %s\n", input);
        }

        if (wolfSSL_write(ssl, msg, sizeof(msg)) != sizeof(msg)) {
            /*err_sys("SSL_write failed");*/
            goto done;
        }
        /* free ssl for this connection */
        wolfSSL_shutdown(ssl);
        wolfSSL_free(ssl); ssl = NULL;
        CloseSocket(clientfd);
        clientfd = -1;

        count++;
    }
#ifdef WOLFSSL_TIRTOS
    Task_yield();
#endif

    opts->return_code = TEST_SUCCESS;

done:
    if (ssl != NULL) {
        wolfSSL_shutdown(ssl);
        wolfSSL_free(ssl);
    }
    if (!sharedCtx)
        wolfSSL_CTX_free(ctx);

    if (clientfd != SOCKET_INVALID)
        CloseSocket(clientfd);

#ifdef WOLFSSL_TIRTOS
    fdCloseSession(Task_self());
#endif

#if defined(NO_MAIN_DRIVER) && defined(HAVE_ECC) && defined(FP_ECC) \
                            && defined(HAVE_THREAD_LS)
    wc_ecc_fp_free();  /* free per thread cache */
#endif

    WOLFSSL_RETURN_FROM_THREAD(0);
}
#endif /* defined(OPENSSL_EXTRA) && !defined(NO_SESSION_CACHE) && !defined(WOLFSSL_TLS13) */

int test_client_nofail(void* args, cbType cb)
{
#if !defined(NO_WOLFSSL_CLIENT)
    SOCKET_T sockfd = 0;
    callback_functions* cbf;

    WOLFSSL_CTX*     ctx     = 0;
    WOLFSSL*         ssl     = 0;
    WOLFSSL_CIPHER*  cipher;

    char msg[64] = "hello wolfssl!";
    char reply[1024];
    int  input;
    int  msgSz = (int)XSTRLEN(msg);
    int  ret, err = 0;
    int  cipherSuite;
    int  sharedCtx = 0;
    int  doUdp = 0;
    const char* cipherName1, *cipherName2;

    wolfSSL_SetLoggingPrefix("client");

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    ((func_args*)args)->return_code = TEST_FAIL;
    cbf = ((func_args*)args)->callbacks;

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (cbf != NULL && cbf->ctx) {
        ctx = cbf->ctx;
        sharedCtx = cbf->isSharedCtx;
    }
    else
#endif
    {
        WOLFSSL_METHOD* method  = NULL;
        if (cbf != NULL && cbf->method != NULL) {
            method = cbf->method();
        }
        else {
            method = wolfSSLv23_client_method();
        }
        ctx = wolfSSL_CTX_new(method);
    }

    if (cbf != NULL)
        doUdp = cbf->doUdp;

#ifdef WOLFSSL_ENCRYPTED_KEYS
    wolfSSL_CTX_set_default_passwd_cb(ctx, PasswordCallBack);
#endif

    /* Do connect here so server detects failures */
    tcp_connect(&sockfd, wolfSSLIP, ((func_args*)args)->signal->port,
                doUdp, 0, NULL);
    /* Connect the socket so that we don't have to set the peer later on */
    if (doUdp)
        udp_connect(&sockfd, wolfSSLIP, ((func_args*)args)->signal->port);

    if (wolfSSL_CTX_load_verify_locations(ctx, caCertFile, 0) != WOLFSSL_SUCCESS)
    {
        /* err_sys("can't load ca file, Please run from wolfSSL home dir");*/
        goto done;
    }
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (!sharedCtx && wolfSSL_CTX_use_certificate_file(ctx, cliCertFile,
                                     CERT_FILETYPE) != WOLFSSL_SUCCESS) {
#else
    if (wolfSSL_CTX_use_certificate_file(ctx, cliCertFile,
                                     CERT_FILETYPE) != WOLFSSL_SUCCESS) {
#endif
        /*err_sys("can't load client cert file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (!sharedCtx && wolfSSL_CTX_use_PrivateKey_file(ctx, cliKeyFile,
                                     CERT_FILETYPE) != WOLFSSL_SUCCESS) {
#else
    if (wolfSSL_CTX_use_PrivateKey_file(ctx, cliKeyFile,
                                     CERT_FILETYPE) != WOLFSSL_SUCCESS) {
#endif

        /*err_sys("can't load client key file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }

#ifdef WOLFSSL_SRTP
    /* make sure that NULL (error condition) returns 1 */
    if (wolfSSL_CTX_set_tlsext_use_srtp(ctx, NULL) != 1) {
         goto done;
    }
#endif

#ifdef HAVE_CRL
    if (cbf != NULL && cbf->crlPemFile != NULL) {
        if (wolfSSL_CTX_EnableCRL(ctx, WOLFSSL_CRL_CHECKALL) != WOLFSSL_SUCCESS)
            goto done;
        if (wolfSSL_CTX_LoadCRLFile(ctx, cbf->crlPemFile, CERT_FILETYPE)
                != WOLFSSL_SUCCESS)
            goto done;
    }
#endif

    /* call ctx setup callback */
    if (cbf != NULL && cbf->ctx_ready != NULL) {
        cbf->ctx_ready(ctx);
    }

    ssl = wolfSSL_new(ctx);
    if (ssl == NULL) {
        goto done;
    }
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (sharedCtx && wolfSSL_use_certificate_file(ssl, cliCertFile,
                                     CERT_FILETYPE) != WOLFSSL_SUCCESS) {
#else
    if (wolfSSL_use_certificate_file(ssl, cliCertFile,
                                     CERT_FILETYPE) != WOLFSSL_SUCCESS) {
#endif
        /*err_sys("can't load client cert file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (sharedCtx && wolfSSL_use_PrivateKey_file(ssl, cliKeyFile,
                                     CERT_FILETYPE) != WOLFSSL_SUCCESS) {
#else
    if (wolfSSL_use_PrivateKey_file(ssl, cliKeyFile,
                                     CERT_FILETYPE) != WOLFSSL_SUCCESS) {
#endif
        /*err_sys("can't load client key file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }

#ifdef WOLFSSL_SRTP
    /* make sure that NULL (error condition) returns 1 */
    if (wolfSSL_set_tlsext_use_srtp(ssl, NULL) != 1) {
         goto done;
    }
#endif

    if (!doUdp) {
        if (wolfSSL_set_fd(ssl, sockfd) != WOLFSSL_SUCCESS) {
            /*err_sys("SSL_set_fd failed");*/
            goto done;
        }
    }
    else {
#ifdef WOLFSSL_DTLS
        if (wolfSSL_set_dtls_fd_connected(ssl, sockfd) != WOLFSSL_SUCCESS) {
            /*err_sys("SSL_set_fd failed");*/
            goto done;
        }
#else
        goto done;
#endif
    }

    /* call ssl setup callback */
    if (cbf != NULL && cbf->ssl_ready != NULL) {
        cbf->ssl_ready(ssl);
    }

    WOLFSSL_ASYNC_WHILE_PENDING(ret = wolfSSL_negotiate(ssl),
                                ret != WOLFSSL_SUCCESS);
    if (ret != WOLFSSL_SUCCESS) {
        char buff[WOLFSSL_MAX_ERROR_SZ];
        fprintf(stderr, "error = %d, %s\n", err,
            wolfSSL_ERR_error_string((word32)err, buff));
        /*err_sys("SSL_connect failed");*/
        goto done;
    }

    /* test the various get cipher methods */
    /* Internal cipher suite names */
    cipherSuite = wolfSSL_get_current_cipher_suite(ssl);
    cipherName1 = wolfSSL_get_cipher_name(ssl);
    cipherName2 = wolfSSL_get_cipher_name_from_suite(
        (byte)(cipherSuite >> 8), cipherSuite & 0xFF);
    AssertStrEQ(cipherName1, cipherName2);

    /* IANA Cipher Suites Names */
    /* Unless WOLFSSL_CIPHER_INTERNALNAME or NO_ERROR_STRINGS,
        then it's the internal cipher suite name */
    cipher = wolfSSL_get_current_cipher(ssl);
    cipherName1 = wolfSSL_CIPHER_get_name(cipher);
    cipherName2 = wolfSSL_get_cipher(ssl);
    AssertStrEQ(cipherName1, cipherName2);
#if !defined(WOLFSSL_CIPHER_INTERNALNAME) && !defined(NO_ERROR_STRINGS) && \
    !defined(WOLFSSL_QT)
    cipherName1 = wolfSSL_get_cipher_name_iana_from_suite(
            (byte)(cipherSuite >> 8), cipherSuite & 0xFF);
    AssertStrEQ(cipherName1, cipherName2);
#endif

    if (cb != NULL)
        (cb)(ctx, ssl);

    if (wolfSSL_write(ssl, msg, msgSz) != msgSz) {
        /*err_sys("SSL_write failed");*/
        goto done;
    }

    input = wolfSSL_read(ssl, reply, sizeof(reply)-1);
    if (input > 0) {
        reply[input] = '\0';
        fprintf(stderr, "Server response: %s\n", reply);
    }

    if (cbf != NULL && cbf->on_result != NULL)
        cbf->on_result(ssl);

    ((func_args*)args)->return_code = TEST_SUCCESS;

done:
    if (cbf != NULL)
        cbf->last_err = err;
    if (cbf != NULL && cbf->on_cleanup != NULL)
        cbf->on_cleanup(ssl);

    wolfSSL_free(ssl);
    if (!sharedCtx)
        wolfSSL_CTX_free(ctx);

    CloseSocket(sockfd);

#ifdef WOLFSSL_TIRTOS
    fdCloseSession(Task_self());
#endif

#if defined(NO_MAIN_DRIVER) && defined(HAVE_ECC) && defined(FP_ECC) \
                            && defined(HAVE_THREAD_LS)
    wc_ecc_fp_free();  /* free per thread cache */
#endif

#else
    (void)args;
    (void)cb;
#endif /* !NO_WOLFSSL_CLIENT */

    wolfSSL_SetLoggingPrefix(NULL);

    return 0;
}

void test_wolfSSL_client_server_nofail_ex(callback_functions* client_cb,
    callback_functions* server_cb, cbType client_on_handshake)
{
    func_args client_args;
    func_args server_args;
    tcp_ready ready;
    THREAD_TYPE serverThread;

    XMEMSET(&client_args, 0, sizeof(func_args));
    XMEMSET(&server_args, 0, sizeof(func_args));

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    StartTCP();
    InitTcpReady(&ready);

#if defined(USE_WINDOWS_API)
    /* use RNG to get random port if using windows */
    ready.port = GetRandomPort();
#endif

    server_args.signal = &ready;
    server_args.callbacks = server_cb;
    client_args.signal = &ready;
    client_args.callbacks = client_cb;

    start_thread(test_server_nofail, &server_args, &serverThread);
    wait_tcp_ready(&server_args);
    test_client_nofail(&client_args, client_on_handshake);
    join_thread(serverThread);

    client_cb->return_code = client_args.return_code;
    server_cb->return_code = server_args.return_code;

    FreeTcpReady(&ready);

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif
}

void test_wolfSSL_client_server_nofail(callback_functions* client_cb,
    callback_functions* server_cb)
{
    test_wolfSSL_client_server_nofail_ex(client_cb, server_cb, NULL);
}


#if defined(OPENSSL_EXTRA) && !defined(NO_SESSION_CACHE) && \
   !defined(WOLFSSL_NO_TLS12) && !defined(NO_WOLFSSL_CLIENT)
static void test_client_reuse_WOLFSSLobj(void* args, cbType cb,
                                         void* server_args)
{
    SOCKET_T sockfd = 0;
    callback_functions* cbf;

    WOLFSSL_CTX*     ctx     = 0;
    WOLFSSL*         ssl     = 0;
    WOLFSSL_SESSION* session = NULL;

    char msg[64] = "hello wolfssl!";
    char reply[1024];
    int  input;
    int  msgSz = (int)XSTRLEN(msg);
    int  ret, err = 0;
    int  sharedCtx = 0;

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    ((func_args*)args)->return_code = TEST_FAIL;
    cbf = ((func_args*)args)->callbacks;

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (cbf != NULL && cbf->ctx) {
        ctx = cbf->ctx;
        sharedCtx = 1;
    }
    else
#endif
    {
        WOLFSSL_METHOD* method  = NULL;
        if (cbf != NULL && cbf->method != NULL) {
            method = cbf->method();
        }
        else {
            method = wolfSSLv23_client_method();
        }
        ctx = wolfSSL_CTX_new(method);
    }

#ifdef WOLFSSL_ENCRYPTED_KEYS
    wolfSSL_CTX_set_default_passwd_cb(ctx, PasswordCallBack);
#endif

    /* Do connect here so server detects failures */
    tcp_connect(&sockfd, wolfSSLIP, ((func_args*)args)->signal->port,
                0, 0, NULL);

    if (wolfSSL_CTX_load_verify_locations(ctx, caCertFile, 0) !=
                                                              WOLFSSL_SUCCESS) {
        /* err_sys("can't load ca file, Please run from wolfSSL home dir");*/
        goto done;
    }
    if (!sharedCtx && wolfSSL_CTX_use_certificate_file(ctx, cliCertFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
        /*err_sys("can't load client cert file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }
    if (!sharedCtx && wolfSSL_CTX_use_PrivateKey_file(ctx, cliKeyFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
        /*err_sys("can't load client key file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }

    /* call ctx setup callback */
    if (cbf != NULL && cbf->ctx_ready != NULL) {
        cbf->ctx_ready(ctx);
    }

    ssl = wolfSSL_new(ctx);
    if (ssl == NULL) {
        goto done;
    }
    /* keep handshake resources for reusing WOLFSSL obj */
    wolfSSL_KeepArrays(ssl);
    if (wolfSSL_KeepHandshakeResources(ssl)) {
        /* err_sys("SSL_KeepHandshakeResources failed"); */
        goto done;
    }
    if (sharedCtx && wolfSSL_use_certificate_file(ssl, cliCertFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
        /*err_sys("can't load client cert file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }
    if (sharedCtx && wolfSSL_use_PrivateKey_file(ssl, cliKeyFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
        /*err_sys("can't load client key file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }

    if (wolfSSL_set_fd(ssl, sockfd) != WOLFSSL_SUCCESS) {
        /*err_sys("SSL_set_fd failed");*/
        goto done;
    }

    /* call ssl setup callback */
    if (cbf != NULL && cbf->ssl_ready != NULL) {
        cbf->ssl_ready(ssl);
    }

    WOLFSSL_ASYNC_WHILE_PENDING(ret = wolfSSL_connect(ssl),
                                ret != WOLFSSL_SUCCESS);
    if (ret != WOLFSSL_SUCCESS) {
        char buff[WOLFSSL_MAX_ERROR_SZ];
        fprintf(stderr, "error = %d, %s\n", err,
            wolfSSL_ERR_error_string(err, buff));
        /*err_sys("SSL_connect failed");*/
        goto done;
    }
    /* Build first session */
    if (cb != NULL)
        cb(ctx, ssl);

    if (wolfSSL_write(ssl, msg, msgSz) != msgSz) {
        /*err_sys("SSL_write failed");*/
        goto done;
    }

    input = wolfSSL_read(ssl, reply, sizeof(reply)-1);
    if (input > 0) {
        reply[input] = '\0';
        fprintf(stderr, "Server response: %s\n", reply);
    }

    /* Session Resumption by reusing WOLFSSL object */
    wolfSSL_set_quiet_shutdown(ssl, 1);
    if (wolfSSL_shutdown(ssl) != WOLFSSL_SUCCESS) {
        /* err_sys ("SSL shutdown failed"); */
        goto done;
    }
    session = wolfSSL_get1_session(ssl);
    if (wolfSSL_clear(ssl) != WOLFSSL_SUCCESS) {
        wolfSSL_SESSION_free(session);
        /* err_sys ("SSL_clear failed"); */
        goto done;
    }
    wolfSSL_set_session(ssl, session);
    wolfSSL_SESSION_free(session);
    session = NULL;
    /* close socket once */
    CloseSocket(sockfd);
    sockfd = 0;
    /* wait until server ready */
    wait_tcp_ready((func_args*)server_args);
    fprintf(stderr, "session resumption\n");
    /* Do re-connect  */
    tcp_connect(&sockfd, wolfSSLIP, ((func_args*)args)->signal->port,
                0, 0, NULL);
    if (wolfSSL_set_fd(ssl, sockfd) != WOLFSSL_SUCCESS) {
        /*err_sys("SSL_set_fd failed");*/
        goto done;
    }

    WOLFSSL_ASYNC_WHILE_PENDING(ret = wolfSSL_connect(ssl),
                                ret != WOLFSSL_SUCCESS);
    if (ret != WOLFSSL_SUCCESS) {
        char buff[WOLFSSL_MAX_ERROR_SZ];
        fprintf(stderr, "error = %d, %s\n", err,
            wolfSSL_ERR_error_string(err, buff));
        /*err_sys("SSL_connect failed");*/
        goto done;
    }
    /* Build first session */
    if (cb != NULL)
        cb(ctx, ssl);

    if (wolfSSL_write(ssl, msg, msgSz) != msgSz) {
        /*err_sys("SSL_write failed");*/
        goto done;
    }

    input = wolfSSL_read(ssl, reply, sizeof(reply)-1);
    if (input > 0) {
        reply[input] = '\0';
        fprintf(stderr, "Server response: %s\n", reply);
    }

    ((func_args*)args)->return_code = TEST_SUCCESS;

done:
    wolfSSL_free(ssl);
    if (!sharedCtx)
        wolfSSL_CTX_free(ctx);

    CloseSocket(sockfd);

#ifdef WOLFSSL_TIRTOS
    fdCloseSession(Task_self());
#endif

    return;
}
#endif /* defined(OPENSSL_EXTRA) && !defined(NO_SESSION_CACHE) &&
         !defined(WOLFSSL_TLS13) && !defined(NO_WOLFSSL_CLIENT) */

#if (defined(OPENSSL_ALL) || defined(WOLFSSL_NGINX) || \
     defined(WOLFSSL_HAPROXY) || defined(HAVE_LIGHTY)) && \
    defined(HAVE_ALPN) && defined(HAVE_SNI) && \
    defined(HAVE_IO_TESTS_DEPENDENCIES) && !defined(NO_BIO)
    #define HAVE_ALPN_PROTOS_SUPPORT
#endif

/* Generic TLS client / server with callbacks for API unit tests
 * Used by SNI / ALPN / crypto callback helper functions */
#if defined(HAVE_IO_TESTS_DEPENDENCIES) && \
    (defined(HAVE_SNI) || defined(HAVE_ALPN) || defined(WOLF_CRYPTO_CB) || \
     defined(HAVE_ALPN_PROTOS_SUPPORT)) || defined(WOLFSSL_STATIC_MEMORY)
    #define ENABLE_TLS_CALLBACK_TEST
#endif

#if defined(ENABLE_TLS_CALLBACK_TEST) || \
    (defined(WOLFSSL_DTLS) && defined(WOLFSSL_SESSION_EXPORT))
/* TLS server for API unit testing - generic */
static THREAD_RETURN WOLFSSL_THREAD run_wolfssl_server(void* args)
{
    callback_functions* callbacks = ((func_args*)args)->callbacks;

    WOLFSSL_CTX*    ctx = NULL;
    WOLFSSL*        ssl = NULL;
    SOCKET_T        sfd = 0;
    SOCKET_T        cfd = 0;
    word16          port;

    char msg[] = "I hear you fa shizzle!";
    int  len   = (int) XSTRLEN(msg);
    char input[1024];
    int  idx;
    int  ret, err = 0;

    ((func_args*)args)->return_code = TEST_FAIL;

#if defined(USE_WINDOWS_API)
    port = ((func_args*)args)->signal->port;
#elif defined(NO_MAIN_DRIVER) && !defined(WOLFSSL_SNIFFER) && \
     !defined(WOLFSSL_MDK_SHELL) && !defined(WOLFSSL_TIRTOS)
    /* Let tcp_listen assign port */
    port = 0;
#else
    /* Use default port */
    port = wolfSSLPort;
#endif

#ifdef WOLFSSL_DTLS
    if (callbacks->method == wolfDTLS_server_method
#ifdef WOLFSSL_STATIC_MEMORY
     || callbacks->method_ex == wolfDTLS_server_method_ex
#endif
#ifndef NO_OLD_TLS
     || callbacks->method == wolfDTLSv1_server_method
#ifdef WOLFSSL_STATIC_MEMORY
     || callbacks->method_ex == wolfDTLSv1_server_method_ex
#endif
#endif
#ifndef WOLFSSL_NO_TLS12
     || callbacks->method == wolfDTLSv1_2_server_method
#ifdef WOLFSSL_STATIC_MEMORY
     || callbacks->method_ex == wolfDTLSv1_2_server_method_ex
#endif
#endif
#ifdef WOLFSSL_DTLS13
     || callbacks->method == wolfDTLSv1_3_server_method
#ifdef WOLFSSL_STATIC_MEMORY
     || callbacks->method_ex == wolfDTLSv1_3_server_method_ex
#endif
#endif
        ) {
        tcp_accept(&sfd, &cfd, (func_args*)args, port, 0, 1, 0, 0, 0, 0, 0);
    }
    else
#endif
    {
        tcp_accept(&sfd, &cfd, (func_args*)args, port, 0, 0, 0, 0, 1, 0, 0);
    }

#ifdef WOLFSSL_STATIC_MEMORY
    if (callbacks->method_ex != NULL && callbacks->mem != NULL &&
        callbacks->memSz > 0) {
        ret = wolfSSL_CTX_load_static_memory(&ctx, callbacks->method_ex,
            callbacks->mem, callbacks->memSz, 0, 1);
        if (ret != WOLFSSL_SUCCESS) {
            fprintf(stderr, "CTX static new failed %d\n", ret);
            goto cleanup;
        }
    }
#else
    ctx = wolfSSL_CTX_new(callbacks->method());
#endif
    if (ctx == NULL) {
        fprintf(stderr, "CTX new failed\n");
        goto cleanup;
    }

    /* set defaults */
    if (callbacks->caPemFile == NULL)
        callbacks->caPemFile = cliCertFile;
    if (callbacks->certPemFile == NULL)
        callbacks->certPemFile = svrCertFile;
    if (callbacks->keyPemFile == NULL)
        callbacks->keyPemFile = svrKeyFile;

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    wolfSSL_CTX_SetDevId(ctx, callbacks->devId);

    wolfSSL_CTX_set_verify(ctx,
                  WOLFSSL_VERIFY_PEER | WOLFSSL_VERIFY_FAIL_IF_NO_PEER_CERT, 0);

#ifdef WOLFSSL_ENCRYPTED_KEYS
    wolfSSL_CTX_set_default_passwd_cb(ctx, PasswordCallBack);
#endif
#if defined(WOLFSSL_SESSION_EXPORT) && defined(WOLFSSL_DTLS)
    if (callbacks->method == wolfDTLSv1_2_server_method) {
        if (wolfSSL_CTX_dtls_set_export(ctx, test_export) != WOLFSSL_SUCCESS)
            goto cleanup;
    }
#endif


    if (wolfSSL_CTX_load_verify_locations(ctx, callbacks->caPemFile, 0) !=
            WOLFSSL_SUCCESS) {
        goto cleanup;
    }

    if (wolfSSL_CTX_use_certificate_file(ctx, callbacks->certPemFile,
            CERT_FILETYPE) != WOLFSSL_SUCCESS) {
        goto cleanup;
    }

    if (wolfSSL_CTX_use_PrivateKey_file(ctx, callbacks->keyPemFile,
            CERT_FILETYPE) != WOLFSSL_SUCCESS) {
        goto cleanup;
    }

#ifdef HAVE_CRL
    if (callbacks->crlPemFile != NULL) {
        if (wolfSSL_CTX_LoadCRLFile(ctx, callbacks->crlPemFile,
                CERT_FILETYPE) != WOLFSSL_SUCCESS) {
            goto cleanup;
        }
    }
#endif

    if (callbacks->ctx_ready)
        callbacks->ctx_ready(ctx);

    ssl = wolfSSL_new(ctx);
    if (ssl == NULL) {
        fprintf(stderr, "SSL new failed\n");
        goto cleanup;
    }
    if (wolfSSL_dtls(ssl)) {
        SOCKADDR_IN_T cliAddr;
        socklen_t     cliLen;

        cliLen = sizeof(cliAddr);
        idx = (int)recvfrom(sfd, input, sizeof(input), MSG_PEEK,
                (struct sockaddr*)&cliAddr, &cliLen);
        if (idx <= 0) {
            goto cleanup;
        }
        wolfSSL_dtls_set_peer(ssl, &cliAddr, cliLen);
    }
    else {
        CloseSocket(sfd);
    }

    if (wolfSSL_set_fd(ssl, cfd) != WOLFSSL_SUCCESS) {
        goto cleanup;
    }

    if (callbacks->loadToSSL) {
        wolfSSL_SetDevId(ssl, callbacks->devId);

        if (wolfSSL_use_certificate_file(ssl, callbacks->certPemFile,
                CERT_FILETYPE) != WOLFSSL_SUCCESS) {
            goto cleanup;
        }

        if (wolfSSL_use_PrivateKey_file(ssl, callbacks->keyPemFile,
                CERT_FILETYPE) != WOLFSSL_SUCCESS) {
            goto cleanup;
        }
    }

#ifdef NO_PSK
    #if !defined(NO_FILESYSTEM) && !defined(NO_DH)
        wolfSSL_SetTmpDH_file(ssl, dhParamFile, CERT_FILETYPE);
    #elif !defined(NO_DH)
        SetDH(ssl);  /* will repick suites with DHE, higher priority than PSK */
    #endif
#endif

    if (callbacks->ssl_ready)
        callbacks->ssl_ready(ssl);

    WOLFSSL_ASYNC_WHILE_PENDING(ret = wolfSSL_accept(ssl),
                                ret != WOLFSSL_SUCCESS);
    if (ret != WOLFSSL_SUCCESS) {
        char buff[WOLFSSL_MAX_ERROR_SZ];
        fprintf(stderr, "accept error = %d, %s\n", err,
            wolfSSL_ERR_error_string((word32)err, buff));
        /*err_sys("SSL_accept failed");*/
    }
    else {
        WOLFSSL_ASYNC_WHILE_PENDING(idx = wolfSSL_read(ssl, input, sizeof(input)-1),
                                    idx <= 0);
        if (idx > 0) {
            input[idx] = 0;
            fprintf(stderr, "Client message: %s\n", input);
        }

        WOLFSSL_ASYNC_WHILE_PENDING(ret = wolfSSL_write(ssl, msg, len),
                                    len != ret);
        if (len != ret) {
            goto cleanup;
        }

#if defined(WOLFSSL_SESSION_EXPORT) && !defined(HAVE_IO_POOL) && \
        defined(WOLFSSL_DTLS)
        if (wolfSSL_dtls(ssl)) {
            byte*  import;
            word32 sz;

            wolfSSL_dtls_export(ssl, NULL, &sz);
            import = (byte*)XMALLOC(sz, NULL, DYNAMIC_TYPE_TMP_BUFFER);
            if (import == NULL) {
                goto cleanup;
            }
            idx = wolfSSL_dtls_export(ssl, import, &sz);
            if (idx < 0) {
                goto cleanup;
            }
            if (wolfSSL_dtls_import(ssl, import, idx) < 0) {
                goto cleanup;
            }
            XFREE(import, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        }
#endif
#ifdef WOLFSSL_TIRTOS
        Task_yield();
#endif
        ((func_args*)args)->return_code = TEST_SUCCESS;
    }

    if (callbacks->on_result)
        callbacks->on_result(ssl);

    wolfSSL_shutdown(ssl);

cleanup:
    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
    CloseSocket(cfd);


#ifdef WOLFSSL_TIRTOS
    fdCloseSession(Task_self());
#endif

#if defined(NO_MAIN_DRIVER) && defined(HAVE_ECC) && defined(FP_ECC) \
                            && defined(HAVE_THREAD_LS)
    wc_ecc_fp_free();  /* free per thread cache */
#endif

    WOLFSSL_RETURN_FROM_THREAD(0);
}

/* TLS Client for API unit testing - generic */
static void run_wolfssl_client(void* args)
{
    callback_functions* callbacks = ((func_args*)args)->callbacks;

    WOLFSSL_CTX* ctx = NULL;
    WOLFSSL*     ssl = NULL;
    SOCKET_T    sfd = 0;

    char msg[] = "hello wolfssl server!";
    int  len   = (int) XSTRLEN(msg);
    char input[1024];
    int  ret, err = 0;

    ((func_args*)args)->return_code = TEST_FAIL;

    /* set defaults */
    if (callbacks->caPemFile == NULL)
        callbacks->caPemFile = caCertFile;
    if (callbacks->certPemFile == NULL)
        callbacks->certPemFile = cliCertFile;
    if (callbacks->keyPemFile == NULL)
        callbacks->keyPemFile = cliKeyFile;

#ifdef WOLFSSL_STATIC_MEMORY
    if (callbacks->method_ex != NULL && callbacks->mem != NULL &&
        callbacks->memSz > 0) {
        ret = wolfSSL_CTX_load_static_memory(&ctx, callbacks->method_ex,
            callbacks->mem, callbacks->memSz, 0, 1);
        if (ret != WOLFSSL_SUCCESS) {
            fprintf(stderr, "CTX static new failed %d\n", ret);
            goto cleanup;
        }
    }
#else
    ctx = wolfSSL_CTX_new(callbacks->method());
#endif
    if (ctx == NULL) {
        fprintf(stderr, "CTX new failed\n");
        goto cleanup;
    }

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    if (!callbacks->loadToSSL) {
        wolfSSL_CTX_SetDevId(ctx, callbacks->devId);
    }

#ifdef WOLFSSL_ENCRYPTED_KEYS
    wolfSSL_CTX_set_default_passwd_cb(ctx, PasswordCallBack);
#endif

    if (wolfSSL_CTX_load_verify_locations(ctx, callbacks->caPemFile, 0) !=
            WOLFSSL_SUCCESS) {
        goto cleanup;
    }

    if (!callbacks->loadToSSL) {
        if (wolfSSL_CTX_use_certificate_file(ctx, callbacks->certPemFile,
                CERT_FILETYPE) != WOLFSSL_SUCCESS) {
            goto cleanup;
        }

        if (wolfSSL_CTX_use_PrivateKey_file(ctx, callbacks->keyPemFile,
                CERT_FILETYPE) != WOLFSSL_SUCCESS) {
            goto cleanup;
        }
    }

#ifdef HAVE_CRL
    if (callbacks->crlPemFile != NULL) {
        if (wolfSSL_CTX_LoadCRLFile(ctx, callbacks->crlPemFile,
                CERT_FILETYPE) != WOLFSSL_SUCCESS) {
            goto cleanup;
        }
    }
#endif

    if (callbacks->ctx_ready)
        callbacks->ctx_ready(ctx);

    ssl = wolfSSL_new(ctx);
    if (wolfSSL_dtls(ssl)) {
        tcp_connect(&sfd, wolfSSLIP, ((func_args*)args)->signal->port,
                    1, 0, ssl);
    }
    else {
        tcp_connect(&sfd, wolfSSLIP, ((func_args*)args)->signal->port,
                    0, 0, ssl);
    }
    if (wolfSSL_set_fd(ssl, sfd) != WOLFSSL_SUCCESS) {
        goto cleanup;
    }

    if (callbacks->loadToSSL) {
        wolfSSL_SetDevId(ssl, callbacks->devId);

        if (wolfSSL_use_certificate_file(ssl, callbacks->certPemFile,
                CERT_FILETYPE) != WOLFSSL_SUCCESS) {
            goto cleanup;
        }

        if (wolfSSL_use_PrivateKey_file(ssl, callbacks->keyPemFile,
                CERT_FILETYPE) != WOLFSSL_SUCCESS) {
            goto cleanup;
        }
    }

    if (callbacks->ssl_ready)
        callbacks->ssl_ready(ssl);

    WOLFSSL_ASYNC_WHILE_PENDING(ret = wolfSSL_connect(ssl),
                                ret != WOLFSSL_SUCCESS);
    if (ret != WOLFSSL_SUCCESS) {
        char buff[WOLFSSL_MAX_ERROR_SZ];
        fprintf(stderr, "error = %d, %s\n", err,
            wolfSSL_ERR_error_string((word32)err, buff));
        /*err_sys("SSL_connect failed");*/
    }
    else {
        WOLFSSL_ASYNC_WHILE_PENDING(ret = wolfSSL_write(ssl, msg, len),
                                    ret != len);
        if (len != ret)
            goto cleanup;

        WOLFSSL_ASYNC_WHILE_PENDING(ret = wolfSSL_read(ssl, input, sizeof(input)-1),
                                    ret <= 0);
        if (ret > 0) {
            input[ret] = '\0'; /* null term */
            fprintf(stderr, "Server response: %s\n", input);
        }
        ((func_args*)args)->return_code = TEST_SUCCESS;
    }

    if (callbacks->on_result)
        callbacks->on_result(ssl);

cleanup:
    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
    CloseSocket(sfd);

#ifdef WOLFSSL_TIRTOS
    fdCloseSession(Task_self());
#endif
}

#endif /* ENABLE_TLS_CALLBACK_TEST */


static int test_wolfSSL_read_write(void)
{
    EXPECT_DECLS;
#ifndef NO_SHA256
    /* The unit testing for read and write shall happen simultaneously, since
     * one can't do anything with one without the other. (Except for a failure
     * test case.) This function will call all the others that will set up,
     * execute, and report their test findings.
     *
     * Set up the success case first. This function will become the template
     * for the other tests. This should eventually be renamed
     *
     * The success case isn't interesting, how can this fail?
     * - Do not give the client context a CA certificate. The connect should
     *   fail. Do not need server for this?
     * - Using NULL for the ssl object on server. Do not need client for this.
     * - Using NULL for the ssl object on client. Do not need server for this.
     * - Good ssl objects for client and server. Client write() without server
     *   read().
     * - Good ssl objects for client and server. Server write() without client
     *   read().
     * - Forgetting the password callback?
    */
    tcp_ready ready;
    func_args client_args;
    func_args server_args;
    THREAD_TYPE serverThread;

    XMEMSET(&client_args, 0, sizeof(func_args));
    XMEMSET(&server_args, 0, sizeof(func_args));
#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    StartTCP();
    InitTcpReady(&ready);

#if defined(USE_WINDOWS_API)
    /* use RNG to get random port if using windows */
    ready.port = GetRandomPort();
#endif

    server_args.signal = &ready;
    client_args.signal = &ready;

    start_thread(test_server_nofail, &server_args, &serverThread);
    wait_tcp_ready(&server_args);
    test_client_nofail(&client_args, NULL);
    join_thread(serverThread);

    ExpectTrue(client_args.return_code);
    ExpectTrue(server_args.return_code);

    FreeTcpReady(&ready);

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_read_write_ex(void)
{
    EXPECT_DECLS;
    WOLFSSL_CTX *ctx_c = NULL;
    WOLFSSL_CTX *ctx_s = NULL;
    WOLFSSL *ssl_c = NULL;
    WOLFSSL *ssl_s = NULL;
    struct test_memio_ctx test_ctx;
    const char *test_str = "test";
    int test_str_size;
    size_t count;
    byte buf[255];

    XMEMSET(&test_ctx, 0, sizeof(test_ctx));
    ExpectIntEQ(test_memio_setup(&test_ctx, &ctx_c, &ctx_s, &ssl_c, &ssl_s,
        wolfSSLv23_client_method, wolfSSLv23_server_method), 0);
    ExpectIntEQ(test_memio_do_handshake(ssl_c, ssl_s, 10, NULL), 0);
    test_str_size = XSTRLEN("test") + 1;
    ExpectIntEQ(wolfSSL_write_ex(ssl_c, test_str, test_str_size, &count),
        WOLFSSL_SUCCESS);
    ExpectIntEQ(count, test_str_size);
    count = 0;
    ExpectIntEQ(wolfSSL_read_ex(ssl_s, buf, sizeof(buf), &count), WOLFSSL_SUCCESS);
    ExpectIntEQ(count, test_str_size);
    ExpectIntEQ(XSTRCMP((char*)buf, test_str), 0);


    ExpectIntEQ(wolfSSL_shutdown(ssl_c), WOLFSSL_SHUTDOWN_NOT_DONE);
    ExpectIntEQ(wolfSSL_shutdown(ssl_s), WOLFSSL_SHUTDOWN_NOT_DONE);
    ExpectIntEQ(wolfSSL_shutdown(ssl_c), 1);
    ExpectIntEQ(wolfSSL_shutdown(ssl_s), 1);

    wolfSSL_free(ssl_c);
    wolfSSL_free(ssl_s);
    wolfSSL_CTX_free(ctx_c);
    wolfSSL_CTX_free(ctx_s);
    return TEST_SUCCESS;
}

static int test_wolfSSL_reuse_WOLFSSLobj(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_SESSION_CACHE) && \
    !defined(WOLFSSL_NO_TLS12)
    /* The unit test for session resumption by reusing WOLFSSL object.
     * WOLFSSL object is not cleared after first session. It reuse the object
     * for second connection.
    */
    tcp_ready ready;
    func_args client_args;
    func_args server_args;
    THREAD_TYPE serverThread;
    callback_functions client_cbf;
    callback_functions server_cbf;

    XMEMSET(&client_args, 0, sizeof(func_args));
    XMEMSET(&server_args, 0, sizeof(func_args));
    XMEMSET(&client_cbf, 0, sizeof(callback_functions));
    XMEMSET(&server_cbf, 0, sizeof(callback_functions));
#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    StartTCP();
    InitTcpReady(&ready);

#if defined(USE_WINDOWS_API)
    /* use RNG to get random port if using windows */
    ready.port = GetRandomPort();
#endif

    client_cbf.method = wolfTLSv1_2_client_method;
    server_cbf.method = wolfTLSv1_2_server_method;
    client_args.callbacks = &client_cbf;
    server_args.callbacks = &server_cbf;

    server_args.signal = &ready;
    client_args.signal = &ready;
    /* the var is used for loop number */
    server_args.argc = 2;

    start_thread(test_server_loop, &server_args, &serverThread);
    wait_tcp_ready(&server_args);
    test_client_reuse_WOLFSSLobj(&client_args, NULL, &server_args);
    join_thread(serverThread);

    ExpectTrue(client_args.return_code);
    ExpectTrue(server_args.return_code);

    FreeTcpReady(&ready);

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif
#endif /* defined(OPENSSL_EXTRA) && !defined(NO_SESSION_CACHE) &&
        * !defined(WOLFSSL_TLS13) */
    return EXPECT_RESULT();
}

#if defined(OPENSSL_EXTRA) && defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES)
static int test_wolfSSL_CTX_verifyDepth_ServerClient_1_ctx_ready(
    WOLFSSL_CTX* ctx)
{
    wolfSSL_CTX_set_verify(ctx, SSL_VERIFY_PEER, myVerify);
    myVerifyAction = VERIFY_USE_PREVERIFY;
    wolfSSL_CTX_set_verify_depth(ctx, 2);
    return TEST_SUCCESS;
}
#endif

static int test_wolfSSL_CTX_verifyDepth_ServerClient_1(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES)
    test_ssl_cbf client_cbf;
    test_ssl_cbf server_cbf;

    XMEMSET(&client_cbf, 0, sizeof(client_cbf));
    XMEMSET(&server_cbf, 0, sizeof(server_cbf));

#ifdef WOLFSSL_TLS13
    client_cbf.method = wolfTLSv1_3_client_method;
#endif /* WOLFSSL_TLS13 */
    client_cbf.ctx_ready =
        test_wolfSSL_CTX_verifyDepth_ServerClient_1_ctx_ready;

    /* test case 1 verify depth is equal to peer chain */
    ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&client_cbf,
        &server_cbf, NULL), TEST_SUCCESS);

    ExpectIntEQ(client_cbf.return_code, TEST_SUCCESS);
    ExpectIntEQ(server_cbf.return_code, TEST_SUCCESS);
#endif /* OPENSSL_EXTRA && HAVE_SSL_MEMIO_TESTS_DEPENDENCIES */

    return EXPECT_RESULT();
}

#if defined(OPENSSL_EXTRA) && defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES)
static int test_wolfSSL_CTX_verifyDepth_ServerClient_2_ctx_ready(
    WOLFSSL_CTX* ctx)
{
    wolfSSL_CTX_set_verify(ctx, SSL_VERIFY_PEER, myVerify);
    myVerifyAction = VERIFY_OVERRIDE_ERROR;
    wolfSSL_CTX_set_verify_depth(ctx, 0);
    return TEST_SUCCESS;
}
#endif

static int test_wolfSSL_CTX_verifyDepth_ServerClient_2(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES)
    test_ssl_cbf client_cbf;
    test_ssl_cbf server_cbf;

    XMEMSET(&client_cbf, 0, sizeof(client_cbf));
    XMEMSET(&server_cbf, 0, sizeof(server_cbf));

#ifdef WOLFSSL_TLS13
    client_cbf.method = wolfTLSv1_3_client_method;
#endif /* WOLFSSL_TLS13 */
    client_cbf.ctx_ready =
        test_wolfSSL_CTX_verifyDepth_ServerClient_2_ctx_ready;

    /* test case 2
     * verify depth is zero, number of peer's chain is 2.
     * verify result becomes MAX_CHAIN_ERROR, but it is overridden in
     * callback.
     */
    ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&client_cbf,
        &server_cbf, NULL), TEST_SUCCESS);

    ExpectIntEQ(client_cbf.return_code, TEST_SUCCESS);
    ExpectIntEQ(server_cbf.return_code, TEST_SUCCESS);
#endif /* OPENSSL_EXTRA && HAVE_SSL_MEMIO_TESTS_DEPENDENCIES */

    return EXPECT_RESULT();
}

#if defined(OPENSSL_EXTRA) && defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES)
static int test_wolfSSL_CTX_verifyDepth_ServerClient_3_ctx_ready(
    WOLFSSL_CTX* ctx)
{
    wolfSSL_CTX_set_verify(ctx, SSL_VERIFY_PEER, myVerify);
    myVerifyAction = VERIFY_USE_PREVERIFY;
    wolfSSL_CTX_set_verify_depth(ctx, 0);
    return TEST_SUCCESS;
}
#endif

static int test_wolfSSL_CTX_verifyDepth_ServerClient_3(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES)
    test_ssl_cbf client_cbf;
    test_ssl_cbf server_cbf;

    XMEMSET(&client_cbf, 0, sizeof(client_cbf));
    XMEMSET(&server_cbf, 0, sizeof(server_cbf));

#ifdef WOLFSSL_TLS13
    client_cbf.method = wolfTLSv1_3_client_method;
#endif /* WOLFSSL_TLS13 */
    client_cbf.ctx_ready =
        test_wolfSSL_CTX_verifyDepth_ServerClient_3_ctx_ready;

    /* test case 3
     * verify depth is zero, number of peer's chain is 2
     * verify result becomes MAX_CHAIN_ERRO. call-back returns failure.
     * therefore, handshake becomes failure.
     */
    ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&client_cbf,
        &server_cbf, NULL), -1001);

    ExpectIntEQ(client_cbf.return_code, -1000);
    ExpectIntEQ(server_cbf.return_code, -1000);
    ExpectIntEQ(client_cbf.last_err, WC_NO_ERR_TRACE(MAX_CHAIN_ERROR));
    ExpectIntEQ(server_cbf.last_err, WC_NO_ERR_TRACE(FATAL_ERROR));
#endif /* OPENSSL_EXTRA && HAVE_SSL_MEMIO_TESTS_DEPENDENCIES */

    return EXPECT_RESULT();
}

#if defined(OPENSSL_ALL) && defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES) && \
    !defined(WOLFSSL_NO_TLS12) && \
    defined(HAVE_ECC) && !defined(NO_AES) && !defined(NO_SHA256)
static int test_wolfSSL_CTX_set_cipher_list_server_ctx_ready(WOLFSSL_CTX* ctx)
{
    EXPECT_DECLS;
    ExpectTrue(wolfSSL_CTX_set_cipher_list(ctx, "DEFAULT:!NULL"));
    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_set_cipher_list_client_ctx_ready(WOLFSSL_CTX* ctx)
{
    EXPECT_DECLS;
    ExpectTrue(wolfSSL_CTX_set_cipher_list(ctx, "ECDHE-RSA-AES128-SHA256"));
    return EXPECT_RESULT();
}
#endif

static int test_wolfSSL_CTX_set_cipher_list(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_ALL) && defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES) && \
    defined(HAVE_ECC) && !defined(NO_AES) && !defined(NO_SHA256)

    #if !defined(WOLFSSL_NO_TLS12)
    WOLFSSL_CTX* ctxClient = NULL;
    WOLFSSL*     sslClient = NULL;
    test_ssl_cbf client_cbf;
    test_ssl_cbf server_cbf;

    XMEMSET(&client_cbf, 0, sizeof(client_cbf));
    XMEMSET(&server_cbf, 0, sizeof(server_cbf));

    server_cbf.method = wolfTLSv1_2_server_method;
    server_cbf.ctx_ready = test_wolfSSL_CTX_set_cipher_list_server_ctx_ready;
    client_cbf.method = wolfTLSv1_2_client_method;
    client_cbf.ctx_ready = test_wolfSSL_CTX_set_cipher_list_client_ctx_ready;

    ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&client_cbf,
        &server_cbf, NULL), TEST_SUCCESS);

    ExpectIntEQ(client_cbf.return_code, TEST_SUCCESS);
    ExpectIntEQ(server_cbf.return_code, TEST_SUCCESS);

    /* check with cipher string that has '+' */
    ExpectNotNull((ctxClient = wolfSSL_CTX_new(wolfTLSv1_2_client_method())));
    /* Use trailing : with nothing to test for ASAN */
    ExpectTrue(wolfSSL_CTX_set_cipher_list(ctxClient, "ECDHE+AESGCM:"));
    ExpectNotNull((sslClient = wolfSSL_new(ctxClient)));

    /* check for the existence of an ECDHE ECDSA cipher suite */
    if (EXPECT_SUCCESS()) {
        int i = 0;
        int found = 0;
        const char* suite;

        WOLF_STACK_OF(WOLFSSL_CIPHER)* sk = NULL;
        WOLFSSL_CIPHER* current;

        ExpectNotNull((sk = wolfSSL_get_ciphers_compat(sslClient)));
        do {
            current = wolfSSL_sk_SSL_CIPHER_value(sk, i++);
            if (current) {
                suite = wolfSSL_CIPHER_get_name(current);
                if (suite && XSTRSTR(suite, "ECDSA")) {
                    found = 1;
                    break;
                }
            }
        } while (current);
        ExpectIntEQ(found, 1);
    }

    wolfSSL_free(sslClient);
    wolfSSL_CTX_free(ctxClient);

    #endif /* !WOLFSSL_NO_TLS12 */
#endif
    return EXPECT_RESULT();
}

#if defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES) && \
    defined(WOLFSSL_HAVE_TLS_UNIQUE)
static int test_wolfSSL_get_finished_client_on_handshake(WOLFSSL_CTX* ctx,
    WOLFSSL* ssl)
{
    EXPECT_DECLS;
    size_t msg_len;

    (void)ctx;

    /* get_finished test */
    /* 1. get own sent message */
    XMEMSET(client_side_msg1, 0, WC_MAX_DIGEST_SIZE);
    msg_len = wolfSSL_get_finished(ssl, client_side_msg1, WC_MAX_DIGEST_SIZE);
    ExpectIntGE(msg_len, 0);
    /* 2. get peer message */
    XMEMSET(client_side_msg2, 0, WC_MAX_DIGEST_SIZE);
    msg_len = wolfSSL_get_peer_finished(ssl, client_side_msg2, WC_MAX_DIGEST_SIZE);
    ExpectIntGE(msg_len, 0);

    return EXPECT_RESULT();
}
#endif

static int test_wolfSSL_get_finished(void)
{
    EXPECT_DECLS;
#if defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES) && \
    defined(WOLFSSL_HAVE_TLS_UNIQUE)
    test_ssl_cbf client_cbf;
    test_ssl_cbf server_cbf;

    XMEMSET(&client_cbf, 0, sizeof(client_cbf));
    XMEMSET(&server_cbf, 0, sizeof(server_cbf));

    ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&client_cbf,
        &server_cbf, test_wolfSSL_get_finished_client_on_handshake),
        TEST_SUCCESS);

    /* test received msg vs sent msg */
    ExpectIntEQ(0, XMEMCMP(client_side_msg1, server_side_msg2, WC_MAX_DIGEST_SIZE));
    ExpectIntEQ(0, XMEMCMP(client_side_msg2, server_side_msg1, WC_MAX_DIGEST_SIZE));
#endif /* HAVE_SSL_MEMIO_TESTS_DEPENDENCIES && WOLFSSL_HAVE_TLS_UNIQUE */

    return EXPECT_RESULT();
}

#if defined(HAVE_IO_TESTS_DEPENDENCIES) && defined(HAVE_EXT_CACHE) && \
    !defined(SINGLE_THREADED) && defined(WOLFSSL_TLS13) && \
    !defined(NO_SESSION_CACHE)

/* Sessions to restore/store */
static WOLFSSL_SESSION* test_wolfSSL_CTX_add_session_client_sess;
static WOLFSSL_SESSION* test_wolfSSL_CTX_add_session_server_sess;
static WOLFSSL_CTX*     test_wolfSSL_CTX_add_session_server_ctx;

static void test_wolfSSL_CTX_add_session_ctx_ready(WOLFSSL_CTX* ctx)
{
    /* Don't store sessions. Lookup is still enabled. */
    AssertIntEQ(wolfSSL_CTX_set_session_cache_mode(ctx,
            WOLFSSL_SESS_CACHE_NO_INTERNAL_STORE), WOLFSSL_SUCCESS);
#ifdef OPENSSL_EXTRA
    AssertIntEQ(wolfSSL_CTX_get_session_cache_mode(ctx) &
            WOLFSSL_SESS_CACHE_NO_INTERNAL_STORE,
            WOLFSSL_SESS_CACHE_NO_INTERNAL_STORE);
#endif
    /* Require both peers to provide certs */
    wolfSSL_CTX_set_verify(ctx, WOLFSSL_VERIFY_PEER, NULL);
}

static void test_wolfSSL_CTX_add_session_on_result(WOLFSSL* ssl)
{
    WOLFSSL_SESSION** sess;
#ifdef WOLFSSL_MUTEX_INITIALIZER
    static wolfSSL_Mutex m = WOLFSSL_MUTEX_INITIALIZER(m);

    (void)wc_LockMutex(&m);
#endif
    if (wolfSSL_is_server(ssl))
        sess = &test_wolfSSL_CTX_add_session_server_sess;
    else
        sess = &test_wolfSSL_CTX_add_session_client_sess;
    if (*sess == NULL) {
#ifdef NO_SESSION_CACHE_REF
        *sess = wolfSSL_get1_session(ssl);
        AssertNotNull(*sess);
#else
        /* Test for backwards compatibility */
        if (wolfSSL_is_server(ssl)) {
            *sess = wolfSSL_get1_session(ssl);
            AssertNotNull(*sess);
        }
        else {
            *sess = wolfSSL_get_session(ssl);
            AssertNotNull(*sess);
        }
#endif
        /* Now save the session in the internal store to make it available
         * for lookup. For TLS 1.3, we can't save the session without
         * WOLFSSL_TICKET_HAVE_ID because there is no way to retrieve the
         * session from cache. */
        if (wolfSSL_is_server(ssl)
#ifndef WOLFSSL_TICKET_HAVE_ID
                && wolfSSL_version(ssl) != TLS1_3_VERSION
#endif
                )
            AssertIntEQ(wolfSSL_CTX_add_session(wolfSSL_get_SSL_CTX(ssl),
                    *sess), WOLFSSL_SUCCESS);
    }
    else {
        /* If we have a session retrieved then remaining connections should be
         * resuming on that session */
        AssertIntEQ(wolfSSL_session_reused(ssl), 1);
    }
#ifdef WOLFSSL_MUTEX_INITIALIZER
    wc_UnLockMutex(&m);
#endif

    /* Save CTX to be able to decrypt tickets */
    if (wolfSSL_is_server(ssl) &&
            test_wolfSSL_CTX_add_session_server_ctx == NULL) {
        test_wolfSSL_CTX_add_session_server_ctx = wolfSSL_get_SSL_CTX(ssl);
        AssertNotNull(test_wolfSSL_CTX_add_session_server_ctx);
        AssertIntEQ(wolfSSL_CTX_up_ref(wolfSSL_get_SSL_CTX(ssl)),
                WOLFSSL_SUCCESS);
    }
#ifdef SESSION_CERTS
#ifndef WOLFSSL_TICKET_HAVE_ID
    if (wolfSSL_version(ssl) != TLS1_3_VERSION &&
            wolfSSL_session_reused(ssl))
#endif
    {
        /* With WOLFSSL_TICKET_HAVE_ID the peer certs should be available
         * for all connections. TLS 1.3 only has tickets so if we don't
         * include the session id in the ticket then the certificates
         * will not be available on resumption. */
    #ifdef KEEP_PEER_CERT
        WOLFSSL_X509* peer = wolfSSL_get_peer_certificate(ssl);
        AssertNotNull(peer);
        wolfSSL_X509_free(peer);
    #endif
        AssertNotNull(wolfSSL_SESSION_get_peer_chain(*sess));
    #ifdef OPENSSL_EXTRA
        AssertNotNull(SSL_SESSION_get0_peer(*sess));
    #endif
    }
#endif /* SESSION_CERTS */
}

static void test_wolfSSL_CTX_add_session_ssl_ready(WOLFSSL* ssl)
{
    /* Set the session to reuse for the client */
    AssertIntEQ(wolfSSL_set_session(ssl,
            test_wolfSSL_CTX_add_session_client_sess), WOLFSSL_SUCCESS);
}
#endif

static int test_wolfSSL_CTX_add_session(void)
{
    EXPECT_DECLS;
#if defined(HAVE_IO_TESTS_DEPENDENCIES) && defined(HAVE_EXT_CACHE) && \
    !defined(SINGLE_THREADED) && defined(WOLFSSL_TLS13) && \
    !defined(NO_SESSION_CACHE)
    tcp_ready ready;
    func_args client_args;
    func_args server_args;
    THREAD_TYPE serverThread;
    callback_functions client_cb;
    callback_functions server_cb;
    method_provider methods[][2] = {
#if !defined(NO_OLD_TLS) && ((!defined(NO_AES) && !defined(NO_AES_CBC)) || \
        !defined(NO_DES3))
        /* Without AES there are almost no ciphersuites available. This leads
         * to no ciphersuites being available and an error. */
        { wolfTLSv1_1_client_method, wolfTLSv1_1_server_method },
#endif
#ifndef WOLFSSL_NO_TLS12
        { wolfTLSv1_2_client_method, wolfTLSv1_2_server_method },
#endif
        /* Needs the default ticket callback since it is tied to the
         * connection context and this makes it easy to carry over the ticket
         * crypto context between connections */
#if defined(WOLFSSL_TLS13) && !defined(WOLFSSL_NO_DEF_TICKET_ENC_CB) && \
    defined(HAVE_SESSION_TICKET)
        { wolfTLSv1_3_client_method, wolfTLSv1_3_server_method },
#endif
    };
    const size_t methodsLen = sizeof(methods)/sizeof(*methods);
    size_t i, j;

    for (i = 0; i < methodsLen; i++) {
        /* First run creates a connection while the second+ run will attempt
         * to resume the connection. The trick is that the internal cache
         * is turned off. wolfSSL_CTX_add_session should put the session in
         * the cache anyway. */
        test_wolfSSL_CTX_add_session_client_sess = NULL;
        test_wolfSSL_CTX_add_session_server_sess = NULL;
        test_wolfSSL_CTX_add_session_server_ctx = NULL;

#ifdef NO_SESSION_CACHE_REF
        for (j = 0; j < 4; j++) {
#else
        /* The session may be overwritten in this case. Do only one resumption
         * to stop this test from failing intermittently. */
        for (j = 0; j < 2; j++) {
#endif
#ifdef WOLFSSL_TIRTOS
            fdOpenSession(Task_self());
#endif

            StartTCP();
            InitTcpReady(&ready);

            XMEMSET(&client_args, 0, sizeof(func_args));
            XMEMSET(&server_args, 0, sizeof(func_args));

            XMEMSET(&client_cb, 0, sizeof(callback_functions));
            XMEMSET(&server_cb, 0, sizeof(callback_functions));
            client_cb.method  = methods[i][0];
            server_cb.method  = methods[i][1];

            server_args.signal    = &ready;
            server_args.callbacks = &server_cb;
            client_args.signal    = &ready;
            client_args.callbacks = &client_cb;

            if (test_wolfSSL_CTX_add_session_server_ctx != NULL) {
                server_cb.ctx = test_wolfSSL_CTX_add_session_server_ctx;
                server_cb.isSharedCtx = 1;
            }
            server_cb.ctx_ready = test_wolfSSL_CTX_add_session_ctx_ready;
            client_cb.ctx_ready = test_wolfSSL_CTX_add_session_ctx_ready;
            if (j != 0)
                client_cb.ssl_ready = test_wolfSSL_CTX_add_session_ssl_ready;
            server_cb.on_result = test_wolfSSL_CTX_add_session_on_result;
            client_cb.on_result = test_wolfSSL_CTX_add_session_on_result;
            server_cb.ticNoInit = 1; /* Use default builtin */

            start_thread(test_server_nofail, &server_args, &serverThread);
            wait_tcp_ready(&server_args);
            test_client_nofail(&client_args, NULL);
            join_thread(serverThread);

            ExpectTrue(client_args.return_code);
            ExpectTrue(server_args.return_code);

            FreeTcpReady(&ready);

            if (EXPECT_FAIL())
                break;
        }
        wolfSSL_SESSION_free(test_wolfSSL_CTX_add_session_client_sess);
        wolfSSL_SESSION_free(test_wolfSSL_CTX_add_session_server_sess);
        wolfSSL_CTX_free(test_wolfSSL_CTX_add_session_server_ctx);

        if (EXPECT_FAIL())
            break;
    }
#endif

    return EXPECT_RESULT();
}
#if defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES) && defined(HAVE_EXT_CACHE) && \
     defined(WOLFSSL_TLS13) && !defined(NO_SESSION_CACHE) && \
     defined(OPENSSL_EXTRA) && defined(SESSION_CERTS) && \
     defined(HAVE_SESSION_TICKET) && \
    !defined(TITAN_SESSION_CACHE) && \
    !defined(HUGE_SESSION_CACHE) && \
    !defined(BIG_SESSION_CACHE) && \
    !defined(MEDIUM_SESSION_CACHE)

/* twcase - prefix for test_wolfSSL_CTX_add_session_ext */
/* Sessions to restore/store */
static WOLFSSL_SESSION* twcase_server_first_session_ptr;
static WOLFSSL_SESSION* twcase_client_first_session_ptr;
static WOLFSSL_CTX*     twcase_server_current_ctx_ptr;
static int twcase_new_session_called    = 0;
static int twcase_remove_session_called = 0;
static int twcase_get_session_called    = 0;

/* Test default, SESSIONS_PER_ROW*SESSION_ROWS = 3*11, see ssl.c */
#define SESSION_CACHE_SIZE 33

typedef struct {
    const byte* key;  /* key, altSessionID, session ID, NULL if empty */
    WOLFSSL_SESSION* value;
} hashTable_entry;

typedef struct {
    hashTable_entry entries[SESSION_CACHE_SIZE];  /* hash slots */
    size_t capacity;                     /* size of entries */
    size_t length;                       /* number of items in the hash table */
    wolfSSL_Mutex htLock;                /* lock */
}hashTable;

static hashTable server_sessionCache;

static int twcase_new_sessionCb(WOLFSSL *ssl, WOLFSSL_SESSION *sess)
{
    int i;
    unsigned int len;
    (void)ssl;

    /*
     * This example uses a hash table.
     * Steps you should take for a non-demo code:
     * - acquire a lock for the file named according to the session id
     * - open the file
     * - encrypt and write the SSL_SESSION object to the file
     * - release the lock
     *
     * Return:
     *  0: The callback does not wish to hold a reference of the sess
     *  1: The callback wants to hold a reference of the sess. The callback is
     *     now also responsible for calling wolfSSL_SESSION_free() on sess.
     */
    if (sess == NULL)
        return 0;

    if (wc_LockMutex(&server_sessionCache.htLock) != 0) {
        return 0;
    }
    for (i = 0; i < SESSION_CACHE_SIZE; i++) {
        if (server_sessionCache.entries[i].value == NULL) {
            server_sessionCache.entries[i].key = SSL_SESSION_get_id(sess, &len);
            server_sessionCache.entries[i].value = sess;
            server_sessionCache.length++;
            break;
        }
    }
    ++twcase_new_session_called;
    wc_UnLockMutex(&server_sessionCache.htLock);
    fprintf(stderr, "\t\ttwcase_new_session_called %d\n",
            twcase_new_session_called);
    return 1;
}

static void twcase_remove_sessionCb(WOLFSSL_CTX *ctx, WOLFSSL_SESSION *sess)
{
    int i;
    (void)ctx;
    (void)sess;

    if (sess == NULL)
        return;
    /*
     * This example uses a hash table.
     * Steps you should take for a non-demo code:
     * - acquire a lock for the file named according to the session id
     * - remove the file
     * - release the lock
     */
    if (wc_LockMutex(&server_sessionCache.htLock) != 0) {
        return;
    }
    for (i = 0; i < SESSION_CACHE_SIZE; i++) {
        if (server_sessionCache.entries[i].key != NULL &&
           XMEMCMP(server_sessionCache.entries[i].key,
                   sess->sessionID, SSL_MAX_SSL_SESSION_ID_LENGTH) == 0) {
            wolfSSL_SESSION_free(server_sessionCache.entries[i].value);
            server_sessionCache.entries[i].value = NULL;
            server_sessionCache.entries[i].key = NULL;
            server_sessionCache.length--;
            break;
        }
    }
    ++twcase_remove_session_called;
    wc_UnLockMutex(&server_sessionCache.htLock);
    fprintf(stderr, "\t\ttwcase_remove_session_called %d\n",
            twcase_remove_session_called);
}

static WOLFSSL_SESSION *twcase_get_sessionCb(WOLFSSL *ssl,
                                  const unsigned char *id, int len, int *ref)
{
    int i;
    (void)ssl;
    (void)id;
    (void)len;

    /*
     * This example uses a hash table.
     * Steps you should take for a non-demo code:
     * - acquire a lock for the file named according to the session id in the
     *   2nd arg
     * - read and decrypt contents of file and create a new SSL_SESSION
     * - object release the lock
     * - return the new session object
     */
    fprintf(stderr, "\t\ttwcase_get_session_called %d\n",
            ++twcase_get_session_called);
    /* This callback want to retain a copy of the object. If we want wolfSSL to
     * be responsible for the pointer then set to 0. */
    *ref = 1;

    for (i = 0; i < SESSION_CACHE_SIZE; i++) {
        if (server_sessionCache.entries[i].key != NULL &&
           XMEMCMP(server_sessionCache.entries[i].key, id,
                   SSL_MAX_SSL_SESSION_ID_LENGTH) == 0) {
           return server_sessionCache.entries[i].value;
        }
    }
    return NULL;
}
static int twcase_get_sessionCb_cleanup(void)
{
    int i;
    int cnt = 0;

    /* If  twcase_get_sessionCb sets *ref = 1, the application is responsible
     * for freeing sessions */

    for (i = 0; i < SESSION_CACHE_SIZE; i++) {
        if (server_sessionCache.entries[i].value != NULL) {
            wolfSSL_SESSION_free(server_sessionCache.entries[i].value);
            cnt++;
        }
    }

    fprintf(stderr, "\t\ttwcase_get_sessionCb_cleanup freed %d sessions\n",
            cnt);

    return TEST_SUCCESS;
}

static int twcase_cache_intOff_extOff(WOLFSSL_CTX* ctx)
{
    EXPECT_DECLS;
    /* off - Disable internal cache */
    ExpectIntEQ(wolfSSL_CTX_set_session_cache_mode(ctx,
            WOLFSSL_SESS_CACHE_NO_INTERNAL_STORE), WOLFSSL_SUCCESS);
#ifdef OPENSSL_EXTRA
    ExpectIntEQ(wolfSSL_CTX_get_session_cache_mode(ctx) &
            WOLFSSL_SESS_CACHE_NO_INTERNAL_STORE,
            WOLFSSL_SESS_CACHE_NO_INTERNAL_STORE);
#endif
    /* off - Do not setup external cache */

    /* Require both peers to provide certs */
    wolfSSL_CTX_set_verify(ctx, WOLFSSL_VERIFY_PEER, NULL);
    return EXPECT_RESULT();
}

static int twcase_cache_intOn_extOff(WOLFSSL_CTX* ctx)
{
    /* on - internal cache is on by default */
    /* off - Do not setup external cache */
    /* Require both peers to provide certs */
    wolfSSL_CTX_set_verify(ctx, WOLFSSL_VERIFY_PEER, NULL);
    return TEST_SUCCESS;
}

static int twcase_cache_intOff_extOn(WOLFSSL_CTX* ctx)
{
    EXPECT_DECLS;
    /* off - Disable internal cache */
    ExpectIntEQ(wolfSSL_CTX_set_session_cache_mode(ctx,
            WOLFSSL_SESS_CACHE_NO_INTERNAL_STORE), WOLFSSL_SUCCESS);
#ifdef OPENSSL_EXTRA
    ExpectIntEQ(wolfSSL_CTX_get_session_cache_mode(ctx) &
            WOLFSSL_SESS_CACHE_NO_INTERNAL_STORE,
            WOLFSSL_SESS_CACHE_NO_INTERNAL_STORE);
#endif
    /* on - Enable external cache */
    wolfSSL_CTX_sess_set_new_cb(ctx, twcase_new_sessionCb);
    wolfSSL_CTX_sess_set_remove_cb(ctx, twcase_remove_sessionCb);
    wolfSSL_CTX_sess_set_get_cb(ctx, twcase_get_sessionCb);

    /* Require both peers to provide certs */
    wolfSSL_CTX_set_verify(ctx, WOLFSSL_VERIFY_PEER, NULL);
    return EXPECT_RESULT();
}

static int twcase_cache_intOn_extOn(WOLFSSL_CTX* ctx)
{
    /* on - internal cache is on by default */
    /* on - Enable external cache */
    wolfSSL_CTX_sess_set_new_cb(ctx, twcase_new_sessionCb);
    wolfSSL_CTX_sess_set_remove_cb(ctx, twcase_remove_sessionCb);
    wolfSSL_CTX_sess_set_get_cb(ctx, twcase_get_sessionCb);

    /* Require both peers to provide certs */
    wolfSSL_CTX_set_verify(ctx, WOLFSSL_VERIFY_PEER, NULL);
    return TEST_SUCCESS;
}
static int twcase_cache_intOn_extOn_noTicket(WOLFSSL_CTX* ctx)
{
    /* on - internal cache is on by default */
    /* on - Enable external cache */
    wolfSSL_CTX_sess_set_new_cb(ctx, twcase_new_sessionCb);
    wolfSSL_CTX_sess_set_remove_cb(ctx, twcase_remove_sessionCb);
    wolfSSL_CTX_sess_set_get_cb(ctx, twcase_get_sessionCb);

    wolfSSL_CTX_set_options(ctx, WOLFSSL_OP_NO_TICKET);
    /* Require both peers to provide certs */
    wolfSSL_CTX_set_verify(ctx, WOLFSSL_VERIFY_PEER, NULL);
    return TEST_SUCCESS;
}
static int twcase_server_sess_ctx_pre_shutdown(WOLFSSL* ssl)
{
    EXPECT_DECLS;
    WOLFSSL_SESSION** sess;
    if (wolfSSL_is_server(ssl))
        sess = &twcase_server_first_session_ptr;
    else
        return TEST_SUCCESS;

    if (*sess == NULL) {
        ExpectNotNull(*sess = wolfSSL_get1_session(ssl));
        /* Now save the session in the internal store to make it available
         * for lookup. For TLS 1.3, we can't save the session without
         * WOLFSSL_TICKET_HAVE_ID because there is no way to retrieve the
         * session from cache. */
        if (wolfSSL_is_server(ssl)
#ifndef WOLFSSL_TICKET_HAVE_ID
                && wolfSSL_version(ssl) != TLS1_3_VERSION
                && wolfSSL_version(ssl) != DTLS1_3_VERSION
#endif
                ) {
            ExpectIntEQ(wolfSSL_CTX_add_session(wolfSSL_get_SSL_CTX(ssl),
                    *sess), WOLFSSL_SUCCESS);
        }
    }
    /* Save CTX to be able to decrypt tickets */
    if (twcase_server_current_ctx_ptr == NULL) {
        ExpectNotNull(twcase_server_current_ctx_ptr = wolfSSL_get_SSL_CTX(ssl));
        ExpectIntEQ(wolfSSL_CTX_up_ref(wolfSSL_get_SSL_CTX(ssl)),
                    WOLFSSL_SUCCESS);
    }
#ifdef SESSION_CERTS
#ifndef WOLFSSL_TICKET_HAVE_ID
    if (wolfSSL_version(ssl) != TLS1_3_VERSION &&
            wolfSSL_session_reused(ssl))
#endif
    {
        /* With WOLFSSL_TICKET_HAVE_ID the peer certs should be available
         * for all connections. TLS 1.3 only has tickets so if we don't
         * include the session id in the ticket then the certificates
         * will not be available on resumption. */
    #ifdef KEEP_PEER_CERT
        WOLFSSL_X509* peer = NULL;
        ExpectNotNull(peer = wolfSSL_get_peer_certificate(ssl));
        wolfSSL_X509_free(peer);
    #endif
        ExpectNotNull(wolfSSL_SESSION_get_peer_chain(*sess));
    }
#endif
    return EXPECT_RESULT();
}

static int twcase_client_sess_ctx_pre_shutdown(WOLFSSL* ssl)
{
    EXPECT_DECLS;
    WOLFSSL_SESSION** sess;
    sess = &twcase_client_first_session_ptr;
    if (*sess == NULL) {
        ExpectNotNull(*sess = wolfSSL_get1_session(ssl));
    }
    else {
        /* If we have a session retrieved then remaining connections should be
         * resuming on that session */
        ExpectIntEQ(wolfSSL_session_reused(ssl), 1);
    }

#ifdef SESSION_CERTS
#ifndef WOLFSSL_TICKET_HAVE_ID
    if (wolfSSL_version(ssl) != TLS1_3_VERSION &&
            wolfSSL_session_reused(ssl))
#endif
    {
    #ifdef KEEP_PEER_CERT
        WOLFSSL_X509* peer = wolfSSL_get_peer_certificate(ssl);
        ExpectNotNull(peer);
        wolfSSL_X509_free(peer);
    #endif
        ExpectNotNull(wolfSSL_SESSION_get_peer_chain(*sess));
#ifdef OPENSSL_EXTRA
        ExpectNotNull(wolfSSL_SESSION_get0_peer(*sess));
#endif
    }
#endif
    return EXPECT_RESULT();
}
static int twcase_client_set_sess_ssl_ready(WOLFSSL* ssl)
{
    EXPECT_DECLS;
    /* Set the session to reuse for the client */
    ExpectNotNull(ssl);
    ExpectNotNull(twcase_client_first_session_ptr);
    ExpectIntEQ(wolfSSL_set_session(ssl,twcase_client_first_session_ptr),
                WOLFSSL_SUCCESS);
    return EXPECT_RESULT();
}

struct test_add_session_ext_params {
    method_provider client_meth;
    method_provider server_meth;
    const char* tls_version;
};

static int test_wolfSSL_CTX_add_session_ext(
    struct test_add_session_ext_params* param)
{
    EXPECT_DECLS;
    /* Test the default 33 sessions */
    int j;

    /* Clear cache before starting */
    wolfSSL_CTX_flush_sessions(NULL, -1);

    XMEMSET(&server_sessionCache, 0, sizeof(hashTable));
    if (wc_InitMutex(&server_sessionCache.htLock) != 0)
        return BAD_MUTEX_E;
    server_sessionCache.capacity = SESSION_CACHE_SIZE;

    fprintf(stderr, "\tBegin %s\n", param->tls_version);
    for (j = 0; j < 5; j++) {
        int tls13 = XSTRSTR(param->tls_version, "TLSv1_3") != NULL;
        int dtls = XSTRSTR(param->tls_version, "DTLS") != NULL;
        test_ssl_cbf client_cb;
        test_ssl_cbf server_cb;

        (void)dtls;

        /* Test five cache configurations */
        twcase_client_first_session_ptr = NULL;
        twcase_server_first_session_ptr = NULL;
        twcase_server_current_ctx_ptr = NULL;
        twcase_new_session_called    = 0;
        twcase_remove_session_called = 0;
        twcase_get_session_called    = 0;

        /* connection 1 - first connection */
        fprintf(stderr, "\tconnect: %s: j=%d\n", param->tls_version, j);

        XMEMSET(&client_cb, 0, sizeof(client_cb));
        XMEMSET(&server_cb, 0, sizeof(server_cb));
        client_cb.method  = param->client_meth;
        server_cb.method  = param->server_meth;

        if (dtls)
            client_cb.doUdp = server_cb.doUdp = 1;

        /* Setup internal and external cache */
        switch (j) {
            case 0:
                /* SSL_OP_NO_TICKET stateful ticket case */
                server_cb.ctx_ready = twcase_cache_intOn_extOn_noTicket;
                break;
            case 1:
                server_cb.ctx_ready = twcase_cache_intOn_extOn;
                break;
            case 2:
                server_cb.ctx_ready = twcase_cache_intOff_extOn;
                break;
            case 3:
                server_cb.ctx_ready = twcase_cache_intOn_extOff;
                break;
            case 4:
                server_cb.ctx_ready = twcase_cache_intOff_extOff;
                break;
        }
        client_cb.ctx_ready = twcase_cache_intOff_extOff;

        /* Add session to internal cache and save SSL session for testing */
        server_cb.on_result = twcase_server_sess_ctx_pre_shutdown;
        /* Save client SSL session for testing */
        client_cb.on_result = twcase_client_sess_ctx_pre_shutdown;
        server_cb.ticNoInit = 1; /* Use default builtin */
        /* Don't free/release ctx */
        server_cb.ctx = twcase_server_current_ctx_ptr;
        server_cb.isSharedCtx = 1;

        ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&client_cb,
            &server_cb, NULL), TEST_SUCCESS);

        ExpectIntEQ(twcase_get_session_called, 0);
        if (EXPECT_FAIL()) {
            wolfSSL_SESSION_free(twcase_client_first_session_ptr);
            wolfSSL_SESSION_free(twcase_server_first_session_ptr);
            wolfSSL_CTX_free(twcase_server_current_ctx_ptr);
            break;
        }

        switch (j) {
            case 0:
            case 1:
            case 2:
                /* cache cannot be searched with out a connection */
                /* Add a new session */
                ExpectIntEQ(twcase_new_session_called, 1);
                /* In twcase_server_sess_ctx_pre_shutdown
                 * wolfSSL_CTX_add_session which evicts the existing session
                 * in cache and adds it back in */
                ExpectIntLE(twcase_remove_session_called, 1);
                break;
            case 3:
            case 4:
                /* no external cache  */
                ExpectIntEQ(twcase_new_session_called, 0);
                ExpectIntEQ(twcase_remove_session_called, 0);
                break;
        }

        /* connection 2 - session resume */
        fprintf(stderr, "\tresume: %s: j=%d\n", param->tls_version, j);
        twcase_new_session_called    = 0;
        twcase_remove_session_called = 0;
        twcase_get_session_called    = 0;
        server_cb.on_result = 0;
        client_cb.on_result = 0;
        server_cb.ticNoInit = 1; /* Use default builtin */

        server_cb.ctx = twcase_server_current_ctx_ptr;

        /* try session resumption */
        client_cb.ssl_ready = twcase_client_set_sess_ssl_ready;

        ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&client_cb,
            &server_cb, NULL), TEST_SUCCESS);

        /* Clear cache before checking */
        wolfSSL_CTX_flush_sessions(NULL, -1);

        switch (j) {
            case 0:
                if (tls13) {
                    /* (D)TLSv1.3 stateful case */
                    /* cache hit */
                    /* DTLS accesses cache once for stateless parsing and
                     * once for stateful parsing */
                    ExpectIntEQ(twcase_get_session_called, !dtls ? 1 : 2);

                    /* (D)TLSv1.3 creates a new ticket,
                     * updates both internal and external cache */
                    ExpectIntEQ(twcase_new_session_called, 1);
                    /* A new session ID is created for a new ticket */
                    ExpectIntEQ(twcase_remove_session_called, 2);

                }
                else {
                    /* non (D)TLSv1.3 case, no update */
                    /* DTLS accesses cache once for stateless parsing and
                     * once for stateful parsing */
#ifdef WOLFSSL_DTLS_NO_HVR_ON_RESUME
                    ExpectIntEQ(twcase_get_session_called, !dtls ? 1 : 2);
#else
                    ExpectIntEQ(twcase_get_session_called, 1);
#endif
                    ExpectIntEQ(twcase_new_session_called, 0);
                    /* Called on session added in
                     * twcase_server_sess_ctx_pre_shutdown */
                    ExpectIntEQ(twcase_remove_session_called, 1);
                }
                break;
            case 1:
                if (tls13) {
                    /* (D)TLSv1.3 case */
                    /* cache hit */
                    ExpectIntEQ(twcase_get_session_called, 1);
                    /* (D)TLSv1.3 creates a new ticket,
                     * updates both internal and external cache */
                    ExpectIntEQ(twcase_new_session_called, 1);
                    /* Called on session added in
                     * twcase_server_sess_ctx_pre_shutdown and by wolfSSL */
                    ExpectIntEQ(twcase_remove_session_called, 1);
                }
                else {
                    /* non (D)TLSv1.3 case */
                    /* cache hit */
                    /* DTLS accesses cache once for stateless parsing and
                     * once for stateful parsing */
#ifdef WOLFSSL_DTLS_NO_HVR_ON_RESUME
                    ExpectIntEQ(twcase_get_session_called, !dtls ? 1 : 2);
#else
                    ExpectIntEQ(twcase_get_session_called, 1);
#endif
                    ExpectIntEQ(twcase_new_session_called, 0);
                    /* Called on session added in
                     * twcase_server_sess_ctx_pre_shutdown */
                    ExpectIntEQ(twcase_remove_session_called, 1);
                }
                break;
            case 2:
                if (tls13) {
                    /* (D)TLSv1.3 case */
                    /* cache hit */
                    ExpectIntEQ(twcase_get_session_called, 1);
                    /* (D)TLSv1.3 creates a new ticket,
                     * updates both internal and external cache */
                    ExpectIntEQ(twcase_new_session_called, 1);
                    /* Called on session added in
                     * twcase_server_sess_ctx_pre_shutdown and by wolfSSL */
                    ExpectIntEQ(twcase_remove_session_called, 1);
                }
                else {
                    /* non (D)TLSv1.3 case */
                    /* cache hit */
                    /* DTLS accesses cache once for stateless parsing and
                     * once for stateful parsing */
#ifdef WOLFSSL_DTLS_NO_HVR_ON_RESUME
                    ExpectIntEQ(twcase_get_session_called, !dtls ? 1 : 2);
#else
                    ExpectIntEQ(twcase_get_session_called, 1);
#endif
                    ExpectIntEQ(twcase_new_session_called, 0);
                    /* Called on session added in
                     * twcase_server_sess_ctx_pre_shutdown */
                    ExpectIntEQ(twcase_remove_session_called, 1);
                }
                break;
            case 3:
            case 4:
                /* no external cache */
                ExpectIntEQ(twcase_get_session_called, 0);
                ExpectIntEQ(twcase_new_session_called, 0);
                ExpectIntEQ(twcase_remove_session_called, 0);
                break;
        }
        wolfSSL_SESSION_free(twcase_client_first_session_ptr);
        wolfSSL_SESSION_free(twcase_server_first_session_ptr);
        wolfSSL_CTX_free(twcase_server_current_ctx_ptr);

        if (EXPECT_FAIL())
            break;
    }
    twcase_get_sessionCb_cleanup();
    XMEMSET(&server_sessionCache.entries, 0,
            sizeof(server_sessionCache.entries));
    fprintf(stderr, "\tEnd %s\n", param->tls_version);

    wc_FreeMutex(&server_sessionCache.htLock);

    return EXPECT_RESULT();
}
#endif

static int test_wolfSSL_CTX_add_session_ext_tls13(void)
{
    EXPECT_DECLS;
#if defined(HAVE_IO_TESTS_DEPENDENCIES) && defined(HAVE_EXT_CACHE) && \
     defined(WOLFSSL_TLS13) && !defined(NO_SESSION_CACHE) && \
     defined(OPENSSL_EXTRA) && defined(SESSION_CERTS) && \
     defined(HAVE_SESSION_TICKET) && \
    !defined(TITAN_SESSION_CACHE) && \
    !defined(HUGE_SESSION_CACHE) && \
    !defined(BIG_SESSION_CACHE) && \
    !defined(MEDIUM_SESSION_CACHE)
#if defined(WOLFSSL_TLS13) && !defined(WOLFSSL_NO_DEF_TICKET_ENC_CB) && \
    defined(HAVE_SESSION_TICKET) && defined(WOLFSSL_TICKET_HAVE_ID)
    struct test_add_session_ext_params param[1] =  {
        { wolfTLSv1_3_client_method, wolfTLSv1_3_server_method, "TLSv1_3" }
    };
    ExpectIntEQ(test_wolfSSL_CTX_add_session_ext(param), TEST_SUCCESS);
#endif
#endif
    return EXPECT_RESULT();
}
static int test_wolfSSL_CTX_add_session_ext_dtls13(void)
{
    EXPECT_DECLS;
#if defined(HAVE_IO_TESTS_DEPENDENCIES) && defined(HAVE_EXT_CACHE) && \
     defined(WOLFSSL_TLS13) && !defined(NO_SESSION_CACHE) && \
     defined(OPENSSL_EXTRA) && defined(SESSION_CERTS) && \
     defined(HAVE_SESSION_TICKET) && \
    !defined(TITAN_SESSION_CACHE) && \
    !defined(HUGE_SESSION_CACHE) && \
    !defined(BIG_SESSION_CACHE) && \
    !defined(MEDIUM_SESSION_CACHE)
#if defined(WOLFSSL_TLS13) && !defined(WOLFSSL_NO_DEF_TICKET_ENC_CB) && \
    defined(HAVE_SESSION_TICKET) && defined(WOLFSSL_TICKET_HAVE_ID)
#ifdef WOLFSSL_DTLS13
    struct test_add_session_ext_params param[1] =  {
        { wolfDTLSv1_3_client_method, wolfDTLSv1_3_server_method, "DTLSv1_3" }
    };
    ExpectIntEQ(test_wolfSSL_CTX_add_session_ext(param), TEST_SUCCESS);
#endif
#endif
#endif
    return EXPECT_RESULT();
}
static int test_wolfSSL_CTX_add_session_ext_tls12(void)
{
    EXPECT_DECLS;
#if defined(HAVE_IO_TESTS_DEPENDENCIES) && defined(HAVE_EXT_CACHE) && \
     defined(WOLFSSL_TLS13) && !defined(NO_SESSION_CACHE) && \
     defined(OPENSSL_EXTRA) && defined(SESSION_CERTS) && \
     defined(HAVE_SESSION_TICKET) && \
    !defined(TITAN_SESSION_CACHE) && \
    !defined(HUGE_SESSION_CACHE) && \
    !defined(BIG_SESSION_CACHE) && \
    !defined(MEDIUM_SESSION_CACHE)
#ifndef WOLFSSL_NO_TLS12
    struct test_add_session_ext_params param[1] =  {
        { wolfTLSv1_2_client_method, wolfTLSv1_2_server_method, "TLSv1_2" }
    };
    ExpectIntEQ(test_wolfSSL_CTX_add_session_ext(param), TEST_SUCCESS);
#endif
#endif
    return EXPECT_RESULT();
}
static int test_wolfSSL_CTX_add_session_ext_dtls12(void)
{
    EXPECT_DECLS;
#if defined(HAVE_IO_TESTS_DEPENDENCIES) && defined(HAVE_EXT_CACHE) && \
     defined(WOLFSSL_TLS13) && !defined(NO_SESSION_CACHE) && \
     defined(OPENSSL_EXTRA) && defined(SESSION_CERTS) && \
     defined(HAVE_SESSION_TICKET) && \
    !defined(TITAN_SESSION_CACHE) && \
    !defined(HUGE_SESSION_CACHE) && \
    !defined(BIG_SESSION_CACHE) && \
    !defined(MEDIUM_SESSION_CACHE)
#ifndef WOLFSSL_NO_TLS12
#ifdef WOLFSSL_DTLS
    struct test_add_session_ext_params param[1] =  {
        { wolfDTLSv1_2_client_method, wolfDTLSv1_2_server_method, "DTLSv1_2" }
    };
    ExpectIntEQ(test_wolfSSL_CTX_add_session_ext(param), TEST_SUCCESS);
#endif
#endif
#endif
    return EXPECT_RESULT();
}
static int test_wolfSSL_CTX_add_session_ext_tls11(void)
{
    EXPECT_DECLS;
#if defined(HAVE_IO_TESTS_DEPENDENCIES) && defined(HAVE_EXT_CACHE) && \
     defined(WOLFSSL_TLS13) && !defined(NO_SESSION_CACHE) && \
     defined(OPENSSL_EXTRA) && defined(SESSION_CERTS) && \
     defined(HAVE_SESSION_TICKET) && \
    !defined(TITAN_SESSION_CACHE) && \
    !defined(HUGE_SESSION_CACHE) && \
    !defined(BIG_SESSION_CACHE) && \
    !defined(MEDIUM_SESSION_CACHE)
#if !defined(NO_OLD_TLS) && ((!defined(NO_AES) && !defined(NO_AES_CBC)) || \
        !defined(NO_DES3))
    struct test_add_session_ext_params param[1] =  {
        { wolfTLSv1_1_client_method, wolfTLSv1_1_server_method, "TLSv1_1" }
    };
    ExpectIntEQ(test_wolfSSL_CTX_add_session_ext(param), TEST_SUCCESS);
#endif
#endif
    return EXPECT_RESULT();
}
static int test_wolfSSL_CTX_add_session_ext_dtls1(void)
{
    EXPECT_DECLS;
#if defined(HAVE_IO_TESTS_DEPENDENCIES) && defined(HAVE_EXT_CACHE) && \
     defined(WOLFSSL_TLS13) && !defined(NO_SESSION_CACHE) && \
     defined(OPENSSL_EXTRA) && defined(SESSION_CERTS) && \
     defined(HAVE_SESSION_TICKET) && \
    !defined(TITAN_SESSION_CACHE) && \
    !defined(HUGE_SESSION_CACHE) && \
    !defined(BIG_SESSION_CACHE) && \
    !defined(MEDIUM_SESSION_CACHE)
#if !defined(NO_OLD_TLS) && ((!defined(NO_AES) && !defined(NO_AES_CBC)) || \
        !defined(NO_DES3))
#ifdef WOLFSSL_DTLS
    struct test_add_session_ext_params param[1] =  {
        { wolfDTLSv1_client_method, wolfDTLSv1_server_method, "DTLSv1_0" }
    };
    ExpectIntEQ(test_wolfSSL_CTX_add_session_ext(param), TEST_SUCCESS);
#endif
#endif
#endif
    return EXPECT_RESULT();
}

#if defined(WOLFSSL_DTLS) && defined(WOLFSSL_SESSION_EXPORT)
/* canned export of a session using older version 3 */
static unsigned char version_3[] = {
    0xA5, 0xA3, 0x01, 0x88, 0x00, 0x3c, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x80, 0x0C, 0x00, 0x00, 0x00,
    0x00, 0x80, 0x00, 0x1C, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC0, 0x30,
    0x05, 0x09, 0x0A, 0x01, 0x01, 0x00, 0x0D, 0x05,
    0xFE, 0xFD, 0x01, 0x25, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x06, 0x00, 0x05, 0x00, 0x06, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x06, 0x00, 0x01, 0x00, 0x07, 0x00, 0x00,
    0x00, 0x30, 0x00, 0x00, 0x00, 0x10, 0x01, 0x01,
    0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x3F,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x05,
    0x12, 0xCF, 0x22, 0xA1, 0x9F, 0x1C, 0x39, 0x1D,
    0x31, 0x11, 0x12, 0x1D, 0x11, 0x18, 0x0D, 0x0B,
    0xF3, 0xE1, 0x4D, 0xDC, 0xB1, 0xF1, 0x39, 0x98,
    0x91, 0x6C, 0x48, 0xE5, 0xED, 0x11, 0x12, 0xA0,
    0x00, 0xF2, 0x25, 0x4C, 0x09, 0x26, 0xD1, 0x74,
    0xDF, 0x23, 0x40, 0x15, 0x6A, 0x42, 0x2A, 0x26,
    0xA5, 0xAC, 0x56, 0xD5, 0x4A, 0x20, 0xB7, 0xE9,
    0xEF, 0xEB, 0xAF, 0xA8, 0x1E, 0x23, 0x7C, 0x04,
    0xAA, 0xA1, 0x6D, 0x92, 0x79, 0x7B, 0xFA, 0x80,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
    0x0C, 0x79, 0x7B, 0xFA, 0x80, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xAA, 0xA1, 0x6D,
    0x92, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x10, 0x00, 0x20, 0x00, 0x04, 0x00,
    0x10, 0x00, 0x10, 0x08, 0x02, 0x05, 0x08, 0x01,
    0x30, 0x28, 0x00, 0x00, 0x0F, 0x00, 0x02, 0x00,
    0x09, 0x31, 0x32, 0x37, 0x2E, 0x30, 0x2E, 0x30,
    0x2E, 0x31, 0xED, 0x4F
};
#endif /* defined(WOLFSSL_DTLS) && defined(WOLFSSL_SESSION_EXPORT) */

static int test_wolfSSL_dtls_export(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_DTLS) && defined(WOLFSSL_SESSION_EXPORT)
    tcp_ready ready;
    func_args client_args;
    func_args server_args;
    THREAD_TYPE serverThread;
    callback_functions server_cbf;
    callback_functions client_cbf;
#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    InitTcpReady(&ready);

#if defined(USE_WINDOWS_API)
    /* use RNG to get random port if using windows */
    ready.port = GetRandomPort();
#endif

    /* set using dtls */
    XMEMSET(&client_args, 0, sizeof(func_args));
    XMEMSET(&server_args, 0, sizeof(func_args));
    XMEMSET(&server_cbf, 0, sizeof(callback_functions));
    XMEMSET(&client_cbf, 0, sizeof(callback_functions));
    server_cbf.method = wolfDTLSv1_2_server_method;
    client_cbf.method = wolfDTLSv1_2_client_method;
    server_args.callbacks = &server_cbf;
    client_args.callbacks = &client_cbf;

    server_args.signal = &ready;
    client_args.signal = &ready;

    start_thread(run_wolfssl_server, &server_args, &serverThread);
    wait_tcp_ready(&server_args);
    run_wolfssl_client(&client_args);
    join_thread(serverThread);

    ExpectTrue(client_args.return_code);
    ExpectTrue(server_args.return_code);

    FreeTcpReady(&ready);

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    if (EXPECT_SUCCESS()) {
        SOCKET_T sockfd = 0;
        WOLFSSL_CTX* ctx = NULL;
        WOLFSSL*     ssl = NULL;
        char msg[64] = "hello wolfssl!";
        char reply[1024];
        int  msgSz = (int)XSTRLEN(msg);
        byte *session, *window;
        unsigned int sessionSz = 0;
        unsigned int windowSz = 0;

#ifndef TEST_IPV6
        struct sockaddr_in peerAddr;
#else
        struct sockaddr_in6 peerAddr;
#endif /* TEST_IPV6 */

        int i;


        /* Set ctx to DTLS 1.2 */
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfDTLSv1_2_server_method()));
        ExpectNotNull(ssl = wolfSSL_new(ctx));

        /* test importing version 3 */
        ExpectIntGE(wolfSSL_dtls_import(ssl, version_3, sizeof(version_3)), 0);

        /* test importing bad length and bad version */
        version_3[2]++;
        ExpectIntLT(wolfSSL_dtls_import(ssl, version_3, sizeof(version_3)), 0);
        version_3[2]--; version_3[1] = 0XA0;
        ExpectIntLT(wolfSSL_dtls_import(ssl, version_3, sizeof(version_3)), 0);
        wolfSSL_free(ssl);
        wolfSSL_CTX_free(ctx);


    /* check storing client state after connection and storing window only */
#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    InitTcpReady(&ready);

#if defined(USE_WINDOWS_API)
    /* use RNG to get random port if using windows */
    ready.port = GetRandomPort();
#endif

    /* set using dtls */
    XMEMSET(&server_args, 0, sizeof(func_args));
    XMEMSET(&server_cbf, 0, sizeof(callback_functions));
    server_cbf.method = wolfDTLSv1_2_server_method;
    server_cbf.doUdp = 1;
    server_args.callbacks = &server_cbf;
    server_args.argc = 3; /* set loop_count to 3 */


    server_args.signal = &ready;
    start_thread(test_server_nofail, &server_args, &serverThread);
    wait_tcp_ready(&server_args);

    /* create and connect with client */
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfDTLSv1_2_client_method()));
    ExpectIntEQ(WOLFSSL_SUCCESS,
            wolfSSL_CTX_load_verify_locations(ctx, caCertFile, 0));
    ExpectIntEQ(WOLFSSL_SUCCESS,
          wolfSSL_CTX_use_certificate_file(ctx, cliCertFile, SSL_FILETYPE_PEM));
    ExpectIntEQ(WOLFSSL_SUCCESS,
            wolfSSL_CTX_use_PrivateKey_file(ctx, cliKeyFile, SSL_FILETYPE_PEM));
    tcp_connect(&sockfd, wolfSSLIP, server_args.signal->port, 1, 0, NULL);
    ExpectNotNull(ssl = wolfSSL_new(ctx));
    ExpectIntEQ(wolfSSL_set_fd(ssl, sockfd), WOLFSSL_SUCCESS);

    /* store server information connected too */
    XMEMSET(&peerAddr, 0, sizeof(peerAddr));
#ifndef TEST_IPV6
    peerAddr.sin_family = AF_INET;
    ExpectIntEQ(XINET_PTON(AF_INET, wolfSSLIP, &peerAddr.sin_addr),1);
    peerAddr.sin_port = XHTONS(server_args.signal->port);
#else
    peerAddr.sin6_family = AF_INET6;
    ExpectIntEQ(
        XINET_PTON(AF_INET6, wolfSSLIP, &peerAddr.sin6_addr),1);
    peerAddr.sin6_port = XHTONS(server_args.signal->port);
#endif

    ExpectIntEQ(wolfSSL_dtls_set_peer(ssl, &peerAddr, sizeof(peerAddr)),
                    WOLFSSL_SUCCESS);

    ExpectIntEQ(wolfSSL_connect(ssl), WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_dtls_export(ssl, NULL, &sessionSz), 0);
    session = (byte*)XMALLOC(sessionSz, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    ExpectIntGT(wolfSSL_dtls_export(ssl, session, &sessionSz), 0);
    ExpectIntEQ(wolfSSL_write(ssl, msg, msgSz), msgSz);
    ExpectIntGT(wolfSSL_read(ssl, reply, sizeof(reply)), 0);
    ExpectIntEQ(wolfSSL_dtls_export_state_only(ssl, NULL, &windowSz), 0);
    window = (byte*)XMALLOC(windowSz, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    ExpectIntGT(wolfSSL_dtls_export_state_only(ssl, window, &windowSz), 0);
    wolfSSL_free(ssl);

    for (i = 1; EXPECT_SUCCESS() && i < server_args.argc; i++) {
        /* restore state */
        ExpectNotNull(ssl = wolfSSL_new(ctx));
        ExpectIntGT(wolfSSL_dtls_import(ssl, session, sessionSz), 0);
        ExpectIntGT(wolfSSL_dtls_import(ssl, window, windowSz), 0);
        ExpectIntEQ(wolfSSL_set_fd(ssl, sockfd), WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_dtls_set_peer(ssl, &peerAddr, sizeof(peerAddr)),
                    WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_write(ssl, msg, msgSz), msgSz);
        ExpectIntGE(wolfSSL_read(ssl, reply, sizeof(reply)), 0);
        ExpectIntGT(wolfSSL_dtls_export_state_only(ssl, window, &windowSz), 0);
        wolfSSL_free(ssl);
    }
    XFREE(session, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    XFREE(window, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    wolfSSL_CTX_free(ctx);

    fprintf(stderr, "done and waiting for server\n");
    join_thread(serverThread);
    ExpectIntEQ(server_args.return_code, TEST_SUCCESS);

    FreeTcpReady(&ready);

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif
    }
#endif

    return EXPECT_RESULT();
}

#if defined(WOLFSSL_SESSION_EXPORT) && !defined(WOLFSSL_NO_TLS12)
#ifdef WOLFSSL_TLS13
static const byte canned_client_tls13_session_v4[] = {
    0xA7, 0xA4, 0x01, 0x18, 0x00, 0x41, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x80, 0x04, 0x00, 0x00, 0x00,
    0x00, 0x80, 0x00, 0x1C, 0x01, 0x00, 0x00, 0x01,
    0x00, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01,
    0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x13,
    0x01, 0x0A, 0x0F, 0x10, 0x01, 0x02, 0x09, 0x00,
    0x05, 0x00, 0x00, 0x00, 0x00, 0x03, 0x04, 0x00,
    0xB7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x27, 0x00, 0x00, 0x00,
    0x11, 0x01, 0x01, 0x00, 0x20, 0x84, 0x4F, 0x18,
    0xD8, 0xC1, 0x24, 0xD8, 0xBB, 0x17, 0x9E, 0x31,
    0xA3, 0xF8, 0xA7, 0x3C, 0xBA, 0xEC, 0xFA, 0xB4,
    0x7F, 0xC5, 0x78, 0xEB, 0x6D, 0xE3, 0x2B, 0x7B,
    0x94, 0xBE, 0x20, 0x11, 0x7E, 0x17, 0x10, 0xA7,
    0x10, 0x19, 0xEC, 0x62, 0xCC, 0xBE, 0xF5, 0x01,
    0x35, 0x3C, 0xEA, 0xEF, 0x44, 0x3C, 0x40, 0xA2,
    0xBC, 0x18, 0x43, 0xA1, 0xA1, 0x65, 0x5C, 0x48,
    0xE2, 0xF9, 0x38, 0xEB, 0x11, 0x10, 0x72, 0x7C,
    0x78, 0x22, 0x13, 0x3B, 0x19, 0x40, 0xF0, 0x73,
    0xBE, 0x96, 0x14, 0x78, 0x26, 0xB9, 0x6B, 0x2E,
    0x72, 0x22, 0x0D, 0x90, 0x94, 0xDD, 0x78, 0x77,
    0xFC, 0x0C, 0x2E, 0x63, 0x6E, 0xF0, 0x0C, 0x35,
    0x41, 0xCD, 0xF3, 0x49, 0x31, 0x08, 0xD0, 0x6F,
    0x02, 0x3D, 0xC1, 0xD3, 0xB7, 0xEE, 0x3A, 0xA0,
    0x8E, 0xA1, 0x4D, 0xC3, 0x2E, 0x5E, 0x06, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0C,
    0x35, 0x41, 0xCD, 0xF3, 0x49, 0x31, 0x08, 0xD0,
    0x6F, 0x02, 0x3D, 0xC1, 0xD3, 0xB7, 0xEE, 0x3A,
    0xA0, 0x8E, 0xA1, 0x4D, 0xC3, 0x2E, 0x5E, 0x06,
    0x00, 0x10, 0x00, 0x10, 0x00, 0x0C, 0x00, 0x10,
    0x00, 0x10, 0x07, 0x02, 0x04, 0x00, 0x00, 0x20,
    0x28, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x03
};

static const byte canned_client_tls13_session_v5[] = {
    0xa7, 0xa5, 0x01, 0x19, 0x00, 0x42, 0x00, 0x00, 0x01, 0x00, 0x00, 0x80,
    0x04, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x1c, 0x01, 0x00, 0x00, 0x01,
    0x00, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x13, 0x01, 0x0a, 0x0f, 0x10,
    0x01, 0x02, 0x09, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x01, 0x03, 0x04,
    0x00, 0xb7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x27, 0x00, 0x00,
    0x00, 0x11, 0x01, 0x01, 0x00, 0x20, 0x84, 0x4f, 0x18, 0xd8, 0xc1, 0x24,
    0xd8, 0xbb, 0x17, 0x9e, 0x31, 0xa3, 0xf8, 0xa7, 0x3c, 0xba, 0xec, 0xfa,
    0xb4, 0x7f, 0xc5, 0x78, 0xeb, 0x6d, 0xe3, 0x2b, 0x7b, 0x94, 0xbe, 0x20,
    0x11, 0x7e, 0x17, 0x10, 0xa7, 0x10, 0x19, 0xec, 0x62, 0xcc, 0xbe, 0xf5,
    0x01, 0x35, 0x3c, 0xea, 0xef, 0x44, 0x3c, 0x40, 0xa2, 0xbc, 0x18, 0x43,
    0xa1, 0xa1, 0x65, 0x5c, 0x48, 0xe2, 0xf9, 0x38, 0xeb, 0x11, 0x10, 0x72,
    0x7c, 0x78, 0x22, 0x13, 0x3b, 0x19, 0x40, 0xf0, 0x73, 0xbe, 0x96, 0x14,
    0x78, 0x26, 0xb9, 0x6b, 0x2e, 0x72, 0x22, 0x0d, 0x90, 0x94, 0xdd, 0x78,
    0x77, 0xfc, 0x0c, 0x2e, 0x63, 0x6e, 0xf0, 0x0c, 0x35, 0x41, 0xcd, 0xf3,
    0x49, 0x31, 0x08, 0xd0, 0x6f, 0x02, 0x3d, 0xc1, 0xd3, 0xb7, 0xee, 0x3a,
    0xa0, 0x8e, 0xa1, 0x4d, 0xc3, 0x2e, 0x5e, 0x06, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0c, 0x35, 0x41, 0xcd, 0xf3, 0x49, 0x31, 0x08,
    0xd0, 0x6f, 0x02, 0x3d, 0xc1, 0xd3, 0xb7, 0xee, 0x3a, 0xa0, 0x8e, 0xa1,
    0x4d, 0xc3, 0x2e, 0x5e, 0x06, 0x00, 0x10, 0x00, 0x10, 0x00, 0x0c, 0x00,
    0x10, 0x00, 0x10, 0x07, 0x02, 0x04, 0x00, 0x00, 0x20, 0x28, 0x00, 0x00,
    0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03
};

static const byte canned_server_tls13_session[] = {
    0xA7, 0xA4, 0x01, 0x18, 0x00, 0x41, 0x01, 0x00,
    0x01, 0x00, 0x00, 0x80, 0x04, 0x00, 0x00, 0x00,
    0x00, 0x80, 0x00, 0x1C, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x13,
    0x01, 0x0A, 0x0F, 0x10, 0x01, 0x02, 0x00, 0x0F,
    0x05, 0x00, 0x00, 0x00, 0x00, 0x03, 0x04, 0x00,
    0xB7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x17, 0x00, 0x00, 0x00,
    0x11, 0x01, 0x01, 0x00, 0x20, 0x84, 0x4F, 0x18,
    0xD8, 0xC1, 0x24, 0xD8, 0xBB, 0x17, 0x9E, 0x31,
    0xA3, 0xF8, 0xA7, 0x3C, 0xBA, 0xEC, 0xFA, 0xB4,
    0x7F, 0xC5, 0x78, 0xEB, 0x6D, 0xE3, 0x2B, 0x7B,
    0x94, 0xBE, 0x20, 0x11, 0x7E, 0x17, 0x10, 0xA7,
    0x10, 0x19, 0xEC, 0x62, 0xCC, 0xBE, 0xF5, 0x01,
    0x35, 0x3C, 0xEA, 0xEF, 0x44, 0x3C, 0x40, 0xA2,
    0xBC, 0x18, 0x43, 0xA1, 0xA1, 0x65, 0x5C, 0x48,
    0xE2, 0xF9, 0x38, 0xEB, 0x11, 0x10, 0x72, 0x7C,
    0x78, 0x22, 0x13, 0x3B, 0x19, 0x40, 0xF0, 0x73,
    0xBE, 0x96, 0x14, 0x78, 0x26, 0xB9, 0x6B, 0x2E,
    0x72, 0x22, 0x0D, 0x90, 0x94, 0xDD, 0x78, 0x77,
    0xFC, 0x0C, 0x2E, 0x63, 0x6E, 0xF0, 0x0C, 0x35,
    0x41, 0xCD, 0xF3, 0x49, 0x31, 0x08, 0xD0, 0x6F,
    0x02, 0x3D, 0xC1, 0xD3, 0xB7, 0xEE, 0x3A, 0xA0,
    0x8E, 0xA1, 0x4D, 0xC3, 0x2E, 0x5E, 0x06, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0C,
    0xD3, 0xB7, 0xEE, 0x3A, 0xA0, 0x8E, 0xA1, 0x4D,
    0xC3, 0x2E, 0x5E, 0x06, 0x35, 0x41, 0xCD, 0xF3,
    0x49, 0x31, 0x08, 0xD0, 0x6F, 0x02, 0x3D, 0xC1,
    0x00, 0x10, 0x00, 0x10, 0x00, 0x0C, 0x00, 0x10,
    0x00, 0x10, 0x07, 0x02, 0x04, 0x00, 0x00, 0x20,
    0x28, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04
};
#endif /* WOLFSSL_TLS13 */

static const byte canned_client_session_v4[] = {
    0xA7, 0xA4, 0x01, 0x40, 0x00, 0x41, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x80, 0x02, 0x00, 0x00, 0x00,
    0x00, 0x80, 0x00, 0x1C, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC0,
    0x27, 0x0A, 0x0D, 0x10, 0x01, 0x01, 0x0A, 0x00,
    0x05, 0x00, 0x01, 0x01, 0x01, 0x03, 0x03, 0x00,
    0xBF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x0A, 0x01, 0x01, 0x00, 0x20, 0x69, 0x11, 0x6D,
    0x97, 0x15, 0x6E, 0x52, 0x27, 0xD6, 0x1D, 0x1D,
    0xF5, 0x0D, 0x59, 0xA5, 0xAC, 0x2E, 0x8C, 0x0E,
    0xCB, 0x26, 0x1E, 0xE2, 0xCE, 0xBB, 0xCE, 0xE1,
    0x7D, 0xD7, 0xEF, 0xA5, 0x44, 0x80, 0x2A, 0xDE,
    0xBB, 0x75, 0xB0, 0x1D, 0x75, 0x17, 0x20, 0x4C,
    0x08, 0x05, 0x1B, 0xBA, 0x60, 0x1F, 0x6C, 0x91,
    0x8C, 0xAA, 0xBB, 0xE5, 0xA3, 0x0B, 0x12, 0x3E,
    0xC0, 0x35, 0x43, 0x1D, 0xE2, 0x10, 0xE2, 0x02,
    0x92, 0x4B, 0x8F, 0x05, 0xA9, 0x4B, 0xCC, 0x90,
    0xC3, 0x0E, 0xC2, 0x0F, 0xE9, 0x33, 0x85, 0x9B,
    0x3C, 0x19, 0x21, 0xD5, 0x62, 0xE5, 0xE1, 0x17,
    0x8F, 0x8C, 0x19, 0x52, 0xD8, 0x59, 0x10, 0x2D,
    0x20, 0x6F, 0xBA, 0xC1, 0x1C, 0xD1, 0x82, 0xC7,
    0x32, 0x1B, 0xBB, 0xCC, 0x30, 0x03, 0xD7, 0x3A,
    0xC8, 0x18, 0xED, 0x58, 0xC8, 0x11, 0xFE, 0x71,
    0x9C, 0x71, 0xD8, 0x6B, 0xE0, 0x25, 0x64, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0C,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x10, 0x00, 0x10, 0x00, 0x10, 0x00, 0x10,
    0x00, 0x00, 0x06, 0x01, 0x04, 0x08, 0x01, 0x20,
    0x28, 0x00, 0x09, 0xE1, 0x50, 0x70, 0x02, 0x2F,
    0x7E, 0xDA, 0xBD, 0x40, 0xC5, 0x58, 0x87, 0xCE,
    0x43, 0xF3, 0xC5, 0x8F, 0xA1, 0x59, 0x93, 0xEF,
    0x7E, 0xD3, 0xD0, 0xB5, 0x87, 0x1D, 0x81, 0x54,
    0x14, 0x63, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x03
};

static const byte canned_client_session_v5[] = {
    0xa7, 0xa5, 0x01, 0x41, 0x00, 0x42, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
    0x02, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x1c, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x27, 0x0a, 0x0d, 0x10,
    0x01, 0x01, 0x0a, 0x00, 0x05, 0x00, 0x01, 0x01, 0x01, 0x01, 0x03, 0x03,
    0x00, 0xbf, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00,
    0x00, 0x0a, 0x01, 0x01, 0x00, 0x20, 0x69, 0x11, 0x6d, 0x97, 0x15, 0x6e,
    0x52, 0x27, 0xd6, 0x1d, 0x1d, 0xf5, 0x0d, 0x59, 0xa5, 0xac, 0x2e, 0x8c,
    0x0e, 0xcb, 0x26, 0x1e, 0xe2, 0xce, 0xbb, 0xce, 0xe1, 0x7d, 0xd7, 0xef,
    0xa5, 0x44, 0x80, 0x2a, 0xde, 0xbb, 0x75, 0xb0, 0x1d, 0x75, 0x17, 0x20,
    0x4c, 0x08, 0x05, 0x1b, 0xba, 0x60, 0x1f, 0x6c, 0x91, 0x8c, 0xaa, 0xbb,
    0xe5, 0xa3, 0x0b, 0x12, 0x3e, 0xc0, 0x35, 0x43, 0x1d, 0xe2, 0x10, 0xe2,
    0x02, 0x92, 0x4b, 0x8f, 0x05, 0xa9, 0x4b, 0xcc, 0x90, 0xc3, 0x0e, 0xc2,
    0x0f, 0xe9, 0x33, 0x85, 0x9b, 0x3c, 0x19, 0x21, 0xd5, 0x62, 0xe5, 0xe1,
    0x17, 0x8f, 0x8c, 0x19, 0x52, 0xd8, 0x59, 0x10, 0x2d, 0x20, 0x6f, 0xba,
    0xc1, 0x1c, 0xd1, 0x82, 0xc7, 0x32, 0x1b, 0xbb, 0xcc, 0x30, 0x03, 0xd7,
    0x3a, 0xc8, 0x18, 0xed, 0x58, 0xc8, 0x11, 0xfe, 0x71, 0x9c, 0x71, 0xd8,
    0x6b, 0xe0, 0x25, 0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x10, 0x00, 0x10, 0x00, 0x10, 0x00, 0x10, 0x00, 0x00, 0x06,
    0x01, 0x04, 0x08, 0x01, 0x20, 0x28, 0x00, 0x09, 0xe1, 0x50, 0x70, 0x02,
    0x2f, 0x7e, 0xda, 0xbd, 0x40, 0xc5, 0x58, 0x87, 0xce, 0x43, 0xf3, 0xc5,
    0x8f, 0xa1, 0x59, 0x93, 0xef, 0x7e, 0xd3, 0xd0, 0xb5, 0x87, 0x1d, 0x81,
    0x54, 0x14, 0x63, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03
};


static const byte canned_server_session[] = {
    0xA7, 0xA4, 0x01, 0x40, 0x00, 0x41, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x80, 0x02, 0x00, 0x00, 0x00,
    0x00, 0x80, 0x00, 0x1C, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC0,
    0x27, 0x08, 0x0F, 0x10, 0x01, 0x01, 0x00, 0x11,
    0x05, 0x00, 0x01, 0x01, 0x01, 0x03, 0x03, 0x00,
    0xBF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
    0x0A, 0x01, 0x01, 0x00, 0x20, 0x69, 0x11, 0x6D,
    0x97, 0x15, 0x6E, 0x52, 0x27, 0xD6, 0x1D, 0x1D,
    0xF5, 0x0D, 0x59, 0xA5, 0xAC, 0x2E, 0x8C, 0x0E,
    0xCB, 0x26, 0x1E, 0xE2, 0xCE, 0xBB, 0xCE, 0xE1,
    0x7D, 0xD7, 0xEF, 0xA5, 0x44, 0x80, 0x2A, 0xDE,
    0xBB, 0x75, 0xB0, 0x1D, 0x75, 0x17, 0x20, 0x4C,
    0x08, 0x05, 0x1B, 0xBA, 0x60, 0x1F, 0x6C, 0x91,
    0x8C, 0xAA, 0xBB, 0xE5, 0xA3, 0x0B, 0x12, 0x3E,
    0xC0, 0x35, 0x43, 0x1D, 0xE2, 0x10, 0xE2, 0x02,
    0x92, 0x4B, 0x8F, 0x05, 0xA9, 0x4B, 0xCC, 0x90,
    0xC3, 0x0E, 0xC2, 0x0F, 0xE9, 0x33, 0x85, 0x9B,
    0x3C, 0x19, 0x21, 0xD5, 0x62, 0xE5, 0xE1, 0x17,
    0x8F, 0x8C, 0x19, 0x52, 0xD8, 0x59, 0x10, 0x2D,
    0x20, 0x6F, 0xBA, 0xC1, 0x1C, 0xD1, 0x82, 0xC7,
    0x32, 0x1B, 0xBB, 0xCC, 0x30, 0x03, 0xD7, 0x3A,
    0xC8, 0x18, 0xED, 0x58, 0xC8, 0x11, 0xFE, 0x71,
    0x9C, 0x71, 0xD8, 0x6B, 0xE0, 0x25, 0x64, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0C,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x10, 0x00, 0x10, 0x00, 0x10, 0x00, 0x10,
    0x00, 0x00, 0x06, 0x01, 0x04, 0x08, 0x01, 0x20,
    0x28, 0x00, 0xC5, 0x8F, 0xA1, 0x59, 0x93, 0xEF,
    0x7E, 0xD3, 0xD0, 0xB5, 0x87, 0x1D, 0x81, 0x54,
    0x14, 0x63, 0x09, 0xE1, 0x50, 0x70, 0x02, 0x2F,
    0x7E, 0xDA, 0xBD, 0x40, 0xC5, 0x58, 0x87, 0xCE,
    0x43, 0xF3, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04
};

static THREAD_RETURN WOLFSSL_THREAD tls_export_server(void* args)
{
    SOCKET_T sockfd = 0;
    SOCKET_T clientfd = 0;
    word16 port;

    callback_functions* cbf;
    WOLFSSL_CTX* ctx = 0;
    WOLFSSL* ssl = 0;

    char msg[] = "I hear you fa shizzle!";
    char input[1024];
    int idx;

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    ((func_args*)args)->return_code = TEST_FAIL;
    cbf = ((func_args*)args)->callbacks;

#if defined(USE_WINDOWS_API)
    port = ((func_args*)args)->signal->port;
#elif defined(NO_MAIN_DRIVER) && !defined(WOLFSSL_SNIFFER) && \
     !defined(WOLFSSL_MDK_SHELL) && !defined(WOLFSSL_TIRTOS)
    /* Let tcp_listen assign port */
    port = 0;
#else
    /* Use default port */
    port = wolfSSLPort;
#endif

    /* do it here to detect failure */
    tcp_accept(&sockfd, &clientfd, (func_args*)args, port, 0, 0, 0, 0, 1, 0, 0);
    CloseSocket(sockfd);

    {
        WOLFSSL_METHOD* method = NULL;
        if (cbf != NULL && cbf->method != NULL) {
            method = cbf->method();
        }
        else {
            method = wolfTLSv1_2_server_method();
        }
        ctx = wolfSSL_CTX_new(method);
    }
    if (ctx == NULL) {
        goto done;
    }
    wolfSSL_CTX_set_cipher_list(ctx, "ECDHE-RSA-AES128-SHA256");

    /* call ctx setup callback */
    if (cbf != NULL && cbf->ctx_ready != NULL) {
        cbf->ctx_ready(ctx);
    }

    ssl = wolfSSL_new(ctx);
    if (ssl == NULL) {
        goto done;
    }
    wolfSSL_set_fd(ssl, clientfd);

    /* call ssl setup callback */
    if (cbf != NULL && cbf->ssl_ready != NULL) {
        cbf->ssl_ready(ssl);
    }
    idx = wolfSSL_read(ssl, input, sizeof(input)-1);
    if (idx > 0) {
        input[idx] = '\0';
        fprintf(stderr, "Client message export/import: %s\n", input);
    }
    else {
        fprintf(stderr, "ret = %d error = %d\n", idx,
            wolfSSL_get_error(ssl, idx));
        goto done;
    }

    if (wolfSSL_write(ssl, msg, sizeof(msg)) != sizeof(msg)) {
        /*err_sys("SSL_write failed");*/
        WOLFSSL_RETURN_FROM_THREAD(0);
    }

#ifdef WOLFSSL_TIRTOS
    Task_yield();
#endif

    ((func_args*)args)->return_code = TEST_SUCCESS;

done:
    wolfSSL_shutdown(ssl);
    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);

    CloseSocket(clientfd);

#ifdef WOLFSSL_TIRTOS
    fdCloseSession(Task_self());
#endif

#if defined(NO_MAIN_DRIVER) && defined(HAVE_ECC) && defined(FP_ECC) \
                            && defined(HAVE_THREAD_LS)
    wc_ecc_fp_free();  /* free per thread cache */
#endif

#if defined(HAVE_SESSION_TICKET) && \
    ((defined(HAVE_CHACHA) && defined(HAVE_POLY1305)) || defined(HAVE_AESGCM))
#if defined(OPENSSL_EXTRA) && defined(HAVE_AESGCM)
    OpenSSLTicketCleanup();
#elif defined(WOLFSSL_NO_DEF_TICKET_ENC_CB)
    TicketCleanup();
#endif
#endif

    WOLFSSL_RETURN_FROM_THREAD(0);
}


static void load_tls12_canned_server(WOLFSSL* ssl)
{
    int clientfd = wolfSSL_get_fd(ssl);
    AssertIntEQ(wolfSSL_tls_import(ssl, canned_server_session,
                sizeof(canned_server_session)), sizeof(canned_server_session));
    wolfSSL_set_fd(ssl, clientfd);
}


#ifdef WOLFSSL_TLS13
static void load_tls13_canned_server(WOLFSSL* ssl)
{
    int clientfd = wolfSSL_get_fd(ssl);
    AssertIntEQ(wolfSSL_tls_import(ssl, canned_server_tls13_session,
            sizeof(canned_server_tls13_session)),
            sizeof(canned_server_tls13_session));
    wolfSSL_set_fd(ssl, clientfd);
}
#endif


/* v is for version WOLFSSL_TLSV1_2 or WOLFSSL_TLSV1_3 */
static int test_wolfSSL_tls_export_run(method_provider server_method,
        method_provider client_method, ssl_callback ssl_ready,
        const byte* clientSession, int clientSessionSz, int cmpSess)
{
    EXPECT_DECLS;
    SOCKET_T sockfd = 0;
    WOLFSSL_CTX*     ctx     = 0;
    WOLFSSL*         ssl     = 0;
    char msg[64] = "hello wolfssl!";
    char reply[1024];
    word32 replySz;
    int  msgSz = (int)XSTRLEN(msg);

    tcp_ready ready;
    func_args server_args;
    THREAD_TYPE serverThread;
    callback_functions server_cbf;

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    (void)cmpSess;

    InitTcpReady(&ready);

#if defined(USE_WINDOWS_API)
    /* use RNG to get random port if using windows */
    ready.port = GetRandomPort();
#endif

    XMEMSET(&server_args, 0, sizeof(func_args));
    XMEMSET(&server_cbf, 0, sizeof(callback_functions));
    server_cbf.method = server_method;
    server_cbf.ssl_ready = ssl_ready;
    ExpectNotNull(ctx = wolfSSL_CTX_new(client_method()));
    server_args.callbacks = &server_cbf;
    server_args.signal    = &ready;

    start_thread(tls_export_server, &server_args, &serverThread);
    wait_tcp_ready(&server_args);


#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    ExpectNotNull(ssl = wolfSSL_new(ctx));
    tcp_connect(&sockfd, wolfSSLIP, ready.port, 0, 0, ssl);
    ExpectIntEQ(wolfSSL_tls_import(ssl, clientSession, clientSessionSz),
                clientSessionSz);
    replySz = sizeof(reply);
    ExpectIntGT(wolfSSL_tls_export(ssl, (byte*)reply, &replySz), 0);
#if !defined(NO_PSK) && defined(HAVE_ANON)
    if (cmpSess) {
        /* index 20 has is setting if PSK was on and 49 is if anon is allowed */
        ExpectIntEQ(replySz, clientSessionSz);
        ExpectBufEQ(reply, clientSession, replySz);
    }
#endif
    wolfSSL_set_fd(ssl, sockfd);

    ExpectIntEQ(wolfSSL_write(ssl, msg, msgSz), msgSz);
    ExpectIntGT(wolfSSL_read(ssl, reply, sizeof(reply)-1), 0);

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);

    CloseSocket(sockfd);

#ifdef WOLFSSL_TIRTOS
    fdCloseSession(Task_self());
#endif

#if defined(NO_MAIN_DRIVER) && defined(HAVE_ECC) && defined(FP_ECC) \
                            && defined(HAVE_THREAD_LS)
    wc_ecc_fp_free();  /* free per thread cache */
#endif

    join_thread(serverThread);

    ExpectIntEQ(server_args.return_code, TEST_SUCCESS);
    FreeTcpReady(&ready);

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    return EXPECT_RESULT();
}
#endif

static int test_wolfSSL_tls_export(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_SESSION_EXPORT) && !defined(WOLFSSL_NO_TLS12)
    EXPECT_TEST(test_wolfSSL_tls_export_run(wolfTLSv1_2_server_method,
        wolfTLSv1_2_client_method, load_tls12_canned_server,
        canned_client_session_v4, sizeof(canned_client_session_v4), 0));
    EXPECT_TEST(test_wolfSSL_tls_export_run(wolfTLSv1_2_server_method,
        wolfTLSv1_2_client_method, load_tls12_canned_server,
        canned_client_session_v5, sizeof(canned_client_session_v5), 1));
    #ifdef WOLFSSL_TLS13
    EXPECT_TEST(test_wolfSSL_tls_export_run(wolfTLSv1_3_server_method,
        wolfTLSv1_3_client_method, load_tls13_canned_server,
        canned_client_tls13_session_v4, sizeof(canned_client_tls13_session_v4),
        0));
    EXPECT_TEST(test_wolfSSL_tls_export_run(wolfTLSv1_3_server_method,
        wolfTLSv1_3_client_method, load_tls13_canned_server,
        canned_client_tls13_session_v5, sizeof(canned_client_tls13_session_v5),
        1));
    #endif
#endif

    return EXPECT_RESULT();
}

/*----------------------------------------------------------------------------*
 | TLS extensions tests
 *----------------------------------------------------------------------------*/

#ifdef ENABLE_TLS_CALLBACK_TEST
/* Connection test runner - generic */
static void test_wolfSSL_client_server(callback_functions* client_callbacks,
                                       callback_functions* server_callbacks)
{
    tcp_ready ready;
    func_args client_args;
    func_args server_args;
    THREAD_TYPE serverThread;

    XMEMSET(&client_args, 0, sizeof(func_args));
    XMEMSET(&server_args, 0, sizeof(func_args));

    StartTCP();

    client_args.callbacks = client_callbacks;
    server_args.callbacks = server_callbacks;

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    /* RUN Server side */
    InitTcpReady(&ready);

#if defined(USE_WINDOWS_API)
    /* use RNG to get random port if using windows */
    ready.port = GetRandomPort();
#endif

    server_args.signal = &ready;
    client_args.signal = &ready;
    start_thread(run_wolfssl_server, &server_args, &serverThread);
    wait_tcp_ready(&server_args);

    /* RUN Client side */
    run_wolfssl_client(&client_args);
    join_thread(serverThread);

    FreeTcpReady(&ready);
#ifdef WOLFSSL_TIRTOS
    fdCloseSession(Task_self());
#endif

    client_callbacks->return_code = client_args.return_code;
    server_callbacks->return_code = server_args.return_code;
}
#endif /* ENABLE_TLS_CALLBACK_TEST */


#ifdef HAVE_SNI
static int test_wolfSSL_UseSNI_params(void)
{
    EXPECT_DECLS;
#if !defined(NO_WOLFSSL_CLIENT)
    WOLFSSL_CTX *ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
    WOLFSSL     *ssl = wolfSSL_new(ctx);

    ExpectNotNull(ctx);
    ExpectNotNull(ssl);

    /* invalid [ctx|ssl] */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_UseSNI(NULL, 0, "ctx", 3));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_UseSNI(    NULL, 0, "ssl", 3));
    /* invalid type */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_UseSNI(ctx, (byte)-1, "ctx", 3));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_UseSNI(    ssl, (byte)-1, "ssl", 3));
    /* invalid data */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_UseSNI(ctx,  0, NULL,  3));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_UseSNI(    ssl,  0, NULL,  3));
    /* success case */
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_UseSNI(ctx,  0, "ctx", 3));
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseSNI(    ssl,  0, "ssl", 3));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif /* !NO_WOLFSSL_CLIENT */

    return EXPECT_RESULT();
}

/* BEGIN of connection tests callbacks */
static void use_SNI_at_ctx(WOLFSSL_CTX* ctx)
{
    AssertIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CTX_UseSNI(ctx, WOLFSSL_SNI_HOST_NAME, "www.wolfssl.com", 15));
}

static void use_SNI_at_ssl(WOLFSSL* ssl)
{
    AssertIntEQ(WOLFSSL_SUCCESS,
             wolfSSL_UseSNI(ssl, WOLFSSL_SNI_HOST_NAME, "www.wolfssl.com", 15));
}

static void different_SNI_at_ssl(WOLFSSL* ssl)
{
    AssertIntEQ(WOLFSSL_SUCCESS,
             wolfSSL_UseSNI(ssl, WOLFSSL_SNI_HOST_NAME, "ww2.wolfssl.com", 15));
}

static void use_SNI_WITH_CONTINUE_at_ssl(WOLFSSL* ssl)
{
    use_SNI_at_ssl(ssl);
    wolfSSL_SNI_SetOptions(ssl, WOLFSSL_SNI_HOST_NAME,
                                              WOLFSSL_SNI_CONTINUE_ON_MISMATCH);
}

static void use_SNI_WITH_FAKE_ANSWER_at_ssl(WOLFSSL* ssl)
{
    use_SNI_at_ssl(ssl);
    wolfSSL_SNI_SetOptions(ssl, WOLFSSL_SNI_HOST_NAME,
                                                WOLFSSL_SNI_ANSWER_ON_MISMATCH);
}

static void use_MANDATORY_SNI_at_ctx(WOLFSSL_CTX* ctx)
{
    use_SNI_at_ctx(ctx);
    wolfSSL_CTX_SNI_SetOptions(ctx, WOLFSSL_SNI_HOST_NAME,
                                                  WOLFSSL_SNI_ABORT_ON_ABSENCE);
}

static void use_MANDATORY_SNI_at_ssl(WOLFSSL* ssl)
{
    use_SNI_at_ssl(ssl);
    wolfSSL_SNI_SetOptions(ssl, WOLFSSL_SNI_HOST_NAME,
                                                  WOLFSSL_SNI_ABORT_ON_ABSENCE);
}

static void use_PSEUDO_MANDATORY_SNI_at_ctx(WOLFSSL_CTX* ctx)
{
    use_SNI_at_ctx(ctx);
    wolfSSL_CTX_SNI_SetOptions(ctx, WOLFSSL_SNI_HOST_NAME,
                 WOLFSSL_SNI_ANSWER_ON_MISMATCH | WOLFSSL_SNI_ABORT_ON_ABSENCE);
}

static void verify_UNKNOWN_SNI_on_server(WOLFSSL* ssl)
{
    AssertIntEQ(WC_NO_ERR_TRACE(UNKNOWN_SNI_HOST_NAME_E),
                wolfSSL_get_error(ssl, 0));
}

static void verify_SNI_ABSENT_on_server(WOLFSSL* ssl)
{
    AssertIntEQ(WC_NO_ERR_TRACE(SNI_ABSENT_ERROR), wolfSSL_get_error(ssl, 0));
}

static void verify_SNI_no_matching(WOLFSSL* ssl)
{
    byte type = WOLFSSL_SNI_HOST_NAME;
    void* request = (void*) &type; /* to be overwritten */

    AssertIntEQ(WOLFSSL_SNI_NO_MATCH, wolfSSL_SNI_Status(ssl, type));
    AssertNotNull(request);
    AssertIntEQ(0, wolfSSL_SNI_GetRequest(ssl, type, &request));
    AssertNull(request);
}

static void verify_SNI_real_matching(WOLFSSL* ssl)
{
    byte type = WOLFSSL_SNI_HOST_NAME;
    void* request = NULL;

    AssertIntEQ(WOLFSSL_SNI_REAL_MATCH, wolfSSL_SNI_Status(ssl, type));
    AssertIntEQ(15, wolfSSL_SNI_GetRequest(ssl, type, &request));
    AssertNotNull(request);
    AssertStrEQ("www.wolfssl.com", (char*)request);
}

static void verify_SNI_fake_matching(WOLFSSL* ssl)
{
    byte type = WOLFSSL_SNI_HOST_NAME;
    void* request = NULL;

    AssertIntEQ(WOLFSSL_SNI_FAKE_MATCH, wolfSSL_SNI_Status(ssl, type));
    AssertIntEQ(15, wolfSSL_SNI_GetRequest(ssl, type, &request));
    AssertNotNull(request);
    AssertStrEQ("ww2.wolfssl.com", (char*)request);
}

static void verify_FATAL_ERROR_on_client(WOLFSSL* ssl)
{
    AssertIntEQ(WC_NO_ERR_TRACE(FATAL_ERROR), wolfSSL_get_error(ssl, 0));
}
/* END of connection tests callbacks */

static int test_wolfSSL_UseSNI_connection(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_WOLFSSL_CLIENT) && !defined(NO_WOLFSSL_SERVER)
    callback_functions client_cb;
    callback_functions server_cb;
    size_t i;
#ifdef WOLFSSL_STATIC_MEMORY
    byte cliMem[TEST_TLS_STATIC_MEMSZ];
    byte svrMem[TEST_TLS_STATIC_MEMSZ];
#endif
    struct {
        method_provider client_meth;
        method_provider server_meth;
    #ifdef WOLFSSL_STATIC_MEMORY
        wolfSSL_method_func client_meth_ex;
        wolfSSL_method_func server_meth_ex;
    #endif
    } methods[] = {
#if defined(WOLFSSL_NO_TLS12) && !defined(WOLFSSL_TLS13)
        {wolfSSLv23_client_method, wolfSSLv23_server_method
        #ifdef WOLFSSL_STATIC_MEMORY
            ,wolfSSLv23_client_method_ex, wolfSSLv23_server_method_ex
        #endif
        },
#endif
#ifndef WOLFSSL_NO_TLS12
        {wolfTLSv1_2_client_method, wolfTLSv1_2_server_method
        #ifdef WOLFSSL_STATIC_MEMORY
            ,wolfTLSv1_2_client_method_ex, wolfTLSv1_2_server_method_ex
        #endif
        },
#endif
#ifdef WOLFSSL_TLS13
        {wolfTLSv1_3_client_method, wolfTLSv1_3_server_method
        #ifdef WOLFSSL_STATIC_MEMORY
            ,wolfTLSv1_3_client_method_ex, wolfTLSv1_3_server_method_ex
        #endif
        },
#endif
    };
    size_t methodsSz = sizeof(methods) / sizeof(*methods);

    for (i = 0; i < methodsSz; i++) {
        XMEMSET(&client_cb, 0, sizeof(callback_functions));
        XMEMSET(&server_cb, 0, sizeof(callback_functions));
        client_cb.method = methods[i].client_meth;
        server_cb.method = methods[i].server_meth;
        client_cb.devId = testDevId;
        server_cb.devId = testDevId;
    #ifdef WOLFSSL_STATIC_MEMORY
        client_cb.method_ex = methods[i].client_meth_ex;
        server_cb.method_ex = methods[i].server_meth_ex;
        client_cb.mem = cliMem;
        client_cb.memSz = (word32)sizeof(cliMem);
        server_cb.mem = svrMem;
        server_cb.memSz = (word32)sizeof(svrMem);;
    #endif

        /* success case at ctx */
        fprintf(stderr, "\n\tsuccess case at ctx\n");
        client_cb.ctx_ready = use_SNI_at_ctx; client_cb.ssl_ready = NULL; client_cb.on_result = NULL;
        server_cb.ctx_ready = use_SNI_at_ctx; server_cb.ssl_ready = NULL; server_cb.on_result = verify_SNI_real_matching;
        test_wolfSSL_client_server(&client_cb, &server_cb);

        /* success case at ssl */
        fprintf(stderr, "\tsuccess case at ssl\n");
        client_cb.ctx_ready = NULL; client_cb.ssl_ready = use_SNI_at_ssl; client_cb.on_result = verify_SNI_real_matching;
        server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_SNI_at_ssl; server_cb.on_result = verify_SNI_real_matching;
        test_wolfSSL_client_server(&client_cb, &server_cb);

        /* default mismatch behavior */
        fprintf(stderr, "\tdefault mismatch behavior\n");
        client_cb.ctx_ready = NULL; client_cb.ssl_ready = different_SNI_at_ssl; client_cb.on_result = verify_FATAL_ERROR_on_client;
        server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_SNI_at_ssl;       server_cb.on_result = verify_UNKNOWN_SNI_on_server;
        test_wolfSSL_client_server(&client_cb, &server_cb);

        /* continue on mismatch */
        fprintf(stderr, "\tcontinue on mismatch\n");
        client_cb.ctx_ready = NULL; client_cb.ssl_ready = different_SNI_at_ssl;         client_cb.on_result = NULL;
        server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_SNI_WITH_CONTINUE_at_ssl; server_cb.on_result = verify_SNI_no_matching;
        test_wolfSSL_client_server(&client_cb, &server_cb);

        /* fake answer on mismatch */
        fprintf(stderr, "\tfake answer on mismatch\n");
        client_cb.ctx_ready = NULL; client_cb.ssl_ready = different_SNI_at_ssl;            client_cb.on_result = NULL;
        server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_SNI_WITH_FAKE_ANSWER_at_ssl; server_cb.on_result = verify_SNI_fake_matching;
        test_wolfSSL_client_server(&client_cb, &server_cb);

        /* sni abort - success */
        fprintf(stderr, "\tsni abort - success\n");
        client_cb.ctx_ready = use_SNI_at_ctx;           client_cb.ssl_ready = NULL; client_cb.on_result = NULL;
        server_cb.ctx_ready = use_MANDATORY_SNI_at_ctx; server_cb.ssl_ready = NULL; server_cb.on_result = verify_SNI_real_matching;
        test_wolfSSL_client_server(&client_cb, &server_cb);

        /* sni abort - abort when absent (ctx) */
        fprintf(stderr, "\tsni abort - abort when absent (ctx)\n");
        client_cb.ctx_ready = NULL;                     client_cb.ssl_ready = NULL; client_cb.on_result = verify_FATAL_ERROR_on_client;
        server_cb.ctx_ready = use_MANDATORY_SNI_at_ctx; server_cb.ssl_ready = NULL; server_cb.on_result = verify_SNI_ABSENT_on_server;
        test_wolfSSL_client_server(&client_cb, &server_cb);

        /* sni abort - abort when absent (ssl) */
        fprintf(stderr, "\tsni abort - abort when absent (ssl)\n");
        client_cb.ctx_ready = NULL; client_cb.ssl_ready = NULL;                     client_cb.on_result = verify_FATAL_ERROR_on_client;
        server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_MANDATORY_SNI_at_ssl; server_cb.on_result = verify_SNI_ABSENT_on_server;
        test_wolfSSL_client_server(&client_cb, &server_cb);

        /* sni abort - success when overwritten */
        fprintf(stderr, "\tsni abort - success when overwritten\n");
        client_cb.ctx_ready = NULL;                     client_cb.ssl_ready = NULL;           client_cb.on_result = NULL;
        server_cb.ctx_ready = use_MANDATORY_SNI_at_ctx; server_cb.ssl_ready = use_SNI_at_ssl; server_cb.on_result = verify_SNI_no_matching;
        test_wolfSSL_client_server(&client_cb, &server_cb);

        /* sni abort - success when allowing mismatches */
        fprintf(stderr, "\tsni abort - success when allowing mismatches\n");
        client_cb.ctx_ready = NULL;                            client_cb.ssl_ready = different_SNI_at_ssl; client_cb.on_result = NULL;
        server_cb.ctx_ready = use_PSEUDO_MANDATORY_SNI_at_ctx; server_cb.ssl_ready = NULL;                 server_cb.on_result = verify_SNI_fake_matching;
        test_wolfSSL_client_server(&client_cb, &server_cb);
    }

    res = TEST_RES_CHECK(1);
#endif /* !NO_WOLFSSL_CLIENT && !NO_WOLFSSL_SERVER */

    return res;
}

static int test_wolfSSL_SNI_GetFromBuffer(void)
{
    EXPECT_DECLS;
    byte buff[] = { /* www.paypal.com */
        0x00, 0x00, 0x00, 0x00, 0xff, 0x01, 0x00, 0x00, 0x60, 0x03, 0x03, 0x5c,
        0xc4, 0xb3, 0x8c, 0x87, 0xef, 0xa4, 0x09, 0xe0, 0x02, 0xab, 0x86, 0xca,
        0x76, 0xf0, 0x9e, 0x01, 0x65, 0xf6, 0xa6, 0x06, 0x13, 0x1d, 0x0f, 0xa5,
        0x79, 0xb0, 0xd4, 0x77, 0x22, 0xeb, 0x1a, 0x00, 0x00, 0x16, 0x00, 0x6b,
        0x00, 0x67, 0x00, 0x39, 0x00, 0x33, 0x00, 0x3d, 0x00, 0x3c, 0x00, 0x35,
        0x00, 0x2f, 0x00, 0x05, 0x00, 0x04, 0x00, 0x0a, 0x01, 0x00, 0x00, 0x21,
        0x00, 0x00, 0x00, 0x13, 0x00, 0x11, 0x00, 0x00, 0x0e, 0x77, 0x77, 0x77,
        0x2e, 0x70, 0x61, 0x79, 0x70, 0x61, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x00,
        0x0d, 0x00, 0x06, 0x00, 0x04, 0x04, 0x01, 0x02, 0x01
    };

    byte buff2[] = { /* api.textmate.org */
        0x16, 0x03, 0x01, 0x00, 0xc6, 0x01, 0x00, 0x00, 0xc2, 0x03, 0x03, 0x52,
        0x8b, 0x7b, 0xca, 0x69, 0xec, 0x97, 0xd5, 0x08, 0x03, 0x50, 0xfe, 0x3b,
        0x99, 0xc3, 0x20, 0xce, 0xa5, 0xf6, 0x99, 0xa5, 0x71, 0xf9, 0x57, 0x7f,
        0x04, 0x38, 0xf6, 0x11, 0x0b, 0xb8, 0xd3, 0x00, 0x00, 0x5e, 0x00, 0xff,
        0xc0, 0x24, 0xc0, 0x23, 0xc0, 0x0a, 0xc0, 0x09, 0xc0, 0x07, 0xc0, 0x08,
        0xc0, 0x28, 0xc0, 0x27, 0xc0, 0x14, 0xc0, 0x13, 0xc0, 0x11, 0xc0, 0x12,
        0xc0, 0x26, 0xc0, 0x25, 0xc0, 0x2a, 0xc0, 0x29, 0xc0, 0x05, 0xc0, 0x04,
        0xc0, 0x02, 0xc0, 0x03, 0xc0, 0x0f, 0xc0, 0x0e, 0xc0, 0x0c, 0xc0, 0x0d,
        0x00, 0x3d, 0x00, 0x3c, 0x00, 0x2f, 0x00, 0x05, 0x00, 0x04, 0x00, 0x35,
        0x00, 0x0a, 0x00, 0x67, 0x00, 0x6b, 0x00, 0x33, 0x00, 0x39, 0x00, 0x16,
        0x00, 0xaf, 0x00, 0xae, 0x00, 0x8d, 0x00, 0x8c, 0x00, 0x8a, 0x00, 0x8b,
        0x00, 0xb1, 0x00, 0xb0, 0x00, 0x2c, 0x00, 0x3b, 0x01, 0x00, 0x00, 0x3b,
        0x00, 0x00, 0x00, 0x15, 0x00, 0x13, 0x00, 0x00, 0x10, 0x61, 0x70, 0x69,
        0x2e, 0x74, 0x65, 0x78, 0x74, 0x6d, 0x61, 0x74, 0x65, 0x2e, 0x6f, 0x72,
        0x67, 0x00, 0x0a, 0x00, 0x08, 0x00, 0x06, 0x00, 0x17, 0x00, 0x18, 0x00,
        0x19, 0x00, 0x0b, 0x00, 0x02, 0x01, 0x00, 0x00, 0x0d, 0x00, 0x0c, 0x00,
        0x0a, 0x05, 0x01, 0x04, 0x01, 0x02, 0x01, 0x04, 0x03, 0x02, 0x03
    };

    byte buff3[] = { /* no sni extension */
        0x16, 0x03, 0x03, 0x00, 0x4d, 0x01, 0x00, 0x00, 0x49, 0x03, 0x03, 0xea,
        0xa1, 0x9f, 0x60, 0xdd, 0x52, 0x12, 0x13, 0xbd, 0x84, 0x34, 0xd5, 0x1c,
        0x38, 0x25, 0xa8, 0x97, 0xd2, 0xd5, 0xc6, 0x45, 0xaf, 0x1b, 0x08, 0xe4,
        0x1e, 0xbb, 0xdf, 0x9d, 0x39, 0xf0, 0x65, 0x00, 0x00, 0x16, 0x00, 0x6b,
        0x00, 0x67, 0x00, 0x39, 0x00, 0x33, 0x00, 0x3d, 0x00, 0x3c, 0x00, 0x35,
        0x00, 0x2f, 0x00, 0x05, 0x00, 0x04, 0x00, 0x0a, 0x01, 0x00, 0x00, 0x0a,
        0x00, 0x0d, 0x00, 0x06, 0x00, 0x04, 0x04, 0x01, 0x02, 0x01
    };

    byte buff4[] = { /* last extension has zero size */
        0x16, 0x03, 0x01, 0x00, 0xba, 0x01, 0x00, 0x00,
        0xb6, 0x03, 0x03, 0x83, 0xa3, 0xe6, 0xdc, 0x16, 0xa1, 0x43, 0xe9, 0x45,
        0x15, 0xbd, 0x64, 0xa9, 0xb6, 0x07, 0xb4, 0x50, 0xc6, 0xdd, 0xff, 0xc2,
        0xd3, 0x0d, 0x4f, 0x36, 0xb4, 0x41, 0x51, 0x61, 0xc1, 0xa5, 0x9e, 0x00,
        0x00, 0x28, 0xcc, 0x14, 0xcc, 0x13, 0xc0, 0x2b, 0xc0, 0x2f, 0x00, 0x9e,
        0xc0, 0x0a, 0xc0, 0x09, 0xc0, 0x13, 0xc0, 0x14, 0xc0, 0x07, 0xc0, 0x11,
        0x00, 0x33, 0x00, 0x32, 0x00, 0x39, 0x00, 0x9c, 0x00, 0x2f, 0x00, 0x35,
        0x00, 0x0a, 0x00, 0x05, 0x00, 0x04, 0x01, 0x00, 0x00, 0x65, 0xff, 0x01,
        0x00, 0x01, 0x00, 0x00, 0x0a, 0x00, 0x08, 0x00, 0x06, 0x00, 0x17, 0x00,
        0x18, 0x00, 0x19, 0x00, 0x0b, 0x00, 0x02, 0x01, 0x00, 0x00, 0x23, 0x00,
        0x00, 0x33, 0x74, 0x00, 0x00, 0x00, 0x10, 0x00, 0x1b, 0x00, 0x19, 0x06,
        0x73, 0x70, 0x64, 0x79, 0x2f, 0x33, 0x08, 0x73, 0x70, 0x64, 0x79, 0x2f,
        0x33, 0x2e, 0x31, 0x08, 0x68, 0x74, 0x74, 0x70, 0x2f, 0x31, 0x2e, 0x31,
        0x75, 0x50, 0x00, 0x00, 0x00, 0x05, 0x00, 0x05, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x0d, 0x00, 0x12, 0x00, 0x10, 0x04, 0x01, 0x05, 0x01, 0x02,
        0x01, 0x04, 0x03, 0x05, 0x03, 0x02, 0x03, 0x04, 0x02, 0x02, 0x02, 0x00,
        0x12, 0x00, 0x00
    };

    byte buff5[] = { /* SSL v2.0 client hello */
        0x00, 0x2b, 0x01, 0x03, 0x01, 0x00, 0x09, 0x00, 0x00,
        /* dummy bytes below, just to pass size check */
        0xb6, 0x03, 0x03, 0x83, 0xa3, 0xe6, 0xdc, 0x16, 0xa1, 0x43, 0xe9, 0x45,
        0x15, 0xbd, 0x64, 0xa9, 0xb6, 0x07, 0xb4, 0x50, 0xc6, 0xdd, 0xff, 0xc2,
        0xd3, 0x0d, 0x4f, 0x36, 0xb4, 0x41, 0x51, 0x61, 0xc1, 0xa5, 0x9e, 0x00,
    };

    byte result[32] = {0};
    word32 length   = 32;

    ExpectIntEQ(0, wolfSSL_SNI_GetFromBuffer(buff4, sizeof(buff4),
                                                           0, result, &length));

    ExpectIntEQ(0, wolfSSL_SNI_GetFromBuffer(buff3, sizeof(buff3),
                                                           0, result, &length));

    ExpectIntEQ(0, wolfSSL_SNI_GetFromBuffer(buff2, sizeof(buff2),
                                                           1, result, &length));

    ExpectIntEQ(WC_NO_ERR_TRACE(BUFFER_ERROR), wolfSSL_SNI_GetFromBuffer(buff, sizeof(buff),
                                                           0, result, &length));
    buff[0] = 0x16;

    ExpectIntEQ(WC_NO_ERR_TRACE(BUFFER_ERROR), wolfSSL_SNI_GetFromBuffer(buff, sizeof(buff),
                                                           0, result, &length));
    buff[1] = 0x03;

    ExpectIntEQ(WC_NO_ERR_TRACE(SNI_UNSUPPORTED), wolfSSL_SNI_GetFromBuffer(buff,
                                           sizeof(buff), 0, result, &length));
    buff[2] = 0x03;

    ExpectIntEQ(WC_NO_ERR_TRACE(INCOMPLETE_DATA), wolfSSL_SNI_GetFromBuffer(buff,
                                           sizeof(buff), 0, result, &length));
    buff[4] = 0x64;

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_SNI_GetFromBuffer(buff, sizeof(buff),
                                                           0, result, &length));
    if (EXPECT_SUCCESS())
        result[length] = 0;
    ExpectStrEQ("www.paypal.com", (const char*) result);

    length = 32;

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_SNI_GetFromBuffer(buff2, sizeof(buff2),
                                                           0, result, &length));
    if (EXPECT_SUCCESS())
        result[length] = 0;
    ExpectStrEQ("api.textmate.org", (const char*) result);

    /* SSL v2.0 tests */
    ExpectIntEQ(WC_NO_ERR_TRACE(SNI_UNSUPPORTED), wolfSSL_SNI_GetFromBuffer(buff5,
                                          sizeof(buff5), 0, result, &length));

    buff5[2] = 0x02;
    ExpectIntEQ(WC_NO_ERR_TRACE(BUFFER_ERROR), wolfSSL_SNI_GetFromBuffer(buff5,
                                          sizeof(buff5), 0, result, &length));

    buff5[2] = 0x01; buff5[6] = 0x08;
    ExpectIntEQ(WC_NO_ERR_TRACE(BUFFER_ERROR), wolfSSL_SNI_GetFromBuffer(buff5,
                                          sizeof(buff5), 0, result, &length));

    buff5[6] = 0x09; buff5[8] = 0x01;
    ExpectIntEQ(WC_NO_ERR_TRACE(BUFFER_ERROR), wolfSSL_SNI_GetFromBuffer(buff5,
                                          sizeof(buff5), 0, result, &length));

    return EXPECT_RESULT();
}

#endif /* HAVE_SNI */

#endif /* HAVE_IO_TESTS_DEPENDENCIES */


#if defined(WOLFSSL_DTLS) && defined(WOLFSSL_SESSION_EXPORT) && \
    defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES)
/* Dummy peer functions to satisfy the exporter/importer */
static int test_wolfSSL_dtls_export_peers_get_peer(WOLFSSL* ssl, char* ip,
        int* ipSz, unsigned short* port, int* fam)
{
    (void)ssl;
    ip[0] = -1;
    *ipSz = 1;
    *port = 1;
    *fam = 2;
    return 1;
}

static int test_wolfSSL_dtls_export_peers_set_peer(WOLFSSL* ssl, char* ip,
        int ipSz, unsigned short port, int fam)
{
    (void)ssl;
    if (ip[0] != -1 || ipSz != 1 || port != 1 || fam != 2)
        return 0;
    return 1;
}

static int test_wolfSSL_dtls_export_peers_on_handshake(WOLFSSL_CTX **ctx,
        WOLFSSL **ssl)
{
    EXPECT_DECLS;
    unsigned char* sessionBuf = NULL;
    unsigned int sessionSz = 0;
    void* ioWriteCtx = wolfSSL_GetIOWriteCtx(*ssl);
    void* ioReadCtx = wolfSSL_GetIOReadCtx(*ssl);

    wolfSSL_CTX_SetIOGetPeer(*ctx, test_wolfSSL_dtls_export_peers_get_peer);
    wolfSSL_CTX_SetIOSetPeer(*ctx, test_wolfSSL_dtls_export_peers_set_peer);
    ExpectIntGE(wolfSSL_dtls_export(*ssl, NULL, &sessionSz), 0);
    ExpectNotNull(sessionBuf =
        (unsigned char*)XMALLOC(sessionSz, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER));
    ExpectIntGE(wolfSSL_dtls_export(*ssl, sessionBuf, &sessionSz), 0);
    wolfSSL_free(*ssl);
    *ssl = NULL;
    ExpectNotNull(*ssl = wolfSSL_new(*ctx));
    ExpectIntGE(wolfSSL_dtls_import(*ssl, sessionBuf, sessionSz), 0);
    wolfSSL_SetIOWriteCtx(*ssl, ioWriteCtx);
    wolfSSL_SetIOReadCtx(*ssl, ioReadCtx);

    XFREE(sessionBuf, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    return EXPECT_RESULT();
}
#endif

static int test_wolfSSL_dtls_export_peers(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_DTLS) && defined(WOLFSSL_SESSION_EXPORT) && \
    defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES)
    test_ssl_cbf client_cbf;
    test_ssl_cbf server_cbf;
    size_t i, j;
    struct test_params {
        method_provider client_meth;
        method_provider server_meth;
        const char* dtls_version;
    } params[] = {
#ifndef NO_OLD_TLS
        {wolfDTLSv1_client_method, wolfDTLSv1_server_method, "1.0"},
#endif
        {wolfDTLSv1_2_client_method, wolfDTLSv1_2_server_method, "1.2"},
        /* TODO DTLS 1.3 exporting not supported
#ifdef WOLFSSL_DTLS13
        {wolfDTLSv1_3_client_method, wolfDTLSv1_3_server_method, "1.3"},
#endif
         */
    };

    for (i = 0; i < sizeof(params)/sizeof(*params); i++) {
        for (j = 0; j <= 0b11; j++) {
            XMEMSET(&client_cbf, 0, sizeof(client_cbf));
            XMEMSET(&server_cbf, 0, sizeof(server_cbf));

            printf("\n\tTesting DTLS %s connection;", params[i].dtls_version);

            client_cbf.method = params[i].client_meth;
            server_cbf.method = params[i].server_meth;

            if (j & 0b01) {
                client_cbf.on_handshake =
                        test_wolfSSL_dtls_export_peers_on_handshake;
                printf(" With client export;");
            }
            if (j & 0b10) {
                server_cbf.on_handshake =
                        test_wolfSSL_dtls_export_peers_on_handshake;
                printf(" With server export;");
            }

            printf("\n");

            ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&client_cbf,
                &server_cbf, NULL), TEST_SUCCESS);
            if (!EXPECT_SUCCESS())
                break;
        }
    }
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_UseTrustedCA(void)
{
    EXPECT_DECLS;
#if defined(HAVE_TRUSTED_CA) && !defined(NO_CERTS) && !defined(NO_FILESYSTEM) \
    && !defined(NO_RSA)
#if !defined(NO_TLS) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
    WOLFSSL_CTX *ctx = NULL;
    WOLFSSL     *ssl = NULL;
    byte        id[20];

#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull((ctx = wolfSSL_CTX_new(wolfSSLv23_server_method())));
    ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, svrCertFile, WOLFSSL_FILETYPE_PEM));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, svrKeyFile, WOLFSSL_FILETYPE_PEM));
#else
    ExpectNotNull((ctx = wolfSSL_CTX_new(wolfSSLv23_client_method())));
#endif
    ExpectNotNull((ssl = wolfSSL_new(ctx)));
    XMEMSET(id, 0, sizeof(id));

    /* error cases */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_UseTrustedCA(NULL, 0, NULL, 0));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_UseTrustedCA(ssl,
                WOLFSSL_TRUSTED_CA_CERT_SHA1+1, NULL, 0));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_UseTrustedCA(ssl,
                WOLFSSL_TRUSTED_CA_CERT_SHA1, NULL, 0));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_UseTrustedCA(ssl,
                WOLFSSL_TRUSTED_CA_CERT_SHA1, id, 5));
#ifdef NO_SHA
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_UseTrustedCA(ssl,
                WOLFSSL_TRUSTED_CA_KEY_SHA1, id, sizeof(id)));
#endif
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_UseTrustedCA(ssl,
                WOLFSSL_TRUSTED_CA_X509_NAME, id, 0));

    /* success cases */
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseTrustedCA(ssl,
                WOLFSSL_TRUSTED_CA_PRE_AGREED, NULL, 0));
#ifndef NO_SHA
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseTrustedCA(ssl,
                WOLFSSL_TRUSTED_CA_KEY_SHA1, id, sizeof(id)));
#endif
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseTrustedCA(ssl,
                WOLFSSL_TRUSTED_CA_X509_NAME, id, 5));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif /* !NO_TLS && (!NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER) */
#endif /* HAVE_TRUSTED_CA */
    return EXPECT_RESULT();
}

static int test_wolfSSL_UseMaxFragment(void)
{
    EXPECT_DECLS;
#if defined(HAVE_MAX_FRAGMENT) && !defined(NO_CERTS) && \
    !defined(NO_FILESYSTEM) && !defined(NO_RSA)

#if !defined(NO_TLS) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
  #ifndef NO_WOLFSSL_SERVER
    WOLFSSL_CTX* ctx = wolfSSL_CTX_new(wolfSSLv23_server_method());
  #else
    WOLFSSL_CTX* ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
  #endif
    WOLFSSL     *ssl = NULL;
  #ifdef OPENSSL_EXTRA
    int (*UseMaxFragment)(SSL *s, unsigned char mode);
    int (*CTX_UseMaxFragment)(SSL_CTX *c, unsigned char mode);
  #else
    int (*UseMaxFragment)(WOLFSSL *s, unsigned char mode);
    int (*CTX_UseMaxFragment)(WOLFSSL_CTX *c, unsigned char mode);
  #endif

  #ifndef NO_WOLFSSL_SERVER
    ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, svrCertFile, WOLFSSL_FILETYPE_PEM));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, svrKeyFile, WOLFSSL_FILETYPE_PEM));
  #endif

    ExpectNotNull(ctx);

    ExpectNotNull(ssl = wolfSSL_new(ctx));

  #ifdef OPENSSL_EXTRA
    CTX_UseMaxFragment = SSL_CTX_set_tlsext_max_fragment_length;
    UseMaxFragment = SSL_set_tlsext_max_fragment_length;
  #else
    UseMaxFragment = wolfSSL_UseMaxFragment;
    CTX_UseMaxFragment = wolfSSL_CTX_UseMaxFragment;
  #endif

    /* error cases */
    ExpectIntNE(WOLFSSL_SUCCESS, CTX_UseMaxFragment(NULL, WOLFSSL_MFL_2_9));
    ExpectIntNE(WOLFSSL_SUCCESS, UseMaxFragment(    NULL, WOLFSSL_MFL_2_9));
    ExpectIntNE(WOLFSSL_SUCCESS, CTX_UseMaxFragment(ctx, WOLFSSL_MFL_MIN-1));
    ExpectIntNE(WOLFSSL_SUCCESS, CTX_UseMaxFragment(ctx, WOLFSSL_MFL_MAX+1));
    ExpectIntNE(WOLFSSL_SUCCESS, UseMaxFragment(ssl, WOLFSSL_MFL_MIN-1));
    ExpectIntNE(WOLFSSL_SUCCESS, UseMaxFragment(ssl, WOLFSSL_MFL_MAX+1));

    /* success case */
  #ifdef OPENSSL_EXTRA
    ExpectIntEQ(WC_NO_ERR_TRACE(BAD_FUNC_ARG), CTX_UseMaxFragment(ctx,  WOLFSSL_MFL_2_8));
  #else
    ExpectIntEQ(WOLFSSL_SUCCESS, CTX_UseMaxFragment(ctx,  WOLFSSL_MFL_2_8));
  #endif
    ExpectIntEQ(WOLFSSL_SUCCESS, CTX_UseMaxFragment(ctx,  WOLFSSL_MFL_2_9));
    ExpectIntEQ(WOLFSSL_SUCCESS, CTX_UseMaxFragment(ctx,  WOLFSSL_MFL_2_10));
    ExpectIntEQ(WOLFSSL_SUCCESS, CTX_UseMaxFragment(ctx,  WOLFSSL_MFL_2_11));
    ExpectIntEQ(WOLFSSL_SUCCESS, CTX_UseMaxFragment(ctx,  WOLFSSL_MFL_2_12));
  #ifdef OPENSSL_EXTRA
    ExpectIntEQ(WC_NO_ERR_TRACE(BAD_FUNC_ARG), CTX_UseMaxFragment(ctx, WOLFSSL_MFL_2_13));

    ExpectIntEQ(WC_NO_ERR_TRACE(BAD_FUNC_ARG), UseMaxFragment(    ssl,  WOLFSSL_MFL_2_8));
  #else
    ExpectIntEQ(WOLFSSL_SUCCESS, CTX_UseMaxFragment(ctx,  WOLFSSL_MFL_2_13));

    ExpectIntEQ(WOLFSSL_SUCCESS, UseMaxFragment(    ssl,  WOLFSSL_MFL_2_8));
  #endif
    ExpectIntEQ(WOLFSSL_SUCCESS, UseMaxFragment(    ssl,  WOLFSSL_MFL_2_9));
    ExpectIntEQ(WOLFSSL_SUCCESS, UseMaxFragment(    ssl,  WOLFSSL_MFL_2_10));
    ExpectIntEQ(WOLFSSL_SUCCESS, UseMaxFragment(    ssl,  WOLFSSL_MFL_2_11));
    ExpectIntEQ(WOLFSSL_SUCCESS, UseMaxFragment(    ssl,  WOLFSSL_MFL_2_12));

  #ifdef OPENSSL_EXTRA
    ExpectIntEQ(WC_NO_ERR_TRACE(BAD_FUNC_ARG), UseMaxFragment(    ssl,  WOLFSSL_MFL_2_13));
  #else
    ExpectIntEQ(WOLFSSL_SUCCESS, UseMaxFragment(    ssl,  WOLFSSL_MFL_2_13));
  #endif

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);

#if defined(OPENSSL_EXTRA) && defined(HAVE_MAX_FRAGMENT) && \
    defined(HAVE_MANUAL_MEMIO_TESTS_DEPENDENCIES)
    /* check negotiated max fragment size */
    {
        WOLFSSL *ssl_c = NULL;
        WOLFSSL *ssl_s = NULL;
        struct test_memio_ctx test_ctx;
        WOLFSSL_CTX *ctx_c = NULL;
        WOLFSSL_CTX *ctx_s = NULL;

        XMEMSET(&test_ctx, 0, sizeof(test_ctx));
        ExpectIntEQ(test_memio_setup(&test_ctx, &ctx_c, &ctx_s, &ssl_c, &ssl_s,
            wolfTLSv1_2_client_method, wolfTLSv1_2_server_method), 0);
        ExpectIntEQ(wolfSSL_UseMaxFragment(ssl_c, WOLFSSL_MFL_2_8),
            WOLFSSL_SUCCESS);
        ExpectIntEQ(test_memio_do_handshake(ssl_c, ssl_s, 10, NULL), 0);
#ifndef NO_SESSION_CACHE
        ExpectIntEQ(SSL_SESSION_get_max_fragment_length(
            wolfSSL_get_session(ssl_c)), WOLFSSL_MFL_2_8);
#endif

        wolfSSL_free(ssl_c);
        wolfSSL_free(ssl_s);
        wolfSSL_CTX_free(ctx_c);
        wolfSSL_CTX_free(ctx_s);
    }
#endif
#endif /* !NO_TLS && (!NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER) */
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_UseTruncatedHMAC(void)
{
    EXPECT_DECLS;
#if defined(HAVE_TRUNCATED_HMAC) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    !defined(NO_FILESYSTEM) && !defined(NO_RSA)
#if !defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)
  #ifndef NO_WOLFSSL_SERVER
    WOLFSSL_CTX* ctx = wolfSSL_CTX_new(wolfSSLv23_server_method());
  #else
    WOLFSSL_CTX* ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
  #endif
    WOLFSSL     *ssl = NULL;

    ExpectNotNull(ctx);

  #ifndef NO_WOLFSSL_SERVER
    ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, svrCertFile, WOLFSSL_FILETYPE_PEM));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, svrKeyFile, WOLFSSL_FILETYPE_PEM));
  #endif

    ExpectNotNull(ssl = wolfSSL_new(ctx));

    /* error cases */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_UseTruncatedHMAC(NULL));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_UseTruncatedHMAC(NULL));

    /* success case */
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_UseTruncatedHMAC(ctx));
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseTruncatedHMAC(ssl));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif /* !NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER */
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_UseSupportedCurve(void)
{
    EXPECT_DECLS;
#if defined(HAVE_SUPPORTED_CURVES) && !defined(NO_WOLFSSL_CLIENT) && \
    !defined(NO_TLS)
    WOLFSSL_CTX* ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
    WOLFSSL     *ssl = wolfSSL_new(ctx);

    ExpectNotNull(ctx);
    ExpectNotNull(ssl);

    /* error cases */
    ExpectIntNE(WOLFSSL_SUCCESS,
                    wolfSSL_CTX_UseSupportedCurve(NULL, WOLFSSL_ECC_SECP256R1));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_UseSupportedCurve(ctx,  0));

    ExpectIntNE(WOLFSSL_SUCCESS,
                        wolfSSL_UseSupportedCurve(NULL, WOLFSSL_ECC_SECP256R1));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_UseSupportedCurve(ssl,  0));

    /* success case */
    ExpectIntEQ(WOLFSSL_SUCCESS,
                     wolfSSL_CTX_UseSupportedCurve(ctx, WOLFSSL_ECC_SECP256R1));
    ExpectIntEQ(WOLFSSL_SUCCESS,
                         wolfSSL_UseSupportedCurve(ssl, WOLFSSL_ECC_SECP256R1));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif

    return EXPECT_RESULT();
}

#if defined(HAVE_ALPN) && defined(HAVE_IO_TESTS_DEPENDENCIES)

static void verify_ALPN_FATAL_ERROR_on_client(WOLFSSL* ssl)
{
    AssertIntEQ(WC_NO_ERR_TRACE(UNKNOWN_ALPN_PROTOCOL_NAME_E), wolfSSL_get_error(ssl, 0));
}

static void use_ALPN_all(WOLFSSL* ssl)
{
    /* http/1.1,spdy/1,spdy/2,spdy/3 */
    char alpn_list[] = {0x68, 0x74, 0x74, 0x70, 0x2f, 0x31, 0x2e, 0x31, 0x2c,
                        0x73, 0x70, 0x64, 0x79, 0x2f, 0x31, 0x2c,
                        0x73, 0x70, 0x64, 0x79, 0x2f, 0x32, 0x2c,
                        0x73, 0x70, 0x64, 0x79, 0x2f, 0x33};
    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseALPN(ssl, alpn_list, sizeof(alpn_list),
                                             WOLFSSL_ALPN_FAILED_ON_MISMATCH));
}

static void use_ALPN_all_continue(WOLFSSL* ssl)
{
    /* http/1.1,spdy/1,spdy/2,spdy/3 */
    char alpn_list[] = {0x68, 0x74, 0x74, 0x70, 0x2f, 0x31, 0x2e, 0x31, 0x2c,
        0x73, 0x70, 0x64, 0x79, 0x2f, 0x31, 0x2c,
        0x73, 0x70, 0x64, 0x79, 0x2f, 0x32, 0x2c,
        0x73, 0x70, 0x64, 0x79, 0x2f, 0x33};
    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseALPN(ssl, alpn_list, sizeof(alpn_list),
                                             WOLFSSL_ALPN_CONTINUE_ON_MISMATCH));
}

static void use_ALPN_one(WOLFSSL* ssl)
{
    /* spdy/2 */
    char proto[] = {0x73, 0x70, 0x64, 0x79, 0x2f, 0x32};

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseALPN(ssl, proto, sizeof(proto),
                                             WOLFSSL_ALPN_FAILED_ON_MISMATCH));
}

static void use_ALPN_unknown(WOLFSSL* ssl)
{
    /* http/2.0 */
    char proto[] = {0x68, 0x74, 0x74, 0x70, 0x2f, 0x32, 0x2e, 0x30};

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseALPN(ssl, proto, sizeof(proto),
                                             WOLFSSL_ALPN_FAILED_ON_MISMATCH));
}

static void use_ALPN_unknown_continue(WOLFSSL* ssl)
{
    /* http/2.0 */
    char proto[] = {0x68, 0x74, 0x74, 0x70, 0x2f, 0x32, 0x2e, 0x30};

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseALPN(ssl, proto, sizeof(proto),
                                             WOLFSSL_ALPN_CONTINUE_ON_MISMATCH));
}

static void verify_ALPN_not_matching_spdy3(WOLFSSL* ssl)
{
    /* spdy/3 */
    char nego_proto[] = {0x73, 0x70, 0x64, 0x79, 0x2f, 0x33};

    char *proto = NULL;
    word16 protoSz = 0;

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_ALPN_GetProtocol(ssl, &proto, &protoSz));

    /* check value */
    AssertIntNE(1, sizeof(nego_proto) == protoSz);
    if (proto) {
        AssertIntNE(0, XMEMCMP(nego_proto, proto, sizeof(nego_proto)));
    }
}

static void verify_ALPN_not_matching_continue(WOLFSSL* ssl)
{
    char *proto = NULL;
    word16 protoSz = 0;

    AssertIntEQ(WC_NO_ERR_TRACE(WOLFSSL_ALPN_NOT_FOUND),
                wolfSSL_ALPN_GetProtocol(ssl, &proto, &protoSz));

    /* check value */
    AssertIntEQ(1, (0 == protoSz));
    AssertIntEQ(1, (NULL == proto));
}

static void verify_ALPN_matching_http1(WOLFSSL* ssl)
{
    /* http/1.1 */
    char nego_proto[] = {0x68, 0x74, 0x74, 0x70, 0x2f, 0x31, 0x2e, 0x31};
    char *proto;
    word16 protoSz = 0;

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_ALPN_GetProtocol(ssl, &proto, &protoSz));

    /* check value */
    AssertIntEQ(1, sizeof(nego_proto) == protoSz);
    AssertIntEQ(0, XMEMCMP(nego_proto, proto, protoSz));
}

static void verify_ALPN_matching_spdy2(WOLFSSL* ssl)
{
    /* spdy/2 */
    char nego_proto[] = {0x73, 0x70, 0x64, 0x79, 0x2f, 0x32};
    char *proto;
    word16 protoSz = 0;

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_ALPN_GetProtocol(ssl, &proto, &protoSz));

    /* check value */
    AssertIntEQ(1, sizeof(nego_proto) == protoSz);
    AssertIntEQ(0, XMEMCMP(nego_proto, proto, protoSz));
}

static void verify_ALPN_client_list(WOLFSSL* ssl)
{
    /* http/1.1,spdy/1,spdy/2,spdy/3 */
    char alpn_list[] = {0x68, 0x74, 0x74, 0x70, 0x2f, 0x31, 0x2e, 0x31, 0x2c,
                        0x73, 0x70, 0x64, 0x79, 0x2f, 0x31, 0x2c,
                        0x73, 0x70, 0x64, 0x79, 0x2f, 0x32, 0x2c,
                        0x73, 0x70, 0x64, 0x79, 0x2f, 0x33};
    char    *clist = NULL;
    word16  clistSz = 0;

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_ALPN_GetPeerProtocol(ssl, &clist,
                                                          &clistSz));

    /* check value */
    AssertIntEQ(1, sizeof(alpn_list) == clistSz);
    AssertIntEQ(0, XMEMCMP(alpn_list, clist, clistSz));

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_ALPN_FreePeerProtocol(ssl, &clist));
}

#if defined(OPENSSL_ALL) || defined(WOLFSSL_NGINX) || \
    defined(WOLFSSL_HAPROXY) || defined(HAVE_LIGHTY)

/* ALPN select callback, success with spdy/2 */
static int select_ALPN_spdy2(WOLFSSL *ssl, const unsigned char **out,
    unsigned char *outlen, const unsigned char *in,
    unsigned int inlen, void *arg)
{
    /* spdy/2 */
    const char proto[] = {0x73, 0x70, 0x64, 0x79, 0x2f, 0x32};

    (void)ssl;
    (void)arg;

    /* adding +1 since LEN byte comes first */
    if (inlen < sizeof(proto) + 1) {
        return SSL_TLSEXT_ERR_ALERT_FATAL;
    }

    if (XMEMCMP(in + 1, proto, sizeof(proto)) == 0) {
        *out = in + 1;
        *outlen = (unsigned char)sizeof(proto);
        return SSL_TLSEXT_ERR_OK;
    }

    return SSL_TLSEXT_ERR_ALERT_FATAL;
}

/* ALPN select callback, force failure */
static int select_ALPN_failure(WOLFSSL *ssl, const unsigned char **out,
    unsigned char *outlen, const unsigned char *in,
    unsigned int inlen, void *arg)
{
    (void)ssl;
    (void)out;
    (void)outlen;
    (void)in;
    (void)inlen;
    (void)arg;

    return SSL_TLSEXT_ERR_ALERT_FATAL;
}

static void use_ALPN_spdy2_callback(WOLFSSL* ssl)
{
    wolfSSL_set_alpn_select_cb(ssl, select_ALPN_spdy2, NULL);
}

static void use_ALPN_failure_callback(WOLFSSL* ssl)
{
    wolfSSL_set_alpn_select_cb(ssl, select_ALPN_failure, NULL);
}
#endif /* OPENSSL_ALL | NGINX | HAPROXY | LIGHTY | QUIC */

static int test_wolfSSL_UseALPN_connection(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_WOLFSSL_CLIENT) && !defined(NO_WOLFSSL_SERVER)
    callback_functions client_cb;
    callback_functions server_cb;

    XMEMSET(&client_cb, 0, sizeof(callback_functions));
    XMEMSET(&server_cb, 0, sizeof(callback_functions));
    client_cb.method = wolfSSLv23_client_method;
    server_cb.method = wolfSSLv23_server_method;
    client_cb.devId = testDevId;
    server_cb.devId = testDevId;

    /* success case same list */
    client_cb.ctx_ready = NULL; client_cb.ssl_ready = use_ALPN_all; client_cb.on_result = NULL;
    server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_ALPN_all; server_cb.on_result = verify_ALPN_matching_http1;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* success case only one for server */
    client_cb.ctx_ready = NULL; client_cb.ssl_ready = use_ALPN_all; client_cb.on_result = NULL;
    server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_ALPN_one; server_cb.on_result = verify_ALPN_matching_spdy2;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* success case only one for client */
    client_cb.ctx_ready = NULL; client_cb.ssl_ready = use_ALPN_one; client_cb.on_result = NULL;
    server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_ALPN_all; server_cb.on_result = verify_ALPN_matching_spdy2;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* success case none for client */
    client_cb.ctx_ready = NULL; client_cb.ssl_ready = NULL;         client_cb.on_result = NULL;
    server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_ALPN_all; server_cb.on_result = NULL;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* success case mismatch behavior but option 'continue' set */
    client_cb.ctx_ready = NULL; client_cb.ssl_ready = use_ALPN_all_continue;     client_cb.on_result = verify_ALPN_not_matching_continue;
    server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_ALPN_unknown_continue; server_cb.on_result = NULL;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* success case read protocol send by client */
    client_cb.ctx_ready = NULL; client_cb.ssl_ready = use_ALPN_all; client_cb.on_result = NULL;
    server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_ALPN_one; server_cb.on_result = verify_ALPN_client_list;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* mismatch behavior with same list
        * the first and only this one must be taken */
    client_cb.ctx_ready = NULL; client_cb.ssl_ready = use_ALPN_all; client_cb.on_result = NULL;
    server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_ALPN_all; server_cb.on_result = verify_ALPN_not_matching_spdy3;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* default mismatch behavior */
    client_cb.ctx_ready = NULL; client_cb.ssl_ready = use_ALPN_all;     client_cb.on_result = NULL;
    server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_ALPN_unknown; server_cb.on_result = verify_ALPN_FATAL_ERROR_on_client;
    test_wolfSSL_client_server(&client_cb, &server_cb);

#if defined(OPENSSL_ALL) || defined(WOLFSSL_NGINX) || \
    defined(WOLFSSL_HAPROXY) || defined(HAVE_LIGHTY)

    /* WOLFSSL-level ALPN select callback tests */
    /* Callback: success (one protocol, spdy/2) */
    client_cb.ctx_ready = NULL;
    client_cb.ssl_ready = use_ALPN_one;
    client_cb.on_result = verify_ALPN_matching_spdy2;
    server_cb.ctx_ready = NULL;
    server_cb.ssl_ready = use_ALPN_spdy2_callback;
    server_cb.on_result = verify_ALPN_matching_spdy2;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* Callback: failure (one client protocol, spdy/2) */
    client_cb.ctx_ready = NULL;
    client_cb.ssl_ready = use_ALPN_one;
    client_cb.on_result = NULL;
    server_cb.ctx_ready = NULL;
    server_cb.ssl_ready = use_ALPN_failure_callback;
    server_cb.on_result = verify_ALPN_FATAL_ERROR_on_client;
    test_wolfSSL_client_server(&client_cb, &server_cb);

#endif /* OPENSSL_ALL | NGINX | HAPROXY | LIGHTY */

    res = TEST_RES_CHECK(1);
#endif /* !NO_WOLFSSL_CLIENT && !NO_WOLFSSL_SERVER */
    return res;
}

static int test_wolfSSL_UseALPN_params(void)
{
    EXPECT_DECLS;
#ifndef NO_WOLFSSL_CLIENT
    /* "http/1.1" */
    char http1[] = {0x68, 0x74, 0x74, 0x70, 0x2f, 0x31, 0x2e, 0x31};
    /* "spdy/1" */
    char spdy1[] = {0x73, 0x70, 0x64, 0x79, 0x2f, 0x31};
    /* "spdy/2" */
    char spdy2[] = {0x73, 0x70, 0x64, 0x79, 0x2f, 0x32};
    /* "spdy/3" */
    char spdy3[] = {0x73, 0x70, 0x64, 0x79, 0x2f, 0x33};
    char buff[256];
    word32 idx;

    WOLFSSL_CTX *ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
    WOLFSSL     *ssl = wolfSSL_new(ctx);

    ExpectNotNull(ctx);
    ExpectNotNull(ssl);

    /* error cases */
    ExpectIntNE(WOLFSSL_SUCCESS,
                wolfSSL_UseALPN(NULL, http1, sizeof(http1),
                                WOLFSSL_ALPN_FAILED_ON_MISMATCH));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_UseALPN(ssl, NULL, 0,
                                              WOLFSSL_ALPN_FAILED_ON_MISMATCH));

    /* success case */
    /* http1 only */
    ExpectIntEQ(WOLFSSL_SUCCESS,
                wolfSSL_UseALPN(ssl, http1, sizeof(http1),
                                WOLFSSL_ALPN_FAILED_ON_MISMATCH));

    /* http1, spdy1 */
    XMEMCPY(buff, http1, sizeof(http1));
    idx = sizeof(http1);
    buff[idx++] = ',';
    XMEMCPY(buff+idx, spdy1, sizeof(spdy1));
    idx += sizeof(spdy1);
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseALPN(ssl, buff, idx,
                                              WOLFSSL_ALPN_FAILED_ON_MISMATCH));

    /* http1, spdy2, spdy1 */
    XMEMCPY(buff, http1, sizeof(http1));
    idx = sizeof(http1);
    buff[idx++] = ',';
    XMEMCPY(buff+idx, spdy2, sizeof(spdy2));
    idx += sizeof(spdy2);
    buff[idx++] = ',';
    XMEMCPY(buff+idx, spdy1, sizeof(spdy1));
    idx += sizeof(spdy1);
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseALPN(ssl, buff, idx,
                                              WOLFSSL_ALPN_FAILED_ON_MISMATCH));

    /* spdy3, http1, spdy2, spdy1 */
    XMEMCPY(buff, spdy3, sizeof(spdy3));
    idx = sizeof(spdy3);
    buff[idx++] = ',';
    XMEMCPY(buff+idx, http1, sizeof(http1));
    idx += sizeof(http1);
    buff[idx++] = ',';
    XMEMCPY(buff+idx, spdy2, sizeof(spdy2));
    idx += sizeof(spdy2);
    buff[idx++] = ',';
    XMEMCPY(buff+idx, spdy1, sizeof(spdy1));
    idx += sizeof(spdy1);
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseALPN(ssl, buff, idx,
                                            WOLFSSL_ALPN_CONTINUE_ON_MISMATCH));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}
#endif /* HAVE_ALPN  */

#ifdef HAVE_ALPN_PROTOS_SUPPORT
static void CTX_set_alpn_protos(SSL_CTX *ctx)
{
    unsigned char p[] = {
       8, 'h', 't', 't', 'p', '/', '1', '.', '1',
       6, 's', 'p', 'd', 'y', '/', '2',
       6, 's', 'p', 'd', 'y', '/', '1',
    };

    unsigned char p_len = sizeof(p);
    int ret;

    ret = SSL_CTX_set_alpn_protos(ctx, p, p_len);

#ifdef WOLFSSL_ERROR_CODE_OPENSSL
    AssertIntEQ(ret, 0);
#else
    AssertIntEQ(ret, SSL_SUCCESS);
#endif
}

static void set_alpn_protos(SSL* ssl)
{
    unsigned char p[] = {
       6, 's', 'p', 'd', 'y', '/', '3',
       8, 'h', 't', 't', 'p', '/', '1', '.', '1',
       6, 's', 'p', 'd', 'y', '/', '2',
       6, 's', 'p', 'd', 'y', '/', '1',
    };

    unsigned char p_len = sizeof(p);
    int ret;

    ret = SSL_set_alpn_protos(ssl, p, p_len);

#ifdef WOLFSSL_ERROR_CODE_OPENSSL
    AssertIntEQ(ret, 0);
#else
    AssertIntEQ(ret, SSL_SUCCESS);
#endif

}

static void verify_alpn_matching_spdy3(WOLFSSL* ssl)
{
    /* "spdy/3" */
    char nego_proto[] = {0x73, 0x70, 0x64, 0x79, 0x2f, 0x33};
    const unsigned char *proto;
    unsigned int protoSz = 0;

    SSL_get0_alpn_selected(ssl, &proto, &protoSz);

    /* check value */
    AssertIntEQ(1, sizeof(nego_proto) == protoSz);
    AssertIntEQ(0, XMEMCMP(nego_proto, proto, protoSz));
}

static void verify_alpn_matching_http1(WOLFSSL* ssl)
{
    /* "http/1.1" */
    char nego_proto[] = {0x68, 0x74, 0x74, 0x70, 0x2f, 0x31, 0x2e, 0x31};
    const unsigned char *proto;
    unsigned int protoSz = 0;

    SSL_get0_alpn_selected(ssl, &proto, &protoSz);

    /* check value */
    AssertIntEQ(1, sizeof(nego_proto) == protoSz);
    AssertIntEQ(0, XMEMCMP(nego_proto, proto, protoSz));
}

static int test_wolfSSL_set_alpn_protos(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_WOLFSSL_CLIENT) && !defined(NO_WOLFSSL_SERVER)
    callback_functions client_cb;
    callback_functions server_cb;

    XMEMSET(&client_cb, 0, sizeof(callback_functions));
    XMEMSET(&server_cb, 0, sizeof(callback_functions));
    client_cb.method = wolfSSLv23_client_method;
    server_cb.method = wolfSSLv23_server_method;
    client_cb.devId = testDevId;
    server_cb.devId = testDevId;

    /* use CTX_alpn_protos */
    client_cb.ctx_ready = CTX_set_alpn_protos;
    client_cb.ssl_ready = NULL;
    client_cb.on_result = NULL;
    server_cb.ctx_ready = CTX_set_alpn_protos;
    server_cb.ssl_ready = NULL;
    server_cb.on_result = verify_alpn_matching_http1;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* use set_alpn_protos */
    client_cb.ctx_ready = NULL;
    client_cb.ssl_ready = set_alpn_protos;
    client_cb.on_result = NULL;
    server_cb.ctx_ready = NULL;
    server_cb.ssl_ready = set_alpn_protos;
    server_cb.on_result = verify_alpn_matching_spdy3;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    res = TEST_SUCCESS;
#endif /* !NO_WOLFSSL_CLIENT && !NO_WOLFSSL_SERVER */
    return res;
}

#endif /* HAVE_ALPN_PROTOS_SUPPORT */

static int test_wolfSSL_wolfSSL_UseSecureRenegotiation(void)
{
    EXPECT_DECLS;
#if defined(HAVE_SECURE_RENEGOTIATION) && !defined(NO_WOLFSSL_CLIENT) && \
    !defined(NO_TLS)
    WOLFSSL_CTX *ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
    WOLFSSL     *ssl = wolfSSL_new(ctx);

    ExpectNotNull(ctx);
    ExpectNotNull(ssl);

    /* error cases */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_UseSecureRenegotiation(NULL));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_UseSecureRenegotiation(NULL));

    /* success cases */
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_UseSecureRenegotiation(ctx));
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseSecureRenegotiation(ssl));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

/* Test reconnecting with a different ciphersuite after a renegotiation. */
static int test_wolfSSL_SCR_Reconnect(void)
{
    EXPECT_DECLS;
#if defined(HAVE_SECURE_RENEGOTIATION) && \
    defined(BUILD_TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384) && \
    defined(BUILD_TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256) && \
    defined(HAVE_MANUAL_MEMIO_TESTS_DEPENDENCIES)
    struct test_memio_ctx test_ctx;
    WOLFSSL_CTX *ctx_c = NULL, *ctx_s = NULL;
    WOLFSSL *ssl_c = NULL, *ssl_s = NULL;
    byte data;

    XMEMSET(&test_ctx, 0, sizeof(test_ctx));
    test_ctx.c_ciphers = "ECDHE-RSA-AES256-GCM-SHA384";
    test_ctx.s_ciphers =
        "ECDHE-RSA-AES256-GCM-SHA384:ECDHE-RSA-CHACHA20-POLY1305";
    ExpectIntEQ(test_memio_setup(&test_ctx, &ctx_c, &ctx_s, &ssl_c, &ssl_s,
        wolfTLSv1_2_client_method, wolfTLSv1_2_server_method), 0);
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_UseSecureRenegotiation(ctx_c));
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_UseSecureRenegotiation(ctx_s));
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseSecureRenegotiation(ssl_c));
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseSecureRenegotiation(ssl_s));
    ExpectIntEQ(test_memio_do_handshake(ssl_c, ssl_s, 10, NULL), 0);
    /* WOLFSSL_FATAL_ERROR since it will block */
    ExpectIntEQ(wolfSSL_Rehandshake(ssl_s), WC_NO_ERR_TRACE(WOLFSSL_FATAL_ERROR));
    ExpectIntEQ(wolfSSL_get_error(ssl_s, WC_NO_ERR_TRACE(WOLFSSL_FATAL_ERROR)),
                WOLFSSL_ERROR_WANT_READ);
    ExpectIntEQ(wolfSSL_read(ssl_c, &data, 1), WC_NO_ERR_TRACE(WOLFSSL_FATAL_ERROR));
    ExpectIntEQ(wolfSSL_get_error(ssl_s, WC_NO_ERR_TRACE(WOLFSSL_FATAL_ERROR)),
                WOLFSSL_ERROR_WANT_READ);
    ExpectIntEQ(test_memio_do_handshake(ssl_c, ssl_s, 10, NULL), 0);
    wolfSSL_free(ssl_c);
    ssl_c = NULL;
    wolfSSL_free(ssl_s);
    ssl_s = NULL;
    wolfSSL_CTX_free(ctx_c);
    ctx_c = NULL;
    test_ctx.c_ciphers = "ECDHE-RSA-CHACHA20-POLY1305";
    ExpectIntEQ(test_memio_setup(&test_ctx, &ctx_c, &ctx_s, &ssl_c, &ssl_s,
        wolfTLSv1_2_client_method, wolfTLSv1_2_server_method), 0);
    ExpectIntEQ(test_memio_do_handshake(ssl_c, ssl_s, 10, NULL), 0);
    wolfSSL_free(ssl_s);
    wolfSSL_free(ssl_c);
    wolfSSL_CTX_free(ctx_s);
    wolfSSL_CTX_free(ctx_c);
#endif
    return EXPECT_RESULT();
}

#if !defined(NO_WOLFSSL_SERVER) && !defined(NO_TLS) && \
    !defined(NO_FILESYSTEM) && (!defined(NO_RSA) || defined(HAVE_ECC))
/* Called when writing. */
static int DummySend(WOLFSSL* ssl, char* buf, int sz, void* ctx)
{
    (void)ssl;
    (void)buf;
    (void)sz;
    (void)ctx;

    /* Force error return from wolfSSL_accept_TLSv13(). */
    return WANT_WRITE;
}
/* Called when reading. */
static int BufferInfoRecv(WOLFSSL* ssl, char* buf, int sz, void* ctx)
{
    WOLFSSL_BUFFER_INFO* msg = (WOLFSSL_BUFFER_INFO*)ctx;
    int len = (int)msg->length;

    (void)ssl;
    (void)sz;

    /* Pass back as much of message as will fit in buffer. */
    if (len > sz)
        len = sz;
    XMEMCPY(buf, msg->buffer, len);
    /* Move over returned data. */
    msg->buffer += len;
    msg->length -= (word32)len;

    /* Amount actually copied. */
    return len;
}
#endif

/* Test the detection of duplicate known TLS extensions.
 * Specifically in a ClientHello.
 */
static int test_tls_ext_duplicate(void)
{
    EXPECT_DECLS;
#if !defined(NO_WOLFSSL_SERVER) && !defined(NO_TLS) && \
    !defined(NO_FILESYSTEM) && (!defined(NO_RSA) || defined(HAVE_ECC))
    const unsigned char clientHelloDupTlsExt[] = {
        0x16, 0x03, 0x03, 0x00, 0x6a, 0x01, 0x00, 0x00,
        0x66, 0x03, 0x03, 0xf4, 0x65, 0xbd, 0x22, 0xfe,
        0x6e, 0xab, 0x66, 0xdd, 0xcf, 0xe9, 0x65, 0x55,
        0xe8, 0xdf, 0xc3, 0x8e, 0x4b, 0x00, 0xbc, 0xf8,
        0x23, 0x57, 0x1b, 0xa0, 0xc8, 0xa9, 0xe2, 0x8c,
        0x91, 0x6e, 0xf9, 0x20, 0xf7, 0x5c, 0xc5, 0x5b,
        0x75, 0x8c, 0x47, 0x0a, 0x0e, 0xc4, 0x1a, 0xda,
        0xef, 0x75, 0xe5, 0x21, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x13, 0x01,
        0x00, 0x9e, 0x01, 0x00,
        /* Extensions - duplicate signature algorithms. */
                                0x00, 0x19, 0x00, 0x0d,
        0x00, 0x04, 0x00, 0x02, 0x04, 0x01, 0x00, 0x0d,
        0x00, 0x04, 0x00, 0x02, 0x04, 0x01,
        /* Supported Versions extension for TLS 1.3. */
                                            0x00, 0x2b,
        0x00, 0x05, 0x04, 0x03, 0x04, 0x03, 0x03
    };
    WOLFSSL_BUFFER_INFO msg;
    const char* testCertFile;
    const char* testKeyFile;
    WOLFSSL_CTX *ctx = NULL;
    WOLFSSL     *ssl = NULL;

#ifndef NO_RSA
    testCertFile = svrCertFile;
    testKeyFile = svrKeyFile;
#elif defined(HAVE_ECC)
    testCertFile = eccCertFile;
    testKeyFile = eccKeyFile;
#endif

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));

    ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, testCertFile,
        CERT_FILETYPE));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, testKeyFile,
        CERT_FILETYPE));

    /* Read from 'msg'. */
    wolfSSL_SetIORecv(ctx, BufferInfoRecv);
    /* No where to send to - dummy sender. */
    wolfSSL_SetIOSend(ctx, DummySend);

    ssl = wolfSSL_new(ctx);
    ExpectNotNull(ssl);

    msg.buffer = (unsigned char*)clientHelloDupTlsExt;
    msg.length = (unsigned int)sizeof(clientHelloDupTlsExt);
    wolfSSL_SetIOReadCtx(ssl, &msg);

    ExpectIntNE(wolfSSL_accept(ssl), WOLFSSL_SUCCESS);
    /* can return duplicate ext error or socket error if the peer closed down
     * while sending alert */
    if (wolfSSL_get_error(ssl, 0) != WC_NO_ERR_TRACE(SOCKET_ERROR_E)) {
        ExpectIntEQ(wolfSSL_get_error(ssl, 0), WC_NO_ERR_TRACE(DUPLICATE_TLS_EXT_E));
    }

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}


/* Test TLS connection abort when legacy version field indicates TLS 1.3 or
 * higher. Based on test_tls_ext_duplicate() but with legacy version modified
 * to 0x0304.
 */
static int test_tls_bad_legacy_version(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_TLS13) && !defined(WOLFSSL_ALLOW_BAD_TLS_LEGACY_VERSION)
#if !defined(NO_WOLFSSL_SERVER) && !defined(NO_TLS) && \
    !defined(NO_FILESYSTEM) && (!defined(NO_RSA) || defined(HAVE_ECC))
    /* This buffer (prior to Extensions) is exactly the same as the buffer in
     * test_tls_ext_duplicate() except the 11th byte is set to 0x04. That
     * change means the legacy protocol version field is invalid. That will be
     * caught before the dulplicate signature algorithms extension. */
    const unsigned char clientHelloBadLegacyVersion[] = {
        0x16, 0x03, 0x03, 0x00, 0x6a, 0x01, 0x00, 0x00,
        0x66, 0x03, 0x04, 0xf4, 0x65, 0xbd, 0x22, 0xfe,
        0x6e, 0xab, 0x66, 0xdd, 0xcf, 0xe9, 0x65, 0x55,
        0xe8, 0xdf, 0xc3, 0x8e, 0x4b, 0x00, 0xbc, 0xf8,
        0x23, 0x57, 0x1b, 0xa0, 0xc8, 0xa9, 0xe2, 0x8c,
        0x91, 0x6e, 0xf9, 0x20, 0xf7, 0x5c, 0xc5, 0x5b,
        0x75, 0x8c, 0x47, 0x0a, 0x0e, 0xc4, 0x1a, 0xda,
        0xef, 0x75, 0xe5, 0x21, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x13, 0x01,
        0x00, 0x9e, 0x01, 0x00,
        /* Extensions */
                                0x00, 0x19, 0x00, 0x0d,
        0x00, 0x04, 0x00, 0x02, 0x04, 0x01, 0x00, 0x15,
        0x00, 0x04, 0x00, 0x02, 0x04, 0x01,
        /* Supported Versions extension for TLS 1.3. */
                                            0x00, 0x2b,
        0x00, 0x05, 0x04, 0x03, 0x04, 0x03, 0x03
    };

    WOLFSSL_BUFFER_INFO msg;
    const char* testCertFile;
    const char* testKeyFile;
    WOLFSSL_CTX *ctx = NULL;
    WOLFSSL     *ssl = NULL;

#ifndef NO_RSA
    testCertFile = svrCertFile;
    testKeyFile = svrKeyFile;
#elif defined(HAVE_ECC)
    testCertFile = eccCertFile;
    testKeyFile = eccKeyFile;
#endif

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));

    ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, testCertFile,
        CERT_FILETYPE));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, testKeyFile,
        CERT_FILETYPE));

    /* Read from 'msg'. */
    wolfSSL_SetIORecv(ctx, BufferInfoRecv);
    /* No where to send to - dummy sender. */
    wolfSSL_SetIOSend(ctx, DummySend);

    ssl = wolfSSL_new(ctx);
    ExpectNotNull(ssl);

    msg.buffer = (unsigned char*)clientHelloBadLegacyVersion;
    msg.length = (unsigned int)sizeof(clientHelloBadLegacyVersion);
    wolfSSL_SetIOReadCtx(ssl, &msg);

    ExpectIntNE(wolfSSL_accept(ssl), WOLFSSL_SUCCESS);
    /* Connection should fail due to bad legacy version field. When that
     * happens the return code is VERSION_ERROR but that gets transformed into
     * SOCKET_ERROR_E. */
    ExpectIntEQ(wolfSSL_get_error(ssl, 0), WC_NO_ERR_TRACE(SOCKET_ERROR_E));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif
#endif
    return EXPECT_RESULT();
}
/*----------------------------------------------------------------------------*
 | X509 Tests
 *----------------------------------------------------------------------------*/
static int test_wolfSSL_X509_NAME_get_entry(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && !defined(NO_RSA) && !defined(NO_FILESYSTEM)
#if defined(OPENSSL_ALL) || \
        (defined(OPENSSL_EXTRA) && \
            (defined(KEEP_PEER_CERT) || defined(SESSION_CERTS)))
    /* use openssl like name to test mapping */
    X509_NAME_ENTRY* ne = NULL;
    X509_NAME* name = NULL;
    X509* x509 = NULL;
    ASN1_STRING* asn = NULL;
    char* subCN = NULL;
    int idx = 0;
    ASN1_OBJECT *object = NULL;
#if defined(WOLFSSL_APACHE_HTTPD) || defined(OPENSSL_ALL) || \
    defined(WOLFSSL_NGINX)
#ifndef NO_BIO
    BIO* bio = NULL;
#endif
#endif

    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(cliCertFile,
        WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = X509_get_subject_name(x509));
    ExpectIntGE(idx = X509_NAME_get_index_by_NID(name, NID_commonName, -1), 0);
    ExpectNotNull(ne = X509_NAME_get_entry(name, idx));
    ExpectNull(X509_NAME_ENTRY_get_data(NULL));
    ExpectNotNull(asn = X509_NAME_ENTRY_get_data(ne));
    ExpectNotNull(subCN = (char*)ASN1_STRING_data(asn));
    wolfSSL_FreeX509(x509);
    x509 = NULL;

    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(cliCertFile,
        WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = X509_get_subject_name(x509));
    ExpectIntGE(idx = X509_NAME_get_index_by_NID(name, NID_commonName, -1), 0);

#if defined(WOLFSSL_APACHE_HTTPD) || defined(OPENSSL_ALL) || \
    defined(WOLFSSL_NGINX)
#ifndef NO_BIO
    ExpectNotNull(bio = BIO_new(BIO_s_mem()));
    ExpectIntEQ(X509_NAME_print_ex(bio, name, 4,
                    (XN_FLAG_RFC2253 & ~XN_FLAG_DN_REV)), WOLFSSL_SUCCESS);
    ExpectIntEQ(X509_NAME_print_ex_fp(XBADFILE, name, 4,
                    (XN_FLAG_RFC2253 & ~XN_FLAG_DN_REV)), WOLFSSL_FAILURE);
    ExpectIntEQ(X509_NAME_print_ex_fp(stderr, name, 4,
                    (XN_FLAG_RFC2253 & ~XN_FLAG_DN_REV)), WOLFSSL_SUCCESS);
    BIO_free(bio);
#endif
#endif

    ExpectNotNull(ne = X509_NAME_get_entry(name, idx));
    ExpectNotNull(object = X509_NAME_ENTRY_get_object(ne));
    wolfSSL_FreeX509(x509);
#endif /* OPENSSL_ALL || (OPENSSL_EXTRA && (KEEP_PEER_CERT || SESSION_CERTS) */
#endif /* !NO_CERTS && !NO_RSA && !NO_FILESYSTEM */

    return EXPECT_RESULT();
}

/* Testing functions dealing with PKCS12 parsing out X509 certs */
static int test_wolfSSL_PKCS12(void)
{
    EXPECT_DECLS;
    /* .p12 file is encrypted with DES3 */
#ifndef HAVE_FIPS /* Password used in cert "wolfSSL test" is only 12-bytes
                   * (96-bit) FIPS mode requires Minimum of 14-byte (112-bit)
                   * Password Key
                   */
#if defined(OPENSSL_EXTRA) && !defined(NO_DES3) && !defined(NO_FILESYSTEM) && \
    !defined(NO_STDIO_FILESYSTEM) && !defined(NO_TLS) && \
    !defined(NO_ASN) && !defined(NO_PWDBASED) && !defined(NO_RSA) && \
    !defined(NO_SHA) && defined(HAVE_PKCS12) && !defined(NO_BIO) && \
    defined(WOLFSSL_AES_256)
    byte buf[6000];
    char file[] = "./certs/test-servercert.p12";
    char order[] = "./certs/ecc-rsa-server.p12";
#ifdef WC_RC2
    char rc2p12[] = "./certs/test-servercert-rc2.p12";
#endif
    char pass[] = "a password";
    const char goodPsw[] = "wolfSSL test";
    const char badPsw[] = "bad";
#ifdef HAVE_ECC
    WOLFSSL_X509_NAME *subject = NULL;
    WOLFSSL_X509      *x509 = NULL;
#endif
    XFILE f = XBADFILE;
    int  bytes = 0, ret = 0, goodPswLen = 0, badPswLen = 0;
    WOLFSSL_BIO      *bio = NULL;
    WOLFSSL_EVP_PKEY *pkey = NULL;
    WC_PKCS12        *pkcs12 = NULL;
    WC_PKCS12        *pkcs12_2 = NULL;
    WOLFSSL_X509     *cert = NULL;
    WOLFSSL_X509     *tmp = NULL;
    WOLF_STACK_OF(WOLFSSL_X509) *ca = NULL;
#if (defined(OPENSSL_ALL) || defined(WOLFSSL_ASIO) || defined(WOLFSSL_HAPROXY) \
    || defined(WOLFSSL_NGINX)) && defined(SESSION_CERTS)
    WOLFSSL_CTX      *ctx = NULL;
    WOLFSSL          *ssl = NULL;
    WOLF_STACK_OF(WOLFSSL_X509) *tmp_ca = NULL;
#endif

    ExpectTrue((f = XFOPEN(file, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buf, 1, sizeof(buf), f), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }

    goodPswLen = (int)XSTRLEN(goodPsw);
    badPswLen = (int)XSTRLEN(badPsw);

    ExpectNotNull(bio = wolfSSL_BIO_new(wolfSSL_BIO_s_mem()));

    ExpectIntEQ(BIO_write(bio, buf, bytes), bytes); /* d2i consumes BIO */
    ExpectNotNull(d2i_PKCS12_bio(bio, &pkcs12));
    ExpectNotNull(pkcs12);
    BIO_free(bio);
    bio = NULL;

    /* check verify MAC directly */
    ExpectIntEQ(ret = PKCS12_verify_mac(pkcs12, goodPsw, goodPswLen), 1);

    /* check verify MAC fail case directly */
    ExpectIntEQ(ret = PKCS12_verify_mac(pkcs12, badPsw, badPswLen), 0);

    /* check verify MAC fail case */
    ExpectIntEQ(ret = PKCS12_parse(pkcs12, "bad", &pkey, &cert, NULL), 0);
    ExpectNull(pkey);
    ExpectNull(cert);

    /* check parse with no extra certs kept */
    ExpectIntEQ(ret = PKCS12_parse(pkcs12, "wolfSSL test", &pkey, &cert, NULL),
        1);
    ExpectNotNull(pkey);
    ExpectNotNull(cert);

    wolfSSL_EVP_PKEY_free(pkey);
    pkey = NULL;
    wolfSSL_X509_free(cert);
    cert = NULL;

    /* check parse with extra certs kept */
    ExpectIntEQ(ret = PKCS12_parse(pkcs12, "wolfSSL test", &pkey, &cert, &ca),
        1);
    ExpectNotNull(pkey);
    ExpectNotNull(cert);
    ExpectNotNull(ca);

#if (defined(OPENSSL_ALL) || defined(WOLFSSL_ASIO) || defined(WOLFSSL_HAPROXY) \
    || defined(WOLFSSL_NGINX)) && defined(SESSION_CERTS)

    /* Check that SSL_CTX_set0_chain correctly sets the certChain buffer */
#if !defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)
#if !defined(NO_WOLFSSL_CLIENT) && defined(SESSION_CERTS)
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
#else
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#endif
    /* Copy stack structure */
    ExpectNotNull(tmp_ca = X509_chain_up_ref(ca));
    ExpectIntEQ(SSL_CTX_set0_chain(ctx, tmp_ca), 1);
    /* CTX now owns the tmp_ca stack structure */
    tmp_ca = NULL;
    ExpectIntEQ(wolfSSL_CTX_get_extra_chain_certs(ctx, &tmp_ca), 1);
    ExpectNotNull(tmp_ca);
    ExpectIntEQ(sk_X509_num(tmp_ca), sk_X509_num(ca));
    /* Check that the main cert is also set */
    ExpectNotNull(SSL_CTX_get0_certificate(ctx));
    ExpectNotNull(ssl = SSL_new(ctx));
    ExpectNotNull(SSL_get_certificate(ssl));
    SSL_free(ssl);
    SSL_CTX_free(ctx);
    ctx = NULL;
#endif
#endif /* !NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER */
    /* should be 2 other certs on stack */
    ExpectNotNull(tmp = sk_X509_pop(ca));
    X509_free(tmp);
    ExpectNotNull(tmp = sk_X509_pop(ca));
    X509_free(tmp);
    ExpectNull(sk_X509_pop(ca));

    EVP_PKEY_free(pkey);
    pkey = NULL;
    X509_free(cert);
    cert = NULL;
    sk_X509_pop_free(ca, X509_free);
    ca = NULL;

    /* check PKCS12_create */
    ExpectNull(PKCS12_create(pass, NULL, NULL, NULL, NULL, -1, -1, -1, -1,0));
    ExpectIntEQ(PKCS12_parse(pkcs12, "wolfSSL test", &pkey, &cert, &ca),
            SSL_SUCCESS);
    ExpectNotNull((pkcs12_2 = PKCS12_create(pass, NULL, pkey, cert, ca,
                    -1, -1, 100, -1, 0)));
    EVP_PKEY_free(pkey);
    pkey = NULL;
    X509_free(cert);
    cert = NULL;
    sk_X509_pop_free(ca, NULL);
    ca = NULL;

    ExpectIntEQ(PKCS12_parse(pkcs12_2, "a password", &pkey, &cert, &ca),
            SSL_SUCCESS);
    PKCS12_free(pkcs12_2);
    pkcs12_2 = NULL;
    ExpectNotNull((pkcs12_2 = PKCS12_create(pass, NULL, pkey, cert, ca,
             NID_pbe_WithSHA1And3_Key_TripleDES_CBC,
             NID_pbe_WithSHA1And3_Key_TripleDES_CBC,
             2000, 1, 0)));
    EVP_PKEY_free(pkey);
    pkey = NULL;
    X509_free(cert);
    cert = NULL;
    sk_X509_pop_free(ca, NULL);
    ca = NULL;

    /* convert to DER then back and parse */
    ExpectNotNull(bio = BIO_new(BIO_s_mem()));
    ExpectIntEQ(i2d_PKCS12_bio(bio, pkcs12_2), SSL_SUCCESS);
    PKCS12_free(pkcs12_2);
    pkcs12_2 = NULL;

    ExpectNotNull(pkcs12_2 = d2i_PKCS12_bio(bio, NULL));
    BIO_free(bio);
    bio = NULL;
    ExpectIntEQ(PKCS12_parse(pkcs12_2, "a password", &pkey, &cert, &ca),
            SSL_SUCCESS);

    /* should be 2 other certs on stack */
    ExpectNotNull(tmp = sk_X509_pop(ca));
    X509_free(tmp);
    ExpectNotNull(tmp = sk_X509_pop(ca));
    X509_free(tmp);
    ExpectNull(sk_X509_pop(ca));


#ifndef NO_RC4
    PKCS12_free(pkcs12_2);
    pkcs12_2 = NULL;
    ExpectNotNull((pkcs12_2 = PKCS12_create(pass, NULL, pkey, cert, NULL,
             NID_pbe_WithSHA1And128BitRC4,
             NID_pbe_WithSHA1And128BitRC4,
             2000, 1, 0)));
    EVP_PKEY_free(pkey);
    pkey = NULL;
    X509_free(cert);
    cert = NULL;
    sk_X509_pop_free(ca, NULL);
    ca = NULL;

    ExpectIntEQ(PKCS12_parse(pkcs12_2, "a password", &pkey, &cert, &ca),
            SSL_SUCCESS);

#endif /* NO_RC4 */

    EVP_PKEY_free(pkey);
    pkey = NULL;
    X509_free(cert);
    cert = NULL;
    PKCS12_free(pkcs12);
    pkcs12 = NULL;
    PKCS12_free(pkcs12_2);
    pkcs12_2 = NULL;
    sk_X509_pop_free(ca, NULL);
    ca = NULL;

#ifdef HAVE_ECC
    /* test order of parsing */
    ExpectTrue((f = XFOPEN(order, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buf, 1, sizeof(buf), f), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }

    ExpectNotNull(bio = BIO_new_mem_buf((void*)buf, bytes));
    ExpectNotNull(pkcs12 = d2i_PKCS12_bio(bio, NULL));
    ExpectIntEQ((ret = PKCS12_parse(pkcs12, "", &pkey, &cert, &ca)),
            WOLFSSL_SUCCESS);

    /* check use of pkey after parse */
#if (defined(OPENSSL_ALL) || defined(WOLFSSL_ASIO) || defined(WOLFSSL_HAPROXY) \
    || defined(WOLFSSL_NGINX)) && defined(SESSION_CERTS)
#if !defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)
#if !defined(NO_WOLFSSL_CLIENT) && defined(SESSION_CERTS)
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
#else
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#endif
    ExpectIntEQ(SSL_CTX_use_PrivateKey(ctx, pkey), WOLFSSL_SUCCESS);
    SSL_CTX_free(ctx);
#endif /* !NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER */
#endif

    ExpectNotNull(pkey);
    ExpectNotNull(cert);
    ExpectNotNull(ca);

    /* compare subject lines of certificates */
    ExpectNotNull(subject = wolfSSL_X509_get_subject_name(cert));
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(eccRsaCertFile,
                SSL_FILETYPE_PEM));
    ExpectIntEQ(wolfSSL_X509_NAME_cmp((const WOLFSSL_X509_NAME*)subject,
            (const WOLFSSL_X509_NAME*)wolfSSL_X509_get_subject_name(x509)), 0);
    X509_free(x509);
    x509 = NULL;

    /* test expected fail case */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(eccCertFile,
                SSL_FILETYPE_PEM));
    ExpectIntNE(wolfSSL_X509_NAME_cmp((const WOLFSSL_X509_NAME*)subject,
            (const WOLFSSL_X509_NAME*)wolfSSL_X509_get_subject_name(x509)), 0);
    X509_free(x509);
    x509 = NULL;
    X509_free(cert);
    cert = NULL;

    /* get subject line from ca stack */
    ExpectNotNull(cert = sk_X509_pop(ca));
    ExpectNotNull(subject = wolfSSL_X509_get_subject_name(cert));

    /* compare subject from certificate in ca to expected */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(eccCertFile,
                SSL_FILETYPE_PEM));
    ExpectIntEQ(wolfSSL_X509_NAME_cmp((const WOLFSSL_X509_NAME*)subject,
            (const WOLFSSL_X509_NAME*)wolfSSL_X509_get_subject_name(x509)), 0);

    /* modify case and compare subject from certificate in ca to expected.
     * The first bit of the name is:
     * /C=US/ST=Washington
     * So we'll change subject->name[1] to 'c' (lower case) */
    if (subject != NULL) {
        subject->name[1] = 'c';
        ExpectIntEQ(wolfSSL_X509_NAME_cmp((const WOLFSSL_X509_NAME*)subject,
            (const WOLFSSL_X509_NAME*)wolfSSL_X509_get_subject_name(x509)), 0);
    }

    EVP_PKEY_free(pkey);
    pkey = NULL;
    X509_free(x509);
    x509 = NULL;
    X509_free(cert);
    cert = NULL;
    BIO_free(bio);
    bio = NULL;
    PKCS12_free(pkcs12);
    pkcs12 = NULL;
    sk_X509_pop_free(ca, NULL); /* TEST d2i_PKCS12_fp */
    ca = NULL;

    /* test order of parsing */
    ExpectTrue((f = XFOPEN(file, "rb")) != XBADFILE);
    ExpectNotNull(pkcs12 = d2i_PKCS12_fp(f, NULL));
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }

    /* check verify MAC fail case */
    ExpectIntEQ(ret = PKCS12_parse(pkcs12, "bad", &pkey, &cert, NULL), 0);
    ExpectNull(pkey);
    ExpectNull(cert);

    /* check parse with no extra certs kept */
    ExpectIntEQ(ret = PKCS12_parse(pkcs12, "wolfSSL test", &pkey, &cert, NULL),
        1);
    ExpectNotNull(pkey);
    ExpectNotNull(cert);

    wolfSSL_EVP_PKEY_free(pkey);
    pkey = NULL;
    wolfSSL_X509_free(cert);
    cert = NULL;

    /* check parse with extra certs kept */
    ExpectIntEQ(ret = PKCS12_parse(pkcs12, "wolfSSL test", &pkey, &cert, &ca),
        1);
    ExpectNotNull(pkey);
    ExpectNotNull(cert);
    ExpectNotNull(ca);

    wolfSSL_EVP_PKEY_free(pkey);
    pkey = NULL;
    wolfSSL_X509_free(cert);
    cert = NULL;
    sk_X509_pop_free(ca, NULL);
    ca = NULL;

    PKCS12_free(pkcs12);
    pkcs12 = NULL;
#endif /* HAVE_ECC */

#ifdef WC_RC2
    /* test PKCS#12 with RC2 encryption */
    ExpectTrue((f = XFOPEN(rc2p12, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buf, 1, sizeof(buf), f), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }

    ExpectNotNull(bio = BIO_new_mem_buf((void*)buf, bytes));
    ExpectNotNull(pkcs12 = d2i_PKCS12_bio(bio, NULL));

    /* check verify MAC fail case */
    ExpectIntEQ(ret = PKCS12_parse(pkcs12, "bad", &pkey, &cert, NULL), 0);
    ExpectNull(pkey);
    ExpectNull(cert);

    /* check parse with not extra certs kept */
    ExpectIntEQ(ret = PKCS12_parse(pkcs12, "wolfSSL test", &pkey, &cert, NULL),
        WOLFSSL_SUCCESS);
    ExpectNotNull(pkey);
    ExpectNotNull(cert);

    wolfSSL_EVP_PKEY_free(pkey);
    pkey = NULL;
    wolfSSL_X509_free(cert);
    cert = NULL;

    /* check parse with extra certs kept */
    ExpectIntEQ(ret = PKCS12_parse(pkcs12, "wolfSSL test", &pkey, &cert, &ca),
        WOLFSSL_SUCCESS);
    ExpectNotNull(pkey);
    ExpectNotNull(cert);
    ExpectNotNull(ca);

    wolfSSL_EVP_PKEY_free(pkey);
    wolfSSL_X509_free(cert);
    sk_X509_pop_free(ca, NULL);

    BIO_free(bio);
    bio = NULL;
    PKCS12_free(pkcs12);
    pkcs12 = NULL;
#endif /* WC_RC2 */

    /* Test i2d_PKCS12_bio */
    ExpectTrue((f = XFOPEN(file, "rb")) != XBADFILE);
    ExpectNotNull(pkcs12 = d2i_PKCS12_fp(f, NULL));
    if (f != XBADFILE)
        XFCLOSE(f);

    ExpectNotNull(bio = BIO_new(BIO_s_mem()));

    ExpectIntEQ(ret = i2d_PKCS12_bio(bio, pkcs12), 1);

    ExpectIntEQ(ret = i2d_PKCS12_bio(NULL, pkcs12), 0);

    ExpectIntEQ(ret = i2d_PKCS12_bio(bio, NULL), 0);

    PKCS12_free(pkcs12);
    BIO_free(bio);

    (void)order;
#endif /* OPENSSL_EXTRA */
#endif /* HAVE_FIPS */
    return EXPECT_RESULT();
}


#if !defined(NO_FILESYSTEM) && !defined(NO_ASN) && defined(HAVE_PKCS8) && \
    defined(WOLFSSL_ENCRYPTED_KEYS) && !defined(NO_DES3) && !defined(NO_PWDBASED) && \
    (!defined(NO_RSA) || defined(HAVE_ECC)) && !defined(NO_MD5)
    #define TEST_PKCS8_ENC
#endif

#if !defined(NO_FILESYSTEM) && !defined(NO_ASN) && defined(HAVE_PKCS8) && \
    defined(HAVE_ECC) && defined(WOLFSSL_ENCRYPTED_KEYS) && !defined(NO_TLS)

/* used to keep track if FailTestCallback was called */
static int failTestCallbackCalled = 0;

static WC_INLINE int FailTestCallBack(char* passwd, int sz, int rw, void* userdata)
{
    (void)passwd;
    (void)sz;
    (void)rw;
    (void)userdata;

    /* mark called, test_wolfSSL_no_password_cb() will check and fail if set */
    failTestCallbackCalled = 1;

    return -1;
}
#endif

static int test_wolfSSL_no_password_cb(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_ASN) && defined(HAVE_PKCS8) && \
    defined(HAVE_ECC) && defined(WOLFSSL_ENCRYPTED_KEYS) && !defined(NO_TLS)
    WOLFSSL_CTX* ctx = NULL;
    byte buff[FOURK_BUF];
    const char eccPkcs8PrivKeyDerFile[] = "./certs/ecc-privkeyPkcs8.der";
    const char eccPkcs8PrivKeyPemFile[] = "./certs/ecc-privkeyPkcs8.pem";
    XFILE f = XBADFILE;
    int bytes = 0;

#ifndef NO_WOLFSSL_CLIENT
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLS_client_method()));
#else
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLS_server_method()));
#endif
    wolfSSL_CTX_set_default_passwd_cb(ctx, FailTestCallBack);

    ExpectTrue((f = XFOPEN(eccPkcs8PrivKeyDerFile, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buff, 1, sizeof(buff), f), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }
    ExpectIntLE(bytes, sizeof(buff));
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    ExpectTrue((f = XFOPEN(eccPkcs8PrivKeyPemFile, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buff, 1, sizeof(buff), f), 0);
    if (f != XBADFILE)
        XFCLOSE(f);
    ExpectIntLE(bytes, sizeof(buff));
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);

    wolfSSL_CTX_free(ctx);

    /* Password callback should not be called by default */
    ExpectIntEQ(failTestCallbackCalled, 0);
#endif
    return EXPECT_RESULT();
}

#if defined(TEST_PKCS8_ENC) && !defined(NO_TLS)
/* for PKCS8 test case */
static int PKCS8TestCallBack(char* passwd, int sz, int rw, void* userdata)
{
    int flag = 0;

    (void)rw;
    if (userdata != NULL) {
        flag = *((int*)userdata); /* user set data */
    }

    switch (flag) {
        case 1: /* flag set for specific WOLFSSL_CTX structure, note userdata
                 * can be anything the user wishes to be passed to the callback
                 * associated with the WOLFSSL_CTX */
            XSTRNCPY(passwd, "yassl123", sz);
            return 8;

        default:
            return BAD_FUNC_ARG;
    }
}
#endif /* TEST_PKCS8_ENC && !NO_TLS */

/* Testing functions dealing with PKCS8 */
static int test_wolfSSL_PKCS8(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_ASN) && defined(HAVE_PKCS8) && \
    !defined(WOLFCRYPT_ONLY) && !defined(NO_TLS) && \
    (!defined(WOLFSSL_NO_TLS12) || defined(WOLFSSL_TLS13))
#if !defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)
    byte buff[FOURK_BUF];
    byte der[FOURK_BUF];
    #ifndef NO_RSA
#ifdef WOLFSSL_PEM_TO_DER
        const char serverKeyPkcs8PemFile[] = "./certs/server-keyPkcs8.pem";
#endif
        const char serverKeyPkcs8DerFile[] = "./certs/server-keyPkcs8.der";
    #endif
#ifdef WOLFSSL_PEM_TO_DER
    const char eccPkcs8PrivKeyPemFile[] = "./certs/ecc-privkeyPkcs8.pem";
#endif
    #ifdef HAVE_ECC
        const char eccPkcs8PrivKeyDerFile[] = "./certs/ecc-privkeyPkcs8.der";
    #endif
    XFILE f = XBADFILE;
    int bytes = 0;
    WOLFSSL_CTX* ctx = NULL;
#if defined(HAVE_ECC) && !defined(NO_CODING) && !defined(WOLFSSL_NO_PEM)
    int ret;
    ecc_key key;
    word32 x = 0;
#endif
#ifdef TEST_PKCS8_ENC
    #if !defined(NO_RSA) && !defined(NO_SHA)
        const char serverKeyPkcs8EncPemFile[] = "./certs/server-keyPkcs8Enc.pem";
        const char serverKeyPkcs8EncDerFile[] = "./certs/server-keyPkcs8Enc.der";
    #endif
    #if defined(HAVE_ECC) && !defined(NO_SHA)
        const char eccPkcs8EncPrivKeyPemFile[] = "./certs/ecc-keyPkcs8Enc.pem";
        const char eccPkcs8EncPrivKeyDerFile[] = "./certs/ecc-keyPkcs8Enc.der";
    #endif
    int flag;
#endif

    (void)der;

#ifndef NO_WOLFSSL_CLIENT
    #ifndef WOLFSSL_NO_TLS12
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_2_client_method()));
    #else
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_3_client_method()));
    #endif
#else
    #ifndef WOLFSSL_NO_TLS12
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_2_server_method()));
    #else
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_3_server_method()));
    #endif
#endif

#ifdef TEST_PKCS8_ENC
    wolfSSL_CTX_set_default_passwd_cb(ctx, PKCS8TestCallBack);
    wolfSSL_CTX_set_default_passwd_cb_userdata(ctx, (void*)&flag);
    flag = 1; /* used by password callback as return code */

    #if !defined(NO_RSA) && !defined(NO_SHA)
    #if defined(WOLFSSL_PEM_TO_DER)
    /* test loading PEM PKCS8 encrypted file */
    ExpectTrue((f = XFOPEN(serverKeyPkcs8EncPemFile, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buff, 1, sizeof(buff), f), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);

    /* this next case should fail because of password callback return code */
    flag = 0; /* used by password callback as return code */
    ExpectIntNE(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);

    /* decrypt PKCS8 PEM to key in DER format with not using WOLFSSL_CTX */
    ExpectIntGT(wc_KeyPemToDer(buff, bytes, der, (word32)sizeof(der),
        "yassl123"), 0);

    /* test that error value is returned with a bad password */
    ExpectIntLT(wc_KeyPemToDer(buff, bytes, der, (word32)sizeof(der),
        "bad"), 0);
    #endif

    /* test loading PEM PKCS8 encrypted file */
    ExpectTrue((f = XFOPEN(serverKeyPkcs8EncDerFile, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buff, 1, sizeof(buff), f), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }
    flag = 1; /* used by password callback as return code */
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* this next case should fail because of password callback return code */
    flag = 0; /* used by password callback as return code */
    ExpectIntNE(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    #endif /* !NO_RSA && !NO_SHA */

    #if defined(HAVE_ECC) && !defined(NO_SHA)
    #if defined(WOLFSSL_PEM_TO_DER)
    /* test loading PEM PKCS8 encrypted ECC Key file */
    ExpectTrue((f = XFOPEN(eccPkcs8EncPrivKeyPemFile, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buff, 1, sizeof(buff), f), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }
    flag = 1; /* used by password callback as return code */
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);

    /* this next case should fail because of password callback return code */
    flag = 0; /* used by password callback as return code */
    ExpectIntNE(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);

    /* decrypt PKCS8 PEM to key in DER format with not using WOLFSSL_CTX */
    ExpectIntGT(wc_KeyPemToDer(buff, bytes, der, (word32)sizeof(der),
        "yassl123"), 0);

    /* test that error value is returned with a bad password */
    ExpectIntLT(wc_KeyPemToDer(buff, bytes, der, (word32)sizeof(der),
        "bad"), 0);
    #endif

    /* test loading DER PKCS8 encrypted ECC Key file */
    ExpectTrue((f = XFOPEN(eccPkcs8EncPrivKeyDerFile, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buff, 1, sizeof(buff), f), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }
    flag = 1; /* used by password callback as return code */
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* this next case should fail because of password callback return code */
    flag = 0; /* used by password callback as return code */
    ExpectIntNE(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* leave flag as "okay" */
    flag = 1;
    #endif /* HAVE_ECC && !NO_SHA */
#endif /* TEST_PKCS8_ENC */


#ifndef NO_RSA
    /* test loading ASN.1 (DER) PKCS8 private key file (not encrypted) */
    ExpectTrue((f = XFOPEN(serverKeyPkcs8DerFile, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buff, 1, sizeof(buff), f), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    #ifdef WOLFSSL_PEM_TO_DER
    /* test loading PEM PKCS8 private key file (not encrypted) */
    ExpectTrue((f = XFOPEN(serverKeyPkcs8PemFile, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buff, 1, sizeof(buff), f), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);
    #endif
#endif /* !NO_RSA */

#ifdef WOLFSSL_PEM_TO_DER
    /* Test PKCS8 PEM ECC key no crypt */
    ExpectTrue((f = XFOPEN(eccPkcs8PrivKeyPemFile, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buff, 1, sizeof(buff), f), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }
#endif
#ifdef HAVE_ECC
#ifdef WOLFSSL_PEM_TO_DER
    /* Test PKCS8 PEM ECC key no crypt */
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);

    /* decrypt PKCS8 PEM to key in DER format */
    ExpectIntGT((bytes = wc_KeyPemToDer(buff, bytes, der,
        (word32)sizeof(der), NULL)), 0);
    ret = wc_ecc_init(&key);
    if (ret == 0) {
        ret = wc_EccPrivateKeyDecode(der, &x, &key, (word32)bytes);
        wc_ecc_free(&key);
    }
    ExpectIntEQ(ret, 0);
#endif

    /* Test PKCS8 DER ECC key no crypt */
    ExpectTrue((f = XFOPEN(eccPkcs8PrivKeyDerFile, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buff, 1, sizeof(buff), f), 0);
    if (f != XBADFILE)
        XFCLOSE(f);

    /* Test using a PKCS8 ECC PEM */
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
#else
#ifdef WOLFSSL_PEM_TO_DER
    /* if HAVE_ECC is not defined then BEGIN EC PRIVATE KEY is not found */
    ExpectIntEQ((bytes = wc_KeyPemToDer(buff, bytes, der,
        (word32)sizeof(der), NULL)), WC_NO_ERR_TRACE(ASN_NO_PEM_HEADER));
#endif
#endif /* HAVE_ECC */

    wolfSSL_CTX_free(ctx);
#endif /* !NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER */
#endif /* !NO_FILESYSTEM && !NO_ASN && HAVE_PKCS8 */
    return EXPECT_RESULT();
}

static int test_wolfSSL_PKCS8_ED25519(void)
{
    EXPECT_DECLS;
#if !defined(NO_ASN) && defined(HAVE_PKCS8) && defined(HAVE_AES_CBC) && \
    defined(WOLFSSL_AES_256) && \
    defined(WOLFSSL_ENCRYPTED_KEYS) && defined(HAVE_ED25519) && \
    defined(HAVE_ED25519_KEY_IMPORT)
    const byte encPrivKey[] = \
    "-----BEGIN ENCRYPTED PRIVATE KEY-----\n"
    "MIGbMFcGCSqGSIb3DQEFDTBKMCkGCSqGSIb3DQEFDDAcBAheCGLmWGh7+AICCAAw\n"
    "DAYIKoZIhvcNAgkFADAdBglghkgBZQMEASoEEC4L5P6GappsTyhOOoQfvh8EQJMX\n"
    "OAdlsYKCOcFo4djg6AI1lRdeBRwVFWkha7gBdoCJOzS8wDvTbYcJMPvANu5ft3nl\n"
    "2L9W4v7swXkV+X+a1ww=\n"
    "-----END ENCRYPTED PRIVATE KEY-----\n";
    const char password[] = "abcdefghijklmnopqrstuvwxyz";
    byte der[FOURK_BUF];
#if !defined(NO_TLS) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
    WOLFSSL_CTX* ctx = NULL;
#endif
    int bytes;

    XMEMSET(der, 0, sizeof(der));
    ExpectIntGT((bytes = wc_KeyPemToDer(encPrivKey, sizeof(encPrivKey), der,
        (word32)sizeof(der), password)), 0);
#if !defined(NO_TLS) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#else
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
#endif
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, der, bytes,
        WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    wolfSSL_CTX_free(ctx);
#endif /* !NO_TLS && (!NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER) */
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_PKCS8_ED448(void)
{
    EXPECT_DECLS;
#if !defined(NO_ASN) && defined(HAVE_PKCS8) && defined(HAVE_AES_CBC) && \
    defined(WOLFSSL_AES_256) && \
    defined(WOLFSSL_ENCRYPTED_KEYS) && defined(HAVE_ED448) && \
    defined(HAVE_ED448_KEY_IMPORT)
    const byte encPrivKey[] = \
    "-----BEGIN ENCRYPTED PRIVATE KEY-----\n"
    "MIGrMFcGCSqGSIb3DQEFDTBKMCkGCSqGSIb3DQEFDDAcBAjSbZKnG4EPggICCAAw\n"
    "DAYIKoZIhvcNAgkFADAdBglghkgBZQMEASoEEFvCFWBBHBlJBsYleBJlJWcEUNC7\n"
    "Tf5pZviT5Btar4D/MNg6BsQHSDf5KW4ix871EsgDY2Zz+euaoWspiMntz7gU+PQu\n"
    "T/JJcbD2Ly8BbE3l5WHMifAQqNLxJBfXrHkfYtAo\n"
    "-----END ENCRYPTED PRIVATE KEY-----\n";
    const char password[] = "abcdefghijklmnopqrstuvwxyz";
    byte der[FOURK_BUF];
#if !defined(NO_TLS) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
    WOLFSSL_CTX* ctx = NULL;
#endif
    int bytes;

    XMEMSET(der, 0, sizeof(der));
    ExpectIntGT((bytes = wc_KeyPemToDer(encPrivKey, sizeof(encPrivKey), der,
        (word32)sizeof(der), password)), 0);
#if !defined(NO_TLS) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#else
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
#endif
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, der, bytes,
        WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    wolfSSL_CTX_free(ctx);
#endif /* !NO_TLS && (!NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER) */
#endif
    return EXPECT_RESULT();
}

/* Testing functions dealing with PKCS5 */
static int test_wolfSSL_PKCS5(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_SHA) && !defined(NO_PWDBASED)
#ifdef HAVE_FIPS /* Password minimum length is 14 (112-bit) in FIPS MODE */
    const char* passwd = "myfipsPa$$W0rd";
#else
    const char *passwd = "pass1234";
#endif
    const unsigned char *salt = (unsigned char *)"salt1234";
    unsigned char *out = (unsigned char *)XMALLOC(WC_SHA_DIGEST_SIZE, NULL,
                                                  DYNAMIC_TYPE_TMP_BUFFER);
    int ret = 0;

    ExpectNotNull(out);
    ExpectIntEQ(ret = PKCS5_PBKDF2_HMAC_SHA1(passwd,(int)XSTRLEN(passwd), salt,
        (int)XSTRLEN((const char *) salt), 10, WC_SHA_DIGEST_SIZE,out),
        WOLFSSL_SUCCESS);

#ifdef WOLFSSL_SHA512
    ExpectIntEQ(ret = PKCS5_PBKDF2_HMAC(passwd,(int)XSTRLEN(passwd), salt,
        (int)XSTRLEN((const char *) salt), 10, wolfSSL_EVP_sha512(),
        WC_SHA_DIGEST_SIZE, out), SSL_SUCCESS);
#endif

    XFREE(out, NULL, DYNAMIC_TYPE_TMP_BUFFER);
#endif /* defined(OPENSSL_EXTRA) && !defined(NO_SHA) */
    return EXPECT_RESULT();
}

/* test parsing URI from certificate */
static int test_wolfSSL_URI(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && !defined(NO_RSA) && !defined(NO_FILESYSTEM) \
    && (defined(KEEP_PEER_CERT) || defined(SESSION_CERTS) || \
    defined(OPENSSL_EXTRA))
    WOLFSSL_X509* x509 = NULL;
    const char uri[] = "./certs/client-uri-cert.pem";
    const char urn[] = "./certs/client-absolute-urn.pem";
    const char badUri[] = "./certs/client-relative-uri.pem";

    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(uri,
        WOLFSSL_FILETYPE_PEM));
    wolfSSL_FreeX509(x509);
    x509 = NULL;

    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(urn,
        WOLFSSL_FILETYPE_PEM));
    wolfSSL_FreeX509(x509);
    x509 = NULL;

#if !defined(IGNORE_NAME_CONSTRAINTS) && !defined(WOLFSSL_NO_ASN_STRICT) \
    && !defined(WOLFSSL_FPKI)
    ExpectNull(x509 = wolfSSL_X509_load_certificate_file(badUri,
        WOLFSSL_FILETYPE_PEM));
#else
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(badUri,
        WOLFSSL_FILETYPE_PEM));
#endif
    wolfSSL_FreeX509(x509);
#endif
    return EXPECT_RESULT();
}


static int test_wolfSSL_TBS(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && !defined(NO_RSA) && !defined(NO_FILESYSTEM) \
    && defined(OPENSSL_EXTRA)
    WOLFSSL_X509* x509 = NULL;
    const unsigned char* tbs;
    int tbsSz;

    ExpectNotNull(x509 = wolfSSL_X509_new());
    ExpectNull(tbs = wolfSSL_X509_get_tbs(x509, &tbsSz));
    wolfSSL_X509_free(x509);
    x509 = NULL;

    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(caCertFile,
        WOLFSSL_FILETYPE_PEM));

    ExpectNull(tbs = wolfSSL_X509_get_tbs(NULL, &tbsSz));
    ExpectNull(tbs = wolfSSL_X509_get_tbs(x509, NULL));
    ExpectNotNull(tbs = wolfSSL_X509_get_tbs(x509, &tbsSz));
    ExpectIntEQ(tbsSz, 1003);

    wolfSSL_FreeX509(x509);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_verify(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && !defined(NO_RSA) && !defined(NO_FILESYSTEM) && \
    defined(OPENSSL_EXTRA)
    WOLFSSL_X509* ca = NULL;
    WOLFSSL_X509* serv = NULL;
    WOLFSSL_EVP_PKEY* pkey = NULL;
    unsigned char buf[2048];
    const unsigned char* pt = NULL;
    int bufSz = 0;

    ExpectNotNull(ca = wolfSSL_X509_load_certificate_file(caCertFile,
        WOLFSSL_FILETYPE_PEM));

    ExpectIntNE(wolfSSL_X509_get_pubkey_buffer(NULL, buf, NULL),
        WOLFSSL_SUCCESS);
    ExpectIntNE(wolfSSL_X509_get_pubkey_buffer(NULL, buf, &bufSz),
        WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_get_pubkey_buffer(ca, NULL, &bufSz),
        WOLFSSL_SUCCESS);
    ExpectIntEQ(bufSz, 294);

    bufSz--;
    ExpectIntNE(wolfSSL_X509_get_pubkey_buffer(ca, buf, &bufSz),
        WOLFSSL_SUCCESS);
    bufSz = 2048;
    ExpectIntEQ(wolfSSL_X509_get_pubkey_buffer(ca, buf, &bufSz),
        WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_get_pubkey_type(NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(wolfSSL_X509_get_pubkey_type(ca), RSAk);


    ExpectNotNull(serv = wolfSSL_X509_load_certificate_file(svrCertFile,
        WOLFSSL_FILETYPE_PEM));

    /* success case */
    pt = buf;
    ExpectNotNull(pkey = wolfSSL_d2i_PUBKEY(NULL, &pt, bufSz));

    ExpectIntEQ(i2d_PUBKEY(pkey, NULL), bufSz);

    ExpectIntEQ(wolfSSL_X509_verify(serv, pkey), WOLFSSL_SUCCESS);
    wolfSSL_EVP_PKEY_free(pkey);
    pkey = NULL;

    /* fail case */
    bufSz = 2048;
    ExpectIntEQ(wolfSSL_X509_get_pubkey_buffer(serv, buf, &bufSz),
        WOLFSSL_SUCCESS);
    pt = buf;
    ExpectNotNull(pkey = wolfSSL_d2i_PUBKEY(NULL, &pt, bufSz));
    ExpectIntEQ(wolfSSL_X509_verify(serv, pkey), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

    ExpectIntEQ(wolfSSL_X509_verify(NULL, pkey), WC_NO_ERR_TRACE(WOLFSSL_FATAL_ERROR));
    ExpectIntEQ(wolfSSL_X509_verify(serv, NULL), WC_NO_ERR_TRACE(WOLFSSL_FATAL_ERROR));

#ifndef NO_WOLFSSL_STUB
    ExpectNull(wolfSSL_X509_get0_pubkey_bitstr(NULL));
    ExpectNull(wolfSSL_X509_get0_pubkey_bitstr(serv));
#endif

    wolfSSL_EVP_PKEY_free(pkey);

    wolfSSL_FreeX509(ca);
    wolfSSL_FreeX509(serv);
#endif
    return EXPECT_RESULT();
}

#if defined(WOLFSSL_ACERT) && !defined(NO_CERTS) && !defined(NO_RSA) && \
    defined(WC_RSA_PSS) && !defined(NO_FILESYSTEM) && defined(OPENSSL_EXTRA)
/* Given acert file and its pubkey file, read them and then
 * attempt to verify signed acert.
 *
 * If expect_pass is true, then verification should pass.
 * If expect_pass is false, then verification should fail.
 * */
static int do_acert_verify_test(const char * acert_file,
                                const char * pkey_file,
                                size_t       expect_pass)
{
    X509_ACERT * x509 = NULL;
    EVP_PKEY *   pkey = NULL;
    BIO *        bp = NULL;
    int          verify_rc = 0;

    /* First read the attribute certificate. */
    bp = BIO_new_file(acert_file, "r");
    if (bp == NULL) {
        return -1;
    }

    x509 = PEM_read_bio_X509_ACERT(bp, NULL, NULL, NULL);
    BIO_free(bp);
    bp = NULL;

    if (x509 == NULL) {
        return -1;
    }

    /* Next read the associated pub key. */
    bp = BIO_new_file(pkey_file, "r");

    if (bp == NULL) {
        X509_ACERT_free(x509);
        x509 = NULL;
        return -1;
    }

    pkey = PEM_read_bio_PUBKEY(bp, &pkey, NULL, NULL);
    BIO_free(bp);
    bp = NULL;

    if (pkey == NULL) {
        X509_ACERT_free(x509);
        x509 = NULL;
        return -1;
    }

    /* Finally, do verification. */
    verify_rc = X509_ACERT_verify(x509, pkey);

    X509_ACERT_free(x509);
    x509 = NULL;

    EVP_PKEY_free(pkey);
    pkey = NULL;

    if (expect_pass && verify_rc != 1) {
        return -1;
    }

    if (!expect_pass && verify_rc == 1) {
        return -1;
    }

    return 0;
}
#endif

static int test_wolfSSL_X509_ACERT_verify(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_ACERT) && !defined(NO_CERTS) && !defined(NO_RSA) && \
    defined(WC_RSA_PSS) && !defined(NO_FILESYSTEM) && defined(OPENSSL_EXTRA)
    /* Walk over list of signed ACERTs and their pubkeys.
     * All should load and pass verification. */
    const char * acerts[4] = {"certs/acert/acert.pem",
                              "certs/acert/acert_ietf.pem",
                              "certs/acert/rsa_pss/acert.pem",
                              "certs/acert/rsa_pss/acert_ietf.pem"};
    const char * pkeys[4] =  {"certs/acert/acert_pubkey.pem",
                              "certs/acert/acert_ietf_pubkey.pem",
                              "certs/acert/rsa_pss/acert_pubkey.pem",
                              "certs/acert/rsa_pss/acert_ietf_pubkey.pem"};
    int    rc = 0;
    size_t i = 0;
    size_t j = 0;

    for (i = 0; i < 4; ++i) {
        for (j = i; j < 4; ++j) {
            rc = do_acert_verify_test(acerts[i], pkeys[j], i == j);

            if (rc) {
                fprintf(stderr, "error: %s: i = %zu, j = %zu, rc = %d\n",
                        "do_acert_verify_test", i, j, rc);
                break;
            }
        }

        if (rc) { break; }
    }

    ExpectIntEQ(rc, 0);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_ACERT_misc_api(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_ACERT) && !defined(NO_CERTS) && !defined(NO_RSA) && \
    !defined(NO_FILESYSTEM) && defined(OPENSSL_EXTRA)
    const char * acerts[4] = {"certs/acert/acert.pem",
                              "certs/acert/acert_ietf.pem",
                              "certs/acert/rsa_pss/acert.pem",
                              "certs/acert/rsa_pss/acert_ietf.pem"};
    int          rc = 0;
    X509_ACERT * x509 = NULL;
    BIO *        bp = NULL;
    long         ver_long = 0;
    int          ver = 0;
    int          nid = 0;
    const byte * raw_attr = NULL;
    word32       attr_len = 0;
    size_t       i = 0;
    int          buf_len = 0;
    byte         ietf_serial[] = {0x03, 0xb5, 0x90, 0x59, 0x02,
                                  0xa2, 0xaa, 0xb5, 0x40, 0x21,
                                  0x44, 0xb8, 0x2c, 0x4f, 0xd9,
                                  0x80, 0x1b, 0x5f, 0x57, 0xc2};

    for (i = 0; i < 4; ++i) {
        const char * acert_file = acerts[i];
        int          is_rsa_pss = 0;
        int          is_ietf_acert = 0;
        byte         serial[64];
        int          serial_len = sizeof(serial);

        XMEMSET(serial, 0, sizeof(serial));

        is_rsa_pss = XSTRSTR(acert_file, "rsa_pss") != NULL ? 1 : 0;
        is_ietf_acert = XSTRSTR(acert_file, "ietf.pem") != NULL ? 1 : 0;

        /* First read the attribute certificate. */
        bp = BIO_new_file(acert_file, "r");
        ExpectNotNull(bp);

        x509 = PEM_read_bio_X509_ACERT(bp, NULL, NULL, NULL);
        ExpectNotNull(x509);

        /* We're done with the bio for now. */
        if (bp != NULL) {
            BIO_free(bp);
            bp = NULL;
        }

        /* Check version and signature NID. */
        ver_long = X509_ACERT_get_version(x509);
        ExpectIntEQ(ver_long, 1);

        ver = wolfSSL_X509_ACERT_version(x509);
        ExpectIntEQ(ver, 2);

        nid = X509_ACERT_get_signature_nid(x509);

        if (is_rsa_pss) {
            ExpectIntEQ(nid, NID_rsassaPss);
        }
        else {
            ExpectIntEQ(nid, NID_sha256WithRSAEncryption);
        }

        /* Get the serial number buffer.
         * The ietf acert example has a 20 byte serial number. */
        rc = wolfSSL_X509_ACERT_get_serial_number(x509, serial, &serial_len);
        ExpectIntEQ(rc, SSL_SUCCESS);

        if (is_ietf_acert) {
            ExpectIntEQ(serial_len, 20);
            ExpectIntEQ(XMEMCMP(serial, ietf_serial, sizeof(ietf_serial)), 0);
        }
        else {
            ExpectIntEQ(serial_len, 1);
            ExpectTrue(serial[0] == 0x01);
        }

        /* Repeat the same but with null serial buffer. This is ok. */
        rc = wolfSSL_X509_ACERT_get_serial_number(x509, NULL, &serial_len);
        ExpectIntEQ(rc, SSL_SUCCESS);

        if (is_ietf_acert) {
            ExpectIntEQ(serial_len, 20);
        }
        else {
            ExpectIntEQ(serial_len, 1);
            ExpectTrue(serial[0] == 0x01);
        }

        /* Get the attributes buffer. */
        rc = wolfSSL_X509_ACERT_get_attr_buf(x509, &raw_attr, &attr_len);
        ExpectIntEQ(rc, SSL_SUCCESS);

        if (is_ietf_acert) {
            /* This cert has a 65 byte attributes field. */
            ExpectNotNull(raw_attr);
            ExpectIntEQ(attr_len, 65);
        }
        else {
            /* This cert has a 237 byte attributes field. */
            ExpectNotNull(raw_attr);
            ExpectIntEQ(attr_len, 237);
        }

        /* Test printing acert to memory bio. */
        ExpectNotNull(bp = BIO_new(BIO_s_mem()));
        rc = X509_ACERT_print(bp, x509);
        ExpectIntEQ(rc, SSL_SUCCESS);

        /* Now do a bunch of invalid stuff with partially valid inputs. */
        rc = wolfSSL_X509_ACERT_get_attr_buf(x509, &raw_attr, NULL);
        ExpectIntEQ(rc, BAD_FUNC_ARG);

        rc = wolfSSL_X509_ACERT_get_attr_buf(x509, NULL, &attr_len);
        ExpectIntEQ(rc, BAD_FUNC_ARG);

        rc = wolfSSL_X509_ACERT_get_attr_buf(NULL, &raw_attr, &attr_len);
        ExpectIntEQ(rc, BAD_FUNC_ARG);

        ver_long = X509_ACERT_get_version(NULL);
        ExpectIntEQ(ver_long, 0);

        ver = wolfSSL_X509_ACERT_version(NULL);
        ExpectIntEQ(ver, 0);

        rc = wolfSSL_X509_ACERT_get_signature(x509, NULL, NULL);
        ExpectIntEQ(rc, WOLFSSL_FATAL_ERROR);

        rc = wolfSSL_X509_ACERT_get_signature(x509, NULL, &buf_len);
        ExpectIntEQ(rc, SSL_SUCCESS);
        ExpectIntEQ(buf_len, 256);

        rc = wolfSSL_X509_ACERT_get_serial_number(x509, serial, NULL);
        ExpectIntEQ(rc, BAD_FUNC_ARG);

        rc = X509_ACERT_print(bp, NULL);
        ExpectIntEQ(rc, WOLFSSL_FAILURE);

        rc = X509_ACERT_print(NULL, x509);
        ExpectIntEQ(rc, WOLFSSL_FAILURE);

        /* Finally free the acert and bio, we're done with them. */
        if (x509 != NULL) {
            X509_ACERT_free(x509);
            x509 = NULL;
        }

        if (bp != NULL) {
            BIO_free(bp);
            bp = NULL;
        }
    }
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_ACERT_buffer(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_ACERT) && !defined(NO_CERTS) && \
    !defined(NO_RSA) && defined(WC_RSA_PSS) && \
    (defined(OPENSSL_EXTRA_X509_SMALL) || defined(OPENSSL_EXTRA))
    const byte acert_ietf[] = \
    "-----BEGIN ATTRIBUTE CERTIFICATE-----\n"
    "MIICPTCCASUCAQEwN6AWMBGkDzANMQswCQYDVQQDDAJDQQIBAqEdpBswGTEXMBUG\n"
    "A1UEAwwOc2VydmVyLmV4YW1wbGWgLTArpCkwJzElMCMGA1UEAwwcQXR0cmlidXRl\n"
    "IENlcnRpZmljYXRlIElzc3VlcjANBgkqhkiG9w0BAQsFAAIUA7WQWQKiqrVAIUS4\n"
    "LE/ZgBtfV8IwIhgPMjAyMTA2MTUxMjM1MDBaGA8yMDMxMDYxMzEyMzUwMFowQTAj\n"
    "BggrBgEFBQcKBDEXMBWgCYYHVGVzdHZhbDAIDAZncm91cDEwGgYDVQRIMRMwEaEP\n"
    "gw1hZG1pbmlzdHJhdG9yMCwwHwYDVR0jBBgwFoAUYm7JaGdsZLtTgt0tqoCK2MrI\n"
    "i10wCQYDVR04BAIFADANBgkqhkiG9w0BAQsFAAOCAQEAlIOJ2Dj3TEUj6BIv6vUs\n"
    "GqFWms05i+d10XSzWrunlUTQPoJcUjYkifOWp/7RpZ2XnRl+6hH+nIbmwSmXWwBn\n"
    "ERw2bQMmw/""/nWuN4Qv9t7ltuovWC0pJX6VMT1IRTuTV4SxuZpFL37vkmnFlPBlb+\n"
    "mn3ESSxLTjThWFIq1tip4IaxE/i5Uh32GlJglatFHM1PCGoJtyLtYb6KHDlvknw6\n"
    "coDyjIcj0FZwtQw41jLwxI8jWNmrpt978wdpprB/URrRs+m02HmeQoiHFi/qvdv8\n"
    "d+5vHf3Pi/ulhz/+dvr0p1vEQSoFnYxLXuty2p5m3PJPZCFmT3gURgmgR3BN9d7A\n"
    "Bw==\n"
    "-----END ATTRIBUTE CERTIFICATE-----\n";
    X509_ACERT * x509 = NULL;
    int          rc = 0;
    byte         ietf_serial[] = {0x03, 0xb5, 0x90, 0x59, 0x02,
                                  0xa2, 0xaa, 0xb5, 0x40, 0x21,
                                  0x44, 0xb8, 0x2c, 0x4f, 0xd9,
                                  0x80, 0x1b, 0x5f, 0x57, 0xc2};
    byte         serial[64];
    int          serial_len = sizeof(serial);
    const byte * raw_attr = NULL;
    word32       attr_len = 0;

    x509 = wolfSSL_X509_ACERT_load_certificate_buffer_ex(acert_ietf,
                                                         sizeof(acert_ietf),
                                                         WOLFSSL_FILETYPE_PEM,
                                                         HEAP_HINT);

    rc = wolfSSL_X509_ACERT_get_serial_number(x509, serial, &serial_len);
    ExpectIntEQ(rc, SSL_SUCCESS);

    ExpectIntEQ(serial_len, 20);
    ExpectIntEQ(XMEMCMP(serial, ietf_serial, sizeof(ietf_serial)), 0);

    /* Get the attributes buffer. */
    rc = wolfSSL_X509_ACERT_get_attr_buf(x509, &raw_attr, &attr_len);
    ExpectIntEQ(rc, SSL_SUCCESS);

    /* This cert has a 65 byte attributes field. */
    ExpectNotNull(raw_attr);
    ExpectIntEQ(attr_len, 65);

    ExpectNotNull(x509);

    if (x509 != NULL) {
        wolfSSL_X509_ACERT_free(x509);
        x509 = NULL;
    }
#endif
    return EXPECT_RESULT();
}

/* note: when ACERT generation and signing are implemented,
 * this test will be filled out appropriately.
 * */
static int test_wolfSSL_X509_ACERT_new_and_sign(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_ACERT) && !defined(NO_CERTS) && \
    !defined(NO_RSA) && defined(WC_RSA_PSS) && \
    (defined(OPENSSL_EXTRA_X509_SMALL) || defined(OPENSSL_EXTRA))
    X509_ACERT * x509 = NULL;
    int          rc = 0;

    x509 = X509_ACERT_new();
    ExpectNotNull(x509);

    if (x509 != NULL) {
        wolfSSL_X509_ACERT_free(x509);
        x509 = NULL;
    }

    /* Same but with static memory hint. */
    x509 = wolfSSL_X509_ACERT_new_ex(HEAP_HINT);
    ExpectNotNull(x509);

    #ifndef NO_WOLFSSL_STUB
    /* ACERT sign not implemented yet. */
    if (x509 != NULL) {
        rc = wolfSSL_X509_ACERT_sign(x509, NULL, NULL);
        ExpectIntEQ(rc, WOLFSSL_NOT_IMPLEMENTED);
    }
    #else
    (void) rc;
    #endif /* NO_WOLFSSL_STUB */

    if (x509 != NULL) {
        wolfSSL_X509_ACERT_free(x509);
        x509 = NULL;
    }

#endif
    return EXPECT_RESULT();
}

/* Test ACERT support, but with ASN functions only.
 *
 * This example acert_ietf has both Holder IssuerSerial
 * and Holder entityName fields.
 * */
static int test_wolfSSL_X509_ACERT_asn(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_ACERT) && !defined(NO_CERTS)
    const byte     acert_ietf[] = \
    "-----BEGIN ATTRIBUTE CERTIFICATE-----\n"
    "MIICPTCCASUCAQEwN6AWMBGkDzANMQswCQYDVQQDDAJDQQIBAqEdpBswGTEXMBUG\n"
    "A1UEAwwOc2VydmVyLmV4YW1wbGWgLTArpCkwJzElMCMGA1UEAwwcQXR0cmlidXRl\n"
    "IENlcnRpZmljYXRlIElzc3VlcjANBgkqhkiG9w0BAQsFAAIUA7WQWQKiqrVAIUS4\n"
    "LE/ZgBtfV8IwIhgPMjAyMTA2MTUxMjM1MDBaGA8yMDMxMDYxMzEyMzUwMFowQTAj\n"
    "BggrBgEFBQcKBDEXMBWgCYYHVGVzdHZhbDAIDAZncm91cDEwGgYDVQRIMRMwEaEP\n"
    "gw1hZG1pbmlzdHJhdG9yMCwwHwYDVR0jBBgwFoAUYm7JaGdsZLtTgt0tqoCK2MrI\n"
    "i10wCQYDVR04BAIFADANBgkqhkiG9w0BAQsFAAOCAQEAlIOJ2Dj3TEUj6BIv6vUs\n"
    "GqFWms05i+d10XSzWrunlUTQPoJcUjYkifOWp/7RpZ2XnRl+6hH+nIbmwSmXWwBn\n"
    "ERw2bQMmw/""/nWuN4Qv9t7ltuovWC0pJX6VMT1IRTuTV4SxuZpFL37vkmnFlPBlb+\n"
    "mn3ESSxLTjThWFIq1tip4IaxE/i5Uh32GlJglatFHM1PCGoJtyLtYb6KHDlvknw6\n"
    "coDyjIcj0FZwtQw41jLwxI8jWNmrpt978wdpprB/URrRs+m02HmeQoiHFi/qvdv8\n"
    "d+5vHf3Pi/ulhz/+dvr0p1vEQSoFnYxLXuty2p5m3PJPZCFmT3gURgmgR3BN9d7A\n"
    "Bw==\n"
    "-----END ATTRIBUTE CERTIFICATE-----\n";
    int            rc = 0;
    int            n_diff = 0;
    byte           ietf_serial[] =      {0x03, 0xb5, 0x90, 0x59, 0x02,
                                         0xa2, 0xaa, 0xb5, 0x40, 0x21,
                                         0x44, 0xb8, 0x2c, 0x4f, 0xd9,
                                         0x80, 0x1b, 0x5f, 0x57, 0xc2};
    byte           holderIssuerName[] = {0x31, 0x0b, 0x30, 0x09, 0x06,
                                         0x03, 0x55, 0x04, 0x03, 0x0c,
                                         0x02, 0x43, 0x41};
    byte           holderEntityName[] = {0x31, 0x17, 0x30, 0x15, 0x06,
                                         0x03, 0x55, 0x04, 0x03, 0x0c,
                                         0x0e, 0x73, 0x65, 0x72, 0x76,
                                         0x65, 0x72, 0x2e, 0x65, 0x78,
                                         0x61, 0x6d, 0x70, 0x6c, 0x65};
    DerBuffer *    der = NULL;
    WC_DECLARE_VAR(acert, DecodedAcert, 1, 0);

    rc = wc_PemToDer(acert_ietf, sizeof(acert_ietf), ACERT_TYPE, &der,
                     HEAP_HINT, NULL, NULL);

    ExpectIntEQ(rc, 0);
    ExpectNotNull(der);

    if (der != NULL) {
        ExpectNotNull(der->buffer);
    }

#ifdef WOLFSSL_SMALL_STACK
    acert = (DecodedAcert*)XMALLOC(sizeof(DecodedAcert), HEAP_HINT,
                                   DYNAMIC_TYPE_DCERT);
    ExpectNotNull(acert);
#else
    XMEMSET(acert, 0, sizeof(DecodedAcert));
#endif

    if (der != NULL && der->buffer != NULL
#ifdef WOLFSSL_SMALL_STACK
        && acert != NULL
#endif
    ) {
        wc_InitDecodedAcert(acert, der->buffer, der->length, HEAP_HINT);
        rc = wc_ParseX509Acert(acert, VERIFY_SKIP_DATE);
        ExpectIntEQ(rc, 0);

        ExpectIntEQ(acert->serialSz, 20);
        ExpectIntEQ(XMEMCMP(acert->serial, ietf_serial, sizeof(ietf_serial)),
                    0);

        /* This cert has a 65 byte attributes field. */
        ExpectNotNull(acert->rawAttr);
        ExpectIntEQ(acert->rawAttrLen, 65);

        ExpectNotNull(acert->holderIssuerName);
        ExpectNotNull(acert->holderEntityName);

        if ((acert->holderIssuerName != NULL) &&
            (acert->holderEntityName != NULL)) {
            ExpectNotNull(acert->holderEntityName->name);
            ExpectNotNull(acert->holderIssuerName->name);
        }

        if ((acert->holderIssuerName != NULL) &&
            (acert->holderEntityName != NULL) &&
            (acert->holderIssuerName->name != NULL) &&
            (acert->holderEntityName->name != NULL)) {
            ExpectIntEQ(acert->holderIssuerName->len,
                        sizeof(holderIssuerName));
            ExpectIntEQ(acert->holderEntityName->len,
                        sizeof(holderEntityName));

            ExpectIntEQ(acert->holderIssuerName->type, ASN_DIR_TYPE);
            ExpectIntEQ(acert->holderEntityName->type, ASN_DIR_TYPE);

            n_diff = XMEMCMP(acert->holderIssuerName->name, holderIssuerName,
                             sizeof(holderIssuerName));
            ExpectIntEQ(n_diff, 0);

            n_diff = XMEMCMP(acert->holderEntityName->name, holderEntityName,
                             sizeof(holderEntityName));
            ExpectIntEQ(n_diff, 0);
        }

        wc_FreeDecodedAcert(acert);
    }

#ifdef WOLFSSL_SMALL_STACK
    if (acert != NULL) {
        XFREE(acert, HEAP_HINT, DYNAMIC_TYPE_DCERT);
        acert = NULL;
    }
#endif

    if (der != NULL) {
        wc_FreeDer(&der);
        der = NULL;
    }

#endif
    return EXPECT_RESULT();
}

#if !defined(NO_DH) && !defined(NO_AES) && defined(WOLFSSL_CERT_GEN) && \
         defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES) && \
         defined(OPENSSL_EXTRA) && !defined(NO_ASN_TIME)
/* create certificate with version 2 */
static int test_set_x509_badversion(WOLFSSL_CTX* ctx)
{
    EXPECT_DECLS;
    WOLFSSL_X509 *x509 = NULL, *x509v2 = NULL;
    WOLFSSL_EVP_PKEY *priv = NULL, *pub = NULL;
    unsigned char *der = NULL, *key = NULL, *pt;
    char *header = NULL, *name = NULL;
    int derSz;
    long keySz;
    XFILE fp = XBADFILE;
    WOLFSSL_ASN1_TIME *notBefore = NULL, *notAfter = NULL;
    time_t t;

    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(cliCertFile,
                WOLFSSL_FILETYPE_PEM));

    ExpectTrue((fp = XFOPEN(cliKeyFile, "rb")) != XBADFILE);
    ExpectIntEQ(wolfSSL_PEM_read(fp, &name, &header, &key, &keySz),
            WOLFSSL_SUCCESS);
    if (fp != XBADFILE)
        XFCLOSE(fp);
    pt = key;
    ExpectNotNull(priv = wolfSSL_d2i_PrivateKey(EVP_PKEY_RSA, NULL,
                (const unsigned char**)&pt, keySz));


    /* create the version 2 certificate */
    ExpectNotNull(x509v2 = X509_new());
    ExpectIntEQ(wolfSSL_X509_set_version(x509v2, 1), WOLFSSL_SUCCESS);

    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509v2,
                wolfSSL_X509_get_subject_name(x509)), WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509v2,
                wolfSSL_X509_get_issuer_name(x509)), WOLFSSL_SUCCESS);
    ExpectNotNull(pub = wolfSSL_X509_get_pubkey(x509));
    ExpectIntEQ(X509_set_pubkey(x509v2, pub), WOLFSSL_SUCCESS);

    t = time(NULL);
    ExpectNotNull(notBefore = wolfSSL_ASN1_TIME_adj(NULL, t, 0, 0));
    ExpectNotNull(notAfter = wolfSSL_ASN1_TIME_adj(NULL, t, 365, 0));
    ExpectTrue(wolfSSL_X509_set_notBefore(x509v2, notBefore));
    ExpectTrue(wolfSSL_X509_set_notAfter(x509v2, notAfter));

    ExpectIntGT(wolfSSL_X509_sign(x509v2, priv, EVP_sha256()), 0);
    derSz = wolfSSL_i2d_X509(x509v2, &der);
    ExpectIntGT(derSz, 0);
    ExpectIntEQ(wolfSSL_CTX_use_certificate_buffer(ctx, der, derSz,
                                     WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    /* TODO: Replace with API call */
    XFREE(der, HEAP_HINT, DYNAMIC_TYPE_OPENSSL);
    XFREE(key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    XFREE(name, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    XFREE(header, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    wolfSSL_X509_free(x509);
    wolfSSL_X509_free(x509v2);
    wolfSSL_EVP_PKEY_free(priv);
    wolfSSL_EVP_PKEY_free(pub);
    wolfSSL_ASN1_TIME_free(notBefore);
    wolfSSL_ASN1_TIME_free(notAfter);

    return EXPECT_RESULT();
}


/* override certificate version error */
static int test_override_x509(int preverify, WOLFSSL_X509_STORE_CTX* store)
{
    EXPECT_DECLS;
#ifndef OPENSSL_COMPATIBLE_DEFAULTS
    ExpectIntEQ(store->error, WC_NO_ERR_TRACE(ASN_VERSION_E));
#else
    ExpectIntEQ(store->error, 0);
#endif
    ExpectIntEQ((int)wolfSSL_X509_get_version(store->current_cert), 1);
    (void)preverify;
    return EXPECT_RESULT() == TEST_SUCCESS;
}


/* set verify callback that will override bad certificate version */
static int test_set_override_x509(WOLFSSL_CTX* ctx)
{
    wolfSSL_CTX_set_verify(ctx, WOLFSSL_VERIFY_PEER, test_override_x509);
    return TEST_SUCCESS;
}
#endif


static int test_wolfSSL_X509_TLS_version_test_1(void)
{
    EXPECT_DECLS;
#if !defined(NO_DH) && !defined(NO_AES) && defined(WOLFSSL_CERT_GEN) && \
         defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES) && \
         defined(OPENSSL_EXTRA) && !defined(NO_ASN_TIME)
    test_ssl_cbf func_cb_client;
    test_ssl_cbf func_cb_server;

    /* test server rejects a client certificate that is not version 3 */
    XMEMSET(&func_cb_client, 0, sizeof(func_cb_client));
    XMEMSET(&func_cb_server, 0, sizeof(func_cb_server));

    func_cb_client.ctx_ready = &test_set_x509_badversion;
#ifndef WOLFSSL_NO_TLS12
    func_cb_client.method = wolfTLSv1_2_client_method;
#else
    func_cb_client.method = wolfTLSv1_3_client_method;
#endif

#ifndef WOLFSSL_NO_TLS12
    func_cb_server.method = wolfTLSv1_2_server_method;
#else
    func_cb_server.method = wolfTLSv1_3_server_method;
#endif

#ifndef OPENSSL_COMPATIBLE_DEFAULTS
    ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&func_cb_client,
        &func_cb_server, NULL), -1001);
#else
    ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&func_cb_client,
        &func_cb_server, NULL), TEST_SUCCESS);
#endif
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_TLS_version_test_2(void)
{
    EXPECT_DECLS;
#if !defined(NO_DH) && !defined(NO_AES) && defined(WOLFSSL_CERT_GEN) && \
         defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES) && \
         defined(OPENSSL_EXTRA) && !defined(NO_ASN_TIME)
    test_ssl_cbf func_cb_client;
    test_ssl_cbf func_cb_server;

    XMEMSET(&func_cb_client, 0, sizeof(func_cb_client));
    XMEMSET(&func_cb_server, 0, sizeof(func_cb_server));

    func_cb_client.ctx_ready = &test_set_x509_badversion;
    func_cb_server.ctx_ready = &test_set_override_x509;
#ifndef WOLFSSL_NO_TLS12
    func_cb_client.method = wolfTLSv1_2_client_method;
#else
    func_cb_client.method = wolfTLSv1_3_client_method;
#endif

#ifndef WOLFSSL_NO_TLS12
    func_cb_server.method = wolfTLSv1_2_server_method;
#else
    func_cb_server.method = wolfTLSv1_3_server_method;
#endif

    ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&func_cb_client,
        &func_cb_server, NULL), TEST_SUCCESS);
#endif

    return EXPECT_RESULT();
}

/* Testing function  wolfSSL_CTX_SetMinVersion; sets the minimum downgrade
 * version allowed.
 * POST: 1 on success.
 */
static int test_wolfSSL_CTX_SetMinVersion(void)
{
    int                     res = TEST_SKIPPED;
#if !defined(NO_WOLFSSL_CLIENT) && !defined(NO_TLS)
    int                     failFlag = WOLFSSL_SUCCESS;
    WOLFSSL_CTX*            ctx;
    int                     itr;

    #ifndef NO_OLD_TLS
        const int versions[]  = {
                            #ifdef WOLFSSL_ALLOW_TLSV10
                                  WOLFSSL_TLSV1,
                            #endif
                                  WOLFSSL_TLSV1_1,
                                  WOLFSSL_TLSV1_2 };
    #elif !defined(WOLFSSL_NO_TLS12)
        const int versions[]  = { WOLFSSL_TLSV1_2 };
    #elif defined(WOLFSSL_TLS13)
        const int versions[]  = { WOLFSSL_TLSV1_3 };
    #else
        const int versions[0];
    #endif

    ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());

    for (itr = 0; itr < (int)(sizeof(versions)/sizeof(int)); itr++) {
        if (wolfSSL_CTX_SetMinVersion(ctx, *(versions + itr))
                != WOLFSSL_SUCCESS) {
            failFlag = WOLFSSL_FAILURE;
        }
    }

    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(failFlag == WOLFSSL_SUCCESS);
#endif
    return res;

} /* END test_wolfSSL_CTX_SetMinVersion */


/*----------------------------------------------------------------------------*
 | OCSP Stapling
 *----------------------------------------------------------------------------*/


/* Testing wolfSSL_UseOCSPStapling function. OCSP stapling eliminates the need
 * need to contact the CA, lowering the cost of cert revocation checking.
 * PRE: HAVE_OCSP and HAVE_CERTIFICATE_STATUS_REQUEST
 * POST: 1 returned for success.
 */
static int test_wolfSSL_UseOCSPStapling(void)
{
    EXPECT_DECLS;
#if defined(HAVE_CERTIFICATE_STATUS_REQUEST) && defined(HAVE_OCSP) && \
    !defined(NO_TLS) && !defined(NO_WOLFSSL_CLIENT)
    WOLFSSL_CTX*    ctx = NULL;
    WOLFSSL*        ssl = NULL;

#ifndef NO_WOLFSSL_CLIENT
    #ifndef WOLFSSL_NO_TLS12
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_2_client_method()));
    #else
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_3_client_method()));
    #endif
#else
    #ifndef WOLFSSL_NO_TLS12
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_2_server_method()));
    #else
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_3_server_method()));
    #endif
#endif
    ExpectNotNull(ssl = wolfSSL_new(ctx));

    ExpectIntEQ(wolfSSL_UseOCSPStapling(NULL, WOLFSSL_CSR2_OCSP,
        WOLFSSL_CSR2_OCSP_USE_NONCE), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
#ifndef NO_WOLFSSL_CLIENT
    ExpectIntEQ(wolfSSL_UseOCSPStapling(ssl, WOLFSSL_CSR2_OCSP,
        WOLFSSL_CSR2_OCSP_USE_NONCE), 1);
#else
    ExpectIntEQ(wolfSSL_UseOCSPStapling(ssl, WOLFSSL_CSR2_OCSP,
        WOLFSSL_CSR2_OCSP_USE_NONCE), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
#endif

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
} /* END test_wolfSSL_UseOCSPStapling */


/* Testing OCSP stapling version 2, wolfSSL_UseOCSPStaplingV2 function. OCSP
 * stapling eliminates the need to contact the CA and lowers cert revocation
 * check.
 * PRE: HAVE_CERTIFICATE_STATUS_REQUEST_V2 and HAVE_OCSP defined.
 */
static int test_wolfSSL_UseOCSPStaplingV2(void)
{
    EXPECT_DECLS;
#if defined(HAVE_CERTIFICATE_STATUS_REQUEST_V2) && defined(HAVE_OCSP) && \
    !defined(NO_TLS) && !defined(NO_WOLFSSL_CLIENT)
    WOLFSSL_CTX*        ctx = NULL;
    WOLFSSL*            ssl = NULL;

#ifndef NO_WOLFSSL_CLIENT
    #ifndef WOLFSSL_NO_TLS12
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_2_client_method()));
    #else
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_3_client_method()));
    #endif
#else
    #ifndef WOLFSSL_NO_TLS12
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_2_server_method()));
    #else
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_3_server_method()));
    #endif
#endif
    ExpectNotNull(ssl = wolfSSL_new(ctx));

    ExpectIntEQ(wolfSSL_UseOCSPStaplingV2(NULL, WOLFSSL_CSR2_OCSP,
        WOLFSSL_CSR2_OCSP_USE_NONCE), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
#ifndef NO_WOLFSSL_CLIENT
    ExpectIntEQ(wolfSSL_UseOCSPStaplingV2(ssl, WOLFSSL_CSR2_OCSP,
        WOLFSSL_CSR2_OCSP_USE_NONCE), 1);
#else
    ExpectIntEQ(wolfSSL_UseOCSPStaplingV2(ssl, WOLFSSL_CSR2_OCSP,
        WOLFSSL_CSR2_OCSP_USE_NONCE), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
#endif

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();

} /* END test_wolfSSL_UseOCSPStaplingV2 */

/*----------------------------------------------------------------------------*
 | Multicast Tests
 *----------------------------------------------------------------------------*/
static int test_wolfSSL_mcast(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_DTLS) && defined(WOLFSSL_MULTICAST) && \
    (defined(WOLFSSL_TLS13) || defined(WOLFSSL_SNIFFER))
    WOLFSSL_CTX* ctx = NULL;
    WOLFSSL* ssl = NULL;
    byte preMasterSecret[512];
    byte clientRandom[32];
    byte serverRandom[32];
    byte suite[2] = {0, 0xfe};  /* WDM_WITH_NULL_SHA256 */
    byte buf[256];
    word16 newId;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfDTLSv1_2_client_method()));

    ExpectIntEQ(wolfSSL_CTX_mcast_set_member_id(ctx, 0), WOLFSSL_SUCCESS);

    ExpectNotNull(ssl = wolfSSL_new(ctx));

    XMEMSET(preMasterSecret, 0x23, sizeof(preMasterSecret));
    XMEMSET(clientRandom, 0xA5, sizeof(clientRandom));
    XMEMSET(serverRandom, 0x5A, sizeof(serverRandom));
    ExpectIntEQ(wolfSSL_set_secret(ssl, 23, preMasterSecret,
        sizeof(preMasterSecret), clientRandom, serverRandom, suite),
        WOLFSSL_SUCCESS);

    ExpectIntLE(wolfSSL_mcast_read(ssl, &newId, buf, sizeof(buf)), 0);
    ExpectIntLE(newId, 100);

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif /* WOLFSSL_DTLS && WOLFSSL_MULTICAST && (WOLFSSL_TLS13 ||
        * WOLFSSL_SNIFFER) */
    return EXPECT_RESULT();
}


/*----------------------------------------------------------------------------*
 |  Wolfcrypt
 *----------------------------------------------------------------------------*/

/*
 * Testing wc_SetKeyUsage()
 */
static int test_wc_SetKeyUsage(void)
{
    EXPECT_DECLS;
#if !defined(NO_RSA) && defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_CERT_GEN) && !defined(HAVE_FIPS)
    Cert myCert;

    ExpectIntEQ(wc_InitCert(&myCert), 0);

    ExpectIntEQ(wc_SetKeyUsage(&myCert, "keyEncipherment,keyAgreement"), 0);
    ExpectIntEQ(wc_SetKeyUsage(&myCert, "digitalSignature,nonRepudiation"), 0);
    ExpectIntEQ(wc_SetKeyUsage(&myCert, "contentCommitment,encipherOnly"), 0);
    ExpectIntEQ(wc_SetKeyUsage(&myCert, "decipherOnly"), 0);
    ExpectIntEQ(wc_SetKeyUsage(&myCert, "cRLSign,keyCertSign"), 0);

    /* Test bad args. */
    ExpectIntEQ(wc_SetKeyUsage(NULL, "decipherOnly"), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_SetKeyUsage(&myCert, NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_SetKeyUsage(&myCert, ""), WC_NO_ERR_TRACE(KEYUSAGE_E));
    ExpectIntEQ(wc_SetKeyUsage(&myCert, ","), WC_NO_ERR_TRACE(KEYUSAGE_E));
    ExpectIntEQ(wc_SetKeyUsage(&myCert, "digitalSignature, cRLSign"),
        WC_NO_ERR_TRACE(KEYUSAGE_E));
#endif
    return EXPECT_RESULT();
} /* END  test_wc_SetKeyUsage */

#if defined(OPENSSL_EXTRA) || defined(HAVE_WEBSERVER)
static void sample_mutex_cb (int flag, int type, const char* file, int line)
{
    (void)flag;
    (void)type;
    (void)file;
    (void)line;
}
#endif
/*
 * Testing wc_LockMutex_ex
 */
static int test_wc_LockMutex_ex(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) || defined(HAVE_WEBSERVER)
    int flag = CRYPTO_LOCK;
    int type = 0;
    const char* file = "./test-LockMutex_ex.txt";
    int line = 0;

    /* without SetMutexCb */
    ExpectIntEQ(wc_LockMutex_ex(flag, type, file, line), WC_NO_ERR_TRACE(BAD_STATE_E));
    /* with SetMutexCb */
    ExpectIntEQ(wc_SetMutexCb(sample_mutex_cb), 0);
    ExpectIntEQ(wc_LockMutex_ex(flag, type, file, line), 0);
    ExpectIntEQ(wc_SetMutexCb(NULL), 0);
#endif
    return EXPECT_RESULT();
} /* End test_wc_LockMutex_ex*/
/*
 * Testing wc_SetMutexCb
 */
static int test_wc_SetMutexCb(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) || defined(HAVE_WEBSERVER)
    ExpectIntEQ(wc_SetMutexCb(sample_mutex_cb), 0);
    ExpectIntEQ(wc_SetMutexCb(NULL), 0);
#endif
    return EXPECT_RESULT();
} /* End test_wc_SetMutexCb*/


/*
 * Testing ToTraditional
 */
static int test_ToTraditional(void)
{
    EXPECT_DECLS;
#if !defined(NO_ASN) && (defined(HAVE_PKCS8) || defined(HAVE_PKCS12)) && \
    (defined(WOLFSSL_TEST_CERT) || defined(OPENSSL_EXTRA) || \
     defined(OPENSSL_EXTRA_X509_SMALL)) && !defined(NO_FILESYSTEM)
    XFILE  f = XBADFILE;
    byte   input[TWOK_BUF];
    word32 sz = 0;

    ExpectTrue((f = XFOPEN("./certs/server-keyPkcs8.der", "rb")) != XBADFILE);
    ExpectTrue((sz = (word32)XFREAD(input, 1, sizeof(input), f)) > 0);
    if (f != XBADFILE)
        XFCLOSE(f);

    /* Good case */
    ExpectIntGT(ToTraditional(input, sz), 0);
    /* Bad cases */
    ExpectIntEQ(ToTraditional(NULL, 0), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(ToTraditional(NULL, sz), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
#ifdef WOLFSSL_ASN_TEMPLATE
    ExpectIntEQ(ToTraditional(input, 0), WC_NO_ERR_TRACE(BUFFER_E));
#else
    ExpectIntEQ(ToTraditional(input, 0), WC_NO_ERR_TRACE(ASN_PARSE_E));
#endif
#endif
    return EXPECT_RESULT();
} /* End test_ToTraditional*/


/*
 * Testing wc_SetSubjectBuffer
 */
static int test_wc_SetSubjectBuffer(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_CERT_GEN) && !defined(NO_RSA) && !defined(NO_FILESYSTEM)
    Cert   cert;
    XFILE  file = XBADFILE;
    byte*  der = NULL;
    word32 derSz;

    derSz = FOURK_BUF;
    ExpectNotNull(der = (byte*)XMALLOC(FOURK_BUF, HEAP_HINT,
        DYNAMIC_TYPE_TMP_BUFFER));
    ExpectTrue((file = XFOPEN("./certs/ca-cert.der", "rb")) != XBADFILE);
    ExpectTrue((derSz = (word32)XFREAD(der, 1, FOURK_BUF, file)) > 0);
    if (file != XBADFILE)
        XFCLOSE(file);

    ExpectIntEQ(wc_InitCert(&cert), 0);
    ExpectIntEQ(wc_SetSubjectBuffer(&cert, der, (int)derSz), 0);
    ExpectIntEQ(wc_SetSubjectBuffer(NULL, der, (int)derSz), WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    XFREE(der, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif
    return EXPECT_RESULT();
} /* End test_wc_SetSubjectBuffer*/

/*
 * Testing wc_SetSubjectKeyIdFromPublicKey_ex
 */
static int test_wc_SetSubjectKeyIdFromPublicKey_ex(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_CERT_GEN)
    WC_RNG      rng;
    Cert        cert;
#if !defined(NO_RSA) && defined(WOLFSSL_KEY_GEN)
    RsaKey      rsaKey;
    int         bits = 2048;
#endif
#if defined(HAVE_ECC)
    ecc_key     eccKey;
    int         ret;
#endif
#if defined(HAVE_ED25519) && defined(HAVE_ED25519_KEY_EXPORT)
    ed25519_key ed25519Key;
#endif
#if defined(HAVE_ED448) && defined(HAVE_ED448_KEY_EXPORT)
    ed448_key   ed448Key;
#endif

#ifndef HAVE_FIPS
    ExpectIntEQ(wc_InitRng_ex(&rng, HEAP_HINT, testDevId), 0);
#else
    ExpectIntEQ(wc_InitRng(&rng), 0);
#endif

    ExpectIntEQ(wc_InitCert(&cert), 0);

#if !defined(NO_RSA) && defined(WOLFSSL_KEY_GEN)
    /* RSA */
    XMEMSET(&rsaKey, 0, sizeof(RsaKey));
    ExpectIntEQ(wc_InitRsaKey(&rsaKey, HEAP_HINT), 0);
    ExpectIntEQ(MAKE_RSA_KEY(&rsaKey, bits, WC_RSA_EXPONENT, &rng), 0);
    ExpectIntEQ(wc_SetSubjectKeyIdFromPublicKey_ex(&cert, RSA_TYPE, &rsaKey),
        0);
    DoExpectIntEQ(wc_FreeRsaKey(&rsaKey), 0);
#endif

#if defined(HAVE_ECC)
    /* ECC */
    XMEMSET(&eccKey, 0, sizeof(ecc_key));
    ExpectIntEQ(wc_ecc_init(&eccKey), 0);
    ret = wc_ecc_make_key(&rng, KEY14, &eccKey);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &eccKey.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    ExpectIntEQ(ret, 0);
    ExpectIntEQ(wc_SetSubjectKeyIdFromPublicKey_ex(&cert, ECC_TYPE, &eccKey),
        0);
    DoExpectIntEQ(wc_ecc_free(&eccKey), 0);
#endif

#if defined(HAVE_ED25519) && defined(HAVE_ED25519_KEY_EXPORT)
    /* ED25519 */
    XMEMSET(&ed25519Key, 0, sizeof(ed25519_key));
    ExpectIntEQ(wc_ed25519_init(&ed25519Key), 0);
    ExpectIntEQ(wc_ed25519_make_key(&rng, ED25519_KEY_SIZE, &ed25519Key), 0);
    ExpectIntEQ(wc_SetSubjectKeyIdFromPublicKey_ex(&cert, ED25519_TYPE,
        &ed25519Key), 0);
    wc_ed25519_free(&ed25519Key);
#endif

#if defined(HAVE_ED448) && defined(HAVE_ED448_KEY_EXPORT)
    /* ED448 */
    XMEMSET(&ed448Key, 0, sizeof(ed448_key));
    ExpectIntEQ(wc_ed448_init(&ed448Key), 0);
    ExpectIntEQ(wc_ed448_make_key(&rng, ED448_KEY_SIZE, &ed448Key), 0);
    ExpectIntEQ(wc_SetSubjectKeyIdFromPublicKey_ex(&cert, ED448_TYPE,
        &ed448Key), 0);
    wc_ed448_free(&ed448Key);
#endif

    wc_FreeRng(&rng);
    DoExpectIntEQ(wc_FreeRng(&rng), 0);
#endif /* WOLFSSL_CERT_EXT && WOLFSSL_CERT_GEN */
    return EXPECT_RESULT();
} /* End test_wc_SetSubjectKeyIdFromPublicKey_ex*/

/*
 * Testing wc_SetAuthKeyIdFromPublicKey_ex
 */
static int test_wc_SetAuthKeyIdFromPublicKey_ex(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_CERT_GEN)
    WC_RNG      rng;
    Cert        cert;
#if !defined(NO_RSA) && defined(WOLFSSL_KEY_GEN)
    RsaKey      rsaKey;
    int         bits = 2048;
#endif
#if defined(HAVE_ECC)
    ecc_key     eccKey;
    int         ret;
#endif
#if defined(HAVE_ED25519) && defined(HAVE_ED25519_KEY_EXPORT)
    ed25519_key ed25519Key;
#endif
#if defined(HAVE_ED448) && defined(HAVE_ED448_KEY_EXPORT)
    ed448_key   ed448Key;
#endif

#ifndef HAVE_FIPS
    ExpectIntEQ(wc_InitRng_ex(&rng, HEAP_HINT, testDevId), 0);
#else
    ExpectIntEQ(wc_InitRng(&rng), 0);
#endif

    ExpectIntEQ(wc_InitCert(&cert), 0);

#if !defined(NO_RSA) && defined(WOLFSSL_KEY_GEN)
    /* RSA */
    XMEMSET(&rsaKey, 0, sizeof(RsaKey));
    ExpectIntEQ(wc_InitRsaKey(&rsaKey, HEAP_HINT), 0);
    ExpectIntEQ(MAKE_RSA_KEY(&rsaKey, bits, WC_RSA_EXPONENT, &rng), 0);
    ExpectIntEQ(wc_SetAuthKeyIdFromPublicKey_ex(&cert, RSA_TYPE, &rsaKey), 0);
    DoExpectIntEQ(wc_FreeRsaKey(&rsaKey), 0);
#endif

#if defined(HAVE_ECC)
    /* ECC */
    XMEMSET(&eccKey, 0, sizeof(ecc_key));
    ExpectIntEQ(wc_ecc_init(&eccKey), 0);
    ret = wc_ecc_make_key(&rng, KEY14, &eccKey);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &eccKey.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    ExpectIntEQ(ret, 0);
    ExpectIntEQ(wc_SetAuthKeyIdFromPublicKey_ex(&cert, ECC_TYPE, &eccKey), 0);
    DoExpectIntEQ(wc_ecc_free(&eccKey), 0);
#endif

#if defined(HAVE_ED25519) && defined(HAVE_ED25519_KEY_EXPORT)
    /* ED25519 */
    XMEMSET(&ed25519Key, 0, sizeof(ed25519_key));
    ExpectIntEQ(wc_ed25519_init(&ed25519Key), 0);
    ExpectIntEQ(wc_ed25519_make_key(&rng, ED25519_KEY_SIZE, &ed25519Key), 0);
    ExpectIntEQ(wc_SetAuthKeyIdFromPublicKey_ex(&cert, ED25519_TYPE,
        &ed25519Key), 0);
    wc_ed25519_free(&ed25519Key);
#endif

#if defined(HAVE_ED448) && defined(HAVE_ED448_KEY_EXPORT)
    /* ED448 */
    XMEMSET(&ed448Key, 0, sizeof(ed448_key));
    ExpectIntEQ(wc_ed448_init(&ed448Key), 0);
    ExpectIntEQ(wc_ed448_make_key(&rng, ED448_KEY_SIZE, &ed448Key), 0);
    ExpectIntEQ(wc_SetAuthKeyIdFromPublicKey_ex(&cert, ED448_TYPE, &ed448Key),
        0);
    wc_ed448_free(&ed448Key);
#endif

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
#endif /* defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_CERT_GEN)*/
    return EXPECT_RESULT();
} /* End test_wc_SetAuthKeyIdFromPublicKey_ex*/

static int test_wolfSSL_lhash(void)
{
    EXPECT_DECLS;
#ifdef OPENSSL_ALL
    const char testStr[] = "Like a true nature's child\n"
                           "We were born\n"
                           "Born to be wild";

#ifdef NO_SHA
    ExpectIntEQ(lh_strhash(testStr), 0xf9dc8a43);
#else
    ExpectIntEQ(lh_strhash(testStr), 0x5b7541dc);
#endif
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_NAME(void)
{
    EXPECT_DECLS;
#if (defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)) && \
    !defined(NO_CERTS) && !defined(NO_FILESYSTEM) && \
    !defined(NO_RSA) && defined(WOLFSSL_CERT_GEN) && \
    (defined(WOLFSSL_CERT_REQ) || defined(WOLFSSL_CERT_EXT) || \
     defined(OPENSSL_EXTRA))
    X509* x509 = NULL;
#ifndef OPENSSL_EXTRA
    const unsigned char* c = NULL;
    int bytes = 0;
#endif
    unsigned char buf[4096];
    XFILE f = XBADFILE;
    const X509_NAME* a = NULL;
    const X509_NAME* b = NULL;
    X509_NAME* d2i_name = NULL;
    int sz = 0;
    unsigned char* tmp = NULL;
    char file[] = "./certs/ca-cert.der";
#ifndef OPENSSL_EXTRA_X509_SMALL
    byte empty[] = { /* CN=empty emailAddress= */
        0x30, 0x21, 0x31, 0x0E, 0x30, 0x0C, 0x06, 0x03,
        0x55, 0x04, 0x03, 0x0C, 0x05, 0x65, 0x6D, 0x70,
        0x74, 0x79, 0x31, 0x0F, 0x30, 0x0D, 0x06, 0x09,
        0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x09,
        0x01, 0x16, 0x00
    };
#endif
#if defined(OPENSSL_EXTRA) && !defined(NO_PWDBASED)
    byte   digest[64]; /* max digest size */
    word32 digestSz;
#endif

#ifndef OPENSSL_EXTRA_X509_SMALL
    /* test compile of deprecated function, returns 0 */
    ExpectIntEQ(CRYPTO_thread_id(), 0);
#endif

    ExpectNotNull(a = X509_NAME_new());
    ExpectNotNull(b = X509_NAME_new());
#ifndef OPENSSL_EXTRA_X509_SMALL
    ExpectIntEQ(X509_NAME_cmp(a, b), 0);
#endif
    X509_NAME_free((X509_NAME*)b);
    X509_NAME_free((X509_NAME*)a);
    a = NULL;

    ExpectTrue((f = XFOPEN(file, "rb")) != XBADFILE);
#ifndef OPENSSL_EXTRA
    ExpectIntGT(bytes = (int)XFREAD(buf, 1, sizeof(buf), f), 0);
    if (f != XBADFILE)
        XFCLOSE(f);

    c = buf;
    ExpectNotNull(x509 = wolfSSL_X509_d2i_ex(NULL, c, bytes, HEAP_HINT));
#else
    ExpectNull(wolfSSL_X509_d2i_fp(NULL, XBADFILE));
    ExpectNotNull(wolfSSL_X509_d2i_fp(&x509, f));
    if (f != XBADFILE)
        XFCLOSE(f);
#endif

    /* test cmp function */
    ExpectNull(X509_get_issuer_name(NULL));
    ExpectNotNull(a = X509_get_issuer_name(x509));
    ExpectNull(X509_get_subject_name(NULL));
    ExpectNotNull(b = X509_get_subject_name(x509));
#ifdef KEEP_PEER_CERT
    ExpectNull(wolfSSL_X509_get_subjectCN(NULL));
    ExpectNotNull(wolfSSL_X509_get_subjectCN(x509));
#endif

#if defined(OPENSSL_EXTRA)
    ExpectIntEQ(X509_check_issued(NULL, NULL),
        WOLFSSL_X509_V_ERR_SUBJECT_ISSUER_MISMATCH);
    ExpectIntEQ(X509_check_issued(x509, NULL),
        WOLFSSL_X509_V_ERR_SUBJECT_ISSUER_MISMATCH);
    ExpectIntEQ(X509_check_issued(NULL, x509),
        WOLFSSL_X509_V_ERR_SUBJECT_ISSUER_MISMATCH);
    ExpectIntEQ(X509_check_issued(x509, x509), WOLFSSL_X509_V_OK);

    ExpectIntEQ(X509_NAME_cmp(NULL, NULL), -2);
    ExpectIntEQ(X509_NAME_cmp(NULL, b), -2);
    ExpectIntEQ(X509_NAME_cmp(a, NULL), -2);
    ExpectIntEQ(X509_NAME_cmp(a, b), 0); /* self signed should be 0 */

#if !defined(NO_PWDBASED)
    ExpectIntEQ(wolfSSL_X509_NAME_digest(NULL, NULL, NULL, NULL), 0);
    ExpectIntEQ(wolfSSL_X509_NAME_digest(a, NULL, NULL, NULL), 0);
#ifndef NO_SHA256
    ExpectIntEQ(wolfSSL_X509_NAME_digest(NULL, wolfSSL_EVP_sha256(), NULL,
        NULL), 0);
#endif
    ExpectIntEQ(wolfSSL_X509_NAME_digest(NULL, NULL, digest, NULL), 0);
    ExpectIntEQ(wolfSSL_X509_NAME_digest(NULL, NULL, NULL, &digestSz), 0);
    ExpectIntEQ(wolfSSL_X509_NAME_digest(a, NULL, digest,
        &digestSz), 0);
#ifndef NO_SHA256
    ExpectIntEQ(wolfSSL_X509_NAME_digest(NULL, wolfSSL_EVP_sha256(), digest,
        &digestSz), 0);
    ExpectIntEQ(wolfSSL_X509_NAME_digest(a, wolfSSL_EVP_sha256(), NULL,
        &digestSz), 0);
    ExpectIntEQ(wolfSSL_X509_NAME_digest(a, wolfSSL_EVP_sha256(), digest,
        NULL), 1);
    ExpectIntEQ(wolfSSL_X509_NAME_digest(a, wolfSSL_EVP_sha256(), digest,
        &digestSz), 1);
    ExpectTrue(digestSz == 32);
#endif
#else
    ExpectIntEQ(wolfSSL_X509_NAME_digest(NULL, NULL, NULL, NULL),
        NOT_COMPILED_IN);
#endif
#endif /* OPENSSL_EXTRA */

    tmp = buf;
    ExpectIntGT((sz = i2d_X509_NAME((X509_NAME*)a, &tmp)), 0);
    if (sz > 0 && tmp == buf) {
        fprintf(stderr, "\nERROR - %s line %d failed with:", __FILE__,
            __LINE__);
        fprintf(stderr, " Expected pointer to be incremented\n");
        abort();
    }

#ifndef OPENSSL_EXTRA_X509_SMALL
    tmp = buf;
    ExpectNotNull(d2i_name = d2i_X509_NAME(NULL, &tmp, sz));
#endif

    /* if output parameter is NULL, should still return required size. */
    ExpectIntGT((sz = i2d_X509_NAME((X509_NAME*)b, NULL)), 0);
    /* retry but with the function creating a buffer */
    tmp = NULL;
    ExpectIntGT((sz = i2d_X509_NAME((X509_NAME*)b, &tmp)), 0);
    XFREE(tmp, NULL, DYNAMIC_TYPE_OPENSSL);
    tmp = NULL;

#ifdef WOLFSSL_CERT_NAME_ALL
    /* test for givenName and name */
    {
        WOLFSSL_X509_NAME_ENTRY* entry = NULL;
        WOLFSSL_X509_NAME_ENTRY empty;
        const byte gName[] = "test-given-name";
        const byte name[] = "test-name";

        XMEMSET(&empty, 0, sizeof(empty));

        ExpectNull(wolfSSL_X509_NAME_ENTRY_create_by_NID(NULL,
            NID_givenName, ASN_UTF8STRING, NULL, sizeof(gName)));
        ExpectNotNull(entry = wolfSSL_X509_NAME_ENTRY_create_by_NID(NULL,
            NID_givenName, ASN_UTF8STRING, gName, sizeof(gName)));
        ExpectNotNull(wolfSSL_X509_NAME_ENTRY_create_by_NID(&entry,
            NID_givenName, ASN_UTF8STRING, gName, sizeof(gName)));
        ExpectIntEQ(wolfSSL_X509_NAME_add_entry(NULL         , NULL  , -1, 0),
            0);
        ExpectIntEQ(wolfSSL_X509_NAME_add_entry((X509_NAME*)b, NULL  , -1, 0),
            0);
        ExpectIntEQ(wolfSSL_X509_NAME_add_entry(NULL         , entry , -1, 0),
            0);
        ExpectIntEQ(wolfSSL_X509_NAME_add_entry((X509_NAME*)b, &empty, -1, 0),
            0);
        ExpectIntEQ(wolfSSL_X509_NAME_add_entry((X509_NAME*)b, entry , 99, 0),
            0);
        ExpectIntEQ(wolfSSL_X509_NAME_add_entry((X509_NAME*)b, entry , -1, 0),
            1);
        wolfSSL_X509_NAME_ENTRY_free(entry);
        entry = NULL;

        ExpectNotNull(wolfSSL_X509_NAME_ENTRY_create_by_NID(&entry,
            NID_name, ASN_UTF8STRING, name, sizeof(name)));
        ExpectIntEQ(wolfSSL_X509_NAME_add_entry((X509_NAME*)b, entry, -1, 0),
            1);
        wolfSSL_X509_NAME_ENTRY_free(entry);

        tmp = NULL;
        ExpectIntGT((sz = i2d_X509_NAME((X509_NAME*)b, &tmp)), 0);
        XFREE(tmp, NULL, DYNAMIC_TYPE_OPENSSL);
    }
#endif

    b = NULL;
    ExpectNull(X509_NAME_dup(NULL));
    ExpectNotNull(b = X509_NAME_dup((X509_NAME*)a));
#ifndef OPENSSL_EXTRA_X509_SMALL
    ExpectIntEQ(X509_NAME_cmp(a, b), 0);
#endif
    ExpectIntEQ(X509_NAME_entry_count(NULL), 0);
    ExpectIntEQ(X509_NAME_entry_count((X509_NAME*)b), 7);
    X509_NAME_free((X509_NAME*)b);
    ExpectNotNull(b = wolfSSL_X509_NAME_new());
    ExpectIntEQ(X509_NAME_entry_count((X509_NAME*)b), 0);
    ExpectIntEQ(wolfSSL_X509_NAME_copy(NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_X509_NAME_copy((X509_NAME*)a, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_X509_NAME_copy(NULL, (X509_NAME*)b), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_X509_NAME_copy((X509_NAME*)a, (X509_NAME*)b), 1);
    ExpectIntEQ(X509_NAME_entry_count((X509_NAME*)b), 7);
    X509_NAME_free((X509_NAME*)b);
    X509_NAME_free(d2i_name);
    d2i_name = NULL;
    X509_free(x509);

#ifndef OPENSSL_EXTRA_X509_SMALL
    /* test with an empty domain component */
    tmp = empty;
    sz  = sizeof(empty);
    ExpectNotNull(d2i_name = d2i_X509_NAME(NULL, &tmp, sz));
    ExpectIntEQ(X509_NAME_entry_count(d2i_name), 2);

    /* size of empty emailAddress will be 0 */
    tmp = buf;
    ExpectIntEQ(X509_NAME_get_text_by_NID(d2i_name, NID_emailAddress,
                (char*)tmp, sizeof(buf)), 0);

    /* should contain no organization name */
    tmp = buf;
    ExpectIntEQ(X509_NAME_get_text_by_NID(d2i_name, NID_organizationName,
                (char*)tmp, sizeof(buf)), -1);
    X509_NAME_free(d2i_name);
#endif
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_NAME_hash(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_FILESYSTEM) && \
    !defined(NO_RSA) && !defined(NO_SHA) && !defined(NO_BIO)
    BIO* bio = NULL;
    X509* x509 = NULL;
    X509_NAME* name = NULL;

    ExpectIntEQ(X509_NAME_hash(NULL), 0);
    ExpectNotNull(name = wolfSSL_X509_NAME_new_ex(NULL));
    ExpectIntEQ(X509_NAME_hash(name), 0);
    X509_NAME_free(name);

    ExpectNotNull(bio = BIO_new(BIO_s_file()));
    ExpectIntGT(BIO_read_filename(bio, svrCertFile), 0);
    ExpectNotNull(PEM_read_bio_X509(bio, &x509, NULL, NULL));
    ExpectIntEQ(X509_NAME_hash(X509_get_subject_name(x509)), 0x137DC03F);
    ExpectIntEQ(X509_NAME_hash(X509_get_issuer_name(x509)), 0xFDB2DA4);
    X509_free(x509);
    BIO_free(bio);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_NAME_print_ex(void)
{
    EXPECT_DECLS;
#if (defined(OPENSSL_ALL) || (defined(OPENSSL_EXTRA) && \
     (defined(HAVE_STUNNEL) || defined(WOLFSSL_NGINX) || \
     defined(HAVE_LIGHTY) || defined(WOLFSSL_HAPROXY) || \
     defined(WOLFSSL_OPENSSH) || defined(HAVE_SBLIM_SFCB)))) && \
    !defined(NO_BIO) && !defined(NO_RSA)
    int memSz = 0;
    byte* mem = NULL;
    BIO* bio = NULL;
    BIO* membio = NULL;
    X509* x509 = NULL;
    X509_NAME* name = NULL;
    X509_NAME* empty = NULL;

    const char* expNormal  = "C=US, CN=wolfssl.com";
    const char* expEqSpace = "C = US, CN = wolfssl.com";
    const char* expReverse = "CN=wolfssl.com, C=US";

    const char* expNotEscaped = "C= US,+\"\\ , CN=#wolfssl.com<>;";
    const char* expNotEscapedRev = "CN=#wolfssl.com<>;, C= US,+\"\\ ";
    const char* expRFC5523 =
        "CN=\\#wolfssl.com\\<\\>\\;, C=\\ US\\,\\+\\\"\\\\\\ ";

    /* Test with real cert (svrCertFile) first */
    ExpectNotNull(bio = BIO_new(BIO_s_file()));
    ExpectIntGT(BIO_read_filename(bio, svrCertFile), 0);
    ExpectNotNull(PEM_read_bio_X509(bio, &x509, NULL, NULL));
    ExpectNotNull(name = X509_get_subject_name(x509));

    /* Test without flags */
    ExpectNotNull(membio = BIO_new(BIO_s_mem()));
    ExpectNotNull(empty = wolfSSL_X509_NAME_new());
    ExpectIntEQ(X509_NAME_print_ex(NULL, NULL, 0, 0), WOLFSSL_FAILURE);
    ExpectIntEQ(X509_NAME_print_ex(membio, NULL, 0, 0), WOLFSSL_FAILURE);
    ExpectIntEQ(X509_NAME_print_ex(NULL, name, 0, 0), WOLFSSL_FAILURE);
    ExpectIntEQ(X509_NAME_print_ex(membio, empty, 0, 0), WOLFSSL_SUCCESS);
    ExpectIntEQ(X509_NAME_print_ex(membio, name, 0, 0), WOLFSSL_SUCCESS);
    wolfSSL_X509_NAME_free(empty);
    BIO_free(membio);
    membio = NULL;

    /* Test flag: XN_FLAG_RFC2253 */
    ExpectNotNull(membio = BIO_new(BIO_s_mem()));
    ExpectIntEQ(X509_NAME_print_ex(membio, name, 0,
                XN_FLAG_RFC2253), WOLFSSL_SUCCESS);
    BIO_free(membio);
    membio = NULL;

    /* Test flag: XN_FLAG_RFC2253 | XN_FLAG_DN_REV */
    ExpectNotNull(membio = BIO_new(BIO_s_mem()));
    ExpectIntEQ(X509_NAME_print_ex(membio, name, 0,
                XN_FLAG_RFC2253 | XN_FLAG_DN_REV), WOLFSSL_SUCCESS);
    BIO_free(membio);
    membio = NULL;

    X509_free(x509);
    BIO_free(bio);
    name = NULL;

    /* Test with empty issuer cert empty-issuer-cert.pem.
     * See notes in certs/test/gen-testcerts.sh for how it was generated. */
    ExpectNotNull(bio = BIO_new(BIO_s_file()));
    ExpectIntGT(BIO_read_filename(bio, noIssuerCertFile), 0);
    ExpectNotNull(PEM_read_bio_X509(bio, &x509, NULL, NULL));
    ExpectNotNull(name = X509_get_subject_name(x509));

    ExpectNotNull(membio = BIO_new(BIO_s_mem()));
    ExpectIntEQ(X509_NAME_print_ex(membio, name, 0, 0), WOLFSSL_SUCCESS);
    /* Should be empty string "" */
    ExpectIntEQ((memSz = BIO_get_mem_data(membio, &mem)), 0);

    BIO_free(membio);
    membio = NULL;
    X509_free(x509);
    BIO_free(bio);
    name = NULL;

    /* Test normal case without escaped characters */
    {
        /* Create name: "/C=US/CN=wolfssl.com" */
        ExpectNotNull(name = X509_NAME_new());
        ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName",
                    MBSTRING_UTF8, (byte*)"US", 2, -1, 0),
                    WOLFSSL_SUCCESS);
        ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName",
                    MBSTRING_UTF8, (byte*)"wolfssl.com", 11, -1, 0),
                    WOLFSSL_SUCCESS);

        /* Test without flags */
        ExpectNotNull(membio = BIO_new(BIO_s_mem()));
        ExpectIntEQ(X509_NAME_print_ex(membio, name, 0, 0), WOLFSSL_SUCCESS);
        ExpectIntGE((memSz = BIO_get_mem_data(membio, &mem)), 0);
        ExpectIntEQ(memSz, XSTRLEN(expNormal));
        ExpectIntEQ(XSTRNCMP((char*)mem, expNormal, XSTRLEN(expNormal)), 0);
        BIO_free(membio);
        membio = NULL;

        /* Test with XN_FLAG_ONELINE which should enable XN_FLAG_SPC_EQ for
           spaces around '=' */
        ExpectNotNull(membio = BIO_new(BIO_s_mem()));
        ExpectIntEQ(X509_NAME_print_ex(membio, name, 0, XN_FLAG_ONELINE),
            WOLFSSL_SUCCESS);
        ExpectIntGE((memSz = BIO_get_mem_data(membio, &mem)), 0);
        ExpectIntEQ(memSz, XSTRLEN(expEqSpace));
        ExpectIntEQ(XSTRNCMP((char*)mem, expEqSpace, XSTRLEN(expEqSpace)), 0);
        BIO_free(membio);
        membio = NULL;

        /* Test flags: XN_FLAG_RFC2253 - should be reversed */
        ExpectNotNull(membio = BIO_new(BIO_s_mem()));
        ExpectIntEQ(X509_NAME_print_ex(membio, name, 0,
                    XN_FLAG_RFC2253), WOLFSSL_SUCCESS);
        ExpectIntGE((memSz = BIO_get_mem_data(membio, &mem)), 0);
        ExpectIntEQ(memSz, XSTRLEN(expReverse));
        BIO_free(membio);
        membio = NULL;

        /* Test flags: XN_FLAG_DN_REV - reversed */
        ExpectNotNull(membio = BIO_new(BIO_s_mem()));
        ExpectIntEQ(X509_NAME_print_ex(membio, name, 0,
                    XN_FLAG_DN_REV), WOLFSSL_SUCCESS);
        ExpectIntGE((memSz = BIO_get_mem_data(membio, &mem)), 0);
        ExpectIntEQ(memSz, XSTRLEN(expReverse));
        ExpectIntEQ(XSTRNCMP((char*)mem, expReverse, XSTRLEN(expReverse)), 0);
        BIO_free(membio);
        membio = NULL;

        X509_NAME_free(name);
        name = NULL;
    }

    /* Test RFC2253 characters are escaped with backslashes */
    {
        ExpectNotNull(name = X509_NAME_new());
        ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName",
                    /* space at beginning and end, and: ,+"\ */
                    MBSTRING_UTF8, (byte*)" US,+\"\\ ", 8, -1, 0),
                    WOLFSSL_SUCCESS);
        ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName",
                    /* # at beginning, and: <>;*/
                    MBSTRING_UTF8, (byte*)"#wolfssl.com<>;", 15, -1, 0),
                    WOLFSSL_SUCCESS);

        /* Test without flags */
        ExpectNotNull(membio = BIO_new(BIO_s_mem()));
        ExpectIntEQ(X509_NAME_print_ex(membio, name, 0, 0), WOLFSSL_SUCCESS);
        ExpectIntGE((memSz = BIO_get_mem_data(membio, &mem)), 0);
        ExpectIntEQ(memSz, XSTRLEN(expNotEscaped));
        ExpectIntEQ(XSTRNCMP((char*)mem, expNotEscaped,
                    XSTRLEN(expNotEscaped)), 0);
        BIO_free(membio);
        membio = NULL;

        /* Test flags: XN_FLAG_RFC5523 - should be reversed and escaped */
        ExpectNotNull(membio = BIO_new(BIO_s_mem()));
        ExpectIntEQ(X509_NAME_print_ex(membio, name, 0,
                    XN_FLAG_RFC2253), WOLFSSL_SUCCESS);
        ExpectIntGE((memSz = BIO_get_mem_data(membio, &mem)), 0);
        ExpectIntEQ(memSz, XSTRLEN(expRFC5523));
        ExpectIntEQ(XSTRNCMP((char*)mem, expRFC5523, XSTRLEN(expRFC5523)), 0);
        BIO_free(membio);
        membio = NULL;

        /* Test flags: XN_FLAG_DN_REV - reversed but not escaped */
        ExpectNotNull(membio = BIO_new(BIO_s_mem()));
        ExpectIntEQ(X509_NAME_print_ex(membio, name, 0,
                    XN_FLAG_DN_REV), WOLFSSL_SUCCESS);
        ExpectIntGE((memSz = BIO_get_mem_data(membio, &mem)), 0);
        ExpectIntEQ(memSz, XSTRLEN(expNotEscapedRev));
        ExpectIntEQ(XSTRNCMP((char*)mem, expNotEscapedRev,
                    XSTRLEN(expNotEscapedRev)), 0);
        BIO_free(membio);

        X509_NAME_free(name);
    }
#endif
    return EXPECT_RESULT();
}

#ifndef NO_BIO
static int test_wolfSSL_X509_INFO_multiple_info(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_ALL) && !defined(NO_RSA)
    STACK_OF(X509_INFO) *info_stack = NULL;
    X509_INFO *info = NULL;
    int len;
    int i;
    const char* files[] = {
        cliCertFile,
        cliKeyFile,
        /* This needs to be the order as svrCertFile contains the
         * intermediate cert as well. */
        svrKeyFile,
        svrCertFile,
        NULL,
    };
    const char** curFile;
    BIO *fileBIO = NULL;
    BIO *concatBIO = NULL;
    byte tmp[FOURK_BUF];

    /* concatenate the cert and the key file to force PEM_X509_INFO_read_bio
     * to group objects together. */
    ExpectNotNull(concatBIO = BIO_new(BIO_s_mem()));
    for (curFile = files; EXPECT_SUCCESS() && *curFile != NULL; curFile++) {
        int fileLen = 0;
        ExpectNotNull(fileBIO = BIO_new_file(*curFile, "rb"));
        ExpectIntGT(fileLen = wolfSSL_BIO_get_len(fileBIO), 0);
        if (EXPECT_SUCCESS()) {
            while ((len = BIO_read(fileBIO, tmp, sizeof(tmp))) > 0) {
                ExpectIntEQ(BIO_write(concatBIO, tmp, len), len);
                fileLen -= len;
                if (EXPECT_FAIL())
                    break;
            }
            /* Make sure we read the entire file */
            ExpectIntEQ(fileLen, 0);
        }
        BIO_free(fileBIO);
        fileBIO = NULL;
    }

    ExpectNotNull(info_stack = PEM_X509_INFO_read_bio(concatBIO, NULL, NULL,
        NULL));
    ExpectIntEQ(sk_X509_INFO_num(info_stack), 3);
    for (i = 0; i < sk_X509_INFO_num(info_stack); i++) {
        ExpectNotNull(info = sk_X509_INFO_value(info_stack, i));
        ExpectNotNull(info->x509);
        ExpectNull(info->crl);
        if (i != 2) {
            ExpectNotNull(info->x_pkey);
            ExpectIntEQ(X509_check_private_key(info->x509,
                                               info->x_pkey->dec_pkey), 1);
        }
        else {
            ExpectNull(info->x_pkey);
        }
    }

    sk_X509_INFO_pop_free(info_stack, X509_INFO_free);
    BIO_free(concatBIO);
#endif
    return EXPECT_RESULT();
}
#endif

#ifndef NO_BIO
static int test_wolfSSL_X509_INFO(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_ALL) && !defined(NO_RSA)
    STACK_OF(X509_INFO) *info_stack = NULL;
    X509_INFO *info = NULL;
    BIO *cert = NULL;
    int i;
    /* PEM in hex format to avoid null terminator */
    byte data[] = {
        0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47,
        0x49, 0x4e, 0x20, 0x43, 0x45, 0x52, 0x54, 0x63, 0x2d, 0x2d, 0x2d, 0x2d,
        0x2d, 0x0a, 0x4d, 0x49, 0x49, 0x44, 0x4d, 0x54, 0x42, 0x75, 0x51, 0x3d,
        0x0a, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x45, 0x4e, 0x44, 0x20, 0x2d, 0x2d,
        0x2d, 0x2d, 0x2d
    };
    /* PEM in hex format to avoid null terminator */
    byte data2[] = {
        0x41, 0x53, 0x4e, 0x31, 0x20, 0x4f, 0x49, 0x44, 0x3a, 0x20, 0x70, 0x72,
        0x69, 0x6d, 0x65, 0x32, 0x35, 0x36, 0x76, 0x31, 0x0a, 0x2d, 0x2d, 0x2d,
        0x2d, 0x2d, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x45, 0x43, 0x20, 0x50,
        0x41, 0x52, 0x41, 0x4d, 0x45, 0x54, 0x45, 0x52, 0x53, 0x2d, 0x2d, 0x2d,
        0x2d, 0x43, 0x65, 0x72, 0x74, 0x69, 0x2d, 0x0a, 0x42, 0x67, 0x67, 0x71,
        0x68, 0x6b, 0x6a, 0x4f, 0x50, 0x51, 0x4d, 0x42, 0x42, 0x77, 0x3d, 0x3d,
        0x0a, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d
    };

    ExpectNotNull(cert = BIO_new_file(cliCertFileExt, "rb"));
    ExpectNotNull(info_stack = PEM_X509_INFO_read_bio(cert, NULL, NULL, NULL));
    for (i = 0; i < sk_X509_INFO_num(info_stack); i++) {
        ExpectNotNull(info = sk_X509_INFO_value(info_stack, i));
        ExpectNotNull(info->x509);
        ExpectNull(info->crl);
        ExpectNull(info->x_pkey);
    }
    sk_X509_INFO_pop_free(info_stack, X509_INFO_free);
    info_stack = NULL;
    BIO_free(cert);
    cert = NULL;

    ExpectNotNull(cert = BIO_new_file(cliCertFileExt, "rb"));
    ExpectNotNull(info_stack = PEM_X509_INFO_read_bio(cert, NULL, NULL, NULL));
    sk_X509_INFO_pop_free(info_stack, X509_INFO_free);
    info_stack = NULL;
    BIO_free(cert);
    cert = NULL;

    /* This case should fail due to invalid input. */
    ExpectNotNull(cert = BIO_new(BIO_s_mem()));
    ExpectIntEQ(BIO_write(cert, data, sizeof(data)), sizeof(data));
    ExpectNull(info_stack = PEM_X509_INFO_read_bio(cert, NULL, NULL, NULL));
    sk_X509_INFO_pop_free(info_stack, X509_INFO_free);
    info_stack = NULL;
    BIO_free(cert);
    cert = NULL;
    ExpectNotNull(cert = BIO_new(BIO_s_mem()));
    ExpectIntEQ(BIO_write(cert, data2, sizeof(data2)), sizeof(data2));
    ExpectNull(info_stack = PEM_X509_INFO_read_bio(cert, NULL, NULL, NULL));
    sk_X509_INFO_pop_free(info_stack, X509_INFO_free);
    BIO_free(cert);
#endif
    return EXPECT_RESULT();
}
#endif

static int test_wolfSSL_X509_subject_name_hash(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_FILESYSTEM) && \
    !defined(NO_RSA) && (!defined(NO_SHA) || !defined(NO_SHA256))
    X509* x509 = NULL;
    X509_NAME* subjectName = NULL;
    unsigned long ret1 = 0;
    unsigned long ret2 = 0;

    ExpectNotNull(x509 = X509_new());
    ExpectIntEQ(X509_subject_name_hash(NULL), 0);
    ExpectIntEQ(X509_subject_name_hash(x509), 0);
    X509_free(x509);
    x509 = NULL;

    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(cliCertFile,
                SSL_FILETYPE_PEM));
    ExpectNotNull(subjectName = wolfSSL_X509_get_subject_name(x509));

    /* These two
     *   - X509_subject_name_hash(x509)
     *   - X509_NAME_hash(X509_get_subject_name(x509))
     *  should give the same hash, if !defined(NO_SHA) is true. */

    ret1 = X509_subject_name_hash(x509);
    ExpectIntNE(ret1, 0);

#if !defined(NO_SHA)
    ret2 = X509_NAME_hash(X509_get_subject_name(x509));
    ExpectIntNE(ret2, 0);

    ExpectIntEQ(ret1, ret2);
#else
    (void) ret2;
#endif

    X509_free(x509);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_issuer_name_hash(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_FILESYSTEM) \
    && !defined(NO_RSA) && (!defined(NO_SHA) || !defined(NO_SHA256))
    X509* x509 = NULL;
    X509_NAME* issuertName = NULL;
    unsigned long ret1 = 0;
    unsigned long ret2 = 0;

    ExpectNotNull(x509 = X509_new());
    ExpectIntEQ(X509_issuer_name_hash(NULL), 0);
    ExpectIntEQ(X509_issuer_name_hash(x509), 0);
    X509_free(x509);
    x509 = NULL;

    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(cliCertFile,
                SSL_FILETYPE_PEM));
    ExpectNotNull(issuertName = wolfSSL_X509_get_issuer_name(x509));

    /* These two
     *   - X509_issuer_name_hash(x509)
     *   - X509_NAME_hash(X509_get_issuer_name(x509))
     *  should give the same hash, if !defined(NO_SHA) is true. */

    ret1 = X509_issuer_name_hash(x509);
    ExpectIntNE(ret1, 0);

#if !defined(NO_SHA)
    ret2 = X509_NAME_hash(X509_get_issuer_name(x509));
    ExpectIntNE(ret2, 0);

    ExpectIntEQ(ret1, ret2);
#else
    (void) ret2;
#endif

    X509_free(x509);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_check_host(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_FILESYSTEM) \
    && !defined(NO_SHA) && !defined(NO_RSA)
    X509* x509 = NULL;
    const char altName[] = "example.com";
    const char badAltName[] = "a.example.com";

    ExpectIntEQ(X509_check_host(NULL, NULL, XSTRLEN(altName), 0, NULL),
            WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

    /* cliCertFile has subjectAltName set to 'example.com', '127.0.0.1' */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(cliCertFile,
                SSL_FILETYPE_PEM));

    ExpectIntEQ(X509_check_host(x509, altName, XSTRLEN(altName), 0, NULL),
            WOLFSSL_SUCCESS);

    ExpectIntEQ(X509_check_host(x509, badAltName, XSTRLEN(badAltName), 0, NULL),
            WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

    ExpectIntEQ(X509_check_host(x509, NULL, 0, 0, NULL),
            WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

    /* Check WOLFSSL_LEFT_MOST_WILDCARD_ONLY flag set */
    ExpectIntEQ(X509_check_host(x509, altName, XSTRLEN(altName),
            WOLFSSL_LEFT_MOST_WILDCARD_ONLY, NULL), WOLFSSL_SUCCESS);

    ExpectIntEQ(X509_check_host(x509, NULL, 0,
            WOLFSSL_LEFT_MOST_WILDCARD_ONLY, NULL),
            WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

    ExpectIntEQ(X509_check_host(x509, badAltName, XSTRLEN(badAltName),
            WOLFSSL_LEFT_MOST_WILDCARD_ONLY, NULL),
            WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

    ExpectIntEQ(wolfSSL_X509_check_host(x509, altName, XSTRLEN(altName),
        WOLFSSL_NO_WILDCARDS, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(wolfSSL_X509_check_host(x509, altName, XSTRLEN(altName),
        WOLFSSL_NO_PARTIAL_WILDCARDS, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(wolfSSL_X509_check_host(x509, altName, XSTRLEN(altName),
        WOLFSSL_MULTI_LABEL_WILDCARDS, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

    X509_free(x509);

    ExpectIntEQ(X509_check_host(NULL, altName, XSTRLEN(altName), 0, NULL),
            WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

    /* Check again with WOLFSSL_LEFT_MOST_WILDCARD_ONLY flag set */
    ExpectIntEQ(X509_check_host(NULL, altName, XSTRLEN(altName),
            WOLFSSL_LEFT_MOST_WILDCARD_ONLY, NULL),
            WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_check_email(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && defined(WOLFSSL_CERT_GEN) && !defined(NO_RSA)
    X509* x509 = NULL;
    X509* empty = NULL;
    const char goodEmail[] = "info@wolfssl.com";
    const char badEmail[] = "disinfo@wolfssl.com";

    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(cliCertFile,
                SSL_FILETYPE_PEM));
    ExpectNotNull(empty = wolfSSL_X509_new());

    ExpectIntEQ(wolfSSL_X509_check_email(NULL, NULL, 0, 0), 0);
    ExpectIntEQ(wolfSSL_X509_check_email(x509, NULL, 0, 0), 0);
    ExpectIntEQ(wolfSSL_X509_check_email(NULL, goodEmail, XSTRLEN(goodEmail),
        0), 0);
    ExpectIntEQ(wolfSSL_X509_check_email(empty, goodEmail, XSTRLEN(goodEmail),
        0), 0);

    /* Should fail on non-matching email address */
    ExpectIntEQ(wolfSSL_X509_check_email(x509, badEmail, XSTRLEN(badEmail), 0),
            WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    /* Should succeed on matching email address */
    ExpectIntEQ(wolfSSL_X509_check_email(x509, goodEmail, XSTRLEN(goodEmail), 0),
            WOLFSSL_SUCCESS);
    /* Should compute length internally when not provided */
    ExpectIntEQ(wolfSSL_X509_check_email(x509, goodEmail, 0, 0),
            WOLFSSL_SUCCESS);
    /* Should fail when email address is NULL */
    ExpectIntEQ(wolfSSL_X509_check_email(x509, NULL, 0, 0),
            WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

    X509_free(empty);
    X509_free(x509);

    /* Should fail when x509 is NULL */
    ExpectIntEQ(wolfSSL_X509_check_email(NULL, goodEmail, 0, 0),
            WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
#endif /* OPENSSL_EXTRA && WOLFSSL_CERT_GEN */
    return EXPECT_RESULT();
}

static int test_wc_PemToDer(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && defined(WOLFSSL_PEM_TO_DER) && !defined(NO_FILESYSTEM)
    int ret;
    DerBuffer* pDer = NULL;
    const char* ca_cert = "./certs/server-cert.pem";
    const char* trusted_cert = "./certs/test/ossl-trusted-cert.pem";
    byte* cert_buf = NULL;
    size_t cert_sz = 0;
    int eccKey = 0;
    EncryptedInfo info;

    XMEMSET(&info, 0, sizeof(info));

    ExpectIntEQ(ret = load_file(ca_cert, &cert_buf, &cert_sz), 0);
    ExpectIntEQ(ret = wc_PemToDer(cert_buf, (long int)cert_sz, CERT_TYPE, &pDer, NULL,
        &info, &eccKey), 0);
    wc_FreeDer(&pDer);
    pDer = NULL;

    if (cert_buf != NULL) {
        free(cert_buf);
        cert_buf = NULL;
    }

    /* Test that -----BEGIN TRUSTED CERTIFICATE----- banner parses OK */
    ExpectIntEQ(ret = load_file(trusted_cert, &cert_buf, &cert_sz), 0);
    ExpectIntEQ(ret = wc_PemToDer(cert_buf, (long int)cert_sz, TRUSTED_CERT_TYPE, &pDer, NULL,
        &info, &eccKey), 0);
    wc_FreeDer(&pDer);
    pDer = NULL;

    if (cert_buf != NULL) {
        free(cert_buf);
        cert_buf = NULL;
    }

#ifdef HAVE_ECC
    {
        const char* ecc_private_key = "./certs/ecc-privOnlyKey.pem";
        byte key_buf[256] = {0};

        /* Test fail of loading a key with cert type */
        ExpectIntEQ(load_file(ecc_private_key, &cert_buf, &cert_sz), 0);
        key_buf[0] = '\n';
        ExpectNotNull(XMEMCPY(key_buf + 1, cert_buf, cert_sz));
        ExpectIntNE((ret = wc_PemToDer(key_buf, (long int)cert_sz + 1, CERT_TYPE,
            &pDer, NULL, &info, &eccKey)), 0);

    #ifdef OPENSSL_EXTRA
        ExpectIntEQ((ret = wc_PemToDer(key_buf, cert_sz + 1, PRIVATEKEY_TYPE,
            &pDer, NULL, &info, &eccKey)), 0);
    #endif
        wc_FreeDer(&pDer);
        if (cert_buf != NULL)
            free(cert_buf);
    }
#endif
#endif
    return EXPECT_RESULT();
}

static int test_wc_AllocDer(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS)
    DerBuffer* pDer = NULL;
    word32 testSize = 1024;

    ExpectIntEQ(wc_AllocDer(NULL, testSize, CERT_TYPE, HEAP_HINT),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_AllocDer(&pDer, testSize, CERT_TYPE, HEAP_HINT), 0);
    ExpectNotNull(pDer);
    wc_FreeDer(&pDer);
#endif
    return EXPECT_RESULT();
}

static int test_wc_CertPemToDer(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && defined(WOLFSSL_PEM_TO_DER) && !defined(NO_FILESYSTEM)
    const char* ca_cert = "./certs/ca-cert.pem";
    byte* cert_buf = NULL;
    size_t cert_sz = 0;
    size_t cert_dersz = 0;
    byte* cert_der = NULL;

    ExpectIntEQ(load_file(ca_cert, &cert_buf, &cert_sz), 0);
    cert_dersz = cert_sz; /* DER will be smaller than PEM */
    ExpectNotNull(cert_der = (byte*)malloc(cert_dersz));
    ExpectIntGE(wc_CertPemToDer(cert_buf, (int)cert_sz, cert_der,
        (int)cert_dersz, CERT_TYPE), 0);

    ExpectIntEQ(wc_CertPemToDer(NULL, (int)cert_sz, NULL, -1, CERT_TYPE),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_CertPemToDer(cert_buf, (int)cert_sz, NULL, -1, CERT_TYPE),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_CertPemToDer(NULL, (int)cert_sz, cert_der, -1, CERT_TYPE),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_CertPemToDer(NULL, (int)cert_sz, NULL, (int)cert_dersz,
        CERT_TYPE), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_CertPemToDer(NULL, (int)cert_sz, cert_der,
        (int)cert_dersz, CERT_TYPE), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_CertPemToDer(cert_buf, (int)cert_sz, NULL,
        (int)cert_dersz, CERT_TYPE), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_CertPemToDer(cert_buf, (int)cert_sz, cert_der, -1,
        CERT_TYPE), WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    if (cert_der != NULL)
        free(cert_der);
    if (cert_buf != NULL)
        free(cert_buf);
#endif
    return EXPECT_RESULT();
}

static int test_wc_KeyPemToDer(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_PEM_TO_DER) && !defined(NO_FILESYSTEM) && !defined(NO_RSA)
    int ret = 0;
    const byte cert_buf[] = \
    "-----BEGIN PRIVATE KEY-----\n"
    "MIIEvgIBADANBgkqhkiG9w0BAQEFAASCBKgwggSkAgEAAoIBAQDMG5KgWxP002pA\n"
    "QJIdA4H5N0oM1Wf0LrHcos5RYUlrHDkC2b5p2BUpVRPmgDAFD2+8leim98x0BvcB\n"
    "k48TNzrVynuwyVEY664+iQyzEBO5v27HPRydOddprbLCvRO036XINGIjauy1jHFi\n"
    "HaDVx3bexSwgp9aefUGAszFXi4q1J4GacV7Cr2b/wBqUHqWv4ZXPu6R9/UYngTkD\n"
    "UDJL5gLlLfcLzNyyodKPHPCIAKdWn6mSVdcHk8XVpK4y9lgz4E7YDWA6ohKZgWgG\n"
    "2RDha8CMilFMDgYa0G0SiS9g3PQx0qh3AMXJJsKSVhScFCZufAE0kV6KvjP7jAqP\n"
    "XBiSkRGPAgMBAAECggEAW7hmRyY2jRX2UMJThrM9VIs6fRLnYI0dQ0tsEJj536ay\n"
    "nevQjArc05KWW0Yujg+WRDZPcry3RUqd9Djlmhp/F3Si6dpF1b+PMS3wJYVrf9Sd\n"
    "SO5W7faArU4vnyBNe0HnY1Ta5xSVI65lg1RSIs88RTZwsooJwXYDGf0shq0/21CE\n"
    "V8HOb27DDYNcEnm35lzaONjFnMqQQT2Vs9anRrPiSEXNleEvTgLVXZtGTyCGTz6v\n"
    "x86Y8eSWL9YNHvPE1I+mDPuocfSR7eRNgRu7SK3mn94W5mqd7Ns072YKX/2XN1mO\n"
    "66+ZFHO6v4dK1u7cSjuwrU1EhLHpUsgDz6Bna5InyQKBgQDv5l8RPy8UneKSADaf\n"
    "M5L/5675I/5t4nqVjvbnQje00YveLTAEjlJBNR93Biln3sYgnvNamYDCxyEuUZ/I\n"
    "S/vmBL9PoxfGZow4FcsIBOEbIn3E0SYJgCBNWthquUvGpKsYDnThJuhO+1cVmxAJ\n"
    "BUOjLFnJYHM0a+Vmk9GexT2OBwKBgQDZzkUBOK7Im3eiYytFocUJyhqMH30d49X9\n"
    "ujC7kGw4UWAqVe7YCSvlBa8nzWpRWK2kRpu3M0272RU0V4geyWqT+nr/SvRRPtNP\n"
    "F5dY8l3yR7hjtSejqqjOfBcZT6ETJxI4tiG0+Nl5BlfM5M+0nxnkWpRcHuOR3j79\n"
    "YUFERyN+OQKBgQCjlOKeUAc6d65W/+4/AFvsQ378Q57qLtSHxsR1TKHPmlNVXFqx\n"
    "wJo1/JNIBduWCEHxXHF0BdfW+RGXE/FwEt/hKLuLAhrkHmjelX2sKieU6R/5ZOQa\n"
    "9lMQbDHGFDOncAF6leD85hriQGBRSzrT69MDIOrYdfwYcroqCAGX0cb3YQKBgQC8\n"
    "iIFQylj5SyHmjcMSNjKSA8CxFDzAV8yPIdE3Oo+CvGXqn5HsrRuy1hXE9VmXapR8\n"
    "A6ackSszdHiXY0FvrNe1mfdH7wDHJwPQjdIzazCJHS3uGQxj7sDKY7226ie6pXJv\n"
    "ZrCMr2/IBAaSVGm6ppHKCeIsT4ybYm7R85KEYLPHeQKBgBeJOMBinXQfWN/1jT9b\n"
    "6Ywrutvp2zP8hVxQGSZJ0WG4iewZyFLsPUlbWRXOSYNPElHmdD0ZomdLVm+lSpAA\n"
    "XSH5FJ/IFCwqq7Eft6Gf8NFRV+NjPMUny+PnjHe4oFP8YK/Ek22K3ttNG8Hw69Aw\n"
    "AQue5o6oVfhgLiJzMdo/77gw\n"
    "-----END PRIVATE KEY-----\n";
    const int cert_sz = sizeof(cert_buf);
    const char cert_pw[] = "password";
    int cert_dersz = 0;
    byte* cert_der = NULL;

    /* Bad arg: Cert buffer is NULL */
    ExpectIntEQ(wc_KeyPemToDer(NULL, cert_sz, cert_der, cert_dersz, ""),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    /* Bad arg: Cert DER buffer non-NULL but size zero (or less) */
    ExpectIntEQ(wc_KeyPemToDer(cert_buf, cert_sz, (byte*)&cert_der, 0, ""),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    /* Test normal operation */
    cert_dersz = cert_sz; /* DER will be smaller than PEM */
    ExpectNotNull(cert_der = (byte*)malloc((size_t)cert_dersz));
    ExpectIntGE(ret = wc_KeyPemToDer(cert_buf, cert_sz, cert_der, cert_dersz,
        cert_pw), 0);
    ExpectIntLE(ret, cert_sz);
    if (cert_der != NULL) {
        free(cert_der);
        cert_der = NULL;
    }

    /* Test NULL for DER buffer to return needed DER buffer size */
    ExpectIntGT(ret = wc_KeyPemToDer(cert_buf, cert_sz, NULL, 0, ""), 0);
    ExpectIntLE(ret, cert_sz);
    if (EXPECT_SUCCESS())
        cert_dersz = ret;
    ExpectNotNull(cert_der = (byte*)malloc((size_t)cert_dersz));
    ExpectIntGE(ret = wc_KeyPemToDer(cert_buf, cert_sz, cert_der, cert_dersz,
        cert_pw), 0);
    ExpectIntLE(ret, cert_sz);
    if (cert_der != NULL)
        free(cert_der);
#endif
    return EXPECT_RESULT();
}

static int test_wc_PubKeyPemToDer(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_PEM_TO_DER) && !defined(NO_FILESYSTEM) && \
   (defined(WOLFSSL_CERT_EXT) || defined(WOLFSSL_PUB_PEM_TO_DER))
    int ret = 0;
    const char* key = "./certs/ecc-client-keyPub.pem";
    byte* cert_buf = NULL;
    size_t cert_sz = 0, cert_dersz = 0;
    byte* cert_der = NULL;

    ExpectIntEQ(wc_PubKeyPemToDer(cert_buf, (int)cert_sz,
        cert_der, (int)cert_dersz), WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    ExpectIntEQ(load_file(key, &cert_buf, &cert_sz), 0);
    cert_dersz = cert_sz; /* DER will be smaller than PEM */
    ExpectNotNull(cert_der = (byte*)malloc(cert_dersz));
    ExpectIntGE(wc_PubKeyPemToDer(cert_buf, (int)cert_sz, cert_der,
        (int)cert_dersz), 0);
    if (cert_der != NULL) {
        free(cert_der);
        cert_der = NULL;
    }

    /* Test NULL for DER buffer to return needed DER buffer size */
    ExpectIntGT(ret = wc_PubKeyPemToDer(cert_buf, (int)cert_sz, NULL, 0), 0);
    ExpectIntLE(ret, cert_sz);
    cert_dersz = (size_t)ret;
    ExpectNotNull(cert_der = (byte*)malloc(cert_dersz));
    ExpectIntGE(wc_PubKeyPemToDer(cert_buf, (int)cert_sz, cert_der,
        (int)cert_dersz), 0);
    if (cert_der != NULL) {
        free(cert_der);
    }

    if (cert_buf != NULL) {
        free(cert_buf);
    }
#endif
    return EXPECT_RESULT();
}

static int test_wc_PemPubKeyToDer(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && \
    (defined(WOLFSSL_CERT_EXT) || defined(WOLFSSL_PUB_PEM_TO_DER))
    const char* key = "./certs/ecc-client-keyPub.pem";
    size_t cert_dersz = 1024;
    byte* cert_der = NULL;

    ExpectIntGE(wc_PemPubKeyToDer(NULL, cert_der, (int)cert_dersz),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    ExpectNotNull(cert_der = (byte*)malloc(cert_dersz));
    ExpectIntGE(wc_PemPubKeyToDer(key, cert_der, (int)cert_dersz), 0);
    if (cert_der != NULL) {
        free(cert_der);
    }
#endif
    return EXPECT_RESULT();
}

static int test_wc_GetPubKeyDerFromCert(void)
{
    EXPECT_DECLS;
#if !defined(NO_RSA) || defined(HAVE_ECC)
    int ret;
    word32 idx = 0;
    byte keyDer[TWOK_BUF];  /* large enough for up to RSA 2048 */
    word32 keyDerSz = (word32)sizeof(keyDer);
    DecodedCert decoded;
#if !defined(NO_RSA) && defined(WOLFSSL_CERT_REQ) && !defined(NO_FILESYSTEM)
    byte certBuf[6000]; /* for PEM and CSR, client-cert.pem is 5-6kB */
    word32 certBufSz = sizeof(certBuf);
#endif
#if ((!defined(USE_CERT_BUFFERS_2048) && !defined(USE_CERT_BUFFERS_1024)) || \
     defined(WOLFSSL_CERT_REQ)) && !defined(NO_RSA) && !defined(NO_FILESYSTEM)
    XFILE fp = XBADFILE;
#endif
#ifndef NO_RSA
    RsaKey rsaKey;
    #if defined(USE_CERT_BUFFERS_2048)
        byte* rsaCertDer = (byte*)client_cert_der_2048;
        word32 rsaCertDerSz = sizeof_client_cert_der_2048;
    #elif defined(USE_CERT_BUFFERS_1024)
        byte* rsaCertDer = (byte*)client_cert_der_1024;
        word32 rsaCertDerSz = sizeof_client_cert_der_1024;
    #else
        unsigned char rsaCertDer[TWOK_BUF];
        word32 rsaCertDerSz;
    #endif
#endif
#ifdef HAVE_ECC
    ecc_key eccKey;
    #if defined(USE_CERT_BUFFERS_256)
        byte* eccCert = (byte*)cliecc_cert_der_256;
        word32 eccCertSz = sizeof_cliecc_cert_der_256;
    #else
        unsigned char eccCert[ONEK_BUF];
        word32 eccCertSz;
        XFILE fp2 = XBADFILE;
    #endif
#endif

#ifndef NO_RSA

#if !defined(USE_CERT_BUFFERS_1024) && !defined(USE_CERT_BUFFERS_2048)
    ExpectTrue((fp = XFOPEN("./certs/1024/client-cert.der", "rb")) != XBADFILE);
    ExpectIntGT(rsaCertDerSz = (word32)XFREAD(rsaCertDer, 1, sizeof(rsaCertDer),
        fp), 0);
    if (fp != XBADFILE) {
        XFCLOSE(fp);
        fp = XBADFILE;
    }
#endif

    /* good test case - RSA DER cert */
    wc_InitDecodedCert(&decoded, rsaCertDer, rsaCertDerSz, NULL);
    ExpectIntEQ(wc_ParseCert(&decoded, CERT_TYPE, NO_VERIFY, NULL), 0);

    ExpectIntEQ(wc_GetPubKeyDerFromCert(&decoded, keyDer, &keyDerSz), 0);
    ExpectIntGT(keyDerSz, 0);

    /* sanity check, verify we can import DER public key */
    ret = wc_InitRsaKey(&rsaKey, HEAP_HINT);
    ExpectIntEQ(ret, 0);
    ExpectIntEQ(wc_RsaPublicKeyDecode(keyDer, &idx, &rsaKey, keyDerSz), 0);
    if (ret == 0) {
        wc_FreeRsaKey(&rsaKey);
    }

    /* test LENGTH_ONLY_E case */
    keyDerSz = 0;
    ExpectIntEQ(wc_GetPubKeyDerFromCert(&decoded, NULL, &keyDerSz),
        WC_NO_ERR_TRACE(LENGTH_ONLY_E));
    ExpectIntGT(keyDerSz, 0);

    /* bad args: DecodedCert NULL */
    ExpectIntEQ(wc_GetPubKeyDerFromCert(NULL, keyDer, &keyDerSz), WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    /* bad args: output key buff size */
    ExpectIntEQ(wc_GetPubKeyDerFromCert(&decoded, keyDer, NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    /* bad args: zero size output key buffer */
    keyDerSz = 0;
    ExpectIntEQ(ret = wc_GetPubKeyDerFromCert(&decoded, keyDer, &keyDerSz),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    wc_FreeDecodedCert(&decoded);

    /* Certificate Request Tests */
    #if defined(WOLFSSL_CERT_REQ) && !defined(NO_FILESYSTEM)
    {
        XMEMSET(certBuf, 0, sizeof(certBuf));
        ExpectTrue((fp = XFOPEN("./certs/csr.signed.der", "rb")) != XBADFILE);
        ExpectIntGT(certBufSz = (word32)XFREAD(certBuf, 1, certBufSz, fp), 0);
        if (fp != XBADFILE) {
            XFCLOSE(fp);
        }

        wc_InitDecodedCert(&decoded, certBuf, certBufSz, NULL);
        ExpectIntEQ(wc_ParseCert(&decoded, CERTREQ_TYPE, VERIFY, NULL), 0);

        /* good test case - RSA DER certificate request */
        keyDerSz = sizeof(keyDer);
        ExpectIntEQ(ret = wc_GetPubKeyDerFromCert(&decoded, keyDer, &keyDerSz),
            0);
        ExpectIntGT(keyDerSz, 0);

        /* sanity check, verify we can import DER public key */
        ret = wc_InitRsaKey(&rsaKey, HEAP_HINT);
        ExpectIntEQ(ret, 0);
        idx = 0;
        ExpectIntEQ(wc_RsaPublicKeyDecode(keyDer, &idx, &rsaKey, keyDerSz), 0);
        if (ret == 0) {
            wc_FreeRsaKey(&rsaKey);
        }

        wc_FreeDecodedCert(&decoded);
    }
    #endif /* WOLFSSL_CERT_REQ */
#endif /* NO_RSA */

#ifdef HAVE_ECC
    #ifndef USE_CERT_BUFFERS_256
        ExpectTrue((fp2 = XFOPEN("./certs/client-ecc-cert.der", "rb")) !=
            XBADFILE);
        ExpectIntGT(eccCertSz = (word32)XFREAD(eccCert, 1, ONEK_BUF, fp2), 0);
        if (fp2 != XBADFILE) {
            XFCLOSE(fp2);
        }
    #endif

    wc_InitDecodedCert(&decoded, eccCert, eccCertSz, NULL);
    ExpectIntEQ(wc_ParseCert(&decoded, CERT_TYPE, NO_VERIFY, NULL), 0);

    /* good test case - ECC */
    XMEMSET(keyDer, 0, sizeof(keyDer));
    keyDerSz = sizeof(keyDer);
    ExpectIntEQ(wc_GetPubKeyDerFromCert(&decoded, keyDer, &keyDerSz), 0);
    ExpectIntGT(keyDerSz, 0);

    /* sanity check, verify we can import DER public key */
    ret = wc_ecc_init(&eccKey);
    ExpectIntEQ(ret, 0);
    idx = 0; /* reset idx to 0, used above in RSA case */
    ExpectIntEQ(wc_EccPublicKeyDecode(keyDer, &idx, &eccKey, keyDerSz), 0);
    if (ret == 0) {
        wc_ecc_free(&eccKey);
    }

    /* test LENGTH_ONLY_E case */
    keyDerSz = 0;
    ExpectIntEQ(wc_GetPubKeyDerFromCert(&decoded, NULL, &keyDerSz),
        WC_NO_ERR_TRACE(LENGTH_ONLY_E));
    ExpectIntGT(keyDerSz, 0);

    wc_FreeDecodedCert(&decoded);
#endif
#endif /* !NO_RSA || HAVE_ECC */
    return EXPECT_RESULT();
}

static int test_wc_GetSubjectPubKeyInfoDerFromCert(void)
{
    EXPECT_DECLS;
#if !defined(NO_RSA) || defined(HAVE_ECC)
    int ret;
    word32 idx = 0;
    byte keyDer[TWOK_BUF];  /* large enough for up to RSA 2048 */
    word32 keyDerSz = (word32)sizeof(keyDer);
#if !defined(NO_RSA) && defined(WOLFSSL_CERT_REQ) && !defined(NO_FILESYSTEM)
    byte certBuf[6000]; /* for PEM and CSR, client-cert.pem is 5-6kB */
    word32 certBufSz = sizeof(certBuf);
#endif
#if ((!defined(USE_CERT_BUFFERS_2048) && !defined(USE_CERT_BUFFERS_1024)) || \
     defined(WOLFSSL_CERT_REQ)) && !defined(NO_RSA) && !defined(NO_FILESYSTEM)
    XFILE fp = XBADFILE;
#endif
#ifndef NO_RSA
    RsaKey rsaKey;
    #if defined(USE_CERT_BUFFERS_2048)
        byte* rsaCertDer = (byte*)client_cert_der_2048;
        word32 rsaCertDerSz = sizeof_client_cert_der_2048;
    #elif defined(USE_CERT_BUFFERS_1024)
        byte* rsaCertDer = (byte*)client_cert_der_1024;
        word32 rsaCertDerSz = sizeof_client_cert_der_1024;
    #else
        unsigned char rsaCertDer[TWOK_BUF];
        word32 rsaCertDerSz;
    #endif
#endif
#ifdef HAVE_ECC
    ecc_key eccKey;
    #if defined(USE_CERT_BUFFERS_256)
        byte* eccCert = (byte*)cliecc_cert_der_256;
        word32 eccCertSz = sizeof_cliecc_cert_der_256;
    #else
        unsigned char eccCert[ONEK_BUF];
        word32 eccCertSz;
        XFILE fp2 = XBADFILE;
    #endif
#endif

#ifndef NO_RSA

#if !defined(USE_CERT_BUFFERS_1024) && !defined(USE_CERT_BUFFERS_2048)
    ExpectTrue((fp = XFOPEN("./certs/1024/client-cert.der", "rb")) != XBADFILE);
    ExpectIntGT(rsaCertDerSz = (word32)XFREAD(rsaCertDer, 1, sizeof(rsaCertDer),
        fp), 0);
    if (fp != XBADFILE) {
        XFCLOSE(fp);
        fp = XBADFILE;
    }
#endif

    /* good test case - RSA DER cert */
    ExpectIntEQ(wc_GetSubjectPubKeyInfoDerFromCert(rsaCertDer, rsaCertDerSz,
                                                   keyDer, &keyDerSz), 0);
    ExpectIntGT(keyDerSz, 0);

    /* sanity check, verify we can import DER public key */
    ret = wc_InitRsaKey(&rsaKey, HEAP_HINT);
    ExpectIntEQ(ret, 0);
    ExpectIntEQ(wc_RsaPublicKeyDecode(keyDer, &idx, &rsaKey, keyDerSz), 0);
    if (ret == 0) {
        wc_FreeRsaKey(&rsaKey);
    }

    /* bad args: certDer */
    keyDerSz = (word32)sizeof(keyDer);
    ExpectIntEQ(wc_GetSubjectPubKeyInfoDerFromCert(NULL, rsaCertDerSz, keyDer,
                                                   &keyDerSz),
                WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    /* bad args: 0 sized certSz */
    keyDerSz = (word32)sizeof(keyDer);
    ExpectIntEQ(wc_GetSubjectPubKeyInfoDerFromCert(rsaCertDer, 0, keyDer,
                                                   &keyDerSz),
                WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    /* bad args: NULL inout size */
    ExpectIntEQ(ret = wc_GetSubjectPubKeyInfoDerFromCert(rsaCertDer,
                                                         rsaCertDerSz, keyDer,
                                                         NULL),
                WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    /* Certificate Request Tests */
    #if defined(WOLFSSL_CERT_REQ) && !defined(NO_FILESYSTEM)
    {
        XMEMSET(certBuf, 0, sizeof(certBuf));
        ExpectTrue((fp = XFOPEN("./certs/csr.signed.der", "rb")) != XBADFILE);
        ExpectIntGT(certBufSz = (word32)XFREAD(certBuf, 1, certBufSz, fp), 0);
        if (fp != XBADFILE) {
            XFCLOSE(fp);
        }

        /* good test case - RSA DER certificate request */
        keyDerSz = sizeof(keyDer);
        ExpectIntEQ(ret = wc_GetSubjectPubKeyInfoDerFromCert(rsaCertDer,
                                                             rsaCertDerSz,
                                                             keyDer,
                                                             &keyDerSz), 0);
        ExpectIntGT(keyDerSz, 0);

        /* sanity check, verify we can import DER public key */
        ret = wc_InitRsaKey(&rsaKey, HEAP_HINT);
        ExpectIntEQ(ret, 0);
        idx = 0;
        ExpectIntEQ(wc_RsaPublicKeyDecode(keyDer, &idx, &rsaKey, keyDerSz), 0);
        if (ret == 0) {
            wc_FreeRsaKey(&rsaKey);
        }
    }
    #endif /* WOLFSSL_CERT_REQ */
#endif /* NO_RSA */

#ifdef HAVE_ECC
    #ifndef USE_CERT_BUFFERS_256
        ExpectTrue((fp2 = XFOPEN("./certs/client-ecc-cert.der", "rb")) !=
            XBADFILE);
        ExpectIntGT(eccCertSz = (word32)XFREAD(eccCert, 1, ONEK_BUF, fp2), 0);
        if (fp2 != XBADFILE) {
            XFCLOSE(fp2);
        }
    #endif

    /* good test case - ECC */
    XMEMSET(keyDer, 0, sizeof(keyDer));
    keyDerSz = sizeof(keyDer);
    ExpectIntEQ(wc_GetSubjectPubKeyInfoDerFromCert(eccCert, eccCertSz, keyDer,
                                                   &keyDerSz), 0);
    ExpectIntGT(keyDerSz, 0);

    /* sanity check, verify we can import DER public key */
    ret = wc_ecc_init(&eccKey);
    ExpectIntEQ(ret, 0);
    idx = 0; /* reset idx to 0, used above in RSA case */
    ExpectIntEQ(wc_EccPublicKeyDecode(keyDer, &idx, &eccKey, keyDerSz), 0);
    if (ret == 0) {
        wc_ecc_free(&eccKey);
    }

#endif
#endif /* !NO_RSA || HAVE_ECC */
    return EXPECT_RESULT();
}

static int test_wc_CheckCertSigPubKey(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_FILESYSTEM) && \
    !defined(NO_RSA) && defined(WOLFSSL_PEM_TO_DER) && defined(HAVE_ECC)
    int ret = 0;
    const char* ca_cert = "./certs/ca-cert.pem";
    byte* cert_buf = NULL;
    size_t cert_sz = 0;
    byte* cert_der = NULL;
    word32 cert_dersz = 0;
    byte keyDer[TWOK_BUF];  /* large enough for up to RSA 2048 */
    word32 keyDerSz = (word32)sizeof(keyDer);
    DecodedCert decoded;

    ExpectIntEQ(load_file(ca_cert, &cert_buf, &cert_sz), 0);
    cert_dersz = (word32)cert_sz; /* DER will be smaller than PEM */
    ExpectNotNull(cert_der = (byte*)malloc(cert_dersz));
    ExpectIntGE(ret = wc_CertPemToDer(cert_buf, (int)cert_sz, cert_der,
        (int)cert_dersz, CERT_TYPE), 0);

    wc_InitDecodedCert(&decoded, cert_der, cert_dersz, NULL);
    ExpectIntEQ(wc_ParseCert(&decoded, CERT_TYPE, NO_VERIFY, NULL), 0);

    ExpectIntEQ(wc_GetPubKeyDerFromCert(&decoded, keyDer, &keyDerSz), 0);
    ExpectIntGT(keyDerSz, 0);

    /* Good test case. */
    ExpectIntEQ(wc_CheckCertSigPubKey(cert_der, cert_dersz, NULL, keyDer,
        keyDerSz, RSAk), 0);

    /* No certificate. */
    ExpectIntEQ(wc_CheckCertSigPubKey(NULL, cert_dersz, NULL, keyDer, keyDerSz,
        ECDSAk), WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    /* Bad cert size. */
    ExpectIntNE(ret = wc_CheckCertSigPubKey(cert_der, 0, NULL, keyDer, keyDerSz,
        RSAk), 0);
    ExpectTrue(ret == WC_NO_ERR_TRACE(ASN_PARSE_E) || ret == WC_NO_ERR_TRACE(BUFFER_E));

    /* No public key. */
    ExpectIntEQ(wc_CheckCertSigPubKey(cert_der, cert_dersz, NULL, NULL,
        keyDerSz, RSAk), WC_NO_ERR_TRACE(ASN_NO_SIGNER_E));

    /* Bad public key size. */
    ExpectIntEQ(wc_CheckCertSigPubKey(cert_der, cert_dersz, NULL, keyDer, 0,
        RSAk), WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    /* Wrong aglo. */
    ExpectIntEQ(wc_CheckCertSigPubKey(cert_der, cert_dersz, NULL, keyDer,
        keyDerSz, ECDSAk), WC_NO_ERR_TRACE(ASN_PARSE_E));

    wc_FreeDecodedCert(&decoded);
    if (cert_der != NULL)
        free(cert_der);
    if (cert_buf != NULL)
        free(cert_buf);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_ext_d2i(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS)
    X509* x509 = NULL;

    ExpectNotNull(x509 = wolfSSL_X509_new());

    ExpectNull(wolfSSL_X509_get_ext_d2i(x509, NID_basic_constraints,
        NULL, NULL));
    ExpectNull(wolfSSL_X509_get_ext_d2i(x509, NID_subject_alt_name,
        NULL, NULL));
    ExpectNull(wolfSSL_X509_get_ext_d2i(x509, NID_authority_key_identifier,
        NULL, NULL));
    ExpectNull(wolfSSL_X509_get_ext_d2i(x509, NID_subject_key_identifier,
        NULL, NULL));
    ExpectNull(wolfSSL_X509_get_ext_d2i(x509, NID_key_usage,
        NULL, NULL));
    ExpectNull(wolfSSL_X509_get_ext_d2i(x509, NID_crl_distribution_points,
        NULL, NULL));
    ExpectNull(wolfSSL_X509_get_ext_d2i(x509, NID_ext_key_usage,
        NULL, NULL));
    ExpectNull(wolfSSL_X509_get_ext_d2i(x509, NID_info_access,
        NULL, NULL));
    ExpectNull(wolfSSL_X509_get_ext_d2i(x509, NID_certificate_policies,
        NULL, NULL));
    /* Invalid NID for an extension. */
    ExpectNull(wolfSSL_X509_get_ext_d2i(x509, NID_description,
        NULL, NULL));

    wolfSSL_X509_free(x509);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_certs(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_FILESYSTEM) && \
    !defined(NO_TLS) && !defined(NO_RSA)
    X509*  x509ext = NULL;
    X509*  x509 = NULL;
#ifdef OPENSSL_ALL
    WOLFSSL_X509_EXTENSION* ext = NULL;
    ASN1_OBJECT* obj = NULL;
#endif
    WOLFSSL*     ssl = NULL;
    WOLFSSL_CTX* ctx = NULL;
    STACK_OF(ASN1_OBJECT)* sk = NULL;
    ASN1_STRING* asn1_str = NULL;
    AUTHORITY_KEYID* akey = NULL;
    WOLFSSL_STACK* skid = NULL;
    BASIC_CONSTRAINTS* bc = NULL;
    int crit = 0;

#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = SSL_CTX_new(SSLv23_server_method()));
#else
    ExpectNotNull(ctx = SSL_CTX_new(SSLv23_client_method()));
#endif
    ExpectTrue(SSL_CTX_use_certificate_file(ctx, svrCertFile, SSL_FILETYPE_PEM));
    ExpectTrue(SSL_CTX_use_PrivateKey_file(ctx, svrKeyFile, SSL_FILETYPE_PEM));
    ExpectTrue(SSL_CTX_use_PrivateKey_file(ctx, cliKeyFile, SSL_FILETYPE_PEM));
    #if !defined(NO_CHECK_PRIVATE_KEY)
    ExpectIntEQ(SSL_CTX_check_private_key(ctx), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    #endif
    ExpectTrue(SSL_CTX_use_PrivateKey_file(ctx, svrKeyFile, SSL_FILETYPE_PEM));
    #if !defined(NO_CHECK_PRIVATE_KEY)
    ExpectIntEQ(SSL_CTX_check_private_key(ctx), SSL_SUCCESS);
    #endif
    ExpectNotNull(ssl = SSL_new(ctx));

    /* Invalid parameters. */
    ExpectIntEQ(SSL_use_certificate_file(NULL, NULL, WOLFSSL_FILETYPE_PEM),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(SSL_use_certificate_file(ssl, NULL, WOLFSSL_FILETYPE_PEM),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(SSL_use_certificate_file(NULL, "./certs/server-cert.pem",
        WOLFSSL_FILETYPE_PEM), WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    #if !defined(NO_CHECK_PRIVATE_KEY)
    ExpectIntEQ(wolfSSL_check_private_key(ssl), WOLFSSL_SUCCESS);
    #endif

    #ifdef HAVE_PK_CALLBACKS
    ExpectIntEQ((int)SSL_set_tlsext_debug_arg(ssl, NULL), WOLFSSL_SUCCESS);
    #endif /* HAVE_PK_CALLBACKS */

    /* Invalid parameters. */
    ExpectNotNull(x509 = wolfSSL_X509_new());
    ExpectIntEQ(SSL_use_certificate(NULL, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(SSL_use_certificate(ssl, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(SSL_use_certificate(NULL, x509), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    /* No data in certificate. */
    ExpectIntEQ(SSL_use_certificate(ssl, x509), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    wolfSSL_X509_free(x509);
    x509 = NULL;

    /* create and use x509 */
    ExpectNull(wolfSSL_X509_load_certificate_file(cliCertFileExt, -1));
    ExpectNull(wolfSSL_X509_load_certificate_file("/tmp/badfile",
        WOLFSSL_FILETYPE_PEM));
    ExpectNull(wolfSSL_X509_load_certificate_file(NULL, WOLFSSL_FILETYPE_PEM));
    ExpectNull(wolfSSL_X509_load_certificate_file(cliCertFileExt,
        WOLFSSL_FILETYPE_ASN1));
#ifdef OPENSSL_ALL
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(cliCertFile,
        WOLFSSL_FILETYPE_PEM));
#endif
    ExpectNotNull(x509ext = wolfSSL_X509_load_certificate_file(cliCertFileExt,
        WOLFSSL_FILETYPE_PEM));
    ExpectIntEQ(SSL_use_certificate(ssl, x509ext), WOLFSSL_SUCCESS);

    #if !defined(NO_CHECK_PRIVATE_KEY)
    /* with loading in a new cert the check on private key should now fail */
    ExpectIntNE(wolfSSL_check_private_key(ssl), WOLFSSL_SUCCESS);
    #endif


    #if defined(USE_CERT_BUFFERS_2048)
        /* Invalid parameters. */
        ExpectIntEQ(SSL_use_certificate_ASN1(NULL, NULL, 0), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
        ExpectIntEQ(SSL_use_certificate_ASN1(ssl, NULL, 0), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
        ExpectIntEQ(SSL_use_certificate_ASN1(NULL,
                (unsigned char*)server_cert_der_2048, 0), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
        /* No data. */
        ExpectIntEQ(SSL_use_certificate_ASN1(ssl,
                (unsigned char*)server_cert_der_2048, 0), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

        ExpectIntEQ(SSL_use_certificate_ASN1(ssl,
                                  (unsigned char*)server_cert_der_2048,
                                  sizeof_server_cert_der_2048), WOLFSSL_SUCCESS);
    #endif

    #if !defined(NO_SHA) && !defined(NO_SHA256) && !defined(NO_PWDBASED)
    /************* Get Digest of Certificate ******************/
    {
        byte   digest[64]; /* max digest size */
        word32 digestSz;
        X509*  x509Empty = NULL;

        XMEMSET(digest, 0, sizeof(digest));
        ExpectIntEQ(X509_digest(NULL, wolfSSL_EVP_sha1(), digest, &digestSz),
                    WOLFSSL_FAILURE);
        ExpectIntEQ(X509_digest(x509ext, NULL, digest, &digestSz),
                    WOLFSSL_FAILURE);
        ExpectIntEQ(X509_digest(x509ext, wolfSSL_EVP_sha1(), NULL, &digestSz),
                    WOLFSSL_FAILURE);
        ExpectIntEQ(X509_digest(x509ext, wolfSSL_EVP_sha1(), digest, NULL),
                    WOLFSSL_SUCCESS);
        ExpectIntEQ(X509_digest(x509ext, wolfSSL_EVP_sha1(), digest, &digestSz),
                    WOLFSSL_SUCCESS);
        ExpectIntEQ(X509_digest(x509ext, wolfSSL_EVP_sha256(), digest,
                    &digestSz), WOLFSSL_SUCCESS);

        ExpectNotNull(x509Empty = wolfSSL_X509_new());
        ExpectIntEQ(X509_digest(x509Empty, wolfSSL_EVP_sha256(), digest,
                    &digestSz), WOLFSSL_FAILURE);
        wolfSSL_X509_free(x509Empty);
    }
    #endif /* !NO_SHA && !NO_SHA256 && !NO_PWDBASED */

    #if !defined(NO_SHA) && !defined(NO_SHA256) && !defined(NO_PWDBASED)
    /************* Get Digest of Certificate ******************/
    {
        byte   digest[64]; /* max digest size */
        word32 digestSz;
        X509*  x509Empty = NULL;

        XMEMSET(digest, 0, sizeof(digest));
        ExpectIntEQ(X509_pubkey_digest(NULL, wolfSSL_EVP_sha1(), digest,
                    &digestSz), WOLFSSL_FAILURE);
        ExpectIntEQ(X509_pubkey_digest(x509ext, NULL, digest, &digestSz),
                    WOLFSSL_FAILURE);
        ExpectIntEQ(X509_pubkey_digest(x509ext, wolfSSL_EVP_sha1(), NULL,
                    &digestSz), WOLFSSL_FAILURE);
        ExpectIntEQ(X509_pubkey_digest(x509ext, wolfSSL_EVP_sha1(), digest,
                    NULL), WOLFSSL_SUCCESS);
        ExpectIntEQ(X509_pubkey_digest(x509ext, wolfSSL_EVP_sha1(), digest,
                    &digestSz), WOLFSSL_SUCCESS);
        ExpectIntEQ(X509_pubkey_digest(x509ext, wolfSSL_EVP_sha256(), digest,
                    &digestSz), WOLFSSL_SUCCESS);

        ExpectNotNull(x509Empty = wolfSSL_X509_new());
        ExpectIntEQ(X509_pubkey_digest(x509Empty, wolfSSL_EVP_sha256(), digest,
                    &digestSz), WOLFSSL_FAILURE);
        wolfSSL_X509_free(x509Empty);
    }
    #endif /* !NO_SHA && !NO_SHA256 && !NO_PWDBASED */

    /* test and checkout X509 extensions */
    ExpectNotNull(bc = (BASIC_CONSTRAINTS*)X509_get_ext_d2i(x509ext,
        NID_basic_constraints, NULL, NULL));
    BASIC_CONSTRAINTS_free(bc);
    bc = NULL;
    ExpectNotNull(bc = (BASIC_CONSTRAINTS*)X509_get_ext_d2i(x509ext,
        NID_basic_constraints, &crit, NULL));
    ExpectIntEQ(crit, 0);

#ifdef OPENSSL_ALL
    ExpectNull(X509V3_EXT_i2d(NID_basic_constraints, crit, NULL));
    {
        int i;
        int unsupportedNid[] = {
            0,
            NID_inhibit_any_policy,
            NID_certificate_policies,
            NID_policy_mappings,
            NID_name_constraints,
            NID_policy_constraints,
            NID_crl_distribution_points
        };
        int unsupportedNidCnt = (int)(sizeof(unsupportedNid) /
                                      sizeof(*unsupportedNid));

        for (i = 0; i < unsupportedNidCnt; i++) {
            ExpectNotNull(ext = X509V3_EXT_i2d(unsupportedNid[i], crit, bc));
            X509_EXTENSION_free(ext);
            ext = NULL;
        }
    }
    ExpectNotNull(ext = X509V3_EXT_i2d(NID_basic_constraints, crit, bc));
    X509_EXTENSION_free(ext);
    ext = NULL;

    ExpectNotNull(ext = X509_EXTENSION_new());
    ExpectIntEQ(X509_EXTENSION_set_critical(NULL, 1), WOLFSSL_FAILURE);
    ExpectIntEQ(X509_EXTENSION_set_critical(ext, 1), WOLFSSL_SUCCESS);
    ExpectNotNull(obj = OBJ_nid2obj(NID_basic_constraints));
    ExpectIntEQ(X509_EXTENSION_set_object(NULL, NULL), SSL_FAILURE);
    ExpectIntEQ(X509_EXTENSION_set_object(NULL, obj), SSL_FAILURE);
    ExpectIntEQ(X509_EXTENSION_set_object(ext, NULL), SSL_SUCCESS);
    ExpectIntEQ(X509_EXTENSION_set_object(ext, obj), SSL_SUCCESS);
    /* Check old object is being freed. */
    ExpectIntEQ(X509_EXTENSION_set_object(ext, obj), SSL_SUCCESS);
    ASN1_OBJECT_free(obj);
    obj = NULL;
    X509_EXTENSION_free(ext);
    ext = NULL;

    ExpectNotNull(ext = X509_EXTENSION_new());
    ExpectIntEQ(X509_EXTENSION_set_critical(ext, 0), WOLFSSL_SUCCESS);
    ExpectIntEQ(X509_EXTENSION_set_data(ext, NULL),
            WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectNotNull(asn1_str = (ASN1_STRING*)X509_get_ext_d2i(x509ext,
            NID_key_usage, NULL, NULL));
    ASN1_STRING_free(asn1_str);
    asn1_str = NULL;
    ExpectNotNull(asn1_str = (ASN1_STRING*)X509_get_ext_d2i(x509ext,
            NID_key_usage, &crit, NULL));
    ExpectIntEQ(X509_EXTENSION_set_data(ext, asn1_str), SSL_SUCCESS);
    ExpectIntEQ(X509_EXTENSION_set_data(ext, asn1_str), SSL_SUCCESS);
    ASN1_STRING_free(asn1_str); /* X509_EXTENSION_set_data has made a copy
                                 * and X509_get_ext_d2i has created new */
    asn1_str = NULL;
    X509_EXTENSION_free(ext);
    ext = NULL;

#endif
    BASIC_CONSTRAINTS_free(NULL);
    BASIC_CONSTRAINTS_free(bc);
    bc = NULL;

    ExpectNotNull(asn1_str = (ASN1_STRING*)X509_get_ext_d2i(x509ext,
        NID_key_usage, NULL, NULL));
    ASN1_STRING_free(asn1_str);
    asn1_str = NULL;
    ExpectNotNull(asn1_str = (ASN1_STRING*)X509_get_ext_d2i(x509ext,
        NID_key_usage, &crit, NULL));
    ExpectIntEQ(crit, 1);
    ExpectIntEQ(asn1_str->type, NID_key_usage);
#ifdef OPENSSL_ALL
    ExpectNotNull(ext = X509V3_EXT_i2d(NID_key_usage, crit, asn1_str));
    X509_EXTENSION_free(ext);
    ext = NULL;
#endif
    ASN1_STRING_free(asn1_str);
    asn1_str = NULL;

#ifdef OPENSSL_ALL
    ExpectNotNull(sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(x509,
        NID_ext_key_usage, NULL, NULL));
    EXTENDED_KEY_USAGE_free(NULL);
    EXTENDED_KEY_USAGE_free(sk);
    sk = NULL;
    ExpectNotNull(sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(x509,
        NID_ext_key_usage, &crit, NULL));
    ExpectNotNull(ext = X509V3_EXT_i2d(NID_ext_key_usage, crit, sk));
    X509_EXTENSION_free(ext);
    ext = NULL;
    EXTENDED_KEY_USAGE_free(sk);
    sk = NULL;
#else
    sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(x509ext, NID_ext_key_usage,
            &crit, NULL);
    ExpectNull(sk);
#endif

    ExpectNotNull(akey = (AUTHORITY_KEYID*)X509_get_ext_d2i(x509ext,
        NID_authority_key_identifier, NULL, NULL));
    wolfSSL_AUTHORITY_KEYID_free(NULL);
    wolfSSL_AUTHORITY_KEYID_free(akey);
    akey = NULL;
    ExpectNotNull(akey = (AUTHORITY_KEYID*)X509_get_ext_d2i(x509ext,
        NID_authority_key_identifier, &crit, NULL));
#ifdef OPENSSL_ALL
    ExpectNotNull(ext = X509V3_EXT_i2d(NID_authority_key_identifier, crit,
        akey));
    X509_EXTENSION_free(ext);
    ext = NULL;
#endif
    wolfSSL_AUTHORITY_KEYID_free(akey);
    akey = NULL;

    ExpectNotNull(skid = (WOLFSSL_STACK*)X509_get_ext_d2i(x509ext,
        NID_subject_key_identifier, NULL, NULL));
    wolfSSL_sk_ASN1_OBJECT_pop_free(skid, wolfSSL_ASN1_OBJECT_free);
    skid = NULL;
    ExpectNotNull(skid = (WOLFSSL_STACK*)X509_get_ext_d2i(x509ext,
        NID_subject_key_identifier, &crit, NULL));
#ifdef OPENSSL_ALL
    ExpectNotNull(ext = X509V3_EXT_i2d(NID_subject_key_identifier, crit,
        skid));
    X509_EXTENSION_free(ext);
    ext = NULL;
#endif
    wolfSSL_sk_ASN1_OBJECT_pop_free(skid, wolfSSL_ASN1_OBJECT_free);
    skid = NULL;

    /* NID not yet supported */
    ExpectNull(sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(x509ext,
        NID_private_key_usage_period, &crit, NULL));
    ExpectIntEQ(crit, -1);
    sk_ASN1_OBJECT_free(sk);
    sk = NULL;

    ExpectNotNull(sk = (STACK_OF(GENERAL_NAME)*)X509_get_ext_d2i(x509ext,
        NID_subject_alt_name, NULL, NULL));
    sk_GENERAL_NAME_free(sk);
    sk = NULL;
    ExpectNotNull(sk = (STACK_OF(GENERAL_NAME)*)X509_get_ext_d2i(x509ext,
        NID_subject_alt_name, &crit, NULL));
    {
        int i;
        for (i = 0; i < sk_GENERAL_NAME_num(sk); i++) {
            GENERAL_NAME* gen = sk_GENERAL_NAME_value(sk, i);
            ExpectIntEQ(gen->type, GEN_DNS);
            ExpectIntEQ(gen->d.dNSName->type, V_ASN1_IA5STRING);
        }
    }
    sk_GENERAL_NAME_free(sk);
    sk = NULL;

    /* NID not yet supported */
    ExpectNull(sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(x509ext,
        NID_issuer_alt_name, &crit, NULL));
    ExpectIntEQ(crit, -1);
    sk_ASN1_OBJECT_free(sk);
    sk = NULL;

    /* NID not yet supported */
    ExpectNull(sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(x509ext,
        NID_info_access, &crit, NULL));
    sk_ASN1_OBJECT_free(sk);
    sk = NULL;

    /* NID not yet supported */
    ExpectNull(sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(x509ext,
        NID_sinfo_access, &crit, NULL));
    ExpectIntEQ(crit, -1);
    sk_ASN1_OBJECT_free(sk);
    sk = NULL;

    /* NID not yet supported */
    ExpectNull(sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(x509ext,
        NID_name_constraints, &crit, NULL));
    ExpectIntEQ(crit, -1);
    sk_ASN1_OBJECT_free(sk);
    sk = NULL;

    /* no cert policy set */
    ExpectNull(sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(x509ext,
            NID_certificate_policies, &crit, NULL));
    sk_ASN1_OBJECT_free(sk);
    sk = NULL;

    /* NID not yet supported */
    ExpectNull(sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(x509ext,
        NID_policy_mappings, &crit, NULL));
    ExpectIntEQ(crit, -1);
    sk_ASN1_OBJECT_free(sk);
    sk = NULL;

    /* NID not yet supported */
    ExpectNull(sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(x509ext,
        NID_policy_constraints, &crit, NULL));
    ExpectIntEQ(crit, -1);
    sk_ASN1_OBJECT_free(sk);
    sk = NULL;

    /* NID not yet supported */
    ExpectNull(sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(x509ext,
        NID_inhibit_any_policy, &crit, NULL));
    ExpectIntEQ(crit, -1);
    sk_ASN1_OBJECT_free(sk);
    sk = NULL;

    /* NID not yet supported */
    ExpectNull(sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(x509ext,
        NID_tlsfeature, &crit, NULL));
    ExpectIntEQ(crit, -1);
    sk_ASN1_OBJECT_free(sk);
    sk = NULL;

    /* test invalid cases */
    crit = 0;
    ExpectNull(sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(x509ext, -1, &crit,
        NULL));
    ExpectIntEQ(crit, -1);
    /* NULL passed for criticality. */
    ExpectNull(sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(NULL,
        NID_tlsfeature, NULL, NULL));

    ExpectIntEQ(SSL_get_hit(ssl), 0);
#ifdef OPENSSL_ALL
    X509_free(x509);
#endif
    X509_free(x509ext);
    SSL_free(ssl);
    SSL_CTX_free(ctx);
#endif /* OPENSSL_EXTRA && !NO_CERTS */
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_check_private_key(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_RSA) && \
        defined(USE_CERT_BUFFERS_2048) && !defined(NO_CHECK_PRIVATE_KEY) && \
        !defined(NO_FILESYSTEM)
    X509*  x509 = NULL;
    EVP_PKEY* pkey = NULL;
    const byte* key;

    /* Check with correct key */
    ExpectNotNull((x509 = X509_load_certificate_file(cliCertFile,
        SSL_FILETYPE_PEM)));
    key = client_key_der_2048;
    ExpectNotNull(d2i_PrivateKey(EVP_PKEY_RSA, &pkey, &key,
        (long)sizeof_client_key_der_2048));
    ExpectIntEQ(X509_check_private_key(x509, pkey), 1);
    EVP_PKEY_free(pkey);
    pkey = NULL;

    /* Check with wrong key */
    key = server_key_der_2048;
    ExpectNotNull(d2i_PrivateKey(EVP_PKEY_RSA, &pkey, &key,
        (long)sizeof_server_key_der_2048));
    ExpectIntEQ(X509_check_private_key(x509, pkey), 0);

    /* test for incorrect parameter */
    ExpectIntEQ(X509_check_private_key(NULL, pkey), 0);
    ExpectIntEQ(X509_check_private_key(x509, NULL), 0);
    ExpectIntEQ(X509_check_private_key(NULL, NULL), 0);

    EVP_PKEY_free(pkey);
    X509_free(x509);
#endif
    return EXPECT_RESULT();
}


static int test_wolfSSL_private_keys(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_TLS) && \
   !defined(NO_FILESYSTEM)
#if !defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)
    WOLFSSL*     ssl = NULL;
    WOLFSSL_CTX* ctx = NULL;
    EVP_PKEY* pkey = NULL;

    OpenSSL_add_all_digests();
    OpenSSL_add_all_algorithms();

#ifndef NO_RSA
    #ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_server_method()));
    #else
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_client_method()));
    #endif
    ExpectTrue(SSL_CTX_use_PrivateKey_file(ctx, svrKeyFile,
        WOLFSSL_FILETYPE_PEM));
    /* Have to load a cert before you can check the private key against that
     * certificates public key! */
    #if !defined(NO_CHECK_PRIVATE_KEY)
    ExpectIntEQ(wolfSSL_CTX_check_private_key(ctx), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    #endif
    ExpectTrue(SSL_CTX_use_certificate_file(ctx, svrCertFile,
        WOLFSSL_FILETYPE_PEM));
    #if !defined(NO_CHECK_PRIVATE_KEY)
    ExpectIntEQ(wolfSSL_CTX_check_private_key(ctx), WOLFSSL_SUCCESS);
    #endif
    ExpectNotNull(ssl = SSL_new(ctx));

    #if !defined(NO_CHECK_PRIVATE_KEY)
    ExpectIntEQ(wolfSSL_check_private_key(ssl), WOLFSSL_SUCCESS);
    #endif

    /* Invalid parameters. */
    ExpectIntEQ(SSL_use_PrivateKey_file(NULL, NULL, WOLFSSL_FILETYPE_PEM),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(SSL_use_PrivateKey_file(NULL, svrKeyFile, WOLFSSL_FILETYPE_PEM),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(SSL_use_PrivateKey_file(ssl, NULL, WOLFSSL_FILETYPE_PEM),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

#ifdef USE_CERT_BUFFERS_2048
    {
    const unsigned char* server_key = (const unsigned char*)server_key_der_2048;
    unsigned char buf[FOURK_BUF];
    word32 bufSz;

    /* Invalid parameters. */
    ExpectIntEQ(SSL_use_RSAPrivateKey_ASN1(NULL, NULL, 0), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(SSL_use_RSAPrivateKey_ASN1(ssl, NULL, 0), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(SSL_use_RSAPrivateKey_ASN1(NULL,
        (unsigned char*)client_key_der_2048, 0), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(SSL_use_PrivateKey_ASN1(0, NULL, NULL, 0), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(SSL_use_PrivateKey_ASN1(0, ssl, NULL, 0), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(SSL_use_PrivateKey_ASN1(0, NULL, (unsigned char*)server_key, 0),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(SSL_CTX_use_PrivateKey_ASN1(0, NULL, NULL, 0), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(SSL_CTX_use_PrivateKey_ASN1(0, ctx, NULL, 0), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(SSL_CTX_use_PrivateKey_ASN1(0, NULL, (unsigned char*)server_key,
        0), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

    ExpectIntEQ(SSL_use_RSAPrivateKey_ASN1(ssl,
                (unsigned char*)client_key_der_2048,
                sizeof_client_key_der_2048), WOLFSSL_SUCCESS);
    #if !defined(NO_CHECK_PRIVATE_KEY)
    /* Should mismatch now that a different private key loaded */
    ExpectIntNE(wolfSSL_check_private_key(ssl), WOLFSSL_SUCCESS);
    #endif

    ExpectIntEQ(SSL_use_PrivateKey_ASN1(0, ssl,
                (unsigned char*)server_key,
                sizeof_server_key_der_2048), WOLFSSL_SUCCESS);
    #if !defined(NO_CHECK_PRIVATE_KEY)
    /* After loading back in DER format of original key, should match */
    ExpectIntEQ(wolfSSL_check_private_key(ssl), WOLFSSL_SUCCESS);
    #endif

    /* test loading private key to the WOLFSSL_CTX */
    ExpectIntEQ(SSL_CTX_use_PrivateKey_ASN1(0, ctx,
                (unsigned char*)client_key_der_2048,
                sizeof_client_key_der_2048), WOLFSSL_SUCCESS);

    #if !defined(NO_CHECK_PRIVATE_KEY)
    /* Should mismatch now that a different private key loaded */
    ExpectIntNE(wolfSSL_CTX_check_private_key(ctx), WOLFSSL_SUCCESS);
    #endif

    ExpectIntEQ(SSL_CTX_use_PrivateKey_ASN1(0, ctx,
                (unsigned char*)server_key,
                sizeof_server_key_der_2048), WOLFSSL_SUCCESS);
    #if !defined(NO_CHECK_PRIVATE_KEY)
    /* After loading back in DER format of original key, should match */
    ExpectIntEQ(wolfSSL_CTX_check_private_key(ctx), WOLFSSL_SUCCESS);
    #endif

    /* Invalid parameters. */
    ExpectNotNull(pkey = wolfSSL_EVP_PKEY_new());
    ExpectIntEQ(SSL_use_PrivateKey(NULL, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(SSL_use_PrivateKey(ssl, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(SSL_use_PrivateKey(NULL, pkey), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    /* pkey is empty - no key data to use. */
    ExpectIntEQ(SSL_use_PrivateKey(ssl, pkey), WC_NO_ERR_TRACE(ASN_PARSE_E));
    wolfSSL_EVP_PKEY_free(pkey);
    pkey = NULL;

    /* set PKEY and test again */
    ExpectNotNull(wolfSSL_d2i_PrivateKey(EVP_PKEY_RSA, &pkey,
                &server_key, (long)sizeof_server_key_der_2048));
    ExpectIntEQ(SSL_use_PrivateKey(ssl, pkey), WOLFSSL_SUCCESS);

    /* reuse PKEY structure and test
     * this should be checked with a memory management sanity checker */
    ExpectFalse(server_key == (const unsigned char*)server_key_der_2048);
    server_key = (const unsigned char*)server_key_der_2048;
    ExpectNotNull(wolfSSL_d2i_PrivateKey(EVP_PKEY_RSA, &pkey,
                &server_key, (long)sizeof_server_key_der_2048));
    ExpectIntEQ(SSL_use_PrivateKey(ssl, pkey), WOLFSSL_SUCCESS);

    /* check striping PKCS8 header with wolfSSL_d2i_PrivateKey */
    bufSz = FOURK_BUF;
    ExpectIntGT((bufSz = (word32)wc_CreatePKCS8Key(buf, &bufSz,
                    (byte*)server_key_der_2048, sizeof_server_key_der_2048,
                    RSAk, NULL, 0)), 0);
    server_key = (const unsigned char*)buf;
    ExpectNotNull(wolfSSL_d2i_PrivateKey(EVP_PKEY_RSA, &pkey, &server_key,
                (long)bufSz));
    }
#endif


    EVP_PKEY_free(pkey);
    pkey = NULL;
    SSL_free(ssl); /* frees x509 also since loaded into ssl */
    ssl = NULL;
    SSL_CTX_free(ctx);
    ctx = NULL;
#endif /* end of RSA private key match tests */


#ifdef HAVE_ECC
    #ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_server_method()));
    #else
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_client_method()));
    #endif
    ExpectTrue(SSL_CTX_use_certificate_file(ctx, eccCertFile,
                                                         WOLFSSL_FILETYPE_PEM));
    ExpectTrue(SSL_CTX_use_PrivateKey_file(ctx, eccKeyFile,
                                                         WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(ssl = SSL_new(ctx));

    #if !defined(NO_CHECK_PRIVATE_KEY)
    ExpectIntEQ(wolfSSL_check_private_key(ssl), WOLFSSL_SUCCESS);
    #endif
    SSL_free(ssl);
    ssl = NULL;


    ExpectTrue(SSL_CTX_use_PrivateKey_file(ctx, cliEccKeyFile,
                                                         WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(ssl = SSL_new(ctx));

    #ifdef WOLFSSL_VALIDATE_ECC_IMPORT
    ExpectIntNE(wolfSSL_check_private_key(ssl), WOLFSSL_SUCCESS);
    #endif

    SSL_free(ssl);
    ssl = NULL;
    SSL_CTX_free(ctx);
    ctx = NULL;
#endif /* end of ECC private key match tests */

#if defined(HAVE_ED25519) && defined(HAVE_ED25519_KEY_IMPORT)
    #ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_server_method()));
    #else
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_client_method()));
    #endif
    ExpectTrue(SSL_CTX_use_certificate_file(ctx, edCertFile,
                                                         WOLFSSL_FILETYPE_PEM));
    ExpectTrue(SSL_CTX_use_PrivateKey_file(ctx, edKeyFile,
                                                         WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(ssl = SSL_new(ctx));

    #if !defined(NO_CHECK_PRIVATE_KEY)
    ExpectIntEQ(wolfSSL_check_private_key(ssl), WOLFSSL_SUCCESS);
    #endif
    SSL_free(ssl);
    ssl = NULL;


    ExpectTrue(SSL_CTX_use_PrivateKey_file(ctx, cliEdKeyFile,
                                                         WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(ssl = SSL_new(ctx));

    #if !defined(NO_CHECK_PRIVATE_KEY)
    #ifdef HAVE_ED25519_MAKE_KEY
    ExpectIntNE(wolfSSL_check_private_key(ssl), WOLFSSL_SUCCESS);
    #else
    ExpectIntEQ(wolfSSL_check_private_key(ssl), WOLFSSL_SUCCESS);
    #endif
    #endif

    SSL_free(ssl);
    ssl = NULL;
    SSL_CTX_free(ctx);
    ctx = NULL;
#endif /* end of Ed25519 private key match tests */

#if defined(HAVE_ED448) && defined(HAVE_ED448_KEY_IMPORT)
    #ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_server_method()));
    #else
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_client_method()));
    #endif
    ExpectTrue(SSL_CTX_use_certificate_file(ctx, ed448CertFile,
                                                         WOLFSSL_FILETYPE_PEM));
    ExpectTrue(SSL_CTX_use_PrivateKey_file(ctx, ed448KeyFile,
                                                         WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(ssl = SSL_new(ctx));

    #if !defined(NO_CHECK_PRIVATE_KEY)
    ExpectIntEQ(wolfSSL_check_private_key(ssl), WOLFSSL_SUCCESS);
    #endif
    SSL_free(ssl);
    ssl = NULL;


    ExpectTrue(SSL_CTX_use_PrivateKey_file(ctx, cliEd448KeyFile,
                                                         WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(ssl = SSL_new(ctx));

    #if !defined(NO_CHECK_PRIVATE_KEY)
    ExpectIntNE(wolfSSL_check_private_key(ssl), WOLFSSL_SUCCESS);
    #endif

    SSL_free(ssl);
    ssl = NULL;
    SSL_CTX_free(ctx);
    ctx = NULL;
#endif /* end of Ed448 private key match tests */

    EVP_cleanup();

    /* test existence of no-op macros in wolfssl/openssl/ssl.h */
    CONF_modules_free();
    ENGINE_cleanup();
    CONF_modules_unload();

    (void)ssl;
    (void)ctx;
    (void)pkey;
#endif /* !NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER */
#endif /* defined(OPENSSL_EXTRA) && !defined(NO_CERTS) */
    return EXPECT_RESULT();
}

static int test_wolfSSL_PEM_def_callback(void)
{
    EXPECT_DECLS;
#ifdef OPENSSL_EXTRA
    char buf[10];
    const char* defpwd = "DEF PWD";
    int defpwdLen = (int)XSTRLEN(defpwd);
    int smallLen = 1;

    /* Bad parameters. */
    ExpectIntEQ(wolfSSL_PEM_def_callback(NULL, sizeof(buf), 0, NULL), 0);
    ExpectIntEQ(wolfSSL_PEM_def_callback(NULL, sizeof(buf), 0, (void*)defpwd),
        0);
    ExpectIntEQ(wolfSSL_PEM_def_callback(buf, sizeof(buf), 0, NULL), 0);

    XMEMSET(buf, 0, sizeof(buf));
    ExpectIntEQ(wolfSSL_PEM_def_callback(buf, sizeof(buf), 0, (void*)defpwd),
        defpwdLen);
    ExpectIntEQ(XMEMCMP(buf, defpwd, defpwdLen), 0);
    ExpectIntEQ(buf[defpwdLen], 0);
    /* Size of buffer is smaller than default password. */
    XMEMSET(buf, 0, sizeof(buf));
    ExpectIntEQ(wolfSSL_PEM_def_callback(buf, smallLen, 0, (void*)defpwd),
        smallLen);
    ExpectIntEQ(XMEMCMP(buf, defpwd, smallLen), 0);
    ExpectIntEQ(buf[smallLen], 0);
#endif /* OPENSSL_EXTRA */
    return EXPECT_RESULT();
}

static int test_wolfSSL_PEM_read_PrivateKey(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_FILESYSTEM) && (!defined(NO_RSA) || \
    !defined(NO_DSA) || defined(HAVE_ECC) || !defined(NO_DH))
    XFILE file = XBADFILE;
#if !defined(NO_RSA)
    const char* fname_rsa = "./certs/server-key.pem";
    RSA* rsa = NULL;
    WOLFSSL_EVP_PKEY_CTX* ctx = NULL;
    unsigned char* sig = NULL;
    size_t sigLen = 0;
    const unsigned char tbs[] = {0, 1, 2, 3, 4, 5, 6, 7};
    size_t tbsLen = sizeof(tbs);
#endif
#if !defined(NO_DSA)
    const char* fname_dsa = "./certs/dsa2048.pem";
#endif
#if defined(HAVE_ECC)
    const char* fname_ec = "./certs/ecc-key.pem";
#endif
#if !defined(NO_DH)
    const char* fname_dh = "./certs/dh-priv-2048.pem";
#endif
    EVP_PKEY* pkey = NULL;

    /* Check error case. */
    ExpectNull(pkey = PEM_read_PrivateKey(NULL, NULL, NULL, NULL));

    /* not a PEM key. */
    ExpectTrue((file = XFOPEN("./certs/ecc-key.der", "rb")) != XBADFILE);
    ExpectNull(PEM_read_PrivateKey(file, NULL, NULL, NULL));
    if (file != XBADFILE)
        XFCLOSE(file);
    file = XBADFILE;

#ifndef NO_RSA
    /* Read in an RSA key. */
    ExpectTrue((file = XFOPEN(fname_rsa, "rb")) != XBADFILE);
    ExpectNotNull(pkey = PEM_read_PrivateKey(file, NULL, NULL, NULL));
    if (file != XBADFILE)
        XFCLOSE(file);
    file = XBADFILE;

    /* Make sure the key is usable by signing some data with it. */
    ExpectNotNull(rsa = EVP_PKEY_get0_RSA(pkey));
    ExpectIntGT((sigLen = RSA_size(rsa)), 0);
    ExpectNotNull(sig = (unsigned char*)XMALLOC(sigLen, HEAP_HINT,
        DYNAMIC_TYPE_TMP_BUFFER));
    ExpectNotNull(ctx = EVP_PKEY_CTX_new(pkey, NULL));
    ExpectIntEQ(EVP_PKEY_sign_init(ctx), WOLFSSL_SUCCESS);
    ExpectIntEQ(EVP_PKEY_sign(ctx, sig, &sigLen, tbs, tbsLen),
        WOLFSSL_SUCCESS);

    XFREE(sig, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    EVP_PKEY_CTX_free(ctx);
    EVP_PKEY_free(pkey);
    pkey = NULL;
#endif

#ifndef NO_DSA
    /* Read in a DSA key. */
    ExpectTrue((file = XFOPEN(fname_dsa, "rb")) != XBADFILE);
#if defined(WOLFSSL_QT) || defined(OPENSSL_ALL) || defined(WOLFSSL_OPENSSH)
    ExpectNotNull(pkey = PEM_read_PrivateKey(file, NULL, NULL, NULL));
    EVP_PKEY_free(pkey);
    pkey = NULL;
#else
    ExpectNull(PEM_read_PrivateKey(file, NULL, NULL, NULL));
#endif
    if (file != XBADFILE)
        XFCLOSE(file);
    file = XBADFILE;
#endif

#ifdef HAVE_ECC
    /* Read in an EC key. */
    ExpectTrue((file = XFOPEN(fname_ec, "rb")) != XBADFILE);
    ExpectNotNull(pkey = EVP_PKEY_new());
    ExpectPtrEq(PEM_read_PrivateKey(file, &pkey, NULL, NULL), pkey);
    if (file != XBADFILE)
        XFCLOSE(file);
    file = XBADFILE;
    EVP_PKEY_free(pkey);
    pkey = NULL;
#endif

#ifndef NO_DH
    /* Read in a DH key. */
    ExpectTrue((file = XFOPEN(fname_dh, "rb")) != XBADFILE);
#if (defined(WOLFSSL_QT) || defined(OPENSSL_ALL) || \
     defined(WOLFSSL_OPENSSH)) && (!defined(HAVE_FIPS) || \
     (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION > 2)))
    ExpectNotNull(pkey = PEM_read_PrivateKey(file, NULL, NULL, NULL));
    EVP_PKEY_free(pkey);
    pkey = NULL;
#else
    ExpectNull(PEM_read_PrivateKey(file, NULL, NULL, NULL));
#endif
    if (file != XBADFILE)
        XFCLOSE(file);
    file = XBADFILE;
#endif
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_PEM_read_PUBKEY(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_RSA) \
    && !defined(NO_FILESYSTEM)
    XFILE file = XBADFILE;
    const char* fname = "./certs/client-keyPub.pem";
    EVP_PKEY* pkey = NULL;

    /* Check error case. */
    ExpectNull(pkey = PEM_read_PUBKEY(NULL, NULL, NULL, NULL));

    /* Read in an RSA key. */
    ExpectTrue((file = XFOPEN(fname, "rb")) != XBADFILE);
    ExpectNotNull(pkey = PEM_read_PUBKEY(file, NULL, NULL, NULL));
    EVP_PKEY_free(pkey);
    pkey = NULL;
    if (file != XBADFILE)
        XFCLOSE(file);
    file = XBADFILE;
    ExpectTrue((file = XFOPEN(fname, "rb")) != XBADFILE);
    ExpectNotNull(pkey = EVP_PKEY_new());
    ExpectPtrEq(PEM_read_PUBKEY(file, &pkey, NULL, NULL), pkey);
    EVP_PKEY_free(pkey);
    if (file != XBADFILE)
        XFCLOSE(file);
#endif
    return EXPECT_RESULT();
}

/* test loading RSA key using BIO */
static int test_wolfSSL_PEM_PrivateKey_rsa(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_RSA) && \
    defined(USE_CERT_BUFFERS_2048) && !defined(NO_FILESYSTEM) && \
    !defined(NO_BIO)
    BIO*      bio = NULL;
    XFILE file = XBADFILE;
    const char* fname = "./certs/server-key.pem";
    const char* fname_rsa_p8 = "./certs/server-keyPkcs8.pem";
    EVP_PKEY* pkey  = NULL;
    size_t sz = 0;
    byte* buf = NULL;
    EVP_PKEY* pkey2 = NULL;
    EVP_PKEY* pkey3 = NULL;
    RSA* rsa_key = NULL;
#if defined(WOLFSSL_KEY_GEN) || defined(WOLFSSL_CERT_GEN)
    unsigned char extra[10];
    int i;
    BIO* pub_bio = NULL;
    const unsigned char* server_key = (const unsigned char*)server_key_der_2048;
#endif

    ExpectTrue((file = XFOPEN(fname, "rb")) != XBADFILE);
    ExpectTrue(XFSEEK(file, 0, XSEEK_END) == 0);
    ExpectIntGT(sz = XFTELL(file), 0);
    ExpectTrue(XFSEEK(file, 0, XSEEK_SET) == 0);
    ExpectNotNull(buf = (byte*)XMALLOC(sz, NULL, DYNAMIC_TYPE_FILE));
    if (buf != NULL) {
        ExpectIntEQ(XFREAD(buf, 1, sz, file), sz);
    }
    if (file != XBADFILE) {
        XFCLOSE(file);
        file = XBADFILE;
    }

    /* Test using BIO new mem and loading PEM private key */
    ExpectNotNull(bio = BIO_new_mem_buf(buf, (int)sz));
    ExpectNotNull((pkey = PEM_read_bio_PrivateKey(bio, NULL, NULL, NULL)));
    XFREE(buf, NULL, DYNAMIC_TYPE_FILE);
    buf = NULL;
    BIO_free(bio);
    bio = NULL;

    /* New empty EVP_PKEY */
    ExpectNotNull(pkey2 = EVP_PKEY_new());
    if (pkey2 != NULL) {
        pkey2->type = EVP_PKEY_RSA;
    }
    /* Test parameter copy */
    ExpectIntEQ(EVP_PKEY_copy_parameters(pkey2, pkey), 0);
    EVP_PKEY_free(pkey2);
    EVP_PKEY_free(pkey);
    pkey  = NULL;

    /* Qt unit test case : rsa pkcs8 key */
    ExpectTrue((file = XFOPEN(fname_rsa_p8, "rb")) != XBADFILE);
    ExpectTrue(XFSEEK(file, 0, XSEEK_END) == 0);
    ExpectIntGT(sz = XFTELL(file), 0);
    ExpectTrue(XFSEEK(file, 0, XSEEK_SET) == 0);
    ExpectNotNull(buf = (byte*)XMALLOC(sz, NULL, DYNAMIC_TYPE_FILE));
    if (buf) {
        ExpectIntEQ(XFREAD(buf, 1, sz, file), sz);
    }
    if (file != XBADFILE) {
        XFCLOSE(file);
        file = XBADFILE;
    }

    ExpectNotNull(bio = BIO_new_mem_buf(buf, (int)sz));
    ExpectNotNull((pkey = PEM_read_bio_PrivateKey(bio, NULL, NULL, NULL)));
    XFREE(buf, NULL, DYNAMIC_TYPE_FILE);
    buf = NULL;
    BIO_free(bio);
    bio = NULL;
    ExpectNotNull(pkey3 = EVP_PKEY_new());

    ExpectNotNull(rsa_key = EVP_PKEY_get1_RSA(pkey));
    ExpectIntEQ(EVP_PKEY_set1_RSA(pkey3, rsa_key), WOLFSSL_SUCCESS);

#ifdef WOLFSSL_ERROR_CODE_OPENSSL
    ExpectIntEQ(EVP_PKEY_cmp(pkey, pkey3), 1/* match */);
#else
    ExpectIntEQ(EVP_PKEY_cmp(pkey, pkey3), 0);
#endif

    RSA_free(rsa_key);
    EVP_PKEY_free(pkey3);
    EVP_PKEY_free(pkey);
    pkey  = NULL;
    pkey2 = NULL;

#if defined(WOLFSSL_KEY_GEN) || defined(WOLFSSL_CERT_GEN)
    #define BIO_PEM_TEST_CHAR 'a'
    XMEMSET(extra, BIO_PEM_TEST_CHAR, sizeof(extra));

    ExpectNotNull(bio = wolfSSL_BIO_new(wolfSSL_BIO_s_mem()));
    ExpectIntEQ(BIO_set_write_buf_size(bio, 4096), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectNotNull(pub_bio = wolfSSL_BIO_new(wolfSSL_BIO_s_mem()));
    ExpectIntEQ(BIO_set_write_buf_size(pub_bio, 4096), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

    ExpectNull(d2i_PrivateKey(EVP_PKEY_EC, &pkey, &server_key,
        (long)sizeof_server_key_der_2048));
    ExpectNull(pkey);

    ExpectNotNull(wolfSSL_d2i_PrivateKey(EVP_PKEY_RSA, &pkey, &server_key,
        (long)sizeof_server_key_der_2048));
    ExpectIntEQ(PEM_write_bio_PrivateKey(NULL, pkey, NULL, NULL, 0, NULL, NULL),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(PEM_write_bio_PrivateKey(bio,  NULL, NULL, NULL, 0, NULL, NULL),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(PEM_write_bio_PrivateKey(bio,  pkey, NULL, NULL, 0, NULL, NULL),
        WOLFSSL_SUCCESS);
    ExpectIntGT(BIO_pending(bio), 0);
    ExpectIntEQ(BIO_pending(bio), 1679);
    /* Check if the pubkey API writes only the public key */
#ifdef WOLFSSL_KEY_GEN
    ExpectIntEQ(PEM_write_bio_PUBKEY(NULL, pkey), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(PEM_write_bio_PUBKEY(pub_bio, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(PEM_write_bio_PUBKEY(pub_bio, pkey), WOLFSSL_SUCCESS);
    ExpectIntGT(BIO_pending(pub_bio), 0);
    /* Previously both the private key and the pubkey calls would write
     * out the private key and the PEM header was the only difference.
     * The public PEM should be significantly shorter than the
     * private key versison. */
    ExpectIntEQ(BIO_pending(pub_bio), 451);
#else
    /* Not supported. */
    ExpectIntEQ(PEM_write_bio_PUBKEY(pub_bio, pkey), 0);
#endif

    /* test creating new EVP_PKEY with good args */
    ExpectNotNull((pkey2 = PEM_read_bio_PrivateKey(bio, NULL, NULL, NULL)));
    if (pkey && pkey->pkey.ptr && pkey2 && pkey2->pkey.ptr) {
        ExpectIntEQ((int)XMEMCMP(pkey->pkey.ptr, pkey2->pkey.ptr,
            pkey->pkey_sz), 0);
    }

    /* test of reuse of EVP_PKEY */
    ExpectNull(PEM_read_bio_PrivateKey(bio, &pkey, NULL, NULL));
    ExpectIntEQ(BIO_pending(bio), 0);
    ExpectIntEQ(PEM_write_bio_PrivateKey(bio, pkey, NULL, NULL, 0, NULL, NULL),
            SSL_SUCCESS);
    /* add 10 extra bytes after PEM */
    ExpectIntEQ(BIO_write(bio, extra, 10), 10);
    ExpectNotNull(PEM_read_bio_PrivateKey(bio, &pkey, NULL, NULL));
    ExpectNotNull(pkey);
    if (pkey && pkey->pkey.ptr && pkey2 && pkey2->pkey.ptr) {
        ExpectIntEQ((int)XMEMCMP(pkey->pkey.ptr, pkey2->pkey.ptr,
            pkey->pkey_sz), 0);
    }
    /* check 10 extra bytes still there */
    ExpectIntEQ(BIO_pending(bio), 10);
    ExpectIntEQ(BIO_read(bio, extra, 10), 10);
    for (i = 0; i < 10; i++) {
        ExpectIntEQ(extra[i], BIO_PEM_TEST_CHAR);
    }

    BIO_free(pub_bio);
    BIO_free(bio);
    bio = NULL;
    EVP_PKEY_free(pkey);
    pkey  = NULL;
    EVP_PKEY_free(pkey2);
#endif /* WOLFSSL_KEY_GEN || WOLFSSL_CERT_GEN */
#endif /* OPENSSL_EXTRA && !NO_CERTS && !NO_RSA && USE_CERT_BUFFERS_2048 &&
        * !NO_FILESYSTEM && !NO_BIO */
    return EXPECT_RESULT();
}

/* test loading ECC key using BIO */
static int test_wolfSSL_PEM_PrivateKey_ecc(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && defined(HAVE_ECC) && \
    !defined(NO_FILESYSTEM) && !defined(NO_BIO)
    BIO*      bio = NULL;
    EVP_PKEY* pkey  = NULL;
    XFILE file = XBADFILE;
    const char* fname = "./certs/ecc-key.pem";
    const char* fname_ecc_p8  = "./certs/ecc-keyPkcs8.pem";

    size_t sz = 0;
    byte* buf = NULL;
    EVP_PKEY* pkey2 = NULL;
    EVP_PKEY* pkey3 = NULL;
    EC_KEY*   ec_key = NULL;
    int nid = 0;
    BIO* pub_bio = NULL;

    ExpectTrue((file = XFOPEN(fname, "rb")) != XBADFILE);
    ExpectTrue(XFSEEK(file, 0, XSEEK_END) == 0);
    ExpectIntGT(sz = XFTELL(file), 0);
    ExpectTrue(XFSEEK(file, 0, XSEEK_SET) == 0);
    ExpectNotNull(buf = (byte*)XMALLOC(sz, NULL, DYNAMIC_TYPE_FILE));
    if (buf) {
        ExpectIntEQ(XFREAD(buf, 1, sz, file), sz);
    }
    if (file != XBADFILE) {
        XFCLOSE(file);
        file = XBADFILE;
    }

    /* Test using BIO new mem and loading PEM private key */
    ExpectNotNull(bio = BIO_new_mem_buf(buf, (int)sz));
    ExpectNotNull((pkey = PEM_read_bio_PrivateKey(bio, NULL, NULL, NULL)));
    BIO_free(bio);
    bio = NULL;
    XFREE(buf, NULL, DYNAMIC_TYPE_FILE);
    buf = NULL;
    ExpectNotNull(bio = wolfSSL_BIO_new(wolfSSL_BIO_s_mem()));
    ExpectNotNull(pub_bio = wolfSSL_BIO_new(wolfSSL_BIO_s_mem()));
    ExpectIntEQ(PEM_write_bio_PrivateKey(bio, pkey, NULL, NULL, 0, NULL, NULL),
        WOLFSSL_SUCCESS);
    ExpectIntGT(BIO_pending(bio), 0);
    /* No parameters. */
    ExpectIntEQ(BIO_pending(bio), 227);
    /* Check if the pubkey API writes only the public key */
#ifdef WOLFSSL_KEY_GEN
    ExpectIntEQ(PEM_write_bio_PUBKEY(pub_bio, pkey), WOLFSSL_SUCCESS);
    ExpectIntGT(BIO_pending(pub_bio), 0);
    /* Previously both the private key and the pubkey calls would write
     * out the private key and the PEM header was the only difference.
     * The public PEM should be significantly shorter than the
     * private key versison. */
    ExpectIntEQ(BIO_pending(pub_bio), 178);
#endif
    BIO_free(pub_bio);
    BIO_free(bio);
    bio = NULL;
    ExpectNotNull(pkey2 = EVP_PKEY_new());
    ExpectNotNull(pkey3 = EVP_PKEY_new());
    if (pkey2 != NULL) {
         pkey2->type = EVP_PKEY_EC;
    }
    /* Test parameter copy */
    ExpectIntEQ(EVP_PKEY_copy_parameters(pkey2, pkey), 1);


    /* Qt unit test case 1*/
    ExpectNotNull(ec_key = EVP_PKEY_get1_EC_KEY(pkey));
    ExpectIntEQ(EVP_PKEY_set1_EC_KEY(pkey3, ec_key), WOLFSSL_SUCCESS);
    #ifdef WOLFSSL_ERROR_CODE_OPENSSL
    ExpectIntEQ(EVP_PKEY_cmp(pkey, pkey3), 1/* match */);
    #else
    ExpectIntEQ(EVP_PKEY_cmp(pkey, pkey3), 0);
    #endif
    /* Test default digest */
    ExpectIntEQ(EVP_PKEY_get_default_digest_nid(pkey, &nid), 1);
    ExpectIntEQ(nid, NID_sha256);
    EC_KEY_free(ec_key);
    ec_key = NULL;
    EVP_PKEY_free(pkey3);
    pkey3 = NULL;
    EVP_PKEY_free(pkey2);
    pkey2 = NULL;
    EVP_PKEY_free(pkey);
    pkey  = NULL;

    /* Qt unit test case ec pkcs8 key */
    ExpectTrue((file = XFOPEN(fname_ecc_p8, "rb")) != XBADFILE);
    ExpectTrue(XFSEEK(file, 0, XSEEK_END) == 0);
    ExpectIntGT(sz = XFTELL(file), 0);
    ExpectTrue(XFSEEK(file, 0, XSEEK_SET) == 0);
    ExpectNotNull(buf = (byte*)XMALLOC(sz, NULL, DYNAMIC_TYPE_FILE));
    if (buf) {
        ExpectIntEQ(XFREAD(buf, 1, sz, file), sz);
    }
    if (file != XBADFILE) {
        XFCLOSE(file);
        file = XBADFILE;
    }

    ExpectNotNull(bio = BIO_new_mem_buf(buf, (int)sz));
    ExpectNotNull((pkey = PEM_read_bio_PrivateKey(bio, NULL, NULL, NULL)));
    XFREE(buf, NULL, DYNAMIC_TYPE_FILE);
    buf = NULL;
    BIO_free(bio);
    bio = NULL;
    ExpectNotNull(pkey3 = EVP_PKEY_new());
    /* Qt unit test case */
    ExpectNotNull(ec_key = EVP_PKEY_get1_EC_KEY(pkey));
    ExpectIntEQ(EVP_PKEY_set1_EC_KEY(pkey3, ec_key), WOLFSSL_SUCCESS);
#ifdef WOLFSSL_ERROR_CODE_OPENSSL
    ExpectIntEQ(EVP_PKEY_cmp(pkey, pkey3), 1/* match */);
#else
    ExpectIntEQ(EVP_PKEY_cmp(pkey, pkey3), 0);
#endif
    EC_KEY_free(ec_key);
    EVP_PKEY_free(pkey3);
    EVP_PKEY_free(pkey);
    pkey  = NULL;
#endif
    return EXPECT_RESULT();
}

/* test loading DSA key using BIO */
static int test_wolfSSL_PEM_PrivateKey_dsa(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_DSA) && \
    !defined(NO_FILESYSTEM) && !defined(NO_BIO)
#if defined(WOLFSSL_QT) || defined(OPENSSL_ALL)
    BIO*      bio = NULL;
    EVP_PKEY* pkey  = NULL;

    ExpectNotNull(bio = BIO_new_file("./certs/dsa2048.pem", "rb"));
    /* Private DSA EVP_PKEY */
    ExpectNotNull(pkey = wolfSSL_PEM_read_bio_PrivateKey(bio, NULL, NULL,
        NULL));
    BIO_free(bio);
    bio = NULL;

    ExpectNotNull(bio = wolfSSL_BIO_new(wolfSSL_BIO_s_mem()));
#if defined(OPENSSL_ALL) && !defined(NO_PWDBASED) && defined(HAVE_PKCS8)
#ifdef WOLFSSL_ASN_TEMPLATE
    ExpectIntEQ(PEM_write_bio_PKCS8PrivateKey(bio, pkey, NULL, NULL, 0, NULL,
        NULL), 1216);
#else
    ExpectIntEQ(PEM_write_bio_PKCS8PrivateKey(bio, pkey, NULL, NULL, 0, NULL,
        NULL), 1212);
#endif
#endif

#ifdef WOLFSSL_KEY_GEN
    ExpectIntEQ(PEM_write_bio_PUBKEY(bio, pkey), 1);
#ifdef WOLFSSL_ASN_TEMPLATE
    ExpectIntEQ(BIO_pending(bio), 2394);
#else
    ExpectIntEQ(BIO_pending(bio), 2390);
#endif
    BIO_reset(bio);
#endif

    ExpectIntEQ(PEM_write_bio_PrivateKey(bio, pkey, NULL, NULL, 0, NULL, NULL),
        1);
    ExpectIntEQ(BIO_pending(bio), 1196);

    BIO_free(bio);
    bio = NULL;

    EVP_PKEY_free(pkey);
    pkey  = NULL;
#endif
#endif
    return EXPECT_RESULT();
}

/* test loading DH key using BIO */
static int test_wolfSSL_PEM_PrivateKey_dh(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_DH) && \
    !defined(NO_FILESYSTEM) && !defined(NO_BIO)
#if (defined(WOLFSSL_QT) || defined(OPENSSL_ALL) || \
     defined(WOLFSSL_OPENSSH)) && (!defined(HAVE_FIPS) || \
     (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION > 2)))
    BIO*      bio = NULL;
    EVP_PKEY* pkey  = NULL;
    int       expectedBytes = 0;

    ExpectNotNull(bio = BIO_new_file("./certs/dh-priv-2048.pem", "rb"));
    /* Private DH EVP_PKEY */
    ExpectNotNull(pkey = wolfSSL_PEM_read_bio_PrivateKey(bio, NULL, NULL,
        NULL));
    BIO_free(bio);
    bio = NULL;

    ExpectNotNull(bio = wolfSSL_BIO_new(wolfSSL_BIO_s_mem()));

#if defined(OPENSSL_ALL) && !defined(NO_PWDBASED) && defined(HAVE_PKCS8)
    expectedBytes += 806;
    ExpectIntEQ(PEM_write_bio_PKCS8PrivateKey(bio, pkey, NULL, NULL, 0, NULL,
        NULL), expectedBytes);
#endif
#ifdef WOLFSSL_KEY_GEN
    ExpectIntEQ(PEM_write_bio_PUBKEY(bio, pkey), 0);
#endif

    ExpectIntEQ(PEM_write_bio_PrivateKey(bio, pkey, NULL, NULL, 0, NULL, NULL),
        1);
    expectedBytes += 806;
    ExpectIntEQ(BIO_pending(bio), expectedBytes);

    BIO_free(bio);
    bio = NULL;

    EVP_PKEY_free(pkey);
    pkey  = NULL;
#endif
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_PEM_PrivateKey(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    (!defined(NO_RSA) || defined(HAVE_ECC)) && defined(USE_CERT_BUFFERS_2048)
#ifndef NO_BIO
    BIO*      bio = NULL;
#endif
    EVP_PKEY* pkey  = NULL;
    const unsigned char* server_key = (const unsigned char*)server_key_der_2048;

#ifndef NO_BIO

    /* test creating new EVP_PKEY with bad arg */
    ExpectNull((pkey = PEM_read_bio_PrivateKey(NULL, NULL, NULL, NULL)));

    /* Test bad EVP_PKEY type. */
    /* New HMAC EVP_PKEY */
    ExpectNotNull(bio = BIO_new_mem_buf("", 1));
    ExpectNotNull(pkey = EVP_PKEY_new());
    if (pkey != NULL) {
        pkey->type = EVP_PKEY_HMAC;
    }
    ExpectIntEQ(PEM_write_bio_PrivateKey(bio, pkey, NULL, NULL, 0, NULL, NULL),
        0);
#if defined(OPENSSL_ALL) && !defined(NO_PWDBASED) && defined(HAVE_PKCS8)
    ExpectIntEQ(PEM_write_bio_PKCS8PrivateKey(bio, pkey, NULL, NULL, 0, NULL,
        NULL), 0);
#endif
#ifdef WOLFSSL_KEY_GEN
    ExpectIntEQ(PEM_write_bio_PUBKEY(bio, pkey), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
#endif
    EVP_PKEY_free(pkey);
    pkey = NULL;
    BIO_free(bio);
    bio = NULL;


    /* key is DES encrypted */
    #if !defined(NO_DES3) && defined(WOLFSSL_ENCRYPTED_KEYS) && \
        !defined(NO_RSA) && !defined(NO_BIO) && !defined(NO_FILESYSTEM) && \
        !defined(NO_MD5) && defined(WOLFSSL_KEY_GEN) && !defined(NO_RSA)
    {
        XFILE f = XBADFILE;
        wc_pem_password_cb* passwd_cb = NULL;
        void* passwd_cb_userdata;
        SSL_CTX* ctx = NULL;
        char passwd[] = "bad password";

    #ifndef WOLFSSL_NO_TLS12
        #ifndef NO_WOLFSSL_SERVER
        ExpectNotNull(ctx = SSL_CTX_new(TLSv1_2_server_method()));
        #else
        ExpectNotNull(ctx = SSL_CTX_new(TLSv1_2_client_method()));
        #endif
    #else
        #ifndef NO_WOLFSSL_SERVER
        ExpectNotNull(ctx = SSL_CTX_new(TLSv1_3_server_method()));
        #else
        ExpectNotNull(ctx = SSL_CTX_new(TLSv1_3_client_method()));
        #endif
    #endif

        ExpectNotNull(bio = BIO_new_file("./certs/server-keyEnc.pem", "rb"));
        SSL_CTX_set_default_passwd_cb(ctx, PasswordCallBack);
        ExpectNotNull(passwd_cb = SSL_CTX_get_default_passwd_cb(ctx));
        ExpectNull(passwd_cb_userdata =
            SSL_CTX_get_default_passwd_cb_userdata(ctx));

        /* fail case with password call back */
        ExpectNull(pkey = PEM_read_bio_PrivateKey(bio, NULL, NULL,
                    (void*)passwd));
        BIO_free(bio);
        ExpectNotNull(bio = BIO_new_file("./certs/server-keyEnc.pem", "rb"));
        ExpectNull(pkey = PEM_read_bio_PrivateKey(bio, NULL, passwd_cb,
                    (void*)passwd));
        BIO_free(bio);

        ExpectTrue((f = XFOPEN("./certs/server-keyEnc.pem", "rb")) != XBADFILE);
        ExpectNotNull(bio = BIO_new_fp(f, BIO_CLOSE));
        if ((bio == NULL) && (f != XBADFILE)) {
            XFCLOSE(f);
        }

        /* use callback that works */
        ExpectNotNull(pkey = PEM_read_bio_PrivateKey(bio, NULL, passwd_cb,
                (void*)"yassl123"));

        ExpectIntEQ(SSL_CTX_use_PrivateKey(ctx, pkey), SSL_SUCCESS);

        EVP_PKEY_free(pkey);
        pkey  = NULL;
        BIO_free(bio);
        bio = NULL;
        SSL_CTX_free(ctx);
    }
    #endif /* !defined(NO_DES3) */

#endif /* !NO_BIO */

    #if defined(HAVE_ECC) && !defined(NO_FILESYSTEM)
    {
        unsigned char buf[2048];
        size_t bytes = 0;
        XFILE f = XBADFILE;
        SSL_CTX* ctx = NULL;

    #ifndef WOLFSSL_NO_TLS12
        #ifndef NO_WOLFSSL_SERVER
        ExpectNotNull(ctx = SSL_CTX_new(TLSv1_2_server_method()));
        #else
        ExpectNotNull(ctx = SSL_CTX_new(TLSv1_2_client_method()));
        #endif
    #else
        #ifndef NO_WOLFSSL_SERVER
        ExpectNotNull(ctx = SSL_CTX_new(wolfTLSv1_3_server_method()));
        #else
        ExpectNotNull(ctx = SSL_CTX_new(wolfTLSv1_3_client_method()));
        #endif
    #endif

        ExpectTrue((f = XFOPEN("./certs/ecc-key.der", "rb")) != XBADFILE);
        ExpectIntGT(bytes = (size_t)XFREAD(buf, 1, sizeof(buf), f), 0);
        if (f != XBADFILE)
            XFCLOSE(f);

        server_key = buf;
        pkey = NULL;
        ExpectNull(d2i_PrivateKey(EVP_PKEY_RSA, &pkey, &server_key, (long int)bytes));
        ExpectNull(pkey);
        ExpectNotNull(d2i_PrivateKey(EVP_PKEY_EC, &pkey, &server_key, (long int)bytes));
        ExpectIntEQ(SSL_CTX_use_PrivateKey(ctx, pkey), SSL_SUCCESS);

        EVP_PKEY_free(pkey);
        pkey = NULL;
        SSL_CTX_free(ctx);
        server_key = NULL;
    }
    #endif

#ifndef NO_BIO
    (void)bio;
#endif
    (void)pkey;
    (void)server_key;
#endif /* OPENSSL_EXTRA && !NO_CERTS && !NO_RSA && USE_CERT_BUFFERS_2048 */
    return EXPECT_RESULT();
}

static int test_wolfSSL_PEM_file_RSAKey(void)
{
    EXPECT_DECLS;
#if (defined(OPENSSL_EXTRA) || defined(OPENSSL_ALL)) && \
    defined(WOLFSSL_KEY_GEN) && !defined(NO_RSA) && \
    !defined(NO_FILESYSTEM) && !defined(NO_CERTS)
    RSA* rsa = NULL;
    XFILE fp = XBADFILE;

    ExpectTrue((fp = XFOPEN("./certs/rsa-pub-2048.pem", "rb")) != XBADFILE);
    ExpectNotNull((rsa = PEM_read_RSA_PUBKEY(fp, NULL, NULL, NULL)));
    if (fp != XBADFILE)
        XFCLOSE(fp);
    ExpectIntEQ(RSA_size(rsa), 256);

    ExpectIntEQ(PEM_write_RSAPublicKey(XBADFILE, rsa), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(PEM_write_RSAPublicKey(stderr, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(PEM_write_RSAPublicKey(stderr, rsa), WOLFSSL_SUCCESS);

    ExpectIntEQ(PEM_write_RSA_PUBKEY(XBADFILE, rsa), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(PEM_write_RSA_PUBKEY(stderr, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(PEM_write_RSA_PUBKEY(stderr, rsa), WOLFSSL_SUCCESS);

    RSA_free(rsa);
#endif /* defined(OPENSSL_EXTRA) || defined(OPENSSL_ALL)) && \
         (defined(WOLFSSL_KEY_GEN) || WOLFSSL_CERT_GEN) && \
         !defined(NO_FILESYSTEM) && !defined(NO_RSA) && !defined(NO_CERTS) */
    return EXPECT_RESULT();
}

static int test_wolfSSL_PEM_file_RSAPrivateKey(void)
{
    EXPECT_DECLS;
#if !defined(NO_RSA) && defined(OPENSSL_EXTRA) && defined(WOLFSSL_KEY_GEN) && \
    !defined(NO_FILESYSTEM) && \
    (defined(WOLFSSL_PEM_TO_DER) || defined(WOLFSSL_DER_TO_PEM))
    RSA* rsa = NULL;
    XFILE f = NULL;

    ExpectTrue((f = XFOPEN(svrKeyFile, "rb")) != XBADFILE);
    ExpectNotNull((rsa = PEM_read_RSAPrivateKey(f, NULL, NULL, NULL)));
    ExpectIntEQ(RSA_size(rsa), 256);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }

    ExpectIntEQ(PEM_write_RSAPrivateKey(XBADFILE, rsa, NULL, NULL, 0, NULL,
        NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(PEM_write_RSAPrivateKey(stderr, NULL, NULL, NULL, 0, NULL,
        NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(PEM_write_RSAPrivateKey(stderr, rsa, NULL, NULL, 0, NULL, NULL),
        WOLFSSL_SUCCESS);

    RSA_free(rsa);

#ifdef HAVE_ECC
    ExpectTrue((f = XFOPEN(eccKeyFile, "rb")) != XBADFILE);
    ExpectNull((rsa = PEM_read_RSAPrivateKey(f, NULL, NULL, NULL)));
    if (f != XBADFILE)
        XFCLOSE(f);
#endif /* HAVE_ECC */
#endif /* defined(OPENSSL_EXTRA) && !defined(NO_CERTS) */
    return EXPECT_RESULT();
}

static int test_wolfSSL_PEM_read_RSA_PUBKEY(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && \
       !defined(NO_FILESYSTEM) && !defined(NO_RSA)
    XFILE file = XBADFILE;
    const char* fname = "./certs/client-keyPub.pem";
    RSA *rsa = NULL;

    ExpectNull(wolfSSL_PEM_read_RSA_PUBKEY(XBADFILE, NULL, NULL, NULL));

    ExpectTrue((file = XFOPEN(fname, "rb")) != XBADFILE);
    ExpectNotNull((rsa = PEM_read_RSA_PUBKEY(file, NULL, NULL, NULL)));
    ExpectIntEQ(RSA_size(rsa), 256);
    RSA_free(rsa);
    if (file != XBADFILE)
       XFCLOSE(file);
#endif /* defined(OPENSSL_EXTRA) && !defined(NO_CERTS) */
    return EXPECT_RESULT();
}

#ifndef NO_BIO
static int test_wolfSSL_PEM_bio_RSAKey(void)
{
    EXPECT_DECLS;
#if (defined(OPENSSL_EXTRA) || defined(OPENSSL_ALL)) && \
    defined(WOLFSSL_KEY_GEN) && !defined(NO_RSA) && \
    !defined(NO_FILESYSTEM) && !defined(NO_CERTS)
    RSA* rsa = NULL;
    BIO* bio = NULL;

    /* PrivateKey */
    ExpectNotNull(bio = BIO_new_file(svrKeyFile, "rb"));
    ExpectNull((rsa = PEM_read_bio_RSAPrivateKey(NULL, NULL, NULL, NULL)));
    ExpectNotNull(PEM_read_bio_RSAPrivateKey(bio, &rsa, NULL, NULL));
    ExpectNotNull(rsa);
    ExpectIntEQ(RSA_size(rsa), 256);
    ExpectIntEQ(PEM_write_bio_RSAPrivateKey(NULL, NULL, NULL, NULL, 0, NULL, \
                                            NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    BIO_free(bio);
    bio = NULL;
    ExpectNotNull(bio = wolfSSL_BIO_new(wolfSSL_BIO_s_mem()));
    ExpectIntEQ(PEM_write_bio_RSAPrivateKey(bio, rsa, NULL, NULL, 0, NULL, \
                                            NULL), WOLFSSL_SUCCESS);
    BIO_free(bio);
    bio = NULL;
    RSA_free(rsa);
    rsa = NULL;

    /* PUBKEY */
    ExpectNotNull(bio = BIO_new_file("./certs/rsa-pub-2048.pem", "rb"));
    ExpectNull((rsa = PEM_read_bio_RSA_PUBKEY(NULL, NULL, NULL, NULL)));
    ExpectNotNull((rsa = PEM_read_bio_RSA_PUBKEY(bio, NULL, NULL, NULL)));
    ExpectIntEQ(RSA_size(rsa), 256);
    ExpectIntEQ(PEM_write_bio_RSA_PUBKEY(NULL, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    BIO_free(bio);
    bio = NULL;
    ExpectNotNull(bio = wolfSSL_BIO_new(wolfSSL_BIO_s_mem()));
    ExpectIntEQ(PEM_write_bio_RSA_PUBKEY(bio, rsa), WOLFSSL_SUCCESS);
    BIO_free(bio);
    bio = NULL;

    RSA_free(rsa);
    rsa = NULL;

    /* Ensure that keys beginning with BEGIN RSA PUBLIC KEY can be read, too. */
    ExpectNotNull(bio = BIO_new_file("./certs/server-keyPub.pem", "rb"));
    ExpectNotNull((rsa = PEM_read_bio_RSA_PUBKEY(bio, NULL, NULL, NULL)));
    BIO_free(bio);
    bio = NULL;
    RSA_free(rsa);
    rsa = NULL;

    #ifdef HAVE_ECC
    /* ensure that non-rsa keys do not work */
    ExpectNotNull(bio = BIO_new_file(eccKeyFile, "rb")); /* ecc key */
    ExpectNull((rsa = PEM_read_bio_RSAPrivateKey(bio, NULL, NULL, NULL)));
    ExpectNull((rsa = PEM_read_bio_RSA_PUBKEY(bio, NULL, NULL, NULL)));
    BIO_free(bio);
    bio = NULL;
    RSA_free(rsa);
    rsa = NULL;
    #endif /* HAVE_ECC */
#endif /* defined(OPENSSL_EXTRA) || defined(OPENSSL_ALL)) && \
         (defined(WOLFSSL_KEY_GEN) || WOLFSSL_CERT_GEN) && \
         !defined(NO_FILESYSTEM) && !defined(NO_RSA) && !defined(NO_CERTS) */
    return EXPECT_RESULT();
}

static int test_wolfSSL_PEM_bio_RSAPrivateKey(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && \
   !defined(NO_FILESYSTEM) && !defined(NO_RSA)
    RSA* rsa = NULL;
    RSA* rsa_dup = NULL;
    BIO* bio = NULL;

    ExpectNotNull(bio = BIO_new_file(svrKeyFile, "rb"));
    ExpectNotNull((rsa = PEM_read_bio_RSAPrivateKey(bio, NULL, NULL, NULL)));
    ExpectIntEQ(RSA_size(rsa), 256);

#if defined(WOLFSSL_KEY_GEN) && !defined(NO_RSA)
    ExpectNull(rsa_dup = RSAPublicKey_dup(NULL));
    /* Test duplicating empty key. */
    ExpectNotNull(rsa_dup = RSA_new());
    ExpectNull(RSAPublicKey_dup(rsa_dup));
    RSA_free(rsa_dup);
    rsa_dup = NULL;
    ExpectNotNull(rsa_dup = RSAPublicKey_dup(rsa));
    ExpectPtrNE(rsa_dup, rsa);
#endif

    /* test if valgrind complains about unreleased memory */
    RSA_up_ref(rsa);
    RSA_free(rsa);

    BIO_free(bio);
    bio = NULL;
    RSA_free(rsa);
    rsa = NULL;
    RSA_free(rsa_dup);
    rsa_dup = NULL;

#ifdef HAVE_ECC
    ExpectNotNull(bio = BIO_new_file(eccKeyFile, "rb"));
    ExpectNull((rsa = PEM_read_bio_RSAPrivateKey(bio, NULL, NULL, NULL)));

    BIO_free(bio);
#endif /* HAVE_ECC */
#endif /* defined(OPENSSL_EXTRA) && !defined(NO_CERTS) */
    return EXPECT_RESULT();
}

static int test_wolfSSL_PEM_bio_DSAKey(void)
{
    EXPECT_DECLS;
#ifndef HAVE_SELFTEST
#if (defined(WOLFSSL_QT) || defined(OPENSSL_ALL)) && !defined(NO_CERTS) && \
    defined(WOLFSSL_KEY_GEN) && !defined(NO_FILESYSTEM) && !defined(NO_DSA)
    DSA* dsa = NULL;
    BIO* bio = NULL;

    /* PrivateKey */
    ExpectNotNull(bio = BIO_new_file("./certs/1024/dsa1024.pem", "rb"));
    ExpectNull((dsa = PEM_read_bio_DSAPrivateKey(NULL, NULL, NULL, NULL)));
    ExpectNotNull((dsa = PEM_read_bio_DSAPrivateKey(bio, NULL, NULL, NULL)));
    ExpectIntEQ(BN_num_bytes(dsa->g), 128);
    ExpectIntEQ(PEM_write_bio_DSAPrivateKey(NULL, NULL, NULL, NULL, 0, NULL,
        NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    BIO_free(bio);
    bio = NULL;
    ExpectNotNull(bio = wolfSSL_BIO_new(wolfSSL_BIO_s_mem()));
    ExpectIntEQ(PEM_write_bio_DSAPrivateKey(bio, dsa, NULL, NULL, 0, NULL,
        NULL), WOLFSSL_SUCCESS);
    BIO_free(bio);
    bio = NULL;
    DSA_free(dsa);
    dsa = NULL;

    /* PUBKEY */
    ExpectNotNull(bio = BIO_new_file("./certs/1024/dsa-pub-1024.pem", "rb"));
    ExpectNull((dsa = PEM_read_bio_DSA_PUBKEY(NULL, NULL, NULL, NULL)));
    ExpectNotNull((dsa = PEM_read_bio_DSA_PUBKEY(bio, NULL, NULL, NULL)));
    ExpectIntEQ(BN_num_bytes(dsa->g), 128);
    ExpectIntEQ(PEM_write_bio_DSA_PUBKEY(NULL, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    BIO_free(bio);
    bio = NULL;
    ExpectNotNull(bio = wolfSSL_BIO_new(wolfSSL_BIO_s_mem()));
    ExpectIntEQ(PEM_write_bio_DSA_PUBKEY(bio, dsa), WOLFSSL_SUCCESS);
    BIO_free(bio);
    bio = NULL;
    DSA_free(dsa);
    dsa = NULL;

    #ifdef HAVE_ECC
    /* ensure that non-dsa keys do not work */
    ExpectNotNull(bio = BIO_new_file(eccKeyFile, "rb")); /* ecc key */
    ExpectNull((dsa = PEM_read_bio_DSAPrivateKey(bio, NULL, NULL, NULL)));
    ExpectNull((dsa = PEM_read_bio_DSA_PUBKEY(bio, NULL, NULL, NULL)));
    BIO_free(bio);
    bio = NULL;
    DSA_free(dsa);
    dsa = NULL;
    #endif /* HAVE_ECC */
#endif /* defined(WOLFSSL_QT) || defined(OPENSSL_ALL)) && \
         !defined(NO_CERTS) && defined(WOLFSSL_KEY_GEN) && \
         !defined(NO_FILESYSTEM) && !defined(NO_DSA) */
#endif /* HAVE_SELFTEST */
    return EXPECT_RESULT();
}

static int test_wolfSSL_PEM_bio_ECKey(void)
{
    EXPECT_DECLS;
#if (defined(OPENSSL_EXTRA) || defined(OPENSSL_ALL)) && \
    defined(WOLFSSL_KEY_GEN) && !defined(NO_FILESYSTEM) && defined(HAVE_ECC)
    EC_KEY* ec = NULL;
    EC_KEY* ec2;
    BIO* bio = NULL;
#if defined(WOLFSSL_PEM_TO_DER) || defined(WOLFSSL_DER_TO_PEM)
    unsigned char* pem = NULL;
    int pLen;
#endif
    static char ec_key_bad_1[] = "-----BEGIN PUBLIC KEY-----\n"
                                 "MAA=\n"
                                 "-----END PUBLIC KEY-----";
    static char ec_priv_key_bad_1[] = "-----BEGIN EC PRIVATE KEY-----\n"
                                      "MAA=\n"
                                      "-----END EC PRIVATE KEY-----";

    /* PrivateKey */
    ExpectNotNull(bio = BIO_new_file("./certs/ecc-key.pem", "rb"));
    ExpectNull((ec = PEM_read_bio_ECPrivateKey(NULL, NULL, NULL, NULL)));
    ec2 = NULL;
    ExpectNotNull((ec = PEM_read_bio_ECPrivateKey(bio, &ec2, NULL, NULL)));
    ExpectIntEQ(ec == ec2, 1);
    ExpectIntEQ(wc_ecc_size((ecc_key*)ec->internal), 32);
    ExpectIntEQ(PEM_write_bio_ECPrivateKey(NULL, NULL, NULL, NULL, 0, NULL,
        NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(PEM_write_bio_ECPrivateKey(bio, NULL, NULL, NULL, 0, NULL,
        NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(PEM_write_bio_ECPrivateKey(NULL, ec, NULL, NULL, 0, NULL, NULL),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    BIO_free(bio);
    bio = NULL;
    /* Public key data - fail. */
    ExpectNotNull(bio = BIO_new_file("./certs/ecc-client-keyPub.pem", "rb"));
    ExpectNull(PEM_read_bio_ECPrivateKey(bio, NULL, NULL, NULL));
    BIO_free(bio);
    bio = NULL;
    ExpectNotNull(bio = wolfSSL_BIO_new(wolfSSL_BIO_s_mem()));
    ExpectIntEQ(PEM_write_bio_ECPrivateKey(bio, ec, NULL, NULL, 0, NULL, \
                                           NULL), WOLFSSL_SUCCESS);
    BIO_free(bio);
    bio = NULL;

    ExpectIntEQ(PEM_write_ECPrivateKey(XBADFILE, NULL, NULL, NULL, 0, NULL,
        NULL),WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(PEM_write_ECPrivateKey(stderr, NULL, NULL, NULL, 0, NULL, NULL),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(PEM_write_ECPrivateKey(XBADFILE, ec, NULL, NULL, 0, NULL, NULL),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(PEM_write_ECPrivateKey(stderr, ec, NULL, NULL, 0, NULL, NULL),
        WOLFSSL_SUCCESS);

    ExpectIntEQ(wolfSSL_PEM_write_mem_ECPrivateKey(NULL, NULL, NULL, 0, NULL,
        NULL), 0);
#if defined(WOLFSSL_PEM_TO_DER) || defined(WOLFSSL_DER_TO_PEM)
    ExpectIntEQ(wolfSSL_PEM_write_mem_ECPrivateKey(ec, NULL, NULL, 0, NULL,
        NULL), 0);
    ExpectIntEQ(wolfSSL_PEM_write_mem_ECPrivateKey(NULL, NULL, NULL, 0, &pem,
        NULL), 0);
    ExpectIntEQ(wolfSSL_PEM_write_mem_ECPrivateKey(NULL, NULL, NULL, 0, NULL,
        &pLen), 0);
    ExpectIntEQ(wolfSSL_PEM_write_mem_ECPrivateKey(NULL, NULL, NULL, 0, &pem,
        &pLen), 0);
    ExpectIntEQ(wolfSSL_PEM_write_mem_ECPrivateKey(ec, NULL, NULL, 0, NULL,
        &pLen), 0);
    ExpectIntEQ(wolfSSL_PEM_write_mem_ECPrivateKey(ec, NULL, NULL, 0, &pem,
        NULL), 0);
    ExpectIntEQ(wolfSSL_PEM_write_mem_ECPrivateKey(ec, NULL, NULL, 0, &pem,
        &pLen), 1);
    ExpectIntGT(pLen, 0);
    XFREE(pem, NULL, DYNAMIC_TYPE_TMP_BUFFER);
#endif

    EC_KEY_free(ec);
    ec = NULL;

    /* PUBKEY */
    ExpectNotNull(bio = BIO_new_file("./certs/ecc-client-keyPub.pem", "rb"));
    ExpectNull((ec = PEM_read_bio_EC_PUBKEY(NULL, NULL, NULL, NULL)));
    ec2 = NULL;
    ExpectNotNull((ec = PEM_read_bio_EC_PUBKEY(bio, &ec2, NULL, NULL)));
    ExpectIntEQ(ec == ec2, 1);
    ExpectIntEQ(wc_ecc_size((ecc_key*)ec->internal), 32);
    ExpectIntEQ(PEM_write_bio_EC_PUBKEY(NULL, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    BIO_free(bio);
    bio = NULL;
    /* Test 0x30, 0x00 fails. */
    ExpectNotNull(bio = BIO_new_mem_buf((unsigned char*)ec_key_bad_1,
        sizeof(ec_key_bad_1)));
    ExpectNull(PEM_read_bio_EC_PUBKEY(bio, NULL, NULL, NULL));
    BIO_free(bio);
    bio = NULL;

    /* Private key data - fail. */
    ExpectNotNull(bio = BIO_new_file("./certs/ecc-key.pem", "rb"));
    ExpectNull(PEM_read_bio_EC_PUBKEY(bio, NULL, NULL, NULL));
    BIO_free(bio);
    bio = NULL;
    ExpectNotNull(bio = wolfSSL_BIO_new(wolfSSL_BIO_s_mem()));
    ExpectIntEQ(PEM_write_bio_EC_PUBKEY(bio, ec), WOLFSSL_SUCCESS);
    BIO_free(bio);
    bio = NULL;

    /* Same test as above, but with a file pointer rather than a BIO. */
    ExpectIntEQ(PEM_write_EC_PUBKEY(NULL, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(PEM_write_EC_PUBKEY(NULL, ec), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(PEM_write_EC_PUBKEY(stderr, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(PEM_write_EC_PUBKEY(stderr, ec), WOLFSSL_SUCCESS);

    EC_KEY_free(ec);
    ec = NULL;

    #ifndef NO_RSA
    /* ensure that non-ec keys do not work */
    ExpectNotNull(bio = BIO_new_file(svrKeyFile, "rb")); /* rsa key */
    ExpectNull((ec = PEM_read_bio_ECPrivateKey(bio, NULL, NULL, NULL)));
    ExpectNull((ec = PEM_read_bio_EC_PUBKEY(bio, NULL, NULL, NULL)));
    BIO_free(bio);
    bio = NULL;
    EC_KEY_free(ec);
    ec = NULL;
    #endif /* !NO_RSA */
    /* Test 0x30, 0x00 fails. */
    ExpectNotNull(bio = BIO_new_mem_buf((unsigned char*)ec_priv_key_bad_1,
        sizeof(ec_priv_key_bad_1)));
    ExpectNull(PEM_read_bio_ECPrivateKey(bio, NULL, NULL, NULL));
    BIO_free(bio);
    bio = NULL;
#endif /* defined(OPENSSL_EXTRA) && !defined(NO_CERTS) */
    return EXPECT_RESULT();
}

static int test_wolfSSL_PEM_PUBKEY(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && defined(HAVE_ECC)
    BIO*      bio = NULL;
    EVP_PKEY* pkey  = NULL;

    /* test creating new EVP_PKEY with bad arg */
    ExpectNull((pkey = PEM_read_bio_PUBKEY(NULL, NULL, NULL, NULL)));

    /* test loading ECC key using BIO */
#if defined(HAVE_ECC) && !defined(NO_FILESYSTEM)
    {
        XFILE file = XBADFILE;
        const char* fname = "./certs/ecc-client-keyPub.pem";
        size_t sz = 0;
        byte* buf = NULL;

        EVP_PKEY* pkey2 = NULL;
        EC_KEY*   ec_key = NULL;

        ExpectTrue((file = XFOPEN(fname, "rb")) != XBADFILE);
        ExpectIntEQ(XFSEEK(file, 0, XSEEK_END), 0);
        ExpectIntGT(sz = XFTELL(file), 0);
        ExpectIntEQ(XFSEEK(file, 0, XSEEK_SET), 0);
        ExpectNotNull(buf = (byte*)XMALLOC(sz, NULL, DYNAMIC_TYPE_FILE));
        if (buf != NULL) {
            ExpectIntEQ(XFREAD(buf, 1, sz, file), sz);
        }
        if (file != XBADFILE) {
            XFCLOSE(file);
        }

        /* Test using BIO new mem and loading PEM private key */
        ExpectNotNull(bio = BIO_new_mem_buf(buf, (int)sz));
        ExpectNotNull((pkey = PEM_read_bio_PUBKEY(bio, NULL, NULL, NULL)));
        BIO_free(bio);
        bio = NULL;
        EVP_PKEY_free(pkey);
        pkey = NULL;
        ExpectNotNull(bio = BIO_new_mem_buf(buf, (int)sz));
        ExpectNotNull(pkey = EVP_PKEY_new());
        ExpectPtrEq(PEM_read_bio_PUBKEY(bio, &pkey, NULL, NULL), pkey);
        XFREE(buf, NULL, DYNAMIC_TYPE_FILE);
        BIO_free(bio);
        bio = NULL;

        /* Qt unit test case*/
        ExpectNotNull(pkey2 = EVP_PKEY_new());
        ExpectNotNull(ec_key = EVP_PKEY_get1_EC_KEY(pkey));
        ExpectIntEQ(EVP_PKEY_set1_EC_KEY(pkey2, ec_key), WOLFSSL_SUCCESS);
    #ifdef WOLFSSL_ERROR_CODE_OPENSSL
        ExpectIntEQ(EVP_PKEY_cmp(pkey, pkey2), 1/* match */);
    #else
        ExpectIntEQ(EVP_PKEY_cmp(pkey, pkey2), 0);
    #endif

        EC_KEY_free(ec_key);
        EVP_PKEY_free(pkey2);
        EVP_PKEY_free(pkey);
        pkey = NULL;
    }
#endif

    (void)bio;
    (void)pkey;
#endif
    return EXPECT_RESULT();
}

#endif /* !NO_BIO */

static int test_wolfSSL_tmp_dh(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_FILESYSTEM) && \
    !defined(NO_RSA) && !defined(NO_DH) && !defined(NO_BIO) && \
    !defined(NO_TLS) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
    byte buff[6000];
    static const unsigned char p[] = {
        0xb0, 0xa1, 0x08, 0x06, 0x9c, 0x08, 0x13, 0xba,
        0x59, 0x06, 0x3c, 0xbc, 0x30, 0xd5, 0xf5, 0x00,
        0xc1, 0x4f, 0x44, 0xa7, 0xd6, 0xef, 0x4a, 0xc6,
        0x25, 0x27, 0x1c, 0xe8, 0xd2, 0x96, 0x53, 0x0a,
        0x5c, 0x91, 0xdd, 0xa2, 0xc2, 0x94, 0x84, 0xbf,
        0x7d, 0xb2, 0x44, 0x9f, 0x9b, 0xd2, 0xc1, 0x8a,
        0xc5, 0xbe, 0x72, 0x5c, 0xa7, 0xe7, 0x91, 0xe6,
        0xd4, 0x9f, 0x73, 0x07, 0x85, 0x5b, 0x66, 0x48,
        0xc7, 0x70, 0xfa, 0xb4, 0xee, 0x02, 0xc9, 0x3d,
        0x9a, 0x4a, 0xda, 0x3d, 0xc1, 0x46, 0x3e, 0x19,
        0x69, 0xd1, 0x17, 0x46, 0x07, 0xa3, 0x4d, 0x9f,
        0x2b, 0x96, 0x17, 0x39, 0x6d, 0x30, 0x8d, 0x2a,
        0xf3, 0x94, 0xd3, 0x75, 0xcf, 0xa0, 0x75, 0xe6,
        0xf2, 0x92, 0x1f, 0x1a, 0x70, 0x05, 0xaa, 0x04,
        0x83, 0x57, 0x30, 0xfb, 0xda, 0x76, 0x93, 0x38,
        0x50, 0xe8, 0x27, 0xfd, 0x63, 0xee, 0x3c, 0xe5,
        0xb7, 0xc8, 0x09, 0xae, 0x6f, 0x50, 0x35, 0x8e,
        0x84, 0xce, 0x4a, 0x00, 0xe9, 0x12, 0x7e, 0x5a,
        0x31, 0xd7, 0x33, 0xfc, 0x21, 0x13, 0x76, 0xcc,
        0x16, 0x30, 0xdb, 0x0c, 0xfc, 0xc5, 0x62, 0xa7,
        0x35, 0xb8, 0xef, 0xb7, 0xb0, 0xac, 0xc0, 0x36,
        0xf6, 0xd9, 0xc9, 0x46, 0x48, 0xf9, 0x40, 0x90,
        0x00, 0x2b, 0x1b, 0xaa, 0x6c, 0xe3, 0x1a, 0xc3,
        0x0b, 0x03, 0x9e, 0x1b, 0xc2, 0x46, 0xe4, 0x48,
        0x4e, 0x22, 0x73, 0x6f, 0xc3, 0x5f, 0xd4, 0x9a,
        0xd6, 0x30, 0x07, 0x48, 0xd6, 0x8c, 0x90, 0xab,
        0xd4, 0xf6, 0xf1, 0xe3, 0x48, 0xd3, 0x58, 0x4b,
        0xa6, 0xb9, 0xcd, 0x29, 0xbf, 0x68, 0x1f, 0x08,
        0x4b, 0x63, 0x86, 0x2f, 0x5c, 0x6b, 0xd6, 0xb6,
        0x06, 0x65, 0xf7, 0xa6, 0xdc, 0x00, 0x67, 0x6b,
        0xbb, 0xc3, 0xa9, 0x41, 0x83, 0xfb, 0xc7, 0xfa,
        0xc8, 0xe2, 0x1e, 0x7e, 0xaf, 0x00, 0x3f, 0x93
    };
    int pSz = (int)sizeof(p);
#if !defined(WOLFSSL_OLD_PRIME_CHECK) && !defined(HAVE_FIPS) && \
    !defined(HAVE_SELFTEST)
    static const unsigned char bad_p[] = {
        0xb0, 0xa1, 0x08, 0x06, 0x9c, 0x08, 0x13, 0xba,
        0x59, 0x06, 0x3c, 0xbc, 0x30, 0xd5, 0xf5, 0x00,
        0xc1, 0x4f, 0x44, 0xa7, 0xd6, 0xef, 0x4a, 0xc6,
        0x25, 0x27, 0x1c, 0xe8, 0xd2, 0x96, 0x53, 0x0a,
        0x5c, 0x91, 0xdd, 0xa2, 0xc2, 0x94, 0x84, 0xbf,
        0x7d, 0xb2, 0x44, 0x9f, 0x9b, 0xd2, 0xc1, 0x8a,
        0xc5, 0xbe, 0x72, 0x5c, 0xa7, 0xe7, 0x91, 0xe6,
        0xd4, 0x9f, 0x73, 0x07, 0x85, 0x5b, 0x66, 0x48,
        0xc7, 0x70, 0xfa, 0xb4, 0xee, 0x02, 0xc9, 0x3d,
        0x9a, 0x4a, 0xda, 0x3d, 0xc1, 0x46, 0x3e, 0x19,
        0x69, 0xd1, 0x17, 0x46, 0x07, 0xa3, 0x4d, 0x9f,
        0x2b, 0x96, 0x17, 0x39, 0x6d, 0x30, 0x8d, 0x2a,
        0xf3, 0x94, 0xd3, 0x75, 0xcf, 0xa0, 0x75, 0xe6,
        0xf2, 0x92, 0x1f, 0x1a, 0x70, 0x05, 0xaa, 0x04,
        0x83, 0x57, 0x30, 0xfb, 0xda, 0x76, 0x93, 0x38,
        0x50, 0xe8, 0x27, 0xfd, 0x63, 0xee, 0x3c, 0xe5,
        0xb7, 0xc8, 0x09, 0xae, 0x6f, 0x50, 0x35, 0x8e,
        0x84, 0xce, 0x4a, 0x00, 0xe9, 0x12, 0x7e, 0x5a,
        0x31, 0xd7, 0x33, 0xfc, 0x21, 0x13, 0x76, 0xcc,
        0x16, 0x30, 0xdb, 0x0c, 0xfc, 0xc5, 0x62, 0xa7,
        0x35, 0xb8, 0xef, 0xb7, 0xb0, 0xac, 0xc0, 0x36,
        0xf6, 0xd9, 0xc9, 0x46, 0x48, 0xf9, 0x40, 0x90,
        0x00, 0x2b, 0x1b, 0xaa, 0x6c, 0xe3, 0x1a, 0xc3,
        0x0b, 0x03, 0x9e, 0x1b, 0xc2, 0x46, 0xe4, 0x48,
        0x4e, 0x22, 0x73, 0x6f, 0xc3, 0x5f, 0xd4, 0x9a,
        0xd6, 0x30, 0x07, 0x48, 0xd6, 0x8c, 0x90, 0xab,
        0xd4, 0xf6, 0xf1, 0xe3, 0x48, 0xd3, 0x58, 0x4b,
        0xa6, 0xb9, 0xcd, 0x29, 0xbf, 0x68, 0x1f, 0x08,
        0x4b, 0x63, 0x86, 0x2f, 0x5c, 0x6b, 0xd6, 0xb6,
        0x06, 0x65, 0xf7, 0xa6, 0xdc, 0x00, 0x67, 0x6b,
        0xbb, 0xc3, 0xa9, 0x41, 0x83, 0xfb, 0xc7, 0xfa,
        0xc8, 0xe2, 0x1e, 0x7e, 0xaf, 0x00, 0x3f, 0x91
    };
#endif
    static const unsigned char g[] = { 0x02 };
    int gSz = (int)sizeof(g);
#if !defined(NO_DSA)
    char file[] = "./certs/dsaparams.pem";
    DSA* dsa = NULL;
#else
    char file[] = "./certs/dh2048.pem";
#endif
    XFILE f = XBADFILE;
    int  bytes = 0;
    DH*  dh = NULL;
    DH*  dh2 = NULL;
    BIO*     bio = NULL;
    SSL*     ssl = NULL;
    SSL_CTX* ctx = NULL;
#ifndef NO_WOLFSSL_CLIENT
    SSL*     ssl_c = NULL;
    SSL_CTX* ctx_c = NULL;
#endif

#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_server_method()));
    ExpectTrue(SSL_CTX_use_certificate_file(ctx, svrCertFile,
        WOLFSSL_FILETYPE_PEM));
    ExpectTrue(SSL_CTX_use_PrivateKey_file(ctx, svrKeyFile,
        WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(ssl = SSL_new(ctx));
#endif
#ifndef NO_WOLFSSL_CLIENT
    ExpectNotNull(ctx_c = SSL_CTX_new(wolfSSLv23_client_method()));
    ExpectTrue(SSL_CTX_use_certificate_file(ctx_c, svrCertFile,
        WOLFSSL_FILETYPE_PEM));
    ExpectTrue(SSL_CTX_use_PrivateKey_file(ctx_c, svrKeyFile,
        WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(ssl_c = SSL_new(ctx_c));
#ifdef NO_WOLFSSL_SERVER
    ctx = ctx_c;
    ssl = ssl_c;
#endif
#endif

    XMEMSET(buff, 0, sizeof(buff));
    ExpectTrue((f = XFOPEN(file, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buff, 1, sizeof(buff), f), 0);
    if (f != XBADFILE)
        XFCLOSE(f);

    ExpectNotNull(bio = BIO_new_mem_buf((void*)buff, bytes));

#if !defined(NO_DSA)
    dsa = wolfSSL_PEM_read_bio_DSAparams(bio, NULL, NULL, NULL);
    ExpectNotNull(dsa);

    dh = wolfSSL_DSA_dup_DH(dsa);
#else
    dh = wolfSSL_PEM_read_bio_DHparams(bio, NULL, NULL, NULL);
#endif
    ExpectNotNull(dh);
#if defined(WOLFSSL_DH_EXTRA) && \
    (defined(WOLFSSL_QT) || defined(OPENSSL_ALL) || defined(WOLFSSL_OPENSSH))
    ExpectNotNull(dh2 = wolfSSL_DH_dup(dh));
    DH_free(dh2);
    dh2 = NULL;
#endif

    /* Failure cases */
    ExpectIntEQ((int)wolfSSL_CTX_SetTmpDH(NULL, NULL, 0, NULL, 0),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ((int)wolfSSL_CTX_SetTmpDH(ctx , NULL, 0, NULL, 0),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ((int)wolfSSL_CTX_SetTmpDH(NULL, p   , 0, NULL, 0),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ((int)wolfSSL_CTX_SetTmpDH(NULL, NULL, 0, g   , 0),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ((int)wolfSSL_CTX_SetTmpDH(ctx , p  , 0, NULL, 0),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ((int)wolfSSL_CTX_SetTmpDH(ctx , NULL, 0, g   , 0),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ((int)wolfSSL_CTX_SetTmpDH(NULL, p   , 0, g   , 0),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ((int)wolfSSL_CTX_SetTmpDH(ctx , p   , 1, g   , 1),
        WC_NO_ERR_TRACE(DH_KEY_SIZE_E));
    ExpectIntEQ((int)wolfSSL_CTX_SetTmpDH(ctx , buff, 6000, g   , 1),
        WC_NO_ERR_TRACE(DH_KEY_SIZE_E));
#if !defined(WOLFSSL_OLD_PRIME_CHECK) && !defined(HAVE_FIPS) && \
    !defined(HAVE_SELFTEST)
    ExpectIntEQ((int)wolfSSL_CTX_SetTmpDH(ctx, bad_p, pSz, g, gSz),
        WC_NO_ERR_TRACE(DH_CHECK_PUB_E));
#endif
    ExpectIntEQ((int)wolfSSL_SetTmpDH(NULL, NULL, 0, NULL, 0),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ((int)wolfSSL_SetTmpDH(ssl , NULL, 0, NULL, 0),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ((int)wolfSSL_SetTmpDH(NULL, p   , 0, NULL, 0),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ((int)wolfSSL_SetTmpDH(NULL, NULL, 0, g   , 0),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ((int)wolfSSL_SetTmpDH(ssl , p  , 0, NULL, 0),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ((int)wolfSSL_SetTmpDH(ssl , NULL, 0, g   , 0),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ((int)wolfSSL_SetTmpDH(NULL, p   , 0, g   , 0),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ((int)wolfSSL_SetTmpDH(ssl , p   , 1, g   , 1),
        WC_NO_ERR_TRACE(DH_KEY_SIZE_E));
    ExpectIntEQ((int)wolfSSL_SetTmpDH(ssl , buff, 6000, g   , 1),
        WC_NO_ERR_TRACE(DH_KEY_SIZE_E));
#if !defined(WOLFSSL_OLD_PRIME_CHECK) && !defined(HAVE_FIPS) && \
    !defined(HAVE_SELFTEST)
#ifndef NO_WOLFSSL_SERVER
    /* Parameters will be tested later so it passes now. */
    ExpectIntEQ((int)wolfSSL_SetTmpDH(ssl, bad_p, pSz, g, gSz),
        WOLFSSL_SUCCESS);
#endif
#endif
#ifndef NO_WOLFSSL_CLIENT
    ExpectIntEQ((int)wolfSSL_SetTmpDH(ssl_c, p, pSz, g, gSz),
        WC_NO_ERR_TRACE(SIDE_ERROR));
#endif
    ExpectIntEQ((int)SSL_CTX_set_tmp_dh(NULL, NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ((int)SSL_CTX_set_tmp_dh(ctx , NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ((int)SSL_CTX_set_tmp_dh(NULL, dh  ), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ((int)SSL_set_tmp_dh(NULL, NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ((int)SSL_set_tmp_dh(ssl , NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ((int)SSL_set_tmp_dh(NULL, dh  ), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    /* No p/g to use. */
    dh2 = wolfSSL_DH_new();
    ExpectIntEQ((int)SSL_CTX_set_tmp_dh(ctx , dh2 ), WC_NO_ERR_TRACE(WOLFSSL_FATAL_ERROR));
    ExpectIntEQ((int)SSL_set_tmp_dh(ssl , dh2 ), WC_NO_ERR_TRACE(WOLFSSL_FATAL_ERROR));
    DH_free(dh2);
    dh2 = NULL;

    ExpectIntEQ((int)wolfSSL_CTX_SetTmpDH(ctx, p, pSz, g, gSz),
        WOLFSSL_SUCCESS);
    ExpectIntEQ((int)SSL_CTX_set_tmp_dh(ctx, dh), WOLFSSL_SUCCESS);
#ifndef NO_WOLFSSL_SERVER
    ExpectIntEQ((int)SSL_set_tmp_dh(ssl, dh), WOLFSSL_SUCCESS);
#else
    ExpectIntEQ((int)SSL_set_tmp_dh(ssl, dh), WC_NO_ERR_TRACE(SIDE_ERROR));
#endif

    BIO_free(bio);
#if !defined(NO_DSA)
    DSA_free(dsa);
#endif
    DH_free(dh);
    dh = NULL;
#ifndef NO_WOLFSSL_CLIENT
    if (ssl != ssl_c) {
        SSL_free(ssl_c);
    }
#endif
    SSL_free(ssl);
#ifndef NO_WOLFSSL_CLIENT
    if (ctx != ctx_c) {
        SSL_CTX_free(ctx_c);
    }
#endif
    SSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_ctrl(void)
{
    EXPECT_DECLS;
#if defined (OPENSSL_EXTRA) && !defined(NO_BIO)
    byte buff[6000];
    BIO* bio = NULL;
    int  bytes;
    BUF_MEM* ptr = NULL;

    XMEMSET(buff, 0, sizeof(buff));

    bytes = sizeof(buff);
    ExpectNotNull(bio = BIO_new_mem_buf((void*)buff, bytes));
    ExpectNotNull(BIO_s_socket());

    ExpectIntEQ((int)wolfSSL_BIO_get_mem_ptr(bio, &ptr), WOLFSSL_SUCCESS);

    /* needs tested after stubs filled out @TODO
        SSL_ctrl
        SSL_CTX_ctrl
    */

    BIO_free(bio);
#endif /* defined(OPENSSL_EXTRA) && !defined(NO_BIO) */
    return EXPECT_RESULT();
}


static int test_wolfSSL_EVP_PKEY_new_mac_key(void)
{
    EXPECT_DECLS;
#ifdef OPENSSL_EXTRA
    static const unsigned char pw[] = "password";
    static const int pwSz = sizeof(pw) - 1;
    size_t checkPwSz = 0;
    const unsigned char* checkPw = NULL;
    WOLFSSL_EVP_PKEY* key = NULL;

    ExpectNull(key = wolfSSL_EVP_PKEY_new_mac_key(0, NULL, pw, pwSz));
    ExpectNull(key = wolfSSL_EVP_PKEY_new_mac_key(0, NULL, NULL, pwSz));

    ExpectNotNull(key = wolfSSL_EVP_PKEY_new_mac_key(EVP_PKEY_HMAC, NULL, pw,
        pwSz));
    if (key != NULL) {
        ExpectIntEQ(key->type, EVP_PKEY_HMAC);
        ExpectIntEQ(key->save_type, EVP_PKEY_HMAC);
        ExpectIntEQ(key->pkey_sz, pwSz);
        ExpectIntEQ(XMEMCMP(key->pkey.ptr, pw, pwSz), 0);
    }
    ExpectNotNull(checkPw = wolfSSL_EVP_PKEY_get0_hmac(key, &checkPwSz));
    ExpectIntEQ((int)checkPwSz, pwSz);
    ExpectIntEQ(XMEMCMP(checkPw, pw, pwSz), 0);
    wolfSSL_EVP_PKEY_free(key);
    key = NULL;

    ExpectNotNull(key = wolfSSL_EVP_PKEY_new_mac_key(EVP_PKEY_HMAC, NULL, pw,
        0));
    ExpectIntEQ(key->pkey_sz, 0);
    if (EXPECT_SUCCESS()) {
        /* Allocation for key->pkey.ptr may fail - OK key len is 0 */
        checkPw = wolfSSL_EVP_PKEY_get0_hmac(key, &checkPwSz);
    }
    ExpectTrue((checkPwSz == 0) || (checkPw != NULL));
    ExpectIntEQ((int)checkPwSz, 0);
    wolfSSL_EVP_PKEY_free(key);
    key = NULL;

    ExpectNotNull(key = wolfSSL_EVP_PKEY_new_mac_key(EVP_PKEY_HMAC, NULL, NULL,
        0));
    ExpectIntEQ(key->pkey_sz, 0);
    if (EXPECT_SUCCESS()) {
        /* Allocation for key->pkey.ptr may fail - OK key len is 0 */
        checkPw = wolfSSL_EVP_PKEY_get0_hmac(key, &checkPwSz);
    }
    ExpectTrue((checkPwSz == 0) || (checkPw != NULL));
    ExpectIntEQ((int)checkPwSz, 0);
    wolfSSL_EVP_PKEY_free(key);
    key = NULL;
#endif /* OPENSSL_EXTRA */
    return EXPECT_RESULT();
}


static int test_wolfSSL_EVP_PKEY_new_CMAC_key(void)
{
    EXPECT_DECLS;
#ifdef OPENSSL_EXTRA
#if defined(WOLFSSL_CMAC) && !defined(NO_AES) && \
    defined(WOLFSSL_AES_DIRECT) && defined(WOLFSSL_AES_128)
    const char *priv = "ABCDEFGHIJKLMNOP";
    const WOLFSSL_EVP_CIPHER* cipher = EVP_aes_128_cbc();
    WOLFSSL_EVP_PKEY* key = NULL;

    ExpectNull(key = wolfSSL_EVP_PKEY_new_CMAC_key(
        NULL, NULL, AES_128_KEY_SIZE, cipher));
    ExpectNull(key = wolfSSL_EVP_PKEY_new_CMAC_key(
        NULL, (const unsigned char *)priv, 0, cipher));
    ExpectNull(key = wolfSSL_EVP_PKEY_new_CMAC_key(
        NULL, (const unsigned char *)priv, AES_128_KEY_SIZE, NULL));

    ExpectNotNull(key = wolfSSL_EVP_PKEY_new_CMAC_key(
        NULL, (const unsigned char *)priv, AES_128_KEY_SIZE, cipher));
    wolfSSL_EVP_PKEY_free(key);
#endif /* WOLFSSL_CMAC && !NO_AES && WOLFSSL_AES_DIRECT && WOLFSSL_AES_128 */
#endif /* OPENSSL_EXTRA */
    return EXPECT_RESULT();
}

static int test_wolfSSL_EVP_Digest(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_SHA256) && !defined(NO_PWDBASED)
    const char* in = "abc";
    int   inLen = (int)XSTRLEN(in);
    byte  out[WC_SHA256_DIGEST_SIZE];
    unsigned int outLen;
    const char* expOut =
        "\xBA\x78\x16\xBF\x8F\x01\xCF\xEA\x41\x41\x40\xDE\x5D\xAE\x22"
        "\x23\xB0\x03\x61\xA3\x96\x17\x7A\x9C\xB4\x10\xFF\x61\xF2\x00"
        "\x15\xAD";

    ExpectIntEQ(wolfSSL_EVP_Digest((unsigned char*)in, inLen, out, &outLen,
        "SHA256", NULL), 1);
    ExpectIntEQ(outLen, WC_SHA256_DIGEST_SIZE);
    ExpectIntEQ(XMEMCMP(out, expOut, WC_SHA256_DIGEST_SIZE), 0);
#endif /* OPEN_EXTRA && ! NO_SHA256 */
    return EXPECT_RESULT();
}

static int test_wolfSSL_EVP_Digest_all(void)
{
    EXPECT_DECLS;
#ifdef OPENSSL_EXTRA
    const char* digests[] = {
#ifndef NO_MD5
        "MD5",
#endif
#ifndef NO_SHA
        "SHA",
#endif
#ifdef WOLFSSL_SHA224
        "SHA224",
#endif
#ifndef NO_SHA256
        "SHA256",
#endif
#ifdef WOLFSSL_SHA384
        "SHA384",
#endif
#ifdef WOLFSSL_SHA512
        "SHA512",
#endif
#if defined(WOLFSSL_SHA512) && !defined(WOLFSSL_NOSHA512_224)
        "SHA512-224",
#endif
#if defined(WOLFSSL_SHA512) && !defined(WOLFSSL_NOSHA512_256)
        "SHA512-256",
#endif
#ifdef WOLFSSL_SHA3
#ifndef WOLFSSL_NOSHA3_224
        "SHA3-224",
#endif
#ifndef WOLFSSL_NOSHA3_256
        "SHA3-256",
#endif
        "SHA3-384",
#ifndef WOLFSSL_NOSHA3_512
        "SHA3-512",
#endif
#endif /* WOLFSSL_SHA3 */
        NULL
    };
    const char** d;
    const unsigned char in[] = "abc";
    int   inLen = XSTR_SIZEOF(in);
    byte  out[WC_MAX_DIGEST_SIZE];
    unsigned int outLen;

    for (d = digests; *d != NULL; d++) {
        ExpectIntEQ(EVP_Digest(in, inLen, out, &outLen, *d, NULL), 1);
        ExpectIntGT(outLen, 0);
        ExpectIntEQ(EVP_MD_size(*d), outLen);
    }
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_EVP_MD_size(void)
{
    EXPECT_DECLS;
#ifdef OPENSSL_EXTRA
    WOLFSSL_EVP_MD_CTX mdCtx;

#ifdef WOLFSSL_SHA3
#ifndef WOLFSSL_NOSHA3_224
    wolfSSL_EVP_MD_CTX_init(&mdCtx);

    ExpectIntEQ(wolfSSL_EVP_DigestInit(&mdCtx, "SHA3-224"), 1);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_size(&mdCtx), WC_SHA3_224_DIGEST_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_block_size(&mdCtx), WC_SHA3_224_BLOCK_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_cleanup(&mdCtx), 1);
#endif
#ifndef WOLFSSL_NOSHA3_256
    wolfSSL_EVP_MD_CTX_init(&mdCtx);

    ExpectIntEQ(wolfSSL_EVP_DigestInit(&mdCtx, "SHA3-256"), 1);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_size(&mdCtx), WC_SHA3_256_DIGEST_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_block_size(&mdCtx), WC_SHA3_256_BLOCK_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_cleanup(&mdCtx), 1);
#endif
    wolfSSL_EVP_MD_CTX_init(&mdCtx);

    ExpectIntEQ(wolfSSL_EVP_DigestInit(&mdCtx, "SHA3-384"), 1);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_size(&mdCtx), WC_SHA3_384_DIGEST_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_block_size(&mdCtx), WC_SHA3_384_BLOCK_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_cleanup(&mdCtx), 1);
#ifndef WOLFSSL_NOSHA3_512
    wolfSSL_EVP_MD_CTX_init(&mdCtx);

    ExpectIntEQ(wolfSSL_EVP_DigestInit(&mdCtx, "SHA3-512"), 1);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_size(&mdCtx), WC_SHA3_512_DIGEST_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_block_size(&mdCtx), WC_SHA3_512_BLOCK_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_cleanup(&mdCtx), 1);
#endif
#endif /* WOLFSSL_SHA3 */

#ifndef NO_SHA256
    wolfSSL_EVP_MD_CTX_init(&mdCtx);

    ExpectIntEQ(wolfSSL_EVP_DigestInit(&mdCtx, "SHA256"), 1);
    ExpectIntEQ(wolfSSL_EVP_MD_size(wolfSSL_EVP_MD_CTX_md(&mdCtx)),
        WC_SHA256_DIGEST_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_block_size(wolfSSL_EVP_MD_CTX_md(&mdCtx)),
        WC_SHA256_BLOCK_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_size(&mdCtx), WC_SHA256_DIGEST_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_block_size(&mdCtx), WC_SHA256_BLOCK_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_cleanup(&mdCtx), 1);

#endif

#ifndef NO_MD5
    wolfSSL_EVP_MD_CTX_init(&mdCtx);

    ExpectIntEQ(wolfSSL_EVP_DigestInit(&mdCtx, "MD5"), 1);
    ExpectIntEQ(wolfSSL_EVP_MD_size(wolfSSL_EVP_MD_CTX_md(&mdCtx)),
        WC_MD5_DIGEST_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_block_size(wolfSSL_EVP_MD_CTX_md(&mdCtx)),
        WC_MD5_BLOCK_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_size(&mdCtx), WC_MD5_DIGEST_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_block_size(&mdCtx), WC_MD5_BLOCK_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_cleanup(&mdCtx), 1);

#endif

#ifdef WOLFSSL_SHA224
    wolfSSL_EVP_MD_CTX_init(&mdCtx);

    ExpectIntEQ(wolfSSL_EVP_DigestInit(&mdCtx, "SHA224"), 1);
    ExpectIntEQ(wolfSSL_EVP_MD_size(wolfSSL_EVP_MD_CTX_md(&mdCtx)),
        WC_SHA224_DIGEST_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_block_size(wolfSSL_EVP_MD_CTX_md(&mdCtx)),
        WC_SHA224_BLOCK_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_size(&mdCtx), WC_SHA224_DIGEST_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_block_size(&mdCtx), WC_SHA224_BLOCK_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_cleanup(&mdCtx), 1);

#endif

#ifdef WOLFSSL_SHA384
    wolfSSL_EVP_MD_CTX_init(&mdCtx);

    ExpectIntEQ(wolfSSL_EVP_DigestInit(&mdCtx, "SHA384"), 1);
    ExpectIntEQ(wolfSSL_EVP_MD_size(wolfSSL_EVP_MD_CTX_md(&mdCtx)),
        WC_SHA384_DIGEST_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_block_size(wolfSSL_EVP_MD_CTX_md(&mdCtx)),
        WC_SHA384_BLOCK_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_size(&mdCtx), WC_SHA384_DIGEST_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_block_size(&mdCtx), WC_SHA384_BLOCK_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_cleanup(&mdCtx), 1);

#endif

#ifdef WOLFSSL_SHA512
    wolfSSL_EVP_MD_CTX_init(&mdCtx);

    ExpectIntEQ(wolfSSL_EVP_DigestInit(&mdCtx, "SHA512"), 1);
    ExpectIntEQ(wolfSSL_EVP_MD_size(wolfSSL_EVP_MD_CTX_md(&mdCtx)),
        WC_SHA512_DIGEST_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_block_size(wolfSSL_EVP_MD_CTX_md(&mdCtx)),
        WC_SHA512_BLOCK_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_size(&mdCtx), WC_SHA512_DIGEST_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_block_size(&mdCtx), WC_SHA512_BLOCK_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_cleanup(&mdCtx), 1);

#endif

#ifndef NO_SHA
    wolfSSL_EVP_MD_CTX_init(&mdCtx);

    ExpectIntEQ(wolfSSL_EVP_DigestInit(&mdCtx, "SHA"), 1);
    ExpectIntEQ(wolfSSL_EVP_MD_size(wolfSSL_EVP_MD_CTX_md(&mdCtx)),
        WC_SHA_DIGEST_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_block_size(wolfSSL_EVP_MD_CTX_md(&mdCtx)),
        WC_SHA_BLOCK_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_size(&mdCtx), WC_SHA_DIGEST_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_block_size(&mdCtx), WC_SHA_BLOCK_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_cleanup(&mdCtx), 1);

    wolfSSL_EVP_MD_CTX_init(&mdCtx);

    ExpectIntEQ(wolfSSL_EVP_DigestInit(&mdCtx, "SHA1"), 1);
    ExpectIntEQ(wolfSSL_EVP_MD_size(wolfSSL_EVP_MD_CTX_md(&mdCtx)),
        WC_SHA_DIGEST_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_block_size(wolfSSL_EVP_MD_CTX_md(&mdCtx)),
        WC_SHA_BLOCK_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_size(&mdCtx), WC_SHA_DIGEST_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_block_size(&mdCtx), WC_SHA_BLOCK_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_cleanup(&mdCtx), 1);
#endif
    /* error case */
    wolfSSL_EVP_MD_CTX_init(&mdCtx);

    ExpectIntEQ(wolfSSL_EVP_DigestInit(&mdCtx, ""), 0);
    ExpectIntEQ(wolfSSL_EVP_MD_size(wolfSSL_EVP_MD_CTX_md(&mdCtx)), 0);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_block_size(&mdCtx), 0);
    /* Cleanup is valid on uninit'ed struct */
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_cleanup(&mdCtx), 1);
#endif /* OPENSSL_EXTRA */
    return EXPECT_RESULT();
}

static int test_wolfSSL_EVP_MD_pkey_type(void)
{
    EXPECT_DECLS;
#ifdef OPENSSL_EXTRA
    const WOLFSSL_EVP_MD* md;

#ifndef NO_MD5
    ExpectNotNull(md = EVP_md5());
    ExpectIntEQ(EVP_MD_pkey_type(md), NID_md5WithRSAEncryption);
#endif
#ifndef NO_SHA
    ExpectNotNull(md = EVP_sha1());
    ExpectIntEQ(EVP_MD_pkey_type(md), NID_sha1WithRSAEncryption);
#endif
#ifdef WOLFSSL_SHA224
    ExpectNotNull(md = EVP_sha224());
    ExpectIntEQ(EVP_MD_pkey_type(md), NID_sha224WithRSAEncryption);
#endif
    ExpectNotNull(md = EVP_sha256());
    ExpectIntEQ(EVP_MD_pkey_type(md), NID_sha256WithRSAEncryption);
#ifdef WOLFSSL_SHA384
    ExpectNotNull(md = EVP_sha384());
    ExpectIntEQ(EVP_MD_pkey_type(md), NID_sha384WithRSAEncryption);
#endif
#ifdef WOLFSSL_SHA512
    ExpectNotNull(md = EVP_sha512());
    ExpectIntEQ(EVP_MD_pkey_type(md), NID_sha512WithRSAEncryption);
#endif
#endif
    return EXPECT_RESULT();
}

#ifdef OPENSSL_EXTRA
static int test_hmac_signing(const WOLFSSL_EVP_MD *type, const byte* testKey,
        size_t testKeySz, const char* testData, size_t testDataSz,
        const byte* testResult, size_t testResultSz)
{
    EXPECT_DECLS;
    unsigned char check[WC_MAX_DIGEST_SIZE];
    size_t checkSz = 0;
    WOLFSSL_EVP_PKEY* key = NULL;
    WOLFSSL_EVP_MD_CTX mdCtx;

    ExpectNotNull(key = wolfSSL_EVP_PKEY_new_mac_key(EVP_PKEY_HMAC, NULL,
                                                      testKey, (int)testKeySz));
    wolfSSL_EVP_MD_CTX_init(&mdCtx);
    ExpectIntEQ(wolfSSL_EVP_DigestSignInit(&mdCtx, NULL, type, NULL, key), 1);
    ExpectIntEQ(wolfSSL_EVP_DigestSignUpdate(&mdCtx, testData,
                                                  (unsigned int)testDataSz), 1);
    checkSz = sizeof(check);
    ExpectIntEQ(wolfSSL_EVP_DigestSignFinal(&mdCtx, NULL, &checkSz), 1);
    ExpectIntEQ((int)checkSz, (int)testResultSz);
    checkSz = sizeof(check);
    ExpectIntEQ(wolfSSL_EVP_DigestSignFinal(&mdCtx, check, &checkSz), 1);
    ExpectIntEQ((int)checkSz,(int)testResultSz);
    ExpectIntEQ(XMEMCMP(testResult, check, testResultSz), 0);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_cleanup(&mdCtx), 1);

    ExpectIntEQ(wolfSSL_EVP_DigestVerifyInit(&mdCtx, NULL, type, NULL, key), 1);
    ExpectIntEQ(wolfSSL_EVP_DigestVerifyUpdate(&mdCtx, testData,
                                                  (unsigned int)testDataSz), 1);
    ExpectIntEQ(wolfSSL_EVP_DigestVerifyFinal(&mdCtx, testResult, checkSz), 1);

    ExpectIntEQ(wolfSSL_EVP_MD_CTX_cleanup(&mdCtx), 1);
    wolfSSL_EVP_MD_CTX_init(&mdCtx);
    ExpectIntEQ(wolfSSL_EVP_DigestSignInit(&mdCtx, NULL, type, NULL, key), 1);
    ExpectIntEQ(wolfSSL_EVP_DigestSignUpdate(&mdCtx, testData, 4), 1);
    checkSz = sizeof(check);
    ExpectIntEQ(wolfSSL_EVP_DigestSignFinal(&mdCtx, NULL, &checkSz), 1);
    ExpectIntEQ((int)checkSz, (int)testResultSz);
    checkSz = sizeof(check);
    ExpectIntEQ(wolfSSL_EVP_DigestSignFinal(&mdCtx, check, &checkSz), 1);
    ExpectIntEQ((int)checkSz,(int)testResultSz);
    ExpectIntEQ(wolfSSL_EVP_DigestSignUpdate(&mdCtx, testData + 4,
                                              (unsigned int)testDataSz - 4), 1);
    checkSz = sizeof(check);
    ExpectIntEQ(wolfSSL_EVP_DigestSignFinal(&mdCtx, check, &checkSz), 1);
    ExpectIntEQ((int)checkSz,(int)testResultSz);
    ExpectIntEQ(XMEMCMP(testResult, check, testResultSz), 0);

    ExpectIntEQ(wolfSSL_EVP_MD_CTX_cleanup(&mdCtx), 1);
    ExpectIntEQ(wolfSSL_EVP_DigestVerifyInit(&mdCtx, NULL, type, NULL, key), 1);
    ExpectIntEQ(wolfSSL_EVP_DigestVerifyUpdate(&mdCtx, testData, 4), 1);
    ExpectIntEQ(wolfSSL_EVP_DigestVerifyUpdate(&mdCtx, testData + 4,
                                              (unsigned int)testDataSz - 4), 1);
    ExpectIntEQ(wolfSSL_EVP_DigestVerifyFinal(&mdCtx, testResult, checkSz), 1);

    ExpectIntEQ(wolfSSL_EVP_MD_CTX_cleanup(&mdCtx), 1);

    wolfSSL_EVP_PKEY_free(key);

    return EXPECT_RESULT();
}
#endif

static int test_wolfSSL_EVP_MD_hmac_signing(void)
{
    EXPECT_DECLS;
#ifdef OPENSSL_EXTRA
    static const unsigned char testKey[] =
    {
        0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
        0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
        0x0b, 0x0b, 0x0b, 0x0b
    };
    static const char testData[] = "Hi There";
#ifdef WOLFSSL_SHA224
    static const unsigned char testResultSha224[] =
    {
        0x89, 0x6f, 0xb1, 0x12, 0x8a, 0xbb, 0xdf, 0x19,
        0x68, 0x32, 0x10, 0x7c, 0xd4, 0x9d, 0xf3, 0x3f,
        0x47, 0xb4, 0xb1, 0x16, 0x99, 0x12, 0xba, 0x4f,
        0x53, 0x68, 0x4b, 0x22
    };
#endif
#ifndef NO_SHA256
    static const unsigned char testResultSha256[] =
    {
        0xb0, 0x34, 0x4c, 0x61, 0xd8, 0xdb, 0x38, 0x53,
        0x5c, 0xa8, 0xaf, 0xce, 0xaf, 0x0b, 0xf1, 0x2b,
        0x88, 0x1d, 0xc2, 0x00, 0xc9, 0x83, 0x3d, 0xa7,
        0x26, 0xe9, 0x37, 0x6c, 0x2e, 0x32, 0xcf, 0xf7
    };
#endif
#ifdef WOLFSSL_SHA384
    static const unsigned char testResultSha384[] =
    {
        0xaf, 0xd0, 0x39, 0x44, 0xd8, 0x48, 0x95, 0x62,
        0x6b, 0x08, 0x25, 0xf4, 0xab, 0x46, 0x90, 0x7f,
        0x15, 0xf9, 0xda, 0xdb, 0xe4, 0x10, 0x1e, 0xc6,
        0x82, 0xaa, 0x03, 0x4c, 0x7c, 0xeb, 0xc5, 0x9c,
        0xfa, 0xea, 0x9e, 0xa9, 0x07, 0x6e, 0xde, 0x7f,
        0x4a, 0xf1, 0x52, 0xe8, 0xb2, 0xfa, 0x9c, 0xb6
    };
#endif
#ifdef WOLFSSL_SHA512
    static const unsigned char testResultSha512[] =
    {
        0x87, 0xaa, 0x7c, 0xde, 0xa5, 0xef, 0x61, 0x9d,
        0x4f, 0xf0, 0xb4, 0x24, 0x1a, 0x1d, 0x6c, 0xb0,
        0x23, 0x79, 0xf4, 0xe2, 0xce, 0x4e, 0xc2, 0x78,
        0x7a, 0xd0, 0xb3, 0x05, 0x45, 0xe1, 0x7c, 0xde,
        0xda, 0xa8, 0x33, 0xb7, 0xd6, 0xb8, 0xa7, 0x02,
        0x03, 0x8b, 0x27, 0x4e, 0xae, 0xa3, 0xf4, 0xe4,
        0xbe, 0x9d, 0x91, 0x4e, 0xeb, 0x61, 0xf1, 0x70,
        0x2e, 0x69, 0x6c, 0x20, 0x3a, 0x12, 0x68, 0x54
    };
#endif
#ifdef WOLFSSL_SHA3
    #ifndef WOLFSSL_NOSHA3_224
    static const unsigned char testResultSha3_224[] =
    {
        0x3b, 0x16, 0x54, 0x6b, 0xbc, 0x7b, 0xe2, 0x70,
        0x6a, 0x03, 0x1d, 0xca, 0xfd, 0x56, 0x37, 0x3d,
        0x98, 0x84, 0x36, 0x76, 0x41, 0xd8, 0xc5, 0x9a,
        0xf3, 0xc8, 0x60, 0xf7
    };
    #endif
    #ifndef WOLFSSL_NOSHA3_256
    static const unsigned char testResultSha3_256[] =
    {
        0xba, 0x85, 0x19, 0x23, 0x10, 0xdf, 0xfa, 0x96,
        0xe2, 0xa3, 0xa4, 0x0e, 0x69, 0x77, 0x43, 0x51,
        0x14, 0x0b, 0xb7, 0x18, 0x5e, 0x12, 0x02, 0xcd,
        0xcc, 0x91, 0x75, 0x89, 0xf9, 0x5e, 0x16, 0xbb
    };
    #endif
    #ifndef WOLFSSL_NOSHA3_384
    static const unsigned char testResultSha3_384[] =
    {
        0x68, 0xd2, 0xdc, 0xf7, 0xfd, 0x4d, 0xdd, 0x0a,
        0x22, 0x40, 0xc8, 0xa4, 0x37, 0x30, 0x5f, 0x61,
        0xfb, 0x73, 0x34, 0xcf, 0xb5, 0xd0, 0x22, 0x6e,
        0x1b, 0xc2, 0x7d, 0xc1, 0x0a, 0x2e, 0x72, 0x3a,
        0x20, 0xd3, 0x70, 0xb4, 0x77, 0x43, 0x13, 0x0e,
        0x26, 0xac, 0x7e, 0x3d, 0x53, 0x28, 0x86, 0xbd
    };
    #endif
    #ifndef WOLFSSL_NOSHA3_512
    static const unsigned char testResultSha3_512[] =
    {
        0xeb, 0x3f, 0xbd, 0x4b, 0x2e, 0xaa, 0xb8, 0xf5,
        0xc5, 0x04, 0xbd, 0x3a, 0x41, 0x46, 0x5a, 0xac,
        0xec, 0x15, 0x77, 0x0a, 0x7c, 0xab, 0xac, 0x53,
        0x1e, 0x48, 0x2f, 0x86, 0x0b, 0x5e, 0xc7, 0xba,
        0x47, 0xcc, 0xb2, 0xc6, 0xf2, 0xaf, 0xce, 0x8f,
        0x88, 0xd2, 0x2b, 0x6d, 0xc6, 0x13, 0x80, 0xf2,
        0x3a, 0x66, 0x8f, 0xd3, 0x88, 0x8b, 0xb8, 0x05,
        0x37, 0xc0, 0xa0, 0xb8, 0x64, 0x07, 0x68, 0x9e
    };
    #endif
#endif

#ifndef NO_SHA256
    ExpectIntEQ(test_hmac_signing(wolfSSL_EVP_sha256(), testKey,
        sizeof(testKey), testData, XSTRLEN(testData), testResultSha256,
        sizeof(testResultSha256)), TEST_SUCCESS);
#endif
#ifdef WOLFSSL_SHA224
    ExpectIntEQ(test_hmac_signing(wolfSSL_EVP_sha224(), testKey,
        sizeof(testKey), testData, XSTRLEN(testData), testResultSha224,
        sizeof(testResultSha224)), TEST_SUCCESS);
#endif
#ifdef WOLFSSL_SHA384
    ExpectIntEQ(test_hmac_signing(wolfSSL_EVP_sha384(), testKey,
        sizeof(testKey), testData, XSTRLEN(testData), testResultSha384,
        sizeof(testResultSha384)), TEST_SUCCESS);
#endif
#ifdef WOLFSSL_SHA512
    ExpectIntEQ(test_hmac_signing(wolfSSL_EVP_sha512(), testKey,
        sizeof(testKey), testData, XSTRLEN(testData), testResultSha512,
        sizeof(testResultSha512)), TEST_SUCCESS);
#endif
#ifdef WOLFSSL_SHA3
    #ifndef WOLFSSL_NOSHA3_224
    ExpectIntEQ(test_hmac_signing(wolfSSL_EVP_sha3_224(), testKey,
        sizeof(testKey), testData, XSTRLEN(testData), testResultSha3_224,
        sizeof(testResultSha3_224)), TEST_SUCCESS);
    #endif
    #ifndef WOLFSSL_NOSHA3_256
    ExpectIntEQ(test_hmac_signing(wolfSSL_EVP_sha3_256(), testKey,
        sizeof(testKey), testData, XSTRLEN(testData), testResultSha3_256,
        sizeof(testResultSha3_256)), TEST_SUCCESS);
    #endif
    #ifndef WOLFSSL_NOSHA3_384
    ExpectIntEQ(test_hmac_signing(wolfSSL_EVP_sha3_384(), testKey,
        sizeof(testKey), testData, XSTRLEN(testData), testResultSha3_384,
        sizeof(testResultSha3_384)), TEST_SUCCESS);
    #endif
    #ifndef WOLFSSL_NOSHA3_512
    ExpectIntEQ(test_hmac_signing(wolfSSL_EVP_sha3_512(), testKey,
        sizeof(testKey), testData, XSTRLEN(testData), testResultSha3_512,
        sizeof(testResultSha3_512)), TEST_SUCCESS);
    #endif
#endif
#endif /* OPENSSL_EXTRA */
    return EXPECT_RESULT();
}


static int test_wolfSSL_EVP_MD_rsa_signing(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_RSA) && defined(USE_CERT_BUFFERS_2048)
    WOLFSSL_EVP_PKEY* privKey = NULL;
    WOLFSSL_EVP_PKEY* pubKey = NULL;
    WOLFSSL_EVP_PKEY_CTX* keyCtx = NULL;
    const char testData[] = "Hi There";
    WOLFSSL_EVP_MD_CTX mdCtx;
    WOLFSSL_EVP_MD_CTX mdCtxCopy;
    int ret;
    size_t checkSz = -1;
    int sz = 2048 / 8;
    const unsigned char* cp;
    const unsigned char* p;
    unsigned char check[2048/8];
    size_t i;
    int paddings[] = {
            RSA_PKCS1_PADDING,
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST) && defined(WC_RSA_PSS)
            RSA_PKCS1_PSS_PADDING,
#endif
    };


    cp = client_key_der_2048;
    ExpectNotNull((privKey = wolfSSL_d2i_PrivateKey(EVP_PKEY_RSA, NULL, &cp,
                                                  sizeof_client_key_der_2048)));
    p = client_keypub_der_2048;
    ExpectNotNull((pubKey = wolfSSL_d2i_PUBKEY(NULL, &p,
                                               sizeof_client_keypub_der_2048)));

    wolfSSL_EVP_MD_CTX_init(&mdCtx);
    wolfSSL_EVP_MD_CTX_init(&mdCtxCopy);
    ExpectIntEQ(wolfSSL_EVP_DigestSignInit(&mdCtx, NULL, wolfSSL_EVP_sha256(),
                                                             NULL, privKey), 1);
    ExpectIntEQ(wolfSSL_EVP_DigestSignUpdate(&mdCtx, testData,
                                          (unsigned int)XSTRLEN(testData)), 1);
    checkSz = sizeof(check);
    ExpectIntEQ(wolfSSL_EVP_DigestSignFinal(&mdCtx, NULL, &checkSz), 1);
    ExpectIntEQ((int)checkSz, sz);
    checkSz = sizeof(check);
    ExpectIntEQ(wolfSSL_EVP_DigestSignFinal(&mdCtx, check, &checkSz), 1);
    ExpectIntEQ((int)checkSz,sz);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_copy_ex(&mdCtxCopy, &mdCtx), 1);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_copy_ex(&mdCtxCopy, &mdCtx), 1);
    ret = wolfSSL_EVP_MD_CTX_cleanup(&mdCtxCopy);
    ExpectIntEQ(ret, 1);
    ret = wolfSSL_EVP_MD_CTX_cleanup(&mdCtx);
    ExpectIntEQ(ret, 1);

    wolfSSL_EVP_MD_CTX_init(&mdCtx);
    ExpectIntEQ(wolfSSL_EVP_DigestVerifyInit(&mdCtx, NULL, wolfSSL_EVP_sha256(),
                                                              NULL, pubKey), 1);
    ExpectIntEQ(wolfSSL_EVP_DigestVerifyUpdate(&mdCtx, testData,
                                               (unsigned int)XSTRLEN(testData)),
                1);
    ExpectIntEQ(wolfSSL_EVP_DigestVerifyFinal(&mdCtx, check, checkSz), 1);
    ret = wolfSSL_EVP_MD_CTX_cleanup(&mdCtx);
    ExpectIntEQ(ret, 1);

    wolfSSL_EVP_MD_CTX_init(&mdCtx);
    ExpectIntEQ(wolfSSL_EVP_DigestSignInit(&mdCtx, NULL, wolfSSL_EVP_sha256(),
                                                             NULL, privKey), 1);
    ExpectIntEQ(wolfSSL_EVP_DigestSignUpdate(&mdCtx, testData, 4), 1);
    checkSz = sizeof(check);
    ExpectIntEQ(wolfSSL_EVP_DigestSignFinal(&mdCtx, NULL, &checkSz), 1);
    ExpectIntEQ((int)checkSz, sz);
    checkSz = sizeof(check);
    ExpectIntEQ(wolfSSL_EVP_DigestSignFinal(&mdCtx, check, &checkSz), 1);
    ExpectIntEQ((int)checkSz, sz);
    ExpectIntEQ(wolfSSL_EVP_DigestSignUpdate(&mdCtx, testData + 4,
                                      (unsigned int)XSTRLEN(testData) - 4), 1);
    checkSz = sizeof(check);
    ExpectIntEQ(wolfSSL_EVP_DigestSignFinal(&mdCtx, check, &checkSz), 1);
    ExpectIntEQ((int)checkSz, sz);
    ret = wolfSSL_EVP_MD_CTX_cleanup(&mdCtx);
    ExpectIntEQ(ret, 1);

    wolfSSL_EVP_MD_CTX_init(&mdCtx);
    ExpectIntEQ(wolfSSL_EVP_DigestVerifyInit(&mdCtx, NULL, wolfSSL_EVP_sha256(),
                                                              NULL, pubKey), 1);
    ExpectIntEQ(wolfSSL_EVP_DigestVerifyUpdate(&mdCtx, testData, 4), 1);
    ExpectIntEQ(wolfSSL_EVP_DigestVerifyUpdate(&mdCtx, testData + 4,
                                           (unsigned int)XSTRLEN(testData) - 4),
                1);
    ExpectIntEQ(wolfSSL_EVP_DigestVerifyFinal(&mdCtx, check, checkSz), 1);
    ret = wolfSSL_EVP_MD_CTX_cleanup(&mdCtx);
    ExpectIntEQ(ret, 1);

    /* Check all signing padding types */
    for (i = 0; i < sizeof(paddings)/sizeof(int); i++) {
        wolfSSL_EVP_MD_CTX_init(&mdCtx);
        ExpectIntEQ(wolfSSL_EVP_DigestSignInit(&mdCtx, &keyCtx,
                wolfSSL_EVP_sha256(), NULL, privKey), 1);
        ExpectIntEQ(wolfSSL_EVP_PKEY_CTX_set_rsa_padding(keyCtx,
                paddings[i]), 1);
        ExpectIntEQ(wolfSSL_EVP_DigestSignUpdate(&mdCtx, testData,
                (unsigned int)XSTRLEN(testData)), 1);
        checkSz = sizeof(check);
        ExpectIntEQ(wolfSSL_EVP_DigestSignFinal(&mdCtx, NULL, &checkSz), 1);
        ExpectIntEQ((int)checkSz, sz);
        checkSz = sizeof(check);
        ExpectIntEQ(wolfSSL_EVP_DigestSignFinal(&mdCtx, check, &checkSz), 1);
        ExpectIntEQ((int)checkSz,sz);
        ret = wolfSSL_EVP_MD_CTX_cleanup(&mdCtx);
        ExpectIntEQ(ret, 1);

        wolfSSL_EVP_MD_CTX_init(&mdCtx);
        ExpectIntEQ(wolfSSL_EVP_DigestVerifyInit(&mdCtx, &keyCtx,
                wolfSSL_EVP_sha256(), NULL, pubKey), 1);
        ExpectIntEQ(wolfSSL_EVP_PKEY_CTX_set_rsa_padding(keyCtx,
                paddings[i]), 1);
        ExpectIntEQ(wolfSSL_EVP_DigestVerifyUpdate(&mdCtx, testData,
                (unsigned int)XSTRLEN(testData)), 1);
        ExpectIntEQ(wolfSSL_EVP_DigestVerifyFinal(&mdCtx, check, checkSz), 1);
        ret = wolfSSL_EVP_MD_CTX_cleanup(&mdCtx);
        ExpectIntEQ(ret, 1);
    }

    wolfSSL_EVP_PKEY_free(pubKey);
    wolfSSL_EVP_PKEY_free(privKey);
#endif
    return EXPECT_RESULT();
}


static int test_wolfSSL_EVP_MD_ecc_signing(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && defined(HAVE_ECC) && defined(USE_CERT_BUFFERS_256)
    WOLFSSL_EVP_PKEY* privKey = NULL;
    WOLFSSL_EVP_PKEY* pubKey = NULL;
    const char testData[] = "Hi There";
    WOLFSSL_EVP_MD_CTX mdCtx;
    int ret;
    const unsigned char* cp;
    const unsigned char* p;
    unsigned char check[2048/8];
    size_t checkSz = sizeof(check);

    XMEMSET(check, 0, sizeof(check));

    cp = ecc_clikey_der_256;
    ExpectNotNull(privKey = wolfSSL_d2i_PrivateKey(EVP_PKEY_EC, NULL, &cp,
                                                   sizeof_ecc_clikey_der_256));
    p = ecc_clikeypub_der_256;
    ExpectNotNull((pubKey = wolfSSL_d2i_PUBKEY(NULL, &p,
                                                sizeof_ecc_clikeypub_der_256)));

    wolfSSL_EVP_MD_CTX_init(&mdCtx);
    ExpectIntEQ(wolfSSL_EVP_DigestSignInit(&mdCtx, NULL, wolfSSL_EVP_sha256(),
                                                             NULL, privKey), 1);
    ExpectIntEQ(wolfSSL_EVP_DigestSignUpdate(&mdCtx, testData,
                                          (unsigned int)XSTRLEN(testData)), 1);
    checkSz = sizeof(check);
    ExpectIntEQ(wolfSSL_EVP_DigestSignFinal(&mdCtx, NULL, &checkSz), 1);
    checkSz = sizeof(check);
    ExpectIntEQ(wolfSSL_EVP_DigestSignFinal(&mdCtx, check, &checkSz), 1);
    ret = wolfSSL_EVP_MD_CTX_cleanup(&mdCtx);
    ExpectIntEQ(ret, 1);

    wolfSSL_EVP_MD_CTX_init(&mdCtx);
    ExpectIntEQ(wolfSSL_EVP_DigestVerifyInit(&mdCtx, NULL, wolfSSL_EVP_sha256(),
                                                              NULL, pubKey), 1);
    ExpectIntEQ(wolfSSL_EVP_DigestVerifyUpdate(&mdCtx, testData,
                                               (unsigned int)XSTRLEN(testData)),
                1);
    ExpectIntEQ(wolfSSL_EVP_DigestVerifyFinal(&mdCtx, check, checkSz), 1);
    ret = wolfSSL_EVP_MD_CTX_cleanup(&mdCtx);
    ExpectIntEQ(ret, 1);

    wolfSSL_EVP_MD_CTX_init(&mdCtx);
    ExpectIntEQ(wolfSSL_EVP_DigestSignInit(&mdCtx, NULL, wolfSSL_EVP_sha256(),
                                                             NULL, privKey), 1);
    ExpectIntEQ(wolfSSL_EVP_DigestSignUpdate(&mdCtx, testData, 4), 1);
    checkSz = sizeof(check);
    ExpectIntEQ(wolfSSL_EVP_DigestSignFinal(&mdCtx, NULL, &checkSz), 1);
    checkSz = sizeof(check);
    ExpectIntEQ(wolfSSL_EVP_DigestSignFinal(&mdCtx, check, &checkSz), 1);
    ExpectIntEQ(wolfSSL_EVP_DigestSignUpdate(&mdCtx, testData + 4,
                                      (unsigned int)XSTRLEN(testData) - 4), 1);
    checkSz = sizeof(check);
    ExpectIntEQ(wolfSSL_EVP_DigestSignFinal(&mdCtx, check, &checkSz), 1);
    ret = wolfSSL_EVP_MD_CTX_cleanup(&mdCtx);
    ExpectIntEQ(ret, 1);

    wolfSSL_EVP_MD_CTX_init(&mdCtx);
    ExpectIntEQ(wolfSSL_EVP_DigestVerifyInit(&mdCtx, NULL, wolfSSL_EVP_sha256(),
                                                              NULL, pubKey), 1);
    ExpectIntEQ(wolfSSL_EVP_DigestVerifyUpdate(&mdCtx, testData, 4), 1);
    ExpectIntEQ(wolfSSL_EVP_DigestVerifyUpdate(&mdCtx, testData + 4,
                                           (unsigned int)XSTRLEN(testData) - 4),
                1);
    ExpectIntEQ(wolfSSL_EVP_DigestVerifyFinal(&mdCtx, check, checkSz), 1);
    ret = wolfSSL_EVP_MD_CTX_cleanup(&mdCtx);
    ExpectIntEQ(ret, 1);

    wolfSSL_EVP_PKEY_free(pubKey);
    wolfSSL_EVP_PKEY_free(privKey);
#endif
    return EXPECT_RESULT();
}


static int test_wolfSSL_CTX_add_extra_chain_cert(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_TLS) && \
   !defined(NO_FILESYSTEM) && !defined(NO_RSA) && !defined(NO_BIO)
#if !defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)
    char caFile[] = "./certs/client-ca.pem";
    char clientFile[] = "./certs/client-cert.pem";
    SSL_CTX* ctx = NULL;
    X509* x509 = NULL;
    BIO *bio = NULL;
    X509 *cert = NULL;
    X509 *ca = NULL;
    STACK_OF(X509) *chain = NULL;
    STACK_OF(X509) *chain2 = NULL;

#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_server_method()));
#else
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_client_method()));
#endif

    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(caFile,
        WOLFSSL_FILETYPE_PEM));

    /* Negative tests. */
    ExpectIntEQ((int)SSL_CTX_add_extra_chain_cert(NULL, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ((int)SSL_CTX_add_extra_chain_cert(ctx, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ((int)SSL_CTX_add_extra_chain_cert(NULL, x509), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

    ExpectIntEQ((int)SSL_CTX_add_extra_chain_cert(ctx, x509), WOLFSSL_SUCCESS);

    ExpectNotNull(x509 = wolfSSL_X509_new());
    /* Empty certificate. */
    ExpectIntEQ((int)SSL_CTX_add_extra_chain_cert(ctx, x509), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    wolfSSL_X509_free(x509);
    x509 = NULL;

    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(clientFile,
        WOLFSSL_FILETYPE_PEM));

    /* additional test of getting EVP_PKEY key size from X509
     * Do not run with user RSA because wolfSSL_RSA_size is not currently
     * allowed with user RSA */
    {
        EVP_PKEY* pkey = NULL;
    #if defined(HAVE_ECC)
        X509* ecX509 = NULL;
    #endif /* HAVE_ECC */

        ExpectNotNull(pkey = X509_get_pubkey(x509));
        /* current RSA key is 2048 bit (256 bytes) */
        ExpectIntEQ(EVP_PKEY_size(pkey), 256);

        EVP_PKEY_free(pkey);
        pkey = NULL;

#if defined(HAVE_ECC)
    #if defined(USE_CERT_BUFFERS_256)
        ExpectNotNull(ecX509 = wolfSSL_X509_load_certificate_buffer(
                    cliecc_cert_der_256, sizeof_cliecc_cert_der_256,
                    SSL_FILETYPE_ASN1));
    #else
        ExpectNotNull(ecX509 = wolfSSL_X509_load_certificate_file(
            cliEccCertFile, SSL_FILETYPE_PEM));
    #endif
        pkey = X509_get_pubkey(ecX509);
        ExpectNotNull(pkey);
        /* current ECC key is 256 bit (32 bytes) */
        ExpectIntGE(EVP_PKEY_size(pkey), 72);

        X509_free(ecX509);
        ecX509 = NULL;
        EVP_PKEY_free(pkey);
        pkey = NULL;
#endif /* HAVE_ECC */
    }

    ExpectIntEQ((int)SSL_CTX_add_extra_chain_cert(ctx, x509), SSL_SUCCESS);
    if (EXPECT_SUCCESS()) {
        x509 = NULL;
    }

#ifdef WOLFSSL_ENCRYPTED_KEYS
    ExpectNull(SSL_CTX_get_default_passwd_cb(ctx));
    ExpectNull(SSL_CTX_get_default_passwd_cb_userdata(ctx));
#endif
    SSL_CTX_free(ctx);
    ctx = NULL;

#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_server_method()));
#else
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_client_method()));
#endif
    /* Test haproxy use case */
    ExpectNotNull(bio = BIO_new_file(svrCertFile, "r"));
    /* Read Certificate */
    ExpectNotNull(cert = PEM_read_bio_X509_AUX(bio, NULL, NULL, NULL));
    ExpectNotNull(ca = PEM_read_bio_X509(bio, NULL, NULL, NULL));
    ExpectNotNull(chain = sk_X509_new_null());
    ExpectIntEQ(sk_X509_push(chain, ca), 1);
    if (EXPECT_SUCCESS()) {
        ca = NULL;
    }
    ExpectNotNull(chain2 = X509_chain_up_ref(chain));
    ExpectNotNull(ca = sk_X509_shift(chain2));
    ExpectIntEQ(SSL_CTX_use_certificate(ctx, cert), 1);
    ExpectIntEQ(SSL_CTX_add_extra_chain_cert(ctx, ca), 1);
    if (EXPECT_SUCCESS()) {
        ca = NULL;
    }

    BIO_free(bio);
    X509_free(cert);
    X509_free(ca);
    X509_free(x509);
    sk_X509_pop_free(chain, X509_free);
    sk_X509_pop_free(chain2, X509_free);
    SSL_CTX_free(ctx);
#endif /* !NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER */
#endif /* defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && \
         !defined(NO_FILESYSTEM) && !defined(NO_RSA) && !defined (NO_BIO) */
    return EXPECT_RESULT();
}

#if !defined(NO_WOLFSSL_CLIENT) && !defined(NO_WOLFSSL_SERVER)
static int test_wolfSSL_ERR_peek_last_error_line(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && \
    !defined(NO_FILESYSTEM) && defined(DEBUG_WOLFSSL) && \
    !defined(NO_OLD_TLS) && !defined(WOLFSSL_NO_TLS12) && \
    defined(HAVE_IO_TESTS_DEPENDENCIES) && !defined(NO_ERROR_QUEUE)
    callback_functions client_cb;
    callback_functions server_cb;
    int         line = 0;
    int         flag = ERR_TXT_STRING;
    const char* file = NULL;
    const char* data = NULL;

    /* create a failed connection and inspect the error */
    XMEMSET(&client_cb, 0, sizeof(callback_functions));
    XMEMSET(&server_cb, 0, sizeof(callback_functions));
    client_cb.method  = wolfTLSv1_1_client_method;
    server_cb.method  = wolfTLSv1_2_server_method;

    test_wolfSSL_client_server_nofail(&client_cb, &server_cb);

    ExpectIntGT(ERR_get_error_line_data(NULL, NULL, &data, &flag), 0);
    ExpectNotNull(data);

    /* check clearing error state */
    ERR_remove_state(0);
    ExpectIntEQ((int)ERR_peek_last_error_line(NULL, NULL), 0);
    ERR_peek_last_error_line(NULL, &line);
    ExpectIntEQ(line, 0);
    ERR_peek_last_error_line(&file, NULL);
    ExpectNull(file);

    /* retry connection to fill error queue */
    XMEMSET(&client_cb, 0, sizeof(callback_functions));
    XMEMSET(&server_cb, 0, sizeof(callback_functions));
    client_cb.method  = wolfTLSv1_1_client_method;
    server_cb.method  = wolfTLSv1_2_server_method;

    test_wolfSSL_client_server_nofail(&client_cb, &server_cb);

    /* check that error code was stored */
    ExpectIntNE((int)ERR_peek_last_error_line(NULL, NULL), 0);
    ERR_peek_last_error_line(NULL, &line);
    ExpectIntNE(line, 0);
    ERR_peek_last_error_line(&file, NULL);
    ExpectNotNull(file);

    fprintf(stderr, "\nTesting error print out\n");
    ERR_print_errors_fp(stderr);
    fprintf(stderr, "Done testing print out\n\n");
#endif /* defined(OPENSSL_EXTRA) && !defined(NO_CERTS) &&
        * !defined(NO_FILESYSTEM) && !defined(DEBUG_WOLFSSL) */
    return EXPECT_RESULT();
}
#endif /* !NO_WOLFSSL_CLIENT && !NO_WOLFSSL_SERVER */

#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && \
       !defined(NO_FILESYSTEM) && !defined(NO_RSA)
static int verify_cb(int ok, X509_STORE_CTX *ctx)
{
    (void) ok;
    (void) ctx;
    fprintf(stderr, "ENTER verify_cb\n");
    return SSL_SUCCESS;
}
#endif

static int test_wolfSSL_X509_Name_canon(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_ALL) && !defined(NO_CERTS) && \
    !defined(NO_FILESYSTEM) && !defined(NO_SHA) && \
     defined(WOLFSSL_CERT_GEN) && \
    (defined(WOLFSSL_CERT_REQ) || defined(WOLFSSL_CERT_EXT)) && !defined(NO_RSA)
    const long ex_hash1 = 0x0fdb2da4;
    const long ex_hash2 = 0x9f3e8c9e;
    X509_NAME *name = NULL;
    X509 *x509 = NULL;
    XFILE file = XBADFILE;
    unsigned long hash = 0;
    byte digest[WC_MAX_DIGEST_SIZE] = {0};
    byte  *pbuf = NULL;
    word32 len = 0;
    (void) ex_hash2;

    ExpectTrue((file = XFOPEN(caCertFile, "rb")) != XBADFILE);
    ExpectNotNull(x509 = PEM_read_X509(file, NULL, NULL, NULL));
    ExpectNotNull(name = X509_get_issuer_name(x509));

    /* When output buffer is NULL, should return necessary output buffer
     * length.*/
    ExpectIntEQ(wolfSSL_i2d_X509_NAME_canon(NULL, NULL), BAD_FUNC_ARG);
    ExpectIntGT(wolfSSL_i2d_X509_NAME_canon(name, NULL), 0);
    ExpectIntGT((len = (word32)wolfSSL_i2d_X509_NAME_canon(name, &pbuf)), 0);
    ExpectIntEQ(wc_ShaHash((const byte*)pbuf, (word32)len, digest), 0);

    hash = (((unsigned long)digest[3] << 24) |
            ((unsigned long)digest[2] << 16) |
            ((unsigned long)digest[1] <<  8) |
            ((unsigned long)digest[0]));
    ExpectIntEQ(hash, ex_hash1);

    if (file != XBADFILE) {
        XFCLOSE(file);
        file = XBADFILE;
    }
    X509_free(x509);
    x509 = NULL;
    XFREE(pbuf, NULL, DYNAMIC_TYPE_OPENSSL);
    pbuf = NULL;

    ExpectTrue((file = XFOPEN(cliCertFile, "rb")) != XBADFILE);
    ExpectNotNull(x509 = PEM_read_X509(file, NULL, NULL, NULL));
    ExpectNotNull(name = X509_get_issuer_name(x509));

    ExpectIntGT((len = (word32)wolfSSL_i2d_X509_NAME_canon(name, &pbuf)), 0);
    ExpectIntEQ(wc_ShaHash((const byte*)pbuf, (word32)len, digest), 0);

    hash = (((unsigned long)digest[3] << 24) |
            ((unsigned long)digest[2] << 16) |
            ((unsigned long)digest[1] <<  8) |
            ((unsigned long)digest[0]));

    ExpectIntEQ(hash, ex_hash2);

    if (file != XBADFILE)
        XFCLOSE(file);
    X509_free(x509);
    XFREE(pbuf, NULL, DYNAMIC_TYPE_OPENSSL);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_LOOKUP_ctrl_hash_dir(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_ALL) && !defined(NO_FILESYSTEM) && !defined(NO_WOLFSSL_DIR)
    const int  MAX_DIR = 4;
    const char paths[][32] = {
                             "./certs/ed25519",
                             "./certs/ecc",
                             "./certs/crl",
                             "./certs/",
                            };

    char CertCrl_path[MAX_FILENAME_SZ];
    char *p;
    X509_STORE* str = NULL;
    X509_LOOKUP* lookup = NULL;
    WOLFSSL_STACK* sk = NULL;
    int len, total_len, i;

    (void)sk;

    XMEMSET(CertCrl_path, 0, MAX_FILENAME_SZ);

    /* illegal string */
    ExpectNotNull((str = wolfSSL_X509_STORE_new()));
    ExpectNotNull(lookup = X509_STORE_add_lookup(str, X509_LOOKUP_file()));
    ExpectIntEQ(X509_LOOKUP_ctrl(lookup, X509_L_ADD_DIR, "",
                                    SSL_FILETYPE_PEM, NULL), 0);
    ExpectIntEQ(X509_LOOKUP_ctrl(lookup, X509_L_ADD_STORE, "",
        SSL_FILETYPE_PEM, NULL), WOLFSSL_NOT_IMPLEMENTED);
    ExpectIntEQ(X509_LOOKUP_ctrl(lookup, X509_L_LOAD_STORE, "",
        SSL_FILETYPE_PEM, NULL), WOLFSSL_NOT_IMPLEMENTED);
    ExpectIntEQ(X509_LOOKUP_ctrl(lookup, 0, "",
        SSL_FILETYPE_PEM, NULL), WOLFSSL_FAILURE);

    /* free store */
    X509_STORE_free(str);
    str = NULL;

    /* short folder string */
    ExpectNotNull((str = wolfSSL_X509_STORE_new()));
    ExpectNotNull(lookup = X509_STORE_add_lookup(str, X509_LOOKUP_file()));
    ExpectIntEQ(X509_LOOKUP_ctrl(lookup, X509_L_ADD_DIR, "./",
                                    SSL_FILETYPE_PEM,NULL), 1);
    #if defined(WOLFSSL_INT_H)
    /* only available when including internal.h */
    ExpectNotNull(sk = lookup->dirs->dir_entry);
    #endif
    /* free store */
    X509_STORE_free(str);
    str = NULL;

    /* typical function check */
    p = &CertCrl_path[0];
    total_len = 0;

    for (i = MAX_DIR - 1; i>=0 && total_len < MAX_FILENAME_SZ; i--) {
        len = (int)XSTRLEN((const char*)&paths[i]);
        total_len += len;
        XSTRNCPY(p, paths[i], MAX_FILENAME_SZ - total_len);
        p += len;
        if (i != 0) *(p++) = SEPARATOR_CHAR;
    }

    ExpectNotNull((str = wolfSSL_X509_STORE_new()));
    ExpectNotNull(lookup = X509_STORE_add_lookup(str, X509_LOOKUP_file()));
    ExpectIntEQ(X509_LOOKUP_ctrl(lookup, X509_L_ADD_DIR, CertCrl_path,
                                    SSL_FILETYPE_PEM,NULL), 1);
    #if defined(WOLFSSL_INT_H)
    /* only available when including internal.h */
    ExpectNotNull(sk = lookup->dirs->dir_entry);
    #endif

    X509_STORE_free(str);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_LOOKUP_ctrl_file(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_ALL) && !defined(NO_CERTS) && \
    !defined(NO_FILESYSTEM) && !defined(NO_RSA) && \
    defined(WOLFSSL_SIGNER_DER_CERT)
    X509_STORE_CTX* ctx = NULL;
    X509_STORE* str = NULL;
    X509_LOOKUP* lookup = NULL;

    X509* cert1 = NULL;
    X509* x509Ca = NULL;
    X509* x509Svr = NULL;
    X509* issuer = NULL;

    WOLFSSL_STACK* sk = NULL;
    X509_NAME* caName = NULL;
    X509_NAME* issuerName = NULL;

    XFILE file1 = XBADFILE;
    int i;
    int cert_count = 0;
    int cmp;

    char der[] = "certs/ca-cert.der";

#ifdef HAVE_CRL
    char pem[][100] = {
        "./certs/crl/crl.pem",
        "./certs/crl/crl2.pem",
        "./certs/crl/caEccCrl.pem",
        "./certs/crl/eccCliCRL.pem",
        "./certs/crl/eccSrvCRL.pem",
        ""
    };
#endif
    ExpectTrue((file1 = XFOPEN("./certs/ca-cert.pem", "rb")) != XBADFILE);
    ExpectNotNull(cert1 = wolfSSL_PEM_read_X509(file1, NULL, NULL, NULL));
    if (file1 != XBADFILE)
        XFCLOSE(file1);

    ExpectNotNull(ctx = X509_STORE_CTX_new());
    ExpectNotNull((str = wolfSSL_X509_STORE_new()));
    ExpectNotNull(lookup = X509_STORE_add_lookup(str, X509_LOOKUP_file()));
    ExpectIntEQ(wolfSSL_X509_load_cert_crl_file(NULL, NULL,
        WOLFSSL_FILETYPE_PEM), 0);
    ExpectIntEQ(wolfSSL_X509_load_cert_crl_file(lookup, NULL,
        WOLFSSL_FILETYPE_PEM), 0);
    ExpectIntEQ(wolfSSL_X509_load_cert_crl_file(NULL, caCertFile,
        WOLFSSL_FILETYPE_PEM), 0);
    ExpectIntEQ(wolfSSL_X509_load_cert_crl_file(NULL, der       ,
        WOLFSSL_FILETYPE_PEM), 0);
    ExpectIntEQ(X509_LOOKUP_ctrl(lookup, X509_L_FILE_LOAD, caCertFile,
                                    SSL_FILETYPE_PEM,NULL), 1);
    ExpectNotNull(sk = wolfSSL_CertManagerGetCerts(str->cm));
    ExpectIntEQ((cert_count = sk_X509_num(sk)), 1);

    /* check if CA cert is loaded into the store */
    for (i = 0; i < cert_count; i++) {
        x509Ca = sk_X509_value(sk, i);
        ExpectIntEQ(0, wolfSSL_X509_cmp(x509Ca, cert1));
    }

    ExpectNotNull((x509Svr =
            wolfSSL_X509_load_certificate_file(svrCertFile, SSL_FILETYPE_PEM)));

    ExpectIntEQ(X509_STORE_CTX_init(ctx, str, x509Svr, NULL), SSL_SUCCESS);

    ExpectNull(X509_STORE_CTX_get0_current_issuer(NULL));
    issuer = X509_STORE_CTX_get0_current_issuer(ctx);
    ExpectNull(issuer);

    ExpectIntEQ(X509_verify_cert(ctx), 1);

    issuer = X509_STORE_CTX_get0_current_issuer(ctx);
    ExpectNotNull(issuer);
    caName = X509_get_subject_name(x509Ca);
    ExpectNotNull(caName);
    issuerName = X509_get_subject_name(issuer);
    ExpectNotNull(issuerName);
    cmp = X509_NAME_cmp(caName, issuerName);
    ExpectIntEQ(cmp, 0);

    /* load der format */
    issuer = NULL;
    X509_STORE_CTX_free(ctx);
    ctx = NULL;
    X509_STORE_free(str);
    str = NULL;
    sk_X509_pop_free(sk, NULL);
    sk = NULL;
    X509_free(x509Svr);
    x509Svr = NULL;

    ExpectNotNull((str = wolfSSL_X509_STORE_new()));
    ExpectNotNull(lookup = X509_STORE_add_lookup(str, X509_LOOKUP_file()));
    ExpectIntEQ(X509_LOOKUP_ctrl(lookup, X509_L_FILE_LOAD, der,
                                    SSL_FILETYPE_ASN1,NULL), 1);
    ExpectNotNull(sk = wolfSSL_CertManagerGetCerts(str->cm));
    ExpectIntEQ((cert_count = sk_X509_num(sk)), 1);
    /* check if CA cert is loaded into the store */
    for (i = 0; i < cert_count; i++) {
        x509Ca = sk_X509_value(sk, i);
        ExpectIntEQ(0, wolfSSL_X509_cmp(x509Ca, cert1));
    }

    X509_STORE_free(str);
    str = NULL;
    sk_X509_pop_free(sk, NULL);
    sk = NULL;
    X509_free(cert1);
    cert1 = NULL;

#ifdef HAVE_CRL
    ExpectNotNull(str = wolfSSL_X509_STORE_new());
    ExpectNotNull(lookup = X509_STORE_add_lookup(str, X509_LOOKUP_file()));
    ExpectIntEQ(X509_LOOKUP_ctrl(lookup, X509_L_FILE_LOAD, caCertFile,
                                                    SSL_FILETYPE_PEM,NULL), 1);
    ExpectIntEQ(X509_LOOKUP_ctrl(lookup, X509_L_FILE_LOAD,
                                "certs/server-revoked-cert.pem",
                                 SSL_FILETYPE_PEM,NULL), 1);
    if (str) {
        ExpectIntEQ(wolfSSL_CertManagerVerify(str->cm, svrCertFile,
                    WOLFSSL_FILETYPE_PEM), 1);
        /* since store hasn't yet known the revoked cert*/
        ExpectIntEQ(wolfSSL_CertManagerVerify(str->cm,
                    "certs/server-revoked-cert.pem",
                    WOLFSSL_FILETYPE_PEM), 1);
    }
    for (i = 0; pem[i][0] != '\0'; i++)
    {
        ExpectIntEQ(X509_LOOKUP_ctrl(lookup, X509_L_FILE_LOAD, pem[i],
                                        SSL_FILETYPE_PEM, NULL), 1);
    }

    if (str) {
        /* since store knows crl list */
        ExpectIntEQ(wolfSSL_CertManagerVerify(str->cm,
                    "certs/server-revoked-cert.pem",
                    WOLFSSL_FILETYPE_PEM ), WC_NO_ERR_TRACE(CRL_CERT_REVOKED));
    }

    ExpectIntEQ(X509_LOOKUP_ctrl(NULL, 0, NULL, 0, NULL), 0);
    X509_STORE_free(str);
#endif
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_STORE_CTX_trusted_stack_cleanup(void)
{
    int res = TEST_SKIPPED;
#if defined(OPENSSL_EXTRA)
    X509_STORE_CTX_cleanup(NULL);
    X509_STORE_CTX_trusted_stack(NULL, NULL);

    res = TEST_SUCCESS;
#endif
    return res;
}

static int test_wolfSSL_X509_STORE_CTX_get_issuer(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_RSA) && !defined(NO_FILESYSTEM)
    X509_STORE_CTX* ctx = NULL;
    X509_STORE* str = NULL;
    X509* x509Ca = NULL;
    X509* x509Svr = NULL;
    X509* issuer = NULL;
    X509_NAME* caName = NULL;
    X509_NAME* issuerName = NULL;

    ExpectNotNull(ctx = X509_STORE_CTX_new());
    ExpectNotNull((str = wolfSSL_X509_STORE_new()));
    ExpectNotNull((x509Ca =
            wolfSSL_X509_load_certificate_file(caCertFile, SSL_FILETYPE_PEM)));
    ExpectIntEQ(X509_STORE_add_cert(str, x509Ca), SSL_SUCCESS);
    ExpectNotNull((x509Svr =
            wolfSSL_X509_load_certificate_file(svrCertFile, SSL_FILETYPE_PEM)));

    ExpectIntEQ(X509_STORE_CTX_init(ctx, str, x509Svr, NULL), SSL_SUCCESS);

    /* Issuer0 is not set until chain is built for verification */
    ExpectNull(X509_STORE_CTX_get0_current_issuer(NULL));
    ExpectNull(issuer = X509_STORE_CTX_get0_current_issuer(ctx));

    /* Issuer1 will use the store to make a new issuer */
    ExpectIntEQ(X509_STORE_CTX_get1_issuer(&issuer, ctx, x509Svr), 1);
    ExpectNotNull(issuer);
    X509_free(issuer);

    ExpectIntEQ(X509_verify_cert(ctx), 1);
    ExpectNotNull(issuer = X509_STORE_CTX_get0_current_issuer(ctx));
    ExpectNotNull(caName = X509_get_subject_name(x509Ca));
    ExpectNotNull(issuerName = X509_get_subject_name(issuer));
#ifdef WOLFSSL_SIGNER_DER_CERT
    ExpectIntEQ(X509_NAME_cmp(caName, issuerName), 0);
#endif

    X509_STORE_CTX_free(ctx);
    X509_free(x509Svr);
    X509_STORE_free(str);
    X509_free(x509Ca);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_PKCS7_certs(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_ALL) && !defined(NO_CERTS) && !defined(NO_BIO) && \
   !defined(NO_FILESYSTEM) && !defined(NO_RSA) && defined(HAVE_PKCS7)
    STACK_OF(X509)* sk = NULL;
    STACK_OF(X509_INFO)* info_sk = NULL;
    PKCS7 *p7 = NULL;
    BIO* bio = NULL;
    const byte* p = NULL;
    int buflen = 0;
    int i;

    /* Test twice. Once with d2i and once without to test
     * that everything is free'd correctly. */
    for (i = 0; i < 2; i++) {
        ExpectNotNull(p7 = PKCS7_new());
        if (p7 != NULL) {
            p7->version = 1;
        #ifdef NO_SHA
            p7->hashOID = SHA256h;
        #else
            p7->hashOID = SHAh;
        #endif
        }
        ExpectNotNull(bio = BIO_new(BIO_s_file()));
        ExpectIntGT(BIO_read_filename(bio, svrCertFile), 0);
        ExpectNotNull(info_sk = PEM_X509_INFO_read_bio(bio, NULL, NULL, NULL));
        ExpectIntEQ(sk_X509_INFO_num(info_sk), 2);
        ExpectNotNull(sk = sk_X509_new_null());
        while (EXPECT_SUCCESS() && (sk_X509_INFO_num(info_sk) > 0)) {
            X509_INFO* info = NULL;
            ExpectNotNull(info = sk_X509_INFO_shift(info_sk));
            if (EXPECT_SUCCESS() && info != NULL) {
                ExpectIntGT(sk_X509_push(sk, info->x509), 0);
                info->x509 = NULL;
            }
            X509_INFO_free(info);
        }
        sk_X509_INFO_pop_free(info_sk, X509_INFO_free);
        info_sk = NULL;
        BIO_free(bio);
        bio = NULL;
        ExpectNotNull(bio = BIO_new(BIO_s_mem()));
        ExpectIntEQ(wolfSSL_PKCS7_encode_certs(p7, sk, bio), 1);
        if ((sk != NULL) && ((p7 == NULL) || (bio == NULL))) {
            sk_X509_pop_free(sk, X509_free);
        }
        sk = NULL;
        ExpectIntGT((buflen = BIO_get_mem_data(bio, &p)), 0);

        if (i == 0) {
            PKCS7_free(p7);
            p7 = NULL;
            ExpectNotNull(d2i_PKCS7(&p7, &p, buflen));
            if (p7 != NULL) {
                /* Reset certs to force wolfSSL_PKCS7_to_stack to regenerate
                 * them */
                ((WOLFSSL_PKCS7*)p7)->certs = NULL;
            }
            /* PKCS7_free free's the certs */
            ExpectNotNull(wolfSSL_PKCS7_to_stack(p7));
        }

        BIO_free(bio);
        bio = NULL;
        PKCS7_free(p7);
        p7 = NULL;
    }
#endif /* defined(OPENSSL_ALL) && !defined(NO_CERTS) && \
         !defined(NO_FILESYSTEM) && !defined(NO_RSA) && defined(HAVE_PKCS7) */
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_STORE_CTX(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && \
   !defined(NO_FILESYSTEM) && !defined(NO_RSA)
    X509_STORE_CTX* ctx = NULL;
    X509_STORE* str = NULL;
    X509* x509 = NULL;
#ifdef OPENSSL_ALL
    X509* x5092 = NULL;
    STACK_OF(X509) *sk = NULL;
    STACK_OF(X509) *sk2 = NULL;
    STACK_OF(X509) *sk3 = NULL;
#endif

    ExpectNotNull(ctx = X509_STORE_CTX_new());
    ExpectNotNull((str = wolfSSL_X509_STORE_new()));
    ExpectNotNull((x509 =
                wolfSSL_X509_load_certificate_file(svrCertFile, SSL_FILETYPE_PEM)));
    ExpectIntEQ(X509_STORE_add_cert(str, x509), SSL_SUCCESS);
#ifdef OPENSSL_ALL
    /* sk_X509_new only in OPENSSL_ALL */
    sk = sk_X509_new_null();
    ExpectNotNull(sk);
    ExpectIntEQ(X509_STORE_CTX_init(ctx, str, x509, sk), SSL_SUCCESS);
#else
    ExpectIntEQ(X509_STORE_CTX_init(ctx, str, x509, NULL), SSL_SUCCESS);
#endif
    ExpectIntEQ(SSL_get_ex_data_X509_STORE_CTX_idx(), 0);
    X509_STORE_CTX_set_error(ctx, -5);
    X509_STORE_CTX_set_error(NULL, -5);

    X509_STORE_CTX_free(ctx);
    ctx = NULL;
#ifdef OPENSSL_ALL
    sk_X509_pop_free(sk, NULL);
    sk = NULL;
#endif
    X509_STORE_free(str);
    str = NULL;
    X509_free(x509);
    x509 = NULL;

    ExpectNotNull(ctx = X509_STORE_CTX_new());
    X509_STORE_CTX_set_verify_cb(ctx, verify_cb);
    X509_STORE_CTX_free(ctx);
    ctx = NULL;

#ifdef OPENSSL_ALL
    /* test X509_STORE_CTX_get(1)_chain */
    ExpectNotNull((x509 = X509_load_certificate_file(svrCertFile,
                                                     SSL_FILETYPE_PEM)));
    ExpectNotNull((x5092 = X509_load_certificate_file(cliCertFile,
                                                     SSL_FILETYPE_PEM)));
    ExpectNotNull((sk = sk_X509_new_null()));
    ExpectIntEQ(sk_X509_push(sk, x509), 1);
    if (EXPECT_FAIL()) {
        X509_free(x509);
        x509 = NULL;
    }
    ExpectNotNull((str = X509_STORE_new()));
    ExpectNotNull((ctx = X509_STORE_CTX_new()));
    ExpectIntEQ(X509_STORE_CTX_init(ctx, str, x5092, sk), 1);
    ExpectNull((sk2 = X509_STORE_CTX_get_chain(NULL)));
    ExpectNull((sk2 = X509_STORE_CTX_get_chain(ctx)));
    ExpectNull((sk3 = X509_STORE_CTX_get1_chain(NULL)));
    ExpectNull((sk3 = X509_STORE_CTX_get1_chain(ctx)));
    X509_STORE_CTX_free(ctx);
    ctx = NULL;
    X509_STORE_free(str);
    str = NULL;
    /* CTX certs not freed yet */
    X509_free(x5092);
    x5092 = NULL;
    sk_X509_pop_free(sk, NULL);
    sk = NULL;
    /* sk3 is dup so free here */
    sk_X509_pop_free(sk3, NULL);
    sk3 = NULL;
#endif

    /* test X509_STORE_CTX_get/set_ex_data */
    {
        int i = 0, tmpData = 5;
        void* tmpDataRet;
        ExpectNotNull(ctx = X509_STORE_CTX_new());
    #ifdef HAVE_EX_DATA
        for (i = 0; i < MAX_EX_DATA; i++) {
            ExpectIntEQ(X509_STORE_CTX_set_ex_data(ctx, i, &tmpData),
                        WOLFSSL_SUCCESS);
            tmpDataRet = (int*)X509_STORE_CTX_get_ex_data(ctx, i);
            ExpectNotNull(tmpDataRet);
            ExpectIntEQ(tmpData, *(int*)tmpDataRet);
        }
    #else
        ExpectIntEQ(X509_STORE_CTX_set_ex_data(ctx, i, &tmpData),
                    WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
        tmpDataRet = (int*)X509_STORE_CTX_get_ex_data(ctx, i);
        ExpectNull(tmpDataRet);
    #endif
        X509_STORE_CTX_free(ctx);
        ctx = NULL;
    }

    /* test X509_STORE_get/set_ex_data */
    {
        int i = 0, tmpData = 99;
        void* tmpDataRet;
        ExpectNotNull(str = X509_STORE_new());
    #ifdef HAVE_EX_DATA
        for (i = 0; i < MAX_EX_DATA; i++) {
            ExpectIntEQ(X509_STORE_set_ex_data(str, i, &tmpData),
                        WOLFSSL_SUCCESS);
            tmpDataRet = (int*)X509_STORE_get_ex_data(str, i);
            ExpectNotNull(tmpDataRet);
            ExpectIntEQ(tmpData, *(int*)tmpDataRet);
        }
    #else
        ExpectIntEQ(X509_STORE_set_ex_data(str, i, &tmpData),
                    WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
        tmpDataRet = (int*)X509_STORE_get_ex_data(str, i);
        ExpectNull(tmpDataRet);
    #endif
        X509_STORE_free(str);
        str = NULL;
    }

#endif /* defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && \
        * !defined(NO_FILESYSTEM) && !defined(NO_RSA) */

    return EXPECT_RESULT();
}

#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && \
   !defined(NO_FILESYSTEM) && !defined(NO_RSA)

typedef struct {
    const char *caFile;
    const char *caIntFile;
    const char *caInt2File;
    const char *leafFile;
    X509 *x509Ca;
    X509 *x509CaInt;
    X509 *x509CaInt2;
    X509 *x509Leaf;
    STACK_OF(X509)* expectedChain;
} X509_STORE_test_data;

static X509 * test_wolfSSL_X509_STORE_CTX_ex_helper(const char *file)
{
    XFILE fp = XBADFILE;
    X509 *x = NULL;

    fp = XFOPEN(file, "rb");
    if (fp == NULL) {
        return NULL;
    }
    x = PEM_read_X509(fp, 0, 0, 0);
    XFCLOSE(fp);

    return x;
}

static int test_wolfSSL_X509_STORE_CTX_ex1(X509_STORE_test_data *testData)
{
    EXPECT_DECLS;
    X509_STORE* store = NULL;
    X509_STORE_CTX* ctx = NULL;
    STACK_OF(X509)* chain = NULL;
    int i = 0;

    /* Test case 1, add X509 certs to store and verify */
    ExpectNotNull(store = X509_STORE_new());
    ExpectIntEQ(X509_STORE_add_cert(store, testData->x509Ca), 1);
    ExpectIntEQ(X509_STORE_add_cert(store, testData->x509CaInt), 1);
    ExpectIntEQ(X509_STORE_add_cert(store, testData->x509CaInt2), 1);
    ExpectNotNull(ctx = X509_STORE_CTX_new());
    ExpectIntEQ(X509_STORE_CTX_init(ctx, store, testData->x509Leaf, NULL), 1);
    ExpectIntEQ(X509_verify_cert(ctx), 1);
    ExpectNotNull(chain = X509_STORE_CTX_get_chain(ctx));
    ExpectIntEQ(sk_X509_num(chain), sk_X509_num(testData->expectedChain));
    for (i = 0; i < sk_X509_num(chain); i++) {
        ExpectIntEQ(X509_cmp(sk_X509_value(chain, i),
                             sk_X509_value(testData->expectedChain, i)), 0);
    }
    X509_STORE_CTX_free(ctx);
    X509_STORE_free(store);
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_STORE_CTX_ex2(X509_STORE_test_data *testData)
{
    EXPECT_DECLS;
    X509_STORE* store = NULL;
    X509_STORE_CTX* ctx = NULL;
    STACK_OF(X509)* chain = NULL;
    int i = 0;

    /* Test case 2, add certs by filename to store and verify */
    ExpectNotNull(store = X509_STORE_new());
    ExpectIntEQ(X509_STORE_load_locations(
        store, testData->caFile, NULL), 1);
    ExpectIntEQ(X509_STORE_load_locations(
        store, testData->caIntFile, NULL), 1);
    ExpectIntEQ(X509_STORE_load_locations(
        store, testData->caInt2File, NULL), 1);
    ExpectNotNull(ctx = X509_STORE_CTX_new());
    ExpectIntEQ(X509_STORE_CTX_init(ctx, store, testData->x509Leaf, NULL), 1);
    ExpectIntEQ(X509_verify_cert(ctx), 1);
    ExpectNotNull(chain = X509_STORE_CTX_get_chain(ctx));
    ExpectIntEQ(sk_X509_num(chain), sk_X509_num(testData->expectedChain));
    for (i = 0; i < sk_X509_num(chain); i++) {
        ExpectIntEQ(X509_cmp(sk_X509_value(chain, i),
                             sk_X509_value(testData->expectedChain, i)), 0);
    }
    X509_STORE_CTX_free(ctx);
    X509_STORE_free(store);
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_STORE_CTX_ex3(X509_STORE_test_data *testData)
{
    EXPECT_DECLS;
    X509_STORE* store = NULL;
    X509_STORE_CTX* ctx = NULL;
    STACK_OF(X509)* chain = NULL;
    int i = 0;

    /* Test case 3, mix and match X509 with files */
    ExpectNotNull(store = X509_STORE_new());
    ExpectIntEQ(X509_STORE_add_cert(store, testData->x509CaInt), 1);
    ExpectIntEQ(X509_STORE_add_cert(store, testData->x509CaInt2), 1);
    ExpectIntEQ(X509_STORE_load_locations(
        store, testData->caFile, NULL), 1);
    ExpectNotNull(ctx = X509_STORE_CTX_new());
    ExpectIntEQ(X509_STORE_CTX_init(ctx, store, testData->x509Leaf, NULL), 1);
    ExpectIntEQ(X509_verify_cert(ctx), 1);
    ExpectNotNull(chain = X509_STORE_CTX_get_chain(ctx));
    ExpectIntEQ(sk_X509_num(chain), sk_X509_num(testData->expectedChain));
    for (i = 0; i < sk_X509_num(chain); i++) {
        ExpectIntEQ(X509_cmp(sk_X509_value(chain, i),
                             sk_X509_value(testData->expectedChain, i)), 0);
    }
    X509_STORE_CTX_free(ctx);
    X509_STORE_free(store);
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_STORE_CTX_ex4(X509_STORE_test_data *testData)
{
    EXPECT_DECLS;
    X509_STORE* store = NULL;
    X509_STORE_CTX* ctx = NULL;
    STACK_OF(X509)* chain = NULL;
    STACK_OF(X509)* inter = NULL;
    int i = 0;

    /* Test case 4, CA loaded by file, intermediates passed on init */
    ExpectNotNull(store = X509_STORE_new());
    ExpectIntEQ(X509_STORE_load_locations(
        store, testData->caFile, NULL), 1);
    ExpectNotNull(inter = sk_X509_new_null());
    ExpectIntGE(sk_X509_push(inter, testData->x509CaInt), 1);
    ExpectIntGE(sk_X509_push(inter, testData->x509CaInt2), 1);
    ExpectNotNull(ctx = X509_STORE_CTX_new());
    ExpectIntEQ(X509_STORE_CTX_init(ctx, store, testData->x509Leaf, inter), 1);
    ExpectIntEQ(X509_verify_cert(ctx), 1);
    ExpectNotNull(chain = X509_STORE_CTX_get_chain(ctx));
    ExpectIntEQ(sk_X509_num(chain), sk_X509_num(testData->expectedChain));
    for (i = 0; i < sk_X509_num(chain); i++) {
        ExpectIntEQ(X509_cmp(sk_X509_value(chain, i),
                             sk_X509_value(testData->expectedChain, i)), 0);
    }
    X509_STORE_CTX_free(ctx);
    X509_STORE_free(store);
    sk_X509_free(inter);
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_STORE_CTX_ex5(X509_STORE_test_data *testData)
{
    EXPECT_DECLS;
    X509_STORE* store = NULL;
    X509_STORE_CTX* ctx = NULL;
    STACK_OF(X509)* chain = NULL;
    STACK_OF(X509)* trusted = NULL;
    int i = 0;

    /* Test case 5, manually set trusted stack */
    ExpectNotNull(store = X509_STORE_new());
    ExpectNotNull(trusted = sk_X509_new_null());
    ExpectIntGE(sk_X509_push(trusted, testData->x509Ca), 1);
    ExpectIntGE(sk_X509_push(trusted, testData->x509CaInt), 1);
    ExpectIntGE(sk_X509_push(trusted, testData->x509CaInt2), 1);
    ExpectNotNull(ctx = X509_STORE_CTX_new());
    ExpectIntEQ(X509_STORE_CTX_init(ctx, store, testData->x509Leaf, NULL), 1);
    X509_STORE_CTX_trusted_stack(ctx, trusted);
    ExpectIntEQ(X509_verify_cert(ctx), 1);
    ExpectNotNull(chain = X509_STORE_CTX_get_chain(ctx));
    ExpectIntEQ(sk_X509_num(chain), sk_X509_num(testData->expectedChain));
    for (i = 0; i < sk_X509_num(chain); i++) {
        ExpectIntEQ(X509_cmp(sk_X509_value(chain, i),
                             sk_X509_value(testData->expectedChain, i)), 0);
    }
    X509_STORE_CTX_free(ctx);
    X509_STORE_free(store);
    sk_X509_free(trusted);
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_STORE_CTX_ex6(X509_STORE_test_data *testData)
{
    EXPECT_DECLS;
    X509_STORE* store = NULL;
    X509_STORE_CTX* ctx = NULL;
    STACK_OF(X509)* chain = NULL;
    STACK_OF(X509)* trusted = NULL;
    STACK_OF(X509)* inter = NULL;
    int i = 0;

    /* Test case 6, manually set trusted stack will be unified with
     * any intermediates provided on init */
    ExpectNotNull(store = X509_STORE_new());
    ExpectNotNull(trusted = sk_X509_new_null());
    ExpectNotNull(inter = sk_X509_new_null());
    ExpectIntGE(sk_X509_push(trusted, testData->x509Ca), 1);
    ExpectIntGE(sk_X509_push(inter, testData->x509CaInt), 1);
    ExpectIntGE(sk_X509_push(inter, testData->x509CaInt2), 1);
    ExpectNotNull(ctx = X509_STORE_CTX_new());
    ExpectIntEQ(X509_STORE_CTX_init(ctx, store, testData->x509Leaf, inter), 1);
    X509_STORE_CTX_trusted_stack(ctx, trusted);
    ExpectIntEQ(X509_verify_cert(ctx), 1);
    ExpectNotNull(chain = X509_STORE_CTX_get_chain(ctx));
    ExpectIntEQ(sk_X509_num(chain), sk_X509_num(testData->expectedChain));
    for (i = 0; i < sk_X509_num(chain); i++) {
        ExpectIntEQ(X509_cmp(sk_X509_value(chain, i),
                             sk_X509_value(testData->expectedChain, i)), 0);
    }
    X509_STORE_CTX_free(ctx);
    X509_STORE_free(store);
    sk_X509_free(trusted);
    sk_X509_free(inter);
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_STORE_CTX_ex7(X509_STORE_test_data *testData)
{
    EXPECT_DECLS;
    X509_STORE* store = NULL;
    X509_STORE_CTX* ctx = NULL;
    STACK_OF(X509)* chain = NULL;
    int i = 0;

    /* Test case 7, certs added to store after ctx init are still used */
    ExpectNotNull(store = X509_STORE_new());
    ExpectNotNull(ctx = X509_STORE_CTX_new());
    ExpectIntEQ(X509_STORE_CTX_init(ctx, store, testData->x509Leaf, NULL), 1);
    ExpectIntNE(X509_verify_cert(ctx), 1);
    ExpectIntEQ(X509_STORE_add_cert(store, testData->x509CaInt2), 1);
    ExpectIntEQ(X509_STORE_add_cert(store, testData->x509CaInt), 1);
    ExpectIntEQ(X509_STORE_add_cert(store, testData->x509Ca), 1);
    ExpectIntEQ(X509_verify_cert(ctx), 1);
    ExpectNotNull(chain = X509_STORE_CTX_get_chain(ctx));
    ExpectIntEQ(sk_X509_num(chain), sk_X509_num(testData->expectedChain));
    for (i = 0; i < sk_X509_num(chain); i++) {
        ExpectIntEQ(X509_cmp(sk_X509_value(chain, i),
                             sk_X509_value(testData->expectedChain, i)), 0);
    }
    X509_STORE_CTX_free(ctx);
    X509_STORE_free(store);
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_STORE_CTX_ex8(X509_STORE_test_data *testData)
{
    EXPECT_DECLS;
    X509_STORE* store = NULL;
    X509_STORE_CTX* ctx = NULL;
    STACK_OF(X509)* chain = NULL;
    int i = 0;

    /* Test case 8, Only full chain verifies */
    ExpectNotNull(store = X509_STORE_new());
    ExpectNotNull(ctx = X509_STORE_CTX_new());
    ExpectIntEQ(X509_STORE_CTX_init(ctx, store, testData->x509Leaf, NULL), 1);
    ExpectIntNE(X509_verify_cert(ctx), 1);
    ExpectIntEQ(X509_STORE_add_cert(store, testData->x509CaInt2), 1);
    ExpectIntNE(X509_verify_cert(ctx), 1);
    ExpectIntEQ(X509_STORE_add_cert(store, testData->x509CaInt), 1);
    ExpectIntNE(X509_verify_cert(ctx), 1);
    ExpectIntEQ(X509_STORE_add_cert(store, testData->x509Ca), 1);
    ExpectIntEQ(X509_verify_cert(ctx), 1);
    ExpectNotNull(chain = X509_STORE_CTX_get_chain(ctx));
    ExpectIntEQ(sk_X509_num(chain), sk_X509_num(testData->expectedChain));
    for (i = 0; i < sk_X509_num(chain); i++) {
        ExpectIntEQ(X509_cmp(sk_X509_value(chain, i),
                             sk_X509_value(testData->expectedChain, i)), 0);
    }
    X509_STORE_CTX_free(ctx);
    X509_STORE_free(store);
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_STORE_CTX_ex9(X509_STORE_test_data *testData)
{
    EXPECT_DECLS;
    X509_STORE* store = NULL;
    X509_STORE_CTX* ctx = NULL;
    X509_STORE_CTX* ctx2 = NULL;
    STACK_OF(X509)* trusted = NULL;

    /* Test case 9, certs added to store should not be reflected in ctx that
     * has been manually set with a trusted stack, but are reflected in ctx
     * that has not set trusted stack */
    ExpectNotNull(store = X509_STORE_new());
    ExpectNotNull(ctx = X509_STORE_CTX_new());
    ExpectNotNull(ctx2 = X509_STORE_CTX_new());
    ExpectNotNull(trusted = sk_X509_new_null());
    ExpectIntGE(sk_X509_push(trusted, testData->x509Ca), 1);
    ExpectIntGE(sk_X509_push(trusted, testData->x509CaInt), 1);
    ExpectIntGE(sk_X509_push(trusted, testData->x509CaInt2), 1);
    ExpectIntEQ(X509_STORE_CTX_init(ctx, store, testData->x509Leaf, NULL), 1);
    ExpectIntEQ(X509_STORE_CTX_init(ctx2, store, testData->x509Leaf, NULL), 1);
    ExpectIntNE(X509_verify_cert(ctx), 1);
    ExpectIntNE(X509_verify_cert(ctx2), 1);
    X509_STORE_CTX_trusted_stack(ctx, trusted);
    /* CTX1 should now verify */
    ExpectIntEQ(X509_verify_cert(ctx), 1);
    ExpectIntNE(X509_verify_cert(ctx2), 1);
    ExpectIntEQ(X509_STORE_add_cert(store, testData->x509Ca), 1);
    ExpectIntEQ(X509_STORE_add_cert(store, testData->x509CaInt), 1);
    ExpectIntEQ(X509_STORE_add_cert(store, testData->x509CaInt2), 1);
    /* CTX2 should now verify */
    ExpectIntEQ(X509_verify_cert(ctx2), 1);
    X509_STORE_CTX_free(ctx);
    X509_STORE_CTX_free(ctx2);
    X509_STORE_free(store);
    sk_X509_free(trusted);
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_STORE_CTX_ex10(X509_STORE_test_data *testData)
{
    EXPECT_DECLS;
    X509_STORE* store = NULL;
    X509_STORE_CTX* ctx = NULL;
    STACK_OF(X509)* chain = NULL;

    /* Test case 10, ensure partial chain flag works */
    ExpectNotNull(store = X509_STORE_new());
    ExpectIntEQ(X509_STORE_add_cert(store, testData->x509CaInt), 1);
    ExpectIntEQ(X509_STORE_add_cert(store, testData->x509CaInt2), 1);
    ExpectNotNull(ctx = X509_STORE_CTX_new());
    ExpectIntEQ(X509_STORE_CTX_init(ctx, store, testData->x509Leaf, NULL), 1);
    /* Fails because chain is incomplete */
    ExpectIntNE(X509_verify_cert(ctx), 1);
    ExpectIntEQ(X509_STORE_set_flags(store, X509_V_FLAG_PARTIAL_CHAIN), 1);
    /* Partial chain now OK */
    ExpectIntEQ(X509_verify_cert(ctx), 1);
    ExpectNotNull(chain = X509_STORE_CTX_get_chain(ctx));
    X509_STORE_CTX_free(ctx);
    X509_STORE_free(store);
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_STORE_CTX_ex11(X509_STORE_test_data *testData)
{
    EXPECT_DECLS;
    X509_STORE* store = NULL;
    X509_STORE_CTX* ctx = NULL;
    STACK_OF(X509)* chain = NULL;

    /* Test case 11, test partial chain flag on ctx itself */
    ExpectNotNull(store = X509_STORE_new());
    ExpectIntEQ(X509_STORE_add_cert(store, testData->x509CaInt), 1);
    ExpectIntEQ(X509_STORE_add_cert(store, testData->x509CaInt2), 1);
    ExpectNotNull(ctx = X509_STORE_CTX_new());
    ExpectIntEQ(X509_STORE_CTX_init(ctx, store, testData->x509Leaf, NULL), 1);
    /* Fails because chain is incomplete */
    ExpectIntNE(X509_verify_cert(ctx), 1);
    X509_STORE_CTX_set_flags(ctx, X509_V_FLAG_PARTIAL_CHAIN);
    /* Partial chain now OK */
    ExpectIntEQ(X509_verify_cert(ctx), 1);
    ExpectNotNull(chain = X509_STORE_CTX_get_chain(ctx));
    X509_STORE_CTX_free(ctx);
    X509_STORE_free(store);
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_STORE_CTX_ex12(void)
{
    EXPECT_DECLS;
#ifdef HAVE_ECC
    X509_STORE* store = NULL;
    X509_STORE_CTX* ctx = NULL;
    STACK_OF(X509)* chain = NULL;
    X509* rootEccX509 = NULL;
    X509* badAkiX509 = NULL;
    X509* ca1X509 = NULL;

    const char* intCARootECCFile    = "./certs/ca-ecc-cert.pem";
    const char* intCA1ECCFile       = "./certs/intermediate/ca-int-ecc-cert.pem";
    const char* intCABadAKIECCFile = "./certs/intermediate/ca-ecc-bad-aki.pem";

    /* Test case 12, multiple CAs with the same SKI including 1 with intentionally
       bad/unregistered AKI.  x509_verify_cert should still form a valid chain
       using the valid CA, ignoring the bad CA. Developed from customer provided
       reproducer. */

    ExpectNotNull(store = X509_STORE_new());
    ExpectNotNull(rootEccX509 = test_wolfSSL_X509_STORE_CTX_ex_helper(intCARootECCFile));
    ExpectIntEQ(X509_STORE_add_cert(store, rootEccX509), 1);
    ExpectNotNull(badAkiX509 = test_wolfSSL_X509_STORE_CTX_ex_helper(intCABadAKIECCFile));
    ExpectNotNull(ctx = X509_STORE_CTX_new());
    ExpectIntEQ(X509_STORE_CTX_init(ctx, store, badAkiX509, NULL), 1);
    ExpectIntEQ(X509_verify_cert(ctx), 0);
    X509_STORE_CTX_cleanup(ctx);

    ExpectIntEQ(X509_STORE_add_cert(store, badAkiX509), 1);
    ExpectNotNull(ca1X509 = test_wolfSSL_X509_STORE_CTX_ex_helper(intCA1ECCFile));
    ExpectIntEQ(X509_STORE_CTX_init(ctx, store, ca1X509, NULL), 1);
    ExpectIntEQ(X509_verify_cert(ctx), 1);
    ExpectNotNull(chain = X509_STORE_CTX_get_chain(ctx));

    X509_STORE_CTX_free(ctx);
    X509_STORE_free(store);
    X509_free(rootEccX509);
    X509_free(badAkiX509);
    X509_free(ca1X509);
#endif
    return EXPECT_RESULT();
}
#endif

static int test_wolfSSL_X509_STORE_CTX_ex(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && \
   !defined(NO_FILESYSTEM) && !defined(NO_RSA)
    X509_STORE_test_data testData;
    XMEMSET((void *)&testData, 0, sizeof(X509_STORE_test_data));
    testData.caFile =     "./certs/ca-cert.pem";
    testData.caIntFile =  "./certs/intermediate/ca-int-cert.pem";
    testData.caInt2File = "./certs/intermediate/ca-int2-cert.pem";
    testData.leafFile =   "./certs/intermediate/server-chain.pem";

    ExpectNotNull(testData.x509Ca = \
                  test_wolfSSL_X509_STORE_CTX_ex_helper(testData.caFile));
    ExpectNotNull(testData.x509CaInt = \
                  test_wolfSSL_X509_STORE_CTX_ex_helper(testData.caIntFile));
    ExpectNotNull(testData.x509CaInt2 = \
                  test_wolfSSL_X509_STORE_CTX_ex_helper(testData.caInt2File));
    ExpectNotNull(testData.x509Leaf = \
                  test_wolfSSL_X509_STORE_CTX_ex_helper(testData.leafFile));
    ExpectNotNull(testData.expectedChain = sk_X509_new_null());
    ExpectIntGE(sk_X509_push(testData.expectedChain, testData.x509Leaf), 1);
    ExpectIntGE(sk_X509_push(testData.expectedChain, testData.x509CaInt2), 1);
    ExpectIntGE(sk_X509_push(testData.expectedChain, testData.x509CaInt), 1);
    ExpectIntGE(sk_X509_push(testData.expectedChain, testData.x509Ca), 1);

    ExpectIntEQ(test_wolfSSL_X509_STORE_CTX_ex1(&testData), 1);
    ExpectIntEQ(test_wolfSSL_X509_STORE_CTX_ex2(&testData), 1);
    ExpectIntEQ(test_wolfSSL_X509_STORE_CTX_ex3(&testData), 1);
    ExpectIntEQ(test_wolfSSL_X509_STORE_CTX_ex4(&testData), 1);
    ExpectIntEQ(test_wolfSSL_X509_STORE_CTX_ex5(&testData), 1);
    ExpectIntEQ(test_wolfSSL_X509_STORE_CTX_ex6(&testData), 1);
    ExpectIntEQ(test_wolfSSL_X509_STORE_CTX_ex7(&testData), 1);
    ExpectIntEQ(test_wolfSSL_X509_STORE_CTX_ex8(&testData), 1);
    ExpectIntEQ(test_wolfSSL_X509_STORE_CTX_ex9(&testData), 1);
    ExpectIntEQ(test_wolfSSL_X509_STORE_CTX_ex10(&testData), 1);
    ExpectIntEQ(test_wolfSSL_X509_STORE_CTX_ex11(&testData), 1);
    test_wolfSSL_X509_STORE_CTX_ex12();

    if(testData.x509Ca) {
        X509_free(testData.x509Ca);
    }
    if(testData.x509CaInt) {
        X509_free(testData.x509CaInt);
    }
    if(testData.x509CaInt2) {
        X509_free(testData.x509CaInt2);
    }
    if(testData.x509Leaf) {
        X509_free(testData.x509Leaf);
    }
    if (testData.expectedChain) {
        sk_X509_free(testData.expectedChain);
    }

#endif /* defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && \
        * !defined(NO_FILESYSTEM) && !defined(NO_RSA) */

    return EXPECT_RESULT();
}


#if defined(OPENSSL_EXTRA) && !defined(NO_RSA) && !defined(NO_FILESYSTEM)
static int test_X509_STORE_untrusted_load_cert_to_stack(const char* filename,
        STACK_OF(X509)* chain)
{
    EXPECT_DECLS;
    XFILE fp = XBADFILE;
    X509* cert = NULL;

    ExpectTrue((fp = XFOPEN(filename, "rb"))
            != XBADFILE);
    ExpectNotNull(cert = PEM_read_X509(fp, 0, 0, 0 ));
    if (fp != XBADFILE) {
        XFCLOSE(fp);
        fp = XBADFILE;
    }
    ExpectIntGT(sk_X509_push(chain, cert), 0);
    if (EXPECT_FAIL())
        X509_free(cert);

    return EXPECT_RESULT();
}

#if defined(OPENSSL_ALL)

static int last_errcode;
static int last_errdepth;

static int X509Callback(int ok, X509_STORE_CTX *ctx)
{

    if (!ok) {
        last_errcode  = X509_STORE_CTX_get_error(ctx);
        last_errdepth = X509_STORE_CTX_get_error_depth(ctx);
    }
    /* Always return OK to allow verification to continue.*/
    return 1;
}

static int test_X509_STORE_InvalidCa(void)
{
    EXPECT_DECLS;
    const char* filename = "./certs/intermediate/ca_false_intermediate/"
                                                    "test_int_not_cacert.pem";
    const char* srvfile = "./certs/intermediate/ca_false_intermediate/"
                                            "test_sign_bynoca_srv.pem";
    X509_STORE_CTX* ctx = NULL;
    X509_STORE* str = NULL;
    XFILE fp = XBADFILE;
    X509* cert = NULL;
    STACK_OF(X509)* untrusted = NULL;

    last_errcode = 0;
    last_errdepth = 0;

    ExpectTrue((fp = XFOPEN(srvfile, "rb"))
            != XBADFILE);
    ExpectNotNull(cert = PEM_read_X509(fp, 0, 0, 0 ));
    if (fp != XBADFILE) {
        XFCLOSE(fp);
        fp = XBADFILE;
    }

    ExpectNotNull(str = X509_STORE_new());
    ExpectNotNull(ctx = X509_STORE_CTX_new());
    ExpectNotNull(untrusted = sk_X509_new_null());

    /* create cert chain stack */
    ExpectIntEQ(test_X509_STORE_untrusted_load_cert_to_stack(filename,
                untrusted), TEST_SUCCESS);

    X509_STORE_set_verify_cb(str, X509Callback);

    ExpectIntEQ(X509_STORE_load_locations(str,
                "./certs/intermediate/ca_false_intermediate/test_ca.pem",
                                                                    NULL), 1);

    ExpectIntEQ(X509_STORE_CTX_init(ctx, str, cert, untrusted), 1);
    ExpectIntEQ(X509_verify_cert(ctx), 1);
    ExpectIntEQ(last_errcode, X509_V_ERR_INVALID_CA);

    X509_free(cert);
    X509_STORE_free(str);
    X509_STORE_CTX_free(ctx);
    sk_X509_pop_free(untrusted, NULL);

    return EXPECT_RESULT();
}
#endif /* OPENSSL_ALL */



static int test_X509_STORE_untrusted_certs(const char** filenames, int ret,
        int err, int loadCA)
{
    EXPECT_DECLS;
    X509_STORE_CTX* ctx = NULL;
    X509_STORE* str = NULL;
    XFILE fp = XBADFILE;
    X509* cert = NULL;
    STACK_OF(X509)* untrusted = NULL;

    ExpectTrue((fp = XFOPEN("./certs/intermediate/server-int-cert.pem", "rb"))
            != XBADFILE);
    ExpectNotNull(cert = PEM_read_X509(fp, 0, 0, 0 ));
    if (fp != XBADFILE) {
        XFCLOSE(fp);
        fp = XBADFILE;
    }

    ExpectNotNull(str = X509_STORE_new());
    ExpectNotNull(ctx = X509_STORE_CTX_new());
    ExpectNotNull(untrusted = sk_X509_new_null());

    ExpectIntEQ(X509_STORE_set_flags(str, 0), 1);
    if (loadCA) {
        ExpectIntEQ(X509_STORE_load_locations(str, "./certs/ca-cert.pem", NULL),
                1);
    }
    for (; *filenames; filenames++) {
        ExpectIntEQ(test_X509_STORE_untrusted_load_cert_to_stack(*filenames,
                untrusted), TEST_SUCCESS);
    }

    ExpectIntEQ(X509_STORE_CTX_init(ctx, str, cert, untrusted), 1);
    ExpectIntEQ(X509_verify_cert(ctx), ret);
    ExpectIntEQ(X509_STORE_CTX_get_error(ctx), err);

    X509_free(cert);
    X509_STORE_free(str);
    X509_STORE_CTX_free(ctx);
    sk_X509_pop_free(untrusted, NULL);

    return EXPECT_RESULT();
}
#endif

static int test_X509_STORE_untrusted(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_RSA) && !defined(NO_FILESYSTEM)
    const char* untrusted1[] = {
        "./certs/intermediate/ca-int2-cert.pem",
        NULL
    };
    const char* untrusted2[] = {
        "./certs/intermediate/ca-int-cert.pem",
        "./certs/intermediate/ca-int2-cert.pem",
        NULL
    };
    const char* untrusted3[] = {
        "./certs/intermediate/ca-int-cert.pem",
        "./certs/intermediate/ca-int2-cert.pem",
        "./certs/ca-cert.pem",
        NULL
    };
    /* Adding unrelated certs that should be ignored */
    const char* untrusted4[] = {
        "./certs/client-ca.pem",
        "./certs/intermediate/ca-int-cert.pem",
        "./certs/server-cert.pem",
        "./certs/intermediate/ca-int2-cert.pem",
        NULL
    };

    /* Only immediate issuer in untrusted chain. Fails since can't build chain
     * to loaded CA. */
    ExpectIntEQ(test_X509_STORE_untrusted_certs(untrusted1, 0,
            X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT_LOCALLY, 1), TEST_SUCCESS);
    /* Succeeds because path to loaded CA is available. */
    ExpectIntEQ(test_X509_STORE_untrusted_certs(untrusted2, 1, 0, 1),
            TEST_SUCCESS);
    /* Root CA in untrusted chain is OK so long as CA has been loaded
     * properly */
    ExpectIntEQ(test_X509_STORE_untrusted_certs(untrusted3, 1, 0, 1),
            TEST_SUCCESS);
    /* Still needs properly loaded CA, while including it in untrusted
     * list is not an error, it also doesn't count for verify */
    ExpectIntEQ(test_X509_STORE_untrusted_certs(untrusted3, 0,
                X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT_LOCALLY, 0),
            TEST_SUCCESS);
    /* Succeeds because path to loaded CA is available. */
    ExpectIntEQ(test_X509_STORE_untrusted_certs(untrusted4, 1, 0, 1),
            TEST_SUCCESS);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_STORE_set_flags(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && \
   !defined(NO_FILESYSTEM) && !defined(NO_RSA)
    X509_STORE* store = NULL;
    X509* x509 = NULL;

    ExpectNotNull((store = wolfSSL_X509_STORE_new()));
    ExpectNotNull((x509 = wolfSSL_X509_load_certificate_file(svrCertFile,
        WOLFSSL_FILETYPE_PEM)));
    ExpectIntEQ(X509_STORE_add_cert(store, x509), WOLFSSL_SUCCESS);

#ifdef HAVE_CRL
    ExpectIntEQ(X509_STORE_set_flags(store, WOLFSSL_CRL_CHECKALL),
        WOLFSSL_SUCCESS);
#else
    ExpectIntEQ(X509_STORE_set_flags(store, WOLFSSL_CRL_CHECKALL),
        WC_NO_ERR_TRACE(NOT_COMPILED_IN));
#endif

    wolfSSL_X509_free(x509);
    wolfSSL_X509_STORE_free(store);
#endif /* defined(OPENSSL_EXTRA) && !defined(NO_CERTS) &&
        * !defined(NO_FILESYSTEM) && !defined(NO_RSA) */
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_LOOKUP_load_file(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && defined(HAVE_CRL) && \
   !defined(NO_FILESYSTEM) && !defined(NO_RSA) && defined(HAVE_ECC) && \
   (!defined(NO_WOLFSSL_CLIENT) || !defined(WOLFSSL_NO_CLIENT_AUTH))
    WOLFSSL_X509_STORE*  store = NULL;
    WOLFSSL_X509_LOOKUP* lookup = NULL;

    ExpectNotNull(store = wolfSSL_X509_STORE_new());
    ExpectNotNull(lookup = X509_STORE_add_lookup(store, X509_LOOKUP_file()));
    /* One RSA and one ECC certificate in file. */
    ExpectIntEQ(wolfSSL_X509_LOOKUP_load_file(lookup, "certs/client-ca.pem",
        X509_FILETYPE_PEM), 1);
    ExpectIntEQ(wolfSSL_X509_LOOKUP_load_file(lookup, "certs/crl/crl2.pem",
        X509_FILETYPE_PEM), 1);

    if (store != NULL) {
        ExpectIntEQ(wolfSSL_CertManagerVerify(store->cm, cliCertFile,
            WOLFSSL_FILETYPE_PEM), 1);
        ExpectIntEQ(wolfSSL_CertManagerVerify(store->cm, svrCertFile,
            WOLFSSL_FILETYPE_PEM), WC_NO_ERR_TRACE(ASN_NO_SIGNER_E));
    }
    ExpectIntEQ(wolfSSL_X509_LOOKUP_load_file(lookup, "certs/ca-cert.pem",
        X509_FILETYPE_PEM), 1);
    if (store != NULL) {
        ExpectIntEQ(wolfSSL_CertManagerVerify(store->cm, svrCertFile,
            WOLFSSL_FILETYPE_PEM), 1);
    }

    wolfSSL_X509_STORE_free(store);
#endif /* defined(OPENSSL_EXTRA) && defined(HAVE_CRL) &&
        * !defined(NO_FILESYSTEM) && !defined(NO_RSA) */
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_STORE_CTX_set_time(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA)
    WOLFSSL_X509_STORE_CTX* ctx = NULL;
    time_t c_time;

    ExpectNotNull(ctx = wolfSSL_X509_STORE_CTX_new());
    c_time = 365*24*60*60;
    wolfSSL_X509_STORE_CTX_set_time(ctx, 0, c_time);
    ExpectTrue((ctx->param->flags & WOLFSSL_USE_CHECK_TIME) ==
        WOLFSSL_USE_CHECK_TIME);
    ExpectTrue(ctx->param->check_time == c_time);
    wolfSSL_X509_STORE_CTX_free(ctx);
#endif /* OPENSSL_EXTRA */
    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_get0_set1_param(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA)
#if !defined(NO_TLS) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
    SSL_CTX* ctx = NULL;
    WOLFSSL_X509_VERIFY_PARAM* pParam = NULL;
    WOLFSSL_X509_VERIFY_PARAM* pvpm = NULL;
    char testIPv4[] = "127.0.0.1";
    char testhostName[] = "foo.hoge.com";

#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_server_method()));
#else
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_client_method()));
#endif

    ExpectNull(SSL_CTX_get0_param(NULL));
    ExpectNotNull(pParam = SSL_CTX_get0_param(ctx));

    ExpectNotNull(pvpm = (WOLFSSL_X509_VERIFY_PARAM *)XMALLOC(
        sizeof(WOLFSSL_X509_VERIFY_PARAM), NULL, DYNAMIC_TYPE_OPENSSL));
    ExpectNotNull(XMEMSET(pvpm, 0, sizeof(WOLFSSL_X509_VERIFY_PARAM)));

    ExpectIntEQ(wolfSSL_X509_VERIFY_PARAM_set1_host(pvpm, testhostName,
        (int)XSTRLEN(testhostName)), WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_VERIFY_PARAM_set1_ip_asc(pvpm, testIPv4),
        WOLFSSL_SUCCESS);
    wolfSSL_X509_VERIFY_PARAM_set_hostflags(pvpm, 0x01);

    ExpectIntEQ(SSL_CTX_set1_param(ctx, pvpm), 1);
    ExpectIntEQ(0, XSTRNCMP(pParam->hostName, testhostName,
        (int)XSTRLEN(testhostName)));
    ExpectIntEQ(0x01, pParam->hostFlags);
    ExpectIntEQ(0, XSTRNCMP(pParam->ipasc, testIPv4, WOLFSSL_MAX_IPSTR));

    /* test for incorrect parameter */
    ExpectIntEQ(1,SSL_CTX_set1_param(ctx, NULL));
    ExpectIntEQ(1,SSL_CTX_set1_param(NULL, pvpm));
    ExpectIntEQ(1,SSL_CTX_set1_param(NULL, NULL));

    SSL_CTX_free(ctx);

    XFREE(pvpm, NULL, DYNAMIC_TYPE_OPENSSL);
#endif /* !NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER */
#endif /* OPENSSL_EXTRA && !defined(NO_RSA)*/
    return EXPECT_RESULT();
}

static int test_wolfSSL_get0_param(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_RSA) && !defined(NO_TLS) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)) && \
    !defined(NO_FILESYSTEM)
    SSL_CTX* ctx = NULL;
    SSL*     ssl = NULL;

#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_server_method()));
#else
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_client_method()));
#endif
    ExpectTrue(SSL_CTX_use_certificate_file(ctx, svrCertFile,
        SSL_FILETYPE_PEM));
    ExpectTrue(SSL_CTX_use_PrivateKey_file(ctx, svrKeyFile, SSL_FILETYPE_PEM));
    ExpectNotNull(ssl = SSL_new(ctx));

    ExpectNotNull(SSL_get0_param(ssl));

    SSL_free(ssl);
    SSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_VERIFY_PARAM_set1_host(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA)
    const char host[] = "www.example.com";
    WOLFSSL_X509_VERIFY_PARAM* pParam = NULL;

    ExpectNotNull(pParam = (WOLFSSL_X509_VERIFY_PARAM*)XMALLOC(
        sizeof(WOLFSSL_X509_VERIFY_PARAM), HEAP_HINT, DYNAMIC_TYPE_OPENSSL));
    if (pParam != NULL) {
        XMEMSET(pParam, 0, sizeof(WOLFSSL_X509_VERIFY_PARAM));

        ExpectIntEQ(X509_VERIFY_PARAM_set1_host(NULL, host, sizeof(host)),
            WOLFSSL_FAILURE);

        X509_VERIFY_PARAM_set1_host(pParam, host, sizeof(host));

        ExpectIntEQ(XMEMCMP(pParam->hostName, host, sizeof(host)), 0);

        XMEMSET(pParam, 0, sizeof(WOLFSSL_X509_VERIFY_PARAM));

        ExpectIntNE(XMEMCMP(pParam->hostName, host, sizeof(host)), 0);

        XFREE(pParam, HEAP_HINT, DYNAMIC_TYPE_OPENSSL);
    }
#endif /* OPENSSL_EXTRA */
    return EXPECT_RESULT();
}

static int test_wolfSSL_set1_host(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_RSA) && !defined(NO_TLS) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)) && \
    !defined(NO_FILESYSTEM)
    const char host[] = "www.test_wolfSSL_set1_host.com";
    const char emptyStr[] = "";
    SSL_CTX*   ctx = NULL;
    SSL*       ssl = NULL;
    WOLFSSL_X509_VERIFY_PARAM* pParam = NULL;

#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_server_method()));
#else
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_client_method()));
#endif
    ExpectTrue(SSL_CTX_use_certificate_file(ctx, svrCertFile,
        SSL_FILETYPE_PEM));
    ExpectTrue(SSL_CTX_use_PrivateKey_file(ctx, svrKeyFile, SSL_FILETYPE_PEM));
    ExpectNotNull(ssl = SSL_new(ctx));

    pParam = SSL_get0_param(ssl);

    /* we should get back host string */
    ExpectIntEQ(SSL_set1_host(ssl, host), WOLFSSL_SUCCESS);
    ExpectIntEQ(XMEMCMP(pParam->hostName, host, sizeof(host)), 0);

    /* we should get back empty string */
    ExpectIntEQ(SSL_set1_host(ssl, emptyStr), WOLFSSL_SUCCESS);
    ExpectIntEQ(XMEMCMP(pParam->hostName, emptyStr, sizeof(emptyStr)), 0);

    /* we should get back host string */
    ExpectIntEQ(SSL_set1_host(ssl, host), WOLFSSL_SUCCESS);
    ExpectIntEQ(XMEMCMP(pParam->hostName, host, sizeof(host)), 0);

    /* we should get back empty string */
    ExpectIntEQ(SSL_set1_host(ssl, NULL), WOLFSSL_SUCCESS);
    ExpectIntEQ(XMEMCMP(pParam->hostName, emptyStr, sizeof(emptyStr)), 0);

    SSL_free(ssl);
    SSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_VERIFY_PARAM_set1_ip(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_FILESYSTEM)
    unsigned char buf[16] = {0};
    WOLFSSL_X509_VERIFY_PARAM* param = NULL;

    ExpectNotNull(param = X509_VERIFY_PARAM_new());

    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip(NULL, NULL, 1), WOLFSSL_FAILURE);
    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip(param, NULL, 1), WOLFSSL_FAILURE);
    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip(NULL, buf, 1), WOLFSSL_FAILURE);
    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip(NULL, NULL, 16), WOLFSSL_FAILURE);
    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip(NULL, NULL, 4), WOLFSSL_FAILURE);
    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip(NULL, NULL, 0), WOLFSSL_FAILURE);
    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip(param, buf, 1), WOLFSSL_FAILURE);
    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip(param, NULL, 16), WOLFSSL_FAILURE);
    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip(param, NULL, 4), WOLFSSL_FAILURE);
    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip(NULL, buf, 16), WOLFSSL_FAILURE);
    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip(NULL, buf, 4), WOLFSSL_FAILURE);
    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip(NULL, buf, 0), WOLFSSL_FAILURE);

    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip(param, NULL, 0), WOLFSSL_SUCCESS);

    /* test 127.0.0.1 */
    buf[0] =0x7f; buf[1] = 0; buf[2] = 0; buf[3] = 1;
    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip(param, &buf[0], 4), SSL_SUCCESS);
    ExpectIntEQ(XSTRNCMP(param->ipasc, "127.0.0.1", sizeof(param->ipasc)), 0);

    /* test 2001:db8:3333:4444:5555:6666:7777:8888 */
    buf[0]=32;buf[1]=1;buf[2]=13;buf[3]=184;
    buf[4]=51;buf[5]=51;buf[6]=68;buf[7]=68;
    buf[8]=85;buf[9]=85;buf[10]=102;buf[11]=102;
    buf[12]=119;buf[13]=119;buf[14]=136;buf[15]=136;
    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip(param, &buf[0], 16), SSL_SUCCESS);
    ExpectIntEQ(XSTRNCMP(param->ipasc,
        "2001:db8:3333:4444:5555:6666:7777:8888", sizeof(param->ipasc)), 0);

    /* test 2001:db8:: */
    buf[0]=32;buf[1]=1;buf[2]=13;buf[3]=184;
    buf[4]=0;buf[5]=0;buf[6]=0;buf[7]=0;
    buf[8]=0;buf[9]=0;buf[10]=0;buf[11]=0;
    buf[12]=0;buf[13]=0;buf[14]=0;buf[15]=0;
    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip(param, &buf[0], 16), SSL_SUCCESS);
    ExpectIntEQ(XSTRNCMP(param->ipasc, "2001:db8::", sizeof(param->ipasc)), 0);

    /* test ::1234:5678 */
    buf[0]=0;buf[1]=0;buf[2]=0;buf[3]=0;
    buf[4]=0;buf[5]=0;buf[6]=0;buf[7]=0;
    buf[8]=0;buf[9]=0;buf[10]=0;buf[11]=0;
    buf[12]=18;buf[13]=52;buf[14]=86;buf[15]=120;
    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip(param, &buf[0], 16), SSL_SUCCESS);
    ExpectIntEQ(XSTRNCMP(param->ipasc, "::1234:5678", sizeof(param->ipasc)), 0);


    /* test 2001:db8::1234:5678 */
    buf[0]=32;buf[1]=1;buf[2]=13;buf[3]=184;
    buf[4]=0;buf[5]=0;buf[6]=0;buf[7]=0;
    buf[8]=0;buf[9]=0;buf[10]=0;buf[11]=0;
    buf[12]=18;buf[13]=52;buf[14]=86;buf[15]=120;
    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip(param, &buf[0], 16), SSL_SUCCESS);
    ExpectIntEQ(XSTRNCMP(param->ipasc, "2001:db8::1234:5678",
                                                sizeof(param->ipasc)), 0);

    /* test 2001:0db8:0001:0000:0000:0ab9:c0a8:0102*/
    /*      2001:db8:1::ab9:c0a8:102 */
    buf[0]=32;buf[1]=1;buf[2]=13;buf[3]=184;
    buf[4]=0;buf[5]=1;buf[6]=0;buf[7]=0;
    buf[8]=0;buf[9]=0;buf[10]=10;buf[11]=185;
    buf[12]=192;buf[13]=168;buf[14]=1;buf[15]=2;
    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip(param, &buf[0], 16), SSL_SUCCESS);
    ExpectIntEQ(XSTRNCMP(param->ipasc, "2001:db8:1::ab9:c0a8:102",
                                                sizeof(param->ipasc)), 0);

    XFREE(param, HEAP_HINT, DYNAMIC_TYPE_OPENSSL);
#endif /* OPENSSL_EXTRA */
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_STORE_CTX_get0_store(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA)
    X509_STORE* store = NULL;
    X509_STORE_CTX* ctx = NULL;
    X509_STORE_CTX* ctx_no_init = NULL;

    ExpectNotNull((store = X509_STORE_new()));
    ExpectNotNull(ctx = X509_STORE_CTX_new());
    ExpectNotNull(ctx_no_init = X509_STORE_CTX_new());
    ExpectIntEQ(X509_STORE_CTX_init(ctx, store, NULL, NULL), SSL_SUCCESS);

    ExpectNull(X509_STORE_CTX_get0_store(NULL));
    /* should return NULL if ctx has not bee initialized */
    ExpectNull(X509_STORE_CTX_get0_store(ctx_no_init));
    ExpectNotNull(X509_STORE_CTX_get0_store(ctx));

    wolfSSL_X509_STORE_CTX_free(ctx);
    wolfSSL_X509_STORE_CTX_free(ctx_no_init);
    X509_STORE_free(store);
#endif /* OPENSSL_EXTRA */
    return EXPECT_RESULT();
}

#if defined(OPENSSL_ALL) && !defined(NO_RSA) && !defined(NO_CERTS) && \
    !defined(NO_WOLFSSL_CLIENT) && !defined(NO_WOLFSSL_SERVER) && \
    defined(HAVE_ECC) && !defined(NO_TLS) && defined(HAVE_AESGCM)
static int test_wolfSSL_get_client_ciphers_ctx_ready(WOLFSSL_CTX* ctx)
{
    EXPECT_DECLS;
    ExpectTrue(wolfSSL_CTX_set_cipher_list(ctx, "ECDHE-RSA-AES128-GCM-SHA256"));
    return EXPECT_RESULT();
}


static int test_wolfSSL_get_client_ciphers_on_result(WOLFSSL* ssl) {
    EXPECT_DECLS;
    WOLF_STACK_OF(WOLFSSL_CIPHER)* ciphers;

    ciphers = SSL_get_client_ciphers(ssl);
    if (wolfSSL_is_server(ssl) == 0) {
        ExpectNull(ciphers);
    }
    else {
        WOLFSSL_CIPHER* current;

        /* client should have only sent over one cipher suite */
        ExpectNotNull(ciphers);
        ExpectIntEQ(sk_SSL_CIPHER_num(ciphers), 1);
        current = sk_SSL_CIPHER_value(ciphers, 0);
        ExpectNotNull(current);
    #if !defined(WOLFSSL_CIPHER_INTERNALNAME) && !defined(NO_ERROR_STRINGS) && \
        !defined(WOLFSSL_QT)
        ExpectStrEQ("TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256",
            SSL_CIPHER_get_name(current));
    #else
        ExpectStrEQ("ECDHE-RSA-AES128-GCM-SHA256",
            SSL_CIPHER_get_name(current));
    #endif
    }
    return EXPECT_RESULT();
}
#endif

static int test_wolfSSL_get_client_ciphers(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_ALL) && !defined(NO_RSA) && !defined(NO_CERTS) && \
    !defined(NO_WOLFSSL_CLIENT) && !defined(NO_WOLFSSL_SERVER) && \
    defined(HAVE_ECC) && !defined(NO_TLS) && defined(HAVE_AESGCM)
    test_ssl_cbf server_cb;
    test_ssl_cbf client_cb;

    XMEMSET(&client_cb, 0, sizeof(test_ssl_cbf));
    XMEMSET(&server_cb, 0, sizeof(test_ssl_cbf));
    client_cb.method = wolfTLSv1_2_client_method;
    server_cb.method = wolfTLSv1_2_server_method;
    client_cb.devId = testDevId;
    server_cb.devId = testDevId;
    client_cb.ctx_ready = test_wolfSSL_get_client_ciphers_ctx_ready;
    client_cb.on_result = test_wolfSSL_get_client_ciphers_on_result;
    server_cb.on_result = test_wolfSSL_get_client_ciphers_on_result;

    ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&client_cb,
        &server_cb, NULL), TEST_SUCCESS);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_set_client_CA_list(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_ALL) && !defined(NO_RSA) && !defined(NO_CERTS) && \
    !defined(NO_WOLFSSL_CLIENT) && !defined(NO_WOLFSSL_SERVER) && \
    !defined(NO_BIO) && !defined(NO_TLS)
    WOLFSSL_CTX* ctx = NULL;
    WOLFSSL* ssl = NULL;
    X509_NAME* name = NULL;
    STACK_OF(X509_NAME)* names = NULL;
    STACK_OF(X509_NAME)* ca_list = NULL;
    int names_len = 0;
    int i;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
    /* Send two X501 names in cert request */
    names = SSL_load_client_CA_file(cliCertFile);
    ExpectNotNull(names);
    ca_list = SSL_load_client_CA_file(caCertFile);
    ExpectNotNull(ca_list);
    ExpectNotNull(name = sk_X509_NAME_value(ca_list, 0));
    ExpectIntEQ(sk_X509_NAME_push(names, name), 2);
    if (EXPECT_FAIL()) {
        wolfSSL_X509_NAME_free(name);
        name = NULL;
    }
    SSL_CTX_set_client_CA_list(ctx, names);
    /* This should only free the stack structure */
    sk_X509_NAME_free(ca_list);
    ca_list = NULL;
    ExpectNotNull(ca_list = SSL_CTX_get_client_CA_list(ctx));
    ExpectIntEQ(sk_X509_NAME_num(ca_list), sk_X509_NAME_num(names));

    ExpectIntEQ(sk_X509_NAME_find(NULL, name), BAD_FUNC_ARG);
    ExpectIntEQ(sk_X509_NAME_find(names, NULL), WOLFSSL_FATAL_ERROR);
    ExpectIntGT((names_len = sk_X509_NAME_num(names)), 0);
    for (i = 0; i < names_len; i++) {
        ExpectNotNull(name = sk_X509_NAME_value(names, i));
        ExpectIntEQ(sk_X509_NAME_find(names, name), i);
    }

    /* Needed to be able to create ssl object */
    ExpectTrue(SSL_CTX_use_certificate_file(ctx, svrCertFile,
        SSL_FILETYPE_PEM));
    ExpectTrue(SSL_CTX_use_PrivateKey_file(ctx, svrKeyFile, SSL_FILETYPE_PEM));
    ExpectNotNull(ssl = wolfSSL_new(ctx));
    /* load again as old names are responsibility of ctx to free*/
    names = SSL_load_client_CA_file(cliCertFile);
    ExpectNotNull(names);
    SSL_set_client_CA_list(ssl, names);
    ExpectNotNull(ca_list = SSL_get_client_CA_list(ssl));
    ExpectIntEQ(sk_X509_NAME_num(ca_list), sk_X509_NAME_num(names));

    ExpectIntGT((names_len = sk_X509_NAME_num(names)), 0);
    for (i = 0; i < names_len; i++) {
        ExpectNotNull(name = sk_X509_NAME_value(names, i));
        ExpectIntEQ(sk_X509_NAME_find(names, name), i);
    }

#if !defined(SINGLE_THREADED) && defined(SESSION_CERTS)
    {
        tcp_ready ready;
        func_args server_args;
        callback_functions server_cb;
        THREAD_TYPE serverThread;
        WOLFSSL* ssl_client = NULL;
        WOLFSSL_CTX* ctx_client = NULL;
        SOCKET_T sockfd = 0;

        /* wolfSSL_get_client_CA_list() with handshake */

        StartTCP();
        InitTcpReady(&ready);

        XMEMSET(&server_args, 0, sizeof(func_args));
        XMEMSET(&server_cb, 0, sizeof(callback_functions));

        server_args.signal    = &ready;
        server_args.callbacks = &server_cb;

        /* we are responsible for free'ing WOLFSSL_CTX */
        server_cb.ctx = ctx;
        server_cb.isSharedCtx = 1;

        ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_load_verify_locations(ctx,
            cliCertFile, 0));

        start_thread(test_server_nofail, &server_args, &serverThread);
        wait_tcp_ready(&server_args);

        tcp_connect(&sockfd, wolfSSLIP, server_args.signal->port, 0, 0, NULL);
        ExpectNotNull(ctx_client =
            wolfSSL_CTX_new(wolfTLSv1_2_client_method()));
        ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_load_verify_locations(
            ctx_client, caCertFile, 0));
        ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_use_certificate_file(
            ctx_client, cliCertFile, SSL_FILETYPE_PEM));
        ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_use_PrivateKey_file(
            ctx_client, cliKeyFile, SSL_FILETYPE_PEM));

        ExpectNotNull(ssl_client = wolfSSL_new(ctx_client));
        ExpectIntEQ(wolfSSL_set_fd(ssl_client, sockfd), WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_connect(ssl_client), WOLFSSL_SUCCESS);

        ExpectNotNull(ca_list = SSL_get_client_CA_list(ssl_client));
        /* We are expecting two cert names to be sent */
        ExpectIntEQ(sk_X509_NAME_num(ca_list), 2);

        ExpectNotNull(names = SSL_CTX_get_client_CA_list(ctx));
        for (i=0; i<sk_X509_NAME_num(ca_list); i++) {
            ExpectNotNull(name = sk_X509_NAME_value(ca_list, i));
            ExpectIntGE(sk_X509_NAME_find(names, name), 0);
        }

        wolfSSL_shutdown(ssl_client);
        wolfSSL_free(ssl_client);
        wolfSSL_CTX_free(ctx_client);

        CloseSocket(sockfd);

        join_thread(serverThread);
        FreeTcpReady(&ready);
    }
#endif

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif /* OPENSSL_EXTRA && !NO_RSA && !NO_CERTS && !NO_WOLFSSL_CLIENT &&
        * !NO_BIO */
    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_add_client_CA(void)
{
    EXPECT_DECLS;
#if !defined(WOLFSSL_NO_CA_NAMES) && defined(OPENSSL_EXTRA) && \
    !defined(NO_RSA) && !defined(NO_CERTS) && \
    !defined(NO_TLS) && !defined(NO_WOLFSSL_CLIENT) && !defined(NO_FILESYSTEM)
    WOLFSSL_CTX* ctx = NULL;
    WOLFSSL_X509* x509 = NULL;
    WOLFSSL_X509* x509_a = NULL;
    STACK_OF(X509_NAME)* ca_list = NULL;

    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_client_method()));
    /* Add client cert */
    ExpectNotNull(x509 = X509_load_certificate_file(cliCertFile,
        SSL_FILETYPE_PEM));
    ExpectIntEQ(SSL_CTX_add_client_CA(ctx, x509), SSL_SUCCESS);
    ExpectNotNull(ca_list = SSL_CTX_get_client_CA_list(ctx));
    /* Add another client cert */
    ExpectNotNull(x509_a = X509_load_certificate_file(cliCertFile,
        SSL_FILETYPE_PEM));
    ExpectIntEQ(SSL_CTX_add_client_CA(ctx, x509_a), SSL_SUCCESS);

    /* test for incorrect parameter */
    ExpectIntEQ(SSL_CTX_add_client_CA(NULL, x509), 0);
    ExpectIntEQ(SSL_CTX_add_client_CA(ctx, NULL), 0);
    ExpectIntEQ(SSL_CTX_add_client_CA(NULL, NULL), 0);

    X509_free(x509);
    X509_free(x509_a);
    SSL_CTX_free(ctx);
#endif /* OPENSSL_EXTRA  && !NO_RSA && !NO_CERTS && !NO_WOLFSSL_CLIENT */
    return EXPECT_RESULT();
}
#if defined(WOLFSSL_TLS13) && defined(HAVE_ECH) && \
    defined(HAVE_IO_TESTS_DEPENDENCIES)
static THREAD_RETURN WOLFSSL_THREAD server_task_ech(void* args)
{
    callback_functions* callbacks = ((func_args*)args)->callbacks;
    WOLFSSL_CTX* ctx       = callbacks->ctx;
    WOLFSSL*  ssl   = NULL;
    SOCKET_T  sfd   = 0;
    SOCKET_T  cfd   = 0;
    word16    port;
    char      input[1024];
    int       idx;
    int       ret, err = 0;
    const char* privateName = "ech-private-name.com";
    int         privateNameLen = (int)XSTRLEN(privateName);

    ((func_args*)args)->return_code = TEST_FAIL;
    port = ((func_args*)args)->signal->port;

    AssertIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CTX_load_verify_locations(ctx, cliCertFile, 0));

    AssertIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CTX_use_certificate_file(ctx, svrCertFile,
            WOLFSSL_FILETYPE_PEM));

    AssertIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CTX_use_PrivateKey_file(ctx, svrKeyFile,
                 WOLFSSL_FILETYPE_PEM));

    if (callbacks->ctx_ready)
        callbacks->ctx_ready(ctx);

    ssl = wolfSSL_new(ctx);

    /* set the sni for the server */
    wolfSSL_UseSNI(ssl, WOLFSSL_SNI_HOST_NAME, privateName, privateNameLen);

    tcp_accept(&sfd, &cfd, (func_args*)args, port, 0, 0, 0, 0, 1, NULL, NULL);
    CloseSocket(sfd);
    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_set_fd(ssl, cfd));

    if (callbacks->ssl_ready)
        callbacks->ssl_ready(ssl);

    do {
        err = 0; /* Reset error */
        ret = wolfSSL_accept(ssl);
        if (ret != WOLFSSL_SUCCESS) {
            err = wolfSSL_get_error(ssl, 0);
        }
    } while (ret != WOLFSSL_SUCCESS && err == WC_NO_ERR_TRACE(WC_PENDING_E));

    if (ret != WOLFSSL_SUCCESS) {
        char buff[WOLFSSL_MAX_ERROR_SZ];
        fprintf(stderr, "error = %d, %s\n", err, wolfSSL_ERR_error_string(err, buff));
    }
    else {
        if (0 < (idx = wolfSSL_read(ssl, input, sizeof(input)-1))) {
            input[idx] = 0;
           fprintf(stderr, "Client message: %s\n", input);
        }

        AssertIntEQ(privateNameLen, wolfSSL_write(ssl, privateName,
            privateNameLen));
        ((func_args*)args)->return_code = TEST_SUCCESS;
    }

    if (callbacks->on_result)
        callbacks->on_result(ssl);

    wolfSSL_shutdown(ssl);
    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
    CloseSocket(cfd);

#ifdef FP_ECC
    wc_ecc_fp_free();
#endif

    WOLFSSL_RETURN_FROM_THREAD(0);
}
#endif /* HAVE_ECH && WOLFSSL_TLS13 */

#if defined(OPENSSL_EXTRA) && defined(HAVE_SECRET_CALLBACK)
static void keyLog_callback(const WOLFSSL* ssl, const char* line)
{
    XFILE fp;
    const byte lf = '\n';

    AssertNotNull(ssl);
    AssertNotNull(line);

    fp = XFOPEN("./MyKeyLog.txt", "a");
    XFWRITE(line, 1, XSTRLEN(line), fp);
    XFWRITE((void*)&lf, 1, 1, fp);
    XFFLUSH(fp);
    XFCLOSE(fp);
}
#endif /* OPENSSL_EXTRA && HAVE_SECRET_CALLBACK */
static int test_wolfSSL_CTX_set_keylog_callback(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && defined(HAVE_SECRET_CALLBACK) && \
                              !defined(NO_WOLFSSL_CLIENT)
    SSL_CTX* ctx = NULL;

    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_client_method()));
    SSL_CTX_set_keylog_callback(ctx, keyLog_callback );
    SSL_CTX_free(ctx);
    SSL_CTX_set_keylog_callback(NULL, NULL);
#endif /* OPENSSL_EXTRA && HAVE_SECRET_CALLBACK && !NO_WOLFSSL_CLIENT */
    return EXPECT_RESULT();
}
static int test_wolfSSL_CTX_get_keylog_callback(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && defined(HAVE_SECRET_CALLBACK) && \
                                !defined(NO_WOLFSSL_CLIENT)
    SSL_CTX* ctx = NULL;

    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_client_method()));
    ExpectPtrEq(SSL_CTX_get_keylog_callback(ctx),NULL);
    SSL_CTX_set_keylog_callback(ctx, keyLog_callback );
    ExpectPtrEq(SSL_CTX_get_keylog_callback(ctx),keyLog_callback);
    SSL_CTX_set_keylog_callback(ctx, NULL );
    ExpectPtrEq(SSL_CTX_get_keylog_callback(ctx),NULL);
    SSL_CTX_free(ctx);
#endif /* OPENSSL_EXTRA && HAVE_SECRET_CALLBACK && !NO_WOLFSSL_CLIENT */
    return EXPECT_RESULT();
}

#if defined(OPENSSL_EXTRA) && defined(HAVE_SECRET_CALLBACK)
static int test_wolfSSL_Tls12_Key_Logging_client_ctx_ready(WOLFSSL_CTX* ctx)
{
    /* set keylog callback */
    wolfSSL_CTX_set_keylog_callback(ctx, keyLog_callback);
    return TEST_SUCCESS;
}
#endif

static int test_wolfSSL_Tls12_Key_Logging_test(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && defined(HAVE_SECRET_CALLBACK)
    /* This test is intended for checking whether keylog callback is called
     * in client during TLS handshake between the client and a server.
     */
    test_ssl_cbf server_cbf;
    test_ssl_cbf client_cbf;
    XFILE fp = XBADFILE;
    char  buff[500];
    int   found = 0;

    XMEMSET(&server_cbf, 0, sizeof(test_ssl_cbf));
    XMEMSET(&client_cbf, 0, sizeof(test_ssl_cbf));
    server_cbf.method     = wolfTLSv1_2_server_method;
    client_cbf.ctx_ready = &test_wolfSSL_Tls12_Key_Logging_client_ctx_ready;

    /* clean up keylog file */
    ExpectTrue((fp = XFOPEN("./MyKeyLog.txt", "w")) != XBADFILE);
    if (fp != XBADFILE) {
        XFFLUSH(fp);
        XFCLOSE(fp);
        fp = XBADFILE;
    }

    ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&client_cbf,
        &server_cbf, NULL), TEST_SUCCESS);

    /* check if the keylog file exists */
    ExpectTrue((fp = XFOPEN("./MyKeyLog.txt", "rb")) != XBADFILE);
    XFFLUSH(fp); /* Just to make sure any buffers get flushed */

    XMEMSET(buff, 0, sizeof(buff));
    while (EXPECT_SUCCESS() && XFGETS(buff, (int)sizeof(buff), fp) != NULL) {
        if (0 == strncmp(buff,"CLIENT_RANDOM ", sizeof("CLIENT_RANDOM ")-1)) {
            found = 1;
            break;
        }
    }
    if (fp != XBADFILE) {
        XFCLOSE(fp);
    }
    /* a log starting with "CLIENT_RANDOM " should exit in the file */
    ExpectIntEQ(found, 1);
    /* clean up */
    ExpectIntEQ(rem_file("./MyKeyLog.txt"), 0);
#endif /* OPENSSL_EXTRA && HAVE_SECRET_CALLBACK */
    return EXPECT_RESULT();
}

#if defined(WOLFSSL_TLS13) && defined(OPENSSL_EXTRA) && \
    defined(HAVE_SECRET_CALLBACK)
static int test_wolfSSL_Tls13_Key_Logging_client_ctx_ready(WOLFSSL_CTX* ctx)
{
    /* set keylog callback */
    wolfSSL_CTX_set_keylog_callback(ctx, keyLog_callback);
    return TEST_SUCCESS;
}
#endif

static int test_wolfSSL_Tls13_Key_Logging_test(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_TLS13) && defined(OPENSSL_EXTRA) && \
    defined(HAVE_SECRET_CALLBACK)
/* This test is intended for checking whether keylog callback is called
 * in client during TLS handshake between the client and a server.
 */
    test_ssl_cbf server_cbf;
    test_ssl_cbf client_cbf;
    XFILE fp = XBADFILE;

    XMEMSET(&server_cbf, 0, sizeof(test_ssl_cbf));
    XMEMSET(&client_cbf, 0, sizeof(test_ssl_cbf));
    server_cbf.method    = wolfTLSv1_3_server_method;  /* TLS1.3 */
    client_cbf.ctx_ready = &test_wolfSSL_Tls13_Key_Logging_client_ctx_ready;

    /* clean up keylog file */
    ExpectTrue((fp = XFOPEN("./MyKeyLog.txt", "w")) != XBADFILE);
    if (fp != XBADFILE) {
        XFCLOSE(fp);
        fp = XBADFILE;
    }

    ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&client_cbf,
        &server_cbf, NULL), TEST_SUCCESS);

    /* check if the keylog file exists */
    {
        char buff[300] = {0};
        int  found[4]   = {0};
        int  numfnd = 0;
        int  i;

        ExpectTrue((fp = XFOPEN("./MyKeyLog.txt", "rb")) != XBADFILE);

        while (EXPECT_SUCCESS() &&
                XFGETS(buff, (int)sizeof(buff), fp) != NULL) {
            if (0 == strncmp(buff, "CLIENT_HANDSHAKE_TRAFFIC_SECRET ",
                    sizeof("CLIENT_HANDSHAKE_TRAFFIC_SECRET ")-1)) {
                found[0] = 1;
                continue;
            }
            else if (0 == strncmp(buff, "SERVER_HANDSHAKE_TRAFFIC_SECRET ",
                    sizeof("SERVER_HANDSHAKE_TRAFFIC_SECRET ")-1)) {
                found[1] = 1;
                continue;
            }
            else if (0 == strncmp(buff, "CLIENT_TRAFFIC_SECRET_0 ",
                    sizeof("CLIENT_TRAFFIC_SECRET_0 ")-1)) {
                found[2] = 1;
                continue;
            }
            else if (0 == strncmp(buff, "SERVER_TRAFFIC_SECRET_0 ",
                    sizeof("SERVER_TRAFFIC_SECRET_0 ")-1)) {
                found[3] = 1;
                continue;
            }
        }
        if (fp != XBADFILE)
            XFCLOSE(fp);
        for (i = 0; i < 4; i++) {
            if (found[i] != 0)
                numfnd++;
        }
        ExpectIntEQ(numfnd, 4);
    }
#endif /* OPENSSL_EXTRA && HAVE_SECRET_CALLBACK && WOLFSSL_TLS13 */
    return EXPECT_RESULT();
}
#if defined(WOLFSSL_TLS13) && defined(HAVE_ECH) && \
    defined(HAVE_IO_TESTS_DEPENDENCIES)
static int test_wolfSSL_Tls13_ECH_params(void)
{
    EXPECT_DECLS;
#if !defined(NO_WOLFSSL_CLIENT)
    word32 outputLen = 0;
    byte testBuf[72];
    WOLFSSL_CTX *ctx = wolfSSL_CTX_new(wolfTLSv1_3_client_method());
    WOLFSSL     *ssl = wolfSSL_new(ctx);

    ExpectNotNull(ctx);
    ExpectNotNull(ssl);

    /* invalid ctx */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_GenerateEchConfig(NULL,
        "ech-public-name.com", 0, 0, 0));
    /* invalid public name */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_GenerateEchConfig(ctx, NULL, 0,
        0, 0));
    /* invalid algorithms */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_GenerateEchConfig(ctx,
        "ech-public-name.com", 1000, 1000, 1000));

    /* invalid ctx */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_SetEchConfigsBase64(NULL,
        (char*)testBuf, sizeof(testBuf)));
    /* invalid base64 configs */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_SetEchConfigsBase64(ctx,
        NULL, sizeof(testBuf)));
    /* invalid length */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_SetEchConfigsBase64(ctx,
        (char*)testBuf, 0));

    /* invalid ctx */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_SetEchConfigs(NULL,
        testBuf, sizeof(testBuf)));
    /* invalid configs */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_SetEchConfigs(ctx,
        NULL, sizeof(testBuf)));
    /* invalid length */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_SetEchConfigs(ctx,
        testBuf, 0));

    /* invalid ctx */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_GetEchConfigs(NULL, NULL,
        &outputLen));
    /* invalid output len */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_GetEchConfigs(ctx, NULL, NULL));

    /* invalid ssl */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_SetEchConfigsBase64(NULL,
        (char*)testBuf, sizeof(testBuf)));
    /* invalid configs64 */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_SetEchConfigsBase64(ssl, NULL,
        sizeof(testBuf)));
    /* invalid size */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_SetEchConfigsBase64(ssl,
        (char*)testBuf, 0));

    /* invalid ssl */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_SetEchConfigs(NULL, testBuf,
        sizeof(testBuf)));
    /* invalid configs */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_SetEchConfigs(ssl, NULL,
        sizeof(testBuf)));
    /* invalid size */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_SetEchConfigs(ssl, testBuf, 0));

    /* invalid ssl */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_GetEchConfigs(NULL, NULL, &outputLen));
    /* invalid size */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_GetEchConfigs(ssl, NULL, NULL));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif /* !NO_WOLFSSL_CLIENT */

    return EXPECT_RESULT();
}

static int test_wolfSSL_Tls13_ECH_ex(int hrr)
{
    EXPECT_DECLS;
    tcp_ready ready;
    func_args client_args;
    func_args server_args;
    THREAD_TYPE serverThread;
    callback_functions server_cbf;
    callback_functions client_cbf;
    SOCKET_T sockfd = 0;
    WOLFSSL_CTX* ctx = NULL;
    WOLFSSL*     ssl = NULL;
    const char* publicName = "ech-public-name.com";
    const char* privateName = "ech-private-name.com";
    int privateNameLen = 20;
    char reply[1024];
    int replyLen = 0;
    byte rawEchConfig[128];
    word32 rawEchConfigLen = sizeof(rawEchConfig);

    InitTcpReady(&ready);
    ready.port = 22222;

    XMEMSET(&client_args, 0, sizeof(func_args));
    XMEMSET(&server_args, 0, sizeof(func_args));
    XMEMSET(&server_cbf, 0, sizeof(callback_functions));
    XMEMSET(&client_cbf, 0, sizeof(callback_functions));
    server_cbf.method     = wolfTLSv1_3_server_method;  /* TLS1.3 */

    /* create the server context here so we can get the ech config */
    ExpectNotNull(server_cbf.ctx =
        wolfSSL_CTX_new(wolfTLSv1_3_server_method()));

    /* generate ech config */
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_GenerateEchConfig(server_cbf.ctx,
        publicName, 0, 0, 0));

    /* get the config for the client to use */
    ExpectIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CTX_GetEchConfigs(server_cbf.ctx, rawEchConfig,
        &rawEchConfigLen));

    server_args.callbacks = &server_cbf;
    server_args.signal    = &ready;

    /* start server task */
    start_thread(server_task_ech, &server_args, &serverThread);
    wait_tcp_ready(&server_args);

    /* run as a TLS1.3 client */
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_3_client_method()));
    ExpectIntEQ(WOLFSSL_SUCCESS,
            wolfSSL_CTX_load_verify_locations(ctx, caCertFile, 0));
    ExpectIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CTX_use_certificate_file(ctx, cliCertFile, SSL_FILETYPE_PEM));
    ExpectIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CTX_use_PrivateKey_file(ctx, cliKeyFile, SSL_FILETYPE_PEM));

    tcp_connect(&sockfd, wolfSSLIP, server_args.signal->port, 0, 0, NULL);

    /* get connected the server task */
    ExpectNotNull(ssl = wolfSSL_new(ctx));

    /* set the ech configs for the client */
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_SetEchConfigs(ssl, rawEchConfig,
        rawEchConfigLen));

    /* set the sni for the client */
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseSNI(ssl, WOLFSSL_SNI_HOST_NAME,
        privateName, privateNameLen));

    /* force hello retry request */
    if (hrr)
        ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_NoKeyShares(ssl));

    /* connect like normal */
    ExpectIntEQ(wolfSSL_set_fd(ssl, sockfd), WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_connect(ssl), WOLFSSL_SUCCESS);
    ExpectIntEQ(ssl->options.echAccepted, 1);
    ExpectIntEQ(wolfSSL_write(ssl, privateName, privateNameLen),
        privateNameLen);
    ExpectIntGT((replyLen = wolfSSL_read(ssl, reply, sizeof(reply))), 0);
    /* add th null terminator for string compare */
    reply[replyLen] = 0;
    /* check that the server replied with the private name */
    ExpectStrEQ(privateName, reply);
    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);

    CloseSocket(sockfd);

    join_thread(serverThread);

    FreeTcpReady(&ready);

    return EXPECT_RESULT();
}

static int test_wolfSSL_Tls13_ECH(void)
{
    return test_wolfSSL_Tls13_ECH_ex(0);
}

static int test_wolfSSL_Tls13_ECH_HRR(void)
{
    return test_wolfSSL_Tls13_ECH_ex(1);
}
#endif /* HAVE_ECH && WOLFSSL_TLS13 */

#if defined(HAVE_IO_TESTS_DEPENDENCIES) && \
defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && \
    defined(WOLFSSL_TLS13) && defined(WOLFSSL_POST_HANDSHAKE_AUTH)
static int post_auth_version_cb(WOLFSSL* ssl)
{
    EXPECT_DECLS;
    /* do handshake and then test version error */
    ExpectIntEQ(wolfSSL_accept(ssl), WOLFSSL_SUCCESS);
    ExpectStrEQ("TLSv1.2", wolfSSL_get_version(ssl));
    return EXPECT_RESULT();
}

static int post_auth_version_client_cb(WOLFSSL* ssl)
{
    EXPECT_DECLS;
    /* do handshake and then test version error */
    ExpectIntEQ(wolfSSL_connect(ssl), WOLFSSL_SUCCESS);
    ExpectStrEQ("TLSv1.2", wolfSSL_get_version(ssl));
    ExpectIntEQ(wolfSSL_verify_client_post_handshake(ssl), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
#if defined(OPENSSL_ALL) && !defined(NO_ERROR_QUEUE)
    /* check was added to error queue */
    ExpectIntEQ(wolfSSL_ERR_get_error(), -WC_NO_ERR_TRACE(UNSUPPORTED_PROTO_VERSION));

    /* check the string matches expected string */
    #ifndef NO_ERROR_STRINGS
    ExpectStrEQ(wolfSSL_ERR_error_string(-WC_NO_ERR_TRACE(UNSUPPORTED_PROTO_VERSION), NULL),
            "WRONG_SSL_VERSION");
    #endif
#endif
    return EXPECT_RESULT();
}

static int post_auth_cb(WOLFSSL* ssl)
{
    EXPECT_DECLS;
    WOLFSSL_X509* x509 = NULL;
    /* do handshake and then test version error */
    ExpectIntEQ(wolfSSL_accept(ssl), WOLFSSL_SUCCESS);
    ExpectStrEQ("TLSv1.3", wolfSSL_get_version(ssl));
    ExpectNull(x509 = wolfSSL_get_peer_certificate(ssl));
    wolfSSL_X509_free(x509);
    ExpectIntEQ(wolfSSL_verify_client_post_handshake(ssl), WOLFSSL_SUCCESS);
    return EXPECT_RESULT();
}

static int set_post_auth_cb(WOLFSSL* ssl)
{
    if (!wolfSSL_is_server(ssl)) {
        EXPECT_DECLS;
        ExpectIntEQ(wolfSSL_allow_post_handshake_auth(ssl), 0);
        return EXPECT_RESULT();
    }
    wolfSSL_set_verify(ssl, WOLFSSL_VERIFY_POST_HANDSHAKE, NULL);
    return TEST_SUCCESS;
}
#endif

static int test_wolfSSL_Tls13_postauth(void)
{
    EXPECT_DECLS;
#if defined(HAVE_IO_TESTS_DEPENDENCIES) && \
    defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && \
    defined(WOLFSSL_TLS13) && defined(WOLFSSL_POST_HANDSHAKE_AUTH)
    test_ssl_cbf server_cbf;
    test_ssl_cbf client_cbf;

    /* test version failure doing post auth with TLS 1.2 connection */
    XMEMSET(&server_cbf, 0, sizeof(server_cbf));
    XMEMSET(&client_cbf, 0, sizeof(client_cbf));
    server_cbf.method = wolfTLSv1_2_server_method;
    server_cbf.ssl_ready = set_post_auth_cb;
    server_cbf.on_result = post_auth_version_cb;
    client_cbf.ssl_ready = set_post_auth_cb;
    client_cbf.on_result = post_auth_version_client_cb;

    ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&client_cbf,
        &server_cbf, NULL), TEST_SUCCESS);

    /* tests on post auth with TLS 1.3 */
    XMEMSET(&server_cbf, 0, sizeof(server_cbf));
    XMEMSET(&client_cbf, 0, sizeof(client_cbf));
    server_cbf.method = wolfTLSv1_3_server_method;
    server_cbf.ssl_ready = set_post_auth_cb;
    client_cbf.ssl_ready = set_post_auth_cb;
    server_cbf.on_result = post_auth_cb;
    client_cbf.on_result = NULL;

    ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&client_cbf,
        &server_cbf, NULL), TEST_SUCCESS);
#endif
    return EXPECT_RESULT();
}


static int test_wolfSSL_X509_NID(void)
{
    EXPECT_DECLS;
#if (defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)) && \
    !defined(NO_RSA) && defined(USE_CERT_BUFFERS_2048) && !defined(NO_ASN)
    int   sigType;
    int   nameSz = 0;

    X509*  cert = NULL;
    EVP_PKEY*  pubKeyTmp = NULL;
    X509_NAME* name = NULL;

    char commonName[80];
    char countryName[80];
    char localityName[80];
    char stateName[80];
    char orgName[80];
    char orgUnit[80];

    /* ------ PARSE ORIGINAL SELF-SIGNED CERTIFICATE ------ */

    /* convert cert from DER to internal WOLFSSL_X509 struct */
    ExpectNotNull(cert = wolfSSL_X509_d2i_ex(&cert, client_cert_der_2048,
            sizeof_client_cert_der_2048, HEAP_HINT));

    /* ------ EXTRACT CERTIFICATE ELEMENTS ------ */

    /* extract PUBLIC KEY from cert */
    ExpectNotNull(pubKeyTmp = X509_get_pubkey(cert));

    /* extract signatureType */
    ExpectIntEQ(wolfSSL_X509_get_signature_type(NULL), 0);
    ExpectIntNE((sigType = wolfSSL_X509_get_signature_type(cert)), 0);

    /* extract subjectName info */
    ExpectNotNull(name = X509_get_subject_name(cert));
    ExpectIntEQ(X509_NAME_get_text_by_NID(name, -1, NULL, 0), -1);
    ExpectIntEQ(X509_NAME_get_text_by_NID(NULL, NID_commonName, NULL, 0), -1);
    ExpectIntEQ(X509_NAME_get_text_by_NID(name, NID_commonName,
        commonName, -2), 0);
    ExpectIntGT((nameSz = X509_NAME_get_text_by_NID(name, NID_commonName,
        NULL, 0)), 0);
    ExpectIntEQ(nameSz, 15);
    ExpectIntGT((nameSz = X509_NAME_get_text_by_NID(name, NID_commonName,
        commonName, sizeof(commonName))), 0);
    ExpectIntEQ(nameSz, 15);
    ExpectIntEQ(XMEMCMP(commonName, "www.wolfssl.com", nameSz), 0);
    ExpectIntGT((nameSz = X509_NAME_get_text_by_NID(name, NID_commonName,
        commonName, 9)), 0);
    ExpectIntEQ(nameSz, 8);
    ExpectIntEQ(XMEMCMP(commonName, "www.wolf", nameSz), 0);

    ExpectIntGT((nameSz = X509_NAME_get_text_by_NID(name, NID_countryName,
        countryName, sizeof(countryName))), 0);
    ExpectIntEQ(XMEMCMP(countryName, "US", nameSz), 0);

    ExpectIntGT((nameSz = X509_NAME_get_text_by_NID(name, NID_localityName,
        localityName, sizeof(localityName))), 0);
    ExpectIntEQ(XMEMCMP(localityName, "Bozeman", nameSz), 0);

    ExpectIntGT((nameSz = X509_NAME_get_text_by_NID(name,
        NID_stateOrProvinceName, stateName, sizeof(stateName))), 0);
    ExpectIntEQ(XMEMCMP(stateName, "Montana", nameSz), 0);

    ExpectIntGT((nameSz = X509_NAME_get_text_by_NID(name, NID_organizationName,
        orgName, sizeof(orgName))), 0);
    ExpectIntEQ(XMEMCMP(orgName, "wolfSSL_2048", nameSz), 0);

    ExpectIntGT((nameSz = X509_NAME_get_text_by_NID(name,
        NID_organizationalUnitName, orgUnit, sizeof(orgUnit))), 0);
    ExpectIntEQ(XMEMCMP(orgUnit, "Programming-2048", nameSz), 0);

    EVP_PKEY_free(pubKeyTmp);
    X509_free(cert);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_set_srp_username(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && defined(WOLFCRYPT_HAVE_SRP) && \
    !defined(NO_SHA256) && !defined(WC_NO_RNG) && !defined(NO_TLS) && \
    !defined(NO_WOLFSSL_CLIENT)
    WOLFSSL_CTX* ctx = NULL;
    WOLFSSL* ssl = NULL;
    const char *username = "TESTUSER";
    const char *password = "TESTPASSWORD";

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
    ExpectIntEQ(wolfSSL_CTX_set_srp_username(ctx, (char *)username),
         SSL_SUCCESS);
    wolfSSL_CTX_free(ctx);
    ctx = NULL;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
    ExpectIntEQ(wolfSSL_CTX_set_srp_password(ctx, (char *)password),
        SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_CTX_set_srp_username(ctx, (char *)username),
        SSL_SUCCESS);

    ExpectNotNull(ssl = SSL_new(ctx));
    ExpectNotNull(SSL_get_srp_username(ssl));
    ExpectStrEQ(SSL_get_srp_username(ssl), username);

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif /* OPENSSL_EXTRA && WOLFCRYPT_HAVE_SRP */
       /* && !NO_SHA256 && !WC_NO_RNG && !NO_WOLFSSL_CLIENT */
    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_set_srp_password(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && defined(WOLFCRYPT_HAVE_SRP) && \
    !defined(NO_SHA256) && !defined(WC_NO_RNG) && !defined(NO_TLS) && \
    !defined(NO_WOLFSSL_CLIENT)
    WOLFSSL_CTX* ctx = NULL;
    const char *username = "TESTUSER";
    const char *password = "TESTPASSWORD";

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
    ExpectIntEQ(wolfSSL_CTX_set_srp_password(ctx, (char *)password),
        SSL_SUCCESS);
    wolfSSL_CTX_free(ctx);
    ctx = NULL;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
    ExpectIntEQ(wolfSSL_CTX_set_srp_username(ctx, (char *)username),
        SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_CTX_set_srp_password(ctx, (char *)password),
        SSL_SUCCESS);
    wolfSSL_CTX_free(ctx);
#endif /* OPENSSL_EXTRA && WOLFCRYPT_HAVE_SRP */
       /* && !NO_SHA256 && !WC_NO_RNG && !NO_WOLFSSL_CLIENT */
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_STORE(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_RSA) && !defined(NO_TLS) && \
    !defined(NO_FILESYSTEM)
    X509_STORE *store = NULL;

#ifdef HAVE_CRL
    X509_STORE_CTX *storeCtx = NULL;
    X509 *ca = NULL;
    X509 *cert = NULL;
    const char srvCert[] = "./certs/server-revoked-cert.pem";
    const char caCert[] = "./certs/ca-cert.pem";
#ifndef WOLFSSL_CRL_ALLOW_MISSING_CDP
    X509_CRL *crl = NULL;
    const char crlPem[] = "./certs/crl/crl.revoked";
    XFILE fp = XBADFILE;
#endif /* !WOLFSSL_CRL_ALLOW_MISSING_CDP */

    ExpectNotNull(store = (X509_STORE *)X509_STORE_new());
    ExpectNotNull((ca = wolfSSL_X509_load_certificate_file(caCert,
                           SSL_FILETYPE_PEM)));
    ExpectIntEQ(X509_STORE_add_cert(store, ca), SSL_SUCCESS);
    ExpectNotNull((cert = wolfSSL_X509_load_certificate_file(srvCert,
                    SSL_FILETYPE_PEM)));
    ExpectNotNull((storeCtx = X509_STORE_CTX_new()));
    ExpectIntEQ(X509_STORE_CTX_init(storeCtx, store, cert, NULL), SSL_SUCCESS);
    ExpectIntEQ(X509_verify_cert(storeCtx), SSL_SUCCESS);
    X509_STORE_free(store);
    store = NULL;
    X509_STORE_CTX_free(storeCtx);
    storeCtx = NULL;
    X509_free(cert);
    cert = NULL;
    X509_free(ca);
    ca = NULL;

#ifndef WOLFSSL_CRL_ALLOW_MISSING_CDP
    /* should fail to verify now after adding in CRL */
    ExpectNotNull(store = (X509_STORE *)X509_STORE_new());
    ExpectNotNull((ca = wolfSSL_X509_load_certificate_file(caCert,
                           SSL_FILETYPE_PEM)));
    ExpectIntEQ(X509_STORE_add_cert(store, ca), SSL_SUCCESS);
    ExpectTrue((fp = XFOPEN(crlPem, "rb")) != XBADFILE);
    ExpectNotNull(crl = (X509_CRL *)PEM_read_X509_CRL(fp, (X509_CRL **)NULL,
                NULL, NULL));
    if (fp != XBADFILE)
        XFCLOSE(fp);
    ExpectIntEQ(X509_STORE_add_crl(store, crl), SSL_SUCCESS);
    ExpectIntEQ(X509_STORE_set_flags(store, X509_V_FLAG_CRL_CHECK),SSL_SUCCESS);
    ExpectNotNull((storeCtx = X509_STORE_CTX_new()));
    ExpectNotNull((cert = wolfSSL_X509_load_certificate_file(srvCert,
                    SSL_FILETYPE_PEM)));
    ExpectIntEQ(X509_STORE_CTX_init(storeCtx, store, cert, NULL), SSL_SUCCESS);
    ExpectIntNE(X509_verify_cert(storeCtx), SSL_SUCCESS);
    ExpectIntEQ(X509_STORE_CTX_get_error(storeCtx),
                WOLFSSL_X509_V_ERR_CERT_REVOKED);
    X509_CRL_free(crl);
    crl = NULL;
    X509_STORE_free(store);
    store = NULL;
    X509_STORE_CTX_free(storeCtx);
    storeCtx = NULL;
    X509_free(cert);
    cert = NULL;
    X509_free(ca);
    ca = NULL;
#endif /* !WOLFSSL_CRL_ALLOW_MISSING_CDP */
#endif /* HAVE_CRL */



#if !defined(WOLFCRYPT_ONLY) && !defined(NO_FILESYSTEM)
    {
    #if !defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)
        SSL_CTX* ctx = NULL;
        SSL* ssl = NULL;
        int i;
        for (i = 0; i < 2; i++) {
        #ifndef NO_WOLFSSL_SERVER
            ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_server_method()));
        #else
            ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_client_method()));
        #endif
            ExpectNotNull(store = (X509_STORE *)X509_STORE_new());
            SSL_CTX_set_cert_store(ctx, store);
            ExpectNotNull(store = (X509_STORE *)X509_STORE_new());
            SSL_CTX_set_cert_store(ctx, store);
            ExpectNotNull(store = (X509_STORE *)X509_STORE_new());
            ExpectIntEQ(SSL_CTX_use_certificate_file(ctx, svrCertFile,
                    SSL_FILETYPE_PEM), SSL_SUCCESS);
            ExpectIntEQ(SSL_CTX_use_PrivateKey_file(ctx, svrKeyFile,
                    SSL_FILETYPE_PEM), SSL_SUCCESS);
            ExpectNotNull(ssl = SSL_new(ctx));
            if (i == 0) {
                ExpectIntEQ(SSL_set0_verify_cert_store(ssl, store),
                    SSL_SUCCESS);
            }
            else {
                ExpectIntEQ(SSL_set1_verify_cert_store(ssl, store), SSL_SUCCESS);
                #ifdef OPENSSL_ALL
                ExpectIntEQ(SSL_CTX_set1_verify_cert_store(ctx, store), SSL_SUCCESS);
                #endif
            }
            if (EXPECT_FAIL() || (i == 1)) {
                X509_STORE_free(store);
                store = NULL;
            }
            SSL_free(ssl);
            ssl = NULL;
            SSL_CTX_free(ctx);
            ctx = NULL;
        }
    #endif /* !NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER */
    }
#endif
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_STORE_load_locations(void)
{
    EXPECT_DECLS;
#if (defined(OPENSSL_ALL) || defined(WOLFSSL_APACHE_HTTPD)) && \
    !defined(NO_FILESYSTEM) && !defined(NO_WOLFSSL_DIR) && !defined(NO_RSA) && \
    !defined(NO_TLS)
    SSL_CTX *ctx = NULL;
    X509_STORE *store = NULL;

    const char ca_file[] = "./certs/ca-cert.pem";
    const char client_pem_file[] = "./certs/client-cert.pem";
    const char client_der_file[] = "./certs/client-cert.der";
    const char ecc_file[] = "./certs/ecc-key.pem";
    const char certs_path[] = "./certs/";
    const char bad_path[] = "./bad-path/";
#ifdef HAVE_CRL
    const char crl_path[] = "./certs/crl/";
    const char crl_file[] = "./certs/crl/crl.pem";
#endif

#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = SSL_CTX_new(SSLv23_server_method()));
#else
    ExpectNotNull(ctx = SSL_CTX_new(SSLv23_client_method()));
#endif
    ExpectNotNull(store = SSL_CTX_get_cert_store(ctx));
    ExpectIntEQ(wolfSSL_CertManagerLoadCA(store->cm, ca_file, NULL),
        WOLFSSL_SUCCESS);

    /* Test bad arguments */
    ExpectIntEQ(X509_STORE_load_locations(NULL, ca_file, NULL),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(X509_STORE_load_locations(store, NULL, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(X509_STORE_load_locations(store, client_der_file, NULL),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(X509_STORE_load_locations(store, ecc_file, NULL),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(X509_STORE_load_locations(store, NULL, bad_path),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

#ifdef HAVE_CRL
    /* Test with CRL */
    ExpectIntEQ(X509_STORE_load_locations(store, crl_file, NULL),
        WOLFSSL_SUCCESS);
    ExpectIntEQ(X509_STORE_load_locations(store, NULL, crl_path),
        WOLFSSL_SUCCESS);
#endif

    /* Test with CA */
    ExpectIntEQ(X509_STORE_load_locations(store, ca_file, NULL),
        WOLFSSL_SUCCESS);

    /* Test with client_cert and certs path */
    ExpectIntEQ(X509_STORE_load_locations(store, client_pem_file, NULL),
        WOLFSSL_SUCCESS);
    ExpectIntEQ(X509_STORE_load_locations(store, NULL, certs_path),
        WOLFSSL_SUCCESS);

#if defined(XGETENV) && !defined(NO_GETENV) && defined(_POSIX_C_SOURCE) && \
    _POSIX_C_SOURCE >= 200112L
    ExpectIntEQ(wolfSSL_CTX_UnloadCAs(ctx), WOLFSSL_SUCCESS);
    /* Test with env vars */
    ExpectIntEQ(setenv("SSL_CERT_FILE", client_pem_file, 1), 0);
    ExpectIntEQ(setenv("SSL_CERT_DIR", certs_path, 1), 0);
    ExpectIntEQ(X509_STORE_set_default_paths(store), WOLFSSL_SUCCESS);
#endif

#if defined(OPENSSL_EXTRA) || defined(DEBUG_WOLFSSL_VERBOSE)
    /* Clear nodes */
    ERR_clear_error();
#endif

    SSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

static int test_X509_STORE_get0_objects(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_ALL) && !defined(NO_FILESYSTEM) && !defined(NO_TLS) && \
    !defined(NO_WOLFSSL_DIR) && !defined(NO_RSA)
    X509_STORE *store = NULL;
    X509_STORE *store_cpy = NULL;
    SSL_CTX *ctx = NULL;
    X509_OBJECT *obj = NULL;
#ifdef HAVE_CRL
    X509_OBJECT *objCopy = NULL;
#endif
    STACK_OF(X509_OBJECT) *objs = NULL;
    STACK_OF(X509_OBJECT) *objsCopy = NULL;
    int i;

    /* Setup store */
#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = SSL_CTX_new(SSLv23_server_method()));
#else
    ExpectNotNull(ctx = SSL_CTX_new(SSLv23_client_method()));
#endif
    ExpectNotNull(store_cpy = X509_STORE_new());
    ExpectNotNull(store = SSL_CTX_get_cert_store(ctx));
    ExpectIntEQ(X509_STORE_load_locations(store, cliCertFile, NULL),
        WOLFSSL_SUCCESS);
    ExpectIntEQ(X509_STORE_load_locations(store, caCertFile, NULL),
        WOLFSSL_SUCCESS);
    ExpectIntEQ(X509_STORE_load_locations(store, svrCertFile, NULL),
        WOLFSSL_SUCCESS);
#ifdef HAVE_CRL
    ExpectIntEQ(X509_STORE_load_locations(store, NULL, crlPemDir),
        WOLFSSL_SUCCESS);
#endif
    /* Store ready */

    /* Similar to HaProxy ssl_set_cert_crl_file use case */
    ExpectNotNull(objs = X509_STORE_get0_objects(store));
#ifdef HAVE_CRL
#ifdef WOLFSSL_SIGNER_DER_CERT
    ExpectIntEQ(sk_X509_OBJECT_num(objs), 4);
#else
    ExpectIntEQ(sk_X509_OBJECT_num(objs), 1);
#endif
#else
#ifdef WOLFSSL_SIGNER_DER_CERT
    ExpectIntEQ(sk_X509_OBJECT_num(objs), 3);
#else
    ExpectIntEQ(sk_X509_OBJECT_num(objs), 0);
#endif
#endif
    ExpectIntEQ(sk_X509_OBJECT_num(NULL), 0);
    ExpectNull(sk_X509_OBJECT_value(NULL, 0));
    ExpectNull(sk_X509_OBJECT_value(NULL, 1));
    ExpectNull(sk_X509_OBJECT_value(objs, sk_X509_OBJECT_num(objs)));
    ExpectNull(sk_X509_OBJECT_value(objs, sk_X509_OBJECT_num(objs) + 1));
#ifndef NO_WOLFSSL_STUB
    ExpectNull(sk_X509_OBJECT_delete(objs, 0));
#endif
    ExpectNotNull(objsCopy = sk_X509_OBJECT_deep_copy(objs, NULL, NULL));
    ExpectIntEQ(sk_X509_OBJECT_num(objs), sk_X509_OBJECT_num(objsCopy));
    for (i = 0; i < sk_X509_OBJECT_num(objs) && EXPECT_SUCCESS(); i++) {
        obj = (X509_OBJECT*)sk_X509_OBJECT_value(objs, i);
    #ifdef HAVE_CRL
        objCopy = (X509_OBJECT*)sk_X509_OBJECT_value(objsCopy, i);
    #endif
        switch (X509_OBJECT_get_type(obj)) {
        case X509_LU_X509:
        {
            X509* x509 = NULL;
            X509_NAME *subj_name = NULL;
            ExpectNull(X509_OBJECT_get0_X509_CRL(NULL));
            ExpectNull(X509_OBJECT_get0_X509_CRL(obj));
            ExpectNotNull(x509 = X509_OBJECT_get0_X509(obj));
            ExpectIntEQ(X509_STORE_add_cert(store_cpy, x509), WOLFSSL_SUCCESS);
            ExpectNotNull(subj_name = X509_get_subject_name(x509));
            ExpectPtrEq(obj, X509_OBJECT_retrieve_by_subject(objs, X509_LU_X509,
                    subj_name));

            break;
        }
        case X509_LU_CRL:
#ifdef HAVE_CRL
        {
            X509_CRL* crl = NULL;
            ExpectNull(X509_OBJECT_get0_X509(NULL));
            ExpectNull(X509_OBJECT_get0_X509(obj));
            ExpectNotNull(crl = X509_OBJECT_get0_X509_CRL(obj));
            ExpectIntEQ(X509_STORE_add_crl(store_cpy, crl), WOLFSSL_SUCCESS);

            ExpectNotNull(crl = X509_OBJECT_get0_X509_CRL(objCopy));
            break;
        }
#endif
        case X509_LU_NONE:
        default:
            Fail(("X509_OBJECT_get_type should return x509 or crl "
                    "(when built with crl support)"),
                    ("Unrecognized X509_OBJECT type or none"));
        }
    }

    X509_STORE_free(store_cpy);
    SSL_CTX_free(ctx);

    wolfSSL_sk_X509_OBJECT_free(NULL);
    objs = NULL;
    wolfSSL_sk_pop_free(objsCopy, NULL);
    objsCopy = NULL;
    ExpectNotNull(objs = wolfSSL_sk_X509_OBJECT_new());
    ExpectIntEQ(wolfSSL_sk_X509_OBJECT_push(NULL, NULL), WOLFSSL_FAILURE);
    ExpectIntEQ(wolfSSL_sk_X509_OBJECT_push(objs, NULL), WOLFSSL_FAILURE);
    ExpectIntEQ(wolfSSL_sk_X509_OBJECT_push(NULL, obj), WOLFSSL_FAILURE);
    ExpectNotNull(objsCopy = sk_X509_OBJECT_deep_copy(objs, NULL, NULL));
    wolfSSL_sk_X509_OBJECT_free(objsCopy);
    wolfSSL_sk_X509_OBJECT_free(objs);
#endif
    return EXPECT_RESULT();
}

#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_TLS) && \
   !defined(NO_FILESYSTEM) && !defined(NO_RSA)
#define TEST_ARG 0x1234
static void msg_cb(int write_p, int version, int content_type,
                   const void *buf, size_t len, SSL *ssl, void *arg)
{
    (void)write_p;
    (void)version;
    (void)content_type;
    (void)buf;
    (void)len;
    (void)ssl;

    AssertTrue(arg == (void*)TEST_ARG);
}
#endif

#if defined(OPENSSL_EXTRA) && defined(DEBUG_WOLFSSL) && \
      defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES)
#if defined(SESSION_CERTS)
#include "wolfssl/internal.h"
#endif
static int msgSrvCb(SSL_CTX *ctx, SSL *ssl)
{
    EXPECT_DECLS;
#if defined(OPENSSL_ALL) && defined(SESSION_CERTS) && !defined(NO_BIO)
    STACK_OF(X509)* sk = NULL;
    X509* x509 = NULL;
    int i, num;
    BIO* bio = NULL;
#endif

    ExpectNotNull(ctx);
    ExpectNotNull(ssl);

    fprintf(stderr, "\n===== msgSrvCb called ====\n");
#if defined(SESSION_CERTS) && defined(TEST_PEER_CERT_CHAIN)
    ExpectTrue(SSL_get_peer_cert_chain(ssl) != NULL);
    ExpectIntEQ(((WOLFSSL_X509_CHAIN *)SSL_get_peer_cert_chain(ssl))->count, 2);
    ExpectNotNull(SSL_get0_verified_chain(ssl));
#endif

#if defined(OPENSSL_ALL) && defined(SESSION_CERTS) && !defined(NO_BIO)
#ifdef KEEP_PEER_CERT
    {
        WOLFSSL_X509* peer = NULL;
        ExpectNotNull(peer= wolfSSL_get_peer_certificate(ssl));
        ExpectNotNull(bio = BIO_new_fp(stderr, BIO_NOCLOSE));
        fprintf(stderr, "Peer Certificate = :\n");
        X509_print(bio, peer);
        X509_free(peer);
    }
#endif

    ExpectNotNull(sk = SSL_get_peer_cert_chain(ssl));
    if (sk == NULL) {
        BIO_free(bio);
        return TEST_FAIL;
    }
    num = sk_X509_num(sk);
    ExpectTrue(num > 0);
    for (i = 0; i < num; i++) {
        ExpectNotNull(x509 = sk_X509_value(sk,i));
        if (x509 == NULL)
            break;
        fprintf(stderr, "Certificate at index [%d] = :\n",i);
        X509_print(bio,x509);
        fprintf(stderr, "\n\n");
    }
    BIO_free(bio);
#endif
    return EXPECT_RESULT();
}

static int msgCb(SSL_CTX *ctx, SSL *ssl)
{
    EXPECT_DECLS;
#if defined(OPENSSL_ALL) && defined(SESSION_CERTS) && !defined(NO_BIO)
    STACK_OF(X509)* sk = NULL;
    X509* x509 = NULL;
    int i, num;
    BIO* bio = NULL;
#endif

    ExpectNotNull(ctx);
    ExpectNotNull(ssl);

    fprintf(stderr, "\n===== msgcb called ====\n");
#if defined(SESSION_CERTS) && defined(TEST_PEER_CERT_CHAIN)
    ExpectTrue(SSL_get_peer_cert_chain(ssl) != NULL);
    ExpectIntEQ(((WOLFSSL_X509_CHAIN *)SSL_get_peer_cert_chain(ssl))->count, 2);
    ExpectNotNull(SSL_get0_verified_chain(ssl));
#endif

#if defined(OPENSSL_ALL) && defined(SESSION_CERTS) && !defined(NO_BIO)
    ExpectNotNull(bio = BIO_new_fp(stderr, BIO_NOCLOSE));
    ExpectNotNull(sk = SSL_get_peer_cert_chain(ssl));
    if (sk == NULL) {
        BIO_free(bio);
        return TEST_FAIL;
    }
    num = sk_X509_num(sk);
    ExpectTrue(num > 0);
    for (i = 0; i < num; i++) {
        ExpectNotNull(x509 = sk_X509_value(sk,i));
        if (x509 == NULL)
            break;
        fprintf(stderr, "Certificate at index [%d] = :\n",i);
        X509_print(bio,x509);
        fprintf(stderr, "\n\n");
    }
    BIO_free(bio);
#endif
    return EXPECT_RESULT();
}
#endif

static int test_wolfSSL_msgCb(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && defined(DEBUG_WOLFSSL) && \
      defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES)
    test_ssl_cbf client_cb;
    test_ssl_cbf server_cb;

    XMEMSET(&client_cb, 0, sizeof(client_cb));
    XMEMSET(&server_cb, 0, sizeof(server_cb));
#ifndef WOLFSSL_NO_TLS12
    client_cb.method  = wolfTLSv1_2_client_method;
    server_cb.method  = wolfTLSv1_2_server_method;
#else
    client_cb.method  = wolfTLSv1_3_client_method;
    server_cb.method  = wolfTLSv1_3_server_method;
#endif
    server_cb.caPemFile = caCertFile;
    client_cb.certPemFile = "./certs/intermediate/client-chain.pem";

    ExpectIntEQ(test_wolfSSL_client_server_nofail_memio_ex(&client_cb,
        &server_cb, msgCb, msgSrvCb), TEST_SUCCESS);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_either_side(void)
{
    EXPECT_DECLS;
#if (defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)) && \
    defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES)
    test_ssl_cbf client_cb;
    test_ssl_cbf server_cb;

    XMEMSET(&client_cb, 0, sizeof(client_cb));
    XMEMSET(&server_cb, 0, sizeof(server_cb));

    /* Use different CTX for client and server */
    client_cb.ctx = wolfSSL_CTX_new(wolfSSLv23_method());
    ExpectNotNull(client_cb.ctx);
    server_cb.ctx = wolfSSL_CTX_new(wolfSSLv23_method());
    ExpectNotNull(server_cb.ctx);
    /* we are responsible for free'ing WOLFSSL_CTX */
    server_cb.isSharedCtx = client_cb.isSharedCtx = 1;

    ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&client_cb,
        &server_cb, NULL), TEST_SUCCESS);

    wolfSSL_CTX_free(client_cb.ctx);
    wolfSSL_CTX_free(server_cb.ctx);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_DTLS_either_side(void)
{
    EXPECT_DECLS;
#if (defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)) && \
    defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES) && defined(WOLFSSL_DTLS)
    test_ssl_cbf client_cb;
    test_ssl_cbf server_cb;

    XMEMSET(&client_cb, 0, sizeof(client_cb));
    XMEMSET(&server_cb, 0, sizeof(server_cb));

    /* Use different CTX for client and server */
    client_cb.ctx = wolfSSL_CTX_new(wolfDTLS_method());
    ExpectNotNull(client_cb.ctx);
    server_cb.ctx = wolfSSL_CTX_new(wolfDTLS_method());
    ExpectNotNull(server_cb.ctx);
    /* we are responsible for free'ing WOLFSSL_CTX */
    server_cb.isSharedCtx = client_cb.isSharedCtx = 1;

    ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&client_cb,
        &server_cb, NULL), TEST_SUCCESS);

    wolfSSL_CTX_free(client_cb.ctx);
    wolfSSL_CTX_free(server_cb.ctx);
#endif
    return EXPECT_RESULT();
}

static int test_generate_cookie(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_DTLS) && defined(OPENSSL_EXTRA) && defined(USE_WOLFSSL_IO)
    SSL_CTX* ctx = NULL;
    SSL* ssl = NULL;
    byte    buf[FOURK_BUF] = {0};

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfDTLS_method()));
    ExpectNotNull(ssl = SSL_new(ctx));

    /* Test unconnected */
    ExpectIntEQ(EmbedGenerateCookie(ssl, buf, FOURK_BUF, NULL), WC_NO_ERR_TRACE(GEN_COOKIE_E));

    wolfSSL_CTX_SetGenCookie(ctx, EmbedGenerateCookie);

    wolfSSL_SetCookieCtx(ssl, ctx);

    ExpectNotNull(wolfSSL_GetCookieCtx(ssl));

    ExpectNull(wolfSSL_GetCookieCtx(NULL));

    SSL_free(ssl);
    SSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_set_options(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && !defined(NO_TLS) && !defined(NO_FILESYSTEM) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)) && \
    !defined(NO_RSA)
    WOLFSSL*     ssl = NULL;
    WOLFSSL_CTX* ctx = NULL;
#if defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)
    char appData[] = "extra msg";
#endif
#ifdef OPENSSL_EXTRA
    unsigned char protos[] = {
        7, 't', 'l', 's', '/', '1', '.', '2',
        8, 'h', 't', 't', 'p', '/', '1', '.', '1'
    };
    unsigned int len = sizeof(protos);
    void *arg = (void *)TEST_ARG;
#endif

#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#else
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
#endif
    ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, svrCertFile,
                                                CERT_FILETYPE));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, svrKeyFile,
                                               CERT_FILETYPE));

    ExpectTrue(wolfSSL_CTX_set_options(ctx, WOLFSSL_OP_NO_TLSv1)
               == WOLFSSL_OP_NO_TLSv1);
    ExpectTrue(wolfSSL_CTX_get_options(ctx) == WOLFSSL_OP_NO_TLSv1);

    ExpectIntGT((int)wolfSSL_CTX_set_options(ctx, (WOLFSSL_OP_COOKIE_EXCHANGE |
                WOLFSSL_OP_NO_SSLv2)), 0);
    ExpectTrue((wolfSSL_CTX_set_options(ctx, WOLFSSL_OP_COOKIE_EXCHANGE) &
                WOLFSSL_OP_COOKIE_EXCHANGE) == WOLFSSL_OP_COOKIE_EXCHANGE);
    ExpectTrue((wolfSSL_CTX_set_options(ctx, WOLFSSL_OP_NO_TLSv1_2) &
                WOLFSSL_OP_NO_TLSv1_2) == WOLFSSL_OP_NO_TLSv1_2);
    ExpectTrue((wolfSSL_CTX_set_options(ctx, WOLFSSL_OP_NO_COMPRESSION) &
                WOLFSSL_OP_NO_COMPRESSION) == WOLFSSL_OP_NO_COMPRESSION);
    ExpectFalse((wolfSSL_CTX_clear_options(ctx, WOLFSSL_OP_NO_COMPRESSION) &
                                           WOLFSSL_OP_NO_COMPRESSION));

    wolfSSL_CTX_free(ctx);
    ctx = NULL;

#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#else
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
#endif
    ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, svrCertFile,
                                               CERT_FILETYPE));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, svrKeyFile,
                                               CERT_FILETYPE));
#ifdef OPENSSL_EXTRA
    ExpectTrue(wolfSSL_CTX_set_msg_callback(ctx, msg_cb) == WOLFSSL_SUCCESS);
#endif

    ExpectNotNull(ssl = wolfSSL_new(ctx));
#if defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)
#ifdef HAVE_EX_DATA
    ExpectIntEQ(wolfSSL_set_app_data(ssl, (void*)appData), WOLFSSL_SUCCESS);
    ExpectNotNull(wolfSSL_get_app_data((const WOLFSSL*)ssl));
    if (ssl != NULL) {
        ExpectIntEQ(XMEMCMP(wolfSSL_get_app_data((const WOLFSSL*)ssl),
                    appData, sizeof(appData)), 0);
    }
#else
    ExpectIntEQ(wolfSSL_set_app_data(ssl, (void*)appData), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectNull(wolfSSL_get_app_data((const WOLFSSL*)ssl));
#endif
#endif

    ExpectTrue(wolfSSL_set_options(ssl, WOLFSSL_OP_NO_TLSv1) ==
                                   WOLFSSL_OP_NO_TLSv1);

    ExpectTrue(wolfSSL_get_options(ssl) == WOLFSSL_OP_NO_TLSv1);

    ExpectIntGT((int)wolfSSL_set_options(ssl, (WOLFSSL_OP_COOKIE_EXCHANGE |
                                         WOLFSSL_OP_NO_SSLv2)), 0);

    ExpectTrue((wolfSSL_set_options(ssl, WOLFSSL_OP_COOKIE_EXCHANGE) &
                WOLFSSL_OP_COOKIE_EXCHANGE) == WOLFSSL_OP_COOKIE_EXCHANGE);

    ExpectTrue((wolfSSL_set_options(ssl, WOLFSSL_OP_NO_TLSv1_2) &
                WOLFSSL_OP_NO_TLSv1_2) == WOLFSSL_OP_NO_TLSv1_2);

    ExpectTrue((wolfSSL_set_options(ssl, WOLFSSL_OP_NO_COMPRESSION) &
               WOLFSSL_OP_NO_COMPRESSION) == WOLFSSL_OP_NO_COMPRESSION);

#ifdef OPENSSL_EXTRA
    ExpectFalse((wolfSSL_clear_options(ssl, WOLFSSL_OP_NO_COMPRESSION) &
                                       WOLFSSL_OP_NO_COMPRESSION));
#endif

#ifdef OPENSSL_EXTRA
    ExpectTrue(wolfSSL_set_msg_callback(ssl, msg_cb) == WOLFSSL_SUCCESS);
    wolfSSL_set_msg_callback_arg(ssl, arg);
#ifdef WOLFSSL_ERROR_CODE_OPENSSL
    ExpectTrue(wolfSSL_CTX_set_alpn_protos(ctx, protos, len) == 0);
#else
    ExpectTrue(wolfSSL_CTX_set_alpn_protos(ctx, protos, len) == WOLFSSL_SUCCESS);
#endif
#endif

#if defined(WOLFSSL_NGINX) || defined(WOLFSSL_HAPROXY) || \
    defined(WOLFSSL_MYSQL_COMPATIBLE) || defined(OPENSSL_ALL) || \
    defined(HAVE_LIGHTY) || defined(HAVE_STUNNEL)

#if defined(HAVE_ALPN) && !defined(NO_BIO)

#ifdef WOLFSSL_ERROR_CODE_OPENSSL
    ExpectTrue(wolfSSL_set_alpn_protos(ssl, protos, len) == 0);
#else
    ExpectTrue(wolfSSL_set_alpn_protos(ssl, protos, len) == WOLFSSL_SUCCESS);
#endif

#endif /* HAVE_ALPN && !NO_BIO */
#endif

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_sk_SSL_CIPHER(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_ALL) && !defined(NO_CERTS) && !defined(NO_TLS) && \
   !defined(NO_FILESYSTEM) && !defined(NO_RSA)
#if !defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)
    SSL*     ssl = NULL;
    SSL_CTX* ctx = NULL;
    STACK_OF(SSL_CIPHER) *sk = NULL;
    STACK_OF(SSL_CIPHER) *dupSk = NULL;

#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_server_method()));
#else
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_client_method()));
#endif
    ExpectTrue(SSL_CTX_use_certificate_file(ctx, svrCertFile, SSL_FILETYPE_PEM));
    ExpectTrue(SSL_CTX_use_PrivateKey_file(ctx, svrKeyFile, SSL_FILETYPE_PEM));
    ExpectNotNull(ssl = SSL_new(ctx));
    ExpectNotNull(sk = SSL_get_ciphers(ssl));
    ExpectNotNull(dupSk = sk_SSL_CIPHER_dup(sk));
    ExpectIntGT(sk_SSL_CIPHER_num(sk), 0);
    ExpectIntEQ(sk_SSL_CIPHER_num(sk), sk_SSL_CIPHER_num(dupSk));

    /* error case because connection has not been established yet */
    ExpectIntEQ(sk_SSL_CIPHER_find(sk, SSL_get_current_cipher(ssl)), -1);
    sk_SSL_CIPHER_free(dupSk);

    /* sk is pointer to internal struct that should be free'd in SSL_free */
    SSL_free(ssl);
    SSL_CTX_free(ctx);
#endif /* !NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER */
#endif /* defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && \
         !defined(NO_FILESYSTEM) && !defined(NO_RSA) */
    return EXPECT_RESULT();
}

static int test_wolfSSL_set1_curves_list(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && defined(HAVE_ECC) && !defined(NO_TLS) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)) && \
    !defined(NO_FILESYSTEM)
    SSL*     ssl = NULL;
    SSL_CTX* ctx = NULL;

#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_server_method()));
#else
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_client_method()));
#endif
    ExpectTrue(SSL_CTX_use_certificate_file(ctx, eccCertFile,
               SSL_FILETYPE_PEM));
    ExpectTrue(SSL_CTX_use_PrivateKey_file(ctx, eccKeyFile, SSL_FILETYPE_PEM));
    ExpectNotNull(ssl = SSL_new(ctx));

    ExpectIntEQ(SSL_CTX_set1_curves_list(ctx, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
#ifdef HAVE_ECC
    ExpectIntEQ(SSL_CTX_set1_curves_list(ctx, "P-25X"), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(SSL_CTX_set1_curves_list(ctx, "P-256"), WOLFSSL_SUCCESS);
#endif
#ifdef HAVE_CURVE25519
    ExpectIntEQ(SSL_CTX_set1_curves_list(ctx, "X25519"), WOLFSSL_SUCCESS);
#else
    ExpectIntEQ(SSL_CTX_set1_curves_list(ctx, "X25519"), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
#endif
#ifdef HAVE_CURVE448
    ExpectIntEQ(SSL_CTX_set1_curves_list(ctx, "X448"), WOLFSSL_SUCCESS);
#else
    ExpectIntEQ(SSL_CTX_set1_curves_list(ctx, "X448"), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
#endif

    ExpectIntEQ(SSL_set1_curves_list(ssl, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
#ifdef HAVE_ECC
    ExpectIntEQ(SSL_set1_curves_list(ssl, "P-25X"), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(SSL_set1_curves_list(ssl, "P-256"), WOLFSSL_SUCCESS);
#endif

#ifdef HAVE_CURVE25519
    ExpectIntEQ(SSL_set1_curves_list(ssl, "X25519"), WOLFSSL_SUCCESS);
#else
    ExpectIntEQ(SSL_set1_curves_list(ssl, "X25519"), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
#endif
#ifdef HAVE_CURVE448
    ExpectIntEQ(SSL_set1_curves_list(ssl, "X448"), WOLFSSL_SUCCESS);
#else
    ExpectIntEQ(SSL_set1_curves_list(ssl, "X448"), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
#endif

    SSL_free(ssl);
    SSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

#if defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES) && \
        (defined(OPENSSL_EXTRA) || defined(HAVE_CURL)) && defined(HAVE_ECC)
static int test_wolfSSL_curves_mismatch_ctx_ready(WOLFSSL_CTX* ctx)
{
    static int counter = 0;
    EXPECT_DECLS;

    if (counter % 2) {
        ExpectIntEQ(wolfSSL_CTX_set1_curves_list(ctx, "P-256"),
                WOLFSSL_SUCCESS);
    }
    else {
        ExpectIntEQ(wolfSSL_CTX_set1_curves_list(ctx, "P-384"),
                WOLFSSL_SUCCESS);
    }

    /* Ciphersuites that require curves */
    wolfSSL_CTX_set_cipher_list(ctx, "TLS13-AES256-GCM-SHA384:"
            "TLS13-CHACHA20-POLY1305-SHA256:TLS13-AES128-GCM-SHA256:"
            "ECDHE-ECDSA-AES256-GCM-SHA384:ECDHE-RSA-AES256-GCM-SHA384:"
            "ECDHE-ECDSA-AES128-GCM-SHA256:"
            "ECDHE-RSA-AES128-GCM-SHA256:ECDHE-RSA-CHACHA20-POLY1305:"
            "ECDHE-ECDSA-CHACHA20-POLY1305");

    counter++;
    return EXPECT_RESULT();
}
#endif

static int test_wolfSSL_curves_mismatch(void)
{
    EXPECT_DECLS;
#if defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES) && \
        (defined(OPENSSL_EXTRA) || defined(HAVE_CURL)) && defined(HAVE_ECC)
    test_ssl_cbf func_cb_client;
    test_ssl_cbf func_cb_server;
    size_t i;
    struct {
        method_provider client_meth;
        method_provider server_meth;
        const char* desc;
        int client_last_err;
        int server_last_err;
    } test_params[] = {
#ifdef WOLFSSL_TLS13
        {wolfTLSv1_3_client_method, wolfTLSv1_3_server_method, "TLS 1.3",
                /* Client gets error because server will attempt HRR */
                WC_NO_ERR_TRACE(BAD_KEY_SHARE_DATA),
                WC_NO_ERR_TRACE(FATAL_ERROR)
        },
#endif
#ifndef WOLFSSL_NO_TLS12
        {wolfTLSv1_2_client_method, wolfTLSv1_2_server_method, "TLS 1.2",
                WC_NO_ERR_TRACE(FATAL_ERROR),
                /* Server gets error because <=1.2 doesn't have a mechanism
                 * to negotiate curves. */
#ifdef OPENSSL_EXTRA
                WC_NO_ERR_TRACE(WOLFSSL_ERROR_SYSCALL)
#else
                WC_NO_ERR_TRACE(MATCH_SUITE_ERROR)
#endif
        },
#endif
#ifndef NO_OLD_TLS
        {wolfTLSv1_1_client_method, wolfTLSv1_1_server_method, "TLS 1.1",
                WC_NO_ERR_TRACE(FATAL_ERROR),
#ifdef OPENSSL_EXTRA
                WC_NO_ERR_TRACE(WOLFSSL_ERROR_SYSCALL)
#else
                WC_NO_ERR_TRACE(MATCH_SUITE_ERROR)
#endif
        },
#endif
    };

    for (i = 0; i < XELEM_CNT(test_params) && !EXPECT_FAIL(); i++) {
        XMEMSET(&func_cb_client, 0, sizeof(func_cb_client));
        XMEMSET(&func_cb_server, 0, sizeof(func_cb_server));

        printf("\tTesting with %s...\n", test_params[i].desc);

        func_cb_client.ctx_ready = &test_wolfSSL_curves_mismatch_ctx_ready;
        func_cb_server.ctx_ready = &test_wolfSSL_curves_mismatch_ctx_ready;

        func_cb_client.method = test_params[i].client_meth;
        func_cb_server.method = test_params[i].server_meth;

        ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&func_cb_client,
            &func_cb_server, NULL), -1001);
        ExpectIntEQ(func_cb_client.last_err, test_params[i].client_last_err);
        ExpectIntEQ(func_cb_server.last_err, test_params[i].server_last_err);

        if (!EXPECT_SUCCESS())
            break;
        printf("\t%s passed\n", test_params[i].desc);
    }
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_set1_sigalgs_list(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_RSA) && \
    !defined(NO_TLS) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)) && \
    !defined(NO_FILESYSTEM)
    SSL*     ssl = NULL;
    SSL_CTX* ctx = NULL;

#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_server_method()));
#else
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_client_method()));
#endif
    ExpectTrue(SSL_CTX_use_certificate_file(ctx, svrCertFile,
               SSL_FILETYPE_PEM));
    ExpectTrue(SSL_CTX_use_PrivateKey_file(ctx, svrKeyFile, SSL_FILETYPE_PEM));
    ExpectNotNull(ssl = SSL_new(ctx));

    ExpectIntEQ(wolfSSL_CTX_set1_sigalgs_list(NULL, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(wolfSSL_CTX_set1_sigalgs_list(ctx, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(wolfSSL_set1_sigalgs_list(NULL, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(wolfSSL_set1_sigalgs_list(ssl, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

    ExpectIntEQ(wolfSSL_CTX_set1_sigalgs_list(ctx, ""), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(wolfSSL_set1_sigalgs_list(ssl, ""), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

#ifndef NO_RSA
    #ifndef NO_SHA256
        ExpectIntEQ(wolfSSL_CTX_set1_sigalgs_list(NULL, "RSA+SHA256"),
                    WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
        ExpectIntEQ(wolfSSL_set1_sigalgs_list(NULL, "RSA+SHA256"),
                    WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

        ExpectIntEQ(wolfSSL_CTX_set1_sigalgs_list(ctx, "RSA+SHA256"),
                    WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_set1_sigalgs_list(ssl, "RSA+SHA256"),
                    WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_CTX_set1_sigalgs_list(ctx, "RSA-SHA256"),
                    WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
        ExpectIntEQ(wolfSSL_set1_sigalgs_list(ssl, "RSA-SHA256"),
                    WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
        #ifdef WC_RSA_PSS
            ExpectIntEQ(wolfSSL_CTX_set1_sigalgs_list(ctx, "RSA-PSS+SHA256"),
                        WOLFSSL_SUCCESS);
            ExpectIntEQ(wolfSSL_set1_sigalgs_list(ssl, "RSA-PSS+SHA256"),
                        WOLFSSL_SUCCESS);
            ExpectIntEQ(wolfSSL_CTX_set1_sigalgs_list(ctx, "PSS+SHA256"),
                        WOLFSSL_SUCCESS);
            ExpectIntEQ(wolfSSL_set1_sigalgs_list(ssl, "PSS+SHA256"),
                        WOLFSSL_SUCCESS);
        #endif
        #ifdef WOLFSSL_SHA512
            ExpectIntEQ(wolfSSL_CTX_set1_sigalgs_list(ctx,
                       "RSA+SHA256:RSA+SHA512"), WOLFSSL_SUCCESS);
            ExpectIntEQ(wolfSSL_set1_sigalgs_list(ssl,
                       "RSA+SHA256:RSA+SHA512"), WOLFSSL_SUCCESS);
        #elif defined(WOLFSSL_SHA384)
            ExpectIntEQ(wolfSSL_CTX_set1_sigalgs_list(ctx,
                       "RSA+SHA256:RSA+SHA384"), WOLFSSL_SUCCESS);
            ExpectIntEQ(wolfSSL_set1_sigalgs_list(ssl,
                       "RSA+SHA256:RSA+SHA384"), WOLFSSL_SUCCESS);
        #endif
        ExpectIntEQ(wolfSSL_CTX_set1_sigalgs_list(ctx, "RSA"), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
        ExpectIntEQ(wolfSSL_set1_sigalgs_list(ssl, "RSA"), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
        ExpectIntEQ(wolfSSL_CTX_set1_sigalgs_list(ctx, "RSA:RSA+SHA256"),
                    WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
        ExpectIntEQ(wolfSSL_set1_sigalgs_list(ssl, "RSA:RSA+SHA256"),
                    WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

        ExpectIntEQ(wolfSSL_CTX_set1_sigalgs_list(ctx, "RSA+SHA256+SHA256"),
                    WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
        ExpectIntEQ(wolfSSL_set1_sigalgs_list(ssl, "RSA+SHA256+RSA"),
                    WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    #endif
#endif
#ifdef HAVE_ECC
    #ifndef NO_SHA256
        ExpectIntEQ(wolfSSL_CTX_set1_sigalgs_list(ctx, "ECDSA+SHA256"),
                    WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_set1_sigalgs_list(ssl, "ECDSA+SHA256"),
            WOLFSSL_SUCCESS);
        #ifdef WOLFSSL_SHA512
            ExpectIntEQ(wolfSSL_CTX_set1_sigalgs_list(ctx,
                       "ECDSA+SHA256:ECDSA+SHA512"), WOLFSSL_SUCCESS);
            ExpectIntEQ(wolfSSL_set1_sigalgs_list(ssl,
                       "ECDSA+SHA256:ECDSA+SHA512"), WOLFSSL_SUCCESS);
        #elif defined(WOLFSSL_SHA384)
            ExpectIntEQ(wolfSSL_CTX_set1_sigalgs_list(ctx,
                       "ECDSA+SHA256:ECDSA+SHA384"), WOLFSSL_SUCCESS);
            ExpectIntEQ(wolfSSL_set1_sigalgs_list(ssl,
                       "ECDSA+SHA256:ECDSA+SHA384"), WOLFSSL_SUCCESS);
        #endif
    #endif
#endif
#ifdef HAVE_ED25519
    ExpectIntEQ(wolfSSL_CTX_set1_sigalgs_list(ctx, "ED25519"), WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_set1_sigalgs_list(ssl, "ED25519"), WOLFSSL_SUCCESS);
#endif
#ifdef HAVE_ED448
    ExpectIntEQ(wolfSSL_CTX_set1_sigalgs_list(ctx, "ED448"), WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_set1_sigalgs_list(ssl, "ED448"), WOLFSSL_SUCCESS);
#endif
#ifndef NO_DSA
    #ifndef NO_SHA256
        ExpectIntEQ(wolfSSL_CTX_set1_sigalgs_list(ctx, "DSA+SHA256"),
                    WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_set1_sigalgs_list(ssl, "DSA+SHA256"),
                    WOLFSSL_SUCCESS);
    #endif
    #if !defined(NO_SHA) && (!defined(NO_OLD_TLS) || \
                                                defined(WOLFSSL_ALLOW_TLS_SHA1))
        ExpectIntEQ(wolfSSL_CTX_set1_sigalgs_list(ctx, "DSA+SHA1"),
                    WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_set1_sigalgs_list(ssl, "DSA+SHA1"),
                    WOLFSSL_SUCCESS);
    #endif
#endif

    SSL_free(ssl);
    SSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

/* Testing  wolfSSL_set_tlsext_status_type function.
 * PRE: OPENSSL and HAVE_CERTIFICATE_STATUS_REQUEST defined.
 */
static int test_wolfSSL_set_tlsext_status_type(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && defined(HAVE_CERTIFICATE_STATUS_REQUEST) && \
    !defined(NO_RSA) && !defined(NO_WOLFSSL_SERVER)
    SSL*     ssl = NULL;
    SSL_CTX* ctx = NULL;

    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_server_method()));
    ExpectTrue(SSL_CTX_use_certificate_file(ctx, svrCertFile,
        SSL_FILETYPE_PEM));
    ExpectTrue(SSL_CTX_use_PrivateKey_file(ctx, svrKeyFile, SSL_FILETYPE_PEM));
    ExpectNotNull(ssl = SSL_new(ctx));
    ExpectIntEQ(SSL_set_tlsext_status_type(ssl,TLSEXT_STATUSTYPE_ocsp),
        SSL_SUCCESS);
    ExpectIntEQ(SSL_get_tlsext_status_type(ssl), TLSEXT_STATUSTYPE_ocsp);
    SSL_free(ssl);
    SSL_CTX_free(ctx);
#endif  /* OPENSSL_EXTRA && HAVE_CERTIFICATE_STATUS_REQUEST && !NO_RSA */
    return EXPECT_RESULT();
}

#ifndef NO_BIO

static int test_wolfSSL_PEM_read_bio(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && \
   !defined(NO_FILESYSTEM) && !defined(NO_RSA)
    byte buff[6000];
    XFILE f = XBADFILE;
    int  bytes = 0;
    X509* x509 = NULL;
    BIO*  bio = NULL;
    BUF_MEM* buf = NULL;

    ExpectTrue((f = XFOPEN(cliCertFile, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buff, 1, sizeof(buff), f), 0);
    if (f != XBADFILE)
        XFCLOSE(f);

    ExpectNull(x509 = PEM_read_bio_X509_AUX(bio, NULL, NULL, NULL));
    ExpectNotNull(bio = BIO_new_mem_buf((void*)buff, bytes));
    ExpectIntEQ(BIO_set_mem_eof_return(bio, -0xDEAD), 1);
    ExpectNotNull(x509 = PEM_read_bio_X509_AUX(bio, NULL, NULL, NULL));
    ExpectIntEQ((int)BIO_set_fd(bio, 0, BIO_CLOSE), 1);
    /* BIO should return the set EOF value */
    ExpectIntEQ(BIO_read(bio, buff, sizeof(buff)), -0xDEAD);
    ExpectIntEQ(BIO_set_close(bio, BIO_NOCLOSE), 1);
    ExpectIntEQ(BIO_set_close(NULL, BIO_NOCLOSE), 1);
    ExpectIntEQ(SSL_SUCCESS, BIO_get_mem_ptr(bio, &buf));

    BIO_free(bio);
    BUF_MEM_free(buf);
    X509_free(x509);
#endif /* defined(OPENSSL_EXTRA) && !defined(NO_CERTS) &&
        * !defined(NO_FILESYSTEM) && !defined(NO_RSA) */
    return EXPECT_RESULT();
}


#if defined(OPENSSL_EXTRA)
static long bioCallback(BIO *bio, int cmd, const char* argp, int argi,
                 long argl, long ret)
{
    (void)bio;
    (void)cmd;
    (void)argp;
    (void)argi;
    (void)argl;
    return ret;
}
#endif


static int test_wolfSSL_BIO(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA)
    const unsigned char* p = NULL;
    byte buff[20];
    BIO* bio1 = NULL;
    BIO* bio2 = NULL;
    BIO* bio3 = NULL;
    char* bufPt = NULL;
    int i;

    for (i = 0; i < 20; i++) {
        buff[i] = i;
    }
    /* test BIO_free with NULL */
    ExpectIntEQ(BIO_free(NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

    /* Creating and testing type BIO_s_bio */
    ExpectNotNull(bio1 = BIO_new(BIO_s_bio()));
    ExpectNotNull(bio2 = BIO_new(BIO_s_bio()));
    ExpectNotNull(bio3 = BIO_new(BIO_s_bio()));

    /* read/write before set up */
    ExpectIntEQ(BIO_read(bio1, buff, 2),  WOLFSSL_BIO_UNSET);
    ExpectIntEQ(BIO_write(bio1, buff, 2), WOLFSSL_BIO_UNSET);

    ExpectIntEQ(BIO_set_nbio(bio1, 1), 1);
    ExpectIntEQ(BIO_set_write_buf_size(bio1, 20), WOLFSSL_SUCCESS);
    ExpectIntEQ(BIO_set_write_buf_size(bio2, 8),  WOLFSSL_SUCCESS);
    ExpectIntEQ(BIO_make_bio_pair(bio1, bio2),    WOLFSSL_SUCCESS);

    ExpectIntEQ(BIO_nwrite(bio1, &bufPt, 10), 10);
    ExpectNotNull(XMEMCPY(bufPt, buff, 10));
    ExpectIntEQ(BIO_write(bio1, buff + 10, 10), 10);
    /* write buffer full */
    ExpectIntEQ(BIO_write(bio1, buff, 10), WOLFSSL_BIO_ERROR);
    ExpectIntEQ(BIO_flush(bio1), WOLFSSL_SUCCESS);
    ExpectIntEQ((int)BIO_ctrl_pending(bio1), 0);

    /* write the other direction with pair */
    ExpectIntEQ((int)BIO_nwrite(bio2, &bufPt, 10), 8);
    ExpectNotNull(XMEMCPY(bufPt, buff, 8));
    ExpectIntEQ(BIO_write(bio2, buff, 10), WOLFSSL_BIO_ERROR);

    /* try read */
    ExpectIntEQ((int)BIO_ctrl_pending(bio1), 8);
    ExpectIntEQ((int)BIO_ctrl_pending(bio2), 20);

    /* try read using ctrl function */
    ExpectIntEQ((int)BIO_ctrl(bio1, BIO_CTRL_WPENDING, 0, NULL), 8);
    ExpectIntEQ((int)BIO_ctrl(bio1, BIO_CTRL_PENDING, 0, NULL), 8);
    ExpectIntEQ((int)BIO_ctrl(bio2, BIO_CTRL_WPENDING, 0, NULL), 20);
    ExpectIntEQ((int)BIO_ctrl(bio2, BIO_CTRL_PENDING, 0, NULL), 20);

    ExpectIntEQ(BIO_nread(bio2, &bufPt, (int)BIO_ctrl_pending(bio2)), 20);
    for (i = 0; i < 20; i++) {
        ExpectIntEQ((int)bufPt[i], i);
    }
    ExpectIntEQ(BIO_nread(bio2, &bufPt, 1), 0);
    ExpectIntEQ(BIO_nread(bio1, &bufPt, (int)BIO_ctrl_pending(bio1)), 8);
    for (i = 0; i < 8; i++) {
        ExpectIntEQ((int)bufPt[i], i);
    }
    ExpectIntEQ(BIO_nread(bio1, &bufPt, 1), 0);
    ExpectIntEQ(BIO_ctrl_reset_read_request(bio1), 1);

    /* new pair */
    ExpectIntEQ(BIO_make_bio_pair(bio1, bio3), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    BIO_free(bio2); /* free bio2 and automatically remove from pair */
    bio2 = NULL;
    ExpectIntEQ(BIO_make_bio_pair(bio1, bio3), WOLFSSL_SUCCESS);
    ExpectIntEQ((int)BIO_ctrl_pending(bio3), 0);
    ExpectIntEQ(BIO_nread(bio3, &bufPt, 10), 0);

    /* test wrap around... */
    ExpectIntEQ(BIO_reset(bio1), 1);
    ExpectIntEQ(BIO_reset(bio3), 1);

    /* fill write buffer, read only small amount then write again */
    ExpectIntEQ(BIO_nwrite(bio1, &bufPt, 20), 20);
    ExpectNotNull(XMEMCPY(bufPt, buff, 20));
    ExpectIntEQ(BIO_nread(bio3, &bufPt, 4), 4);
    for (i = 0; i < 4; i++) {
        ExpectIntEQ(bufPt[i], i);
    }

    /* try writing over read index */
    ExpectIntEQ(BIO_nwrite(bio1, &bufPt, 5), 4);
    ExpectNotNull(XMEMSET(bufPt, 0, 4));
    ExpectIntEQ((int)BIO_ctrl_pending(bio3), 20);

    /* read and write 0 bytes */
    ExpectIntEQ(BIO_nread(bio3, &bufPt, 0), 0);
    ExpectIntEQ(BIO_nwrite(bio1, &bufPt, 0), 0);

    /* should read only to end of write buffer then need to read again */
    ExpectIntEQ(BIO_nread(bio3, &bufPt, 20), 16);
    for (i = 0; i < 16; i++) {
        ExpectIntEQ(bufPt[i], buff[4 + i]);
    }

    ExpectIntEQ(BIO_nread(bio3, NULL, 0), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(BIO_nread0(bio3, &bufPt), 4);
    for (i = 0; i < 4; i++) {
        ExpectIntEQ(bufPt[i], 0);
    }

    /* read index should not have advanced with nread0 */
    ExpectIntEQ(BIO_nread(bio3, &bufPt, 5), 4);
    for (i = 0; i < 4; i++) {
        ExpectIntEQ(bufPt[i], 0);
    }

    /* write and fill up buffer checking reset of index state */
    ExpectIntEQ(BIO_nwrite(bio1, &bufPt, 20), 20);
    ExpectNotNull(XMEMCPY(bufPt, buff, 20));

    /* test reset on data in bio1 write buffer */
    ExpectIntEQ(BIO_reset(bio1), 1);
    ExpectIntEQ((int)BIO_ctrl_pending(bio3), 0);
    ExpectIntEQ(BIO_nread(bio3, &bufPt, 3), 0);
    ExpectIntEQ(BIO_nwrite(bio1, &bufPt, 20), 20);
    ExpectIntEQ((int)BIO_ctrl(bio1, BIO_CTRL_INFO, 0, &p), 20);
    ExpectNotNull(p);
    ExpectNotNull(XMEMCPY(bufPt, buff, 20));
    ExpectIntEQ(BIO_nread(bio3, &bufPt, 6), 6);
    for (i = 0; i < 6; i++) {
        ExpectIntEQ(bufPt[i], i);
    }

    /* test case of writing twice with offset read index */
    ExpectIntEQ(BIO_nwrite(bio1, &bufPt, 3), 3);
    ExpectIntEQ(BIO_nwrite(bio1, &bufPt, 4), 3); /* try overwriting */
    ExpectIntEQ(BIO_nwrite(bio1, &bufPt, 4), WOLFSSL_BIO_ERROR);
    ExpectIntEQ(BIO_nread(bio3, &bufPt, 0), 0);
    ExpectIntEQ(BIO_nwrite(bio1, &bufPt, 4), WOLFSSL_BIO_ERROR);
    ExpectIntEQ(BIO_nread(bio3, &bufPt, 1), 1);
    ExpectIntEQ(BIO_nwrite(bio1, &bufPt, 4), 1);
    ExpectIntEQ(BIO_nwrite(bio1, &bufPt, 4), WOLFSSL_BIO_ERROR);

    BIO_free(bio1);
    bio1 = NULL;
    BIO_free(bio3);
    bio3 = NULL;

    #if defined(OPENSSL_ALL) || defined(WOLFSSL_ASIO)
    {
        BIO* bioA = NULL;
        BIO* bioB = NULL;
        ExpectIntEQ(BIO_new_bio_pair(NULL, 256, NULL, 256), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
        ExpectIntEQ(BIO_new_bio_pair(&bioA, 256, &bioB, 256), WOLFSSL_SUCCESS);
        BIO_free(bioA);
        bioA = NULL;
        BIO_free(bioB);
        bioB = NULL;
    }
    #endif /* OPENSSL_ALL || WOLFSSL_ASIO */

    /* BIOs with file pointers */
    #if !defined(NO_FILESYSTEM)
    {
        XFILE f1 = XBADFILE;
        XFILE f2 = XBADFILE;
        BIO*  f_bio1 = NULL;
        BIO*  f_bio2 = NULL;
        unsigned char cert[300];
        char testFile[] = "tests/bio_write_test.txt";
        char msg[]      = "bio_write_test.txt contains the first 300 bytes of certs/server-cert.pem\ncreated by tests/unit.test\n\n";

        ExpectNotNull(f_bio1 = BIO_new(BIO_s_file()));
        ExpectNotNull(f_bio2 = BIO_new(BIO_s_file()));

        /* Failure due to wrong BIO type */
        ExpectIntEQ((int)BIO_set_mem_eof_return(f_bio1, -1), 0);
        ExpectIntEQ((int)BIO_set_mem_eof_return(NULL, -1),   0);

        ExpectTrue((f1 = XFOPEN(svrCertFile, "rb+")) != XBADFILE);
        ExpectIntEQ((int)BIO_set_fp(f_bio1, f1, BIO_CLOSE), WOLFSSL_SUCCESS);
        ExpectIntEQ(BIO_write_filename(f_bio2, testFile),
                WOLFSSL_SUCCESS);

        ExpectIntEQ(BIO_read(f_bio1, cert, sizeof(cert)), sizeof(cert));
        ExpectIntEQ(BIO_tell(f_bio1),sizeof(cert));
        ExpectIntEQ(BIO_write(f_bio2, msg, sizeof(msg)), sizeof(msg));
        ExpectIntEQ(BIO_tell(f_bio2),sizeof(msg));
        ExpectIntEQ(BIO_write(f_bio2, cert, sizeof(cert)), sizeof(cert));
        ExpectIntEQ(BIO_tell(f_bio2),sizeof(cert) + sizeof(msg));

        ExpectIntEQ((int)BIO_get_fp(f_bio2, &f2), WOLFSSL_SUCCESS);
        ExpectIntEQ(BIO_reset(f_bio2), 1);
        ExpectIntEQ(BIO_tell(NULL),-1);
        ExpectIntEQ(BIO_tell(f_bio2),0);
        ExpectIntEQ(BIO_seek(f_bio2, 4), 0);
        ExpectIntEQ(BIO_tell(f_bio2),4);

        BIO_free(f_bio1);
        f_bio1 = NULL;
        BIO_free(f_bio2);
        f_bio2 = NULL;

        ExpectNotNull(f_bio1 = BIO_new_file(svrCertFile, "rb+"));
        ExpectIntEQ((int)BIO_set_mem_eof_return(f_bio1, -1), 0);
        ExpectIntEQ(BIO_read(f_bio1, cert, sizeof(cert)), sizeof(cert));
        BIO_free(f_bio1);
        f_bio1 = NULL;
    }
    #endif /* !defined(NO_FILESYSTEM) */

    /* BIO info callback */
    {
        const char* testArg = "test";
        BIO* cb_bio = NULL;
        ExpectNotNull(cb_bio = BIO_new(BIO_s_mem()));

        BIO_set_callback(cb_bio, bioCallback);
        ExpectNotNull(BIO_get_callback(cb_bio));
        BIO_set_callback(cb_bio, NULL);
        ExpectNull(BIO_get_callback(cb_bio));

        BIO_set_callback_arg(cb_bio, (char*)testArg);
        ExpectStrEQ(BIO_get_callback_arg(cb_bio), testArg);
        ExpectNull(BIO_get_callback_arg(NULL));

        BIO_free(cb_bio);
        cb_bio = NULL;
    }

    /* BIO_vfree */
    ExpectNotNull(bio1 = BIO_new(BIO_s_bio()));
    BIO_vfree(NULL);
    BIO_vfree(bio1);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_BIO_BIO_ring_read(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_ALL)
    BIO* bio1 = NULL;
    BIO* bio2 = NULL;
    byte data[50];
    byte tmp[50];

    XMEMSET(data, 42, sizeof(data));


    ExpectIntEQ(BIO_new_bio_pair(&bio1, sizeof(data), &bio2, sizeof(data)),
            SSL_SUCCESS);

    ExpectIntEQ(BIO_write(bio1, data, 40), 40);
    ExpectIntEQ(BIO_read(bio1, tmp, 20), -1);
    ExpectIntEQ(BIO_read(bio2, tmp, 20), 20);
    ExpectBufEQ(tmp, data, 20);
    ExpectIntEQ(BIO_write(bio1, data, 20), 20);
    ExpectIntEQ(BIO_read(bio2, tmp, 40), 40);
    ExpectBufEQ(tmp, data, 40);

    BIO_free(bio1);
    BIO_free(bio2);
#endif
    return EXPECT_RESULT();
}

#endif /* !NO_BIO */


static int test_wolfSSL_a2i_IPADDRESS(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_ALL) && !defined(WOLFSSL_USER_IO)
    const unsigned char* data = NULL;
    int dataSz = 0;
    ASN1_OCTET_STRING *st = NULL;

    const unsigned char ipv4_exp[] = {0x7F, 0, 0, 1};
    const unsigned char ipv6_exp[] = {
        0x20, 0x21, 0x0d, 0xb8, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0xff, 0x00, 0x00, 0x42, 0x77, 0x77
    };
    const unsigned char ipv6_home[] = {
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01
    };

    ExpectNull(st = a2i_IPADDRESS("127.0.0.1bad"));
    ExpectNotNull(st = a2i_IPADDRESS("127.0.0.1"));
    ExpectNotNull(data = ASN1_STRING_get0_data(st));
    ExpectIntEQ(dataSz = ASN1_STRING_length(st), WOLFSSL_IP4_ADDR_LEN);
    ExpectIntEQ(XMEMCMP(data, ipv4_exp, dataSz), 0);
    ASN1_STRING_free(st);
    st = NULL;

    ExpectNotNull(st = a2i_IPADDRESS("::1"));
    ExpectNotNull(data = ASN1_STRING_get0_data(st));
    ExpectIntEQ(dataSz = ASN1_STRING_length(st), WOLFSSL_IP6_ADDR_LEN);
    ExpectIntEQ(XMEMCMP(data, ipv6_home, dataSz), 0);
    ASN1_STRING_free(st);
    st = NULL;

    ExpectNotNull(st = a2i_IPADDRESS("2021:db8::ff00:42:7777"));
    ExpectNotNull(data = ASN1_STRING_get0_data(st));
    ExpectIntEQ(dataSz = ASN1_STRING_length(st), WOLFSSL_IP6_ADDR_LEN);
    ExpectIntEQ(XMEMCMP(data, ipv6_exp, dataSz), 0);
    ASN1_STRING_free(st);
#endif
    return EXPECT_RESULT();
}


static int test_wolfSSL_X509_cmp_time(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_ASN_TIME) \
&& !defined(USER_TIME) && !defined(TIME_OVERRIDES)
    WOLFSSL_ASN1_TIME asn_time;
    time_t t;

    ExpectIntEQ(0, wolfSSL_X509_cmp_time(NULL, &t));
    XMEMSET(&asn_time, 0, sizeof(WOLFSSL_ASN1_TIME));
    ExpectIntEQ(0, wolfSSL_X509_cmp_time(&asn_time, &t));

    ExpectIntEQ(ASN1_TIME_set_string(&asn_time, "000222211515Z"), 1);
    ExpectIntEQ(-1, wolfSSL_X509_cmp_time(&asn_time, NULL));
    ExpectIntEQ(-1, wolfSSL_X509_cmp_current_time(&asn_time));
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_time_adj(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_ASN_TIME) && \
    !defined(USER_TIME) && !defined(TIME_OVERRIDES) && \
    defined(USE_CERT_BUFFERS_2048) && !defined(NO_RSA) && \
    !defined(NO_ASN_TIME)
    X509*  x509 = NULL;
    time_t t;
    time_t not_before;
    time_t not_after;

    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_buffer(
        client_cert_der_2048, sizeof_client_cert_der_2048,
        WOLFSSL_FILETYPE_ASN1));

    t = 0;
    not_before = wc_Time(0);
    not_after = wc_Time(0) + (60 * 24 * 30); /* 30 days after */
    ExpectNotNull(X509_time_adj(X509_get_notBefore(x509), not_before, &t));
    ExpectNotNull(X509_time_adj(X509_get_notAfter(x509), not_after, &t));
    /* Check X509_gmtime_adj, too. */
    ExpectNotNull(X509_gmtime_adj(X509_get_notAfter(x509), not_after));

    X509_free(x509);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_bad_altname(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_RSA)
    const unsigned char malformed_alt_name_cert[] = {
        0x30, 0x82, 0x02, 0xf9, 0x30, 0x82, 0x01, 0xe1, 0xa0, 0x03, 0x02, 0x01,
        0x02, 0x02, 0x02, 0x10, 0x21, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48,
        0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x30, 0x0f, 0x31, 0x0d,
        0x30, 0x0b, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x04, 0x61, 0x61, 0x31,
        0x31, 0x30, 0x1e, 0x17, 0x0d, 0x31, 0x36, 0x30, 0x32, 0x30, 0x37, 0x31,
        0x37, 0x32, 0x34, 0x30, 0x30, 0x5a, 0x17, 0x0d, 0x33, 0x34, 0x30, 0x32,
        0x31, 0x34, 0x30, 0x36, 0x32, 0x36, 0x35, 0x33, 0x5a, 0x30, 0x0f, 0x31,
        0x0d, 0x30, 0x0b, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x04, 0x61, 0x61,
        0x61, 0x61, 0x30, 0x82, 0x01, 0x20, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86,
        0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01,
        0x0d, 0x00, 0x30, 0x82, 0x01, 0x08, 0x02, 0x82, 0x01, 0x01, 0x00, 0xa8,
        0x8a, 0x5e, 0x26, 0x23, 0x1b, 0x31, 0xd3, 0x37, 0x1a, 0x70, 0xb2, 0xec,
        0x3f, 0x74, 0xd4, 0xb4, 0x44, 0xe3, 0x7a, 0xa5, 0xc0, 0xf5, 0xaa, 0x97,
        0x26, 0x9a, 0x04, 0xff, 0xda, 0xbe, 0xe5, 0x09, 0x03, 0x98, 0x3d, 0xb5,
        0xbf, 0x01, 0x2c, 0x9a, 0x0a, 0x3a, 0xfb, 0xbc, 0x3c, 0xe7, 0xbe, 0x83,
        0x5c, 0xb3, 0x70, 0xe8, 0x5c, 0xe3, 0xd1, 0x83, 0xc3, 0x94, 0x08, 0xcd,
        0x1a, 0x87, 0xe5, 0xe0, 0x5b, 0x9c, 0x5c, 0x6e, 0xb0, 0x7d, 0xe2, 0x58,
        0x6c, 0xc3, 0xb5, 0xc8, 0x9d, 0x11, 0xf1, 0x5d, 0x96, 0x0d, 0x66, 0x1e,
        0x56, 0x7f, 0x8f, 0x59, 0xa7, 0xa5, 0xe1, 0xc5, 0xe7, 0x81, 0x4c, 0x09,
        0x9d, 0x5e, 0x96, 0xf0, 0x9a, 0xc2, 0x8b, 0x70, 0xd5, 0xab, 0x79, 0x58,
        0x5d, 0xb7, 0x58, 0xaa, 0xfd, 0x75, 0x52, 0xaa, 0x4b, 0xa7, 0x25, 0x68,
        0x76, 0x59, 0x00, 0xee, 0x78, 0x2b, 0x91, 0xc6, 0x59, 0x91, 0x99, 0x38,
        0x3e, 0xa1, 0x76, 0xc3, 0xf5, 0x23, 0x6b, 0xe6, 0x07, 0xea, 0x63, 0x1c,
        0x97, 0x49, 0xef, 0xa0, 0xfe, 0xfd, 0x13, 0xc9, 0xa9, 0x9f, 0xc2, 0x0b,
        0xe6, 0x87, 0x92, 0x5b, 0xcc, 0xf5, 0x42, 0x95, 0x4a, 0xa4, 0x6d, 0x64,
        0xba, 0x7d, 0xce, 0xcb, 0x04, 0xd0, 0xf8, 0xe7, 0xe3, 0xda, 0x75, 0x60,
        0xd3, 0x8b, 0x6a, 0x64, 0xfc, 0x78, 0x56, 0x21, 0x69, 0x5a, 0xe8, 0xa7,
        0x8f, 0xfb, 0x8f, 0x82, 0xe3, 0xae, 0x36, 0xa2, 0x93, 0x66, 0x92, 0xcb,
        0x82, 0xa3, 0xbe, 0x84, 0x00, 0x86, 0xdc, 0x7e, 0x6d, 0x53, 0x77, 0x84,
        0x17, 0xb9, 0x55, 0x43, 0x0d, 0xf1, 0x16, 0x1f, 0xd5, 0x43, 0x75, 0x99,
        0x66, 0x19, 0x52, 0xd0, 0xac, 0x5f, 0x74, 0xad, 0xb2, 0x90, 0x15, 0x50,
        0x04, 0x74, 0x43, 0xdf, 0x6c, 0x35, 0xd0, 0xfd, 0x32, 0x37, 0xb3, 0x8d,
        0xf5, 0xe5, 0x09, 0x02, 0x01, 0x03, 0xa3, 0x61, 0x30, 0x5f, 0x30, 0x0c,
        0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00,
        0x30, 0x0f, 0x06, 0x03, 0x55, 0x1d, 0x11, 0x04, 0x08, 0x30, 0x06, 0x82,
        0x04, 0x61, 0x2a, 0x00, 0x2a, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e,
        0x04, 0x16, 0x04, 0x14, 0x92, 0x6a, 0x1e, 0x52, 0x3a, 0x1a, 0x57, 0x9f,
        0xc9, 0x82, 0x9a, 0xce, 0xc8, 0xc0, 0xa9, 0x51, 0x9d, 0x2f, 0xc7, 0x72,
        0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80,
        0x14, 0x6b, 0xf9, 0xa4, 0x2d, 0xa5, 0xe9, 0x39, 0x89, 0xa8, 0x24, 0x58,
        0x79, 0x87, 0x11, 0xfc, 0x6f, 0x07, 0x91, 0xef, 0xa6, 0x30, 0x0d, 0x06,
        0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00,
        0x03, 0x82, 0x01, 0x01, 0x00, 0x3f, 0xd5, 0x37, 0x2f, 0xc7, 0xf8, 0x8b,
        0x39, 0x1c, 0xe3, 0xdf, 0x77, 0xee, 0xc6, 0x4b, 0x5f, 0x84, 0xcf, 0xfa,
        0x33, 0x2c, 0xb2, 0xb5, 0x4b, 0x09, 0xee, 0x56, 0xc0, 0xf2, 0xf0, 0xeb,
        0xad, 0x1c, 0x02, 0xef, 0xae, 0x09, 0x53, 0xc0, 0x06, 0xad, 0x4e, 0xfd,
        0x3e, 0x8c, 0x13, 0xb3, 0xbf, 0x80, 0x05, 0x36, 0xb5, 0x3f, 0x2b, 0xc7,
        0x60, 0x53, 0x14, 0xbf, 0x33, 0x63, 0x47, 0xc3, 0xc6, 0x28, 0xda, 0x10,
        0x12, 0xe2, 0xc4, 0xeb, 0xc5, 0x64, 0x66, 0xc0, 0xcc, 0x6b, 0x84, 0xda,
        0x0c, 0xe9, 0xf6, 0xe3, 0xf8, 0x8e, 0x3d, 0x95, 0x5f, 0xba, 0x9f, 0xe1,
        0xc7, 0xed, 0x6e, 0x97, 0xcc, 0xbd, 0x7d, 0xe5, 0x4e, 0xab, 0xbc, 0x1b,
        0xf1, 0x3a, 0x09, 0x33, 0x09, 0xe1, 0xcc, 0xec, 0x21, 0x16, 0x8e, 0xb1,
        0x74, 0x9e, 0xc8, 0x13, 0x7c, 0xdf, 0x07, 0xaa, 0xeb, 0x70, 0xd7, 0x91,
        0x5c, 0xc4, 0xef, 0x83, 0x88, 0xc3, 0xe4, 0x97, 0xfa, 0xe4, 0xdf, 0xd7,
        0x0d, 0xff, 0xba, 0x78, 0x22, 0xfc, 0x3f, 0xdc, 0xd8, 0x02, 0x8d, 0x93,
        0x57, 0xf9, 0x9e, 0x39, 0x3a, 0x77, 0x00, 0xd9, 0x19, 0xaa, 0x68, 0xa1,
        0xe6, 0x9e, 0x13, 0xeb, 0x37, 0x16, 0xf5, 0x77, 0xa4, 0x0b, 0x40, 0x04,
        0xd3, 0xa5, 0x49, 0x78, 0x35, 0xfa, 0x3b, 0xf6, 0x02, 0xab, 0x85, 0xee,
        0xcb, 0x9b, 0x62, 0xda, 0x05, 0x00, 0x22, 0x2f, 0xf8, 0xbd, 0x0b, 0xe5,
        0x2c, 0xb2, 0x53, 0x78, 0x0a, 0xcb, 0x69, 0xc0, 0xb6, 0x9f, 0x96, 0xff,
        0x58, 0x22, 0x70, 0x9c, 0x01, 0x2e, 0x56, 0x60, 0x5d, 0x37, 0xe3, 0x40,
        0x25, 0xc9, 0x90, 0xc8, 0x0f, 0x41, 0x68, 0xb4, 0xfd, 0x10, 0xe2, 0x09,
        0x99, 0x08, 0x5d, 0x7b, 0xc9, 0xe3, 0x29, 0xd4, 0x5a, 0xcf, 0xc9, 0x34,
        0x55, 0xa1, 0x40, 0x44, 0xd6, 0x88, 0x16, 0xbb, 0xdd
    };

    X509* x509 = NULL;
    int certSize = (int)sizeof(malformed_alt_name_cert) / sizeof(unsigned char);
    const char *name = "aaaaa";
    int nameLen = (int)XSTRLEN(name);

    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_buffer(
        malformed_alt_name_cert, certSize, SSL_FILETYPE_ASN1));

    /* malformed_alt_name_cert has a malformed alternative
     * name of "a*\0*". Ensure that it does not match "aaaaa" */
    ExpectIntNE(wolfSSL_X509_check_host(x509, name, nameLen,
        WOLFSSL_ALWAYS_CHECK_SUBJECT, NULL), 1);

    /* Also make sure WOLFSSL_LEFT_MOST_WILDCARD_ONLY fails too */
    ExpectIntNE(wolfSSL_X509_check_host(x509, name, nameLen,
        WOLFSSL_ALWAYS_CHECK_SUBJECT | WOLFSSL_LEFT_MOST_WILDCARD_ONLY,
        NULL), 1);

    X509_free(x509);

#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_name_match(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_RSA)
    /* A certificate with the subject alternative name a* */
    const unsigned char cert_der[] = {
        0x30, 0x82, 0x03, 0xac, 0x30, 0x82, 0x02, 0x94, 0xa0, 0x03, 0x02, 0x01,
        0x02, 0x02, 0x14, 0x0f, 0xa5, 0x10, 0x85, 0xef, 0x58, 0x10, 0x59, 0xfc,
        0x0f, 0x20, 0x1f, 0x53, 0xf5, 0x30, 0x39, 0x34, 0x49, 0x54, 0x05, 0x30,
        0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b,
        0x05, 0x00, 0x30, 0x77, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04,
        0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55,
        0x04, 0x08, 0x0c, 0x07, 0x4d, 0x6f, 0x6e, 0x74, 0x61, 0x6e, 0x61, 0x31,
        0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x07, 0x0c, 0x07, 0x42, 0x6f,
        0x7a, 0x65, 0x6d, 0x61, 0x6e, 0x31, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55,
        0x04, 0x0a, 0x0c, 0x0b, 0x77, 0x6f, 0x6c, 0x66, 0x53, 0x53, 0x4c, 0x20,
        0x49, 0x6e, 0x63, 0x31, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55, 0x04, 0x0b,
        0x0c, 0x0b, 0x45, 0x6e, 0x67, 0x69, 0x6e, 0x65, 0x65, 0x72, 0x69, 0x6e,
        0x67, 0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f,
        0x77, 0x77, 0x77, 0x2e, 0x77, 0x6f, 0x6c, 0x66, 0x73, 0x73, 0x6c, 0x2e,
        0x63, 0x6f, 0x6d, 0x30, 0x1e, 0x17, 0x0d, 0x32, 0x34, 0x30, 0x35, 0x33,
        0x30, 0x32, 0x30, 0x31, 0x35, 0x35, 0x38, 0x5a, 0x17, 0x0d, 0x33, 0x34,
        0x30, 0x35, 0x32, 0x38, 0x32, 0x30, 0x31, 0x35, 0x35, 0x38, 0x5a, 0x30,
        0x77, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02,
        0x55, 0x53, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c,
        0x07, 0x4d, 0x6f, 0x6e, 0x74, 0x61, 0x6e, 0x61, 0x31, 0x10, 0x30, 0x0e,
        0x06, 0x03, 0x55, 0x04, 0x07, 0x0c, 0x07, 0x42, 0x6f, 0x7a, 0x65, 0x6d,
        0x61, 0x6e, 0x31, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c,
        0x0b, 0x77, 0x6f, 0x6c, 0x66, 0x53, 0x53, 0x4c, 0x20, 0x49, 0x6e, 0x63,
        0x31, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x0c, 0x0b, 0x45,
        0x6e, 0x67, 0x69, 0x6e, 0x65, 0x65, 0x72, 0x69, 0x6e, 0x67, 0x31, 0x18,
        0x30, 0x16, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x77, 0x77, 0x77,
        0x2e, 0x77, 0x6f, 0x6c, 0x66, 0x73, 0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d,
        0x30, 0x82, 0x01, 0x22, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86,
        0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0f, 0x00,
        0x30, 0x82, 0x01, 0x0a, 0x02, 0x82, 0x01, 0x01, 0x00, 0xf4, 0xca, 0x3d,
        0xd4, 0xbc, 0x9b, 0xea, 0x74, 0xfe, 0x73, 0xf4, 0x16, 0x23, 0x0b, 0x4a,
        0x09, 0x54, 0xf6, 0x7b, 0x10, 0x99, 0x11, 0x93, 0xb2, 0xdb, 0x4d, 0x7d,
        0x23, 0xab, 0xf9, 0xcd, 0xf6, 0x54, 0xd4, 0xf6, 0x39, 0x57, 0xee, 0x97,
        0xb2, 0xb9, 0xfc, 0x7e, 0x9c, 0xb3, 0xfb, 0x56, 0xb6, 0x84, 0xd6, 0x2d,
        0x59, 0x1c, 0xed, 0xda, 0x9b, 0x19, 0xf5, 0x8a, 0xa7, 0x8a, 0x89, 0xd6,
        0xa1, 0xc0, 0xe6, 0x16, 0xad, 0x04, 0xcf, 0x5a, 0x1f, 0xdf, 0x62, 0x6c,
        0x68, 0x45, 0xe9, 0x55, 0x2e, 0x42, 0xa3, 0x1b, 0x3b, 0x86, 0x23, 0x22,
        0xa1, 0x20, 0x48, 0xd1, 0x52, 0xc0, 0x8b, 0xab, 0xe2, 0x8a, 0x15, 0x68,
        0xbd, 0x89, 0x6f, 0x9f, 0x45, 0x75, 0xb4, 0x27, 0xc1, 0x72, 0x41, 0xfd,
        0x79, 0x89, 0xb0, 0x74, 0xa2, 0xe9, 0x61, 0x48, 0x4c, 0x54, 0xad, 0x6b,
        0x61, 0xbf, 0x0e, 0x27, 0x58, 0xb4, 0xf6, 0x9c, 0x2c, 0x9f, 0xc2, 0x3e,
        0x3b, 0xb3, 0x90, 0x41, 0xbc, 0x61, 0xcd, 0x01, 0x57, 0x90, 0x82, 0xec,
        0x46, 0xba, 0x4f, 0x89, 0x8e, 0x7f, 0x49, 0x4f, 0x46, 0x69, 0x37, 0x8b,
        0xa0, 0xba, 0x85, 0xe8, 0x42, 0xff, 0x9a, 0xa1, 0x53, 0x81, 0x5c, 0xf3,
        0x8e, 0x85, 0x1c, 0xd4, 0x90, 0x60, 0xa0, 0x37, 0x59, 0x04, 0x65, 0xa6,
        0xb5, 0x12, 0x00, 0xc3, 0x04, 0x51, 0xa7, 0x83, 0x96, 0x62, 0x3d, 0x49,
        0x97, 0xe8, 0x6b, 0x9a, 0x5d, 0x51, 0x24, 0xee, 0xad, 0x45, 0x18, 0x0f,
        0x3f, 0x97, 0xec, 0xdf, 0xcf, 0x42, 0x8a, 0x96, 0xc7, 0xd8, 0x82, 0x87,
        0x7f, 0x57, 0x70, 0x22, 0xfb, 0x29, 0x3e, 0x3c, 0xa3, 0xc1, 0xd5, 0x71,
        0xb3, 0x84, 0x06, 0x53, 0xa3, 0x86, 0x20, 0x35, 0xe3, 0x41, 0xb9, 0xd8,
        0x00, 0x22, 0x4f, 0x6d, 0xe6, 0xfd, 0xf0, 0xf4, 0xa2, 0x39, 0x0a, 0x1a,
        0x23, 0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x30, 0x30, 0x2e, 0x30, 0x0d,
        0x06, 0x03, 0x55, 0x1d, 0x11, 0x04, 0x06, 0x30, 0x04, 0x82, 0x02, 0x61,
        0x2a, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14,
        0x45, 0x05, 0xf3, 0x4d, 0x3e, 0x7e, 0x9c, 0xf5, 0x08, 0xee, 0x2c, 0x13,
        0x32, 0xe3, 0xf2, 0x14, 0xe8, 0x0e, 0x71, 0x21, 0x30, 0x0d, 0x06, 0x09,
        0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x03,
        0x82, 0x01, 0x01, 0x00, 0xa8, 0x28, 0xe5, 0x22, 0x65, 0xcf, 0x47, 0xfe,
        0x82, 0x17, 0x99, 0x20, 0xdb, 0xb1, 0x57, 0xd4, 0x91, 0x1a, 0x83, 0xde,
        0xc1, 0xaf, 0xc4, 0x1f, 0xfb, 0xa4, 0x6a, 0xad, 0xdc, 0x58, 0x72, 0xd9,
        0x9b, 0xab, 0xa5, 0xbb, 0xf4, 0x98, 0xd4, 0xdf, 0x36, 0xcb, 0xb5, 0x78,
        0xce, 0x4b, 0x25, 0x5b, 0x24, 0x92, 0xfe, 0xe8, 0xd4, 0xe4, 0xbd, 0x6f,
        0x71, 0x1a, 0x81, 0x2a, 0x6f, 0x35, 0x93, 0xf7, 0xcc, 0xed, 0xe5, 0x06,
        0xd2, 0x96, 0x41, 0xb5, 0xa9, 0x8a, 0xc0, 0xc9, 0x17, 0xe3, 0x13, 0x5e,
        0x94, 0x5e, 0xfa, 0xfc, 0xf0, 0x00, 0x2e, 0xe1, 0xd8, 0x1b, 0x23, 0x3f,
        0x7c, 0x4d, 0x9f, 0xfb, 0xb7, 0x95, 0xc1, 0x94, 0x7f, 0x7f, 0xb5, 0x4f,
        0x93, 0x6d, 0xc3, 0x2b, 0xb2, 0x28, 0x36, 0xd2, 0x7c, 0x01, 0x3c, 0xae,
        0x35, 0xdb, 0xc8, 0x95, 0x1b, 0x5f, 0x6c, 0x0f, 0x57, 0xb3, 0xcc, 0x97,
        0x98, 0x80, 0x06, 0xaa, 0xe4, 0x93, 0x1f, 0xb7, 0xa0, 0x54, 0xf1, 0x4f,
        0x6f, 0x11, 0xdf, 0xab, 0xd3, 0xbf, 0xf0, 0x3a, 0x81, 0x60, 0xaf, 0x7a,
        0xf7, 0x09, 0xd5, 0xae, 0x0c, 0x7d, 0xae, 0x8d, 0x47, 0x06, 0xbe, 0x11,
        0x6e, 0xf8, 0x7e, 0x49, 0xf8, 0xac, 0x24, 0x0a, 0x4b, 0xc2, 0xf6, 0xe8,
        0x2c, 0xec, 0x35, 0xef, 0xa9, 0x13, 0xb8, 0xd2, 0x9c, 0x92, 0x61, 0x91,
        0xec, 0x7b, 0x0c, 0xea, 0x9a, 0x71, 0x36, 0x15, 0x34, 0x2b, 0x7a, 0x25,
        0xac, 0xfe, 0xc7, 0x26, 0x89, 0x70, 0x3e, 0x64, 0x68, 0x97, 0x4b, 0xaa,
        0xc1, 0x24, 0x14, 0xbd, 0x45, 0x2f, 0xe0, 0xfe, 0xf4, 0x2b, 0x8e, 0x08,
        0x3e, 0xe4, 0xb5, 0x3d, 0x5d, 0xf4, 0xc3, 0xd6, 0x9c, 0xb5, 0x33, 0x1b,
        0x3b, 0xda, 0x6e, 0x99, 0x7b, 0x09, 0xd1, 0x30, 0x97, 0x23, 0x52, 0x6d,
        0x1b, 0x71, 0x3a, 0xf4, 0x54, 0xf0, 0xe5, 0x9e
        };

    WOLFSSL_X509* x509 = NULL;
    int certSize = (int)(sizeof(cert_der) / sizeof(unsigned char));
    const char *name1 = "aaaaa";
    int nameLen1 = (int)(XSTRLEN(name1));
    const char *name2 = "a";
    int nameLen2 = (int)(XSTRLEN(name2));
    const char *name3 = "abbbb";
    int nameLen3 = (int)(XSTRLEN(name3));
    const char *name4 = "bbb";
    int nameLen4 = (int)(XSTRLEN(name4));

    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_buffer(
        cert_der, certSize, WOLFSSL_FILETYPE_ASN1));

    /* Ensure that "a*" matches "aaaaa" */
    ExpectIntEQ(wolfSSL_X509_check_host(x509, name1, nameLen1,
        WOLFSSL_ALWAYS_CHECK_SUBJECT, NULL), WOLFSSL_SUCCESS);
    /* Ensure that "a*" matches "a" */
    ExpectIntEQ(wolfSSL_X509_check_host(x509, name2, nameLen2,
        WOLFSSL_ALWAYS_CHECK_SUBJECT, NULL), WOLFSSL_SUCCESS);
    /* Ensure that "a*" matches "abbbb" */
    ExpectIntEQ(wolfSSL_X509_check_host(x509, name3, nameLen3,
        WOLFSSL_ALWAYS_CHECK_SUBJECT, NULL), WOLFSSL_SUCCESS);
    /* Ensure that "a*" does not match "bbb" */
    ExpectIntNE(wolfSSL_X509_check_host(x509, name4, nameLen4,
        WOLFSSL_ALWAYS_CHECK_SUBJECT, NULL), 1);

    /* WOLFSSL_LEFT_MOST_WILDCARD_ONLY flag should fail on all cases, since
     * 'a*' alt name does not have wildcard left-most */

    /* Ensure that "a*" does not match "aaaaa" */
    ExpectIntNE(wolfSSL_X509_check_host(x509, name1, nameLen1,
        WOLFSSL_ALWAYS_CHECK_SUBJECT | WOLFSSL_LEFT_MOST_WILDCARD_ONLY,
        NULL), WOLFSSL_SUCCESS);
    /* Ensure that "a*" does not match "a" */
    ExpectIntNE(wolfSSL_X509_check_host(x509, name2, nameLen2,
        WOLFSSL_ALWAYS_CHECK_SUBJECT | WOLFSSL_LEFT_MOST_WILDCARD_ONLY,
        NULL), WOLFSSL_SUCCESS);
    /* Ensure that "a*" does not match "abbbb" */
    ExpectIntNE(wolfSSL_X509_check_host(x509, name3, nameLen3,
        WOLFSSL_ALWAYS_CHECK_SUBJECT | WOLFSSL_LEFT_MOST_WILDCARD_ONLY,
        NULL), WOLFSSL_SUCCESS);
    /* Ensure that "a*" does not match "bbb" */
    ExpectIntNE(wolfSSL_X509_check_host(x509, name4, nameLen4,
        WOLFSSL_ALWAYS_CHECK_SUBJECT | WOLFSSL_LEFT_MOST_WILDCARD_ONLY,
        NULL), WOLFSSL_SUCCESS);

    wolfSSL_X509_free(x509);

#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_name_match2(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_RSA)
    /* A certificate with the subject alternative name a*b* */
    const unsigned char cert_der[] = {
        0x30, 0x82, 0x03, 0xae, 0x30, 0x82, 0x02, 0x96, 0xa0, 0x03, 0x02, 0x01,
        0x02, 0x02, 0x14, 0x41, 0x8c, 0x8b, 0xaa, 0x0e, 0xd8, 0x5a, 0xc0, 0x52,
        0x46, 0x0e, 0xe5, 0xd8, 0xb9, 0x48, 0x93, 0x7e, 0x8a, 0x7c, 0x65, 0x30,
        0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b,
        0x05, 0x00, 0x30, 0x77, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04,
        0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55,
        0x04, 0x08, 0x0c, 0x07, 0x4d, 0x6f, 0x6e, 0x74, 0x61, 0x6e, 0x61, 0x31,
        0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x07, 0x0c, 0x07, 0x42, 0x6f,
        0x7a, 0x65, 0x6d, 0x61, 0x6e, 0x31, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55,
        0x04, 0x0a, 0x0c, 0x0b, 0x77, 0x6f, 0x6c, 0x66, 0x53, 0x53, 0x4c, 0x20,
        0x49, 0x6e, 0x63, 0x31, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55, 0x04, 0x0b,
        0x0c, 0x0b, 0x45, 0x6e, 0x67, 0x69, 0x6e, 0x65, 0x65, 0x72, 0x69, 0x6e,
        0x67, 0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f,
        0x77, 0x77, 0x77, 0x2e, 0x77, 0x6f, 0x6c, 0x66, 0x73, 0x73, 0x6c, 0x2e,
        0x63, 0x6f, 0x6d, 0x30, 0x1e, 0x17, 0x0d, 0x32, 0x34, 0x30, 0x35, 0x33,
        0x30, 0x32, 0x30, 0x34, 0x33, 0x34, 0x30, 0x5a, 0x17, 0x0d, 0x33, 0x34,
        0x30, 0x35, 0x32, 0x38, 0x32, 0x30, 0x34, 0x33, 0x34, 0x30, 0x5a, 0x30,
        0x77, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02,
        0x55, 0x53, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c,
        0x07, 0x4d, 0x6f, 0x6e, 0x74, 0x61, 0x6e, 0x61, 0x31, 0x10, 0x30, 0x0e,
        0x06, 0x03, 0x55, 0x04, 0x07, 0x0c, 0x07, 0x42, 0x6f, 0x7a, 0x65, 0x6d,
        0x61, 0x6e, 0x31, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c,
        0x0b, 0x77, 0x6f, 0x6c, 0x66, 0x53, 0x53, 0x4c, 0x20, 0x49, 0x6e, 0x63,
        0x31, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x0c, 0x0b, 0x45,
        0x6e, 0x67, 0x69, 0x6e, 0x65, 0x65, 0x72, 0x69, 0x6e, 0x67, 0x31, 0x18,
        0x30, 0x16, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x77, 0x77, 0x77,
        0x2e, 0x77, 0x6f, 0x6c, 0x66, 0x73, 0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d,
        0x30, 0x82, 0x01, 0x22, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86,
        0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0f, 0x00,
        0x30, 0x82, 0x01, 0x0a, 0x02, 0x82, 0x01, 0x01, 0x00, 0xa5, 0x60, 0x80,
        0xf3, 0xee, 0x19, 0xd2, 0xe4, 0x15, 0x94, 0x54, 0x12, 0x88, 0xee, 0xda,
        0x11, 0x11, 0x87, 0x99, 0x88, 0xb3, 0x71, 0xc7, 0x97, 0x78, 0x1b, 0x57,
        0x37, 0x1d, 0x0b, 0x1f, 0x2f, 0x2c, 0x35, 0x13, 0x75, 0xd3, 0x31, 0x3e,
        0x6f, 0x80, 0x21, 0xa5, 0xa3, 0xad, 0x10, 0x81, 0xb6, 0x37, 0xd4, 0x55,
        0x2e, 0xc1, 0xb8, 0x37, 0xa3, 0x3c, 0xe8, 0x81, 0x03, 0x3c, 0xda, 0x5f,
        0x6f, 0x45, 0x32, 0x2b, 0x0e, 0x99, 0x27, 0xfd, 0xe5, 0x6c, 0x07, 0xd9,
        0x4e, 0x0a, 0x8b, 0x23, 0x74, 0x96, 0x25, 0x97, 0xae, 0x6d, 0x19, 0xba,
        0xbf, 0x0f, 0xc8, 0xa1, 0xe5, 0xea, 0xa8, 0x00, 0x09, 0xc3, 0x9a, 0xef,
        0x09, 0x33, 0xc1, 0x33, 0x2e, 0x7b, 0x6d, 0xa7, 0x66, 0x87, 0xb6, 0x3a,
        0xb9, 0xdb, 0x4c, 0x5e, 0xb5, 0x55, 0x69, 0x37, 0x17, 0x92, 0x1f, 0xe3,
        0x53, 0x1a, 0x2d, 0x25, 0xd0, 0xcf, 0x72, 0x37, 0xc2, 0x89, 0x83, 0x78,
        0xcf, 0xac, 0x2e, 0x46, 0x92, 0x5c, 0x4a, 0xba, 0x7d, 0xa0, 0x22, 0x34,
        0xb1, 0x22, 0x26, 0x99, 0xda, 0xe8, 0x97, 0xe2, 0x0c, 0xd3, 0xbc, 0x97,
        0x7e, 0xa8, 0xb9, 0xe3, 0xe2, 0x7f, 0x56, 0xef, 0x22, 0xee, 0x15, 0x95,
        0xa6, 0xd1, 0xf4, 0xa7, 0xac, 0x4a, 0xab, 0xc1, 0x1a, 0xda, 0xc5, 0x5f,
        0xa5, 0x5e, 0x2f, 0x15, 0x9c, 0x36, 0xbe, 0xd3, 0x47, 0xb6, 0x86, 0xb9,
        0xc6, 0x59, 0x39, 0x36, 0xad, 0x84, 0x53, 0x95, 0x72, 0x91, 0x89, 0x51,
        0x32, 0x77, 0xf1, 0xa5, 0x93, 0xfe, 0xf0, 0x41, 0x7c, 0x64, 0xf1, 0xb0,
        0x8b, 0x81, 0x8d, 0x3a, 0x2c, 0x9e, 0xbe, 0x2e, 0x8b, 0xf7, 0x80, 0x63,
        0x35, 0x32, 0xfa, 0x26, 0xe0, 0x63, 0xbf, 0x5e, 0xaf, 0xf0, 0x08, 0xe0,
        0x80, 0x65, 0x38, 0xfa, 0x21, 0xaa, 0x91, 0x34, 0x48, 0x3d, 0x32, 0x5c,
        0xbf, 0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x32, 0x30, 0x30, 0x30, 0x0f,
        0x06, 0x03, 0x55, 0x1d, 0x11, 0x04, 0x08, 0x30, 0x06, 0x82, 0x04, 0x61,
        0x2a, 0x62, 0x2a, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16,
        0x04, 0x14, 0x3d, 0x55, 0x74, 0xf8, 0x3a, 0x26, 0x03, 0x8c, 0x6a, 0x2e,
        0x91, 0x0e, 0x18, 0x70, 0xb4, 0xa4, 0xcc, 0x04, 0x00, 0xd3, 0x30, 0x0d,
        0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05,
        0x00, 0x03, 0x82, 0x01, 0x01, 0x00, 0x8f, 0x3b, 0xff, 0x46, 0x0c, 0xb5,
        0x21, 0xdc, 0xcf, 0x61, 0x9a, 0x25, 0x93, 0x99, 0x68, 0x2f, 0x16, 0x71,
        0x15, 0x00, 0x5f, 0xb0, 0x9b, 0x43, 0x5c, 0x47, 0xe2, 0x8e, 0xc8, 0xea,
        0xb3, 0x30, 0x4d, 0x87, 0x90, 0xcf, 0x24, 0x37, 0x5c, 0xfd, 0xc8, 0xc6,
        0x09, 0x36, 0xb2, 0xfb, 0xfd, 0xc1, 0x82, 0x92, 0x77, 0x5b, 0x9d, 0xeb,
        0xac, 0x47, 0xbc, 0xda, 0x7c, 0x89, 0x19, 0x03, 0x9e, 0xcd, 0x96, 0x2a,
        0x90, 0x55, 0x23, 0x19, 0xac, 0x9d, 0x49, 0xfb, 0xa0, 0x31, 0x7d, 0x6b,
        0x1a, 0x16, 0x13, 0xb1, 0xa9, 0xc9, 0xc4, 0xaf, 0xf1, 0xb4, 0xa7, 0x9b,
        0x08, 0x64, 0x6a, 0x09, 0xcd, 0x4a, 0x03, 0x4c, 0x93, 0xb6, 0xcf, 0x29,
        0xdb, 0x56, 0x88, 0x8e, 0xed, 0x08, 0x6d, 0x8d, 0x76, 0xa3, 0xd7, 0xc6,
        0x69, 0xa1, 0xf5, 0xd2, 0xd0, 0x0a, 0x4b, 0xfa, 0x88, 0x66, 0x6c, 0xe5,
        0x4a, 0xee, 0x13, 0xad, 0xad, 0x22, 0x25, 0x73, 0x39, 0x56, 0x74, 0x0e,
        0xda, 0xcd, 0x35, 0x67, 0xe3, 0x81, 0x5c, 0xc5, 0xae, 0x3c, 0x4f, 0x47,
        0x3e, 0x97, 0xde, 0xac, 0xf6, 0xe1, 0x26, 0xe2, 0xe0, 0x66, 0x48, 0x20,
        0x7c, 0x02, 0x81, 0x3e, 0x7d, 0x34, 0xb7, 0x73, 0x3e, 0x2e, 0xd6, 0x20,
        0x1c, 0xdf, 0xf1, 0xae, 0x86, 0x8b, 0xb2, 0xc2, 0x9b, 0x68, 0x9c, 0xf6,
        0x1a, 0x5e, 0x30, 0x06, 0x39, 0x0a, 0x1f, 0x7b, 0xd7, 0x18, 0x4b, 0x06,
        0x9d, 0xff, 0x84, 0x57, 0xcc, 0x92, 0xad, 0x81, 0x0a, 0x19, 0x11, 0xc4,
        0xac, 0x59, 0x00, 0xe8, 0x5a, 0x70, 0x78, 0xd6, 0x9f, 0xe0, 0x82, 0x2a,
        0x1f, 0x09, 0x36, 0x1c, 0x52, 0x98, 0xf7, 0x95, 0x8f, 0xf9, 0x48, 0x4f,
        0x30, 0x52, 0xb5, 0xf3, 0x8d, 0x13, 0x93, 0x27, 0xbe, 0xb4, 0x75, 0x39,
        0x65, 0xc6, 0x48, 0x4e, 0x32, 0xd7, 0xf4, 0xc3, 0x26, 0x8d
        };

    WOLFSSL_X509* x509 = NULL;
    int certSize = (int)(sizeof(cert_der) / sizeof(unsigned char));
    const char *name1 = "ab";
    int nameLen1 = (int)(XSTRLEN(name1));
    const char *name2 = "acccbccc";
    int nameLen2 = (int)(XSTRLEN(name2));
    const char *name3 = "accb";
    int nameLen3 = (int)(XSTRLEN(name3));
    const char *name4 = "accda";
    int nameLen4 = (int)(XSTRLEN(name4));
    const char *name5 = "acc\0bcc";
    int nameLen5 = 7;

    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_buffer(
        cert_der, certSize, WOLFSSL_FILETYPE_ASN1));

    /* Ensure that "a*b*" matches "ab" */
    ExpectIntEQ(wolfSSL_X509_check_host(x509, name1, nameLen1,
        WOLFSSL_ALWAYS_CHECK_SUBJECT, NULL), WOLFSSL_SUCCESS);
    /* Ensure that "a*b*" matches "acccbccc" */
    ExpectIntEQ(wolfSSL_X509_check_host(x509, name2, nameLen2,
        WOLFSSL_ALWAYS_CHECK_SUBJECT, NULL), WOLFSSL_SUCCESS);
    /* Ensure that "a*b*" matches "accb" */
    ExpectIntEQ(wolfSSL_X509_check_host(x509, name3, nameLen3,
        WOLFSSL_ALWAYS_CHECK_SUBJECT, NULL), WOLFSSL_SUCCESS);
    /* Ensure that "a*b*" does not match "accda" */
    ExpectIntNE(wolfSSL_X509_check_host(x509, name4, nameLen4,
        WOLFSSL_ALWAYS_CHECK_SUBJECT, NULL), WOLFSSL_SUCCESS);

    /* WOLFSSL_LEFT_MOST_WILDCARD_ONLY flag should fail on all cases, since
     * 'a*b*' alt name does not have wildcard left-most */
    ExpectIntEQ(wolfSSL_X509_check_host(x509, name1, nameLen1,
        WOLFSSL_ALWAYS_CHECK_SUBJECT | WOLFSSL_LEFT_MOST_WILDCARD_ONLY,
        NULL), WOLFSSL_FAILURE);
    ExpectIntEQ(wolfSSL_X509_check_host(x509, name2, nameLen2,
        WOLFSSL_ALWAYS_CHECK_SUBJECT | WOLFSSL_LEFT_MOST_WILDCARD_ONLY,
        NULL), WOLFSSL_FAILURE);
    ExpectIntEQ(wolfSSL_X509_check_host(x509, name3, nameLen3,
        WOLFSSL_ALWAYS_CHECK_SUBJECT | WOLFSSL_LEFT_MOST_WILDCARD_ONLY,
        NULL), WOLFSSL_FAILURE);
    ExpectIntEQ(wolfSSL_X509_check_host(x509, name4, nameLen4,
        WOLFSSL_ALWAYS_CHECK_SUBJECT | WOLFSSL_LEFT_MOST_WILDCARD_ONLY,
        NULL), WOLFSSL_FAILURE);

    /* Ensure that "a*b*" matches "ab", testing openssl behavior replication
     * on check len input handling, 0 for len is OK as it should then use
     * strlen(name1) */
    ExpectIntEQ(wolfSSL_X509_check_host(x509, name1, 0,
        WOLFSSL_ALWAYS_CHECK_SUBJECT, NULL), WOLFSSL_SUCCESS);
    /* Openssl also allows for len to include NULL terminator */
    ExpectIntEQ(wolfSSL_X509_check_host(x509, name1, nameLen1 + 1,
        WOLFSSL_ALWAYS_CHECK_SUBJECT, NULL), WOLFSSL_SUCCESS);
    /* Ensure that check string with NULL terminator in middle is
     * rejected */
    ExpectIntNE(wolfSSL_X509_check_host(x509, name5, nameLen5,
        WOLFSSL_ALWAYS_CHECK_SUBJECT, NULL), WOLFSSL_SUCCESS);

    wolfSSL_X509_free(x509);

#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_name_match3(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_RSA)
    /* A certificate with the subject alternative name *.example.com */
    const unsigned char cert_der[] = {
        0x30, 0x82, 0x03, 0xb7, 0x30, 0x82, 0x02, 0x9f, 0xa0, 0x03, 0x02, 0x01,
        0x02, 0x02, 0x14, 0x59, 0xbb, 0xf6, 0xde, 0xb8, 0x3d, 0x0e, 0x8c, 0xe4,
        0xbd, 0x98, 0xa3, 0xbe, 0x3e, 0x8f, 0xdc, 0xbd, 0x7f, 0xcc, 0xae, 0x30,
        0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b,
        0x05, 0x00, 0x30, 0x77, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04,
        0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55,
        0x04, 0x08, 0x0c, 0x07, 0x4d, 0x6f, 0x6e, 0x74, 0x61, 0x6e, 0x61, 0x31,
        0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x07, 0x0c, 0x07, 0x42, 0x6f,
        0x7a, 0x65, 0x6d, 0x61, 0x6e, 0x31, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55,
        0x04, 0x0a, 0x0c, 0x0b, 0x77, 0x6f, 0x6c, 0x66, 0x53, 0x53, 0x4c, 0x20,
        0x49, 0x6e, 0x63, 0x31, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55, 0x04, 0x0b,
        0x0c, 0x0b, 0x45, 0x6e, 0x67, 0x69, 0x6e, 0x65, 0x65, 0x72, 0x69, 0x6e,
        0x67, 0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f,
        0x77, 0x77, 0x77, 0x2e, 0x77, 0x6f, 0x6c, 0x66, 0x73, 0x73, 0x6c, 0x2e,
        0x63, 0x6f, 0x6d, 0x30, 0x1e, 0x17, 0x0d, 0x32, 0x34, 0x30, 0x35, 0x33,
        0x31, 0x30, 0x30, 0x33, 0x37, 0x34, 0x39, 0x5a, 0x17, 0x0d, 0x33, 0x34,
        0x30, 0x35, 0x32, 0x39, 0x30, 0x30, 0x33, 0x37, 0x34, 0x39, 0x5a, 0x30,
        0x77, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02,
        0x55, 0x53, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c,
        0x07, 0x4d, 0x6f, 0x6e, 0x74, 0x61, 0x6e, 0x61, 0x31, 0x10, 0x30, 0x0e,
        0x06, 0x03, 0x55, 0x04, 0x07, 0x0c, 0x07, 0x42, 0x6f, 0x7a, 0x65, 0x6d,
        0x61, 0x6e, 0x31, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c,
        0x0b, 0x77, 0x6f, 0x6c, 0x66, 0x53, 0x53, 0x4c, 0x20, 0x49, 0x6e, 0x63,
        0x31, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x0c, 0x0b, 0x45,
        0x6e, 0x67, 0x69, 0x6e, 0x65, 0x65, 0x72, 0x69, 0x6e, 0x67, 0x31, 0x18,
        0x30, 0x16, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x77, 0x77, 0x77,
        0x2e, 0x77, 0x6f, 0x6c, 0x66, 0x73, 0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d,
        0x30, 0x82, 0x01, 0x22, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86,
        0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0f, 0x00,
        0x30, 0x82, 0x01, 0x0a, 0x02, 0x82, 0x01, 0x01, 0x00, 0xda, 0x78, 0x16,
        0x05, 0x65, 0xf2, 0x85, 0xf2, 0x61, 0x7f, 0xb1, 0x4d, 0x73, 0xe2, 0x82,
        0xb5, 0x3d, 0xf7, 0x9d, 0x05, 0x65, 0xed, 0x9d, 0xc3, 0x29, 0x7a, 0x92,
        0x2c, 0x06, 0x5f, 0xc8, 0x13, 0x55, 0x42, 0x4e, 0xbd, 0xe2, 0x56, 0x2a,
        0x4b, 0xac, 0xe6, 0x1b, 0x10, 0xc9, 0xdb, 0x9a, 0x45, 0x36, 0xed, 0xf3,
        0x26, 0x8c, 0x22, 0x88, 0x1e, 0x6d, 0x2b, 0x41, 0xfa, 0x0d, 0x43, 0x88,
        0x88, 0xde, 0x8d, 0x2e, 0xca, 0x6e, 0x7c, 0x62, 0x66, 0x3e, 0xfa, 0x4e,
        0x71, 0xea, 0x7d, 0x3b, 0x32, 0x33, 0x5c, 0x7a, 0x7e, 0xea, 0x74, 0xbd,
        0xb6, 0x8f, 0x4c, 0x1c, 0x7a, 0x79, 0x94, 0xf1, 0xe8, 0x02, 0x67, 0x98,
        0x25, 0xb4, 0x31, 0x80, 0xc1, 0xae, 0xbf, 0xef, 0xf2, 0x6c, 0x78, 0x42,
        0xef, 0xb5, 0xc6, 0x01, 0x47, 0x79, 0x8d, 0x92, 0xce, 0xc1, 0xb5, 0x98,
        0x76, 0xf0, 0x84, 0xa2, 0x53, 0x90, 0xe5, 0x39, 0xc7, 0xbd, 0xf2, 0xbb,
        0xe3, 0x3f, 0x00, 0xf6, 0xf0, 0x46, 0x86, 0xee, 0x55, 0xbd, 0x2c, 0x1f,
        0x97, 0x24, 0x7c, 0xbc, 0xda, 0x2f, 0x1b, 0x53, 0xef, 0x26, 0x56, 0xcc,
        0xb7, 0xd8, 0xca, 0x17, 0x20, 0x4e, 0x62, 0x03, 0x66, 0x32, 0xb3, 0xd1,
        0x71, 0x26, 0x6c, 0xff, 0xd1, 0x9e, 0x44, 0x86, 0x2a, 0xae, 0xba, 0x43,
        0x00, 0x13, 0x7e, 0x50, 0xdd, 0x3e, 0x27, 0x39, 0x70, 0x1c, 0x0c, 0x0b,
        0xe8, 0xa2, 0xae, 0x03, 0x09, 0x2e, 0xd8, 0x71, 0xee, 0x7b, 0x1a, 0x09,
        0x2d, 0xe1, 0xd5, 0xde, 0xf5, 0xa3, 0x36, 0x77, 0x90, 0x97, 0x99, 0xd7,
        0x6c, 0xb7, 0x5c, 0x9d, 0xf7, 0x7e, 0x41, 0x89, 0xfe, 0xe4, 0x08, 0xc6,
        0x0b, 0xe4, 0x9b, 0x5f, 0x51, 0xa6, 0x08, 0xb8, 0x99, 0x81, 0xe9, 0xce,
        0xb4, 0x2d, 0xb2, 0x92, 0x9f, 0xe5, 0x1a, 0x98, 0x76, 0x20, 0x70, 0x54,
        0x93, 0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x3b, 0x30, 0x39, 0x30, 0x18,
        0x06, 0x03, 0x55, 0x1d, 0x11, 0x04, 0x11, 0x30, 0x0f, 0x82, 0x0d, 0x2a,
        0x2e, 0x65, 0x78, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x2e, 0x63, 0x6f, 0x6d,
        0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x60,
        0xd4, 0x26, 0xbb, 0xcc, 0x7c, 0x29, 0xa2, 0x88, 0x3c, 0x76, 0x7d, 0xb4,
        0x86, 0x8b, 0x47, 0x64, 0x5b, 0x87, 0xe0, 0x30, 0x0d, 0x06, 0x09, 0x2a,
        0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x03, 0x82,
        0x01, 0x01, 0x00, 0xc3, 0x0d, 0x03, 0x67, 0xbb, 0x47, 0x8b, 0xf3, 0x20,
        0xdc, 0x7d, 0x2e, 0xe1, 0xd9, 0xf0, 0x01, 0xc4, 0x66, 0xc2, 0xe1, 0xcd,
        0xc3, 0x4a, 0x72, 0xf0, 0x6e, 0x38, 0xcf, 0x63, 0x01, 0x96, 0x9e, 0x84,
        0xb9, 0xce, 0x1d, 0xba, 0x4b, 0xe0, 0x70, 0x86, 0x2b, 0x5a, 0xab, 0xec,
        0xbf, 0xc2, 0xaa, 0x64, 0xa2, 0x6c, 0xd2, 0x42, 0x52, 0xd4, 0xbe, 0x8a,
        0xca, 0x9c, 0x03, 0xf3, 0xd6, 0x5f, 0xcd, 0x23, 0x9f, 0xf5, 0xa9, 0x04,
        0x40, 0x5b, 0x66, 0x78, 0xc0, 0xac, 0xa1, 0xdb, 0x5d, 0xd1, 0x94, 0xfc,
        0x47, 0x94, 0xf5, 0x45, 0xe3, 0x70, 0x13, 0x3f, 0x66, 0x6d, 0xdd, 0x73,
        0x68, 0x68, 0xe2, 0xd2, 0x89, 0xcb, 0x7f, 0xc6, 0xca, 0xd6, 0x96, 0x0b,
        0xcc, 0xdd, 0xa1, 0x74, 0xda, 0x33, 0xe8, 0x9e, 0xda, 0xb7, 0xd9, 0x12,
        0xab, 0x85, 0x9d, 0x0c, 0xde, 0xa0, 0x7d, 0x7e, 0xa1, 0x91, 0xed, 0xe5,
        0x32, 0x7c, 0xc5, 0xea, 0x1d, 0x4a, 0xb5, 0x38, 0x63, 0x17, 0xf3, 0x4f,
        0x2c, 0x4a, 0x58, 0x86, 0x09, 0x33, 0x86, 0xc4, 0xe7, 0x56, 0x6f, 0x32,
        0x71, 0xb7, 0xd0, 0x83, 0x12, 0x9e, 0x26, 0x0a, 0x3a, 0x45, 0xcb, 0xd7,
        0x4e, 0xab, 0xa4, 0xc3, 0xee, 0x4c, 0xc0, 0x38, 0xa1, 0xfa, 0xba, 0xfa,
        0xb7, 0x80, 0x69, 0x67, 0xa3, 0xef, 0x89, 0xba, 0xce, 0x89, 0x91, 0x3d,
        0x6a, 0x76, 0xe9, 0x3b, 0x32, 0x86, 0x76, 0x85, 0x6b, 0x4f, 0x7f, 0xbc,
        0x7a, 0x5b, 0x31, 0x92, 0x79, 0x35, 0xf8, 0xb9, 0xb1, 0xd7, 0xdb, 0xa9,
        0x6a, 0x8a, 0x91, 0x60, 0x65, 0xd4, 0x76, 0x54, 0x55, 0x57, 0xb9, 0x35,
        0xe0, 0xf5, 0xbb, 0x8f, 0xd4, 0x40, 0x75, 0xbb, 0x47, 0xa8, 0xf9, 0x0f,
        0xea, 0xc9, 0x6e, 0x84, 0xd5, 0xf5, 0x58, 0x2d, 0xe5, 0x76, 0x7b, 0xdf,
        0x97, 0x05, 0x5e, 0xaf, 0x50, 0xf5, 0x48
        };

    WOLFSSL_X509* x509 = NULL;
    int certSize = (int)(sizeof(cert_der) / sizeof(unsigned char));
    const char *name1 = "foo.example.com";
    int nameLen1 = (int)(XSTRLEN(name1));
    const char *name2 = "x.y.example.com";
    int nameLen2 = (int)(XSTRLEN(name2));
    const char *name3 = "example.com";
    int nameLen3 = (int)(XSTRLEN(name3));

    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_buffer(
        cert_der, certSize, WOLFSSL_FILETYPE_ASN1));

    /* Ensure that "*.example.com" matches "foo.example.com" */
    ExpectIntEQ(wolfSSL_X509_check_host(x509, name1, nameLen1,
        WOLFSSL_ALWAYS_CHECK_SUBJECT, NULL), WOLFSSL_SUCCESS);
    /* Ensure that "*.example.com" does NOT match "x.y.example.com" */
    ExpectIntNE(wolfSSL_X509_check_host(x509, name2, nameLen2,
        WOLFSSL_ALWAYS_CHECK_SUBJECT, NULL), WOLFSSL_SUCCESS);
    /* Ensure that "*.example.com" does NOT match "example.com" */
    ExpectIntNE(wolfSSL_X509_check_host(x509, name3, nameLen3,
        WOLFSSL_ALWAYS_CHECK_SUBJECT, NULL), WOLFSSL_SUCCESS);

    /* WOLFSSL_LEFT_MOST_WILDCARD_ONLY, should match "foo.example.com" */
    ExpectIntEQ(wolfSSL_X509_check_host(x509, name1, nameLen1,
        WOLFSSL_ALWAYS_CHECK_SUBJECT | WOLFSSL_LEFT_MOST_WILDCARD_ONLY,
        NULL), WOLFSSL_SUCCESS);
    /* WOLFSSL_LEFT_MOST_WILDCARD_ONLY, should NOT  match "x.y.example.com" */
    ExpectIntNE(wolfSSL_X509_check_host(x509, name2, nameLen2,
        WOLFSSL_ALWAYS_CHECK_SUBJECT | WOLFSSL_LEFT_MOST_WILDCARD_ONLY,
        NULL), WOLFSSL_SUCCESS);
    /* WOLFSSL_LEFT_MOST_WILDCARD_ONLY, should NOT match "example.com" */
    ExpectIntNE(wolfSSL_X509_check_host(x509, name3, nameLen3,
        WOLFSSL_ALWAYS_CHECK_SUBJECT | WOLFSSL_LEFT_MOST_WILDCARD_ONLY,
        NULL), WOLFSSL_SUCCESS);

    wolfSSL_X509_free(x509);

#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_max_altnames(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    !defined(NO_RSA)

    /* Only test if max alt names has not been modified */
#if WOLFSSL_MAX_ALT_NAMES <= 1024

    WOLFSSL_CTX* ctx = NULL;
    /* File contains a certificate encoded with 130 subject alternative names */
    const char* over_max_altnames_cert = \
        "./certs/test/cert-over-max-altnames.pem";

#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#else
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
#endif

    ExpectIntNE(wolfSSL_CTX_load_verify_locations_ex(ctx,
            over_max_altnames_cert, NULL, WOLFSSL_LOAD_FLAG_NONE),
            WOLFSSL_SUCCESS);
    wolfSSL_CTX_free(ctx);
#endif
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_max_name_constraints(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    !defined(NO_RSA) && !defined(IGNORE_NAME_CONSTRAINTS)

    /* Only test if max name constraints has not been modified */
#if WOLFSSL_MAX_NAME_CONSTRAINTS == 128

    WOLFSSL_CTX* ctx = NULL;
    /* File contains a certificate with 130 name constraints */
    const char* over_max_nc = "./certs/test/cert-over-max-nc.pem";

#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#else
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
#endif

    ExpectIntNE(wolfSSL_CTX_load_verify_locations_ex(ctx, over_max_nc,
            NULL, WOLFSSL_LOAD_FLAG_NONE), WOLFSSL_SUCCESS);
    wolfSSL_CTX_free(ctx);
#endif

#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_FILESYSTEM) && \
    !defined(NO_RSA)
    X509* x509 = NULL;
#ifndef NO_BIO
    BIO*  bio = NULL;
    X509_STORE_CTX* ctx = NULL;
    X509_STORE* store = NULL;
#endif
    char der[] = "certs/ca-cert.der";
    XFILE fp = XBADFILE;
    int derSz = 0;

#ifndef NO_BIO
    ExpectNotNull(bio = BIO_new(BIO_s_mem()));
#endif

    ExpectNotNull(x509 = X509_new());
    ExpectNull(wolfSSL_X509_get_der(x509, &derSz));
#if !defined(NO_BIO) && defined(WOLFSSL_CERT_GEN)
    ExpectIntEQ(i2d_X509_bio(bio, x509), WOLFSSL_FAILURE);
#endif
    ExpectNull(wolfSSL_X509_dup(x509));
    X509_free(x509);
    x509 = NULL;

#ifndef NO_BIO
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(cliCertFile,
        SSL_FILETYPE_PEM));

#ifdef WOLFSSL_CERT_GEN
    ExpectIntEQ(i2d_X509_bio(NULL, NULL), WOLFSSL_FAILURE);
    ExpectIntEQ(i2d_X509_bio(bio, NULL), WOLFSSL_FAILURE);
    ExpectIntEQ(i2d_X509_bio(NULL, x509), WOLFSSL_FAILURE);
    ExpectIntEQ(i2d_X509_bio(bio, x509), SSL_SUCCESS);
#endif

    ExpectNotNull(ctx = X509_STORE_CTX_new());

    ExpectIntEQ(X509_verify_cert(ctx), WC_NO_ERR_TRACE(WOLFSSL_FATAL_ERROR));
    ExpectNotNull(wolfSSL_X509_verify_cert_error_string(CRL_MISSING));

    ExpectNotNull(store = X509_STORE_new());
    ExpectIntEQ(X509_STORE_add_cert(store, x509), SSL_SUCCESS);
    ExpectIntEQ(X509_STORE_CTX_init(ctx, store, x509, NULL), SSL_SUCCESS);
    ExpectIntEQ(X509_verify_cert(ctx), SSL_SUCCESS);

#ifndef NO_WOLFSSL_STUB
    ExpectNull(X509_get_default_cert_file_env());
    ExpectNull(X509_get_default_cert_file());
    ExpectNull(X509_get_default_cert_dir_env());
    ExpectNull(X509_get_default_cert_dir());
#endif

    ExpectNull(wolfSSL_X509_get_der(NULL, NULL));
    ExpectNull(wolfSSL_X509_get_der(x509, NULL));
    ExpectNull(wolfSSL_X509_get_der(NULL, &derSz));

    ExpectIntEQ(wolfSSL_X509_version(NULL), 0);
    ExpectIntEQ(wolfSSL_X509_version(x509), 3);

    X509_STORE_CTX_free(ctx);
    X509_STORE_free(store);
    X509_free(x509);
    x509 = NULL;
    BIO_free(bio);
    bio = NULL;
#endif

    /** d2i_X509_fp test **/
    ExpectTrue((fp = XFOPEN(der, "rb")) != XBADFILE);
    ExpectNotNull(x509 = (X509 *)d2i_X509_fp(fp, (X509 **)NULL));
    ExpectNotNull(x509);

#ifdef HAVE_EX_DATA_CRYPTO
    ExpectIntEQ(wolfSSL_X509_get_ex_new_index(1, NULL, NULL, NULL, NULL), 0);
#endif
    ExpectNull(wolfSSL_X509_get_ex_data(NULL, 1));
    ExpectNull(wolfSSL_X509_get_ex_data(x509, 1));
#ifdef HAVE_EX_DATA
    ExpectIntEQ(wolfSSL_X509_set_ex_data(NULL, 1, der), 0);
    ExpectIntEQ(wolfSSL_X509_set_ex_data(x509, 1, der), 1);
    ExpectPtrEq(wolfSSL_X509_get_ex_data(x509, 1), der);
#else
    ExpectIntEQ(wolfSSL_X509_set_ex_data(NULL, 1, der), 0);
    ExpectIntEQ(wolfSSL_X509_set_ex_data(x509, 1, der), 0);
    ExpectNull(wolfSSL_X509_get_ex_data(x509, 1));
#endif

    X509_free(x509);
    x509 = NULL;
    if (fp != XBADFILE) {
        XFCLOSE(fp);
        fp = XBADFILE;
    }
    ExpectTrue((fp = XFOPEN(der, "rb")) != XBADFILE);
    ExpectNull((X509 *)d2i_X509_fp(XBADFILE, (X509 **)&x509));
    ExpectNotNull((X509 *)d2i_X509_fp(fp, (X509 **)&x509));
    ExpectNotNull(x509);
    X509_free(x509);
    x509 = NULL;
    if (fp != XBADFILE)
        XFCLOSE(fp);

#ifndef NO_BIO
    ExpectNotNull(bio = BIO_new_file(der, "rb"));
    ExpectNull(d2i_X509_bio(NULL, &x509));
    ExpectNotNull(x509 = d2i_X509_bio(bio, NULL));
    ExpectNotNull(x509);
    X509_free(x509);
    BIO_free(bio);
    bio = NULL;
#endif

    /* X509_up_ref test */
    ExpectIntEQ(X509_up_ref(NULL), 0);
    ExpectNotNull(x509 = X509_new());   /* refCount = 1 */
    ExpectIntEQ(X509_up_ref(x509), 1);  /* refCount = 2 */
    ExpectIntEQ(X509_up_ref(x509), 1);  /* refCount = 3 */
    X509_free(x509); /* refCount = 2 */
    X509_free(x509); /* refCount = 1 */
    X509_free(x509); /* refCount = 0, free */

#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_get_ext_count(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_ALL) && !defined(NO_CERTS) && !defined(NO_FILESYSTEM) && \
    !defined(NO_RSA)
    int ret = 0;
    WOLFSSL_X509* x509 = NULL;
    const char ocspRootCaFile[] = "./certs/ocsp/root-ca-cert.pem";
    XFILE f = XBADFILE;

    /* NULL parameter check */
    ExpectIntEQ(X509_get_ext_count(NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectNotNull(x509 = wolfSSL_X509_new());
    ExpectIntEQ(X509_get_ext_count(x509), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    wolfSSL_X509_free(x509);
    x509 = NULL;

    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(svrCertFile,
        SSL_FILETYPE_PEM));
    ExpectIntEQ(X509_get_ext_count(x509), 5);
    wolfSSL_X509_free(x509);

    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(ocspRootCaFile,
        SSL_FILETYPE_PEM));
    ExpectIntEQ(X509_get_ext_count(x509), 5);
    wolfSSL_X509_free(x509);

    ExpectTrue((f = XFOPEN("./certs/server-cert.pem", "rb")) != XBADFILE);
    ExpectNotNull(x509 = wolfSSL_PEM_read_X509(f, NULL, NULL, NULL));
    if (f != XBADFILE)
        XFCLOSE(f);

    /* wolfSSL_X509_get_ext_count() valid input */
    ExpectIntEQ((ret = wolfSSL_X509_get_ext_count(x509)), 5);

    wolfSSL_X509_free(x509);
#endif
    return EXPECT_RESULT();
}


/* Tests X509v3_get_ext_count, X509v3_get_ext_by_NID, and X509v3_get_ext
 * working with a stack retrieved from wolfSSL_X509_get0_extensions().
 */
static int test_wolfSSL_X509_stack_extensions(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_FILESYSTEM) && \
    !defined(NO_RSA)
    WOLFSSL_X509* x509 = NULL;
    const WOLFSSL_STACK* ext_stack = NULL;
    WOLFSSL_X509_EXTENSION* ext = NULL;
    int idx = -1;
    int count = 0;
    XFILE f = XBADFILE;

    /* Load a certificate */
    ExpectTrue((f = XFOPEN("./certs/server-cert.pem", "rb")) != XBADFILE);
    ExpectNotNull(x509 = wolfSSL_PEM_read_X509(f, NULL, NULL, NULL));
    if (f != XBADFILE)
        XFCLOSE(f);

    /* Get the stack of extensions */
    ExpectNotNull(ext_stack = wolfSSL_X509_get0_extensions(x509));

    /* Test X509v3_get_ext_count */
    ExpectIntGT((count = X509v3_get_ext_count(ext_stack)), 0);

    /* Test X509v3_get_ext_by_NID - find Basic Constraints extension */
    ExpectIntGE((idx = X509v3_get_ext_by_NID(ext_stack, NID_basic_constraints,
                -1)), 0);

    /* Test X509v3_get_ext - get extension by index */
    ExpectNotNull(ext = X509v3_get_ext(ext_stack, idx));

    /* Verify that the extension is the correct one */
    ExpectIntEQ(wolfSSL_OBJ_obj2nid(wolfSSL_X509_EXTENSION_get_object(ext)),
               NID_basic_constraints);

    /* Test negative cases */
    ExpectIntEQ(X509v3_get_ext_by_NID(NULL, NID_basic_constraints, -1),
               WOLFSSL_FATAL_ERROR);
    ExpectNull(X509v3_get_ext(NULL, 0));
    ExpectNull(X509v3_get_ext(ext_stack, -1));
    ExpectNull(X509v3_get_ext(ext_stack, count));

    wolfSSL_X509_free(x509);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_sign2(void)
{
    EXPECT_DECLS;
    /* test requires WOLFSSL_AKID_NAME to match expected output */
#if defined(OPENSSL_EXTRA) && !defined(NO_RSA) && !defined(NO_CERTS) && \
    defined(WOLFSSL_CERT_GEN) && defined(WOLFSSL_ALT_NAMES) && \
    defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_AKID_NAME) && \
    (defined(WOLFSSL_QT) || defined(OPENSSL_ALL) || \
    defined(WOLFSSL_IP_ALT_NAME))
    WOLFSSL_X509 *x509 = NULL;
    WOLFSSL_X509 *ca = NULL;
    const unsigned char *der = NULL;
    const unsigned char *pt = NULL;
    WOLFSSL_EVP_PKEY  *priv = NULL;
    WOLFSSL_X509_NAME *name = NULL;
    int derSz;
#ifndef NO_ASN_TIME
    WOLFSSL_ASN1_TIME *notBefore = NULL;
    WOLFSSL_ASN1_TIME *notAfter = NULL;

    const int year = 365*24*60*60;
    const int day  = 24*60*60;
    const int hour = 60*60;
    const int mini = 60;
    time_t t;
#endif

    const unsigned char expected[] = {
        0x30, 0x82, 0x05, 0x13, 0x30, 0x82, 0x03, 0xFB, 0xA0, 0x03, 0x02, 0x01,
        0x02, 0x02, 0x14, 0x6B, 0x61, 0x49, 0x45, 0xFF, 0x4A, 0xD1, 0x54, 0x16,
        0xB4, 0x35, 0x37, 0xC4, 0x98, 0x5D, 0xA9, 0xF6, 0x67, 0x60, 0x91, 0x30,
        0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x0B,
        0x05, 0x00, 0x30, 0x81, 0x94, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55,
        0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x10, 0x30, 0x0E, 0x06, 0x03,
        0x55, 0x04, 0x08, 0x0C, 0x07, 0x4D, 0x6F, 0x6E, 0x74, 0x61, 0x6E, 0x61,
        0x31, 0x10, 0x30, 0x0E, 0x06, 0x03, 0x55, 0x04, 0x07, 0x0C, 0x07, 0x42,
        0x6F, 0x7A, 0x65, 0x6D, 0x61, 0x6E, 0x31, 0x11, 0x30, 0x0F, 0x06, 0x03,
        0x55, 0x04, 0x0A, 0x0C, 0x08, 0x53, 0x61, 0x77, 0x74, 0x6F, 0x6F, 0x74,
        0x68, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x0B, 0x0C, 0x0A,
        0x43, 0x6F, 0x6E, 0x73, 0x75, 0x6C, 0x74, 0x69, 0x6E, 0x67, 0x31, 0x18,
        0x30, 0x16, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0C, 0x0F, 0x77, 0x77, 0x77,
        0x2E, 0x77, 0x6F, 0x6C, 0x66, 0x73, 0x73, 0x6C, 0x2E, 0x63, 0x6F, 0x6D,
        0x31, 0x1F, 0x30, 0x1D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D,
        0x01, 0x09, 0x01, 0x16, 0x10, 0x69, 0x6E, 0x66, 0x6F, 0x40, 0x77, 0x6F,
        0x6C, 0x66, 0x73, 0x73, 0x6C, 0x2E, 0x63, 0x6F, 0x6D, 0x30, 0x1E, 0x17,
        0x0D, 0x30, 0x30, 0x30, 0x32, 0x31, 0x35, 0x32, 0x30, 0x33, 0x30, 0x30,
        0x30, 0x5A, 0x17, 0x0D, 0x30, 0x31, 0x30, 0x32, 0x31, 0x34, 0x32, 0x30,
        0x33, 0x30, 0x30, 0x30, 0x5A, 0x30, 0x81, 0x9E, 0x31, 0x0B, 0x30, 0x09,
        0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x10, 0x30,
        0x0E, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0C, 0x07, 0x4D, 0x6F, 0x6E, 0x74,
        0x61, 0x6E, 0x61, 0x31, 0x10, 0x30, 0x0E, 0x06, 0x03, 0x55, 0x04, 0x07,
        0x0C, 0x07, 0x42, 0x6F, 0x7A, 0x65, 0x6D, 0x61, 0x6E, 0x31, 0x15, 0x30,
        0x13, 0x06, 0x03, 0x55, 0x04, 0x0A, 0x0C, 0x0C, 0x77, 0x6F, 0x6C, 0x66,
        0x53, 0x53, 0x4C, 0x5F, 0x32, 0x30, 0x34, 0x38, 0x31, 0x19, 0x30, 0x17,
        0x06, 0x03, 0x55, 0x04, 0x0B, 0x0C, 0x10, 0x50, 0x72, 0x6F, 0x67, 0x72,
        0x61, 0x6D, 0x6D, 0x69, 0x6E, 0x67, 0x2D, 0x32, 0x30, 0x34, 0x38, 0x31,
        0x18, 0x30, 0x16, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0C, 0x0F, 0x77, 0x77,
        0x77, 0x2E, 0x77, 0x6F, 0x6C, 0x66, 0x73, 0x73, 0x6C, 0x2E, 0x63, 0x6F,
        0x6D, 0x31, 0x1F, 0x30, 0x1D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7,
        0x0D, 0x01, 0x09, 0x01, 0x16, 0x10, 0x69, 0x6E, 0x66, 0x6F, 0x40, 0x77,
        0x6F, 0x6C, 0x66, 0x73, 0x73, 0x6C, 0x2E, 0x63, 0x6F, 0x6D, 0x30, 0x82,
        0x01, 0x22, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D,
        0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0F, 0x00, 0x30, 0x82,
        0x01, 0x0A, 0x02, 0x82, 0x01, 0x01, 0x00, 0xC3, 0x03, 0xD1, 0x2B, 0xFE,
        0x39, 0xA4, 0x32, 0x45, 0x3B, 0x53, 0xC8, 0x84, 0x2B, 0x2A, 0x7C, 0x74,
        0x9A, 0xBD, 0xAA, 0x2A, 0x52, 0x07, 0x47, 0xD6, 0xA6, 0x36, 0xB2, 0x07,
        0x32, 0x8E, 0xD0, 0xBA, 0x69, 0x7B, 0xC6, 0xC3, 0x44, 0x9E, 0xD4, 0x81,
        0x48, 0xFD, 0x2D, 0x68, 0xA2, 0x8B, 0x67, 0xBB, 0xA1, 0x75, 0xC8, 0x36,
        0x2C, 0x4A, 0xD2, 0x1B, 0xF7, 0x8B, 0xBA, 0xCF, 0x0D, 0xF9, 0xEF, 0xEC,
        0xF1, 0x81, 0x1E, 0x7B, 0x9B, 0x03, 0x47, 0x9A, 0xBF, 0x65, 0xCC, 0x7F,
        0x65, 0x24, 0x69, 0xA6, 0xE8, 0x14, 0x89, 0x5B, 0xE4, 0x34, 0xF7, 0xC5,
        0xB0, 0x14, 0x93, 0xF5, 0x67, 0x7B, 0x3A, 0x7A, 0x78, 0xE1, 0x01, 0x56,
        0x56, 0x91, 0xA6, 0x13, 0x42, 0x8D, 0xD2, 0x3C, 0x40, 0x9C, 0x4C, 0xEF,
        0xD1, 0x86, 0xDF, 0x37, 0x51, 0x1B, 0x0C, 0xA1, 0x3B, 0xF5, 0xF1, 0xA3,
        0x4A, 0x35, 0xE4, 0xE1, 0xCE, 0x96, 0xDF, 0x1B, 0x7E, 0xBF, 0x4E, 0x97,
        0xD0, 0x10, 0xE8, 0xA8, 0x08, 0x30, 0x81, 0xAF, 0x20, 0x0B, 0x43, 0x14,
        0xC5, 0x74, 0x67, 0xB4, 0x32, 0x82, 0x6F, 0x8D, 0x86, 0xC2, 0x88, 0x40,
        0x99, 0x36, 0x83, 0xBA, 0x1E, 0x40, 0x72, 0x22, 0x17, 0xD7, 0x52, 0x65,
        0x24, 0x73, 0xB0, 0xCE, 0xEF, 0x19, 0xCD, 0xAE, 0xFF, 0x78, 0x6C, 0x7B,
        0xC0, 0x12, 0x03, 0xD4, 0x4E, 0x72, 0x0D, 0x50, 0x6D, 0x3B, 0xA3, 0x3B,
        0xA3, 0x99, 0x5E, 0x9D, 0xC8, 0xD9, 0x0C, 0x85, 0xB3, 0xD9, 0x8A, 0xD9,
        0x54, 0x26, 0xDB, 0x6D, 0xFA, 0xAC, 0xBB, 0xFF, 0x25, 0x4C, 0xC4, 0xD1,
        0x79, 0xF4, 0x71, 0xD3, 0x86, 0x40, 0x18, 0x13, 0xB0, 0x63, 0xB5, 0x72,
        0x4E, 0x30, 0xC4, 0x97, 0x84, 0x86, 0x2D, 0x56, 0x2F, 0xD7, 0x15, 0xF7,
        0x7F, 0xC0, 0xAE, 0xF5, 0xFC, 0x5B, 0xE5, 0xFB, 0xA1, 0xBA, 0xD3, 0x02,
        0x03, 0x01, 0x00, 0x01, 0xA3, 0x82, 0x01, 0x4F, 0x30, 0x82, 0x01, 0x4B,
        0x30, 0x0C, 0x06, 0x03, 0x55, 0x1D, 0x13, 0x04, 0x05, 0x30, 0x03, 0x01,
        0x01, 0xFF, 0x30, 0x1C, 0x06, 0x03, 0x55, 0x1D, 0x11, 0x04, 0x15, 0x30,
        0x13, 0x82, 0x0B, 0x65, 0x78, 0x61, 0x6D, 0x70, 0x6C, 0x65, 0x2E, 0x63,
        0x6F, 0x6D, 0x87, 0x04, 0x7F, 0x00, 0x00, 0x01, 0x30, 0x1D, 0x06, 0x03,
        0x55, 0x1D, 0x0E, 0x04, 0x16, 0x04, 0x14, 0x33, 0xD8, 0x45, 0x66, 0xD7,
        0x68, 0x87, 0x18, 0x7E, 0x54, 0x0D, 0x70, 0x27, 0x91, 0xC7, 0x26, 0xD7,
        0x85, 0x65, 0xC0, 0x30, 0x81, 0xDE, 0x06, 0x03, 0x55, 0x1D, 0x23, 0x04,
        0x81, 0xD6, 0x30, 0x81, 0xD3, 0x80, 0x14, 0x33, 0xD8, 0x45, 0x66, 0xD7,
        0x68, 0x87, 0x18, 0x7E, 0x54, 0x0D, 0x70, 0x27, 0x91, 0xC7, 0x26, 0xD7,
        0x85, 0x65, 0xC0, 0xA1, 0x81, 0xA4, 0xA4, 0x81, 0xA1, 0x30, 0x81, 0x9E,
        0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55,
        0x53, 0x31, 0x10, 0x30, 0x0E, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0C, 0x07,
        0x4D, 0x6F, 0x6E, 0x74, 0x61, 0x6E, 0x61, 0x31, 0x10, 0x30, 0x0E, 0x06,
        0x03, 0x55, 0x04, 0x07, 0x0C, 0x07, 0x42, 0x6F, 0x7A, 0x65, 0x6D, 0x61,
        0x6E, 0x31, 0x15, 0x30, 0x13, 0x06, 0x03, 0x55, 0x04, 0x0A, 0x0C, 0x0C,
        0x77, 0x6F, 0x6C, 0x66, 0x53, 0x53, 0x4C, 0x5F, 0x32, 0x30, 0x34, 0x38,
        0x31, 0x19, 0x30, 0x17, 0x06, 0x03, 0x55, 0x04, 0x0B, 0x0C, 0x10, 0x50,
        0x72, 0x6F, 0x67, 0x72, 0x61, 0x6D, 0x6D, 0x69, 0x6E, 0x67, 0x2D, 0x32,
        0x30, 0x34, 0x38, 0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55, 0x04, 0x03,
        0x0C, 0x0F, 0x77, 0x77, 0x77, 0x2E, 0x77, 0x6F, 0x6C, 0x66, 0x73, 0x73,
        0x6C, 0x2E, 0x63, 0x6F, 0x6D, 0x31, 0x1F, 0x30, 0x1D, 0x06, 0x09, 0x2A,
        0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x09, 0x01, 0x16, 0x10, 0x69, 0x6E,
        0x66, 0x6F, 0x40, 0x77, 0x6F, 0x6C, 0x66, 0x73, 0x73, 0x6C, 0x2E, 0x63,
        0x6F, 0x6D, 0x82, 0x14, 0x6B, 0x61, 0x49, 0x45, 0xFF, 0x4A, 0xD1, 0x54,
        0x16, 0xB4, 0x35, 0x37, 0xC4, 0x98, 0x5D, 0xA9, 0xF6, 0x67, 0x60, 0x91,
        0x30, 0x1D, 0x06, 0x03, 0x55, 0x1D, 0x25, 0x04, 0x16, 0x30, 0x14, 0x06,
        0x08, 0x2B, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x01, 0x06, 0x08, 0x2B,
        0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x02, 0x30, 0x0D, 0x06, 0x09, 0x2A,
        0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x0B, 0x05, 0x00, 0x03, 0x82,
        0x01, 0x01, 0x00, 0x2F, 0x9F, 0x83, 0x05, 0x15, 0x1E, 0x5D, 0x7C, 0x22,
        0x12, 0x20, 0xEE, 0x07, 0x35, 0x25, 0x39, 0xDD, 0x34, 0x06, 0xD3, 0x89,
        0x31, 0x51, 0x8B, 0x9A, 0xE5, 0xE8, 0x60, 0x30, 0x07, 0x7A, 0xBB, 0x17,
        0xB9, 0x54, 0x72, 0x83, 0xA2, 0x1F, 0x62, 0xE0, 0x18, 0xAC, 0x93, 0x5E,
        0x63, 0xC7, 0xDD, 0x12, 0x58, 0x96, 0xC7, 0x90, 0x8B, 0x12, 0x50, 0xD2,
        0x60, 0x0E, 0x24, 0x07, 0x53, 0x55, 0xD7, 0x8E, 0xC9, 0x56, 0x12, 0x28,
        0xD8, 0xFD, 0x47, 0xE3, 0x13, 0xFB, 0x3C, 0xD6, 0x3D, 0x82, 0x09, 0x7E,
        0x10, 0x19, 0xE1, 0xCD, 0xCC, 0x4C, 0x78, 0xDF, 0xE5, 0xFB, 0x2C, 0x8C,
        0x88, 0xF7, 0x5B, 0x99, 0x93, 0xC6, 0xC7, 0x22, 0xA5, 0xFA, 0x76, 0x6C,
        0xE9, 0xBC, 0x69, 0xBA, 0x02, 0x82, 0x18, 0xAF, 0x47, 0xD0, 0x9C, 0x5F,
        0xED, 0xAE, 0x5A, 0x95, 0x59, 0x78, 0x86, 0x24, 0x22, 0xB6, 0x81, 0x03,
        0x58, 0x9A, 0x14, 0x93, 0xDC, 0x24, 0x58, 0xF3, 0xD2, 0x6C, 0x8E, 0xD2,
        0x6D, 0x8B, 0xE8, 0x4E, 0xC6, 0xA0, 0x2B, 0x0D, 0xDB, 0x1A, 0x76, 0x28,
        0xA9, 0x8D, 0xFB, 0x51, 0xA6, 0xF0, 0x82, 0x30, 0xEE, 0x78, 0x1C, 0x71,
        0xA8, 0x11, 0x8A, 0xA5, 0xC3, 0x91, 0xAB, 0x9A, 0x46, 0xFF, 0x8D, 0xCD,
        0x82, 0x3F, 0x5D, 0xB6, 0x28, 0x46, 0x6D, 0x66, 0xE2, 0xEE, 0x1E, 0x82,
        0x0D, 0x1A, 0x74, 0x87, 0xFB, 0xFD, 0x96, 0x26, 0x50, 0x09, 0xEC, 0xA7,
        0x73, 0x89, 0x43, 0x3B, 0x42, 0x2D, 0xA9, 0x6B, 0x0F, 0x61, 0x81, 0x97,
        0x11, 0x71, 0xF9, 0xDB, 0x9B, 0x69, 0x4B, 0x6E, 0xD3, 0x7D, 0xDA, 0xC6,
        0x61, 0x9F, 0x39, 0x87, 0x53, 0x52, 0xA8, 0x4D, 0xAD, 0x80, 0x29, 0x6C,
        0x19, 0xF0, 0x8D, 0xB1, 0x0D, 0x4E, 0xFB, 0x1B, 0xB7, 0xF1, 0x85, 0x49,
        0x08, 0x2A, 0x94, 0xD0, 0x4E, 0x0B, 0x8F
    };

    pt = ca_key_der_2048;
    ExpectNotNull(priv = wolfSSL_d2i_PrivateKey(EVP_PKEY_RSA, NULL, &pt,
        sizeof_ca_key_der_2048));

    pt = client_cert_der_2048;
    ExpectNotNull(x509 = wolfSSL_d2i_X509(NULL, &pt,
        sizeof_client_cert_der_2048));

    pt = ca_cert_der_2048;
    ExpectNotNull(ca = wolfSSL_d2i_X509(NULL, &pt, sizeof_ca_cert_der_2048));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);

#ifndef NO_ASN_TIME
    t = (time_t)30 * year + 45 * day + 20 * hour + 30 * mini + 7 * day;
    ExpectNotNull(notBefore = wolfSSL_ASN1_TIME_adj(NULL, t, 0, 0));
    ExpectNotNull(notAfter = wolfSSL_ASN1_TIME_adj(NULL, t, 365, 0));
    ExpectIntEQ(notAfter->length, 13);

    ExpectTrue(wolfSSL_X509_set_notBefore(x509, notBefore));
    ExpectTrue(wolfSSL_X509_set1_notBefore(x509, notBefore));
    ExpectTrue(wolfSSL_X509_set_notAfter(x509, notAfter));
    ExpectTrue(wolfSSL_X509_set1_notAfter(x509, notAfter));
#endif

    ExpectNull(wolfSSL_X509_notBefore(NULL));
    ExpectNotNull(wolfSSL_X509_notBefore(x509));
    ExpectNull(wolfSSL_X509_notAfter(NULL));
    ExpectNotNull(wolfSSL_X509_notAfter(x509));

    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    ExpectNotNull((der = wolfSSL_X509_get_der(x509, &derSz)));

    ExpectIntEQ(derSz, sizeof(expected));
#ifndef NO_ASN_TIME
    ExpectIntEQ(XMEMCMP(der, expected, derSz), 0);
#endif

    wolfSSL_X509_free(ca);
    wolfSSL_X509_free(x509);
    wolfSSL_EVP_PKEY_free(priv);
#ifndef NO_ASN_TIME
    wolfSSL_ASN1_TIME_free(notBefore);
    wolfSSL_ASN1_TIME_free(notAfter);
#endif
#endif
    return EXPECT_RESULT();
}


static int test_wolfSSL_X509_sign(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_ASN_TIME) && \
    defined(WOLFSSL_CERT_GEN) && defined(WOLFSSL_CERT_REQ) && !defined(NO_RSA)
    int ret = 0;
    char *cn = NULL;
    word32 cnSz = 0;
    X509_NAME *name = NULL;
    X509_NAME *emptyName = NULL;
    X509 *x509 = NULL;
    X509 *ca = NULL;
    DecodedCert dCert;
    EVP_PKEY *pub = NULL;
    EVP_PKEY *priv = NULL;
    EVP_MD_CTX *mctx = NULL;
#if defined(USE_CERT_BUFFERS_1024)
    const unsigned char* rsaPriv = client_key_der_1024;
    const unsigned char* rsaPub = client_keypub_der_1024;
    const unsigned char* certIssuer = client_cert_der_1024;
    long clientKeySz = (long)sizeof_client_key_der_1024;
    long clientPubKeySz = (long)sizeof_client_keypub_der_1024;
    long certIssuerSz = (long)sizeof_client_cert_der_1024;
#elif defined(USE_CERT_BUFFERS_2048)
    const unsigned char* rsaPriv = client_key_der_2048;
    const unsigned char* rsaPub = client_keypub_der_2048;
    const unsigned char* certIssuer = client_cert_der_2048;
    long clientKeySz = (long)sizeof_client_key_der_2048;
    long clientPubKeySz = (long)sizeof_client_keypub_der_2048;
    long certIssuerSz = (long)sizeof_client_cert_der_2048;
#endif
    byte sn[16];
    int snSz = sizeof(sn);
    int sigSz = 0;
#ifndef NO_WOLFSSL_STUB
    const WOLFSSL_ASN1_BIT_STRING* sig = NULL;
    const WOLFSSL_X509_ALGOR* alg = NULL;
#endif

    /* Set X509_NAME fields */
    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"wolfssl.com", 11, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "emailAddress", MBSTRING_UTF8,
                     (byte*)"support@wolfssl.com", 19, -1, 0), SSL_SUCCESS);

    /* Get private and public keys */
    ExpectNotNull(priv = wolfSSL_d2i_PrivateKey(EVP_PKEY_RSA, NULL, &rsaPriv,
                                                                  clientKeySz));
    ExpectNotNull(pub = wolfSSL_d2i_PUBKEY(NULL, &rsaPub, clientPubKeySz));
    ExpectNotNull(x509 = X509_new());
    ExpectIntEQ(X509_sign(x509, priv, EVP_sha256()), 0);
    /* Set version 3 */
    ExpectIntNE(X509_set_version(x509, 2L), 0);
    /* Set subject name, add pubkey, and sign certificate */
    ExpectIntEQ(X509_set_subject_name(x509, name), SSL_SUCCESS);
    X509_NAME_free(name);
    name = NULL;
    ExpectIntEQ(X509_set_pubkey(x509, pub), SSL_SUCCESS);
#ifdef WOLFSSL_ALT_NAMES
    ExpectNull(wolfSSL_X509_get_next_altname(NULL));
    ExpectNull(wolfSSL_X509_get_next_altname(x509));

    /* Add some subject alt names */
    ExpectIntNE(wolfSSL_X509_add_altname(NULL,
                "ipsum", ASN_DNS_TYPE), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_add_altname(x509,
                NULL, ASN_DNS_TYPE), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_add_altname(x509,
                "sphygmomanometer",
                ASN_DNS_TYPE), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_add_altname(x509,
                "supercalifragilisticexpialidocious",
                ASN_DNS_TYPE), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_add_altname(x509,
                "Llanfairpwllgwyngyllgogerychwyrndrobwllllantysiliogogogoch",
                ASN_DNS_TYPE), SSL_SUCCESS);
#ifdef WOLFSSL_IP_ALT_NAME
    {
        unsigned char ip4_type[] = {127,128,0,255};
        unsigned char ip6_type[] = {0xdd, 0xcc, 0xba, 0xab,
                                    0xff, 0xee, 0x99, 0x88,
                                    0x77, 0x66, 0x55, 0x44,
                                    0x00, 0x33, 0x22, 0x11};
        ExpectIntEQ(wolfSSL_X509_add_altname_ex(x509, (char*)ip4_type,
                sizeof(ip4_type), ASN_IP_TYPE), SSL_SUCCESS);
        ExpectIntEQ(wolfSSL_X509_add_altname_ex(x509, (char*)ip6_type,
                sizeof(ip6_type), ASN_IP_TYPE), SSL_SUCCESS);
    }
#endif

    {
        int i;

        if (x509 != NULL) {
            x509->altNamesNext = x509->altNames;
        }
#ifdef WOLFSSL_IP_ALT_NAME
        /* No names in IP address. */
        ExpectNull(wolfSSL_X509_get_next_altname(x509));
        ExpectNull(wolfSSL_X509_get_next_altname(x509));
#endif
        for (i = 0; i < 3; i++) {
            ExpectNotNull(wolfSSL_X509_get_next_altname(x509));
        }
        ExpectNull(wolfSSL_X509_get_next_altname(x509));
#ifdef WOLFSSL_MULTICIRCULATE_ALTNAMELIST
        ExpectNotNull(wolfSSL_X509_get_next_altname(x509));
#endif
    }
#endif /* WOLFSSL_ALT_NAMES */

    {
        ASN1_UTCTIME* infinite_past = NULL;
        ExpectNotNull(infinite_past = ASN1_UTCTIME_set(NULL, 0));
        ExpectIntEQ(X509_set1_notBefore(x509, infinite_past), 1);
        ASN1_UTCTIME_free(infinite_past);
    }

    /* test valid sign case */
    ExpectIntGT(ret = X509_sign(x509, priv, EVP_sha256()), 0);
    /* test getting signature */
#ifndef NO_WOLFSSL_STUB
    wolfSSL_X509_get0_signature(&sig, &alg, x509);
#endif
    ExpectIntEQ(wolfSSL_X509_get_signature(x509, NULL, &sigSz),
        WOLFSSL_SUCCESS);
    ExpectIntGT(sigSz, 0);
    ExpectIntEQ(wolfSSL_X509_get_signature(NULL, NULL, NULL),
        WOLFSSL_FATAL_ERROR);
    ExpectIntEQ(wolfSSL_X509_get_signature(x509, NULL, NULL),
        WOLFSSL_FATAL_ERROR);
    ExpectIntEQ(wolfSSL_X509_get_signature(NULL, NULL, &sigSz),
        WOLFSSL_FATAL_ERROR);
    sigSz = 0;
    ExpectIntEQ(wolfSSL_X509_get_signature(x509, sn, &sigSz),
        WOLFSSL_FATAL_ERROR);

    /* test valid X509_sign_ctx case */
    ExpectNotNull(mctx = EVP_MD_CTX_new());
    ExpectIntEQ(EVP_DigestSignInit(mctx, NULL, EVP_sha256(), NULL, priv), 1);
    ExpectIntGT(X509_sign_ctx(x509, mctx), 0);

#if defined(OPENSSL_ALL) && defined(WOLFSSL_ALT_NAMES)
    ExpectIntEQ(X509_get_ext_count(x509), 1);
#endif
#if defined(WOLFSSL_ALT_NAMES) && defined(WOLFSSL_IP_ALT_NAME)
    ExpectIntEQ(wolfSSL_X509_check_ip_asc(x509, "127.128.0.255", 0), 1);
    ExpectIntEQ(wolfSSL_X509_check_ip_asc(x509, "DDCC:BAAB:FFEE:9988:7766:5544:0033:2211", 0), 1);
#endif

    ExpectIntEQ(wolfSSL_X509_get_serial_number(x509, sn, &snSz),
                WOLFSSL_SUCCESS);
    DEBUG_WRITE_CERT_X509(x509, "signed.pem");

    /* Variation in size depends on ASN.1 encoding when MSB is set.
     * WOLFSSL_ASN_TEMPLATE code does not generate a serial number
     * with the MSB set. See GenerateInteger in asn.c */
#ifndef USE_CERT_BUFFERS_1024
#ifndef WOLFSSL_ALT_NAMES
    /* Valid case - size should be 781-786 with 16 byte serial number */
    ExpectTrue((781 + snSz <= ret) && (ret <= 781 + 5 + snSz));
#elif defined(WOLFSSL_IP_ALT_NAME)
    /* Valid case - size should be 955-960 with 16 byte serial number */
    ExpectTrue((939 + snSz <= ret) && (ret <= 939 + 5 + snSz));
#else
    /* Valid case - size should be 926-931 with 16 byte serial number */
    ExpectTrue((910 + snSz <= ret) && (ret <= 910 + 5 + snSz));
#endif
#else
#ifndef WOLFSSL_ALT_NAMES
    /* Valid case - size should be 537-542 with 16 byte serial number */
    ExpectTrue((521 + snSz <= ret) && (ret <= 521 + 5 + snSz));
#elif defined(OPENSSL_ALL) || defined(WOLFSSL_IP_ALT_NAME)
    /* Valid case - size should be 695-670 with 16 byte serial number */
    ExpectTrue((679 + snSz <= ret) && (ret <= 679 + 5 + snSz));
#else
    /* Valid case - size should be 666-671 with 16 byte serial number */
    ExpectTrue((650 + snSz <= ret) && (ret <= 650 + 5 + snSz));
#endif
#endif
    /* check that issuer name is as expected after signature */
    InitDecodedCert(&dCert, certIssuer, (word32)certIssuerSz, 0);
    ExpectIntEQ(ParseCert(&dCert, CERT_TYPE, NO_VERIFY, NULL), 0);

    ExpectNotNull(emptyName = X509_NAME_new());
    ExpectNotNull(ca = d2i_X509(NULL, &certIssuer, (int)certIssuerSz));
    ExpectIntEQ(wolfSSL_X509_get_isCA(NULL), 0);
    ExpectIntEQ(wolfSSL_X509_get_isCA(ca), 1);
    ExpectNotNull(name = X509_get_subject_name(ca));
    ExpectIntEQ(X509_NAME_get_sz(NULL), WOLFSSL_FATAL_ERROR);
    ExpectIntGT(cnSz = X509_NAME_get_sz(name), 0);
    ExpectNotNull(cn = (char*)XMALLOC(cnSz, HEAP_HINT, DYNAMIC_TYPE_OPENSSL));
    ExpectNull(X509_NAME_oneline(NULL, cn, (int)cnSz));
    ExpectPtrEq(X509_NAME_oneline(name, cn, 0), cn);
    ExpectPtrEq(X509_NAME_oneline(emptyName, cn, (int)cnSz), cn);
    ExpectNull(X509_NAME_oneline(emptyName, NULL, 0));
    ExpectPtrEq(X509_NAME_oneline(name, cn, (int)cnSz), cn);
    ExpectIntEQ(0, XSTRNCMP(cn, dCert.subject, XSTRLEN(cn)));
    XFREE(cn, HEAP_HINT, DYNAMIC_TYPE_OPENSSL);
    cn = NULL;

#if defined(XSNPRINTF)
    ExpectNull(wolfSSL_X509_get_name_oneline(NULL, NULL, 0));
    ExpectNotNull(cn = wolfSSL_X509_get_name_oneline(name, NULL, 0));
    ExpectIntGT((int)(cnSz = (word32)XSTRLEN(cn) + 1), 0);
    ExpectPtrEq(wolfSSL_X509_get_name_oneline(name, cn, (int)cnSz), cn);
    ExpectNull(wolfSSL_X509_get_name_oneline(NULL, cn, (int)cnSz));
    ExpectNull(wolfSSL_X509_get_name_oneline(name, cn, cnSz - 1));
    ExpectPtrEq(wolfSSL_X509_get_name_oneline(name, cn, (int)cnSz), cn);
    ExpectPtrEq(wolfSSL_X509_get_name_oneline(emptyName, cn, (int)cnSz), cn);
    XFREE(cn, HEAP_HINT, DYNAMIC_TYPE_OPENSSL);
    cn = NULL;
#endif
    X509_NAME_free(emptyName);

#ifdef WOLFSSL_MULTI_ATTRIB
    /* test adding multiple OU's to the signer */
    ExpectNotNull(name = X509_get_subject_name(ca));
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "OU", MBSTRING_UTF8,
                                       (byte*)"OU1", 3, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "OU", MBSTRING_UTF8,
                                       (byte*)"OU2", 3, -1, 0), SSL_SUCCESS);
    ExpectIntGT(X509_sign(ca, priv, EVP_sha256()), 0);
#endif

    ExpectNotNull(name = X509_get_subject_name(ca));
    ExpectIntEQ(X509_set_issuer_name(x509, name), SSL_SUCCESS);

    ExpectIntGT(X509_sign(x509, priv, EVP_sha256()), 0);
    ExpectNotNull(name = X509_get_issuer_name(x509));
    cnSz = X509_NAME_get_sz(name);
    ExpectNotNull(cn = (char*)XMALLOC(cnSz, HEAP_HINT, DYNAMIC_TYPE_OPENSSL));
    ExpectNotNull(cn = X509_NAME_oneline(name, cn, (int)cnSz));
    /* compare and don't include the multi-attrib "/OU=OU1/OU=OU2" above */
    ExpectIntEQ(0, XSTRNCMP(cn, dCert.issuer, XSTRLEN(dCert.issuer)));
    XFREE(cn, HEAP_HINT, DYNAMIC_TYPE_OPENSSL);
    cn = NULL;

    FreeDecodedCert(&dCert);

    /* Test invalid parameters */
    ExpectIntEQ(X509_sign(NULL, priv, EVP_sha256()), 0);
    ExpectIntEQ(X509_sign(x509, NULL, EVP_sha256()), 0);
    ExpectIntEQ(X509_sign(x509, priv, NULL), 0);

    ExpectIntEQ(X509_sign_ctx(NULL, mctx), 0);
    EVP_MD_CTX_free(mctx);
    mctx = NULL;
    ExpectNotNull(mctx = EVP_MD_CTX_new());
    ExpectIntEQ(X509_sign_ctx(x509, mctx), 0);
    ExpectIntEQ(X509_sign_ctx(x509, NULL), 0);

    /* test invalid version number */
#if defined(OPENSSL_ALL)
    ExpectIntNE(X509_set_version(x509, 6L), 0);
    ExpectIntGT(X509_sign(x509, priv, EVP_sha256()), 0);

    /* uses ParseCert which fails on bad version number */
    ExpectIntEQ(X509_get_ext_count(x509), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
#endif

    EVP_MD_CTX_free(mctx);
    EVP_PKEY_free(priv);
    EVP_PKEY_free(pub);
    X509_free(x509);
    X509_free(ca);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_get0_tbs_sigalg(void)
{
    EXPECT_DECLS;
#if (defined(OPENSSL_ALL) || defined(WOLFSSL_APACHE_HTTPD))
    X509* x509 = NULL;
    const X509_ALGOR* alg;

    ExpectNotNull(x509 = X509_new());

    ExpectNull(alg = X509_get0_tbs_sigalg(NULL));
    ExpectNotNull(alg = X509_get0_tbs_sigalg(x509));

    X509_free(x509);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_ALGOR_get0(void)
{
    EXPECT_DECLS;
#if (defined(OPENSSL_ALL) || defined(WOLFSSL_APACHE_HTTPD)) && \
    !defined(NO_SHA256) && !defined(NO_RSA)
    X509* x509 = NULL;
    const ASN1_OBJECT* obj = NULL;
    const X509_ALGOR* alg = NULL;
    X509_ALGOR* alg2 = NULL;
    int pptype = 0;
    const void *ppval = NULL;
    byte* der = NULL;
    const byte* tmp = NULL;
    const byte badObj[] = { 0x06, 0x00 };

    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(cliCertFile,
        SSL_FILETYPE_PEM));
    ExpectNotNull(alg = X509_get0_tbs_sigalg(x509));

    /* Invalid case */
    X509_ALGOR_get0(&obj, NULL, NULL, NULL);
    ExpectNull(obj);

    /* Valid case */
    X509_ALGOR_get0(NULL, NULL, NULL, alg);
    X509_ALGOR_get0(&obj, &pptype, &ppval, alg);
    ExpectNotNull(obj);
    ExpectNull(ppval);
    ExpectIntNE(pptype, 0);
    /* Make sure NID of X509_ALGOR is Sha256 with RSA */
    ExpectIntEQ(OBJ_obj2nid(obj), NID_sha256WithRSAEncryption);

    ExpectIntEQ(i2d_X509_ALGOR(NULL, NULL), WOLFSSL_FATAL_ERROR);
    ExpectIntEQ(i2d_X509_ALGOR(alg, &der), 15);
    ExpectNull(d2i_X509_ALGOR(NULL, NULL, 0));
    /* tmp is NULL. */
    ExpectNull(d2i_X509_ALGOR(NULL, &tmp, 0));
    tmp = badObj;
    ExpectNull(d2i_X509_ALGOR(NULL, &tmp, (long)sizeof(badObj)));
    tmp = der;
    ExpectNull(d2i_X509_ALGOR(NULL, &tmp, 0));
    ExpectNotNull(d2i_X509_ALGOR(&alg2, &tmp, 15));
    tmp = der;
    ExpectNotNull(d2i_X509_ALGOR(&alg2, &tmp, 15));

    XFREE(der, NULL, DYNAMIC_TYPE_ASN1);
    X509_free(x509);
    X509_ALGOR_free(NULL);
    X509_ALGOR_free(alg2);
    alg2 = NULL;
#endif
    return EXPECT_RESULT();
}


static int test_wolfSSL_X509_VERIFY_PARAM(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA)
    X509_VERIFY_PARAM *paramTo = NULL;
    X509_VERIFY_PARAM *paramFrom = NULL;
    char testIPv4[] = "127.0.0.1";
    char testIPv6[] = "0001:0000:0000:0000:0000:0000:0000:0000/32";
    char testhostName1[] = "foo.hoge.com";
    char testhostName2[] = "foobar.hoge.com";

    ExpectNotNull(paramTo = X509_VERIFY_PARAM_new());
    ExpectNotNull(XMEMSET(paramTo, 0, sizeof(X509_VERIFY_PARAM)));

    ExpectNotNull(paramFrom = X509_VERIFY_PARAM_new());
    ExpectNotNull(XMEMSET(paramFrom, 0, sizeof(X509_VERIFY_PARAM)));

    ExpectIntEQ(X509_VERIFY_PARAM_set1_host(paramFrom, testhostName1,
        (int)XSTRLEN(testhostName1)), 1);
    ExpectIntEQ(0, XSTRNCMP(paramFrom->hostName, testhostName1,
        (int)XSTRLEN(testhostName1)));

    X509_VERIFY_PARAM_set_hostflags(NULL, 0x00);

    X509_VERIFY_PARAM_set_hostflags(paramFrom, 0x01);
    ExpectIntEQ(0x01, paramFrom->hostFlags);

    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip_asc(NULL, testIPv4), 0);

    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip_asc(paramFrom, testIPv4), 1);
    ExpectIntEQ(0, XSTRNCMP(paramFrom->ipasc, testIPv4, WOLFSSL_MAX_IPSTR));

    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip_asc(paramFrom, NULL), 1);

    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip_asc(paramFrom, testIPv6), 1);
    ExpectIntEQ(0, XSTRNCMP(paramFrom->ipasc, testIPv6, WOLFSSL_MAX_IPSTR));

    /* null pointer */
    ExpectIntEQ(X509_VERIFY_PARAM_set1(NULL, paramFrom), 0);
    /* in the case of "from" null, returns success */
    ExpectIntEQ(X509_VERIFY_PARAM_set1(paramTo, NULL), 1);

    ExpectIntEQ(X509_VERIFY_PARAM_set1(NULL, NULL), 0);

    /* inherit flags test : VPARAM_DEFAULT */
    ExpectIntEQ(X509_VERIFY_PARAM_set1(paramTo, paramFrom), 1);
    ExpectIntEQ(0, XSTRNCMP(paramTo->hostName, testhostName1,
                                    (int)XSTRLEN(testhostName1)));
    ExpectIntEQ(0x01, paramTo->hostFlags);
    ExpectIntEQ(0, XSTRNCMP(paramTo->ipasc, testIPv6, WOLFSSL_MAX_IPSTR));

    /* inherit flags test : VPARAM OVERWRITE */
    ExpectIntEQ(X509_VERIFY_PARAM_set1_host(paramTo, testhostName2,
        (int)XSTRLEN(testhostName2)), 1);
    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip_asc(paramTo, testIPv4), 1);
    X509_VERIFY_PARAM_set_hostflags(paramTo, 0x00);

    if (paramTo != NULL) {
        paramTo->inherit_flags = X509_VP_FLAG_OVERWRITE;
    }

    ExpectIntEQ(X509_VERIFY_PARAM_set1(paramTo, paramFrom), 1);
    ExpectIntEQ(0, XSTRNCMP(paramTo->hostName, testhostName1,
        (int)XSTRLEN(testhostName1)));
    ExpectIntEQ(0x01, paramTo->hostFlags);
    ExpectIntEQ(0, XSTRNCMP(paramTo->ipasc, testIPv6, WOLFSSL_MAX_IPSTR));

    /* inherit flags test : VPARAM_RESET_FLAGS */
    ExpectIntEQ(X509_VERIFY_PARAM_set1_host(paramTo, testhostName2,
        (int)XSTRLEN(testhostName2)), 1);
    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip_asc(paramTo, testIPv4), 1);
    X509_VERIFY_PARAM_set_hostflags(paramTo, 0x10);

    if (paramTo != NULL) {
        paramTo->inherit_flags = X509_VP_FLAG_RESET_FLAGS;
    }

    ExpectIntEQ(X509_VERIFY_PARAM_set1(paramTo, paramFrom), 1);
    ExpectIntEQ(0, XSTRNCMP(paramTo->hostName, testhostName1,
                                        (int)XSTRLEN(testhostName1)));
    ExpectIntEQ(0x01, paramTo->hostFlags);
    ExpectIntEQ(0, XSTRNCMP(paramTo->ipasc, testIPv6, WOLFSSL_MAX_IPSTR));

    /* inherit flags test : VPARAM_LOCKED */
    ExpectIntEQ(X509_VERIFY_PARAM_set1_host(paramTo, testhostName2,
        (int)XSTRLEN(testhostName2)), 1);
    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip_asc(paramTo, testIPv4), 1);
    X509_VERIFY_PARAM_set_hostflags(paramTo, 0x00);

    if (paramTo != NULL) {
        paramTo->inherit_flags = X509_VP_FLAG_LOCKED;
    }

    ExpectIntEQ(X509_VERIFY_PARAM_set1(paramTo, paramFrom), 1);
    ExpectIntEQ(0, XSTRNCMP(paramTo->hostName, testhostName2,
                                    (int)XSTRLEN(testhostName2)));
    ExpectIntEQ(0x00, paramTo->hostFlags);
    ExpectIntEQ(0, XSTRNCMP(paramTo->ipasc, testIPv4, WOLFSSL_MAX_IPSTR));

    /* test for incorrect parameters */
    ExpectIntEQ(X509_VERIFY_PARAM_set_flags(NULL, X509_V_FLAG_CRL_CHECK_ALL),
        0);

    ExpectIntEQ(X509_VERIFY_PARAM_set_flags(NULL, 0), 0);

    /* inherit flags test : VPARAM_ONCE, not testable yet */

    ExpectIntEQ(X509_VERIFY_PARAM_set_flags(paramTo, X509_V_FLAG_CRL_CHECK_ALL),
        1);

    ExpectIntEQ(X509_VERIFY_PARAM_get_flags(NULL), 0);
    ExpectIntEQ(X509_VERIFY_PARAM_get_flags(paramTo),
        X509_V_FLAG_CRL_CHECK_ALL);

    ExpectIntEQ(X509_VERIFY_PARAM_clear_flags(NULL, X509_V_FLAG_CRL_CHECK_ALL),
        WOLFSSL_FAILURE);
    ExpectIntEQ(X509_VERIFY_PARAM_clear_flags(paramTo,
        X509_V_FLAG_CRL_CHECK_ALL), 1);

    ExpectIntEQ(X509_VERIFY_PARAM_get_flags(paramTo), 0);

    ExpectNull(wolfSSL_X509_VERIFY_PARAM_lookup(NULL));
    ExpectNull(wolfSSL_X509_VERIFY_PARAM_lookup(""));
    ExpectNotNull(wolfSSL_X509_VERIFY_PARAM_lookup("ssl_client"));
    ExpectNotNull(wolfSSL_X509_VERIFY_PARAM_lookup("ssl_server"));

    X509_VERIFY_PARAM_free(paramTo);
    X509_VERIFY_PARAM_free(paramFrom);
    X509_VERIFY_PARAM_free(NULL); /* to confirm NULL parameter gives no harm */
#endif
    return EXPECT_RESULT();
}

#if defined(OPENSSL_EXTRA) && defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES) && \
    !defined(WOLFSSL_HOSTNAME_VERIFY_ALT_NAME_ONLY)

static int test_wolfSSL_check_domain_verify_count = 0;

static WC_INLINE int test_wolfSSL_check_domain_verify_cb(int preverify,
        WOLFSSL_X509_STORE_CTX* store)
{
    EXPECT_DECLS;
    ExpectIntEQ(X509_STORE_CTX_get_error(store), 0);
    ExpectIntEQ(preverify, 1);
    ExpectIntGT(++test_wolfSSL_check_domain_verify_count, 0);
    return EXPECT_SUCCESS();
}

static int test_wolfSSL_check_domain_client_cb(WOLFSSL* ssl)
{
    EXPECT_DECLS;
    X509_VERIFY_PARAM *param = NULL;

    ExpectNotNull(param = SSL_get0_param(ssl));

    /* Domain check should only be done on the leaf cert */
    X509_VERIFY_PARAM_set_hostflags(param,
        X509_CHECK_FLAG_NO_PARTIAL_WILDCARDS);
    ExpectIntEQ(X509_VERIFY_PARAM_set1_host(param,
        "wolfSSL Server Chain", 0), 1);
    wolfSSL_set_verify(ssl, WOLFSSL_VERIFY_PEER,
        test_wolfSSL_check_domain_verify_cb);
    return EXPECT_RESULT();
}

static int test_wolfSSL_check_domain_server_cb(WOLFSSL_CTX* ctx)
{
    EXPECT_DECLS;
    /* Use a cert with different domains in chain */
    ExpectIntEQ(wolfSSL_CTX_use_certificate_chain_file(ctx,
        "certs/intermediate/server-chain.pem"), WOLFSSL_SUCCESS);
    return EXPECT_RESULT();
}

static int test_wolfSSL_check_domain(void)
{
    EXPECT_DECLS;
    test_ssl_cbf func_cb_client;
    test_ssl_cbf func_cb_server;

    XMEMSET(&func_cb_client, 0, sizeof(func_cb_client));
    XMEMSET(&func_cb_server, 0, sizeof(func_cb_server));

    func_cb_client.ssl_ready = &test_wolfSSL_check_domain_client_cb;
    func_cb_server.ctx_ready = &test_wolfSSL_check_domain_server_cb;

    ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&func_cb_client,
        &func_cb_server, NULL), TEST_SUCCESS);

    /* Should have been called once for each cert in sent chain */
#ifdef WOLFSSL_VERIFY_CB_ALL_CERTS
    ExpectIntEQ(test_wolfSSL_check_domain_verify_count, 3);
#else
    ExpectIntEQ(test_wolfSSL_check_domain_verify_count, 1);
#endif

    return EXPECT_RESULT();
}

#else

static int test_wolfSSL_check_domain(void)
{
    EXPECT_DECLS;
    return EXPECT_RESULT();
}

#endif /* OPENSSL_EXTRA && HAVE_SSL_MEMIO_TESTS_DEPENDENCIES */
#if defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES) && \
    !defined(OPENSSL_COMPATIBLE_DEFAULTS) && !defined(NO_SHA256)
static const char* dn = NULL;
static int test_wolfSSL_check_domain_basic_client_ssl(WOLFSSL* ssl)
{
    EXPECT_DECLS;

    ExpectIntEQ(wolfSSL_check_domain_name(ssl, dn), WOLFSSL_SUCCESS);

    return EXPECT_RESULT();
}
static int test_wolfSSL_check_domain_basic(void)
{
    EXPECT_DECLS;
    test_ssl_cbf func_cb_client;
    test_ssl_cbf func_cb_server;

    XMEMSET(&func_cb_client, 0, sizeof(func_cb_client));
    XMEMSET(&func_cb_server, 0, sizeof(func_cb_server));

    dn = "invalid.com";
    func_cb_client.ssl_ready = &test_wolfSSL_check_domain_basic_client_ssl;

    /* Expect to fail */
    ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&func_cb_client,
        &func_cb_server, NULL), -1001);

    dn = "example.com";

    /* Expect to succeed */
    ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&func_cb_client,
        &func_cb_server, NULL), TEST_SUCCESS);

    return EXPECT_RESULT();
}
#else
static int test_wolfSSL_check_domain_basic(void)
{
    EXPECT_DECLS;
    return EXPECT_RESULT();
}
#endif /* HAVE_SSL_MEMIO_TESTS_DEPENDENCIES */

static int test_wolfSSL_X509_get_X509_PUBKEY(void)
{
    EXPECT_DECLS;
#if (defined(OPENSSL_ALL) || defined(WOLFSSL_APACHE_HTTPD))
    X509* x509 = NULL;
    X509_PUBKEY* pubKey;

    ExpectNotNull(x509 = X509_new());

    ExpectNull(pubKey = wolfSSL_X509_get_X509_PUBKEY(NULL));
    ExpectNotNull(pubKey = wolfSSL_X509_get_X509_PUBKEY(x509));

    X509_free(x509);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_PUBKEY_RSA(void)
{
    EXPECT_DECLS;
#if (defined(OPENSSL_ALL) || defined(WOLFSSL_APACHE_HTTPD)) && \
    !defined(NO_SHA256) && !defined(NO_RSA)
    X509* x509 = NULL;
    ASN1_OBJECT* obj = NULL;
    const ASN1_OBJECT* pa_oid = NULL;
    X509_PUBKEY* pubKey = NULL;
    X509_PUBKEY* pubKey2 = NULL;
    EVP_PKEY* evpKey = NULL;
    byte buf[1024];
    byte* tmp;

    const unsigned char *pk = NULL;
    int ppklen;
    int pptype;
    X509_ALGOR *pa = NULL;
    const void *pval;

    ExpectNotNull(x509 = X509_load_certificate_file(cliCertFile,
        SSL_FILETYPE_PEM));

    ExpectNotNull(pubKey = X509_get_X509_PUBKEY(x509));
    ExpectIntEQ(X509_PUBKEY_get0_param(&obj, &pk, &ppklen, &pa, pubKey), 1);
    ExpectNotNull(pk);
    ExpectNotNull(pa);
    ExpectNotNull(pubKey);
    ExpectIntGT(ppklen, 0);

    tmp = buf;
    ExpectIntEQ(wolfSSL_i2d_X509_PUBKEY(NULL, NULL), WOLFSSL_FATAL_ERROR);
    ExpectIntEQ(wolfSSL_i2d_X509_PUBKEY(NULL, &tmp), WOLFSSL_FATAL_ERROR);
    ExpectIntEQ(wolfSSL_i2d_X509_PUBKEY(pubKey, NULL), 294);
    ExpectIntEQ(wolfSSL_i2d_X509_PUBKEY(pubKey, &tmp), 294);

    ExpectIntEQ(OBJ_obj2nid(obj), NID_rsaEncryption);

    ExpectNotNull(evpKey = X509_PUBKEY_get(pubKey));
    ExpectNotNull(pubKey2 = X509_PUBKEY_new());
    ExpectIntEQ(X509_PUBKEY_get0_param(&obj, &pk, &ppklen, &pa, NULL), 0);
    ExpectIntEQ(X509_PUBKEY_get0_param(&obj, &pk, &ppklen, &pa, pubKey2), 0);
    ExpectIntEQ(X509_PUBKEY_set(NULL, NULL), 0);
    ExpectIntEQ(X509_PUBKEY_set(&pubKey2, NULL), 0);
    ExpectIntEQ(X509_PUBKEY_set(NULL, evpKey), 0);
    ExpectIntEQ(X509_PUBKEY_set(&pubKey2, evpKey), 1);
    ExpectIntEQ(X509_PUBKEY_get0_param(NULL, NULL, NULL, NULL, pubKey2), 1);
    ExpectIntEQ(X509_PUBKEY_get0_param(&obj, &pk, &ppklen, &pa, pubKey2), 1);
    ExpectNotNull(pk);
    ExpectNotNull(pa);
    ExpectIntGT(ppklen, 0);
    X509_ALGOR_get0(&pa_oid, &pptype, &pval, pa);
    ExpectNotNull(pa_oid);
    ExpectNull(pval);
    ExpectIntEQ(pptype, V_ASN1_NULL);
    ExpectIntEQ(OBJ_obj2nid(pa_oid), EVP_PKEY_RSA);

    X509_PUBKEY_free(NULL);
    X509_PUBKEY_free(pubKey2);
    X509_free(x509);
    EVP_PKEY_free(evpKey);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_PUBKEY_EC(void)
{
    EXPECT_DECLS;
#if (defined(OPENSSL_ALL) || defined(WOLFSSL_APACHE_HTTPD)) && defined(HAVE_ECC)
    X509* x509 = NULL;
    ASN1_OBJECT* obj = NULL;
    ASN1_OBJECT* poid = NULL;
    const ASN1_OBJECT* pa_oid = NULL;
    X509_PUBKEY* pubKey = NULL;
    X509_PUBKEY* pubKey2 = NULL;
    EVP_PKEY* evpKey = NULL;

    const unsigned char *pk = NULL;
    int ppklen;
    int pptype;
    X509_ALGOR *pa = NULL;
    const void *pval;
    char buf[50];

    ExpectNotNull(x509 = X509_load_certificate_file(cliEccCertFile,
                                                    SSL_FILETYPE_PEM));
    ExpectNotNull(pubKey = X509_get_X509_PUBKEY(x509));
    ExpectNotNull(evpKey = X509_PUBKEY_get(pubKey));
    ExpectNotNull(pubKey2 = X509_PUBKEY_new());
    ExpectIntEQ(X509_PUBKEY_set(&pubKey2, evpKey), 1);
    ExpectIntEQ(X509_PUBKEY_get0_param(&obj, &pk, &ppklen, &pa, pubKey2), 1);
    ExpectNotNull(pk);
    ExpectNotNull(pa);
    ExpectIntGT(ppklen, 0);
    X509_ALGOR_get0(&pa_oid, &pptype, &pval, pa);
    ExpectNotNull(pa_oid);
    ExpectNotNull(pval);
    ExpectIntEQ(pptype, V_ASN1_OBJECT);
    ExpectIntEQ(OBJ_obj2nid(pa_oid), EVP_PKEY_EC);
    poid = (ASN1_OBJECT *)pval;
    ExpectIntGT(OBJ_obj2txt(buf, (int)sizeof(buf), poid, 0), 0);
    ExpectIntEQ(OBJ_txt2nid(buf), NID_X9_62_prime256v1);

    X509_PUBKEY_free(pubKey2);
    X509_free(x509);
    EVP_PKEY_free(evpKey);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_PUBKEY_DSA(void)
{
    EXPECT_DECLS;
#if (defined(OPENSSL_ALL) || defined(WOLFSSL_APACHE_HTTPD)) && !defined(NO_DSA)
    word32  bytes;
#ifdef USE_CERT_BUFFERS_1024
    byte    tmp[ONEK_BUF];
#elif defined(USE_CERT_BUFFERS_2048)
    byte    tmp[TWOK_BUF];
#else
    byte    tmp[TWOK_BUF];
#endif /* END USE_CERT_BUFFERS_1024 */
    const unsigned char* dsaKeyDer = tmp;

    ASN1_OBJECT* obj = NULL;
    ASN1_STRING* str;
    const ASN1_OBJECT* pa_oid = NULL;
    X509_PUBKEY* pubKey = NULL;
    EVP_PKEY* evpKey = NULL;

    const unsigned char *pk = NULL;
    int ppklen, pptype;
    X509_ALGOR *pa = NULL;
    const void *pval;

#ifdef USE_CERT_BUFFERS_1024
    XMEMSET(tmp, 0, sizeof(tmp));
    XMEMCPY(tmp, dsa_key_der_1024, sizeof_dsa_key_der_1024);
    bytes = sizeof_dsa_key_der_1024;
#elif defined(USE_CERT_BUFFERS_2048)
    XMEMSET(tmp, 0, sizeof(tmp));
    XMEMCPY(tmp, dsa_key_der_2048, sizeof_dsa_key_der_2048);
    bytes = sizeof_dsa_key_der_2048;
#else
    {
        XFILE fp = XBADFILE;
        XMEMSET(tmp, 0, sizeof(tmp));
        ExpectTrue((fp = XFOPEN("./certs/dsa2048.der", "rb")) != XBADFILE);
        ExpectIntGT(bytes = (word32) XFREAD(tmp, 1, sizeof(tmp), fp), 0);
        if (fp != XBADFILE)
            XFCLOSE(fp);
    }
#endif

    /* Initialize pkey with der format dsa key */
    ExpectNotNull(d2i_PrivateKey(EVP_PKEY_DSA, &evpKey, &dsaKeyDer, bytes));

    ExpectNotNull(pubKey = X509_PUBKEY_new());
    ExpectIntEQ(X509_PUBKEY_set(&pubKey, evpKey), 1);
    ExpectIntEQ(X509_PUBKEY_get0_param(&obj, &pk, &ppklen, &pa, pubKey), 1);
    ExpectNotNull(pk);
    ExpectNotNull(pa);
    ExpectIntGT(ppklen, 0);
    X509_ALGOR_get0(&pa_oid, &pptype, &pval, pa);
    ExpectNotNull(pa_oid);
    ExpectNotNull(pval);
    ExpectIntEQ(pptype, V_ASN1_SEQUENCE);
    ExpectIntEQ(OBJ_obj2nid(pa_oid), EVP_PKEY_DSA);
    str = (ASN1_STRING *)pval;
    DEBUG_WRITE_DER(ASN1_STRING_data(str), ASN1_STRING_length(str), "str.der");
#ifdef USE_CERT_BUFFERS_1024
    ExpectIntEQ(ASN1_STRING_length(str), 291);
#else
    ExpectIntEQ(ASN1_STRING_length(str), 549);
#endif /* END USE_CERT_BUFFERS_1024 */

    X509_PUBKEY_free(pubKey);
    EVP_PKEY_free(evpKey);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_BUF(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA)
    BUF_MEM* buf = NULL;
    ExpectNotNull(buf = BUF_MEM_new());
    ExpectIntEQ(BUF_MEM_grow(buf, 10), 10);
    ExpectIntEQ(BUF_MEM_grow(buf, -1), 0);
    BUF_MEM_free(buf);
#endif
    return EXPECT_RESULT();
}

#if defined(OPENSSL_EXTRA) && !defined(WOLFSSL_NO_OPENSSL_RAND_CB)
static int stub_rand_seed(const void *buf, int num)
{
    (void)buf;
    (void)num;

    return 123;
}

static int stub_rand_bytes(unsigned char *buf, int num)
{
    (void)buf;
    (void)num;

    return 456;
}

static byte* was_stub_rand_cleanup_called(void)
{
    static byte was_called = 0;

    return &was_called;
}

static void stub_rand_cleanup(void)
{
    byte* was_called = was_stub_rand_cleanup_called();

    *was_called = 1;

    return;
}

static byte* was_stub_rand_add_called(void)
{
    static byte was_called = 0;

    return &was_called;
}

static int stub_rand_add(const void *buf, int num, double entropy)
{
    byte* was_called = was_stub_rand_add_called();

    (void)buf;
    (void)num;
    (void)entropy;

    *was_called = 1;

    return 0;
}

static int stub_rand_pseudo_bytes(unsigned char *buf, int num)
{
    (void)buf;
    (void)num;

    return 9876;
}

static int stub_rand_status(void)
{
    return 5432;
}
#endif /* OPENSSL_EXTRA && !WOLFSSL_NO_OPENSSL_RAND_CB */

static int test_wolfSSL_RAND_set_rand_method(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(WOLFSSL_NO_OPENSSL_RAND_CB)
    RAND_METHOD rand_methods = {NULL, NULL, NULL, NULL, NULL, NULL};
    unsigned char* buf = NULL;
    int num = 0;
    double entropy = 0;
    int ret;
    byte* was_cleanup_called = was_stub_rand_cleanup_called();
    byte* was_add_called = was_stub_rand_add_called();

    ExpectNotNull(buf = (byte*)XMALLOC(32 * sizeof(byte), NULL,
        DYNAMIC_TYPE_TMP_BUFFER));

    ExpectIntNE(wolfSSL_RAND_status(), 5432);
    ExpectIntEQ(*was_cleanup_called, 0);
    RAND_cleanup();
    ExpectIntEQ(*was_cleanup_called, 0);


    rand_methods.seed = &stub_rand_seed;
    rand_methods.bytes = &stub_rand_bytes;
    rand_methods.cleanup = &stub_rand_cleanup;
    rand_methods.add = &stub_rand_add;
    rand_methods.pseudorand = &stub_rand_pseudo_bytes;
    rand_methods.status = &stub_rand_status;

    ExpectIntEQ(RAND_set_rand_method(&rand_methods), WOLFSSL_SUCCESS);
    ExpectIntEQ(RAND_seed(buf, num), 123);
    ExpectIntEQ(RAND_bytes(buf, num), 456);
    ExpectIntEQ(RAND_pseudo_bytes(buf, num), 9876);
    ExpectIntEQ(RAND_status(), 5432);

    ExpectIntEQ(*was_add_called, 0);
    /* The function pointer for RAND_add returns int, but RAND_add itself
     * returns void. */
    RAND_add(buf, num, entropy);
    ExpectIntEQ(*was_add_called, 1);
    was_add_called = 0;
    ExpectIntEQ(*was_cleanup_called, 0);
    RAND_cleanup();
    ExpectIntEQ(*was_cleanup_called, 1);
    *was_cleanup_called = 0;


    ret = RAND_set_rand_method(NULL);
    ExpectIntEQ(ret, WOLFSSL_SUCCESS);
    ExpectIntNE(RAND_status(), 5432);
    ExpectIntEQ(*was_cleanup_called, 0);
    RAND_cleanup();
    ExpectIntEQ(*was_cleanup_called, 0);

    RAND_set_rand_method(NULL);

    XFREE(buf, NULL, DYNAMIC_TYPE_TMP_BUFFER);
#endif /* OPENSSL_EXTRA && !WOLFSSL_NO_OPENSSL_RAND_CB */
    return EXPECT_RESULT();
}

static int test_wolfSSL_RAND_bytes(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA)
    const int size1 = RNG_MAX_BLOCK_LEN;        /* in bytes */
    const int size2 = RNG_MAX_BLOCK_LEN + 1;    /* in bytes */
    const int size3 = RNG_MAX_BLOCK_LEN * 2;    /* in bytes */
    const int size4 = RNG_MAX_BLOCK_LEN * 4;    /* in bytes */
    int  max_bufsize;
    byte *my_buf = NULL;
#if defined(OPENSSL_EXTRA) && defined(HAVE_GETPID) && !defined(__MINGW64__) && \
    !defined(__MINGW32__)
    byte seed[16] = {0};
    byte randbuf[8] = {0};
    int pipefds[2] = {0};
    pid_t pid = 0;
#endif

    /* sanity check */
    ExpectIntEQ(RAND_bytes(NULL, 16), 0);
    ExpectIntEQ(RAND_bytes(NULL, 0), 0);

    max_bufsize = size4;

    ExpectNotNull(my_buf = (byte*)XMALLOC(max_bufsize * sizeof(byte), HEAP_HINT,
        DYNAMIC_TYPE_TMP_BUFFER));

    ExpectIntEQ(RAND_bytes(my_buf, 0), 1);
    ExpectIntEQ(RAND_bytes(my_buf, -1), 0);

    ExpectNotNull(XMEMSET(my_buf, 0, max_bufsize));
    ExpectIntEQ(RAND_bytes(my_buf, size1), 1);
    ExpectIntEQ(RAND_bytes(my_buf, size2), 1);
    ExpectIntEQ(RAND_bytes(my_buf, size3), 1);
    ExpectIntEQ(RAND_bytes(my_buf, size4), 1);
    XFREE(my_buf, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);

#if defined(OPENSSL_EXTRA) && defined(HAVE_GETPID) && !defined(__MINGW64__) && \
    !defined(__MINGW32__)
    XMEMSET(seed, 0, sizeof(seed));
    RAND_cleanup();

    /* No global methods set. */
    ExpectIntEQ(RAND_seed(seed, sizeof(seed)), 1);

    ExpectIntEQ(pipe(pipefds), 0);
    pid = fork();
    ExpectIntGE(pid, 0);
    if (pid == 0) {
        ssize_t n_written = 0;

        /* Child process. */
        close(pipefds[0]);
        RAND_bytes(randbuf, sizeof(randbuf));
        n_written = write(pipefds[1], randbuf, sizeof(randbuf));
        close(pipefds[1]);
        exit(n_written == sizeof(randbuf) ? 0 : 1);
    }
    else {
        /* Parent process. */
        byte childrand[8] = {0};
        int waitstatus = 0;

        close(pipefds[1]);
        ExpectIntEQ(RAND_bytes(randbuf, sizeof(randbuf)), 1);
        ExpectIntEQ(read(pipefds[0], childrand, sizeof(childrand)),
            sizeof(childrand));
    #ifdef WOLFSSL_NO_GETPID
        ExpectBufEQ(randbuf, childrand, sizeof(randbuf));
    #else
        ExpectBufNE(randbuf, childrand, sizeof(randbuf));
    #endif
        close(pipefds[0]);
        waitpid(pid, &waitstatus, 0);
    }
    RAND_cleanup();
#endif
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_RAND(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA)
    byte seed[16];

    XMEMSET(seed, 0, sizeof(seed));

    /* No global methods set. */
    ExpectIntEQ(RAND_seed(seed, sizeof(seed)), 1);
    ExpectIntEQ(RAND_poll(), 1);
    RAND_cleanup();

    ExpectIntEQ(RAND_egd(NULL), -1);
#ifndef NO_FILESYSTEM
    {
        char fname[100];

        ExpectNotNull(RAND_file_name(fname, (sizeof(fname) - 1)));
        ExpectIntEQ(RAND_write_file(NULL), 0);
    }
#endif
#endif
    return EXPECT_RESULT();
}


#if defined(WC_RNG_SEED_CB) && defined(OPENSSL_EXTRA)
static int wc_DummyGenerateSeed(OS_Seed* os, byte* output, word32 sz)
{
    word32 i;
    for (i = 0; i < sz; i++ )
        output[i] = (byte)i;

    (void)os;

    return 0;
}
#endif /* WC_RNG_SEED_CB */


static int test_wolfSSL_RAND_poll(void)
{
    EXPECT_DECLS;

#if defined(OPENSSL_EXTRA)
     byte seed[16];
    byte rand1[16];
#ifdef WC_RNG_SEED_CB
    byte rand2[16];
#endif

    XMEMSET(seed, 0, sizeof(seed));
    ExpectIntEQ(RAND_seed(seed, sizeof(seed)), 1);
    ExpectIntEQ(RAND_poll(), 1);
    ExpectIntEQ(RAND_bytes(rand1, 16), 1);
    RAND_cleanup();

#ifdef WC_RNG_SEED_CB
    /* Test with custom seed and poll */
    wc_SetSeed_Cb(wc_DummyGenerateSeed);

    ExpectIntEQ(RAND_seed(seed, sizeof(seed)), 1);
    ExpectIntEQ(RAND_bytes(rand1, 16), 1);
    RAND_cleanup();

    /* test that the same value is generated twice with dummy seed function */
    ExpectIntEQ(RAND_seed(seed, sizeof(seed)), 1);
    ExpectIntEQ(RAND_bytes(rand2, 16), 1);
    ExpectIntEQ(XMEMCMP(rand1, rand2, 16), 0);
    RAND_cleanup();

    /* test that doing a poll is reseeding RNG */
    ExpectIntEQ(RAND_seed(seed, sizeof(seed)), 1);
    ExpectIntEQ(RAND_poll(), 1);
    ExpectIntEQ(RAND_bytes(rand2, 16), 1);
    ExpectIntNE(XMEMCMP(rand1, rand2, 16), 0);

    /* reset the seed function used */
    wc_SetSeed_Cb(WC_GENERATE_SEED_DEFAULT);
#endif
    RAND_cleanup();

    ExpectIntEQ(RAND_egd(NULL), -1);
#endif

    return EXPECT_RESULT();
}


static int test_wolfSSL_PKCS8_Compat(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_FILESYSTEM) && defined(HAVE_ECC) && \
    !defined(NO_BIO)
    PKCS8_PRIV_KEY_INFO* pt = NULL;
    PKCS8_PRIV_KEY_INFO* pt2 = NULL;
    BIO* bio = NULL;
    XFILE f = XBADFILE;
    int bytes = 0;
    char pkcs8_buffer[512];
#if defined(OPENSSL_ALL) || defined(WOLFSSL_WPAS_SMALL)
    EVP_PKEY *pkey = NULL;
#endif

    /* file from wolfssl/certs/ directory */
    ExpectTrue((f = XFOPEN("./certs/ecc-keyPkcs8.pem", "rb")) != XBADFILE);
    ExpectIntGT((bytes = (int)XFREAD(pkcs8_buffer, 1, sizeof(pkcs8_buffer), f)),
        0);
    if (f != XBADFILE)
        XFCLOSE(f);
    ExpectNotNull(bio = BIO_new_mem_buf((void*)pkcs8_buffer, bytes));
    ExpectNotNull(pt = d2i_PKCS8_PRIV_KEY_INFO_bio(bio, NULL));

#if defined(OPENSSL_ALL) || defined(WOLFSSL_WPAS_SMALL)
    ExpectNotNull(pkey = EVP_PKCS82PKEY(pt));
    ExpectIntEQ(EVP_PKEY_type(pkey->type), EVP_PKEY_EC);

    /* gets PKCS8 pointer to pkey */
    ExpectNotNull(pt2 = EVP_PKEY2PKCS8(pkey));

    EVP_PKEY_free(pkey);
#endif

    BIO_free(bio);
    PKCS8_PRIV_KEY_INFO_free(pt);
    PKCS8_PRIV_KEY_INFO_free(pt2);
#endif
    return EXPECT_RESULT();
}

#if defined(OPENSSL_EXTRA) && !defined(NO_FILESYSTEM) && !defined(NO_BIO)
static int NoPasswordCallBack(char* passwd, int sz, int rw, void* userdata)
{
    (void)passwd;
    (void)sz;
    (void)rw;
    (void)userdata;

    return -1;
}
#endif

static int test_wolfSSL_PKCS8_d2i(void)
{
    EXPECT_DECLS;
#if !defined(HAVE_FIPS) && defined(OPENSSL_EXTRA)
    /* This test ends up using HMAC as a part of PBKDF2, and HMAC
     * requires a 12 byte password in FIPS mode. This test ends up
     * trying to use an 8 byte password. */

#ifndef NO_FILESYSTEM
    unsigned char pkcs8_buffer[2048];
    const unsigned char* p = NULL;
    int bytes = 0;
    XFILE file = XBADFILE;
    WOLFSSL_EVP_PKEY* pkey = NULL;
#ifndef NO_BIO
    BIO* bio = NULL;
    #if defined(OPENSSL_ALL) && \
            ((!defined(NO_RSA) && !defined(NO_DES3)) || \
               defined(HAVE_ECC)) && \
        !defined(NO_BIO) && !defined(NO_PWDBASED) && defined(HAVE_PKCS8)
    WOLFSSL_EVP_PKEY* evpPkey = NULL;
    #endif
#endif
#ifndef NO_RSA
    const char rsaDerPkcs8File[] = "./certs/server-keyPkcs8.der";
    const char rsaPemPkcs8File[] = "./certs/server-keyPkcs8.pem";
    #ifndef NO_DES3
    const char rsaDerPkcs8EncFile[] = "./certs/server-keyPkcs8Enc.der";
    #endif
#endif /* NO_RSA */
#ifdef HAVE_ECC
    const char ecDerPkcs8File[] = "certs/ecc-keyPkcs8.der";
    const char ecPemPkcs8File[] = "certs/ecc-keyPkcs8.pem";
    #ifndef NO_DES3
    const char ecDerPkcs8EncFile[] = "certs/ecc-keyPkcs8Enc.der";
    #endif
#endif /* HAVE_ECC */
#endif /* !NO_FILESYSTEM */

#if defined(OPENSSL_ALL) && (!defined(NO_RSA) || defined(HAVE_ECC))
#ifndef NO_RSA
    #ifdef USE_CERT_BUFFERS_1024
    const unsigned char* rsa = (unsigned char*)server_key_der_1024;
    int rsaSz = sizeof_server_key_der_1024;
    #else
    const unsigned char* rsa = (unsigned char*)server_key_der_2048;
    int rsaSz = sizeof_server_key_der_2048;
    #endif
#endif
#ifdef HAVE_ECC
    const unsigned char* ec = (unsigned char*)ecc_key_der_256;
    int ecSz = sizeof_ecc_key_der_256;
#endif
#endif /* OPENSSL_ALL && (!NO_RSA || HAVE_ECC) */


#ifndef NO_FILESYSTEM
   (void)pkcs8_buffer;
   (void)p;
   (void)bytes;
   (void)file;
#ifndef NO_BIO
   (void)bio;
#endif
#endif

#ifdef OPENSSL_ALL
#ifndef NO_RSA
    /* Try to auto-detect normal RSA private key */
    ExpectNotNull(pkey = d2i_AutoPrivateKey(NULL, &rsa, rsaSz));
    EVP_PKEY_free(pkey);
    pkey = NULL;
#endif
#ifdef HAVE_ECC
    /* Try to auto-detect normal EC private key */
    ExpectNotNull(pkey = d2i_AutoPrivateKey(NULL, &ec, ecSz));
    EVP_PKEY_free(pkey);
    pkey = NULL;
#endif
#endif /* OPENSSL_ALL */

#ifndef NO_FILESYSTEM
#if defined(OPENSSL_ALL) && !defined(NO_PWDBASED) && defined(HAVE_PKCS8)
    ExpectIntEQ(PEM_write_PKCS8PrivateKey(XBADFILE, pkey, NULL, NULL, 0, NULL,
        NULL), 0);
    ExpectIntEQ(PEM_write_PKCS8PrivateKey(stderr, NULL, NULL, NULL, 0, NULL,
        NULL), 0);
#endif

#ifndef NO_RSA
    /* Get DER encoded RSA PKCS#8 data. */
    ExpectTrue((file = XFOPEN(rsaDerPkcs8File, "rb")) != XBADFILE);
    ExpectNotNull(XMEMSET(pkcs8_buffer, 0, sizeof(pkcs8_buffer)));
    ExpectIntGT((bytes = (int)XFREAD(pkcs8_buffer, 1, sizeof(pkcs8_buffer),
        file)), 0);
    if (file != XBADFILE) {
        XFCLOSE(file);
        file = XBADFILE;
    }

    p = pkcs8_buffer;
#ifdef OPENSSL_ALL
    /* Try to decode - auto-detect key type. */
    ExpectNotNull(pkey = d2i_AutoPrivateKey(NULL, &p, bytes));
#else
    ExpectNotNull(pkey = d2i_PrivateKey(EVP_PKEY_RSA, NULL, &p, bytes));
#endif

    /* Get PEM encoded RSA PKCS#8 data. */
    ExpectTrue((file = XFOPEN(rsaPemPkcs8File, "rb")) != XBADFILE);
    ExpectIntGT((bytes = (int)XFREAD(pkcs8_buffer, 1, sizeof(pkcs8_buffer),
        file)), 0);
    if (file != XBADFILE) {
        XFCLOSE(file);
        file = XBADFILE;
    }
#if defined(OPENSSL_ALL) && \
    !defined(NO_BIO) && !defined(NO_PWDBASED) && defined(HAVE_PKCS8)
    ExpectNotNull(bio = BIO_new(BIO_s_mem()));
    ExpectIntEQ(PEM_write_bio_PKCS8PrivateKey(NULL, pkey, NULL, NULL, 0, NULL,
        NULL), 0);
    ExpectIntEQ(PEM_write_bio_PKCS8PrivateKey(bio, NULL, NULL, NULL, 0, NULL,
        NULL), 0);
    /* Write PKCS#8 PEM to BIO. */
    ExpectIntEQ(PEM_write_bio_PKCS8PrivateKey(bio, pkey, NULL, NULL, 0, NULL,
        NULL), bytes);
    /* Write PKCS#8 PEM to stderr. */
    ExpectIntEQ(PEM_write_PKCS8PrivateKey(stderr, pkey, NULL, NULL, 0, NULL,
        NULL), bytes);
    /* Compare file and written data */
    ExpectIntEQ(BIO_get_mem_data(bio, &p), bytes);
    ExpectIntEQ(XMEMCMP(p, pkcs8_buffer, bytes), 0);
    BIO_free(bio);
    bio = NULL;
#if !defined(NO_AES) && defined(HAVE_AESGCM)
    ExpectIntEQ(PEM_write_PKCS8PrivateKey(stderr, pkey, EVP_aes_128_gcm(),
        NULL, 0, PasswordCallBack, (void*)"yassl123"), 0);
#endif
#if !defined(NO_DES3) && !defined(NO_SHA)
    ExpectNotNull(bio = BIO_new(BIO_s_mem()));
    /* Write Encrypted PKCS#8 PEM to BIO. */
    bytes = 1834;
    ExpectIntEQ(PEM_write_bio_PKCS8PrivateKey(bio, pkey, EVP_des_ede3_cbc(),
        NULL, 0, PasswordCallBack, (void*)"yassl123"), bytes);
    ExpectIntEQ(PEM_write_PKCS8PrivateKey(stderr, pkey, EVP_des_ede3_cbc(),
        NULL, 0, PasswordCallBack, (void*)"yassl123"), bytes);
    ExpectNotNull(evpPkey = PEM_read_bio_PrivateKey(bio, NULL, PasswordCallBack,
        (void*)"yassl123"));
    EVP_PKEY_free(evpPkey);
    evpPkey = NULL;
    BIO_free(bio);
    bio = NULL;
#endif /* !NO_DES3 && !NO_SHA */
#endif /* !NO_BIO && !NO_PWDBASED && HAVE_PKCS8 */
    EVP_PKEY_free(pkey);
    pkey = NULL;

    /* PKCS#8 encrypted RSA key */
#ifndef NO_DES3
    ExpectTrue((file = XFOPEN(rsaDerPkcs8EncFile, "rb")) != XBADFILE);
    ExpectNotNull(XMEMSET(pkcs8_buffer, 0, sizeof(pkcs8_buffer)));
    ExpectIntGT((bytes = (int)XFREAD(pkcs8_buffer, 1, sizeof(pkcs8_buffer),
        file)), 0);
    if (file != XBADFILE) {
        XFCLOSE(file);
        file = XBADFILE;
    }
#if defined(OPENSSL_ALL) && \
    !defined(NO_BIO) && !defined(NO_PWDBASED) && defined(HAVE_PKCS8)
    ExpectNotNull(bio = BIO_new_mem_buf((void*)pkcs8_buffer, bytes));
    ExpectNotNull(pkey = d2i_PKCS8PrivateKey_bio(bio, NULL, PasswordCallBack,
        (void*)"yassl123"));
    EVP_PKEY_free(pkey);
    pkey = NULL;
    BIO_free(bio);
    bio = NULL;
#endif /* OPENSSL_ALL && !NO_BIO && !NO_PWDBASED && HAVE_PKCS8 */
#endif /* !NO_DES3 */
#endif /* NO_RSA */

#ifdef HAVE_ECC
    /* PKCS#8 encode EC key */
    ExpectTrue((file = XFOPEN(ecDerPkcs8File, "rb")) != XBADFILE);
    ExpectNotNull(XMEMSET(pkcs8_buffer, 0, sizeof(pkcs8_buffer)));
    ExpectIntGT((bytes = (int)XFREAD(pkcs8_buffer, 1, sizeof(pkcs8_buffer),
        file)), 0);
    if (file != XBADFILE) {
        XFCLOSE(file);
        file = XBADFILE;
    }

    p = pkcs8_buffer;
#ifdef OPENSSL_ALL
    /* Try to decode - auto-detect key type. */
    ExpectNotNull(pkey = d2i_AutoPrivateKey(NULL, &p, bytes));
#else
    ExpectNotNull(pkey = d2i_PrivateKey(EVP_PKEY_EC, NULL, &p, bytes));
#endif

    /* Get PEM encoded RSA PKCS#8 data. */
    ExpectTrue((file = XFOPEN(ecPemPkcs8File, "rb")) != XBADFILE);
    ExpectNotNull(XMEMSET(pkcs8_buffer, 0, sizeof(pkcs8_buffer)));
    ExpectIntGT((bytes = (int)XFREAD(pkcs8_buffer, 1, sizeof(pkcs8_buffer),
        file)), 0);
    if (file != XBADFILE) {
        XFCLOSE(file);
        file = XBADFILE;
    }
#if defined(OPENSSL_ALL) && \
    !defined(NO_BIO) && !defined(NO_PWDBASED) && defined(HAVE_PKCS8) && \
    defined(HAVE_AES_CBC) && defined(WOLFSSL_AES_256)
    ExpectNotNull(bio = BIO_new(BIO_s_mem()));
    /* Write PKCS#8 PEM to BIO. */
    ExpectIntEQ(PEM_write_bio_PKCS8PrivateKey(bio, pkey, NULL, NULL, 0, NULL,
        NULL), bytes);
    ExpectIntEQ(PEM_write_PKCS8PrivateKey(stderr, pkey, NULL, NULL, 0, NULL,
        NULL), bytes);
    /* Compare file and written data */
    ExpectIntEQ(BIO_get_mem_data(bio, &p), bytes);
    ExpectIntEQ(XMEMCMP(p, pkcs8_buffer, bytes), 0);
    BIO_free(bio);
    bio = NULL;
    ExpectNotNull(bio = BIO_new(BIO_s_mem()));
    /* Write Encrypted PKCS#8 PEM to BIO (test write 0 then 379) */
    bytes = 379;
    ExpectIntEQ(PEM_write_bio_PKCS8PrivateKey(bio, pkey, EVP_aes_256_cbc(),
        NULL, 0, NoPasswordCallBack, (void*)"yassl123"), 0);
    ExpectIntEQ(PEM_write_bio_PKCS8PrivateKey(bio, pkey, EVP_aes_256_cbc(),
        NULL, 0, PasswordCallBack, (void*)"yassl123"), bytes);

    /* invalid cases to stderr */
    #ifdef WOLFSSL_AES_128
    ExpectIntEQ(PEM_write_PKCS8PrivateKey(stderr, pkey, EVP_aes_128_cbc(),
        NULL, 0, PasswordCallBack, (void*)"yassl123"), bytes);
    ExpectIntEQ(PEM_write_PKCS8PrivateKey(stderr, pkey, EVP_aes_128_cbc(),
        (char*)"yassl123", 8, PasswordCallBack, NULL), bytes);
    #endif
    ExpectIntEQ(PEM_write_PKCS8PrivateKey(stderr, pkey, EVP_aes_256_cbc(),
        NULL, 0, PasswordCallBack, (void*)"yassl123"), bytes);
    ExpectIntEQ(PEM_write_PKCS8PrivateKey(stderr, pkey, EVP_aes_256_cbc(),
        (char*)"yassl123", 8, PasswordCallBack, NULL), bytes);

    /* read/decode private key with password */
    ExpectNotNull(evpPkey = PEM_read_bio_PrivateKey(bio, NULL, PasswordCallBack,
        (void*)"yassl123"));
    EVP_PKEY_free(evpPkey);
    evpPkey = NULL;
    BIO_free(bio);
    bio = NULL;

    /* https://github.com/wolfSSL/wolfssl/issues/8610 */
    bytes = (int)XSTRLEN((char *)pkcs8_buffer);
    ExpectNotNull(bio = BIO_new_mem_buf((void*)pkcs8_buffer, bytes));
    ExpectIntEQ(BIO_get_mem_data(bio, &p), bytes);
    ExpectIntEQ(XMEMCMP(p, pkcs8_buffer, bytes), 0);

    ExpectNotNull(evpPkey = PEM_read_bio_PrivateKey(bio, NULL, NULL,
            (void*)"yassl123"));
    ExpectIntEQ(PEM_write_PKCS8PrivateKey(stderr, evpPkey, NULL,
            NULL, 0, NULL, NULL), bytes);
    EVP_PKEY_free(evpPkey);
    evpPkey = NULL;
    BIO_free(bio);
    bio = NULL;
#endif /* OPENSSL_ALL && !NO_BIO && !NO_PWDBASED && HAVE_PKCS8 && HAVE_AES_CBC */
    EVP_PKEY_free(pkey);
    pkey = NULL;

    /* PKCS#8 encrypted EC key */
#ifndef NO_DES3
    ExpectTrue((file = XFOPEN(ecDerPkcs8EncFile, "rb")) != XBADFILE);
    ExpectNotNull(XMEMSET(pkcs8_buffer, 0, sizeof(pkcs8_buffer)));
    ExpectIntGT((bytes = (int)XFREAD(pkcs8_buffer, 1, sizeof(pkcs8_buffer),
        file)), 0);
    if (file != XBADFILE) {
        XFCLOSE(file);
        file = XBADFILE;
    }
#if defined(OPENSSL_ALL) && \
    !defined(NO_BIO) && !defined(NO_PWDBASED) && defined(HAVE_PKCS8)
    ExpectNotNull(bio = BIO_new_mem_buf((void*)pkcs8_buffer, bytes));
    ExpectNotNull(pkey = d2i_PKCS8PrivateKey_bio(bio, NULL, PasswordCallBack,
                                                            (void*)"yassl123"));
    EVP_PKEY_free(pkey);
    pkey = NULL;
    BIO_free(bio);
    bio = NULL;
#endif /* OPENSSL_ALL && !NO_BIO && !NO_PWDBASED && HAVE_PKCS8 */
#endif /* !NO_DES3 */
#endif /* HAVE_ECC */

#endif /* !NO_FILESYSTEM */
#endif /* HAVE_FIPS && OPENSSL_EXTRA */
    return EXPECT_RESULT();
}

#if !defined(SINGLE_THREADED) && defined(ERROR_QUEUE_PER_THREAD) && \
    !defined(NO_ERROR_QUEUE) && defined(OPENSSL_EXTRA) && \
    defined(DEBUG_WOLFSSL)
#define LOGGING_THREADS 5
#define ERROR_COUNT 10
/* copied from logging.c since this is not exposed otherwise */
#ifndef ERROR_QUEUE_MAX
#ifdef ERROR_QUEUE_PER_THREAD
    #define ERROR_QUEUE_MAX 16
#else
    /* this breaks from compat of unlimited error queue size */
    #define ERROR_QUEUE_MAX 100
#endif
#endif

static volatile int loggingThreadsReady;
static THREAD_RETURN WOLFSSL_THREAD test_logging(void* args)
{
    const char* file;
    int line;
    unsigned long err;
    int errorCount = 0;
    int i;

    (void)args;

    while (!loggingThreadsReady);
    for (i = 0; i < ERROR_COUNT; i++)
        ERR_put_error(ERR_LIB_PEM, SYS_F_ACCEPT, -990 - i, __FILE__, __LINE__);

    while ((err = ERR_get_error_line(&file, &line))) {
        AssertIntEQ(err, 990 + errorCount);
        errorCount++;
    }
    AssertIntEQ(errorCount, ERROR_COUNT);

    /* test max queue behavior, trying to add an arbitrary 3 errors over */
    ERR_clear_error(); /* ERR_get_error_line() does not remove */
    errorCount = 0;
    for (i = 0; i < ERROR_QUEUE_MAX + 3; i++)
        ERR_put_error(ERR_LIB_PEM, SYS_F_ACCEPT, -990 - i, __FILE__, __LINE__);

    while ((err = ERR_get_error_line(&file, &line))) {
        AssertIntEQ(err, 990 + errorCount);
        errorCount++;
    }

    /* test that the 3 errors over the max were dropped */
    AssertIntEQ(errorCount, ERROR_QUEUE_MAX);

    WOLFSSL_RETURN_FROM_THREAD(0);
}
#endif

static int test_error_queue_per_thread(void)
{
    int res = TEST_SKIPPED;
#if !defined(SINGLE_THREADED) && defined(ERROR_QUEUE_PER_THREAD) && \
    !defined(NO_ERROR_QUEUE) && defined(OPENSSL_EXTRA) && \
    defined(DEBUG_WOLFSSL)
    THREAD_TYPE loggingThreads[LOGGING_THREADS];
    int i;

    ERR_clear_error(); /* clear out any error nodes */

    loggingThreadsReady = 0;
    for (i = 0; i < LOGGING_THREADS; i++)
        start_thread(test_logging, NULL, &loggingThreads[i]);
    loggingThreadsReady = 1;
    for (i = 0; i < LOGGING_THREADS; i++)
        join_thread(loggingThreads[i]);

    res = TEST_SUCCESS;
#endif
    return res;
}

static int test_wolfSSL_ERR_put_error(void)
{
    EXPECT_DECLS;
#if !defined(NO_ERROR_QUEUE) && defined(OPENSSL_EXTRA) && \
    defined(DEBUG_WOLFSSL)
    const char* file;
    int line;

    ERR_clear_error(); /* clear out any error nodes */
    ERR_put_error(0,SYS_F_ACCEPT, 0, "this file", 0);
    ExpectIntEQ(ERR_get_error_line(&file, &line), 0);
    ERR_put_error(0,SYS_F_BIND, 1, "this file", 1);
    ExpectIntEQ(ERR_get_error_line(&file, &line), 1);
    ERR_put_error(0,SYS_F_CONNECT, 2, "this file", 2);
    ExpectIntEQ(ERR_get_error_line(&file, &line), 2);
    ERR_put_error(0,SYS_F_FOPEN, 3, "this file", 3);
    ExpectIntEQ(ERR_get_error_line(&file, &line), 3);
    ERR_put_error(0,SYS_F_FREAD, 4, "this file", 4);
    ExpectIntEQ(ERR_get_error_line(&file, &line), 4);
    ERR_put_error(0,SYS_F_GETADDRINFO, 5, "this file", 5);
    ExpectIntEQ(ERR_get_error_line(&file, &line), 5);
    ERR_put_error(0,SYS_F_GETSOCKOPT, 6, "this file", 6);
    ExpectIntEQ(ERR_get_error_line(&file, &line), 6);
    ERR_put_error(0,SYS_F_GETSOCKNAME, 7, "this file", 7);
    ExpectIntEQ(ERR_get_error_line(&file, &line), 7);
    ERR_put_error(0,SYS_F_GETHOSTBYNAME, 8, "this file", 8);
    ExpectIntEQ(ERR_get_error_line(&file, &line), 8);
    ERR_put_error(0,SYS_F_GETNAMEINFO, 9, "this file", 9);
    ExpectIntEQ(ERR_get_error_line(&file, &line), 9);
    ERR_put_error(0,SYS_F_GETSERVBYNAME, 10, "this file", 10);
    ExpectIntEQ(ERR_get_error_line(&file, &line), 10);
    ERR_put_error(0,SYS_F_IOCTLSOCKET, 11, "this file", 11);
    ExpectIntEQ(ERR_get_error_line(&file, &line), 11);
    ERR_put_error(0,SYS_F_LISTEN, 12, "this file", 12);
    ExpectIntEQ(ERR_get_error_line(&file, &line), 12);
    ERR_put_error(0,SYS_F_OPENDIR, 13, "this file", 13);
    ExpectIntEQ(ERR_get_error_line(&file, &line), 13);
    ERR_put_error(0,SYS_F_SETSOCKOPT, 14, "this file", 14);
    ExpectIntEQ(ERR_get_error_line(&file, &line), 14);
    ERR_put_error(0,SYS_F_SOCKET, 15, "this file", 15);
    ExpectIntEQ(ERR_get_error_line(&file, &line), 15);

#if defined(OPENSSL_ALL) && defined(WOLFSSL_PYTHON)
    ERR_put_error(ERR_LIB_ASN1, SYS_F_ACCEPT, ASN1_R_HEADER_TOO_LONG,
            "this file", 100);
    ExpectIntEQ(wolfSSL_ERR_peek_last_error_line(&file, &line),
            (ERR_LIB_ASN1 << 24) | ASN1_R_HEADER_TOO_LONG);
    ExpectIntEQ(line, 100);
    ExpectIntEQ(wolfSSL_ERR_peek_error(),
            (ERR_LIB_ASN1 << 24) | ASN1_R_HEADER_TOO_LONG);
    ExpectIntEQ(ERR_get_error_line(&file, &line), ASN1_R_HEADER_TOO_LONG);
#endif

    /* try reading past end of error queue */
    file = NULL;
    ExpectIntEQ(ERR_get_error_line(&file, &line), 0);
    ExpectNull(file);
    ExpectIntEQ(ERR_get_error_line_data(&file, &line, NULL, NULL), 0);

    PEMerr(4,4);
    ExpectIntEQ(ERR_get_error(), 4);
    /* Empty and free up all error nodes */
    ERR_clear_error();

    /* Verify all nodes are cleared */
    ERR_put_error(0,SYS_F_ACCEPT, 0, "this file", 0);
    ERR_clear_error();
    ExpectIntEQ(ERR_get_error_line(&file, &line), 0);
#endif
    return EXPECT_RESULT();
}

/*
 * This is a regression test for a bug where the peek/get error functions were
 * drawing from the end of the queue rather than the front.
 */
static int test_wolfSSL_ERR_get_error_order(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_HAVE_ERROR_QUEUE) && defined(OPENSSL_EXTRA)
    /* Empty the queue. */
    wolfSSL_ERR_clear_error();

    wolfSSL_ERR_put_error(0, 0, WC_NO_ERR_TRACE(ASN_NO_SIGNER_E), "test", 0);
    wolfSSL_ERR_put_error(0, 0, WC_NO_ERR_TRACE(ASN_SELF_SIGNED_E), "test", 0);

    ExpectIntEQ(wolfSSL_ERR_peek_error(), -WC_NO_ERR_TRACE(ASN_NO_SIGNER_E));
    ExpectIntEQ(wolfSSL_ERR_get_error(), -WC_NO_ERR_TRACE(ASN_NO_SIGNER_E));
    ExpectIntEQ(wolfSSL_ERR_peek_error(), -WC_NO_ERR_TRACE(ASN_SELF_SIGNED_E));
    ExpectIntEQ(wolfSSL_ERR_get_error(), -WC_NO_ERR_TRACE(ASN_SELF_SIGNED_E));
#endif /* WOLFSSL_HAVE_ERROR_QUEUE && OPENSSL_EXTRA */
    return EXPECT_RESULT();
}

#ifndef NO_BIO

static int test_wolfSSL_ERR_print_errors(void)
{
    EXPECT_DECLS;
#if !defined(NO_ERROR_QUEUE) && defined(OPENSSL_EXTRA) && \
    defined(DEBUG_WOLFSSL) && !defined(NO_ERROR_STRINGS)
    BIO* bio = NULL;
    char buf[1024];

    ExpectNotNull(bio = BIO_new(BIO_s_mem()));
    ERR_clear_error(); /* clear out any error nodes */
    ERR_put_error(0,SYS_F_ACCEPT, -173, "ssl.c", 0);
    /* Choosing -600 as an unused errno. */
    ERR_put_error(0,SYS_F_BIND, -600, "asn.c", 100);

    ERR_print_errors(bio);
    ExpectIntEQ(BIO_gets(bio, buf, sizeof(buf)), 56);
    ExpectIntEQ(XSTRNCMP(
        "error:173:wolfSSL library:Bad function argument:ssl.c:0",
        buf, 55), 0);
    ExpectIntEQ(BIO_gets(bio, buf, sizeof(buf)), 57);
    ExpectIntEQ(XSTRNCMP(
        "error:600:wolfSSL library:unknown error number:asn.c:100",
        buf, 56), 0);
    ExpectIntEQ(BIO_gets(bio, buf, sizeof(buf)), 1);
    ExpectIntEQ(buf[0], '\0');
    ExpectIntEQ(ERR_get_error_line(NULL, NULL), 0);

    BIO_free(bio);
#endif
    return EXPECT_RESULT();
}

#if !defined(NO_ERROR_QUEUE) && defined(OPENSSL_EXTRA) && \
    defined(DEBUG_WOLFSSL)
static int test_wolfSSL_error_cb(const char *str, size_t len, void *u)
{
    if (u != NULL) {
        wolfSSL_BIO_write((BIO*)u, str, (int)len);
    }
    return 0;
}
#endif

static int test_wolfSSL_ERR_print_errors_cb(void)
{
    EXPECT_DECLS;
#if !defined(NO_ERROR_QUEUE) && defined(OPENSSL_EXTRA) && \
    defined(DEBUG_WOLFSSL)
    BIO* bio = NULL;
    char buf[1024];

    ExpectNotNull(bio = BIO_new(BIO_s_mem()));
    ERR_clear_error(); /* clear out any error nodes */
    ERR_put_error(0,SYS_F_ACCEPT, -173, "ssl.c", 0);
    ERR_put_error(0,SYS_F_BIND, -275, "asn.c", 100);

    ERR_print_errors_cb(test_wolfSSL_error_cb, bio);
    ExpectIntEQ(BIO_gets(bio, buf, sizeof(buf)), 108);
    ExpectIntEQ(XSTRNCMP(
        "wolfSSL error occurred, error = 173 line:0 file:ssl.c",
        buf, 53), 0);
    ExpectIntEQ(XSTRNCMP(
         "wolfSSL error occurred, error = 275 line:100 file:asn.c",
         buf + 53, 55), 0);
    ExpectIntEQ(BIO_gets(bio, buf, sizeof(buf)), 0);

    BIO_free(bio);
#endif

    return EXPECT_RESULT();
}
/*
 * Testing WOLFSSL_ERROR_MSG
 */
static int test_WOLFSSL_ERROR_MSG(void)
{
    int res = TEST_SKIPPED;
#if defined(DEBUG_WOLFSSL) || defined(OPENSSL_ALL) || defined(WOLFSSL_NGINX) ||\
    defined(WOLFSSL_HAPROXY) || defined(OPENSSL_EXTRA)
    const char* msg = TEST_STRING;

    WOLFSSL_ERROR_MSG(msg);

    res = TEST_SUCCESS;
#endif
    return res;
} /* End test_WOLFSSL_ERROR_MSG */
/*
 * Testing wc_ERR_remove_state
 */
static int test_wc_ERR_remove_state(void)
{
    int res = TEST_SKIPPED;
#if defined(OPENSSL_EXTRA) || defined(DEBUG_WOLFSSL_VERBOSE)
    wc_ERR_remove_state();

    res = TEST_SUCCESS;
#endif
    return res;
} /* End test_wc_ERR_remove_state */
/*
 * Testing wc_ERR_print_errors_fp
 */
static int test_wc_ERR_print_errors_fp(void)
{
    EXPECT_DECLS;
#if (defined(OPENSSL_EXTRA) || defined(DEBUG_WOLFSSL_VERBOSE)) && \
    (!defined(NO_FILESYSTEM) && !defined(NO_STDIO_FILESYSTEM))
    long sz;
    XFILE fp = XBADFILE;

    WOLFSSL_ERROR(WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectTrue((fp = XFOPEN("./tests/test-log-dump-to-file.txt", "a+")) !=
        XBADFILE);
    wc_ERR_print_errors_fp(fp);
#if defined(DEBUG_WOLFSSL)
    ExpectTrue(XFSEEK(fp, 0, XSEEK_END) == 0);
    #ifdef NO_ERROR_QUEUE
        ExpectIntEQ(sz = XFTELL(fp), 0);
    #else
        ExpectIntNE(sz = XFTELL(fp), 0);
    #endif
#endif
    if (fp != XBADFILE)
        XFCLOSE(fp);
    (void)sz;
#endif
    return EXPECT_RESULT();
} /* End test_wc_ERR_print_errors_fp */
#ifdef DEBUG_WOLFSSL
static void Logging_cb(const int logLevel, const char *const logMessage)
{
    (void)logLevel;
    (void)logMessage;
}
#endif
/*
 * Testing wolfSSL_GetLoggingCb
 */
static int test_wolfSSL_GetLoggingCb(void)
{
    EXPECT_DECLS;
#ifdef DEBUG_WOLFSSL
    /* Testing without wolfSSL_SetLoggingCb() */
    ExpectNull(wolfSSL_GetLoggingCb());
    /* Testing with wolfSSL_SetLoggingCb() */
    ExpectIntEQ(wolfSSL_SetLoggingCb(Logging_cb), 0);
    ExpectNotNull(wolfSSL_GetLoggingCb());
    ExpectIntEQ(wolfSSL_SetLoggingCb(NULL), 0);
#endif
    ExpectNull(wolfSSL_GetLoggingCb());

    return EXPECT_RESULT();
} /* End test_wolfSSL_GetLoggingCb */

#endif /* !NO_BIO */

static int test_wolfSSL_OBJ(void)
{
/* Password "wolfSSL test" is only 12 (96-bit) too short for testing in FIPS
 * mode
 */
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_SHA256) && !defined(NO_ASN) && \
    !defined(HAVE_FIPS) && !defined(NO_SHA) && defined(WOLFSSL_CERT_EXT) && \
    defined(WOLFSSL_CERT_GEN) && !defined(NO_BIO) && \
    !defined(NO_FILESYSTEM) && !defined(NO_STDIO_FILESYSTEM)
    ASN1_OBJECT *obj = NULL;
    ASN1_OBJECT *obj2 = NULL;
    char buf[50];

    XFILE fp = XBADFILE;
    X509 *x509 = NULL;
    X509_NAME *x509Name = NULL;
    X509_NAME_ENTRY *x509NameEntry = NULL;
    ASN1_OBJECT *asn1Name = NULL;
    int numNames = 0;
    BIO *bio = NULL;
    int nid;
    int i, j;
    const char *f[] = {
    #ifndef NO_RSA
        "./certs/ca-cert.der",
    #endif
    #ifdef HAVE_ECC
        "./certs/ca-ecc-cert.der",
        "./certs/ca-ecc384-cert.der",
    #endif
        NULL};
    ASN1_OBJECT *field_name_obj = NULL;
    int lastpos = -1;
    int tmp = -1;
    ASN1_STRING *asn1 = NULL;
    unsigned char *buf_dyn = NULL;

    ExpectIntEQ(OBJ_obj2txt(buf, (int)sizeof(buf), obj, 1), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectNotNull(obj = OBJ_nid2obj(NID_any_policy));
    ExpectIntEQ(OBJ_obj2nid(obj), NID_any_policy);
    ExpectIntEQ(OBJ_obj2txt(buf, (int)sizeof(buf), obj, 1), 11);
    ExpectIntGT(OBJ_obj2txt(buf, (int)sizeof(buf), obj, 0), 0);
    ASN1_OBJECT_free(obj);
    obj = NULL;

    ExpectNotNull(obj = OBJ_nid2obj(NID_sha256));
    ExpectIntEQ(OBJ_obj2nid(obj), NID_sha256);
    ExpectIntEQ(OBJ_obj2txt(buf, (int)sizeof(buf), obj, 1), 22);
#ifdef WOLFSSL_CERT_EXT
    ExpectIntEQ(OBJ_txt2nid(buf), NID_sha256);
#endif
    ExpectIntGT(OBJ_obj2txt(buf, (int)sizeof(buf), obj, 0), 0);
    ExpectNotNull(obj2 = OBJ_dup(obj));
    ExpectIntEQ(OBJ_cmp(obj, obj2), 0);
    ASN1_OBJECT_free(obj);
    obj = NULL;
    ASN1_OBJECT_free(obj2);
    obj2 = NULL;

    for (i = 0; f[i] != NULL; i++)
    {
        ExpectTrue((fp = XFOPEN(f[i], "rb")) != XBADFILE);
        ExpectNotNull(x509 = d2i_X509_fp(fp, NULL));
        if (fp != XBADFILE) {
            XFCLOSE(fp);
            fp = XBADFILE;
        }
        ExpectNotNull(x509Name = X509_get_issuer_name(x509));
        ExpectIntNE((numNames = X509_NAME_entry_count(x509Name)), 0);

        /* Get the Common Name by using OBJ_txt2obj */
        ExpectNotNull(field_name_obj = OBJ_txt2obj("CN", 0));
        ExpectIntEQ(X509_NAME_get_index_by_OBJ(NULL, NULL, 99),
            WOLFSSL_FATAL_ERROR);
        ExpectIntEQ(X509_NAME_get_index_by_OBJ(x509Name, NULL, 99),
            WOLFSSL_FATAL_ERROR);
        ExpectIntEQ(X509_NAME_get_index_by_OBJ(NULL, field_name_obj, 99),
            WOLFSSL_FATAL_ERROR);
        ExpectIntEQ(X509_NAME_get_index_by_OBJ(x509Name, field_name_obj, 99),
            WOLFSSL_FATAL_ERROR);
        ExpectIntEQ(X509_NAME_get_index_by_OBJ(x509Name, NULL, 0),
            WOLFSSL_FATAL_ERROR);
        do
        {
            lastpos = tmp;
            tmp = X509_NAME_get_index_by_OBJ(x509Name, field_name_obj, lastpos);
        } while (tmp > -1);
        ExpectIntNE(lastpos, -1);
        ASN1_OBJECT_free(field_name_obj);
        field_name_obj = NULL;
        ExpectNotNull(x509NameEntry = X509_NAME_get_entry(x509Name, lastpos));
        ExpectNotNull(asn1 = X509_NAME_ENTRY_get_data(x509NameEntry));
        ExpectIntGE(ASN1_STRING_to_UTF8(&buf_dyn, asn1), 0);
        /*
         * All Common Names should be www.wolfssl.com
         * This makes testing easier as we can test for the expected value.
         */
        ExpectStrEQ((char*)buf_dyn, "www.wolfssl.com");
        OPENSSL_free(buf_dyn);
        buf_dyn = NULL;
        bio = BIO_new(BIO_s_mem());
        ExpectTrue(bio != NULL);
        for (j = 0; j < numNames; j++)
        {
            ExpectNotNull(x509NameEntry = X509_NAME_get_entry(x509Name, j));
            ExpectNotNull(asn1Name = X509_NAME_ENTRY_get_object(x509NameEntry));
            ExpectTrue((nid = OBJ_obj2nid(asn1Name)) > 0);
        }
        BIO_free(bio);
        bio = NULL;
        X509_free(x509);
        x509 = NULL;

    }

#ifdef HAVE_PKCS12
    {
        PKCS12 *p12 = NULL;
        int boolRet;
        EVP_PKEY *pkey = NULL;
        const char *p12_f[] = {
            /* bundle uses AES-CBC 256 and PKCS7 key uses DES3 */
        #if !defined(NO_DES3) && defined(WOLFSSL_AES_256) && !defined(NO_RSA)
            "./certs/test-servercert.p12",
        #endif
            NULL
        };

        for (i = 0; p12_f[i] != NULL; i++)
        {
            ExpectTrue((fp = XFOPEN(p12_f[i], "rb")) != XBADFILE);
            ExpectNotNull(p12 = d2i_PKCS12_fp(fp, NULL));
            if (fp != XBADFILE) {
                XFCLOSE(fp);
                fp = XBADFILE;
            }
            ExpectTrue((boolRet = PKCS12_parse(p12, "wolfSSL test",
                                               &pkey, &x509, NULL)) > 0);
            wc_PKCS12_free(p12);
            p12 = NULL;
            EVP_PKEY_free(pkey);
            x509Name = X509_get_issuer_name(x509);
            ExpectNotNull(x509Name);
            ExpectIntNE((numNames = X509_NAME_entry_count(x509Name)), 0);
            ExpectTrue((bio = BIO_new(BIO_s_mem())) != NULL);
            for (j = 0; j < numNames; j++)
            {
                ExpectNotNull(x509NameEntry = X509_NAME_get_entry(x509Name, j));
                ExpectNotNull(asn1Name =
                        X509_NAME_ENTRY_get_object(x509NameEntry));
                ExpectTrue((nid = OBJ_obj2nid(asn1Name)) > 0);
            }
            BIO_free(bio);
            bio = NULL;
            X509_free(x509);
            x509 = NULL;
        }
    }
#endif /* HAVE_PKCS12 */
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_OBJ_cmp(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_SHA256)
    ASN1_OBJECT *obj = NULL;
    ASN1_OBJECT *obj2 = NULL;

    ExpectNotNull(obj = OBJ_nid2obj(NID_any_policy));
    ExpectNotNull(obj2 = OBJ_nid2obj(NID_sha256));

    ExpectIntEQ(OBJ_cmp(NULL, NULL), WC_NO_ERR_TRACE(WOLFSSL_FATAL_ERROR));
    ExpectIntEQ(OBJ_cmp(obj, NULL), WC_NO_ERR_TRACE(WOLFSSL_FATAL_ERROR));
    ExpectIntEQ(OBJ_cmp(NULL, obj2), WC_NO_ERR_TRACE(WOLFSSL_FATAL_ERROR));
    ExpectIntEQ(OBJ_cmp(obj, obj2), WC_NO_ERR_TRACE(WOLFSSL_FATAL_ERROR));
    ExpectIntEQ(OBJ_cmp(obj, obj), 0);
    ExpectIntEQ(OBJ_cmp(obj2, obj2), 0);

    ASN1_OBJECT_free(obj);
    ASN1_OBJECT_free(obj2);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_OBJ_txt2nid(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL) || \
    defined(WOLFSSL_APACHE_HTTPD)
    int i;
    static const struct {
        const char* sn;
        const char* ln;
        const char* oid;
        int nid;
    } testVals[] = {
#ifdef WOLFSSL_APACHE_HTTPD
        { "tlsfeature", "TLS Feature", "1.3.6.1.5.5.7.1.24", NID_tlsfeature },
        { "id-on-dnsSRV", "SRVName", "1.3.6.1.5.5.7.8.7",
                                                             NID_id_on_dnsSRV },
        { "msUPN", "Microsoft User Principal Name",
                                         "1.3.6.1.4.1.311.20.2.3", NID_ms_upn },
#endif
        { NULL, NULL, NULL, NID_undef }
    };

    /* Invalid cases */
    ExpectIntEQ(OBJ_txt2nid(NULL), NID_undef);
    ExpectIntEQ(OBJ_txt2nid("Bad name"), NID_undef);

    /* Valid cases */
    for (i = 0; testVals[i].sn != NULL; i++) {
        ExpectIntEQ(OBJ_txt2nid(testVals[i].sn), testVals[i].nid);
        ExpectIntEQ(OBJ_txt2nid(testVals[i].ln), testVals[i].nid);
        ExpectIntEQ(OBJ_txt2nid(testVals[i].oid), testVals[i].nid);
    }
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_OBJ_txt2obj(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_APACHE_HTTPD) || (defined(OPENSSL_EXTRA) && \
        defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_CERT_GEN))
    int i;
    char buf[50];
    ASN1_OBJECT* obj = NULL;
    static const struct {
        const char* oidStr;
        const char* sn;
        const char* ln;
    } objs_list[] = {
    #if defined(WOLFSSL_APACHE_HTTPD)
        { "1.3.6.1.5.5.7.1.24", "tlsfeature", "TLS Feature" },
        { "1.3.6.1.5.5.7.8.7", "id-on-dnsSRV", "SRVName" },
    #endif
        { "2.5.29.19", "basicConstraints", "X509v3 Basic Constraints"},
        { NULL, NULL, NULL }
    };
    static const struct {
        const char* numeric;
        const char* name;
    } objs_named[] = {
        /* In dictionary but not in normal list. */
        { "1.3.6.1.5.5.7.3.8", "Time Stamping" },
        /* Made up OID. */
        { "1.3.5.7",           "1.3.5.7" },
        { NULL, NULL }
    };

    ExpectNull(obj = OBJ_txt2obj("Bad name", 0));
    ASN1_OBJECT_free(obj);
    obj = NULL;
    ExpectNull(obj = OBJ_txt2obj(NULL, 0));
    ASN1_OBJECT_free(obj);
    obj = NULL;

    for (i = 0; objs_list[i].oidStr != NULL; i++) {
        /* Test numerical value of oid (oidStr) */
        ExpectNotNull(obj = OBJ_txt2obj(objs_list[i].oidStr, 1));
        /* Convert object back to text to confirm oid is correct */
        wolfSSL_OBJ_obj2txt(buf, (int)sizeof(buf), obj, 1);
        ExpectIntEQ(XSTRNCMP(buf, objs_list[i].oidStr, (int)XSTRLEN(buf)), 0);
        ASN1_OBJECT_free(obj);
        obj = NULL;
        XMEMSET(buf, 0, sizeof(buf));

        /* Test short name (sn) */
        ExpectNull(obj = OBJ_txt2obj(objs_list[i].sn, 1));
        ExpectNotNull(obj = OBJ_txt2obj(objs_list[i].sn, 0));
        /* Convert object back to text to confirm oid is correct */
        wolfSSL_OBJ_obj2txt(buf, (int)sizeof(buf), obj, 1);
        ExpectIntEQ(XSTRNCMP(buf, objs_list[i].oidStr, (int)XSTRLEN(buf)), 0);
        ASN1_OBJECT_free(obj);
        obj = NULL;
        XMEMSET(buf, 0, sizeof(buf));

        /* Test long name (ln) - should fail when no_name = 1 */
        ExpectNull(obj = OBJ_txt2obj(objs_list[i].ln, 1));
        ExpectNotNull(obj = OBJ_txt2obj(objs_list[i].ln, 0));
        /* Convert object back to text to confirm oid is correct */
        wolfSSL_OBJ_obj2txt(buf, (int)sizeof(buf), obj, 1);
        ExpectIntEQ(XSTRNCMP(buf, objs_list[i].oidStr, (int)XSTRLEN(buf)), 0);
        ASN1_OBJECT_free(obj);
        obj = NULL;
        XMEMSET(buf, 0, sizeof(buf));
    }

    for (i = 0; objs_named[i].numeric != NULL; i++) {
        ExpectNotNull(obj = OBJ_txt2obj(objs_named[i].numeric, 1));
        wolfSSL_OBJ_obj2txt(buf, (int)sizeof(buf), obj, 0);
        ExpectIntEQ(XSTRNCMP(buf, objs_named[i].name, (int)XSTRLEN(buf)), 0);
        wolfSSL_OBJ_obj2txt(buf, (int)sizeof(buf), obj, 1);
        ExpectIntEQ(XSTRNCMP(buf, objs_named[i].numeric, (int)XSTRLEN(buf)), 0);
        ASN1_OBJECT_free(obj);
        obj = NULL;
    }
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_PEM_write_bio_X509(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && defined(OPENSSL_ALL) && \
    defined(WOLFSSL_AKID_NAME) && defined(WOLFSSL_CERT_EXT) && \
    defined(WOLFSSL_CERT_GEN) && !defined(NO_BIO) && !defined(NO_RSA) && \
    !defined(NO_FILESYSTEM)
    /* This test contains the hard coded expected
     * lengths. Update if necessary */
    XFILE fp = XBADFILE;
    WOLFSSL_EVP_PKEY *priv = NULL;

    BIO* input = NULL;
    BIO* output = NULL;
    X509* x509a = NULL;
    X509* x509b = NULL;
    X509* empty = NULL;

    ASN1_TIME* notBeforeA = NULL;
    ASN1_TIME* notAfterA  = NULL;
#ifndef NO_ASN_TIME
    ASN1_TIME* notBeforeB = NULL;
    ASN1_TIME* notAfterB  = NULL;
#endif
    int expectedLen;

    ExpectTrue((fp = XFOPEN("certs/server-key.pem", "rb")) != XBADFILE);
    ExpectNotNull(priv = wolfSSL_PEM_read_PrivateKey(fp, NULL, NULL, NULL));
    if (fp != XBADFILE) {
        XFCLOSE(fp);
        fp = XBADFILE;
    }

    ExpectNotNull(input = BIO_new_file("certs/test/cert-ext-multiple.pem",
        "rb"));
    ExpectIntEQ(wolfSSL_BIO_get_len(input), 2000);

    /* read PEM into X509 struct, get notBefore / notAfter to verify against */
    ExpectNotNull(PEM_read_bio_X509(input, &x509a, NULL, NULL));
    ExpectNotNull(notBeforeA = X509_get_notBefore(x509a));
    ExpectNotNull(notAfterA = X509_get_notAfter(x509a));

    /* write X509 back to PEM BIO; no need to sign as nothing changed. */
    ExpectNotNull(output = BIO_new(wolfSSL_BIO_s_mem()));
    ExpectNotNull(empty = wolfSSL_X509_new());
    ExpectIntEQ(PEM_write_bio_X509(NULL, NULL), WOLFSSL_FAILURE);
    ExpectIntEQ(PEM_write_bio_X509(output, NULL), WOLFSSL_FAILURE);
    ExpectIntEQ(PEM_write_bio_X509(NULL, x509a), WOLFSSL_FAILURE);
    ExpectIntEQ(PEM_write_bio_X509(output, empty), WOLFSSL_FAILURE);
    ExpectIntEQ(PEM_write_bio_X509(output, x509a), WOLFSSL_SUCCESS);
    /* compare length against expected */
    expectedLen = 2000;
    ExpectIntEQ(wolfSSL_BIO_get_len(output), expectedLen);
    wolfSSL_X509_free(empty);

#ifndef NO_ASN_TIME
    /* read exported X509 PEM back into struct, sanity check on export,
     * make sure notBefore/notAfter are the same and certs are identical. */
    ExpectNotNull(PEM_read_bio_X509(output, &x509b, NULL, NULL));
    ExpectNotNull(notBeforeB = X509_get_notBefore(x509b));
    ExpectNotNull(notAfterB = X509_get_notAfter(x509b));
    ExpectIntEQ(ASN1_TIME_compare(notBeforeA, notBeforeB), 0);
    ExpectIntEQ(ASN1_TIME_compare(notAfterA, notAfterB), 0);
    ExpectIntEQ(0, wolfSSL_X509_cmp(x509a, x509b));
    X509_free(x509b);
    x509b = NULL;
#endif

    /* Reset output buffer */
    BIO_free(output);
    output = NULL;
    ExpectNotNull(output = BIO_new(wolfSSL_BIO_s_mem()));

    /* Test forcing the AKID to be generated just from KeyIdentifier */
    if (EXPECT_SUCCESS() && x509a->authKeyIdSrc != NULL) {
        XMEMMOVE(x509a->authKeyIdSrc, x509a->authKeyId, x509a->authKeyIdSz);
        x509a->authKeyId = x509a->authKeyIdSrc;
        x509a->authKeyIdSrc = NULL;
        x509a->authKeyIdSrcSz = 0;
    }

    /* Resign to re-generate the der */
    ExpectIntGT(wolfSSL_X509_sign(x509a, priv, EVP_sha256()), 0);

    ExpectIntEQ(PEM_write_bio_X509(output, x509a), WOLFSSL_SUCCESS);

    /* Check that we generate a smaller output since the AKID will
     * only contain the KeyIdentifier without any additional
     * information */

    /* Here we copy the validity struct from the original */
    expectedLen = 1688;
    ExpectIntEQ(wolfSSL_BIO_get_len(output), expectedLen);

    /* Reset buffers and x509 */
    BIO_free(input);
    input = NULL;
    BIO_free(output);
    output = NULL;
    X509_free(x509a);
    x509a = NULL;

    /* test CA and basicConstSet values are encoded when
     * the cert is a CA */
    ExpectNotNull(input = BIO_new_file("certs/server-cert.pem", "rb"));

    /* read PEM into X509 struct */
    ExpectNotNull(PEM_read_bio_X509(input, &x509a, NULL, NULL));

    /* write X509 back to PEM BIO; no need to sign as nothing changed */
    ExpectNotNull(output = BIO_new(wolfSSL_BIO_s_mem()));
    ExpectIntEQ(PEM_write_bio_X509(output, x509a), WOLFSSL_SUCCESS);

    /* read exported X509 PEM back into struct, ensure isCa and basicConstSet
     * values are maintained and certs are identical.*/
    ExpectNotNull(PEM_read_bio_X509(output, &x509b, NULL, NULL));
    ExpectIntEQ(x509b->isCa, 1);
    ExpectIntEQ(x509b->basicConstSet, 1);
    ExpectIntEQ(0, wolfSSL_X509_cmp(x509a, x509b));

    X509_free(x509a);
    x509a = NULL;
    X509_free(x509b);
    x509b = NULL;
    BIO_free(input);
    input = NULL;
    BIO_free(output);
    output = NULL;

    /* test CA and basicConstSet values are encoded when
     * the cert is not CA */
    ExpectNotNull(input = BIO_new_file("certs/client-uri-cert.pem", "rb"));

    /* read PEM into X509 struct */
    ExpectNotNull(PEM_read_bio_X509(input, &x509a, NULL, NULL));

    /* write X509 back to PEM BIO; no need to sign as nothing changed */
    ExpectNotNull(output = BIO_new(wolfSSL_BIO_s_mem()));
    ExpectIntEQ(PEM_write_bio_X509(output, x509a), WOLFSSL_SUCCESS);

    /* read exported X509 PEM back into struct, ensure isCa and
     * basicConstSet values are maintained and certs are identical */
    ExpectNotNull(PEM_read_bio_X509(output, &x509b, NULL, NULL));
    ExpectIntEQ(x509b->isCa, 0);
    ExpectIntEQ(x509b->basicConstSet, 1);
    ExpectIntEQ(0, wolfSSL_X509_cmp(x509a, x509b));

    wolfSSL_EVP_PKEY_free(priv);
    X509_free(x509a);
    X509_free(x509b);
    BIO_free(input);
    BIO_free(output);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_NAME_ENTRY(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_FILESYSTEM) && \
    !defined(NO_RSA) && defined(WOLFSSL_CERT_GEN)
    X509*      x509 = NULL;
#ifndef NO_BIO
    X509*      empty = NULL;
    BIO*       bio = NULL;
#endif
    X509_NAME* nm = NULL;
    X509_NAME_ENTRY* entry = NULL;
    WOLF_STACK_OF(WOLFSSL_X509_NAME_ENTRY)* entries = NULL;
    unsigned char cn[] = "another name to add";
#ifdef OPENSSL_ALL
    int i;
    int names_len = 0;
#endif

    ExpectNotNull(x509 =
            wolfSSL_X509_load_certificate_file(cliCertFile, SSL_FILETYPE_PEM));
#ifndef NO_BIO
    ExpectNotNull(empty = wolfSSL_X509_new());
    ExpectNotNull(bio = BIO_new(BIO_s_mem()));
    ExpectIntEQ(PEM_write_bio_X509_AUX(NULL, NULL), WOLFSSL_FAILURE);
    ExpectIntEQ(PEM_write_bio_X509_AUX(bio, NULL), WOLFSSL_FAILURE);
    ExpectIntEQ(PEM_write_bio_X509_AUX(NULL, x509), WOLFSSL_FAILURE);
    ExpectIntEQ(PEM_write_bio_X509_AUX(bio, empty), WOLFSSL_FAILURE);
    ExpectIntEQ(PEM_write_bio_X509_AUX(bio, x509), SSL_SUCCESS);
    wolfSSL_X509_free(empty);
#endif

#ifdef WOLFSSL_CERT_REQ
    {
        X509_REQ* req = NULL;
#ifndef NO_BIO
        X509_REQ* emptyReq = NULL;
        BIO*      bReq = NULL;
#endif

        ExpectNotNull(req =
            wolfSSL_X509_load_certificate_file(cliCertFile, SSL_FILETYPE_PEM));
#ifndef NO_BIO
        ExpectNotNull(emptyReq = wolfSSL_X509_REQ_new());
        ExpectNotNull(bReq = BIO_new(BIO_s_mem()));
        ExpectIntEQ(PEM_write_bio_X509_REQ(NULL, NULL), WOLFSSL_FAILURE);
        ExpectIntEQ(PEM_write_bio_X509_REQ(bReq, NULL), WOLFSSL_FAILURE);
        ExpectIntEQ(PEM_write_bio_X509_REQ(NULL, req), WOLFSSL_FAILURE);
        ExpectIntEQ(PEM_write_bio_X509_REQ(bReq, emptyReq), WOLFSSL_FAILURE);
        ExpectIntEQ(PEM_write_bio_X509_REQ(bReq, req), SSL_SUCCESS);

        BIO_free(bReq);
        X509_REQ_free(emptyReq);
#endif
        X509_free(req);
    }
#endif

    ExpectNotNull(nm = X509_get_subject_name(x509));

    /* Test add entry */
    ExpectNotNull(entry = X509_NAME_ENTRY_create_by_NID(NULL, NID_commonName,
                0x0c, cn, (int)sizeof(cn)));
    ExpectIntEQ(X509_NAME_add_entry(nm, entry, -1, 0), SSL_SUCCESS);

    /* @TODO the internal name entry set value needs investigated for matching
     * behavior with OpenSSL. At the moment the getter function for the set
     * value is being tested only in that it succeeds in getting the internal
     * value. */
    ExpectIntGT(X509_NAME_ENTRY_set(X509_NAME_get_entry(nm, 1)), 0);

#ifdef WOLFSSL_CERT_EXT
    ExpectIntEQ(X509_NAME_add_entry_by_txt(NULL, NULL, MBSTRING_UTF8,
        (byte*)"support@wolfssl.com", 19, -1, 1), WOLFSSL_FAILURE);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(nm, NULL, MBSTRING_UTF8,
        (byte*)"support@wolfssl.com", 19, -1, 1), WOLFSSL_FAILURE);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(NULL, "emailAddress", MBSTRING_UTF8,
        (byte*)"support@wolfssl.com", 19, -1, 1), WOLFSSL_FAILURE);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(nm, "emailAddress", MBSTRING_UTF8,
        (byte*)"support@wolfssl.com", 19, -1, 1), WOLFSSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(nm, "commonName", MBSTRING_UTF8,
        (byte*)"wolfssl.com", 11, 0, 1), WOLFSSL_SUCCESS);
    ExpectNull(wolfSSL_X509_NAME_delete_entry(NULL, -1));
    ExpectNull(wolfSSL_X509_NAME_delete_entry(nm, -1));
    ExpectNotNull(wolfSSL_X509_NAME_delete_entry(nm, 0));
#endif
    X509_NAME_ENTRY_free(entry);
    entry = NULL;

#ifdef WOLFSSL_CERT_REQ
    {
        unsigned char srv_pkcs9p[] = "Server";
        unsigned char rfc822Mlbx[] = "support@wolfssl.com";
        unsigned char fvrtDrnk[] = "tequila";
        unsigned char* der = NULL;
        char* subject = NULL;

        ExpectIntEQ(X509_NAME_add_entry_by_NID(nm, NID_pkcs9_contentType,
            MBSTRING_ASC, srv_pkcs9p, -1, -1, 0), SSL_SUCCESS);

        ExpectIntEQ(X509_NAME_add_entry_by_NID(nm, NID_rfc822Mailbox,
            MBSTRING_ASC, rfc822Mlbx, -1, -1, 0), SSL_SUCCESS);

        ExpectIntEQ(X509_NAME_add_entry_by_NID(nm, NID_favouriteDrink,
            MBSTRING_ASC, fvrtDrnk, -1, -1, 0), SSL_SUCCESS);

        ExpectIntEQ(wolfSSL_i2d_X509_NAME(NULL, &der), BAD_FUNC_ARG);
        ExpectIntGT(wolfSSL_i2d_X509_NAME(nm, &der), 0);
        ExpectNotNull(der);

        ExpectNotNull(subject = X509_NAME_oneline(nm, NULL, 0));
        ExpectNotNull(XSTRSTR(subject, "rfc822Mailbox=support@wolfssl.com"));
        ExpectNotNull(XSTRSTR(subject, "favouriteDrink=tequila"));
        ExpectNotNull(XSTRSTR(subject, "contentType=Server"));
    #ifdef DEBUG_WOLFSSL
        if (subject != NULL) {
            fprintf(stderr, "\n\t%s\n", subject);
        }
    #endif
        XFREE(subject, 0, DYNAMIC_TYPE_OPENSSL);
        XFREE(der, NULL, DYNAMIC_TYPE_OPENSSL);
    }
#endif

    ExpectNull(entry = X509_NAME_ENTRY_create_by_txt(NULL, NULL, 0x0c, cn,
        (int)sizeof(cn)));
    /* Test add entry by text */
    ExpectNotNull(entry = X509_NAME_ENTRY_create_by_txt(NULL, "commonName",
                0x0c, cn, (int)sizeof(cn)));
    ExpectPtrEq(X509_NAME_ENTRY_create_by_txt(&entry, "commonName",
                0x0c, cn, (int)sizeof(cn)), entry);
    #if defined(OPENSSL_ALL) || defined(WOLFSSL_ASIO) \
    || defined(WOLFSSL_HAPROXY) || defined(WOLFSSL_NGINX)
    ExpectNull(X509_NAME_ENTRY_create_by_txt(&entry, "unknown",
                V_ASN1_UTF8STRING, cn, (int)sizeof(cn)));
    #endif
    ExpectIntEQ(X509_NAME_add_entry(nm, entry, -1, 0), SSL_SUCCESS);
    X509_NAME_ENTRY_free(entry);
    entry = NULL;

    /* Test add entry by NID */
    ExpectIntEQ(X509_NAME_add_entry_by_NID(nm, NID_commonName, MBSTRING_UTF8,
                                       cn, -1, -1, 0), SSL_SUCCESS);

#ifdef OPENSSL_ALL
    /* stack of name entry */
    ExpectIntGT((names_len = sk_X509_NAME_ENTRY_num(nm->entries)), 0);
    for (i = 0; i < names_len; i++) {
        ExpectNotNull(entry = sk_X509_NAME_ENTRY_value(nm->entries, i));
    }
#endif

    ExpectNotNull(entries = wolfSSL_sk_X509_NAME_ENTRY_new(NULL));
    ExpectIntEQ(sk_X509_NAME_ENTRY_num(NULL), BAD_FUNC_ARG);
    ExpectIntEQ(sk_X509_NAME_ENTRY_num(entries), 0);
    ExpectNull(sk_X509_NAME_ENTRY_value(NULL, 0));
    ExpectNull(sk_X509_NAME_ENTRY_value(entries, 0));
    wolfSSL_sk_X509_NAME_ENTRY_free(entries);

#ifndef NO_BIO
    BIO_free(bio);
#endif
    X509_free(x509); /* free's nm */
#endif
    return EXPECT_RESULT();
}

/* Note the lack of wolfSSL_ prefix...this is a compatibility layer test. */
static int test_GENERAL_NAME_set0_othername(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && \
    defined(WOLFSSL_CERT_GEN) && defined(WOLFSSL_CERT_REQ) && \
    defined(WOLFSSL_CUSTOM_OID) && defined(WOLFSSL_ALT_NAMES) && \
    defined(WOLFSSL_CERT_EXT) && !defined(NO_FILESYSTEM) && \
    defined(WOLFSSL_FPKI) && !defined(NO_RSA)
    /* ./configure --enable-opensslall --enable-certgen --enable-certreq
     *  --enable-certext --enable-debug 'CPPFLAGS=-DWOLFSSL_CUSTOM_OID
     *  -DWOLFSSL_ALT_NAMES  -DWOLFSSL_FPKI' */
    const char * cert_fname = "./certs/server-cert.der";
    const char * key_fname = "./certs/server-key.der";
    X509* x509 = NULL;
    GENERAL_NAME* gn = NULL;
    GENERAL_NAMES* gns = NULL;
    ASN1_OBJECT* upn_oid = NULL;
    ASN1_UTF8STRING *utf8str = NULL;
    ASN1_TYPE *value = NULL;
    X509_EXTENSION * ext = NULL;

    byte* pt = NULL;
    byte der[4096];
    int derSz = 0;
    EVP_PKEY* priv = NULL;
    XFILE f = XBADFILE;

    ExpectTrue((f = XFOPEN(cert_fname, "rb")) != XBADFILE);
    ExpectNotNull(x509 = d2i_X509_fp(f, NULL));
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }
    ExpectNotNull(gn = GENERAL_NAME_new());
    ExpectNotNull(upn_oid = OBJ_txt2obj("1.3.6.1.4.1.311.20.2.3", 1));
    ExpectNotNull(utf8str = ASN1_UTF8STRING_new());
    ExpectIntEQ(ASN1_STRING_set(utf8str, "othername@wolfssl.com", -1), 1);
    ExpectNotNull(value = ASN1_TYPE_new());
    ASN1_TYPE_set(value, V_ASN1_UTF8STRING, utf8str);
    if ((value == NULL) || (value->value.ptr != (char*)utf8str)) {
        wolfSSL_ASN1_STRING_free(utf8str);
    }
    ExpectIntEQ(GENERAL_NAME_set0_othername(NULL, NULL   , NULL ),
        WOLFSSL_FAILURE);
    ExpectIntEQ(GENERAL_NAME_set0_othername(gn  , NULL   , NULL ),
        WOLFSSL_FAILURE);
    ExpectIntEQ(GENERAL_NAME_set0_othername(NULL, upn_oid, NULL ),
        WOLFSSL_FAILURE);
    ExpectIntEQ(GENERAL_NAME_set0_othername(NULL, NULL   , value),
        WOLFSSL_FAILURE);
    ExpectIntEQ(GENERAL_NAME_set0_othername(gn  , upn_oid, NULL ),
        WOLFSSL_FAILURE);
    ExpectIntEQ(GENERAL_NAME_set0_othername(gn  , NULL   , value),
        WOLFSSL_FAILURE);
    ExpectIntEQ(GENERAL_NAME_set0_othername(NULL, upn_oid, value ),
        WOLFSSL_FAILURE);
    ExpectIntEQ(GENERAL_NAME_set0_othername(gn, upn_oid, value), 1);
    if (EXPECT_FAIL()) {
        ASN1_TYPE_free(value);
    }
    ExpectNotNull(gns = sk_GENERAL_NAME_new(NULL));
    ExpectIntEQ(sk_GENERAL_NAME_push(gns, gn), 1);
    if (EXPECT_FAIL()) {
        GENERAL_NAME_free(gn);
        gn = NULL;
    }
    ExpectNotNull(ext = X509V3_EXT_i2d(NID_subject_alt_name, 0, gns));
    ExpectIntEQ(X509_add_ext(x509, ext, -1), 1);
    ExpectTrue((f = XFOPEN(key_fname, "rb")) != XBADFILE);
    ExpectIntGT(derSz = (int)XFREAD(der, 1, sizeof(der), f), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }
    pt = der;
    ExpectNotNull(priv = d2i_PrivateKey(EVP_PKEY_RSA, NULL,
        (const unsigned char**)&pt, derSz));
    ExpectIntGT(X509_sign(x509, priv, EVP_sha256()), 0);
    sk_GENERAL_NAME_pop_free(gns, GENERAL_NAME_free);
    gns = NULL;
    ExpectNotNull(gns = (GENERAL_NAMES*)X509_get_ext_d2i(x509,
        NID_subject_alt_name, NULL, NULL));

    ExpectIntEQ(sk_GENERAL_NAME_num(NULL), 0);
    ExpectIntEQ(sk_GENERAL_NAME_num(gns), 3);

    ExpectNull(sk_GENERAL_NAME_value(NULL, 0));
    ExpectNull(sk_GENERAL_NAME_value(gns, 20));
    ExpectNotNull(gn = sk_GENERAL_NAME_value(gns, 2));
    ExpectIntEQ(gn->type, 0);

    sk_GENERAL_NAME_pop_free(gns, GENERAL_NAME_free);

    ASN1_OBJECT_free(upn_oid);
    X509_EXTENSION_free(ext);
    X509_free(x509);
    EVP_PKEY_free(priv);
#endif
    return EXPECT_RESULT();
}

/* Note the lack of wolfSSL_ prefix...this is a compatibility layer test. */
static int test_othername_and_SID_ext(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_ALL) && !defined(NO_CERTS) && \
    defined(WOLFSSL_CERT_GEN) && defined(WOLFSSL_CERT_REQ) && \
    defined(WOLFSSL_CUSTOM_OID) && defined(WOLFSSL_ALT_NAMES) && \
    defined(WOLFSSL_CERT_EXT) && !defined(NO_FILESYSTEM) && \
    defined(WOLFSSL_FPKI) && defined(WOLFSSL_ASN_TEMPLATE) && !defined(NO_RSA)
    /* ./configure --enable-opensslall --enable-certgen --enable-certreq
     *  --enable-certext --enable-debug 'CPPFLAGS=-DWOLFSSL_CUSTOM_OID
     *  -DWOLFSSL_ALT_NAMES  -DWOLFSSL_FPKI' */
    const char* csr_fname = "./certs/csr.signed.der";
    const char* key_fname = "./certs/server-key.der";

    byte der[4096];
    int derSz = 0;
    byte badDer[2] = { 0x30, 0x00 };
    X509_REQ* x509 = NULL;
    STACK_OF(X509_EXTENSION) *exts = NULL;

    X509_EXTENSION * san_ext = NULL;
    X509_EXTENSION * ext = NULL;
    GENERAL_NAME* gn = NULL;
    GENERAL_NAMES* gns = NULL;
    ASN1_OBJECT* upn_oid = NULL;
    ASN1_UTF8STRING *utf8str = NULL;
    ASN1_TYPE *value = NULL;
    ASN1_STRING *extval = NULL;

    /* SID extension. SID data format explained here:
     * https://blog.qdsecurity.se/2022/05/27/manually-injecting-a-sid-in-a-certificate/
     */
    byte SidExtension[] = {
    48, 64, 160, 62, 6,  10, 43, 6,  1,  4,  1,  130, 55, 25, 2,  1,  160,
    48, 4,  46,  83, 45, 49, 45, 53, 45, 50, 49, 45,  50, 56, 52, 51, 57,
    48, 55, 52,  49, 56, 45, 51, 57, 50, 54, 50, 55,  55, 52, 50, 49, 45,
    51, 56, 49,  53, 57, 57, 51, 57, 55, 50, 45, 52,  54, 48, 49};

    byte expectedAltName[] = {
    0x30, 0x27, 0xA0, 0x25, 0x06, 0x0A, 0x2B, 0x06, 0x01, 0x04, 0x01, 0x82,
    0x37, 0x14, 0x02, 0x03, 0xA0, 0x17, 0x0C, 0x15, 0x6F, 0x74, 0x68, 0x65,
    0x72, 0x6E, 0x61, 0x6D, 0x65, 0x40, 0x77, 0x6F, 0x6C, 0x66, 0x73, 0x73,
    0x6C, 0x2E, 0x63, 0x6F, 0x6D};

    X509_EXTENSION *sid_ext = NULL;
    ASN1_OBJECT* sid_oid = NULL;
    ASN1_OCTET_STRING *sid_data = NULL;

    ASN1_OBJECT* alt_names_oid = NULL;

    EVP_PKEY* priv = NULL;
    XFILE f = XBADFILE;
    byte* pt = NULL;
    BIO* bio = NULL;

    ExpectTrue((f = XFOPEN(csr_fname, "rb")) != XBADFILE);
    ExpectNotNull(x509 = d2i_X509_REQ_fp(f, NULL));
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }
    ExpectIntEQ(X509_REQ_set_version(x509, 2), 1);
    ExpectNotNull(gn = GENERAL_NAME_new());
    ExpectNotNull(upn_oid = OBJ_txt2obj("1.3.6.1.4.1.311.20.2.3", 1));
    ExpectNotNull(utf8str = ASN1_UTF8STRING_new());
    ExpectIntEQ(ASN1_STRING_set(utf8str, "othername@wolfssl.com", -1), 1);
    ExpectNotNull(value = ASN1_TYPE_new());
    ASN1_TYPE_set(value, V_ASN1_UTF8STRING, utf8str);
    if (EXPECT_FAIL()) {
        ASN1_UTF8STRING_free(utf8str);
    }
    ExpectIntEQ(GENERAL_NAME_set0_othername(gn, upn_oid, value), 1);
    if (EXPECT_FAIL()) {
        ASN1_TYPE_free(value);
        GENERAL_NAME_free(gn);
        gn = NULL;
    }
    ExpectNotNull(gns = sk_GENERAL_NAME_new(NULL));
    ExpectIntEQ(sk_GENERAL_NAME_push(gns, gn), 1);
    if (EXPECT_FAIL()) {
        GENERAL_NAME_free(gn);
    }
    ExpectNotNull(san_ext = X509V3_EXT_i2d(NID_subject_alt_name, 0, gns));
    ExpectNotNull(sid_oid = OBJ_txt2obj("1.3.6.1.4.1.311.25.2", 1));
    ExpectNotNull(sid_data = ASN1_OCTET_STRING_new());
    ASN1_OCTET_STRING_set(sid_data, SidExtension, sizeof(SidExtension));
    ExpectNotNull(sid_ext = X509_EXTENSION_create_by_OBJ(NULL, sid_oid, 0,
                                                         sid_data));
    ExpectNotNull(exts = sk_X509_EXTENSION_new_null());
    wolfSSL_sk_X509_EXTENSION_free(exts);
    exts = NULL;
    ExpectNotNull(exts = sk_X509_EXTENSION_new_null());
    /* Ensure an empty stack doesn't raise an error. */
    ExpectIntEQ(X509_REQ_add_extensions(NULL, NULL), 0);
    ExpectIntEQ(X509_REQ_add_extensions(x509, NULL), 0);
    ExpectIntEQ(X509_REQ_add_extensions(NULL, exts), 0);
    ExpectIntEQ(X509_REQ_add_extensions(x509, exts), 1);
    ExpectIntEQ(sk_X509_EXTENSION_push(exts, san_ext), 1);
    if (EXPECT_FAIL()) {
        X509_EXTENSION_free(san_ext);
    }
    ExpectIntEQ(sk_X509_EXTENSION_push(exts, sid_ext), 2);
    if (EXPECT_FAIL()) {
        X509_EXTENSION_free(sid_ext);
    }
    ExpectIntEQ(X509_REQ_add_extensions(x509, exts), 1);
    ExpectTrue((f = XFOPEN(key_fname, "rb")) != XBADFILE);
    ExpectIntGT(derSz = (int)XFREAD(der, 1, sizeof(der), f), 0);
    if (f != XBADFILE)
        XFCLOSE(f);
    pt = der;
    ExpectNotNull(priv = d2i_PrivateKey(EVP_PKEY_RSA, NULL,
                                        (const unsigned char**)&pt, derSz));
    ExpectIntGT(X509_REQ_sign(x509, priv, EVP_sha256()), 0);
    pt = der;
    ExpectIntGT(derSz = i2d_X509_REQ(x509, &pt), 0);
    X509_REQ_free(x509);
    x509 = NULL;
    ExpectNull(d2i_X509_REQ_INFO(&x509, NULL, derSz));
    pt = badDer;
    ExpectNull(d2i_X509_REQ_INFO(&x509, (const unsigned char**)&pt,
        sizeof(badDer)));
    pt = der;
    ExpectNotNull(d2i_X509_REQ_INFO(&x509, (const unsigned char**)&pt, derSz));
    sk_GENERAL_NAME_pop_free(gns, GENERAL_NAME_free);
    gns = NULL;
    sk_X509_EXTENSION_pop_free(exts, X509_EXTENSION_free);
    exts = NULL;
    ASN1_OBJECT_free(upn_oid);
    ASN1_OBJECT_free(sid_oid);
    sid_oid = NULL;
    ASN1_OCTET_STRING_free(sid_data);
    X509_REQ_free(x509);
    EVP_PKEY_free(priv);

    /* At this point everything used to generate what is in der is cleaned up.
     * We now read back from der to confirm the extensions were inserted
     * correctly. */
    bio = wolfSSL_BIO_new(wolfSSL_BIO_s_mem());
    ExpectNotNull(bio);

    ExpectIntEQ(BIO_write(bio, der, derSz), derSz); /* d2i consumes BIO */
    ExpectNotNull(d2i_X509_REQ_bio(bio, &x509));
    ExpectNotNull(x509);
    BIO_free(bio);
    ExpectNotNull(exts = (STACK_OF(X509_EXTENSION)*)X509_REQ_get_extensions(
        x509));
    ExpectIntEQ(sk_X509_EXTENSION_num(NULL), WOLFSSL_FATAL_ERROR);
    ExpectIntEQ(sk_X509_EXTENSION_num(exts), 2);

    /* Check the SID extension. */
    ExpectNotNull(sid_oid = OBJ_txt2obj("1.3.6.1.4.1.311.25.2", 1));
    ExpectNotNull(ext = sk_X509_EXTENSION_value(exts,
            X509_get_ext_by_OBJ(x509, sid_oid, -1)));
    ExpectNotNull(extval = X509_EXTENSION_get_data(ext));
    ExpectIntEQ(extval->length, sizeof(SidExtension));
    ExpectIntEQ(XMEMCMP(SidExtension, extval->data, sizeof(SidExtension)), 0);
    ASN1_OBJECT_free(sid_oid);

    /* Check the AltNames extension. */
    ExpectNotNull(alt_names_oid = OBJ_txt2obj("subjectAltName", 0));
    ExpectNotNull(ext = sk_X509_EXTENSION_value(exts,
            X509_get_ext_by_OBJ(x509, alt_names_oid, -1)));
    ExpectNotNull(extval = X509_EXTENSION_get_data(ext));
    ExpectIntEQ(extval->length, sizeof(expectedAltName));
    ExpectIntEQ(XMEMCMP(expectedAltName, extval->data, sizeof(expectedAltName)),
                0);
    ASN1_OBJECT_free(alt_names_oid);

    /* Cleanup */
    ExpectNotNull(gns = (GENERAL_NAMES*)X509_get_ext_d2i(x509,
        NID_subject_alt_name, NULL, NULL));
    ExpectIntEQ(sk_GENERAL_NAME_num(gns), 1);
    ExpectNotNull(gn = sk_GENERAL_NAME_value(gns, 0));
    ExpectIntEQ(gn->type, 0);

    sk_GENERAL_NAME_pop_free(gns, GENERAL_NAME_free);

    sk_X509_EXTENSION_pop_free(exts, X509_EXTENSION_free);
    X509_REQ_free(x509);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_set_name(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && \
    defined(WOLFSSL_CERT_GEN) && defined(WOLFSSL_CERT_REQ)
    X509* x509 = NULL;
    X509_NAME* name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                                           (byte*)"wolfssl.com", 11, 0, 1),
                WOLFSSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "emailAddress", MBSTRING_UTF8,
                                           (byte*)"support@wolfssl.com", 19, -1,
                                           1), WOLFSSL_SUCCESS);
    ExpectNotNull(x509 = X509_new());

    ExpectIntEQ(X509_set_subject_name(NULL, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(X509_set_subject_name(x509, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(X509_set_subject_name(NULL, name), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);

    ExpectIntEQ(X509_set_issuer_name(NULL, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(X509_set_issuer_name(x509, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(X509_set_issuer_name(NULL, name), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);

    X509_free(x509);
    X509_NAME_free(name);
#endif /* OPENSSL_ALL && !NO_CERTS */
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_set_notAfter(void)
{
    EXPECT_DECLS;
#if (defined(OPENSSL_ALL) || defined(WOLFSSL_APACHE_HTTPD)) \
    && !defined(NO_ASN_TIME) && !defined(USER_TIME) && \
    !defined(TIME_OVERRIDES) && !defined(NO_CERTS) && \
    defined(WOLFSSL_CERT_GEN) && defined(WOLFSSL_CERT_REQ) &&\
    !defined(TIME_T_NOT_64BIT) && !defined(NO_64BIT) && !defined(NO_BIO)
    /* Generalized time will overflow time_t if not long */
    X509* x = NULL;
    BIO*  bio = NULL;
    ASN1_TIME *asn_time = NULL;
    ASN1_TIME *time_check = NULL;
    const int year = 365*24*60*60;
    const int day  = 24*60*60;
    const int hour = 60*60;
    const int mini = 60;
    int offset_day;
    unsigned char buf[25];
    time_t t;

    /*
     * Setup asn_time. APACHE HTTPD uses time(NULL)
     */
    t = (time_t)107 * year + 31 * day + 34 * hour + 30 * mini + 7 * day;
    offset_day = 7;
    /*
     * Free these.
     */
    asn_time = wolfSSL_ASN1_TIME_adj(NULL, t, offset_day, 0);
    ExpectNotNull(asn_time);
    ExpectNotNull(x = X509_new());
    ExpectNotNull(bio = BIO_new(BIO_s_mem()));
    /*
     * Tests
     */
    ExpectTrue(wolfSSL_X509_set_notAfter(x, asn_time));
    /* time_check is simply (ANS1_TIME*)x->notAfter */
    ExpectNotNull(time_check = X509_get_notAfter(x));
    /* ANS1_TIME_check validates by checking if argument can be parsed */
    ExpectIntEQ(ASN1_TIME_check(time_check), WOLFSSL_SUCCESS);
    /* Convert to human readable format and compare to intended date */
    ExpectIntEQ(ASN1_TIME_print(bio, time_check), 1);
    ExpectIntEQ(BIO_read(bio, buf, sizeof(buf)), 24);
    ExpectIntEQ(XMEMCMP(buf, "Jan 20 10:30:00 2077 GMT", sizeof(buf) - 1), 0);

    ExpectFalse(wolfSSL_X509_set_notAfter(NULL, NULL));
    ExpectFalse(wolfSSL_X509_set_notAfter(x, NULL));
    ExpectFalse(wolfSSL_X509_set_notAfter(NULL, asn_time));

    /*
     * Cleanup
     */
    XFREE(asn_time, NULL, DYNAMIC_TYPE_OPENSSL);
    X509_free(x);
    BIO_free(bio);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_set_notBefore(void)
{
    EXPECT_DECLS;
#if (defined(OPENSSL_ALL) || defined(WOLFSSL_APACHE_HTTPD)) \
    && !defined(NO_ASN_TIME) && !defined(USER_TIME) && \
    !defined(TIME_OVERRIDES) && !defined(NO_CERTS) && \
    defined(WOLFSSL_CERT_GEN) && defined(WOLFSSL_CERT_REQ) && !defined(NO_BIO)
    X509* x = NULL;
    BIO*  bio = NULL;
    ASN1_TIME *asn_time = NULL;
    ASN1_TIME *time_check = NULL;
    const int year = 365*24*60*60;
    const int day  = 24*60*60;
    const int hour = 60*60;
    const int mini = 60;
    int offset_day;
    unsigned char buf[25];
    time_t t;

    /*
     * Setup asn_time. APACHE HTTPD uses time(NULL)
     */
    t = (time_t)49 * year + 125 * day + 20 * hour + 30 * mini + 7 * day;
    offset_day = 7;

    /*
     * Free these.
     */
    asn_time = wolfSSL_ASN1_TIME_adj(NUL