/* Write a 1D file. */

/* Copyright (c) 2014-2025 MJ Rutter 
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3
 * of the Licence, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see http://www.gnu.org/licenses/
 */ 

#include<stdio.h>
#include<stdlib.h>
#include<ctype.h>
#include<math.h>
#include<string.h>

#include "c2xsf.h"

/* line_spec must be AtN:AtN:n
 * with At - atomic symbol of element present
 *      N  - index of required atom within species (1 assumed if omitted)
 *      n  - number of points
 */

void interpolate1d(struct grid *gptr, double st[3], double end[3],
                   int n, double *pts);

void interpolate3d(struct grid *old_grid, struct grid *new_grid);
void  interpolate_r(struct grid *gptr, struct unit_cell *c, double *start,
		    double len, int n, double *points, char axis, char mode);
void  interpolate_fr(struct grid *gptr, int comp, struct unit_cell *c,
		     double *start, double len, int n,
		     double *points, char axis, char mode);
double interpolate_spherical(struct grid *gptr, int comp, struct unit_cell *c,
			   double *origin_frac, double len, int n,
			   double *points, char mode);
double interpolate_fspherical(struct grid *gptr, int comp, struct unit_cell *c,
			   double *origin_frac, double len, int n,
			   double *points, char mode);
double integrate_fspherical(struct grid *g, int comp, struct unit_cell *c,
			    double *origin_frac, double rad);
double accumulate(double *points, double *acc, double len, int n_in);
double acc_simpson(double *points, double *acc, double len, int n_in);
void reduce2d(double *data, int grid[3], char axis);
void reduce2d_z(double *data, int grid[3], char axis);

#ifndef ACC_TEST

/* lscan: parse "(x,y,z)" or "AtN" or "At"
 *        expects termination by : or null
 *        used by main() for point value, and this file for line
 *
 *        accept 0 as shorthand for (0,0,0)
 *        accept M as shorthand for (0.5,0.5,0.5)
 */
 
void lscan(char **p, struct contents *m, double x[3]){
  char *ptr,sym[5];
  int i,j,n,atno,ns,found;

  ptr=*p;

  if (*ptr=='('){
    /*
    ptr++;
    if (sscanf(ptr,"%lf,%lf,%lf",x,x+1,x+2)!=3){
      fprintf(stderr,"Malformed line_spec: %s\n",ptr);
      exit(1);
    }
    while(*ptr&&(*ptr!=')')) ptr++;
    if (*ptr!=')'){
      fprintf(stderr,"Malformed line_spec: %s\n",ptr);
      exit(1);
    }
    ptr++;
    */
    if (point_scan(ptr,x,&n)!=1){
      fprintf(stderr,"Malformed line_spec: %s\n",ptr);
      exit(1);
    }
    ptr+=n;
  }
  else if ((*ptr=='0')&&((*(ptr+1)==':')||(*(ptr+1)==0))){
    ptr++;
    x[0]=x[1]=x[2]=0;
  }
  else if ((*ptr=='M')&&((*(ptr+1)==':')||(*(ptr+1)==0))){
    ptr++;
    x[0]=x[1]=x[2]=0.5;
  }
  else{
    for(i=0;i<4;i++){
      if(isalpha(*ptr)) sym[i]=*ptr;
      else break;
      ptr++;
    }

    sym[i]=0;
    if ((!isdigit(*ptr))&&(*ptr!=':')&&(*ptr!=0)){
      fprintf(stderr,"Malformed line_spec: %s\n",ptr);
      exit(1);
    }
    atno=atsym2no(sym);
    if (atno==0) error_exit("Invalid atom in line_spec");
    if ((*ptr==':')||(*ptr==0))
      ns=1;
    else
      if(sscanf(ptr,"%d",&ns)!=1)
	error_exit("Invalid number in atom spec in line_spec");

    if (ns<1) error_exit("Invalid atom number in line_spec");
    if (m->n<1) error_exit("Atom position specified, but no atoms read");
    
    found=0;
    for(i=0;i<m->n;i++){
      if (m->atoms[i].atno==atno){
        found++;
        if (found==ns){
          for(j=0;j<3;j++) x[j]=m->atoms[i].frac[j];
          break;
        }
      }
    }
    if (found!=ns) error_exit("Failed to find atom in line_spec");
    while(*ptr&&(*ptr!=':')) ptr++;
  }

  *p=ptr;

}

/* lflags&1 -> use gnuplot syntax */

void line_write(FILE* outfile, struct unit_cell *c,
                struct contents *m, struct grid *gptr, char *line_spec,
                int lflags){
  char *ptr,*ptr2,mode,axis;
  int i,j,n,radial,tmp;
  double start[3],end[3],v[3],v2[3],x,*points,len;
  struct grid *g;
  char *fmt;
  struct cmt *comment;

  mode=' ';
  axis='c';
  
  if (!gptr||(!gptr->data))
    error_exit("No data found to plot");

  if (flags&HIPREC)
    fmt="%.12g %.12g\n";
  else
    fmt="%8g %g\n";

  /* parse line spec */

  radial=0;
  ptr=line_spec;

  if (!strcmp(line_spec,"a")){
    start[0]=start[1]=start[2]=0;
    end[0]=1;
    end[1]=end[2]=0;
    n=gptr->size[0]+1;
  }
  else if (!strcmp(line_spec,"b")){
    start[0]=start[1]=start[2]=0;
    end[1]=1;
    end[0]=end[2]=0;
    n=gptr->size[1]+1;
  }
  else if (!strcmp(line_spec,"c")){
    start[0]=start[1]=start[2]=0;
    end[2]=1;
    end[0]=end[1]=0;
    n=gptr->size[2]+1;
  }
  else{
  
    lscan(&ptr,m,start);

    if (*ptr!=':') error_exit("Failed to find first colon in line_spec");
    ptr++;

    if ((*ptr=='r')||(*ptr=='R')){
      radial=1;
      if (*ptr=='R') radial=2;
      ptr++;
      sscanf(ptr,"%lf%n",&len,&n);
      ptr+=n;
      if (*ptr=='B'){
	len*=BOHR;
	ptr++;
      }
      if (radial==1){
	if ((*ptr>='a')&&(*ptr<='c')){
	  axis=*ptr;
	  ptr++;
	}
      }
    }
    else
      lscan(&ptr,m,end);

    if (*ptr!=':') error_exit("Failed to find second colon in line_spec");
    ptr++;

    if(sscanf(ptr,"%d%n",&n,&i)!=1)
      error_exit("Invalid number of points in line_spec");

    if (ptr[i]=='w') mode='w';
    else if (ptr[i]=='a') mode='a';
    else if ((ptr[i])&&(!isspace(ptr[i])))
      fprintf(stderr,"Warning: unexpected trailing characters in line_spec\n");
  }

  if ((radial==2)&&(len==0)){ /* maximal sphere requested -- what is it? */
    /* centre of cell in abs co-ords */
    for(i=0;i<3;i++){
      v[i]=0;
      for(j=0;j<3;j++)
	v[i]+=0.5*c->basis[j][i];
    }

    for(i=0;i<3;i++){
      /* for each of the cell's faces, find perp unit vector */
      vcross(c->basis[(i+1)%3],c->basis[(i+2)%3],v2);
      x=sqrt(vmod2(v2));
      for(j=0;j<3;j++) v2[j]/=x;
      /* dot with vector to centre of cell to find perp dist */
      x=0;
      for(j=0;j<3;j++) x+=v[j]*v2[j];
      if (i==0)
	len=x;
      else
	len=min(len,x);
    }
  }

  if ((radial==0)&&(n==0)){
    for(i=0;i<3;i++){
      if (start[i]==end[i]) continue;
      if (aeq(start[i]*gptr->size[i],floor(start[i]*gptr->size[i]+0.5))&&
	  aeq(end[i]*gptr->size[i],floor(end[i]*gptr->size[i]+0.5))){
	tmp=1+floor((end[i]-start[i])*gptr->size[i]+0.5);
	if (n==0) n=tmp;
	else if (tmp!=n){
	  n=0;
	  break;
	}
      }
    }
    if (n){
      for(i=0;i<3;i++){
	if (aeq(start[i]*gptr->size[i],floor(start[i]*gptr->size[i]+0.5)))
	  start[i]=floor(start[i]*gptr->size[i]+0.5)/gptr->size[i];
	if (aeq(end[i]*gptr->size[i],floor(end[i]*gptr->size[i]+0.5)))
	  end[i]=floor(end[i]*gptr->size[i]+0.5)/gptr->size[i];
      }
      fprintf(stderr,"Line chosen (%f,%f,%f):(%f,%f,%f):%d\n",
	      start[0],start[1],start[2],end[0],end[1],end[2],n);
    }
  }

  if (n<1){
    fprintf(stderr,"Invalid number of points for line. Have %d\n",n);
    exit(1);
  }
  
  points=malloc(n*sizeof(double));
  if(!points) error_exit("Malloc error for points in line_write");

  if (debug){
    if (radial){
      fprintf(stderr,"Requested line centre (%f,%f,%f) ",
	      start[0],start[1],start[2]);
      if (radial==1) fprintf(stderr,"cylinder axis %c",axis);
      else fprintf(stderr,"sphere");
      fprintf(stderr," radius %f A with %d points.\n",len,n);
    }
    else
      fprintf(stderr,
	      "Requested line (%f,%f,%f) to (%f,%f,%f) with %d points.\n",
	      start[0],start[1],start[2],end[0],end[1],end[2],n);
  }
  
  /* Find length of line */

  /* Convert to absolute co-ords */

  if (!radial){
    for(i=0;i<3;i++){
      v[i]=0;
      for(j=0;j<3;j++)
	v[i]+=(end[j]-start[j])*c->basis[j][i];
    }

    len=0;
    for(i=0;i<3;i++) len+=v[i]*v[i];
    len=sqrt(len);
    if (flags&AU) len=len/BOHR;
  }
  
  if (m->title) fprintf(outfile,"# %s\n\n",m->title);

  fprintf(outfile,"# %s\n",line_spec);
  if (radial)
    fprintf(outfile,"# centre (%f,%f,%f) length %f with %d points\n",
	    start[0],start[1],start[2],len,n);
  else
    fprintf(outfile,"# (%f,%f,%f) to (%f,%f,%f) with %d points\n",
	    start[0],start[1],start[2],end[0],end[1],end[2],n);
  fprintf(outfile,"# distance in %s\n",(flags&AU)?"Bohr":"Angstrom");
  if (m->comment->txt){
    fprintf(outfile,"\n");
    comment=m->comment;
    while((comment)&&(comment->txt)){
      fprintf(outfile,"# %s\n",comment->txt);
      comment=comment->next;
    }
    fprintf(outfile,"\n");
  }


  if (lflags&1){
    fprintf(outfile,"set xlabel \"%s\"\n",(flags&AU)?"Bohr":"Angstrom");
    fprintf(outfile,"set title \"%s (length %g %s)",line_spec,len,
            (flags&AU)?"Bohr":"A");
    if ((mode=='w')||(mode=='a')) fprintf(outfile,", %s weighted",
			   (radial==1)?"2pi r":"4pi r^2");
    if (mode=='a') fprintf(outfile,", cumulative");
    fprintf(outfile,"\"\n");
    fprintf(outfile,"plot [0:%g] ",len);
    g=gptr;
    ptr=NULL;
    while(g&&(g->data)){ 
      if (ptr) free(ptr); /* Remove _ as gnuplot treats as subscript */
      ptr=malloc(strlen(g->name)+1);
      if (!ptr) error_exit("malloc error for grid title!");
      strcpy(ptr,g->name);
      while((ptr2=strchr(ptr,'_'))) *ptr2=' ';
      fprintf(outfile,"'-' w lp title \"%s\"",ptr);
      g=g->next;
      if (g&&(g->data)) fprintf(outfile,",");
    }
    if (ptr) free(ptr);
    fprintf(outfile,"\n");
  }

  while(gptr&&(gptr->data)){
    if (gptr->comps!=1)
      fprintf(stderr,"Warning: using first component"
	      " of dataset with %d components\n",gptr->comps);
    if (radial==1)
      interpolate_r(gptr,c,start,len,n,points,axis,mode);
    else if (radial==2)
      (void)interpolate_spherical(gptr,1,c,start,len,n,points,mode);
    else
      interpolate1d(gptr,start,end,n,points);

    fprintf(outfile,"# %s\n",gptr->name);

    for(i=0;i<n;i++) fprintf(outfile,fmt,i*len/(n-1),points[i]);

    if (lflags&1) fprintf(outfile,"e\n");

    gptr=gptr->next;

    if (gptr&&(gptr->data)) fprintf(outfile,"\n");
  }

  if (lflags&1) fprintf(outfile,"pause -1 \"Press return to exit\"\n");

  free(points);
}

/* Returns average value over cylinder
 *
 * Either points==NULL, or points a double array of length n and is
 *  filled with radial values, weighted if mode='w', weighted and
 *  accumulated (i.e. integrated) if mode='a'.
 *
 * Weighting is by 2 pi r c, i.e. a volume element
 */

void interpolate_r(struct grid *gptr, struct unit_cell *c, double *start,
		   double len, int n, double *pts, char axis, char mode){
  int i,j,jmax,fft;
  double origin[3],pt[3],frac[3],radius,sum,integral,length,dot;
  double abc[6],basis[2][2],recip[2][2];
  double *points,*wpoints;

  if (n==0) return;
  
  if (!is_orthog(axis-'a',c->basis))
    fprintf(stderr,"WARNING: cell axis %c not orthogonal to other axes\n",
	    axis);

  if (debug>1)
    fprintf(stderr,"interpolate_r called centre (%f,%f,%f) axis %c radius %fA"
	    " points %d mode %c\n",start[0],start[1],start[2],axis,
	    len,n,mode);
  
  if (flags&FFT){
    interpolate_fr(gptr,1,c,start,len,n,pts,axis,mode);
    return;
  }
  
  /* Reduce grid */

#if 0
  if (gptr->size[2]!=1){
    for(i=0;i<gptr->size[0]*gptr->size[1];i++){
      sum=0;
      for(j=0;j<gptr->size[2];j++)
	sum+=gptr->data[i*gptr->size[2]+j];
      gptr->data[i]=sum/gptr->size[2];
    }
    gptr->data=realloc(gptr->data,gptr->size[0]*gptr->size[1]*sizeof(double));
    if(!gptr->data) error_exit("realloc error in interpolate_r");
    gptr->size[2]=1;
  }
#endif
  
  reduce2d(gptr->data,gptr->size,axis);
  basis2abc(c->basis,abc);

  fft=0;
  if (flags&FFT){
    fft=1;
    flags^=FFT;
  }

  /* Move to wholly 2D co-ordinate system */
  
  for(i=0;i<3;i++) origin[i]=start[i];
  origin[axis-'a']=0;
  vfrac2abs(origin,c->basis);

  basis[0][1]=0;
  if (axis=='c'){
    basis[0][0]=abc[0];
    basis[1][0]=abc[1]*cos(M_PI*abc[5]/180);
    basis[1][1]=abc[1]*sin(M_PI*abc[5]/180);
    frac[0]=start[0];
    frac[1]=start[1];
  }
  else if (axis=='b'){
    basis[0][0]=abc[0];
    basis[1][0]=abc[2]*cos(M_PI*abc[4]/180);
    basis[1][1]=abc[2]*sin(M_PI*abc[4]/180);
    frac[0]=start[0];
    frac[1]=start[2];
  }
  else if (axis=='a'){
    basis[0][0]=abc[1];
    basis[1][0]=abc[2]*cos(M_PI*abc[3]/180);
    basis[1][1]=abc[2]*sin(M_PI*abc[3]/180);
    frac[0]=start[1];
    frac[1]=start[2];
  }
  else
    error_exit("Impossible axis");

  recip[0][0]=basis[1][1];
  recip[0][1]=-basis[1][0];
  dot=basis[0][0]*recip[0][0]+basis[0][1]*recip[0][1];
  recip[0][0]/=dot;
  recip[0][1]/=dot;

  recip[1][0]=basis[0][1];
  recip[1][1]=-basis[0][0];
  dot=basis[1][0]*recip[1][0]+basis[1][1]*recip[1][1];
  recip[1][0]/=dot;
  recip[1][1]/=dot;

  origin[0]=frac[0]*basis[0][0]+frac[1]*basis[1][0];
  origin[1]=frac[0]*basis[0][1]+frac[1]*basis[1][1];
  
  if (debug)
    fprintf(stderr,"Origin, abs coords, flattened: (%f,%f)\n",origin[0],
	    origin[1]);

  length=abc[axis-'a'];

  points=pts;
  if (mode=='a') points=NULL;
  if ((pts)&&(mode=='w')){
    wpoints=pts;
    points=NULL;
  }
  else{
    wpoints=malloc(n*sizeof(double));
    if (!wpoints) error_exit("malloc error in interpolate_r");
  }

  integral=0;
  for(i=0;i<n;i++){
    if (n==1)
      radius=len;
    else
      radius=i*len/(n-1);
    jmax=2+2*M_PI*n*radius/len;
    sum=0;
    for(j=0;j<jmax;j++){
      pt[0]=origin[0]+radius*cos((2*M_PI*j)/jmax);
      pt[1]=origin[1]+radius*sin((2*M_PI*j)/jmax);
      pt[2]=0;
      frac[0]=pt[0]*recip[0][0]+pt[1]*recip[0][1];
      frac[1]=pt[0]*recip[1][0]+pt[1]*recip[1][1];
      frac[2]=0;
      sum+=interpolate0d(gptr,frac,1);
    }
    sum=sum/jmax;
    if (points) points[i]=sum;
    wpoints[i]=2*M_PI*radius*length*sum;
  }

  if (mode=='a')
    integral=accumulate(wpoints,pts,len,n);
  else
    integral=accumulate(wpoints,NULL,len,n);
  
  if (debug){
    fprintf(stderr,"Radial integral: %f\n",integral);
  }

  if (fft) flags|=FFT;
  if (mode!='w') free(wpoints);
  
}

/* Average over spherical shells using Fibonacci spiral to sample shell */

/* Returns average value over sphere
 *
 * Either points==NULL, or points a double array of length n and is
 *  filled with radial values, weighted if mode='w', weighted and
 *  accumulated (i.e. integrated) if mode='a'.
 *
 * Weighting is by 4 pi r^2.
 */

double interpolate_spherical(struct grid *gptr, int comp, struct unit_cell *c,
			   double *origin_frac, double len, int n,
			   double *pts, char mode){
  int i,j,jj,k,jmax;
  double origin[3],pt[3],frac[3],radius,sum,integral,golden,r,x,y,z,theta;
  double *points,*wpoints;


  if ((comp<=0)||(comp>gptr->comps)){
    fprintf(stderr,"Invalid value of comp in interpolate_spherical. "
            "Have %d, valid 1 to %d\n",comp,gptr->comps);
    exit(1);
  }

  if (flags&FFT){
    return interpolate_fspherical(gptr,comp,c,origin_frac,len,n,pts,mode);
  }
  
  if ((len==0)&&(pts==NULL))
    return interpolate0d(gptr,origin_frac,comp);

  if (n==0) return 0;
  
  points=pts;
  if (mode=='a') points=NULL;
  if ((pts)&&(mode=='w')){
    wpoints=pts;
    points=NULL;
  }
  else{
    wpoints=malloc(n*sizeof(double));
    if (!wpoints) error_exit("malloc error in interpolate_spherical");
  }
  
  for(i=0;i<3;i++)
    origin[i]=origin_frac[0]*c->basis[0][i]+origin_frac[1]*c->basis[1][i]+
      origin_frac[2]*c->basis[2][i];

  if ((debug>1)&&(comp==1))
    fprintf(stderr,"Origin, abs coords: (%f,%f,%f)\n",origin[0],
	    origin[1],origin[2]);

  integral=0;
  golden=(1+sqrt(5.0))/2;
  for(i=0;i<n;i++){
    if (n==1)
      radius=len;
    else
      radius=i*len/(n-1);
    jmax=2+4*M_PI*n*n*radius*radius/(len*len);
    sum=0;
    for(j=0;j<jmax;j++){
      z=1-2*(j+0.5)/jmax;
      r=sqrt(1-z*z);
      theta=2*M_PI*j/golden;
      x=r*cos(theta);
      y=r*sin(theta);
      pt[0]=origin[0]+x*radius;
      pt[1]=origin[1]+y*radius;
      pt[2]=origin[2]+z*radius;
      for(jj=0;jj<3;jj++){
	frac[jj]=0;
	for(k=0;k<3;k++)
	  frac[jj]+=pt[k]*c->recip[jj][k];
      }
      sum+=interpolate0d(gptr,frac,comp);
    }
    sum=sum/jmax;
    if (points) points[i]=sum;
    wpoints[i]=4*M_PI*radius*radius*sum;
  }
  
  if (mode=='a')
    integral=accumulate(wpoints,pts,len,n);
  else
    integral=accumulate(wpoints,NULL,len,n);

  if ((debug)&&(gptr->comps==1)){
    fprintf(stderr,"Spherical integral:   %f\n",integral);
  }
  
  if (mode!='w') free(wpoints);
  return (3*integral/(4*M_PI*len*len*len));
}

void vinterpolate_spherical(struct grid *gptr, struct unit_cell *c,
			    double *origin_frac, double radius, int n,
			    double *z, char mode){
  int i;

  for(i=0;i<gptr->comps;i++)
    z[i]=interpolate_spherical(gptr,i+1,c,origin_frac,radius,n,NULL,mode);

}

void z_spec(char *spec, struct grid *gptr, struct unit_cell *c,
	    double *rad, double *scale, int *n){
  int i;
  double abc[6];
  
  *rad=0;
  *scale=1;
  *n=0;

  if ((!spec)||(!(*spec))) return;

  if (sscanf(spec,"%lf%n",rad,&i)!=1)
    error_exit("error parsing point spec");

  spec+=i;
  
  if (*spec=='B') {
    *rad*=BOHR;
    spec++;
  }
  if (*spec=='i'){
    *scale=4*M_PI*(*rad)*(*rad)*(*rad)/3;
    spec++;
  }

  cart2abc(c,NULL,abc,NULL);
  *n=max(*rad*gptr->size[0]/abc[0],*rad*gptr->size[1]/abc[1]);
  *n=max(*n,*rad*gptr->size[2]/abc[2]);
  *n=2*(*n)+2;

  while (*spec=='f'){
    *n*=2;
    spec++;
  }

}

#endif

/* Simple trapezium rule */
double accumulate(double *points, double *acc, double len, int n){
  int i;
  double width,sum;

  if (n<1) return 0;
  if (n==1){
    if (acc) acc[0]=0;
    return 0;
  }
  if (n==2){
    if (acc){
      acc[0]=0;
      acc[1]=len*(points[0]+points[1])/2;
    }
    return len*(points[0]+points[1])/2;
  }

  if (flags&SIMPSON) return acc_simpson(points,acc,len,n);

  width=len/(n-1);

  if (!acc){
    sum=(points[0]+points[n-1])/2;
    for(i=1;i<n-1;i++)
      sum+=points[i];
    sum*=width;
    return(sum);
  }
      
  acc[0]=0;
  for(i=1;i<n;i++)
    acc[i]=acc[i-1]+0.5*(points[i-1]+points[i])*width;

  return acc[n-1];
}

/* This routine uses Simpson's 1/3 and 3/8 rules, and should be
 * exact for cubics. Simpson's 1/3 rule used for n_in odd, else
 * three subtracted to make n odd, and cubic fitted to last four
 * points.
 */
double acc_simpson(double *points, double *acc, double len, int n_in){
  int i,n;
  double width,sum;

  if (n_in<1) return 0;
  if (n_in==1){
    if (acc) acc[0]=0;
    return 0;
  }
  if (n_in==2){
    if (acc){
      acc[0]=0;
      acc[1]=len*(points[0]+points[1])/2;
    }
    return len*(points[0]+points[1])/2;
  }
  
  width=len/(n_in-1);
  
  if (!acc){
    n=n_in;
    if ((n&1)==0) n=n-3;
    sum=0;
    if (n>1){
      sum=points[0];
      for(i=1;i<n-1;i++){
	if (i&1) sum+=4*points[i];
	else sum+=2*points[i];
      }
      sum+=points[n-1];
      sum=sum/3;
    }
    if (n!=n_in)
      sum+=3*(points[n-1]+3*points[n]+3*points[n+1]+points[n+2])/8;
    sum*=width;
    return(sum);
  }
  
  n=n_in;
  if ((n&1)==0) n=n-1;
  acc[0]=0;
  if (n_in==3)
    acc[1]=(5*points[0]+8*points[1]-points[2])/12;
  else
    acc[1]=(9*points[0]+19*points[1]-5*points[2]+points[3])/24;
  acc[2]=(points[0]+4*points[1]+points[2])/3;
  for(i=2;i<n-2;i+=2){
    acc[i+1]=acc[i-2]+3*(points[i-2]+3*points[i-1]+3*points[i]+points[i+1])/8;
    acc[i+2]=acc[i]+(points[i]+4*points[i+1]+points[i+2])/3;
  }
  if (i==n_in-2){
    acc[n]=acc[n-3]+3*(points[n-3]+3*points[n-2]+3*points[n-1]+points[n])/8;
  }
  for(i=0;i<n_in;i++) acc[i]*=width;
  return(acc[n_in-1]);
}


double interpolate_fspherical(struct grid *g, int comp, struct unit_cell *c,
			      double *origin_frac, double rad, int n,
			      double *points, char mode){
  int nn[3],i,j,k,ii,jj,kk,ngx,ngy,ngz,ngpts;
  double *rgrid,*ptr,*ptr2,gv[3],gva[3],theta,sum,integral,radius;
  double dot,*modk,store[2];
  double *pts;
  
  if ((comp<=0)||(comp>g->comps)){
    fprintf(stderr,"Invalid value of comp in interpolate_fspherical. "
            "Have %d, valid 1 to %d\n",comp,g->comps);
    exit(1);
  }

  if ((rad==0)&&(n>1))
    error_exit("zero radius but multiple points in interpolate_fspherical");

  if (debug)
    fprintf(stderr,"Performing spherical integrals in reciprocal space\n");
  
  if (n==0) return 0;
  if (points==NULL){
    pts=store;
    n=2;
  }
  else
    pts=points;

  /* Copy real grid data to complex rgrid prior to FFT */
  ngpts=g->size[0]*g->size[1]*g->size[2];
  rgrid=malloc(2*ngpts*sizeof(double));
  if (!rgrid) error_exit("Malloc error in interpolate_fspherical");
  
  ptr=g->data+(comp-1)*ngpts;
  ptr2=rgrid;
  for(i=0;i<ngpts;i++){
    *(ptr2++)=*(ptr++);
    *(ptr2++)=0;
  }

  /* FFT */

  nn[0]=g->size[2];
  nn[1]=g->size[1];
  nn[2]=g->size[0];
  
  fft3d(rgrid,nn,-1);

  ngx=g->size[0];
  ngy=g->size[1];
  ngz=g->size[2];

  /* Create array for storing |k| */

  modk=malloc(ngpts*sizeof(double));

  if (!modk) error_exit("malloc error in interpolate_spherical");

  kk=0;
  for(i=0;i<ngx;i++){
    gv[0]=i;
    if (gv[0]>ngx/2) gv[0]=gv[0]-ngx;
    for(j=0;j<ngy;j++){
      gv[1]=j;
      if (gv[1]>ngy/2) gv[1]=gv[1]-ngy;
      for(k=0;k<ngz;k++){
	gv[2]=k;
	if (gv[2]>ngz/2) gv[2]=gv[2]-ngz;

	for(ii=0;ii<3;ii++){
	  gva[ii]=0;
	  for(jj=0;jj<3;jj++)
	    gva[ii]+=gv[jj]*c->recip[jj][ii];
	}
	
	dot=2*M_PI*
	  (gv[0]*origin_frac[0]+gv[1]*origin_frac[1]+gv[2]*origin_frac[2]);
	rgrid[kk]=rgrid[2*kk]*cos(dot)-rgrid[2*kk+1]*sin(dot);
	modk[kk]=2*M_PI*sqrt(gva[0]*gva[0]+gva[1]*gva[1]+gva[2]*gva[2]);
	kk++;
      }
    }
  }

  /* radius=0 terms */
  if ((mode=='a')||(mode=='w'))
    pts[0]=0;
  else{
    sum=rgrid[0];
    for(j=1;j<ngpts;j++){
      sum+=rgrid[j];
    }
    sum/=ngpts;
    if (rad==0){
      free(modk);
      free(rgrid);
      return sum;
    }
    pts[0]=sum;
  }
  /* radius!=0 */
  for(i=1;i<n;i++){
    radius=i*rad/(n-1);
    /* g=0 contribution */
    if (mode=='a'){
      sum=radius*radius*radius*rgrid[0]/3;
      for(j=1;j<ngpts;j++){
	theta=radius*modk[j];
	sum+=rgrid[j]*(sin(theta)/modk[j]-radius*cos(theta))/
	  (modk[j]*modk[j]);
      }
      sum*=4*M_PI/ngpts;
      pts[i]=sum;
    }
    else{
      sum=rgrid[0];
      for(j=1;j<ngpts;j++){
	sum+=rgrid[j]*sin(modk[j]*radius)/(modk[j]*radius);
      }
      sum/=ngpts;
      if (mode=='w')
	pts[i]=4*M_PI*radius*radius*sum;
      else
	pts[i]=sum;
    }
  }

  if (mode!='a'){
  /* g=0 contribution */
    integral=rad*rad*rad*rgrid[0]/3;
    for(j=1;j<ngpts;j++){
      theta=rad*modk[j];
      integral+=rgrid[j]*(sin(theta)/modk[j]-rad*cos(theta))/
	(modk[j]*modk[j]);
    }
    integral*=4*M_PI/ngpts;
  }
  else
    integral=pts[n-1];
  
  free(modk);
  free(rgrid);

  if ((debug)&&(g->comps==1)){
    fprintf(stderr,"Spherical integral:   %f\n",integral);
  }

  return (3*integral/(4*M_PI*rad*rad*rad));
  
}

/* Returns average over sphere using Fourier method
 *
 * BUG FOR COMP!=1
 *
 */
 
void interpolate_fr(struct grid *g, int comp, struct unit_cell *c,
		    double *origin_frac_in,
		    double rad, int n, double *pts, char axis, char mode){
  int nn[3],i,j,ii,jj,kk,ngx,ngy,ngpts;
  double *rgrid,*ptr,*ptr2,gv[3],gva[3],theta,sum,integral,radius,len_c;
  double dot,*modk,recip[2][3],origin_frac[2];
  double *points,*wpoints;

  if ((comp<=0)||(comp>g->comps)){
    fprintf(stderr,"Invalid value of comp in interpolate_fr. "
            "Have %d, valid 1 to %d\n",comp,g->comps);
    exit(1);
  }

  if (n==0) return;
  
  if (debug)
    fprintf(stderr,"Performing cylindrical integrals in reciprocal space\n");

  reduce2d(g->data,g->size,axis);

  if (axis=='c'){
    origin_frac[0]=origin_frac_in[0];
    origin_frac[1]=origin_frac_in[1];
    for(i=0;i<3;i++) recip[0][i]=c->recip[0][i];
    for(i=0;i<3;i++) recip[1][i]=c->recip[1][i];
  }
  else if (axis=='b'){
    origin_frac[0]=origin_frac_in[0];
    origin_frac[1]=origin_frac_in[2];
    for(i=0;i<3;i++) recip[0][i]=c->recip[0][i];
    for(i=0;i<3;i++) recip[1][i]=c->recip[2][i];
  }
  else if (axis=='a'){
    origin_frac[0]=origin_frac_in[1];
    origin_frac[1]=origin_frac_in[2];
    for(i=0;i<3;i++) recip[0][i]=c->recip[1][i];
    for(i=0;i<3;i++) recip[1][i]=c->recip[2][i];
  }
  else
    error_exit("Impossible axis");
  
  points=pts;
  if (mode=='a') points=NULL;
  if ((pts)&&(mode=='w')){
    wpoints=pts;
    points=NULL;
  }
  else{
    wpoints=malloc(n*sizeof(double));
    if (!wpoints) error_exit("malloc error in interpolate_r");
  }

  len_c=0;
  for(i=0;i<3;i++) len_c+=c->basis[axis-'a'][i]*c->basis[axis-'a'][i];
  len_c=sqrt(len_c);
  
  /* Copy real grid data to complex rgrid prior to FFT */
  ngpts=g->size[0]*g->size[1];
  rgrid=malloc(2*ngpts*sizeof(double));
  if (!rgrid) error_exit("Malloc error in interpolate_fr");
  
  ptr=g->data;
  ptr2=rgrid+(comp-1)*ngpts;
  for(i=0;i<ngpts;i++){
    *(ptr2++)=*(ptr++);
    *(ptr2++)=0;
  }

  /* FFT */

  nn[0]=g->size[2];
  nn[1]=g->size[1];
  nn[2]=g->size[0];
  
  fft3d(rgrid,nn,-1);

  ngx=g->size[0];
  ngy=g->size[1];

  /* Create array for storing |k| */

  modk=malloc(ngpts*sizeof(double));

  if (!modk) error_exit("malloc error in interpolate_fr");

  kk=0;
  for(i=0;i<ngx;i++){
    gv[0]=i;
    if (gv[0]>ngx/2) gv[0]=gv[0]-ngx;
    for(j=0;j<ngy;j++){
      gv[1]=j;
      if (gv[1]>ngy/2) gv[1]=gv[1]-ngy;

      for(ii=0;ii<3;ii++){
	gva[ii]=0;
	for(jj=0;jj<2;jj++)
	  gva[ii]+=gv[jj]*recip[jj][ii];
      }
	
      dot=2*M_PI*(gv[0]*origin_frac[0]+gv[1]*origin_frac[1]);
      rgrid[kk]=rgrid[2*kk]*cos(dot)-rgrid[2*kk+1]*sin(dot);
      modk[kk]=2*M_PI*sqrt(gva[0]*gva[0]+gva[1]*gva[1]+gva[2]*gva[2]);
      kk++;
    }
  }

    /* radius=0 terms */
  if (wpoints) wpoints[0]=0;
  if ((mode=='a')&&(pts))
    pts[0]=0;
  if ((points)||(rad==0)){
    sum=rgrid[0];
    for(j=1;j<ngpts;j++){
      sum+=rgrid[j];
    }
    sum/=ngpts;
    if (rad==0){
      free(modk);
      free(rgrid);
      return;
    }
    points[0]=sum;
  }

  /* radius!=0 */
  for(i=1;i<n;i++){
    radius=i*rad/(n-1);
    /* g=0 contribution */
    if (mode=='a'){
      sum=M_PI*radius*radius*rgrid[0];
      for(j=1;j<ngpts;j++){
      	theta=radius*modk[j];
      	sum+=2*M_PI*rgrid[j]*radius*j1(theta)/modk[j];
      }
      sum*=len_c/ngpts;
      pts[i]=sum;
    }
    else{
      sum=rgrid[0];
      for(j=1;j<ngpts;j++){
	theta=radius*modk[j];
	sum+=rgrid[j]*(j1(theta)/(modk[j]*radius)+0.5*(j0(theta)-
						       jn(2,theta)));
      }
      sum/=ngpts;
      if (points) points[i]=sum;
      wpoints[i]=2*M_PI*radius*len_c*sum;
    }
  }

  if (mode=='a')
    integral=pts[n-1];
  else
     integral=accumulate(wpoints,NULL,rad,n);
  
  if (debug){
    fprintf(stderr,"Radial integral: %f\n",integral);
  }

  if (mode!='w') free(wpoints);
  
}

void reduce2d(double *data, int grid[3], char axis){
  int i,j,k;
  double sum;

  if (debug>1){
    sum=0;
    for(i=0;i<grid[0]*grid[1]*grid[2];i++)
      sum+=data[i];
    fprintf(stderr,"reduce2d called, grid %d %d %d sum %g\n",
	    grid[0],grid[1],grid[2],sum);
  }
  
  if (axis=='c'){
    for(i=0;i<grid[0];i++){
      for(j=0;j<grid[1];j++){
	sum=0;
	for(k=0;k<grid[2];k++)
	  sum+=data[i*grid[1]*grid[2]+j*grid[2]+k];
	data[i*grid[1]+j]=sum/grid[2];
      }
    }
    grid[2]=1;
  }
  else if (axis=='b'){
    for(i=0;i<grid[0];i++){
      for(j=0;j<grid[2];j++){
	sum=0;
	for(k=0;k<grid[1];k++)
	   sum+=data[i*grid[1]*grid[2]+k*grid[2]+j];
	data[i*grid[2]+j]=sum/grid[1];
      }
    }
    grid[1]=grid[2];
    grid[2]=1;
  }
  else if (axis=='a'){
    for(i=0;i<grid[1];i++){
      for(j=0;j<grid[2];j++){
	sum=0;
	for(k=0;k<grid[0];k++)
	   sum+=data[k*grid[1]*grid[2]+i*grid[2]+j];
	data[i*grid[2]+j]=sum/grid[0];
      }
    }
    grid[0]=grid[1];
    grid[1]=grid[2];
    grid[2]=1;
  }
  else
    error_exit("invalid axis in reduce2d");

  if (debug>1){
    sum=0;
    for(i=0;i<grid[0]*grid[1]*grid[2];i++)
      sum+=data[i];
    fprintf(stderr,"reduce2d ends, grid %d %d %d sum %g\n",
	    grid[0],grid[1],grid[2],sum);
  }
  

}

/* as reduce2d, but for grid storing reals as complex */
void reduce2d_z(double *data, int grid[3], char axis){
  int i,j,k;
  double sum;

  if (axis=='c'){
    for(i=0;i<grid[0];i++){
      for(j=0;j<grid[1];j++){
	sum=0;
	for(k=0;k<grid[2];k++)
	  sum+=data[2*(i*grid[1]*grid[2]+j*grid[2]+k)];
	data[2*(i*grid[1]+j)]=sum/grid[2];
	data[2*(i*grid[1]+j)+1]=0;
      }
    }
    grid[2]=1;
  }
  else if (axis=='b'){
    for(i=0;i<grid[0];i++){
      for(j=0;j<grid[2];j++){
	sum=0;
	for(k=0;k<grid[1];k++)
	  sum+=data[2*(i*grid[1]*grid[2]+k*grid[2]+j)];
	data[2*(i*grid[2]+j)]=sum/grid[1];
	data[2*(i*grid[2]+j)+1]=0;
      }
    }
    grid[1]=grid[2];
    grid[2]=1;
  }
  else if (axis=='a'){
    for(i=0;i<grid[1];i++){
      for(j=0;j<grid[2];j++){
	sum=0;
	for(k=0;k<grid[0];k++)
	  sum+=data[2*(k*grid[1]*grid[2]+i*grid[2]+j)];
	data[2*(i*grid[2]+j)]=sum/grid[0];
	data[2*(i*grid[2]+j)+1]=0;
      }
    }
    grid[0]=grid[1];
    grid[1]=grid[2];
    grid[2]=1;
  }
  else
    error_exit("invalid axis in reduce2d_z");
}
