import 'react/jsx-runtime';
import '@emotion/css';
import { AuthMethod } from './types.js';
import './auth-method/AuthMethodSettings.js';
import '@grafana/ui';
import 'react';
import '@grafana/data';

const headerNamePrefix = "httpHeaderName";
const headerValuePrefix = "httpHeaderValue";
function convertLegacyAuthProps({
  config,
  onChange
}) {
  const props = {
    selectedMethod: getSelectedMethod(config),
    onAuthMethodSelect: getOnAuthMethodSelectHandler(config, onChange),
    basicAuth: getBasicAuthProps(config, onChange),
    TLS: getTLSProps(config, onChange),
    customHeaders: getCustomHeaders(config, onChange),
    readOnly: config.readOnly
  };
  return props;
}
function getSelectedMethod(config) {
  if (config.basicAuth) {
    return AuthMethod.BasicAuth;
  }
  if (config.withCredentials) {
    return AuthMethod.CrossSiteCredentials;
  }
  if (config.jsonData.oauthPassThru) {
    return AuthMethod.OAuthForward;
  }
  return AuthMethod.NoAuth;
}
function getOnAuthMethodSelectHandler(config, onChange) {
  return (method) => {
    onChange({
      ...config,
      basicAuth: method === AuthMethod.BasicAuth,
      withCredentials: method === AuthMethod.CrossSiteCredentials,
      jsonData: {
        ...config.jsonData,
        oauthPassThru: method === AuthMethod.OAuthForward
      }
    });
  };
}
function getBasicAuthProps(config, onChange) {
  return {
    user: config.basicAuthUser,
    passwordConfigured: config.secureJsonFields.basicAuthPassword,
    onUserChange: (user) => onChange({ ...config, basicAuthUser: user }),
    onPasswordChange: (password) => onChange({
      ...config,
      secureJsonData: {
        ...config.secureJsonData,
        basicAuthPassword: password
      }
    }),
    onPasswordReset: () => onChange({
      ...config,
      secureJsonData: { ...config.secureJsonData, basicAuthPassword: "" },
      secureJsonFields: {
        ...config.secureJsonFields,
        basicAuthPassword: false
      }
    })
  };
}
function getTLSProps(config, onChange) {
  var _a, _b, _c;
  return {
    selfSignedCertificate: {
      enabled: Boolean(config.jsonData.tlsAuthWithCACert),
      certificateConfigured: !!((_a = config.secureJsonFields) == null ? undefined : _a.tlsCACert),
      onToggle: (enabled) => enabled ? onChange({
        ...config,
        jsonData: { ...config.jsonData, tlsAuthWithCACert: enabled }
      }) : onChange({
        ...config,
        jsonData: { ...config.jsonData, tlsAuthWithCACert: enabled },
        secureJsonData: { ...config.secureJsonData, tlsCACert: "" },
        secureJsonFields: { ...config.secureJsonFields, tlsCACert: false }
      }),
      onCertificateChange: (certificate) => onChange({
        ...config,
        secureJsonData: { ...config.secureJsonData, tlsCACert: certificate }
      }),
      onCertificateReset: () => onChange({
        ...config,
        secureJsonData: { ...config.secureJsonData, tlsCACert: "" },
        secureJsonFields: { ...config.secureJsonFields, tlsCACert: false }
      })
    },
    TLSClientAuth: {
      enabled: config.jsonData.tlsAuth,
      serverName: config.jsonData.serverName,
      clientCertificateConfigured: !!((_b = config.secureJsonFields) == null ? undefined : _b.tlsClientCert),
      clientKeyConfigured: !!((_c = config.secureJsonFields) == null ? undefined : _c.tlsClientKey),
      onToggle: (enabled) => enabled ? onChange({
        ...config,
        jsonData: { ...config.jsonData, tlsAuth: enabled }
      }) : onChange({
        ...config,
        jsonData: { ...config.jsonData, tlsAuth: enabled, serverName: "" },
        secureJsonData: { ...config.secureJsonData, tlsClientCert: "", tlsClientKey: "" },
        secureJsonFields: { ...config.secureJsonFields, tlsClientCert: false, tlsClientKey: false }
      }),
      onServerNameChange: (serverName) => onChange({
        ...config,
        jsonData: { ...config.jsonData, serverName }
      }),
      onClientCertificateChange: (clientCertificate) => onChange({
        ...config,
        secureJsonData: {
          ...config.secureJsonData,
          tlsClientCert: clientCertificate
        }
      }),
      onClientCertificateReset: () => onChange({
        ...config,
        secureJsonData: {
          ...config.secureJsonData,
          tlsClientCert: ""
        },
        secureJsonFields: {
          ...config.secureJsonFields,
          tlsClientCert: false
        }
      }),
      onClientKeyChange: (clientKey) => onChange({
        ...config,
        secureJsonData: {
          ...config.secureJsonData,
          tlsClientKey: clientKey
        }
      }),
      onClientKeyReset: () => onChange({
        ...config,
        secureJsonData: {
          ...config.secureJsonData,
          tlsClientKey: ""
        },
        secureJsonFields: {
          ...config.secureJsonFields,
          tlsClientKey: false
        }
      })
    },
    skipTLSVerification: {
      enabled: config.jsonData.tlsSkipVerify,
      onToggle: (enabled) => onChange({
        ...config,
        jsonData: { ...config.jsonData, tlsSkipVerify: enabled }
      })
    }
  };
}
function getCustomHeaders(config, onChange) {
  const headers = Object.keys(config.jsonData).filter((key) => key.startsWith(headerNamePrefix)).sort().map((key) => {
    var _a;
    const index = key.slice(headerNamePrefix.length);
    return {
      name: config.jsonData[key],
      configured: (_a = config.secureJsonFields[`${headerValuePrefix}${index}`]) != null ? _a : false
    };
  });
  return {
    headers,
    onChange: (headers2) => {
      const newJsonData = Object.fromEntries(
        Object.entries(config.jsonData).filter(([key]) => !key.startsWith(headerNamePrefix))
      );
      const newSecureJsonData = Object.fromEntries(
        Object.entries(config.secureJsonData || {}).filter(([key]) => !key.startsWith(headerValuePrefix))
      );
      const newSecureJsonFields = Object.fromEntries(
        Object.entries(config.secureJsonFields).filter(([key]) => !key.startsWith(headerValuePrefix))
      );
      headers2.forEach((header, index) => {
        newJsonData[`${headerNamePrefix}${index + 1}`] = header.name;
        if (header.configured) {
          newSecureJsonFields[`${headerValuePrefix}${index + 1}`] = true;
        } else {
          newSecureJsonData[`${headerValuePrefix}${index + 1}`] = header.value;
        }
      });
      onChange({
        ...config,
        jsonData: newJsonData,
        secureJsonData: newSecureJsonData,
        secureJsonFields: newSecureJsonFields
      });
    }
  };
}

export { convertLegacyAuthProps, getBasicAuthProps, getCustomHeaders, getOnAuthMethodSelectHandler, getSelectedMethod, getTLSProps };
//# sourceMappingURL=utils.js.map
