# frozen_string_literal: true

module Gitlab
  module BackgroundMigration
    class BackfillVulnerabilityDetectionTransitions < BatchedMigrationJob
      operation_name :backfill_vulnerability_detection_transitions
      feature_category :vulnerability_management

      class DetectionTransition < ::SecApplicationRecord
        self.table_name = 'vulnerability_detection_transitions'
      end

      def perform
        timestamp = Time.current

        each_sub_batch do |sub_batch|
          backfill_detection_transitions(sub_batch, timestamp)
        end
      end

      private

      def backfill_detection_transitions(sub_batch, timestamp)
        existing_occurrence_ids = DetectionTransition
          .where(vulnerability_occurrence_id: sub_batch.select(:id))
          .pluck(:vulnerability_occurrence_id)

        records = sub_batch
          .joins("INNER JOIN vulnerabilities ON vulnerabilities.id = vulnerability_occurrences.vulnerability_id")
          .where(vulnerabilities: { resolved_on_default_branch: true })
          .where.not(id: existing_occurrence_ids)
          .pluck(:id, :project_id)
          .map do |occurrence_id, project_id|
            {
              vulnerability_occurrence_id: occurrence_id,
              project_id: project_id,
              detected: false,
              created_at: timestamp,
              updated_at: timestamp
            }
          end

        return if records.empty?

        DetectionTransition.insert_all!(records)
      end
    end
  end
end
