// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_dk_dv.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_dk_dv__A0__F298

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 32 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_128_3_F_T_0___gfx950)xyzw" };
#else
{ R"xyzw(amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_128_3_F_T_0___gfx950)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊bf16@16_128_3_F_T_0)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx950)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0x8e8d7e73u, 0x1b39cbddu, 0, 6 }, // 8e8d7e731b39cbdd = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_128_3_F_T_0___gfx950__P__16_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0xf51b28c7u, 0x5d1b6311u, 0, 46 }, // f51b28c75d1b6311 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_128_3_F_T_0___gfx950__P__16_16__CO__wave3_warp2_stg1--Arch_gfx950
    { 0x0d5c6aa7u, 0x65c66097u, 23, 29 }, // 0d5c6aa765c66097 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_128_3_F_T_0___gfx950__P__32_16__CO__wave1_warp2_stg1--Arch_gfx950
    { 0x153d6d8eu, 0xeb07cba1u, 23, 6 }, // 153d6d8eeb07cba1 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_128_3_F_T_0___gfx950__P__32_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0x679b1829u, 0xbc9a9977u, 80, 6 }, // 679b1829bc9a9977 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_128_3_F_T_0___gfx950__P__32_32__CO__wave2_warp2_stg1--Arch_gfx950
    { 0x622786f0u, 0x37107acdu, 103, 29 }, // 622786f037107acd = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_128_3_F_T_0___gfx950__P__64_16__CO__wave1_warp2_stg1--Arch_gfx950
    { 0xbacc9397u, 0xf0847c6eu, 103, 109 }, // bacc9397f0847c6e = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_128_3_F_T_0___gfx950__P__64_16__CO__wave1_warp4_stg1--Arch_gfx950
    { 0x7981993au, 0xc8efe5f1u, 103, 6 }, // 7981993ac8efe5f1 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_128_3_F_T_0___gfx950__P__64_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0x5245d858u, 0xba745eaeu, 103, 126 }, // 5245d858ba745eae = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_128_3_F_T_0___gfx950__P__64_16__CO__wave2_warp4_stg1--Arch_gfx950
    { 0x477e60b7u, 0xccb969a5u, 160, 29 }, // 477e60b7ccb969a5 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_128_3_F_T_0___gfx950__P__64_32__CO__wave1_warp2_stg1--Arch_gfx950
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_dk_dv_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_dk_dv_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx950_mod0
{{1,0,1,1,1,0,0,1,1,1},
 {3,3,3,3,0,3,1,0,1,1},
 {6,6,6,3,1,2,2,0,0,1},
 {8,8,8,2,0,2,2,2,1,0},
 {8,8,8,2,0,9,2,2,2,1},
 {8,8,8,9,4,9,7,2,2,2},
 {8,8,5,9,4,9,7,7,7,2},
 {8,8,8,7,4,9,9,7,7,8},
 {8,8,5,7,4,9,9,9,9,9},
 {8,8,8,7,4,9,9,5,9,9}}
// End of GPU gfx950_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelDkDvContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelDkDvContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_dk_dv__A0__F298 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 4;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_dk_dv__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 4;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*bf16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// D = "*fp32:16"
// stride_bz = 0
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = 0
// philox_seed_ptr = 0
// philox_offset1 = 0
// philox_offset2 = 0
// Window_left = "i32"
// BLOCK_DMODEL = 128
// CAUSAL_TYPE = 3
// ENABLE_DROPOUT = False
// PADDED_HEAD = True
// BIAS_TYPE = 0

// SELECT * FROM FLASH$bwd_kernel_dk_dv WHERE gpu IN ('gfx950_mod0') AND inputs$Q_dtype = 'torch.bfloat16' AND inputs$BLOCK_DMODEL = 128 AND inputs$CAUSAL_TYPE = 3 AND inputs$ENABLE_DROPOUT = False AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 0

// vim: set fileencoding=utf-8

