/*
 * This file is part of OpenTTD.
 * OpenTTD is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, version 2.
 * OpenTTD is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details. You should have received a copy of the GNU General Public License along with OpenTTD. If not, see <https://www.gnu.org/licenses/old-licenses/gpl-2.0>.
 */

/** @file table/settings.h Settings to save in the savegame and config file. */

#include "table/strings.h"

/* Callback function used in _settings[] as well as _company_settings[] */
static std::optional<uint32_t> ConvertLandscape(std::string_view value);

static StringID SettingTitleWallclock(const IntSettingDesc &sd);
static StringID SettingHelpWallclock(const IntSettingDesc &sd);

/****************************
 * OTTD specific INI stuff
 ****************************
 * DO NOT edit this file. This file is automatically generated from the contents of /src/table/
 */

/**
 * Settings-macro usage:
 * The list might look daunting at first, but is in general easy to understand.
 * The macros can be grouped depending on where the config variable is
 * stored:
 * 1. SDTG_something
 *    These are for global variables. Here 'var' refers to a global variable.
 * 2. SDTC_something
 *    These are for client-only variables. Here the 'var' refers to an
 *    entry inside _settings_client.
 * 3. SDT_something
 *    These are for members in the struct described by the current
 *    #SettingDesc list / .ini file.  Here, 'base' specifies type of the
 *    struct while 'var' points out the member of the struct (the actual
 *    struct to store it in is implicitly defined by the #SettingDesc
 *    list / .ini file preamble the entry is in).
 *
 * The something part defines the type of variable to store. There are a
 * lot of types. Easy ones are:
 * - VAR:  any number type, 'type' field specifies what number. eg int8_t or uint32
 * - BOOL: a boolean number type
 * - STR:  a string or character. 'type' field specifies what string. Normal, string, or quoted
 * A bit more difficult to use are MMANY (meaning ManyOfMany) and OMANY (OneOfMany)
 * These are actually normal numbers, only bitmasked. In MMANY several bits can
 * be set, in the other only one.
 * If nothing fits you, you can use the GENERAL macros, but it exposes the
 * internal structure somewhat so it needs a little looking.
 *
 * While reading values from openttd.cfg, some values may not be converted
 * properly, for any kind of reasons.  In order to allow a process of self-cleaning
 * mechanism, a callback procedure is made available.  You will have to supply the function, which
 * will work on a string, one function per setting. And of course, enable the callback param
 * on the appropriate macro.
 */

#define NSD(type, ...) SettingVariant { std::in_place_type<type##SettingDesc>, __VA_ARGS__ }

/* Macros for various objects to go in the configuration file.
 * This section is for global variables */
#define SDTG_VAR(name, type, flags, var, def, min, max, interval, str, strhelp, strval, pre_check, post_callback, get_title_hook, get_help_hook, set_value_dparams_hook, get_def_hook, get_range_hook, from, to, cat, extra, startup)\
	NSD(Int, SLEG_GENERAL(name, SL_VAR, var, type, 1, from, to, extra), flags, startup, def, min, max, interval, str, strhelp, strval, cat, pre_check, post_callback, get_title_hook, get_help_hook, set_value_dparams_hook, get_def_hook, get_range_hook)

#define SDTG_BOOL(name, flags, var, def, str, strhelp, strval, pre_check, post_callback, get_title_hook, get_help_hook, set_value_dparams_hook, get_def_hook, from, to, cat, extra, startup)\
	NSD(Bool, SLEG_GENERAL(name, SL_VAR, var, SLE_BOOL, 1, from, to, extra), flags, startup, def, str, strhelp, strval, cat, pre_check, post_callback, get_title_hook, get_help_hook, set_value_dparams_hook, get_def_hook)

#define SDTG_LIST(name, type, flags, var, def, length, from, to, cat, extra, startup)\
	NSD(List, SLEG_GENERAL(name, SL_ARR, var, type, length, from, to, extra),flags, startup, def)

#define SDTG_SSTR(name, type, flags, var, def, max_length, pre_check, post_callback, from, to, cat, extra, startup)\
	NSD(String, SLEG_GENERAL(name, SL_STDSTR, var, type, sizeof(var), from, to, extra), flags, startup, def, max_length, pre_check, post_callback)

#define SDTG_OMANY(name, type, flags, var, def, max, full, str, strhelp, strval, pre_check, post_callback, get_title_hook, get_help_hook, set_value_dparams_hook, get_def_hook, from, to, cat, extra, startup)\
	NSD(OneOfMany, SLEG_GENERAL(name, SL_VAR, var, type, 1, from, to, extra), flags, startup, def, max, str, strhelp, strval, cat, pre_check, post_callback, get_title_hook, get_help_hook, set_value_dparams_hook, get_def_hook, full, nullptr)

#define SDTG_MMANY(name, type, flags, var, def, full, str, strhelp, strval, pre_check, post_callback, get_title_hook, get_help_hook, set_value_dparams_hook, get_def_hook, from, to, cat, extra, startup)\
	NSD(ManyOfMany, SLEG_GENERAL(name, SL_VAR, var, type, 1, from, to, extra), flags, startup, def, str, strhelp, strval, cat, pre_check, post_callback, get_title_hook, get_help_hook, set_value_dparams_hook, get_def_hook, full, nullptr)

/* Macros for various objects to go in the configuration file.
 * This section is for structures where their various members are saved */
#define SDT_VAR(base, var, type, flags, def, min, max, interval, str, strhelp, strval, pre_check, post_callback, get_title_hook, get_help_hook, set_value_dparams_hook, get_def_hook, get_range_hook, from, to, cat, extra, startup)\
	NSD(Int, SLE_GENERAL(SL_VAR, base, var, type, 1, from, to, extra), flags, startup, def, min, max, interval, str, strhelp, strval, cat, pre_check, post_callback, get_title_hook, get_help_hook, set_value_dparams_hook, get_def_hook, get_range_hook)

#define SDT_VAR_NAME(base, var, type, flags, def, min, max, interval, str, strhelp, strval, pre_check, post_callback, get_title_hook, get_help_hook, set_value_dparams_hook, get_def_hook, from, to, cat, extra, startup, name)\
	NSD(Int, SLE_GENERAL_NAME(SL_VAR, name, base, var, type, 1, from, to, extra), flags, startup, def, min, max, interval, str, strhelp, strval, cat, pre_check, post_callback, get_title_hook, get_help_hook, set_value_dparams_hook, get_def_hook)

#define SDT_BOOL(base, var, flags, def, str, strhelp, strval, pre_check, post_callback, get_title_hook, get_help_hook, set_value_dparams_hook, get_def_hook, from, to, cat, extra, startup)\
	NSD(Bool, SLE_GENERAL(SL_VAR, base, var, SLE_BOOL, 1, from, to, extra), flags, startup, def, str, strhelp, strval, cat, pre_check, post_callback, get_title_hook, get_help_hook, set_value_dparams_hook, get_def_hook)

#define SDT_LIST(base, var, type, flags, def, from, to, cat, extra, startup)\
	NSD(List, SLE_GENERAL(SL_ARR, base, var, type, lengthof(((base*)8)->var), from, to, extra), flags, startup, def)

#define SDT_SSTR(base, var, type, flags, def, pre_check, post_callback, from, to, cat, extra, startup)\
	NSD(String, SLE_GENERAL(SL_STDSTR, base, var, type, sizeof(((base*)8)->var), from, to, extra), flags, startup, def, 0, pre_check, post_callback)

#define SDT_OMANY(base, var, type, flags, def, max, full, str, strhelp, strval, pre_check, post_callback, get_title_hook, get_help_hook, set_value_dparams_hook, get_def_hook, from, to, load, cat, extra, startup)\
	NSD(OneOfMany, SLE_GENERAL(SL_VAR, base, var, type, 1, from, to, extra), flags, startup, def, max, str, strhelp, strval, cat, pre_check, post_callback, get_title_hook, get_help_hook, set_value_dparams_hook, get_def_hook, full, load)

#define SDT_MMANY(base, var, type, flags, def, full, str, pre_check, post_callback, get_title_hook, get_help_hook, set_value_dparams_hook, get_def_hook, strhelp, strval, from, to, cat, extra, startup)\
	NSD(ManyOfMany, SLE_GENERAL(SL_VAR, base, var, type, 1, from, to, extra), flags, startup, def, str, strhelp, strval, cat, pre_check, post_callback, get_title_hook, get_help_hook, set_value_dparams_hook, get_def_hook, full, nullptr)


#define SDTC_VAR(var, type, flags, def, min, max, interval, str, strhelp, strval, pre_check, post_callback, get_title_hook, get_help_hook, set_value_dparams_hook, get_def_hook, get_range_hook, from, to, cat, extra, startup)\
	SDTG_VAR(#var, type, flags, _settings_client.var, def, min, max, interval, str, strhelp, strval, pre_check, post_callback, get_title_hook, get_help_hook, set_value_dparams_hook, get_def_hook, get_range_hook, from, to, cat, extra, startup)

#define SDTC_BOOL(var, flags, def, str, strhelp, strval, pre_check, post_callback, get_title_hook, get_help_hook, set_value_dparams_hook, get_def_hook, from, to, cat, extra, startup)\
	SDTG_BOOL(#var, flags, _settings_client.var, def, str, strhelp, strval, pre_check, post_callback, get_title_hook, get_help_hook, set_value_dparams_hook, get_def_hook, from, to, cat, extra, startup)

#define SDTC_LIST(var, type, flags, def, from, to, cat, extra, startup)\
	SDTG_LIST(#var, type, flags, _settings_client.var, def, lengthof(_settings_client.var), from, to, cat, extra, startup)

#define SDTC_SSTR(var, type, flags, def, max_length, pre_check, post_callback, from, to, cat, extra, startup)\
	SDTG_SSTR(#var, type, flags, _settings_client.var, def, max_length, pre_check, post_callback, from, to, cat, extra, startup)\

#define SDTC_OMANY(var, type, flags, def, max, full, str, strhelp, strval, pre_check, post_callback, get_title_hook, get_help_hook, set_value_dparams_hook, get_def_hook, from, to, cat, extra, startup)\
	SDTG_OMANY(#var, type, flags, _settings_client.var, def, max, full, str, strhelp, strval, pre_check, post_callback, get_title_hook, get_help_hook, set_value_dparams_hook, get_def_hook, from, to, cat, extra, startup)
static void UpdateAllServiceInterval(int32_t new_value);
static bool CanUpdateServiceInterval(VehicleType type, int32_t &new_value);
static void UpdateServiceInterval(VehicleType type, int32_t new_value);
static std::pair<StringParameter, StringParameter> SettingsValueAbsolute(const IntSettingDesc &sd, int32_t value);
static std::pair<StringParameter, StringParameter> ServiceIntervalSettingsValueText(const IntSettingDesc &sd, int32_t value);
static int32_t GetDefaultServiceInterval(const IntSettingDesc &sd, VehicleType type);
static std::tuple<int32_t, uint32_t> GetServiceIntervalRange(const IntSettingDesc &sd);
static const SettingVariant _company_settings_table[] = {
SDT_BOOL(CompanySettings, engine_renew,        SettingFlags({SettingFlag::PerCompany}), true,                        STR_CONFIG_SETTING_AUTORENEW_VEHICLE, STR_CONFIG_SETTING_AUTORENEW_VEHICLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_BASIC, 0, false),
SDT_VAR(CompanySettings, engine_renew_months, SLE_INT16, SettingFlags({SettingFlag::PerCompany}), 6, -12, 12, 0, STR_CONFIG_SETTING_AUTORENEW_MONTHS, STR_CONFIG_SETTING_AUTORENEW_MONTHS_HELPTEXT, STR_CONFIG_SETTING_AUTORENEW_MONTHS_VALUE_BEFORE, nullptr, nullptr, nullptr, nullptr, SettingsValueAbsolute, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_ADVANCED, 0, false),
SDT_VAR(CompanySettings, engine_renew_money, SLE_UINT, SettingFlags({SettingFlag::PerCompany, SettingFlag::GuiCurrency}), 100000, 0, 2000000, 0, STR_CONFIG_SETTING_AUTORENEW_MONEY, STR_CONFIG_SETTING_AUTORENEW_MONEY_HELPTEXT, STR_JUST_CURRENCY_LONG, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_ADVANCED, 0, false),
SDT_BOOL(CompanySettings, renew_keep_length,        SettingFlags({SettingFlag::PerCompany}), false,                        STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_ADVANCED, 0, false),
SDT_BOOL(CompanySettings, vehicle.servint_ispercent,        SettingFlags({SettingFlag::PerCompany}), false,                        STR_CONFIG_SETTING_SERVINT_ISPERCENT, STR_CONFIG_SETTING_SERVINT_ISPERCENT_HELPTEXT, STR_NULL, nullptr, UpdateAllServiceInterval, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_ADVANCED, 0, false),
SDT_VAR(CompanySettings, vehicle.servint_trains, SLE_UINT16, SettingFlags({SettingFlag::PerCompany, SettingFlag::GuiZeroIsSpecial}), DEF_SERVINT_DAYS_TRAINS, MIN_SERVINT_MINUTES, MAX_SERVINT_DAYS, 1, STR_CONFIG_SETTING_SERVINT_TRAINS, STR_CONFIG_SETTING_SERVINT_TRAINS_HELPTEXT, STR_CONFIG_SETTING_SERVINT_VALUE_DAYS, [](auto &new_value) { return CanUpdateServiceInterval(VEH_TRAIN, new_value); }, [](auto new_value) { UpdateServiceInterval(VEH_TRAIN, new_value); }, nullptr, nullptr, ServiceIntervalSettingsValueText, [](auto &sd) { return GetDefaultServiceInterval(sd, VEH_TRAIN); }, GetServiceIntervalRange, SL_MIN_VERSION, SL_MAX_VERSION, SC_ADVANCED, 0, false),
SDT_VAR(CompanySettings, vehicle.servint_roadveh, SLE_UINT16, SettingFlags({SettingFlag::PerCompany, SettingFlag::GuiZeroIsSpecial}), DEF_SERVINT_DAYS_ROADVEH, MIN_SERVINT_MINUTES, MAX_SERVINT_DAYS, 1, STR_CONFIG_SETTING_SERVINT_ROAD_VEHICLES, STR_CONFIG_SETTING_SERVINT_ROAD_VEHICLES_HELPTEXT, STR_CONFIG_SETTING_SERVINT_VALUE_DAYS, [](auto &new_value) { return CanUpdateServiceInterval(VEH_ROAD, new_value); }, [](auto new_value) { UpdateServiceInterval(VEH_ROAD, new_value); }, nullptr, nullptr, ServiceIntervalSettingsValueText, [](auto &sd) { return GetDefaultServiceInterval(sd, VEH_ROAD); }, GetServiceIntervalRange, SL_MIN_VERSION, SL_MAX_VERSION, SC_ADVANCED, 0, false),
SDT_VAR(CompanySettings, vehicle.servint_ships, SLE_UINT16, SettingFlags({SettingFlag::PerCompany, SettingFlag::GuiZeroIsSpecial}), DEF_SERVINT_DAYS_SHIPS, MIN_SERVINT_MINUTES, MAX_SERVINT_DAYS, 1, STR_CONFIG_SETTING_SERVINT_SHIPS, STR_CONFIG_SETTING_SERVINT_SHIPS_HELPTEXT, STR_CONFIG_SETTING_SERVINT_VALUE_DAYS, [](auto &new_value) { return CanUpdateServiceInterval(VEH_SHIP, new_value); }, [](auto new_value) { UpdateServiceInterval(VEH_SHIP, new_value); }, nullptr, nullptr, ServiceIntervalSettingsValueText, [](auto &sd) { return GetDefaultServiceInterval(sd, VEH_SHIP); }, GetServiceIntervalRange, SL_MIN_VERSION, SL_MAX_VERSION, SC_ADVANCED, 0, false),
SDT_VAR(CompanySettings, vehicle.servint_aircraft, SLE_UINT16, SettingFlags({SettingFlag::PerCompany, SettingFlag::GuiZeroIsSpecial}), DEF_SERVINT_DAYS_AIRCRAFT, MIN_SERVINT_MINUTES, MAX_SERVINT_DAYS, 1, STR_CONFIG_SETTING_SERVINT_AIRCRAFT, STR_CONFIG_SETTING_SERVINT_AIRCRAFT_HELPTEXT, STR_CONFIG_SETTING_SERVINT_VALUE_DAYS, [](auto &new_value) { return CanUpdateServiceInterval(VEH_AIRCRAFT, new_value); }, [](auto new_value) { UpdateServiceInterval(VEH_AIRCRAFT, new_value); }, nullptr, nullptr, ServiceIntervalSettingsValueText, [](auto &sd) { return GetDefaultServiceInterval(sd, VEH_AIRCRAFT); }, GetServiceIntervalRange, SL_MIN_VERSION, SL_MAX_VERSION, SC_ADVANCED, 0, false),
};
static const SettingVariant _currency_settings_table[] = {
SDT_VAR (CurrencySpec, rate, SLE_UINT16, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), 1, 0, UINT16_MAX, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_ADVANCED, 0, false),
SDT_SSTR(CurrencySpec, separator, SLE_STRQ, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), ".",                                                 nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_BASIC, 0, false),
SDT_VAR (CurrencySpec, to_euro, SLE_INT32, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), 0, CalendarTime::MIN_YEAR, CalendarTime::MAX_YEAR, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_ADVANCED, 0, false),
SDT_SSTR(CurrencySpec, prefix, SLE_STRQ, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), "",                                                 nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_ADVANCED, 0, false),
SDT_SSTR(CurrencySpec, suffix, SLE_STRQ, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), " credits",                                                 nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_ADVANCED, 0, false),
};
const std::array<std::string, GAME_DIFFICULTY_NUM> _old_diff_settings{"max_no_competitors", "competitor_start_time", "number_towns", "industry_density", "max_loan", "initial_interest", "vehicle_costs", "competitor_speed", "competitor_intelligence", "vehicle_breakdowns", "subsidy_multiplier", "construction_cost", "terrain_type", "quantity_sea_lakes", "economy", "line_reverse_mode", "disasters", "town_council_tolerance"};
std::array<uint16_t, GAME_DIFFICULTY_NUM> _old_diff_custom;
uint8_t _old_diff_level;                                 ///< Old difficulty level from old savegames
static void DifficultyNoiseChange(int32_t new_value);
static void MaxNoAIsChange(int32_t new_value);
static const SettingVariant _difficulty_settings_table[] = {
SDT_VAR (GameSettings, difficulty.max_no_competitors, SLE_UINT8, SettingFlags({}), 0,       0, MAX_COMPANIES - 1, 1, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, MaxNoAIsChange, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_97, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_VAR (GameSettings, difficulty.competitors_interval, SLE_UINT16, SettingFlags({}), 10,       MIN_COMPETITORS_INTERVAL, MAX_COMPETITORS_INTERVAL, 1, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_AI_START_DATE, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_VAR (GameSettings, difficulty.competitor_start_time, SLE_UINT8, SettingFlags({}), 2,       0, 3, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_97, SLV_110,        SC_ADVANCED, 0, false),
SDT_VAR (GameSettings, difficulty.number_towns, SLE_UINT8, SettingFlags({SettingFlag::NewgameOnly}), 2,       0, 4, 1, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NUM_VERY_LOW, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_97, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_VAR (GameSettings, difficulty.industry_density, SLE_UINT8, SettingFlags({SettingFlag::GuiDropdown}), ID_NORMAL,       0, ID_END - 1, 1, STR_CONFIG_SETTING_INDUSTRY_DENSITY, STR_CONFIG_SETTING_INDUSTRY_DENSITY_HELPTEXT, STR_FUNDING_ONLY, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_97, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_VAR (GameSettings, difficulty.max_loan, SLE_UINT32, SettingFlags({SettingFlag::NewgameOnly, SettingFlag::SceneditToo, SettingFlag::GuiCurrency, SettingFlag::GuiZeroIsSpecial}), 300000,       LOAN_INTERVAL, MAX_LOAN_LIMIT, LOAN_INTERVAL, STR_CONFIG_SETTING_MAXIMUM_INITIAL_LOAN, STR_CONFIG_SETTING_MAXIMUM_INITIAL_LOAN_HELPTEXT, STR_CONFIG_SETTING_MAXIMUM_INITIAL_LOAN_VALUE, [](auto &new_value) { new_value = (new_value + LOAN_INTERVAL / 2) / LOAN_INTERVAL * LOAN_INTERVAL; return true; }, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_97, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_VAR (GameSettings, difficulty.initial_interest, SLE_UINT8, SettingFlags({SettingFlag::NewgameOnly, SettingFlag::SceneditToo}), 2,       2, 4, 1, STR_CONFIG_SETTING_INTEREST_RATE, STR_CONFIG_SETTING_INTEREST_RATE_HELPTEXT, STR_CONFIG_SETTING_PERCENTAGE, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_97, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_VAR (GameSettings, difficulty.vehicle_costs, SLE_UINT8, SettingFlags({SettingFlag::NewgameOnly, SettingFlag::SceneditToo, SettingFlag::GuiDropdown}), 0,       0, 2, 1, STR_CONFIG_SETTING_RUNNING_COSTS, STR_CONFIG_SETTING_RUNNING_COSTS_HELPTEXT, STR_CONFIG_SETTING_RUNNING_COSTS_LOW, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_97, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_VAR (GameSettings, difficulty.competitor_speed, SLE_UINT8, SettingFlags({SettingFlag::GuiDropdown}), 2,       0, 4, 1, STR_CONFIG_SETTING_CONSTRUCTION_SPEED, STR_CONFIG_SETTING_CONSTRUCTION_SPEED_HELPTEXT, STR_AI_SPEED_VERY_SLOW, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_97, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_VAR (GameSettings, difficulty.competitor_intelligence, SLE_UINT8, SettingFlags({}), 0,       0, 2, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_97, SLV_110,        SC_ADVANCED, 0, false),
SDT_VAR (GameSettings, difficulty.vehicle_breakdowns, SLE_UINT8, SettingFlags({SettingFlag::GuiDropdown}), 1,       0, 2, 1, STR_CONFIG_SETTING_VEHICLE_BREAKDOWNS, STR_CONFIG_SETTING_VEHICLE_BREAKDOWNS_HELPTEXT, STR_DISASTER_NONE, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_97, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_VAR (GameSettings, difficulty.subsidy_multiplier, SLE_UINT8, SettingFlags({SettingFlag::GuiDropdown}), 2,       0, 3, 1, STR_CONFIG_SETTING_SUBSIDY_MULTIPLIER, STR_CONFIG_SETTING_SUBSIDY_MULTIPLIER_HELPTEXT, STR_SUBSIDY_X1_5, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_97, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_VAR (GameSettings, difficulty.subsidy_duration, SLE_UINT16, SettingFlags({SettingFlag::GuiZeroIsSpecial}), 1,       0, 5000, 1, STR_CONFIG_SETTING_SUBSIDY_DURATION, STR_CONFIG_SETTING_SUBSIDY_DURATION_HELPTEXT, STR_CONFIG_SETTING_SUBSIDY_DURATION_VALUE, nullptr, nullptr, nullptr, SettingHelpWallclock, nullptr, nullptr, nullptr, SLV_CUSTOM_SUBSIDY_DURATION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_VAR (GameSettings, difficulty.construction_cost, SLE_UINT8, SettingFlags({SettingFlag::NewgameOnly, SettingFlag::SceneditToo, SettingFlag::GuiDropdown}), 0,       0, 2, 1, STR_CONFIG_SETTING_CONSTRUCTION_COSTS, STR_CONFIG_SETTING_CONSTRUCTION_COSTS_HELPTEXT, STR_CONFIG_SETTING_CONSTRUCTION_COSTS_LOW, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_97, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_VAR (GameSettings, difficulty.terrain_type, SLE_UINT8, SettingFlags({SettingFlag::GuiDropdown, SettingFlag::NewgameOnly}), 1,       0, 5, 1, STR_CONFIG_SETTING_TERRAIN_TYPE, STR_CONFIG_SETTING_TERRAIN_TYPE_HELPTEXT, STR_TERRAIN_TYPE_VERY_FLAT, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_97, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_VAR (GameSettings, difficulty.quantity_sea_lakes, SLE_UINT8, SettingFlags({SettingFlag::NewgameOnly}), 0,       0, 4, 1, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_SEA_LEVEL_VERY_LOW, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_97, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_BOOL(GameSettings, difficulty.economy,        SettingFlags({}), false,                              STR_CONFIG_SETTING_RECESSIONS, STR_CONFIG_SETTING_RECESSIONS_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_97, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_BOOL(GameSettings, difficulty.line_reverse_mode,        SettingFlags({}), false,                              STR_CONFIG_SETTING_TRAIN_REVERSING, STR_CONFIG_SETTING_TRAIN_REVERSING_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_97, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_BOOL(GameSettings, difficulty.disasters,        SettingFlags({}), false,                              STR_CONFIG_SETTING_DISASTERS, STR_CONFIG_SETTING_DISASTERS_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_97, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_VAR (GameSettings, difficulty.town_council_tolerance, SLE_UINT8, SettingFlags({SettingFlag::GuiDropdown}), 0,       0, 3, 1, STR_CONFIG_SETTING_CITY_APPROVAL, STR_CONFIG_SETTING_CITY_APPROVAL_HELPTEXT, STR_CITY_APPROVAL_LENIENT, nullptr, DifficultyNoiseChange, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_97, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTG_VAR("diff_level",              SLE_UINT8, SettingFlags({SettingFlag::NotInConfig}), _old_diff_level, 3, 0, 3, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_97, SLV_178,        SC_BASIC, 0, false),
SDT_BOOL(GameSettings, difficulty.infinite_money,        SettingFlags({SettingFlag::Sandbox}), false,                              STR_CONFIG_SETTING_INFINITE_MONEY, STR_CONFIG_SETTING_INFINITE_MONEY_HELPTEXT, STR_NULL, nullptr, [](auto) { SetWindowDirty(WC_STATUS_BAR, 0); }, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
};
static void TownFoundingChanged(int32_t new_value);
static void ChangeTimekeepingUnits(int32_t new_value);
static void ChangeMinutesPerYear(int32_t new_value);
static std::tuple<int32_t, uint32_t> GetMinutesPerYearRange(const IntSettingDesc &sd);
static constexpr std::initializer_list<std::string_view> _place_houses{"forbidden"sv, "allowed"sv, "fully constructed"sv};
static const SettingVariant _economy_settings_table[] = {
SDT_VAR (GameSettings, economy.town_layout, SLE_UINT8, SettingFlags({SettingFlag::GuiDropdown}), TL_ORIGINAL,       TL_BEGIN, NUM_TLS - 1, 1, STR_CONFIG_SETTING_TOWN_LAYOUT, STR_CONFIG_SETTING_TOWN_LAYOUT_HELPTEXT, STR_CONFIG_SETTING_TOWN_LAYOUT_DEFAULT, nullptr, TownFoundingChanged, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_59, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_BOOL(GameSettings, economy.allow_town_roads,        SettingFlags({SettingFlag::NoNetwork}), true,                              STR_CONFIG_SETTING_ALLOW_TOWN_ROADS, STR_CONFIG_SETTING_ALLOW_TOWN_ROADS_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_113, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_VAR (GameSettings, economy.found_town, SLE_UINT8, SettingFlags({SettingFlag::GuiDropdown}), TF_FORBIDDEN,       TF_BEGIN, TF_END - 1, 1, STR_CONFIG_SETTING_TOWN_FOUNDING, STR_CONFIG_SETTING_TOWN_FOUNDING_HELPTEXT, STR_CONFIG_SETTING_TOWN_FOUNDING_FORBIDDEN, nullptr, TownFoundingChanged, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_128, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_VAR (GameSettings, economy.place_houses, SLE_UINT8, SettingFlags({SettingFlag::GuiDropdown}), PH_FORBIDDEN,       PH_FORBIDDEN, PH_ALLOWED_CONSTRUCTED, 0, STR_CONFIG_SETTING_HOUSE_PLACER, STR_CONFIG_SETTING_HOUSE_PLACER_HELPTEXT, STR_CONFIG_SETTING_HOUSE_PLACER_FORBIDDEN, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_BOOL(GameSettings, economy.allow_town_level_crossings,        SettingFlags({SettingFlag::NoNetwork}), true,                              STR_CONFIG_SETTING_ALLOW_TOWN_LEVEL_CROSSINGS, STR_CONFIG_SETTING_ALLOW_TOWN_LEVEL_CROSSINGS_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_143, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_VAR (GameSettings, economy.town_cargogen_mode, SLE_UINT8, SettingFlags({SettingFlag::GuiDropdown}), TCGM_BITCOUNT,       TCGM_BEGIN, TCGM_END - 1, 1, STR_CONFIG_SETTING_TOWN_CARGOGENMODE, STR_CONFIG_SETTING_TOWN_CARGOGENMODE_HELPTEXT, STR_CONFIG_SETTING_TOWN_CARGOGENMODE_ORIGINAL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_TOWN_CARGOGEN, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_BOOL(GameSettings, economy.station_noise_level,        SettingFlags({SettingFlag::NoNetwork}), false,                              STR_CONFIG_SETTING_NOISE_LEVEL, STR_CONFIG_SETTING_NOISE_LEVEL_HELPTEXT, STR_NULL, nullptr, [](auto new_value) { InvalidateWindowClassesData(WC_TOWN_VIEW, new_value); }, nullptr, nullptr, nullptr, nullptr, SLV_96, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_BOOL(GameSettings, economy.inflation,        SettingFlags({SettingFlag::NoNetwork}), false,                              STR_CONFIG_SETTING_INFLATION, STR_CONFIG_SETTING_INFLATION_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_BOOL(GameSettings, economy.multiple_industry_per_town,        SettingFlags({}), false,                              STR_CONFIG_SETTING_MULTIPINDTOWN, STR_CONFIG_SETTING_MULTIPINDTOWN_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_BOOL(GameSettings, economy.bribe,        SettingFlags({}), true,                              STR_CONFIG_SETTING_BRIBE, STR_CONFIG_SETTING_BRIBE_HELPTEXT, STR_NULL, nullptr, [](auto) { InvalidateWindowClassesData(WC_TOWN_AUTHORITY); }, nullptr, SettingHelpWallclock, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_BOOL(GameSettings, economy.exclusive_rights,        SettingFlags({}), true,                              STR_CONFIG_SETTING_ALLOW_EXCLUSIVE, STR_CONFIG_SETTING_ALLOW_EXCLUSIVE_HELPTEXT, STR_NULL, nullptr, [](auto) { InvalidateWindowClassesData(WC_TOWN_AUTHORITY); }, nullptr, SettingHelpWallclock, nullptr, nullptr, SLV_79, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_BOOL(GameSettings, economy.fund_buildings,        SettingFlags({}), true,                              STR_CONFIG_SETTING_ALLOW_FUND_BUILDINGS, STR_CONFIG_SETTING_ALLOW_FUND_BUILDINGS_HELPTEXT, STR_NULL, nullptr, [](auto) { InvalidateWindowClassesData(WC_TOWN_AUTHORITY); }, nullptr, nullptr, nullptr, nullptr, SLV_165, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_BOOL(GameSettings, economy.fund_roads,        SettingFlags({}), true,                              STR_CONFIG_SETTING_ALLOW_FUND_ROAD, STR_CONFIG_SETTING_ALLOW_FUND_ROAD_HELPTEXT, STR_NULL, nullptr, [](auto) { InvalidateWindowClassesData(WC_TOWN_AUTHORITY); }, nullptr, nullptr, nullptr, nullptr, SLV_160, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_BOOL(GameSettings, economy.give_money,        SettingFlags({}), true,                              STR_CONFIG_SETTING_ALLOW_GIVE_MONEY, STR_CONFIG_SETTING_ALLOW_GIVE_MONEY_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_79, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_VAR (GameSettings, economy.type, SLE_UINT8, SettingFlags({SettingFlag::GuiDropdown}), ET_SMOOTH,       ET_BEGIN, ET_END - 1, 0, STR_CONFIG_SETTING_ECONOMY_TYPE, STR_CONFIG_SETTING_ECONOMY_TYPE_HELPTEXT, STR_CONFIG_SETTING_ECONOMY_TYPE_ORIGINAL, nullptr, [](auto) { InvalidateWindowClassesData(WC_INDUSTRY_VIEW); }, nullptr, SettingHelpWallclock, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_VAR (GameSettings, economy.feeder_payment_share, SLE_UINT8, SettingFlags({}), 75,       0, 100, 0, STR_CONFIG_SETTING_FEEDER_PAYMENT_SHARE, STR_CONFIG_SETTING_FEEDER_PAYMENT_SHARE_HELPTEXT, STR_CONFIG_SETTING_PERCENTAGE, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_134, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR (GameSettings, economy.town_growth_rate, SLE_UINT8, SettingFlags({SettingFlag::GuiDropdown}), 2,       0, 4, 0, STR_CONFIG_SETTING_TOWN_GROWTH, STR_CONFIG_SETTING_TOWN_GROWTH_HELPTEXT, STR_CONFIG_SETTING_TOWN_GROWTH_NONE, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_54, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_VAR (GameSettings, economy.larger_towns, SLE_UINT8, SettingFlags({SettingFlag::GuiZeroIsSpecial}), 4,       0, 255, 1, STR_CONFIG_SETTING_LARGER_TOWNS, STR_CONFIG_SETTING_LARGER_TOWNS_HELPTEXT, STR_CONFIG_SETTING_LARGER_TOWNS_VALUE, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_54, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_VAR (GameSettings, economy.initial_city_size, SLE_UINT8, SettingFlags({}), 2,       1, 10, 1, STR_CONFIG_SETTING_CITY_SIZE_MULTIPLIER, STR_CONFIG_SETTING_CITY_SIZE_MULTIPLIER_HELPTEXT, STR_JUST_COMMA, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_56, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_BOOL(GameSettings, economy.mod_road_rebuild,        SettingFlags({}), true,                              STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_77, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR (GameSettings, economy.dist_local_authority, SLE_UINT8, SettingFlags({}), 20,       5, 60, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR (GameSettings, economy.town_noise_population[0], SLE_UINT16, SettingFlags({}), 800,       200, 65535, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_96, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR (GameSettings, economy.town_noise_population[1], SLE_UINT16, SettingFlags({}), 2000,       400, 65535, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_96, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR (GameSettings, economy.town_noise_population[2], SLE_UINT16, SettingFlags({}), 4000,       800, 65535, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_96, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR (GameSettings, economy.town_noise_population[3], SLE_UINT16, SettingFlags({}), 400,       100, 65535, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_BOOL(GameSettings, economy.infrastructure_maintenance,        SettingFlags({}), false,                              STR_CONFIG_SETTING_INFRASTRUCTURE_MAINTENANCE, STR_CONFIG_SETTING_INFRASTRUCTURE_MAINTENANCE_HELPTEXT, STR_NULL, nullptr, [](auto) { InvalidateWindowClassesData(WC_COMPANY_INFRASTRUCTURE); }, nullptr, nullptr, nullptr, nullptr, SLV_166, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_VAR (GameSettings, economy.timekeeping_units, SLE_UINT8, SettingFlags({SettingFlag::GuiDropdown, SettingFlag::NewgameOnly, SettingFlag::SceneditToo}), TKU_CALENDAR,       TKU_CALENDAR, TKU_WALLCLOCK, 0, STR_CONFIG_SETTING_TIMEKEEPING_UNITS, STR_CONFIG_SETTING_TIMEKEEPING_UNITS_HELPTEXT, STR_CONFIG_SETTING_TIMEKEEPING_UNITS_CALENDAR, nullptr, ChangeTimekeepingUnits, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_VAR (GameSettings, economy.minutes_per_calendar_year, SLE_UINT16, SettingFlags({SettingFlag::GuiZeroIsSpecial, SettingFlag::NoNetwork}), CalendarTime::DEF_MINUTES_PER_YEAR,       CalendarTime::FROZEN_MINUTES_PER_YEAR, CalendarTime::MAX_MINUTES_PER_YEAR, 1, STR_CONFIG_SETTING_MINUTES_PER_YEAR, STR_CONFIG_SETTING_MINUTES_PER_YEAR_HELPTEXT, STR_CONFIG_SETTING_MINUTES_PER_YEAR_VALUE, [](auto) { return _game_mode == GM_MENU || _settings_game.economy.timekeeping_units == 1; }, ChangeMinutesPerYear, nullptr, nullptr, nullptr, nullptr, GetMinutesPerYearRange, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_VAR (GameSettings, economy.town_cargo_scale, SLE_UINT16, SettingFlags({SettingFlag::NoNetwork}), DEF_CARGO_SCALE,       MIN_CARGO_SCALE, MAX_CARGO_SCALE, 1, STR_CONFIG_SETTING_TOWN_CARGO_SCALE, STR_CONFIG_SETTING_TOWN_CARGO_SCALE_HELPTEXT, STR_CONFIG_SETTING_CARGO_SCALE_VALUE, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_VAR (GameSettings, economy.industry_cargo_scale, SLE_UINT16, SettingFlags({SettingFlag::NoNetwork}), DEF_CARGO_SCALE,       MIN_CARGO_SCALE, MAX_CARGO_SCALE, 1, STR_CONFIG_SETTING_INDUSTRY_CARGO_SCALE, STR_CONFIG_SETTING_INDUSTRY_CARGO_SCALE_HELPTEXT, STR_CONFIG_SETTING_CARGO_SCALE_VALUE, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
};
static constexpr std::initializer_list<std::string_view> _roadsides{"left"sv, "right"sv};
static void StationSpreadChanged(int32_t new_value);
static void UpdateConsists(int32_t new_value);
static void TrainAccelerationModelChanged(int32_t new_value);
static void RoadVehAccelerationModelChanged(int32_t new_value);
static void TrainSlopeSteepnessChanged(int32_t new_value);
static void RoadVehSlopeSteepnessChanged(int32_t new_value);
static bool CheckRoadSide(int32_t &new_value);
static bool CheckDynamicEngines(int32_t &new_value);
static void StationCatchmentChanged(int32_t new_value);
static void MaxVehiclesChanged(int32_t new_value);
static const SettingVariant _game_settings_table[] = {
SDT_BOOL(GameSettings, order.no_servicing_if_no_breakdowns,        SettingFlags({}), true,                              STR_CONFIG_SETTING_NOSERVICE, STR_CONFIG_SETTING_NOSERVICE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_BOOL(GameSettings, order.improved_load,        SettingFlags({SettingFlag::NoNetwork}), true,                              STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_BOOL(GameSettings, order.selectgoods,        SettingFlags({}), true,                              STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_BOOL(GameSettings, order.serviceathelipad,        SettingFlags({}), true,                              STR_CONFIG_SETTING_SERVICEATHELIPAD, STR_CONFIG_SETTING_SERVICEATHELIPAD_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_BOOL(GameSettings, order.station_length_loading_penalty,        SettingFlags({}), true,                              STR_CONFIG_SETTING_STATION_LENGTH_LOADING_PENALTY, STR_CONFIG_SETTING_STATION_LENGTH_LOADING_PENALTY_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_BOOL(GameSettings, order.gradual_loading,        SettingFlags({SettingFlag::NoNetwork}), true,                              STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_40, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_BOOL(GameSettings, station.never_expire_airports,        SettingFlags({SettingFlag::NoNetwork}), false,                              STR_CONFIG_SETTING_NEVER_EXPIRE_AIRPORTS, STR_CONFIG_SETTING_NEVER_EXPIRE_AIRPORTS_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_VAR(GameSettings, station.station_spread, SLE_UINT8, SettingFlags({}), 12,       4, 64, 0, STR_CONFIG_SETTING_STATION_SPREAD, STR_CONFIG_SETTING_STATION_SPREAD_HELPTEXT, STR_CONFIG_SETTING_TILE_LENGTH, nullptr, StationSpreadChanged, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_BOOL(GameSettings, station.modified_catchment,        SettingFlags({}), true,                              STR_CONFIG_SETTING_CATCHMENT, STR_CONFIG_SETTING_CATCHMENT_HELPTEXT, STR_NULL, nullptr, StationCatchmentChanged, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_BOOL(GameSettings, station.serve_neutral_industries,        SettingFlags({}), true,                              STR_CONFIG_SETTING_SERVE_NEUTRAL_INDUSTRIES, STR_CONFIG_SETTING_SERVE_NEUTRAL_INDUSTRIES_HELPTEXT, STR_NULL, nullptr, StationCatchmentChanged, nullptr, nullptr, nullptr, nullptr, SLV_SERVE_NEUTRAL_INDUSTRIES, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_BOOL(GameSettings, station.distant_join_stations,        SettingFlags({}), true,                              STR_CONFIG_SETTING_DISTANT_JOIN_STATIONS, STR_CONFIG_SETTING_DISTANT_JOIN_STATIONS_HELPTEXT, STR_NULL, nullptr, [](auto) { CloseWindowById(WC_SELECT_STATION, 0); }, nullptr, nullptr, nullptr, nullptr, SLV_106, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_OMANY(GameSettings, vehicle.road_side, SLE_UINT8, SettingFlags({SettingFlag::GuiDropdown, SettingFlag::NoNetwork}), 1,             1, _roadsides,     STR_CONFIG_SETTING_ROAD_SIDE, STR_CONFIG_SETTING_ROAD_SIDE_HELPTEXT, STR_CONFIG_SETTING_ROAD_SIDE_LEFT, CheckRoadSide, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_97, SL_MAX_VERSION, nullptr, SC_BASIC, 0, false),
SDT_VAR(GameSettings, vehicle.train_acceleration_model, SLE_UINT8, SettingFlags({SettingFlag::GuiDropdown}), 1,       0, 1, 1, STR_CONFIG_SETTING_TRAIN_ACCELERATION_MODEL, STR_CONFIG_SETTING_TRAIN_ACCELERATION_MODEL_HELPTEXT, STR_CONFIG_SETTING_ORIGINAL, nullptr, TrainAccelerationModelChanged, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_VAR(GameSettings, vehicle.roadveh_acceleration_model, SLE_UINT8, SettingFlags({SettingFlag::GuiDropdown}), 1,       0, 1, 1, STR_CONFIG_SETTING_ROAD_VEHICLE_ACCELERATION_MODEL, STR_CONFIG_SETTING_ROAD_VEHICLE_ACCELERATION_MODEL_HELPTEXT, STR_CONFIG_SETTING_ORIGINAL, nullptr, RoadVehAccelerationModelChanged, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_139, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_VAR(GameSettings, vehicle.train_slope_steepness, SLE_UINT8, SettingFlags({}), 3,       0, 10, 1, STR_CONFIG_SETTING_TRAIN_SLOPE_STEEPNESS, STR_CONFIG_SETTING_TRAIN_SLOPE_STEEPNESS_HELPTEXT, STR_CONFIG_SETTING_PERCENTAGE, nullptr, TrainSlopeSteepnessChanged, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_133, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, vehicle.roadveh_slope_steepness, SLE_UINT8, SettingFlags({}), 7,       0, 10, 1, STR_CONFIG_SETTING_ROAD_VEHICLE_SLOPE_STEEPNESS, STR_CONFIG_SETTING_ROAD_VEHICLE_SLOPE_STEEPNESS_HELPTEXT, STR_CONFIG_SETTING_PERCENTAGE, nullptr, RoadVehSlopeSteepnessChanged, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_139, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, vehicle.max_train_length, SLE_UINT8, SettingFlags({}), 7,       1, 64, 1, STR_CONFIG_SETTING_TRAIN_LENGTH, STR_CONFIG_SETTING_TRAIN_LENGTH_HELPTEXT, STR_CONFIG_SETTING_TILE_LENGTH, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_159, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_VAR(GameSettings, vehicle.smoke_amount, SLE_UINT8, SettingFlags({SettingFlag::GuiDropdown}), 2,       0, 2, 0, STR_CONFIG_SETTING_SMOKE_AMOUNT, STR_CONFIG_SETTING_SMOKE_AMOUNT_HELPTEXT, STR_CONFIG_SETTING_NONE, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_145, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_BOOL(GameSettings, vehicle.never_expire_vehicles,        SettingFlags({SettingFlag::NoNetwork}), false,                              STR_CONFIG_SETTING_NEVER_EXPIRE_VEHICLES, STR_CONFIG_SETTING_NEVER_EXPIRE_VEHICLES_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_VAR(GameSettings, vehicle.max_trains, SLE_UINT16, SettingFlags({}), 500,       0, 5000, 0, STR_CONFIG_SETTING_MAX_TRAINS, STR_CONFIG_SETTING_MAX_TRAINS_HELPTEXT, STR_JUST_COMMA, nullptr, MaxVehiclesChanged, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_VAR(GameSettings, vehicle.max_roadveh, SLE_UINT16, SettingFlags({}), 500,       0, 5000, 0, STR_CONFIG_SETTING_MAX_ROAD_VEHICLES, STR_CONFIG_SETTING_MAX_ROAD_VEHICLES_HELPTEXT, STR_JUST_COMMA, nullptr, MaxVehiclesChanged, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_VAR(GameSettings, vehicle.max_aircraft, SLE_UINT16, SettingFlags({}), 200,       0, 5000, 0, STR_CONFIG_SETTING_MAX_AIRCRAFT, STR_CONFIG_SETTING_MAX_AIRCRAFT_HELPTEXT, STR_JUST_COMMA, nullptr, MaxVehiclesChanged, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_VAR(GameSettings, vehicle.max_ships, SLE_UINT16, SettingFlags({}), 300,       0, 5000, 0, STR_CONFIG_SETTING_MAX_SHIPS, STR_CONFIG_SETTING_MAX_SHIPS_HELPTEXT, STR_JUST_COMMA, nullptr, MaxVehiclesChanged, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_BOOL(GameSettings, vehicle.wagon_speed_limits,        SettingFlags({SettingFlag::NoNetwork}), true,                              STR_CONFIG_SETTING_WAGONSPEEDLIMITS, STR_CONFIG_SETTING_WAGONSPEEDLIMITS_HELPTEXT, STR_NULL, nullptr, UpdateConsists, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_BOOL(GameSettings, vehicle.disable_elrails,        SettingFlags({SettingFlag::NoNetwork}), false,                              STR_CONFIG_SETTING_DISABLE_ELRAILS, STR_CONFIG_SETTING_DISABLE_ELRAILS_HELPTEXT, STR_NULL, nullptr, SettingsDisableElrail, nullptr, nullptr, nullptr, nullptr, SLV_38, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, vehicle.freight_trains, SLE_UINT8, SettingFlags({SettingFlag::NoNetwork}), 1,       1, 255, 1, STR_CONFIG_SETTING_FREIGHT_TRAINS, STR_CONFIG_SETTING_FREIGHT_TRAINS_HELPTEXT, STR_JUST_COMMA, nullptr, UpdateConsists, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_39, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_VAR(GameSettings, vehicle.plane_speed, SLE_UINT8, SettingFlags({SettingFlag::NoNetwork}), 4,       1, 4, 0, STR_CONFIG_SETTING_PLANE_SPEED, STR_CONFIG_SETTING_PLANE_SPEED_HELPTEXT, STR_CONFIG_SETTING_PLANE_SPEED_VALUE, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_90, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_BOOL(GameSettings, vehicle.dynamic_engines,        SettingFlags({SettingFlag::NoNetwork}), true,                              STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, CheckDynamicEngines, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_95, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, vehicle.plane_crashes, SLE_UINT8, SettingFlags({SettingFlag::GuiDropdown}), 2,       0, 2, 1, STR_CONFIG_SETTING_PLANE_CRASHES, STR_CONFIG_SETTING_PLANE_CRASHES_HELPTEXT, STR_CONFIG_SETTING_PLANE_CRASHES_NONE, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_138, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_VAR(GameSettings, vehicle.extend_vehicle_life, SLE_UINT8, SettingFlags({}), 0,       0, 100, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDTG_BOOL("vehicle.servint_ispercent",                     SettingFlags({SettingFlag::NoNetwork}), _old_vds.servint_ispercent, false,                        STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SLV_120,        SC_ADVANCED, 0, false),
SDTG_VAR("vehicle.servint_trains",              SLE_UINT16, SettingFlags({SettingFlag::GuiZeroIsSpecial}), _old_vds.servint_trains, 150, 5, 800, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SLV_120,        SC_ADVANCED, 0, false),
SDTG_VAR("vehicle.servint_roadveh",              SLE_UINT16, SettingFlags({SettingFlag::GuiZeroIsSpecial}), _old_vds.servint_roadveh, 150, 5, 800, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SLV_120,        SC_ADVANCED, 0, false),
SDTG_VAR("vehicle.servint_ships",              SLE_UINT16, SettingFlags({SettingFlag::GuiZeroIsSpecial}), _old_vds.servint_ships, 360, 5, 800, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SLV_120,        SC_ADVANCED, 0, false),
SDTG_VAR("vehicle.servint_aircraft",              SLE_UINT16, SettingFlags({SettingFlag::GuiZeroIsSpecial}), _old_vds.servint_aircraft, 150, 5, 800, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SLV_120,        SC_ADVANCED, 0, false),
SDTC_BOOL(              gui.sg_full_load_any,        SettingFlags({}), true,                              STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_22, SLV_93,        SC_ADVANCED, 0, false),
SDTC_BOOL(              gui.sg_new_nonstop,        SettingFlags({}), false,                              STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_22, SLV_93,        SC_ADVANCED, 0, false),
};
static bool CanChangeTimetableMode(int32_t &new_value);
static void v_PositionMainToolbar(int32_t new_value);
static void v_PositionStatusbar(int32_t new_value);
static void RedrawSmallmap(int32_t new_value);
static void UpdateLinkgraphColours(int32_t new_value);
static void InvalidateCompanyLiveryWindow(int32_t new_value);
static void InvalidateNewGRFChangeWindows(int32_t new_value);
static void ZoomMinMaxChanged(int32_t new_value);
static void SpriteZoomMinChanged(int32_t new_value);
static constexpr std::initializer_list<std::string_view> _osk_activation{"disabled"sv, "double"sv, "single"sv, "immediately"sv};
static constexpr std::initializer_list<std::string_view> _savegame_date{"long"sv, "short"sv, "iso"sv};
static constexpr std::initializer_list<std::string_view> _right_click_close{"no"sv, "yes"sv, "except sticky"sv};
static const SettingVariant _gui_settings_table[] = {
SDTC_VAR(              gui.autosave_interval, SLE_UINT32, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), 10,       0, 1440, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_BOOL(              gui.threaded_saves,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), true,                              STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDTC_OMANY(              gui.date_format_in_default_names, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), 2,             2, _savegame_date,     STR_CONFIG_SETTING_DATE_FORMAT_IN_SAVE_NAMES, STR_CONFIG_SETTING_DATE_FORMAT_IN_SAVE_NAMES_HELPTEXT, STR_CONFIG_SETTING_DATE_FORMAT_IN_SAVE_NAMES_LONG, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_BOOL(              gui.show_finances,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), true,                              STR_CONFIG_SETTING_SHOWFINANCES, STR_CONFIG_SETTING_SHOWFINANCES_HELPTEXT, STR_NULL, nullptr, nullptr, SettingTitleWallclock, SettingHelpWallclock, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDTC_VAR(              gui.auto_scrolling, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), 0,       0, 3, 0, STR_CONFIG_SETTING_AUTOSCROLL, STR_CONFIG_SETTING_AUTOSCROLL_HELPTEXT, STR_CONFIG_SETTING_AUTOSCROLL_DISABLED, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
#ifdef UNIX
SDTC_VAR(              gui.scroll_mode, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), VSM_MAP_RMB,       VSM_VIEWPORT_RMB_FIXED, VSM_MAP_LMB, 0, STR_CONFIG_SETTING_SCROLLMODE, STR_CONFIG_SETTING_SCROLLMODE_HELPTEXT, STR_CONFIG_SETTING_SCROLLMODE_DEFAULT, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
#endif
#ifndef UNIX
SDTC_VAR(              gui.scroll_mode, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), VSM_VIEWPORT_RMB_FIXED,       VSM_VIEWPORT_RMB_FIXED, VSM_MAP_LMB, 0, STR_CONFIG_SETTING_SCROLLMODE, STR_CONFIG_SETTING_SCROLLMODE_HELPTEXT, STR_CONFIG_SETTING_SCROLLMODE_DEFAULT, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
#endif
SDTC_BOOL(              gui.smooth_scroll,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), true,                              STR_CONFIG_SETTING_SMOOTH_SCROLLING, STR_CONFIG_SETTING_SMOOTH_SCROLLING_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_OMANY(              gui.right_click_wnd_close, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), RCC_NO,             RCC_YES_EXCEPT_STICKY, _right_click_close,     STR_CONFIG_SETTING_RIGHT_MOUSE_WND_CLOSE, STR_CONFIG_SETTING_RIGHT_MOUSE_WND_CLOSE_HELPTEXT, STR_CONFIG_SETTING_RIGHT_MOUSE_WND_CLOSE_NO, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDTC_BOOL(              gui.toolbar_dropdown_autoselect,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), false,                              STR_CONFIG_SETTING_TOOLBAR_DROPDOWN_AUTOSELECT, STR_CONFIG_SETTING_TOOLBAR_DROPDOWN_AUTOSELECT_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
#ifdef __APPLE__
SDTC_VAR(              gui.right_mouse_btn_emulation, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), 0,       0, 2, 0, STR_CONFIG_SETTING_RIGHT_MOUSE_BTN_EMU, STR_CONFIG_SETTING_RIGHT_MOUSE_BTN_EMU_HELPTEXT, STR_CONFIG_SETTING_RIGHT_MOUSE_BTN_EMU_COMMAND, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
#endif
SDTC_BOOL(              gui.measure_tooltip,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), true,                              STR_CONFIG_SETTING_MEASURE_TOOLTIP, STR_CONFIG_SETTING_MEASURE_TOOLTIP_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDTC_VAR(              gui.errmsg_duration, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), 5,       0, 20, 0, STR_CONFIG_SETTING_ERRMSG_DURATION, STR_CONFIG_SETTING_ERRMSG_DURATION_HELPTEXT, STR_CONFIG_SETTING_SECONDS_VALUE, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_VAR(              gui.hover_delay_ms, SLE_UINT16, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiZeroIsSpecial}), 250,       50, 6000, 50, STR_CONFIG_SETTING_HOVER_DELAY, STR_CONFIG_SETTING_HOVER_DELAY_HELPTEXT, STR_CONFIG_SETTING_HOVER_DELAY_VALUE, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_OMANY(              gui.osk_activation, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), 1,             3, _osk_activation,     STR_CONFIG_SETTING_OSK_ACTIVATION, STR_CONFIG_SETTING_OSK_ACTIVATION_HELPTEXT, STR_CONFIG_SETTING_OSK_ACTIVATION_DISABLED, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDTC_VAR(              gui.toolbar_pos, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), 1,       0, 2, 0, STR_CONFIG_SETTING_TOOLBAR_POS, STR_CONFIG_SETTING_TOOLBAR_POS_HELPTEXT, STR_CONFIG_SETTING_HORIZONTAL_POS_LEFT, nullptr, v_PositionMainToolbar, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDTC_VAR(              gui.statusbar_pos, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), 1,       0, 2, 0, STR_CONFIG_SETTING_STATUSBAR_POS, STR_CONFIG_SETTING_STATUSBAR_POS_HELPTEXT, STR_CONFIG_SETTING_HORIZONTAL_POS_LEFT, nullptr, v_PositionStatusbar, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDTC_VAR(              gui.window_snap_radius, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiZeroIsSpecial}), 10,       1, 32, 0, STR_CONFIG_SETTING_SNAP_RADIUS, STR_CONFIG_SETTING_SNAP_RADIUS_HELPTEXT, STR_CONFIG_SETTING_SNAP_RADIUS_VALUE, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDTC_VAR(              gui.window_soft_limit, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiZeroIsSpecial}), 20,       5, 255, 1, STR_CONFIG_SETTING_SOFT_LIMIT, STR_CONFIG_SETTING_SOFT_LIMIT_HELPTEXT, STR_CONFIG_SETTING_SOFT_LIMIT_VALUE, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDTC_VAR(              gui.zoom_min, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), to_underlying(ZoomLevel::Min),       to_underlying(ZoomLevel::Min), to_underlying(ZoomLevel::Normal), 0, STR_CONFIG_SETTING_ZOOM_MIN, STR_CONFIG_SETTING_ZOOM_MIN_HELPTEXT, STR_CONFIG_SETTING_ZOOM_LVL_MIN, nullptr, ZoomMinMaxChanged, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, true),
SDTC_VAR(              gui.zoom_max, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), to_underlying(ZoomLevel::Max),       to_underlying(ZoomLevel::Out2x), to_underlying(ZoomLevel::Max), 0, STR_CONFIG_SETTING_ZOOM_MAX, STR_CONFIG_SETTING_ZOOM_MAX_HELPTEXT, STR_CONFIG_SETTING_ZOOM_LVL_OUT_2X, nullptr, ZoomMinMaxChanged, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, true),
SDTC_VAR(              gui.sprite_zoom_min, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), to_underlying(ZoomLevel::Min),       to_underlying(ZoomLevel::Min), to_underlying(ZoomLevel::Normal), 0, STR_CONFIG_SETTING_SPRITE_ZOOM_MIN, STR_CONFIG_SETTING_SPRITE_ZOOM_MIN_HELPTEXT, STR_CONFIG_SETTING_SPRITE_ZOOM_LVL_MIN, nullptr, SpriteZoomMinChanged, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_BOOL(              gui.population_in_label,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), true,                              STR_CONFIG_SETTING_POPULATION_IN_LABEL, STR_CONFIG_SETTING_POPULATION_IN_LABEL_HELPTEXT, STR_NULL, nullptr, [](auto) { UpdateAllTownVirtCoords(); }, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_BOOL(              gui.link_terraform_toolbar,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), false,                              STR_CONFIG_SETTING_LINK_TERRAFORM_TOOLBAR, STR_CONFIG_SETTING_LINK_TERRAFORM_TOOLBAR_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_VAR(              gui.smallmap_land_colour, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), 0,       0, 2, 0, STR_CONFIG_SETTING_SMALLMAP_LAND_COLOUR, STR_CONFIG_SETTING_SMALLMAP_LAND_COLOUR_HELPTEXT, STR_CONFIG_SETTING_SMALLMAP_LAND_COLOUR_GREEN, nullptr, RedrawSmallmap, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_VAR(              gui.linkgraph_colours, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), 0,       0, 3, 0, STR_CONFIG_SETTING_LINKGRAPH_COLOURS, STR_CONFIG_SETTING_LINKGRAPH_COLOURS_HELPTEXT, STR_CONFIG_SETTING_LINKGRAPH_COLOURS_GREEN_TO_RED, nullptr, UpdateLinkgraphColours, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDTC_VAR(              gui.liveries, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), LIT_ALL,       LIT_NONE, LIT_ALL, 0, STR_CONFIG_SETTING_LIVERIES, STR_CONFIG_SETTING_LIVERIES_HELPTEXT, STR_CONFIG_SETTING_LIVERIES_NONE, nullptr, InvalidateCompanyLiveryWindow, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_VAR(              gui.starting_colour, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), COLOUR_END,       0, COLOUR_END, 0, STR_CONFIG_SETTING_COMPANY_STARTING_COLOUR, STR_CONFIG_SETTING_COMPANY_STARTING_COLOUR_HELPTEXT, STR_COLOUR_DARK_BLUE, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_VAR(              gui.starting_colour_secondary, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), COLOUR_END,       0, COLOUR_END, 0, STR_CONFIG_SETTING_COMPANY_STARTING_COLOUR_SECONDARY, STR_CONFIG_SETTING_COMPANY_STARTING_COLOUR_SECONDARY_HELPTEXT, STR_COLOUR_SECONDARY_DARK_BLUE, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_BOOL(              gui.auto_remove_signals,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), false,                              STR_CONFIG_SETTING_AUTO_REMOVE_SIGNALS, STR_CONFIG_SETTING_AUTO_REMOVE_SIGNALS_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_BOOL(              gui.prefer_teamchat,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), false,                              STR_CONFIG_SETTING_PREFER_TEAMCHAT, STR_CONFIG_SETTING_PREFER_TEAMCHAT_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDTC_VAR(              gui.scrollwheel_scrolling, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), SWS_ZOOM_MAP,       SWS_ZOOM_MAP, SWS_OFF, 0, STR_CONFIG_SETTING_SCROLLWHEEL_SCROLLING, STR_CONFIG_SETTING_SCROLLWHEEL_SCROLLING_HELPTEXT, STR_CONFIG_SETTING_SCROLLWHEEL_ZOOM, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDTC_VAR(              gui.scrollwheel_multiplier, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), 5,       1, 15, 1, STR_CONFIG_SETTING_SCROLLWHEEL_MULTIPLIER, STR_CONFIG_SETTING_SCROLLWHEEL_MULTIPLIER_HELPTEXT, STR_JUST_COMMA, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDTC_BOOL(              gui.pause_on_newgame,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), false,                              STR_CONFIG_SETTING_PAUSE_ON_NEW_GAME, STR_CONFIG_SETTING_PAUSE_ON_NEW_GAME_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDTC_VAR(              gui.advanced_vehicle_list, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), 1,       0, 2, 0, STR_CONFIG_SETTING_ADVANCED_VEHICLE_LISTS, STR_CONFIG_SETTING_ADVANCED_VEHICLE_LISTS_HELPTEXT, STR_CONFIG_SETTING_COMPANIES_OFF, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_VAR(              gui.timetable_mode, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), 0,       0, 2, 0, STR_CONFIG_SETTING_TIMETABLE_MODE, STR_CONFIG_SETTING_TIMETABLE_MODE_HELPTEXT, STR_CONFIG_SETTING_TIMETABLE_MODE_DAYS, CanChangeTimetableMode, [](auto) { InvalidateWindowClassesData(WC_VEHICLE_TIMETABLE, VIWD_MODIFY_ORDERS); }, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_BOOL(              gui.timetable_arrival_departure,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), true,                              STR_CONFIG_SETTING_TIMETABLE_SHOW_ARRIVAL_DEPARTURE, STR_CONFIG_SETTING_TIMETABLE_SHOW_ARRIVAL_DEPARTURE_HELPTEXT, STR_NULL, nullptr, [](auto) { InvalidateWindowClassesData(WC_VEHICLE_TIMETABLE, VIWD_MODIFY_ORDERS); }, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_BOOL(              gui.quick_goto,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), true,                              STR_CONFIG_SETTING_QUICKGOTO, STR_CONFIG_SETTING_QUICKGOTO_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDTC_VAR(              gui.loading_indicators, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), 1,       0, 2, 0, STR_CONFIG_SETTING_LOADING_INDICATORS, STR_CONFIG_SETTING_LOADING_INDICATORS_HELPTEXT, STR_CONFIG_SETTING_COMPANIES_OFF, nullptr, [](auto) { MarkWholeScreenDirty(); }, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDTC_VAR(              gui.default_rail_type, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), 0,       0, 2, 0, STR_CONFIG_SETTING_DEFAULT_RAIL_TYPE, STR_CONFIG_SETTING_DEFAULT_RAIL_TYPE_HELPTEXT, STR_CONFIG_SETTING_DEFAULT_RAIL_TYPE_FIRST, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDTC_VAR(              gui.signal_gui_mode, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), 0,       0, 1, 0, STR_CONFIG_SETTING_SIGNAL_GUI_MODE, STR_CONFIG_SETTING_SIGNAL_GUI_MODE_HELPTEXT, STR_CONFIG_SETTING_SIGNAL_GUI_MODE_PATH, nullptr, [](auto) { CloseWindowByClass(WC_BUILD_SIGNAL); }, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_VAR(              gui.default_signal_type, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), 5,       0, 5, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_VAR(              gui.coloured_news_year, SLE_INT32, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), 2000,       CalendarTime::MIN_YEAR, CalendarTime::MAX_YEAR, 1, STR_CONFIG_SETTING_COLOURED_NEWS_YEAR, STR_CONFIG_SETTING_COLOURED_NEWS_YEAR_HELPTEXT, STR_JUST_INT, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDTC_VAR(              gui.cycle_signal_types, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), SIGNAL_CYCLE_GROUP,       SIGNAL_CYCLE_GROUP, SIGNAL_CYCLE_ALL, 1, STR_CONFIG_SETTING_CYCLE_SIGNAL_TYPES, STR_CONFIG_SETTING_CYCLE_SIGNAL_TYPES_HELPTEXT, STR_CONFIG_SETTING_CYCLE_SIGNAL_GROUP, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_VAR(              gui.drag_signals_density, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), 4,       1, 20, 0, STR_CONFIG_SETTING_DRAG_SIGNALS_DENSITY, STR_CONFIG_SETTING_DRAG_SIGNALS_DENSITY_HELPTEXT, STR_CONFIG_SETTING_DRAG_SIGNALS_DENSITY_VALUE, nullptr, [](auto) { InvalidateWindowData(WC_BUILD_SIGNAL, 0); }, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDTC_BOOL(              gui.drag_signals_fixed_distance,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), false,                              STR_CONFIG_SETTING_DRAG_SIGNALS_FIXED_DISTANCE, STR_CONFIG_SETTING_DRAG_SIGNALS_FIXED_DISTANCE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDTC_VAR(              gui.semaphore_build_before, SLE_INT32, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), 1950,       CalendarTime::MIN_YEAR, CalendarTime::MAX_YEAR, 1, STR_CONFIG_SETTING_SEMAPHORE_BUILD_BEFORE_DATE, STR_CONFIG_SETTING_SEMAPHORE_BUILD_BEFORE_DATE_HELPTEXT, STR_JUST_INT, nullptr, ResetSignalVariant, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_BOOL(              gui.vehicle_income_warn,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), true,                              STR_CONFIG_SETTING_WARN_INCOME_LESS, STR_CONFIG_SETTING_WARN_INCOME_LESS_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, SettingHelpWallclock, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDTC_VAR(              gui.order_review_system, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), 2,       0, 2, 0, STR_CONFIG_SETTING_ORDER_REVIEW, STR_CONFIG_SETTING_ORDER_REVIEW_HELPTEXT, STR_CONFIG_SETTING_ORDER_REVIEW_OFF, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDTC_BOOL(              gui.lost_vehicle_warn,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), true,                              STR_CONFIG_SETTING_WARN_LOST_VEHICLE, STR_CONFIG_SETTING_WARN_LOST_VEHICLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_BOOL(              gui.old_vehicle_warn,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), true,                              STR_CONFIG_SETTING_WARN_OLD_VEHICLE, STR_CONFIG_SETTING_WARN_OLD_VEHICLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_BOOL(              gui.new_nonstop,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), true,                              STR_CONFIG_SETTING_NONSTOP_BY_DEFAULT, STR_CONFIG_SETTING_NONSTOP_BY_DEFAULT_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDTC_VAR(              gui.stop_location, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), 1,       0, 2, 1, STR_CONFIG_SETTING_STOP_LOCATION, STR_CONFIG_SETTING_STOP_LOCATION_HELPTEXT, STR_CONFIG_SETTING_STOP_LOCATION_NEAR_END, nullptr, [](auto) { SetWindowClassesDirty(WC_VEHICLE_ORDERS); }, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDTC_BOOL(              gui.keep_all_autosave,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), false,                              STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_BOOL(              gui.autosave_on_exit,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), false,                              STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDTC_BOOL(              gui.autosave_on_network_disconnect,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), true,                              STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDTC_VAR(              gui.max_num_autosaves, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), 16,       0, 255, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_BOOL(              gui.auto_euro,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), true,                              STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_VAR(              gui.news_message_timeout, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), 2,       1, 255, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_BOOL(              gui.show_track_reservation,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), true,                              STR_CONFIG_SETTING_SHOW_TRACK_RESERVATION, STR_CONFIG_SETTING_SHOW_TRACK_RESERVATION_HELPTEXT, STR_NULL, nullptr, [](auto) { MarkWholeScreenDirty(); }, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDTC_VAR(              gui.station_numtracks, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), 1,       1, 7, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_VAR(              gui.station_platlength, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), 5,       1, 7, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDTC_BOOL(              gui.station_dragdrop,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), true,                              STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDTC_BOOL(              gui.station_show_coverage,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), false,                              STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDTC_BOOL(              gui.persistent_buildingtools,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), true,                              STR_CONFIG_SETTING_PERSISTENT_BUILDINGTOOLS, STR_CONFIG_SETTING_PERSISTENT_BUILDINGTOOLS_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDTC_VAR(              gui.station_gui_group_order, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), 3,       0, 5, 1, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_VAR(              gui.station_gui_sort_by, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), 0,       0, 3, 1, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_VAR(              gui.station_gui_sort_order, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), 0,       0, 1, 1, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_VAR(              gui.missing_strings_threshold, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), 25,       1, UINT8_MAX, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDTC_VAR(              gui.graph_line_thickness, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), 3,       1, 5, 0, STR_CONFIG_SETTING_GRAPH_LINE_THICKNESS, STR_CONFIG_SETTING_GRAPH_LINE_THICKNESS_HELPTEXT, STR_JUST_COMMA, nullptr, [](auto) { MarkWholeScreenDirty(); }, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_BOOL(              gui.show_newgrf_name,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), false,                              STR_CONFIG_SETTING_SHOW_NEWGRF_NAME, STR_CONFIG_SETTING_SHOW_NEWGRF_NAME_HELPTEXT, STR_NULL, nullptr, [](auto) { MarkWholeScreenDirty(); }, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_BOOL(              gui.show_cargo_in_vehicle_lists,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), false,                              STR_CONFIG_SETTING_SHOW_CARGO_IN_LISTS, STR_CONFIG_SETTING_SHOW_CARGO_IN_LISTS_HELPTEXT, STR_NULL, nullptr, [](auto) { MarkWholeScreenDirty(); }, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
#ifdef DEDICATED
SDTC_BOOL(              gui.show_date_in_logs,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), true,                              STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
#endif
#ifndef DEDICATED
SDTC_BOOL(              gui.show_date_in_logs,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), false,                              STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
#endif
SDTC_VAR(              gui.settings_restriction_mode, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), 0,       0, 2, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_VAR(              gui.developer, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), 1,       0, 2, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, [](auto) { DebugReconsiderSendRemoteMessages(); }, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDTC_BOOL(              gui.newgrf_developer_tools,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), false,                              STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, InvalidateNewGRFChangeWindows, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDTC_BOOL(              gui.ai_developer_tools,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), false,                              STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, [](auto) { InvalidateWindowClassesData(WC_GAME_OPTIONS); InvalidateWindowClassesData(WC_SCRIPT_DEBUG); InvalidateWindowClassesData(WC_SCRIPT_SETTINGS); }, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDTC_BOOL(              gui.scenario_developer,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), false,                              STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, InvalidateNewGRFChangeWindows, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_BOOL(              gui.newgrf_show_old_versions,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), false,                              STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDTC_VAR(              gui.newgrf_default_palette, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), 1,       0, 1, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, UpdateNewGRFConfigPalette, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDTC_VAR(              gui.console_backlog_timeout, SLE_UINT16, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), 100,       10, 65500, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_VAR(              gui.console_backlog_length, SLE_UINT16, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), 100,       10, 65500, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_VAR(              gui.refresh_rate, SLE_UINT16, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), 60,       10, 1000, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, true),
SDTC_VAR(              gui.fast_forward_speed_limit, SLE_UINT16, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiZeroIsSpecial, SettingFlag::NoNetwork}), 2500,       0, 50000, 10, STR_CONFIG_SETTING_FAST_FORWARD_SPEED_LIMIT, STR_CONFIG_SETTING_FAST_FORWARD_SPEED_LIMIT_HELPTEXT, STR_CONFIG_SETTING_FAST_FORWARD_SPEED_LIMIT_VAL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDTC_VAR(              gui.network_chat_box_width_pct, SLE_UINT16, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), 40,       10, 100, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDTC_VAR(              gui.network_chat_box_height, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), 25,       5, 255, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDTC_VAR(              gui.network_chat_timeout, SLE_UINT16, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), 20,       1, 65535, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDTC_BOOL(              gui.scale_bevels,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), true,                              STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, [](auto) { SetupWidgetDimensions(); ReInitAllWindows(true); }, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, true),
};
static const SettingVariant _linkgraph_settings_table[] = {
SDT_VAR(GameSettings, linkgraph.recalc_interval, SLE_UINT16, SettingFlags({}), 8,       4, 90, 1, STR_CONFIG_SETTING_LINKGRAPH_RECALC_INTERVAL, STR_CONFIG_SETTING_LINKGRAPH_RECALC_INTERVAL_HELPTEXT, STR_CONFIG_SETTING_SECONDS_VALUE, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_183, SL_MAX_VERSION,        SC_ADVANCED, offsetof(LinkGraphSettings, recalc_interval), false),
SDT_VAR(GameSettings, linkgraph.recalc_time, SLE_UINT16, SettingFlags({}), 32,       1, 9000, 1, STR_CONFIG_SETTING_LINKGRAPH_RECALC_TIME, STR_CONFIG_SETTING_LINKGRAPH_RECALC_TIME_HELPTEXT, STR_CONFIG_SETTING_SECONDS_VALUE, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_183, SL_MAX_VERSION,        SC_ADVANCED, offsetof(LinkGraphSettings, recalc_time), false),
SDT_VAR(GameSettings, linkgraph.distribution_pax, SLE_UINT8, SettingFlags({SettingFlag::GuiDropdown}), DT_MANUAL,       DT_MIN, DT_MAX, 1, STR_CONFIG_SETTING_DISTRIBUTION_PAX, STR_CONFIG_SETTING_DISTRIBUTION_PAX_HELPTEXT, STR_CONFIG_SETTING_DISTRIBUTION_MANUAL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_183, SL_MAX_VERSION,        SC_ADVANCED, offsetof(LinkGraphSettings, distribution_pax), false),
SDT_VAR(GameSettings, linkgraph.distribution_mail, SLE_UINT8, SettingFlags({SettingFlag::GuiDropdown}), DT_MANUAL,       DT_MIN, DT_MAX, 1, STR_CONFIG_SETTING_DISTRIBUTION_MAIL, STR_CONFIG_SETTING_DISTRIBUTION_MAIL_HELPTEXT, STR_CONFIG_SETTING_DISTRIBUTION_MANUAL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_183, SL_MAX_VERSION,        SC_ADVANCED, offsetof(LinkGraphSettings, distribution_mail), false),
SDT_VAR(GameSettings, linkgraph.distribution_armoured, SLE_UINT8, SettingFlags({SettingFlag::GuiDropdown}), DT_MANUAL,       DT_MIN, DT_MAX, 1, STR_CONFIG_SETTING_DISTRIBUTION_ARMOURED, STR_CONFIG_SETTING_DISTRIBUTION_ARMOURED_HELPTEXT, STR_CONFIG_SETTING_DISTRIBUTION_MANUAL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_183, SL_MAX_VERSION,        SC_ADVANCED, offsetof(LinkGraphSettings, distribution_armoured), false),
SDT_VAR(GameSettings, linkgraph.distribution_default, SLE_UINT8, SettingFlags({SettingFlag::GuiDropdown}), DT_MANUAL,       DT_BEGIN, DT_MAX_NONSYMMETRIC, 1, STR_CONFIG_SETTING_DISTRIBUTION_DEFAULT, STR_CONFIG_SETTING_DISTRIBUTION_DEFAULT_HELPTEXT, STR_CONFIG_SETTING_DISTRIBUTION_MANUAL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_183, SL_MAX_VERSION,        SC_ADVANCED, offsetof(LinkGraphSettings, distribution_default), false),
SDT_VAR(GameSettings, linkgraph.accuracy, SLE_UINT8, SettingFlags({}), 16,       2, 64, 1, STR_CONFIG_SETTING_LINKGRAPH_ACCURACY, STR_CONFIG_SETTING_LINKGRAPH_ACCURACY_HELPTEXT, STR_JUST_COMMA, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_183, SL_MAX_VERSION,        SC_ADVANCED, offsetof(LinkGraphSettings, accuracy), false),
SDT_VAR(GameSettings, linkgraph.demand_distance, SLE_UINT8, SettingFlags({}), 100,       0, 255, 5, STR_CONFIG_SETTING_DEMAND_DISTANCE, STR_CONFIG_SETTING_DEMAND_DISTANCE_HELPTEXT, STR_CONFIG_SETTING_PERCENTAGE, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_183, SL_MAX_VERSION,        SC_ADVANCED, offsetof(LinkGraphSettings, demand_distance), false),
SDT_VAR(GameSettings, linkgraph.demand_size, SLE_UINT8, SettingFlags({}), 100,       0, 100, 5, STR_CONFIG_SETTING_DEMAND_SIZE, STR_CONFIG_SETTING_DEMAND_SIZE_HELPTEXT, STR_CONFIG_SETTING_PERCENTAGE, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_183, SL_MAX_VERSION,        SC_ADVANCED, offsetof(LinkGraphSettings, demand_size), false),
SDT_VAR(GameSettings, linkgraph.short_path_saturation, SLE_UINT8, SettingFlags({}), 80,       0, 250, 5, STR_CONFIG_SETTING_SHORT_PATH_SATURATION, STR_CONFIG_SETTING_SHORT_PATH_SATURATION_HELPTEXT, STR_CONFIG_SETTING_PERCENTAGE, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_183, SL_MAX_VERSION,        SC_ADVANCED, offsetof(LinkGraphSettings, short_path_saturation), false),
};
static std::pair<StringParameter, StringParameter> SettingsValueVelocityUnit(const IntSettingDesc &sd, int32_t value);
uint8_t _old_units;                                      ///< Old units from old savegames
static constexpr std::initializer_list<std::string_view> _locale_currencies{"GBP"sv, "USD"sv, "EUR"sv, "JPY"sv, "ATS"sv, "BEF"sv, "CHF"sv, "CZK"sv, "DEM"sv, "DKK"sv, "ESP"sv, "FIM"sv, "FRF"sv, "GRD"sv, "HUF"sv, "ISK"sv, "ITL"sv, "NLG"sv, "NOK"sv, "PLN"sv, "RON"sv, "RUR"sv, "SIT"sv, "SEK"sv, "TRY"sv, "SKK"sv, "BRL"sv, "EEK"sv, "LTL"sv, "KRW"sv, "ZAR"sv, "custom"sv, "GEL"sv, "IRR"sv, "RUB"sv, "MXN"sv, "NTD"sv, "CNY"sv, "HKD"sv, "INR"sv, "IDR"sv, "MYR"sv, "LVL"sv, "PTE"sv, "UAH"sv, "VND"sv};
static constexpr std::initializer_list<std::string_view> _locale_units{"imperial"sv, "metric"sv, "si"sv, "gameunits"sv, "knots"sv};
static_assert(_locale_currencies.size() == CURRENCY_END);
static const SettingVariant _locale_settings_table[] = {
SDT_OMANY(GameSettings, locale.currency, SLE_UINT8, SettingFlags({SettingFlag::NoNetworkSync}), 0,             CURRENCY_END - 1, _locale_currencies,     STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, [](auto) { MarkWholeScreenDirty(); }, nullptr, nullptr, nullptr, nullptr, SLV_97, SL_MAX_VERSION, nullptr, SC_BASIC, 0, false),
SDTG_OMANY("units",              SLE_UINT8, SettingFlags({SettingFlag::NotInConfig}), _old_units, 1,       2, _locale_units,     STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, [](auto) { MarkWholeScreenDirty(); }, nullptr, nullptr, nullptr, nullptr, SLV_97, SLV_184,        SC_BASIC, 0, false),
SDT_OMANY(GameSettings, locale.units_velocity, SLE_UINT8, SettingFlags({SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), 1,             4, _locale_units,     STR_CONFIG_SETTING_LOCALISATION_UNITS_VELOCITY, STR_CONFIG_SETTING_LOCALISATION_UNITS_VELOCITY_HELPTEXT, STR_NULL, nullptr, [](auto) { MarkWholeScreenDirty(); }, nullptr, nullptr, SettingsValueVelocityUnit, nullptr, SLV_184, SL_MAX_VERSION, nullptr, SC_BASIC, 0, false),
SDT_OMANY(GameSettings, locale.units_velocity_nautical, SLE_UINT8, SettingFlags({SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), 1,             4, _locale_units,     STR_CONFIG_SETTING_LOCALISATION_UNITS_VELOCITY_NAUTICAL, STR_CONFIG_SETTING_LOCALISATION_UNITS_VELOCITY_HELPTEXT, STR_NULL, nullptr, [](auto) { MarkWholeScreenDirty(); }, nullptr, nullptr, SettingsValueVelocityUnit, nullptr, SLV_VELOCITY_NAUTICAL, SL_MAX_VERSION, nullptr, SC_BASIC, 0, false),
SDT_OMANY(GameSettings, locale.units_power, SLE_UINT8, SettingFlags({SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), 1,             2, _locale_units,     STR_CONFIG_SETTING_LOCALISATION_UNITS_POWER, STR_CONFIG_SETTING_LOCALISATION_UNITS_POWER_HELPTEXT, STR_CONFIG_SETTING_LOCALISATION_UNITS_POWER_IMPERIAL, nullptr, [](auto) { MarkWholeScreenDirty(); }, nullptr, nullptr, nullptr, nullptr, SLV_184, SL_MAX_VERSION, nullptr, SC_BASIC, 0, false),
SDT_OMANY(GameSettings, locale.units_weight, SLE_UINT8, SettingFlags({SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), 1,             2, _locale_units,     STR_CONFIG_SETTING_LOCALISATION_UNITS_WEIGHT, STR_CONFIG_SETTING_LOCALISATION_UNITS_WEIGHT_HELPTEXT, STR_CONFIG_SETTING_LOCALISATION_UNITS_WEIGHT_IMPERIAL, nullptr, [](auto) { MarkWholeScreenDirty(); }, nullptr, nullptr, nullptr, nullptr, SLV_184, SL_MAX_VERSION, nullptr, SC_BASIC, 0, false),
SDT_OMANY(GameSettings, locale.units_volume, SLE_UINT8, SettingFlags({SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), 1,             2, _locale_units,     STR_CONFIG_SETTING_LOCALISATION_UNITS_VOLUME, STR_CONFIG_SETTING_LOCALISATION_UNITS_VOLUME_HELPTEXT, STR_CONFIG_SETTING_LOCALISATION_UNITS_VOLUME_IMPERIAL, nullptr, [](auto) { MarkWholeScreenDirty(); }, nullptr, nullptr, nullptr, nullptr, SLV_184, SL_MAX_VERSION, nullptr, SC_BASIC, 0, false),
SDT_OMANY(GameSettings, locale.units_force, SLE_UINT8, SettingFlags({SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), 2,             2, _locale_units,     STR_CONFIG_SETTING_LOCALISATION_UNITS_FORCE, STR_CONFIG_SETTING_LOCALISATION_UNITS_FORCE_HELPTEXT, STR_CONFIG_SETTING_LOCALISATION_UNITS_FORCE_IMPERIAL, nullptr, [](auto) { MarkWholeScreenDirty(); }, nullptr, nullptr, nullptr, nullptr, SLV_184, SL_MAX_VERSION, nullptr, SC_BASIC, 0, false),
SDT_OMANY(GameSettings, locale.units_height, SLE_UINT8, SettingFlags({SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), 1,             3, _locale_units,     STR_CONFIG_SETTING_LOCALISATION_UNITS_HEIGHT, STR_CONFIG_SETTING_LOCALISATION_UNITS_HEIGHT_HELPTEXT, STR_CONFIG_SETTING_LOCALISATION_UNITS_HEIGHT_IMPERIAL, nullptr, [](auto) { MarkWholeScreenDirty(); }, nullptr, nullptr, nullptr, nullptr, SLV_184, SL_MAX_VERSION, nullptr, SC_BASIC, 0, false),
SDT_SSTR(GameSettings, locale.digit_group_separator, SLE_STRQ, SettingFlags({SettingFlag::NoNetworkSync}), "",                                                       nullptr, [](auto) { MarkWholeScreenDirty(); }, SLV_118, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_SSTR(GameSettings, locale.digit_group_separator_currency, SLE_STRQ, SettingFlags({SettingFlag::NoNetworkSync}), "",                                                       nullptr, [](auto) { MarkWholeScreenDirty(); }, SLV_118, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_SSTR(GameSettings, locale.digit_decimal_separator, SLE_STRQ, SettingFlags({SettingFlag::NoNetworkSync}), "",                                                       nullptr, [](auto) { MarkWholeScreenDirty(); }, SLV_126, SL_MAX_VERSION,        SC_BASIC, 0, false),
};
extern std::string _config_language_file;
static constexpr std::initializer_list<std::string_view> _support8bppmodes{"no"sv, "system"sv, "hardware"sv};
static constexpr std::initializer_list<std::string_view> _display_opt_modes{"SHOW_TOWN_NAMES"sv, "SHOW_STATION_NAMES"sv, "SHOW_SIGNS"sv, "FULL_ANIMATION"sv, ""sv, "FULL_DETAIL"sv, "WAYPOINTS"sv, "SHOW_COMPETITOR_SIGNS"sv};
static constexpr std::initializer_list<std::string_view> _facility_display_opt_modes{"TRAIN"sv, "TRUCK_STOP"sv, "BUS_STOP"sv, "AIRPORT"sv, "DOCK"sv, ""sv, "GHOST"sv};
#ifdef WITH_COCOA
extern bool _allow_hidpi_window;
#endif
#ifndef WITH_COCOA
#define WITHOUT_COCOA
#endif
static const SettingVariant _misc_settings_table[] = {
SDTG_MMANY("display_opt", SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), _display_opt, (1 << DO_SHOW_TOWN_NAMES | 1 << DO_SHOW_STATION_NAMES | 1 << DO_SHOW_SIGNS | 1 << DO_FULL_ANIMATION | 1 << DO_FULL_DETAIL | 1 << DO_SHOW_WAYPOINT_NAMES | 1 << DO_SHOW_COMPETITOR_SIGNS),                        _display_opt_modes, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_ADVANCED, 0, true),
SDTG_MMANY("facility_display_opt", SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), _facility_display_opt, StationFacilities({StationFacility::Train, StationFacility::TruckStop, StationFacility::BusStop, StationFacility::Airport, StationFacility::Dock, STATION_FACILITY_GHOST}),                        _facility_display_opt_modes, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_ADVANCED, 0, true),
SDTG_BOOL("fullscreen",        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), _fullscreen, false,                               STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_BASIC, 0, true),
#ifdef WITH_COCOA
SDTG_BOOL("video_hw_accel",        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), _video_hw_accel, false,                               STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_BASIC, 0, true),
#endif
#ifdef WITHOUT_COCOA
SDTG_BOOL("video_hw_accel",        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), _video_hw_accel, true,                               STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_BASIC, 0, true),
#endif
SDTG_BOOL("video_vsync",        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), _video_vsync, false,                               STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_BASIC, 0, true),
SDTG_OMANY("support8bpp", SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), _support8bpp, 0,       2,            _support8bppmodes, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_BASIC, 0, true),
SDTG_SSTR("soundsset", SLE_STRQ, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), BaseSounds::ini_set, "",       0,                                               nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_BASIC, 0, true),
SDTG_SSTR("musicset", SLE_STRQ, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), BaseMusic::ini_set, "",       0,                                               nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_BASIC, 0, true),
SDTG_SSTR("videodriver", SLE_STRQ, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), _ini_videodriver, "",       0,                                               nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_EXPERT, 0, true),
SDTG_SSTR("musicdriver", SLE_STRQ, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), _ini_musicdriver, "",       0,                                               nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_EXPERT, 0, true),
SDTG_SSTR("sounddriver", SLE_STRQ, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), _ini_sounddriver, "",       0,                                               nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_EXPERT, 0, true),
SDTG_SSTR("blitter", SLE_STRQ, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), _ini_blitter, "",       0,                                               nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_ADVANCED, 0, true),
SDTG_SSTR("language", SLE_STR, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), _config_language_file, "",       0,                                               nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_BASIC, 0, true),
SDTG_LIST("resolution", SLE_UINT, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), _cur_resolution, "0,0",       2,                                                            SL_MIN_VERSION, SL_MAX_VERSION, SC_BASIC, 0, true),
SDTG_SSTR("screenshot_format", SLE_STR, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), _screenshot_format_name, "",       0,                                               nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_EXPERT, 0, true),
SDTG_SSTR("savegame_format", SLE_STR, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), _savegame_format, "",       0,                                               nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_EXPERT, 0, true),
SDTG_BOOL("rightclick_emulate",        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), _rightclick_emulate, false,                               STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_ADVANCED, 0, true),
#ifdef HAS_TRUETYPE_FONT
SDTG_SSTR("small_font", SLE_STR, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), _fcsettings.small.font, "",       0,                                               nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_ADVANCED, 0, true),
#endif
#ifdef HAS_TRUETYPE_FONT
SDTG_SSTR("medium_font", SLE_STR, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), _fcsettings.medium.font, "",       0,                                               nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_ADVANCED, 0, true),
#endif
#ifdef HAS_TRUETYPE_FONT
SDTG_SSTR("large_font", SLE_STR, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), _fcsettings.large.font, "",       0,                                               nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_ADVANCED, 0, true),
#endif
#ifdef HAS_TRUETYPE_FONT
SDTG_SSTR("mono_font", SLE_STR, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), _fcsettings.mono.font, "",       0,                                               nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_ADVANCED, 0, true),
#endif
#ifdef HAS_TRUETYPE_FONT
SDTG_VAR("small_size", SLE_UINT, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), _fcsettings.small.size, 0, 0, 72, 0,        STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_ADVANCED, 0, true),
#endif
#ifdef HAS_TRUETYPE_FONT
SDTG_VAR("medium_size", SLE_UINT, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), _fcsettings.medium.size, 0, 0, 72, 0,        STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_ADVANCED, 0, true),
#endif
#ifdef HAS_TRUETYPE_FONT
SDTG_VAR("large_size", SLE_UINT, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), _fcsettings.large.size, 0, 0, 72, 0,        STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_ADVANCED, 0, true),
#endif
#ifdef HAS_TRUETYPE_FONT
SDTG_VAR("mono_size", SLE_UINT, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), _fcsettings.mono.size, 0, 0, 72, 0,        STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_ADVANCED, 0, true),
#endif
#ifdef HAS_TRUETYPE_FONT
SDTG_BOOL("global_aa",        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), _fcsettings.global_aa, true,                               STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_ADVANCED, 0, true),
#endif
#ifdef HAS_TRUETYPE_FONT
SDTG_BOOL("prefer_sprite_font",        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), _fcsettings.prefer_sprite, false,                               STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_ADVANCED, 0, true),
#endif
SDTG_VAR("sprite_cache_size_px", SLE_UINT, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), _sprite_cache_size, 128, 1, 512, 0,        STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_EXPERT, 0, true),
SDTG_SSTR("player_face", SLE_STR, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), _company_manager_face, "",       0,                                               nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_BASIC, 0, true),
SDTG_VAR("transparency_options", SLE_UINT, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), _transparency_opt, 0, 0, 0x1FF, 0,        STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_BASIC, 0, true),
SDTG_VAR("transparency_locks", SLE_UINT, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), _transparency_lock, 0, 0, 0x1FF, 0,        STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_BASIC, 0, true),
SDTG_VAR("invisibility_options", SLE_UINT, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), _invisibility_opt, 0, 0, 0xFF, 0,        STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_BASIC, 0, true),
SDTG_SSTR("keyboard", SLE_STR, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), _keyboard_opt[0], "",       0,                                               nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_EXPERT, 0, true),
SDTG_SSTR("keyboard_caps", SLE_STR, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), _keyboard_opt[1], "",       0,                                               nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_EXPERT, 0, true),
SDTG_VAR("last_newgrf_count", SLE_UINT32, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), _settings_client.gui.last_newgrf_count, 100, 0, UINT32_MAX, 0,        STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_EXPERT, 0, true),
SDTG_VAR("gui_scale", SLE_INT32, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), _gui_scale_cfg, -1, -1, MAX_INTERFACE_SCALE, 0,        STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_BASIC, 0, true),
#ifdef WITH_COCOA
SDTG_BOOL("allow_hidpi",        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), _allow_hidpi_window, true,                               STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_ADVANCED, 0, true),
#endif
};
static const SettingVariant _multimedia_settings_table[] = {
SDTC_BOOL(              sound.news_ticker,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), true,                              STR_CONFIG_SETTING_SOUND_TICKER, STR_CONFIG_SETTING_SOUND_TICKER_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_BOOL(              sound.news_full,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), true,                              STR_CONFIG_SETTING_SOUND_NEWS, STR_CONFIG_SETTING_SOUND_NEWS_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_BOOL(              sound.new_year,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), true,                              STR_CONFIG_SETTING_SOUND_NEW_YEAR, STR_CONFIG_SETTING_SOUND_NEW_YEAR_HELPTEXT, STR_NULL, nullptr, nullptr, SettingTitleWallclock, SettingHelpWallclock, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_BOOL(              sound.confirm,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), true,                              STR_CONFIG_SETTING_SOUND_CONFIRM, STR_CONFIG_SETTING_SOUND_CONFIRM_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_BOOL(              sound.click_beep,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), true,                              STR_CONFIG_SETTING_SOUND_CLICK, STR_CONFIG_SETTING_SOUND_CLICK_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_BOOL(              sound.disaster,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), true,                              STR_CONFIG_SETTING_SOUND_DISASTER, STR_CONFIG_SETTING_SOUND_DISASTER_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_BOOL(              sound.vehicle,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), true,                              STR_CONFIG_SETTING_SOUND_VEHICLE, STR_CONFIG_SETTING_SOUND_VEHICLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_BOOL(              sound.ambient,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), true,                              STR_CONFIG_SETTING_SOUND_AMBIENT, STR_CONFIG_SETTING_SOUND_AMBIENT_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_VAR(              music.playlist, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), 0,       0, 5, 1, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDTC_VAR(              music.music_vol, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), 50,       0, 127, 1, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDTC_VAR(              music.effect_vol, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), 100,       0, 127, 1, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDTC_LIST(              music.custom_1, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), "",                                                                          SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDTC_LIST(              music.custom_2, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), "",                                                                          SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDTC_BOOL(              music.playing,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), true,                              STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDTC_BOOL(              music.shuffle,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), false,                              STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
};
static void UpdateClientConfigValues();
void ChangeNetworkRestartTime(bool reset);
static constexpr std::initializer_list<std::string_view> _server_game_type{"local"sv, "public"sv, "invite-only"sv};
static const SettingVariant _network_settings_table[] = {
SDTC_VAR(              network.sync_freq, SLE_UINT16, SettingFlags({SettingFlag::NotInSave, SettingFlag::NotInConfig, SettingFlag::NoNetworkSync, SettingFlag::NetworkOnly}), 100,       0, 100, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDTC_VAR(              network.frame_freq, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NotInConfig, SettingFlag::NoNetworkSync, SettingFlag::NetworkOnly}), 0,       0, 100, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDTC_VAR(              network.commands_per_frame, SLE_UINT16, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::NetworkOnly}), 2,       1, 65535, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDTC_VAR(              network.commands_per_frame_server, SLE_UINT16, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::NetworkOnly}), 16,       1, 65535, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDTC_VAR(              network.max_commands_in_queue, SLE_UINT16, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::NetworkOnly}), 16,       1, 65535, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDTC_VAR(              network.bytes_per_frame, SLE_UINT16, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::NetworkOnly}), 8,       1, 65535, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDTC_VAR(              network.bytes_per_frame_burst, SLE_UINT16, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::NetworkOnly}), 256,       1, 65535, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDTC_VAR(              network.max_init_time, SLE_UINT16, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::NetworkOnly}), 100,       0, 32000, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDTC_VAR(              network.max_join_time, SLE_UINT16, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::NetworkOnly}), 500,       0, 32000, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_VAR(              network.max_download_time, SLE_UINT16, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::NetworkOnly}), 1000,       0, 32000, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_VAR(              network.max_password_time, SLE_UINT16, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::NetworkOnly}), 2000,       0, 32000, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_VAR(              network.max_lag_time, SLE_UINT16, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::NetworkOnly}), 500,       0, 32000, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_BOOL(              network.pause_on_join,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::NetworkOnly}), true,                              STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_VAR(              network.server_port, SLE_UINT16, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::NetworkOnly}), NETWORK_DEFAULT_PORT,       0, 65535, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDTC_VAR(              network.server_admin_port, SLE_UINT16, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::NetworkOnly}), NETWORK_ADMIN_PORT,       0, 65535, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDTC_BOOL(              network.server_admin_chat,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::NetworkOnly}), true,                              STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDTC_BOOL(              network.allow_insecure_admin_login,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::NetworkOnly}), false,                              STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDTC_OMANY(              network.server_game_type, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::NetworkOnly}), SERVER_GAME_TYPE_LOCAL,             SERVER_GAME_TYPE_INVITE_ONLY, _server_game_type,     STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, [](auto) { NetworkUpdateServerGameType(); }, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDTC_BOOL(              network.autoclean_companies,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::NetworkOnly}), false,                              STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_VAR(              network.autoclean_protected, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiZeroIsSpecial, SettingFlag::NetworkOnly}), 36,       0, 240, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_VAR(              network.autoclean_novehicles, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiZeroIsSpecial, SettingFlag::NetworkOnly}), 0,       0, 240, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_VAR(              network.max_companies, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::NetworkOnly}), 15,       1, MAX_COMPANIES, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, [](auto) { UpdateClientConfigValues(); }, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDTC_VAR(              network.max_clients, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::NetworkOnly}), 25,       2, MAX_CLIENTS, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, [](auto) { UpdateClientConfigValues(); }, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDTC_VAR(              network.restart_game_year, SLE_INT32, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiZeroIsSpecial, SettingFlag::NetworkOnly}), 0,       CalendarTime::MIN_YEAR, CalendarTime::MAX_YEAR, 1, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_VAR(              network.restart_hours, SLE_UINT16, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiZeroIsSpecial, SettingFlag::NetworkOnly}), 0,       0, UINT16_MAX, 1, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, [](auto) { ChangeNetworkRestartTime(false); }, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_VAR(              network.min_active_clients, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::NetworkOnly}), 0,       0, MAX_CLIENTS, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_BOOL(              network.reload_cfg,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::NetworkOnly}), false,                              STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
};
static constexpr std::initializer_list<std::string_view> _use_relay_service{"never"sv, "ask"sv, "allow"sv};
static constexpr std::initializer_list<std::string_view> _participate_survey{"ask"sv, "no"sv, "yes"sv};
static const SettingVariant _network_private_settings_table[] = {
SDTC_SSTR(              network.client_name, SLE_STR, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), "",             NETWORK_CLIENT_NAME_LENGTH,                                  NetworkValidateClientName, NetworkUpdateClientName, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDTC_SSTR(              network.server_name, SLE_STR, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::NetworkOnly}), "",             NETWORK_NAME_LENGTH,                                  NetworkValidateServerName, [](auto) { UpdateClientConfigValues(); }, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDTC_SSTR(              network.connect_to_ip, SLE_STR, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), "",             0,                                  nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_SSTR(              network.last_joined, SLE_STR, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), "",             0,                                  nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDTC_OMANY(              network.use_relay_service, SLE_UINT8, SettingFlags({SettingFlag::GuiDropdown, SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), URS_ASK,             URS_ALLOW, _use_relay_service,     STR_CONFIG_SETTING_USE_RELAY_SERVICE, STR_CONFIG_SETTING_USE_RELAY_SERVICE_HELPTEXT, STR_CONFIG_SETTING_USE_RELAY_SERVICE_NEVER, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDTC_OMANY(              network.participate_survey, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), PS_ASK,             PS_YES, _participate_survey,     STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
};
static bool ReplaceAsteriskWithEmptyPassword(std::string &newval);
static const SettingVariant _network_secrets_settings_table[] = {
SDTC_SSTR(              network.server_password, SLE_STR, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::NetworkOnly}), "",             NETWORK_PASSWORD_LENGTH,                                  ReplaceAsteriskWithEmptyPassword, [](auto) { NetworkServerUpdateGameInfo(); }, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDTC_SSTR(              network.rcon_password, SLE_STR, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::NetworkOnly}), "",             NETWORK_PASSWORD_LENGTH,                                  ReplaceAsteriskWithEmptyPassword, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDTC_SSTR(              network.admin_password, SLE_STR, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::NetworkOnly}), "",             NETWORK_PASSWORD_LENGTH,                                  nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDTC_SSTR(              network.client_secret_key, SLE_STR, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), "",             NETWORK_SECRET_KEY_LENGTH,                                  [](auto) { return false; }, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_SSTR(              network.client_public_key, SLE_STR, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), "",             NETWORK_SECRET_KEY_LENGTH,                                  [](auto) { return false; }, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_SSTR(              network.server_invite_code, SLE_STR, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::NetworkOnly}), "",             NETWORK_INVITE_CODE_LENGTH,                                  nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_SSTR(              network.server_invite_code_secret, SLE_STR, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::NetworkOnly}), "",             NETWORK_INVITE_CODE_SECRET_LENGTH,                                  nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
};
static constexpr std::initializer_list<std::string_view> _news_display{"off"sv, "summarized"sv, "full"sv};
static const SettingVariant _news_display_settings_table[] = {
SDTC_OMANY(              news_display.arrival_player, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), 2,             2, _news_display,     STR_CONFIG_SETTING_NEWS_ARRIVAL_FIRST_VEHICLE_OWN, STR_CONFIG_SETTING_NEWS_ARRIVAL_FIRST_VEHICLE_OWN_HELPTEXT, STR_CONFIG_SETTING_NEWS_MESSAGES_OFF, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_OMANY(              news_display.arrival_other, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), 1,             2, _news_display,     STR_CONFIG_SETTING_NEWS_ARRIVAL_FIRST_VEHICLE_OTHER, STR_CONFIG_SETTING_NEWS_ARRIVAL_FIRST_VEHICLE_OTHER_HELPTEXT, STR_CONFIG_SETTING_NEWS_MESSAGES_OFF, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_OMANY(              news_display.accident, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), 2,             2, _news_display,     STR_CONFIG_SETTING_NEWS_ACCIDENTS_DISASTERS, STR_CONFIG_SETTING_NEWS_ACCIDENTS_DISASTERS_HELPTEXT, STR_CONFIG_SETTING_NEWS_MESSAGES_OFF, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_OMANY(              news_display.accident_other, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), 2,             2, _news_display,     STR_CONFIG_SETTING_NEWS_ACCIDENT_OTHER, STR_CONFIG_SETTING_NEWS_ACCIDENT_OTHER_HELPTEXT, STR_CONFIG_SETTING_NEWS_MESSAGES_OFF, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_OMANY(              news_display.company_info, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), 2,             2, _news_display,     STR_CONFIG_SETTING_NEWS_COMPANY_INFORMATION, STR_CONFIG_SETTING_NEWS_COMPANY_INFORMATION_HELPTEXT, STR_CONFIG_SETTING_NEWS_MESSAGES_OFF, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_OMANY(              news_display.open, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), 1,             2, _news_display,     STR_CONFIG_SETTING_NEWS_INDUSTRY_OPEN, STR_CONFIG_SETTING_NEWS_INDUSTRY_OPEN_HELPTEXT, STR_CONFIG_SETTING_NEWS_MESSAGES_OFF, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_OMANY(              news_display.close, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), 1,             2, _news_display,     STR_CONFIG_SETTING_NEWS_INDUSTRY_CLOSE, STR_CONFIG_SETTING_NEWS_INDUSTRY_CLOSE_HELPTEXT, STR_CONFIG_SETTING_NEWS_MESSAGES_OFF, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_OMANY(              news_display.economy, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), 2,             2, _news_display,     STR_CONFIG_SETTING_NEWS_ECONOMY_CHANGES, STR_CONFIG_SETTING_NEWS_ECONOMY_CHANGES_HELPTEXT, STR_CONFIG_SETTING_NEWS_MESSAGES_OFF, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_OMANY(              news_display.production_player, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), 1,             2, _news_display,     STR_CONFIG_SETTING_NEWS_INDUSTRY_CHANGES_COMPANY, STR_CONFIG_SETTING_NEWS_INDUSTRY_CHANGES_COMPANY_HELPTEXT, STR_CONFIG_SETTING_NEWS_MESSAGES_OFF, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_OMANY(              news_display.production_other, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), 0,             2, _news_display,     STR_CONFIG_SETTING_NEWS_INDUSTRY_CHANGES_OTHER, STR_CONFIG_SETTING_NEWS_INDUSTRY_CHANGES_OTHER_HELPTEXT, STR_CONFIG_SETTING_NEWS_MESSAGES_OFF, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_OMANY(              news_display.production_nobody, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), 0,             2, _news_display,     STR_CONFIG_SETTING_NEWS_INDUSTRY_CHANGES_UNSERVED, STR_CONFIG_SETTING_NEWS_INDUSTRY_CHANGES_UNSERVED_HELPTEXT, STR_CONFIG_SETTING_NEWS_MESSAGES_OFF, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_OMANY(              news_display.advice, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), 2,             2, _news_display,     STR_CONFIG_SETTING_NEWS_ADVICE, STR_CONFIG_SETTING_NEWS_ADVICE_HELPTEXT, STR_CONFIG_SETTING_NEWS_MESSAGES_OFF, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_OMANY(              news_display.new_vehicles, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), 2,             2, _news_display,     STR_CONFIG_SETTING_NEWS_NEW_VEHICLES, STR_CONFIG_SETTING_NEWS_NEW_VEHICLES_HELPTEXT, STR_CONFIG_SETTING_NEWS_MESSAGES_OFF, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_OMANY(              news_display.acceptance, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), 2,             2, _news_display,     STR_CONFIG_SETTING_NEWS_CHANGES_ACCEPTANCE, STR_CONFIG_SETTING_NEWS_CHANGES_ACCEPTANCE_HELPTEXT, STR_CONFIG_SETTING_NEWS_MESSAGES_OFF, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_OMANY(              news_display.subsidies, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), 1,             2, _news_display,     STR_CONFIG_SETTING_NEWS_SUBSIDIES, STR_CONFIG_SETTING_NEWS_SUBSIDIES_HELPTEXT, STR_CONFIG_SETTING_NEWS_MESSAGES_OFF, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDTC_OMANY(              news_display.general, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), 2,             2, _news_display,     STR_CONFIG_SETTING_NEWS_GENERAL_INFORMATION, STR_CONFIG_SETTING_NEWS_GENERAL_INFORMATION_HELPTEXT, STR_CONFIG_SETTING_NEWS_MESSAGES_OFF, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
};
static constexpr std::initializer_list<std::string_view> _town_names{"english"sv, "french"sv, "german"sv, "american"sv, "latin"sv, "silly"sv, "swedish"sv, "dutch"sv, "finnish"sv, "polish"sv, "slovak"sv, "norwegian"sv, "hungarian"sv, "austrian"sv, "romanian"sv, "czech"sv, "swiss"sv, "danish"sv, "turkish"sv, "italian"sv, "catalan"sv};
static constexpr std::initializer_list<std::string_view> _climates{"temperate"sv, "arctic"sv, "tropic"sv, "toyland"sv};
static const SettingVariant _old_gameopt_settings_table[] = {
/* In version 4 a new difficulty setting has been added to the difficulty settings,
* town attitude towards demolishing. Needs special handling because some dimwit thought
* it funny to have the GameDifficulty struct be an array while it is a struct of
* same-sized members
* XXX - To save file-space and since values are never bigger than about 10? only
* save the first 16 bits in the savegame. Question is why the values are still int32
* and why not byte for example?
* 'SLE_FILE_I16 | SLE_VAR_U16' in "diff_custom" is needed to get around SlCopy() hack
* for savegames version 0 - though it is an array, it has to go through the byteswap process */
SDTG_LIST("diff_custom",              SLE_FILE_I16 | SLE_VAR_U16, SettingFlags({SettingFlag::NotInConfig}), _old_diff_custom, "", 17, SL_MIN_VERSION, SLV_4, SC_ADVANCED, 0, false),
SDTG_LIST("diff_custom",              SLE_UINT16, SettingFlags({SettingFlag::NotInConfig}), _old_diff_custom, "", 18, SLV_4, SL_MAX_VERSION, SC_ADVANCED, 0, false),
SDTG_VAR("diff_level",              SLE_UINT8, SettingFlags({SettingFlag::NotInConfig}), _old_diff_level, SP_CUSTOM, SP_EASY, SP_CUSTOM, 0,  STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_BASIC, 0, false),
SDT_OMANY(GameSettings, locale.currency, SLE_UINT8, SettingFlags({SettingFlag::NoNetworkSync}), 0,       CURRENCY_END - 1, _locale_currencies,            STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, nullptr, SC_BASIC, 0, false),
SDTG_OMANY("units",              SLE_UINT8, SettingFlags({SettingFlag::NotInConfig}), _old_units, 1, 2, _locale_units,            STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_OMANY(GameSettings, game_creation.town_name, SLE_UINT8, SettingFlags({}), 0,       255, _town_names,            STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, nullptr, SC_BASIC, 0, false),
SDT_OMANY(GameSettings, game_creation.landscape, SLE_UINT8, SettingFlags({}), 0,       3, _climates,            STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, ConvertLandscape, SC_BASIC, 0, false),
SDT_VAR(GameSettings, game_creation.snow_line_height, SLE_UINT8, SettingFlags({}), DEF_SNOWLINE_HEIGHT * TILE_HEIGHT, MIN_SNOWLINE_HEIGHT * TILE_HEIGHT, UINT8_MAX,        0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SLV_22,        SC_ADVANCED, 0, false),
SDT_OMANY(GameSettings, vehicle.road_side, SLE_UINT8, SettingFlags({}), 1,       1, _roadsides,            STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, nullptr, SC_BASIC, 0, false),
};
static const SettingVariant _pathfinding_settings_table[] = {
SDT_BOOL(GameSettings, pf.forbid_90_deg,        SettingFlags({}), true,                              STR_CONFIG_SETTING_FORBID_90_DEG, STR_CONFIG_SETTING_FORBID_90_DEG_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_BOOL(GameSettings, pf.roadveh_queue,        SettingFlags({}), true,                              STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_BOOL(GameSettings, pf.reverse_at_signals,        SettingFlags({}), false,                              STR_CONFIG_SETTING_REVERSE_AT_SIGNALS, STR_CONFIG_SETTING_REVERSE_AT_SIGNALS_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_159, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_VAR(GameSettings, pf.wait_oneway_signal, SLE_UINT8, SettingFlags({}), 15,       2, 255, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, pf.wait_twoway_signal, SLE_UINT8, SettingFlags({}), 41,       2, 255, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, pf.wait_for_pbs_path, SLE_UINT8, SettingFlags({}), 30,       2, 255, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_100, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_BOOL(GameSettings, pf.reserve_paths,        SettingFlags({}), false,                              STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_100, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, pf.path_backoff_interval, SLE_UINT8, SettingFlags({}), 20,       1, 255, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_100, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, pf.yapf.max_search_nodes, SLE_UINT, SettingFlags({}), 10000,       500, 1000000, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_28, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_BOOL(GameSettings, pf.yapf.rail_firstred_twoway_eol,        SettingFlags({}), true,                              STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_28, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, pf.yapf.rail_firstred_penalty, SLE_UINT, SettingFlags({}), 10 * YAPF_TILE_LENGTH,       0, 1000000, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_28, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, pf.yapf.rail_firstred_exit_penalty, SLE_UINT, SettingFlags({}), 100 * YAPF_TILE_LENGTH,       0, 1000000, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_28, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, pf.yapf.rail_lastred_penalty, SLE_UINT, SettingFlags({}), 10 * YAPF_TILE_LENGTH,       0, 1000000, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_28, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, pf.yapf.rail_lastred_exit_penalty, SLE_UINT, SettingFlags({}), 100 * YAPF_TILE_LENGTH,       0, 1000000, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_28, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, pf.yapf.rail_station_penalty, SLE_UINT, SettingFlags({}), 10 * YAPF_TILE_LENGTH,       0, 1000000, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_28, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, pf.yapf.rail_slope_penalty, SLE_UINT, SettingFlags({}), 2 * YAPF_TILE_LENGTH,       0, 1000000, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_28, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, pf.yapf.rail_curve45_penalty, SLE_UINT, SettingFlags({}), 1 * YAPF_TILE_LENGTH,       0, 1000000, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_28, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, pf.yapf.rail_curve90_penalty, SLE_UINT, SettingFlags({}), 6 * YAPF_TILE_LENGTH,       0, 1000000, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_28, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, pf.yapf.rail_depot_reverse_penalty, SLE_UINT, SettingFlags({}), 50 * YAPF_TILE_LENGTH,       0, 1000000, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_28, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, pf.yapf.rail_crossing_penalty, SLE_UINT, SettingFlags({}), 3 * YAPF_TILE_LENGTH,       0, 1000000, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_28, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, pf.yapf.rail_look_ahead_max_signals, SLE_UINT, SettingFlags({}), 10,       1, 100, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_28, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, pf.yapf.rail_look_ahead_signal_p0, SLE_INT, SettingFlags({}), 500,       -1000000, 1000000, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_28, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, pf.yapf.rail_look_ahead_signal_p1, SLE_INT, SettingFlags({}), -100,       -1000000, 1000000, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_28, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, pf.yapf.rail_look_ahead_signal_p2, SLE_INT, SettingFlags({}), 5,       -1000000, 1000000, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_28, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, pf.yapf.rail_pbs_cross_penalty, SLE_UINT, SettingFlags({}), 3 * YAPF_TILE_LENGTH,       0, 1000000, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_100, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, pf.yapf.rail_pbs_station_penalty, SLE_UINT, SettingFlags({}), 8 * YAPF_TILE_LENGTH,       0, 1000000, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_100, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, pf.yapf.rail_pbs_signal_back_penalty, SLE_UINT, SettingFlags({}), 15 * YAPF_TILE_LENGTH,       0, 1000000, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_100, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, pf.yapf.rail_doubleslip_penalty, SLE_UINT, SettingFlags({}), 1 * YAPF_TILE_LENGTH,       0, 1000000, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_100, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, pf.yapf.rail_longer_platform_penalty, SLE_UINT, SettingFlags({}), 8 * YAPF_TILE_LENGTH,       0, 20000, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_33, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, pf.yapf.rail_longer_platform_per_tile_penalty, SLE_UINT, SettingFlags({}), 0 * YAPF_TILE_LENGTH,       0, 20000, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_33, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, pf.yapf.rail_shorter_platform_penalty, SLE_UINT, SettingFlags({}), 40 * YAPF_TILE_LENGTH,       0, 20000, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_33, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, pf.yapf.rail_shorter_platform_per_tile_penalty, SLE_UINT, SettingFlags({}), 0 * YAPF_TILE_LENGTH,       0, 20000, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_33, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, pf.yapf.road_slope_penalty, SLE_UINT, SettingFlags({}), 2 * YAPF_TILE_LENGTH,       0, 1000000, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_33, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, pf.yapf.road_curve_penalty, SLE_UINT, SettingFlags({}), 1 * YAPF_TILE_LENGTH,       0, 1000000, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_33, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, pf.yapf.road_crossing_penalty, SLE_UINT, SettingFlags({}), 3 * YAPF_TILE_LENGTH,       0, 1000000, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_33, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, pf.yapf.road_stop_penalty, SLE_UINT, SettingFlags({}), 8 * YAPF_TILE_LENGTH,       0, 1000000, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_47, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, pf.yapf.road_stop_occupied_penalty, SLE_UINT, SettingFlags({}), 8 * YAPF_TILE_LENGTH,       0, 1000000, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_130, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, pf.yapf.road_stop_bay_occupied_penalty, SLE_UINT, SettingFlags({}), 15 * YAPF_TILE_LENGTH,       0, 1000000, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_130, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, pf.yapf.maximum_go_to_depot_penalty, SLE_UINT, SettingFlags({}), 20 * YAPF_TILE_LENGTH,       0, 1000000, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_131, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, pf.yapf.ship_curve45_penalty, SLE_UINT, SettingFlags({}), 1 * YAPF_TILE_LENGTH,       0, 1000000, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_SHIP_CURVE_PENALTY, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, pf.yapf.ship_curve90_penalty, SLE_UINT, SettingFlags({}), 6 * YAPF_TILE_LENGTH,       0, 1000000, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_SHIP_CURVE_PENALTY, SL_MAX_VERSION,        SC_EXPERT, 0, false),
};
static const SettingVariant _script_settings_table[] = {
SDT_VAR(GameSettings, script.script_max_opcode_till_suspend, SLE_UINT32, SettingFlags({SettingFlag::NewgameOnly}), 10000,       500, 250000, 2500, STR_CONFIG_SETTING_SCRIPT_MAX_OPCODES, STR_CONFIG_SETTING_SCRIPT_MAX_OPCODES_HELPTEXT, STR_JUST_COMMA, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_107, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, script.script_max_memory_megabytes, SLE_UINT32, SettingFlags({SettingFlag::NewgameOnly}), 1024,       8, 8192, 8, STR_CONFIG_SETTING_SCRIPT_MAX_MEMORY, STR_CONFIG_SETTING_SCRIPT_MAX_MEMORY_HELPTEXT, STR_CONFIG_SETTING_SCRIPT_MAX_MEMORY_VALUE, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_SCRIPT_MEMLIMIT, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_BOOL(GameSettings, ai.ai_in_multiplayer,        SettingFlags({}), true,                              STR_CONFIG_SETTING_AI_IN_MULTIPLAYER, STR_CONFIG_SETTING_AI_IN_MULTIPLAYER_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_BOOL(GameSettings, ai.ai_disable_veh_train,        SettingFlags({}), false,                              STR_CONFIG_SETTING_AI_BUILDS_TRAINS, STR_CONFIG_SETTING_AI_BUILDS_TRAINS_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_BOOL(GameSettings, ai.ai_disable_veh_roadveh,        SettingFlags({}), false,                              STR_CONFIG_SETTING_AI_BUILDS_ROAD_VEHICLES, STR_CONFIG_SETTING_AI_BUILDS_ROAD_VEHICLES_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_BOOL(GameSettings, ai.ai_disable_veh_aircraft,        SettingFlags({}), false,                              STR_CONFIG_SETTING_AI_BUILDS_AIRCRAFT, STR_CONFIG_SETTING_AI_BUILDS_AIRCRAFT_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_BOOL(GameSettings, ai.ai_disable_veh_ship,        SettingFlags({}), false,                              STR_CONFIG_SETTING_AI_BUILDS_SHIPS, STR_CONFIG_SETTING_AI_BUILDS_SHIPS_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
};
/* win32_v.cpp only settings */
#if defined(_WIN32) && !defined(DEDICATED)
extern bool _window_maximize;
static const SettingVariant _win32_settings_table[] = {
SDTG_BOOL("window_maximize",        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), _window_maximize, false,                        STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_BASIC, 0, true),
};
#endif /* _WIN32 */
static const SettingVariant _window_settings_table[] = {
SDT_BOOL(WindowDesc, pref_sticky,        SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), false,                        STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_ADVANCED, 0, false),
SDT_VAR(WindowDesc, pref_width, SLE_INT16, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), 0, 0, 32000, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_ADVANCED, 0, false),
SDT_VAR(WindowDesc, pref_height, SLE_INT16, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), 0, 0, 32000, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION, SC_ADVANCED, 0, false),
};
static bool CheckMaxHeightLevel(int32_t &new_value);
static bool CheckFreeformEdges(int32_t &new_value);
static void UpdateFreeformEdges(int32_t new_value);
static const SettingVariant _world_settings_table[] = {
SDT_OMANY(GameSettings, game_creation.town_name, SLE_UINT8, SettingFlags({SettingFlag::NoNetwork}), 0,             255, _town_names,     STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_97, SL_MAX_VERSION, nullptr, SC_BASIC, 0, false),
SDT_OMANY(GameSettings, game_creation.landscape, SLE_UINT8, SettingFlags({SettingFlag::GuiDropdown, SettingFlag::NewgameOnly}), 0,             3, _climates,     STR_CONFIG_SETTING_LANDSCAPE, STR_CONFIG_SETTING_LANDSCAPE_HELPTEXT, STR_CLIMATE_TEMPERATE_LANDSCAPE, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_97, SL_MAX_VERSION, ConvertLandscape, SC_BASIC, 0, false),
SDT_VAR(GameSettings, game_creation.heightmap_height, SLE_UINT8, SettingFlags({SettingFlag::NewgameOnly}), MAP_HEIGHT_LIMIT_AUTO_MINIMUM,       MIN_HEIGHTMAP_HEIGHT, MAX_MAP_HEIGHT_LIMIT, 1, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_MAPGEN_SETTINGS_REVAMP, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_VAR(GameSettings, game_creation.snow_line_height, SLE_UINT8, SettingFlags({SettingFlag::SceneditOnly}), DEF_SNOWLINE_HEIGHT,       MIN_SNOWLINE_HEIGHT, MAX_SNOWLINE_HEIGHT, 1, STR_CONFIG_SETTING_SNOWLINE_HEIGHT, STR_CONFIG_SETTING_SNOWLINE_HEIGHT_HELPTEXT, STR_JUST_COMMA, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_VAR(GameSettings, game_creation.snow_coverage, SLE_UINT8, SettingFlags({SettingFlag::NewgameOnly}), DEF_SNOW_COVERAGE,       0, 100, 10, STR_CONFIG_SETTING_SNOW_COVERAGE, STR_CONFIG_SETTING_SNOW_COVERAGE_HELPTEXT, STR_CONFIG_SETTING_SNOW_COVERAGE_VALUE, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_MAPGEN_SETTINGS_REVAMP, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_VAR(GameSettings, game_creation.desert_coverage, SLE_UINT8, SettingFlags({SettingFlag::NewgameOnly}), DEF_DESERT_COVERAGE,       0, 100, 10, STR_CONFIG_SETTING_DESERT_COVERAGE, STR_CONFIG_SETTING_DESERT_COVERAGE_HELPTEXT, STR_CONFIG_SETTING_DESERT_COVERAGE_VALUE, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_MAPGEN_SETTINGS_REVAMP, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_VAR(GameSettings, game_creation.starting_year, SLE_INT32, SettingFlags({}), CalendarTime::DEF_START_YEAR,       CalendarTime::MIN_YEAR, CalendarTime::MAX_YEAR, 1, STR_CONFIG_SETTING_STARTING_YEAR, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_JUST_INT, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_VAR(GameSettings, game_creation.ending_year, SLE_INT32, SettingFlags({SettingFlag::GuiZeroIsSpecial}), CalendarTime::DEF_END_YEAR,       CalendarTime::MIN_YEAR, CalendarTime::MAX_YEAR - 1, 1, STR_CONFIG_SETTING_ENDING_YEAR, STR_CONFIG_SETTING_ENDING_YEAR_HELPTEXT, STR_CONFIG_SETTING_ENDING_YEAR_VALUE, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_ENDING_YEAR, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_VAR(GameSettings, game_creation.land_generator, SLE_UINT8, SettingFlags({SettingFlag::GuiDropdown, SettingFlag::NewgameOnly}), 1,       0, 1, 0, STR_CONFIG_SETTING_LAND_GENERATOR, STR_CONFIG_SETTING_LAND_GENERATOR_HELPTEXT, STR_CONFIG_SETTING_LAND_GENERATOR_ORIGINAL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_30, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_VAR(GameSettings, game_creation.oil_refinery_limit, SLE_UINT8, SettingFlags({}), 32,       12, 128, 0, STR_CONFIG_SETTING_OIL_REF_EDGE_DISTANCE, STR_CONFIG_SETTING_OIL_REF_EDGE_DISTANCE_HELPTEXT, STR_CONFIG_SETTING_TILE_LENGTH, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_30, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_VAR(GameSettings, game_creation.tgen_smoothness, SLE_UINT8, SettingFlags({SettingFlag::GuiDropdown, SettingFlag::NewgameOnly}), 1,       TGEN_SMOOTHNESS_BEGIN, TGEN_SMOOTHNESS_END - 1, 0, STR_CONFIG_SETTING_ROUGHNESS_OF_TERRAIN, STR_CONFIG_SETTING_ROUGHNESS_OF_TERRAIN_HELPTEXT, STR_CONFIG_SETTING_ROUGHNESS_OF_TERRAIN_VERY_SMOOTH, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_30, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_VAR(GameSettings, game_creation.variety, SLE_UINT8, SettingFlags({SettingFlag::GuiDropdown, SettingFlag::NewgameOnly}), 0,       0, 5, 0, STR_CONFIG_SETTING_VARIETY, STR_CONFIG_SETTING_VARIETY_HELPTEXT, STR_VARIETY_NONE, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_197, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_VAR(GameSettings, game_creation.generation_seed, SLE_UINT32, SettingFlags({SettingFlag::NotInConfig}), GENERATE_NEW_SEED,       0, UINT32_MAX, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_30, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, game_creation.tree_placer, SLE_UINT8, SettingFlags({SettingFlag::GuiDropdown, SettingFlag::NewgameOnly, SettingFlag::SceneditToo}), 2,       0, 2, 0, STR_CONFIG_SETTING_TREE_PLACER, STR_CONFIG_SETTING_TREE_PLACER_HELPTEXT, STR_CONFIG_SETTING_TREE_PLACER_NONE, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_30, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_VAR(GameSettings, game_creation.heightmap_rotation, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync, SettingFlag::GuiDropdown}), 0,       0, 1, 0, STR_CONFIG_SETTING_HEIGHTMAP_ROTATION, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_CONFIG_SETTING_HEIGHTMAP_ROTATION_COUNTER_CLOCKWISE, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_VAR(GameSettings, game_creation.se_flat_world_height, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), 1,       0, 15, 0, STR_CONFIG_SETTING_SE_FLAT_WORLD_HEIGHT, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_JUST_COMMA, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_VAR(GameSettings, game_creation.map_x, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), 8,       MIN_MAP_SIZE_BITS, MAX_MAP_SIZE_BITS, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_VAR(GameSettings, game_creation.map_y, SLE_UINT8, SettingFlags({SettingFlag::NotInSave, SettingFlag::NoNetworkSync}), 8,       MIN_MAP_SIZE_BITS, MAX_MAP_SIZE_BITS, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_VAR(GameSettings, game_creation.water_borders, SLE_UINT8, SettingFlags({}), 16,       0, 16, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_111, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_VAR(GameSettings, game_creation.water_border_presets, SLE_UINT8, SettingFlags({}), BFP_RANDOM,       BFP_RANDOM, BFP_INFINITE_WATER, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_VAR(GameSettings, game_creation.custom_town_number, SLE_UINT16, SettingFlags({}), 1,       1, 5000, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_115, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_VAR(GameSettings, game_creation.custom_industry_number, SLE_UINT16, SettingFlags({}), 1,       1, 64000, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_VAR(GameSettings, game_creation.custom_terrain_type, SLE_UINT8, SettingFlags({SettingFlag::NewgameOnly}), MAP_HEIGHT_LIMIT_AUTO_MINIMUM,       MIN_CUSTOM_TERRAIN_TYPE, MAX_MAP_HEIGHT_LIMIT, 1, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_MAPGEN_SETTINGS_REVAMP, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_VAR(GameSettings, game_creation.custom_sea_level, SLE_UINT8, SettingFlags({}), CUSTOM_SEA_LEVEL_MIN_PERCENTAGE,       CUSTOM_SEA_LEVEL_MIN_PERCENTAGE, CUSTOM_SEA_LEVEL_MAX_PERCENTAGE, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_149, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_VAR(GameSettings, game_creation.min_river_length, SLE_UINT8, SettingFlags({}), 16,       2, 255, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_163, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, game_creation.river_route_random, SLE_UINT8, SettingFlags({}), 5,       1, 255, 0, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_163, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, game_creation.amount_of_rivers, SLE_UINT8, SettingFlags({SettingFlag::GuiDropdown, SettingFlag::NewgameOnly}), 2,       0, 3, 0, STR_CONFIG_SETTING_RIVER_AMOUNT, STR_CONFIG_SETTING_RIVER_AMOUNT_HELPTEXT, STR_RIVERS_NONE, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_163, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_VAR(GameSettings, construction.map_height_limit, SLE_UINT8, SettingFlags({SettingFlag::NewgameOnly, SettingFlag::SceneditToo, SettingFlag::GuiZeroIsSpecial}), 0,       MIN_MAP_HEIGHT_LIMIT, MAX_MAP_HEIGHT_LIMIT, 1, STR_CONFIG_SETTING_MAP_HEIGHT_LIMIT, STR_CONFIG_SETTING_MAP_HEIGHT_LIMIT_HELPTEXT, STR_CONFIG_SETTING_MAP_HEIGHT_LIMIT_VALUE, CheckMaxHeightLevel, [](auto) { InvalidateWindowClassesData(WC_SMALLMAP, 2); }, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_194, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_BOOL(GameSettings, construction.build_on_slopes,        SettingFlags({SettingFlag::NoNetwork}), true,                              STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, construction.command_pause_level, SLE_UINT8, SettingFlags({SettingFlag::GuiDropdown, SettingFlag::NoNetwork}), 1,       0, 3, 1, STR_CONFIG_SETTING_COMMAND_PAUSE_LEVEL, STR_CONFIG_SETTING_COMMAND_PAUSE_LEVEL_HELPTEXT, STR_CONFIG_SETTING_COMMAND_PAUSE_LEVEL_NO_ACTIONS, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_154, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_VAR(GameSettings, construction.terraform_per_64k_frames, SLE_UINT32, SettingFlags({}), 64 << 16,       0, 1 << 30, 1, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_156, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, construction.terraform_frame_burst, SLE_UINT16, SettingFlags({}), 4096,       0, 1 << 15, 1, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_156, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, construction.clear_per_64k_frames, SLE_UINT32, SettingFlags({}), 64 << 16,       0, 1 << 30, 1, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_156, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, construction.clear_frame_burst, SLE_UINT16, SettingFlags({}), 4096,       0, 1 << 15, 1, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_156, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, construction.tree_per_64k_frames, SLE_UINT32, SettingFlags({}), 64 << 16,       0, 1 << 30, 1, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_175, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, construction.tree_frame_burst, SLE_UINT16, SettingFlags({}), 4096,       0, 1 << 15, 1, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_175, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, construction.build_object_per_64k_frames, SLE_UINT32, SettingFlags({}), 32 << 16,       0, 1 << 30, 1, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, construction.build_object_frame_burst, SLE_UINT16, SettingFlags({}), 2048,       0, 1 << 15, 1, STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_BOOL(GameSettings, construction.autoslope,        SettingFlags({}), true,                              STR_CONFIG_SETTING_AUTOSLOPE, STR_CONFIG_SETTING_AUTOSLOPE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_75, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_BOOL(GameSettings, construction.extra_dynamite,        SettingFlags({}), true,                              STR_CONFIG_SETTING_EXTRADYNAMITE, STR_CONFIG_SETTING_EXTRADYNAMITE_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_VAR(GameSettings, construction.max_bridge_length, SLE_UINT16, SettingFlags({SettingFlag::NoNetwork}), 64,       1, MAX_MAP_SIZE, 1, STR_CONFIG_SETTING_MAX_BRIDGE_LENGTH, STR_CONFIG_SETTING_MAX_BRIDGE_LENGTH_HELPTEXT, STR_CONFIG_SETTING_TILE_LENGTH, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_159, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_VAR(GameSettings, construction.max_bridge_height, SLE_UINT8, SettingFlags({SettingFlag::NoNetwork}), 12,       1, MAX_TILE_HEIGHT, 1, STR_CONFIG_SETTING_MAX_BRIDGE_HEIGHT, STR_CONFIG_SETTING_MAX_BRIDGE_HEIGHT_HELPTEXT, STR_JUST_COMMA, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_194, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, construction.max_tunnel_length, SLE_UINT16, SettingFlags({SettingFlag::NoNetwork}), 64,       1, MAX_MAP_SIZE, 1, STR_CONFIG_SETTING_MAX_TUNNEL_LENGTH, STR_CONFIG_SETTING_MAX_TUNNEL_LENGTH_HELPTEXT, STR_CONFIG_SETTING_TILE_LENGTH, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_159, SL_MAX_VERSION,        SC_ADVANCED, 0, false),
SDT_VAR(GameSettings, construction.train_signal_side, SLE_UINT8, SettingFlags({SettingFlag::GuiDropdown, SettingFlag::NoNetwork}), 1,       0, 2, 0, STR_CONFIG_SETTING_SIGNALSIDE, STR_CONFIG_SETTING_SIGNALSIDE_HELPTEXT, STR_CONFIG_SETTING_SIGNALSIDE_LEFT, nullptr, [](auto) { MarkWholeScreenDirty(); }, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_BOOL(GameSettings, construction.road_stop_on_town_road,        SettingFlags({}), true,                              STR_CONFIG_SETTING_STOP_ON_TOWN_ROAD, STR_CONFIG_SETTING_STOP_ON_TOWN_ROAD_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_47, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_BOOL(GameSettings, construction.road_stop_on_competitor_road,        SettingFlags({}), true,                              STR_CONFIG_SETTING_STOP_ON_COMPETITOR_ROAD, STR_CONFIG_SETTING_STOP_ON_COMPETITOR_ROAD_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_114, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_BOOL(GameSettings, construction.crossing_with_competitor,        SettingFlags({}), true,                              STR_CONFIG_SETTING_CROSSING_WITH_COMPETITOR, STR_CONFIG_SETTING_CROSSING_WITH_COMPETITOR_HELPTEXT, STR_NULL, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_VAR(GameSettings, construction.raw_industry_construction, SLE_UINT8, SettingFlags({SettingFlag::GuiDropdown}), 0,       0, 2, 0, STR_CONFIG_SETTING_RAW_INDUSTRY_CONSTRUCTION_METHOD, STR_CONFIG_SETTING_RAW_INDUSTRY_CONSTRUCTION_METHOD_HELPTEXT, STR_CONFIG_SETTING_RAW_INDUSTRY_CONSTRUCTION_METHOD_NONE, nullptr, [](auto) { InvalidateWindowData(WC_BUILD_INDUSTRY, 0); }, nullptr, nullptr, nullptr, nullptr, nullptr, SL_MIN_VERSION, SL_MAX_VERSION,        SC_BASIC, 0, false),
SDT_VAR(GameSettings, construction.industry_platform, SLE_UINT8, SettingFlags({}), 1,       0, 4, 0, STR_CONFIG_SETTING_INDUSTRY_PLATFORM, STR_CONFIG_SETTING_INDUSTRY_PLATFORM_HELPTEXT, STR_CONFIG_SETTING_TILE_LENGTH, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_148, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_BOOL(GameSettings, construction.freeform_edges,        SettingFlags({}), true,                              STR_NULL, STR_CONFIG_SETTING_NO_EXPLANATION_AVAILABLE_HELPTEXT, STR_NULL, CheckFreeformEdges, UpdateFreeformEdges, nullptr, nullptr, nullptr, nullptr, SLV_111, SL_MAX_VERSION,        SC_EXPERT, 0, false),
SDT_VAR(GameSettings, construction.extra_tree_placement, SLE_UINT8, SettingFlags({SettingFlag::GuiDropdown}), 2,       0, 3, 0, STR_CONFIG_SETTING_EXTRA_TREE_PLACEMENT, STR_CONFIG_SETTING_EXTRA_TREE_PLACEMENT_HELPTEXT, STR_CONFIG_SETTING_EXTRA_TREE_PLACEMENT_NO_SPREAD, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, nullptr, SLV_132, SL_MAX_VERSION,        SC_BASIC, 0, false),
};
static_assert(12 <= MAX_SLE_INT16, "Maximum value for CompanySettings.engine_renew_months exceeds storage size");
static_assert(2000000 <= MAX_SLE_UINT, "Maximum value for CompanySettings.engine_renew_money exceeds storage size");
static_assert(MAX_SERVINT_DAYS <= MAX_SLE_UINT16, "Maximum value for CompanySettings.vehicle.servint_trains exceeds storage size");
static_assert(MAX_SERVINT_DAYS <= MAX_SLE_UINT16, "Maximum value for CompanySettings.vehicle.servint_roadveh exceeds storage size");
static_assert(MAX_SERVINT_DAYS <= MAX_SLE_UINT16, "Maximum value for CompanySettings.vehicle.servint_ships exceeds storage size");
static_assert(MAX_SERVINT_DAYS <= MAX_SLE_UINT16, "Maximum value for CompanySettings.vehicle.servint_aircraft exceeds storage size");
static_assert(UINT16_MAX <= MAX_SLE_UINT16, "Maximum value for CurrencySpec.rate exceeds storage size");
static_assert(CalendarTime::MAX_YEAR <= MAX_SLE_INT32, "Maximum value for CurrencySpec.to_euro exceeds storage size");
static_assert(MAX_COMPANIES - 1 <= MAX_SLE_UINT8, "Maximum value for GameSettings.difficulty.max_no_competitors exceeds storage size");
static_assert(MAX_COMPETITORS_INTERVAL <= MAX_SLE_UINT16, "Maximum value for GameSettings.difficulty.competitors_interval exceeds storage size");
static_assert(3 <= MAX_SLE_UINT8, "Maximum value for GameSettings.difficulty.competitor_start_time exceeds storage size");
static_assert(4 <= MAX_SLE_UINT8, "Maximum value for GameSettings.difficulty.number_towns exceeds storage size");
static_assert(ID_END - 1 <= MAX_SLE_UINT8, "Maximum value for GameSettings.difficulty.industry_density exceeds storage size");
static_assert(MAX_LOAN_LIMIT <= MAX_SLE_UINT32, "Maximum value for GameSettings.difficulty.max_loan exceeds storage size");
static_assert(4 <= MAX_SLE_UINT8, "Maximum value for GameSettings.difficulty.initial_interest exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for GameSettings.difficulty.vehicle_costs exceeds storage size");
static_assert(4 <= MAX_SLE_UINT8, "Maximum value for GameSettings.difficulty.competitor_speed exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for GameSettings.difficulty.competitor_intelligence exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for GameSettings.difficulty.vehicle_breakdowns exceeds storage size");
static_assert(3 <= MAX_SLE_UINT8, "Maximum value for GameSettings.difficulty.subsidy_multiplier exceeds storage size");
static_assert(5000 <= MAX_SLE_UINT16, "Maximum value for GameSettings.difficulty.subsidy_duration exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for GameSettings.difficulty.construction_cost exceeds storage size");
static_assert(5 <= MAX_SLE_UINT8, "Maximum value for GameSettings.difficulty.terrain_type exceeds storage size");
static_assert(4 <= MAX_SLE_UINT8, "Maximum value for GameSettings.difficulty.quantity_sea_lakes exceeds storage size");
static_assert(3 <= MAX_SLE_UINT8, "Maximum value for GameSettings.difficulty.town_council_tolerance exceeds storage size");
static_assert(3 <= MAX_SLE_UINT8, "Maximum value for _old_diff_level exceeds storage size");
static_assert(NUM_TLS - 1 <= MAX_SLE_UINT8, "Maximum value for GameSettings.economy.town_layout exceeds storage size");
static_assert(TF_END - 1 <= MAX_SLE_UINT8, "Maximum value for GameSettings.economy.found_town exceeds storage size");
static_assert(PH_ALLOWED_CONSTRUCTED <= MAX_SLE_UINT8, "Maximum value for GameSettings.economy.place_houses exceeds storage size");
static_assert(TCGM_END - 1 <= MAX_SLE_UINT8, "Maximum value for GameSettings.economy.town_cargogen_mode exceeds storage size");
static_assert(ET_END - 1 <= MAX_SLE_UINT8, "Maximum value for GameSettings.economy.type exceeds storage size");
static_assert(100 <= MAX_SLE_UINT8, "Maximum value for GameSettings.economy.feeder_payment_share exceeds storage size");
static_assert(4 <= MAX_SLE_UINT8, "Maximum value for GameSettings.economy.town_growth_rate exceeds storage size");
static_assert(255 <= MAX_SLE_UINT8, "Maximum value for GameSettings.economy.larger_towns exceeds storage size");
static_assert(10 <= MAX_SLE_UINT8, "Maximum value for GameSettings.economy.initial_city_size exceeds storage size");
static_assert(60 <= MAX_SLE_UINT8, "Maximum value for GameSettings.economy.dist_local_authority exceeds storage size");
static_assert(65535 <= MAX_SLE_UINT16, "Maximum value for GameSettings.economy.town_noise_population[0] exceeds storage size");
static_assert(65535 <= MAX_SLE_UINT16, "Maximum value for GameSettings.economy.town_noise_population[1] exceeds storage size");
static_assert(65535 <= MAX_SLE_UINT16, "Maximum value for GameSettings.economy.town_noise_population[2] exceeds storage size");
static_assert(65535 <= MAX_SLE_UINT16, "Maximum value for GameSettings.economy.town_noise_population[3] exceeds storage size");
static_assert(TKU_WALLCLOCK <= MAX_SLE_UINT8, "Maximum value for GameSettings.economy.timekeeping_units exceeds storage size");
static_assert(CalendarTime::MAX_MINUTES_PER_YEAR <= MAX_SLE_UINT16, "Maximum value for GameSettings.economy.minutes_per_calendar_year exceeds storage size");
static_assert(MAX_CARGO_SCALE <= MAX_SLE_UINT16, "Maximum value for GameSettings.economy.town_cargo_scale exceeds storage size");
static_assert(MAX_CARGO_SCALE <= MAX_SLE_UINT16, "Maximum value for GameSettings.economy.industry_cargo_scale exceeds storage size");
static_assert(64 <= MAX_SLE_UINT8, "Maximum value for GameSettings.station.station_spread exceeds storage size");
static_assert(1 <= MAX_SLE_UINT8, "Maximum value for GameSettings.vehicle.road_side exceeds storage size");
static_assert(1 <= MAX_SLE_UINT8, "Maximum value for GameSettings.vehicle.train_acceleration_model exceeds storage size");
static_assert(1 <= MAX_SLE_UINT8, "Maximum value for GameSettings.vehicle.roadveh_acceleration_model exceeds storage size");
static_assert(10 <= MAX_SLE_UINT8, "Maximum value for GameSettings.vehicle.train_slope_steepness exceeds storage size");
static_assert(10 <= MAX_SLE_UINT8, "Maximum value for GameSettings.vehicle.roadveh_slope_steepness exceeds storage size");
static_assert(64 <= MAX_SLE_UINT8, "Maximum value for GameSettings.vehicle.max_train_length exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for GameSettings.vehicle.smoke_amount exceeds storage size");
static_assert(5000 <= MAX_SLE_UINT16, "Maximum value for GameSettings.vehicle.max_trains exceeds storage size");
static_assert(5000 <= MAX_SLE_UINT16, "Maximum value for GameSettings.vehicle.max_roadveh exceeds storage size");
static_assert(5000 <= MAX_SLE_UINT16, "Maximum value for GameSettings.vehicle.max_aircraft exceeds storage size");
static_assert(5000 <= MAX_SLE_UINT16, "Maximum value for GameSettings.vehicle.max_ships exceeds storage size");
static_assert(255 <= MAX_SLE_UINT8, "Maximum value for GameSettings.vehicle.freight_trains exceeds storage size");
static_assert(4 <= MAX_SLE_UINT8, "Maximum value for GameSettings.vehicle.plane_speed exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for GameSettings.vehicle.plane_crashes exceeds storage size");
static_assert(100 <= MAX_SLE_UINT8, "Maximum value for GameSettings.vehicle.extend_vehicle_life exceeds storage size");
static_assert(800 <= MAX_SLE_UINT16, "Maximum value for _old_vds.servint_trains exceeds storage size");
static_assert(800 <= MAX_SLE_UINT16, "Maximum value for _old_vds.servint_roadveh exceeds storage size");
static_assert(800 <= MAX_SLE_UINT16, "Maximum value for _old_vds.servint_ships exceeds storage size");
static_assert(800 <= MAX_SLE_UINT16, "Maximum value for _old_vds.servint_aircraft exceeds storage size");
static_assert(1440 <= MAX_SLE_UINT32, "Maximum value for gui.autosave_interval exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for gui.date_format_in_default_names exceeds storage size");
static_assert(3 <= MAX_SLE_UINT8, "Maximum value for gui.auto_scrolling exceeds storage size");
#ifdef UNIX
static_assert(VSM_MAP_LMB <= MAX_SLE_UINT8, "Maximum value for gui.scroll_mode exceeds storage size");
#endif
#ifndef UNIX
static_assert(VSM_MAP_LMB <= MAX_SLE_UINT8, "Maximum value for gui.scroll_mode exceeds storage size");
#endif
static_assert(RCC_YES_EXCEPT_STICKY <= MAX_SLE_UINT8, "Maximum value for gui.right_click_wnd_close exceeds storage size");
#ifdef __APPLE__
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for gui.right_mouse_btn_emulation exceeds storage size");
#endif
static_assert(20 <= MAX_SLE_UINT8, "Maximum value for gui.errmsg_duration exceeds storage size");
static_assert(6000 <= MAX_SLE_UINT16, "Maximum value for gui.hover_delay_ms exceeds storage size");
static_assert(3 <= MAX_SLE_UINT8, "Maximum value for gui.osk_activation exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for gui.toolbar_pos exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for gui.statusbar_pos exceeds storage size");
static_assert(32 <= MAX_SLE_UINT8, "Maximum value for gui.window_snap_radius exceeds storage size");
static_assert(255 <= MAX_SLE_UINT8, "Maximum value for gui.window_soft_limit exceeds storage size");
static_assert(to_underlying(ZoomLevel::Normal) <= MAX_SLE_UINT8, "Maximum value for gui.zoom_min exceeds storage size");
static_assert(to_underlying(ZoomLevel::Max) <= MAX_SLE_UINT8, "Maximum value for gui.zoom_max exceeds storage size");
static_assert(to_underlying(ZoomLevel::Normal) <= MAX_SLE_UINT8, "Maximum value for gui.sprite_zoom_min exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for gui.smallmap_land_colour exceeds storage size");
static_assert(3 <= MAX_SLE_UINT8, "Maximum value for gui.linkgraph_colours exceeds storage size");
static_assert(LIT_ALL <= MAX_SLE_UINT8, "Maximum value for gui.liveries exceeds storage size");
static_assert(COLOUR_END <= MAX_SLE_UINT8, "Maximum value for gui.starting_colour exceeds storage size");
static_assert(COLOUR_END <= MAX_SLE_UINT8, "Maximum value for gui.starting_colour_secondary exceeds storage size");
static_assert(SWS_OFF <= MAX_SLE_UINT8, "Maximum value for gui.scrollwheel_scrolling exceeds storage size");
static_assert(15 <= MAX_SLE_UINT8, "Maximum value for gui.scrollwheel_multiplier exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for gui.advanced_vehicle_list exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for gui.timetable_mode exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for gui.loading_indicators exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for gui.default_rail_type exceeds storage size");
static_assert(1 <= MAX_SLE_UINT8, "Maximum value for gui.signal_gui_mode exceeds storage size");
static_assert(5 <= MAX_SLE_UINT8, "Maximum value for gui.default_signal_type exceeds storage size");
static_assert(CalendarTime::MAX_YEAR <= MAX_SLE_INT32, "Maximum value for gui.coloured_news_year exceeds storage size");
static_assert(SIGNAL_CYCLE_ALL <= MAX_SLE_UINT8, "Maximum value for gui.cycle_signal_types exceeds storage size");
static_assert(20 <= MAX_SLE_UINT8, "Maximum value for gui.drag_signals_density exceeds storage size");
static_assert(CalendarTime::MAX_YEAR <= MAX_SLE_INT32, "Maximum value for gui.semaphore_build_before exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for gui.order_review_system exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for gui.stop_location exceeds storage size");
static_assert(255 <= MAX_SLE_UINT8, "Maximum value for gui.max_num_autosaves exceeds storage size");
static_assert(255 <= MAX_SLE_UINT8, "Maximum value for gui.news_message_timeout exceeds storage size");
static_assert(7 <= MAX_SLE_UINT8, "Maximum value for gui.station_numtracks exceeds storage size");
static_assert(7 <= MAX_SLE_UINT8, "Maximum value for gui.station_platlength exceeds storage size");
static_assert(5 <= MAX_SLE_UINT8, "Maximum value for gui.station_gui_group_order exceeds storage size");
static_assert(3 <= MAX_SLE_UINT8, "Maximum value for gui.station_gui_sort_by exceeds storage size");
static_assert(1 <= MAX_SLE_UINT8, "Maximum value for gui.station_gui_sort_order exceeds storage size");
static_assert(UINT8_MAX <= MAX_SLE_UINT8, "Maximum value for gui.missing_strings_threshold exceeds storage size");
static_assert(5 <= MAX_SLE_UINT8, "Maximum value for gui.graph_line_thickness exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for gui.settings_restriction_mode exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for gui.developer exceeds storage size");
static_assert(1 <= MAX_SLE_UINT8, "Maximum value for gui.newgrf_default_palette exceeds storage size");
static_assert(65500 <= MAX_SLE_UINT16, "Maximum value for gui.console_backlog_timeout exceeds storage size");
static_assert(65500 <= MAX_SLE_UINT16, "Maximum value for gui.console_backlog_length exceeds storage size");
static_assert(1000 <= MAX_SLE_UINT16, "Maximum value for gui.refresh_rate exceeds storage size");
static_assert(50000 <= MAX_SLE_UINT16, "Maximum value for gui.fast_forward_speed_limit exceeds storage size");
static_assert(100 <= MAX_SLE_UINT16, "Maximum value for gui.network_chat_box_width_pct exceeds storage size");
static_assert(255 <= MAX_SLE_UINT8, "Maximum value for gui.network_chat_box_height exceeds storage size");
static_assert(65535 <= MAX_SLE_UINT16, "Maximum value for gui.network_chat_timeout exceeds storage size");
static_assert(90 <= MAX_SLE_UINT16, "Maximum value for GameSettings.linkgraph.recalc_interval exceeds storage size");
static_assert(9000 <= MAX_SLE_UINT16, "Maximum value for GameSettings.linkgraph.recalc_time exceeds storage size");
static_assert(DT_MAX <= MAX_SLE_UINT8, "Maximum value for GameSettings.linkgraph.distribution_pax exceeds storage size");
static_assert(DT_MAX <= MAX_SLE_UINT8, "Maximum value for GameSettings.linkgraph.distribution_mail exceeds storage size");
static_assert(DT_MAX <= MAX_SLE_UINT8, "Maximum value for GameSettings.linkgraph.distribution_armoured exceeds storage size");
static_assert(DT_MAX_NONSYMMETRIC <= MAX_SLE_UINT8, "Maximum value for GameSettings.linkgraph.distribution_default exceeds storage size");
static_assert(64 <= MAX_SLE_UINT8, "Maximum value for GameSettings.linkgraph.accuracy exceeds storage size");
static_assert(255 <= MAX_SLE_UINT8, "Maximum value for GameSettings.linkgraph.demand_distance exceeds storage size");
static_assert(100 <= MAX_SLE_UINT8, "Maximum value for GameSettings.linkgraph.demand_size exceeds storage size");
static_assert(250 <= MAX_SLE_UINT8, "Maximum value for GameSettings.linkgraph.short_path_saturation exceeds storage size");
static_assert(CURRENCY_END - 1 <= MAX_SLE_UINT8, "Maximum value for GameSettings.locale.currency exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for _old_units exceeds storage size");
static_assert(4 <= MAX_SLE_UINT8, "Maximum value for GameSettings.locale.units_velocity exceeds storage size");
static_assert(4 <= MAX_SLE_UINT8, "Maximum value for GameSettings.locale.units_velocity_nautical exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for GameSettings.locale.units_power exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for GameSettings.locale.units_weight exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for GameSettings.locale.units_volume exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for GameSettings.locale.units_force exceeds storage size");
static_assert(3 <= MAX_SLE_UINT8, "Maximum value for GameSettings.locale.units_height exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for _support8bpp exceeds storage size");
#ifdef HAS_TRUETYPE_FONT
static_assert(72 <= MAX_SLE_UINT, "Maximum value for _fcsettings.small.size exceeds storage size");
#endif
#ifdef HAS_TRUETYPE_FONT
static_assert(72 <= MAX_SLE_UINT, "Maximum value for _fcsettings.medium.size exceeds storage size");
#endif
#ifdef HAS_TRUETYPE_FONT
static_assert(72 <= MAX_SLE_UINT, "Maximum value for _fcsettings.large.size exceeds storage size");
#endif
#ifdef HAS_TRUETYPE_FONT
static_assert(72 <= MAX_SLE_UINT, "Maximum value for _fcsettings.mono.size exceeds storage size");
#endif
static_assert(512 <= MAX_SLE_UINT, "Maximum value for _sprite_cache_size exceeds storage size");
static_assert(0x1FF <= MAX_SLE_UINT, "Maximum value for _transparency_opt exceeds storage size");
static_assert(0x1FF <= MAX_SLE_UINT, "Maximum value for _transparency_lock exceeds storage size");
static_assert(0xFF <= MAX_SLE_UINT, "Maximum value for _invisibility_opt exceeds storage size");
static_assert(UINT32_MAX <= MAX_SLE_UINT32, "Maximum value for _settings_client.gui.last_newgrf_count exceeds storage size");
static_assert(MAX_INTERFACE_SCALE <= MAX_SLE_INT32, "Maximum value for _gui_scale_cfg exceeds storage size");
static_assert(5 <= MAX_SLE_UINT8, "Maximum value for music.playlist exceeds storage size");
static_assert(127 <= MAX_SLE_UINT8, "Maximum value for music.music_vol exceeds storage size");
static_assert(127 <= MAX_SLE_UINT8, "Maximum value for music.effect_vol exceeds storage size");
static_assert(100 <= MAX_SLE_UINT16, "Maximum value for network.sync_freq exceeds storage size");
static_assert(100 <= MAX_SLE_UINT8, "Maximum value for network.frame_freq exceeds storage size");
static_assert(65535 <= MAX_SLE_UINT16, "Maximum value for network.commands_per_frame exceeds storage size");
static_assert(65535 <= MAX_SLE_UINT16, "Maximum value for network.commands_per_frame_server exceeds storage size");
static_assert(65535 <= MAX_SLE_UINT16, "Maximum value for network.max_commands_in_queue exceeds storage size");
static_assert(65535 <= MAX_SLE_UINT16, "Maximum value for network.bytes_per_frame exceeds storage size");
static_assert(65535 <= MAX_SLE_UINT16, "Maximum value for network.bytes_per_frame_burst exceeds storage size");
static_assert(32000 <= MAX_SLE_UINT16, "Maximum value for network.max_init_time exceeds storage size");
static_assert(32000 <= MAX_SLE_UINT16, "Maximum value for network.max_join_time exceeds storage size");
static_assert(32000 <= MAX_SLE_UINT16, "Maximum value for network.max_download_time exceeds storage size");
static_assert(32000 <= MAX_SLE_UINT16, "Maximum value for network.max_password_time exceeds storage size");
static_assert(32000 <= MAX_SLE_UINT16, "Maximum value for network.max_lag_time exceeds storage size");
static_assert(65535 <= MAX_SLE_UINT16, "Maximum value for network.server_port exceeds storage size");
static_assert(65535 <= MAX_SLE_UINT16, "Maximum value for network.server_admin_port exceeds storage size");
static_assert(SERVER_GAME_TYPE_INVITE_ONLY <= MAX_SLE_UINT8, "Maximum value for network.server_game_type exceeds storage size");
static_assert(240 <= MAX_SLE_UINT8, "Maximum value for network.autoclean_protected exceeds storage size");
static_assert(240 <= MAX_SLE_UINT8, "Maximum value for network.autoclean_novehicles exceeds storage size");
static_assert(MAX_COMPANIES <= MAX_SLE_UINT8, "Maximum value for network.max_companies exceeds storage size");
static_assert(MAX_CLIENTS <= MAX_SLE_UINT8, "Maximum value for network.max_clients exceeds storage size");
static_assert(CalendarTime::MAX_YEAR <= MAX_SLE_INT32, "Maximum value for network.restart_game_year exceeds storage size");
static_assert(UINT16_MAX <= MAX_SLE_UINT16, "Maximum value for network.restart_hours exceeds storage size");
static_assert(MAX_CLIENTS <= MAX_SLE_UINT8, "Maximum value for network.min_active_clients exceeds storage size");
static_assert(URS_ALLOW <= MAX_SLE_UINT8, "Maximum value for network.use_relay_service exceeds storage size");
static_assert(PS_YES <= MAX_SLE_UINT8, "Maximum value for network.participate_survey exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for news_display.arrival_player exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for news_display.arrival_other exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for news_display.accident exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for news_display.accident_other exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for news_display.company_info exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for news_display.open exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for news_display.close exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for news_display.economy exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for news_display.production_player exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for news_display.production_other exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for news_display.production_nobody exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for news_display.advice exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for news_display.new_vehicles exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for news_display.acceptance exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for news_display.subsidies exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for news_display.general exceeds storage size");
static_assert(SP_CUSTOM <= MAX_SLE_UINT8, "Maximum value for _old_diff_level exceeds storage size");
static_assert(CURRENCY_END - 1 <= MAX_SLE_UINT8, "Maximum value for GameSettings.locale.currency exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for _old_units exceeds storage size");
static_assert(255 <= MAX_SLE_UINT8, "Maximum value for GameSettings.game_creation.town_name exceeds storage size");
static_assert(3 <= MAX_SLE_UINT8, "Maximum value for GameSettings.game_creation.landscape exceeds storage size");
static_assert(UINT8_MAX <= MAX_SLE_UINT8, "Maximum value for GameSettings.game_creation.snow_line_height exceeds storage size");
static_assert(1 <= MAX_SLE_UINT8, "Maximum value for GameSettings.vehicle.road_side exceeds storage size");
static_assert(255 <= MAX_SLE_UINT8, "Maximum value for GameSettings.pf.wait_oneway_signal exceeds storage size");
static_assert(255 <= MAX_SLE_UINT8, "Maximum value for GameSettings.pf.wait_twoway_signal exceeds storage size");
static_assert(255 <= MAX_SLE_UINT8, "Maximum value for GameSettings.pf.wait_for_pbs_path exceeds storage size");
static_assert(255 <= MAX_SLE_UINT8, "Maximum value for GameSettings.pf.path_backoff_interval exceeds storage size");
static_assert(1000000 <= MAX_SLE_UINT, "Maximum value for GameSettings.pf.yapf.max_search_nodes exceeds storage size");
static_assert(1000000 <= MAX_SLE_UINT, "Maximum value for GameSettings.pf.yapf.rail_firstred_penalty exceeds storage size");
static_assert(1000000 <= MAX_SLE_UINT, "Maximum value for GameSettings.pf.yapf.rail_firstred_exit_penalty exceeds storage size");
static_assert(1000000 <= MAX_SLE_UINT, "Maximum value for GameSettings.pf.yapf.rail_lastred_penalty exceeds storage size");
static_assert(1000000 <= MAX_SLE_UINT, "Maximum value for GameSettings.pf.yapf.rail_lastred_exit_penalty exceeds storage size");
static_assert(1000000 <= MAX_SLE_UINT, "Maximum value for GameSettings.pf.yapf.rail_station_penalty exceeds storage size");
static_assert(1000000 <= MAX_SLE_UINT, "Maximum value for GameSettings.pf.yapf.rail_slope_penalty exceeds storage size");
static_assert(1000000 <= MAX_SLE_UINT, "Maximum value for GameSettings.pf.yapf.rail_curve45_penalty exceeds storage size");
static_assert(1000000 <= MAX_SLE_UINT, "Maximum value for GameSettings.pf.yapf.rail_curve90_penalty exceeds storage size");
static_assert(1000000 <= MAX_SLE_UINT, "Maximum value for GameSettings.pf.yapf.rail_depot_reverse_penalty exceeds storage size");
static_assert(1000000 <= MAX_SLE_UINT, "Maximum value for GameSettings.pf.yapf.rail_crossing_penalty exceeds storage size");
static_assert(100 <= MAX_SLE_UINT, "Maximum value for GameSettings.pf.yapf.rail_look_ahead_max_signals exceeds storage size");
static_assert(1000000 <= MAX_SLE_INT, "Maximum value for GameSettings.pf.yapf.rail_look_ahead_signal_p0 exceeds storage size");
static_assert(1000000 <= MAX_SLE_INT, "Maximum value for GameSettings.pf.yapf.rail_look_ahead_signal_p1 exceeds storage size");
static_assert(1000000 <= MAX_SLE_INT, "Maximum value for GameSettings.pf.yapf.rail_look_ahead_signal_p2 exceeds storage size");
static_assert(1000000 <= MAX_SLE_UINT, "Maximum value for GameSettings.pf.yapf.rail_pbs_cross_penalty exceeds storage size");
static_assert(1000000 <= MAX_SLE_UINT, "Maximum value for GameSettings.pf.yapf.rail_pbs_station_penalty exceeds storage size");
static_assert(1000000 <= MAX_SLE_UINT, "Maximum value for GameSettings.pf.yapf.rail_pbs_signal_back_penalty exceeds storage size");
static_assert(1000000 <= MAX_SLE_UINT, "Maximum value for GameSettings.pf.yapf.rail_doubleslip_penalty exceeds storage size");
static_assert(20000 <= MAX_SLE_UINT, "Maximum value for GameSettings.pf.yapf.rail_longer_platform_penalty exceeds storage size");
static_assert(20000 <= MAX_SLE_UINT, "Maximum value for GameSettings.pf.yapf.rail_longer_platform_per_tile_penalty exceeds storage size");
static_assert(20000 <= MAX_SLE_UINT, "Maximum value for GameSettings.pf.yapf.rail_shorter_platform_penalty exceeds storage size");
static_assert(20000 <= MAX_SLE_UINT, "Maximum value for GameSettings.pf.yapf.rail_shorter_platform_per_tile_penalty exceeds storage size");
static_assert(1000000 <= MAX_SLE_UINT, "Maximum value for GameSettings.pf.yapf.road_slope_penalty exceeds storage size");
static_assert(1000000 <= MAX_SLE_UINT, "Maximum value for GameSettings.pf.yapf.road_curve_penalty exceeds storage size");
static_assert(1000000 <= MAX_SLE_UINT, "Maximum value for GameSettings.pf.yapf.road_crossing_penalty exceeds storage size");
static_assert(1000000 <= MAX_SLE_UINT, "Maximum value for GameSettings.pf.yapf.road_stop_penalty exceeds storage size");
static_assert(1000000 <= MAX_SLE_UINT, "Maximum value for GameSettings.pf.yapf.road_stop_occupied_penalty exceeds storage size");
static_assert(1000000 <= MAX_SLE_UINT, "Maximum value for GameSettings.pf.yapf.road_stop_bay_occupied_penalty exceeds storage size");
static_assert(1000000 <= MAX_SLE_UINT, "Maximum value for GameSettings.pf.yapf.maximum_go_to_depot_penalty exceeds storage size");
static_assert(1000000 <= MAX_SLE_UINT, "Maximum value for GameSettings.pf.yapf.ship_curve45_penalty exceeds storage size");
static_assert(1000000 <= MAX_SLE_UINT, "Maximum value for GameSettings.pf.yapf.ship_curve90_penalty exceeds storage size");
static_assert(250000 <= MAX_SLE_UINT32, "Maximum value for GameSettings.script.script_max_opcode_till_suspend exceeds storage size");
static_assert(8192 <= MAX_SLE_UINT32, "Maximum value for GameSettings.script.script_max_memory_megabytes exceeds storage size");
static_assert(32000 <= MAX_SLE_INT16, "Maximum value for WindowDesc.pref_width exceeds storage size");
static_assert(32000 <= MAX_SLE_INT16, "Maximum value for WindowDesc.pref_height exceeds storage size");
static_assert(255 <= MAX_SLE_UINT8, "Maximum value for GameSettings.game_creation.town_name exceeds storage size");
static_assert(3 <= MAX_SLE_UINT8, "Maximum value for GameSettings.game_creation.landscape exceeds storage size");
static_assert(MAX_MAP_HEIGHT_LIMIT <= MAX_SLE_UINT8, "Maximum value for GameSettings.game_creation.heightmap_height exceeds storage size");
static_assert(MAX_SNOWLINE_HEIGHT <= MAX_SLE_UINT8, "Maximum value for GameSettings.game_creation.snow_line_height exceeds storage size");
static_assert(100 <= MAX_SLE_UINT8, "Maximum value for GameSettings.game_creation.snow_coverage exceeds storage size");
static_assert(100 <= MAX_SLE_UINT8, "Maximum value for GameSettings.game_creation.desert_coverage exceeds storage size");
static_assert(CalendarTime::MAX_YEAR <= MAX_SLE_INT32, "Maximum value for GameSettings.game_creation.starting_year exceeds storage size");
static_assert(CalendarTime::MAX_YEAR - 1 <= MAX_SLE_INT32, "Maximum value for GameSettings.game_creation.ending_year exceeds storage size");
static_assert(1 <= MAX_SLE_UINT8, "Maximum value for GameSettings.game_creation.land_generator exceeds storage size");
static_assert(128 <= MAX_SLE_UINT8, "Maximum value for GameSettings.game_creation.oil_refinery_limit exceeds storage size");
static_assert(TGEN_SMOOTHNESS_END - 1 <= MAX_SLE_UINT8, "Maximum value for GameSettings.game_creation.tgen_smoothness exceeds storage size");
static_assert(5 <= MAX_SLE_UINT8, "Maximum value for GameSettings.game_creation.variety exceeds storage size");
static_assert(UINT32_MAX <= MAX_SLE_UINT32, "Maximum value for GameSettings.game_creation.generation_seed exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for GameSettings.game_creation.tree_placer exceeds storage size");
static_assert(1 <= MAX_SLE_UINT8, "Maximum value for GameSettings.game_creation.heightmap_rotation exceeds storage size");
static_assert(15 <= MAX_SLE_UINT8, "Maximum value for GameSettings.game_creation.se_flat_world_height exceeds storage size");
static_assert(MAX_MAP_SIZE_BITS <= MAX_SLE_UINT8, "Maximum value for GameSettings.game_creation.map_x exceeds storage size");
static_assert(MAX_MAP_SIZE_BITS <= MAX_SLE_UINT8, "Maximum value for GameSettings.game_creation.map_y exceeds storage size");
static_assert(16 <= MAX_SLE_UINT8, "Maximum value for GameSettings.game_creation.water_borders exceeds storage size");
static_assert(BFP_INFINITE_WATER <= MAX_SLE_UINT8, "Maximum value for GameSettings.game_creation.water_border_presets exceeds storage size");
static_assert(5000 <= MAX_SLE_UINT16, "Maximum value for GameSettings.game_creation.custom_town_number exceeds storage size");
static_assert(64000 <= MAX_SLE_UINT16, "Maximum value for GameSettings.game_creation.custom_industry_number exceeds storage size");
static_assert(MAX_MAP_HEIGHT_LIMIT <= MAX_SLE_UINT8, "Maximum value for GameSettings.game_creation.custom_terrain_type exceeds storage size");
static_assert(CUSTOM_SEA_LEVEL_MAX_PERCENTAGE <= MAX_SLE_UINT8, "Maximum value for GameSettings.game_creation.custom_sea_level exceeds storage size");
static_assert(255 <= MAX_SLE_UINT8, "Maximum value for GameSettings.game_creation.min_river_length exceeds storage size");
static_assert(255 <= MAX_SLE_UINT8, "Maximum value for GameSettings.game_creation.river_route_random exceeds storage size");
static_assert(3 <= MAX_SLE_UINT8, "Maximum value for GameSettings.game_creation.amount_of_rivers exceeds storage size");
static_assert(MAX_MAP_HEIGHT_LIMIT <= MAX_SLE_UINT8, "Maximum value for GameSettings.construction.map_height_limit exceeds storage size");
static_assert(3 <= MAX_SLE_UINT8, "Maximum value for GameSettings.construction.command_pause_level exceeds storage size");
static_assert(1 << 30 <= MAX_SLE_UINT32, "Maximum value for GameSettings.construction.terraform_per_64k_frames exceeds storage size");
static_assert(1 << 15 <= MAX_SLE_UINT16, "Maximum value for GameSettings.construction.terraform_frame_burst exceeds storage size");
static_assert(1 << 30 <= MAX_SLE_UINT32, "Maximum value for GameSettings.construction.clear_per_64k_frames exceeds storage size");
static_assert(1 << 15 <= MAX_SLE_UINT16, "Maximum value for GameSettings.construction.clear_frame_burst exceeds storage size");
static_assert(1 << 30 <= MAX_SLE_UINT32, "Maximum value for GameSettings.construction.tree_per_64k_frames exceeds storage size");
static_assert(1 << 15 <= MAX_SLE_UINT16, "Maximum value for GameSettings.construction.tree_frame_burst exceeds storage size");
static_assert(1 << 30 <= MAX_SLE_UINT32, "Maximum value for GameSettings.construction.build_object_per_64k_frames exceeds storage size");
static_assert(1 << 15 <= MAX_SLE_UINT16, "Maximum value for GameSettings.construction.build_object_frame_burst exceeds storage size");
static_assert(MAX_MAP_SIZE <= MAX_SLE_UINT16, "Maximum value for GameSettings.construction.max_bridge_length exceeds storage size");
static_assert(MAX_TILE_HEIGHT <= MAX_SLE_UINT8, "Maximum value for GameSettings.construction.max_bridge_height exceeds storage size");
static_assert(MAX_MAP_SIZE <= MAX_SLE_UINT16, "Maximum value for GameSettings.construction.max_tunnel_length exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for GameSettings.construction.train_signal_side exceeds storage size");
static_assert(2 <= MAX_SLE_UINT8, "Maximum value for GameSettings.construction.raw_industry_construction exceeds storage size");
static_assert(4 <= MAX_SLE_UINT8, "Maximum value for GameSettings.construction.industry_platform exceeds storage size");
static_assert(3 <= MAX_SLE_UINT8, "Maximum value for GameSettings.construction.extra_tree_placement exceeds storage size");
