/*
  SPDX-FileCopyrightText: 2026 Laurent Montel <montel@kde.org>

  SPDX-License-Identifier: GPL-2.0-or-later
*/
#include "ollamacloudconfigurewidget.h"
#include "autogeneratetext_ollamacloud_debug.h"
#include "ollamacloudmanager.h"
#include "ollamacloudsettings.h"
#include "ollamacommoncomboboxwidget.h"
#include "ollamacommonkeepaliveparameterswidget.h"
#include "ollamacommonoptionswidget.h"
#include "ollamacommonoverrideparameterswidget.h"
#include "widgets/common/textautogenerateshowmodelinfodialog.h"
#include <KAuthorized>
#include <KLineEditEventHandler>
#include <KLocalizedString>
#include <KPasswordLineEdit>
#include <QFormLayout>
#include <QLineEdit>
#include <QPointer>
#include <QScrollArea>
#include <QVBoxLayout>

using namespace Qt::Literals::StringLiterals;
OllamaCloudConfigureWidget::OllamaCloudConfigureWidget(OllamaCloudManager *manager, QWidget *parent)
    : QWidget{parent}
    , mName(new QLineEdit(this))
    , mApiKey(new KPasswordLineEdit(this))
    , mManager(manager)
    , mOllamaComboBoxWidget(new OllamaCommonComboBoxWidget(this))
    , mOllamaCommonOverrideParametersWidget(new OllamaCommonOverrideParametersWidget(this))
    , mOllamaCommonKeepAliveParametersWidget(new OllamaCommonKeepAliveParametersWidget(this))
    , mOllamaCommonOptionsWidget(new OllamaCommonOptionsWidget(OllamaCommonOptionsWidget::ExtraOptions(OllamaCommonOptionsWidget::ExtraOption::None), this))

{
    connect(mManager, &OllamaCloudManager::modelsLoadDone, this, [this](const OllamaCloudManager::ModelsInfo &modelinfo) {
        // qDebug() << " OllamaConfigureWidget::fillModels() " << modelinfo;
        if (modelinfo.hasError) {
            qCWarning(AUTOGENERATETEXT_OLLAMACLOUD_LOG) << "load model failed";
        } else {
            mOllamaComboBoxWidget->setModels(modelinfo.models);
        }
    });

    auto centralLayout = new QVBoxLayout(this);
    centralLayout->setObjectName(u"centralLayout"_s);
    centralLayout->setContentsMargins({});

    auto area = new QScrollArea(this);
    centralLayout->addWidget(area);
    area->setWidgetResizable(true);

    auto mainWidget = new QWidget(this);
    area->setWidget(mainWidget);

    auto mainLayout = new QVBoxLayout(mainWidget);

    mainLayout->setObjectName(u"mainLayout"_s);
    mainLayout->setContentsMargins({});

    auto formLayout = new QFormLayout;
    formLayout->setObjectName(u"mainLayout"_s);
    formLayout->setContentsMargins({});
    mainLayout->addLayout(formLayout);

    mName->setObjectName(u"mName"_s);
    KLineEditEventHandler::catchReturnKey(mName);
    formLayout->addRow(i18n("Name:"), mName);

    mApiKey->setObjectName(u"mApiKey"_s);
    mApiKey->setRevealPasswordMode(KAuthorized::authorize(u"lineedit_reveal_password"_s) ? KPassword::RevealMode::OnlyNew : KPassword::RevealMode::Never);
    KLineEditEventHandler::catchReturnKey(mApiKey->lineEdit());
    formLayout->addRow(i18n("Api Key:"), mApiKey);

    mOllamaComboBoxWidget->setObjectName(u"mOllamaComboBoxWidget"_s);
    formLayout->addRow(i18n("Model:"), mOllamaComboBoxWidget);

    mOllamaCommonOverrideParametersWidget->setObjectName(u"mOllamaCommonOverrideParametersWidget"_s);
    mainLayout->addWidget(mOllamaCommonOverrideParametersWidget);

    mOllamaCommonKeepAliveParametersWidget->setObjectName(u"mOllamaCommonKeepAliveParametersWidget"_s);
    mainLayout->addWidget(mOllamaCommonKeepAliveParametersWidget);

    mOllamaCommonOptionsWidget->setObjectName(u"mOllamaCommonOptionsWidget"_s);
    mainLayout->addWidget(mOllamaCommonOptionsWidget);

    mainLayout->addStretch(1);

    connect(mName, &QLineEdit::textChanged, this, [this](const QString &str) {
        Q_EMIT enableOkButton(!str.trimmed().isEmpty());
    });

    connect(mOllamaComboBoxWidget, &OllamaCommonComboBoxWidget::showModelInfoRequested, this, &OllamaCloudConfigureWidget::showModelInfo);
    connect(mOllamaComboBoxWidget, &OllamaCommonComboBoxWidget::reloadModel, this, &OllamaCloudConfigureWidget::fillModels);
    connect(mManager, &OllamaCloudManager::showModelInfoDone, this, &OllamaCloudConfigureWidget::displayModelInfo);
    loadSettings();
    fillModels();
}

OllamaCloudConfigureWidget::~OllamaCloudConfigureWidget() = default;

void OllamaCloudConfigureWidget::restoreToDefaults()
{
    const OllamaCommonOverrideParametersWidget::OverrideParametersInfo parametersInfo{
        .temperature = mManager->ollamaCloudSettings()->defaultTemperature(),
        .seed = mManager->ollamaCloudSettings()->defaultSeed(),
        .contextWindowSize = mManager->ollamaCloudSettings()->defaultContextWindowSize(),
    };
    mOllamaCommonOverrideParametersWidget->setParametersInfo(parametersInfo);
    const OllamaCommonKeepAliveParametersWidget::KeepAliveInfo keepAliveInfo{
        .keepAliveType = mManager->ollamaCloudSettings()->defaultKeepAliveType(),
        .minutes = mManager->ollamaCloudSettings()->defaultKeepAliveMinutes(),
    };
    mOllamaCommonKeepAliveParametersWidget->setKeepAliveInfo(keepAliveInfo);
    const OllamaCommonOptionsWidget::OllamaCommonOptionsInfo optionsInfo{
        .exposeToNetwork = false,
        .thoughtProcessing = mManager->ollamaCloudSettings()->defaultThoughtProcessing(),
        .shareNameType = mManager->ollamaCloudSettings()->defaultShareNameType(),
    };
    mOllamaCommonOptionsWidget->setOptionsInfo(optionsInfo);
}

void OllamaCloudConfigureWidget::fillModels()
{
    mManager->loadModels();
}

void OllamaCloudConfigureWidget::displayModelInfo(const QString &modelStr)
{
    TextAutoGenerateText::TextAutoGenerateShowModelInfoDialog dlg(this);
    dlg.setText(modelStr);
    dlg.exec();
}

void OllamaCloudConfigureWidget::showModelInfo(const QString &modelName)
{
    // qDebug() << " showModelInfo " << modelName;
    mManager->showModelInfo(modelName);
}

void OllamaCloudConfigureWidget::loadSettings()
{
    mName->setText(mManager->ollamaCloudSettings()->displayName());
    mApiKey->setPassword(mManager->apiKey());
    const OllamaCommonOverrideParametersWidget::OverrideParametersInfo parametersInfo{
        .temperature = mManager->ollamaCloudSettings()->temperature(),
        .seed = mManager->ollamaCloudSettings()->seed(),
        .contextWindowSize = mManager->ollamaCloudSettings()->contextWindowSize(),
    };
    mOllamaCommonOverrideParametersWidget->setParametersInfo(parametersInfo);

    const OllamaCommonKeepAliveParametersWidget::KeepAliveInfo keepAliveInfo{
        .keepAliveType = mManager->ollamaCloudSettings()->keepAliveType(),
        .minutes = mManager->ollamaCloudSettings()->keepAliveMinutes(),
    };
    mOllamaCommonKeepAliveParametersWidget->setKeepAliveInfo(keepAliveInfo);

    const OllamaCommonOptionsWidget::OllamaCommonOptionsInfo optionsInfo{
        .exposeToNetwork = false,
        .thoughtProcessing = mManager->ollamaCloudSettings()->thoughtProcessing(),
        .shareNameType = mManager->ollamaCloudSettings()->shareNameType(),
    };
    mOllamaCommonOptionsWidget->setOptionsInfo(optionsInfo);
}

void OllamaCloudConfigureWidget::saveSettings()
{
    mManager->ollamaCloudSettings()->setDisplayName(mName->text());
    mManager->setApiKey(mApiKey->password());
    const auto parametersInfo = mOllamaCommonOverrideParametersWidget->parametersInfo();
    mManager->ollamaCloudSettings()->setTemperature(parametersInfo.temperature);
    mManager->ollamaCloudSettings()->setSeed(parametersInfo.seed);
    mManager->ollamaCloudSettings()->setContextWindowSize(parametersInfo.contextWindowSize);

    const auto keepAliveInfo = mOllamaCommonKeepAliveParametersWidget->keepAliveInfo();
    mManager->ollamaCloudSettings()->setKeepAliveMinutes(keepAliveInfo.minutes);
    mManager->ollamaCloudSettings()->setKeepAliveType(keepAliveInfo.keepAliveType);

    const auto optionsInfo = mOllamaCommonOptionsWidget->optionsInfo();
    mManager->ollamaCloudSettings()->setThoughtProcessing(optionsInfo.thoughtProcessing);
    mManager->ollamaCloudSettings()->setShareNameType(optionsInfo.shareNameType);
}

#include "moc_ollamacloudconfigurewidget.cpp"
